#include "cs.h"
#include "cs_json.h"
#include "gool/gool-types.h"

namespace tis {
  struct FrameDispatch;
  extern FrameDispatch CsTopCDispatch;
  extern int           Send(VM *c, FrameDispatch *d, int argc);

  bool CsSendMessageByNameJSON(VM *c, value obj, const char *sname, int argc,
                               const tool::value *argv, tool::value &retval) {
    if (obj == UNDEFINED_VALUE) { return false; }

    int n;

    tool::ustring funcname(sname);

    value tag    = CsSymbolOf(sname);
    value method = UNDEFINED_VALUE;

    PROTECT(obj, method);

    if (CsNamespaceP(obj)) {
      auto_scope as(c, obj);
      method = CsEvalString(&as, obj, funcname, funcname.length());
    }
    else if (!CsGetProperty(c, obj, tag, &method)) 
    {
      if (funcname().contains_one_of(WCHARS(".[]")))
      {
        auto ct = CsReferenceOf(c, obj, funcname());
        obj = ct.first;
        tag = ct.second;
      }
      dispatch* pd = CsGetDispatch(obj);
      if (pd->handleCall) {

        CsPush(c, obj);
        CsPush(c, UNDEFINED_VALUE); /* filled in below */
        CsPush(c, obj);             /* _next */

        for (n = 0; n < argc; ++n)
          CsPush(c, value_to_value(c, argv[n]));
        value rv = 0;
        if (pd->handleCall(c, obj, tag, argc + 2, &rv) && rv) {
          retval = value_to_value(c, rv);
          CsDrop(c, argc + 3);
          return true;
        }
        CsDrop(c, argc + 3);
      }
      return false;
    }

    c->val = UNDEFINED_VALUE;

    TRY {
      if (!CsMethodP(method) && !CsCMethodP(method)) {
        // tag = method;
        return false;
        // if( optional && method == UNDEFINED_VALUE)
        //  return tool::value();
        // else
        //  CsThrowKnownError(c, CsErrUnexpectedTypeError, method, sname );
      }

      /* reserve space for the obj, selector, _next and arguments */
      CsCheck(c, argc + 3);

      /* push the obj, selector and _next argument */
      CsPush(c, obj);
      CsPush(c, UNDEFINED_VALUE); /* filled in below */
      CsPush(c, obj);             /* _next */

      /* push the arguments */
      for (n = 0; n < argc; ++n)
        CsPush(c, value_to_value(c, argv[n]));

      /* fill in the selector (because interning can cons) */
      c->sp[argc + 1] = method;

      /* setup the call */
      if (Send(c, &tis::CsTopCDispatch, argc + 2)) {
        retval = value_to_value(c, c->val);
        return true;
      }

      /* execute the method */
      c->exec->run(c);
    }
    CATCH_ERROR_NE {
      CsHandleUnhandledError(c);
      retval = value_to_value(c, c->val);
      return false;
    }
    retval = value_to_value(c, c->val);
    return true;
  }

  tool::value call_by_tool(tis::pvalue &method, value self, uint argc,
                           const tool::value *argv) {
    VM *c = method.pvm;

    if (!self || self == UNDEFINED_VALUE) self = CsCurrentScope(c)->globals;

    auto_scope as(c, self);

    CsSavedState state(c);
    c->val = UNDEFINED_VALUE;
    TRY {
      if (!CsMethodP(method.val) && !CsCMethodP(method.val)) {
        CsThrowKnownError(c, CsErrUnexpectedTypeError, method.val, "(?)");
      }

      /* reserve space for the obj, selector, _next and arguments */
      CsCheck(c, argc + 3);

      /* push the obj, selector and _next argument */
      // value vself = as.globals;
      CsPush(c, CsCurrentScope(c)->globals);
      CsPush(c, method.val);                 /* filled in below */
      CsPush(c, CsCurrentScope(c)->globals); /* _next */

      /* push the arguments */
      for (uint n = 0; n < argc; ++n) {
        value arg = value_to_value(c, argv[n]);
        CsPush(c, arg);
      }

      /* fill in the selector (because interning can cons) */
      // c->sp[argc + 1] = method.val; //CsSymbolOf(sname);

      /* setup the call */
      if (Send(c, &tis::CsTopCDispatch, argc + 2)) {
        return value_to_value(c, c->val);
      }

      /* execute the method */
      c->exec->run(c);
    }
    catch (tis::script_exception &) {
      tool::value r = tis::value_to_value(c, c->val);
      // e;
      CsHandleUnhandledError(c);
      state.restore();
      return r;
    }
    return value_to_value(c, c->val);
  }

#if JSON_ISOLATION < 3
  struct object_proxy : public tool::object_proxy {
    // object_proxy* next;
    pvalue pin;
    object_proxy(VM *c, tis::value obj = 0) : pin(c, obj) {}

    virtual ~object_proxy() { pin.unpin(); }

      // virtual void finalize();

#define EXEC_START pin.pvm->send([this,&r]() -> bool {
#define EXEC_START_1(p) pin.pvm->send([this,&r,p]() -> bool {
#define EXEC_START_2(p1, p2) pin.pvm->send([this,&r,p1,p2]() -> bool {
#define EXEC_START_3(p1, p2, p3) pin.pvm->send([this,&r,p1,p2,p3]() -> bool {
#define EXEC_END                                                               \
  return true;                                                                 \
  });

    virtual tool::ustring class_name() const {
      if (!pin.pvm || !pin.val) return tool::ustring();
      tool::ustring r;
      EXEC_START
      r = ustring(CsTypeName(pin.val));
      EXEC_END
      return r;
    }
    virtual uint size() const {
      if (!pin.pvm || !pin.val) return 0;
      uint r = 0;

      EXEC_START
      if (CsVectorP(pin.val))
        r = CsVectorSize(pin.pvm, pin.val);
      else if (CsObjectP(pin.val))
        r = CsObjectPropertyCount(pin.val);
      else if (CsByteVectorP(pin.val))
        r = (uint)CsByteVectorSize(pin.val);
      else if (CsMethodP(pin.val))
        r = (uint)CsMethodArgCnt(pin.val);
      EXEC_END
      return r;
    }
    virtual tool::value get_by_index(uint n) const {
      if (!pin.pvm || !pin.val) return tool::value();

      tool::value r;

      EXEC_START_1(n)

      if (CsVectorP(pin.val)) {
        if (n < uint(CsVectorSize(pin.pvm, pin.val)))
          r = value_to_value(pin.pvm, CsVectorElement(pin.pvm, pin.val, n));
      }

      /*else if( CsObjectP(pin.val))
        return CsObjectSize(pin.pvm,pin.val);
      else if( CsByteVectorP(pin.val))
        return CsByteVectorSize(pin.pvm,pin.val);*/

      EXEC_END
      return r;
    }
    virtual bool set_by_index(uint n, const tool::value &val) {
      if (!pin.pvm || !pin.val) return false;

      tool::value v = val;
      bool        r = false;

      EXEC_START_2(n, v)

      if (CsVectorP(pin.val)) {
        if (n >= uint(CsVectorSize(pin.pvm, pin.val)))
          pin.val = CsResizeVector(pin.pvm, pin.val, n);
        tis::value tv = value_to_value(pin.pvm, v);
        CsSetVectorElement(pin.pvm, pin.val, n, tv);
        r = true;
      }
      /*else if( CsObjectP(pin.val))
        return CsObjectSize(pin.pvm,pin.val);
      else if( CsByteVectorP(pin.val))
        return CsByteVectorSize(pin.pvm,pin.val);*/
      EXEC_END
      return r;
    }

    virtual tool::value get_by_key(const tool::value &key) const {
      if (!pin.pvm || !pin.val) return tool::value();

      tool::value r;
      tool::value k = key;

      EXEC_START_1(k)

      value vk = value_to_value(pin.pvm, k);
      if (CsSymbolP(vk)) {
        value rv;
        if (CsGetProperty(pin.pvm, pin.val, vk, &rv))
          r = value_to_value(pin.pvm, rv);
        else
          r = tool::value();
      } else
        r = value_to_value(pin.pvm, CsGetItem(pin.pvm, pin.val, vk));
      EXEC_END
      return r;
    }
    virtual bool set_by_key(const tool::value &key, const tool::value &val) {
      if (!pin.pvm || !pin.val) return false;

      bool        r = false;
      tool::value k = key;
      tool::value v = val;

      EXEC_START_2(k, v)

      CsPush(pin.pvm, value_to_value(pin.pvm, k));
      value vv = value_to_value(pin.pvm, v);
      value vk = CsPop(pin.pvm);
      try {
        if (CsSymbolP(vk))
          CsSetProperty(pin.pvm, pin.val, vk, vv);
        else
          CsSetItem(pin.pvm, pin.val, vk, vv);
        r = true;
      } catch (script_exception &) { r = false; }
      EXEC_END
      return r;
    }

    virtual tool::value invoke(const tool::value &This, uint argc,
                               const tool::value *argv) {
      if (!pin.pvm || !pin.val) return tool::value();

      tool::value r;
      tool::value self = This;

      EXEC_START_3(argc, argv, self)
      r = call_by_tool(pin, value_to_value(pin.pvm, self), argc, argv);
      EXEC_END
      return r;
    }

    virtual bool get_user_data(void **ppv) const {
      if (!pin.pvm || !pin.val) return false;

      bool r = false;
      EXEC_START_1(ppv)
      if (tis::CsCObjectP(pin.val)) {
        *ppv = tis::ptr<tis::c_ptr_object>(pin.val)->ptr;
        r    = true;
      } else
        r = false;
      EXEC_END
      return r;
    }
    virtual bool set_user_data(void *pv) {
      if (!pin.pvm || !pin.val) return false;

      bool r = false;
      EXEC_START_1(pv)
      if (tis::CsCObjectP(pin.val)) {
        tis::ptr<tis::c_ptr_object>(pin.val)->ptr = pv;
        r                                         = true;
      } else
        r = false;
      EXEC_END
      return r;
    }

    virtual bool visit(const tool::kv_visitor &vis) const {
      if (!pin.pvm || !pin.val) 
        return false;

      //tool::kv_visitor vis = vs;
      bool             r   = false;

      EXEC_START_1(vis)
      if (CsVectorP(pin.val)) {
        for (int n = 0; n < CsVectorSize(pin.pvm, pin.val); ++n) {
          tool::value k = tool::value(n);
          tool::value v =
              value_to_value(pin.pvm, CsVectorElement(pin.pvm, pin.val, n));
          if (vis(k, v)) continue;
          break;
        }
        r = true;
      } else if (CsObjectP(pin.val) || CsCObjectP(pin.val)) {
        tis::each_property gen(pin.pvm, pin.val);
        for (tis::value key, val; gen(key, val);) {
          tool::value k = value_to_value(pin.pvm, key);
          tool::value v = value_to_value(pin.pvm, val);
          if (!vis(k, v)) break;
        }
        r = true;
      } else
        r = false;
      EXEC_END
      return r;
    }
    virtual bool equal(const tool::object_proxy *pv) const {
      const object_proxy *pop = (object_proxy *)pv;
      return pop->pin.val == pin.val && pop->pin.pvm == pin.pvm;
    }
    virtual bool isolate(tool::value &r) const {
      if (!pin.pvm || !pin.val) return false;

#if JSON_ISOLATION < 3

      EXEC_START

      tis::value v = pin.val;
      VM *       c = pin.pvm;

      if (CsVectorP(v)) {
        int sz = CsVectorSize(c, v);
        r      = tool::value::make_array(sz);
        for (int i = 0; i < sz; ++i) {
          tool::value t = value_to_value(c, CsVectorElement(c, v, i));
          t.isolate();
          r.set_element(i, t);
        }
        return true;
      } else if (CsObjectOrMethodP(v) || CsTypeP(c, v)) {
        r = tool::value::make_map();
        each_property gen(c, v);
        for (tis::value key, val; gen(key, val);) {
          tool::value tk = value_to_value(c, key);
          tk.isolate();
          tool::value tv = value_to_value(c, val);
          tv.isolate();
          r.set_prop(tk, tv);
        }
        return true;
      } else if (CsErrorP(v)) {
        string_stream s;
        CsDisplay(c, v, &s);
        r = tool::value::make_error(s.to_ustring());
        return true;
      } else
        r = tool::value::null_val();

      EXEC_END

#endif
      return true;
    }

#undef EXEC_START
#undef EXEC_START_1
#undef EXEC_START_2
#undef EXEC_END
  };

  /*struct object_proxy_ctx
  {
    object_proxy* free_list;
    tool::mutex   guard;
    object_proxy_ctx():free_list(0) {}
    ~object_proxy_ctx()
    {
      while(free_list)
      {
        object_proxy* t = free_list;
        free_list = free_list->next;
        delete t;
      }
    }
    static object_proxy_ctx& get()
    {
      static object_proxy_ctx _inst;
      return _inst;
    }
  }; */

  // static int count = 0;

  static object_proxy *create_proxy(VM *c, value v) {
    /*     object_proxy_ctx& ctx = object_proxy_ctx::get();
         tool::critical_section cs(ctx.guard);
         //++count;
         if( ctx.free_list )
         {
           object_proxy* nt = ctx.free_list;
           ctx.free_list = ctx.free_list->next;
           nt->next = 0;
           nt->pin.pin(c,v);
           return nt;
         }*/
    return new object_proxy(c, v);
  }

    /*void object_proxy::finalize()
      {
        //--count;
        object_proxy_ctx& ctx = object_proxy_ctx::get();
        tool::critical_section cs(ctx.guard);
        next = ctx.free_list;
        ctx.free_list = this;
        pin.unpin();
      }*/

#endif // JSON_ISOLATION < 3

  tool::value value_to_value(VM *c, value v, bool isolate) {

#if defined(JSON_ISOLATION)
    isolate = true
#endif

    if (!CsIsValidPtr(c, v)) {
      CsThrowKnownError(c, CsErrGenericErrorW,
                        W("Attempt to access corrupted value"));
    }
    if (CsStringP(v))
      return tool::value(tool::wchars(CsStringAddress(v), CsStringSize(v)));
    if (v == UNDEFINED_VALUE || v == NOTHING_VALUE) return tool::value();
    if (v == NULL_VALUE) return tool::value::null_val();
    if (v == TRUE_VALUE) return tool::value(true);
    if (v == FALSE_VALUE) return tool::value(false);
    if (CsSymbolP(v))
      return tool::value(tool::ustring(CsSymbolName(v)), 0xFFFF);
    if (CsIntegerP(v)) return tool::value(CsIntegerValue(v));
    if (CsFloatP(v)) return tool::value(CsFloatValue(v));
    if (CsTupleP(v)) {
      handle<function_value> pfv   = new function_value();
      pfv->name             = value_to_string(CsTupleName(v));
      slice<value> elements = CsTupleElements(v);
      for (uint n = 0; n < elements.length; ++n) {
        pfv->params.push(value_to_value(c, elements[n], isolate));
      }
      if (pfv->name == "url" && pfv->params.size() == 1 && pfv->params.value(0).is_string())
      {
        //tool::value t = pfv->params.value(0);
        //t.units(tool::value::uri);
        return tool::value::make_url(pfv->params.value(0).get_string());
      }
      return tool::value::make_function(pfv);
    }
    if (CsVectorP(v)) {
      if (isolate)
      {
        int         sz = CsVectorSize(c, v);
        tool::value va = tool::value::make_array(sz);
        for (int i = 0; i < sz; ++i) {
          va.set_element(i,value_to_value(c, CsVectorElement(c, v, i), isolate));
          if (va[i].is_length())
            va[i] = va[i];
        }
        return va;
      }
      return tool::value::make_proxy(create_proxy(c, v), tool::value::UT_OBJECT_ARRAY);
    }
    if (CsTypeP(c, v) || CsClassP(v) || CsNamespaceP(v)) {
      if (isolate) 
      {
        tool::value   map;
        each_property gen(c, v);
        for (value key, val; gen(key, val);)
          map.set_prop(value_to_value(c, key, isolate), value_to_value(c, val, isolate));
        return map;
      }
      return tool::value::make_proxy(create_proxy(c, v), tool::value::UT_OBJECT_CLASS);
    }
    if (CsObjectP(v)) {
      if (isolate) 
      {
        tool::value   map;
        each_property gen(c, v);
        for (value key, val; gen(key, val);)
          map.set_prop(value_to_value(c, key), value_to_value(c, val,isolate));
        return map;
      }
      return tool::value::make_proxy(create_proxy(c, v), tool::value::UT_OBJECT_OBJECT);
    }
    if (CsMethodP(v) || CsCMethodP(v)) {
#if JSON_ISOLATION >= 2
      return tool::value::make_error(W("function"));
#else
      return tool::value::make_proxy(create_proxy(c, v),
                                     tool::value::UT_OBJECT_FUNCTION);
#endif
    }
    if (CsCFunctorP(v)) {
#if JSON_ISOLATION >= 3
      return tool::value::make_error(W("native function"));
#else
      return tool::value::make_functor(CsCFunctorValue(v));
#endif
    }
    if (CsDateP(c, v)) {
      return tool::value::make_date(CsDateValue(c, v),
                                    date_time::DT_UTC | date_time::DT_HAS_DATE |
                                        date_time::DT_HAS_TIME |
                                        date_time::DT_HAS_SECONDS);
    }
    if (CsErrorP(v)) {
      string_stream s;
      CsDisplay(c, v, &s);
      return tool::value::make_error(s.to_ustring());
      // return tool::value::make_proxy(create_proxy(c,v),
      // tool::value::UT_OBJECT_ERROR);
      /*tool::value map;
      each_property gen(c, v);
      for( value key,val; gen(key,val);)
        map[value_to_value(c,key)] = value_to_value(c,val);
      return map;*/
    }

    if (CsByteVectorP(v)) {
      return tool::value::make_bytes(CsByteVectorArray(v));
    }

    if (CsAssetP(v)) {
      if (som_asset_t* pa = CsAssetPtr(v))
        return tool::value::wrap_asset(pa);
      return tool::value::null_val();
    }

    if (CsCObjectP(v)) {
#if defined(JSON_ISOLATION)
      tool::value   map;
      each_property gen(c, v);
      for (value key, val; gen(key, val);)
        map.set_prop(value_to_value(c, key), value_to_value(c, val));
      return map;
#else
      tool::value tv = c->x_value_to_value(v);
      if( tv.is_undefined() )
        return tool::value::make_proxy(create_proxy(c, v), tool::value::UT_OBJECT_NATIVE);
      return tv;
#endif
    }
    if (CsColorP(v)) {
      uint i = unpack_uint32(v);
      return tool::value::make_packed_color(i);
    }
    if (CsLengthP(v)) {
      // int u;
      // i = to_unit(v,u);
      // return tool::value::make_length(d,u);
      return CsLengthToolValue(v);
    }
    if (CsAngleP(v)) {
      //double d = CsAngleRadians(v); // returns radians
      //return tool::value::make_angle(d);
      double d = CsAngleRadians(v); // returns radians
      auto   u = CsAngleUnits(v);
      return tool::value::make_angle(d, u, true);
    }
    if (CsDurationP(v)) {
      double d = CsDurationSeconds(v); // returns seconds
      return tool::value::make_duration(d, CsDurationUnits(v), true);
    }

    return c->x_value_to_value(v);
  }


  static tool::value make_tuple(array<value> &stack, wchars tag, tool::value p1, tool::value p2 = tool::value(), tool::value p3 = tool::value()) {
    function_value *pfv = new function_value();
    pfv->name = tag;

    pfv->params.push(p1);
    pfv->params.push(p2);
    pfv->params.push(p3);

    return tool::value::make_function(pfv);
  }
  
  tool::value _value_to_value_def(VM *c, value v, array<value> &stack) {

    //dispatch *pd = CsGetDispatch(v);
    if (CsStringP(v)) return tool::value(tool::wchars(CsStringAddress(v), CsStringSize(v)));
    if (v == UNDEFINED_VALUE || v == NOTHING_VALUE) return tool::value();
    if (v == NULL_VALUE) return tool::value::null_val();
    if (v == TRUE_VALUE) return tool::value(true);
    if (v == FALSE_VALUE) return tool::value(false);
    if (CsSymbolP(v)) return tool::value(tool::ustring(CsSymbolName(v)), 0xFFFF);
    if (CsIntegerP(v)) return tool::value(CsIntegerValue(v));
    if (CsFloatP(v)) 
      return tool::value(CsFloatValue(v));
    if (CsTupleP(v)) {
      function_value *pfv   = new function_value();
      pfv->name             = value_to_string(CsTupleName(v));
      slice<value> elements = slice<value>(CsTupleAddress(v), CsTupleSize(v));
      for (uint n = 0; n < elements.length; ++n) {
        pfv->params.push(_value_to_value_def(c, elements[n], stack));
      }
      return tool::value::make_function(pfv);
    }
    if (CsVectorP(v)) {
      if (stack.get_index(v) >= 0)
        return //tool::value(WTEXT("(recursive-array)"), 0xFFFF);
               make_tuple(stack, WCHARS("problem"), tool::value(WCHARS("recursive array")));
      stack.push(v);
      int         sz = CsVectorSize(c, v);
      tool::value va = tool::value::make_array(sz);
      for (int i = 0; i < sz; ++i)
        va.set_element(i, _value_to_value_def(c, CsVectorElement(c, v, i),stack));
      stack.pop();
      return va;
    }
    if (CsDateP(c, v)) {
      return tool::value::make_date(CsDateValue(c, v), date_time::DT_UTC | date_time::DT_HAS_DATE | date_time::DT_HAS_TIME | date_time::DT_HAS_SECONDS);
    }
    if (CsAssetP(v)) {
      return make_tuple(stack, WCHARS("proxy"), tool::value("Asset"), tool::value(CsAssetTypeName(v)), CsAssetProps(v));
    }
    if (CsObjectP(v)) {
      if (stack.get_index(v) >= 0)
        //return tool::value(WTEXT("(recursive-object)"), 0xFFFF);
        return make_tuple(stack, WCHARS("problem"), tool::value(WCHARS("recursive object")));
      if (CsObjectClass(v) == c->objectObject || CsObjectClass(v) == UNDEFINED_VALUE)
      {
        stack.push(v);
        tool::value   map = tool::value::make_map();
        each_property gen(c, v);
        for (value key, val; gen(key, val);)
          map.set_prop(value_to_string(key), _value_to_value_def(c, val, stack));
        stack.pop();
        return map;
      }
      // else fall through:
    }
    if (CsCObjectP(v)) {
      if (stack.get_index(v) >= 0)
        return make_tuple(stack, WCHARS("problem"), tool::value(WCHARS("recursive object")));
      stack.push(v);
      tool::value   props = tool::value::make_map();
      each_property gen(c, v);
      for (value key, val; gen(key, val);) {
        props.set_prop(value_to_string(key), _value_to_value_def(c, val, stack));
      }
      stack.pop();

      string_stream ss;
      CsPrintType(c, v, &ss);

      return make_tuple(stack, WCHARS("proxy"), tool::value(ss.to_ustring()) , tool::value(CsObjectClassName(c, v)), props);
    }
    if (CsObjectP(v) || CsNamespaceP(v) || CsCObjectP(v) || CsTypeP(c, v) || CsMethodP(v) /* || CsCObjectP(v)*/ ) {
      if (stack.get_index(v) >= 0)
        return make_tuple(stack, WCHARS("problem"), tool::value(WCHARS("recursive object")));
      stack.push(v);
      tool::value   props = tool::value::make_map();
      each_property gen(c, v);
      for (value key, val; gen(key, val);) {
        props.set_prop(value_to_string(key), _value_to_value_def(c, val, stack));
      }
      stack.pop();
      return make_tuple(stack, WCHARS("proxy"), _value_to_value_def(c, CsTypeOf(c, v), stack), tool::value(CsObjectClassName(c, v)), props);
    }
    if (CsCMethodP(v))
    {
      return make_tuple(stack, WCHARS("proxy"), _value_to_value_def(c, CsTypeOf(c, v), stack),tool::value(CsCMethodName(v)));
    }
    if (CsCFunctorP(v))
    {
      return make_tuple(stack, WCHARS("proxy"), _value_to_value_def(c, CsTypeOf(c, v), stack));
    }
    if (CsByteVectorP(v))
      return tool::value::make_bytes(
          tool::bytes(CsByteVectorAddress(v), CsByteVectorSize(v)));
    if (CsColorP(v)) {
      uint i = unpack_uint32(v);
      return tool::value::make_packed_color(i);
    }
    if (CsLengthP(v)) { return CsLengthToolValue(v); }
    if (CsAngleP(v)) {
      double d = CsAngleRadians(v); // returns radians
      auto   u = CsAngleUnits(v);
      return tool::value::make_angle(d,u,true);
    }
    if (CsDurationP(v)) {
      double d = CsDurationSeconds(v); // returns seconds
      return tool::value::make_duration(d, CsDurationUnits(v), true);
    }
    // unknown
    return make_tuple(stack, WCHARS("proxy"), _value_to_value_def(c, CsTypeOf(c, v), stack));
  }

  tool::value value_to_value_def(VM *c, value v) {
    array<value> stack;
    return _value_to_value_def(c, v, stack);
  }

  struct json_to_value_ctx : public gc_callback {
    VM *           c;
    volatile value string_map;
    volatile value replacer;
    json_to_value_ctx(VM *vm)
        : c(vm), string_map(0), replacer(0), gc_callback(vm) {}

    value cvt_value(const tool::value &v, bool onreturn = false);
    value cvt_string(const ustring &s);

    virtual void on_GC(VM *c) {
      if (string_map) string_map = CsCopyValue(c, string_map);
      if (replacer) replacer = CsCopyValue(c, replacer);
    }

    value dictionary_to_value(
        const tool::dictionary<tool::value, tool::value> &params);
  };

  value value_to_value(VM *c, const tool::value &v, bool onreturn) {
    json_to_value_ctx ctx(c);
    return ctx.cvt_value(v, onreturn);
  }

  value dictionary_to_value(
      VM *c, const tool::dictionary<tool::value, tool::value> &params) {
    json_to_value_ctx ctx(c);
    return ctx.dictionary_to_value(params);
  }

  value json_to_value_ctx::cvt_string(const ustring &s) {
    if (!string_map) string_map = CsMakeObject(c, UNDEFINED_VALUE);
    value sv = string_to_value(c, s);
    CsPush(c, sv);
    value ev = 0;
    if (CsGetProperty(c, string_map, CsTop(c), &ev)) {
      CsPop(c);
      return ev;
    }
    CsSetProperty(c, string_map, CsTop(c), CsTop(c));
    return CsPop(c);
  }

  value json_to_value_ctx::cvt_value(const tool::value &v, bool onreturn) {
    switch (v.type()) {
    case tool::value::t_undefined: return v.units() == tool::value::UT_NOTHING ? NOTHING_VALUE : UNDEFINED_VALUE;
    case tool::value::t_null: return NULL_VALUE;
    case tool::value::t_bool: return v.get(false) ? TRUE_VALUE : FALSE_VALUE;
    case tool::value::t_int:
      if (v.is_color()) return pack_value(PT_COLOR, 0, v._int());
      return CsMakeInteger(v.get(0));
    case tool::value::t_double:
      return CsMakeFloat(v.get(0.0));
    case tool::value::t_string: {
      if (v.is_string_symbol()) {
        tool::wchars wc = v.get_chars();
        return CsMakeSymbol(c, wc.start, wc.size());
      }
      else if (onreturn && v.is_error_string()) {
        tool::ustring msg = v.get_string();
        CsThrowKnownError(c, CsErrGenericErrorW, msg.c_str());
      }
      return cvt_string(v.get(W("")));
    }
    case tool::value::t_array: {
      int   sz = v.size();
      value vo = CsMakeVector(c, sz);
      PROTECT(vo);
      for (int i = 0; i < sz; ++i) {
        value t = cvt_value(v.get_element(i));
        CsSetVectorElement(c, vo, i, t);
      }
      return vo;
    }
    case tool::value::t_map: {
      value vo = CsMakeObject(c, c->objectObject);
      int   sz = v.size();
      value key = 0, val = 0;
      PROTECT(key, val, vo);
      if (replacer) {
        auto_scope as(c, CsMethodNamespace(replacer));
        for (int n = 0; n < sz; ++n) {
          tool::value vk = v.key(n);
          // if( vk.is_string())
          //  vk.units(tool::value::UT_SYMBOL);
          key = cvt_value(vk);
          if (key == PROTOTYPE_SYM) key = CsMakeString(c, WCHARS("prototype"));
          val = cvt_value(v.get_element(n));
          val = CsCallMethod(c, vo, replacer, vo, 2, key, val);
          //if (c->vals == 2) // return (k,v);
          //  key = c->val[1];
          CsSetItem(c, vo, key, val);
        }
      }
      else
        for (int n = 0; n < sz; ++n) {
          tool::value vk = v.key(n);
          // if( vk.is_string())
          //  vk.units(tool::value::UT_SYMBOL);
          key = cvt_value(vk);
          if (key == PROTOTYPE_SYM) key = CsMakeString(c, WCHARS("prototype"));
          val = cvt_value(v.get_element(n));
          // CsSetObjectProperty(c,vo,key,val);
          CsSetItem(c, vo, key, val);
        }
      return vo;
    }
    case tool::value::t_function: {
      function_value *pf = v.get_function();
      value           vo = CsMakeTuple(c, v.size());
      PROTECT(vo);
      CsSetTupleName(vo, CsSymbolOf(pf->name));
      int sz = v.size();
      for (int n = 0; n < sz; ++n) {
        value val = cvt_value(v.get_element(n));
        CsSetTupleElement(vo, n, val);
      }
      return vo;
    }
    case tool::value::t_bytes: {
      return CsMakeByteVector(c, v.get_bytes_array());
    }
    case tool::value::t_date: {
      tool::date_time dt = v.get_date();
      if (v.units() & date_time::DT_UTC)
        return CsMakeDate(c, dt.time());
      else
        return CsMakeDate(c, dt.time() - date_time::local_offset());
    }
#if JSON_ISOLATION < 3
    case tool::value::t_object_proxy: {
      object_proxy *op = (object_proxy *)v.get_proxy();
      if (op)
        return op->pin.val;
      else
        return NULL_VALUE;
    } break;
#endif
    case tool::value::t_length:
      // return unit_value(v._int(),v.units());
      return CsMakeLengthFromTool(v);
    case tool::value::t_resource:
    {
      tool::handle<tool::resource> pt = v.get_resource();
      tis::value tisv = c->x_value_to_value(v);
      if (tisv != NULL_VALUE)
        return tisv;
      if (pt->is_of_type<tool::native_functor_holder>())
        return CsMakeCFunctor(c, pt.ptr_of<tool::native_functor_holder>());

      return CsMakeString(c, pt->resource_class_name());
    }
    case tool::value::t_asset:
    {
      som_asset_t* pa = v.get_asset();
      return CsMakeAssetObject(c, pa);
    }
    case tool::value::t_angle: return CsMakeAngle(v.get_angle(), ANGLE_UNIT_TYPE(v.units()), true);
    case tool::value::t_duration: return CsMakeDuration(v.get_duration(), DURATION_UNIT_TYPE(v.units()), true);
    case tool::value::t_color: {
#ifdef SCITER
      gool::color_v vc(v);
      return CsMakeColor(vc.to_color());
#else
      return CsMakeColor(v.get(0));
#endif
    }
    case tool::value::t_enum:
      return CsMakeSymbol(c, v.to_string()());

    }
    // assert(false);
    return UNDEFINED_VALUE;
  }

  value json_to_value_ctx::dictionary_to_value(
      const tool::dictionary<tool::value, tool::value> &params) {
    value vo  = CsMakeObject(c, c->objectObject);
    int   sz  = params.size();
    value key = 0, val = 0;
    PROTECT(key, val, vo);
    for (int n = 0; n < sz; ++n) {
      tool::value vk = params.key(n);
      if (vk.is_undefined())
        key = CsMakeInteger(n);
      else
        key = cvt_value(vk);
      if (key != PROTOTYPE_SYM) {
        val = cvt_value(params.value(n));
        CsSetItem(c, vo, key, val);
      }
      // else
      // key = CsMakeString(c,WCHARS("prototype"));
    }
    return vo;
  }

  struct value_printer : public gc_callback {
    stream *           s;
    int                tabs;
    tool::array<value> stack; // cyclic reference detection.

    bool    verbose;
    ustring spacer;
    bool    canonic;  // canonic JSON
    value   replacer; // replacer function
    value   keys;

    value_printer(VM *pc, stream *ps, bool pcanonic,
                  wchars space = WCHARS("\t"))
        : gc_callback(pc), s(ps), canonic(pcanonic), replacer(0), tabs(0),
          keys(0) {
      set_format(space);
    }
    ~value_printer() {}

    void set_format(wchars space) {
      spacer  = space;
      verbose = spacer.size() > 0;
    }

    bool print_value(value val, bool on_right_side = true);
    bool print_vector(value val);
    bool print_object(value obj);
    bool print_numeric(value obj);
    bool print_string(value obj);

    virtual void on_GC(VM *c) {
      for (index_t n = 0; n < stack.size(); ++n)
        stack[n] = CsCopyValue(c, stack[n]);
      if (replacer) replacer = CsCopyValue(c, replacer);
      if (keys) keys = CsCopyValue(c, keys);
    }
  };

  bool value_printer::print_numeric(value val) {
    return CsGetDispatch(val)->print(c, val, s, false);
  }

  bool value_printer::print_string(value val) {
    wchar *p    = CsStringAddress(val);
    int    size = (int)CsStringSize(val);
    s->put_str("\"");
    while (--size >= 0)
      switch (*p) {
      case '"': {
        if (!s->put_str("\\\"")) return false;
        ++p;
      } break;
      case '\r': {
        if (!s->put_str("\\r")) return false;
        ++p;
      } break;
      case '\n': {
        if (!s->put_str("\\n")) return false;
        ++p;
      } break;
      case '\t': {
        if (!s->put_str("\\t")) return false;
        ++p;
      } break;
      case '\a': {
        if (!s->put_str("\\a")) return false;
        ++p;
      } break;
      case '\b': {
        if (!s->put_str("\\b")) return false;
        ++p;
      } break;
      case '\f': {
        if (!s->put_str("\\f")) return false;
        ++p;
      } break;
      case '\v': {
        if (!s->put_str("\\v")) return false;
        ++p;
      } break;
      case '\\': {
        if (!s->put_str("\\\\")) return false;
        ++p;
      } break;
      default:
        if (!s->put(*p++)) return false;
        break;
      }
    s->put_str("\"");
    return true;
  }

  // static bool PrintData( VM *c, value val, stream* s, int* tabs,
  // tool::pool<value>& emited, bool on_right_side = true);

  bool value_printer::print_vector(value val) {
#if defined(TISCRIPT_USE_PERSISTENCE)
    if (_CsIsPersistent(val)) val = CsFetchVectorData(c, val);
#endif

    int_t size, i;
    size = CsVectorSize(c, val);

    if (size == 0) return s->put_str("[]");

    PROTECT(val);

    ++tabs;

    if (!s->put('[')) return false;

    for (i = 0; i < size; ++i) {

      if (i > 0) s->put(',');
      if (verbose && canonic) {
        s->put('\n');
        for (int t = 0; t < tabs; ++t)
          s->put_str(spacer);
      }

      value el = CsVectorElement(c, val, i);
      if (replacer)
        el = CsCallMethod(c, val, replacer, val, 2, CsMakeInteger(i), el);
      if (!print_value(el)) return false;
    }

    --tabs;

    if (verbose && canonic) {
      s->put('\n');
      for (int t = 0; t < tabs; ++t)
        s->put_str(spacer);
    }

    if (!s->put(']')) return false;
    return true;
  }

  bool value_printer::print_object(value obj) {
    if (obj == NULL_VALUE) return s->put_str("null");
    if (obj == UNDEFINED_VALUE) return s->put_str("undefined");

#if defined(TISCRIPT_USE_PERSISTENCE)
    if (_CsIsPersistent(obj)) obj = CsFetchObjectData(c, obj);
#endif

    if (CsObjectPropertyCount(obj) == 0) return s->put_str("{}");

    ++tabs;

    if (!s->put('{')) return false;
    // if( verbose ) {
    //  if (!s->put('\r'))
    //    return false;
    //}

    /*bool is_large = CsObjectPropertyCount(obj) > 4;

    if(!is_large)
    {
      each_property gen(c, obj);
      for( value key,val; gen(key,val);)
      {
        if( CsObjectP(val) || CsVectorP(val) )
        {
          is_large = true;
          break;
        }
      }
    }*/

    if (replacer) {
      each_property gen(c, obj);
      value         key = 0, val = 0;
      PROTECT(key, val);
      int cnt = 0;
      for (; gen(key, val);) {
        if (cnt) s->put(',');
        if (verbose) {
          s->put('\n');
          for (int t = 0; t < tabs; ++t)
            s->put_str(spacer);
        }
        val = CsCallMethod(c, obj, replacer, obj, 2, key, val);
        //if (c->vals == 2) // return (k,v);
        //  key = c->val[1];
        if (val == UNDEFINED_VALUE || val == NOTHING_VALUE) continue;
        if (!print_value(key, false)) break;
        if (!s->put_str(":")) return false;
        if (!print_value(val, true)) break;
        ++cnt;
      }
    } else if (keys) {
      int cnt = 0;
      for (int n = 0; n < CsVectorSize(c, keys); ++n) {
        value key = CsVectorElement(c, keys, n);
        value val = UNDEFINED_VALUE;
        if (!CsGetProperty(c, obj, key, &val)) continue;
        if (++cnt > 1) s->put(',');
        if (verbose) {
          s->put('\n');
          for (int t = 0; t < tabs; ++t)
            s->put_str(spacer);
        }
        if (!print_value(key, false)) break;
        if (!s->put_str(":")) return false;
        if (!print_value(val, true)) break;
      }
    } else {

      struct pd {
        uint  ordinal;
        value k;
        value v;
      };
      buffer<pd, 32> ordered(CsObjectPropertyCount(obj));
      each_property  gen(c, obj);
      uint           n = 0;
      for (value key, val; gen(key, val);) {
        pd d         = {gen.ordinal, key, val};
        ordered[n++] = d;
      }
      tool::sort(
          ordered.begin(), ordered.length(),
          [](const pd &l, const pd &r) { return l.ordinal < r.ordinal; });

      for (n = 0; n < ordered.length(); ++n) {
        if (n) s->put(',');
        if (verbose) {
          s->put('\n');
          for (int t = 0; t < tabs; ++t)
            s->put_str(spacer);
        }
        if (!print_value(ordered[n].k, false)) break;
        if (!s->put_str(":")) return false;
        if (!print_value(ordered[n].v, true)) break;
      }
    }

    --tabs;

    if (verbose) {
      s->put('\n');
      for (int t = 0; t < tabs; ++t)
        s->put_str(spacer);
    }
    if (!s->put('}')) return false;
    return true;
  }

  extern bool isidchar(int ch);
  static bool is_name_token(const wchar *str) {
    while (str && *str) {
      if (!isidchar(*str)) return false;
      ++str;
    }
    return true;
  }

  extern void CsColorToString(char *buf, value obj);

  bool value_printer::print_value(value val, bool right_side) {
    if (CsIntegerP(val) || CsFloatP(val))
      return print_numeric(val);
    else if (val == UNDEFINED_VALUE) {
      if (canonic) // NOTE: 'undefined' is not a value in canonical JSON
        // return s->put_str("null");
        CsThrowKnownError(c, CsErrUnexpectedTypeError, val,
                          "JSON: incompatible value");
      else
        return s->put_str("undefined");
    } else if (val == NULL_VALUE || val == TRUE_VALUE || val == FALSE_VALUE) {
      return s->put_str(CsSymbolName(val));
    } else if (CsSymbolP(val)) {
      if (this->canonic) {
        if (!s->put_str("\"")) return false;
        if (!s->put_str(CsSymbolName(val))) return false;
        return s->put_str("\"");
      } else {
        if (right_side || !is_name_token(CsSymbolName(val))) s->put_str("#");
        return s->put_str(CsSymbolName(val));
      }
    } else if (CsStringP(val))
      return print_string(val);
    else if (CsVectorP(val)) {
      if (stack.get_index(val) >= 0)
        //CsThrowKnownError(c, CsErrUnexpectedTypeError, val,
        //                  "JSON: recursive reference");
        return s->put_str("<recursive reference!>");
      else {
        stack.push(val);
        bool r = print_vector(val);
        stack.pop();
        return r;
      }
    }
    else if (CsAssetP(val)) {
      return c->assetDispatch->print(c, val, s, false);
    }
    else if (CsObjectP(val)) {
      if (stack.get_index(val) >= 0)
        //CsThrowKnownError(c, CsErrUnexpectedTypeError, val,
        //                  "JSON: recursive reference");
        return s->put_str("<recursive reference!>");
      else {
        stack.push(val);
        bool r = print_object(val);
        stack.pop();
        return r;
      }
    } else if (CsDateP(c, val)) {
      if (canonic) // NOTE: JSON has no notion of Date
      {
        if (!s->put_str("\"")) return false;
        if (!CsPrintDate(c, val, s)) return false;
        return s->put_str("\"");
      } else {
        s->put_str("new Date(\"");
        CsPrintDate(c, val, s);
        return s->put_str("\")");
      }
    } else if (canonic)
      CsThrowKnownError(c, CsErrUnexpectedTypeError, val,
                        "JSON: incompatible value");

    if (CsLengthP(val)) {
      return CsLengthPrintFx(c, val, s);
    } else if (CsColorP(val)) {
      char buf[256];
      CsColorToString(buf, val);
      return s->put_str(buf);
    } else if (CsAngleP(val)) {
      return s->put_str(ustring::format(W("%.4frad"), CsAngleRadians(val)));
    } else if (CsDurationP(val)) {
      return s->put_str(ustring::format(W("%.4fs"), CsDurationSeconds(val)));
    } else if (CsTupleP(val)) {
      s->put_str("[");
      print_value(CsTupleName(val), false);
      s->put_str(":");
      int cnt = CsTupleSize(val);
      for (int n = 0; n < cnt; ++n) {
        if (n) s->put_str(",");
        print_value(CsTupleElement(val, n));
      }
      return s->put_str("]");
    } else {
      if (!s->put_str("null")) return false;
      if (!canonic)
        return s->printf(W(" /*object of class %S */"), CsTypeName(val));
      else
        return true;
    }
    return false;
  }

  bool CsPrintData(VM *c, value val, stream *s, bool verbose) {
    value_printer vp(c, s, false, verbose ? WCHARS("\t") : wchars());
    return vp.print_value(val);
  }

  bool CsPrintJsonData(VM *c, value val, stream *s, bool verbose) {
    value_printer vp(c, s, true, verbose ? WCHARS("\t") : wchars());
    return vp.print_value(val);
  }

  value CSF_JSON_stringify(VM *c) {
    value reviewer = UNDEFINED_VALUE;
    value v        = UNDEFINED_VALUE;
    value space    = UNDEFINED_VALUE;

    CsParseArguments(c, "**V|V|V", &v, &reviewer, &space);

    if (CsStringP(reviewer) || CsIntegerP(reviewer)) {
      space    = reviewer;
      reviewer = UNDEFINED_VALUE;
    };

    string_stream ss;
    value_printer vp(c, &ss, true);

    if (CsMethodP(reviewer))
      vp.replacer = reviewer;
    else if (CsVectorP(reviewer))
      vp.keys = reviewer;

    if (CsStringP(space)) {
      ustring spacer = CsStringChars(space);
      vp.set_format(spacer);
    } else if (CsIntegerP(space)) {
      ustring spacer = ustring(' ', CsIntegerValue(space));
      vp.set_format(spacer);
    } else
      vp.set_format(wchars());

    vp.print_value(v);
    return ss.string_o(c);
  }

  value CSF_JSON_parse(VM *c) {
    value  reviewer = 0;
    wchars text;

    CsParseArguments(c, "**S#|m", &text.start, &text.length, &reviewer);

    tool::ustring errmsg;

    auto on_error = [&](wchars err) { errmsg = err; };

    tool::value tv = tool::xjson::parse(text, false, on_error);
    if (tv.is_undefined())
      CsThrowKnownError(c, CsErrGenericErrorW, errmsg.c_str());
    json_to_value_ctx ctx(c);
    if (reviewer)
      ctx.replacer = reviewer;
    return ctx.cvt_value(tv);
  }

  value CsParseJson(VM *c, bytes data) {
    if (data.starts_with(BYTES(UTF8_BOM))) data.prune(3);
    tool::ustring text = u8::cvt(data);

    tool::ustring errmsg;
    auto          on_error = [&](wchars err) { errmsg = err; };

    wchars      toparse = text();
    tool::value tv      = tool::xjson::parse(toparse, false, on_error);

    json_to_value_ctx ctx(c);
    return ctx.cvt_value(tv);
  }

} // namespace tis

/*#ifdef _DEBUG
  void object_proxy_report()
  {
    dbg_printf("tis::count=%d\n",tis::count);
  }
#endif*/
