/* cs_angle.c - 'Angle' type */
/*
        Copyright (c) 2001-2013 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include "cs.h"

#include "cs_int.h"
#include <limits.h>

namespace tis {

  /* method handlers */
  static value CSF_parse(VM *c);
  static value CSF_toFloat(VM *c);
  // static value CSF_toDegrees(VM *c);
  static value CSF_toInteger(VM *c);
  static value CSF_toString(VM *c);
  static value CSF_morph(VM *c);

  /* Integer methods */
  static c_method methods[] = {
      C_METHOD_ENTRY("parse", CSF_parse),
      C_METHOD_ENTRY("toFloat", CSF_toFloat),
      C_METHOD_ENTRY("toInteger", CSF_toInteger),
      // C_METHOD_ENTRY( "toDegrees",        CSF_toDegrees       ),
      C_METHOD_ENTRY("toString", CSF_toString),
      C_METHOD_ENTRY("toHtmlString", CSF_toString),
      C_METHOD_ENTRY("toCssString", CSF_toString),
      C_METHOD_ENTRY("toUrlString", CSF_toString),
      C_METHOD_ENTRY("morph", CSF_morph), C_METHOD_ENTRY(0, 0)};

  static value CSF_degrees(VM *c, value obj);
  static value CSF_radians(VM *c, value obj);

  /* Integer properties */
  static vp_method properties[] = {VP_METHOD_ENTRY("degrees", CSF_degrees, 0),
                                   VP_METHOD_ENTRY("radians", CSF_radians, 0),
                                   VP_METHOD_ENTRY(0, 0, 0)};

  static constant constants[] = {
      // CONSTANT_ENTRY("MAX"    , int_value(INT_MAX     )),
      // CONSTANT_ENTRY("MIN"    , int_value(INT_MIN     )),
      CONSTANT_ENTRY(0, 0)};

  /* CsInitAngle - initialize the 'Angle' obj */
  void CsInitAngle(VM *c) {
    c->angleObject = CsEnterType(CsGlobalScope(c), "Angle", &CsAngleDispatch);
    CsEnterMethods(c, c->angleObject, methods);
    CsEnterVPMethods(c, c->angleObject, properties);
    CsEnterConstants(c, c->angleObject, constants);
  }

  /* CSF_parse - built-in method 'parse' */
  static value CSF_parse(VM *c) {
    wchars str;
    value  defval = UNDEFINED_VALUE;
    CsParseArguments(c, "**S#|V", &str.start, &str.length, &defval);

    double val = tool::str_to_f(str, std::numeric_limits<double>::quiet_NaN());
    if (val == std::numeric_limits<double>::quiet_NaN()) return defval;
    if (str.starts_with(WCHARS("rad")))
      return CsMakeAngle(val,ANGLE_UNIT_TYPE::UT_RAD);
    else if (str.starts_with(WCHARS("deg")))
      return CsMakeAngle(val, ANGLE_UNIT_TYPE::UT_DEG);
    else if (str.starts_with(WCHARS("turn")))
      return CsMakeAngle(val, ANGLE_UNIT_TYPE::UT_TURN);
    else if (str.starts_with(WCHARS("grad")))
      return CsMakeAngle(val, ANGLE_UNIT_TYPE::UT_GRAD);
    return defval;
  }

  /* CSF_parse - built-in method 'parse'
  static value CSF_parse(VM *c)
  {
      wchar* str = 0;
      value defval = UNDEFINED_VALUE;
      CsParseArguments(c,"**S|V",&str,&defval);
      tool::value vv = tool::value::parse(str);
      if( vv.is_angle() )
        return unit_value( int(vv.get_angle() * 10000), ut_rad);
      else if(vv.is_number() )
        return unit_value( int(vv.get_double() * 10000), ut_rad);
      return defval;
  }*/

  static value CSF_radians(VM *c, value obj) {
    // UNIT_TYPE u = unpack_unit(obj);
    int t = unpack_int32(obj);
    return CsMakeFloat(t / 10000.0);
  }

  static value CSF_degrees(VM *c, value obj) {
    // UNIT_TYPE u = unpack_unit(obj);
    int t = unpack_int32(obj);
    return CsMakeFloat(t / 10000.0 * 57.2957795);
  }

  static value CSF_morph(VM *c) {
    value  from;
    value  to;
    double ratio = 0;

    CsParseArguments(c, "**V=V=d", &from, &CsAngleDispatch, &to,
                     &CsAngleDispatch, &ratio);

    double df = CsAngleRadians(from);
    double dt = CsAngleRadians(to);

    return CsMakeAngle(df + (dt - df) * ratio, ANGLE_UNIT_TYPE::UT_RAD);
  }

  double CsAngleRadians(value obj) {
    assert(CsAngleP(obj));
    int v = unpack_int32(obj);
    return v / 10000.0;
  }

  ANGLE_UNIT_TYPE CsAngleUnits(value obj) {
    assert(CsAngleP(obj));
    return unpack_unit<ANGLE_UNIT_TYPE>(obj);
  }

  /* CSF_toFloat - built-in method 'toFloat' */
  static value CSF_toFloat(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsAngleDispatch);
    int v = unpack_int32(obj);
    return CsMakeFloat(v / 10000.0);
  }

  /* CSF_toInteger - built-in method 'toInteger' */
  static value CSF_toInteger(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsAngleDispatch);
    int v = unpack_int32(obj);
    return CsMakeInteger(int((v / 10000.0) * 57.2957795));
  }

  /* CSF_toString - built-in method 'toString' */
  static value CSF_toString(VM *c) {
    value obj;
    CsParseArguments(c, "V=*|i", &obj, &CsAngleDispatch);
    double radians = CsAngleRadians(obj);
    string s = tool::value::angle_to_string(radians, CsAngleUnits(obj));
    return CsMakeCString(c, s);
    //int v = unpack_int32(obj);
    //return CsMakeCString(c, fixedtow(v, 4, W(""), W("rad")));
  }

  // value CSF_angle(VM *c) { return angle_value( c ); }
  value CSF_deg(VM *c) {
    double deg;
    CsParseArguments(c, "**d", &deg);
    return CsMakeAngle(deg, ANGLE_UNIT_TYPE::UT_DEG);
  }

  value CSF_rad(VM *c) {
    double deg;
    CsParseArguments(c, "**d", &deg);
    return CsMakeAngle(deg, ANGLE_UNIT_TYPE::UT_RAD);
  }
   

  /* CsMakeAngle - make a new length value */
  value CsMakeAngle(float_t v, ANGLE_UNIT_TYPE ut, bool raw) {
    if(raw)
      return pack_angle(v, ut);
    else 
      switch (ut) {
        default: assert(false);
        case ANGLE_UNIT_TYPE::UT_RAD: return pack_angle(v, ut);
        case ANGLE_UNIT_TYPE::UT_DEG: return pack_angle(v / 57.2957795, ut);
        case ANGLE_UNIT_TYPE::UT_TURN: return pack_angle(360 * v / 57.2957795, ut);
        case ANGLE_UNIT_TYPE::UT_GRAD: return pack_angle(v * 0.015707963, ut);
      }
  }

  static bool  GetAngleProperty(VM *c, value &obj, value tag, value *pValue);
  static bool  SetAngleProperty(VM *c, value obj, value tag, value value);
  static bool  AnglePrint(VM *c, value obj, stream *s, bool toLocale);
  static long  AngleSize(value obj);
  static value AngleCopy(VM *c, value obj);
  static int_t AngleHash(value obj);

  dispatch CsAngleDispatch = {"Angle",
                              &CsAngleDispatch,
                              GetAngleProperty,
                              SetAngleProperty,
                              CsDefaultNewInstance,
                              AnglePrint,
                              AngleSize,
                              AngleCopy,
                              CsDefaultScan,
                              AngleHash,
                              CsDefaultGetItem,
                              CsDefaultSetItem};

  /* GetAngleProperty - Angle get property handler */
  static bool GetAngleProperty(VM *c, value &obj, value tag, value *pValue) {
    return CsGetVirtualProperty(c, obj, c->angleObject, tag, pValue);
  }

  /* SetAngleProperty - Angle set property handler */
  static bool SetAngleProperty(VM *c, value obj, value tag, value value) {
    return CsSetVirtualProperty(c, obj, c->angleObject, tag, value);
  }

  /* AnglePrint - Angle print handler */
  static bool AnglePrint(VM *c, value obj, stream *s, bool toLocale) {
    double radians = CsAngleRadians(obj);
    string sa = tool::value::angle_to_string(radians, CsAngleUnits(obj));
    return s->put_str(sa);
    //int v = unpack_int32(obj);
    //return s->put_str(fixedtow(v, 4, W(""), W("rad")));
  }

  /* AngleSize - Angle size handler */
  static long AngleSize(value obj) {
    // return sizeof(CsAngle);
    return sizeof(value);
  }

  /* AngleCopy - Angle copy handler */
  static value AngleCopy(VM *c, value obj) {
    // return CsPointerP(obj) ? CsDefaultCopy(c,obj) : obj;
    return obj;
  }

  /* AngleHash - Angle hash handler */
  static int_t AngleHash(value obj) { return (int_t)obj; }

  value CsAngleBinaryOp(VM *c, int op, value p1, value p2) {
    if (!CsAngleP(p1)) CsUnsupportedBinaryOp(c, op, p1, p2);

    ANGLE_UNIT_TYPE u  = unpack_unit<ANGLE_UNIT_TYPE>(p1);
    float_t   fv = CsAngleRadians(p1);

    if (CsIntegerP(p2)) {
      int i = CsIntegerValue(p2);
      switch (op) {
      case BC_ADD: fv += CsAngleRadians(CsMakeAngle(i, u)); break;
      case BC_SUB: assert(false); break;
      case BC_MUL: fv *= i; break;
      case BC_DIV:
        if (i != 0)
          fv /= i;
        else
          fv = 0;
        break;
      default:
        CsUnsupportedBinaryOp(c, op, p1, p2);
        fv = 0; /* never reached */
        break;
      }
    } else if (CsFloatP(p2)) {
      double f = CsFloatValue(p2);
      switch (op) {
      case BC_ADD: fv += f; break;
      case BC_SUB: assert(false); break;
      case BC_MUL: fv *= f; break;
      case BC_DIV:
        if (f != 0)
          fv /= f;
        else
          fv = 0;
        break;
      default:
        CsUnsupportedBinaryOp(c, op, p1, p2);
        fv = 0; /* never reached */
        break;
      }
    } else if (CsAngleP(p2)) {
      // int v2; UNIT_TYPE u2;
      // v2 = to_unit(p2,u2);
      switch (op) {
      case BC_ADD: fv += CsAngleRadians(p2); break;
      case BC_SUB: fv -= CsAngleRadians(p2); break;
      default:
        CsUnsupportedBinaryOp(c, op, p1, p2);
        fv = 0; /* never reached */
        break;
      }
    } else
      CsUnsupportedBinaryOp(c, op, p1, p2);

    return CsMakeAngle(fv, ANGLE_UNIT_TYPE::UT_RAD);

    // CsUnsupportedBinaryOp(c,op, p1,p2);
    // return UNDEFINED_VALUE;
  }

  int CsAnglesCompare(VM *c, int op, value obj1, value obj2) {
    int v1 = unpack_int32(obj1);
    int v2 = unpack_int32(obj2);
    return v1 - v2;
  }

} // namespace tis
