#ifndef CS_PVAL_H
#define CS_PVAL_H

#pragma once

#include "cs.h"

namespace tis {

  // variable access function codes
  enum VAR_ACC_CODE {
    LOAD = 1,
    STORE = 2,
    PUSH = 3,
    DUP = 4,
    DEL = 5,
  };

  class PVAL;
  typedef void pval_ctl_t(CsCompiler *, int, PVAL *);
  static void code_literal(CsCompiler *c, int fcn, PVAL *pv);

  // partial value structure
  class PVAL {
    PVAL(const PVAL &);
    PVAL &operator=(const PVAL &);
    void  swap(PVAL &r) {
      tool::swap(fcn, r.fcn);
      tool::swap(val, r.val);
      tool::swap(val2, r.val2);
      tool::swap(prev, r.prev);
      tool::swap(name, r.name);
    }

  public:
    enum PVAL_TYPE { UNKNOWN_PVAL, VAR_PVAL, CONST_PVAL, EMPTY_PVAL };

    pval_ctl_t *fcn;
    int         val, val2;
    PVAL *      prev;
    PVAL_TYPE   type; // true if this is a part of var ...; declaration
    string      name;
    PVAL() : fcn(0), val(0), val2(0), prev(0), type(UNKNOWN_PVAL) {}
    ~PVAL() { delete prev; }

    void push(CsCompiler *c) {
      PVAL *npv = new PVAL();
      npv->type = type;
      swap(*npv);
      prev = npv;
    }

    void copy_if_literal(PVAL& other) {
      if (!prev && fcn == code_literal)
      {
        other.fcn = code_literal;
        other.val = val;
        other.val2 = val2;
      }
    }

    bool is_list() const { return prev != 0; }
    int  count() const { return (prev ? prev->count() : 0) + 1; }

    int r_push(CsCompiler *c) {
      int r = 1;
      if (prev) {
        r = prev->r_push(c) + 1;
        delete prev;
        prev = nullptr;
      }
      if (fcn) {
        (*fcn)(c, LOAD, this);
        fcn = NULL;
      }
      putcbyte(c, BC_PUSH);
      return r;
    }

    void r_valuate(CsCompiler *c) {
      if (prev) {
        int n = r_push(c);
        putcbyte(c, BC_LIST_RVALS);
        putcword(c, n);
      }
      else {
        if (fcn) {
          (*fcn)(c, LOAD, this);
          fcn = NULL;
        }
      }
    }

    void r_valuate_single(CsCompiler *c) {
      if (prev) CsParseError(c, "only simple expressions in lists");
      if (fcn) {
        (*fcn)(c, LOAD, this);
        fcn = NULL;
      }
    }

    bool is_lvalue() const {
      return (fcn != 0) && (prev ? prev->is_lvalue() : true);
    }

    void push_l_value(CsCompiler *c) {
      if (prev) prev->push_l_value(c);
      (*fcn)(c, PUSH, 0);
    }
    void store_l_value(CsCompiler *c) {
      if (prev) {
        putcbyte(c, BC_PUSH);
        putcbyte(c, BC_POP_RVAL);
        (*fcn)(c, STORE, this);
        for (PVAL* p = prev; p; p = p->prev)
        {
          putcbyte(c, BC_POP_RVAL);
          (*fcn)(c, STORE, p);
        }
        putcbyte(c, BC_DROP);
      }
      else {
        (*fcn)(c, STORE, this);
        //if (prev) {
        //  putcbyte(c, BC_POP_RVAL);
        //  prev->store_l_value(c);
        //}
      }
    }

    void d_valuate(CsCompiler *c) {
      if (prev) {
        prev->d_valuate(c);
        delete prev;
        prev = 0;
      }
      if (fcn) {
        (*fcn)(c, DEL, this);
        fcn = NULL;
      }
    }

    void set_empty() {
      fcn = 0;
      val = 0;
      val2 = 0;
      type = EMPTY_PVAL;
    }

    bool is_empty() const {
      return type == EMPTY_PVAL;
    }

    bool is_nmtoken_list() const {
      if (type == EMPTY_PVAL)
        return true;
      if (name.is_undefined())
        return false;
      if (prev && !prev->is_nmtoken_list())
        return false;
      return true;
    }

    string get_nmtoken() const {
      return name;
    }

    void clear() {
      if (prev) { delete prev; prev = nullptr; }
      fcn = nullptr;
      val = val2 = 0;
      type = UNKNOWN_PVAL;
      name.clear();
    }

  };

}

#endif