/* cs_async.h - cs <-> libuv bridge */
/*
        Copyright (c) 2001-2017 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#ifndef __CS_ASYNC_H__
#define __CS_ASYNC_H__

#include "cs.h"

namespace tis {

  template <class TA, class TVM, bool HAS_EVENTS = true>
  struct async_object : public gcable 
  {
    value _this = 0;

    async_object() {}
    virtual ~async_object() { assert(_this == 0); }

    virtual void on_gc(VM *c) override {
      if(_this)
        _this = CsCopyValue(c, _this);
    }

    void detach() {
      if (_this)
      {
        CsSetCObjectValue(_this, 0);
        _this = 0;
        release();
      }
    }

    static TVM* get_vm() { return static_cast<TVM*>(VM::get_current()); }

    bool notify(wchars ename, value param = 0) 
    {
      if (!_this)
        return false;
      return CsEventObjectFire(get_vm(), _this, CsSymbolOf(ename), param) == TRUE_VALUE;
    }

    //static dispatch* this_dispatch(TVM *pvm)

    static value this_object(TVM *vm, TA *ta) {
      if (!ta) return NULL_VALUE;
      value obj = CsMakeCPtrObject(vm, TA::this_dispatch(vm), ta);
      ta->add_ref();
      return ta->_this = obj;
    }

    static TA *object_ptr(TVM *c, value obj) {
      dispatch *pd = CsQuickGetDispatch(obj);
      if (pd != TA::this_dispatch(c))
        CsThrowKnownError(c, CsErrUnexpectedTypeError, obj, TA::type_name());
      TA *ta = static_cast<TA *>(CsCObjectValue(obj));
      //assert(ta); - finalized object
      return ta;
    }

    static void destroy_object(TVM *c, value obj) {
      TA *ta = object_ptr(c, obj);
      if (ta) {
        ta->close();
        CsSetCObjectValue(obj, 0);
        ta->_this = 0;
        ta->release();
      }
    }

    static void cls_scan(VM *c, value obj) {
      TA *p = object_ptr(static_cast<TVM*>(c), obj);
      if (p) p->_this = CsCopyValue(c, p->_this);
      CsCObjectScan(c, obj);
    }

    static int_t cls_hash(value obj) {
      // its address is a perfect hash
      return (int)(uint_ptr)CsCObjectValue(obj);
    }

    static bool cls_print(VM *c, value obj, stream *s) {
      TA *p = object_ptr(static_cast<TVM*>(c), obj);
      s->put_str(TA::type_name());
      if (p && p->is_live())
        s->put_str("(active)");
      else
        s->put_str("(closed)");
      return true;
    }

    // script methods and properties

    static value CSF_active(TVM *c, value obj) {
      TA *ta = object_ptr(c, obj);
      if (!ta) return FALSE_VALUE;
      return ta->is_live() ? TRUE_VALUE : FALSE_VALUE;
    }

    static value CSF_close(TVM *c) {
      value obj;
      CsParseArguments(c, "V=*", &obj, TA::this_dispatch(c));
      TA *ta = object_ptr(c, obj);
      if (!ta)
        CsThrowKnownError(c, CsErrGenericError, "already closed");
      else
        ta->close();
      return obj;
    }

    static value CSF_on(TVM *c) {
      value  obj, callback;
      wchars name;
      CsParseArguments(c, "V=*S#V=", &obj, TA::this_dispatch(c), &name.start,
        &name.length, &callback, &CsMethodDispatch);
      TA *ta = object_ptr(c, obj);
      if (!ta)
        CsThrowKnownError(c, CsErrGenericError, "already closed");
      else {
        PROTECT(obj);
        CsEventObjectAdd(c, obj, callback, name);
      }
      return obj;
    }

    static value CSF_off(TVM *c) {
      value obj, p = UNDEFINED_VALUE;
      CsParseArguments(c, "V=*|V", &obj, TA::this_dispatch(c), &p);
      TA *ta = object_ptr(c, obj);
      if (!ta)
        CsThrowKnownError(c, CsErrGenericError, "already closed");
      else {
        PROTECT(obj);
        CsEventObjectRemoveV(c, obj, p);
      }
      return obj;
    }

    static dispatch* init_class(TVM*c, c_method* methods, vp_method* properties, constant* constants) {

      dispatch *pd = CsEnterCPtrObjectType(CsGlobalScope(c), TA::type_name(), methods, properties,constants);

      /* create the type */
      if (!pd) CsInsufficientMemory(c);

      pd->scan = cls_scan;
      pd->hash = cls_hash;
      pd->print = (print_t)cls_print;
      pd->baseType = &CsCObjectDispatch;
      pd->binOp = CsDefaultObjectBinOp;

      pd->destroy = (destructor_t)destroy_object;

      if (HAS_EVENTS) {
        static c_method this_methods[] = {
          C_METHOD_ENTRY("close", CSF_close),
          C_METHOD_ENTRY("on", CSF_on),
          C_METHOD_ENTRY("off", CSF_off),
          //C_METHOD_ENTRY_X("once", CSF_once),
          C_METHOD_ENTRY(0, 0)
        };
        static vp_method this_properties[] = {
          VP_METHOD_ENTRY("active", CSF_active, 0),
          VP_METHOD_ENTRY(0, 0, 0)
        };
        CsEnterCObjectMethods(c, pd, this_methods, this_properties, nullptr);
      }

      return pd;
    }
  };

  // the same as above but also has .then(), .catch(), .finally() methods
  template <class TA, class TVM, bool HAS_EVENTS = true>
  struct thenable_async_object : public async_object<TA,TVM,HAS_EVENTS>
  {
    typedef async_object<TA,TVM,HAS_EVENTS> super;
    value _subs = UNDEFINED_VALUE; // then/catc/finally subscription chain

    thenable_async_object() {}
    virtual ~thenable_async_object() { assert(super::_this == 0); }

    virtual void on_gc(VM *c) {
      async_object<TA, TVM,HAS_EVENTS>::on_gc(c);
      if (_subs)
        _subs = CsCopyValue(c, _subs);
    }

    static value CSF_then(VM *c) {
      value obj;
      value onFulfilled, onRejected = NULL_VALUE;
      CsParseArguments(c, "V=*V|V", &obj, TA::this_dispatch(c), &onFulfilled, &onRejected);

      TA *ta = super::object_ptr(c, obj);
      if (!ta)
        CsThrowKnownError(c, CsErrGenericError, "already closed");

      if (!CsAnyMethodP(onFulfilled)) onFulfilled = NULL_VALUE;
      if (!CsAnyMethodP(onRejected)) onRejected = NULL_VALUE;

      PROTECT(obj);
      ta->_subs = CsMakeTuple(c, "sub", onFulfilled, onRejected, ta->_subs);
      return obj;
    }

    static value CSF_catch(VM *c) {
      value obj;
      value onRejected = NULL_VALUE;
      CsParseArguments(c, "V=*V", &obj, TA::this_dispatch(c), &onRejected);

      TA *ta = super::object_ptr(c, obj);
      if (!ta)
        CsThrowKnownError(c, CsErrGenericError, "already closed");
      
      if (!CsAnyMethodP(onRejected)) onRejected = NULL_VALUE;

      PROTECT(obj);
      ta->_subs = CsMakeTuple(c, "sub", NULL_VALUE, onRejected, ta->_subs);
      return obj;
    }

    static value CSF_finally(VM *c) {
      value obj;
      value onFinally;
      CsParseArguments(c, "V=*V", &obj, TA::this_dispatch(c), &onFinally);
      if (!CsTaskP(c, obj)) CsUnexpectedTypeError(c, obj, "Task");

      TA *ta = super::object_ptr(c, obj);
      if (!ta)
        CsThrowKnownError(c, CsErrGenericError, "already closed");

      if (!CsAnyMethodP(onFinally)) onFinally = NULL_VALUE;

      PROTECT(obj);
      ta->_subs = CsMakeTuple(c, "sub", onFinally, onFinally, ta->_subs);
      return obj;
    }

    void notify_completion(tool::value val, bool ok) 
    {
      value cval = value_to_value(super::get_vm(), val);
      value onFulfilled = NULL_VALUE;
      value onRejected = NULL_VALUE;
      value list = this->_subs;

      VM* c = super::get_vm();

      PROTECT(list, cval, onFulfilled, onRejected);

      while (CsTupleP(list)) {
        onFulfilled = CsTupleElement(list, 0);
        onRejected = CsTupleElement(list, 1);
        list = CsTupleElement(list, 2);
        value func = ok ? onFulfilled : onRejected;
        if (CsMethodP(func)) {
          auto_scope as(c, CsMethodNamespace(func));
          CsCallMethod(c, super::_this, func, super::_this, 1, cval);
        }
        else if (CsAnyMethodP(func)) {
          CsCallMethod(c, super::_this, func, super::_this, 1, cval);
        }
      }
    }

    static dispatch* init_class(TVM*c, c_method* methods, vp_method* properties, constant* constants) {
      dispatch* pd = async_object<TA, TVM>::init_class(c, methods, properties, constants);

      static c_method this_methods[] = {
        C_METHOD_ENTRY("then", CSF_then),
        C_METHOD_ENTRY("catch", CSF_catch),
        C_METHOD_ENTRY("finally", CSF_finally),
        C_METHOD_ENTRY(0, 0)
      };

      CsEnterCObjectMethods(c, pd, this_methods, nullptr, nullptr);
      return pd;
    }

  };


}

#endif
