/* cs.h - c-smile definitions */
/*
        Copyright (c) 2001-2010 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#ifndef __CS_H__
#define __CS_H__

#include <stdio.h>
#include <stdarg.h>
#include "config.h"
#include "tool/tool.h"
#include <math.h>

#if defined(_MSC_VER)
#define isnan _isnan
#define finite _finite
#else
#define isnan std::isnan
#define finite std::isfinite
#endif

#define CS_VERSION SCITER_VERSION

#pragma pack(push, 8) // is a must for x64!

namespace tis {
  using namespace tool;
  struct VM;

  /* basic types */
  typedef uint64 value;

  /* float or integer */
  typedef uint64 uint_float_t;

  typedef int              int_t;
  typedef uint32           symbol_t;
  typedef double           float_t;
  typedef tool::datetime_t datetime_t;
} // namespace tis

#include "cs_streams.h"

// struct regexp;
//#ifdef WINDOWS
//  #define snprintf _snprintf
//#endif

namespace tis {

  // events
  struct return_event {
    bool dummy;
    return_event() : dummy(false) {}
  };
  struct script_exception {
    int number;
    script_exception(int n) : number(n) {}
    script_exception(const script_exception &e) : number(e.number) {}
  };

#define TRY try
#define CATCH_ERROR(e) catch (tis::script_exception e)
#define CATCH_ERROR_NE catch (tis::script_exception)
#define RETHROW(e) throw e
#define THROW_ERROR(code) throw tis::script_exception(code)

  /* default values */

#if defined(TISCRIPT_DISABLE_FEATURE_FILE_IO)
#define FEATURE_FILE_IO 0x00000000
#else
#define FEATURE_FILE_IO 0x00000001
#endif

#if defined(TISCRIPT_DISABLE_FEATURE_SOCKET_IO)
#define FEATURE_SOCKET_IO 0x00000000
#else
#define FEATURE_SOCKET_IO 0x00000002
#endif

#if defined(TISCRIPT_DISABLE_FEATURE_EVAL)
#define FEATURE_EVAL 0x00000000
#else
#define FEATURE_EVAL 0x00000004
#endif

#if defined(TISCRIPT_DISABLE_FEATURE_SYSINFO)
#define FEATURE_SYSINFO 0x00000000
#else
#define FEATURE_SYSINFO 0x00000008
#endif

  // execution starts from creating VM (scapp.exe case)
  //#define FEATURE_STANDALONE  0x00000010

  //#define FEATURE_GC        0x00000100
  //#define FEATURE_MATH      0x00000002
  //#define FEATURE_EVAL      0x00000004

  void CsSetFeatures(VM *pvmOrNull, uint features /*FEATURE_*** combination */);

//#define MEM_EXPANDABLE

/* determine whether the machine is little endian */
//#if defined(WIN32) || defined
#define CS_REVERSE_FLOATS_ON_READ
#define CS_REVERSE_FLOATS_ON_WRITE
//#endif

/* obj file version number */
// #define CsFaslVersion      0x500

/* hash table thresholds */
#define CsHashTableCreateThreshold 8         /* power of 2 */
#define CsHashTableExpandThreshold 2         /* power of 2 */
#define CsHashTableExpandMaximum (64 * 1024) /* power of 2 */

/* vector expansion thresholds */
/* amount to expand is:
    min(neededSize,
        min(CsVectorExpandMaximum,
            max(CsVectorExpandMinimum,
                currentSize / CsVectorExpandDivisor)))
*/
#define CsVectorExpandMinimum 8
#define CsVectorExpandMaximum 128
#define CsVectorExpandDivisor 2

#define CS_NATIVE_NESTING_MAX 100

  /* obj file tags */
  enum CsFaslTags {
    CsFaslTagUndefined = 0,
    CsFaslTagNull      = 1,
    CsFaslTagTrue      = 2,
    CsFaslTagFalse     = 3,
    CsFaslTagProxy     = 4, // object proxy in case of cyclic reference
    CsFaslTagCode      = 5,
    CsFaslTagVector    = 6,
    CsFaslTagObject    = 7,
    CsFaslTagSymbol    = 8,
    CsFaslTagString    = 9,
    CsFaslTagInteger   = 10,
    CsFaslTagFloat     = 11,
    CsFaslTagBytes     = 12,
    CsFaslTagDate      = 13,
    CsFaslTagColor     = 14,
    CsFaslTagLength    = 15,
    CsFaslTagTuple     = 16,
    CsFaslTagAngle     = 17,
    CsFaslTagDuration  = 18,
  };

  /*  #ifdef X64BITS
  #ifdef __GNUC__
  #define PTR64_MASK    0xFFFFFFFFFFFFLL // 48bits on x64
  #else
  #define PTR64_MASK    0xFFFFFFFFFFFFi64 // 48bits on x64
  #endif
  #else
  #ifdef __GNUC__
  #define PTR32_MASK    0xFFFFFFFFLL // 32bits on x32
  #else
  #define PTR32_MASK    0xFFFFFFFFi64 // 32bits on x32
  #endif
  #endif*/

#define PTR64_MASK 0xFFFFFFFFFFFFULL // 48bits on x64
#define PTR32_MASK 0xFFFFFFFFULL     // 32bits on x32

  template <size_t> /*constexpr*/ inline uint64 _ptr_mask();

  template <> /*constexpr*/ inline uint64 _ptr_mask<8>() { return PTR64_MASK; }
  template <> /*constexpr*/ inline uint64 _ptr_mask<4>() { return PTR32_MASK; }

  template <typename T> inline T *ptr(value v) {
    return (T *)(void *)(v & _ptr_mask<sizeof(void *)>());
  }

  bool is_valid_ptr_value(VM *c, value v);

  // pinned value
  struct pvalue {
  private:
    pvalue(pvalue &&pv);

  public:
    value   val;
    VM *    pvm;
    pvalue *next;
    pvalue *prev;

    pvalue() : val(0), pvm(nullptr), next(nullptr), prev(nullptr) {}
    pvalue(const pvalue &pv)
        : val(0), pvm(nullptr), next(nullptr), prev(nullptr) {
      pin(pv.pvm, pv.val);
    }

    explicit pvalue(VM *c)
        : val(0), pvm(nullptr), next(nullptr), prev(nullptr) {
      pin(c, 0);
    }
    explicit pvalue(VM *c, value v)
        : val(0), pvm(nullptr), next(nullptr), prev(nullptr) {
      pin(c, v);
    }

    ~pvalue() { unpin(); }

    pvalue &operator=(pvalue v) {
      unpin();
      pin(v.pvm, v.val);
      return *this;
    }

    void prune() {
      val  = 0;
      pvm  = 0;
      next = 0;
      prev = 0;
    }

    void pin(VM *c, value v = 0);
    void unpin();

    operator value() const { return val; }

    pvalue &operator=(value v) {
      assert(!v || pvm);
      val = v;
      return *this;
    }
    void set(value v) {
      assert(!v || pvm);
      val = v;
    }

    bool is_set() const { return val != 0; }
    bool is_alive() const { return pvm && (val != 0); }
  };

  struct header;

  typedef tool::value::unit_type_length LENGTH_UNIT_TYPE;
  typedef tool::value::unit_type_duration DURATION_UNIT_TYPE;
  typedef tool::value::unit_type_angle ANGLE_UNIT_TYPE;

  /*enum UNIT_TYPE {
    ut_none = 0,
    // lengths
    ut_em,             // 1
    ut_ex,             // 2
    ut_pr,             // 3
    ut_flex,           // 4
    ut_px,             // 5
    ut_in,             // 6
    ut_cm,             // 7
    ut_mm,             // 8
    ut_pt,             // 9
    ut_pc,             // 10
    ut_dip,            // 11
    ut_pr_width,       // 12
    ut_pr_height,      // 13
    ut_pr_view_width,  // 14
    ut_pr_view_height, // 15
    ut_pr_view_min,    // 16
    ut_pr_view_max,    // 17

    // angles
    ut_deg,  // 18 - shall match value::UT_DEG
    ut_rad,  // 19
    ut_grad, // 20
    ut_turn, // 21
    // duration
    ut_s,    // 22 - shall match value::UT_S
    ut_ms,   // 23
    //
    0
  };

#define IS_LENGTH_UNIT(ut) (ut >= ut_em && ut <= ut_pr_view_max)
#define IS_ANGLE_UNIT(ut) (ut >= ut_deg && ut <= ut_turn)
#define IS_DURATION_UNIT(ut) (ut >= ut_s && ut <= ut_ms)
#define IS_COLOR_UNIT(ut) (ut == 0)*/

  inline dword hidword(const value &v) {
    return static_cast<dword>((v >> 32) & 0xFFFFFFFF);
  }
  inline dword lodword(const value &v) {
    return static_cast<dword>(v & 0xFFFFFFFF);
  }

  //#define FLOAT_EPSILON (DBL_EPSILON*1000)

  // This is a method of storing values in the IEEE 754 double-precision
  // floating-point number. double type is 64-bit, comprised of 1 sign bit, 11
  // exponent bits and 52 mantissa bits:
  //    7         6        5        4        3        2        1        0
  // seeeeeee|eeeemmmm|mmmmmmmm|mmmmmmmm|mmmmmmmm|mmmmmmmm|mmmmmmmm|mmmmmmmm
  //

  // 11111111|1111tttt|vvvvvvvv|vvvvvvvv|vvvvvvvv|vvvvvvvv|vvvvvvvv|vvvvvvvv
  // NaN marker  |type|  48-bit placeholder for values: pointers, strings

  enum PRIMITIVE_TYPE {
    PT_NULL, // value == 0 (nullptr) normally impossible, memory corruption?
    PT_FLOAT,
    PT_SYMBOL,
    PT_INTEGER,
    PT_COLOR,
    PT_LENGTH,
    PT_ANGLE,
    PT_DURATION,
    PT_PTR,       // on heap - object, cobject, vector, etc.
    PT_IFACE_PTR, // iface ptr
  };

  constexpr inline value pack_value(PRIMITIVE_TYPE t, uint64 d) {
    // 11111111|1111tttt|vvvvvvvv|vvvvvvvv|vvvvvvvv|vvvvvvvv|vvvvvvvv|vvvvvvvv
    return 0x0000000000000000ull | ((uint64(t) & 0xFull) << 48) | (d & 0x0000FFFFFFFFFFFFull);
  }
  constexpr inline value pack_value(PRIMITIVE_TYPE t, uint32 ut, uint32 d) {
    // 11111111|1111tttt|vvvvvvvv|vvvvvvvv|vvvvvvvv|vvvvvvvv|vvvvvvvv|vvvvvvvv
    return 0x0000000000000000ull | ((uint64(t) & 0xFull) << 48) | ((uint64(ut) & 0xFFull) << 32) | (d & 0x00000000FFFFFFFFull);
  }

  constexpr inline uint32 unpack_uint32(uint64 v) {
    // assert(!is_float(v));
    return uint32(v & 0x00000000FFFFFFFFull);
  }
  constexpr inline int32 unpack_int32(uint64 v) {
    return (int32)unpack_uint32(v);
  }

  template<typename E>
  constexpr inline E unpack_unit(uint64 v) {
    // assert(!is_float(v));
    return E((v & 0x000000FF00000000ull) >> 32);
  }

  enum WELL_KNOWN_SYMBOLS // this enum must match well_known_symbols table
  {
    S_NOTHING = 1,
    S_UNDEFINED,
    S_NULL,
    S_TRUE_,
    S_FALSE_,
    S_PROTOTYPE,
    S_TO_STRING,
    S_VALUE_OF,
    S_THIS,
    S_ANY,
    S_NAN,
    S_INF,
    S_CONSTRUCTOR,
    S_BOOLEAN,
    S_INTEGER,
    S_FLOAT,
    S_STRING,
    S_DATE,
    S_ARRAY,
    S_OBJECT,
    S_CLASS,
    S_SYMBOL,
    S_FUNCTION,
    S_COLOR,
    S_LENGTH,
    S_DURATION,
    S_ANGLE,
    S_STORAGE,
    S_INDEX,
    S_TUPLE,
    S_NAMESPACE,
  };

#define NIL_VALUE (tis::value(0))
#define NOTHING_VALUE pack_value(tis::PT_SYMBOL, tis::S_NOTHING)
#define UNDEFINED_VALUE pack_value(tis::PT_SYMBOL, tis::S_UNDEFINED)
#define NULL_VALUE pack_value(tis::PT_SYMBOL, tis::S_NULL)
#define TRUE_VALUE pack_value(tis::PT_SYMBOL, tis::S_TRUE_)
#define FALSE_VALUE pack_value(tis::PT_SYMBOL, tis::S_FALSE_)
#define PROTOTYPE_SYM pack_value(tis::PT_SYMBOL, tis::S_PROTOTYPE)
#define TO_STRING_SYM pack_value(tis::PT_SYMBOL, tis::S_TO_STRING)
#define VALUE_OF_SYM pack_value(tis::PT_SYMBOL, tis::S_VALUE_OF)
#define THIS_SYM pack_value(tis::PT_SYMBOL, tis::S_THIS)
#define ANY_SYM pack_value(tis::PT_SYMBOL, tis::S_ANY)
#define NAN_VALUE pack_value(tis::PT_SYMBOL, tis::S_NAN)
#define INF_VALUE pack_value(tis::PT_SYMBOL, tis::S_INF)
#define CONSTRUCTOR_SYM pack_value(tis::PT_SYMBOL, tis::S_CONSTRUCTOR)
#define VALUE_SYM get_sym_by_id(id_value)

  /*constexpr*/ inline bool is_float(value v) {
    return //(v == NAN_VALUE) || - WRONG! will break all binary ops of floats 
           (v & 0xFFF0000000000000ull) != 0x0000000000000000ull;
  }

  /*constexpr*/ inline PRIMITIVE_TYPE primitive_type(uint64 v) {
    return is_float(v) ? PT_FLOAT
                       : PRIMITIVE_TYPE((v & 0x000F000000000000ull) >> 48);
  }

  inline PRIMITIVE_TYPE unpack_value(uint64 v, uint64 &d) {
    if (is_float(v)) {
      d = v;
      return PT_FLOAT;
    }
    d = v & 0x0000FFFFFFFFFFFFull;
    return PRIMITIVE_TYPE((v & 0x000F000000000000ull) >> 48);
  }

  inline value iface_value(header *ph, uint n) {
    return pack_value(PRIMITIVE_TYPE(PT_IFACE_PTR + n), uint64(ph));
  }
  inline value iface_value(value base, int n) {
    return pack_value(PRIMITIVE_TYPE(PT_IFACE_PTR + n), uint64(base));
  }
  inline value iface_base(value v) {
    uint64 d;
    unpack_value(v, d);
    return pack_value(PT_PTR, d);
  }

  inline value ptr_value(header *ph) {
    return pack_value(PT_PTR, uint64(uint_ptr(ph)));
  }
  inline value symbol_value(symbol_t i) {
    return pack_value(PT_SYMBOL, uint64(i));
  }
  value        symbol_value(const char *str);
  inline value int_value(int_t i) {
    return pack_value(PT_INTEGER, uint64(uint(i)));
  }
  // inline value unit_value( int_t i, UNIT_TYPE u )
  inline value float_value(float_t f) {
    if (isnan(f)) 
      return NAN_VALUE;
    if (isinf(f))
      return INF_VALUE;
    return ~*(value *)&f;
  }

  inline value pack_duration(float_t seconds, DURATION_UNIT_TYPE units) {
    return pack_value(PT_DURATION, units, int(seconds * 10000.0));
  }
  inline value pack_angle(float_t radians, ANGLE_UNIT_TYPE units) {
    return pack_value(PT_ANGLE, units, int(radians * 10000.0));
  }


  inline value bool_value(bool b) { return b ? TRUE_VALUE : FALSE_VALUE; }

  //#define ptr_value(h) ( (value)(h) )

  inline bool is_symbol(const value &v) {
    return primitive_type(v) == PT_SYMBOL;
  }
  inline bool is_int(const value &v) { return primitive_type(v) == PT_INTEGER; }
  // inline bool is_float( const value& v)     { return primitive_type(v) ==
  // PT_FLOAT; }  inline bool is_unit( const value& v)      { return (hidword(v) &
  // 0xF0000000) == 0x40000000 && hidword(v) != 0x40000000; }
  inline bool is_ptr(const value &v) { return primitive_type(v) == PT_PTR; }
  inline bool is_iface_ptr(const value &v) {
    return primitive_type(v) >= PT_IFACE_PTR;
  }

  inline int_t    to_int(const value &v) { return (int_t)unpack_uint32(v); }
  inline symbol_t to_symbol(const value &v) {
    return (symbol_t)unpack_uint32(v);
  }
  inline float_t to_float(value v) {
    if (v == NAN_VALUE)
      return std::numeric_limits<float_t>::quiet_NaN();
    else {
      v = ~v;
      float_t rv = *((float_t *)(&v));
      return rv;
    }
  }

  // inline int_t     to_unit( const value& v, UNIT_TYPE& u )  { u =
  // UNIT_TYPE(hidword(v) & 0xFF); return (int_t)lodword(v); }

  template<typename E> 
  inline E get_unit(const value &v) {
    return unpack_unit<E>(v);
  }

  inline int iface_no(const value &v) {
    return primitive_type(v) - PT_IFACE_PTR;
  } // only 16 ifaces so far
  inline int symbol_idx(const value &v) {
    assert(is_symbol(v));
    return unpack_uint32(v);
  }

  inline void dprint_value(value v) {
    dword d1 = hidword(v);
    dword d2 = lodword(v);
    printf("value=%x %x\n", d1, d2);
  }

  inline PRIMITIVE_TYPE CsPrimitiveType(value v) { return primitive_type(v); }

  inline value value_to_set(value v);

  /* output conversion functions
  inline void CsFloatToString(char *buf, float_t v)
  {
     sprintf(buf,"%f",(double)(v));
     if (!strchr(buf,'.'))
       strcat(buf,".0");
     else
     { // cut last zeroes:
        char* plast = buf + strlen(buf);
        while( --plast > (buf + 1) )
        {
          if( *plast != '0' ) break;
          if( *(plast-1) == '.' ) break;
          *plast = 0;
        }
     }
  }*/

  /* forward types */
  typedef struct CsScope    CsScope;
  typedef struct VM         VM;
  typedef struct CsCompiler CsCompiler;
  typedef struct dispatch   dispatch;
  struct stream;
  typedef struct CsStreamDispatch CsStreamDispatch;
  typedef struct CsFrame          CsFrame;
  typedef struct c_method         c_method;
  typedef struct vp_method        vp_method;
  typedef struct constant         constant;

/* round a size up to a multiple of the size of a int64 */
#define CsRoundSize(x) (((x) + 0x7) & ~0x7)

  /* saved interpreter state */
  struct CsSavedState {
    VM *     vm;
    value    globals;             // global variables
    value    ns;                  // namespace
    value *  sp;                  // stack pointer
    CsFrame *fp;                  // frame pointer
    value    env;                 // environment
    value    code;                // code obj
    long     pcoff;               // program counter offset
    value    currentTaskInstance; // current task instance
    value *  stack;               // stack base
    value *  stackTop;            // stack top
    // bool  isDetached;          // true if this was created on heap (Task)
    long     catch_pcoff;         // program counter offset for catch part

    CsSavedState *next; // next saved state structure

    CsSavedState(VM *vm);
    void store(VM *vm);
    void restore();
    void scan(VM *c);
    bool isTryState() const { return catch_pcoff >= 0; }

    ~CsSavedState();
  };

  // memory space block
  struct CsMemorySpace {
    // CsMemorySpace *next;
    byte *base;
    byte *free;
    byte *top;
    value cObjects; // c objects
    value bObjects; // byte objects
  };

  /* c_method handler */
  typedef value (*c_method_t)(VM *c);
  typedef value (*c_method_ext_t)(VM *c, value self, void *tag);

  /* handling [] access */
  typedef value (*c_get_item_t)(VM *c, value obj, value key);
  typedef void (*c_set_item_t)(VM *c, value obj, value key, value value);

#define C_METHOD_ENTRY(name, fcn) c_method(name, c_method_t(fcn))

  /* virtual property handlers */
  typedef value (*vp_get_t)(VM *c, value obj);
  typedef void (*vp_set_t)(VM *c, value obj, value value);

  typedef value (*vp_get_ext_t)(VM *c, value obj, void *tag);
  typedef void (*vp_set_ext_t)(VM *c, value obj, value value, void *tag);

  typedef bool (*set_value_t)(VM *c, value tag, value val);
  typedef bool (*get_value_t)(VM *c, value tag, value *pval);

#define VP_METHOD_ENTRY(name, get, set) vp_method(name, vp_get_t(get), vp_set_t(set))

#define CONSTANT_ENTRY(name, val) constant(name, val)

  /* scope structure */
  struct CsScope {
    VM *  c = nullptr;
    value globals = 0; // object/map of globals and default namespace in this scope
    value previous_ns = 0;     // saved namesapce
    bool  silent_unknowns = false; // unknown vars resolve to undefined. Otherwise error
                           // is generated.
    virtual bool setValue(value tag, value val) { return false; }
    virtual bool getValue(value tag, value *pval) { return false; }
    virtual ~CsScope() {}

    //CsScope() : c(0), globals(0), previous_ns(0), silent_unknowns(false), getter(nullptr), setter(nullptr) {}
  };

  /* get the obj holding the variables in a scope */
  inline value &CsScopeObject(CsScope *s) { return s->globals; }

  void CsTooManyArguments(VM *c);
  void CsTooFewArguments(VM *c);
  void CsWrongNumberOfArguments(VM *c);
  void CsTypeError(VM *c, value v);
  void CsUnexpectedTypeError(VM *c, value v, const char *expectedTypes);
  void CsStackOverflow(VM *c);
  void CsBadValue(VM *c, value v);
  void CsAlreadyDefined(VM *c, value tag);
  void CsUnsupportedBinaryOp(VM *c, int op, value v1, value v2);
  // void CsAddValueRef(VM* c,value* pval);
  // void CsFreeValueRef(value* pval);

  // protected value on C stack
  /*struct svalue
  {
  private:
    value   val;
    svalue(const svalue& pv) {;}
  public:
    explicit svalue(VM* c):val(0) { CsAddValueRef(c,&val); }
    explicit svalue(VM* c, value v):val(v) { CsAddValueRef(c,&val); val = v; }
    ~svalue() {}

    operator value() const { return val; }
    svalue& operator =( value v) { val = v; return *this; }
    value*  operator& () { return &val; }
    bool is_set() const { return val != 0; }
    operator bool() const { return is_set(); }
  };*/

  template<class TVM> 
  struct protector_t {
  protected:
    protector_t() = delete;
    protector_t(const protector_t &) = delete;
    
    TVM *c;
    int n;

  public:
    protector_t(TVM *c, value &v1);
    protector_t(TVM *c, value &v1, value &v2);
    protector_t(TVM *c, value &v1, value &v2, value &v3);
    protector_t(TVM *c, value &v1, value &v2, value &v3, value &v4);
    protector_t(TVM *c, value &v1, value &v2, value &v3, value &v4, value &v5);
    protector_t(TVM *c, value &v1, value &v2, value &v3, value &v4, value &v5, value &v6);
    protector_t(TVM *c, value &v1, value &v2, value &v3, value &v4, value &v5, value &v6, value &v7);
    protector_t(TVM *c, value &v1, value &v2, value &v3, value &v4, value &v5, value &v6, value &v7, value &v8);
    ~protector_t();
  };
  
  class storage;


// muti return helepers:

#define CS_RETURN4(c, r1, r2, r3, r4) { return cs_return(c, r1, r2, r3, r4); }
#define CS_RETURN3(c, r1, r2, r3) { return cs_return(c, r1, r2, r3); }
#define CS_RETURN2(c, r1, r2) {  return cs_return(c, r1, r2); }

  struct Exec;

  struct Exec {
    CsSavedState *states;
    uint          tryLevel;
    bool          awaiting;
    value         taskInstance;
    Exec()
        : states(nullptr), tryLevel(0), awaiting(false),
          taskInstance(NULL_VALUE) {}
    ~Exec() {}
    bool         run(VM *c, bool riseError = false);
    virtual void scan(VM *c);
  };

  struct loader {
    virtual tool::ustring do_resolve_url(const tool::ustring &base,
                                         const tool::ustring &relative)    = 0;
    virtual stream *do_open_stream(const tool::ustring &url, bool as_text) = 0;
  };

  struct gcable : virtual public resource 
  {
    TOOL_INTERFACE_1(tis, gcable, resource);
    virtual void on_gc(VM *c) = 0;
  };

  struct gc_callback {
    VM *c;
    gc_callback(VM *pvm);
    virtual ~gc_callback();
    virtual void on_GC(VM *c) = 0;
    virtual void detach_vm() { c = 0; }
    NONCOPYABLE(gc_callback)
  };

  struct debug_peer;



  /* interpreter context structure */
  struct VM : loader, tool::resource_x<VM> 
  {
    unsigned int valid = 0xAFED; /* must be a first member */

    CsCompiler *compiler = nullptr; /* compiler structure */
                          // CsScope  currentScope;       /* current scope */
                          //CsScope globalScope; /* the global scope */

                          // CsSavedState *        savedState;      /* saved state */
                          // int                   savedStateLevel; /* saved state nesting level -
                          // number of nesting native calls */
    Exec *exec = nullptr; /* current Exec */

    tool::array<CsScope *> scopes;

    value *argv = nullptr; /* arguments for current function */
    int    argc = 0; /* argument count for current function */

    value *  stack = nullptr;     /* stack base */
    value *  stackTop = nullptr;  /* stack top */
    value *  sp = nullptr;        /* stack pointer */
    CsFrame *fp = nullptr;        /* frame pointer */
    value    code = 0;            /* code obj */
    byte *   cbase = nullptr;     /* code base */
    byte *   pc = nullptr;        /* program counter */
    value    env = 0;             /* environment register */
    value    currentNS = 0;       /* current namespace register */

    value    val = 0;             /* value register */

    bool     enableDebug = false;

    stream *standardInput = nullptr;  /* standard input stream */
    stream *standardOutput = nullptr; /* standard output stream */
    stream *standardError = nullptr;  /* standard error stream */

    value objectObject = 0;   /* base of obj inheritance tree */
    value classObject = 0;    /* rack for Class methods and properties */
    value methodObject = 0;   /* obj for the Method type */
    value propertyObject = 0; /* obj for the Property type */
    value vectorObject = 0;   /* obj for the Vector type */
    value symbolObject = 0;   /* obj for the Symbol type */
    value stringObject = 0;   /* obj for the String type */
    value integerObject = 0;  /* obj for the Integer type */
    value colorObject = 0;    /* obj for the Color type */
    value lengthObject = 0;   /* obj for the Length type */
    value angleObject = 0;    /* obj for the Angle type */
    value durationObject = 0; /* obj for the Duration type */
    value floatObject = 0;    /* obj for the Float type */
    value errorObject = 0;    /* obj for the Error type */
                          // value regexpObject;          /* obj for the Regexp type */
    value byteVectorObject = 0; /* obj for the Bytes type */
    value tupleObject = 0;      /* obj for the Tuple type */
    value dateObject = 0;       /* obj for the Tuple type */

    value notificationList = 0; /* head of notifications list */

    value unhandledExceptionHandler = 0; /* remote debugger support */

                                     // suggest to declare it as storage* storage_list; - this needs l2elem
                                     // enabled in storage class.
    tool::array<value> storages; /* obj for opened storages */

    CsMemorySpace *oldSpace = nullptr;    /* old memory space */
    CsMemorySpace *newSpace = nullptr;    /* new memory space */
    long           expandSize = 0;  /* size of each expansion */
    unsigned long  totalMemory = 0; /* total memory allocated */
    unsigned long  allocCount = 0;  /* number of calls to CsAlloc */
    int            page_size = 0;

    uint32         sequentialId = 0;
    //void *         virtual_memory;

    dispatch *fileDispatch = nullptr;
    dispatch *storageDispatch = nullptr;
    dispatch *dbIndexDispatch = nullptr;
    dispatch *regexpDispatch = nullptr;
    dispatch *dateDispatch = nullptr;

    dispatch *systemDispatch = nullptr;
    dispatch *fileMonitorDispatch = nullptr;
#ifdef PROCESS_SUPPORT
    dispatch *processDispatch = nullptr;
#endif
    dispatch *fileOperationDispatch = nullptr;

    dispatch *assetDispatch = nullptr;

    dispatch *netDispatch = nullptr;
    dispatch *datasocketDispatch = nullptr;
    dispatch *socketDispatch = nullptr;
#ifdef WEBSOCKETS_SUPPORT
    dispatch *websocketDispatch = nullptr;
#endif

    dispatch *vmDispatch = nullptr;
    dispatch *xmlScannerDispatch = nullptr;
    dispatch *coroutineDispatch = nullptr;

    dispatch *typeDispatch = nullptr; /* the Type type */
    dispatch *types = nullptr;        /* derived types */
    void(*protectHandler)(VM *c, void *data) = nullptr;
    void *protectData = nullptr;

    pvalue pins;
    // protector *protector_top; /* values on C stack */

    bool collecting_garbage = false; /* true if in GC */

    unsigned int features = 0; /* must be very last */

    unsigned int gc_generation = 0; /* each CsCollectGarbage increases this number */

    void *extraData = nullptr;

    CsScope                    globalScope; /* the global scope */
    loader *                   ploader;
    handle<debug_peer>         pdebug;
    tool::mutex                guard;
    tool::array<gc_callback *> gc_callbacks;
    tool::ustring              script_url; // url of loading script
    tool::array<value *>       values_on_c_stack;
    tool::array<value>         pending_observers;
    tool::array<value>         pending_tasks;
    tool::array<value>         active_tasks;
    //tool::array<value>         generators;
    handle<tool::async::dispatch> queue;

    Exec main_exec;

    static unsigned int default_features;

    VM(unsigned int features = FEATURE_SYSINFO | FEATURE_EVAL |
                               FEATURE_FILE_IO | FEATURE_SOCKET_IO,
       long size = TISCRIPT_HEAP_SIZE, long expandSize = TISCRIPT_EXPAND_SIZE,
       long stackSize = TISCRIPT_STACK_SIZE);

    inline CsScope *currentScope() const {
      return scopes.last();
    } /* current scope */

    // reset current VM
    void reset();

    void set_loader(loader *pl = 0) { ploader = pl ? pl : this; }

    virtual void GC_started() {}
    virtual void GC_ended() {}

    virtual bool has_log_values() { return false; }
    virtual bool log_values(slice<value> list, value mode = UNDEFINED_VALUE, value location = UNDEFINED_VALUE, bool as_stringizer = false) { return false; }

    //virtual tool::async::tasks *get_tasks() { return nullptr; }

    inline value getCurrentNS() const {
      return currentNS == UNDEFINED_VALUE ? currentScope()->globals : currentNS;
    }

    virtual ~VM();

    bool can_use_feature(uint flag) { return (features & flag) != 0; }

    tool::ustring nativeThrowValue;

    static VM *get_current();

    // loader stuff +++
    virtual tool::ustring do_resolve_url(const tool::ustring &base,
                                         const tool::ustring &relative);
    virtual stream *do_open_stream(const tool::ustring &url, bool as_text);
    // loader stuff ---

    virtual tool::ustring resolve_url(const tool::ustring &base,
                                      const tool::ustring &relative) {
      return ploader->do_resolve_url(base, relative);
    }
    virtual stream *open_stream(const tool::ustring &url, bool as_text) {
      return ploader->do_open_stream(url, as_text);
    }

    virtual bool post(function<void()> &&pc) {
      queue->exec(std::move(pc), false);
      return false;
    }

    virtual bool send(function<void()> &&pc) {
      if (VM::get_current() == this) {
        pc();
        return true;
      }
      else {
        queue->exec(std::move(pc), true); 
        return false;
      }
    }

    void add_gc_callback(gc_callback *nc) { gc_callbacks.push(nc); }
    void remove_gc_callback(gc_callback *nc) {
      if (gc_callbacks.last() == nc)
        gc_callbacks.pop();
      else {
        bool r = gc_callbacks.remove_by_value(nc);
        assert(r);
        r = r;
      }
    }
    void detach_all_gc_callbacks() {
      FOREACH(n, gc_callbacks)
      gc_callbacks[n]->detach_vm();
    }

    bool delivering_notifications = false;
    bool deliver_notifications();

    virtual int pixels_per_inch() { return 96; }

    virtual bool resolveNamedColor(value tuple, uint* pout) { return false; }
    virtual bool resolveNamedLength(value tuple, value* pout) { return false; }

    virtual tool::ustring lang() { return tool::ustring(); }

    // extra convertors
    virtual tool::value x_value_to_value(value v) { return tool::value(); }
    virtual value       x_value_to_value(tool::value v) { return NULL_VALUE; }

  };

  typedef protector_t<VM> protector;

  struct CsTask : public Exec {
    bool          isTask; //  true - task, false - generator
    int           argc;   //  number of arguments passed in initial call
    CsSavedState *current;
    value callbacks;        // list of onFulfilled, onRejected pairs, only for the task
    value ns;               // namespace at task creation
    value method;
    value stack[TISCRIPT_STACK_SIZE / 4];
    CsTask(VM *c, bool isTask);
    ~CsTask();
  };

  inline gc_callback::gc_callback(VM *pvm) : c(pvm) {
    if (c) c->add_gc_callback(this);
  }
  inline gc_callback::~gc_callback() {
    if (c) c->remove_gc_callback(this);
  }

  /*enum CONNECT_RESULT
  {
    INVALID_PARAMETERS = -1,
    CONNECTED_OK = 0,
    REJECTED_BY_HOST,
    REJECTED_BY_ALIEN,
  };
  CONNECT_RESULT CsConnect(VM* host, const char* host_method_path, VM* alien,
  const char* alien_method_path);*/

  // bool CsConnect(VM* c, pvalue& input, pvalue& output, pvalue& error);

  /* pop a saved state off the stack */
  // inline void CsPopSavedState(VM* c)              { c->savedState =
  // c->savedState->next; }
  //       void CsSaveInterpreterState(VM *c,CsSavedState *s);
  //       void CsRestoreInterpreterState(VM *c);

  /*  boolean macros */
  // inline value    CsToBooleanB(VM* c, bool b)     { return b? TRUE_VALUE :
  // FALSE_VALUE; }
  inline bool CsTrueP(value v) { return v != FALSE_VALUE; }
  inline bool CsFalseP(value v) { return v == FALSE_VALUE; }

  value CsToBoolean(VM *c, value v);

  inline bool CsBooleanP(value v) {
    return v == TRUE_VALUE || v == FALSE_VALUE;
  }
  inline value CsMakeBoolean(bool b) { return b ? TRUE_VALUE : FALSE_VALUE; }

  /* get the global scope */
  inline CsScope *CsGlobalScope(VM *c) { return &c->globalScope; }
  inline CsScope *CsCurrentScope(VM *c) { return c->currentScope(); }

  /* argument list */
  inline void CsCheckArgRange(VM *c, int mn, int mx) {
    int n = c->argc;
    if (n < mn)
      CsTooFewArguments(c);
    else if (n > mx)
      CsTooManyArguments(c);
  }
  inline void CsCheckArgCnt(VM *c, int m) { CsCheckArgRange(c, m, m); }

  inline void CsCheckArgMin(VM *c, int m) {
    if (c->argc < m) CsTooFewArguments(c);
  }

#define CsCheckType(c, n, tp)                                                  \
  do {                                                                         \
    if (!tp(CsGetArg(c, n))) CsTypeError(c, CsGetArg(c, n));                   \
  } while (0)

#define CsCheckTypeOr(c, n, tp, tpa)                                           \
  do {                                                                         \
    if (!tp(CsGetArg(c, n)) && !tpa(CsGetArg(c, n)))                           \
      CsTypeError(c, CsGetArg(c, n));                                          \
  } while (0)

  //inline void CsSetRVal(VM *c, int n, value v) {
  //  c->vals   = 1 + n;
  //  c->val[n] = v;
  //}

  inline int    CsArgCnt(VM *c) { return c->argc; }
  inline value *CsArgPtr(VM *c) { return c->argv; }
  inline value &CsGetArg(VM *c, int n) { return c->argv[-n]; }

  inline value &CsCtorRes(VM *c) { return c->argv[1]; }
  inline value  CsGetArgSafe(VM *c, int n) {
    return n <= c->argc ? c->argv[-(n)] : NOTHING_VALUE;
  }

  /* stack manipulation macros */
  inline void CsCheck(VM *c, int n) {
    if (c->sp - n < &c->stack[0]) CsStackOverflow(c);
  }
  inline void CsPush(VM *c, const value &v) {
    *--(c)->sp = v;
    assert(v > 10);
  }
  inline void CsCPush(VM *c, const value &v) {
    if (c->sp > &c->stack[0])
      CsPush(c, v);
    else
      CsStackOverflow(c);
  }
  inline value &CsTop(VM *c) { return c->sp[0]; }
  inline value &CsTop(VM *c, int n) { return c->sp[n]; }
  inline void   CsSetTop(VM *c, const value &v) { *c->sp = v; }
  inline value &CsPop(VM *c) { return *c->sp++; }
  inline void   CsDrop(VM *c, int n) { c->sp += n; }

  /* destructor type */
  // typedef void (CsDestructor)(VM *c,void *data,void *ptr);
  typedef void (*destructor_t)(VM *c, value obj);
  typedef value (*new_instance_t)(VM *c, value proto);
  typedef bool (*print_t)(VM *c, value obj, stream *s, bool toLocale);
  typedef bool (*get_property_t)(VM *c, value &obj, value tag,
                                 value *pValue); // in: obj is the obj, out:
                                                 // "vtbl" (class) where
                                                 // property is found
  typedef bool (*set_property_t)(VM *c, value obj, value tag, value value);
  typedef bool (*add_constant_t)(VM *c, value obj, value tag, value value);

  typedef value (*get_item_t)(VM *c, value obj, value key);
  typedef void (*set_item_t)(VM *c, value obj, value key, value value);
  typedef bool (*del_item_t)(VM *c, value obj, value key);
  typedef void (*scan_t)(VM *c, value obj);

  typedef value (*get_next_element_t)(VM *c, value *index, value obj, int nr);
  typedef long (*thing_size_t)(value obj);
  typedef value (*thing_copy_t)(VM *c, value obj);
  typedef int_t (*thing_hash_t)(value obj);

  typedef bool (*call_method_t)(VM *c, value obj, value tag, int argc, value *pretval);

  // obj << operand
  // obj >> operand
  typedef value (*binary_operator_t)(VM *c, int op, value obj, value operand);

  typedef value& (*meta_accessor_t)(value thing);
  
  /* type pdispatch structure */
  struct dispatch {
    const char *       typeName;
    dispatch *         baseType;
    get_property_t     getProperty;
    set_property_t     setProperty;
    new_instance_t     newInstance;
    print_t            print;
    thing_size_t       size;
    thing_copy_t       copy;
    scan_t             scan;
    thing_hash_t       hash;
    get_item_t         getItem;
    set_item_t         setItem;
    get_next_element_t getNextElement;
    add_constant_t     addConstant;
    del_item_t         delItem;
    call_method_t      handleCall; // native call used by e.g. Sciter behavior
    dispatch **        interfaces;
    value              obj; // a.k.a. class vtbl;
    long               dataSize;
    destructor_t       destroy;
    void *             destroyParam;
    binary_operator_t  binOp; // address of function responsible for '<<'
                              // (BC_SHL) and '>>' (BC_SHR) operations.
    print_t            printType;
    meta_accessor_t    metaAccessor;

    dispatch *next;    // next in the list of dynamically allocated types
  };

#define BEGIN_DISPATCH(name) \
  static dispatch init_##name() { \
  dispatch d = {};

#define END_DISPATCH(name) \
  return d; } \
  dispatch name = init_##name();

  extern dispatch CsIntegerDispatch;
  extern dispatch CsColorDispatch;
  extern dispatch CsLengthDispatch;
  extern dispatch CsAngleDispatch;
  extern dispatch CsDurationDispatch;
  extern dispatch CsSymbolDispatch;
  extern dispatch CsFloatDispatch;

  /* VALUE */

  struct header {
    dispatch *pdispatch;
    uint32    id; // quasi unique id, see VM.entityId(thing)
    header() : pdispatch(0), id(0) {}
  };

#if defined(TISCRIPT_USE_PERSISTENCE)
  struct persistent_header : header {
    value   vstorage;
    uint32  oid;
    uint32  status;

    uint32  flags;
    value   proto;
    value   observer;
    
    bool loaded() const { return tool::get_bit(0x1, status); }
    void loaded(bool v) { tool::set_bit(0x1, status, v); }

    bool modified() const { return tool::get_bit(0x2, status); }
    void modified(bool v) { tool::set_bit(0x2, status, v); }

    persistent_header() : vstorage(0), oid(0), status(0), flags(0), proto(UNDEFINED_VALUE), observer(0) {}
  };
#else
  typedef header persistent_header;
#endif

  /* type macros */
  inline bool CsPointerP(value o) { return is_ptr(o); }

  extern dispatch CsNilDispatch;

  INLINE dispatch *CsQuickGetDispatch(value o) {
    header *ph = ptr<header>(o);
    assert(ph && ph->pdispatch);
    if(!ph || !ph->pdispatch)
      return &CsNilDispatch; // memory corruption, the best we can do is to return CsNilDispatch here
    return ph->pdispatch;
  }

  INLINE bool CsIsValidPtr(VM *c, value o) {
    if (!CsPointerP(o)) return true;
    byte *ph = ptr<byte>(o);
    assert(ph >= c->newSpace->base && ph < c->newSpace->free);
    return ph >= c->newSpace->base && ph < c->newSpace->free;
  }

  inline bool CsColorP(value o) { return primitive_type(o) == PT_COLOR; }
  inline bool CsLengthP(value o) { return primitive_type(o) == PT_LENGTH; }
  inline bool CsAngleP(value o) { return primitive_type(o) == PT_ANGLE; }
  inline bool CsDurationP(value o) { return primitive_type(o) == PT_DURATION; }

  
  inline bool CsUnitsP(value o) {
    auto pt = primitive_type(o);
    return pt == PT_COLOR || pt == PT_LENGTH || pt == PT_ANGLE ||
           pt == PT_DURATION;
  }

  value   CsMakeLength(float_t v, LENGTH_UNIT_TYPE ut);
  value   CsMakeAngle(float_t v, ANGLE_UNIT_TYPE ut, bool raw = false /* true: v are radians */ );

  value   CsMakeDuration(float_t v, DURATION_UNIT_TYPE ut, bool raw = false /* true: v are seconds */ );
  float_t CsDurationSeconds(value obj);
  DURATION_UNIT_TYPE CsDurationUnits(value obj);

  unsigned color_value(value obj);

  extern dispatch CsStringDispatch;

  dispatch *CsGetDispatch(value o);

  INLINE bool CsIsType(value o, dispatch *t) { return CsGetDispatch(o) == t; }
  INLINE bool CsIsBaseType(value o, dispatch *t) 
  {
    dispatch *pd = CsGetDispatch(o);
    assert(pd);
    if (pd) 
    {
      if (pd == t) return true;
      if (pd->baseType == t) return true;
      pd = pd->baseType;
      if (pd->baseType == t) return true;
    }
    return false;
  }
  INLINE bool CsIsBaseType(value o, dispatch *t1, dispatch *t2, dispatch *t3)
  {
    dispatch *pd = CsGetDispatch(o);
    assert(pd);
    if (pd)
    {
      if (pd == t1) return true;
      if (pd == t2) return true;
      if (pd == t3) return true;
      if (pd->baseType == t1) return true;
      if (pd->baseType == t2) return true;
      if (pd->baseType == t3) return true;
      pd = pd->baseType; 
      if (pd) {
        if (pd->baseType == t1) return true;
        if (pd->baseType == t2) return true;
        if (pd->baseType == t3) return true;
      }
    }
    return false;
  }


  INLINE bool CsQuickIsType(value o, dispatch *t) {
    return CsQuickGetDispatch(o) == t;
  }
  inline const char *CsTypeName(value o) { return CsGetDispatch(o)->typeName; }
  inline const char *CsQuickTypeName(value o) {
    return CsQuickGetDispatch(o)->typeName;
  }

  inline void CsCheckArgType(VM *c, int n, dispatch *t) {
    if (!CsIsType(CsGetArg(c, n), t))
      CsUnexpectedTypeError(c, CsGetArg(c, n), t->typeName);
  }
  inline void CsCheckArgType(VM *c, int n, dispatch *t1, dispatch *t2) {
    if (!CsIsType(CsGetArg(c, n), t1) && !CsIsType(CsGetArg(c, n), t2))
      CsUnexpectedTypeError(c, CsGetArg(c, n), t1->typeName);
  }

  inline void CsSetDispatch(value o, dispatch *d) {
    ptr<header>(o)->pdispatch = d;
  }
  inline bool CsGetProperty1(VM *c, value &o, value t, value *pv) {
    return CsGetDispatch(o)->getProperty(c, o, t, pv);
  }
  inline bool CsSetProperty(VM *c, value o, value t, value v) {
    return CsGetDispatch(o)->setProperty(c, o, t, v);
  }

  inline bool CsAddConstant(VM *c, value o, value t, value v) {
    dispatch *pd = CsGetDispatch(o);
    if (pd->addConstant) return pd->addConstant(c, o, t, v);
    return false;
  }
  bool CsSetProperty1(VM *c, value o, value t, value v);

  inline value CsNewInstance(VM *c, value o) {
    return CsGetDispatch(o)->newInstance(c, o);
  }
  extern value CsNewClassInstance(VM *c, value parentClass, value nameSymbol);
  extern value CsNewNamespaceInstance(VM *c, value parentClass,
                                      value nameSymbol);

  inline bool CsPrintValue(VM *c, value o, stream *s, bool toLocale = false) {
    dispatch *d = CsGetDispatch(o);
    return d->print(c, o, s, toLocale);
  }

  inline bool CsPrintType(VM *c, value o, stream *s, bool toLocale = false) {
    dispatch *d = CsGetDispatch(o);
    if (d->printType)
      return d->printType(c, o, s, toLocale);
    else
      return s->put_str(d->typeName);
  }

  inline value CsCopyValue(VM *c, value o) {
    if (is_iface_ptr(o)) {
      value obase = iface_base(o);
      assert(!is_iface_ptr(obase));
      int n = iface_no(o);
      return iface_value((tis::header *)CsGetDispatch(obase)->copy(c, obase),n);
    }
    dispatch *pd = CsGetDispatch(o); // will never return nullptr
    //assert(pd);
    //if (!pd) return NOTHING_VALUE; // something is terribly wrong
    return pd->copy(c, o);
  }
  inline int_t CsHashValue(value o) { return CsGetDispatch(o)->hash(o); }

  bool CsGetObjectProperty(VM *c, value &obj, value tag, value *pValue);
  bool CsSetObjectPropertyNoLoad(VM *c, value obj, value tag, value value);
  bool CsSetObjectProperty(VM *c, value obj, value tag, value value);
  long CsObjectSize(value obj);

  bool CsFetchProperty(VM *c, value& obj, value& self, value tag, value *pValue);
  bool CsTryStoreProperty(VM *c, value obj, value self, value tag, value val, int_t *pHashValue = nullptr, int_t *pIndex = nullptr);
  tristate_v CsStoreProperty(VM *c, value obj, value self, value tag, value val, int_t *pHashValue = nullptr, int_t *pIndex = nullptr);

  void CsObjectScan(VM *c, value obj);
  void CsCObjectScan(VM *c, value obj);

  inline value CsGetItem(VM *c, value o, value key) {
    return CsGetDispatch(o)->getItem(c, o, key);
  }
  inline void CsSetItem(VM *c, value o, value key, value val) {
    CsGetDispatch(o)->setItem(c, o, key, val);
  }

  bool CsGetProperty(VM *c, value obj, value tag, value *pValue);

  bool CsGetProperty(VM *c, value obj, const char *tag, ustring &val);
  bool CsGetProperty(VM *c, value obj, const char *tag, bool &val);
  bool CsGetProperty(VM *c, value obj, const char *tag, int_t &val);
  bool CsGetProperty(VM *c, value obj, const char *tag, float_t &val);
  bool CsGetPropertyMilliseconds(VM *c, value obj, const char *tag, int_t &val);
  bool CsGetProperty(VM *c, value obj, const char *tag, value &val, dispatch *pd);

  bool CsSetProperty(VM *c, value obj, const char *tag, wchars str);
  bool CsSetProperty(VM *c, value obj, const char *tag, chars str);
  bool CsSetProperty(VM *c, value obj, const char *tag, bool val);
  bool CsSetProperty(VM *c, value obj, const char *tag, int_t val);
  bool CsSetProperty(VM *c, value obj, const char *tag, float_t val);
  bool CsSetProperty(VM *c, value obj, const char *tag, value val);
  // bool    CsSetProperty(VM *c,value obj,const char* tag, value& val,
  // dispatch* pd);

  /* BROKEN HEART */

  struct CsBrokenHeart : public header {
    value forwardingAddr;
  };

  extern dispatch CsBrokenHeartDispatch;

  inline bool CsBrokenHeartP(value o) {
    return CsIsType(o, &CsBrokenHeartDispatch);
  }
  inline value CsBrokenHeartForwardingAddr(value o) {
    return ptr<CsBrokenHeart>(o)->forwardingAddr;
  }
  inline void CsBrokenHeartSetForwardingAddr(value o, value v) {
    ptr<CsBrokenHeart>(o)->forwardingAddr = v;
  }

  inline bool  CsIntegerP(value o) { return is_int(o); }
  inline value CsMakeInteger(int val) { return int_value(val); }
  // inline value CsMakeInteger(uint val)         { return int_value(val); }
  //       value CsMakeInteger(value);
  inline int_t CsIntegerValue(value o) { return to_int(o); }
  //       int_t CsIntegerValue(int);

  inline bool    CsFloatP(value o) { return is_float(o); }
  inline float_t CsFloatValue(value o) {
    assert(CsFloatP(o));
    return to_float(o);
  }
  // inline void CsSetFloatValue(value o,float_t v)  { ptr<CsFloat>(o)->value =
  // v; }
  inline value CsMakeFloat(float_t v) { return float_value(v); }

  inline uint CsColorValue(value o) { return unpack_uint32(o); }
  value       CSF_color(VM *c);

  inline value CsMakeColor(int_t colorref) {
    return pack_value(PT_COLOR, 0, colorref);
  }
  value CsMakeColor(byte r, byte g, byte b, byte a);

  float_t CsFloatOrPixelsValue(VM *c, value o, float_t defval = 0.0f);
  float_t CsFloatOrRadiansValue(value o, float_t defval = 0.0f);

  int pixelsPerInch(VM *c);

  value CsMakeLengthFromTool(const tool::value& v);
  bool  CsLengthPrintFx(VM *c, value obj, stream *s, bool toLocale = false);
  tool::value CsLengthToolValue(value lv);
  double      CsLengthPixels(VM *c, value lv);

  // value  CsMakeAngle(double radians);
  double CsAngleRadians(value v); // radians
  ANGLE_UNIT_TYPE CsAngleUnits(value v);

  /* NUMBER */

  inline bool CsNumberP(value o) { return CsIntegerP(o) || CsFloatP(o); }
  inline bool CsNumberOrLengthP(value o) { return CsNumberP(o) || CsLengthP(o); }

  //uint lu2tool(UNIT_TYPE ut);
  //UNIT_TYPE tool2lu(uint ut);

  /* STRING */

  struct CsString : public header {
    uint  size;
    uint  allocatedSize;
    int_t hash;
    /*  unsigned wchar data[0]; */
  };

  extern dispatch CsStringDispatch;

  inline bool   CsStringP(value o) { return CsIsType(o, &CsStringDispatch); }
  inline wchar *CsStringAddress(value o) {
    return (wchar *)(ptr<byte>(o) + sizeof(CsString));
  }
  inline wchar CsStringChar(value o, int i) { return CsStringAddress(o)[i]; }
  inline void  CsSetStringChar(value o, int i, wchar b) {
    CsStringAddress(o)[i] = b;
  }

  inline bool is_string(value o) { return CsStringP(o); }

  inline uint CsStringSize(value o) { return ptr<CsString>(o)->size; }

  inline tool::wchars CsStringChars(value o) {
    return tool::wchars((wchar *)(ptr<byte>(o) + sizeof(CsString)),
                        ptr<CsString>(o)->size);
  }
  inline tool::bytes CsStringBytes(value o) {
    return tool::bytes(ptr<byte>(o) + sizeof(CsString),
                       ptr<CsString>(o)->size * sizeof(wchar));
  }
  inline tool::tslice<wchar> CsStringTargetChars(value o) {
    return tool::tslice<wchar>((wchar *)(ptr<byte>(o) + sizeof(CsString)),
                               ptr<CsString>(o)->size);
  }
  inline tool::tslice<wchar> CsStringAllocatedChars(value o) {
    return tool::tslice<wchar>((wchar *)(ptr<byte>(o) + sizeof(CsString)),
                               ptr<CsString>(o)->allocatedSize);
  }

  inline void CsSetStringSize(value o, size_t sz) {
    ptr<CsString>(o)->size = uint(sz);
  }
  inline void CsSetStringAllocatedSize(value o, size_t sz) {
    ptr<CsString>(o)->allocatedSize = uint(sz);
  }
  inline uint CsStringAllocatedSize(value o) {
    return ptr<CsString>(o)->allocatedSize;
  }

  extern value CsStringHead(VM *c, value s,
                            value d); //  "one.two.three" ~/ "." == "one"
  extern value CsStringTail(VM *c, value s,
                            value d); //  "one.two.three" ~% "." == "two.three"
  extern value CsStringHeadR(VM *c, value s,
                             value d); //  "one.two.three" /~ "." == "one.two"
  extern value CsStringTailR(VM *c, value s,
                             value d); //  "one.two.three" %~ "." == "three"

  tool::string utf8_string(value o);

  /* STRING */

  struct byte_vector : public header {
    size_t                   size;
    value                    type;
    value                    name;
    value                    next; // in chain of CObjects
    array<byte>::array_data *heap_data;

    /*  unsigned byte data[0]; */
  };

  extern dispatch CsByteVectorDispatch;

  inline bool CsByteVectorP(value o) {
    return CsIsType(o, &CsByteVectorDispatch);
  }
  size_t             CsByteVectorSize(value o);
  byte *             CsByteVectorAddress(value o);
  inline tool::bytes CsByteVectorBytes(value o) {
    return tool::bytes(CsByteVectorAddress(o), CsByteVectorSize(o));
  }
  tool::array<byte> CsByteVectorArray(value o);
  inline byte       CsByteVectorByte(value o, int i) {
    return CsByteVectorAddress(o)[i];
  }

  inline value CsByteVectorType(value o) { return ptr<byte_vector>(o)->type; }
  inline void  CsSetByteVectorType(value o, value t) {
    ptr<byte_vector>(o)->type = t;
  }
  inline value CsByteVectorName(value o) { return ptr<byte_vector>(o)->name; }
  inline void  CsSetByteVectorName(value o, value t) {
    ptr<byte_vector>(o)->name = t;
  }

  value CsMakeByteVector(VM *c, const byte *data, int_t size);
  value CsMakeFilledByteVector(VM *c, byte fill, int_t size);

  value CsMakeByteVector(VM *c, const array<byte> &data);

  value CsMakeCharString(VM *c, const wchar *data, size_t size);
  value CsMakeCString(VM *c, const char *str);
  value CsMakeCString(VM *c, const wchar *str);

  value CsConcatStrings(VM *c, value firststr, value secondstr, bool append);

  value CsMakeString(VM *c, tool::chars str);
  value CsMakeString(VM *c, tool::wchars str);
  value CsMakeFilledString(VM *c, wchar fill, size_t size);

  value CsStringSlice(VM *c, value s, int start, int end);
  value CsVectorSlice(VM *c, value s, int start, int end);
  value CsMakeSubString(VM *c, value s, int start, int length);

/* SYMBOL */

/*struct CsSymbol: public header
{
    int_t hashValue;
    int printNameLength;
    unsigned char printName[1];
};*/

// symbol_t here is a number between 0..2**30
#define CsSymbolIndexMax ((1 << 29) - 1)

  typedef unsigned int symbol_t;

  inline bool CsSymbolP(value o) { return is_symbol(o); }

  tool::ustring CsSymbolName(value o);
  inline value  CsSymbol(symbol_t idx) {
    // assert(idx < CsSymbolIndexMax); // shall I rise error here?
    return symbol_value(idx);
  }

  tool::ustring symbol_name(symbol_t idx);

  // value   CsMakeSymbol(VM *c,const char *printName,int length = 0);
  value CsMakeSymbol(VM *c, const wchar *printName, int length = 0);
  value CsMakeSymbol(VM *c, wchars name);
  value CsIntern(VM *c, value printName);
  // value   CsInternCString(VM *c,const char *printName);
  value CsSymbolOf(const char *printName);
  value CsSymbolOf(chars printName);
  value CsSymbolOf(const wchar *printName);
  value CsSymbolOf(wchars printName);
  value CsSymbolOf(const ustring &printName);

  uint CsSymbolIdx(const wchar *printName);
  uint CsSymbolIdx(wchars printName);
  // const char*  CsSymbolIntern(const char *printName);

  inline value symbol_value(const char *str) { return CsSymbolOf(str); }

  value CsInternString(VM *c, const char *printName, int length);
  // bool    CsGlobalValue(CsScope *scope,value sym,value *pValue);

  bool CsGetGlobalValue(VM *c, value sym, value *pValue);
  void CsSetGlobalValue(CsScope *scope, value sym, value value,
                        bool create = true);

  bool CsGetGlobalOrNamespaceValue(VM *c, value tag, value *pValue);
  bool CsSetGlobalOrNamespaceValue(VM *c, value tag, value val, bool create);

  void CsSetNamespaceValue(VM *c, value sym, value val, bool create, bool force = false);
  void CsSetNamespaceConst(VM *c, value sym, value val);
  void CsSetNamespaceGetterSetter(VM *c, value sym, value fun, bool setter);

  void CsSetClassMemberTemplateVar(VM *c, value sym, value val); // class is c->currentNS
  
  value CsGetGlobalValueByPath(VM *c, const char *path);
  value CsGetGlobalValueByPath(VM *c, value root_ns, const char *path);
  bool  CsSetGlobalValueByPath(VM *c, value root_ns, const char *path, value val);

  // for(var el in coll) enumerator
  value CsGetNextMember(VM *c, value *index, value collection, int nr);

  // get_prop helper, fetches property value
  inline bool _CsGetProp(VM *c, value o, const char *name, value *pv) {
    return CsGetDispatch(o)->getProperty(c, o, CsSymbolOf(name), pv);
  }

  inline int_t CsGetProp(VM *c, value o, const char *name, int_t defval) {
    value v;
    if (!_CsGetProp(c, o, name, &v) || !CsIntegerP(v)) return defval;
    return CsIntegerValue(v);
  }
  inline float_t CsGetProp(VM *c, value o, const char *name, float_t defval) {
    value v;
    if (!_CsGetProp(c, o, name, &v) || !CsIntegerP(v)) return defval;
    return CsFloatValue(v);
  }
  inline bool CsGetProp(VM *c, value o, const char *name, bool defval) {
    value v;
    if (!_CsGetProp(c, o, name, &v) || !CsBooleanP(v)) return defval;
    return v == TRUE_VALUE;
  }

  /* FIXED VECTOR */
  struct CsFixedVector : public header {
    union {
      ssize_t size;
      value   placeholder; // it shall be at least of CsBrokenHeart size 
                         // that is header + value (forwarding address)
    };
  };

  extern dispatch CsFixedVectorDispatch;
  extern dispatch CsSpreadDispatch; // CsFixedVectorDispatch
  extern dispatch CsValueListDispatch; // CsFixedVectorDispatch
  extern dispatch CsVirtualPropertyDispatch; // CsFixedVectorDispatch

  inline bool CsFixedVectorP(value o) {
    return CsIsBaseType(o, &CsFixedVectorDispatch);
  }

  inline bool CsValueListP(value o) {
    return CsIsType(o, &CsValueListDispatch);
  }

  inline bool CsSpreadP(value o) {
    return CsIsBaseType(o, &CsSpreadDispatch);
  }

  inline bool CsVirtualPropertyP(value o) {
    return CsIsBaseType(o, &CsVirtualPropertyDispatch);
  }
 
  inline value *CsFixedVectorAddress(value o) {
    return ((value *)(ptr<char>(o) + sizeof(CsFixedVector)));
  }
  inline value CsFixedVectorElement(value o, int i) {
    return CsFixedVectorAddress(o)[i];
  }
  inline void CsSetFixedVectorElement(value o, int i, value v) {
    CsFixedVectorAddress(o)[i] = v;
  }
  inline int_t CsFixedVectorSize(value o) {
    return (int_t) ptr<CsFixedVector>(o)->size;
  }
  inline void CsSetFixedVectorSize(value o, int_t ns) {
    ptr<CsFixedVector>(o)->size = ns;
  }

  inline int_t CsValueListSize(value o) {
    return CsFixedVectorSize(o);
  }

  inline value CsValueListElement(value o, int_t n) {
    //n = CsValueListSize(o) - 1 - n;
    return n >= 0 && n < CsValueListSize(o) ? CsFixedVectorElement(o, n) : NOTHING_VALUE;
  }

  inline value CsValueListLastElement(value o) {
    int n = CsValueListSize(o);
    return CsFixedVectorElement(o, n - 1);
  }


  value CsMakeFixedVectorValue(VM *c, dispatch *type, int size);
  value CsMakeFixedVectorCopy(VM *c, dispatch *type, int size, value otherFW);
  /* construct FV */
  value        CsMakeFixedVector(VM *c, dispatch *type, int argc, value *argv);
  inline value CsMakeFixedVector(VM *c, int size) { return CsMakeFixedVectorValue(c, &CsFixedVectorDispatch, size); }
 
    /* OBJECT */

#define IS_SEALED 0x0001
#define IS_STRONG_SEALED 0x0002
#define IS_FROZEN 0x0004
#define IS_PROTOTYPE_OBJECT 0x0008

  struct object : public persistent_header {
    value properties = UNDEFINED_VALUE;
    value events = NULL_VALUE;
    value meta = 0;
    int_t propertyCount = 0;
  };

  struct klass : public object {
    value name; // fully qualified name
    value undefinedPropHandler;
    value ns;       // namespace this class is in
    value thisVars; // object that holds instance template.
  };

  extern dispatch CsObjectDispatch;

  inline bool  CsObjectP(value o) { return CsIsType(o, &CsObjectDispatch); }
  inline value CsObjectClass(value o) { return ptr<object>(o)->proto; }
  inline void  CsSetObjectClass(value o, value v) { ptr<object>(o)->proto = v; }
  inline value CsObjectProperties(value o) {
    return ptr<object>(o)->properties;
  }
  inline void CsSetObjectProperties(value o, value v) {
    ptr<object>(o)->properties = v;
  }
  inline int_t CsObjectPropertyCount(value o) {
    return ptr<object>(o)->propertyCount;
  }
  inline void CsSetObjectPropertyCount(value o, int_t v) {
    ptr<object>(o)->propertyCount = v;
  }

  inline value CsObjectEvents(value o) { return ptr<object>(o)->events; }
  inline void  CsSetObjectEvents(value o, value v) {  ptr<object>(o)->events = v; }

  inline value CsObjectMeta(value o) { return ptr<object>(o)->meta; }
  inline void  CsSetObjectMeta(value o, value v) { ptr<object>(o)->meta = v; }

  //value CsFlattenObject(VM *c, value obj);
  //value CsStringifyCollection(VM *c, value obj);
  

  inline uint &CsEntityFlags(value o);
  inline value CsEntityObserver(value o) { return ptr<persistent_header>(o)->observer; }
  inline void  CsSetEntityObserver(value o, value v) { ptr<persistent_header>(o)->observer = v; }

  inline value CsEntityClass(value o) { return ptr<persistent_header>(o)->proto; }
  inline void  CsSetEntityClass(value o, value v) { ptr<persistent_header>(o)->proto = v; }

  inline void CsPurgeObject(value obj) // makes it "pristine"
  {
    CsSetObjectClass(obj, UNDEFINED_VALUE);
    CsSetObjectProperties(obj, UNDEFINED_VALUE);
    CsSetObjectPropertyCount(obj, 0);
    CsSetEntityObserver(obj, 0);
    CsSetObjectEvents(obj, NULL_VALUE);
    CsEntityFlags(obj) = 0;
  }
    
  ustring CsObjectClassName(VM *c, value obj);
  ustring CsClassClassName(VM *c, value cls);
  ustring CsClassClassFullName(VM *c, value cls);

  enum OBJECT_MUTATION_OP {
    ADD_PROPERTY,
    UPDATE_PROPERTY,
    DELETE_PROPERTY,
    ADD_RANGE,
    UPDATE_RANGE,
    DELETE_RANGE,
    MAX_OBJECT_MUTATION_CODE
  };

  void CsEnqueueNotification(VM *c, value observer, value obj, value tag,
                             value newval, value oldval,
                             OBJECT_MUTATION_OP mop);

  // void CsEnqueueNotification(VM* c, value observer,
  //                                   value obj,
  //                                   value tag,
  //                                   value newval, value oldval,
  //                                   OBJECT_MUTATION_OP mop);

  extern dispatch CsClassDispatch;
  extern dispatch CsNamespaceDispatch;
  inline bool     CsClassP(value o) { return CsIsType(o, &CsClassDispatch); }
  inline bool     CsNamespaceP(value o) {
    return CsIsType(o, &CsNamespaceDispatch);
  }

  inline bool CsTypeP(VM *c, value o) { return CsIsType(o, c->typeDispatch); }

  inline value CsClassName(value o) { return ptr<klass>(o)->name; }
  inline value CsClassNamespace(value o) { return ptr<klass>(o)->ns; }
  inline void  CsSetClassName(value o, value v) { ptr<klass>(o)->name = v; }
  inline void  CsSetClassNamespace(value o, value v) { ptr<klass>(o)->ns = v; }

  inline value CsClassUndefinedPropHandler(value o) {
    return ptr<klass>(o)->undefinedPropHandler;
  }
  inline void CsSetClassUndefinedPropHandler(value o, value v) {
    ptr<klass>(o)->undefinedPropHandler = v;
  }

  inline value CsClassThisVars(value o) { return ptr<klass>(o)->thisVars; }
  inline void  CsSetClassThisVars(value o, value v) { ptr<klass>(o)->thisVars = v; }
  //value CsGetClassInstancePrototype(VM* c, value o);

  /*inline bool CsTryGetClassInstancePrototype(value o, value& proto) { 
    if (CsClassP(o) && CsObjectP(ptr<klass>(o)->prototype)) {
      proto = ptr<klass>(o)->prototype;
      return true;
    }
    return false;
  }*/

  inline value CsNamespaceIncludedList(value o) {
    return ptr<klass>(o)->thisVars; // Note hijacking this filed that is not used in namespaces
  }
  inline void CsSetNamespaceIncludedList(value o, value v) {
    ptr<klass>(o)->thisVars = v; // Note hijacking this filed that is not used in namespaces
  }

    // set modified flag in this object

#if defined(TISCRIPT_USE_PERSISTENCE)
  inline void CsSetModified(value obj, bool onOff) {
    ptr<persistent_header>(obj)->modified(onOff);
  }
  inline bool CsIsModified(value obj) {
    return ptr<persistent_header>(obj)->modified();
  }
#else
  inline void    CsSetModified(value, bool) { ; }
  inline bool    CsIsModified(value) { return false; }
#endif

  value CsMakeObject(VM *c, value proto);
  value CsCloneObject(VM *c, value obj, bool deep = false);
  value CsCloneObject(VM *c, value obj, slice<value> inc, slice<value> exc);
  value CsExtendObject(VM *c, value obj, value other, bool deep);

  value CsFindProperty(VM *c, value obj, value tag, int_t *pHashValue,
                       int_t *pIndex);
  value CsFindFirstSymbol(VM *c, value obj);
  value CsFindNextSymbol(VM *c, value obj, value tag);
  bool  CsRemoveObjectProperty(VM *c, value obj, value tag);
  bool  CsAddObjectConstant(VM *c, value obj, value tag, value val);
  void  CsMergeThisVarsFromClass(VM *c, value obj,value proto); // used by Behavior.attached to merge 'this var's

  bool CsObjectsEqual(VM *c, value v1, value v2, array<value> &stack);

  value CsClone(VM *c, value obj, bool deep);

  //#ifndef KEY_STRING_TO_SYMBOL
  // value CsMakeJsonObject(VM *c,value proto);
  //#endif

  // value CsMakeClass(VM *c,value proto);

  /* CsScanOnject - scans all key/values in the given object

  struct object_scanner
  {
    virtual bool item( VM *c, value key, value val ) = 0; // true - continue,
  false - stop;
  };


  void   CsScanObject( VM *c, value o, object_scanner& osc );
  void   CsScanObjectNoLoad( VM *c, value o, object_scanner& osc );
  */

  /* COBJECT */

  struct c_object : public object {
    value next;
  };

  struct c_ptr_object : public c_object {
    void *ptr;
  };

  // inline int_t CsCObjectSize(value o)  { return ((c_object *)o.ptr())->size;
  // }  inline void  CsSetCObjectSize(value o,int_t v) { ((c_object
  // *)o.ptr())->size = v; }

  //#define CsCObjectPrototype(o)       (((c_object *)o.ptr())->prototype)
  //#define CsSetCObjectPrototype(o,v)  (((c_object *)o.ptr())->prototype = (v))

  inline void *CsCObjectDataAddress(value o) {
    return (void *)(ptr<char>(o) + sizeof(c_object));
  }
  inline void *CsCObjectValue(value o) { return ptr<c_ptr_object>(o)->ptr; }
  inline void  CsSetCObjectValue(value o, void *v) {
    ptr<c_ptr_object>(o)->ptr = v;
  }

  bool      CsCObjectP(value val);
  dispatch *CsMakeCObjectType(VM *c, const char *typeName,
                              c_method *methods, vp_method *properties,
                              constant *constants, long size);
  value     CsMakeCObject(VM *c, dispatch *d);
  dispatch *CsMakeCPtrObjectType(VM *c, const char *typeName,
                                 c_method *methods, vp_method *properties,
                                 constant *constants = nullptr);
  value     CsMakeCPtrObject(VM *c, dispatch *d, void *ptr);
  void      CsEnterCObjectMethods(VM *c, dispatch *d, c_method *methods,
                                  vp_method *properties, constant *constants = 0);

  bool CsGetVirtualProperty(VM *c, value &obj, value proto, value tag,
                            value *pValue);
  bool CsSetVirtualProperty(VM *c, value obj, value proto, value tag,
                            value value);

  bool CsGetCObjectProperty(VM *c, value &obj, value tag, value *pValue);
  bool CsSetCObjectProperty(VM *c, value obj, value tag, value val);
  bool CsAddCObjectConstant(VM *c, value obj, value tag, value val);

  bool CsGetNonObjectProperty(VM *c, value &obj, value proto, value tag,
                              value *pValue);
  bool CsSetNonObjectProperty(VM *c, value obj, value proto, value tag,
                              value value);

  extern dispatch CsCObjectDispatch;
  //extern dispatch CsCPtrObjectDispatch;

  // value CSF_prototype(VM *c,value obj);

  /* VECTOR */

  extern dispatch CsVectorDispatch;
  // extern dispatch CsMovedVectorDispatch;

  struct vector : public persistent_header {
    int_t size;
    value data; // fixed vector
  };

  inline bool CsVectorP(value o) { return CsIsBaseType(o, &CsVectorDispatch); }
  // inline bool   CsMovedVectorP(value o)           { return
  // CsIsType(o,&CsMovedVectorDispatch); }
  inline int_t CsVectorSizeI(value o) { return ptr<vector>(o)->size; }
  inline void  CsSetVectorSizeI(value o, int_t s) { ptr<vector>(o)->size = s; }
  inline void  CsSetVectorSize(value o, int_t s) {
    // assert(!CsMovedVectorP(o));
    CsSetModified(o, true);
    ptr<vector>(o)->size = s;
  }
  // inline value  CsVectorForwardingAddr(value o)   { return
  // ptr<vector>(o)->d.forwardingAddr; }  inline void
  // CsSetVectorForwardingAddr(value o, value a) {
  // ptr<vector>(o)->d.forwardingAddr = a; }
  inline int_t CsVectorMaxSize(value o) {
    return CsFixedVectorSize(ptr<vector>(o)->data);
  }
  // inline void   CsSetVectorMaxSize(value o,int_t s) { ptr<vector>(o)->maxSize
  // = s; }
  inline value *CsVectorAddressI(value o) {
    return CsFixedVectorAddress(ptr<vector>(o)->data);
  }
  value *      CsVectorAddress(VM *c, value obj);
  inline value CsVectorElementI(value o, int_t i) {
    return CsVectorAddressI(o)[i];
  }
  inline void CsSetVectorElementI(value o, int_t i, value v) {
    CsVectorAddressI(o)[i] = value_to_set(v);
  }
  value CsMakeVector(VM *c, int_t size, value proto = UNDEFINED_VALUE);
  value CsCloneVector(VM *c, value obj, bool deep = false);
  int_t CsVectorSize(VM *c, value obj);
  int_t CsVectorSizeNoLoad(VM *c, value obj);
  value CsVectorElement(VM *c, value obj, int_t i);
  value CsVectorElementNoLoad(VM *c, value obj, int_t i);
  void  CsSetVectorElement(VM *c, value obj, int_t i, value val);
  void  CsSetVectorElementNoLoad(VM *c, value obj, int_t i, value val);

  bool CsVectorsEqual(VM *c, value v1, value v2, array<value> &stack);
  int  CsCompareVectors(VM *c, value v1, value v2, bool suppressError);

  value CsResizeVector(VM *c, value obj, int_t newSize);
  value CsResizeVectorNoLoad(VM *c, value obj, int_t newSize);

  inline value CsVectorClass(value o) { return ptr<vector>(o)->proto; }
  inline void  CsSetVectorClass(value o, value v) { ptr<vector>(o)->proto = v; }

  inline value CsVectorObserver(value o) { return ptr<vector>(o)->observer; }
  inline void  CsSetVectorObserver(value o, value v) {
    ptr<vector>(o)->observer = v;
  }

  //value CsUnspreadVector(VM *c, value vec);

  inline value CsVectorData(value o) {
    assert(!ptr<vector>(o)->data || CsFixedVectorP(ptr<vector>(o)->data));
    return ptr<vector>(o)->data;
  }
  inline void CsSetVectorData(value o, value v) {
    ptr<vector>(o)->data = v;
    assert(CsFixedVectorP(v));
  }

  void CsVectorPush(VM *c, value obj, value v);
  void CsVectorRemove(VM *c, value obj, int idx);
  void CsVectorRemoveI(value obj, int idx);
  value CsVectorInsertReplace(VM *c, value obj, int at, value replacementVector); // element 'at' removed and elements of replacementVector injected in its place

  inline slice<value> CsVectorElements(VM *c, value obj) {
    return slice<value>(CsVectorAddress(c, obj), CsVectorSize(c, obj));
  }
  inline tslice<value> CsVectorTargetElements(value obj) {
    return tslice<value>(CsVectorAddressI(obj), CsVectorSizeI(obj));
  }

  inline bool CsDerivedFromObjectP(value o);

  inline uint &CsEntityFlags(value o) {
#ifdef _DEBUG
    dispatch* pd = CsGetDispatch(o);
    pd = pd;
#endif 
    //assert(CsDerivedFromObjectP(o) || CsVectorP(o));
    return ptr<object>(o)->flags;
  }

  /* Tuple or BASIC VECTOR */

  struct tuple : public header // n-tuple in fact
  {
    value meta; // metadata
    value name; // message/tuple name (if the tuple is a message)
    int_t size;
    /*  value data[0]; */
  };

  extern dispatch CsTupleDispatch;
  inline bool     CsTupleP(value o) { return CsIsType(o, &CsTupleDispatch); }

  inline void     CsSetTupleName(value o, value nm) { ptr<tuple>(o)->name = nm; }
  inline value    CsTupleName(value o) { return ptr<tuple>(o)->name; }
  // aliases of the above
  inline void     CsSetTupleTag(value o, value nm) { CsSetTupleName(o,nm); }
  inline value    CsTupleTag(value o) { return CsTupleName(o); }


  inline int_t  CsTupleSize(value o) { return ptr<tuple>(o)->size; }
  inline void   CsSetTupleSize(value o, int_t s) { ptr<tuple>(o)->size = s; }
  inline value *CsTupleAddress(value o) {
    return (value *)(ptr<char>(o) + sizeof(tuple));
  }
  inline value CsTupleElement(value o, int_t i) {
    assert(i < CsTupleSize(o));
    return CsTupleAddress(o)[i];
  }
  inline void CsSetTupleElement(value o, int_t i, value v) {
    assert(i < CsTupleSize(o));
    CsTupleAddress(o)[i] = v;
  }

  inline value CsTupleElement(value o, int_t i, value def) {
    if(i < CsTupleSize(o))
      return CsTupleAddress(o)[i];
    return def;
  }

  inline slice<value> CsTupleElements(value o) {
    return slice<value>(CsTupleAddress(o), CsTupleSize(o));
  }
  inline tslice<value> CsTupleTargetElements(value obj) {
    return tslice<value>(CsTupleAddress(obj), CsTupleSize(obj));
  }

  value CsMakeTuple(VM *c, dispatch *type, int_t size);
  value CsMakeTuple(VM *c, int_t size);
  bool  CsTuplesEqual(VM* c, value o1, value o2, array<value> &stack);
  value CsMakeTuple(VM *c, const char *tag, value v1);
  value CsMakeTuple(VM *c, const char *tag, value v1, value v2);
  value CsMakeTuple(VM *c, const char *tag, value v1, value v2, value v3);
  value CsMakeTuple(VM *c, const char *tag, value v1, value v2, value v3,
                    value v4);
  value CsMakeTuple(VM *c, const char *tag, value v1, value v2, value v3,
                    value v4, value v5);
  value CsMakeTuple(VM *c, const char *tag, value v1, value v2, value v3,
                    value v4, value v5, value v6);

  
  /* CMETHOD */

  extern dispatch CsCMethodDispatch;

  struct c_method : public header {
    const char *name;
    c_method_t  handler;
    void *      tag;
    value       payload;
    c_method() : name(0), handler(0), tag(0), payload(0) {
      pdispatch = &CsCMethodDispatch;
    }
    c_method(const char *n, c_method_t h, dispatch* pd = &CsCMethodDispatch)
        : name(n), handler(h), tag(0), payload(0) {
      pdispatch = pd;
    }
    c_method(const char *n, c_method_ext_t h, void *t)
        : name(n), handler((c_method_t)h), tag(t), payload(0) {
      pdispatch = &CsCMethodDispatch;
    }
    
    c_method(dispatch* pd, const char *n, c_method_t h)
      : name(n), handler(h), tag(0), payload(0) {
      pdispatch = pd;
    }

    inline value call(VM *c, value self) {
      if (tag) return (*((c_method_ext_t)handler))(c, self, tag);
      return (*handler)(c);
    }
  };

  inline bool CsCMethodP(value o) { return CsIsBaseType(o, &CsCMethodDispatch); }
  inline const char *CsCMethodName(value o) { return ptr<c_method>(o)->name; }
  inline c_method_t  CsCMethodValue(value o) {
    return ptr<c_method>(o)->handler;
  }
  inline c_method *CsCMethodPtr(value o) { return ptr<c_method>(o); }
  inline value CsCMethodPayload(value o) { return ptr<c_method>(o)->payload; }
  inline void  CsSetCMethodPayload(value o, value v) {
    ptr<c_method>(o)->payload = v;
  }

  value CsMakeCMethod(VM *c, const char *name, c_method_t handler);
  void  CsInitCMethod(c_method *ptr);

  void check_thrown_error(VM *c);

  /* VIRTUAL PROPERTY METHOD */

  extern dispatch CsVPMethodDispatch;

  struct vp_method : public header {
    const char *name;
    vp_get_t    get_handler;
    vp_set_t    set_handler;
    void *      tag;
    vp_method() : name(0), get_handler(0), set_handler(0), tag(0) {}
    vp_method(const char *n, vp_get_t gh, vp_set_t sh)
        : name(n), tag(0), get_handler(gh), set_handler(sh) {
      pdispatch = &CsVPMethodDispatch;
    }

    vp_method(const char *n, vp_get_ext_t gh, vp_set_ext_t sh, void *t)
        : name(n), tag(t), get_handler((vp_get_t)gh),
          set_handler((vp_set_t)sh) {
      pdispatch = &CsVPMethodDispatch;
    }

    inline bool get(VM *c, value obj, value &val) {
      if (!get_handler) return false;
      if (tag)
        val = (*((vp_get_ext_t)get_handler))(c, obj, tag);
      else
        val = (*get_handler)(c, obj);
      check_thrown_error(c);
      return true;
    }
    bool set(VM *c, value tag, value obj, value val);
  };

  inline bool CsVPMethodP(value o) { return CsIsType(o, &CsVPMethodDispatch); }
  inline const char *CsVPMethodName(value o) { return ptr<vp_method>(o)->name; }

  /*struct c_functor: public header
  {
      tool::handle<tool::native_functor_holder> handler;
      c_functor() {}
      value call(VM *c, value self);
  };*/

  extern dispatch CsCFunctorDispatch;

  inline bool CsCFunctorP(value o) { return CsIsType(o, &CsCFunctorDispatch); }
  value       CsMakeCFunctor(VM *c, native_functor_holder *handler);
  native_functor_holder *CsCFunctorValue(value val);
  value                  CsCFunctorCall(VM *c, value func);

  /* CONSTANTS */

  struct constant //: public header
  {
    const char *name;
    value       val;
    constant() : name(0), val(0) {}
    constant(const char *n, value v)
        : name(n), val(v) {} //   { pdispatch = &CsConstantDispatch; }
  };

  /* HASH TABLE */
  extern dispatch CsHashTableDispatch;

  inline bool CsHashTableP(value o) {
    return CsIsType(o, &CsHashTableDispatch);
  }
  inline int_t CsHashTableSize(value o) { return CsFixedVectorSize(o); }
  inline value CsHashTableElement(value o, int_t i) {
    return CsFixedVectorElement(o, i);
  }
  inline void CsSetHashTableElement(value o, int_t i, value v) {
    CsSetFixedVectorElement(o, i, v);
  }
  value CsMakeHashTable(VM *c, long size);

  /* PROPERTY */

  extern dispatch CsPropertyDispatch;

#pragma warning(push)
#pragma warning(disable : 4480) // warning C4480: nonstandard extension used:
                                // specifying underlying type for enum

  enum PROP_FLAGS : uint {
    PROP_CONST        = 0x80000000,
    PROP_ORDINAL_MASK = 0x0FFFFFFF,
    //PROP_EXPORT       = 0x10000000,
    //PROP_DEFAULT      = 0x20000000,
    //PROP_EXPORT_DEFAULT = 0x30000000,
  };

#pragma warning(pop)

  inline bool  CsPropertyP(value o) { return CsIsType(o, &CsPropertyDispatch); }
  inline value CsPropertyTag(value o) { return CsFixedVectorElement(o, 0); }
  inline value CsPropertyValue(value o) { return CsFixedVectorElement(o, 1); }

  inline void CsSetPropertyValue(value o, value v) {
    CsSetFixedVectorElement(o, 1, value_to_set(v));
  }

  inline value CsPropertyNext(value o) { return CsFixedVectorElement(o, 2); }
  inline void  CsSetPropertyNext(value o, value v) {
    CsSetFixedVectorElement(o, 2, v);
  }

  inline int_t CsPropertyFlags(value o) {
    return to_int(CsFixedVectorElement(o, 3));
  }
  inline void CsSetPropertyFlags(value o, uint v) {
    CsSetFixedVectorElement(o, 3, int_value(v));
  }

  inline bool CsPropertyIsConst(value o) {
    return (CsPropertyFlags(o) & PROP_CONST) == PROP_CONST;
  }
  inline uint CsPropertyOrdinal(value o) {
    return (CsPropertyFlags(o) & PROP_ORDINAL_MASK);
  }
  inline void CsSetPropertyOrdinal(value o, uint ord) {
    return CsSetPropertyFlags(o,
                              CsPropertyFlags(o) | (ord & PROP_ORDINAL_MASK));
  }

#define CsPropertySize 4

  value CsMakeProperty(VM *c, value &key, value &val, uint flags);

  /* METHOD */

  struct method : public object /* method is an Object! */
  {
    value code;
    value env;
    value ns;        // namespace
    value nlist;     // notification list
    value prototype; // JS style of object creation 
  };

  extern dispatch CsMethodDispatch;
  extern dispatch CsPropertyMethodDispatch;

  inline bool  CsMethodP(value o) { return CsIsBaseType(o, &CsMethodDispatch); }
  inline value CsMethodCode(value o) { return ptr<method>(o)->code; }
  inline value CsMethodEnv(value o) { return ptr<method>(o)->env; }
  //inline value CsMethodGlobals(value o) { return ptr<method>(o)->globals; }
  inline value CsMethodNamespace(value o) { return ptr<method>(o)->ns; }
  inline void  CsSetMethodCode(value o, value v) { ptr<method>(o)->code = (v); }
  inline void  CsSetMethodEnv(value o, value v) { ptr<method>(o)->env = (v); }
  inline void  CsSetMethodPrototype(value o, value v) { ptr<method>(o)->prototype = v; }
  inline value CsMethodPrototype(value o) { return ptr<method>(o)->prototype; }

  // creates method prototype if does not exist
  value CsGetMethodPrototype(VM* c, value o);

  inline void CsSetMethodNamespace(value o, value v) {
    ptr<method>(o)->ns = (v);
  }
  value  CsMethodShortName(value o); // returns function name (as it declared)
  size_t CsMethodArgCnt(value o);
  size_t CsMethodOptionalArgCnt(value o);
  value  CsMethodName(value o); // returns function name (as it declared)

  inline bool CsMethodEventP(value o) {
    return CsMethodP(o) && CsTupleP(CsMethodName(o));
  }

  inline value CsMethodNotificationList(value o) {
    return ptr<method>(o)->nlist;
  }
  inline void CsSetMethodNotificationList(value o, value v) {
    ptr<method>(o)->nlist = (v);
  }

  inline bool CsPropertyMethodP(value o) {
    return CsIsBaseType(o, &CsPropertyMethodDispatch);
  }

  inline bool CsMethodsOfSameOrigin(value m1, value m2) {
    if (m1 == m2) return true;
    return CsMethodP(m1) && CsMethodP(m2) &&
           CsMethodCode(m1) == CsMethodCode(m2);
  }

  inline bool CsObjectOrMethodP(value o) {
    return CsIsBaseType(o, &CsObjectDispatch, &CsMethodDispatch, &CsCObjectDispatch);
  }
  inline bool CsDerivedFromObjectP(value o) {
    return CsIsBaseType(o, &CsObjectDispatch, &CsMethodDispatch, &CsCObjectDispatch);
  }

  // true if it is persitent_header derived
  inline bool CsEntityP(value o) { return CsIsBaseType(o, &CsObjectDispatch, &CsMethodDispatch, &CsVectorDispatch) || CsCObjectP(o); }

  value CsMakeMethod(VM *c, value code, value env, value ns);
  value CsMakePropertyMethod(VM *c, value code, value env, value ns);

  /* COMPILED CODE */

  enum CODE_TYPE_FLAGS {
    CODE_TYPE_GENERATOR = 1 << 0,
    CODE_TYPE_TASK      = 1 << 1
  };

  extern dispatch CsCompiledCodeDispatch;

  inline bool CsCompiledCodeP(value o) {
    return CsIsType(o, &CsCompiledCodeDispatch);
  }
  inline value *CsCompiledCodeLiterals(value o) { return CsTupleAddress(o); }
  inline value  CsCompiledCodeLiteral(value o, int_t i) { return CsTupleElement(o, i); }
  inline value CsCompiledCodeBytecodes(value o) { return CsTupleElement(o, 0); }
  inline value CsCompiledCodeLineNumbers(value o) { return CsTupleElement(o, 1); }
  inline value CsCompiledCodeFileName(value o) { return CsTupleElement(o, 2); }
  inline value CsCompiledCodeArgNames(value o) { return CsTupleElement(o, 3); }
  inline value CsCompiledCodeName(value o) { return CsTupleElement(o, 4); }
  inline uint  CsCompiledCodeFlags(value o) { value v = CsTupleElement(o, 5); return CsIntegerP(v) ? CsIntegerValue(v) : 0; }

  inline void CsSetCompiledCodeBytecodes(value o, value v) {  CsSetTupleElement(o, 0, v); }
  inline void CsSetCompiledCodeLineNumbers(value o, value v) {  CsSetTupleElement(o, 1, v); }
  inline void CsSetCompiledCodeFileName(value o, value fn) { CsSetTupleElement(o, 2, fn);  }
  inline void CsSetCompiledCodeArgNames(value o, value nv) { CsSetTupleElement(o, 3, nv); }
  inline void CsSetCompiledCodeName(value o, value nv) { CsSetTupleElement(o, 4, nv); }
  inline void CsSetCompiledCodeTypeFlags(value o, uint fl) { CsSetTupleElement(o, 5, CsMakeInteger(fl)); }

#define CsFirstLiteral 6

  value CsMakeCompiledCode(VM *c, long size, value bytecodes, value linenumbers, value argnames, value fileNameSymbol);

  inline bool CsCompiledCodeIsTask(value o) {
    return (CsCompiledCodeFlags(o) & CODE_TYPE_TASK) == CODE_TYPE_TASK;
  }
  inline bool CsCompiledCodeIsGenerator(value o) {
    return (CsCompiledCodeFlags(o) & CODE_TYPE_GENERATOR) ==
           CODE_TYPE_GENERATOR;
  }
  inline bool CsCompiledCodeIsCoroutine(value o) {
    return (CsCompiledCodeFlags(o) & (CODE_TYPE_GENERATOR | CODE_TYPE_TASK)) !=
           0;
  }

  /* SSX */

  // canonical VNode element
  inline bool     CsVNodeP(value o) {
    //return CsIsType(o, &CsTupleDispatch); 
    if (!CsTupleP(o)) return false;
    value tag = CsTupleName(o);
    if (!CsSymbolP(tag) && !CsMethodP(tag) && !CsClassP(tag)) return false;
    if (CsTupleSize(o) != 2 && CsTupleSize(o) != 3) return false;
    value v0 = CsTupleElement(o, 0);
    if (v0 != NULL_VALUE && !CsObjectP(v0)) return false;
    value v1 = CsTupleElement(o, 1);
    if (v1 != NULL_VALUE && !CsVectorP(v1)) return false;
    return true;
  }

  inline value CsVNodeTag(value o) { return CsTupleName(o); }
  inline value CsVNodeAtts(value o) { return CsTupleElement(o, 0, NULL_VALUE); }
  inline value CsVNodeKids(value o) { return CsTupleElement(o, 1, NULL_VALUE); }
  inline void  CsSetVNodeKids(value o,value kids) { CsSetTupleElement(o, 1, kids); }
  inline value CsVNodeStates(value o) { return CsTupleElement(o, 2, NULL_VALUE); }
  // returns true if vnode is passive - its tag is symbol and not function or class
  inline bool CsVNodeIsPassive(value o) { return CsSymbolP(CsVNodeTag(o)); }

  inline bool CsVNodeFunctionP(value o) {
    if (!CsVNodeP(o)) return false;
    return CsMethodP(CsVNodeTag(o));
  }

  value CsMakeVNode(VM *c, value tag, value atts = NULL_VALUE, value kids = NULL_VALUE, value states = NULL_VALUE);

  /* TASK, await thing */

  void    CsInitCoroutine(VM *c);
  CsTask *CsCoroutineValue(VM *c, value obj);
  bool    CsCoroutineP(VM *c, value obj);
  bool    CsTaskP(VM *c, value obj);
  bool    CsGeneratorP(VM *c, value obj);
  value   CsMakeCoroutine(VM *c, CsTask *prg);
  void    CsPersistCurrentCoroutine(VM *c);
  void    CsTaskNotifyCompletion(VM *c, bool reject, value list);
  void    CsExecTask(VM *c, value taskInstance,
                     bool error /*true if to throw an error in val[0]*/);
  value   CsExecGenerator(VM *c, value genInstance);

  /* ENVIRONMENT */

#define CsEnvironmentP(o) CsIsBaseType(o, &CsEnvironmentDispatch)
#define CsEnvSize(o) CsTupleSize(o)
#define CsSetEnvSize(o, v) CsSetTupleSize(o, v)
#define CsEnvAddress(o) CsTupleAddress(o)
#define CsEnvElement(o, i) CsTupleElement(o, i)
#define CsSetEnvElement(o, i, v) CsSetTupleElement(o, i, v)
#define CsEnvElementRef(o,i) (CsTupleAddress(o) + i)
#define CsEnvNextFrame(o) CsTupleElement(o, 0)
#define CsSetEnvNextFrame(o, v) CsSetTupleElement(o, 0, v)
#define CsEnvNames(o) CsTupleElement(o, 1)
#define CsSetEnvNames(o, v) CsSetTupleElement(o, 1, v)
#define CsFirstEnvElement 2

  typedef tuple   CsEnvironment;
  value           CsMakeEnvironment(VM *c, long size);
  extern dispatch CsEnvironmentDispatch;

  /* STACK ENVIRONMENT */
  extern dispatch CsStackEnvironmentDispatch;
  inline bool     CsStackEnvironmentP(value o) {
    return CsIsType(o, &CsStackEnvironmentDispatch);
  }

    /* MOVED ENVIRONMENT */

#define CsMovedEnvironmentP(o) CsIsType(o, &CsMovedEnvironmentDispatch)
#define CsMovedEnvForwardingAddr(o) CsEnvNextFrame(o)
#define CsSetMovedEnvForwardingAddr(o, v) CsSetEnvNextFrame(o, v)

  extern dispatch CsMovedEnvironmentDispatch;

  /* FILE */

  //#define CsFileP(o)                     CsIsType(o,CsFileDispatch)
  bool           CsFileP(VM *c, value o);
  inline stream *CsFileStream(value o) { return (stream *)CsCObjectValue(o); }
  inline void CsFileSetStream(value o, stream *v) { CsSetCObjectValue(o, v); }

  // extern  dispatch *CsFileDispatch;

  /* TYPE */

  inline dispatch *CsTypeDispatch(value o) {
    return (dispatch *)CsCObjectValue(o);
  }
  inline void CsTypeSetDispatch(value o, dispatch *v) {
    CsSetCObjectValue(o, v);
  }

  bool CsGetConstantValue(VM *c, const char *tname, const char *cname,
                          value *pValue);

  /* default handlers */
  bool  CsDefaultPrint(VM *c, value obj, stream *s, bool toLocale);
  value CSF_std_toLocaleString(VM *c);
  value CSF_std_toString(VM *c);
  value CSF_std_valueOf(VM *c);
  // value CSF_call(VM *c);
  // value CSF_apply(VM *c);

  /* error codes */
  enum CsErrorCodes {
    CsErrExit,
    CsErrInsufficientMemory,
    CsErrStackOverflow,
    CsErrTooManyArguments,
    CsErrTooFewArguments,
    CsErrTypeError,
    CsErrUnexpectedTypeError,
    CsErrUnboundVariable,
    CsErrIndexOutOfBounds,
    CsErrNoMethod,
    CsErrNoProperty,
    CsErrBadOpcode,
    CsErrRestart,
    CsErrWrite,
    CsErrBadParseCode,
    CsErrImpossible,
    CsErrNoHashValue,
    CsErrReadOnlyProperty,
    CsErrWriteOnlyProperty,
    CsErrFileNotFound,
    CsErrNewInstance,
    CsErrStackEmpty,
    CsErrNotAnObjectFile,
    CsErrWrongObjectVersion,
    CsErrValueError,
    CsErrRegexpError,
    CsErrIOError,
    CsErrIOTimeout,
    CsErrNoSuchFeature,
    CsErrNotAllowed,
    CsErrAlreadyDefined,
    CsErrGenericError,
    CsErrGenericErrorW,
    CsErrPersistError,
    CsErrArguments,
    CsErrAssertion,
    CsErrAssertion2,
    CsErrIsSealed,
    CsErrIsFrozen,
    CsErrUnsupportedBinaryOp,
    CsErrHTTPError,

    /* compiler error codes */
    CsErrSyntaxError       = 0x1000,
    CsErrStoreIntoConstant = 0x1001,
    CsErrTooMuchCode       = 0x1002,
    CsErrTooManyLiterals   = 0x1003,
    CsErrIsNotLValue       = 0x1004,

    /* throw statement  */
    CsErrThrown = 0x2000,

  };

  /* cs_com.c prototypes */
  void        CsInitScanner(CsCompiler *c, stream *s);
  CsCompiler *CsMakeCompiler(VM *ic, long csize, long lsize);
  void        CsFreeCompiler(CsCompiler *c);
  value       CsCompileExpr(CsScope *scope, bool add_this,tool::slice<tool::ustring> argnames = tool::slice<tool::ustring>());
  /* compile sequence of expressions */
  value CsCompileExpressions(CsScope *scope, bool serverScript,int line_no = 0);

  /* compile data expression, JSON style of data declaration.
     Example: { one:1, two:2 }
     is a valid data declaration
  */
  value CsCompileDataExpr(CsScope *scope);

  /* cs_int.c prototypes */

  // value CsCallFunctionByName(CsScope *scope,char *fname,int argc,...);

  struct vargs {
    virtual int   count()    = 0;
    virtual value nth(int n) = 0;
    virtual ~vargs() {}
  };

  value CsCallFunction(CsScope *scope, value fun, int argc, ...);
  value CsCallFunction(CsScope *scope, value fun, vargs &args);
  value CsCallMethod(VM *c, value obj, value method, value ofClass, int argc, ...);
  // bool  Execute(VM *c, value gen = 0);

  value CsSendMessage(CsScope *scope, value obj, value selectorOrMethod, int argc, ...);
  value CsSendMessage(VM *c, value obj, value selectorOrMethod, int argc, ...);
  value CsSendMessage(VM *c, value obj, value selectorOrMethod,  const value *argv = 0, int argc = 0);
  value CsSendMessageByName(VM *c, value obj, char *sname, int argc, ...);

  value CSF_eval(VM *c);

  value CsInternalSend(VM *c, int argc);
  value CsCallCompletion(VM *c, value fun, value param);
  // void CsPushUnwindTarget(VM *c,CsUnwindTarget *target);
  // void CsPopAndUnwind(VM *c,int value);
  // void CsAbort(VM *c);
  void CsThrowExit(VM *c, value v);
  bool CsKeysAreEqual(VM *c, value obj1, value obj2);
  bool CsStrongEql(value obj1, value obj2);
  void CsCopyStack(VM *c);
  // void CsCopyRegisters(VM *c, VM_REGISTERS* vmr);
  void CsStackTrace(VM *c);
  // streams stack trace into s
  void CsStreamStackTrace(VM *c, stream *s);
  // make stack trace as a vector of triplets:
  //  [0] - lineNo
  //  [1] - functionName
  //  [2] - fileName
  value       CsMakeStackTrace(VM *c);
  tool::value CsMakeStackTraceV(VM *c);
  tool::value CsLocalVarsV(VM *c);
  void        CsDumpSymbolTable(VM *c);

  /* cs_enter.c prototypes */
  void  CsEnterVariable(CsScope *scope, const char *name, value value);
  void  CsEnterFunction(CsScope *scope, c_method *function);
  void  CsEnterFunctions(CsScope *scope, c_method *functions);
  value CsEnterObject(CsScope *scope, const char *name, value proto,
                      c_method *methods, vp_method *properties);

  dispatch *CsEnterCObjectType(CsScope *scop, const char *typeName, c_method *methods,
                               vp_method *properties, constant *constants,
                               long size);
  dispatch *CsEnterCPtrObjectType(CsScope *scop, const char *typeName, c_method *methods,
                                  vp_method *properties,
                                  constant * constants = nullptr);



  dispatch *CsEnterTupleType(CsScope *scop, dispatch *proto,
                             const char *typeName, c_method *methods,
                             vp_method *properties, int size);

  void CsEnterMethods(VM *c, value &obj, c_method *methods);
  void CsEnterMethod(VM *c, value &obj, c_method *method);
  void CsEnterVPMethods(VM *c, value &obj, vp_method *methods);
  void CsEnterVPMethod(VM *c, value &obj, vp_method *method);
  void CsEnterProperty(VM *c, value &obj, const char *selector, value value);
  void CsEnterConstants(VM *c, value &obj, constant *constants);

  /* cs_parse.c prototypes */
  int CsParseArguments(VM *c, const char *fmt, ...);

  /* cs_heap.c prototypes */
  CsScope *CsMakeScope(VM *c, CsScope *proto);
  CsScope *CsMakeScopeFromObject(VM *c, value gobject);
  void     CsFreeScope(CsScope *scope);
  void     CsInitScope(CsScope *scope);

  struct CsHeapStats {
    uint total;
    uint free;
    uint used;
  };

  CsHeapStats CsCollectGarbage(VM *c); // gc()
  CsHeapStats CsCollectGarbageAndReclaim(VM *c, int factor); // gc() and free not needed heap
                                                             // if factor == 1 then free == used
                                                             // if factor == 2 then free == used * 2
                                                             // if factor == 0 then free == 0

  uint      CsFreeSpace(VM *c);
  void      CsDumpHeap(VM *c);
  void      CsDumpScopes(VM *c);
  void      CsDumpObject(VM *c, value obj);
  dispatch *CsMakeDispatch(VM *c, const char *typeName, dispatch *prototype);
  void      CsFreeDispatch(VM *c, dispatch *d);
  // bool CsProtectPointer(VM *c,value *pp);
  // bool CsUnprotectPointer(VM *c,value *pp);
  value        CsAllocate(VM *c, size_t size);
  //tslice<byte> CsAllocate(VM *c, size_t size, value &obj);
  void *       CsAlloc(VM *c, unsigned long size);
  void *       CsMalloc(VM *c, size_t size);
  void         CsFree(VM *c, void *ptr);
  void         CsInsufficientMemory(VM *c);

  /* default type handlers */
  bool  CsDefaultGetProperty(VM *c, value &obj, value tag, value *pValue);
  bool  CsDefaultSetProperty(VM *c, value obj, value tag, value value);
  value CsDefaultNewInstance(VM *c, value proto);
  value CsDefaultCopy(VM *c, value obj);
  void  CsDefaultScan(VM *c, value obj);
  int_t CsDefaultHash(value obj);
  value CsDefaultObjectBinOp(VM *c, int op, value obj, value operand);

  /* default index [tag] = value getter and setter */
  value CsDefaultGetItem(VM *c, value obj, value tag);
  void  CsDefaultSetItem(VM *c, value obj, value tag, value value);

  value CsObjectGetItem(VM *c, value obj, value tag);
  void  CsObjectSetItem(VM *c, value obj, value tag, value value);
  value CsObjectNextElement(VM *c, value *index, value obj, int nr);

  /* cs_hash.c prototypes */
  int_t CsHashString(wchars);
  int_t CsHashBytes(bytes);

  /* cs_type.c prototypes */
  void  CsInitTypes(VM *c);
  void  CsAddTypeSymbols(VM *c);
  value CsEnterType(CsScope *scope, const char *name, dispatch *d);

  /* cs_stream.c prototypes */
  int CsPrint(VM *c, value val, stream *s);
  int CsDisplay(VM *c, value val, stream *s);

  stream *CsMakeIndirectStream(VM *c, stream **pStream);
  stream *CsMakeFileStream(VM *c, FILE *fp);
  stream *OpenFileStream(VM *c, const wchar *fname, const wchar *mode);
  /*stream *OpenSocketStream(VM *c, const wchar *domainAndPort,
                           int timeoutSeconds, int maxattempts, bool binstream);*/

  /* cs_stdio.c prototypes */
  void CsUseStandardIO(VM *c);

  /* cs_math.c prototypes */
  void CsUseMath(VM *c);

  /* cs_obj.c prototypes */
  void CsInitObject(VM *c);
  void CsAddProperty(VM *c, value obj, value tag, value value, int_t hashValue,
                     int_t i, uint flags = 0);

  /* cs_method.c prototypes */
  void CsInitMethod(VM *c);

  /* cs_symbol.c prototypes */
  void CsInitSymbol(VM *c);

  /* cs_vector.c prototypes */
  void CsInitVector(VM *c);

  void CsInitTuple(VM *c);
  long CsTupleSizeHandler(value obj);
  void CsTupleScanHandler(VM *c, value obj);

  /* cs_string.c prototypes */
  void CsInitString(VM *c);

  /* cs_cobject.c prototypes */
  void CsDestroyUnreachableCObjects(VM *c);
  void CsDestroyAllCObjects(VM *c);

  /* cs_bytevector.c prototypes */
  void CsDestroyUnreachableByteVectors(VM *c);
  void CsDestroyAllByteVectors(VM *c);

  /* cs_integer.c prototypes */
  void CsInitInteger(VM *c);

  /* cs_float.c prototypes */
  void CsInitFloat(VM *c);

  /* cs_color.c prototypes */
  void CsInitColor(VM *c);

  /* cs_length.c prototypes */
  void CsInitLength(VM *c);
  void CsInitAngle(VM *c);

  /* cs_bytevector prototypes */
  void CsInitByteVector(VM *c);

  /* cs_file.c prototypes */
  void  CsInitFile(VM *c);
  value CsMakeFile(VM *c, stream *s);
  /* prints data suitable for parsing */
  bool CsPrintData(VM *c, value v, stream *s, bool verbose);
  /* prints data as JSON source */
  bool CsPrintJsonData(VM *c, value val, stream *s, bool verbose);
  /* parse utf-8 encoded JSON */
  value CsParseJson(VM *c, bytes data);

  /* cs_datetime.cpp prototypes */
  void        CsInitDate(VM *c);
  bool        CsDateP(VM *c, value obj);
  bool        CsPrintDate(VM *c, value v, stream *s);
  datetime_t &CsDateValue(VM *c, value obj);
  value       CsMakeDate(VM *c, datetime_t dta);

  /* cs_xml_scanner.cpp prototypes */
  void CsInitXmlScanner(VM *c);

  /* cs_fcn.c prototypes */
  void CsEnterLibrarySymbols(VM *c);

  /* cs_eval.c prototypes */
  void  CsUseEval(VM *c);
  void  CsUnuseEval(VM *c);
  value CsEvalString(CsScope *scope, value self, const wchar *str,
                     size_t length);
  value CsEvalStream(CsScope *scope, value self, stream *s, int lineNo = 0);

  /* evaluate (parse) JSON++ literal in the string and stream */
  value CsEvalDataString(CsScope *scope, const wchar *str, size_t length);
  value CsEvalDataStream(CsScope *scope, stream *s);

  // these two return result of last return statement seen
  value CsLoadFile(CsScope *scope, const wchar *fname, stream *os);
  value CsLoadStream(CsScope *scope, stream *is, stream *os = 0,
                     int line_no = 0);

  value CsInclude(CsScope *scope, value what, bool no_throw = false);
  value CsImport(VM *c, value what);

  value CsIncludeLibrary(CsScope *scope, const tool::ustring &name);

  bool CsLoadExtLibrary(CsScope *scope, tool::ustring fullpath, tool::value& val);

  /* cs_wcode.c prototypes */
  bool CsCompileFile(CsScope *scope, const wchar *iname, const wchar *oname,
                     bool serverScript);
  // int  CsCompileString(CsScope *scope,char *str,stream *os);
  bool CsCompileStream(CsScope *scope, stream *is, stream *os,
                       bool serverScript);

  bool CsCompile(VM *c, stream *is, stream *os, bool serverScript);

  /* cs_rcode.c prototypes */
  bool  CsLoadObjectFile(CsScope *scope, const wchar *fname);
  bool  CsLoadObjectStream(CsScope *scope, stream *s);
  value CsEvalObjectStream(CsScope *scope, value self, stream *s);

  bool CsReadBytecodePreamble(VM *c, stream *s,
                              bool riseError); // true - it a BC file of proper
                                               // version, false - too bad.

  /* cs_debug.c prototypes */
  void CsDecodeProcedure(VM *c, value method, stream *stream);
  int  CsDecodeInstruction(VM *c, value code, int lc, stream *stream);

  /* cs_error.c prototypes */

  extern dispatch CsErrorDispatch;

  inline bool CsErrorP(value o) { return CsIsType(o, &CsErrorDispatch); }
#define CsErrorName(o) CsFixedVectorElement(o, 0)
#define CsSetErrorName(o, v) CsSetFixedVectorElement(o, 0, v)
#define CsErrorMessage(o) CsFixedVectorElement(o, 1)
#define CsSetErrorMessage(o, v) CsSetFixedVectorElement(o, 1, v)
#define CsErrorStackTrace(o) CsFixedVectorElement(o, 2)
#define CsSetErrorStackTrace(o, v) CsSetFixedVectorElement(o, 2, v)
#define CsErrorNo(o) CsFixedVectorElement(o, 3)
#define CsSetErrorNo(o, v) CsSetFixedVectorElement(o, 3, v)
#define CsErrorData(o) CsFixedVectorElement(o, 4)
#define CsSetErrorData(o, v) CsSetFixedVectorElement(o, 4, v)

  void CsThrowKnownError(VM *c, int code, ...);
  void CsThrowError(VM *c, const char *msg, ...);
  // void CsShowError(VM *c,int code,va_list ap);
  const char *CsGetErrorText(int code);
  void        CsInitErrorType(VM *c);
  /* construct error obj, automaticly builds stack trace */
  value CsError(VM *c, int n, value message, value data = UNDEFINED_VALUE);

  void CsWarning(VM *c, const char *msg);

  void CsHandleUnhandledError(VM *c);

  /* cs_instanceof.c prototypes */
  bool  CsInstanceOf(VM *c, value obj, value cls);
  value CsTypeOf(VM *c, value val);
  bool  CsHasMember(VM *c, value obj, value tag);
  /* a == b */
  bool CsEqualOp(VM *c, value obj1, value obj2);
  bool CsEqualOp(VM *c, value obj1, value obj2, array<value> &stack);

  bool CsIsLike(VM *c, value what, value mask);

  /* returns [-1,0,1] */
  int CsCompareObjects(VM *c, value obj1, value obj2,
                       bool suppressError = false);

  bool CsMatch(VM *c, value tmpl, value obj);

  /* cvt everything into String */
  value CsToString(VM *c, value val);
  void  CsToString(VM *c, value val, stream &s);
  void  CsToHtmlString(VM *c, value val, stream &s);
  void  CsToCssString(VM *c, value val, stream &s);
  void  CsToUrlString(VM *c, value val, stream &s);
  value CsToInteger(VM *c, value val, bool throw_error = true);
  value CsToFloat(VM *c, value val, bool throw_error = true);

  bool CsRegExpP(VM *c, value obj);
  void CsInitRegExp(VM *c);

  typedef tool::lookup_tbl<ustring, false, uint, uint32> symtab;
  symtab &                                               symbol_table();

  value         string_to_value(VM *c, const tool::ustring &s);
  tool::ustring value_to_string(value v);
  tool::wchars  value_to_wchars(value v);
  tool::ustring value_to_string(VM* c, value v);

  value value_to_value(VM *c, const tool::value &v, bool onreturn = false);
  value dictionary_to_value(VM *c, const tool::dictionary<tool::value, tool::value> &params);
  tool::value value_to_value(VM *c, value v, bool isolate = false);
  tool::value value_to_value_def(VM *c, value v); // all objects get transformed
                                                  // to object definitions [proxy: typename, class, {props}]

  bool isolate_value(tool::value &v); // removes all object proxies - converst
                                      // all managed objects to tool::value
                                      // counterparts

  bool CsStoreValue(VM *c, value v, stream *s);
  bool CsFetchValue(VM *c, value *pv, stream *s);

  // CsScope(VM* vm):c(0),globals(0),ns(0),next(0) {}

  struct auto_scope : CsScope {
    auto_scope(VM *pvm, value gobject, bool silent_unknowns = false) {
      c = pvm;
      this->previous_ns     = c->currentNS;
      this->globals         = gobject;
      this->silent_unknowns = silent_unknowns;
      c->currentNS          = gobject;
      c->scopes.push(this);
    }
    virtual ~auto_scope() {
      assert(c->currentScope() == this);
      c->scopes.pop();
      c->currentNS = this->previous_ns; // restore namespace
    }
  };

  struct CsDebugScope : auto_scope
  {
    CsDebugScope(VM *pvm, value gobject, bool silent_unknowns = false) : auto_scope(pvm, gobject, silent_unknowns) {}
    virtual bool setValue(value tag, value val) override;
    virtual bool getValue(value tag, value *pval) override;
  };

  /* CsInitSystem - initialize the 'System' obj */
  void CsInitSystem(VM *c);
  /* CsInitVM - initialize the 'VM' obj */
  void CsInitVM(VM *c);
  void CsInitNet(VM *c);
  
#if defined(TISCRIPT_USE_PERSISTENCE)

  /* cs_storage.cpp */
  typedef unsigned                       oid_t;
  typedef void *                         storage_t;
  typedef void *                         iterator_t;
  typedef tool::hash_table<oid_t, value> storageHash;

  class storage // to enable it as a member of l2 list use following: :public
                // tool::l2elem<storage>
  {
  public:
    storage_t                             dbS;
    storageHash                           hashS;
    tool::hash_table<tool::string, value> hashNameProto;
    bool                                  autocommit;

  public:
    storage() : dbS(0), hashS(1024), hashNameProto(16), autocommit(true) {}
    ~storage();

  public:
    /* remove obj from hash */
    void DetachObj(oid_t oid);
    void DetachAllObjs(VM *c = 0);

    void CommitHash(VM *c);

    bool IsInHash(oid_t oid) {
      return (!this->hashS.is_empty() && this->hashS.exists(oid));
    }
    bool IsInHash(oid_t oid, value &obj) {
      return (!this->hashS.is_empty() && this->hashS.find(oid, obj));
    }

    value GetFromHash(oid_t oid);
    void  InsertInHash(oid_t oid, value obj);

    bool IsHashEmpty() { return this->hashS.is_empty(); }

    /*returns proto value OR 0 */
    value GetProtoByName(VM *c, const tool::string &className) {
      value v = 0;
      if (this->hashNameProto.find(className, v)) return v;

      v = CsGetGlobalValueByPath(c, className);
      if (v && v != UNDEFINED_VALUE)
        this->hashNameProto[className] = v;
      else
        c->standardError->printf(
            W("class %S not found while loading object from Storage\n"),
            className.c_str());

      return v;
      // if( this->hashNameProto.exists(className) )
      //  return this->hashNameProto[className];
      // else
      //  return 0;
    }

    tool::string GetNameByProto(VM *c, value proto);

  private:
    void resetPersistHdr(value &obj);
  };

  union dbvtag {
    byte    b;
    int_t   i;
    float_t d;
    byte *  s; // string
    uint64  i64;
    oid_t   oid;
  };
  typedef dbvtag dbvtype;

  struct db_triplet {
    dbvtype data;
    int_t   type;
    int_t   len;
    // init as NULL string
    db_triplet();
    ~db_triplet();
    db_triplet &operator=(const db_triplet &obj);
    bool        is_null() const;
  };

  enum EBlobType {
    db_blob  = (byte)0,
    db_char  = (byte)1,
    db_wchar = (byte)2,
    // db_bytes = (byte)3 // tagged blob
  };

  void CsInitStorage(VM *c);

  /* restore persistent obj from an assosiated storage */
  value CsRestoreObj(VM *c, value obj);

  inline value &CsStorageOfPersistent(value obj) {
    return ptr<persistent_header>(obj)->vstorage;
  }

  void StoragePreGC(VM *c, value vs);
  void StoragePostGC(VM *c, value vs);
  bool IsEmptyStorage(value s);
  void DestroyStorage(VM *c, value obj);

  // used inside vector and object methods.
  inline bool _CsIsPersistent(value v) {
    return ptr<persistent_header>(v)->vstorage != 0;
  }
  // init persistent header. Used for persistent objects only
  inline void _CsInitPersistent(value obj) {
    ptr<persistent_header>(obj)->vstorage = 0;
    ptr<persistent_header>(obj)->oid      = 0;
    ptr<persistent_header>(obj)->status   = 0;
  }
  // make obj persistent - fill its storage/oid info and register it in the
  // storage
  oid_t CsSetPersistent(VM *c, value vs, value obj);

  value CsFetchObject(VM *c, value vs, oid_t oid);
  value CsFetchVector(VM *c, value vs, oid_t oid);
  // fetch data of objects with storage/oid set (persistent)
  value CsFetchObjectData(VM *c, value obj);
  value CsFetchVectorData(VM *c, value vec);

  inline value CsFetchObject(VM *c, value obj) {
    return _CsIsPersistent(obj) ? CsFetchObjectData(c, obj) : obj;
  }

  oid_t CsStoreObject(VM *c, value vs, value obj);
  oid_t CsStoreVector(VM *c, value vs, value obj);
  // store data of objects with storage/oid set (persistent)
  void CsStoreObjectData(VM *c, value obj);
  void CsStoreVectorData(VM *c, value vec);

  /* storage utils */
  void Transform(VM *c, value vs, value val, db_triplet &db_v);

  /* cs_persistent.cpp */
  void  CsInitDbIndex(VM *c);
  bool  CsDbIndexP(VM *c, value obj);
  value CsMakeDbIndex(VM *c, value storage, oid_t oid);
  value CsDbIndexSlice(VM *c, value obj, value start, value end, bool ascent,
                       bool startInclusive, bool endInclusive);

  // check for persistent type
  inline bool CsPersistentP(VM *c, value obj) {
    return (CsObjectP(obj) || CsVectorP(obj) || CsDbIndexP(c, obj));
  }
  // true if object has assosiated storage/oid with it
  inline bool CsIsPersistent(VM *c, value v) {
    return CsPersistentP(c, v) && ptr<persistent_header>(v)->vstorage != 0;
  }

#endif // TISCRIPT_USE_PERSISTENCE

  value CsBinaryOp(VM *c, int op, value p1, value p2);

  // each_property generator

  $generator(each_property) {
    int    i, cnt;
    pvalue props;
    pvalue prop;
    uint   ordinal;

    each_property(VM * c, value obj, bool fetch = true) : i(), cnt(), ordinal() {
      props.pin(c);
      prop.pin(c);
#if defined(TISCRIPT_USE_PERSISTENCE)
      if (fetch) obj = CsFetchObject(c, obj);
#endif
      props = CsObjectProperties(obj);
    }
    ~each_property() {
      props.unpin();
      prop.unpin();
    }

    //#pragma optimize( "g", off )
    $emit2(value, value) // will emit key/value pair
      if (CsHashTableP(props.val)) {
        cnt = CsHashTableSize(props.val);
        for (i = 0; i < cnt; ++i) {
          prop.val = CsHashTableElement(props.val, i);
          for (; prop.val != UNDEFINED_VALUE;
               prop.val = CsPropertyNext(prop.val)) {
            ordinal = CsPropertyOrdinal(prop.val);
            $yield2(CsPropertyTag(prop.val), CsPropertyValue(prop.val));
          }
        }
      }
      else for (prop.val = props.val; prop.val != UNDEFINED_VALUE;
                prop.val = CsPropertyNext(prop.val)) {
        ordinal = CsPropertyOrdinal(prop.val);
        $yield2(CsPropertyTag(prop.val), CsPropertyValue(prop.val));
      }
    $stop; // stop, end of sequence. End of body of the generator.
    //#pragma optimize( "g", on )
  };

  $generator(each_property_i) {
    int    i, cnt;
    value props;
    value prop;
    uint   ordinal;

    each_property_i(value obj, bool fetch = true)
      : i(), cnt(), ordinal() {
      props = CsObjectProperties(obj);
    }
    //#pragma optimize( "g", off )
    $emit2(value, value) // will emit key/value pair

      if (CsHashTableP(props)) {
        cnt = CsHashTableSize(props);
        for (i = 0; i < cnt; ++i) {
          prop = CsHashTableElement(props, i);
          for (; prop != UNDEFINED_VALUE;
            prop = CsPropertyNext(prop)) {
            ordinal = CsPropertyOrdinal(prop);
            $yield2(CsPropertyTag(prop), CsPropertyValue(prop));
          }
        }
      }
      else for (prop = props; prop != UNDEFINED_VALUE;
        prop = CsPropertyNext(prop)) {
        ordinal = CsPropertyOrdinal(prop);
        $yield2(CsPropertyTag(prop), CsPropertyValue(prop));
      }
      $stop; // stop, end of sequence. End of body of the generator.
             //#pragma optimize( "g", on )
  };
  
  inline bool CsNilP(value o) { return CsIsType(o, &CsNilDispatch); }

  inline dispatch *CsGetDispatch(value o) {
    auto pt = CsPrimitiveType(o);
    switch (pt) {
    case PT_NULL: return &CsNilDispatch;
    case PT_PTR: return CsQuickGetDispatch(o);
    case PT_SYMBOL: return &CsSymbolDispatch;
    case PT_INTEGER: return &CsIntegerDispatch;
    case PT_FLOAT: return &CsFloatDispatch;
    case PT_COLOR: return &CsColorDispatch;
    case PT_LENGTH: return &CsLengthDispatch;
    case PT_ANGLE: return &CsAngleDispatch;
    case PT_DURATION: return &CsDurationDispatch;

    case PT_IFACE_PTR:
    default: {
      assert(CsPrimitiveType(o) >= PT_IFACE_PTR);
      dispatch *pd = CsQuickGetDispatch(iface_base(o));
      if (pd->interfaces) {
        int ifno = iface_no(o);
        assert(ifno < 16);
        assert(pd->interfaces[ifno]);
        return pd->interfaces[ifno];
      }
      assert(false);
      return &CsBrokenHeartDispatch;
    }
    }
  }

  // Pub Sub

  value CsEventObjectAdd(VM *c, value obj, value fun, value ename,
                         value enamespace = NULL_VALUE,
                         value eselector  = NULL_VALUE);
  value CsEventObjectAdd(VM *c, value obj, value fun, wchars name_namespace, wchars selector = wchars());
  value CsEventObjectAddEF(VM *c, value obj, value fun);
  value CsEventObjectRemove(VM *c, value obj, value fun,
                            value ename      = UNDEFINED_VALUE,
                            value enamespace = UNDEFINED_VALUE,
                            value eselector  = UNDEFINED_VALUE);
  value CsEventObjectRemoveV(VM *c, value obj, value v);
  value CsEventObjectFire(VM *c, value obj, value name, value param = 0);
  value CsEventObjectFire(VM *c, value obj, value name, tool::function<value()> param);

#define EVENT_DEF_LENGTH 5 /* 4 see below + next link */

  inline value CsEventDefFunction(value ed) { return CsFixedVectorElement(ed, 0); }
  inline void  CsSetEventDefFunction(value ed, value func) { CsSetFixedVectorElement(ed, 0, func); }
  inline value CsEventDefName(value ed) { return CsFixedVectorElement(ed, 1); }
  inline value CsEventDefNamespace(value ed) { return CsFixedVectorElement(ed, 2); }
  inline value CsEventDefSelector(value ed) { return CsFixedVectorElement(ed, 3); }

  $generator(each_event_handler) // handler
  {
    pvalue erec;
    value  curr;
    each_event_handler(VM * c, value obj) { erec.pin(c, CsObjectEvents(obj)); }
    ~each_event_handler() { erec.unpin(); }

    $emit4(value, value, value,value) // will emit func, name, namespace, selector
      while (erec && CsFixedVectorP(erec)) {
        curr = erec;
        erec = CsFixedVectorElement(curr, EVENT_DEF_LENGTH - 1);
        $yield4(CsEventDefFunction(curr), CsEventDefName(curr),
                CsEventDefNamespace(curr), CsEventDefSelector(curr));
      }
    $stop; // stop, end of sequence. End of body of the generator.
  };

  void finalize();

  tool::value call_by_tool(tis::pvalue &method, value self, uint argc,
                           const tool::value *argv);

  inline long ValueSize(value o) { return CsQuickGetDispatch(o)->size(o); }

  inline bool CsDelProperty(VM *c, value o, value sym) {
    dispatch *pd = CsGetDispatch(o);
    if (pd->delItem) return pd->delItem(c, o, sym);
    return false;
  }
  // inline bool CsDelGlobalOrNamespaceValue(VM* c, value sym)
  //{
  //  return false; //?
  //}
  bool CsDelGlobalOrNamespaceValue(VM *c, value sym);

  inline bool CsDelItem(VM *c, value o, value sym_or_index) {
    dispatch *pd = CsGetDispatch(o);
    if (pd->delItem) return pd->delItem(c, o, sym_or_index);
    return false;
  }

  /* virtual property */
  inline value CsMakeVirtualProperty(VM *c) {
    value pair[2] = { UNDEFINED_VALUE, UNDEFINED_VALUE };
    return CsMakeFixedVector(c, &CsVirtualPropertyDispatch, 2, pair);
  }

  inline value CsGetVirtualPropertyGetter(value vp) { return CsFixedVectorElement(vp, 0); }
  inline void  CsSetVirtualPropertyGetter(value vp, value fun) { assert(CsMethodP(fun)); return CsSetFixedVectorElement(vp, 0, fun); }
  inline value CsGetVirtualPropertySetter(value vp) { return CsFixedVectorElement(vp, 1); }
  inline void  CsSetVirtualPropertySetter(value vp, value fun) { assert(CsMethodP(fun)); return CsSetFixedVectorElement(vp, 1, fun); }

  // protector
  inline void protected_push(VM *c, value &v) {
    c->values_on_c_stack.push(
        &v); /*assert(v == 0 || is_ptr(v) || is_symbol(v));*/
  }
  inline void protected_pop(VM *c) { c->values_on_c_stack.pop(); }

  template<class TVM>
  inline protector_t<TVM>::protector_t(TVM *vm, value &v1) {
    c  = vm;
    n  = 1;
    protected_push(c, v1);
  }
  template<class TVM>
  inline protector_t<TVM>::protector_t(TVM *vm, value &v1, value &v2) {
    c = vm;
    n  = 2;
    protected_push(c, v1);
    protected_push(c, v2);
  }
  template<class TVM>
  inline protector_t<TVM>::protector_t(TVM *vm, value &v1, value &v2, value &v3) {
    c = vm;
    n  = 3;
    protected_push(c, v1);
    protected_push(c, v2);
    protected_push(c, v3);
  }

  template<class TVM>
  inline protector_t<TVM>::protector_t(TVM *vm, value &v1, value &v2, value &v3,
                              value &v4) {
    c = vm;
    n  = 4;
    protected_push(c, v1);
    protected_push(c, v2);
    protected_push(c, v3);
    protected_push(c, v4);
  }
  template<class TVM>
  inline protector_t<TVM>::protector_t(TVM *vm, value &v1, value &v2, value &v3, value &v4,
                              value &v5) {
    c = vm;
    n  = 5;
    protected_push(c, v1);
    protected_push(c, v2);
    protected_push(c, v3);
    protected_push(c, v4);
    protected_push(c, v5);
  }
  template<class TVM>
  inline protector_t<TVM>::protector_t(TVM *vm, value &v1, value &v2, value &v3, value &v4,
                              value &v5, value &v6) {
    c = vm;
    n  = 6;
    protected_push(c, v1);
    protected_push(c, v2);
    protected_push(c, v3);
    protected_push(c, v4);
    protected_push(c, v5);
    protected_push(c, v6);
  }
  template<class TVM>
  inline protector_t<TVM>::protector_t(TVM *vm, value &v1, value &v2, value &v3, value &v4,
                              value &v5, value &v6, value &v7) {
    c = vm;
    n  = 7;
    protected_push(c, v1);
    protected_push(c, v2);
    protected_push(c, v3);
    protected_push(c, v4);
    protected_push(c, v5);
    protected_push(c, v6);
    protected_push(c, v7);
  }
  template<class TVM>
  inline protector_t<TVM>::protector_t(TVM *vm, value &v1, value &v2, value &v3, value &v4,
                              value &v5, value &v6, value &v7, value &v8) {
    c = vm;
    n  = 8;
    protected_push(c, v1);
    protected_push(c, v2);
    protected_push(c, v3);
    protected_push(c, v4);
    protected_push(c, v5);
    protected_push(c, v6);
    protected_push(c, v7);
    protected_push(c, v8);
  }
  template<class TVM>
  inline protector_t<TVM>::~protector_t() {
    while (n--)
      protected_pop(c);
  }

#define PROTECT(...) protector CONCAT(_protect_, __LINE__)(c, __VA_ARGS__);
#define PROTECT_VM(...) protector CONCAT(_protect_, __LINE__)(vm, __VA_ARGS__);
#define PROTECT_THIS(...) protector CONCAT(_protect_, __LINE__)(this, __VA_ARGS__);
#define PROTECT_P(vm, ...) protector CONCAT(_protect_, __LINE__)(vm, __VA_ARGS__);

  // GC safe vector
  struct GC_vector : public gc_callback {
    tool::array<value> elements;
    GC_vector(VM *c) : gc_callback(c) {}
    virtual void on_GC(VM *c) {
      for (int n = 0; n < elements.size(); ++n)
        elements[n] = CsCopyValue(c, elements[n]);
    }
  };

  /*struct protect1: protector
  {
    value& ref;
    protect1(VM* c, value& p1 ): protector(c), ref(p1) {}
    virtual void on_gc() { if(ref) ref = CsCopyValue(vm,ref); }
  };

  struct protect2: protect1
  {
    value& ref;
  public:
    protect2(VM* c, value& p1, value& p2 ): protect1(c,p1), ref(p2) {}
    virtual void on_gc() { protect1::on_gc(); if(ref) ref = CsCopyValue(vm,ref);
  }
  };

  struct protect3: protect2
  {
    value& ref;
  public:
    protect3(VM* c, value& p1, value& p2, value& p3 ): protect2(c,p1,p2),
  ref(p3) {} virtual void on_gc() { protect2::on_gc(); if(ref) ref =
  CsCopyValue(vm,ref); }
  };

  struct protect4: protect3
  {
    value& ref;
  public:
    protect4(VM* c, value& p1, value& p2, value& p3, value& p4 ):
  protect3(c,p1,p2,p3), ref(p4) {} virtual void on_gc() { protect3::on_gc();
  if(ref) ref = CsCopyValue(vm,ref); }
  };*/

  INLINE bool CsIsNewObjectP(VM *c, value obj) {
    return (ptr<byte>(obj) >= (c)->newSpace->base &&
            ptr<byte>(obj) < (c)->newSpace->free);
  }
  INLINE bool CsAnyMethodP(value o) {
    return CsMethodP(o) || CsCMethodP(o) || CsCFunctorP(o);
  }

  inline bool CsPrimitiveValueP(VM* c, value o) {
    return CsIntegerP(o)
      || CsFloatP(o)
      || CsStringP(o)
      || CsSymbolP(o)
      || CsLengthP(o)
      || CsDurationP(o)
      || CsDateP(c, o)
      || CsColorP(o);
  }

  inline bool CsCollectGarbageIf(
      VM *c, size_t threshold) // ... if it is enough garbage to collect.
  {
    CsMemorySpace *space = c->newSpace;

    if (threshold == 0) {
      if ((space->free - space->base) <
          ((space->top - space->base) * 2) / 3) // 2/3 threshold
        return false;
    } else if (size_t(space->top - space->free) > threshold)
      return false;

    CsCollectGarbage(c);
    return true;
  }

  template <typename T> bool parse_int(slice<T> text, int_t &r) {
    int64 li = to_int64(text);
    if (text.length != 0) return false; // not fully parsed
    if (li > 0x7fffffffll && li <= 0xffffffffll)
      r = (int_t)li;
    else if (li > (int64)limits<uint>::max_value() ||
             li < (int64)limits<int>::min_value())
      return false;
    else
      r = (int_t)li;
    return true;
  }

  int CsGetLineNumber(VM *c, value ccode, int pc);

  struct breakpoint {
    value fileUrlSym; // symbol, file name
    int   lineNo;
    breakpoint() : fileUrlSym(0), lineNo(0) {}
    bool operator==(const breakpoint &other) const {
      return fileUrlSym == other.fileUrlSym && lineNo == other.lineNo;
    }
    bool operator!=(const breakpoint &other) const {
      return fileUrlSym != other.fileUrlSym || lineNo != other.lineNo;
    }
    bool isDefined() const { return fileUrlSym != 0; }
  };

  enum SCRIPT_DEBUG_COMMANDS {
    SCRIPT_DEBUG_CONTINUE = 1,
    SCRIPT_DEBUG_STEP_INTO,
    SCRIPT_DEBUG_STEP_OVER,
    SCRIPT_DEBUG_STEP_OUT,
    SCRIPT_DEBUG_STOP
  };

  template <class T> struct stack_ptr {
    uint_ptr top_off = 0;

    T *get(VM *c) {  return top_off ? ((T *)(uint_ptr(c->stackTop) - top_off + 1)) : 0; }
    T *get(value *stackTop) { return top_off ? ((T *)(uint_ptr(stackTop) - top_off + 1)) : 0; }
    
    void set(VM *c, const T *ptr) { top_off = uint_ptr(c->stackTop) - uint_ptr(ptr) + 1; }

    bool is_set() const { return top_off != 0; }

  };

  struct debug_peer : public resource {

    virtual SCRIPT_DEBUG_COMMANDS
                 breakpoint_hit(VM *vm, const wchar *fileUrl, int atLine,
                                const tool::value &envData) = 0;
    virtual void requested_data(VM *vm, uint command,
                                const tool::value &data)    = 0;

    virtual void enter_execution(VM *vm) {
      nextBreakpoint = breakpoint();
      //lastCmd.clear();
    }
    virtual void leave_execution(VM *vm) {
      nextBreakpoint = breakpoint();
      nextFunctionOut = false;
      nextUnconditionally = false;
      //lastCmd.clear();
    }

    virtual void enter_function(VM *vm, value functionCode, int firstLine,
                                int lastLine);
    virtual void leave_function(VM *vm, value callerCode, int callLine);
    virtual void check_line_no(VM *vm, int lineNo);

    void handle_breakpoint_hit(VM *vm, int lineNo);

    void add_breakpoint(wchars fileUrl, uint lineNo);
    void remove_breakpoint(wchars fileUrl, uint lineNo);

    array<breakpoint>  breakpoints;
    breakpoint         nextBreakpoint;
    bool               nextUnconditionally = false;
    value              currentFunctionFileNameSym = 0;
    bool               gatherVisitStats = false;
    stack_ptr<CsFrame> breakpointFrame;
    bool               skipLineNoCheck = false;
    bool               nextFunctionOut = false;

    struct SourceFileDef : public resource {
      value      fileNameSym;
      array<int> visitedLines;
    };

    handle<SourceFileDef>                    currentFunctionFileDef;
    hash_table<value, handle<SourceFileDef>> sourceFileDefs;

    void start_coverage_gathering() {
      // sourceFileDefs.clear();
      for (auto def : sourceFileDefs.elements())
        def->visitedLines.clear();

      currentFunctionFileDef = nullptr;
      gatherVisitStats       = true;
    }

    void stop_coverage_gathering() {
      // sourceFileDefs.clear();
      // currentFunctionFileDef = nullptr;
      gatherVisitStats = false;
    }

    void register_source_file(VM *vm, value fileNameSym) {
      auto ctor = [this, fileNameSym](handle<SourceFileDef> &ref) {
        ref              = new SourceFileDef;
        ref->fileNameSym = fileNameSym;
        // dbg_printf("register_source_file %S\n",
        // value_to_string(fileNameSym).c_str());
      };
      sourceFileDefs.get_ref(fileNameSym, ctor);
    }

    void check_file(VM *vm) {
      if (!currentFunctionFileDef ||
          (currentFunctionFileDef->fileNameSym != currentFunctionFileNameSym)) {
        auto ctor = [this](handle<SourceFileDef> &ref) {
          ref              = new SourceFileDef;
          ref->fileNameSym = currentFunctionFileNameSym;
          // dbg_printf("check_file %S\n",
          // value_to_string(currentFunctionFileNameSym).c_str());
        };
        currentFunctionFileDef =
            sourceFileDefs.get_ref(currentFunctionFileNameSym, ctor);
      }
    }

    void visit_line_no(VM *vm, int lineNo) {
      assert(currentFunctionFileNameSym == CsCompiledCodeFileName(vm->code));
      // if (currentFunctionFileNameSym != CsCompiledCodeFileName(vm->code)) {
      //  currentFunctionFileNameSym = CsCompiledCodeFileName(vm->code);
      //  check_file(vm);
      //}
      if (currentFunctionFileDef) {
        if (lineNo >= currentFunctionFileDef->visitedLines.size())
          currentFunctionFileDef->visitedLines.size(lineNo + 1);
        ++currentFunctionFileDef->visitedLines[lineNo];
      }
    }
  };

  struct debug_env {
    debug_env(VM *c) {
      this->c = c;
      if (c->pdebug) c->pdebug->enter_execution(c);
    }

    ~debug_env() {
      if (c->pdebug) c->pdebug->leave_execution(c);
    }
    VM *c;
  };

  inline value cs_return(VM* c, value r1, value r2, value r3, value r4)
  {
    PROTECT(r1, r2, r3, r4);
    value list = CsMakeFixedVectorValue(c, &CsValueListDispatch,4);
    CsSetFixedVectorElement(list, 0, r1);
    CsSetFixedVectorElement(list, 1, r2);
    CsSetFixedVectorElement(list, 2, r3);
    CsSetFixedVectorElement(list, 3, r4);
    return list;
  }
  inline value cs_return(VM* c, value r1, value r2, value r3)
  {
    PROTECT(r1, r2, r3);
    value list = CsMakeFixedVectorValue(c, &CsValueListDispatch, 3);
    CsSetFixedVectorElement(list, 0, r1);
    CsSetFixedVectorElement(list, 1, r2);
    CsSetFixedVectorElement(list, 2, r3);
    return list;
  }
  inline value cs_return(VM* c, value r1, value r2)
  {
    PROTECT(r1, r2);
    value list = CsMakeFixedVectorValue(c, &CsValueListDispatch, 2);
    CsSetFixedVectorElement(list, 0, r1);
    CsSetFixedVectorElement(list, 1, r2);
    return list;
  }

  inline value scalar_value(value v) {
    if (CsValueListP(v))
      v = CsValueListLastElement(v);
    return v;
  }
  inline value value_to_set(value v) {
    if (CsValueListP(v))
      v = CsValueListLastElement(v);
    if (v == NOTHING_VALUE)
      return UNDEFINED_VALUE;
    else
      return v;
    // return v == NOTHING_VALUE? UNDEFINED_VALUE: v;
  }

  void CsSilentPropertySetter(VM *c, value obj, value value);

  tool::pair<value, value> CsReferenceOf(VM *c, value ns, wchars path);

  inline bool CsEntityMeta(value o, value &val) {
    auto ma = CsGetDispatch(o)->metaAccessor;
    assert(ma);
    if (!ma) return false;
    val = ma(o);
    if (!val) val = NOTHING_VALUE;
    return true;
  }
  inline bool CsSetEntityMeta(value o, value v) {
    auto ma = CsGetDispatch(o)->metaAccessor;
    assert(ma);
    if (!ma) return false;
    ma(o) = v;
    return true;
  }

  bool CsEntityMeta(VM* c, value o, value tag, value& val);
  bool CsSetEntityMeta(VM* c, value &o, value tag, value &val);
  
  extern value sym_instance;
  extern value sym_ref;
  extern value sym_factory;

  inline value CsMakeVNode(VM *c, value tag, value atts, value kids, value states)
  {
    PROTECT(tag, atts, kids, states);
    value vn = CsMakeTuple(c, 3);
    CsSetTupleTag(vn, tag);
    CsSetTupleElement(vn, 0, atts);
    CsSetTupleElement(vn, 1, kids);
    CsSetTupleElement(vn, 2, states);
    return vn;
  }


  void  CsInitAsset(VM* c);
  bool  CsAssetP(value val);
  value CsMakeAssetObject(VM *c, som_asset_t *ptr);
  ustring CsAssetTypeName(value obj);
  tool::value CsAssetProps(value obj);
  som_asset_t* CsAssetPtr(value obj);

} // namespace tis


#pragma pack(pop)

#endif
