#include "html.h"
#include "tool/tl_zip.h"

#if defined(HTTP_SUPPORT)

#include <wininet.h>

#define BUFLEN 4096 // Size of HTTP Buffer...
#define MAX_REDIRECTS 6

namespace html {
  class IHANDLE {
  public:
    HINTERNET h;
    IHANDLE() : h(0) {}
    IHANDLE(HINTERNET p) : h(p) {}
    ~IHANDLE() { close(); }
    void close() {
      HINTERNET th = 0; swap(th, h);
      if (th)
        InternetCloseHandle(th);
    }
    IHANDLE &operator=(HINTERNET p) {
      close();
      h = p;
      return *this;
    }
    operator HINTERNET() { return h; }

    NONCOPYABLE(IHANDLE)
  };

  class task : public tool::async::task {
    typedef tool::async::task super;
    handle<request> hrq;
    IHANDLE         hConnection;

    task(request *prq) : hConnection(0) { hrq = prq; }

  public:

    static void create(request *prq) {
      task *nt = new task(prq);
      nt->start();
    }

    virtual ~task() {}

    virtual void exec() {
      do_exec();
      stop();
    }
    virtual void stop() { 
      hConnection.close(); 
      super::stop();
    }
    void         do_exec();

    void data_ready(uint status) {
      hrq->done_with_success(status);
    }
    void failed(uint status) {
      hrq->done_with_failure(status);
    }
    bool is_active() {
      if (!hrq->the_pump) return false;
      return hrq->the_pump->is_active();
    }
  };

  class win_inet_client : public inet_client {
    bool         active;

  public:
    win_inet_client() : active(0) {}

    virtual ~win_inet_client() {}
    virtual void exec(pump *pm, request *rq) override;
  };

  static IHANDLE    h_root_handle = NULL;
  static tristate_v proxy_configured;

  void pump::open_internet() {
    if (!h_root_handle) {
      h_root_handle =
          InternetOpenA(get_user_agent(), INTERNET_OPEN_TYPE_PRECONFIG, NULL, NULL, 0);
      if (!h_root_handle) return;

      BOOL t = TRUE;
      InternetSetOption(h_root_handle, INTERNET_OPTION_CONNECT_TIMEOUT,
                        &connection_timeout, sizeof(connection_timeout));
      InternetSetOption(h_root_handle, INTERNET_OPTION_HTTP_DECODING, &t,sizeof(t));
      // InternetSetOption(h_root_handle, INTERNET_OPTION_CLIENT_CERT_CONTEXT, NULL, 0); ???
    }
    if (!inet) inet = new win_inet_client();
  }

#define HTTPS_FLAGS                                                            \
  (INTERNET_FLAG_SECURE /*| INTERNET_FLAG_IGNORE_CERT_CN_INVALID | INTERNET_FLAG_IGNORE_CERT_DATE_INVALID*/)

  static uint http_flags(const string &surl) {
    url u(surl);
    return (u.protocol == CHARS("https")) ? HTTPS_FLAGS : 0;
  }

  bool send_request(HINTERNET hRequest, chars headers, bytes data = bytes()) {
    if (HttpSendRequestA(hRequest, headers.start, DWORD(headers.length),
                         (void *)data.start, DWORD(data.length)))
      return true;

    DWORD dwError = GetLastError();
    if (dwError == ERROR_INTERNET_INVALID_CA &&
        html::pump::https_error_action) {
      if (html::pump::https_error_action == 1) {
        if (ERROR_CANCELLED !=
            InternetErrorDlg(GetDesktopWindow(), hRequest,
                             ERROR_INTERNET_INVALID_CA,
                             FLAGS_ERROR_UI_FILTER_FOR_ERRORS |
                                 FLAGS_ERROR_UI_FLAGS_GENERATE_DATA |
                                 FLAGS_ERROR_UI_FLAGS_CHANGE_OPTIONS,
                             NULL))
          return HttpSendRequestA(hRequest, headers.start,
                                  DWORD(headers.length), (void *)data.start,
                                  DWORD(data.length)) != FALSE;
      } else if (html::pump::https_error_action == 2) {
        DWORD dwFlags;
        DWORD dwBuffLen = sizeof(dwFlags);
        InternetQueryOption(hRequest, INTERNET_OPTION_SECURITY_FLAGS,
                            (LPVOID)&dwFlags, &dwBuffLen);

        dwFlags |= SECURITY_FLAG_IGNORE_UNKNOWN_CA;
        InternetSetOption(hRequest, INTERNET_OPTION_SECURITY_FLAGS, &dwFlags,
                          sizeof(dwFlags));
        return HttpSendRequestA(hRequest, headers.start, DWORD(headers.length),
                                (void *)data.start,
                                DWORD(data.length)) != FALSE;
      }
    }
    return FALSE;
  }

  void win_inet_client::exec(pump *pm, request *prq) {
    active = true;
    task::create(prq);
  }

  uint get_status(HINTERNET hOpenUrlHandle) {
    char  info_buffer[128]   = {0};
    DWORD info_buffer_length = 128;
    DWORD info_buffer_index  = 0;

    if (HttpQueryInfoA(hOpenUrlHandle, HTTP_QUERY_STATUS_CODE, info_buffer,
                       &info_buffer_length, &info_buffer_index)) {
      return (uint)atoi(info_buffer);
    } else
      return 404;
  }

  void configure_proxy(HINTERNET h, request *prq) {
    if (prq->proxy_host.is_undefined() || prq->proxy_port.is_undefined()) {
      if (proxy_configured) {
        /*INTERNET_PROXY_INFO pi; memzero(pi);
        pi.dwAccessType = INTERNET_OPEN_TYPE_PRECONFIG;
        InternetSetOption(h,INTERNET_OPTION_PROXY, &pi, sizeof(pi));*/

        INTERNET_PER_CONN_OPTION_LIST List;
        INTERNET_PER_CONN_OPTION      Option[1];
        unsigned long nSize = sizeof(INTERNET_PER_CONN_OPTION_LIST);

        Option[0].dwOption      = INTERNET_PER_CONN_FLAGS;
        Option[0].Value.dwValue = PROXY_TYPE_DIRECT | PROXY_TYPE_AUTO_DETECT;

        List.dwSize        = sizeof(INTERNET_PER_CONN_OPTION_LIST);
        List.pszConnection = NULL;
        List.dwOptionCount = 1;
        List.dwOptionError = 0;
        List.pOptions      = Option;

        InternetSetOption(h, INTERNET_OPTION_PER_CONNECTION_OPTION, &List,
                          nSize);

        proxy_configured = FALSE;
      }
      return;
    }

    INTERNET_PER_CONN_OPTION_LIST list;
    BOOL                          bReturn;
    DWORD                         dwBufSize = sizeof(list);

    // Fill the list structure.
    list.dwSize = sizeof(list);

    // NULL == LAN, otherwise connectoid name.
    list.pszConnection = NULL;

    // Set three options.
    list.pOptions      = new INTERNET_PER_CONN_OPTION[3];
    list.dwOptionCount = 3;

    // Ensure that the memory was allocated.
    if (NULL == list.pOptions) {
      // Return FALSE if the memory wasn't allocated.
      return;
    }

    ustring proxy_addr =
        string::format("http=http://%s:%d;https=https://%s:%d",
                       prq->proxy_host.c_str(), prq->proxy_port.val(0),
                       prq->proxy_host.c_str(), prq->proxy_port.val(0));
    // Set flags.

    list.pOptions[0].dwOption      = INTERNET_PER_CONN_FLAGS;
    list.pOptions[0].Value.dwValue = PROXY_TYPE_DIRECT | PROXY_TYPE_PROXY;

    // Set proxy name.
    list.pOptions[1].dwOption       = INTERNET_PER_CONN_PROXY_SERVER;
    list.pOptions[1].Value.pszValue = (LPWSTR)proxy_addr.c_str();

    // Set proxy override.
    list.pOptions[2].dwOption       = INTERNET_PER_CONN_PROXY_BYPASS;
    list.pOptions[2].Value.pszValue = L"local";

    // Set the options on the connection.
    bReturn = InternetSetOption(h, INTERNET_OPTION_PER_CONNECTION_OPTION, &list,
                                dwBufSize);

    proxy_configured = TRUE;

    // Free the allocated memory.
    delete[] list.pOptions;
  }

  void task::do_exec() {
    handle<request> keeper    = hrq;
    int             redirects = 0;

    configure_proxy(h_root_handle, hrq);

    hrq->seal_request();

  RESTART:

    // strings are in principle not thread safe so make local copy of strings
    // used.
    string real_url;
    if (hrq->content_url.length())
      real_url = hrq->content_url.c_str(); // scondary request after redirect
    else
      real_url = hrq->url.c_str();

    tool::url uri = hrq->used_url();

    if (uri.hostname.length() == 0) {
      assert(false);
      failed(0);
      return;
    }

    bool is_https = uri.protocol == CHARS("https");

    UINT connection_flags = INTERNET_FLAG_NO_AUTO_REDIRECT;
    if (hrq->rq_type == RQ_POST)
      connection_flags |= INTERNET_FLAG_KEEP_CONNECTION |
                          INTERNET_FLAG_NO_CACHE_WRITE |
                          INTERNET_FLAG_FORMS_SUBMIT;
    else
      connection_flags |= INTERNET_FLAG_KEEP_CONNECTION;
    if (is_https) connection_flags |= HTTPS_FLAGS;

    if( hrq->no_cache )
      connection_flags |= (INTERNET_FLAG_PRAGMA_NOCACHE | INTERNET_FLAG_NO_CACHE_WRITE);

    hConnection = ::InternetConnectA(h_root_handle, // internet opened handle
                                     uri.hostname,  // server name
                                     INTERNET_PORT(uri.port), // ports
                                     uri.username,            // user name
                                     uri.password,            // password
                                     INTERNET_SERVICE_HTTP,   // service type
                                     connection_flags,        // service option
                                     0); // context call-back option

    if (!hConnection) {
      failed(::GetLastError());
      return;
    }

    string objname = uri.compose_object();

    IHANDLE hRequest = ::HttpOpenRequestA(hConnection,
                                          get_http_verb(hrq), // HTTP Verb
                                          objname,            // Object Name
                                          HTTP_VERSIONA,      // Version
                                          "",                 // Reference
                                          HTTP_ACCEPT_DATA,   // Accept Type
                                          connection_flags,
                                          0); // context call-back point

    if (!hRequest) {
      failed(::GetLastError());
      return;
    }

    // send it

    if (!send_request(hRequest, hrq->headers_list(), hrq->data())) {
      failed(::GetLastError());
      return;
    }

    uint status = get_status(hRequest);

    char info_buffer[1024];
    // info_buffer[1023] = 0;
    memset(info_buffer, 0, sizeof(info_buffer));
    DWORD info_buffer_length = 1024;
    DWORD info_buffer_index  = 0;

    if ( // if it was not redirected before.
        status == HTTP_STATUS_MOVED || status == HTTP_STATUS_REDIRECT ||
        status == HTTP_STATUS_REDIRECT_METHOD) {
      if (++redirects >= MAX_REDIRECTS) goto FINISHED;

      if (HttpQueryInfoA(hRequest, HTTP_QUERY_LOCATION, info_buffer,
                         &info_buffer_length, &info_buffer_index)) {
        if (info_buffer_length) {
          hrq->content_url =
              combine_url(hrq->url, string(info_buffer, info_buffer_length));
          goto RESTART;
          // return;
        }
      }
      info_buffer_length = 1024;
      info_buffer_index  = 0;
    }

    byte buff[BUFLEN];

    if (HttpQueryInfoA(hRequest, HTTP_QUERY_CONTENT_TYPE, info_buffer,
                       &info_buffer_length, &info_buffer_index)) {
      if (!verify_content_type(hrq, info_buffer)) goto FINISHED;
    }

    {
      DWORD length   = 0;
      DWORD szlength = sizeof(length);
      DWORD dwindex  = 0;
      if (HttpQueryInfo(hRequest,
                        HTTP_QUERY_CONTENT_LENGTH | HTTP_QUERY_FLAG_NUMBER,
                        &length, &szlength, &dwindex))
        hrq->content_length = length;
    }

    bool compressed = false;

    {
      array<char> headers_buffer(1024);
      DWORD length = 1024;

      for (int n = 0; n < 2; ++n)
        if (!HttpQueryInfoA(hRequest, HTTP_QUERY_RAW_HEADERS_CRLF, (LPVOID)headers_buffer.begin(), &length, NULL))
        {
          if (GetLastError() == ERROR_HTTP_HEADER_NOT_FOUND)
            break;
          // Check for an insufficient buffer.
          if (GetLastError() == ERROR_INSUFFICIENT_BUFFER)
          {
            // allocate the necessary buffer.
            headers_buffer.length(length);
            // Retry the call.
            continue;
          }
        }
        else {
          headers_buffer.length(length);
          chars all = headers_buffer();
          while (all.length) {
            chars one = all.chop(CHARS("\r\n"));
            if (one.length) {
              chars name, value;
              if (one.split(':', name, value))
                hrq->rs_headers[trim(name)] = trim(value);
            }
          }
          break;
        }
    }

#if 0
    {
      char buffer[128] = {0};
      strcpy_s(buffer, "Content-Encoding");
      DWORD length = sizeof(buffer);

      HttpQueryInfoA(hRequest, HTTP_QUERY_CUSTOM, (LPVOID)buffer, &length, NULL);

      if (GetLastError() == ERROR_HTTP_HEADER_NOT_FOUND)
        ;
      else if (strcmp(buffer, "deflate") == 0)
        compressed = true;
      else if (strcmp(buffer, "gzip") == 0)
        compressed = true;
    }
#endif
    {
      string enc = hrq->rs_headers("Content-Encoding");
      if (enc == CHARS("deflate") || enc == CHARS("gzip"))
        compressed = true;
    }

    hrq->data.clear();

    while (hrq->data.length() < pump::max_http_data_length * 1024 * 1024) {
      if (!is_active()) {
        failed(status);
        return;
      }

      DWORD dwBytesRead = 0;
      BOOL  r = InternetReadFile(hRequest, buff, BUFLEN, &dwBytesRead);
      if (!r || !dwBytesRead) {
        status = r ? status : ::GetLastError(); // not 0 : ::GetLastError();
        break;
      }
      hrq->data_chunk_arrived(bytes(buff, dwBytesRead));
    }
    hrq->data_chunk_arrived(bytes());

    if (compressed) {
      array<byte> data;
      if (tool::gzip_uncompress(hrq->data(), data))
        hrq->data.transfer_from(data);
    }

  FINISHED:
    data_ready(status);
  }

} // namespace html

#endif