//
//  macosx_bits.mm
//
//  Copyright (c) 2014 andrew fedoniouk. All rights reserved.
//
#import <Cocoa/Cocoa.h>
#include "gool/gool.h"
#include "html/html.h"

#include <ApplicationServices/ApplicationServices.h>
#include <Carbon/Carbon.h>
#import <IOKit/graphics/IOGraphicsLib.h>

#if defined(NATIVE_OSX_BACKEND)

gool::bitmap* osx_get_icon_for_path(const tool::ustring& path, gool::size sz, bool must_exist )
{
    @autoreleasepool {
        NSString *spath = cvt(path);
        
        if(path.like(W("~*")))
           spath = [spath stringByExpandingTildeInPath];
        
        sz;
        
        BOOL isDir;
        NSImage* pbm = nullptr;
        if(must_exist) {
        if(![[NSFileManager defaultManager] fileExistsAtPath:spath isDirectory:&isDir])
            return nullptr;
            pbm = [[NSWorkspace sharedWorkspace] iconForFile:spath];
        } else {
            NSString *ext = cvt(path().r_tail('.'));
            pbm = [[NSWorkspace sharedWorkspace] iconForFileType:ext];
//          (NSImage *)iconForContentType:(UTType *)contentType;
        }
        
        if(pbm) {
            //gool::bitmap* gbm = new gool::bitmap(sz,true,false);
            NSRect r; r.origin.x = 0; r.origin.y = 0; r.size.width = sz.x; r.size.height = sz.y;
            //assert ([ NSGraphicsContext currentContext]);
            //gbm->cg_image = CGImageCreateCopy([pbm CGImageForProposedRect:&r
            //                                    context:[ NSGraphicsContext currentContext]
            //                                      hints:NULL]);
            //return gbm;
            
            auto cgimg = [pbm CGImageForProposedRect:&r context:[ NSGraphicsContext currentContext] hints:NULL];
            
            return gool::bitmap::create(cgimg, true, sz);
        }
        return nullptr;
    }
}

static void ArrayBufReleaseDataCallback(void *info, const void *data, size_t size) {
  tool::array<gool::argb>::array_data *pa = (tool::array<gool::argb>::array_data *)info;
  tool::array<gool::argb>::array_data::release(pa);
}

CGImageRef osx_ttb_image( gool::bitmap* bmp, gool::graphics* gfx )
{
    if( !bmp )
        return NULL;
    
    if( !bmp->_pixels.length() )
        return NULL;
    
    tool::array<gool::argb> bits(bmp->_pixels.length()); bits.clear();
    tool::array<gool::argb> row(bmp->_dim.x);

    for(int n = 0; n < bmp->_dim.y; ++n )
       bits.push( (*bmp)[n] );

    auto adata = bits.detach_data();
    CF::ref<CGDataProviderRef> provider = CGDataProviderCreateWithData(adata, adata->elements, adata->size * sizeof( gool::argb), &ArrayBufReleaseDataCallback);
            
    CF::ref<CGColorSpaceRef> colorSpace = CGColorSpaceCreateDeviceRGB();
            
    CGBitmapInfo flags = kCGImageAlphaPremultipliedFirst | kCGBitmapByteOrder32Little;
            
    CGImageRef cgi = CGImageCreate ((size_t) bmp->_dim.x,
                                    (size_t) bmp->_dim.y,
                                    8, sizeof(gool::argb) * 8,
                                    (size_t) bmp->stride(),
                                    colorSpace, flags, provider,
                                    0, false, kCGRenderingIntentDefault);
    return cgi;
}

void osx_image_release_data( void *info, const void *data, size_t size )
{
    gool::argb* image_data = static_cast<gool::argb*>(info);
    delete[] image_data;
}


//==============================================================================
CGImageRef osx_image( gool::bitmap* bmp, gool::graphics* gfx )
{
    if( !bmp )
        return NULL;
    
    gool::size dim = bmp->dim();
    
    if( !bmp->cg_image
         || CGImageGetWidth(bmp->cg_image) != dim.x
         || CGImageGetHeight(bmp->cg_image) != dim.y
         || bmp->_generation != bmp->_used_generation
       ) {
        
        bmp->_used_generation = bmp->_generation;
        
        if( bmp->_pixels.length() ) {
            
          critical_section _(gool::lock);
            
          auto src_bits = bmp->get_bits();
            
          gool::argb* image_data = new gool::argb[src_bits.length];
          if( !image_data )
              return bmp->cg_image;

          target(image_data,src_bits.length).copy(src_bits.start);
        
          CF::ref<CGDataProviderRef> provider = CGDataProviderCreateWithData (image_data, image_data, src_bits.length * sizeof( gool::argb), osx_image_release_data);
        
          CF::ref<CGColorSpaceRef> colorSpace = CGColorSpaceCreateDeviceRGB();
        
          CGBitmapInfo flags = kCGImageAlphaPremultipliedFirst | kCGBitmapByteOrder32Little;
        
          CF::ref<CGImageRef> cgi = CGImageCreate ((size_t) dim.x,
                                       (size_t) dim.y,
                                       8, sizeof(gool::argb) * 8,
                                       (size_t) bmp->stride(),
                                       colorSpace, flags, provider,
                                       0, false, kCGRenderingIntentDefault);
          if(cgi)
            bmp->cg_image = cgi;
        } else {
            //bmp->cg_image = bmp->cg_image;
        }
    }
    
    return bmp->cg_image;
}


html::clipboard::data* osx_get_clipboard_data(NSPasteboard* pb)
{
  html::clipboard::data* pdata = new html::clipboard::data();
  NSString* text = [pb stringForType:NSStringPboardType];
  if( text ) {
    ustring data = cvt(text);
    pdata->add(new html::clipboard::text_item(data));
  }
  
  @autoreleasepool {
    
    NSArray *classes = [NSArray arrayWithObject:[NSURL class]];
    NSDictionary *options = [NSDictionary dictionaryWithObject:
                             [NSNumber numberWithBool:YES] forKey:NSPasteboardURLReadingFileURLsOnlyKey];
    
    NSArray *fileURLs = [pb readObjectsForClasses:classes options:options];
    array<ustring> urls;
    for( NSUInteger n = 0; n < fileURLs.count; ++n ) {
      NSURL * url = [fileURLs objectAtIndex:n];
      if(url) {
        ustring tt = cvt([ url path ]);
        urls.push( ustring::format( W("file://%s"),tt.c_str() ) );
      }
    }
    pdata->add(new html::clipboard::file_item(urls));
  }
  
  @autoreleasepool {
     NSArray *classes = [NSArray arrayWithObject:[NSImage class]];
     NSDictionary *options = [NSDictionary dictionary];
     BOOL ok = [pb canReadObjectForClasses:classes options:options];
     if (ok) {
        NSArray *objects = [pb readObjectsForClasses:classes options:options];
        NSImage *image = [objects objectAtIndex:0];
        auto cgimg = [image CGImageForProposedRect:NULL context:[ NSGraphicsContext currentContext] hints:NULL];
        handle<gool::bitmap> pbm = gool::bitmap::create(cgimg, true);
        pdata->add(new html::clipboard::image_item(pbm));
     }
  }
  return pdata;
}


bool osx_set_clipboard_data(html::clipboard::data *cbdata, NSPasteboard *pasteboard )
{
  [pasteboard clearContents];
  
  int cnt = 0;
  for (int n = 0; n < cbdata->items.size(); ++n) {
    switch (cbdata->items[n]->data_type) {
      case html::clipboard::cf_text: {
        html::clipboard::text_item *it = cbdata->items[n].ptr_of<html::clipboard::text_item>();
        NSString* str = cvt(it->val());
        [pasteboard setString: str forType:NSStringPboardType];
        ++cnt;
      } break;
      case html::clipboard::cf_html: {
        html::clipboard::html_item *it = cbdata->items[n].ptr_of<html::clipboard::html_item>();
        ustring us = u8::cvt(it->val());
        [pasteboard setString: cvt(us) forType:NSPasteboardTypeHTML];
        ++cnt;
      } break;
      case html::clipboard::cf_picture: {
         html::clipboard::image_item *it = cbdata->items[n].ptr_of<html::clipboard::image_item>();
         CGImageRef cgimg = osx_ttb_image( const_cast<gool::image*>(it->image.ptr())->get_bitmap(nullptr,it->image->dim()), nullptr );
         NSImage* nsimg = [[NSImage alloc] initWithCGImage:cgimg size:NSZeroSize];
         if(nsimg) {
           NSArray *copiedObjects = [NSArray arrayWithObject:nsimg];
           [pasteboard writeObjects:copiedObjects];
         }
         ++cnt;
         } break;
      case html::clipboard::cf_file: {
        html::clipboard::file_item *it = cbdata->items[n].ptr_of<html::clipboard::file_item>();
        @autoreleasepool {
          NSMutableArray * fileList = [NSMutableArray new];
          for( auto str : it->filenames )
            [fileList addObject: cvt(str)];
          [pasteboard declareTypes:[NSArray arrayWithObject:NSFilenamesPboardType] owner:nil];
          [pasteboard setPropertyList:fileList forType:NSFilenamesPboardType];
        }
        ++cnt;
      } break;
      default:
        break;
    }
  }
  return cnt > 0;
}

namespace html {

  void clipboard::empty()
  {
    [[NSPasteboard generalPasteboard] clearContents];
  }

  void clipboard::available_formats( tool::function<bool(clipboard::clipboard_format cf)> cb )
  {
    //if(!juce::SystemClipboard::getTextFromClipboard().isEmpty())
    //  cb.has(cf_text);
      
      @autoreleasepool {
          NSArray* types = [[NSPasteboard generalPasteboard] types];
          for(NSString* ty in types) {
              if( [ty isEqualToString: NSStringPboardType] ) {
                  if(!cb(cf_text)) return;
              }
              if( [ty isEqualToString: NSHTMLPboardType] ) {
                  if(!cb(cf_html)) return;
              }
              if( [ty isEqualToString: NSTIFFPboardType] ) {
                  if(!cb(cf_picture)) return;
              }
              if( [ty isEqualToString: NSURLPboardType] ) {
                  if(!cb(cf_hyperlink)) return;
              }
              if( [ty isEqualToString: NSFilenamesPboardType] || [ty isEqualToString: NSFilesPromisePboardType] ) {
                  if(!cb(cf_file)) return;
              }
          }
      }

//      NSArray *classArray = [NSArray arrayWithObject:[NSImage class]];
//      NSDictionary *options = [NSDictionary dictionary];
//      BOOL ok = [pasteboard canReadObjectForClasses:classArray options:options]
      
      
    //  NSArray *supportedTypes =
    //  [NSArray arrayWithObjects: NSRTFDPboardType, NSRTFPboardType, NSStringPboardType, nil];
    //  NSString *bestType = [[NSPasteboard generalPasteboard]
    //                        availableTypeFromArray:supportedTypes];
      
  }

  bool clipboard::set_text(wchars text)
  {
      empty();
      return _set_text(text);
  }
  
  void clipboard::set(view& v, selection_ctx& sctx)
  //void clipboard::set(view& v, bookmark start, bookmark end)
  {
     array<wchar> text;
     array<char>  html;
     text_cf(v, sctx, text);
     html_cf(v, sctx, html);
     empty();
     _set_html(html());
     _set_text(text());
  }

  bool clipboard::get(ustring& data)
  {
      @autoreleasepool {
          NSString* text = [[NSPasteboard generalPasteboard] stringForType:NSPasteboardTypeString];
          if( !text )
              return false;
          data = cvt(text);
          return true;
      }
  }

  uint clipboard::get_sequence_num()
  {
    static uint fix_me = 0;
    return ++fix_me;
  }

  bool clipboard::get_html(array<byte>& html, string& url )
  {
/*    @autoreleasepool {
      NSString* text = [[NSPasteboard generalPasteboard] stringForType:NSPasteboardTypeHTML];
      if( !text )
        return false;
      ustring data = cvt(text);
      data.utf8(html);
      return true;
    } */
    @autoreleasepool {
      
      NSPasteboard *pasteboard = [NSPasteboard generalPasteboard];
      
      NSString* nshtml = [pasteboard stringForType:NSPasteboardTypeHTML];
      if( !nshtml )
        return false;
    
      
      html = u8::cvt(cvt(nshtml)).chars_as_bytes();
      
      NSData *data = [pasteboard dataForType:@"Apple Web Archive pasteboard type"];
      if(data) {
        NSDictionary *plist = [NSPropertyListSerialization propertyListWithData:data
                                                                        options:NSPropertyListImmutable
                                                                         format:nil
                                                                          error:nil];
        
        NSDictionary *values = [plist objectForKey:@"WebMainResource"];
        NSString *nsurl = [values objectForKey:@"WebResourceURL"];
        
        url = cvt(nsurl);
      }
    }
    return true;
  }

  clipboard::html_item* clipboard::html_item::from_pasteboard(void* nspasteboard)
  {
    clipboard::html_item* phi = new clipboard::html_item();
    if(!phi) return nullptr;
    
    @autoreleasepool {
      
      NSPasteboard *pasteboard = nspasteboard ? (NSPasteboard *) nspasteboard : [NSPasteboard generalPasteboard];
      
      NSString* nshtml = [pasteboard stringForType:NSPasteboardTypeHTML];
      if( !nshtml ) {
        delete phi;
        return nullptr;
      }
      
      ustring html = cvt(nshtml);
      phi->val = u8::cvt(html);
      
      NSData *data = [pasteboard dataForType:@"Apple Web Archive pasteboard type"];
      if(data) {
        NSDictionary *plist = [NSPropertyListSerialization propertyListWithData:data
                                                                        options:NSPropertyListImmutable
                                                                         format:nil
                                                                          error:nil];
        if(plist) {
          NSDictionary *values = [plist objectForKey:@"WebMainResource"];
          NSString *url = [values objectForKey:@"WebResourceURL"];
          phi->url = cvt(url);
        }
      }
    }
    return phi;
    
    
  }
  
  bool clipboard::_set_text( wchars text )
    {
      @autoreleasepool {
        NSPasteboard *pasteboard = [NSPasteboard generalPasteboard];
        NSString* str = cvt(text);
        [pasteboard setString: str forType:NSStringPboardType];
        return true;
      }
    }
  void clipboard::_set_html( chars html )
  {
    @autoreleasepool {
      NSPasteboard *pasteboard = [NSPasteboard generalPasteboard];
      ustring us = u8::cvt(html);
      [pasteboard setString: cvt(us) forType:NSPasteboardTypeHTML];
    }
  }
  
  void clipboard::_set_files(const array<ustring>& filenames) {

    @autoreleasepool {
      NSMutableArray * fileList = [NSMutableArray new];
      for( auto str : filenames )
        [fileList addObject: cvt(str)];
      
      NSPasteboard *pasteboard = [NSPasteboard generalPasteboard];
      
      [pasteboard declareTypes:[NSArray arrayWithObject:NSFilenamesPboardType] owner:nil];
      [pasteboard setPropertyList:fileList forType:NSFilenamesPboardType];
    }
  }

 
  void clipboard::_set_image(const gool::image* data)
  {
    NSPasteboard *pasteboard = [NSPasteboard generalPasteboard];

    CGImageRef cgimg = osx_ttb_image( const_cast<gool::image*>(data)->get_bitmap(nullptr,data->dim()), nullptr );
    NSImage* nsimg = [[NSImage alloc] initWithCGImage:cgimg size:NSZeroSize];
    if(nsimg) {
      NSArray *copiedObjects = [NSArray arrayWithObject:nsimg];
      [pasteboard writeObjects:copiedObjects];
    }
  }
  void clipboard::_set_link(wchars url,wchars caption)
  {
  }
  
  bool clipboard::get_image(handle<image>& data) {
      
    NSPasteboard *pb = [NSPasteboard generalPasteboard];
      
    @autoreleasepool {
       NSArray *classes = [NSArray arrayWithObject:[NSImage class]];
       NSDictionary *options = [NSDictionary dictionary];
       BOOL ok = [pb canReadObjectForClasses:classes options:options];
       if (ok) {
          NSArray *objects = [pb readObjectsForClasses:classes options:options];
          NSImage *image = [objects objectAtIndex:0];
          auto cgimg = [image CGImageForProposedRect:NULL context:[ NSGraphicsContext currentContext] hints:NULL];
          handle<gool::bitmap> pbm = gool::bitmap::create(cgimg, true);
          data = pbm;
          return true;
       }
    }
    return false;
  }
  
  void clipboard::set(const gool::image* data) {
    empty();
    _set_image(data);
  }
  
  bool clipboard::get_link(ustring& url, ustring& caption )
  {
    @autoreleasepool {
      NSPasteboard *pb = [NSPasteboard generalPasteboard];
      NSURL * nsurl =  [NSURL URLFromPasteboard:pb];
      if( nsurl ) {
        url = cvt([nsurl absoluteString]);
        caption = cvt([pb stringForType:NSPasteboardTypeURL]);
        if(caption.is_undefined())
          caption = cvt([pb stringForType:NSPasteboardTypeString]);
        return true;
      }
    }
    return false;
  }

  bool clipboard::get_files(array<ustring> &filenames)
  {
    @autoreleasepool {
        
      NSPasteboard *pb = [NSPasteboard generalPasteboard];
        
      NSArray *classes = [NSArray arrayWithObject:[NSURL class]];
      NSDictionary *options = [NSDictionary dictionaryWithObject:
                              [NSNumber numberWithBool:YES] forKey:NSPasteboardURLReadingFileURLsOnlyKey];
        
      NSArray *fileURLs = [pb readObjectsForClasses:classes options:options];
      for( NSUInteger n = 0; n < fileURLs.count; ++n ) {
        NSURL * url = [fileURLs objectAtIndex:n];
        if(url) {
            ustring tt = cvt([ url path ]);
            filenames.push( tt );
        }
      }
      
    }
    return true;
  }
}

#endif
