#ifndef __html_window_h__
#define __html_window_h__

#include "tool/tool.h"
#include "gool/gool.h"

#include "html-style.h"
#include "html-pump.h"

typedef bool data_loader(html::pump::request *rq_on_heap);

namespace html {
  using namespace tool;
  using namespace gool;

  struct element;
  class view;

  enum ACTIVATE_MODE {
    INACTIVE,
    ACTIVATED,
    MOUSE_ACTIVATED,
  };

  enum VIEW_TYPE {
    VIEW_PAGES,
    VIEW_WINDOW,
    VIEW_DIALOG,
  };

  enum DO_EVENT_MANNER {
    DO_EVENT_WAIT,
    DO_EVENT_NOWAIT,
    DO_EVENT_ALL,
    DO_EVENT_UNTIL_MOUSE_UP,
    DO_EVENT_ONLY_IO
  };

  enum BLUR_BEHIND {
    BLUR_BEHIND_NONE        = 0,
    BLUR_BEHIND_ULTRA_DARK  = 1,
    BLUR_BEHIND_DARK        = 2,
    BLUR_BEHIND_LIGHT       = 3,
    BLUR_BEHIND_ULTRA_LIGHT = 4,
    BLUR_BEHIND_CUSTOM      = 5,
    BLUR_BEHIND_SIZE,
  };

  extern enum_item_def blur_edef[BLUR_BEHIND_SIZE];

  class application : public gool::application {
    typedef gool::application super;

  public:
    application() { html::init_symbols(true); }
    virtual ~application() {}
    virtual void final_release() override;

    static style_bag &stock_styles();
    static void       set_stock_styles(chars text, bool append);
    static void       destroy_stock_styles();

    static data_loader *    primordial_loader;
    static handle<document> stock_styles_doc;
  };

  inline application *html_app() {
    return static_cast<application *>(gool::app());
  }


  enum ELEMENT_WINDOW_MODE {
    ELEMENT_WINDOW_UNKNOWN,
    ELEMENT_WINDOW_AUTO,     // window gets created if needed
    ELEMENT_WINDOW_ATTACHED, // window always created, attached (moves in sync
                             // with host window)
    ELEMENT_WINDOW_DETACHED, // window always created, detached, live by its own
    ELEMENT_WINDOW_DETACHED_TOPMOST,
    ELEMENT_WINDOW_DOM_ATTACHED, // window always created, attached to element's
                                 // position
  };

  enum FRAME_TYPE {
    STANDARD,
    LAYERED,
    SOLID,
    SOLID_WITH_SHADOW,
    STANDARD_EXTENDED,
  };

  struct iwindow : public virtual  resource
  {
  protected:
    HWINDOW     _hwnd;
  public:
    WINDOW_TYPE type;
    rect        invalid; // invalid area
    tristate_v  _is_layered;
    bool        is_painting = false;
    //tristate_v  is_awaiting_update_layered;
    tristate_v  _is_visible; // true if content is visible (not closed/hiden or
                            // iconified)
    BLUR_BEHIND _blur_behind;
    tristate_v  _resizeable;

    size dim;
    point pos;

    handle<gool::application> app;

    ELEMENT_WINDOW_MODE       windowing_mode, used_windowing_mode;
    int                       animation_count;
    uint                      frame_updater_counter = 0;

    iwindow();
    virtual ~iwindow();

    virtual void init_type(WINDOW_TYPE tp) { type = tp; }

    virtual graphics *surface() { return 0; }
    virtual void      reset_surface() {}
    virtual point     screen_pos();
    virtual point     client_screen_pos();
    virtual point     cursor_pos();
    virtual size      client_dim();
    virtual size      window_dim();

    virtual bool uses_physical_size() const { return false; } // osx specific

    virtual rect client_screen_place() { return rect(client_screen_pos(), client_dim()); }
    virtual rect screen_place() { return rect(screen_pos(), window_dim()); }

    virtual void invalidate(const rect &area);
    virtual void request_render() { assert(false); }

    virtual void refresh(const rect &area) {
      rect rc = area; if (rc.empty()) rc = rect(client_dim());
      invalidate(rc);
    }
    virtual void refresh() { 
      invalidate(rect(client_dim())); 
    }

    virtual void update();

    virtual bool draw(void *dc, rect client_rc) {
      if (is_layered())
        return render_layered(dc,rect(client_dim()));
      else
        return render(dc, client_rc);
    }

    virtual bool render(void *dc, rect client_rc) {
      assert(false);
      return false;
    }

    virtual bool render_layered(void *dc, rect client_rc) {
      //is_awaiting_update_layered = false;
      //refresh();
      return false;
    }

    virtual void attached(HWINDOW hw) {} // got HWINDOW
    virtual void detached(HWINDOW hw) {} // lost HWINDOW

    virtual HWINDOW get_hwnd() const { return _hwnd; }
    virtual void    set_hwnd(HWINDOW hw) {
      if (hw) {
        //assert(!_hwnd);
        _hwnd = hw;
        attached(hw);
      }
      else if(_hwnd) {
        auto t = _hwnd;
        _hwnd = nullptr;
        detached(t);
      }
    }

    virtual void    dismiss(){}; // has to be implemented for popups.

    virtual bool is_airborn() const {
      return is_popup() || is_tool() || is_tooltip();
    }
    virtual bool is_popup() const { return type == POPUP_WINDOW; }
    virtual bool is_tool() const { return type == TOOL_WINDOW; }
    virtual bool is_tooltip() const { return type == TOOLTIP_WINDOW; }
    virtual bool is_child() const { return type == CHILD_WINDOW; }

    virtual element *root() const = 0; //{ return _root; } // root element hosted by the window
    virtual void     root(element *r) { assert(false); /*_root = r;*/ }                // root element hosted by the window

    virtual bool is_layered() const { return _is_layered.val(0) != 0; }

    virtual void set_layered(bool onoff);

    virtual void update_window_frame();

    virtual bool is_direct() const { return false; } // not a directcomposition window

    virtual bool         set_icon(gool::image *pimg) { return false; }
    virtual gool::image *get_icon() const { return nullptr; }

    enum ANIMATE_WINDOW_FLAGS {
      AWF_HOR_POSITIVE = 0x00000001,
      AWF_HOR_NEGATIVE = 0x00000002,
      AWF_VER_POSITIVE = 0x00000004,
      AWF_VER_NEGATIVE = 0x00000008,
      AWF_CENTER       = 0x00000010,
      AWF_HIDE         = 0x00010000,
      AWF_ACTIVATE     = 0x00020000,
      AWF_SLIDE        = 0x00040000,
      AWF_BLEND        = 0x00080000,
      AWF_ROLL         = 0x00000000,
    };

    virtual bool show(uint animate_window_flags, uint ms = 200);

    virtual rect screen_workarea(rect rc = rect()) const;

    virtual element *anchor() const { return 0; }
    virtual element *pfocus() const { return 0; }

    virtual void anchor(element *el) {}

    virtual void pfocus(element *el) {}

    virtual bool is_window_visible();

    virtual bool       set_frame_type(FRAME_TYPE on) { return false; }
    virtual FRAME_TYPE get_frame_type() const { return STANDARD; }
    virtual bool       set_resizeable(bool on) { _resizeable = on; return false; }
    virtual bool       get_resizeable() const { return _resizeable.val(0) != 0; }

    virtual bool       set_resize_policy(int layer_resize_policy) { return false; }
    virtual int        get_resize_policy() const { return 0; }

    virtual bool       set_topmost(bool on);
    virtual bool       get_topmost();
    virtual bool       activate(bool and_bring_to_front) { return false; }

    virtual bool       set_minimizable(bool on) { return false; }
    virtual bool       get_minimizable() { return false; }
    virtual bool       set_maximizable(bool on) { return false; }
    virtual bool       get_maximizable() { return false; }

    virtual bool set_blurbehind(BLUR_BEHIND bb) {
      if (_blur_behind != bb) {
        _blur_behind = bb;
        notify_media_change();
        return true;
      }
      return false;
      
    }
    virtual BLUR_BEHIND get_blurbehind() const { return _blur_behind; }

    virtual void set_transparency(bool on_off) {}
    virtual bool get_transparency() const { return false; }

#if defined(WINDOWS)
    virtual void set_enabled(bool on_off);
    virtual bool is_enabled() const;
#elif defined(WINDOWLESS)
    virtual void set_enabled(bool on_off);
    virtual bool is_enabled() const;
#else
    virtual void set_enabled(bool on_off) {}
    virtual bool is_enabled() const { return true; }
#endif

    //virtual void detach() {} // detach it from windows collection

    virtual void on_visibility_changed(bool visible) {}
    virtual void notify_media_change() { ; }

    bool check_visibility(tristate_v force = tristate_v()) {
      bool p_visibility = !!_is_visible.val(0);
      bool n_visibility =
          force.is_defined() ? (!!force.val(0)) : is_window_visible();
      if (p_visibility != n_visibility) {
        _is_visible = n_visibility;
        on_visibility_changed(n_visibility);
      }
      return n_visibility;
    }

    virtual int get_system_metrics(int m) { assert(false); return 10; } // GetSysteMetrics but DPI aware
  };

  struct window_frame_updater {
    window_frame_updater(iwindow* pw);
    ~window_frame_updater();
    handle<iwindow> that;
  };


  enum AIRBORN_TYPE {
    AIRBORN_INPLACE,
    AIRBORN_WINDOW_VIEW,
    AIRBORN_WINDOW_SCREEN
  };

  struct airborn_ctx : public resource {
    AIRBORN_TYPE              type;
    point                     pos;
    gool::geom::size_t<int_v> dim;
  };

  class popup : public iwindow {
    friend class window;

  protected:
    size                 _dim;
    handle<element>      _root;
    handle<element>      _anchor;
    weak_handle<element> _pfocus; // if any, saved focus

    popup(WINDOW_TYPE wt = POPUP_WINDOW) { init_type(wt); }
    virtual ~popup() {}

  public:

    virtual bool is_airborn() const { return true; }

    virtual element *root() const override; // root element hosted by the window
    virtual void root(element *r) override; // root element hosted by the window

    virtual element *anchor() const override;
    virtual element *pfocus() const override;

    virtual void anchor(element *el) override;
    virtual void pfocus(element *el) override;
  };

  struct screen_info {
    rect    monitor;
    rect    workarea;
    bool    is_primary;
    ustring device_name;
    size    dpi = size(96,96);
    int_v   sid;
  };

  int  number_of_screens();
  int  screen_of(iwindow *pw);
  bool get_screen_info(int n, screen_info &si);
#ifdef SCREENSHOT_SUPPORT
  bitmap *get_screen_shot(int n);
#endif

} // namespace html

struct window_params
{
  handle<html::view> parent;
  tool::tristate_v   owns_vm;
  tool::tristate_v   debug_mode;
  tool::tristate_v   rtl;
#ifdef SCITERJS
  qjs::hvalue        window_parameters;
  qjs::hvalue        window_obj;
#else
  tool::value        window_parameters;
#endif
  html::WINDOW_TYPE  window_type;
  html::WINDOW_STATE window_state = html::WINDOW_STATE_NA;

  tool::tristate_v direct_window;
  bool             is_detached = false; // detached window - it will not close in sync with its parent.

  tool::ustring         caption;
  tool::string          url;
  tool::bytes           data;
  html::point_v_t pos;
  html::dim_v_t   dim;
  int             alignment = 0;
  tool::int_v     screen_no;
  bool            client_coordinates = false;

  bool is_glassy = false;
  bool is_transparent = false;
  bool is_main = false;
  HWINDOW owner = NULL;    // foreign owner window

#ifdef WINDOWS
  void *delegate = nullptr;
  void *delegateParam = nullptr;

  mutable UINT window_style = 0;
  mutable UINT window_style_ex = 0;

  window_params(html::WINDOW_TYPE ft = html::FRAME_WINDOW) :window_type(ft) {
    init_window_styles();
  }

  void set(html::WINDOW_TYPE wt) {
    window_type = wt;
    init_window_styles();
  }

  void init_window_styles() const {
    switch (window_type) {
    case html::FRAME_WINDOW:
      window_style = WS_OVERLAPPEDWINDOW;
      window_style_ex = WS_EX_APPWINDOW | WS_EX_WINDOWEDGE;
      break;
    case html::DIALOG_WINDOW:
      window_style = WS_DLGFRAME | WS_POPUP | WS_CAPTION | WS_SYSMENU | WS_THICKFRAME;
      window_style_ex = WS_EX_DLGMODALFRAME | WS_EX_WINDOWEDGE;
      break;
    case html::TOOL_WINDOW:
      window_style = WS_OVERLAPPEDWINDOW;
      window_style_ex = WS_EX_TOOLWINDOW | WS_EX_WINDOWEDGE;
      break;
    case html::POPUP_WINDOW:
      window_style = WS_POPUP;
      window_style_ex = WS_EX_TOPMOST;
      break;
    case html::CHILD_WINDOW:
      window_style = WS_CHILD;
      window_style_ex = WS_EX_CLIENTEDGE;
      break;
    }
  }

#else
  window_params(html::WINDOW_TYPE ft = html::FRAME_WINDOW) : window_type(ft) {}
  void set(html::WINDOW_TYPE wt) { window_type = wt; }
#endif
};


#endif
