#ifndef __dom_view_h__
#define __dom_view_h__

#include "tool/tool.h"
#include "gool/gool.h"
#include "html-dom.h"
#include "html-document.h"
#include "html-pump.h"
#include "html-behavior.h"
#include "html-window.h"

namespace html {
  using namespace tool;
  using namespace gool;

#define MOUSE_DRAG_THRESHOLD 3 /*dips*/

  extern bool use_platform_theming;
  
  STYLE_CHANGE_TYPE changes(const style *s1, const style *s2);
  STYLE_CHANGE_TYPE changes(const style *s1);

#if defined(SCITER)
  struct posted_event : event_behavior {
    tis::value cbf;
    posted_event() : cbf(0) {}
    posted_event(const event_behavior &eb) : event_behavior(eb), cbf(0) {}
    posted_event(const posted_event &eb) : event_behavior(eb), cbf(eb.cbf) {}
    inline bool operator==(const posted_event &rs) const {
      return cbf == rs.cbf && event_behavior::operator==(rs);
    }
    inline bool operator!=(const posted_event &rs) const {
      return cbf != rs.cbf || event_behavior::operator!=(rs);
    }
    virtual bool  is_posted() const override { return true; }
  };
#else
  struct posted_event : public event_behavior {

    qjs::hvalue cbf;

    posted_event() {}
    posted_event(const event_behavior &eb) : event_behavior(eb) {}
    posted_event(const posted_event &eb) : event_behavior(eb), cbf(eb.cbf) {}
    inline bool operator==(const posted_event &rs) const { return cbf == rs.cbf && event_behavior::operator==(rs); }
    inline bool operator!=(const posted_event &rs) const { return cbf != rs.cbf || event_behavior::operator!=(rs); }
    virtual bool is_posted() const override { return true; }
  };
#endif

  struct update_queue {
    int  nesting;
    int  update_rq;
    bool require_screen_update;
    bool inside_update;
    bool require_touch_mouse;

    array<helement> _set; // queue of layout root elements with invalid content
                          // dimensions (width:... or else)
    array<helement> _working_set; // working set

    struct qitem {
      helement          el;
      STYLE_CHANGE_TYPE sct;
    };
    array<qitem> _queue;

    struct spitem {
      weak_helement el;
      tristate_v    to_top; // if is_undefined() then below are valid
      point         pos;
      bool          smooth;
      bool          allow_out_of_bounds;
    };
    array<spitem> _spqueue;

    // helement ensure_visible;      // to ensure the element visibility after
    // update  bool     ensure_visible_to_top; // where  ENSURE_VISIBLE_MANNER
    //       ensure_visible_manner; // and how

    void queue(view *v, element *b,
               STYLE_CHANGE_TYPE ut); // queue add to update set
    void add(view *v, element *b, STYLE_CHANGE_TYPE ut);    // add to update set
    void do_add(view *v, helement b, STYLE_CHANGE_TYPE ut); // add to update set
    void update(view *v);
    // void update_model(view* v, element *root);
    void clear(element *b);
    void clear() {
      _set.clear();
      _queue.clear();
    }
    void reduce_set(view *v);
    bool is_empty() const { return _set.size() == 0 && _queue.size() == 0; }
    bool is_covered_by(element *b, helement &by);

    void mark_changing_dimension(view *v, element *b);
    void mark_invalid_model(view *v, element *b);

    void request_scroll_pos(helement el, point pos, bool smooth,
                            bool allow_out_of_bounds);
    void request_ensure_visible(helement el, bool to_top, bool smooth);

    void foreach_element(function<bool(element *el)> f) {
      _set.each(f);
      _queue.each([f](const qitem &qi) -> bool { return f(qi.el); });
      _spqueue.each([f](const spitem &qi) -> bool { return f(qi.el); });
    }

    update_queue()
        : nesting(0), update_rq(0), require_screen_update(false),
          require_touch_mouse(false), inside_update(false) {}
  };

  struct current_view_state {
    current_view_state(view *that);
    ~current_view_state();
    void         drop_current();
    handle<view> self;
    handle<view> previous_self;
  };

  class updater : public current_view_state {
    helement over;
    int      level;
    bool     discard_update;
    bool     root;
    updater();
    updater(const updater &s);
    updater &operator=(const updater &s);

  public:
    updater(view *v, element *mouse_over, bool passive, bool is_root);
    ~updater();
  };

  enum OUTPUT_SUBSYTEMS {
    OT_DOM = 0,     // html parser & runtime
    OT_CSSS,        // csss! parser & runtime
    OT_CSS,         // css parser
    OT_TIS,         // TIS parser & runtime
    OT_JS = OT_TIS, // JS parser & runtime
  };
  enum OUTPUT_SEVERITY {
    OS_INFO,
    OS_WARNING,
    OS_ERROR,
  };

  struct selection_ctx;

  struct url_data : public resource // data item loaded into the view
  {
    string             url;
    string             content_url;
    RESOURCE_DATA_TYPE data_type;
    string             headers;
    array<byte>        data;
    string             data_content_type;     // mime type
    string             data_content_encoding; // content-encoding

    static url_data *from(pump::request *rq);
  };

  struct view_callback : public resource {
    virtual bool load_data(view *pv, pump::request *rq)                    = 0;
    virtual bool data_loaded(view *pv, pump::request *rq)                  = 0;
    virtual ctl *create_behavior(view *pv, element *b, const string &name) = 0;
    virtual void post_notify(view *pv, uint_ptr wparam, uint_ptr lparam)   = 0;
    virtual bool request_keyboard(html::view *pv, uint mode) { return false; }
    virtual void invalidate_rect(html::view *pv, const rect& rc) {}

    virtual void graphics_critical_failure(view *pv) {}
    // virtual bool  host_callback( int channel, const tool::value& p1, const
    // tool::value& p2, tool::value& r ) = 0;
    handle<view_callback> next; // they can be chained
  };

  struct view_debug_output : virtual public resource {
    virtual void print(uint subs, uint sev, wchars msg) = 0;
    virtual bool is_debug_peer() const { return false; }
  };

  namespace behavior {
    struct highlighted_ctl;
  }

  class view : public virtual resource,
               public iwindow,
               public weakable,
               public event_handler,
               public pump,
               public gool::resolution_provider {
    friend class z_ctx;
    friend struct update_queue;
    friend class updater;
    friend struct element;
    friend struct document;
    template <typename EVT> friend struct traverser;

  private:
    bool load_html(bytes html, const string &url = string(), uint encoding = 0);
    // bool load_html(chars html, const string& url = string(), uint encoding =
    // 0) { return load_html(bytes((const
    // byte*)html.start,html.length),url,encoding); }
  protected:
    view(const window_params &params);

  public:
    virtual ~view();
    virtual graphics *surface() = 0;

    virtual void finalize() override {
      handle<gool::application> app = this->app; // application must be destroyed only after the view destructors
      resource::finalize();
    }

    document *doc() const;

    virtual element *root() const override {
      return doc();
    } // root element hosted by the window
    virtual void root(element *r) override {
      assert(false);
    } // root element hosted by the window

    virtual bool is_printing() const { return false; }

    static view *get_current(); // returns current view (for the GUI thread)
                                // processing UI message
    static void  debug_printf(uint subsystem, uint severity, const char *utf8, ...);
    virtual void debug_print(uint subsystem, uint severity, wchars msg);
    virtual void debug_print(uint subsystem, uint severity, chars msg) { debug_print(subsystem, severity, u8::cvt(msg)()); }
    virtual void do_debug_print(uint subsystem, uint severity, wchars msg);

    // metrics:

    virtual void attached(HWINDOW hw) override;
    virtual void detached(HWINDOW hw) override;

    template <typename F> static void each(F f) {
      for (int n = all.size() - 1; n >= 0; --n)
        f(all(n));
    }

    template <typename F> static bool each_if(F f) {
      for (int n = all.size() - 1; n >= 0; --n)
        if (f(all(n)))
          return true;
      return false;
    }

    virtual bool can_use_feature(uint flag) { return false; }
    
    virtual void refresh() { this->refresh(rect(client_dim())); }
    virtual void refresh(const rect &area) { iwindow::refresh(area); }

    bool request_render_requested = false;
    virtual void request_render();

    virtual size dimension() const override {
      return const_cast<view *>(this)->client_dim();
    }

    // document loading
    bool load_url(const string &url, bool now = false);

    bool load(handle<request> rq);

    virtual void start(const window_params &params);
    virtual void stop();

    // initialization
    enum UNLOAD_REASON {
      UNLOAD_BY_CHROME, // user clicked close button on window
      UNLOAD_BY_CODE,   // view.close() issued
      UNLOAD_BY_LOAD,   // document unload-old/load-new
    };
    virtual bool ask_unload(document *d, UNLOAD_REASON from_code);
    virtual bool on_before_unload(document *d); // document is still operational but will be closed 
    virtual value get_retval() { return value(); }
    virtual bool on_unload(document *d); // final 
    virtual void on_load_start(document *d) {
      event_behavior evt(d, d, DOCUMENT_CREATED, 0);
      evt.data = value(d->uri().src);
      send_behavior_event(evt);
      //??? if( d->parent )
      //???   d->parent->on(*this,evt);
      //??? else
      //???   on(*this,d,evt);
    }

    virtual void purge_deleted_elements() {}
    //   happens after loading new document but before on_document_complete
    virtual void on_load_end(document *d, bool existing_doc_replaced) {
      d->state.ready(true);
      event_behavior evt(d, d, DOCUMENT_READY, existing_doc_replaced);
      evt.data = value(d->uri().src);
      post_behavior_event(evt);
    }

    virtual void measure_document() {
      if (doc()) doc()->measure(*this, dimension());
    }

    //   happens after loading new document but before on_document_complete
    virtual void on_dom_parsed(document *d) {
      event_behavior evt(d, d, DOCUMENT_PARSED, 0);
      evt.data = value(d->uri().src);
      send_behavior_event(evt);
    }

    virtual void on_document_complete(document *d) {
      {
        event_behavior evt(d, d, DOCUMENT_COMPLETE, 1);
        evt.data = value(d->uri().src);
        post_behavior_event(evt);
      }
#if defined(SCITERJS)       
      {
        event_behavior evt(d, d, LOAD_SUCCEEDED, 1);
        evt.data = value(d->uri().src);
        post_behavior_event(evt);
      }
#endif
    }
    virtual void
    on_root_doc_parsed(document *d) // first root doc parsed into the view
    {
      if (this->type == FRAME_WINDOW || this->type == DIALOG_WINDOW ||
          this->type == TOOL_WINDOW || this->type == POPUP_WINDOW) 
      {

        frame_attributes fatts;

        required_frame_attributes(fatts);
        window_frame_updater _(this);

        if (fatts.frame_type.is_defined())
          set_frame_type(FRAME_TYPE(fatts.frame_type.val(0)));
       
        if (fatts.resizeable.is_defined()) {
          bool resizeable = fatts.resizeable.val(0) != FALSE;
          set_resizeable(resizeable);
          set_minimizable(fatts.minimizable.val(resizeable));
          set_maximizable(fatts.maximizable.val(resizeable));
        }
        else {
          if(fatts.minimizable.is_defined()) set_minimizable(fatts.minimizable.val(0));
          if(fatts.maximizable.is_defined()) set_maximizable(fatts.maximizable.val(0));
        }
            if (fatts.glassy.is_defined())
          set_blurbehind((BLUR_BEHIND)fatts.glassy.val(0));

        if (fatts.min_width.is_defined() && fatts.min_height.is_defined()) {
          gool::size sw_size = screen_workarea().size();
          gool::size w_size;
          w_size.x = dips(*this, doc(), fatts.min_width, sw_size).width();
          w_size.y = dips(*this, doc(), fatts.min_height, sw_size).height();
          this->set_min_size(w_size);
        }

        if (fatts.max_width.is_defined() && fatts.max_height.is_defined()) {
          gool::size sw_size = screen_workarea().size();
          gool::size w_size;
          w_size.x = dips(*this, doc(), fatts.max_width, sw_size).width();
          w_size.y = dips(*this, doc(), fatts.max_height, sw_size).height();
          this->set_max_size(w_size);
        }

        if (fatts.width.is_defined() && fatts.height.is_defined()) {
          gool::size sw_size = screen_workarea().size();
          gool::size w_size;
          w_size.x = pixels(*this, doc(), fatts.width,sw_size).width();
          w_size.y = pixels(*this, doc(), fatts.height, sw_size).height();
          rect wrc = rect(screen_pos(), w_size);
          this->move_window(wrc);
          this->window_was_moved = false;
        }
        
        static name_or_symbol attn_icon = name_or_symbol("window-icon");
        string                icon_url  = doc()->atts.get_string(attn_icon);
        if (icon_url.is_defined()) set_icon_url(icon_url);

        html::element *title =
            html::find_first(*this, d, WCHARS("head>title"), false);
        if (title) {
          ustring t = title->get_text(*this);
          set_window_title(t);
        }
      }
    }

    virtual void on_root_doc_loaded(document *d) { 
      string initial_state = d->atts.get_string(attr::a_window_state);
      if (initial_state.is_undefined()) return;
      WINDOW_STATE ws = WINDOW_SHOWN;
      if (initial_state == CHARS("hidden")) ws = WINDOW_HIDDEN;
      else if (initial_state == CHARS("shown")) ws = WINDOW_SHOWN;
      else if (initial_state == CHARS("maximized")) ws = WINDOW_MAXIMIZED;
      else if (initial_state == CHARS("minimized")) ws = WINDOW_MINIMIZED;
      else if (initial_state == CHARS("full-screen")) ws = WINDOW_FULL_SCREEN;
      this->set_window_state(ws);
    }

    virtual void update_window_frame() override {
      iwindow::update_window_frame();
      notify_media_change();
      on_size(client_dim());
    }

    virtual void set_icon_url(string icon_url) {
      icon_url                  = html::combine_url(doc()->uri().src, icon_url);
      handle<pump::request> prq = new pump::request(icon_url, DATA_RAW_DATA);
      prq->add([icon_url, this](request *rq) -> bool {
        handle<gool::image> img =
            image::create(rq->data, icon_url, this->doc());
        if (img.is_defined()) this->set_icon(img);
        return true;
      });
      this->load_data(prq);
    }

    virtual bool       set_frame_type(FRAME_TYPE on) override;
    virtual FRAME_TYPE get_frame_type() const override;

    virtual void set_enabled(bool on_off) override;
    virtual bool is_enabled() const override;

    // virtual bool on(view& v, element* self, event_behavior& evt) override;

    // operations
    void parse_html(istream &inp);
    bool unload_doc(bool uconditionally);

    virtual style *get_default_style();

    virtual ustring get_lang() const;
    virtual ustring get_theme() const;

    virtual ustring get_input_lang() const;
    virtual void    on_input_lang_change(const ustring &iso_lang_country);

    bool        set_element_html(helement &el, bytes html, SE_TYPE where,
                                 const string &url = string());
    bool        set_element_html(helement &el, wchars html, SE_TYPE where,
                                 const string &url = string());
    bool        construct_element_from_html(bytes html, helement &el);
    gool::font *get_font(style *cs);
    // bool set_element_html(element* b, chars html, SE_TYPE where, const char*
    // url = 0);  bool set_element_html(element* b, wchars html, SE_TYPE where,
    // const char* url = 0);

    virtual void refresh(element *el, rect zarea = rect());
    virtual bool needs_full_window_refresh() const { return false; }
    // events

    virtual void on_visibility_changed(bool visible);

    virtual bool on_mouse(int cmd, uint buttons, uint alts, point pt);

    bool handle_mouse(int cmd, uint buttons, uint alts, point pt);

    virtual void check_mouse(bool always = false); // emulates MOUSE_MOVE
    /*virtual bool do_touch_mouse() {
      on_mouse(MOUSE_CHECK, mouse_buttons, last_alts_keys_state, point(-1, -1));
      return true;
    }*/
    virtual bool on_mouse_wheel(point pt, int steps, uint keys) { return false; }

#ifdef USE_TOUCH
    struct touch_point {
      uint  id;
      point pos;
    };
    struct touch_move {
      uint ticks;
      array<touch_point> points;
    };
    circular_buffer<touch_move, 8> touch_history;
    //touch_move touch_last_start;
    point     touch_last_start_pos;
    uint_v    touch_last_start_time;
    bool      touch_was_double_tap = false;

    tristate_v is_touched;

    bool      handle_touch(array<touch_point> touch_points, uint alts);
    bool      handle_touch_end(uint alts);
    bool      handle_touch_start(uint alts);
    bool      handle_touch_move(uint alts);
    element*  get_target_touch_element(point pt, element* fallback = nullptr);
#endif

    virtual bool on_key(int cmd, int code, int alts);
    virtual bool on_key_nothandled(int cmd, int code, int alts);
    virtual bool on_ime_chars(bool fin, wchars ime_string,
                              range ime_target    = range(),
                              int_v ime_caret_pos = int_v());
    virtual void on_size(size sz);
    virtual bool on_size_request(int side_id, rect &rc);
    virtual void on_size_changed() { }

    virtual float_v aspect_ratio() const { return float_v(); }
    virtual void    aspect_ratio(float_v v) { ; }

    virtual bool get_min_size(gool::size &sz) { return false; }
    virtual bool get_max_size(gool::size &sz) { return false; }
    virtual bool set_min_size(gool::size sz = gool::size()) { return false; }
    virtual bool set_max_size(gool::size sz = gool::size()) { return false; }

    virtual void on_move();
    virtual void on_state_changed() {
      // is_awaiting_update_layered = false;
      bool mouse_leave = false;
      if (doc()) {
        close_popup_tree(doc());
        auto    ws = get_window_state();
        bool    update_it = false;
        ustring val;
        switch (ws) {
          case WINDOW_STATE_NA:
          case WINDOW_HIDDEN: val = W("hidden"); mouse_leave = true; break;
          case WINDOW_SHOWN: val = W("shown"); update_it = true; break;
          case WINDOW_MAXIMIZED: val = W("maximized"); update_it = true; break;
          case WINDOW_MINIMIZED: val = W("minimized"); mouse_leave = true; break;
          case WINDOW_FULL_SCREEN: val = W("full-screen"); update_it = true; break;
        }
        doc()->set_attr(attr::a_window_state, val);
/*        if (update_it) {
          this->post([this]() -> bool {
            size sz = this->client_dim();
            this->on_size(sz);
            return true;
          });
        }*/
      }
      else {
        auto ws = get_window_state();
        if (ws == WINDOW_HIDDEN || ws == WINDOW_MINIMIZED)
          mouse_leave = true;
      }
      if (mouse_leave)
        handle_mouse(MOUSE_LEAVE, 0, 0, point());
      else
        check_mouse();
    }
    virtual bool on_timer(uint_ptr id);
            bool handle_view_timer(timer_id& id, TIMER_KIND kind);
    virtual bool on_activate(ACTIVATE_MODE am) {
      if (doc()) {
        if (am) doc()->state_on(*this, S_OWNS_FOCUS);
        else doc()->state_off(*this, S_OWNS_FOCUS);
      }
      refresh();
      return false;
    }

    virtual bool on_context_menu(point pt);
    virtual void on_start_ui_replacement(tristate_v sizing = tristate_v()) { in_ui_replacement = true; }
    virtual void on_end_ui_replacement(tristate_v sizing = tristate_v()) { in_ui_replacement = false; }
    virtual bool on_move_request(gool::rect &rc) { return false; }

    virtual bool will_handle_gesture_at(point pt, uint &flags);
    virtual bool handle_gesture(gesture_cmd cmd, uint st /*gesture_state's*/,
                                point pt, double val);

    // system shell drag-n-drop support
    virtual bool on_drop(unsigned long &clipboard_dd_modes, clipboard::data *pd,
                         point pt, element *src);
    virtual bool on_drag_enter(unsigned long &  clipboard_dd_modes,
                               clipboard::data *pd, point pt, element *src);
    virtual bool on_drag_over(unsigned long &  clipboard_dd_modes,
                              clipboard::data *pd, point pt, element *src);
    virtual void on_drag_leave(element *src);
    virtual void on_drag_cancel(element *src);
    virtual bool handle_exchange_event(exchange_cmd     cmd,
                                       unsigned long &  clipboard_dd_modes,
                                       clipboard::data *pd, element *b, point p,
                                       element *src);

    virtual handle<gool::bitmap> get_backbuffer() { return nullptr; }

    //virtual void on_


    bool is_dragging = false;

    bool exec_drag(clipboard::data *pd,  uint &ddm /*in out, DD_MODE flags*/, element *src, gool::bitmap *drag_image, point offset) {
      auto_state<bool> _(is_dragging, true);
      return do_drag(pd,ddm,src,drag_image,offset);
    }
    
    virtual bool do_drag(clipboard::data *pd,
                         uint &ddm /*in out, DD_MODE flags*/, element *src,
                         gool::bitmap *drag_image, point offset) {
      return false;
    }
    virtual void create_drag_cursors(gool::bitmap *drag_image, point &offset,
                                     handle<gool::bitmap> &move,
                                     handle<gool::bitmap> &copy,
                                     handle<gool::bitmap> &none) {
      move = drag_image;
      copy = drag_image;
      none = drag_image;
    }

    virtual bool draw(void *dc, rect client_rc) override {
#if defined(WINDOWS)
      if (has_animations())
        on_animation_tick();
      check_timers_overdue_in_all_views();
#endif
      return iwindow::draw(dc, client_rc);
    }
                   
    virtual uint dblclick_timeout() const { return 400; }

    // paint it
    virtual void paint(element *to_draw = nullptr, bool front_layer = true);
    //virtual void paint(function<void(gool::bitmap *dst, point view_pos)> cb, element *to_draw = nullptr, bool front_layer = true);
    // virtual void render(element* el = 0) = 0;

    virtual bool render_opengl(element *p, bool fore) { return false; }

    // marks end of drawing on buffers, see html-animator-blend
    virtual void commit_drawing_buffers() {}

    virtual void setup_layout(text_block *ptb, wchars text,
                              slice<helement> elements) {}
    // virtual void update_layout( text_block* ptb, wchars text, slice<helement>
    // elements ) {}

    virtual void on_graphics_critical_failure() {
      if (callback)
        callback->graphics_critical_failure(this);
      else if (parent())
        parent()->on_graphics_critical_failure();
    }

    // loader
    virtual bool load_data(pump::request *rq, bool now = false) {
      rq->is_sync  = now;
      rq->dst_view = this;

      element *dst = rq->dst ? rq->dst : doc();
      if (dst) dst->on_data_request(*this, rq);

      if (rq->ready_flag) return true;

      if (do_load_data(rq)) return true;
      if (!now) { return false; }
      // if (rq->status != 0) return true; ????
      uint th = get_ticks() + 60000; // one minute threshold
      while (!rq->ready_flag) {
        yield();
        if (get_ticks() > th) {
          // on_data_loaded(rq);
          return false;
        }
      }
      // on_data_loaded(rq);
      return true;
    }

    virtual bool on_load_data(pump::request *prq) { return false; }
    virtual bool on_data_loaded(pump::request *prq);
    virtual bool postprocess_loaded_data(pump::request *prq);
    virtual bool dispatch_request(handle<pump::request> rq)
        override; // returns true if document was reloaded
    virtual bool request_data(pump::request *rq); // delayed

    void stop_pump(bool restart);

    // scripts handling
    virtual bool load_script(document *root, const string &url,
                             const string &mime_type, wchars text) {
      return false;
    }
    virtual bool include_script(document *root, const string &url,
                                const string &mime_type) {
      return false;
    }

    // popups and embedded windows handling
    /*struct popup_def
    {
      helement anchor;
      helement popup;
      helement focus;
      bool     is_tool;
      popup_def(): is_tool(false) {}
    };*/
    element *    popup_anchor(element *of_this);
    element *    popup_of_anchor(element *anchor);
    element *    popup_saved_focus(element *popup);
    virtual bool close_popup(element *p, bool set_auto_focus);
    // virtual  bool close_tool( element* p );
    virtual bool close_popup_tree(element *b);
    void         close_owned_popups(element *b);
    bool         is_anchored_popup(element *b);
    // bool is_tool(element* b);
    virtual bool     forget_window(iwindow *iw);
    virtual iwindow *create_window(
        element *forel, element *anchor, WINDOW_TYPE wt,
        function<rect(view &, element *, element *)> setup_and_calc_place,
        ELEMENT_WINDOW_MODE mode = ELEMENT_WINDOW_AUTO) = 0;
    virtual iwindow *get_iwindow_of(element *el);

    virtual bool show_popup(element *el, element *anchor, WINDOW_TYPE wt,
                            uint placement, point vpt_at = point(),
                            ELEMENT_WINDOW_MODE mode = ELEMENT_WINDOW_AUTO);
    virtual void kill_sibling_popups(element *anchor);
    virtual bool close_all_popups();
    virtual void post_close_all_popups();
    void         notify_popup_show(element *anchor, element *popup, int code,
                                   int reason = 0);

    struct frame_attributes {
      enum_v     frame_type;
      tristate_v resizeable;
      tristate_v minimizable;
      tristate_v maximizable;
      enum_v     glassy; /*0-none, 1-dark, 2-light*/
      size_v     width;
      size_v     height;
      enum_v     alignment; /* 1..9 */

      size_v     min_width;
      size_v     min_height;
      size_v     max_width;
      size_v     max_height;

    };

    void required_frame_attributes(frame_attributes& fatts);

    virtual void on_element_visibility_changed(element *v, bool on_off);

    // update queue
    void         add_to_update(element *el, STYLE_CHANGE_TYPE ut);
    void         add_to_update(element *el, bool need_remeasure) { add_to_update(el, need_remeasure ? CHANGES_DIMENSION : CHANGES_VISUAL); }

    virtual void update_element(element *b);
    virtual void commit_update(bool force_render = false);


    // update queue 2.0
    weak_handle<element> invalid_style_root;

    bool drop_styles(helement el);
    bool _drop_styles(helement el);


    // thunks
    virtual void on_element_behavior_changed(element *     el,
                                             const string &old_names,
                                             const string &new_names);
    virtual void on_style_resolved(handle<element> psb,
                                   const style *   prev_style) {}

    // events & posts
    virtual void post_behavior_event(event_behavior &evt,
                                     bool            only_if_not_exist = false);
    virtual bool send_behavior_event(event_behavior &evt);
    virtual bool post(functor *pf, bool only_once = false);
    virtual bool exec(function<bool()> f) {
      if (dispatch_queue == async::dispatch::current()) return f();
#ifdef _DEBUG
      // most probably: Sciter API access from worker thread:
      // note: Sciter API access from worker thread is supported but may
      // lead to non-effective execution as **each** API call is posted
      // separately. More effective way is to use GUI_CODE_START/GUI_CODE_END
      // blocks for multiple DOM access calls from worker threads.
      dbg_printf("WARNING: view::exec() - execution from non-GUI thread\n");
#endif
      // call from other thread
      handle<function_functor> func = new function_functor(f);
      post(func, false);
      uint Until = tool::timer::get_ticks() + ASYNC_MAX_TIMEOUT;
      while (func->processed == 0) {
        if (tool::timer::get_ticks() > Until) {
          assert(false); // couldn't reach UI thread in ASYNC_MAX_TIMEOUT time
          return false;
        }
        yield();
      }

      return true;
    }
    virtual void async(function<bool()> f, bool wait = false) {
      handle<function_functor> func = new function_functor(f);
      post(func, false);
      if (wait) {
        uint Until = tool::timer::get_ticks() + ASYNC_MAX_TIMEOUT;
        while (func->processed == 0) {
          yield();
          if (tool::timer::get_ticks() > Until) {
            assert(false); // couldn't reach UI thread in ASYNC_MAX_TIMEOUT time
            return;
          }
        }
      }
    }

    void post(function<bool()> f, bool only_once = false) {
      handle<function_functor> func = new function_functor(f);
      post(func, only_once);
    }

    virtual void dispatch_posted_event(handle<posted_event> pe);
    virtual void handle_posted_event(handle<posted_event> pe);
    virtual void on_content_change(element *el, uint content_change_bits);
    void         process_posted_things(bool repost);

    // animations
    virtual bool add_animation(element *b, animation *pba,
                               const style *new_style, const style *old_style);
    virtual bool add_animation(element *b, animation *pba) {
      return add_animation(b, pba, b->c_style, b->p_style);
    }
    virtual void remove_animation(handle<element> b, animation *pa = nullptr);
    virtual void remove_animations(handle<element>                   b,
                                   function<bool(handle<animation>)> filter);
    virtual void remove_all_animations();
    virtual void remove_finite_animations();
    
    virtual uint do_animation(uint current_clock); // returns delay for the next animation step
    virtual void on_animation_tick();

    virtual bool has_animations() const { return animating.size() != 0; }
    virtual bool request_animation_frame(uint delay = 0) = 0;
    virtual void stop_animation_frames()                 = 0;

    // true if the view will do perform_window_move()
    virtual bool supports_native_ui_window_move() const {
#if defined(WINDOWS)
      return true;
#else
      return false;
#endif
    }

    virtual bool perform_window_move() { return false; }

#ifdef _DEBUG
    virtual void dbg_print_stack_trace(){};
#endif // _DEBUG

    // behaviors
    void         detach_all_behaviors();                 // from self
    void         attach_behavior(ctl *pc);               // to self
    void         detach_behavior(ctl *pc);               // from self
    void         detach_behavior(chars name);            // from self
    virtual bool call_behavior_method(method_params *p); // on self
#ifdef SCITER
    //virtual bool call_behavior_method(element *b, tis::VM *c, tis::value tag,
    //                                  tis::value &retval);
#endif
    virtual bool call_behavior_method(element *b, const char *name,
                                      const tool::value *argv, size_t argc,
                                      tool::value &retval);

    virtual bool call_element_method(element *b, const char *name, 
                                     slice<tool::value> args, 
                                     tool::value &retval) { return false; }

    // element behaviors
    ctl *        create_behavior_for(element *b, const string &name);
    virtual ctl *create_behavior(element *b, const string &name);
    // virtual void      on_element_lost_behaviors(element* psb) {}
    virtual bool call_behavior_method(element *b, method_params *p);
    // events

    virtual void on_element_removed(html::element *psb) {
      // note: theses are weak_handles now:
      // if (mouse_over_element == el) mouse_over_element = nullptr;
      // if (mouse_down_element == el) mouse_down_element = nullptr;
      // if (mouse_capture_element == el) mouse_capture_element = nullptr;
      // if (focus_element == el) focus_element = nullptr;
      // if (event_capture_element == el) event_capture_element = nullptr;
    }

    virtual void on_element_removing(html::element *psb) {}
    
    virtual element *find_element(point pt);
    virtual element *element_under_cursor(/* out, view related */ gool::point &cursor_pos) {
      return find_element(cursor_pos);
    } // finds element out of the view
    element *get_current(element *b);
    bool     reset_current_in(element *container);

    // capture and focus .
    virtual void set_capture(element *b);
    virtual void set_capture_strict(element *b) {
      set_capture(b);
      mouse_capture_strict = true;
    }
    virtual element *get_capture();
    virtual void     stop_capture(element *b) {
      if (get_capture() == b) set_capture(0);
    }

    // set the block as a filter for all UI events, used
    // in Element.showModal implementation - lghtweight dialogs
    virtual void     set_event_capture(element *b);
    virtual element *get_event_capture();
    // virtual void      setup_custom_frame();

    virtual bool is_active() const = 0;

    enum FOCUS_CMD_TYPE {
      FOCUS_NEXT,
      FOCUS_PREV,
      FOCUS_HOME,
      FOCUS_END,
      FOCUS_LEFT,
      FOCUS_RIGHT,
      FOCUS_UP,
      FOCUS_DOWN,  // all these - by key
      FOCUS_FIRST, // these two - by_code
      FOCUS_LAST,  //
      FOCUS_END_REACHED = 0x8000
    };
    bool     set_focus_on(FOCUS_CMD_TYPE cmd);
    element *get_next_focus_element(FOCUS_CMD_TYPE cmd,
                                    element *      current_element,
                                    bool &         end_reached);
    // tab traversal helper. this is fired when focus is on the last element,
    // parent shall return true if it knows where to set next focus.
    virtual bool request_parent_to_advance_focus(bool next) { return false; }
    virtual bool is_closed_tab_cycle() { return true; } // true if tabs traversal needs to be closed inside this view

    virtual bool set_focus(
        helement b, FOCUS_CAUSE cause,
        bool postfactum = false /* true -shall not try to set focus on HWND*/);
    bool post_set_focus(
        helement b, FOCUS_CAUSE cause,
        bool postfactum = false /* true -shall not try to set focus on HWND*/);
    virtual bool do_set_focus(
        helement b, FOCUS_CAUSE cause,
        bool postfactum = false /* true -shall not try to set focus on HWND*/);
    virtual void on_focus_changed(element *b, bool got);
    virtual void on_current_changed(element *b) {}
    element *    get_focus_element() const { return focus_element; }
    virtual bool on_focus(bool got);

    virtual bool activate_keyboard_for(element *b, uint type) {
      if (callback)
        return callback->request_keyboard(this, type);
      return false;
    }

    selection_ctx *get_selection_ctx();

    virtual bool     set_highlighted(element *el);
    virtual element *get_highlighted() const;

    // events traversal

    void check_internal_dd_event(event_mouse &evt);
    void check_internal_dd_event(event &evt) {}

    template <class TEVT>
    bool traverse_mouse_child_parent(element *b, element *p, TEVT &evt,
                                     element **accepted_by = nullptr);
    template <class TEVT>
    bool traverse_mouse_parent_child(element *b, element *p, TEVT &evt,
                                     element **accepted_by = nullptr);
    template <class TEVT> bool traverse_mouse(element *b, TEVT &evt);
    template <class TEVT> bool _traverse_mouse(element *b, TEVT &evt);

    virtual bool on_element_event(element *b, event_mouse &evt);
    virtual bool on_element_event(element *b, event_key &evt);
    virtual bool on_element_event(element *b, event_focus &evt);
    virtual bool on_element_event(element *b, event_scroll &evt);
    virtual bool on_element_event(element *b, event_behavior &evt);
    virtual bool on_element_event(element *b, event_command &evt);
    virtual bool on_element_event(element *b, event_exchange &evt);
    virtual bool on_element_event(element *b, event_gesture &evt);
    virtual bool on_element_timer(element *b, timer_def &td);

    virtual bool handle_element_event(helement b, event &evt) { return false; }

    // virtual bool get_element_text_representation(element *b, ustring& us) {
    // return false; }

    virtual bool on(view &v, element *self, event_mouse &evt) { return event_handler::on(v, self, evt); }
    virtual bool on(view &v, element *self, event_key &evt) { return event_handler::on(v, self, evt); }
    virtual bool on(view &v, element *self, event_focus &evt) { return event_handler::on(v, self, evt); }
    virtual bool on(view &v, element *self, event_command &evt) { return event_handler::on(v, self, evt); }
    virtual bool on(view &v, element *self, event_scroll &evt) { return event_handler::on(v, self, evt); }
    virtual bool on(view &v, element *self, event_exchange &evt) { return event_handler::on(v, self, evt); }
    virtual bool on(view &v, element *self, event_gesture &evt) { return event_handler::on(v, self, evt); }
    
    virtual bool on(view &v, element *self, event_behavior &evt) override;

    virtual void on_element_client_size_changing(element *b, bool horizontal, int val) {}  // immediate
    virtual void on_element_client_size_changed(element *b);
    virtual bool may_have_on_size_handler(element *b);

    // called when zpt is inside element's box
    virtual bool hit_test_element(element *b, point zpt) { return true; }

    virtual bool on_element_data_arrived(element *b, pump::request *prq);
    virtual bool on_element_data_request(element *b, pump::request *prq);

    void notify_data_request(element *self, pump::request *rq) {
      if (!rq->request_notify_receivers().contains(uint_ptr(self))) {
        rq->request_notify_receivers.push(uint_ptr(self));
        on_data_request_notify(self, rq);
      }
    }
    void notify_data_arrived(element *self, pump::request *rq) {
      if (!rq->arrived_notify_receivers().contains(uint_ptr(self))) {
        rq->arrived_notify_receivers.push(uint_ptr(self));
        on_data_arrived_notify(self, rq);
      }
    }

    virtual void on_data_request_notify(element *self, pump::request *rq) {  }
    virtual void on_data_arrived_notify(element *self, pump::request *rq) {  }


    virtual bool on_element_draw_background(graphics *pg, element *el,
                                            point pos) {
      return false;
    }
    virtual bool on_element_draw_foreground(graphics *pg, element *el,
                                            point pos) {
      return false;
    }
    virtual bool on_element_draw_content(graphics *pg, element *el, point pos) {
      return false;
    }
    virtual bool on_element_draw_outline(graphics *pg, element *el, point pos) {
      return false;
    }

    virtual bool disable_element_draw(graphics *pg, element *el) {
      if (el->state.popup() && pg->get_drawing_root() == doc()) return true;
      return false;
    }
    // virtual bool disable_glyph_run_draw(graphics* pg, const
    // html::tflow::text_flow& tf, const tflow::glyph_run& gr ) { return false;
    // }

    // passes the evt to event_handlers/behaviors attached to the element.
    template <class EVT> bool traverse_element_event(element *b, EVT &evt);

    // cursor handling
    virtual void    set_cursor(cursor *pcur) = 0;
    virtual cursor *get_cursor()             = 0;
    // virtual uint  get_cursor_id( string uri ) { return 0; }

    virtual uint get_animation_ticks() { return get_ticks(); }
    virtual uint get_ticks() { return tool::get_ticks(); }
    virtual timer_def& start_timer(element *b, uint ms, timer_id id, TIMER_KIND kind);
    virtual void       stop_timer(element *b, timer_id id, TIMER_KIND kind);
   void                stop_timer_if(element * b,  function<bool(timer_id id, TIMER_KIND kind)> pred);
    void               check_timers_overdue();
    static void        check_timers_overdue_in_all_views();

    virtual void set_timer(uint_ptr id, uint ms, uint_ptr &sys_id) = 0;
    virtual void stop_all_timers();

    virtual void do_request_idle() = 0;

    void         request_idle();
    bool         on_idle();
    virtual bool handle_on_idle();
    size_t       items_in_idle_queue() const;
    bool         idle_pileup_detected() const;
    virtual bool wants_idle() const {
      auto pd = tool::async::dispatch::current();
      return pd ? pd->is_alive() : false;
    }
    virtual bool exec_idle() {
      auto pd = tool::async::dispatch::current();
      return pd ? pd->heartbeat() : false;
    }

    virtual void enable_ime(bool on) {}

    // scrolling
    virtual bool scroll_to_view(element *scrollable, const rect &view_rc, bool toTop, ENSURE_VISIBLE_MANNER how = AS_PER_CSS);
    virtual bool ensure_visible(element *b, bool toTop, ENSURE_VISIBLE_MANNER how = AS_PER_CSS);
    bool         _ensure_visible(element *b, bool toTop, ENSURE_VISIBLE_MANNER how);
    virtual bool scroll_window(point target_pos, element *b,
                               ENSURE_VISIBLE_MANNER manner,
                               bool                  shall_close_popups = true,
                               bool allow_out_of_bounds = false);

    virtual bool set_h_scrollbar(range content_h, bool noremove, size &sz) {
      return false;
    }
    virtual bool set_v_scrollbar(range content_v, bool noremove, size &sz) {
      return false;
    }
    virtual point get_scroll_pos() { return point(); }
    virtual void  set_scroll_pos(point pos) {}
    virtual bool  has_h_scrollbar(bool &yes) { return false; }
    virtual bool  has_v_scrollbar(bool &yes) { return false; }

    // media vars
    virtual media_variables &media_vars() { return _media_vars; }
    static media_variables & default_media_vars();

    virtual void set_media_vars(
        wchars mvt, bool reset = true,
        bool reapply_styles = true); // accepts comma separated list of vars
    virtual void set_media_vars(
        value map, bool reset = true,
        bool reapply_styles = true); // accepts value-map - list of vars
    static void set_default_media_vars(
        value map, bool reset = true); // accepts value-map - list of vars
    void update_media_vars(bool reset_styles); // re-evaluate media expressions
                                               // [and reset all styles].

    bool match_media_type(wchars media_var_list) const;

    virtual bool on_media_changed();
    virtual void init_media_vars();
    virtual void on_dpi_changed(size dpi, rect proposed_window_rect) {
      pixels_per_inch(dpi);
      if(auto d = doc())
        d->drop_caches(); // to reset fonts!
      notify_media_change();
    }

    void notify_media_change() override;

    virtual GRAPHICS_CAPS graphics_caps() const { return UNKNOWN_GRAPHICS; }
    virtual ustring       graphics_backend() const { return W("{unknown}"); }

    // virtual bool set_environment(const tool::environment::info &penv) {
    // return false; }

    // value handling

    bool get_element_native_value(helement b, value &v, bool ignore_text_value);
    bool set_element_native_value(helement b, const value &v,
                                  bool ignore_text_value);

    virtual bool get_element_value(helement b, value &v,
                                   bool ignore_text_value) {
      return get_element_native_value(b, v, ignore_text_value);
    }
    virtual bool set_element_value(helement b, const value &v,
                                   bool ignore_text_value) {
      return set_element_native_value(b, v, ignore_text_value);
    }

    virtual bool get_element_property(helement b, wchars name, ustring &v) {
      return false;
    }

    virtual void move_element(element *b, point view_pos, size *pdim = 0,
                              ELEMENT_WINDOW_MODE mode = ELEMENT_WINDOW_AUTO,
                              int                 ref_point = 0);
    virtual void stop_move_element(element *b);

    // aux
    // used by sciter to output std consoles and callabcks

    /*virtual bool host_callback( int channel, const tool::value& p1, const
    tool::value& p2, tool::value& r )
    {
      if( callback && callback->host_callback(channel,p1,p2,r))
        return true;
      return false;
    }*/
    virtual view *    parent() const { return _parent_view; }
    virtual void      parent(view *parent) { _parent_view = parent; }
    virtual VIEW_TYPE view_type() const { return VIEW_WINDOW; }
    virtual bool      is_detached() const { return _is_detached; }
    virtual void      is_detached(bool d) { _is_detached = d; }
    virtual bool      is_main() const { return _is_main; }
    virtual void      is_main(bool d) { _is_main = d; }

    // view as a window
    virtual bool ask_close_window(bool by_chrome = false) { return false; }
    virtual bool close_window() { return false; }
    virtual bool show_modal() = 0;
    enum AFN_MODE {
      AFN_OPEN,
      AFN_OPEN_MULTIPLE,
      AFN_SAVE,
    };
    virtual array<ustring> ask_file_name(AFN_MODE mode, const ustring &caption,
                                         const ustring &filename,
                                         const wchar *  def_ext,
                                         const wchar *  filter) {
      return array<ustring>();
    }
    virtual bool ask_folder_name(const ustring &caption, ustring &foldername) {
      return false;
    }
    virtual rect window_decoration() const {
      return rect(0, 0, 0, 0);
    } // return width of window decorations
    virtual void adjust_window_rect(rect &rc) const {
      rc >>= window_decoration();
    }
    virtual void move_window(const rect &spos, bool client_rc = false) {
      window_was_moved = true;
    }
    virtual WINDOW_STATE get_window_state() const { return WINDOW_STATE_NA; }
    virtual bool         set_window_state(WINDOW_STATE ws) { return false; }
    virtual ustring      get_window_title() const { return ustring(); }
    virtual bool         set_window_title(const wchar *title) { return false; }

    virtual bool do_event(DO_EVENT_MANNER m, bool &result) {
      return false;
    }
    bool do_event(DO_EVENT_MANNER m) {
      bool dummy = false;
      return do_event(m, dummy);
    }

    bool drag_element(element *self, clipboard::data *with_data, uint &ddm);
    bool drag_data(bitmap *pb, point off, clipboard::data *with_data, uint &ddm);

    // support of standalone frames && dialogs
    virtual void finish(bool f = true) { _finished = f; }
    virtual bool finished() { return _finished; }

    virtual void replace_windowed() = 0;

    virtual void           set_callback(view_callback *cb) { callback = cb; }
    virtual view_callback *get_callback() const { return callback; }
    void                   remove_tooltips();
    uint                   get_message_time() { return message_time; }

    // floats support, uses view::bfc
    element *get_bfc(element *for_el);
    range    get_x_space_at(int at_y, element *of_container);
    int      find_free_space(
             element *for_el, int at_y,
             int width); // returns first y where we can place rect.size().x == width
    range push_left(element *elem, int at_y,
                    element *of_container); // returns range at y;
    range push_right(element *elem, int at_y,
                     element *of_container); // returns range at y;

    // update stack support +
    // this us used by reactor style updates - container then its children

    bool has_mutator() const { return mutator_stack.length() > 0; }

  protected:
    friend struct mutator_ctx;

    bool mutator_push(element *p);
    void mutator_pop();
    //    bool no_mutator() const { return mutator_stack.length() == 0; }
    struct mutator_rec {
      helement el;
      uint     flags;
    };
    array<mutator_rec> mutator_stack;

  public:
    bool mutator_rq(element *p,
                    uint flags); // requests mutation: true - if element is NOT
                                 // under mutation stack, false - it is already
                                 // scheduled for later fixup.

    // update stack support -

    void set_debug_output(view_debug_output *pdo);
    void set_native_debug_output(view_debug_output *pdo) {
      native_debug_output = pdo;
    }

    virtual void
    setup_text_flow(html::element *elem, html::tflow::text_flow &tf,
                    tool::slice<tool::handle<html::node>> nodes) = 0;
    // virtual void draw_glyph_runs( html::element* elem, const
    // html::tflow::text_flow& tf, gool::graphics* gfx, gool::point at,
    // html::tflow::sel_context* sctx);
    virtual void draw_glyph_run(gool::graphics *              gfx,
                                const html::tflow::text_flow &tf,
                                const html::tflow::glyph_run &gr,
                                gool::pointf at, argb color,
                                const style *run_style) = 0;
    virtual void draw_glyph_run_back(gool::graphics *              gfx,
                                     const html::tflow::text_flow &tf,
                                     const html::tflow::glyph_run &gr,
                                     gool::rectf rc, argb back_color) {
      gfx->fill(back_color, rc);
    }

    virtual handle<gool::text_layout> create_text_layout(wchars text);

    // resolution_provider
    virtual size pixels_per_inch() override;
    virtual void pixels_per_inch(size sz) override;
    virtual void reset_resolution() override;
    virtual bool px_as_dip() const override;

    // debug support
    virtual void debug_mode(bool onoff) { _debug_mode = onoff; }
    virtual bool debug_mode() const { return _debug_mode.val(default_debug_mode) != 0; }

    static uint default_debug_mode;

    // view specific entities:
    virtual path *create_path() { return app->create_path(); }

    /*void unobserve_element(helement el) {
      if (mouse_over_element == el) mouse_over_element = nullptr;
      if (mouse_down_element == el) mouse_down_element = nullptr;
      if (mouse_capture_element == el) mouse_capture_element = nullptr;
      if (focus_element == el) focus_element = nullptr;
      if (event_capture_element == el) event_capture_element = nullptr;
    }*/

    struct tray_icon_params {
      handle<gool::image> img;
      ustring             tooltip;
    };

    virtual bool trayicon_setup(const tray_icon_params &params) {
      return false;
    }
    virtual bool trayicon_remove() { return false; }
    virtual bool trayicon_place(rect &rc) { return false; }
    virtual bool trayicon_notify(point screen_pt, uint buttons, mouse_events evt) { return false; }

    enum WRA_MODE {
      WRA_STOP = 0,
      WRA_ATTENTION = 1,
      WRA_ATTENTION_CRITICAL = 2,
    };
    virtual bool request_attention(WRA_MODE wm) { return false; }

    template <typename W = view> static W *ptr(HWINDOW hwnd);
    template <typename W = view> static W *last();

    // true if window at screen_pos is this one or one of its popups
    virtual bool is_at_position(point screen_pos) { return true; }
    
    static uint view_count();

    static array<handle<view>> get_all();

  public:
    virtual bool            do_load_data(pump::request *rq);
    mutex                   guard;
    bool                    rtl_ctx; // is true if the window is RTL
    handle<document>        pdoc;
    array<handle<posted_event>> posted_events;
    array<handle<functor>>  posted_functors;
#ifdef SCITER
    array<handle<posted_event>> _posted_events;
    array<handle<functor>>  _posted_functors;
#endif
    bool                    in_ui_replacement = false;

    //uint                    idle_requests_counter = 0;


    mutable mutex    posted_guard;
    array<timer_def> element_timers;
    bool             use_smooth_scroll;
    static bool      use_smooth_scroll_default;

    // array<popup_def>         popup_defs;
    // array<helement>          tool_elements;

    array<handle<iwindow>> windows; // popup, tool and child windows

    hstyle                default_style;
    handle<ctl>           ground_behavior;
    update_queue          to_update;
    handle<view_callback> callback;

    tristate_v      animations_disabled; // CSS animations disabled
    array<helement> animating;
    tristate_v      inside_animation_frame;
    uint            animation_frame_no = 0;

    dim_v_t dpi;          // resolution_provider
    dim_v_t physical_dpi; // resolution_provider

    // mouse states
    point mouse_pos;
    int   mouse_buttons;
    int   mouse_msg;
    uint  message_time; // message time
    uint  mouse_dblclick_time = 0; // last dblclick
    //uint  mouse_up_time = 0; // last dblclick

    weak_helement mouse_over_element;
    weak_helement mouse_down_element; // mouse down was here, valid until mouse up.
    weak_helement x_drop_target_element;
    point         mouse_down_pos;
    tristate_v    mouse_drag_request_issued;
    tristate_v    mouse_down_on_caption; // last mouse down was on window caption element
    weak_helement mouse_capture_element;
    bool          mouse_capture_strict;
    weak_helement tooltip_owner_element;
    element*      checking_animation_on = nullptr;

    tristate_v    is_mouse_inside;
    tristate_v    is_changing_focus; //
    tristate_v    is_setting_focus;  // processing set_focus
    tristate_v    focus_by_mouse;    // setting focus by mouse click
    tristate_v    dismissing;
    tristate_v    is_measuring;
    tristate_v    dont_change_focus;
    tristate_v    is_disabled;             // true if disabled
    tristate_v    has_view_relative_units; // content uses vw,vh,vmin,vmax

    bool notifying_media_change = false;

    weak_helement focus_element;
    weak_helement old_focus_element;
    weak_helement event_capture_element;
    // helement      dragging_source_element; // dragging source block
    // helement      system_dragging_source_element; // system dragging source
    // block
    //circular_buffer<uint> last_wheel_ticks;
    //int                   last_wheel_direction;

    helement sticky_scrollable;        // see sticky_scroll_ctl
    helement sticky_scrollable_anchor; // see sticky_scroll_ctl

    handle<behavior::highlighted_ctl> highlighted_ctl;

    weak_helement gesture_element;
    point         gesture_pos;
    double        gesture_val;  // angle or zoom factor
    uint          gesture_time; // time of last gesture event since system start

    media_variables _media_vars;
    // static media_variables default_media_vars;
    ustring _input_lang;

    int _paint_ops;

    rect              caret_pos;
    weak_handle<view> _parent_view;
    bool              _is_detached =
        false;             // true if window was created with params.is_detached
    bool _is_main = false; // true if window was created with params.is_main -
                           // will destroy all dependent windows
    // VIEW_TYPE     _view_type;
    bool _finished;
    // array<helement> anonymous_elements; // pool of anonymous elements.
    element *get_anonymous_para(tag::symbol_t sym = tag::T_TEXT);

    selection_ctx *_selection_ctx;       // current _selection_ctx
    tristate_v     _draw_selection_only; // used by drag_image implementation

    hash_table<box_shadow_params, handle<shadow_bitmap>> _box_shadow_cache;

    tristate_v window_was_moved;

    handle<gool::image> _icon;

    graphics *drawing_surface = nullptr;

    helement bfc; // current layout root element - establishes BFC -
                  // block formatting context.

    // function<void(html::url_data*)>            url_data_tracker;

    tristate_v _debug_mode; // is on when debugger is enabled.

#if defined(ZIP_SUPPORT)
    static hash_table<string, handle<cabinet>> zip_cache; // cache of zip files
#endif

    array<handle<element>>  moved_elements;
    handle<async::dispatch> dispatch_queue;

    point moved_element_pos(const element *el, int_v mwidx = int_v()) {
      index_t idx =
          mwidx.is_defined() ? mwidx.val(0) : moved_elements.get_index(el);
      assert(idx >= 0);
      if (idx >= 0) {
        handle<element> tel = moved_elements[idx];
        if (tel->airborn) {
          point pos = tel->airborn->pos;
          if (tel->airborn->type == AIRBORN_WINDOW_SCREEN)
            pos -= this->client_screen_pos();
          return pos;
        }
        /*
        iwindow* piw = me.el->window(*this);
        if( piw ) {
          point ppos = piw->client_screen_pos() - this->client_screen_pos() +
        me.el->border_distance(*this).origin; return ppos;
        }*/
        return tel->pos();
      }
      return point(0, 0);
    }

    void error_report(const char *cap, element *el);

  public:
    handle<pump::request> current_rq; // current rq processing, SCN_LOAD_DATA/SciterDataReady
    handle<pump::request> pdocrq;     // doc rq being loaded

    handle<document> current_doc; // being measured

    struct log_item {
      uint    subsystem;
      uint    severity;
      ustring msg;
    };
    circular_buffer<log_item> debug_output_buffer;
    handle<view_debug_output> debug_output;
    handle<view_debug_output> native_debug_output;

    static uint last_alts_keys_state;

    bool alt_key_down_handled = false;

    bool treat_px_as_dip;

  protected:
    static hash_table<uint_ptr, handle<view>> all;
    static mutex                              all_guard;
  };

  // virtual print_target* select_printer(view* ui_parent = nullptr) ; /*if
  // ui_parent == null then it returns default printer params*/  bool
  // set_printer(print_target& target, const ustring& printer_id);

  void draw_glyph_runs(view &v, element *elem, const tflow::text_flow &tf,
                       gool::graphics *g, point at, tflow::sel_context *sctx);

  element *get_enabled(view &v, element *b);
  void     measure_out_of_flow(view &v, element *b);

  bool inscribe_to_screen(iwindow *pw, rect &target_place,
                          const rect &anchor_place);
  // bool replace_on_screen( iwindow* pw, rect& target_place /* screen relative
  // */, const rect& initiator_place/* screen relative */, WINDOW_TYPE for_type
  // );
  bool replace_on_screen(rect        screen_place,
                         rect &      target_place /* screen relative */,
                         const rect &initiator_place /* screen relative */,
                         WINDOW_TYPE for_type);

  template <class TEVT>
  inline bool view::traverse_mouse_child_parent(element *b, element *p,
                                                TEVT &    evt,
                                                element **accepted_by) {
    helement t;

    check_internal_dd_event(evt);

    for (t = b; t && t != p; t = TEVT::get_logical_parent(t)) {
      if (!t || (t->pview() != this)) return (evt.cmd & EVENT_HANDLED) != 0;
      evt.pos = t->transform_view_to_local(*this, evt.pos_view);
      if (t->on(*this, evt)) {
        if (accepted_by && !*accepted_by) *accepted_by = t;
        evt.cmd |= EVENT_HANDLED;
      }
    }
    return (evt.cmd & EVENT_HANDLED) != 0;
  }

  template <class TEVT>
  inline bool view::traverse_mouse_parent_child(element *b, element *p,
                                                TEVT &    evt,
                                                element **accepted_by) {
    helement hb = b;
    helement hp = p;

    check_internal_dd_event(evt);

    if (p == b) return (evt.cmd & EVENT_HANDLED) != 0;
    if (!b || (b->pview() != this)) return (evt.cmd & EVENT_HANDLED) != 0;
    traverse_mouse_parent_child(TEVT::get_logical_parent(b), p, evt,
                                accepted_by);
    evt.pos = b->transform_view_to_local(*this, evt.pos_view);
    if (b->on(*this, evt)) {
      if (accepted_by && !*accepted_by) *accepted_by = b;
      evt.cmd |= EVENT_HANDLED;
    }
    return (evt.cmd & EVENT_HANDLED) != 0;
  }

  template <class TEVT>
  inline bool view::traverse_mouse(element *b, TEVT &evt) {
    if (!b || !b->pview()) return false;

    if (event_capture_element &&
        !b->belongs_to(*this, event_capture_element, true)) {
      return _traverse_mouse(event_capture_element, evt);
    }
    return _traverse_mouse(b, evt);
  }

  template <class TEVT>
  inline bool view::_traverse_mouse(element *b, TEVT &evt) {
    b = get_enabled(*this, b);
    if (!b) return false;

    handle<view> holder(this);

    check_internal_dd_event(evt);

    evt.cmd |= EVENT_SINKING;

    if (this->on(*this, b, evt)) evt.cmd |= EVENT_HANDLED;

    handle<document> bd = doc();

    behavior_h pc = ground_behavior;
    while (pc) {
      if (pc->will_handle(TEVT::EVENT_GROUP) && pc->on(*this, bd, evt))
        evt.cmd |= EVENT_HANDLED;
      behavior_h next = pc->next;
      pc              = next;
    }

    traverse_mouse_parent_child(b, 0, evt); // to give chance behavior to catch it...

    evt.cmd &= ~EVENT_SINKING;
    // if((evt.cmd & EVENT_HANDLED) == 0)
    traverse_mouse_child_parent(b, 0, evt);

    pc = ground_behavior;
    while (pc) {
      if (pc->will_handle(TEVT::EVENT_GROUP) && pc->on(*this, bd, evt))
        evt.cmd |= EVENT_HANDLED;
      behavior_h next = pc->next;
      pc              = next;
    }

    if (this->on(*this, b, evt)) evt.cmd |= EVENT_HANDLED;

    return (evt.cmd & EVENT_HANDLED) != 0;
  }

  template <class EVT>
  inline bool view::traverse_element_event(element *b, EVT &evt) {
    behavior_h pc    = b->behavior;
    helement   guard = b;
    while (pc) {
      if (pc->will_handle(EVT::EVENT_GROUP) && pc->on(*this, b, evt)) {
        evt.cmd |= EVENT_HANDLED;
        if (b->pview() !=
            this) // element was detached from the DOM in the event handler.
          break;
      }
      behavior_h next = pc->next;
      pc              = next;
    }
    return (evt.cmd & EVENT_HANDLED) != 0;
  }

  template <typename EVT> struct traverser {
    view &v;
    element* handled_by = nullptr;
    traverser(view &vv) : v(vv) {}
    bool traverse_child_parent(element *b, element *p, EVT &evt) {
      if (evt.event_bubbling()) {
        for (helement t = b; t && t != p; t = EVT::get_parent(t)) {
          if (!t || (t->pview() != &v)) return (evt.cmd & EVENT_HANDLED) != 0;
          if (t->on(v, evt)) {
            evt.cmd |= EVENT_HANDLED;
            if (!handled_by) handled_by = t;
          }
        }
      }
      else {
        if (!b) return (evt.cmd & EVENT_HANDLED) != 0;
        if (b->on(v, evt)) {
          evt.cmd |= EVENT_HANDLED;
          if (!handled_by) handled_by = b;
        }
      }
      return (evt.cmd & EVENT_HANDLED) != 0;
    }
    bool traverse_parent_child(element *b, element *p, EVT &evt) {
      if (p == b) return false;
      if (b == 0 || (b->pview() != &v)) return false;

      helement hb = b;
      helement hp = p;

      if (evt.event_bubbling()) {
        traverse_parent_child(EVT::get_parent(b), p, evt);
        if (b->pview() == &v && b->on(v, evt)) {
          evt.cmd |= EVENT_HANDLED;
          if (!handled_by) handled_by = b;
        }
      }
      else {
        if (b->on(v, evt)) {
          evt.cmd |= EVENT_HANDLED;
          if (!handled_by) handled_by = b;
        }
      }
      return (evt.cmd & EVENT_HANDLED) != 0;
    }

    bool traverse(element *b, EVT &evt, bool check_enabled = true) {
      handle<view> holder(&v);
      if (v.event_capture_element && evt.event_bubbling() &&
          (!b || !b->belongs_to(v.event_capture_element, true))) {
        if (_traverse(v.event_capture_element, evt, check_enabled && evt.event_bubbling())) 
          return true;
      }
      return _traverse(b, evt, check_enabled && evt.event_bubbling());
    }

    bool _traverse(element *b, EVT &evt, bool check_enabled = true) {
      if (b && check_enabled) {
        b = get_enabled(v, b);
        if (!b) return false;
      }

      helement guard(b);

      evt.cmd |= EVENT_SINKING;

      if (v.on(v, b, evt)) evt.cmd |= EVENT_HANDLED;

      behavior_h pc = v.ground_behavior;

      while (pc) {
        if (pc->will_handle(EVT::EVENT_GROUP) && pc->on(v, b, evt))
          evt.cmd |= EVENT_HANDLED;
        behavior_h next = pc->next;
        pc              = next;
      }

      if (b) {
        traverse_parent_child(b, 0, evt);

        evt.cmd &= ~EVENT_SINKING;

        traverse_child_parent(b, 0, evt);
      } else
        evt.cmd &= ~EVENT_SINKING;

      pc = v.ground_behavior;
      while (pc) {
        if (pc->will_handle(EVT::EVENT_GROUP) && pc->on(v, b, evt))
          evt.cmd |= EVENT_HANDLED;
        behavior_h next = pc->next;
        pc              = next;
      }

      if (v.on(v, b, evt)) evt.cmd |= EVENT_HANDLED;

      return (evt.cmd & EVENT_HANDLED) != 0;
    }
    NONCOPYABLE(traverser)
  };

  inline updater::updater(view *v, element *mouse_over, bool passive,
                          bool is_root)
      : current_view_state(v), over(mouse_over), discard_update(passive),
        root(is_root), level(0) {
    if (!over) over = v->mouse_over_element;
    level = v->to_update.nesting++;
  }
  inline updater::~updater() {
    if (!self) return;
    self->to_update.nesting = level;
    if (level != 0 && !root) { return; }
    self->to_update.nesting = 0;
    if (discard_update) return;
    self->to_update.update(self);
    if (self->to_update.require_touch_mouse && over) self->check_mouse(true);
    self->to_update.require_touch_mouse = false;
  }

  inline bool load_data(pump::request *rq, view *pv, bool now = false) {
    if (pv) {
      if (pv->load_data(rq, now)) return true;
    } else if (rq->url.like("sciter:*")) {
      ustring     name = rq->url.c_str() + 7;
      tool::bytes data = app()->get_resource(name);
      if (data.length) {
        rq->data = data;
        return true;
      }
    } else if (application::primordial_loader)
      return application::primordial_loader(rq);
    return false;
  }

  template <typename W> inline W *view::ptr(HWINDOW hwnd) {
    critical_section _(all_guard);
    handle<view>     ref;
    all.find(uint_ptr(hwnd), ref);
    return ref.ptr_of<W>();
  }

  template <typename W> inline W *view::last() {
    critical_section _(all_guard);
    handle<view>     ref = all.elements().last(nullptr);
    return ref.ptr_of<W>();
  }

  inline uint view::view_count() {
    critical_section _(all_guard);
    return (uint)all.size();
  }

  inline array<handle<view>> view::get_all()
  {
    critical_section _(all_guard);
    return all.elements();
  }

  array<ustring> file_filter_to_ext_list(const wchar *filter);

  struct mutator_ctx {
    view *vref;
    bool  needs_pop;
    mutator_ctx(element *pel, view *pv) : vref(pv), needs_pop(false) {
      if (vref) needs_pop = vref->mutator_push(pel);
    }
    ~mutator_ctx() {
      if (vref && needs_pop) vref->mutator_pop();
    }
  };

} // namespace html

struct load_file_thumbnail_callback : virtual public tool::resource {
  virtual void on_thumbnail_created(html::bitmap *pthumbnail) = 0;
};

bool load_file_thumbnail(const ustring &path, uint size,
                         load_file_thumbnail_callback *pcb);

#endif
