#ifndef __html_style_h
#define __html_style_h

#include "tool/tool.h"
#include "tool/eval/tl_eval.h"
#include "gool/gool.h"
#include "html-style-atts.h"
#include "html-style-types.h"
#include "html-style-enums.h"
#include "html-primitives.h"
#ifdef USE_D2D
#include "d2d/d2d-primitives.h"
#endif

namespace html {
  using namespace tool;
  using namespace gool;

  class view;
  struct document;
  struct style;
  struct css_istream;
  class style_bag;

  /*enum STYLE_CHANGE_TYPE {
    NO_CHANGES,
    CHANGES_VISUAL,
    CHANGES_POSITION,
    CHANGES_DIMENSION,
    CHANGES_MODEL,
    RESOLVE_STYLE,
  };*/

#pragma warning(push)
#pragma warning(disable : 4315) // 'this' pointer for member
                                // 'html::style::before' may not be aligned 8 as
                                // expected by the constructor

#pragma pack(push, 4) // IT MUST BE 4 bytes aligned!!!!

  value resolve_var(const element_context& ctx, value val);

  struct char_style // inheritable properties
  {
    ustring                font_family;
    size_v                 font_size;
    font_style_ev          font_style;
    font_rendering_mode_ev font_rendering_mode;
    int_v                  font_weight;
    color_v                font_color;
    font_variant_ligatures_ev font_variant_ligatures;
    font_variant_caps_ev      font_variant_caps;

    size_v               line_height;
    size_v               letter_spacing;
    white_space_ev       white_space;
    overflow_wrap_ev     word_wrap;   // 0 - normal, 1 - break-word
    word_break_ev        word_break;  // 0 - normal, 1 - keep-all, 2 - break-all
    handle<gool::cursor> cursor;
    direction_ev         direction;
    color_v              text_selection_color;
    color_v              text_selection_background_color;
    color_v              text_selection_caret_color;
    text_transform_ev    text_transform;

    text_decoration_ev        text_decoration_line;     // 0 - none, 1 - underline, 2 - overline, 3 - line-through
    text_decoration_style_ev  text_decoration_style;    // 0 - solid, 1 - double, 2 - dotted, 3 - dashed, 4 - wavy
    color_v                   text_decoration_color;   //
    size_v                    text_decoration_thickness;

    value_handle<shadow_def> text_shadow;
    int_v                    tab_size;

    mapping_v mapping;

    // SVG specific styles
    color_v       fill_color;
    string        fill_id;
    opacity_v     fill_opacity;
    fill_rule_ev  fill_rule;
    color_v       stroke_color;
    string        stroke_id;
    size_v        stroke_width;
    stroke_linecap_ev   stroke_linecap;
    stroke_linejoin_ev  stroke_linejoin;
    float_v       stroke_miterlimit;
    value         stroke_dasharray;
    size_v        stroke_dashoffset;
    opacity_v     stroke_opacity;

    image_rendering_mode_ev image_rendering;

    float_v zoom;
    value   scroll_manner_x;
    value   scroll_manner_y;

    static char_style null_val;

    INLINE bool eq(const char_style &rs) const {
      //return memcmp((byte *)this, &rs, sizeof(char_style)) == 0;

#define PROP(name) (name == rs.name)

      return
           PROP(font_family)
        && PROP(font_size)
        && PROP(font_style)
        && PROP(font_rendering_mode)
        && PROP(font_variant_ligatures)
        && PROP(font_variant_caps)
        && PROP(font_weight)
        && PROP(font_color)
        && PROP(line_height)
        && PROP(letter_spacing)
        && PROP(white_space)
        && PROP(word_wrap)
        && PROP(word_break)
        && PROP(cursor)
        && PROP(direction)
        && PROP(text_selection_color)
        && PROP(text_selection_background_color)
        && PROP(text_selection_caret_color)
        && PROP(text_transform)
        && PROP(text_decoration_line)
        && PROP(text_decoration_style)
        && PROP(text_decoration_color)
        && PROP(text_decoration_thickness)
        && PROP(text_shadow)
        && PROP(tab_size)
        && PROP(mapping)
        && PROP(fill_color)
        && PROP(fill_id)
        && PROP(fill_opacity)
        && PROP(fill_rule)
        && PROP(stroke_color)
        && PROP(stroke_id)
        && PROP(stroke_width)
        && PROP(stroke_linecap)
        && PROP(stroke_linejoin)
        && PROP(stroke_miterlimit)
        && PROP(stroke_dasharray)
        && PROP(stroke_dashoffset)
        && PROP(stroke_opacity)
        && PROP(image_rendering)
        && PROP(zoom)
        && PROP(scroll_manner_x)
        && PROP(scroll_manner_y);
#undef PROP
    }

    inline uint hash() const {
      //STATIC_ASSERT((sizeof(char_style) & 3) == 0);
      //slice<uint> s((const uint *)this, sizeof(char_style) >> 2);
      //return hash_value(s);
#define PROP(name) hash_combine(h, tool::hash(name));

        uint h = 157;

        PROP(font_family)
        PROP(font_size)
        PROP(font_style)
        PROP(font_rendering_mode)
        PROP(font_variant_ligatures)
        PROP(font_variant_caps)
        PROP(font_weight)
        PROP(font_color)
        PROP(line_height)
        PROP(letter_spacing)
        PROP(white_space)
        PROP(word_wrap)
        PROP(word_break)
        PROP(cursor)
        PROP(direction)
        PROP(text_selection_color)
        PROP(text_selection_background_color)
        PROP(text_selection_caret_color)
        PROP(text_transform)
        PROP(text_decoration_line)
        PROP(text_decoration_style)
        PROP(text_decoration_color)
        PROP(text_decoration_thickness)
        PROP(text_shadow)
        PROP(tab_size)
        PROP(mapping)
        PROP(fill_color)
        PROP(fill_id)
        PROP(fill_opacity)
        PROP(fill_rule)
        PROP(stroke_color)
        PROP(stroke_id)
        PROP(stroke_width)
        PROP(stroke_linecap)
        PROP(stroke_linejoin)
        PROP(stroke_miterlimit)
        PROP(stroke_dasharray)
        PROP(stroke_dashoffset)
        PROP(stroke_opacity)
        PROP(image_rendering)
        PROP(zoom)
        PROP(scroll_manner_x)
        PROP(scroll_manner_y);
#undef PROP
        return h;
    }


    inline bool can_wrap() const {
      return white_space == white_space_normal ||
             white_space == white_space_prewrap;
    }

    inline bool cannot_wrap() const {
      return white_space == white_space_nowrap;
    }

    inline bool wrap_unrestricted() const {
      return word_wrap == overflow_wrap_break_word ||
             word_break == overflow_wrap_anywhere;
    }

    inline bool collapse_ws() const {
      return white_space == white_space_normal ||
             white_space == white_space_nowrap;
    }
    inline bool preserve_ws() const { return !collapse_ws(); }

    inline bool is_empty() const { return *this == null_val; }

    void inherit(const char_style *src);


    void resolve(view &v, const element* pel, const style &on_that, const char_style &src);

    // mostly inherit but with special %, em, ex interpretation.
    // void derive(const char_style& src);

    inline bool operator==(const char_style &rs) const { return eq(rs); }
    inline bool operator!=(const char_style &rs) const { return !eq(rs); }

    inline bool shape_is_equal(const char_style &cs) const {
      return font_family == cs.font_family && font_size == cs.font_size &&
             font_style == cs.font_style && font_weight == cs.font_weight &&
             line_height == cs.line_height && white_space == cs.white_space &&
             direction == cs.direction && text_transform == cs.text_transform &&
             letter_spacing == cs.letter_spacing && mapping == cs.mapping;
    }

    inline bool changes_dimension() const {
      return font_family.is_defined() || font_size.is_defined() ||
             line_height.is_defined() || font_weight.is_defined() ||
             letter_spacing.is_defined() ||
             text_transform > text_transform_none;
    }

    inline bool changes_dimension(const char_style &cs) const {
      return (font_family.is_defined() && font_family != cs.font_family) ||
             (font_size.is_defined() && font_size != cs.font_size) ||
             (font_style.is_defined() && font_style != cs.font_style) ||
             (font_weight.is_defined() && font_weight != cs.font_weight) ||
             (line_height.is_defined() && line_height != cs.line_height) ||
             (white_space.is_defined() && white_space != cs.white_space) ||
             (letter_spacing.is_defined() &&
              letter_spacing != cs.letter_spacing) ||
             (text_transform.is_defined() &&
              text_transform != cs.text_transform);
    }

    inline bool is_content_changing() const {
      return text_transform > text_transform_none;
    }
  };

  static_assert((sizeof(char_style) % 4) == 0, "wrong alignment of char_style");

  struct para_style {
    halign_ev  text_align;
    size_v     text_indent;
    image_ref  list_style_image;
    list_style_position_ev list_style_position;
    list_style_type_ev     list_style_type;
    color_v    list_marker_color;
    size_v     list_marker_size;
    border_style_ev list_marker_style;
    string        style_set;
    string        style_set_base;
    appearance_ev appearance;
    string     vscrollbar; // set name for scrollbar styles
    string     hscrollbar; // set name for scrollbar styles

    // tristate_v  style_set_root;     // true if this is a style set root
    // style.

    INLINE bool eq(const para_style &rs) const {
      //return memcmp((byte *)this, &rs, sizeof(para_style)) == 0;

#define PROP(name) && (name == rs.name)
      return true
        PROP(text_align)
        PROP(text_indent)
        PROP(list_style_image)
        PROP(list_style_position)
        PROP(list_style_type)
        PROP(list_marker_color)
        PROP(list_marker_size)
        PROP(list_marker_style)
        PROP(style_set)
        PROP(style_set_base)
        PROP(appearance)
        PROP(vscrollbar)
        PROP(hscrollbar)
        ;
#undef PROP
    }

    inline uint hash() const {
      //STATIC_ASSERT((sizeof(para_style) & 3) == 0);
      //slice<uint> s((const uint *)this, sizeof(para_style) >> 2);
      //return hash_value(s);
#define PROP(name) hash_combine(h, tool::hash(name));
      uint h = 237;
        PROP(text_align)
        PROP(text_indent)
        PROP(list_style_image)
        PROP(list_style_position)
        PROP(list_style_type)
        PROP(list_marker_color)
        PROP(list_marker_size)
        PROP(list_marker_style)
        PROP(style_set)
        PROP(style_set_base)
        PROP(appearance)
        PROP(vscrollbar)
        PROP(hscrollbar)
#undef PROP
        return h;
    }


    static para_style null_val;

    bool is_empty() const { return *this == null_val; }

    void inherit(const para_style *src);


    void resolve(resolution_provider &v, const style &for_that,
                 const style &src);
    // void derive(const style& src);

    inline bool operator==(const para_style &rs) const { return eq(rs); }
    inline bool operator!=(const para_style &rs) const { return !eq(rs); }

    inline bool shape_is_equal(const para_style &cs) const {
      return text_align == cs.text_align && text_indent == cs.text_indent &&
             list_style_position == cs.list_style_position;
    }

    inline bool changes_dimension() const { return text_indent.is_defined(); }
    inline bool changes_dimension(const para_style &cs) const {
      return text_indent.is_defined() && text_indent != cs.text_indent;
    }

  };

  static_assert((sizeof(para_style) % 4) == 0, "wrong alignment of para_style");

  struct animated_effect;

  ustring length_value_string(const size_v &sz);

  enum IMAGE_FRAME_NO {
    FRAME_CURRENT = -2,
    FRAME_LAST    = -1,
    FRAME_ANIMATE = 0,
    FRAME_FIRST   = 1,
  };

  enum TRANSFORMATION_OP {
    X_TRANSLATION,
    X_ROTATION,
    X_SCALING,
    X_SKEW,
    X_MATRIX
  };
  struct transform_t : resource {
    virtual TRANSFORMATION_OP type() const                                 = 0;
    virtual void         apply(view &v, element *el, affine_mtx_f &to_mtx) = 0;
    virtual transform_t *make_initial()                                    = 0;
    virtual transform_t *clone()                                           = 0;
    virtual bool         is_identical(const transform_t *rs) const         = 0;
    virtual void         morph(view &v, element *b, const transform_t *s,
                               const transform_t *e, float n)              = 0;
    virtual ustring      to_string() const                                 = 0;
    virtual value        to_value() const                                  = 0;
    virtual uint         hash() const                                      = 0;
  };

  struct x_translation : transform_t {
    size_v x;
    size_v y;
    x_translation() {
      x.set_ppx(0);
      y.set_ppx(0);
    }
    x_translation(const x_translation *p) {
      x = p->x;
      y = p->y;
    }
    virtual TRANSFORMATION_OP type() const override { return X_TRANSLATION; }
    virtual void apply(view &v, element *el, affine_mtx_f &to_mtx) override;
    virtual transform_t *make_initial() override { return new x_translation(); }
    virtual transform_t *clone() override { return new x_translation(this); }
    virtual bool         is_identical(const transform_t *rs) const override {
      return rs->type() == type() &&
             x == static_cast<const x_translation *>(rs)->x &&
             y == static_cast<const x_translation *>(rs)->y;
    }
    virtual uint hash() const override { return tool::hash(x) + tool::hash(y); }
    virtual void morph(view &v, element *b, const transform_t *s,
                       const transform_t *e, float n) override {
      size_v sx = static_cast<const x_translation *>(s)->x;
      size_v sy = static_cast<const x_translation *>(s)->y;
      size_v ex = static_cast<const x_translation *>(e)->x;
      size_v ey = static_cast<const x_translation *>(e)->y;
      x.morph(v, b, sx, ex, n);
      y.morph(v, b, sy, ey, n);
      // x = n*(r_end - r_start) + r_start;
      // x == static_cast<const x_translation*>(rs)->x && y == static_cast<const
      // x_translation*>(rs)->y;
    }
    virtual ustring to_string() const override {
      return ustring::format(W("translate(%s,%s)"),
                             length_value_string(x).c_str(),
                             length_value_string(y).c_str());
    }
    virtual value to_value() const override {
      function_value *pf = new function_value();
      pf->name           = WCHARS("translate");
      pf->params.push(value(x));
      pf->params.push(value(y));
      return value::make_function(pf);
    }
  };
  struct x_rotation : transform_t {
    float  radians;
    size_v x;
    size_v y;
    x_rotation() : radians(0) {}
    x_rotation(const x_rotation *p) : radians(p->radians) {}
    virtual TRANSFORMATION_OP type() const override { return X_ROTATION; }
    virtual void apply(view &v, element *el, affine_mtx_f &to_mtx) override;
    virtual transform_t *make_initial() override { return new x_rotation(); }
    virtual transform_t *clone() override { return new x_rotation(this); }
    virtual uint         hash() const override {
      return tool::hash(x) + tool::hash(y) + uint(radians*1000);
    }

    virtual bool is_identical(const transform_t *rs) const override {
      return rs->type() == type() &&
             radians == static_cast<const x_rotation *>(rs)->radians &&
             x == static_cast<const x_rotation *>(rs)->x &&
             y == static_cast<const x_rotation *>(rs)->y;
    }
    virtual void morph(view &v, element *b, const transform_t *s,
                       const transform_t *e, float n) override {
      float sradians = static_cast<const x_rotation *>(s)->radians;
      float eradians = static_cast<const x_rotation *>(e)->radians;
      radians        = n * (eradians - sradians) + sradians;
    }
    virtual ustring to_string() const override {
      return ustring::format(W("rotate(%d)"), int(radians * 360));
    }
    virtual value to_value() const override {
      function_value *pf = new function_value();
      pf->name           = WCHARS("rotate");
      pf->params.push(value::make_angle(radians));
      return value::make_function(pf);
    }
  };
  struct x_scaling : transform_t {
    float x;
    float y;
    x_scaling() : x(1), y(1) {}
    x_scaling(const x_scaling *p) : x(p->x), y(p->y) {}
    virtual TRANSFORMATION_OP type() const override { return X_SCALING; }
    virtual void              apply(view & /*v*/, element * /*el*/,
                                    affine_mtx_f &to_mtx) override {
      to_mtx *= affine_scaling_f(x, y);
    }
    virtual transform_t *make_initial() override { return new x_scaling(); }
    virtual transform_t *clone() override { return new x_scaling(this); }
    virtual uint hash() const override { return tool::hash(x) + tool::hash(y); }
    virtual bool is_identical(const transform_t *rs) const override {
      return rs->type() == type() &&
             x == static_cast<const x_scaling *>(rs)->x &&
             y == static_cast<const x_scaling *>(rs)->y;
    }
    virtual void morph(view &v, element *b, const transform_t *s,
                       const transform_t *e, float n) override {
      float sx = static_cast<const x_scaling *>(s)->x;
      float ex = static_cast<const x_scaling *>(e)->x;
      float sy = static_cast<const x_scaling *>(s)->y;
      float ey = static_cast<const x_scaling *>(e)->y;
      x        = n * (ex - sx) + sx;
      y        = n * (ey - sy) + sy;
    }
    virtual ustring to_string() const override {
      return ustring::format(W("scale(%f,%f)"), x, y);
    }
    virtual value to_value() const override {
      function_value *pf = new function_value();
      pf->name           = WCHARS("scale");
      pf->params.push(value(x));
      pf->params.push(value(y));
      return value::make_function(pf);
    }
  };

  struct x_skew : transform_t {
    float x; // angle, radians
    float y; // angle, radians
    x_skew() : x(0), y(0) {}
    x_skew(const x_skew *p) : x(p->x), y(p->y) {}
    virtual TRANSFORMATION_OP type() const override { return X_SKEW; }
    virtual void              apply(view & /*v*/, element * /*el*/,
                                    affine_mtx_f &to_mtx) override {
      to_mtx *= affine_skewing_f(x, y);
    }
    virtual transform_t *make_initial() override { return new x_skew(); }
    virtual transform_t *clone() override { return new x_skew(this); }
    virtual uint hash() const override { return tool::hash(x) + tool::hash(y); }
    virtual bool is_identical(const transform_t *rs) const override {
      return rs->type() == type() && x == static_cast<const x_skew *>(rs)->x &&
             y == static_cast<const x_skew *>(rs)->y;
    }
    virtual void morph(view &v, element *b, const transform_t *s,
                       const transform_t *e, float n) override {
      float sx = static_cast<const x_skew *>(s)->x;
      float ex = static_cast<const x_skew *>(e)->x;
      float sy = static_cast<const x_skew *>(s)->y;
      float ey = static_cast<const x_skew *>(e)->y;
      x        = n * (ex - sx) + sx;
      y        = n * (ey - sy) + sy;
    }
    virtual ustring to_string() const override {
      return ustring::format(W("skew(%d,%d)"), int(x * 360), int(y * 360));
    }
    virtual value to_value() const override {
      function_value *pf = new function_value();
      pf->name           = WCHARS("skew");
      pf->params.push(value::make_angle(x));
      pf->params.push(value::make_angle(y));
      return value::make_function(pf);
    }
  };
  struct x_matrix : transform_t {
    affine_mtx_f mtx;
    x_matrix() {}
    x_matrix(const x_matrix *p) : mtx(p->mtx) {}
    virtual TRANSFORMATION_OP type() const override { return X_MATRIX; }
    virtual void              apply(view & /*v*/, element * /*el*/,
                                    affine_mtx_f &to_mtx) override {
      to_mtx *= mtx;
    }
    virtual transform_t *make_initial() override { return new x_matrix(); }
    virtual transform_t *clone() override { return new x_matrix(this); }
    virtual uint         hash() const override {
      uint r = 37;
      hash_combine(r, tool::hash(mtx.sx));
      hash_combine(r, tool::hash(mtx.shy));
      hash_combine(r, tool::hash(mtx.shx));
      hash_combine(r, tool::hash(mtx.sy));
      hash_combine(r, tool::hash(mtx.tx));
      hash_combine(r, tool::hash(mtx.ty));
      return r;
    }
    virtual bool is_identical(const transform_t *rs) const override {
      return rs->type() == type() &&
             mtx == static_cast<const x_matrix *>(rs)->mtx;
    }
    virtual void morph(view &v, element *b, const transform_t *s,
                       const transform_t *e, float n) override {
#pragma TODO("morph x_matrix")
      /*float sx = static_cast<const x_skew*>(s)->x;
      float ex = static_cast<const x_skew*>(e)->x;
      float sy = static_cast<const x_skew*>(s)->y;
      float ey = static_cast<const x_skew*>(e)->y;
      x = n * (ex - sx) + sx;
      y = n * (ey - sy) + sy; */
    }
    virtual ustring to_string() const override {
      return ustring::format(W("matrix(%d,%d,%d,%d,%d,%d)"), mtx.sx, mtx.shy,
                             mtx.shx, mtx.sy, mtx.tx, mtx.ty);
    }
    virtual value to_value() const override {
      function_value *pf = new function_value();
      pf->name           = WCHARS("matrix");
      pf->params.push(value(mtx.sx));
      pf->params.push(value(mtx.shy));
      pf->params.push(value(mtx.shx));
      pf->params.push(value(mtx.sy));
      pf->params.push(value(mtx.tx));
      pf->params.push(value(mtx.ty));
      return value::make_function(pf);
    }
  };

  struct transforms : public resource {
    array<handle<transform_t>> items;
    transforms() {}
    virtual ~transforms() { items.destroy(); }
    // affine_mtx_f mtx;
    void apply(view &v, element *el, affine_mtx_f &to_mtx) {
      index_t nm = items.size();
      for (index_t n = nm - 1; n >= 0; --n)
        items[n]->apply(v, el, to_mtx);
    }

    transforms *clone() const;
    transforms *make_initial() const;
    bool        is_compatible(const transforms *rs) const;
    bool        is_identical(const transforms *rs) const;
    void        morph(view &v, element *b, const transforms *start,
                      const transforms *end, float n);

    static bool is_identical(const transforms *ls, const transforms *rs);

    uint hash() const {
      uint r = items.size();
      FOREACH(i, items)
      tool::hash_combine(r, items[i]->hash());
      return r;
    }

    bool operator==(const transforms &other) const {
      return is_identical(&other);
    }
    bool operator!=(const transforms &other) const {
      return !is_identical(&other);
    }

    ustring to_string() const {
      ustring r;
      index_t nm = items.size();
      for (index_t n = 0; n < nm; ++n) {
        if (r.length()) r += W(" ");
        r += items[n]->to_string();
      }
      return r;
    }

    value to_value() const {
      array<value> ar;
      index_t      nm = items.size();
      for (index_t n = 0; n < nm; ++n)
        ar.push(items[n]->to_value());
      return value::make_array(ar());
    }
  };

  handle<transforms> parse_transform(slice<value> a_values);

struct handler_list_v {
  struct item : public resource_x<item> {
    string                   funcname;
    string                   url;
    dictionary<value, value> params;
  };
  struct items : public resource, public array<handle<item>> {
    items() {}
    items(const items *pi) : array<handle<item>>(*pi) {}

    unsigned int hash() const {
      uint r = 157;
      hash_combine(r, tool::hash(this->size()));
      for (int n = 0; n < this->size(); ++n) {
        const item* pi = this->operator[](n);
        hash_combine(r, tool::hash(pi->funcname));
        hash_combine(r, tool::hash(pi->url));
        hash_combine(r, tool::hash(pi->params));
      }
      return r;
    }

    bool is_equal(const items* other) const {
      if (size() != other->size())
        return false;
      for (int n = 0; n < size(); ++n) {
        const item* pa = this->operator[](n);
        const item* pb = other->operator[](n);
        if (pa->funcname != pb->funcname) return false;
        if (pa->url != pb->url) return false;
        if (pa->params != pb->params) return false;
      }
      return true;
    }

  };

  handle<items> list;
  uint          list_owner;

  handler_list_v() : list_owner(false) {}
  handler_list_v(const handler_list_v &other)
    : list_owner(false), list(other.list) {}
  handler_list_v &operator=(const handler_list_v &other) {
    if (this != &other) {
      list = other.list;
      list_owner = false;
    }
    return *this;
  }

  void inherit(
    const handler_list_v &v) // note, this uses non standard inheritance.
  {
    if (!v.list) return;
    if (!list) {
      list = v.list;
      list_owner = false;
      return;
    }
    if (list == v.list) return;
    for (int n = 0; n < v.list->size(); ++n) {
      const item *pi = v.list->operator[](n);
      set(pi);
    }
  }

  void set(const item *pi) {
    int   existing_idx;
    item *existing = contains_name(pi, existing_idx);
    if (!existing) {
      if (!list_owner) {
        list_owner = true;
        list = list ? new items(list) : new items();
      }
      list->push(pi);
    }
    else if (existing->params != pi->params) {
      if (!list_owner) {
        list_owner = true;
        list = new items(list);
      }
      (*list)[existing_idx] = new item(*pi);
    }
  }

  item *contains_name(const item *pi, int &at_index) {
    if (list)
      for (int n = 0; n < list->size(); ++n) {
        item *tpi = list->operator[](n);
        if (tpi->funcname == pi->funcname /*&& tpi->params == pi->params*/) {
          at_index = n;
          return tpi;
        }
      }
    return nullptr;
  }

  void append(const string &fn, const string &ur,
    dictionary<value, value>
    params) // note, this uses non standard inheritance.
  {
    handle<item> it = new item();
    it->funcname = fn;
    it->url = ur;
    it->params = params;
    set(it);
  }

  // array<shandler> list;
  // margin[0].inherit(src.margin[0]);
  bool operator==(const handler_list_v &other) const {
    if (!list && !other.list) return true;
    if (list && other.list)
      return list->is_equal(other.list);
    return false;
  }
  bool operator!=(const handler_list_v &other) const {
    if (!list && !other.list) return false;
    if (list && other.list)
      return !list->is_equal(other.list);
    return true;
  }

  void   clear() { list = nullptr; }
  string to_string() const {
    if (!list) return string();
    array<char> out;
    for (int n = 0; n < list->size(); ++n) {
      const item *pi = list->operator[](n);
      if (out.length()) out.push(' ');
      out.push(pi->funcname());
      if (pi->url.is_defined()) {
        out.push(CHARS(" url("));
        out.push(pi->url());
        out.push(CHARS(")"));
      }
    }
    return out();
  }
  value to_value() const {
    if (!list) return value();
    array<value> out;
    for (int n = 0; n < list->size(); ++n) {
      const item *pi = list->operator[](n);
      value va[2] = {
        value(pi->funcname()),
        value(ustring(pi->url()), value::UT_URL)
      };
      out.push(value::make_array(items_of(va)));
    }
    return value::make_array(out());
  }

  unsigned int hash() const { return list ? list->hash() : 157; }

  bool undefined() const { return list.ptr() == 0; }
  bool defined() const { return list.ptr() != 0; }
};

  struct rect_style {

    color_v          back_color;
    value_handle<gradient> back_gradient;

    struct image_def {
      image_ref id;
      image_repeat_ev       repeat;      // repeat(tile), expandable(tile)
      image_attachment_ev   attachment;
      size_v    margin[4];   // background-postion [0] [1] / expandable margins (background-repeat:expand)
      size_v    dim[2];      // background-size [0] [1]
      image_filters filters; // list of transformation filters
      clip_ev       clip;
      int_v         frame_no;
      //blend_mode_ev blend_mode;

      image_def() {}

      uint hash() const {
        uint h = 237;
        hash_combine(h, tool::hash(id));
        hash_combine(h, tool::hash(repeat));
        hash_combine(h, tool::hash(attachment));
        hash_combine(h, tool::hash(margin));
        hash_combine(h, tool::hash(dim));
        hash_combine(h, tool::hash(filters));
        hash_combine(h, tool::hash(clip));
        hash_combine(h, tool::hash(frame_no));
        //hash_combine(h, tool::hash(blend_mode));
        return h;
      }

      bool operator==(const image_def& rs) const {
        return id == rs.id
          && repeat == rs.repeat
          && attachment == rs.attachment
          && tool::equal(margin, rs.margin)
          && tool::equal(dim, rs.dim)
          && filters == rs.filters
          && clip == clip
          && frame_no == rs.frame_no;
          //&& blend_mode == rs.blend_mode;
      }

      void inherit(const image_def &src) {
        static_assert(sizeof(*this) % 4 == 0,
                      "image_def sizeof is not compatible with class style");

        id.inherit(src.id);
        margin[0].inherit(src.margin[0]);
        margin[1].inherit(src.margin[1]);
        margin[2].inherit(src.margin[2]);
        margin[3].inherit(src.margin[3]);
        //margin[4].inherit(src.margin[4]);
        repeat.inherit(src.repeat);
        attachment.inherit(src.attachment);
        // transformation_id.inherit( src.transformation_id );
        filters.inherit(src.filters);
        dim[0].inherit(src.dim[0]);
        dim[1].inherit(src.dim[1]);
        clip.inherit(src.clip);
        frame_no.inherit(src.frame_no);
        //blend_mode.inherit(src.blend_mode);
      }

      INLINE rect calc_sections() const {
        rect margins = rect::make_ltrb(margin[0].pixels(), margin[1].pixels(), margin[2].pixels(),margin[3].pixels());
        // if(!id.img()) return margins;
        // if(!section_colors_valid)
        //  section_colors_valid =
        //  id.img()->get_section_colors(margins,section_colors,
        //  section_colors_solid_flags);
        return margins;
      }

    };
    image_def back_image;
    image_def fore_image;

    color_v          fore_color;
    value_handle<gradient> fore_gradient;

    size_v           border_width[4]; // 0-left, 1-top, 2-right, 3-bottom
    color_v          border_color[4];
    border_style_ev  border_style[4];
    size_v           border_radius[8];

    size_v padding_width[4];
    size_v margin[4];
    size_v hit_margin[4];

    box_sizing_ev   box_sizing;
    clip_box_ev     clip_box;

    color_v         outline_color;
    size_v          outline_width;
    border_style_ev outline_style;
    size_v          outline_offset;
    size_v          outline_shift_x;
    size_v          outline_shift_y;

    //border_style_e get_outline_style() const { return outline_style.val(); }

    size_v width;
    size_v height;

    size_v min_width;
    size_v min_height;

    size_v max_width;
    size_v max_height;

    display_ev display;
    //enum_v display_model;
    visibility_ev visibility;

    //enum_str_v floats;
    //float_v    floats;
    value      floats;
    clear_ev   clears;
    valign_ev  vertical_align;
    valign_ev  content_vertical_align;
    halign_ev  horizontal_align;

    flow_v flow;

    overflow_ev overflow_x;
    overflow_ev overflow_y;

    string behavior;
#if defined(SCITER) || defined(SCITERJS)
    string prototype;     // prototype object
    string prototype_url; // script url
#endif
    string context_menu_sel; // context menu CSS selector.
    string context_menu_url; // context menu CSS selector.

    ustring role;

    text_overflow_ev text_overflow; // 0-none (clip), 1 - ellipsis, 2 - path-ellipsis

    handle<cursor> foreground_image_cursor; // cursor for no-repeat gool::image

    position_ev  position;

    size_v border_spacing_x;
    size_v border_spacing_y;
    border_collapse_ev border_collapse;

    opacity_v opacity; // 0-255, 0 - fully transparent, 255 - not transparent

    int_v page_break_before;
    int_v page_break_after;
    int_v page_break_inside;

    size_v left;
    size_v top;
    size_v right;
    size_v bottom;
    int_v  z_index;
    value  content;

    enum_v draggable;   // none | copy-move | only-move | only-copy
    string accept_drop; // CSS selector of elements to be dragged onto this one.
    enum_v drop;        // insert | append | prepend

    size_v transform_origin_x;
    size_v transform_origin_y;

    enum_v popup_aref_point;
    enum_v popup_pref_point;

    popup_attachment_ev popup_attachment;

    value_handle<shadow_def> box_shadow;
    filter_v        filter;
    filter_v        backdrop_filter;

    weak_handle<style_bag> animation_style_bag;
    string          animation_name;
    duration_v      animation_delay;
    duration_v      animation_duration;
    counter_v       animation_iteration_count;
    ease::function  animation_timing_function;
    animation_fill_mode_ev   animation_fill_mode;
    animation_direction_ev   animation_direction;
    animation_play_state_ev  animation_play_state;

    INLINE bool eq(const rect_style &rs) const {
      //return memcmp((byte *)this, &rs, sizeof(*this)) == 0;

      #define PROP(name) && (name == rs.name)

      #define PROP_A(name) && tool::equal(name,rs.name)


      return true
        PROP(back_color)
        PROP(back_gradient)
        PROP(back_image)
        PROP(fore_image)
        PROP(fore_color)
        PROP(fore_gradient)
        PROP_A(border_width)
        PROP_A(border_color)
        PROP_A(border_style)
        PROP_A(border_radius)
        PROP_A(padding_width)
        PROP_A(margin)
        PROP_A(hit_margin)
        PROP(box_sizing)
        PROP(clip_box)
        PROP(outline_color)
        PROP(outline_width)
        PROP(outline_style)
        PROP(outline_offset)
        PROP(outline_shift_x)
        PROP(outline_shift_y)
        PROP(width)
        PROP(height)
        PROP(min_width)
        PROP(min_height)
        PROP(max_width)
        PROP(max_height)
        PROP(display)
        PROP(visibility)
        PROP(floats)
        PROP(clears)
        PROP(vertical_align)
        PROP(content_vertical_align)
        PROP(horizontal_align)
        PROP(flow)
        PROP(overflow_x)
        PROP(overflow_y)
        PROP(behavior)
        PROP(prototype)
        PROP(prototype_url)
        PROP(context_menu_sel)
        PROP(context_menu_url)
        PROP(role)
        PROP(text_overflow)
        PROP(foreground_image_cursor)
        PROP(position)
        PROP(border_spacing_x)
        PROP(border_spacing_y)
        PROP(border_collapse)
        PROP(opacity)
        PROP(page_break_before)
        PROP(page_break_after)
        PROP(page_break_inside)
        PROP(left)
        PROP(top)
        PROP(right)
        PROP(bottom)
        PROP(z_index)
        PROP(content)
        PROP(draggable)
        PROP(accept_drop)
        PROP(drop)
        PROP(transform_origin_x)
        PROP(transform_origin_y)
        PROP(popup_aref_point)
        PROP(popup_pref_point)
        PROP(popup_attachment)
        PROP(box_shadow)
        PROP(filter)
        PROP(backdrop_filter)
        PROP(animation_name)
        PROP(animation_delay)
        PROP(animation_duration)
        PROP(animation_iteration_count)
        PROP(animation_timing_function)
        PROP(animation_fill_mode)
        PROP(animation_direction)
        PROP(animation_play_state)
        ;
#undef PROP
#undef PROP_A


    }

    inline uint hash() const {
      //STATIC_ASSERT((sizeof(rect_style) & 3) == 0);
      //slice<uint> s((const uint *)this, sizeof(rect_style) >> 2);
      //return hash_value(s);
      uint h = 237;

#define PROP(name) hash_combine(h, tool::hash(name));

        PROP(back_color)
        PROP(back_gradient)
        PROP(back_image)
        PROP(fore_image)
        PROP(fore_color)
        PROP(fore_gradient)
        PROP(border_width)
        PROP(border_color)
        PROP(border_style)
        PROP(border_radius)
        PROP(padding_width)
        PROP(margin)
        PROP(hit_margin)
        PROP(box_sizing)
        PROP(clip_box)
        PROP(outline_color)
        PROP(outline_width)
        PROP(outline_style)
        PROP(outline_offset)
        PROP(outline_shift_x)
        PROP(outline_shift_y)
        PROP(width)
        PROP(height)
        PROP(min_width)
        PROP(min_height)
        PROP(max_width)
        PROP(max_height)
        PROP(display)
        PROP(visibility)
        PROP(floats)
        PROP(clears)
        PROP(vertical_align)
        PROP(content_vertical_align)
        PROP(horizontal_align)
        PROP(flow)
        PROP(overflow_x)
        PROP(overflow_y)
        PROP(behavior)
        PROP(prototype)
        PROP(prototype_url)
        PROP(context_menu_sel)
        PROP(context_menu_url)
        PROP(role)
        PROP(text_overflow)
        PROP(foreground_image_cursor)
        PROP(position)
        PROP(border_spacing_x)
        PROP(border_spacing_y)
        PROP(border_collapse)
        PROP(opacity)
        PROP(page_break_before)
        PROP(page_break_after)
        PROP(page_break_inside)
        PROP(left)
        PROP(top)
        PROP(right)
        PROP(bottom)
        PROP(z_index)
        PROP(content)
        PROP(draggable)
        PROP(accept_drop)
        PROP(drop)
        PROP(transform_origin_x)
        PROP(transform_origin_y)
        PROP(popup_aref_point)
        PROP(popup_pref_point)
        PROP(popup_attachment)
        PROP(box_shadow)
        PROP(filter)
        PROP(backdrop_filter)
        PROP(animation_name)
        PROP(animation_delay)
        PROP(animation_duration)
        PROP(animation_iteration_count)
        PROP(animation_timing_function)
        PROP(animation_fill_mode)
        PROP(animation_direction)
        PROP(animation_play_state)

#undef PROP

        return h;

    }

    bool has_animations() const {
      return animation_name.is_defined() && animation_duration && animation_iteration_count;
    }

    box_sizing_e box_sizing_x() const {
      if (box_sizing.is_undefined())
        return scrollable_x() ? box_sizing_padding : box_sizing_content;
      return box_sizing.val();
    }

    box_sizing_e box_sizing_y() const {
      if (box_sizing.is_undefined())
        return scrollable_y() ? box_sizing_padding : box_sizing_content;
      return box_sizing.val();
    }

    int_v overflow() const {
      if (overflow_x.is_undefined() && overflow_y.is_undefined())
        return overflow_x;
      return max(int(overflow_x), int(overflow_y));
    }

    bool want_scrollbar_x() const {
      return overflow_x == overflow_scroll || overflow_x == overflow_auto ||
             overflow_x == overflow_scroll_indicator;
    }
    bool want_scrollbar_y() const {
      return overflow_y == overflow_scroll || overflow_y == overflow_auto ||
             overflow_y == overflow_scroll_indicator;
    }
    bool want_scrollbars() const {
      return want_scrollbar_x() || want_scrollbar_y();
    }

    bool overflows_x() const { return overflow_x > overflow_visible; }
    bool overflows_y() const { return overflow_y > overflow_visible; }

    bool scrollable_x() const {
      return overflow_x == overflow_scroll || overflow_x == overflow_auto || overflow_x == overflow_hidden_scroll ||
        overflow_x == overflow_scroll_indicator;
    }
    bool scrollable_y() const {
      return overflow_y == overflow_scroll || overflow_y == overflow_auto || overflow_y == overflow_hidden_scroll ||
        overflow_y == overflow_scroll_indicator;
    }


    void overflow(int v) {
      overflow_x = v;
      overflow_y = v;
    }
    bool clip_overflow() const {
      return overflow_x > overflow_visible || overflow_y > overflow_visible;
    }

    bool is_display_none() const {
      return display == display_none || visibility == visibility_none;
    }

    bool collapsed() const {
      return is_display_none() || visibility == visibility_collapse;
    }

    bool visible() const {
      return !is_display_none() && visibility == visibility_visible;
    }

    bool take_space() const {
      return !is_display_none() && visibility != visibility_none;
    }

    inline bool af_positioned() const {
      return position == position_absolute || position == position_fixed;
    }

    // symbol_list     role;

    static rect_style null_val;

    rect_style(const rect_style *src = 0) {
      // transform_origin_x.set_percents(50);
      // transform_origin_y.set_percents(50);
      // inherit()
      // scroll_manner_x = value::inherit_val();
      // scroll_manner_y = value::inherit_val();
      if (src) *this = *src;
    }

    void inherit(const rect_style *src, bool background = true);

    void inherit_background(const rect_style *src);

    bool is_empty() const { return *this == null_val; }

    // void store(archive& bs);
    // void fetch(archive& bs);

    static void store_empty(document *doc, attribute_bag &pb);
    void        store(document *doc, attribute_bag &pb);
    void        fetch(document *doc, const attribute_bag &pb);


    // size_v used_border_width(int n) const   { return border_style[n]?
    // border_width[n]:size_v(); }

    // bool back_colors_equal() const {
    //  return back_color[0] == back_color[1] &&
    //    back_color[0] == back_color[2] &&
    //    back_color[0] == back_color[3]; }

    bool border_colors_equal() const {
      return border_color[0] == border_color[1] &&
             border_color[0] == border_color[2] &&
             border_color[0] == border_color[3];
    }

    bool border_widths_equal() const {
      return border_width[0] == border_width[1] &&
             border_width[0] == border_width[2] &&
             border_width[0] == border_width[3];
    }

    bool border_styles_equal() const {
      return border_style[0] == border_style[1] &&
             border_style[0] == border_style[2] &&
             border_style[0] == border_style[3];
    }

    int margin_left() const { return margin[0].pixels(); }
    int margin_top() const { return margin[1].pixels(); }
    int margin_right() const { return margin[2].pixels(); }
    int margin_bottom() const { return margin[3].pixels(); }

    // void paddings(size_v p) {
    //  padding_width[3] = padding_width[2] = padding_width[1] =
    //  padding_width[0] = p;
    //}

    // int padding_left() const { return padding[0].pixels(0); }
    // int padding_top() const { return padding[1].pixels(0); }
    // int padding_right() const { return padding[2].pixels(0); }
    // int padding_bottom() const { return padding[3].pixels(0); }

    inline int h_flex1000() const {
      return width.flex1000() + used_padding(0).flex1000() +
             margin[0].flex1000() + border_width[0].flex1000() +
             used_padding(2).flex1000() + margin[2].flex1000() +
             used_border_width(2).flex1000();
    }

    inline int v_flex1000() const {
      return height.flex1000() + used_padding(1).flex1000() +
             margin[1].flex1000() + border_width[1].flex1000() +
             used_padding(3).flex1000() + margin[3].flex1000() +
             border_width[3].flex1000();
    }

    // inline int v_percent() const {
    //  return height.percent() + used_padding(1).percent() +
    //  margin[1].percent() +
    //         border_width[1].percent() + used_padding(3).percent() +
    //         margin[3].percent() + border_width[3].percent();
    //}

    // inline int padding_h_flex1000() const {
    //  return used_padding(0).flex1000() + used_padding(2).flex1000();
    //}
    // inline int padding_v_flex1000() const {
    //  return used_padding(1).flex1000() + used_padding(3).flex1000();
    //}
    // inline int padding_flex1000() const {
    //  return padding[0].flex1000() + padding[1].flex1000() +
    //         padding[2].flex1000() + padding[3].flex1000();
    //}
    // inline int border_h_flex1000() const {
    //  return border_width[0].flex1000() + border_width[2].flex1000();
    //}
    // inline int border_v_flex1000() const {
    //  return border_width[1].flex1000() + border_width[3].flex1000();
    //}
    // inline int border_flex1000() const {
    //  return border_width[0].flex1000() + border_width[1].flex1000() +
    //         border_width[2].flex1000() + border_width[3].flex1000();
    //}

    size_v used_border_width(int n) const {
      if (border_style[n].val() == border_none) return 0;
      switch (n) {
        case 1:
        case 3:
          if (box_sizing_y() >= box_sizing_border) return 0;
          break;
        case 0:
        case 2:
          if (box_sizing_x() >= box_sizing_border) return 0;
          break;
      }
      if (border_width[n].is_undefined())
        return size_v(size_v::special_values::$medium, size_v::unit_type::as);
      return border_width[n];
    }

    size_v inner_used_border_width(int n) const {
      if (border_style[n].val() == border_none) return 0;
      switch (n) {
      case 1:
      case 3:
        if (box_sizing_y() < box_sizing_border) return 0;
        break;
      case 0:
      case 2:
        if (box_sizing_x() < box_sizing_border) return 0;
        break;
      }
      if (border_width[n].is_undefined())
        return size_v(size_v::special_values::$medium, size_v::unit_type::as);
      return border_width[n];
    }


    size_v used_padding(int n) const {
      switch (n) {
        case 1:
        case 3:
          if (box_sizing_y() >= box_sizing_padding) return 0;
          break;
        case 0:
        case 2:
          if (box_sizing_x() >= box_sizing_padding) return 0;
          break;
      }
      return padding_width[n];
    }

    size_v inner_used_padding(int n) const {
      switch (n) {
        case 1:
        case 3:
          if (box_sizing_y() >= box_sizing_padding) return padding_width[n];
          break;
        case 0:
        case 2:
          if (box_sizing_x() >= box_sizing_padding) return padding_width[n];
          break;
      }
      return 0;
    }

    INLINE bool has_background_color() const {
      return back_color.is_defined() && !back_color.is_transparent();
    }
    INLINE bool has_background_gradient() const {
      return back_gradient && !back_gradient->is_none();
    }

    INLINE bool has_foreground_color() const {
      return fore_color.is_defined() && !fore_color.is_transparent();
    }
    INLINE bool has_foreground_gradient() const {
      return fore_gradient && !fore_gradient->is_none();
    }

    INLINE bool has_background() const {
      return has_background_color()
          || has_background_image()
          || has_background_gradient()
          || backdrop_filter.has_items();
    }

    INLINE bool has_background_image() const {
      return back_image.id.is_defined() && back_image.id.img();
    }

    INLINE bool has_foreground_image() const {
      return fore_image.id.is_defined() && fore_image.id.img();
    }

    INLINE bool has_foreground() const {
      return has_foreground_color() || has_foreground_image() ||
             has_foreground_gradient();
    }

    /*INLINE bool back_transparent() const {
      return !has_background_color() && back_image.id.is_undefined() &&
             border_style[0] == border_none && border_style[1] == border_none &&
             border_style[2] == border_none && border_style[3] == border_none;
    }*/

    INLINE bool fore_transparent() const {
      if (fore_image.id.is_defined() && fore_image.id.img() &&
          !fore_image.id.img()->is_transparent() &&
          (fore_image.repeat == background_repeat ||
           fore_image.repeat == background_stretch))
        return false;
      return true;
    }

    INLINE bool can_scroll_fast() const {
      return back_image.attachment == 0 && fore_image.attachment == 0;
    }

    /*INLINE bool is_solid_background() const
    {
       if( opacity == 255 )
       {
         if( back_image.id.is_defined() && back_image.id.img() )
         {
           if( back_image.repeat == background_stretch &&
    !back_image.id.img()->is_transparent() ) return true; return false;
         }

         if(!no_color(back_color[0]))
         {
           return back_colors_equal();
         }

       }
       return false;
    }*/

    bool has_outline() const { return (outline_style.val() != border_none); }

    /*{
      if( uint(opacity) == 0)
        return;

      if( uint(opacity) == 255 )
      {
        draw_background(v, sf, rc);
        draw_foreground(v, sf, rc);
      }
      else
      {
        gool::rect area = rc;
        gool::size csz = area.size();

        gool::dib32 base(csz); base.clear_all(0xFFFFFFFF); // fully transparent
        {
          gool::dib_graphics tsf(&base, uint(opacity) != 255);
          tsf.offset( area.origin);
          draw_background(v, tsf, rc);
          draw_foreground(v, tsf, rc);
        }

        sf.blend( rc.origin, base, uint(opacity) );

      }
    }*/

    void resolve(resolution_provider &v, const style &for_that,
                 const style &parent_resolved_style);
    // void derive(const style& src);

    inline bool operator==(const rect_style &rs) const { return eq(rs); }
    inline bool operator!=(const rect_style &rs) const { return !eq(rs); }

    void get_pixels_h(int width, gool::rect &margin_widths,
                      gool::rect &border_widths, gool::rect &padding_widths);
    void get_pixels_v(int height, gool::rect &margin_widths,
                      gool::rect &border_widths, gool::rect &padding_widths);

    inline bool has_border(int idx) const {
      return border_style[idx].val() &&
             (border_width[idx].is_undefined() || !border_width[idx].is_zero());
    }
    inline bool has_borders() const {
      return has_border(0) || has_border(1) || has_border(2) || has_border(3);
    }

    inline bool has_transparent_border(int idx) const {
      if (has_border(idx)) {
        if (border_color[idx].is_transparent()) return true;
        border_style_e bs = border_style[idx].val();
        return bs != border_solid; // e.g. dashed and dotted have transparent pixels.
      }
      return false;
    }

    bool has_padding(int idx) const {
      auto p = used_padding(idx);
      return p.is_defined() && !p.is_zero();
    }

    inline bool is_content_changing() const { return content.is_defined(); }

    inline bool shape_is_equal(const rect_style &cs) const {
      //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
      // return
      bool r =

          border_width[0] == cs.border_width[0] &&
          border_width[1] == cs.border_width[1] &&
          border_width[2] == cs.border_width[2] &&
          border_width[3] == cs.border_width[3] &&
          padding_width[0] == cs.padding_width[0] &&
          padding_width[1] == cs.padding_width[1] &&
          padding_width[2] == cs.padding_width[2] &&
          padding_width[3] == cs.padding_width[3] &&
          margin[0] == cs.margin[0] && margin[1] == cs.margin[1] &&
          margin[2] == cs.margin[2] && margin[3] == cs.margin[3] &&

          (width == cs.width) && (height == cs.height) &&
          (min_width == cs.min_width) && (min_height == cs.min_height) &&
          (max_width == cs.max_width) && (max_height == cs.max_height) &&
          //(display == cs.display) &&
          collapsed() == cs.collapsed() && (floats == cs.floats) &&
          (flow == cs.flow) && (overflow_x == cs.overflow_x) &&
          (overflow_y == cs.overflow_y) && (clears == cs.clears) &&
          content == cs.content;

      return r;
    }

    bool is_position_detached() const {
      if (position < position_absolute) return false;
      return (left.is_not_auto() && top.is_not_auto()) ||
             (right.is_not_auto() && bottom.is_not_auto());
    }

    inline bool stops_layout_propagation() const {
      if (overflow_x < overflow_hidden || overflow_y < overflow_hidden)
        return false;
      if (position > position_relative && !left.undefined_or_auto() &&
          !top.undefined_or_auto())
        return true;
      if (display == display_inline) return false;
      // if( display == display_table_cell )
      //  return false;
      return !has_auto_dimensions();
    }

    inline bool has_auto_dimensions() const {
      if (width.is_min_intrinsic() || width.is_max_intrinsic()) return true;
      if (min_width.is_min_intrinsic() || min_width.is_max_intrinsic())
        return true;
      if (max_width.is_min_intrinsic() || max_width.is_max_intrinsic())
        return true;

      if (height.is_undefined()) return true;
      if (height.is_min_intrinsic() || height.is_max_intrinsic() ||
          height.is_auto())
        return true;
      if (min_height.is_min_intrinsic() || min_height.is_max_intrinsic())
        return true;
      if (max_height.is_min_intrinsic() || max_height.is_max_intrinsic())
        return true;
      return false;
    }

    inline bool may_stop_layout_propagation() const {
      if (overflow_x < overflow_hidden || overflow_y < overflow_hidden)
        return false;
      return true;
    }

    inline bool changes_dimension() const {
      return visibility.is_defined() || display.is_defined() ||
             border_width[0].is_defined() || border_width[1].is_defined() ||
             border_width[2].is_defined() || border_width[3].is_defined() ||
             padding_width[0].is_defined() || padding_width[1].is_defined() ||
             padding_width[2].is_defined() || padding_width[3].is_defined() ||
             margin[0].is_defined() || margin[1].is_defined() ||
             margin[2].is_defined() || margin[3].is_defined() ||

             width.is_defined() || height.is_defined() ||
             min_width.is_defined() || min_height.is_defined() ||
             max_width.is_defined() || max_height.is_defined() ||

             (left.is_defined() && right.is_defined()) ||
             (top.is_defined() && bottom.is_defined()) ||

             overflow_x.is_defined() || overflow_y.is_defined() ||
             border_spacing_x.is_defined() || border_spacing_y.is_defined() ||

             is_content_changing();
    }

    inline bool changes_dimension(const rect_style &cs) const {
      bool r =
          //((cs.border_width[0].is_defined() || border_width[0].is_defined())
          //&&
          // border_width[0] != cs.border_width[0]) ||
          //((cs.border_width[1].is_defined() || border_width[1].is_defined())
          //&&
          // border_width[1] != cs.border_width[1]) ||
          //((cs.border_width[2].is_defined() || border_width[2].is_defined())
          //&&
          // border_width[2] != cs.border_width[2]) ||
          //((cs.border_width[3].is_defined() || border_width[3].is_defined())
          //&&
          // border_width[3] != cs.border_width[3]) ||
          used_border_width(0) != cs.used_border_width(0) ||
          used_border_width(1) != cs.used_border_width(1) ||
          used_border_width(2) != cs.used_border_width(2) ||
          used_border_width(3) != cs.used_border_width(3) ||
          ((cs.padding_width[0].is_defined() ||
            padding_width[0].is_defined()) &&
           padding_width[0] != cs.padding_width[0]) ||
          ((cs.padding_width[1].is_defined() ||
            padding_width[1].is_defined()) &&
           padding_width[1] != cs.padding_width[1]) ||
          ((cs.padding_width[2].is_defined() ||
            padding_width[2].is_defined()) &&
           padding_width[2] != cs.padding_width[2]) ||
          ((cs.padding_width[3].is_defined() ||
            padding_width[3].is_defined()) &&
           padding_width[3] != cs.padding_width[3]) ||
          ((margin[0].is_defined() || cs.margin[0].is_defined()) &&
           margin[0] != cs.margin[0]) ||
          ((margin[1].is_defined() || cs.margin[1].is_defined()) &&
           margin[1] != cs.margin[1]) ||
          ((margin[2].is_defined() || cs.margin[2].is_defined()) &&
           margin[2] != cs.margin[2]) ||
          ((margin[3].is_defined() || cs.margin[3].is_defined()) &&
           margin[3] != cs.margin[3]) ||

          ((cs.width.is_defined() || width.is_defined()) &&
           width != cs.width) ||
          ((cs.height.is_defined() || height.is_defined()) &&
           height != cs.height) ||
          ((cs.min_width.is_defined() || min_width.is_defined()) &&
           min_width != cs.min_width) ||
          ((cs.min_height.is_defined() || min_height.is_defined()) &&
           min_height != cs.min_height) ||
          ((cs.max_width.is_defined() || max_width.is_defined()) &&
           max_width != cs.max_width) ||
          ((cs.max_height.is_defined() || max_height.is_defined()) &&
           max_height != cs.max_height) ||
          ((cs.border_spacing_x.is_defined() ||
            border_spacing_x.is_defined()) &&
           border_spacing_x != cs.border_spacing_x) ||
          ((cs.border_spacing_y.is_defined() ||
            border_spacing_y.is_defined()) &&
           border_spacing_y != cs.border_spacing_y) ||
          ((cs.visibility.is_defined() || visibility.is_defined()) &&
           (cs.visibility != visibility));

      return r;
    }

    bool width_depends_on_intrinsic() const {
      return width.is_min_intrinsic() || width.is_max_intrinsic() ||
             min_width.is_min_intrinsic() || min_width.is_max_intrinsic() ||
             max_width.is_min_intrinsic() || max_width.is_max_intrinsic();
    }
    bool height_depends_on_intrinsic() const {
      return height.undefined_or_auto() || height.is_min_intrinsic() ||
             height.is_max_intrinsic() || min_height.is_min_intrinsic() ||
             min_height.is_max_intrinsic() || max_height.is_min_intrinsic() ||
             max_height.is_max_intrinsic();
    }
  };

  static_assert((sizeof(rect_style) % 4) == 0, "wrong alignment of rect_style");

#if 0
  struct action_style {
    static action_style null_val;

    action_style(const action_style *src = 0) {
      if (src) *this = *src;
    }

    inline bool is_empty() const { return *this == null_val; }
    INLINE bool eq(const action_style &rs) const {
      return memcmp((byte *)this, &rs, sizeof(action_style)) == 0;
    }

    inline bool operator==(const action_style &rs) const { return eq(rs); }
    inline bool operator!=(const action_style &rs) const { return !eq(rs); }

    // inline uint hash() const
    //{
    //  const byte* p = (byte*)this;
    //  return hashlittle(p,sizeof(*this), initial);
    //}

    inline bool changes_dimension() const { return false; }

    void inherit(const action_style *src) {
      act_hover_on.inherit(src->act_hover_on);
      act_hover_off.inherit(src->act_hover_off);
      act_active_on.inherit(src->act_active_on);
      act_active_off.inherit(src->act_active_off);
      act_focus_on.inherit(src->act_focus_on);
      act_focus_off.inherit(src->act_focus_off);
      act_click.inherit(src->act_click);
      act_value_changed.inherit(src->act_value_changed);
      act_size_changed.inherit(src->act_size_changed);
      act_assigned.inherit(src->act_assigned);
      act_validate.inherit(src->act_validate);
      act_key_on.inherit(src->act_key_on);
      act_key_off.inherit(src->act_key_off);
      act_double_click.inherit(src->act_double_click);
      act_timer.inherit(src->act_timer);
      act_animation_step.inherit(src->act_animation_step);
      act_animation_end.inherit(src->act_animation_end);
      act_animation_start.inherit(src->act_animation_start);
    }

    handle<eval::conduit> act_hover_on;
    handle<eval::conduit> act_hover_off;
    handle<eval::conduit> act_active_on;
    handle<eval::conduit> act_active_off;
    handle<eval::conduit> act_focus_on;
    handle<eval::conduit> act_focus_off;
    handle<eval::conduit> act_click;
    handle<eval::conduit> act_value_changed;
    handle<eval::conduit> act_size_changed;
    handle<eval::conduit> act_assigned;
    handle<eval::conduit> act_validate;
    handle<eval::conduit> act_key_on;
    handle<eval::conduit> act_key_off;
    handle<eval::conduit> act_double_click;
    handle<eval::conduit> act_timer;
    handle<eval::conduit> act_animation_step;
    handle<eval::conduit> act_animation_end;
    handle<eval::conduit> act_animation_start;
  };


  static_assert((sizeof(action_style) % 4) == 0,
                "wrong alignment of action_style");
#endif

  string combine_behavior(const string &ov, const string &nv);

  extern bool parse_state_flag(const string &val, ui_state &s);

  struct transition_item : resource {
    uint           prop_sym;
    ease::function easef;
    uint_v         duration;
    uint_v         delay;
    ease::function reverse_easef;
    uint_v         reverse_duration;
    uint_v         reverse_delay;
    transition_item() : prop_sym(0), easef(0), reverse_easef(0), delay(0), duration(0) {}
    transition_item(const transition_item &rs)
        : prop_sym(rs.prop_sym), easef(rs.easef), duration(rs.duration), delay(rs.delay),
          reverse_easef(rs.reverse_easef),
          reverse_duration(rs.reverse_duration),
          reverse_delay(rs.reverse_delay) {}
    transition_item &operator=(const transition_item &rs) {
      if (this == &rs) return *this;
      prop_sym         = (rs.prop_sym);
      easef            = (rs.easef);
      duration         = (rs.duration);
      delay            = (rs.delay);
      reverse_easef    = (rs.reverse_easef);
      reverse_duration = (rs.reverse_duration);
      reverse_delay    = (rs.reverse_delay);
      return *this;
    }

    bool valid() const { return easef != 0 || reverse_easef != 0; }
    void clear() {
      easef         = 0;
      reverse_easef = 0;
    }
    uint hash() const {
      uint r = 33;
      tool::hash_combine(r, tool::hash(prop_sym));
      tool::hash_combine(r, tool::hash(easef));
      tool::hash_combine(r, tool::hash(duration));
      tool::hash_combine(r, tool::hash(delay));
      tool::hash_combine(r, tool::hash(reverse_easef));
      tool::hash_combine(r, tool::hash(reverse_duration));
      tool::hash_combine(r, tool::hash(reverse_delay));
      return r;
    }

    bool operator==(const transition_item &rs) const {
      return prop_sym == rs.prop_sym && easef == rs.easef && duration == rs.duration &&
             delay == rs.delay && reverse_easef == rs.reverse_easef &&
             reverse_duration == rs.reverse_duration &&
             reverse_delay == rs.reverse_delay;
    }
    bool operator!=(const transition_item &rs) const { return !operator==(rs); }
  };

  struct transition_def : resource {

    dictionary<uint, transition_item> props;
    value                  delay;
    value                  duration;
    value                  prop_name;
    value                  timing_func;

    transition_def() {}

    static transition_def *null() {
      static handle<transition_def> _null = new transition_def();
      return _null;
    }
  };

  struct animated_effect : transition_item {
    EFFECT_TYPE etype;

    animated_effect() : etype(transition_none) {}
    animated_effect(const transition_item& p) : etype(transition_none), transition_item(p) {}

    bool valid() const {
      if (etype == transition_none) return false;
      return transition_item::valid();
    }

    uint hash() const {
      uint r = transition_item::hash();
      tool::hash_combine(r, etype);
      return r;
    }
    bool operator==(const animated_effect &rs) const {
      return etype == rs.etype && transition_item::operator==(rs);
    }
    bool operator!=(const animated_effect &rs) const {
      return etype != rs.etype || transition_item::operator!=(rs);
    }
  };

  bool parse_effect_type(const string &name, EFFECT_TYPE &et);

  enum PSEUDO_ELEMENT_ID {
    PSEUDO_NONE   = 0,
    PSEUDO_BEFORE = 0x10000,
    PSEUDO_AFTER  = 0x20000,
    PSEUDO_MARKER = 0x40000,
    PSEUDO_SHADE  = 0x80000,
  };

  // abstract source of style properties
  // known implementations:
  //   - struct style - compiled property struct
  //   - struct style_def - selector + style pair
  //   - struct style_prop_list - dictionary of prop/values
  struct style_prop_bag : resource
  {
#if 0
    enum key_item_e {
      key_address, // unique memory address, e.g. style_def* or style*
      key_hash,    // hash value, e.g. the thing produced by apply_attributes()
    };
    struct key_item {
      key_item_e typ;
      uint_ptr   val;

      bool operator==(const key_item& other) const { return typ == other.typ && val == other.val; }
      bool operator!=(const key_item& other) const { return typ != other.typ || val != other.val; }

      unsigned int hash() const {
        unsigned int r = typ ? 6113 : 9679;
        hash_combine(r, (unsigned int)val);
        return r;
      }
    };

    struct key
    {
      mutable unsigned int _hash = 0;
      array<key_item> _items;
      bool is_valid() const { return _items.length() > 0; }
      void push(key_item ki) { _items.push(ki); _hash = 0; }
      bool operator==(const key& other) const { assert(_items.length()); return hash() == other.hash() && _items == other._items; }
      bool operator!=(const key& other) const { assert(_items.length()); return hash() != other.hash() || _items != other._items; }
      unsigned int hash() const { assert(_items.length()); if (!_hash) _hash = _items.hash(); return _hash; }
    };
#endif

    // updates s style by props contained in this thing,
    // returns: true if this thing has importants
    virtual bool apply_to(const element_context& ctx, style &s, uint inherit_mode, bool apply_important) = 0;
    virtual void apply_variables(const element_context& ctx, style &s) = 0;
    virtual bool apply_char_mark_styles_to(const element_context& ctx, style &s, uint inherit_mode, char_mark_t pseudo_id) { return 0; }

    virtual void get_style_set_name(string& style_set) const = 0;
    virtual void get_style_set_name_base(string& style_set) const = 0;
    virtual void get_appearance(appearance_e& appearance) const = 0;

    virtual bool has_importants() const { return false; }
    virtual bool is_unique() const { return false; }

#if 0
    virtual bool get_key_item(key_item& ki) const = 0;
#endif

    virtual bool is_prop_list() const { return false; }
    virtual bool is_prop_map() const { return false; }
    virtual bool is_style() const { return false; }
    virtual bool is_style_def() const { return false; }

    virtual value definition() const = 0;

  };

  struct style_prop_list : public style_prop_bag {

    struct prop_item {
      cssa::symbol_t sym;
      value          val;

      unsigned int hash() const {
        uint r = sym;
        hash_combine(r, val.hash());
        return r;
      }
    };

    array<prop_item> props;
    array<prop_item> important_props;

    dictionary<string, value> actions;
    dictionary<string, value> custom;
    dictionary<string, value> vars;

    weak_handle<style_bag> pbag;

    string style_set_name;
    string style_set_name_base;
    appearance_ev appearance;

    style_prop_list() = default;
    style_prop_list(const style_prop_list& other)
      : props(other.props)
      , important_props(other.important_props)
      , actions(other.actions)
      , custom(other.custom)
      , vars(other.vars)
      , style_set_name(other.style_set_name)
      , style_set_name_base(other.style_set_name_base)
      , appearance(other.appearance)
    {
    }

    virtual bool is_prop_list() const override { return true; }
    virtual bool has_importants() const override { return important_props.length() > 0; }

    virtual bool apply_to(const element_context& ctx, style &s, uint inherit_mode, bool apply_important) override;
    virtual void apply_variables(const element_context& ctx, style &s) override;

    virtual void get_style_set_name(string& ssn) const override {
      if (style_set_name.is_defined())
        ssn = style_set_name;
    }

    virtual void get_style_set_name_base(string& ssn) const override {
      if (style_set_name_base.is_defined())
        ssn = style_set_name_base;
    }

    virtual void get_appearance(appearance_e& appr) const override {
      if (appearance.is_defined())
        appr = appearance.val();
    }


    virtual bool set(cssa::name_or_symbol name, slice<value> vals, bool important = false);
    virtual bool set(cssa::name_or_symbol name, value val, bool important = false);

    void set_action(const string& name, value val);
    void set_custom(const string& name, value val);
    bool set_var(const string& name, value val);

#if 0
    virtual bool get_key_item(key_item& ki) const {
      unsigned int r = 20011;
      hash_combine(r, props.hash());
      hash_combine(r, important_props.hash());
      hash_combine(r, actions.hash());
      hash_combine(r, custom.hash());
      hash_combine(r, vars.hash());
      ki.typ = key_hash;
      ki.val = r;
      return true;
    }
#endif

#pragma TODO("CSS:changes_dimension")
    bool changes_dimension() const {
      return true;
    }

    virtual value definition() const override {
      value map = value::make_map();
      map.set_prop("type", value(WCHARS("default-style")));
      //rd.push(value("text"), value(WCHARS("")));
      map.set_prop("properties", definition_list());
      return map;
    }

    value definition_list() const {
      map_value* mv = new map_value();
      for (auto& pair : props) {
        value k = value(cssa::symbol_name(pair.sym));
        value v = pair.val;
        mv->params[k] = v;
      }
      for (auto& pair : important_props) {
        value k = value(cssa::symbol_name(pair.sym));
        function_value* pf = new function_value();
        pf->name = WCHARS("!important");
        pf->params.push(pair.val);
        mv->params[k] = value::make_function(pf);
      }
      return value::make_map(mv);
    }

    /*ustring to_string() const {
      ustring out;
      for (auto& av: props) {
        if(out.length()) out += WCHARS(";");
        out += ustring(cssa::symbol_name(av.sym));
        out += WCHARS(":");
        out += av.val.to_string();
      }
      return out;
    }*/

  };

  typedef handle<style_prop_list> hstyle_prop_list;

  hstyle_prop_list parse_style_prop_list(document *pd, const ustring& text);

  struct inline_style_prop_list : public style_prop_list {
    virtual value definition() const override {
      value map = value::make_map();
      map.set_prop("type", value(WCHARS("inline-style")));
      map.set_prop("properties", definition_list());
      return map;
    }
  };

  // a_style map
  struct style_prop_map : public style_prop_list {
    typedef style_prop_list super;
    //dictionary<cssa::symbol_t,value> props;
    //dictionary<string, value>        vars;
    STYLE_CHANGE_TYPE udt = NO_CHANGES;

    style_prop_map() : super() {}
    style_prop_map(const style_prop_map& other) : super(other) {}

    virtual bool is_prop_map() const { return true; }

    virtual bool set(cssa::name_or_symbol name, value val, bool important = false) override {
      val.isolate();
      for (int n = 0; n < props.size(); ++n) {
        if (props[n].sym == name.att_sym) {
          if (props[n].val == val) return false;
          props.remove(n);
          if (val.is_undefined())
            return true;
          break;
        }
      }
      if (val.is_undefined())
        return false;
      udt = max(udt, cssa::symbol_change_type(name.att_sym));
      return super::set(name, val, false);
    }

#if 0
    virtual bool get_key_item(key_item& ki) const {
      // this thing should not participate in cacheable styles
      return false;
    }
#endif

    STYLE_CHANGE_TYPE changes() const {
      return udt;
    }

    virtual value definition() const override {
      value map = value::make_map();
      map.set_prop("type", value(WCHARS("runtime-style")));
      //rd.push(value("text"), value(WCHARS("")));
      map.set_prop("properties", definition_list());
      return map;
    }
  };

  typedef handle<style_prop_map> hstyle_prop_map;


  struct style
      : public style_prop_bag
      , public char_style
      , public para_style
      , public rect_style
#if 0
      ,  public action_style
#endif
  {
    string       name;
    bool         unique;
    bool         animated;
    bool         not_empty;

    //style_prop_bag::key _key; // used by instatniation
    mutable uint hash_v = 0;

    handle<gool::font> font;

    handle<animated_effect>  transition_effect;
    handle<transition_def>   transitions;
    handle<html::transforms> transforms;

#if defined(SCITER) || defined(SCITERJS)
    handler_list_v handler_list;
#endif

#ifdef _DEBUG
    static int _total;
#endif

    attribute_bag_v    custom;     // custom attributes
    attribute_bag_v    vars;       // variables, local and inherited
    handle<style>      importants; // attributes that have !important flag set

    uint pseudo_elements = PSEUDO_NONE; // ORed set of flags - ::after, ::before, ::marker, ::shade exist for the element

    /*struct span_mark {
      uint                id;
      handle<html::style> style;
      bool operator==(const span_mark &rs) const { return id == rs.id && style == rs.style; }
      bool operator!=(const span_mark &rs) const { return id != rs.id || style != rs.style; }
    };*/
    //array<span_mark> span_marks;

    void *operator new(size_t sz) {
      byte *nb = new byte[sz];
      if (nb) memset(nb, 0, sz);
      return nb;
    }
    void operator delete(void *pv) {
      byte *pb = (byte *)pv;
      delete[] pb;
    }

    style(const string &n)
        : name(n), unique(false), animated(false), not_empty(false) {
#ifdef _DEBUG
      ++_total;
#endif
    }
    style() : unique(false), animated(false), not_empty(false) {
#ifdef _DEBUG
      ++_total;
#endif
    }

    /*    style(const rect_style& rs): hash_v(0), unique(false), font(0)
        {
          *((rect_style*)this) = rs;
    #ifdef _DEBUG
        ++_total;
    #endif
        }*/

    style(const style &rs)
        : name(rs.name)
      , char_style(rs)
      , para_style(rs)
      , rect_style(rs)
#if 0
      , action_style(rs)
#endif
      , unique(false)
      , not_empty(rs.not_empty)
      , transforms(rs.transforms)
      , transitions(rs.transitions)
      , transition_effect(rs.transition_effect)
      , pseudo_elements(rs.pseudo_elements)
      , handler_list(rs.handler_list)
      , animated(false), vars(rs.vars) {
#ifdef _DEBUG
      ++_total;
#endif
    }

    ~style() {
#ifdef _DEBUG
      --_total;
#endif
    }

    virtual bool is_style() const { return true; }

    static style *create_unique(const style* from = nullptr) {
      style *t  = new style();
      if (from) *t = *from;
      t->unique = true;
      return t;
    }

#if 0
    virtual bool get_key_item(key_item& ki) const {
      if (unique) // this style cannot participate in key creation,
                  // means: it cannot be cached / shared
        return false;
      ki.typ = key_address;
      ki.val = (uint_ptr) this;
      return true;
    }
#endif

    enum {
      INHERIT_CHAR        = 0x01,
      INHERIT_PARA        = 0x02,
      INHERIT_RECT_BACK   = 0x04,
      INHERIT_RECT        = 0x08 + 0x04,
      INHERIT_ACT         = 0x10,
      INHERIT_TRNS        = 0x20,
      INHERIT_X           = 0x40,
      INHERIT_ALL         = 0x7F,
      INHERIT_INHERITABLE = INHERIT_CHAR | INHERIT_PARA,

    };

    virtual bool apply_to(const element_context& ctx, style &s, uint inherit_mode, bool apply_important) override;
    virtual void apply_variables(const element_context& ctx, style &s) override;
    virtual void get_style_set_name(string& style_set_name) const override { if (style_set.is_defined()) style_set_name = style_set; }
    virtual void get_style_set_name_base(string& style_set_name) const override { if (style_set_base.is_defined()) style_set_name = style_set_base; }
    virtual void get_appearance(appearance_e& appr) const override { if (appearance.is_defined()) appr = appearance.val(); }

    string used_style_set_name() const { return appearance == appearance_none ? style_set_base : style_set; }

    virtual bool has_importants() const override { return importants != nullptr; }


    bool has_custom_atts() const { return custom.items.size() != 0; }

    bool get_custom_attr(name_or_symbol ns, value &v) const {
      return custom.get(ns, v);
    }

    inline style *get_importants(bool create = false) {
      if (create && !importants) importants = new style();
      return importants;
    }

    inline void inherit(const style *src, int what = INHERIT_ALL) {
      if (!src) return;
      not_empty = true;
      if (what & INHERIT_CHAR) char_style::inherit(src);
      if (what & INHERIT_PARA) para_style::inherit(src);
      if (what & INHERIT_RECT)
        rect_style::inherit(src, (what & INHERIT_RECT_BACK) != 0);
      if (what == INHERIT_ALL && src->custom.size()) {
        custom.inherit(src->custom);
        //vars.inherit(src->vars);
      }
      //if (((what & INHERIT_INHERITABLE) == INHERIT_INHERITABLE) &&
      //    src->variables.size())
      //  variables.inherit(src->variables);
#if 0
      if (what & INHERIT_ACT) action_style::inherit(src);
#endif
      if (what & INHERIT_TRNS) {
        if(src->transitions)
          transitions = src->transitions;
        if(src->transition_effect)
          transition_effect = src->transition_effect;
      }
      if ((what & INHERIT_X)) {
        if (src->transforms) transforms = src->transforms;
        //handler_list.inherit(src->handler_list);
      }
    }

    //void set_key(key k) {
    //  _key = k;
    //}

    inline uint hash() const {
#if 0
      return _key.hash();
#else
      if (hash_v) return hash_v;

      uint r = tool::hash<string>(name);

      tool::hash_combine(r, char_style::hash());
      tool::hash_combine(r, para_style::hash());
      tool::hash_combine(r, rect_style::hash());
      if (transforms) tool::hash_combine(r, tool::hash(transforms->hash()));
      if (transition_effect) tool::hash_combine(r, tool::hash(transition_effect->hash()));
      tool::hash_combine(r, pseudo_elements);
      tool::hash_combine(r, handler_list.hash());

      // participate in hash as styles with custom attributes are not getting
      // into the common hash table.
      if (!vars.is_empty()) tool::hash_combine(r, vars.hash());
      hash_v = r;
      return r;
#endif
    }

    bool operator!=(const style &rs) const {
#if 0
      return _key != rs._key;
#else
      if (!char_style::eq(rs)) return true;
      if (!para_style::eq(rs)) return true;
      if (!rect_style::eq(rs)) return true;
#if 0
      if (!action_style::eq(rs)) return true;
#endif
      if (!transition_effect.is_identical(rs.transition_effect)) return true;
      if (!transforms.is_identical(rs.transforms)) return true;
      if (pseudo_elements != rs.pseudo_elements) return true;
      if (handler_list != rs.handler_list) return true;
      if (custom != rs.custom) return true;
      if (vars != rs.vars) return true;
      return false;
#endif
    }
    INLINE bool operator==(const style &rs) const {
#if 0
      return _key == rs._key;
#else
      return char_style::eq(rs)
        && para_style::eq(rs)
        && rect_style::eq(rs)
#if 0
        && action_style::eq(rs)
#endif
        && transition_effect.is_identical(rs.transition_effect)
        && transforms.is_identical(rs.transforms)
        && pseudo_elements == rs.pseudo_elements
        && handler_list == rs.handler_list
        && custom == rs.custom
        && vars == rs.vars;
#endif
    }

    inline bool shape_is_equal(const style &rs) const {
      if (this == &rs) return true;
      return char_style::shape_is_equal(rs) && para_style::shape_is_equal(rs) &&
             rect_style::shape_is_equal(rs);
    }

    inline bool changes_dimension() const {
      return char_style::changes_dimension()
        || para_style::changes_dimension()
        || rect_style::changes_dimension()
#if 0
        || action_style::changes_dimension()
#endif
        ;
    }

    inline bool changes_dimension(const style &other) const {
      return char_style::changes_dimension(other) ||
             para_style::changes_dimension(other) ||
             rect_style::changes_dimension(other);
    }

    inline bool changes_position(const style &other) const {
      if (position != other.position) return true;
      if (z_index != other.z_index) return true;
      if (position >= position_relative) {
        if (collapsed() != other.collapsed()) return true;
        return left != other.left || right != other.right || top != other.top ||
               bottom != other.bottom;
      }
      return false;
    }

    inline static bool different_models(const style *a, const style *b) {
      if (a == b) return false;
      if (a && b) return a->changes_model(*b);
      return a != nullptr;
    }

    inline bool changes_model(const style &other) const {
      if (display.is_defined() && display != other.display) return true;
      if (visibility.val() != other.visibility.val()) return true;
      if (flow.is_defined() && flow != other.flow) return true;
      if (content.is_defined() && content != other.content) return true;
      if (overflow_x.is_defined() && overflow_x != other.overflow_x)
        return true;
      if (overflow_y.is_defined() && overflow_y != other.overflow_y)
        return true;
      if (pseudo_elements != other.pseudo_elements) return true;
      if (position != other.position) return true;
      if (floats != other.floats) return true;
      if (font_size != other.font_size) return true;
      if (font_family != other.font_family) return true;
      if (font_style != other.font_style) return true;
      if (font_weight != other.font_weight) return true;
      if (font_rendering_mode.is_defined() && font_rendering_mode != other.font_rendering_mode) return true;
      if (font_variant_ligatures != other.font_variant_ligatures) return true;
      if (font_variant_caps != other.font_variant_caps) return true;
      if (text_transform != other.text_transform) return true;
      if (behavior.is_defined() && behavior != other.behavior) return true;
      return false;
    }

    inline bool changes_model() const {
      return display.is_defined() || flow.is_defined() ||
        content.is_defined() || overflow_x.is_defined() ||
        overflow_y.is_defined() ||
        //after.is_defined() || before.is_defined() ||
        position.is_defined() ||
        floats.is_defined() || visibility.is_defined() ||
        font_size.is_defined() || font_family.is_defined() ||
        font_style.is_defined() || font_weight.is_defined() ||
        font_rendering_mode.is_defined() ||
        font_variant_ligatures.is_defined() ||
        font_variant_caps.is_defined();
    }

    inline bool changes_position() const {
      return left.is_defined() || right.is_defined() || top.is_defined() ||
             bottom.is_defined();
    }

    inline bool is_content_changing() const {
      return char_style::is_content_changing() ||
             rect_style::is_content_changing();
      // para_style::changes_dimension() ||
      // action_style::changes_dimension();
    }

    inline bool has_transition_effect() const {
      if (!transition_effect) return false;
      if (transition_effect->etype == transition_none) return false;
      return true;
    }

    inline bool is_inline_block() const {
      if (is_display_none()) return false;
      return display == display_inline_block || display == display_inline_table;
    }

    inline bool is_block() const {
      if (is_display_none()) return false;
      return !is_inline_block();
    }

    inline bool is_inline_span() const {
      if (is_display_none()) return false;
      return display == display_inline || display == display_contents;
    }

    bool may_need_ellipsis() const {
      return (overflow_x == overflow_hidden || overflow_x == overflow_hidden_scroll) && !can_wrap() && text_overflow;
    }
    bool may_need_multiline_ellipsis() const {
      return overflow_y == overflow_hidden && !cannot_wrap() && text_overflow;
    }

    style &operator=(const style &rs) {
      set(rs);
      return *this;
    }

    void set(const style &rs) {
      name                    = rs.name;
      not_empty               = rs.not_empty;
      hash_v                  = rs.hash_v;
      *((char_style *)this)   = *((const char_style *)&rs);
      *((para_style *)this)   = *((const para_style *)&rs);
      *((rect_style *)this)   = *((const rect_style *)&rs);
#if 0
      *((action_style *)this) = *((const action_style *)&rs);
#endif
      custom                  = rs.custom;
      vars                    = rs.vars;
      transitions             = rs.transitions;
      transforms              = rs.transforms;
/*      after                   = rs.after;
      before                  = rs.before;
      shade                   = rs.shade;
      marker                  = rs.marker;
      pseudo_element_id       = rs.pseudo_element_id;*/
      pseudo_elements         = rs.pseudo_elements;
      handler_list            = rs.handler_list;
      // ss.clear();
    }

    // gool::text_format get_text_format(view &v, size box = size(24));

    bool is_defined() const {
      return !char_style::is_empty() || !rect_style::is_empty() ||
             !para_style::is_empty() || transforms;
    }

    bool is_incomplete() const {
      if (fore_image.id.is_defined()) {
        gool::image *img = fore_image.id.img();
        if (img == 0) return true;
      }
      /*
      if(back_image.id.is_defined())
      {
        gool::image* img = back_image.id.img();
        if( img == 0 )
          return true;
      }
      if(list_style_image.is_defined())
      {
        gool::image* img = list_style_image.img();
        if( img == 0 )
          return true;
      }*/
      return false;
    }

    bool has_image(const image_ref &ir) const {
      return fore_image.id == ir || back_image.id == ir ||
             list_style_image == ir;
    }

    // int pixels(const size_v& sz, view& v) const;
    void         resolve(view &v, const element* el, const style &parent_resolved_style);
    //gool::argb   resolve_color(color_v c) const;
    //virtual bool resolve_color(uint attr_sym, color_v &val) const override;
    //size_v       resolve_length(size_v v) const;

    //tool::ustring string_value(const string &name) const;
    //tool::ustring string_value(uint css_att) const;

    tool::value   to_value(const string &name) const;
    tool::value   to_value(uint css_att) const;

    struct property_callback {
      virtual void on_prop(const tool::string &name,
                           const tool::value & val) = 0;
    };
    void enum_properties(property_callback &cb) const {
      for (uint n = 0; n < CSSA_ATTR_MAX; ++n) {
        tool::string name  = cssa::symbol_name(n);
        tool::value  value = to_value(n);
        if (!value.is_undefined())
          cb.on_prop(name, value);
      }
    }

    INLINE bool has_solid_background() const {
      if (uint(opacity) != 255) return false;

      if (back_color.is_defined()) {
        if (back_color.to_argb().alfa == 255) return true;
      }

      do {
        if (!back_image.id.is_defined()) return false;
        if (!back_image.id.img()) return false;
        if (back_image.id.img()->is_transparent()) return false;
        if (back_image.repeat == background_repeat)
          break;
        if (back_image.repeat == background_stretch)
          break;
        if ((back_image.dim[0] == back_image.dim[1]) && back_image.dim[1].is_literal(size_v::special_values::$cover) )
          break;
        return false;
      } while (false);

      if (has_rounded_corners())
        return false;

      if (has_transparent_border(0) || has_transparent_border(1) || has_transparent_border(2) || has_transparent_border(3))
        return false;
      return true;
    }

    INLINE bool is_transparent() const {
      if (uint(opacity) != 255) return true;

      while (back_color.is_undefined() ||
             !back_color.to_argb().is_opaque()) {
        if (this->back_gradient && !this->back_gradient->is_transparent())
          break;
        if (!back_image.id.is_defined()) return true;
        if (!back_image.id.img()) return true;
        if (back_image.id.img()->is_transparent()) return true;

        if (back_image.repeat != background_repeat &&
            back_image.repeat != background_stretch &&
            back_image.repeat != background_expand)
          return true;

        if (!fore_transparent()) return false;

        break;
      }
      /*if (back_image.offset[0].is_defined() ||
          back_image.offset[1].is_defined() ||
          back_image.offset[2].is_defined() ||
          back_image.offset[3].is_defined())
        return true;*/

      return has_transparent_border(0) || has_transparent_border(1) ||
             has_transparent_border(2) || has_transparent_border(3);
      // return false;
    }

    INLINE bool has_percentage_widths() const {
      if (text_indent.is_percent()) return true;
      if (width.is_percent()) return true;
      if (min_width.is_percent()) return true;
      if (max_width.is_percent()) return true;
      if (margin[0].is_percent() || margin[2].is_percent()) return true;
      if (padding_width[0].is_percent() || padding_width[2].is_percent())
        return true;
      // if( border_width[0].is_percent() || border_width[2].is_percent())
      // return true;
      return false;
    }

    INLINE bool has_rounded_corners() const {
      for (int n = 0; n < 8; n += 2)
        if (border_radius[n].is_defined() && !border_radius[n].is_zero() &&
            border_radius[n + 1].is_defined() &&
            !border_radius[n + 1].is_zero())
          return true;
      return false;
    }

    bool get_rounded_corners(view &v, element* b, size &rtl, size &rtr, size &rbr,
                             size &rbl, size borderboxsz) const;

    INLINE bool need_offscreen_drawing() const {
      return has_rounded_corners()
          //|| has_csss_draw()
          //|| has_gradient_back()
          ;
    }

    bool add_transition(const transition_item &p);

    void fetch_images(view &v, document *pd);

    void background_outline(view &v, graphics *sf, rect box, gool::rect &rect, handle<gool::path> &path, element* site);

    void draw_background(view &v, graphics *sf, const rect &backrc, element *site /*can be null*/, point pos);
    void _draw_background(view &v, graphics *sf, const rect &backrc, element *site /*can be null*/);
    bool _draw_background(view &v, graphics *sf, const path* p, element *site /*can be null*/);

    void draw_border(view &v, graphics *sf, const rect &borderrc, const rect &border_width, element *site /*can be null*/, point pos);
    void _draw_border(view &v, graphics *sf, const rect &borderrc, const rect &border_width, element *site /*can be null*/);
    void draw_foreground(view &v, graphics *sf, const rect &backrc, element *site /*can be null*/, point pos);
    void draw_image(view &v, graphics *sf, const rect_style::image_def &imd, const gool::rect &backrc, bool foreground, element *site /*can be null*/);
    void draw_outline(view &v, graphics *sf, const rect &border_rc, element *site, point pos);
    void draw_shape(view &v, graphics *sf, const gool::rect &rc, element *site);
    void draw_box_shadow(view &v, graphics *sf, const rect &rcborder, const rect &border_width, element *site);

    halign_e get_text_align() const {
      switch (text_align.val()) {
        default:
        case align_auto:
          return direction == direction_rtl ? align_right : align_left;
        case align_left:
          return this->mapping.u.parts.alignment == mapping_left_to_right ?
                     align_right :
                     align_left;
        case align_right:
          return this->mapping.u.parts.alignment == mapping_left_to_right ?
                     align_left :
                     align_right;
        case align_center: return align_center;
        case align_justify: return align_justify;
        case align_start:
          return direction == direction_ltr ? align_left : align_right;
        case align_end:
          return direction == direction_rtl ? align_left : align_right;
      }
    }

    halign_e get_horizontal_align() const {
      switch (horizontal_align) {
        default:
        case align_justify:
        case align_auto:
          return direction == direction_rtl ? align_right : align_left;
        case align_left:
          return this->mapping.u.parts.alignment == mapping_left_to_right ?
                     align_right :
                     align_left;
        case align_right:
          return this->mapping.u.parts.alignment == mapping_left_to_right ?
                     align_left :
                     align_right;
        case align_center:
          return halign_e(horizontal_align.val(align_left));
        case align_start:
          return direction == direction_ltr ? align_left : align_right;
        case align_end:
          return direction == direction_rtl ? align_left : align_right;
      }
    }

    valign_e get_inline_vertical_align() const {
      if (!this->v_flex1000()) {
        if (this->is_inline_block() || this->is_inline_span())
          return (valign_e)vertical_align.val(valign_baseline);
      }
      return valign_middle;
    }

    valign_e get_block_vertical_align() const {
      if (content_vertical_align.is_defined()) {
        valign_e va = content_vertical_align.val(valign_top);
        switch (va) {
          default:
          case valign_auto: return valign_top;
          case valign_top: return valign_top;
          case valign_middle: return valign_middle;
          case valign_bottom: return valign_bottom;
        }
      }
      if (display == display_inline_block || display == display_inline)
        return valign_top;
      valign_e va = vertical_align.val(valign_top);
      if (va == valign_auto) return valign_top;
      return va;
    }

    valign_e get_text_vertical_align() const {
      valign_e va = vertical_align.val(valign_baseline);
      if (va == valign_auto) return valign_baseline;
      return va;
    }

    INLINE float_e get_float() const {
      if(floats.is_enum<float_e>())
        return (float_e)floats.get_enum_code();
      return float_none;
    }

    INLINE float_e used_float() const {
      float_e t = get_float();
      switch (mapping.u.parts.layout) {
        case mapping_left_to_right:
          if (t == float_left) return float_right;
          else if (t == float_right) return float_left;
          // else fall through
        default: return t;
      }
    }


    INLINE bool clears_left() const {
      switch (mapping.u.parts.layout) {
        case mapping_left_to_right: return (clears & clear_right) != 0;
        default: return (clears & clear_left) != 0;
      }
    }
    INLINE bool clears_right() const {
      switch (mapping.u.parts.layout) {
        case mapping_left_to_right: return (clears & clear_left) != 0;
        default: return (clears & clear_right) != 0;
      }
    }

    bool smooth_scroll(bool y) const {
      value scroll_manner = y ? scroll_manner_y : scroll_manner_x;
      if (!scroll_manner.is_function()) return SMOOTH_SCROLL_DEFAULT;
      function_value *fv = scroll_manner.get_function();
      static value    k  = value(WCHARS("animation"));
      value           v  = fv->get(k);
      return v.get(SMOOTH_SCROLL_DEFAULT);
    }

    bool smooth_scroll_page(bool y) const {
      value scroll_manner = y ? scroll_manner_y : scroll_manner_x;
      if (!scroll_manner.is_function()) return smooth_scroll(y);
      function_value *fv = scroll_manner.get_function();
      static value    k  = value(WCHARS("page-animation"));
      value           v  = fv->get(k);
      return v.get(smooth_scroll(y));
    }
    bool smooth_scroll_step(bool y) const {
      value scroll_manner = y ? scroll_manner_y : scroll_manner_x;
      if (!scroll_manner.is_function()) return smooth_scroll(y);
      function_value *fv = scroll_manner.get_function();
      static value    k  = value(WCHARS("step-animation"));
      value           v  = fv->get(k);
      return v.get(smooth_scroll(y));
    }

    value scroll_step(bool y) const {
      value scroll_manner = y ? scroll_manner_y : scroll_manner_x;
      if (!scroll_manner.is_function()) return value();
      function_value *fv = scroll_manner.get_function();
      static value    k  = value(WCHARS("step"));
      return fv->get(k);
    }
    value scroll_page(bool y) const {
      value scroll_manner = y ? scroll_manner_y : scroll_manner_x;
      if (!scroll_manner.is_function()) return value();
      function_value *fv = scroll_manner.get_function();
      static value    k  = value(WCHARS("page"));
      return fv->get(k);
    }

    bool smooth_scroll_home(bool y) const {
      value scroll_manner = y ? scroll_manner_y : scroll_manner_x;
      if (!scroll_manner.is_function()) return smooth_scroll(y);
      function_value *fv = scroll_manner.get_function();
      static value    k  = value(WCHARS("home-animation"));
      value           v  = fv->get(k);
      return v.get(smooth_scroll(y));
    }
    bool smooth_scroll_wheel(bool y) const {
      value scroll_manner = y ? scroll_manner_y : scroll_manner_x;
      if (!scroll_manner.is_function()) return smooth_scroll(y);
      function_value *fv = scroll_manner.get_function();
      static value    k  = value(WCHARS("wheel-animation"));
      value           v  = fv->get(k);
      return v.get(smooth_scroll(y));
    }
    size_v wheel_step(bool y, size_v df) const {
      value scroll_manner = y ? scroll_manner_y : scroll_manner_x;
      if (!scroll_manner.is_function()) return df;
      function_value *fv = scroll_manner.get_function();
      static value    k  = value(WCHARS("wheel-step"));
      value           v  = fv->get(k);
      return v.is_defined() ? size_v(v) : df;
    }

    virtual value definition() const override;

  };

  struct style_list
  {
    bool   has_uniques = false;
    string style_set;
    string style_set_base;
    appearance_e appearance = appearance_auto;
    uint   n_importants = 0;
    uint   pseudo_element_ids = 0;

    struct list_item {
      uint inherit_mode;
      handle<style_prop_bag> prop_bag;
      weak_handle<html::style_bag> style_bag;
    };

    array<list_item> list; // rules applicable to the element itself
    array<list_item> marks_list; // ::mark(name) rules only

    void add(style* ps, uint inherit_mode) {
#ifdef DEBUG
      if (ps->unique && !has_uniques)
        ps = ps;
        //dbg_report("unique style for");
#endif
      if(ps) has_uniques = has_uniques || ps->unique;
      list_item li = { inherit_mode, ps };
      ps->get_style_set_name(style_set);
      ps->get_style_set_name_base(style_set_base);
      ps->get_appearance(appearance);
      list.push(li);
    }

    void add(style_prop_bag* ps, uint inherit_mode) {
      list_item li = { inherit_mode, ps };
      ps->get_style_set_name(style_set);
      ps->get_style_set_name_base(style_set_base);
      ps->get_appearance(appearance);
      if (ps->has_importants())
        ++n_importants;
      list.push(li);
    }

    void marks_add(style_prop_bag* ps, uint inherit_mode) {
      list_item li = { inherit_mode, ps };
      marks_list.push(li);
    }

    void apply_variables(const element_context& ctx, style &s) {
      for (auto& li : list)
        li.prop_bag->apply_variables(ctx, s);
    }

    bool apply_to(const element_context& ctx, style &s, bool apply_important) {
      uint important_counter = 0;
      for (auto& li : list) {
        if (li.prop_bag->apply_to(ctx, s, li.inherit_mode, apply_important))
          ++important_counter;
      }
      s.pseudo_elements = this->pseudo_element_ids;
      return important_counter > 0;
    }

    bool apply_char_mark_styles_to(const element_context& ctx, style &s, char_mark_t pseudo_id) {
      uint counter = 0;
      for (auto& li : marks_list) {
        if (li.prop_bag->apply_char_mark_styles_to(ctx, s, li.inherit_mode, pseudo_id))
          ++counter;
      }
      return counter > 0;
    }

    void append(const style_list& other)
    {
      list.push(other.list());
      marks_list.push(other.marks_list());

      has_uniques = has_uniques || other.has_uniques;
      style_set.inherit(other.style_set);

      n_importants += other.n_importants;
      pseudo_element_ids |= other.pseudo_element_ids;
    }

#ifdef DEBUG
    void dbg_report() {
      dbg_printf("style rules:\n");
      int i = 0;
      for (auto& li : list) {
        ustring s = li.prop_bag->definition().to_string();
        dbg_printf("rule %d:%S\n", ++i, s.c_str());
      }
    }
#endif

  };

  struct style_set_holder;

#pragma pack(pop)

  static_assert((sizeof(style) % 4) == 0, "wrong alignment of style");

  typedef tool::handle<style> hstyle;
  // typedef hash_table<string,hstyle> style_bag;

  #define TAG_ANY uint(-1) // `*` selector

  class style_def : public style_prop_bag {
  public:
    friend class style_bag;

    string  src_url;
    int     src_line_no;
    ustring selector_text;
    uint    tag;
    string  tag_name;
    ustring id;
    struct attr_def {
      uint    attr_name;
      ustring attr_value;
      uint    attr_eq_op;
      attr_def() : attr_name(0), attr_eq_op(0) {}
    };
    tool::array<attr_def> atts;

    string            seqid; // style sheet sequentional id
    uint              no;
    uint              depth;
    uint              inheritance_level = 0; // filled by owning style set
    handle<style_def> next;     // in context
    handle<style_def> negation; // not thing

    enum SIBLING_TYPE {
      ST_NONE = 0,
      ST_ADJACENT = 1,
      ST_NEXT = 2,
    };

    SIBLING_TYPE sibling;

    int index;
    int step;
    int last_index;
    int last_step;

    tag::symbol_t
        child_of_type; // has-child-of-type(menu), symbol of the 'menu'
    tag::symbol_t
         children_of_type;   // has-children-of-type(menu), symbol of the 'menu'
    bool only_child_of_type; // only-of-type(menu), symbol of the 'menu'
    bool only_child; // only-child(), that is the only child of its owner

    ui_state pseudos;        // block_states: active, hover
    enum_v   pseudo_element; // id defined it is one of PSEUDO_ELEMENT_TYPES:
                             // 0-before, 1-after
    bool has_bound_attributes;

    ustring lang;
    ustring theme;

    handle<eval::conduit> media_expr;

    uint usage_counter;

  public:
    //hstyle pstyle;
    handle<style_prop_list> plist;

    style_def()
        : no(0), tag(0), next(0), depth(0), index(0), step(0), last_index(0),
          last_step(0), sibling(ST_NONE), negation(0), child_of_type(0),
          children_of_type(0), only_child_of_type(false), only_child(false),
          src_line_no(0), usage_counter(0), has_bound_attributes(false) {}

    style_def(const style_def &proto)
        : src_url(proto.src_url), src_line_no(proto.src_line_no),
          selector_text(proto.selector_text), tag(proto.tag),
          tag_name(proto.tag_name), id(proto.id), atts(proto.atts),
          seqid(proto.seqid), no(proto.no), depth(proto.depth),
          negation(proto.negation), sibling(proto.sibling),
          index(proto.index), step(proto.step), last_index(proto.last_index),
          last_step(proto.last_step), child_of_type(proto.child_of_type),
          children_of_type(proto.children_of_type),
          only_child_of_type(proto.only_child_of_type),
          only_child(proto.only_child), pseudos(proto.pseudos),
          pseudo_element(proto.pseudo_element), lang(proto.lang),
          media_expr(proto.media_expr),
          has_bound_attributes(proto.has_bound_attributes) {}

    ~style_def() {}

    virtual bool is_style_def() const { return true; }

    virtual bool apply_to(const element_context& ctx, style &s, uint inherit_mode, bool apply_important) override;
    virtual void apply_variables(const element_context& ctx, style &s) override;
    virtual bool apply_char_mark_styles_to(const element_context& ctx, style &s, uint inherit_mode, char_mark_t pseudo_id) override;
    virtual void get_style_set_name(string& style_set) const override { plist->get_style_set_name(style_set); }
    virtual void get_style_set_name_base(string& style_set) const override { plist->get_style_set_name_base(style_set); }
    virtual void get_appearance(appearance_e& appr) const override { plist->get_appearance(appr); }

    virtual bool has_importants() const override { return plist->has_importants(); }
#if 0
    virtual bool get_key_item(key_item& ki) const {
      ki.typ = key_address;
      ki.val = (uint_ptr) this;
      return true;
    }
#endif

    // deep copy
    style_def *copy() const {
      style_def *cpy = new style_def(*this);
      if (next) cpy->next = next->copy();
      return cpy;
    }

    // last rule in chain
    style_def *last() const {
      if (next) return next->last();
      return const_cast<style_def *>(this);
    }

    bool is_tagname_only() const {
      if (tag_name.length() == 0) return false;
      if (atts.size()) return false;
      if (id.length()) return false;
      if (next) return false;
      if (sibling) return false;
      if (index || step || last_index || last_step) return false;
      if (pseudos.is_defined()) return false;
      return true;
    }

    static bool       parse_list(style_bag *psb, css_istream &s, tool::array<handle<style_def>> &list);
    static style_def *parse(style_bag *psb, css_istream &s);
    bool              parse_single(style_bag *psb, css_istream &s, bool allow_pseudo_elements);
    bool              parse_pseudo_class(style_bag *psb, css_istream &s);
    bool              parse_pseudo_element(style_bag *psb, css_istream &s);
    bool              parse_span_pseudo_element(style_bag *psb, css_istream &s, enum_v &selector);
    bool              parse_attr(css_istream &s);
    bool              parse_nth(css_istream &s, int &step, int &index);
    bool              parse_not(style_bag *psb, css_istream &s);
    bool              parse_hover(css_istream &s);
    bool              parse_child_type(css_istream &s, tag::symbol_t &type);
    bool              parse_named(css_istream &s, ustring &named); // :lang(en), :theme(dark)

    bool match_single(const element *el, const element *root_element) const;
    bool match_single(const element *el, ui_state st,
                      const element *root_element) const;
    bool _match_single(const element *el, ui_state st,
                       const element *root_element, bool depends_on) const;
    bool _match_single_negation(const element *el, ui_state st,
                                const element *root_element,
                                bool           depends_on) const;

    bool        depends_single(const element *el, ui_state st,
                               const element *root_element) const;
    INLINE bool match_single(const ustring &class_name, ui_state st) const;

    bool match(const element *el, const element *root_element = 0, bool match_pseudos = false) const;
    bool match(const element *el, ui_state st, const element *root_element = 0) const;
    // checks if style is related to this element/state, used in
    // has_dependent_pseudo_classes_for()
    bool depends_on(const element *el, ui_state st,
                    const element *root_element = 0) const;

    /*
    Sort by specificity of selector: more specific selectors will override more
    general ones. To find the specificity, count the number of ID attributes in
    the selector (a), the number of CLASS attributes in the selector (b), and
    the number of tag names in the selector (c). Concatenating the three numbers
    (in a number system with a large base) gives the specificity.
    */
    void specificity(uint &b, uint &c, uint &d) const {
      for (const style_def *p = this; p; p = p->next) {
        if (p->id.length()) ++b;
        c += (uint)p->atts.length();
        c += p->pseudos.specificity();
        if (p->index != 0 || p->step != 0) ++c;
        if (p->last_index != 0 || p->last_step != 0) ++c;
        if (p->tag && (p->tag != TAG_ANY)) ++d;
      }
    }
    bool operator<(const style_def &rs) const {
      uint b = 0;
      uint c = 0;
      uint d = 0;
      specificity(b, c, d);

      uint rb = 0;
      uint rc = 0;
      uint rd = 0;
      rs.specificity(rb, rc, rd);

      if (b < rb) return true;
      if (b > rb) return false;
      if (c < rc) return true;
      if (c > rc) return false;
      if (d < rd) return true;
      if (d > rd) return false;

      if (inheritance_level < rs.inheritance_level) return true;
      if (inheritance_level > rs.inheritance_level) return false;

      int ds = seqid().cmp(rs.seqid());
      if (ds == 0) return no < rs.no;
      return ds < 0;
    }

    //bool has_pseudo_classes_for(const element *el, ui_state st, bool deep, const element *root_element = 0) const;
    //bool has_dependent_pseudo_classes_for(const element *el, ui_state st, const element *child_el, const element *root_element = 0) const;

    struct callback {
      virtual void has_rule(const style_def * /*r*/) {}
      virtual void has_inline_style(const style_prop_list * /*r*/) {}
      virtual void has_assigned_style(const style_prop_map * /*r*/) {}
    };

    virtual value definition() const override {
      value map = value::make_map();
      map.set_prop("type", value(WCHARS("style-rule")));
      map.set_prop("selector", value(trim(selector_text())));
      map.set_prop("file", value(src_url, value::UT_STRING)/*value::make_url(src_url)*/);
      map.set_prop("lineNo", value(src_line_no));
      map.set_prop("properties", plist->definition_list());
      return map;
    }


  };

  typedef handle<style_def> hstyle_def;

  class style_bag : public tool::resource,
                    public tool::eval::const_resolver,
                    public tool::weakable {

  public:
    friend class view;

    struct const_def : resource {
      string name;
      value  val;
      string seqid; // sequential id of the style where it was defined
    };

  public:
    tool::handle_pool<style>       _resolved_pool;

    //tool::hash_table<style::key,handle<style>> _cached_styles;

    uint                           _counter;
    tool::array<handle<style_def>> _defs;

    uint inheritance_depth = 0;  // or _parent->inheritance_depth + 1

    tool::hash_table<string, tool::handle<style_bag>> _sets;

    style_bag *_parent; // not null for inner style sets

    tool::hash_table<ustring, handle<const_def>> _constants;
    tool::hash_table<ustring, handle<mixin>>     _mixins;
    tool::hash_table<ustring, handle<keyframes>> _keyframes;

    tool::hash_table<uint, bool>                _known_attributes;

  public:
    style_bag(style_bag *p = 0)
        : _counter(0),
          //_cached_styles(64), // 47, 137
          _resolved_pool(128),
          _parent(p) /*, _defs_start_idx(0)*/ {}
    virtual ~style_bag() { clear(); }

    // tool::eval::get_const impl
    virtual bool get_const(wchars name, value &val) {
      val = get_const(string(name));
      return val.is_defined();
    }

    void clear() {
      // FOREACH(i, _defs) delete _defs[i];
      _counter = 0;
      //_pool.clear();
      //_anonymous_pool.clear();
      _resolved_pool.clear();
      //_cached_styles.clear();
      _sets.clear();
      _defs.destroy();
    }

    void clear_cache() {
      //_cached_styles.clear();
      _resolved_pool.clear();
    }

    void derive_from(const style_bag *super_sb) {
      _defs           = super_sb->_defs;
      //_defs_start_idx = _defs.length();
      _sets           = super_sb->_sets;
      _constants      = super_sb->_constants;
      _counter        = super_sb->_counter;
      inheritance_depth = 1 + super_sb->inheritance_depth;
    }

    hstyle intern_resolved(const style &st, bool force_cached = false) {
      // dbg_printf("intern_resolved size %d\n", _resolved_pool.size());
      if (force_cached)
        return _resolved_pool.intern(st);
      else if (st.custom.is_empty() && !st.transitions) {
        return _resolved_pool.intern(st);
      } else { // there are custom attributes or it has transitions, don't put
               // it then into common pool
        hstyle ds = style::create_unique();
        *ds       = st;
        return ds;
      }
    }

#ifdef _DEBUG
    void usage_stats();
#endif

    void add_style(string sn, style_def *sd, style_prop_list *ps) {
      //hstyle hs  = ps;
      sd->no     = get_next_no();
      sd->seqid  = sn;
      sd->plist  = ps;
      sd->inheritance_level = this->inheritance_depth;
      _defs.push(sd);

      bool changes_dimension = ps->changes_dimension();
      style_def *t = sd;
      while (t) {
        for (index_t an = t->atts.last_index(); an >= 0; --an) {
          style_def::attr_def &atd = t->atts[an];
          bool& cd  = _known_attributes[atd.attr_name];
          if (!cd) cd = changes_dimension;
        }
        t = t->next;
      }
    }

    void remove_styles(element *link_or_style);

    void add_named_set(const string &name, style_bag *sb) {
      assert(name.length());

      _sets[name] = sb;
      sb->_parent = this;
    }

    void add_const(const ustring &name, const array<value> &values,
                   const string &seq_id) {
      if (_constants.exists(name)) return;
      handle<const_def> cd = new const_def();
      cd->name             = name;
      cd->val              = value::make_array(values());
      cd->seqid            = seq_id;
      _constants[name]     = cd;
    }

    void add_const(const ustring &name, const value &val, const string &seq_id) {
      if (_constants.exists(name)) return;
      handle<const_def> cd = new const_def();
      cd->name             = name;
      cd->val              = val;
      cd->seqid            = seq_id;
      _constants[name]     = cd;
    }

    tool::hash_table<ustring, handle<const_def>> get_constants() {
      return _constants;
    }
    void set_constants(const tool::hash_table<ustring, handle<const_def>> &cc) {
      _constants = cc;
    }

    value get_const(const string &name) {
      handle<const_def> r;
      _constants.find(name, r);
      if (!r) {
        if (_parent) return _parent->get_const(name);
        return value();
      }
      return r->val;
    }

    void add_mixin(mixin *pmx) {
      if (_mixins.exists(pmx->name)) return;
      _mixins[pmx->name] = pmx;
    }

    mixin *get_mixin(const ustring &name) {
      handle<mixin> r;
      _mixins.find(name, r);
      if (!r) {
        if (_parent) return _parent->get_mixin(name);
        return nullptr;
      }
      return r;
    }

    void add_keyframes(handle<keyframes> pkf) {
      _keyframes[pkf->name] = pkf;
    }

    handle<keyframes> get_keyframes(const string& name) const {
      return _keyframes[name];
    }

    uint get_next_no() { return ++_counter; }

    void reorder();

    slice<handle<style_def>> rules() const { return _defs(); }

#if 0
    bool apply(style &s, const element *el, const element *root_element = 0,
               style_def::callback *pcb = 0 /*used in get_applied_styles*/,
               bool                 apply_important = false);
#endif

    hstyle style_for(const ustring &class_name, ui_state st, const element *root_element = 0);

    //bool has_pseudo_classes_for(const element *el, ui_state st, bool deep,
    //                            const element *root_element) const;
    //bool has_dependent_pseudo_classes_for(const element *el, ui_state el_state,
    //                                      const element *child_el,
    //                                      const element *root_element) const;

    virtual style_bag *get_named_set(const string &name_of_set) const {
      handle<style_bag> sb;
      _sets.find(name_of_set, sb);
      return sb;
    }

    bool is_sensitive_attr(const name_or_symbol &ns, bool &changes_dimension) {
      if (_known_attributes.find(ns.att_sym, changes_dimension)) return true;
      FOREACH(i, _sets) {
        style_bag *psb = _sets(i);
        if (psb->is_sensitive_attr(ns, changes_dimension)) return true;
      }
      return false;
    }
    /*void register_attr(const name_or_symbol& ns, bool changes_dimension)
    {
      bool& cd = _known_attributes[ns.att_sym];
      if(!cd) cd = changes_dimension;
    }*/
    void drop_caches() {
      //for (auto& hs : _cached_styles.elements())
      //  hs->font = nullptr;
      //_cached_styles.clear();

      auto drop_fonts = [](handle_pool<style> &hp) {
        FOREACH(i, hp.elements()) {
          auto ps = hp.elements()[i];
          ps->font = nullptr;
        }
      };
      drop_fonts(_resolved_pool);


    }

    int rules_for(const element *el, const element *root_element, style_list &list);
    int style_set_rules_for(const element *el, const element *root_element, style_list &list, const string &style_set);

    //bool apply_list(style &s, const element *el, const element *root_element,slice<hstyle> list, bool apply_important);
    //bool apply_style_set(style &s, const element *el, const element *root_element, const string &set_name);

    //bool apply_forced_style_set(style &s, const element *el, document *d,
    //                            const style_set_holder *pssh,
    //                            style_def::callback *   pcb = 0);
    int forced_style_set_rules_for(const element *el, document *d, const style_set_holder *pssh, style_list &list);
  };

  class doc_style_bag : public style_bag {

    typedef style_bag super;

  protected:
    friend struct document;
    friend class application;

  public:
    bool is_master;

    doc_style_bag() : is_master(false) {}
    virtual style_bag *get_named_set(const string &name_of_set) const;
  };

  extern bool set_attribute_value(const element_context& pd, style& pcs, cssa::name_or_symbol attr, slice<value> a_values, style_bag* pb = nullptr);
  extern void clear_attribute_value(style& pcs, cssa::name_or_symbol attr);

  // extern pool<string> string_pool; // string_pool is used for interning
  // strings  extern pool<ustring> ustring_pool; // string_pool is used for
  // interning strings  extern pool<ustring, string_ignore_case> font_name_pool;
  // // font_name_pool is used for interning font names

  bool is_none_value(const value &val);
  bool is_auto_value(const value &val);
  bool is_inherit_value(const value &val);

  //bool  direction_value(tristate_v &v, const value &val);
  bool  flow_value(enum_v &v, const value &val);
  //bool  visibility_value(enum_v &v, const value &val);
  bool  font_variant(tristate_v &v, const value &val);
  //bool  display_style(enum_v &v, const value &val);
  bool  background_image_attachment(enum_v &v, const value &val);
  //bool  list_style_position(tristate_v &v, const value &val);
  //bool  text_decoration(enum_v &v, const array<value> &a_values);
  bool  white_space(enum_v &v, const value &val);
  //bool  text_align(halign_v &v, const value &val);
  //bool  vertical_align(valign_v &v, const value &val);
  //bool  horizontal_align(halign_v &v, const value &val);
  bool  font_weight(int_v &v, const value &val);
  bool  color_value(color_v &c, const value &v, const style *cs = nullptr);
  value color_value(const color_v &c);
  //bool  border_style(enum_v &v, const value &val);
  bool  overflow_value(enum_v &v, const value &val);

  bool  length_value(size_v &sz, const value &v, size_v::NUMERIC_CONVERSION number_cvt = size_v::NUMERIC_TO_DIP);
  value length_value(const size_v &sz);
  //bool  list_style_type(int_v &v, const value &val);
  //bool  font_style(tristate_v &v, const value &val);
  //bool  image_repeat(int_v &v, const value &val);
  bool  image_position(size_v &v, const value &val);

  //ustring direction_value_string(const tristate_v &v);
  ustring flow_value_string(const enum_v &v);
  ustring visibility_value_string(const enum_v &v);
  ustring font_variant(const tristate_v &v);
  ustring display_style_sting(const enum_v &v);
  ustring background_image_attachment_string(const enum_v &v);
  //ustring list_style_position_string(const tristate_v &v);
  ustring text_decoration_string(const enum_v &v);
  //ustring text_align_string(const halign_v &v);
  //ustring vertical_align_string(const valign_v &v);
  //ustring horizontal_align_string(const valign_v &v);
  ustring font_weight_string(const int_v &v);
  //ustring color_value_string(const color_v &c);
  //ustring border_style_string(const enum_v &v);
  //ustring overflow_value_string(const enum_v &v);
  //ustring image_repeat_string(const int_v &v);
  ustring length_value_string(const size_v &sz);
  ustring list_style_type_string(const int_v &v);
  ustring font_style(const tristate_v &v);
  //ustring image_repeat_string(const int_v &v);
  ustring border_style_string(const int_v &v);

  ustring floats_string(const enum_v &v);
  ustring clears_string(const enum_v &v);

  bool cursor_value(const element_context& ctx, handle<cursor> &v, slice<value> a_values);

  // extern bool match_list(const ustring& item, const ustring& instr);
  // extern bool match_list_ci(const ustring& item, const ustring& instr);

  extern bool need_animation(const style *from, const style *to);
  extern bool need_animation(const transition_def *transitions,
                             const style *from, const style *to);

#pragma warning(pop)

} // namespace html

namespace tool {
  template <>
  inline unsigned int hash<html::style>(const html::style &the_style) {
    return the_style.hash();
  }

  template <> struct __type_traits<html::style> {
    typedef __true_type has_trivial_default_constructor;
    typedef __true_type has_trivial_copy_constructor;
    typedef __true_type has_trivial_assignment_operator;
    typedef __true_type has_trivial_destructor;
    typedef __true_type is_POD_type;
  };

  /*template<>
    struct pool_traits<html::image_color_transformation>
  {
    static unsigned int hash(const html::image_color_transformation& e) { return
  e.colors[0] + e.colors[1] + e.colors[2] + e.colors[3] + e.colors[4]; } static
  bool equal(const html::image_color_transformation& l, const
  html::image_color_transformation& r) { return l == r; } static
  html::image_color_transformation create(const
  html::image_color_transformation& key) { return key; }
  };*/

} // namespace tool

#endif
