#ifndef __html_style_types_h
#define __html_style_types_h

#include "tool/tool.h"
#include "tool/eval/tl_eval.h"
#include "gool/gool.h"
#include "html-primitives.h"
#include "html-spring.h"
#include "html-style-atts.h"

//#define FLEX_PLUS

namespace html {
  using namespace tool;
  using namespace gool;
  
  class view;
  struct element;
  struct document;

  typedef uint16 char_mark_t;
  #define char_mark_mask 0xFFFF

  enum text_decoration_e {
    text_decoration_none        = 0,
    text_decoration_underline   = 0x1,
    text_decoration_overline    = 0x2,
    text_decoration_linethrough = 0x4,
  };

  /*enum TEXT_DECORATION_STYLE {
    text_decoration_solid  = 0,
    text_decoration_double = 1,
    text_decoration_dotted = 2,
    text_decoration_dashed = 3,
    text_decoration_wavy   = 4,
  };*/

  /*enum WHITE_SPACE {
    white_space_normal,
    white_space_nowrap,
    white_space_pre,
    white_space_prewrap,
  };*/

  enum clear_e {
    clear_none  = 0,
    clear_left  = 1,
    clear_right = 2,
    clear_both  = 3
  };

  enum clip_e {
    clip_border_box = 0,
    clip_padding_box = 1,
    clip_margin_box = 2,
    clip_content_box = 3,
    clip_hit_margin_box = 4,
  };

  enum CURSOR_TYPE {
    cursor_auto      = 0,
    cursor_default   = 0,
    cursor_text      = 1,
    cursor_wait      = 2,
    cursor_crosshair = 3,
    cursor_uparrow   = 4,
    cursor_nw_resize = 5,
    cursor_ne_resize = 6,
    cursor_ew_resize  = 7,
    cursor_ns_resize  = 8,
    cursor_move      = 9,
    cursor_no        = 10,
    cursor_starting  = 11,
    cursor_help      = 12,
    cursor_pointer   = 13,
    cursor_drag_copy = 14,
    cursor_drag_move = 15,

    cursor_w_resize = cursor_ew_resize,
    cursor_e_resize  = cursor_ew_resize,
    cursor_se_resize = cursor_nw_resize,
    cursor_sw_resize = cursor_ne_resize,
    cursor_s_resize  = cursor_ns_resize,
    cursor_n_resize = cursor_ns_resize,

    cursor_none = 16,

  };


  // extern handle_pool<function_value> function_pool;


  enum display_e {
    display_none   = 0, // Object is not rendered.
    display_inline = 1, // Default. Object is rendered as an inline element
                        // sized by the dimensions of the content.
    display_inline_block = 2, // Object is rendered inline, but the contents of
                              // the object are rendered as a element element.
                              // Adjacent inline elements are rendered on the
                              // same line, space permitting.
    display_block = 3,    // Object is rendered as a block element.
    display_contents = 4, // Object provides its content to the container
    display_list_item,
    display_table,
    display_inline_table,
    display_table_body,
    display_table_row,
    display_table_cell,
  };

/*  enum DISPLAY_MODEL {
    display_model_inline_inside,
    display_model_block_inside,
    display_model_table,
    display_model_text_only,
  };*/

  enum overflow_e {
    overflow_none    = -1,
    overflow_visible = 0,
    overflow_hidden,
    overflow_scroll,
    overflow_auto,
    overflow_hidden_scroll,
    overflow_scroll_indicator,
  };

  enum EFFECT_TYPE {
    transition_none,
    transition_blend,
    transition_blend_atop,
    transition_slide_top,
    transition_slide_bottom,
    transition_slide_left,
    transition_slide_right,
    transition_slide_over_top,
    transition_slide_over_bottom,
    transition_slide_over_left,
    transition_slide_over_right,
    transition_remove_top,
    transition_remove_bottom,
    transition_remove_left,
    transition_remove_right,
    transition_scroll_top,
    transition_scroll_bottom,
    transition_scroll_left,
    transition_scroll_right,
    // transition_slide,
    // transition_image,
    transition_window_blend,
    transition_window_slide_ltr,
    transition_window_slide_rtl,
    transition_window_slide_ttb,
    transition_window_slide_btt,
  };

  enum PAGE_BREAK {
    page_break_auto   = -1,
    page_break_always = 0,
    page_break_avoid  = 100,
  };

  enum li_style {
    none,        // No marker is shown.
    disc,        // default. dolid circles.
    circle,      // outlined circles.
    square,      // solid squares.
    decimal,     // 1, 2, 3, 4, and so on.
    lower_roman, // i, ii, iii, iv, and so on.
    upper_roman, // I, II, III, IV, and so on.
    lower_alpha, // a, b, c, d, and so on.
    upper_alpha  // A, B, C, D, and so on.
  };

  /*enum DIRECTION_TYPE {
    direction_ltr = 0,
    direction_rtl = 1,
    direction_ttb = 2,
  };*/

  enum DRAGGABLE_TYPE {
    draggable_none      = 0,
    draggable_only_copy = 1,
    draggable_only_move = 2,
    draggable_copy_move = 3
  };
  enum DROP_OP {
    drop_insert  = 0,
    drop_recycle = 1,
    drop_append  = 2,
    drop_prepend = 3,
    drop_replace = 4
  };

#pragma warning(push)
#pragma warning(disable : 4480) // warning C4480: nonstandard extension used:
                                // specifying underlying type for enum
                                // 'html::MAPPING_TYPE'
  enum MAPPING_TYPE : uint {
    mapping_default       = 0,
    mapping_inherit       = 1,
    mapping_none          = 2,
    mapping_left_to_right = 3,
    mapping_top_to_right  = 4,
  };
#pragma warning(pop)

  inline bool mapping_is_inherit(uint mt) {
    return mt == mapping_default || mt == mapping_inherit;
  }

#pragma pack(push, 4) // IT MUST BE 4 bytes aligned!!!!
  struct mapping_v {
    union {
      struct {
        uint margin : 3;
        uint padding : 3;
        uint border : 3;
        uint background_position : 3; // position
        uint background_image : 3;    // image and gradient
        uint foreground_position : 3; // position
        uint foreground_image : 3;    // image
        uint list_image : 3;          // image
        uint layout : 3;
        uint alignment : 3; // text-align, horizontal-align
      } parts;
      uint all;
    } u;

    mapping_v() { u.all = 0; }

    bool is_undefined() const { return u.all == 0; }
    bool is_defined() const { return u.all != 0; }
    // bool inherit() const { return _v == inherit_value; }
    bool operator==(const mapping_v &rs) const { return u.all == rs.u.all; }
    bool operator!=(const mapping_v &rs) const { return u.all != rs.u.all; }

    uint hash() const { return hash_value(u.all); }

    void inherit(const mapping_v &v) {
      if (v.is_undefined()) return;
      u.all = v.u.all;
    }
    void resolve(const mapping_v &v) {
      if (u.all == 0)
        u.all = v.u.all;
      else {
        if (mapping_is_inherit(u.parts.margin))
          u.parts.margin = v.u.parts.margin;
        if (mapping_is_inherit(u.parts.padding))
          u.parts.padding = v.u.parts.padding;
        if (mapping_is_inherit(u.parts.border))
          u.parts.border = v.u.parts.border;
        if (mapping_is_inherit(u.parts.background_position))
          u.parts.background_position = v.u.parts.background_position;
        if (mapping_is_inherit(u.parts.background_image))
          u.parts.background_image = v.u.parts.background_image;
        if (mapping_is_inherit(u.parts.foreground_position))
          u.parts.foreground_position = v.u.parts.foreground_position;
        if (mapping_is_inherit(u.parts.foreground_image))
          u.parts.foreground_image = v.u.parts.foreground_image;
        if (mapping_is_inherit(u.parts.layout))
          u.parts.layout = v.u.parts.layout;
        if (mapping_is_inherit(u.parts.list_image))
          u.parts.list_image = v.u.parts.list_image;
        if (mapping_is_inherit(u.parts.alignment))
          u.parts.alignment = v.u.parts.alignment;
      }
    }
    void clear() { u.all = 0; }

    value to_value() const {
      if(u.all == 0)
        return value();
      //assert(false); // todo
      return value();
    }

  };

  struct filter_v {
    struct list_t : resource {
      array<value> elements;
    };
    handle<list_t> list;

    filter_v() {}

    uint hash() const {
      return list ? list->elements.hash() : 0;
    }

    bool is_undefined() const { return !list; }
    bool is_defined() const { return !!list; }
    bool is_none() const { return list && list->elements.is_empty(); }
    bool has_items() const { return list && !list->elements.is_empty(); }
    // bool inherit() const { return _v == inherit_value; }
    bool operator==(const filter_v &rs) const {
      return elements() == rs.elements();
    }
    bool operator!=(const filter_v &rs) const {
      return elements() != rs.elements();
    }

    slice<value> elements() const {
      return list ? list->elements() : slice<value>();
    }

    filter_v &operator=(slice<value> elements) {
      if (elements.length) {
        list           = new list_t();
        list->elements = elements;
      }
      return *this;
    }

    void inherit(const filter_v &v) {
      if (v.is_defined()) list = v.list;
    }
    void resolve(const filter_v &v) {}
    void clear() { list.clear(); }
    void set_none() {
      list.clear();
      list = new list_t();
    }
  };

#pragma pack(pop) // IT MUST BE 4 bytes aligned!!!!

  /*enum MAPPING_ITEM {
    mapping_margin    = 0x010,
    mapping_padding   = 0x020,
    mapping_border    = 0x040,
    mapping_background_position = 0x080,
    mapping_foreground_position = 0x100,
    mapping_all       = 0x1f0,
  };*/

  typedef tool::t_value<int, 0, 0xffff, 0xffff> border_v;
  // typedef tool::t_value<int,0,0,0>                    image_align_v;
  //typedef tool::t_value<int, align_auto, 0, -1>             halign_v;
  //typedef tool::t_value<int, valign_baseline, 0, -1>        valign_v;
  typedef tool::t_value<int, 0, -32768, -32768>             space_v;
  typedef tool::t_value<int, 0, 0x7FFFFFFF, 0x7FFFFFFF>     dimension_v;
  typedef tool::t_value<int, 1, 0, 0>                       span_v;

  // typedef tool::t_value<uint,0xFF000000,0xFF000000,0xFFFDFDFD> color_v;

  tool::ustring to_string(const space_v &v);
  // tool::ustring to_string(const image_align_v& v);
  //tool::ustring to_string(const valign_v &v);
  //tool::ustring to_string(const halign_v &v);
  // tool::ustring to_string(const dimension_v& v);
  tool::ustring to_string(const tool::int_v &v);
  tool::ustring to_string(double d);
  tool::ustring to_string(const color_v &v);

  // void from_string(image_align_v& v,        wchars s);
  //void from_string(valign_v &v, wchars s);
  //void from_string(halign_v &v, wchars s);
  void from_string(dimension_v &v, wchars s);
  void from_string(space_v &v, wchars s);
  void from_string(color_v &v, wchars s);
  void from_string(tool::int_v &v, wchars s);
  void from_string(tool::float_v &v, wchars s);

  struct image_rec : public resource {
    tristate_v   requested;
    gool::himage img;
    tool::string url;
  };

  class image_map;
  typedef tool::dictionary<tool::string, handle<image_rec>> image_bag;
  typedef tool::hash_table<ustring, handle<image_map>>      image_map_bag;

#pragma pack(push, 4) // IT MUST BE 4 bytes aligned!!!!
  struct image_ref {
    handle<image_rec> rec;
    handle<image_rec> base_rec;

    uint hash() const { return rec.hash() + base_rec.hash(); }

    inline image_ref() {}
    inline image_ref(const image_ref &i) : rec(i.rec), base_rec(i.base_rec) {}
    inline image_ref(image_rec *prec, image_rec *pbase_rec = nullptr)
        : rec(prec), base_rec(pbase_rec) {}

    inline bool is_defined() const { return rec.is_defined(); }
    inline bool is_undefined() const { return rec.is_undefined(); }
    inline bool is_empty() const { return rec.is_undefined(); }

    inline void clear() {
      rec      = nullptr;
      base_rec = nullptr;
    }

    inline void
    set_none() { // mark it as null - background-image explicitly set to none
      static handle<image_rec>_null      = new image_rec();
      rec      = _null;
      base_rec = nullptr;
    }

    static image_ref none() {
      image_ref t;
      t.set_none();
      return t;
    }

    inline gool::image *img() const {
      return rec.is_defined() ? rec->img.ptr() : nullptr;
    }
    inline void img(gool::image *pimg) {
      assert(rec);
      if (rec) {
        rec->img       = pimg;
        rec->requested = 1;
      }
    }
    inline tool::string url() const {
      return rec.is_defined() ? rec->url : tool::string();
    }
    inline void inherit(const image_ref &src) {
      if (src.is_defined()) {
        rec      = src.rec;
        base_rec = src.base_rec;
      }
    }

    static image_ref create(image_bag &ib, const tool::string &url,
                            gool::image *pimg = 0);

    inline bool operator==(const image_ref &i) const {
      return rec == i.rec && base_rec == i.base_rec;
    }

    bool fetch(view &v, document *pd);

    value to_value() const {
      return rec.is_defined()
        ? value::make_url(rec->url)
        : value();
    }

  };

  struct opacity_v : public tool::t_value<uint, 0xFF, 0xFFFFFFFF, 0xFFFFFFFE>
  {
    typedef tool::t_value<uint, 0xFF, 0xFFFFFFFF, 0xFFFFFFFE> super;
    opacity_v() : super() {}
    opacity_v(const value& val): super() {
      if (val.is_inherit())
        _v = inherit_val();
      else if (val.is_number()) {
        int v = int(255 * val.get_double());
        _v = limit(v, 0, 255);
      }
    }
    ustring to_string() const {
      if (is_defined())
        return ftow(_v / 255.0);
      return ustring();
    }
    value to_value() const {
      if (is_inherit())
        return value::inherit_val();
      if(is_defined())
        return value(val(0) / 255.0);
      return value();
    }
  };

  // struct image_arrived_cb: public virtual  resource
  //{
  //  virtual void on_image_arrived(document* pd, image* pimg) = 0;
  //};

#ifdef FLEX_PLUS
#define PREF_UNIT_INIT , pref_unit(0)
#else
#define PREF_UNIT_INIT
#endif

#pragma pack(pop)

  inline ustring to_string(const image_ref &imr) {
    if (imr.is_defined())
      return ustring::format(W("url(%S)"), imr.url().c_str());
    return ustring();
  }

  struct size_v;

  enum SIZE_FROM_STRING {
    NUMBER_NUMBER,
    NUMBER_PX,
    NUMBER_DIP,
    NUMBER_PERCENT, // 0..1 mapped to 0%..100%
  };

  void          from_string(size_v &v, wchars s, SIZE_FROM_STRING num_flavour = NUMBER_PX);
  tool::ustring to_string(const size_v &v);

  // parse canonical size_v
  size_v parse_size_v(wchars &ss); // after - non-parsed string in ss

  struct named_length_provider {
    virtual bool get_named_length(uint sym, size_v &sz) const = 0;
  };

#pragma pack(push, 4)

  struct size_v {
    uint unit;
    uint min_unit; // note: unit must 32 bit wide because it may store name_id:
    uint max_unit; //    unit |= name_id << 8;

    union _d {
      int            value;
      eval::conduit *expr;
      uint_ptr       cap;
      _d() { cap = 0; }
    };
    _d d;
    _d min_d;
    _d max_d;

#ifdef FLEX_PLUS
    uint pref_unit;
    _d   pref_d;
#endif


/*    enum unit_type {
      //-------- rel
      em = 1, // height of the element's font.
      ex = 2, // height of letter 'x'
      pr = 3, //%
      sp = 4, //%% "springs"
      rs = 5, // value is -1,1 // smaller larger
              //-------- abs
      as = 6, // value is 1,2,3,4,5,6,7
      px = 7, // pixels, native device pixels (on printer surface they are
              // dip's, see below)
      in = 8,   // Inches (1 inch = 2.54 centimeters).
      cm = 9,   // Centimeters.
      mm = 10,  // Millimeters.
      pt = 11,  // Points (1 point = 1/72 inches).
      pc  = 12, // Picas (1 pica = 12 points).
      dip = 13, // device independent pixels, 1/96 of inch. Thus on 96 DPI
                // screen these correspond to a single physical pixel
      nm             = 14, // Number
      expr           = 15, // calc expression
      pr_width       = 16, // width(n%)
      pr_height      = 17, // height(n%)
      pr_view_width  = 18, // vw
      pr_view_height = 19, // vh
      pr_view_min    = 20, // vmin
      pr_view_max    = 21, // vmax
      max_direct_unit =
          0xFF // all unit - max_direct_unit == name id (attr::symbol_t)
    }; */

    using unit_type = tool::value::unit_type_length;

    enum { max_direct_unit = 0xFF };

    /*enum length_special_values {
      $none    = tool::value::$none,
      $smaller = tool::value::$smaller,
      $larger  = tool::value::$larger,

      $xx_small = tool::value::$xx_small,
      $x_small  = tool::value::$x_small,
      $small    = tool::value::$small,
      $medium   = tool::value::$medium,
      $large    = tool::value::$large,
      $x_large  = tool::value::$x_large,
      $xx_large = tool::value::$xx_large,

      $thin  = tool::value::$thin,
      $thick = tool::value::$thick,

      $auto = tool::value::$auto,

      $inherit     = tool::value::$inherit,
      $min_content = tool::value::$min_content,
      $max_content = tool::value::$max_content,

      $ui_scale = tool::value::$ui_scale,
      $cover = tool::value::$cover,
      $contain = tool::value::$contain,

      $system_metrics_first = tool::value::$system_metrics_first,

        $scrollbar_height  = tool::value::$scrollbar_height,
        $scrollbar_width   = tool::value::$scrollbar_width,
        $small_icon_height = tool::value::$small_icon_height,
        $small_icon_width  = tool::value::$small_icon_width,
        $border_width      = tool::value::$border_width,
        $border_3d_width   = tool::value::$border_3d_width,

        $window_caption_height = tool::value::$window_caption_height,
        $window_button_height = tool::value::$window_button_height,
        $window_button_width = tool::value::$window_button_width,
        $window_frame_width = tool::value::$window_frame_width,

      $system_metrics_last = tool::value::$system_metrics_last,
    }; */

    using special_values = tool::value::length_special_values;

    size_v() : unit(0),min_unit(0),max_unit(0) PREF_UNIT_INIT {}
    ~size_v() { clear(); }

    size_v(space_v v) : unit(0), min_unit(0), max_unit(0) PREF_UNIT_INIT { if (v.is_defined()) set_px(v);  }
    size_v(int v, unit_type u) : unit(0), min_unit(0), max_unit(0) PREF_UNIT_INIT { set(v, u); }
    size_v(double v, unit_type u) : unit(0), min_unit(0), max_unit(0) PREF_UNIT_INIT { set(v, u); }
    size_v(int px) : unit(0), min_unit(0), max_unit(0) PREF_UNIT_INIT { set_ppx(px); }
    size_v(const tool::value &v) : unit(0), min_unit(0), max_unit(0) PREF_UNIT_INIT { set(v); }
    size_v(const size_v &v) : unit(0), min_unit(0), max_unit(0) PREF_UNIT_INIT {
      _set(v.unit, v.d);
#ifdef FLEX_PLUS
      if (v.pref_unit) _set_pref(v.pref_unit, v.pref_d);
#endif
      if(v.min_unit) set_min(v.get_min());
      if(v.max_unit) set_max(v.get_max());
    }

    static bool from_value(const tool::value &v, size_v &sz);

    static size_v make_literal(special_values lit) {
      size_v t;
      t.d.value = lit;
      t.unit    = unit_type::as;
      return t;
    }
    static size_v make_auto() { return make_literal(special_values::$auto); }

    static size_v make_percent(float v) {
      size_v t;
      t.d.value = m1000(v);
      t.unit    = unit_type::pr;
      return t;
    }

    inline size_v &operator=(const size_v &v) {
      if(this == &v)
        return *this;
      clear();
      _set(v.unit, v.d);
#ifdef FLEX_PLUS
      if (v.pref_unit) _set_pref(v.pref_unit, v.pref_d);
#endif
      set_min(v.get_min());
      set_max(v.get_max());
      return *this;
    }
    inline bool operator==(const size_v &rs) const {
      if (is_zero() && rs.is_zero())
        return true;
      return d.value == rs.d.value && unit == rs.unit
          && min_d.value == rs.min_d.value && min_unit == rs.min_unit
          && max_d.value == rs.max_d.value && max_unit == rs.max_unit
#ifdef FLEX_PLUS
             && pref_d.value == rs.pref_d.value && pref_unit == rs.pref_unit
#endif
          ;
    }
    inline bool operator!=(const size_v &rs) const {
      if (is_zero() && rs.is_zero())
        return false;
      return d.value != rs.d.value || unit != rs.unit
        || min_d.value != rs.min_d.value || min_unit != rs.min_unit
        || max_d.value != rs.max_d.value || max_unit != rs.max_unit
#ifdef FLEX_PLUS
             || pref_d.value != rs.pref_d.value || pref_unit != rs.pref_unit
#endif
          ;
    }

    void _set(uint u1, _d d1) {
      unit = u1;
      if ((unit & max_direct_unit) == unit_type::expr) {
        d.expr = d1.expr;
        d.expr->add_ref();
      } else
        d.value = d1.value;
    }
#ifdef FLEX_PLUS
    void _set_pref(uint u1, _d d1) {
      pref_unit = u1;
      if (pref_unit == expr) {
        pref_d.expr = d1.expr;
        pref_d.expr->add_ref();
      } else
        pref_d.value = d1.value;
    }
    inline size_v preferred() const {
      size_v pv;
      pv._set(pref_unit, pref_d);
      return pv;
    }
    inline void preferred(const size_v &pr) {
      clear_pref();
      _set_pref(pr.unit, pr.d);
    }
#else
    inline size_v preferred() const { return size_v(); }
      // inline void   preferred(const size_v& pr) { /*clear_pref(); _set_pref(
      // pr.unit,pr.d );*/ }
#endif

    inline size_v get_min() const {
      size_v pv;
      pv._set(min_unit, min_d);
      return pv;
    }

    inline void clear_min() {
      if (min_unit == unit_type::expr && min_d.expr)
        min_d.expr->release();
      min_unit = 0;
      min_d.cap = 0;
    }

    inline void set_min(const size_v& minv) {
      clear_min();
      min_unit = minv.unit;
      if (min_unit == unit_type::expr) {
        min_d.expr = minv.d.expr;
        min_d.expr->add_ref();
      }
      else
        min_d.value = minv.d.value;
    }

    inline size_v get_max() const {
      size_v pv;
      pv._set(max_unit, max_d);
      return pv;
    }

    inline void clear_max() {
      if (max_unit == unit_type::expr && max_d.expr)
        max_d.expr->release();
      max_unit = 0;
      max_d.cap = 0;
    }

    inline void set_max(const size_v& maxv) {
      clear_max();
      max_unit = maxv.unit;
      if (max_unit == unit_type::expr) {
        max_d.expr = maxv.d.expr;
        max_d.expr->add_ref();
      }
      else
        max_d.value = maxv.d.value;
    }

    inline bool relative() const { return unit > 0 && unit < unit_type::as; }
    // bool  relative_to_font_size() const { return unit == em || unit == ex; }
    inline bool absolute() const { return unit >= unit_type::as; }

    inline float flex() const { return unit == unit_type::sp ? (max(0, d.value) / 1000.0f) : 0; }
    inline int   flex1000() const { return unit == unit_type::sp ? max(0,d.value) : 0; }
    inline int   percent() const { return is_any_percent() ? (max(0, d.value) / 1000) : 0; }
    inline float percent1() const { return is_any_percent() ? (max(0, d.value) / 100000.f) : 0; }

    inline bool is_px() const { return unit == unit_type::px; }
    inline bool is_ppx() const { return unit == unit_type::ppx; }
    inline bool is_spring() const { return unit == unit_type::sp; }
    inline bool is_percent() const { return unit == unit_type::pr; }
    inline bool is_any_percent() const { return unit == unit_type::pr || unit == unit_type::pr_width || unit == unit_type::pr_height || is_percent_of_view() || unit == unit_type::expr; }
    inline bool is_percent_of_width() const { return unit == unit_type::pr_width; }
    inline bool is_percent_of_height() const { return unit == unit_type::pr_height; }
    inline bool is_percent_of_view() const { return unit == unit_type::pr_view_width || unit == unit_type::pr_view_height || unit == unit_type::pr_view_min || unit == unit_type::pr_view_max; }
    inline bool is_percent_of_view_width() const { return unit == unit_type::pr_view_width; }
    inline bool is_percent_of_view_height() const { return unit == unit_type::pr_view_height; }
    //inline bool is_named() const { return unit >= max_direct_unit; }

    inline bool is_points() const { return unit == unit_type::pt; }

    inline bool is_fixed() const {
      return unit != 0 && unit != unit_type::sp && !is_any_percent();
    }
    inline bool is_literal(int v) const { return unit == unit_type::as && d.value == v; }

    inline bool is_auto() const { return unit == unit_type::as && d.value == special_values::$auto; }
    inline bool is_min_intrinsic() const { return unit == unit_type::as && d.value == special_values::$min_content; }
    inline bool is_max_intrinsic() const { return unit == unit_type::as && d.value == special_values::$max_content; }
    inline bool is_none() const { return d.value == 0 && unit == unit_type::as; }
    inline bool is_ui_scale() const { return unit == unit_type::as && d.value == special_values::$ui_scale; }

    inline bool is_zero() const { return d.value == 0 || is_undefined(); }
    inline bool is_expr() const { return unit == unit_type::expr && d.expr; }

    inline bool is_defined() const { return unit > 0; }
    inline bool is_undefined() const { return unit == 0; }
    inline bool is_inherit() const { return d.value == special_values::$inherit && unit == unit_type::as; }

    inline bool is_not_auto() const { return is_defined() && !is_auto(); }

    inline bool undefined_or_auto() const { return (unit == 0) || is_auto(); }

    //inline uint get_name() const { return unit >> 8; }
    //inline uint get_base_unit() const { return unit & max_direct_unit; }

    inline void inherit(const size_v &src) {
      if (src.is_defined()) *this = src;
    }
    void derive(const size_v &src);

    static size_v inherit_val() {
      size_v t;
      t.unit    = unit_type::as;
      t.d.value = special_values::$inherit;
      return t;
    }

    // void inherit_resolve(view& v, const size_v& base);
    void resolve(view &v, const size_v &parent); // to points
    void resolve_pixels(resolution_provider &v, const size_v &base); // to pixels

    // int  pixels(const size_v& font_size, int width) const;
    int pixels(const size_v &font_size, size sz, bool vertical = false,
               resolution_provider *pv = nullptr) const;

    void clear() {
      if ((unit & max_direct_unit) == unit_type::expr) d.expr->release();
      unit  = 0;
      d.cap = 0;
#ifdef FLEX_PLUS
      clear_pref();
#endif
      clear_min();
      clear_max();
    }

#ifdef FLEX_PLUS
    void clear_pref() {
      if (pref_unit == expr) pref_d.expr->release();
      pref_unit  = 0;
      pref_d.cap = 0;
    }
#endif
    operator tool::value() const { return to_value(); }
    tool::value to_value() const;
    tool::ustring  to_string() const;

    inline void set(double v, unit_type u) {
      clear();
      if (u == unit_type::rs || u == unit_type::as) // || u == px
        d.value = int(v);
      else
        d.value = m1000(v);
      unit = u;
    }

    inline void set_raw(int v, unit_type u) {
      clear();
      d.value = v;
      unit    = u;
    }

    inline void set_literal(int v) {
      clear();
      // assert(v >= 1 && v <= 7);
      d.value = v;
      unit    = unit_type::as;
    }
    inline void set_auto() { set_literal(special_values::$auto); }

    inline void set_relative(int v) {
      clear();
      // assert(v >= 1 && v <= 7);
      d.value = v;
      unit    = unit_type::rs;
    }

    /*inline void set_named(uint name_id, size_v fallback) {
      (*this) = fallback;
      unit |= name_id << 8;
    }

    static size_v make_named(uint name_id, size_v fallback) {
      size_v t;
      t.set_named(name_id, fallback);
      return t;
    }*/

    /*void  set_pixels(int v)
    {
      d.value = v;
      unit = px;
    }*/

    inline bool set_literal(wchars css) {
      if (css == WCHARS("*"))
        set_flex(1.0f);
      else if (css == WCHARS("xx-small"))
        set_literal(size_v::special_values::$xx_small);
      else if (css == WCHARS("x-small"))
        set_literal(size_v::special_values::$x_small);
      else if (css == WCHARS("small"))
        set_literal(size_v::special_values::$small);
      else if (css == WCHARS("medium"))
        set_literal(size_v::special_values::$medium);
      else if (css == WCHARS("large"))
        set_literal(size_v::special_values::$large);
      else if (css == WCHARS("x-large"))
        set_literal(size_v::special_values::$x_large);
      else if (css == WCHARS("xx-large"))
        set_literal(size_v::special_values::$xx_large);
      else if (css == WCHARS("smaller"))
        set(false);
      else if (css == WCHARS("larger"))
        set(true);
      else if (css == WCHARS("thin"))
        set_literal(size_v::special_values::$thin);
      else if (css == WCHARS("thick"))
        set_literal(size_v::special_values::$thick);
      //else if (css == WCHARS("auto"))
      //  set_literal(size_v::special_values::$auto);
      //else if (css == WCHARS("none"))
      //  set_literal(size_v::special_values::$none);
      //else if (css == WCHARS("inherit"))
      //  set_literal(size_v::special_values::$inherit);
      else if (css == WCHARS("min-content") || css == WCHARS("min-intrinsic"))
        set_literal(size_v::special_values::$min_content);
      else if (css == WCHARS("max-content") || css == WCHARS("max-intrinsic"))
        set_literal(size_v::special_values::$max_content);

      else if (css == WCHARS("system-scrollbar-height")) set_literal(size_v::special_values::$scrollbar_height);
      else if (css == WCHARS("system-scrollbar-width")) set_literal(size_v::special_values::$scrollbar_width);
      else if (css == WCHARS("system-border-width")) set_literal(size_v::special_values::$border_width);
      else if (css == WCHARS("system-3d-border-width")) set_literal(size_v::special_values::$border_3d_width);
      else if (css == WCHARS("system-small-icon-height")) set_literal(size_v::special_values::$small_icon_height);
      else if (css == WCHARS("system-small-icon-width")) set_literal(size_v::special_values::$small_icon_width);

      else if (css == WCHARS("window-scrollbar-height")) set_literal(size_v::special_values::$scrollbar_height);
      else if (css == WCHARS("window-scrollbar-width")) set_literal(size_v::special_values::$scrollbar_width);
      else if (css == WCHARS("window-border-width")) set_literal(size_v::special_values::$border_width);
      else if (css == WCHARS("window-icon-height")) set_literal(size_v::special_values::$small_icon_height);
      else if (css == WCHARS("window-icon-width")) set_literal(size_v::special_values::$small_icon_width);

      else if (css == WCHARS("window-caption-height")) set_literal(size_v::special_values::$window_caption_height);
      else if (css == WCHARS("window-button-height")) set_literal(size_v::special_values::$window_button_height);
      else if (css == WCHARS("window-button-width")) set_literal(size_v::special_values::$window_button_width);
      else if (css == WCHARS("window-frame-width")) set_literal(size_v::special_values::$window_frame_width);

      else if (css == CHARS("ui-scale"))
        set_literal(size_v::special_values::$ui_scale);
      else
        return false;
      return true;
    }

    inline void set_percents(float v) {
      clear();
      d.value = m1000(v);
      unit    = unit_type::pr;
    }

    inline void set_percents_of_width(float v) {
      clear();
      d.value = m1000(v);
      unit    = unit_type::pr_width;
    }
    inline void set_percents_of_height(float v) {
      clear();
      d.value = m1000(v);
      unit    = unit_type::pr_height;
    }

    inline void set_percents_of_view_width(float v) {
      clear();
      d.value = m1000(v);
      unit    = unit_type::pr_view_width;
    }
    inline void set_percents_of_view_height(float v) {
      clear();
      d.value = m1000(v);
      unit    = unit_type::pr_view_height;
    }

    inline void set_percents_of_view_min(float v) {
      clear();
      d.value = m1000(v);
      unit    = unit_type::pr_view_min;
    }
    inline void set_percents_of_view_max(float v) {
      clear();
      d.value = m1000(v);
      unit    = unit_type::pr_view_max;
    }

    inline void set_flex(double v) {
      clear();
      d.value = m1000(v);
      unit    = unit_type::sp;
    }

    inline void set_number(int v) {
      clear();
      d.value = m1000(v);
      unit    = unit_type::nm;
    }

    inline void set_number(double v) {
      clear();
      d.value = m1000(v);
      unit = unit_type::nm;
    }

    inline void set_calc_expr(eval::conduit *cexpr) {
      clear();
      d.expr = cexpr;
      d.expr->add_ref();
      unit = unit_type::expr;
    }

    inline void set_ppx(int v) {
      clear();
      d.value = m1000(v);
      unit = unit_type::ppx;
    }

    inline void set_px(int v) {
      clear();
      d.value = m1000(v);
      unit    = unit_type::px;
    }
    inline void set_px(float v) {
      clear();
      d.value = m1000(v);
      unit    = unit_type::px;
    }
    inline void set_px(double v) {
      clear();
      d.value = m1000(v);
      unit    = unit_type::px;
    }

    inline void set_dips(int v) {
      clear();
      d.value = m1000(v);
      unit    = unit_type::dip;
    }

    inline int dpoints() const {
      assert(unit == unit_type::pt);
      return d.value / 100;
    }

    inline int kpoints() const {
      assert(unit == unit_type::pt);
      return d.value;
    }

    inline int number(int defv = 0) const {
      return unit == unit_type::nm ? d.value : defv;
    }

    inline double number(double defv) const {
      return unit == unit_type::nm ? d1000(d.value) : defv;
    }

    inline float number(float defv) const {
      return unit == unit_type::nm ? f1000(d.value) : defv;
    }


    inline eval::conduit *cexpr() const {
      if (is_expr()) return d.expr;
      return 0;
    }

    int pixels(int defv = 0) const;

    /*int pixels_width(view &v, element *b) const;
    int pixels_height(view &v, element *b) const;

    int pixels_width(view &v, element *b, int_v base) const;
    int pixels_height(view &v, element *b, int_v base) const;

    float pixels_width_f(view &v, element *b, float base) const;
    float pixels_height_f(view &v, element *b, float base) const;

    float pixels_width_f(view &v, element *b, int base) const { return pixels_width_f(v, b, float(base)); }
    float pixels_height_f(view &v, element *b, int base) const { return pixels_height_f(v, b, float(base)); }

    float pixels_width_f(view &v, element *b) const;
    float pixels_height_f(view &v, element *b) const;*/

    INLINE int preferred_pixels_width(view &v, element *b, int base) const {
#ifdef FLEX_PLUS
      if (pref_unit) return preferred().pixels_width(v, b, base);
#endif
      return 0;
    }
    INLINE int preferred_pixels_height(view &v, element *b, int base) const {
#ifdef FLEX_PLUS
      if (pref_unit) return preferred().pixels_height(v, b, base);
#endif
      return 0;
    }

    void pixels_n_spring_w(view &v, element *b, int base_size, int &pix, int &spr) const;
    void pixels_n_spring_h(view &v, element *b, int base_size, int &pix, int &spr) const;

    void pixels_n_spring_w(view &v, element *b, int base_size, flex_value &fv) const {
      pixels_n_spring_w(v, b, base_size, fv.px, fv.flex);
    }
    void pixels_n_spring_h(view &v, element *b, int base_size, flex_value &fv) const {
      pixels_n_spring_h(v, b, base_size, fv.px, fv.flex);
    }

    enum NUMERIC_CONVERSION {
      NUMERIC_TO_NUMERIC,
      NUMERIC_TO_PIXEL,
      NUMERIC_TO_DIP,
      NUMERIC_TO_PERCENT,
    };

    bool set(const value& v, NUMERIC_CONVERSION number_cvt = NUMERIC_TO_NUMERIC);

    void set(bool bigger) {
      clear();
      d.value = bigger ? +1 : -1;
      unit    = unit_type::rs;
    }

    inline size_v &operator=(int v) {
      clear();
      d.value = m1000(v);
      unit    = unit_type::px;
      return *this;
    }

    inline size_v &operator=(const dimension_v &v) {
      clear();
      if (v.is_defined()) {
        if (v._v < 0) {
          d.value = m1000(-v._v);
          unit    = unit_type::pr;
        } else {
          d.value = m1000(v._v);
          unit    = unit_type::px;
        }
      }
      return *this;
    }

    bool morph(view& v, element* b, size_v start, size_v end, real n);


#ifdef FLEX_PLUS
    inline uint hash() const {
      return d.value ^ int(unit) ^ pref_d.value ^ int(pref_unit);
    }
#else
    inline uint hash() const {
      uint v = 61;
      tool::hash_combine(v, hash_value(d.value));
      tool::hash_combine(v, hash_value(int(unit)));
      if (min_unit) {
        tool::hash_combine(v, hash_value(min_d.value));
        tool::hash_combine(v, hash_value(int(min_unit)));
      }
      if (max_unit) {
        tool::hash_combine(v, hash_value(max_d.value));
        tool::hash_combine(v, hash_value(int(max_unit)));
      }
      return v;
    }
#endif
  };
#pragma pack(pop)

  //size_v unname(size_v v, const element* cs, const style* ps = nullptr);
  //color_v unname(color_v v, const element* cs, const style* ps = nullptr);
    
  extern color_v morph_color(view& v, element* b, color_v c1, color_v c2, real n);

  typedef tool::t_value<uint, 0, uint(-1), uint(-1)> id_v;

  struct pixels 
  {
    view&    v;
    element* b;
    size_v   s;
    size     dim;
    pixels(view &v_, element *b_, const size_v& sz_, size d = size()) : v(v_), b(b_), s(sz_), dim(d) {}

    int width() { return int(roundf(width_f())); }
    int height() { return int(roundf(height_f())); }

    virtual float width_f();
    virtual float height_f();

    virtual float resolve_percents_width(float percent);
    virtual float resolve_percents_height(float percent);
  };

  struct dips : public pixels {

    dips(view &v_, element *b_, const size_v& sz_, size d = size()) : pixels(v_, b_, sz_, d) {}

    float width_f() override;
    float height_f() override;
  };

  // variant where 100% is border rect dimensions 
  struct image_position_pixels : public pixels 
  {
    size im_dim;
    image_position_pixels(view &v_, element *b_, const size_v& sz_, size d, size id) : pixels(v_, b_, sz_, d), im_dim(id) {}

    virtual float resolve_percents_width(float percent) override;
    virtual float resolve_percents_height(float percent) override;

  };

    

  class gradient : public resource
  {
  public:
    struct color_stop {
      color_v      color;
      float_v      position;
      unsigned int hash() const {
        return color.hash() ^ tool::hash((float)position);
      }
      bool operator==(const color_stop &rs) const {
        return color == rs.color && position == rs.position;
      }
      bool operator!=(const color_stop &rs) const {
        return color != rs.color || position != rs.position;
      }
    };

    enum TYPE {
      NO_GRADIENT,
      LINEAR,
      RADIAL,
    };

    virtual ~gradient() {}

    virtual TYPE type() const = 0; // { return NO_GRADIENT; }

    slice<color_stop> stops() const { return _stops(); }
    bool add_stop(float_v position, color_v color) {
      color_stop cs;
      cs.position = position;
      cs.color    = color;
      _stops.push(cs);
      return true;
    };
    bool is_none() const { return _stops.size() == 0; }
    bool normalize_stops() {
      if (is_none()) return false;
      float   miv   = 0.f;
      float   mav   = 1.f;
      index_t first = 0;
      index_t last  = _stops.size() - 1;
      for (index_t n = 0; n < _stops.size();) {
        if (_stops[n].position.is_defined()) {
          _stops[n].position = max(miv, _stops[n].position);
          miv                = _stops[n].position;
          first              = n++;
          continue;
        }
        mav  = 1.f;
        last = n;
        for (++n; n < _stops.size(); ++n)
          if (_stops[n].position.is_defined()) {
            mav  = max(miv, _stops[n].position);
            last = n;
            break;
          } else {
            mav  = 1.f;
            last = n;
          }
        index_t i              = 0;
        _stops[first].position = miv;
        for (index_t k = first + 1; k < last; ++k)
          _stops[k].position = miv + ((mav - miv) * ++i) / (last - first);
        _stops[last].position = mav;
      }
      return true;
    }
    virtual unsigned hash() const {
      unsigned h = type();
      hash_combine(h, uint(_stops.length()));
      for (index_t n = 0; n < _stops.size(); ++n)
        hash_combine(h, _stops[n].hash());
      return h;
    }

    virtual bool is_compatible(const gradient *other) const {
      return type() == other->type() &&
             _stops.length() == other->_stops.length();
    }

    static bool is_compatible(const gradient *a, const gradient *b) {
      if (!b || !a) return false;
      return a->type() == b->type() &&
        a->_stops.length() == b->_stops.length();
    }

    virtual bool is_transparent() const {
      if (_stops.length() == 0) return true;
      return _stops()
                 .find_if([](const color_stop &cs) {
                   return !cs.color.to_argb().is_opaque();
                 })
                 .length != 0;
    }

    virtual bool operator==(const gradient &r) const {
      if (r.type() != NO_GRADIENT) return false;
      return stops() == r.stops();
    }
    virtual bool operator!=(const gradient &r) const { return !operator==(r); }
    virtual gradient *clone() const = 0;

    static gradient *none_gradient();

    virtual value to_value() const = 0;

  protected:
    array<color_stop> _stops;
    gradient() {}
    bool morph(view& v, element *b, const gradient *start, const gradient *end, real n);
  };

  typedef gool::geom::point_t<size_v> point_v;

  class null_gradient : public gradient {
  public:
    DEFINE_TYPE_ID_DERIVED(null_gradient,gradient);

    virtual TYPE      type() const { return NO_GRADIENT; }
    virtual gradient *clone() const { return new null_gradient(); }

    value to_value() const override { return value(); }
  };

  inline gradient *gradient::none_gradient() {
    static handle<gradient> ng = new null_gradient();
    return ng;
  }

  class linear_gradient : public gradient {
  public:
    point_v pos;
    point_v dim;
    float_v angle;

    linear_gradient() {}
    virtual ~linear_gradient() {}
    virtual TYPE     type() const { return LINEAR; }
    virtual unsigned hash() const {
      unsigned h = gradient::hash();
      hash_combine(h, pos.x.hash());
      hash_combine(h, pos.y.hash());
      hash_combine(h, dim.x.hash());
      hash_combine(h, dim.y.hash());
      hash_combine(h, angle.hash());
      return h;
    }
    virtual bool operator==(const gradient &r) const {
      if (r.type() != LINEAR) return false;
      const linear_gradient *pb = static_cast<const linear_gradient *>(&r);
      return pos == pb->pos && dim == pb->dim && angle == pb->angle &&
             stops() == pb->stops();
    }
    DEFINE_TYPE_ID_DERIVED(linear_gradient,gradient)

    bool morph(view& v, element* b, const linear_gradient *start, const linear_gradient *end,
               real n);

    virtual bool is_compatible(const gradient *other) const {
      if (!gradient::is_compatible(other)) return false;
      const linear_gradient *po = static_cast<const linear_gradient *>(other);

      return pos.x.is_defined() == po->pos.x.is_defined() &&
             pos.y.is_defined() == po->pos.y.is_defined() &&
             dim.x.is_defined() == po->dim.x.is_defined() &&
             dim.y.is_defined() == po->dim.y.is_defined() &&
             angle.is_defined() == po->angle.is_defined();
    }
    virtual gradient *clone() const {
      linear_gradient *p = new linear_gradient();
      p->pos             = pos;
      p->dim             = dim;
      p->angle           = angle;
      p->_stops          = stops();
      return p;
    }

    value to_value() const override {
      function_value* pf = new function_value();
      pf->name = WCHARS("linear-gradient");
      return value::make_function(pf);
    }
  };

  class radial_gradient : public gradient {
  public:
    enum SIZE {
      CLOSEST_SIDE, // contain
      CLOSEST_CORNER,
      FARTHEST_SIDE,
      FARTHEST_CORNER, // cover
    };

    enum SHAPE {
      EXPLICIT,
      CIRCLE,
      ELLIPSE,
    };
    SHAPE   shape;
    SIZE    size;
    point_v center;
    point_v radius;
    point_v focus;

    radial_gradient(SHAPE sh = ELLIPSE, SIZE sz = FARTHEST_CORNER)
        : shape(sh), size(sz) {}

    virtual ~radial_gradient() {}
    virtual TYPE     type() const override { return RADIAL; }
    virtual unsigned hash() const override {
      unsigned h = gradient::hash();
      hash_combine(h, radius.x.hash());
      hash_combine(h, radius.y.hash());
      hash_combine(h, focus.x.hash());
      hash_combine(h, focus.y.hash());
      hash_combine(h, center.x.hash());
      hash_combine(h, center.y.hash());
      hash_combine(h, uint(shape));
      hash_combine(h, uint(size));
      return h;
    }
    virtual bool operator==(const gradient &rb) const override {
      if (rb.type() != RADIAL) return false;
      const radial_gradient *pb = static_cast<const radial_gradient *>(&rb);
      return radius == pb->radius && focus == pb->focus &&
             center == pb->center && shape == pb->shape && size == pb->size &&
             stops() == pb->stops();
    }

    DEFINE_TYPE_ID_DERIVED(radial_gradient,gradient)

    bool morph(view& v, element *b, const radial_gradient *start, const radial_gradient *end,
               real n);

    virtual bool is_compatible(const gradient *other) const override {
      if (!gradient::is_compatible(other)) return false;
      const radial_gradient *po = static_cast<const radial_gradient *>(other);

      return shape == po->shape && size == po->size &&
             center.x.is_defined() == po->center.x.is_defined() &&
             center.y.is_defined() == po->center.y.is_defined() &&
             radius.x.is_defined() == po->radius.x.is_defined() &&
             radius.y.is_defined() == po->radius.y.is_defined() &&
             focus.x.is_defined() == po->focus.x.is_defined() &&
             focus.y.is_defined() == po->focus.y.is_defined();
    }

    virtual gradient *clone() const override {
      radial_gradient *p = new radial_gradient();
      p->radius          = radius;
      p->focus           = focus;
      p->center          = center;
      p->shape           = shape;
      p->size            = size;
      p->_stops          = stops();
      return p;
    }

    value to_value() const override {
      function_value* pf = new function_value();
      pf->name = WCHARS("radial-gradient");
      return value::make_function(pf);
    }


  };

  struct shadow_def : public resource {
    tristate_v         inset;
    size_v             offset_x;
    size_v             offset_y;
    size_v             radius;
    size_v             spread;
    color_v            color;
    value_handle<shadow_def> next;

    shadow_def() {}

    DEFINE_TYPE_ID(shadow_def)

    bool operator==(const shadow_def &rs) const { 
      return inset == rs.inset
        && offset_x == rs.offset_x
        && offset_y == rs.offset_y
        && radius == rs.radius
        && spread == rs.spread
        && color == rs.color
        && next == rs.next;
    }
    bool operator!=(const shadow_def &rs) const { return !operator==(rs); }

    inline uint hash() const {
      uint v = 61;
      tool::hash_combine(v, tool::hash(inset));
      tool::hash_combine(v, tool::hash(offset_x));
      tool::hash_combine(v, tool::hash(offset_y));
      tool::hash_combine(v, tool::hash(radius));
      tool::hash_combine(v, tool::hash(spread));
      tool::hash_combine(v, tool::hash(color));
      tool::hash_combine(v, tool::hash(next));
      return v;
    }
    
    static const shadow_def *null_val() {
      static handle<shadow_def> _val = new shadow_def();
      return _val;
    }
    static const shadow_def *inherit_val() {
      static handle<shadow_def> _val = new shadow_def();
      return _val;
    }

    value to_value() const
    {
      array<value> list;
      for (const shadow_def* p = this; p; p = p->next)
      {
        array<value> r;
        if (p->inset)
          r.push(value(CHARS("inset")));
        r.push(p->offset_x.to_value());
        r.push(p->offset_y.to_value());
        r.push(p->radius.to_value());
        if( p->spread.is_defined() )
          r.push(p->spread.to_value());
        list.push(value::make_array(r()));
      }
      if (list.size() > 1)
        return value::make_array(list());
      else
        return list[0];
    }

    // bool is_undefined() const { return !is_defined(); }
    // bool is_defined() const   { return items.ptr() != 0; }
    // bool is_inherit() const   { return items && items->list.size() == 0; }
    // void clear()           { items = 0; }
    // void inherit(const shadow_v& v) { if(v.is_defined()) items = v.items; }

    // void push(const item_t& it)
    //{
    //  if( !items ) items = new items_t;
    //  items->list.push(it);
    //}

    shadow_def *clone() const {
      shadow_def *t = new shadow_def(*this);
      if (t->next) t->next = t->next->clone();
      return t;
    }

    shadow_def *clone_invisible() const {
      shadow_def *t = new shadow_def(*this);
      t->color = color_v::transparent_val();
      if (t->next) t->next = t->next->clone_invisible();
      return t;
    }


    bool morph(view& v, element* b,const shadow_def *start, const shadow_def *end, real n) {
      color = morph_color(v,b,start->color, end->color, n);
      offset_x.morph(v,b,start->offset_x, end->offset_x, n);
      offset_y.morph(v, b, start->offset_y, end->offset_y, n);
      radius.morph(v, b, start->radius, end->radius, n);
      spread.morph(v, b, start->spread, end->spread, n);
      if (next) return next->morph(v, b, start->next, end->next, n);
      return true;
    }

    virtual bool is_compatible(const shadow_def *other) const {
      if (inset != other->inset) return false;
      if (next && other->next) return next->is_compatible(other->next);
      if (!next && !other->next) return true;
      return false;
    }

    static bool is_compatible(const shadow_def *a, const shadow_def *b) {
      if (!a || !b) return false;
      if (a->inset != b->inset) return false;
      if (a->next && b->next) return a->next->is_compatible(b->next);
      if (!a->next && !b->next) return true;
      return false;
    }


  };

#pragma pack(push, 4) // IT MUST BE 4 bytes aligned!!!!

  /*struct shadow_v
  {
    struct item_t
    {
      tristate_v inset;
      size_v     offset_x;
      size_v     offset_y;
      size_v     radius;
      size_v     spread;
      color_v    color;
    };
    struct items_t: public resource
    {
      array<item_t> list;
    };
    handle<items_t> items;

    shadow_v() {}

    static shadow_v null_val() { return shadow_v(); }
    static shadow_v inherit_val() { return shadow_v(true); }

    bool is_undefined() const { return !is_defined(); }
    bool is_defined() const   { return items.ptr() != 0; }
    bool is_inherit() const   { return items && items->list.size() == 0; }

    void clear()           { items = 0; }

    void inherit(const shadow_v& v) { if(v.is_defined()) items = v.items; }

    void push(const item_t& it)
    {
      if( !items ) items = new items_t;
      items->list.push(it);
    }

  private:
    shadow_v(bool) { items = new items_t(); }
  }; */

  struct mixin : resource {
    struct item {
      string       attr_name;
      array<value> attr_value;
    };
    ustring        name;
    array<ustring> params;
    array<item>    items;
  };

  struct keyframes : resource
  {
    struct prop_def {
      cssa::symbol_t prop_symbol;
      array<value>   prop_value;
    };

    struct edge_def {
      float           key_position;
      array<prop_def> key_props;
    };

    ustring           name;
    array<edge_def>   edges;
  };

#pragma pack(pop)


} // namespace html

#endif
