#include "html.h"

namespace html {

  using namespace tool;
  using namespace gool;

  const ustring anv_many(WCHARS("(many)"));

  element_context::element_context(view& v, const element* pe, const style* ps) 
    : document_context(v, pe->doc()), _pel(const_cast<element*>(pe)), _ps(const_cast<style*>(ps)) {}

  gool::argb element_context::color_value(tool::value val) const {
    if (val.is_variable_color())
      val = resolve_color_function(*this, val);
    return color_v(val).to_argb();
  }
  size_v element_context::length_value(tool::value val) const {
    if (val.is_variable_length())
      val = resolve_length_function(*this, val);
    return size_v(val);
  }

  value resolve_length_function(const element_context& ctx, value val) {
    function_value *vf = val.get_function();
    assert(vf);
    if (vf->name == WCHARS("length")) {
      if (!vf->params.value(0).is_string()) return value();
      string name = vf->params.value(0).get_string();
      value dv; if (vf->params.size() >= 2) dv = vf->params.value(1);
      if (!ctx.pel()->get_var(ctx, name, val))
        return dv;
    }
    return val;
  }

  value resolve_color_function(const element_context& ctx, value val) {

    function_value *vf = val.get_function();
    assert(vf);

    auto get_color_component = [](const value & v, gool::argb::channel_t &cc) -> bool {
        if (v.is_int()) { cc = gool::argb::channel_t(v.get(0)); return true; }
        if (v.is_percent()) { cc = gool::argb::channel_t(255 * v.get_percent()); return true; }
        if (v.is_double()) { cc = gool::argb::channel_t(255 * v.get(0.0)); return true; }
        return false;
    };
    auto get_alpha_component = [](const value & v, gool::argb::channel_t &cc) -> bool {
        if (v.is_int()) { cc = gool::argb::channel_t(255 * v.get(0)); return true; }
        if (v.is_percent()) { cc = gool::argb::channel_t(255 * v.get_percent()); return true; }
        if (v.is_double()) { cc = gool::argb::channel_t(255 * v.get(0.0)); return true; }
        return false;
    };

    //MOTE: value::is_variable_color() needs to be updated if functions get added here

    if (vf->name == WCHARS("rgb") || vf->name == WCHARS("rgba")) {
      if (vf->params.size() < 3) return val;
      argb cv;
      if(!get_color_component(vf->params.value(0), cv.red) 
      || !get_color_component(vf->params.value(1), cv.green) 
      || !get_color_component(vf->params.value(2), cv.blue)) 
        return val;
      if (vf->params.size() == 4 && !get_alpha_component(vf->params.value(3),cv.alfa))
        return val;
      return cv.to_value();
    }
    else if (vf->name == WCHARS("color")) {
      if (vf->params.size() == 0) return value();
      if (!vf->params.value(0).is_string()) return value();
      string name = vf->params.value(0).get_string();
      value dv; if(vf->params.size() >= 2) dv = vf->params.value(1);
      if (!ctx.pel()->get_var(ctx, name, val))
        return dv;
      return val;
    }
    else if (vf->name == WCHARS("currentcolor")) {
      if (ctx.pstyle())
        return ctx.pstyle()->font_color.to_value();
      return val;
    }
    else if (vf->name == WCHARS("hsl")) {
      if (vf->params.size() != 3) return value();
      gool::hsl z;
      z.h = (float)limit(vf->params.value(0).get(0), 0, 360);
      z.s = (float)(limit(vf->params.value(1).get(0), 0, 100)) / 100.0f;
      z.l = (float)(limit(vf->params.value(2).get(0), 0, 100)) / 100.0f;
      return rgb(z).to_value();
    }
    if (vf->name == WCHARS("hsv")) {
      if (vf->params.size() != 3) return value();
      gool::hsv z;
      z.h = (float)limit(vf->params.value(0).get(0), 0, 360);
      z.s = (float)(limit(vf->params.value(1).get(0), 0, 100)) / 100.0f;
      z.v = (float)(limit(vf->params.value(2).get(0), 0, 100)) / 100.0f;
      return rgb(z).to_value();
    }
    if (vf->name == WCHARS("tint"))
    {
      float   saturation = 0, luminance = 1;
      if (vf->params.size() < 2) return value();
      value basecolor = resolve_var(ctx,vf->params.value(0));
      if (!basecolor.is_color())
        return value();
      gool::rgb i = color_v(basecolor).to_rgb();
      luminance = vf->params.value(1).get_percent(1);
      if (vf->params.size() >= 3)
        saturation = vf->params.value(2).get_percent(0);

      gool::hsl z = i;

      saturation = limit(saturation, -1.0f, 1.0f);
      luminance = limit(luminance, -1.0f, 1.0f);

      if (saturation < 0.0f)
        z.s -= z.s * (-saturation);
      else if (saturation > 0.0)
        z.s += (1.0f - z.s) * (saturation);

      if (luminance < 0.0f)
        z.l -= z.l * (-luminance);
      else if (luminance > 0.0f)
        z.l += (1.0f - z.l) * (luminance);

      return rgb(z).to_value();
    }
    if (vf->name == WCHARS("morph"))
    {
      //Hue, Saturation, Lightness, Alpha, etc. in one bottle.

      if (vf->params.size() < 2) return value();
      value basecolor = resolve_var(ctx,vf->params.value(0));
      if (!basecolor.is_color())
        return value();

      /*if (basecolor.is_named()) {
        if (!pr) {
          c.set_function(vf); // it cannot be resolved now, store it as it is
          return true;
        }
        else if (!pr->resolve_color(basecolor.get_name(), basecolor))
          basecolor = basecolor._color();
      }
      else if (basecolor.is_current()) {
        if (!pr) {
          c.set_function(vf); // it cannot be resolved now, store it as it is
          return true;
        }
        else
          pr->resolve_current_color(basecolor);
      }*/

      gool::argb ic = color_v(basecolor).to_argb();
      //bool       hslop = false;
      bool       alt = false;
      bool       alt2 = false;

      for (int n = 1; n < vf->params.size(); ++n) {
        ustring key = vf->params.key(n).to_string();
        value   val = vf->params.value(n);
        if (key.is_undefined() || val.is_undefined())
          continue;
        if (key == WCHARS("hue") || (alt = (key == WCHARS("rotate"))))
        {
          gool::hsl  z = gool::rgb(ic);
          float hue;
          if (val.is_angle())
          {
            hue = val.get_angle();
            hue = 360.f * hue / real_traits<float>::pi2();
          }
          else
          {
            hue = val.get_percent(1);
            hue *= 360.0f;
          }
          hue = //limit(hue, -360.0f, 360.0f);
            fmod(hue, 360.0f);
          if (alt) // increment
            z.h += hue;
          else
            z.h = hue;
          if (z.h < 0.0f) z.h += 360.0f;
          else if (z.h > 360.0f) z.h -= 360.0f;
          rgb t(z);
          ic.red = t.red;
          ic.green = t.green;
          ic.blue = t.blue;
        }

        else if (key == WCHARS("lightness") || (alt = (key == WCHARS("lighten"))) || (alt2 = (key == WCHARS("darken"))))
        {
          gool::hsl  z = gool::rgb(ic);
          float luminance = val.get_percent(1);
          if (alt) // increment
            z.l = limit(z.l + luminance, 0.0f, 1.0f);
          else if (alt2) // decrement
            z.l = limit(z.l - luminance, 0.0f, 1.0f);
          else // abs value
            z.l = limit(luminance, 0.0f, 1.0f);
          rgb t(z);
          ic.red = t.red;
          ic.green = t.green;
          ic.blue = t.blue;
        }

        else if (key == WCHARS("saturation") || (alt = (key == WCHARS("saturate"))) || (alt2 = (key == WCHARS("desaturate"))))
        {
          gool::hsl  z = gool::rgb(ic);
          float sat = val.get_percent(1);
          if (alt) // increment
            z.s = limit(z.s + sat, 0.0f, 1.0f);
          else if (alt2) // decrement
            z.s = limit(z.s - sat, 0.0f, 1.0f);
          else // abs value
            z.s = limit(sat, 0.0f, 1.0f);
          rgb t(z);
          ic.red = t.red;
          ic.green = t.green;
          ic.blue = t.blue;
        }

        else if (key == WCHARS("opacity") || (alt = (key == WCHARS("opacify"))))
        {
          int a = limit<int>(int(val.get_percent(1) * 255), -255, 255);
          if (alt) // increment
            ic.alfa = (argb::channel_t) limit(ic.alfa + a, 0, 255);
          else // abs value
            ic.alfa = (argb::channel_t)a;
        }
        else if (key == WCHARS("mix") && vf->params.size() >= 3 && vf->params.value(1).is_color())
        {
          color_v color2 = resolve_var(ctx,vf->params.value(1));
          argb gc = color2.to_argb();
          float ratio = limit(val.get_percent(1), 0.0f, 1.0f);
          ic = argb::morph(ic, gc, ratio);
        }
        else if (key == WCHARS("grayscale"))
        {
          uint luma = ic.luminance();
          float ratio = limit(val.get_percent(1), 0.0f, 1.0f);
          argb gc(luma, luma, luma, ic.alfa);
          ic = argb::morph(ic, gc, ratio);
        }
        else if (key == WCHARS("sepia"))
        {
          float ratio = limit(val.get_percent(1), 0.0f, 1.0f);
          argb gc = ic;
          gc.red = argb::channel_t(limit((ic.red * .393f) + (ic.green *.769f) + (ic.blue * .189f), 0.0f, 255.0f));
          gc.green = argb::channel_t(limit((ic.red * .349f) + (ic.green *.686f) + (ic.blue * .168f), 0.0f, 255.0f));
          gc.blue = argb::channel_t(limit((ic.red * .272f) + (ic.green *.534f) + (ic.blue * .131f), 0.0f, 255.0f));
          ic = argb::morph(ic, gc, ratio);
        }
        else
          return value();
      }
      return ic.to_value();
    }

    return value();
  }


  bool flow_v::set(slice<value> values) {

    if (values.length > 1) {
      handle<function_value> fn = new function_value();
      fn->name = WCHARS("grid");

      for (uint n = 0; n < values.length; ++n) {
        array<value> row;
        if (values[n].is_string() && !values[n].is_string_symbol()) {
          ustring row_token = values[n].to_string();
          wtokens cell_toks(row_token, WCHARS(" "));
          wchars  cell_token;
          while (cell_toks.next(cell_token)) {
            if (cell_token.length == 0) continue;
            if (is_digit(cell_token[0])) {
              uint i = to_uint(cell_token);
              row.push(value(i));
            }
            else {
              row.push(value(cell_token));
            }
          }
        }
        fn->params.push(value::make_array(row()));
      }
      ev = flow_grid;
      func = fn;
      return true;
    }
    else
      return set(values[0]);
  }

  bool flow_v::set(const value& val) {
    if (val.is_function()) {
      function_value *pfv = val.get_function();
      if (pfv->name == WCHARS("row"))
        ev = flow_grid;
      else if (pfv->name == WCHARS("columns"))
        ev = flow_columns;
      else if (pfv->name == WCHARS("grid"))
        ev = flow_grid;
      else
        return false;
      func = pfv;
      return true;
    }

    if (val.is_array()) {
      return set(val.values());
    }

    return ev.set(val);
  }

  bool image_repeat_ev::set(slice<value>& vals) 
  {
    if (vals[0].is_function() && vals[0].get_function()->name == "expand") {
      auto fmap = vals[0].get_function();

      int r = background_expand;

      for (int n = 0; n < fmap->params.size(); ++n) {
        ustring s = fmap->params.value(n).to_string();
        if (s == WCHARS("stretch-top"))
          r |= background_stretch_top;
        else if (s == WCHARS("stretch-bottom"))
          r |= background_stretch_bottom;
        else if (s == WCHARS("stretch-left"))
          r |= background_stretch_left;
        else if (s == WCHARS("stretch-right"))
          r |= background_stretch_right;
        else if (s == WCHARS("stretch-middle"))
          r |= background_stretch_center;
      }
      _v = r;
      ++vals;
      return true;
    }
    else if (vals[0].is_function() && vals[0].get_function()->name == "stretch") {
      auto fmap = vals[0].get_function();

      int tval = background_stretch;
      for (int n = 0; n < fmap->params.size(); ++n) {
        ustring s = fmap->params.value(n).to_string();
        if (s == WCHARS("keep-ratio")) tval |= background_keep_ratio;
      }
      _v = tval;
      ++vals;
      return true;
    }
    else if (vals[0].is_function() && vals[0].get_function()->name == "no-repeat") {
      auto fmap = vals[0].get_function();
      int  tval = background_no_repeat;
      for (int n = 0; n < fmap->params.size(); ++n) {
        ustring s = fmap->params.value(n).to_string();
        if (s == WCHARS("keep-ratio")) tval |= background_keep_ratio;
      }
      _v = tval;
      ++vals;
      return true;
    }

    if (!vals[0].is_string()) { clear(); return false; }
    ustring s = vals[0].to_string();

    if (s.length() == 0) { clear(); return false; }
    if (s == WCHARS("repeat")) {
      _v = background_repeat;
      ++vals;
    }
    else if (s == WCHARS("no-repeat")) {
      _v = background_no_repeat;
      ++vals;
    }
    else if (s == WCHARS("repeat-x")) {
      _v = background_repeat_x;
      ++vals;
    }
    else if (s == WCHARS("repeat-y")) {
      _v = background_repeat_y;
      ++vals;
    }
    else if (s == WCHARS("expand")) {
      _v = background_expand;
      ++vals;
    }
    else if (s == WCHARS("stretch")) {
      _v = background_stretch;
      ++vals;
    }
    else
    {
      clear(); 
      return false;
    }

    while (vals)
    {
      s = vals->to_string();
      if (s == WCHARS("keep-ratio")) 
      {
        if ((_v & 0xf) == background_stretch) {
          _v |= background_keep_ratio;
          ++vals;
          continue;
        }
        else if ((_v & 0xf) == background_no_repeat) {
          _v |= background_keep_ratio;
          ++vals;
          continue;
        }
      }
      else if (s == WCHARS("stretch-top")) {
        if ((_v & 0xf) == background_expand) {
          _v |= background_stretch_top;
          ++vals;
          continue;
        }
      }
      else if (s == WCHARS("stretch-bottom")) {
        if ((_v & 0xf) == background_expand) {
          _v |= background_stretch_bottom;
          ++vals;
          continue;
        }

      }
      else if (s == WCHARS("stretch-left")) {
        if ((_v & 0xf) == background_expand) {
          _v |= background_stretch_left;
          ++vals;
          continue;
        }
      }
      else if (s == WCHARS("stretch-right")) {
        if ((_v & 0xf) == background_expand) {
          _v |= background_stretch_right;
          ++vals;
          continue;
        }
      }
      else if (s == WCHARS("stretch-middle") || s == WCHARS("stretch-center")) {
        if ((_v & 0xf) == background_expand) {
          _v |= background_stretch_center;
          ++vals;
          continue;
        }
      }
      break;
    }
    return true;
  }

  ustring image_repeat_ev::to_string() const {
    if (is_undefined()) return ustring();
    int iv = int(_v);
    switch (iv & 0xF) {
      case background_repeat: return WCHARS("repeat");
      case background_no_repeat: return (iv & background_keep_ratio) ? WCHARS("no-repeat keep-ratio") : WCHARS("no-repeat");
      case background_repeat_x: return WCHARS("repeat-x");
      case background_repeat_y: return WCHARS("repeat-y");
      case background_stretch:
        if (iv & background_keep_ratio)
          return WCHARS("stretch keep-ratio");
        else
          return WCHARS("stretch");
      case background_expand: {
        ustring r("expand");
        if ((iv & background_stretch_top) == background_stretch_top)
          r += ustring(" ") + W("stretch-top");
        if ((iv & background_stretch_bottom) == background_stretch_bottom)
          r += ustring(" ") + W("stretch-bottom");
        if ((iv & background_stretch_left) == background_stretch_left)
          r += ustring(" ") + W("stretch-left");
        if ((iv & background_stretch_right) == background_stretch_right)
          r += ustring(" ") + W("stretch-right");
        return r;
      }
    }
    return ustring();
  }

  value image_repeat_ev::to_value() const {
    if (is_undefined()) return value();
    int iv = int(_v);

    switch (iv & 0xF) 
    {
      case background_repeat:   return value(CHARS("repeat"));
      case background_repeat_x: return value(CHARS("repeat-x"));
      case background_repeat_y: return value(CHARS("repeat-y"));
      case background_stretch:  
        if (iv & background_keep_ratio) {
          value v[2] = { value(CHARS("stretch")), value(CHARS("keep-ratio")) };
          return value::make_array(items_of(v));
        }
        else
          return value(CHARS("stretch"));

      case background_no_repeat: 
        if (iv & background_keep_ratio) {
          value v[2] = { value(CHARS("no-repeat")), value(CHARS("keep-ratio")) };
          return value::make_array(items_of(v));
        } 
        else
          return value(CHARS("no-repeat"));
      case background_expand: {
        if((iv & ~0xF) == 0)
          return value(CHARS("expand"));
        array<value> r;
        r.push(value(CHARS("expand")));
        if ((iv & background_stretch_top) == background_stretch_top)
          r.push(value(CHARS("stretch-top")));
        if ((iv & background_stretch_bottom) == background_stretch_bottom)
          r.push(value(CHARS("stretch-bottom")));
        if ((iv & background_stretch_left) == background_stretch_left)
          r.push(value(CHARS("stretch-left")));
        if ((iv & background_stretch_right) == background_stretch_right)
          r.push(value(CHARS("stretch-right")));
        return value::make_array(r());
      }
    }
    return value();
  }


  bool size_v::from_value(const tool::value &v, size_v &sz) {
    return length_value(sz, v, NUMERIC_TO_NUMERIC);
  }

  tool::value size_v::to_value() const 
  {
    value val;
    if(is_undefined())
      return value();
    /*else if (is_named()) {
      tool::handle<tool::function_value> pf = new tool::function_value();
      pf->name                              = WCHARS("var");
      pf->params.push(tool::value(attr::symbol_name(get_name())));
      size_v t = *this;
      t.unit   = get_base_unit();
      pf->params.push(tool::value(t));
      return value::make_function(pf);
    }*/
    else if (is_expr()) {
      return value(WCHARS("calc(...)"));
      //val.set_object(d.expr);
    } 
    else if (is_percent_of_height()) {
      tool::handle<tool::function_value> pf = new tool::function_value();
      pf->name = WCHARS("height");
      size_v t; t.set_percents((float)d1000(d.value));
      pf->params.push(tool::value(t));
    }
    else if (is_percent_of_width()) {
      tool::handle<tool::function_value> pf = new tool::function_value();
      pf->name = WCHARS("width");
      size_v t; t.set_percents((float)d1000(d.value));
      pf->params.push(tool::value(t));
    }
    else
      val._set_(d.value, value::t_length, unit);
    return val;
  }

  bool size_v::set(const value &v, NUMERIC_CONVERSION number_cvt) {
    if (is_none_value(v)) {
      this->clear();
      return true;
    }
    else if (v.is_inherit()) {
      this->set_literal(size_v::special_values::$inherit);
      return true;
    }
    else if (v.is_auto()) {
      this->set_literal(size_v::special_values::$auto);
      return true;
    }
    else if (v.is_function()) {
      function_value *fv = v.get_function();
      if (fv->params.size() == 0) return false;
      value p = fv->params.value(0);
      if (fv->name == WCHARS("width")) {
        if (fv->params.size() != 1) return false;
        if (!p.is_percent()) return false;
        this->set_percents_of_width(float(p.get(0.0)));
      }
      else if (fv->name == WCHARS("height")) {
        if (fv->params.size() != 1) return false;
        if (!p.is_percent()) return false;
        this->set_percents_of_height(float(p.get(0.0)));
      }
      else if (fv->name == WCHARS("fx") && fv->params.size() >= 1 && fv->params.value(0).is_number()) {
        this->set_flex(fv->params.value(0).to_float());
        if (fv->params.size() == 1)
          return true;
        if (fv->params.key(1).is_defined()) { // named parameters
          for (int n = 1; n < fv->params.size(); ++n) {
            if (fv->params.key(n) == value(WCHARS("min")) && fv->params.value(n).is_length())
              this->set_min(size_v(fv->params.value(n)));
            else if (fv->params.key(n) == value(WCHARS("max")) && fv->params.value(n).is_length())
              this->set_max(size_v(fv->params.value(n)));
            else
              return false;
          }
        }
        else {
          if (fv->params.size() >= 2) {
            if (fv->params.value(1).is_length())
              this->set_min(size_v(fv->params.value(1)));
            else
              return false;
          }
          if (fv->params.size() >= 3) {
            if (fv->params.value(2).is_length())
              this->set_max(size_v(fv->params.value(2)));
            else
              return false;
          }
        }
        return true;
      }
      else
        return false;
      return true;
    }
    else if (v.is_conduit()) {
      this->set_calc_expr(static_cast<eval::conduit *>(v.get_object()));
      return true;
    }
    else if (v.is_string_symbol() || v.is_string()) {
      ustring us = v.get_string();
      if (us == WCHARS("left") || us == WCHARS("top"))
        this->set_percents(0);
      else if (us == WCHARS("center"))
        this->set_percents(50);
      else if (us == WCHARS("right") || us == WCHARS("bottom"))
        this->set_percents(100);
      else
        return this->set_literal(us);
      return true;
    }
    else {
      uint u = v.units();
      if (v.is_int() && u >= value::em && u <= value::nm) {
        this->d.value = v.get_int();
        this->unit = v.units();
        return true;
      }

      if (v.is_number()) {

        switch (number_cvt) {
          case NUMERIC_TO_NUMERIC:  this->set_number(v.get(0.0)); break;
          case NUMERIC_TO_PIXEL:    this->set_px(v.get(0.0)); break;
          case NUMERIC_TO_DIP:      if(v.is_double()) this->set_dips(int(v.get(0.0) + 0.5)); else this->set_dips(v.get_int()); break;
          case NUMERIC_TO_PERCENT:  this->set_percents(100.0f * v.get(0.0f)); break;
        }
        /*if (!number_to_pixels && v.is_int() && u == 0) {
          this->set_dips(v.get_int());
          //this->set_number(v.get_int());
          return true;
        }
        if (!number_to_pixels && v.is_double() && u == 0) {
          this->set_dips(int(v.get_double() + 0.5));
          //this->set_number(v.get_double());
          return true;
        }*/
        return true;
      }
      else if (v.is_length()) {
        this->d.value = v._int();
        this->unit = u;
        return true;
      }
      /*if (v.is_number()) {
        this->set_pixels(v.get(0.0));
        return true;
      }*/
    }
    return false;
  }


  ustring size_v::to_string() const {
    if (is_undefined()) return ustring();
    const wchar *sunits = W("");

    switch (unit) {
    case size_v::unit_type::em: sunits = W("em"); break;
    case size_v::unit_type::rem: sunits = W("rem"); break;
    case size_v::unit_type::ex: sunits = W("ex"); break;
    case size_v::unit_type::ch: sunits = W("ch"); break;
    case size_v::unit_type::pr: return ustring::format(W("%d%%"), d.value / 1000);
    case size_v::unit_type::pr_width: return ustring::format(W("width(%d%%)"), d.value / 1000);
    case size_v::unit_type::pr_height: return ustring::format(W("height(%d%%)"), d.value / 1000);
    case size_v::unit_type::pr_view_width: sunits = W("vw"); break;
    case size_v::unit_type::pr_view_height: sunits = W("vh"); break;
    case size_v::unit_type::pr_view_min: sunits = W("vmin"); break;
    case size_v::unit_type::pr_view_max: sunits = W("vmax"); break;
    case size_v::unit_type::sp: // sunits = L"%%";
                                // return ustring::format(W("%f%%%%"),v.d.value / 1000.0);
      sunits = W("*");
      break;
    case size_v::unit_type::rs: assert(false); break;
    case size_v::unit_type::as:
      switch (d.value) {
      case size_v::special_values::$xx_small: return WCHARS("xx-small");
      case size_v::special_values::$x_small: return WCHARS("x-small");
      case size_v::special_values::$small: return WCHARS("small");
      case size_v::special_values::$medium: return WCHARS("medium");
      case size_v::special_values::$large: return WCHARS("large");
      case size_v::special_values::$x_large: return WCHARS("x-large");
      case size_v::special_values::$xx_large: return WCHARS("xx-large");
      case size_v::special_values::$thin: return WCHARS("thin");
      case size_v::special_values::$thick: return WCHARS("thick");
      case size_v::special_values::$auto: return WCHARS("auto");
      case size_v::special_values::$none: return WCHARS("none");
      case size_v::special_values::$inherit: return WCHARS("inherit");
      case size_v::special_values::$min_content: return WCHARS("min-content");
      case size_v::special_values::$max_content: return WCHARS("max-content");

      case size_v::special_values::$scrollbar_height: return WCHARS("system-scrollbar-height");
      case size_v::special_values::$scrollbar_width: return WCHARS("system-scrollbar-width");
      case size_v::special_values::$border_width: return WCHARS("system-border-width");

      case size_v::special_values::$border_3d_width: return WCHARS("system-3d-border-width");
      case size_v::special_values::$small_icon_height: return WCHARS("system-small-icon-height");
      case size_v::special_values::$small_icon_width: return WCHARS("system-small-icon-width");
      case size_v::special_values::$ui_scale: return WCHARS("ui-scale");
      case size_v::special_values::$cover: return WCHARS("cover");
      case size_v::special_values::$contain: return WCHARS("contain");
      }
      break;
    case size_v::unit_type::px:  sunits = W("px"); break;
    case size_v::unit_type::ppx: sunits = W("ppx"); break;
    case size_v::unit_type::dip: sunits = W("dip"); break;
    case size_v::unit_type::in: sunits = W("in"); break;
    case size_v::unit_type::pt: sunits = W("pt"); break;
    case size_v::unit_type::pc: sunits = W("pc"); break;
    case size_v::unit_type::cm: sunits = W("cm"); break;
    case size_v::unit_type::mm: sunits = W("mm"); break;
    case size_v::unit_type::nm: sunits = W("#"); break;
    case size_v::unit_type::expr: return WCHARS("calc()"); break;
    }
    if (d.value % 1000)
      return ustring::format(W("%f%s"), double(d.value) / 1000.0, sunits);
    else
      return ustring::format(W("%d%s"), d.value / 1000, sunits);
  }

  string attribute_bag::get_url(const string & base_url,
                                name_or_symbol ns) const {
    // return
    ustring s = get_string(ns);
    if (s.length() == 0) return string();

    if (s[0] == '#') return s;

    string as;
    if (url::need_escapement(s))
      as = url::escape(s);
    else
      as = s;

    url u(as);

    if (!u.is_absolute()) as = combine_url(base_url, as);
    // s = url::unescape(s);
    return as;
  }

  ustring to_string(const space_v &v) {
    if (v.is_undefined()) return ustring();
    return ustring::format(W("%d"), v._v);
  }

  ustring to_string(const int_v &v) {
    if (v.is_undefined()) return ustring();
    return ustring::format(W("%d"), v._v);
  }

  ustring to_string(const color_v &v) {
    return v.to_string();
  }

  ustring to_string(double d) { return ustring::format(W("%g"), d); }

  void from_string(space_v &v, wchars ss) {
    v.clear();
    if (ss.length == 0) return;
    int n = str_to_i(ss, 0);
    if (ss.length != 0) return;
    v._v = n;
  }

  void from_string(int_v &v, wchars ss) {
    v.clear();
    if (ss.length == 0) return;
    int n = str_to_i(ss, 0);
    if (ss.length != 0) return;
    v._v = n;
  }

  void from_string(float_v &v, wchars ss) {
    v.clear();
    size_t l = ss.length;
    if (l == 0) return;
    double n = str_to_f(ss, 0.0);
    if (ss.length == l) return;
    v._v = float(n);
  }

  void from_string(dimension_v &v, wchars ss) {
    v.clear();
    if (ss.length == 0) return;
    v._v = str_to_i(ss, 0);
    if (ss.index_of('%') == 0) v._v = -v._v;
  }

  void from_string(color_v &v, wchars ss) {
    v.clear();
    if (ss.length == 0) return;
    if (ss == WCHARS("inherit")) {
      v = color_v::inherit_val();
      return;
    } else if (ss == WCHARS("none")) {
      v = color_v::transparent_val();
      return;
    }

    string s = ss;
    s.to_lower();
    color_v c = parse_color(s);
    if (c.is_defined()) v = c;
  }

  /*inline bool streq2(const char* s, const char* s1)
  {
    return s[0] == s1[0] && s[1] == s1[1];
  }

  inline bool streq3(const char* s, const char* s1)
  {
    return s[0] == s1[0] && s[1] == s1[1] && s[2] == s1[2];
  }*/

  // parse canonical size_v
  size_v parse_size_v(wchars &ss) {
    size_v v;
    if (ss.length == 0) return v;

    wchars css = ss;
    float  d   = str_to_f(ss, 0.0f);
    if (css == ss) return v; // nothing has been parsed

    // parsed something, we have units left there
    // int unit_length = endptr - lastptr;

    if (ss.starts_with(WCHARS("px"))) {
      v.set(d, size_v::unit_type::px);
      ss.prune(2);
    } else if (ss.starts_with(WCHARS("ppx"))) {
      v.set(d, size_v::unit_type::ppx);
      ss.prune(3);
    } else if (ss.starts_with(WCHARS("dip"))) {
      v.set(d, size_v::unit_type::dip);
      ss.prune(3);
    } else if (ss.starts_with(WCHARS("pt"))) {
      v.set(d, size_v::unit_type::pt);
      ss.prune(2);
    } else if (ss.starts_with(WCHARS("em"))) {
      v.set(d, size_v::unit_type::em);
      ss.prune(2);
    } else if (ss.starts_with(WCHARS("ch"))) {
      v.set(d, size_v::unit_type::ch);
      ss.prune(2);
    } else if (ss.starts_with(WCHARS("rem"))) {
      v.set(d, size_v::unit_type::rem);
      ss.prune(2);
    } else if (ss.starts_with(WCHARS("ex"))) {
      v.set(d, size_v::unit_type::ex);
      ss.prune(2);
    } else if (ss.starts_with(WCHARS("in"))) {
      v.set(d, size_v::unit_type::in);
      ss.prune(2);
    } else if (ss.starts_with(WCHARS("cm"))) {
      v.set(d, size_v::unit_type::cm);
      ss.prune(2);
    } else if (ss.starts_with(WCHARS("mm"))) {
      v.set(d, size_v::unit_type::mm);
      ss.prune(2);
    } else if (ss.starts_with(WCHARS("pc"))) {
      v.set(d, size_v::unit_type::pc);
      ss.prune(2);
    } else if (ss.starts_with(WCHARS("vw"))) {
      v.set_percents_of_view_width(d);
      ss.prune(2);
    } else if (ss.starts_with(WCHARS("vh"))) {
      v.set_percents_of_view_height(d);
      ss.prune(2);
    } else if (ss.starts_with(WCHARS("vmin"))) {
      v.set_percents_of_view_min(d);
      ss.prune(4);
    } else if (ss.starts_with(WCHARS("vmax"))) {
      v.set_percents_of_view_max(d);
      ss.prune(4);
    } else if (ss.starts_with(WCHARS("%%"))) {
      v.set_flex(d / 100);
      ss.prune(2);
    } else if (ss.starts_with(WCHARS("*"))) {
      v.set_flex(d);
      ss.prune(1);
    } else if (ss.starts_with(WCHARS("%"))) {
      v.set_percents(d);
      ss.prune(1);
    } else if (ss.starts_with(WCHARS("#"))) {
      v.set_number(int(d));
      ss.prune(1);
    } else
      v.set(d, size_v::unit_type::nm);

    return v;
  }

  /*enum SIZE_FROM_STRING {
    NUMBER_PX,
    NUMBER_DIP,
    NUMBER_PERCENT, // 0..1 mapped to 0%..100%
  };*/

  void from_string(size_v &v, wchars ss, SIZE_FROM_STRING num_flavour) {
    v.clear();
    if (ss.length == 0) return;

    wchars css = ss;

    if (ss == WCHARS("*")) {
      v.set_flex(1);
      return;
    }

    float d = str_to_f(ss, 0.0f);

    if (css != ss) // parsed something, we have units left there
    {
      // int unit_length = endptr - lastptr;

      size_v::unit_type t = size_v::unit_type::px;
      if (ss.length == 0) { // parsed in full, unitless pixel - dip
        switch (num_flavour) {
          case NUMBER_PX: v.set(d, size_v::unit_type::ppx); break;
          case NUMBER_DIP: v.set(d, size_v::unit_type::dip); break;
          case NUMBER_PERCENT: v.set(d*100.0, size_v::unit_type::pr); break;
          case NUMBER_NUMBER: v.set(d, size_v::unit_type::nm); break;
        }
        return;
      }
      if (ss == WCHARS("px"))
        t = size_v::unit_type::px;
      else if (ss == WCHARS("dip"))
        t = size_v::unit_type::dip;
      else if (ss == WCHARS("ppx"))
        t = size_v::unit_type::ppx;
      else if (ss == WCHARS("pt"))
        t = size_v::unit_type::pt;
      else if (ss == WCHARS("em"))
        t = size_v::unit_type::em;
      else if (ss == WCHARS("rem"))
        t = size_v::unit_type::rem;
      else if (ss == WCHARS("ex"))
        t = size_v::unit_type::ex;
      else if (ss == WCHARS("ch"))
        t = size_v::unit_type::ch;
      else if (ss == WCHARS("%")) {
        v.set_percents(d);
        return;
      } else if (ss == WCHARS("%%")) {
        v.set_flex(d / 100);
        return;
      } else if (ss == WCHARS("in"))
        t = size_v::unit_type::in;
      else if (ss == WCHARS("cm"))
        t = size_v::unit_type::cm;
      else if (ss == WCHARS("mm"))
        t = size_v::unit_type::mm;
      else if (ss == WCHARS("pc"))
        t = size_v::unit_type::pc;
      else if (ss == WCHARS("vw"))
        t = size_v::unit_type::pr_view_width;
      else if (ss == WCHARS("vh"))
        t = size_v::unit_type::pr_view_height;
      else if (ss == WCHARS("vmin"))
        t = size_v::unit_type::pr_view_min;
      else if (ss == WCHARS("vmax"))
        t = size_v::unit_type::pr_view_max;
      else if (ss == WCHARS("*")) {
        v.set_flex(d);
        return;
      }
      else if (ss == WCHARS("#")) {
        v.set_number(int(d));
        return;
      } else {
        v.set(d, size_v::unit_type::px);
        return;
      }
      v.set(d, t);
    } else {
      /*if (css == WCHARS("*"))
        v.set_flex(1.0f);
      else if (css == WCHARS("xx-small"))
        v.set_literal(size_v::special_values::$xx_small);
      else if (css == WCHARS("x-small"))
        v.set_literal(size_v::special_values::$x_small);
      else if (css == WCHARS("small"))
        v.set_literal(size_v::special_values::$small);
      else if (css == WCHARS("medium"))
        v.set_literal(size_v::special_values::$medium);
      else if (css == WCHARS("large"))
        v.set_literal(size_v::special_values::$large);
      else if (css == WCHARS("x-large"))
        v.set_literal(size_v::special_values::$x_large);
      else if (css == WCHARS("xx-large"))
        v.set_literal(size_v::special_values::$xx_large);
      else if (css == WCHARS("smaller"))
        v.set(false);
      else if (css == WCHARS("larger"))
        v.set(true);
      else if (css == WCHARS("thin"))
        v.set_literal(size_v::special_values::$thin);
      else if (css == WCHARS("thick"))
        v.set_literal(size_v::special_values::$thick);
      else if (css == WCHARS("auto"))
        v.set_literal(size_v::special_values::$auto);
      else if (css == WCHARS("none"))
        v.set_literal(size_v::special_values::$none);
      else if (css == WCHARS("inherit"))
        v.set_literal(size_v::special_values::$inherit);
      else if (css == WCHARS("min-content") || css == WCHARS("min-intrinsic"))
        v.set_literal(size_v::special_values::$min_content);
      else if (css == WCHARS("max-content") || css == WCHARS("max-intrinsic"))
        v.set_literal(size_v::special_values::$max_content);

      else if (css == WCHARS("system-scrollbar-height")) v.set_literal(size_v::special_values::$scrollbar_height);
      else if (css == WCHARS("system-scrollbar-width")) v.set_literal(size_v::special_values::$scrollbar_width);
      else if (css == WCHARS("system-border-width")) v.set_literal(size_v::special_values::$border_width);
      else if (css == WCHARS("system-3d-border-width")) v.set_literal(size_v::special_values::$border_3d_width);
      else if (css == WCHARS("system-small-icon-height")) v.set_literal(size_v::special_values::$small_icon_height);
      else if (css == WCHARS("system-small-icon-width")) v.set_literal(size_v::special_values::$small_icon_width);

      else if (css == WCHARS("window-scrollbar-height")) v.set_literal(size_v::special_values::$scrollbar_height);
      else if (css == WCHARS("window-scrollbar-width")) v.set_literal(size_v::special_values::$scrollbar_width);
      else if (css == WCHARS("window-border-width")) v.set_literal(size_v::special_values::$border_width);
      else if (css == WCHARS("window-icon-height")) v.set_literal(size_v::special_values::$small_icon_height);
      else if (css == WCHARS("window-icon-width")) v.set_literal(size_v::special_values::$small_icon_width);

      else if (css == WCHARS("window-caption-height")) v.set_literal(size_v::special_values::$window_caption_height);
      else if (css == WCHARS("window-button-height")) v.set_literal(size_v::special_values::$window_button_height);
      else if (css == WCHARS("window-button-width")) v.set_literal(size_v::special_values::$window_button_width);
      else if (css == WCHARS("window-frame-width")) v.set_literal(size_v::special_values::$window_frame_width);

      else if (css == CHARS("ui-scale"))
        v.set_literal(size_v::special_values::$ui_scale);*/
      v.set_literal(css);
    }
  }

  static int map[] = {75, 100, 120, 140, 180, 240, 360};

  /*size_v unname(size_v v, const element* pel, const style* ps) {
    if (v.is_named()) {
      uint sym = v.get_name();
      if (!pel->get_length_var(sym, v, ps)) v.unit = v.get_base_unit();
    }
    return v;
  }*/

  /*color_v unname(color_v v, const element* pel, const style* ps)
  {
    if (v.is_named()) {
      uint sym = v.get_name();
      if (!pel->get_color_var(sym, v, ps))
        v = v._color();
    }
    return v;
  }*/
  
  bool size_v::morph(view& v, element* b, size_v start, size_v end, real n) {

    uint start_u = start.unit;
    uint end_u = end.unit;

    if (start_u != end_u) {
      if (start_u == size_v::unit_type::sp)
        end = size_v(0, unit_type(end_u = size_v::unit_type::sp));
      else if (end_u == size_v::unit_type::sp)
        start = size_v(0, unit_type(start_u = size_v::unit_type::sp));
      else if (start_u == size_v::unit_type::pr)
        end = size_v(0, unit_type(end_u = size_v::unit_type::pr));
      else if (end_u == size_v::unit_type::pr)
        start = size_v(0, unit_type(start_u = size_v::unit_type::pr));
      else {
        start.set_ppx(html::pixels(v, b, start).width());
        end.set_ppx(html::pixels(v, b, end).width());
        start_u = end_u = size_v::unit_type::ppx;
      }
    }
    int r_start = start.d.value;
    int r_end = end.d.value;
    int r_next = int(n * (r_end - r_start)) + r_start;
    int p_val = d.value;
    d.value = r_next;
    unit = start_u;
    return d.value != p_val;
  }

  int size_v::pixels(int defv) const {
    int kp = 0;
    switch (unit) {
    default: return defv;
    case size_v::unit_type::rs: assert(false); break;
    case size_v::unit_type::as:
      switch (d.value) {
      case size_v::special_values::$medium: // medium
        return 2;
      case size_v::special_values::$thin: // thin
        return 1;
      case size_v::special_values::$thick: // thick
        return 3;
      default:
        if (d.value >= size_v::special_values::$xx_small && d.value <= size_v::special_values::$xx_large)
          return map[limit<int>(d.value, size_v::special_values::$xx_small, size_v::special_values::$xx_large) - size_v::special_values::$xx_small] * 100;
        else if (d.value >= size_v::special_values::$system_metrics_first && d.value <= size_v::special_values::$system_metrics_last)
          return resolution_provider::desktop().get_window_metrics(value::length_special_values(d.value));
        else
          return defv;
      }
      break;
    case size_v::unit_type::nm: return i1000(d.value);
    case size_v::unit_type::px: return i1000(d.value);
    case size_v::unit_type::ppx: return i1000(d.value);
    case size_v::unit_type::in: kp = d.value * 72; break;
    case size_v::unit_type::pt: // Points (1 point = 1/72 inches).
      kp = d.value;
      break;
    case size_v::unit_type::dip: // Dips (1 dip = 1/96 inches).
      kp = muldiv(d.value, 72, 96);
      break;
    case size_v::unit_type::pc: // Picas (1 pica = 12 points).
      kp = d.value * 12;
      break;
    case size_v::unit_type::cm: kp = muldiv(d.value, 72 * 100, 254); break;
    case size_v::unit_type::mm: kp = muldiv(d.value, 72 * 100, 2540); break;
    }
    return muldiv(kp, resolution_provider::desktop().pixels_per_inch().y,72000);
  }

  void size_v::resolve(view &v, const size_v &base) {
    switch (unit) {
    case size_v::unit_type::em: d.value = muldiv(d.value, base.kpoints(), 1000); break;
    case size_v::unit_type::rem: d.value = muldiv(d.value, base.kpoints(), 1000); break;
    case size_v::unit_type::ex: d.value = muldiv(d.value, base.kpoints(), 2000); break;
    case size_v::unit_type::pr: d.value = muldiv(d.value, base.kpoints(), 100000); break;
    case size_v::unit_type::pr_width:
    case size_v::unit_type::pr_height:
      // assert(false);
      return;
      // break;
    case size_v::unit_type::pr_view_width: {
      v.has_view_relative_units = true;
      d.value                   = muldiv(d.value, v.dimension().x, 100);
      goto TO_KPOINTS;
    }
    case size_v::unit_type::pr_view_height: {
      v.has_view_relative_units = true;
      d.value                   = muldiv(d.value, v.dimension().y, 100);
      goto TO_KPOINTS;
    }
    case size_v::unit_type::pr_view_max: {
      v.has_view_relative_units = true;
      int tbase                 = max(v.dimension().x, v.dimension().y);
      d.value                   = muldiv(d.value, tbase, 100);
      goto TO_KPOINTS;
    }
    case size_v::unit_type::pr_view_min: {
      v.has_view_relative_units = true;
      int tbase                 = min(v.dimension().x, v.dimension().y);
      d.value                   = muldiv(d.value, tbase, 100);
      goto TO_KPOINTS;
    }

    case size_v::unit_type::rs:
      // assert(false);
      {
        int idx = 7;
        int bdp = base.dpoints();
        for (int i = 0; i <= 6; ++i) {
          if (bdp <= map[i]) {
            idx = i;
            break;
          }
        }
        idx += d.value;
        d.value = map[limit<int>(idx, 0, 6)] * 100;
      }
      break;

    case size_v::unit_type::as:

      switch (d.value) {

        case size_v::special_values::$inherit:
          *this = base;
          break;
        case size_v::special_values::$smaller:
          d.value = d.value * 100 / 120;
          break;
        case size_v::special_values::$larger:
          d.value = d.value * 120 / 100;
          break;
        case size_v::special_values::$xx_small:
        case size_v::special_values::$x_small:
        case size_v::special_values::$small:
        case size_v::special_values::$medium:
        case size_v::special_values::$large:
        case size_v::special_values::$x_large:
        case size_v::special_values::$xx_large:
          d.value = map[limit<int>(d.value, size_v::special_values::$xx_small, size_v::special_values::$xx_large) - size_v::special_values::$xx_small] * 100;
          break;
        default:
          set_ppx(v.get_window_metrics(value::length_special_values(d.value)));
          goto TO_KPOINTS;
      }
      break;

    case size_v::unit_type::nm:
      // fall through
    case size_v::unit_type::ppx:
    TO_KPOINTS:
      d.value = muldiv(d.value, 72, v.pixels_per_inch().y);
      break;
    case size_v::unit_type::px:
      if (!v.px_as_dip()) goto TO_KPOINTS;
      // else fall through 
    case size_v::unit_type::dip: d.value = muldiv(d.value, 72, 96); break;
    case size_v::unit_type::in: d.value = d.value * 72; break;
    case size_v::unit_type::pt: // Points (1 point = 1/72 inches).
      // value = value / 100;
      break;
    case size_v::unit_type::pc: // Picas (1 pica = 12 points).
      d.value = d.value * 12;
      break;
    case size_v::unit_type::cm: d.value = muldiv(d.value, 72 * 100, 254); break;
    case size_v::unit_type::mm: d.value = muldiv(d.value, 72 * 100, 2540); break;
    case size_v::unit_type::expr: assert(false); return;
    default:
      // value = v.get_default_style().font_size.value;
      d.value = base.kpoints();
    }
    // assert(value < 48000);
    unit = size_v::unit_type::pt;
  }

  void size_v::derive(const size_v &descendant) {
    if (is_undefined()) {
      // assert(descendant.defined());
      *this = descendant;
      return;
    }
    switch (descendant.unit) {
    case size_v::unit_type::em:
      d.value = int((int64(d.value) * descendant.d.value) / 1000);
      return;
    case size_v::unit_type::rem:
      d.value = int((int64(d.value) * descendant.d.value) / 1000);
      return;
    case size_v::unit_type::ex:
      d.value = int((int64(d.value) * descendant.d.value) / 2000);
      return;
    case size_v::unit_type::pr:
      d.value = int(d.value * descendant.d.value) / 100000;
      return;
    case size_v::unit_type::pr_width:
    case size_v::unit_type::pr_height:
    case size_v::unit_type::pr_view_width:
    case size_v::unit_type::pr_view_height:
    case size_v::unit_type::pr_view_max:
    case size_v::unit_type::pr_view_min: assert(false); break;

    case size_v::unit_type::rs:
      if (unit == size_v::unit_type::pt) {
        int idx = 7;
        int bdp = dpoints();
        for (int i = 1; i <= 7; ++i) {
          if (bdp <= map[i - 1]) {
            idx = i;
            break;
          }
        }
        idx += descendant.d.value;
        d.value = map[limit<int>(idx, 1, 7) - 1] * 100;
      } else
        switch (descendant.d.value) {
        case -3: d.value = int(float(d.value) / 1.2 / 1.2 / 1.2); break;
        case -2: d.value = int(float(d.value) / 1.2 / 1.2); break;
        case -1: d.value = int(float(d.value) / 1.2); break;
        case 1: d.value = int(float(d.value) * 1.2); break;
        case 2: d.value = int(float(d.value) * 1.2 * 1.2); break;
        case 3: d.value = int(float(d.value) * 1.2 * 1.2 * 1.2); break;
        }
      unit = size_v::unit_type::pt;
      return;
    case size_v::unit_type::as:
      d.value = map[limit<int>(descendant.d.value, 1, 7) - 1] * 100;
      unit    = size_v::unit_type::pt;
      return;
    default: *this = descendant;
    }
  }

  void size_v::resolve_pixels(resolution_provider &v, const size_v &base) {
    int kp = 0;
    switch (unit) {
    case size_v::unit_type::em: kp = int((int64(d.value) * base.kpoints()) / 1000); break;
    case size_v::unit_type::rem: kp = int((int64(d.value) * base.kpoints()) / 1000); break;
    case size_v::unit_type::ex: kp = int((int64(d.value) * base.kpoints()) / 2000); break;

    case size_v::unit_type::pr:
    case size_v::unit_type::pr_width:
    case size_v::unit_type::pr_height:
    case size_v::unit_type::pr_view_width:
    case size_v::unit_type::pr_view_height:
    case size_v::unit_type::pr_view_max:
    case size_v::unit_type::pr_view_min:
    case size_v::unit_type::sp:
      return;
      // assert(false);
      // break;
    case size_v::unit_type::rs: assert(false); break;
    case size_v::unit_type::as:
      // assert(false);
      switch (d.value) {
        case size_v::special_values::$medium: set_dips(2); return;
        case size_v::special_values::$thin:   set_dips(1); return;
        case size_v::special_values::$thick:  set_dips(3); return;

        case size_v::special_values::$auto:
        case size_v::special_values::$none: return;

        default:  set_ppx(v.get_window_metrics(value::length_special_values(d.value))); return;
      }
      unit = size_v::unit_type::px;
      return;
    case size_v::unit_type::in: kp = d.value * 72; break;
    case size_v::unit_type::pt: // Points (1 point = 1/72 inches).
      kp = d.value;
      break;
    case size_v::unit_type::ppx: return;
    case size_v::unit_type::px: if(!v.px_as_dip()) return; // else fall through
    case size_v::unit_type::dip: kp = muldiv(d.value, 72, 96); break;
    case size_v::unit_type::pc: // Picas (1 pica = 12 points).
      kp = d.value * 12;
      break;
    case size_v::unit_type::cm: kp = muldiv(d.value, 72 * 100, 254); break;
    case size_v::unit_type::mm: kp = muldiv(d.value, 72 * 100, 2540); break;
    case size_v::unit_type::expr:
      clear();
      assert(false);
      return;
    default: clear(); return;
    }
    unit    = size_v::unit_type::px;
    d.value = int((int64(kp) * v.pixels_per_inch().y) / 72);
    // MulDiv(kp,v.pixels_per_inch(),72000);
  }


  int size_v::pixels(const size_v &base, size sz, bool vertical,
                     resolution_provider *pv) const {
    auto dips2pixels = [pv](int dips) -> int {
      return int(pv ? pv->pixels_per_dip(size(dips)).x
                    : resolution_provider::desktop().pixels_per_dip(size(dips)).x);
    };
    int kp = 0;
    switch (unit) {
    case size_v::unit_type::em:
      if (base.is_points())
        kp = int((int64(d.value) * base.kpoints()) / 1000);
      else
        return base.pixels() * d.value / 1000;
      break;
    case size_v::unit_type::ex:
      if (base.is_points())
        kp = int((int64(d.value) * base.kpoints()) / 2000);
      else
        return base.pixels() * d.value / 2000;
      break;
    case size_v::unit_type::pr: return ((vertical ? sz.y : sz.x) * d.value) / 100000;
    case size_v::unit_type::pr_width: return (sz.x * d.value) / 100000;
    case size_v::unit_type::pr_height: return (sz.y * d.value) / 100000;
    case size_v::unit_type::pr_view_width: { return pv ? d.value * pv->dimension().x / 100000 : 0; }
    case size_v::unit_type::pr_view_height: { return pv ? d.value * pv->dimension().y / 100000 : 0; }
    case size_v::unit_type::pr_view_max: { int tbase = max(pv->dimension().x, pv->dimension().y); return pv ? d.value * tbase / 100000 : 0; }
    case size_v::unit_type::pr_view_min: { int tbase = min(pv->dimension().x, pv->dimension().y); return pv ? d.value * tbase / 100000 : 0; }

    case size_v::unit_type::sp:
      return 0;
      // assert(false);
      // break;
    case size_v::unit_type::rs: assert(false); break;
    case size_v::unit_type::as:
      switch (d.value) {
      case size_v::special_values::$medium: // medium
        return dips2pixels(2);
      case size_v::special_values::$thin: // thin
        return dips2pixels(1);
      case size_v::special_values::$thick: // thick
        return dips2pixels(3);
      // case size_v::special_values::$auto:
      // case size_v::special_values::$none:
      default:
        if(pv)
          return pv->get_window_metrics(value::length_special_values(d.value));
        else
          return resolution_provider::desktop().get_window_metrics(value::length_special_values(d.value));
      }
      break;
    case size_v::unit_type::in: kp = d.value * 72; break;
    case size_v::unit_type::pt: // Points (1 point = 1/72 inches).
      kp = d.value;
      break;
    case size_v::unit_type::ppx: return i1000(d.value);
    case size_v::unit_type::px: if(!pv || !pv->px_as_dip()) return i1000(d.value);
    case size_v::unit_type::dip: kp = muldiv(d.value, 72, 96); break;
    case size_v::unit_type::pc: // Picas (1 pica = 12 points).
      kp = d.value * 12;
      break;
    case size_v::unit_type::cm: kp = muldiv(d.value, 72 * 100, 254); break;
    case size_v::unit_type::mm: kp = muldiv(d.value, 72 * 100, 2540); break;
    case size_v::unit_type::expr: assert(0); return 0;
    default: return 0;
    }
    return muldiv(kp,
                  pv ? pv->pixels_per_inch().y
                     : resolution_provider::desktop().pixels_per_inch().y,
                  72000);
  }

  // int known_wisth_of_parent(view& v, element *b);
  // int known_height_of_parent(view& v, element *b);

  /*float pixels::width_f() {
    if (s.is_any_percent()) {
      if (b->flags.intrinsic_width_processing) return 0;
      size     sz;
      element *bp = b->parent;
      if (s.is_percent_of_view())
        sz = v.dimension();
      else if (bp) {
        // sz = bp->flags.intrinsic_width_processing?
        // size(0,0):bp->client_rect(v).size();
        sz.x = known_width_of_parent(v, b);
        sz.y = known_height_of_parent(v, b);
      } else
        sz = v.dimension();

      int base = 0;
      switch (s.unit) {
      case size_v::unit_type::pr_view_width: v.has_view_relative_units = true;
      case size_v::unit_type::pr_width: base = sz.x; break;
      case size_v::unit_type::pr_view_height: v.has_view_relative_units = true;
      case size_v::unit_type::pr_height: base = sz.y; break;
      case size_v::unit_type::pr_view_min:
        base = min(sz.x, sz.y);
        v.has_view_relative_units = true;
        break;
      case size_v::unit_type::pr_view_max:
        base = max(sz.x, sz.y);
        v.has_view_relative_units = true;
        break;
      case size_v::unit_type::expr: 
        return width_f(float(dim.x));
      default: 
        return resolve_percents_width(s.d.value / 1000.0f);
      }
      return (base * s.d.value) / 100000.0f;
    }
    return width_f(0.0f);
  }*/

  float pixels::resolve_percents_width(float percent) {
    float base;
    if (dim.x)
      base = float(dim.x);
    else if (!b->parent)
      base = float(v.screen_workarea().width());
    else 
      base = float(known_width_of_parent(v, b));
    return (base * percent) / 100.0f;
  }

  float pixels::resolve_percents_height(float percent) {
    float base;
    if (dim.y)
      base = float(dim.y);
    else if (!b->parent)
      base = float(v.screen_workarea().height());
    else
      base = float(known_height_of_parent(v, b));
    return (base * percent) / 100.0f;
  }

  float image_position_pixels::resolve_percents_width(float percent)
  {
    float xi = (im_dim.x * percent) / 100.0f;
    float xw = (dim.x * percent) / 100.0f;
    return xw - xi;
  }

  float image_position_pixels::resolve_percents_height(float percent)
  {
    float yi = (im_dim.y * percent) / 100.0f;
    float yw = (dim.y * percent) / 100.0f;
    return yw - yi;
  }
    
  /*float pixels::height_f() {
    if (s.is_any_percent()) {
      size     sz;
      element *bp = b->parent;
      if (s.is_percent_of_view()) sz = v.dimension();
      if (bp)
      {
        sz.x = known_width_of_parent(v, b);
        sz.y = known_height_of_parent(v, b);
      } else
        sz = v.dimension();
      int base = 0;
      switch (s.unit) {
      case size_v::unit_type::pr_view_width: v.has_view_relative_units = true;
      case size_v::unit_type::pr_width: base = sz.x; break;
      case size_v::unit_type::pr_view_height: v.has_view_relative_units = true;
      case size_v::unit_type::pr_height: base = sz.y; break;
      case size_v::unit_type::pr_view_min:
        base                      = min(sz.x, sz.y);
        v.has_view_relative_units = true;
        break;
      case size_v::unit_type::pr_view_max:
        base                      = max(sz.x, sz.y);
        v.has_view_relative_units = true;
        break;
      case size_v::unit_type::expr: 
        return height_f(float(dim.y));
      default: 
        return resolve_percents_height(s.d.value / 1000.0f);
      }
      //return height_f(base);
      return (base * s.d.value) / 100000.0f;
    }
    return height_f(0.0f);
  }*/

  inline float zoomify(element *b, float val) {
    if (b && b->parent && b->c_style->zoom.is_defined()) {
      float_v pz = b->parent->c_style->zoom;
      if (pz.is_defined()) val *= pz.val(1.0f);
    }
    return val;
  }

  float pixels::width_f() {

    auto dips2pixels = [this](float dips) {
      return float(v.pixels_per_dip(sizef(dips)).x);
    };

    int kp = 0;
    size_v pfsz;

    switch (s.unit) {
    case size_v::unit_type::ch:
      return (v.get_font(b->get_style(v))->ch() * s.d.value) / 1000;
    case size_v::unit_type::em:
      pfsz = b->get_style(v)->font_size;
      if(pfsz.is_points())
        kp = int((int64(s.d.value) * pfsz.kpoints()) / 1000);
      else
        kp = 0;
      break;
    case size_v::unit_type::rem:
      if(document* pd = b->doc()) 
        pfsz = pd->get_style(v)->font_size;
      else
        pfsz = b->get_style(v)->font_size;
      if (pfsz.is_points())
        kp = int((int64(s.d.value) * pfsz.kpoints()) / 1000);
      else
        kp = 0;
      break;
    case size_v::unit_type::ex:
      pfsz = b->get_style(v)->font_size;
      if(pfsz.is_points())
        kp = int((int64(s.d.value) * pfsz.kpoints()) / 2000);
      else
        kp = 0;
      break;
    case size_v::unit_type::pr:
      return resolve_percents_width(s.d.value / 1000.0f);
    case size_v::unit_type::pr_width:
    {
      if (b->flags.intrinsic_width_processing) return 0;
      float base = float(b->dim().x);
      return float((base * s.d.value) / 100000.0f);
    }
    case size_v::unit_type::pr_height: {
      if (b->flags.intrinsic_width_processing) return 0;
      float base = float(b->dim().y);
      return float((base * s.d.value) / 100000.0f);
    }
    case size_v::unit_type::pr_view_width: {
      v.has_view_relative_units = true;
      if (b->flags.intrinsic_width_processing) return 0;
      float base = float(v.dimension().x);
      return float((base * s.d.value) / 100000.0f);
    }
    case size_v::unit_type::pr_view_height: {
      v.has_view_relative_units = true;
      if (b->flags.intrinsic_width_processing) return 0;
      float base = float(v.dimension().y);
      return float((base * s.d.value) / 100000.0f);
    }
    case size_v::unit_type::pr_view_min:
    {
      v.has_view_relative_units = true;
      if (b->flags.intrinsic_width_processing) return 0;
      auto dim = v.dimension();
      float base = float(min(dim.x, dim.y));
      return float((base * s.d.value) / 100000.0f);
    }
    case size_v::unit_type::pr_view_max: {
      v.has_view_relative_units = true;
      if (b->flags.intrinsic_width_processing) return 0;
      auto dim = v.dimension();
      float base = float(max(dim.x, dim.y));
      return float((base * s.d.value) / 100000.0f);
    }
    case size_v::unit_type::sp:
      return 0;
      // assert(false);
      // break;
    case size_v::unit_type::rs: assert(false); break;
    case size_v::unit_type::as:
      switch (s.d.value) {
      case size_v::special_values::$medium: // medium
        return zoomify(b, dips2pixels(2));
      case size_v::special_values::$thin: // thin
        return zoomify(b, dips2pixels(1));
      case size_v::special_values::$thick: // thick
        return zoomify(b, dips2pixels(3));
      case size_v::special_values::$auto:
        // return (float)b->ldata->dim_min.x;//b->min_width(v);
        return 0;
      case size_v::special_values::$min_content:
        return (float)b->ldata->dim_min.x; // min_width(v);
      case size_v::special_values::$max_content:
        return (float)b->ldata->dim_max.x; // b->max_width(v);
      // case size_v::special_values::$none:
      default:
        return zoomify(b, ((float)v.get_window_metrics(value::length_special_values(s.d.value))));
      }
      break;
    case size_v::unit_type::nm: return zoomify(b, /*dips2pixels(*/ s.number(0.0f) /*)*/);
    case size_v::unit_type::in: kp = s.d.value * 72; break;
    case size_v::unit_type::pt: // Points (1 point = 1/72 inches).
      kp = s.d.value;
      break;
    case size_v::unit_type::ppx: return f1000(s.d.value);
    case size_v::unit_type::px: if(!v.px_as_dip())return f1000(s.d.value); // else fall through                                
    case size_v::unit_type::dip: return zoomify(b, dips2pixels(f1000(s.d.value)));
      //kp = muldiv(s.d.value, 72, 96); break;
    case size_v::unit_type::pc: // Picas (1 pica = 12 points).
      kp = s.d.value * 12;
      break;
    case size_v::unit_type::cm: kp = muldiv(s.d.value, 72 * 100, 254); break;
    case size_v::unit_type::mm: kp = muldiv(s.d.value, 72 * 100, 2540); break;
    case size_v::unit_type::expr: return (float)b->eval_calc(s.d.expr, true, *this).get(0);
    default: return 0;
    }
    return zoomify(b, (kp / 72000.f) * v.pixels_per_inch().x);
  }

  float pixels::height_f() {
    auto dips2pixels = [this](float dips) {
      return float(v.pixels_per_dip(sizef(dips)).y);
    };
    int kp = 0;
    size_v pfsz;
    switch (s.unit) {
    case size_v::unit_type::em:
      pfsz = b->get_style(v)->font_size;
      if (pfsz.is_points())
        kp = int((int64(s.d.value) * pfsz.kpoints()) / 1000);
      else
        kp = 0;
      break;
    case size_v::unit_type::ch:
      return (v.get_font(b->get_style(v))->ch() * s.d.value) / 1000;
    case size_v::unit_type::rem:
      if (document* pd = b->doc())
        pfsz = pd->get_style(v)->font_size;
      else
        pfsz = b->get_style(v)->font_size;
      if (pfsz.is_points())
        kp = int((int64(s.d.value) * pfsz.kpoints()) / 1000);
      else
        kp = 0;
      break;
    case size_v::unit_type::ex:
      kp = int((int64(s.d.value) * b->get_style(v)->font_size.kpoints()) / 2000);
      break;
    case size_v::unit_type::pr:
      if (b->flags.intrinsic_width_processing) return 0;
      //return float((base * s.d.value) / 100000.0f);
      return resolve_percents_height(s.d.value / 1000.0f);
    case size_v::unit_type::pr_width: {
      if (b->flags.intrinsic_width_processing) return 0;
      float base = float(b->ldata->dim.x);
      return float((base * s.d.value) / 100000.0f);
    }
    case size_v::unit_type::pr_height: {
      if (b->c_style->height.is_fixed()) {
        size_v height = b->c_style->height;
        float base = zoomify(b, (float)height.pixels(b->c_style->font_size, b->ldata->dim, true, &v));
        return float((base * s.d.value) / 100000.0f);
      }
      else if (dim.y > 0.0f) {
        return float((dim.y * s.d.value) / 100000.0f);
      }
      else {
        return (float)b->ldata->dim_max.y;
      }
      return 0;
    }
    case size_v::unit_type::pr_view_width: {
      v.has_view_relative_units = true;
      if (b->flags.intrinsic_width_processing) return 0;
      float base = float(v.dimension().x);
      return float((base * s.d.value) / 100000.0f);
    }
    case size_v::unit_type::pr_view_height: {
      v.has_view_relative_units = true;
      if (b->flags.intrinsic_width_processing) return 0;
      float base = float(v.dimension().y);
      return float((base * s.d.value) / 100000.0f);
    }
    case size_v::unit_type::pr_view_min: {
      v.has_view_relative_units = true;
      if (b->flags.intrinsic_width_processing) return 0;
      auto dim = v.dimension();
      float base = float(min(dim.x, dim.y));
      return float((base * s.d.value) / 100000.0f);
    }
    case size_v::unit_type::pr_view_max: {
      v.has_view_relative_units = true;
      if (b->flags.intrinsic_width_processing) return 0;
      auto dim = v.dimension();
      float base = float(max(dim.x, dim.y));
      return float((base * s.d.value) / 100000.0f);
    }
    case size_v::unit_type::sp:
      return 0;
      // assert(false);
      // break;
    case size_v::unit_type::rs: assert(false); break;
    case size_v::unit_type::as:
      switch (s.d.value) {
      case size_v::special_values::$medium: // medium
        return zoomify(b, dips2pixels(2));
      case size_v::special_values::$thin: // thin
        return zoomify(b, dips2pixels(1));
      case size_v::special_values::$thick: // thick
        return zoomify(b, dips2pixels(3));

      case size_v::special_values::$auto: return 0;
      case size_v::special_values::$min_content:
        return (float)b->ldata->dim_min.y; // b->min_height();
      case size_v::special_values::$max_content:
        return (float)b->ldata->dim_max.y; // b->max_height();
      // case size_v::special_values::$none:
      default:
        return zoomify(b, ((float)v.get_window_metrics(value::length_special_values(s.d.value))));
      }
      break;
    case size_v::unit_type::nm: return zoomify(b, /*dips2pixels(*/ s.number(0.0f) /*)*/);
    case size_v::unit_type::in: kp = s.d.value * 72; break;
    case size_v::unit_type::pt: // Points (1 point = 1/72 inches).
      kp = s.d.value;
      break;
    case size_v::unit_type::ppx: return f1000(s.d.value); // else fall through                                
    case size_v::unit_type::px: if (!v.px_as_dip()) return f1000(s.d.value); // else fall through                                
    case size_v::unit_type::dip: return zoomify(b, dips2pixels(f1000(s.d.value)));
      //kp = muldiv(s.d.value, 72, 96); break;
    case size_v::unit_type::pc: // Picas (1 pica = 12 points).
      kp = s.d.value * 12;
      break;
    case size_v::unit_type::cm: kp = muldiv(s.d.value, 72 * 100, 254); break;
    case size_v::unit_type::mm: kp = muldiv(s.d.value, 72 * 100, 2540); break;
    case size_v::unit_type::expr: return (float)b->eval_calc(s.d.expr, false,*this).get(0);
    default:
      return 0;
    }
    return zoomify(b, (kp / 72000.f) * v.pixels_per_inch().y);
  }

  float dips::width_f(){
    float t = pixels::width_f();
    return v.ppx_to_dip(sizef(t)).x;
  }
  float dips::height_f() {
    float t = pixels::height_f();
    return v.ppx_to_dip(sizef(t)).y;
  }


  /*void size_v::pixels_n_spring_w(const element* b, int base_size, int& pix,
    int& spr) const
    {
      pix = 0;
      spr = 0;
      if(unit == sp)
         spr = d.value;
      else
         pix = pixels_width(b,base_size);
    }*/
  void size_v::pixels_n_spring_w(view &v, element *b, int base_size, int &pix, int &spr) const {
    pix      = 0;
    spr      = 0;
    if (is_defined()) {
      if (unit == size_v::unit_type::sp)
        spr = d.value;
      else
        pix = html::pixels(v, b, *this, base_size).width();
#ifdef FLEX_PLUS
      pref_pix = (int)preferred_pixels_width(v, b, base_size);
#endif
    }
  }

  /*void size_v::pixels_n_spring_h(const element* b, int base_size, int& pix,
    int& spr) const
    {
      pix = 0;
      spr = 0;
      if(unit == sp)
         spr = d.value;
      else
         pix = pixels_height(b,base_size);
    }*/

  void size_v::pixels_n_spring_h(view &v, element *b, int base_size, int &pix, int &spr) const {
    pix      = 0;
    spr      = 0;
    if (is_defined()) {
      if (unit == size_v::unit_type::sp)
        spr = d.value;
      else
        pix = html::pixels(v, b, *this, base_size).height();
#ifdef FLEX_PLUS
      pref_pix = (int)preferred_pixels_height(v, b, base_size);
#endif
    }
  }

  ustring to_string(const size_v &v) { return v.to_string(); }

  //argb  morph_color(argb start, argb end, real n);
  float morph_float(float start, float end, real n);

  bool gradient::morph(view& v, element* b, const gradient *start, const gradient *end, real n) {
    slice<color_stop> scs = start->stops();
    slice<color_stop> ecs = end->stops();
    for (int i = 0; i < _stops.size(); ++i) {
      color_stop &cs = _stops[i];
      if (scs[i].position.is_defined())
        cs.position = morph_float(scs[i].position, ecs[i].position, n);
      cs.color = morph_color(v,b,scs[i].color, ecs[i].color, n);
    }
    normalize_stops();
    return false; // no remeasure
  }

  bool linear_gradient::morph(view& v, element* b,
                              const linear_gradient *start,
                              const linear_gradient *end, real n) {
    gradient::morph(v,b,start, end, n);
    pos.x.morph(v,b,start->pos.x, end->pos.x, n);
    pos.y.morph(v, b, start->pos.y, end->pos.y, n);
    dim.x.morph(v, b, start->dim.x, end->dim.x, n);
    dim.y.morph(v, b, start->dim.y, end->dim.y, n);
    if (start->angle.is_defined())
      angle = morph_float(start->angle, end->angle, n);
    else
      angle.clear();
    return false; // no remeasure
  }

  bool radial_gradient::morph(view& v, element *b,
                              const radial_gradient *start,
                              const radial_gradient *end, real n) {
    gradient::morph(v, b,start, end, n);
    center.x.morph(v,b,start->center.x, end->center.x, n);
    center.y.morph(v, b, start->center.y, end->center.y, n);
    radius.x.morph(v, b, start->radius.x, end->radius.x, n);
    radius.y.morph(v, b, start->radius.y, end->radius.y, n);
    focus.x.morph(v, b, start->focus.x, end->focus.x, n);
    focus.y.morph(v, b, start->focus.y, end->focus.y, n);
    return false; // no remeasure
  }


} // namespace html
