#ifndef __html_streams_h__
#define __html_streams_h__

#include "tool/tool.h"
//#include "html-style.h"
//#include "html-style-atts.h"

namespace html {
  using namespace tool;
  // using namespace gool;

  typedef uint get_char_ft(uint encoding, bytes &buf);

  uint get_char_utf8(uint encoding, bytes &buf);
  uint get_char_utf16(uint encoding, bytes &buf);
  uint get_char_latin1(uint encoding, bytes &buf);
  uint get_char_singlebyte(uint encoding, bytes &buf);
  uint get_char_multibyte(uint encoding, bytes &buf);
  uint get_char_acp(uint encoding, bytes &buf);

  class istream : public markup::instream<wchar> {
    typedef markup::instream<wchar> super;
    struct stack_item {
      uint         encoding; // saved
      bytes        buf;      // saved
      string       url;      // saved
      get_char_ft *gcf;      // saved
      uint         line_no;  // saved

      array<byte> data;
      stack_item *prev;

      stack_item(const istream *i)
          : encoding(i->encoding), buf(i->buf), url(i->url), gcf(i->gcf),
            prev(i->stack), line_no(i->line_no) {}

      stack_item *restore(istream *i);
    };
    uint         encoding;
    bytes        input;
    bytes        buf;
    get_char_ft *gcf; // current get_char function
    wchar        next_utf16;
    stack_item * stack;
    array<wchar> processed_chars;
    bool         produce_chars;

    wchar fetch_char();
    void  init_encoding();

  public:
    istream(bytes text, const string &url, bool store_chars)
        : super(url), buf(text), input(text), next_utf16(0)
          //      , line_no(1)
          ,
          stack(0), produce_chars(store_chars) {
      init_encoding();
    }

    istream(chars text, const string &url, bool store_chars)
        : super(url), buf(bytes((const byte *)text.start, text.length)),
          next_utf16(0)
          //      , line_no(1)
          ,
          stack(0), produce_chars(store_chars) {
      init_encoding();
    }

    wchar  get_char();
    void   push(const string &doc_url, const string &url, array<byte> data);
    void   set_encoding(const string &enc_name);
    void   set_utf8_encoding();
    void   set_utf16_encoding();
    wchars parsed_chars() const { return processed_chars(); }

    void rewind();
  };

  struct ostream {
    string url;
    bool   xlate;
    ostream() : xlate(true) {}
    virtual void write(chars)  = 0;
    virtual void write(wchars) = 0;
    virtual void xwrite(wchars);

    virtual uint_ptr position() { return 0; }

    ostream &operator<<(chars str) {
      write(str);
      return *this;
    }
    ostream &operator<<(wchars str) {
      xlate ? xwrite(str) : write(str);
      return *this;
    }
    ostream &operator<<(wchar c) {
      xlate ? xwrite(wchars(c)) : write(wchars(c));
      return *this;
    }
  };

  struct ostream_w : public ostream {
    ostream_w() {}
    array<wchar>     data;
    virtual uint_ptr position() { return data.length(); }
    virtual void     write(chars a) {
      while (a.length)
        data.push(a++);
    }
    virtual void write(wchars w) { data.push(w); }
  };

  struct ostream_8 : public ostream {
    ostream_8() {}
    array<byte>      data;
    virtual uint_ptr position() { return data.length(); }
    virtual void     write(chars a) { while (a.length) data.push(a++); }
    virtual void     write(wchars w) { u8::from_utf16(w, data); }
  };

  inline void istream::push(const string &doc_url, const string &url,
                            array<byte> data) {
    if (doc_url == url) {
      string msg = string::format(
          "<p style='color:red'>ERROR: cyclic INCLUDE of url %s</p>",
          url.c_str());
      data.clear();
      data.push((const byte *)msg.c_str(), msg.length());
    } else
      for (stack_item *pi = stack; pi; pi = pi->prev) {
        if (pi->url == url) {
          string msg = string::format(
              "<p style='color:red'>ERROR: cyclic INCLUDE of url %s</p>",
              url.c_str());
          data.clear();
          data.push((const byte *)msg.c_str(), msg.length());
          break;
        }
      }
    stack = new stack_item(this);
    stack->data.transfer_from(data);
    this->url     = url;
    this->line_no = 1;
    buf           = stack->data();
    if (buf.starts_with(BYTES(UTF8_BOM))) {
      buf.prune(BYTES(UTF8_BOM).length);
      encoding = 65001;
      gcf      = &get_char_utf8;
    } else if (buf.starts_with(BYTES(UTF16LE_BOM))) {
      buf.prune(BYTES(UTF16LE_BOM).length);
      encoding = 65002;
      gcf = &get_char_utf16;
    }
  }

  inline void istream::init_encoding() {
    if (buf.length == 0)
      return;
    if (buf.starts_with(BYTES(UTF8_BOM))) {
      buf.prune(BYTES(UTF8_BOM).length);
      encoding = 65001;
      gcf      = &get_char_utf8;
    } else if (buf.starts_with(BYTES(UTF16LE_BOM))) {
      buf.prune(BYTES(UTF16LE_BOM).length);
      encoding = 65002;
      gcf = &get_char_utf16;
    } else if (buf[0] == 0 && buf[1] != 0) {
     encoding = 65002;
     gcf = &get_char_utf16;
   }
   else
    {
      encoding = 0; // a.k.a. CP_ACP
      gcf      = &get_char_multibyte;
    }
  }

  inline void istream::set_utf8_encoding() {
    encoding = 65001;
    gcf      = &get_char_utf8;
  }
  inline void istream::set_utf16_encoding() {
    encoding = 65002;
    gcf      = &get_char_utf16;
  }

  inline void istream::set_encoding(const string &name) {
    // assert(encoding == 0);
    if (encoding) return; // already set
#ifdef WINDOWS
    if (encoding == 65001) // UTF8 already recognized by initial BOM!
    {
      // assert(name.equals("utf-8"));
      return;
    }
    encoding = get_lang_id(name);
    //#if !defined(PLATFORM_WINCE)
    //    SetThreadLocale(_code_page);
    //#endif
    if (encoding == 65001)
      gcf = get_char_utf8;
    else if (encoding == 65002)
      gcf = get_char_utf16;
    else {
      CPINFO CPInfo;
      GetCPInfo(encoding, &CPInfo);
      if (CPInfo.MaxCharSize == 1)
        gcf = get_char_singlebyte;
      else
        gcf = get_char_multibyte;
    }
#else
    encoding = get_lang_id(name);
    if (encoding == 65001)
      gcf = get_char_utf8;
    else if (encoding == 65002)
      gcf = get_char_utf16;
    else
      gcf = get_char_utf8;
#endif
  }

  inline wchar istream::fetch_char() {
    if (next_utf16) {
      wchar t    = next_utf16;
      next_utf16 = 0;
      return t;
    }

    if (buf.length == 0) {
      if (stack) {
        delete stack->restore(this);
        return fetch_char();
      } else
        return 0;
    }

    uint wc = (*gcf)(encoding, buf);
    if (wc > 0xffff) {
      wchar lead                = wchar(0xD7C0 + (wc >> 10));
      /*wchar tail*/ next_utf16 = 0xDC00 | (wc & 0x3FF);
      return lead;
    }
    if (wc == '\r') {
      wchar next = fetch_char();
      if (next == '\n') 
        return next; 
      else {
        next_utf16 = next;
        return '\r';
      }
    } else if (wc == '\n') 
      ++line_no;
    return wchar(wc);
  }

  inline wchar istream::get_char() {
    wchar wc = fetch_char();
    if (wc && produce_chars) processed_chars.push(wc);
    return wc;
  }

  inline istream::stack_item *istream::stack_item::restore(istream *i) {
    i->encoding = encoding;
    i->buf      = buf;
    i->url      = url;
    i->gcf      = gcf;
    i->stack    = prev;
    i->line_no  = line_no;
    return this;
  }

  inline void istream::rewind() {
    while (stack)
      delete stack->restore(this);
    buf        = input;
    next_utf16 = 0;
    line_no    = 1;
    init_encoding();
  }

} // namespace html

#endif