#ifndef __html_spring_h__
#define __html_spring_h__

#include "tool/tool.h"

#include "kiwi/kiwi.h"

// subset of "Cassowary constraint solving algorithm" for simple flex cases.

namespace html {

  using namespace tool;

  struct flex_value {
    int px; // min
    int px_max;
    int flex; // 1000 == 1fx
    int content_px_max;
    flex_value() : px(0), px_max(0), flex(0), content_px_max(0) {}

    inline void accum(int px_, int flex_) {
      if (px_ > px) px = px_;
      if (flex_ > flex) flex = flex_;
    }
    inline void accum(int px_, int flex_, int px_max_) {
      if (px_ > px) px = px_;
      if (flex_ > flex) flex = flex_;
      if (px_max_ > px_max) px_max = px_max_;
    }
    inline void accum(const flex_value &v) {
      accum(v.px, v.flex, v.px_max);
    }
    inline void clear() {
      px      = 0;
      px_max  = 0;
      flex    = 0;
    }

    inline int_v vmax() const { return px_max ? int_v(px_max) : int_v(); }
  };

#if 0
  namespace spring {
    enum align {
      ALIGN_START  = 1, // shall match align_left and valign_top
      ALIGN_CENTER = 2, // shall match align_center and valign_middle
      ALIGN_END    = 3, // shall match align_right and valign_bottom
    };

    struct item {
      // if the item is not flexible then it should have vmin defined - others
      // are zeroes in this case.
      int   vmin;   // min value.
      int_v vmax;   // max value or zero.
      int   v;      // final, computed value.
      int   p;      // weight, usually 1* is 100 here.
      //int   pref_v; // preferred value.
      //int   accum;  // generational accumulator.
    };

    template <
        typename IC,
        typename ITEM = spring::item> 
                  //  IC is a collection of items, shall have the following
                  //  methods: void  IC::add_item(const ITEM& it); item*
                  //  IC::head(); - a.k.a begin() item* IC::tail(); - a.k.a
                  //  end()

                  // intended usage - CRTP, see:
                  // http://en.wikipedia.org/wiki/Curiously_recurring_template_pattern
                  // and the spring_calc class below.
                  struct engine {
      int ptotal; // total flexes in distribution
      int vmin;   // sum of mins of elements - min-intrinsic of the whole set -
                  // except of flexes
      int vtotalmin; // sum of mins of elements - min-intrinsic of the whole set
      //int vtotalpreferred; // sum of all preferred values
      // bool        has_max_constraints;  // true if max constraint is defined
      // on any item
      int freespace; // free space to distribute,
                     // freespace is an output of the algorithm:
                     // after the calc() and if max100 cap is used it contains
                     // undistributed space left for e.g.
                     // vertical/horizontal-align
      int vtotalmax;
      int pmax;
      int pmin;

      engine() { clear(); }

      void clear() {
        ptotal          = 0;
        vmin            = 0;
        vtotalmin       = 0;
        freespace       = 0;
        //vtotalpreferred = 0;
        vtotalmax       = 0;
        pmax            = 0;
        pmin            = limits<int>::max_value();
      }

      // add an item to the distribution (flexible or not)
      // param: valmin, min width of the element
      // param: valmax, max width of the element, 0 if max width is not defined
      // param: percent, weight, flex value, normalized to 100 if calc(max100)
      // is used param: preferred, preferred value - that strange thing used in
      // XUL

      void add(int valmin, int_v valmax = int_v(), int percent = 0/*, int preferred = 0*/) {
        ITEM i;

        // if(valmax.is_defined() && valmax == 0)
        //  valmax.clear();
        i.v = i.vmin = valmin;
        if (valmax.is_defined()) {
          if (valmax < valmin)
            i.vmax = valmin;
          else
            i.vmax = valmax;
          vtotalmax += i.vmax;
        }
        i.p = percent;
        if (i.p) {
          if (i.p > pmax) pmax = i.p;
          if (i.p < pmin) pmin = i.p;
        }
        ptotal += percent;
        //i.pref_v = preferred;
        //vtotalpreferred += preferred;
        if (percent == 0) vmin += valmin;
        vtotalmin += valmin;
        static_cast<IC *>(this)->add_item(i);
      }

      void add(const flex_value &fv) {
        add(fv.px, fv.px_max ? int_v(fv.px_max) : int_v(), fv.flex);
      }

      inline int val(int i) {
        ITEM *start = static_cast<IC *>(this)->head();
        return (start + i)->v;
      }

      // param: total, total width [of container] to distribute.
      // param: max1000, flag - true - do 100%% normalization that means:
      //                              if sum of weights is less than 1000 then
      //                              some free space will be left undistributed
      // returns: actual distributed content width. if max1000 cap is used then
      // it can be less than 'total' after the call all item::v fields will have
      // computed values - result of distribution.

      inline int calc(int total, bool max1000 = true) {
        freespace = total - vmin; // free space to distribute
        if (freespace <= 0 || ptotal == 0) return vmin;

        ITEM *start = static_cast<IC *>(this)->head();
        ITEM *i = start;
        ITEM *end = static_cast<IC *>(this)->tail();

        //int ptotalmin = ptotal;

        if (ptotal == 0) // no flexes, nothing to do
          return max(0, total - freespace);

        if (max1000) {
          if (ptotal < 1000 && freespace >= 0) ptotal = 1000; // key point
        }

        array<ITEM *> items;
        for (i = start; i < end; ++i)
          if (i->p) items.push(i);

        // sort the list so most "valuable" (potentially wide) items go first (as we will scan them last)
        // the algorithm below is sensitive to scan order
        sort(items.head(), items.length(), [](ITEM* a, ITEM* b)->bool {
          if (a->p > b->p) return true;
          if (a->vmax.val(32000) > b->vmax.val(32000)) return true;
          if (a->vmin > b->vmin) return true;
          return false;
        });

        // check max constraints  
        if (ptotal > 0)
          for (int n = items.last_index(); n >= 0; --n)
          {
            i = items[n];
            if (i->p == 0) continue; // skip elements having fixed size

            int v = (freespace * i->p) / ptotal; // v, value - width candidate
            if (i->vmax.is_defined() && (v > i->vmax)) {
              // maximum reached, exclude from distribution list
              i->v = i->vmax;
              ptotal -= i->p;
              i->p = 0;
              freespace -= i->v;
            }
            if (ptotal <= 0)
              break; // done
          }

        // check min constraints
        if (ptotal > 0)
          for (int n = items.last_index(); n >= 0; --n) {

            i = items[n];
            if (i->p == 0) continue; // skip elements having fixed size

            int v = (freespace * i->p) / ptotal;
            if (v < i->vmin) {
              // minimum reached, exclude from distribution list
              i->v = i->vmin;
              ptotal -= i->p;
              i->p = 0;
              freespace -= i->v;
            }

            if (ptotal <= 0)
              break;
          }

        if (ptotal > 0)
          for (int n = items.last_index(); n >= 0; --n) {

            i = items[n];
            if (i->p != 0)
            {
              int v = (freespace * i->p) / ptotal; // v, value - width candidate
              i->v = max(v, i->vmin);
              if (i->vmax.is_defined()) i->v = min(i->v, i->vmax.val(0));
              ptotal -= i->p;
              freespace -= i->v;
            }

            if (ptotal <= 0)
              break;
          }
        return max(0, total - freespace);
      }
    };

  } // namespace spring

  struct spring_calc : public spring::engine<spring_calc> {
    typedef spring::engine<spring_calc> super;

    array<spring::item> items;

    spring_calc(int num_items_max) {
      items.size(num_items_max);
      items.size(0);
    }
    void clear() {
      items.clear();
      super::clear();
    }

    // spring::engine traits:
    void          add_item(const spring::item &it) { items.push(it); }
    spring::item *head() { return &items[0]; }
    spring::item *tail() { return &items[0] + items.size(); }
  };

  struct spring7 : public spring::engine<spring7> {
    typedef spring::engine<spring7> super;
    spring::item                    items[7];
    int                             counter;

    spring7() : counter(0) {}
    void          add_item(const spring::item &it) { items[counter++] = it; }
    spring::item *head() { return &items[0]; }
    spring::item *tail() { return &items[counter]; }
    void          clear() {
      counter = 0;
      super::clear();
    }
  }; 
#endif

  // spring engine with constrains with cassowar solver
  struct spring_board {

    enum align {
      ALIGN_START = 1, // shall match align_left and valign_top
      ALIGN_CENTER = 2, // shall match align_center and valign_middle
      ALIGN_END = 3, // shall match align_right and valign_bottom
    };


    struct body_item {
      int b_min;  // body min width
      int b_max;  // body max width (specified)
      int b_cmax; // body content max width
      int b_val;  // body width (computed)
      int b_flex; // body flex

      kiwi::Variable b_var;
      body_item()
          : b_min(0), b_max(limits<int>::max_value()), b_val(0), b_flex(0),
            b_cmax(0) {}
    };

    struct item : public body_item {
      int m_min;  // left margin min width
      int m_flex; // left margin flex
      int m_val;  // margin computed value

      int pos; // calculated layout pos;

      kiwi::Variable m_var;
      item() : m_min(0), m_flex(0), pos(0) {}
    };

    struct span_item : public body_item {
      int first; // first span column
      int last;  // last span column
      span_item() : first(0), last(0) {}
    };

    array<item>      items;
    array<span_item> spans;

    int total_min;
    int total_max;
    int total_flex;

    int           solved_width;
    align         solved_align;
    int           solved_offset;

    spring_board()
        : total_min(0), total_flex(0), total_max(0), solved_width(0) {}
    void clear() {
      items.clear();
      spans.clear();
      total_max    = 0;
      total_min    = 0;
      total_flex   = 0;
      solved_width = 0;
    }

    void restart() {
      items.set_all_to(item());
      spans.clear();
      total_max    = 0;
      total_min    = 0;
      total_flex   = 0;
      solved_width = 0;
    }

    // returns number of real columns
    int  size() const { return items.size() ? items.size() - 1 : 0; }
    void size(int n) {
      if (items.size() - 1 < n) items.size(n + 1);
    }

    // spring::engine traits:
    void accum(int at, int bmin, int_v bmax, int bflex, int contentmax,
               int lmin, int lflex, int rmin, int rflex) {
      if (at >= items.size())
        items.size(
            at +
            2); // rightmost column is dummy one - only (left) margin is used

      item &it = items[at];

      // body
      if (bmin > it.b_min) {
        total_min += (bmin - it.b_min);
        it.b_val = it.b_min = bmin;
      }

      if (bmax.is_defined()) {
        if (bmax < it.b_max) it.b_max = bmax;
        if (bmax < contentmax) contentmax = bmax;
      }

      if (it.b_cmax < contentmax) {
        total_max += contentmax - it.b_cmax;
        it.b_cmax = contentmax;
      }

      if (bflex > it.b_flex) {
        it.b_flex = bflex;
        total_flex += bflex;
      }
      // left margin
      if (lmin > it.m_min) {
        total_max += (lmin - it.m_min);
        total_min += (lmin - it.m_min);
        it.m_min = lmin;
      }

      if (lflex > it.m_flex) {
        it.m_flex = lflex;
        total_flex += bflex;
      }
      // right margin
      item &it_right = items[at + 1];
      if (rmin > it_right.m_min) {
        total_max += (rmin - it_right.m_min);
        total_min += (rmin - it_right.m_min);
        it_right.m_min = rmin;
      }
      if (rflex > it_right.m_flex) {
        it_right.m_flex = rflex;
        total_flex += bflex;
      }
    }

    void accum(int at, const flex_value &body, const flex_value &ml,
               const flex_value &mr) {
      accum(at, body.px, body.vmax(), body.flex,
            max(body.px, body.content_px_max), ml.px, ml.flex, mr.px, mr.flex);
    }

    void accum_span(int first, int last, int bmin, int_v bmax, int bflex,
                    int contentmax, int lmin, int lflex, int rmin, int rflex) {
      if (last >= items.size())
        items.size(last + 2); // rightmost column is dummy one - only (left) margin is used

#if 0
      item &it_first = items[first];

      // left margin
      if (lmin > it_first.m_min) {
        total_min += (lmin - it_first.m_min);
        it_first.m_min = lmin;
      }
      if (lflex > it_first.m_flex) {
        it_first.m_flex = lflex;
        total_flex += lflex;
      }
      // right margin
      item &it_end = items[last + 1];
      if (rmin > it_end.m_min) {
        total_min += (rmin - it_end.m_min);
        it_end.m_min = rmin;
      }
      if (rflex > it_end.m_flex) {
        it_end.m_flex = rflex;
        total_flex += rflex;
      }

      int spanned_min = it_first.b_min;
      int spanned_max = it_first.b_cmax;
      // int spanned_flex =  it_first.b_flex;

      for (int i = first + 1; i <= last; ++i) {
        spanned_min += items[i].b_min + items[i].m_min;
        spanned_max += items[i].b_cmax + items[i].m_min;
        // spanned_flex += items[i].b_flex + items[i].m_flex;
      }

      if (spanned_max < (contentmax + lmin + rmin))
        total_max += (contentmax + lmin + rmin) - spanned_max;

      if (spanned_min < bmin) total_min += (bmin - spanned_min);

      if (spanned_min > bmin && bmax.is_undefined() && bflex == 0)
        return; // nothing to do ?

      span_item sit;
      sit.first = first;
      sit.last  = last;
      sit.b_min = bmin;
      if (bmax.is_defined()) sit.b_max = bmax;
      sit.b_flex = bflex;

      spans.push(sit);
#else 
      if (last >= items.size())
        items.size(last + 2); // rightmost column is dummy one - only (left) margin is used

      item &it_first = items[first];

      // left margin
      if (lmin > it_first.m_min) {
        total_min += (lmin - it_first.m_min);
        it_first.m_min = lmin;
      }
      if (lflex > it_first.m_flex) {
        it_first.m_flex = lflex;
        total_flex += lflex;
      }
      // right margin
      item &it_end = items[last + 1];
      if (rmin > it_end.m_min) {
        total_min += (rmin - it_end.m_min);
        it_end.m_min = rmin;
      }
      if (rflex > it_end.m_flex) {
        it_end.m_flex = rflex;
        total_flex += rflex;
      }

      int spanned_min = it_first.b_min;

      for (int i = first + 1; i <= last; ++i)
        spanned_min += items[i].b_min + items[i].m_min;

      if (spanned_min > bmin && bmax.is_undefined() && bflex == 0)
        return; // nothing to do ?

      int t_min = bmin;
      int t_max = bmax.val(contentmax);

      for (int i = 0; i < first; ++i) {
        t_min += items[i].b_min + items[i].m_min;
        t_max += items[i].b_cmax + items[i].m_min;
      }
      for (int i = last + 1; i < items.size(); ++i) {
        t_min += items[i].b_min + items[i].m_min;
        t_max += items[i].b_cmax + items[i].m_min;
      }

      total_min = max(t_min, total_min);
      total_max = max(t_max, total_max);

      span_item sit;
      sit.first = first;
      sit.last = last;
      sit.b_min = bmin;
      if (bmax.is_defined()) sit.b_max = bmax;
      sit.b_flex = bflex;

      spans.push(sit);

#endif

    }

    void accum_span(int first, int last, const flex_value &body,
                    const flex_value &ml, const flex_value &mr) {
      accum_span(first, last, body.px, body.vmax(), body.flex,
                 max(body.px, body.content_px_max), ml.px, ml.flex, mr.px,
                 mr.flex);
    }

    int span_width(int first, int last) {
      int spanned_min = items[first].b_val;
      for (int i = first + 1; i <= last; ++i)
        spanned_min += items[i].b_val + items[i].m_val;
      return spanned_min;
    }

    // returns offset
    int solve(int width, align align);
  };


#if 1
  // v.12 of initial flex calculator
  namespace flex {
    struct item {
      // if the item is not flexible then it should have vmin defined - others
      // are zeroes in this case.
      int   vmin;   // min value.
      int_v vmax;   // max value or zero.
      int   v;      // final, computed value.
      int   p;      // weight, usually 1* is 100 here.
    };

    struct engine {
      int ptotal; // total flexes in distribution
      int vmin;   // sum of mins of elements - min-intrinsic of the whole set -
                  // except of flexes
      int freespace; // free space to distribute,
                     // freespace is an output of the algorithm:
                     // after the calc() and if max100 cap is used it contains
                     // undistributed space left for e.g.
                     // vertical/horizontal-align
      int vmax;

      array<item> items;

      engine(int_v nitems = int_v()) { clear(); if (nitems.is_defined()) items.reserve(nitems); }

      void clear() {
        items.clear();
        ptotal = 0;
        vmin = 0;
        freespace = 0;
        vmax = 0;
      }

      // add an item to the distribution (flexible or not)
      // param: valmin, min width of the element
      // param: valmax, max width of the element, 0 if max width is not defined
      // param: percent, weight, flex value, normalized to 100 if calc(max100)
      // is used param: preferred, preferred value - that strange thing used in
      // XUL

      void add(int valmin, int_v valmax = int_v(), int percent = 0/*, int preferred = 0*/) {
        item i;

        i.v = i.vmin = valmin;
        if (valmax.is_defined()) {
          if (valmax < valmin)
            i.vmax = valmin;
          else
            i.vmax = valmax;
          vmax += i.vmax;
        }
        else if (percent)
          vmax += 32000;
        else
          vmax += valmin;
        i.p = percent;
        ptotal += percent;
        vmin += valmin;
        items.push(i);
      }

      void add(const flex_value &fv) {
        add(fv.px, fv.px_max ? int_v(fv.px_max) : int_v(), fv.flex);
      }

      inline int val(int i) {
        return items[i].v;
      }

      int calc(int total, bool max1000cap = true) {

        freespace = total;

        if (ptotal == 0 || total == 0) {
          freespace = total - vmin;
          return vmin;
        }

        bool max_cap_total = false;
        if (max1000cap && (ptotal < 1000)) {
          ptotal = 1000;
          max_cap_total = true;
        }

        array<item*> flexes;

        int  pmin = 0;
        int  pmax = std::numeric_limits<int>::max();
        
        for (int i = 0; i < items.size(); ++i) {
          item& it = items[i];
          it.v = it.vmin;
          if (it.p) {
            flexes.push(&it);
            pmin = max(pmin, it.p);
            pmax = min(pmax, it.p);
          }
          else
            freespace -= it.v;
        }

        if( pmin != pmax ) // items with larger weights shall come first
          tool::sort(flexes.begin(), flexes.length(), [](const item* a, const item* b) -> bool { return a->p > b->p; });

        for (int k = 0; k < flexes.size(); ++k)
        {
          if (ptotal == 0)
            break;
          int tptotal = ptotal;
          int tspace = freespace;
          for (auto item : flexes) {
            if (item->p == 0) continue;
            item->v = (tspace * item->p) / tptotal;
            if (item->vmax.is_defined() && item->v >= item->vmax)
            {
              item->v = item->vmax;
              freespace -= item->v;
              ptotal -= item->p;
              item->p = 0;
              goto NEXT;
            }
            if (item->v <= item->vmin)
            {
              item->v = item->vmin;
              freespace -= item->v;
              ptotal -= item->p;
              item->p = 0;
              goto NEXT;
            }
            tspace -= item->v;
            tptotal -= item->p;
          }
          freespace = tspace;
          ptotal = tptotal;
          if (max_cap_total)
            break; // don't need to iterate further as ptotal < 1000 - freespace was allowed and no mins and maxes
        NEXT:
          continue;
        }
        return total - freespace;
      }
    };

  }

#else 
  namespace flex {

    // cassowary variant of the above - slow. Investigate why: either their implementation is bad or because of iterative nature of simplex method in general. 

    struct item {
      // if the item is not flexible then it should have vmin defined - others
      // are zeroes in this case.
      int   vmin;   // min value.
      int_v vmax;   // max value or zero.
      int   v;      // final, computed value.
      int   p;      // weight, usually 1* is 100 here.
                    //int   pref_v; // preferred value.
                    //int   accum;  // generational accumulator.
      kiwi::Variable var;
    };


      // intended usage - CRTP, see:
      // http://en.wikipedia.org/wiki/Curiously_recurring_template_pattern
      // and the spring_calc class below.
   struct engine {
      int ptotal; // total flexes in distribution
      int vmin;   // sum of mins of elements - min-intrinsic of the whole set -
                  // except of flexes
      int freespace; // free space to distribute,
                     // freespace is an output of the algorithm:
                     // after the calc() and if max100 cap is used it contains
                     // undistributed space left for e.g.
                     // vertical/horizontal-align
      int vmax;
      //int pmax;      // max p in the set

      array<item> items;

      engine(int_v nitems = int_v()) { clear(); if (nitems.is_defined()) items.reserve(nitems); }

      void clear() {
        items.clear();
        ptotal = 0;
        vmin = 0;
        freespace = 0;
        vmax = 0;
        //pmax = 0;
      }

      // add an item to the distribution (flexible or not)
      // param: valmin, min width of the element
      // param: valmax, max width of the element, 0 if max width is not defined
      // param: percent, weight, flex value, normalized to 100 if calc(max100)
      // is used param: preferred, preferred value - that strange thing used in
      // XUL

      void add(int valmin, int_v valmax = int_v(), int percent = 0/*, int preferred = 0*/) {
        item i;

        i.v = i.vmin = valmin;
        if (valmax.is_defined()) {
          if (valmax < valmin)
            i.vmax = valmin;
          else
            i.vmax = valmax;
          vmax += i.vmax;
        }
        else if(percent)
          vmax += 32000;
        else 
          vmax += valmin;
        i.p = percent;
        //if (i.p > pmax) pmax = i.p;
        ptotal += percent;
        vmin += valmin;
        items.push(i);
      }

      void add(const flex_value &fv) {
        add(fv.px, fv.px_max ? int_v(fv.px_max) : int_v(), fv.flex);
      }

      inline int val(int i) {
        return items[i].v;
      }

      // param: total, total width [of container] to distribute.
      // param: max1000, flag - true - do 100%% normalization that means:
      //                              if sum of weights is less than 1000 then
      //                              some free space will be left undistributed
      // returns: actual distributed content width. if max1000 cap is used then
      // it can be less than 'total' after the call all item::v fields will have
      // computed values - result of distribution.

      int calc(int total);
    };

  } // namespace flex
#endif

} // namespace html

#endif
