#ifndef __html_scrollbar_h__
#define __html_scrollbar_h__

#include "tool/tool.h"
#include "gool/gool.h"
#include "html-primitives.h"
#include "html-events.h"
#include "html-style.h"

namespace html {
  using namespace tool;
  using namespace gool;

  class view;
  struct document;
  struct element;

  struct layout_data;

  struct scroll_data {
    point pos;
    size  dim_view;
    rect  content_outline;
  };

  struct scrollbar : public virtual resource {
    friend struct scrollbars;

    enum ORIENTATION { HORIZONTAL, VERTICAL };

    enum part {
      BASE,       //= theme::SCROLLBAR_BASE,
      PLUS,       //= theme::SCROLLBAR_PLUS,
      MINUS,      //= theme::SCROLLBAR_MINUS,
      SLIDER,     //= theme::SCROLLBAR_SLIDER,
      PAGE_MINUS, //= theme::SCROLLBAR_PAGE_MINUS,
      PAGE_PLUS,  //= theme::SCROLLBAR_PAGE_PLUS,
      CORNER,     //= theme::SCROLLBAR_CORNER,
    };
    enum state {
      NORMAL = 1,
      HOVER,
      PRESSED,
      DISABLED,
    };

    scrollbar(bool vertical = false, bool inverse = false)
        : //_layout_valid(false),
          _vertical(vertical), _min(0), _max(100), _page(10), _step(1),
          _tracking(false), _active_element(0), _inverse(inverse),

          _style_tracking(false), _style_active_element(uint(-1)),
          _style_enabled(false),

          _min_pos(), _slider_size(), _max_pos(), _slider_pos(), _min_size(),
          _max_size(), _tracking_offset() {}

    ~scrollbar() {}

    virtual bool on(view &v, element *b, event_mouse &evt);
    virtual bool on_timer_tick(view & /*v*/, element * /*el*/, timer_id id) { return false; }
    virtual void dismiss(view & /*v*/, element * /*b*/) {}

    virtual void do_layout(view &v, element *b);

    // int           desired_width() { return 16; }
    // int           desired_height() { return 16; }

    // set scrollbar values
    virtual void set_ranges(view &v, element *b, int min, int max, int page,
                            int step = 1);

    // ivalue stuff
    virtual int  get_value();
    virtual void set_value(view &v, element *el, int val,
                           bool allow_out_of_bounds = false);

    virtual rect client_place() const { return rect(_place.dimension()); }

    virtual void draw(view &v, graphics *sf, element *b, const rect &r,
                      const rect &corner_r = rect());

    virtual void refresh(view &v, element *b);

    virtual const rect &place() const { return _place; }
    virtual void        place(const rect &r) {
      _place = r;
      //_layout_valid = false;
    }

    // bool layout_valid() const { return _layout_valid; }

    int get_max() { return _max; }
    int get_min() { return _min; }
    int get_page() { return _page; }
    int get_step() { return _step; }

    inline bool enabled() {
      if (_min >= _max) return false;
      if ((_max - _min + 1) <= _page) return false;
      return true;
    }

    bool is_vertical() { return _vertical; }
    virtual bool is_external() const { return false; }

    void vertical(bool v) { _vertical = v; }

    virtual int defined_width(view &v, element *b);
    virtual int defined_height(view &v, element *b);

    // true if this is mobile like scroll position indicator
    virtual bool is_indicator() const { return false; }
    virtual void on_mouse_enters_element(view & /*v*/, element * /*b*/) {}
    virtual void on_mouse_leaves_element(view & /*v*/, element * /*b*/) {}

    virtual uint hit_test(point zpt) const;

    virtual void clear_styling();

  protected:
    bool  _vertical;
    int   _min, _max, _page, _step;
    int_v _val;
    int   _min_pos, _slider_size, _max_pos, _slider_pos;
    int   _min_size, _max_size;
    int   _tracking_offset;
    bool  _tracking;
    uint  _active_element;
    rect  _place;
    bool  _inverse;

    hstyle _style_base;
    hstyle _style_plus;
    hstyle _style_minus;
    hstyle _style_page_plus;
    hstyle _style_page_minus;
    hstyle _style_slider;
    hstyle _style_corner;

    bool _style_tracking;       //
    uint _style_active_element; // snapshot of states when styles were resolved
    bool _style_enabled;        //
    string _style_set_name;     //

    // element*     _parent;

    //virtual uint hit_test(const event_mouse &evt);
    virtual int  pos_by_val(view &v, element *b, int val, int button_size, int csize);
    virtual int  val_by_pos(view &v, element *b, int p, int button_size, int csize);

    virtual state    element_state(uint element);
    virtual ui_state part_state(uint element);

    virtual void update_styles(view &v, element *b);

    virtual int  slider_min_size(view &v, element *b,int defsize);
    virtual int  minus_size(view &v, element *b,int defsize);
    virtual int  plus_size(view &v, element *b,int defsize);

    int position() const;

    virtual void notify(view &v, element *b, /*view::scroll_cmd*/ int cmd, int val, uint part); // notify listener
    virtual bool takes_space() const { return is_external()? false : true; }
  };

  struct scrollbar_indicator : public scrollbar {
    typedef scrollbar super;

    bool hover;
    int  mode_progress = 0; // 0 - compact, 100 - full form
    int  mode_progressing = 0; // -1 - to compact, +1 to full
    scrollbar_indicator(bool vertical, bool inverse)
        : scrollbar(vertical, inverse), hover(false) {}

    int progress(int min, int max);
    
    virtual bool on(view &v, element *b, event_mouse &evt) override;
    virtual bool takes_space() const override { return false; }
    virtual void draw(view &v, graphics *sf, element *b, const rect &r,
                      const rect &corner_r = rect()) override;
    virtual void do_layout(view &v, element *b) override;
    virtual int  minus_size(view &v, element *b, int defsize) override { return 0; }
    virtual int  plus_size(view &v, element *b, int defsize) override { return 0; }
    virtual int  defined_width(view &v, element *b) override;
    virtual int  defined_height(view &v, element *b) override;
    virtual void set_value(view &v, element *el, int val,
                           bool allow_out_of_bounds = false) override;
    virtual bool on_timer_tick(view &v, element *el, timer_id id) override;

    virtual bool is_indicator() const { return true; }
    virtual void on_mouse_enters_element(view &v, element *b);
    virtual void on_mouse_leaves_element(view &v, element *b);
    void         request_expansion(view &v, element *b);
    void         start_expansion(view &v, element *b);
    void         start_collapse(view &v, element *b);

    virtual void refresh(view &v, element *b) override;

  };

  enum SB_MODE { SBM_NONE, SBM_OPTIONAL, SBM_ALWAYS, SBM_INDICATOR, SBM_PER_CSS  };

  struct scrollbars {
    handle<scrollbar> _vsb;
    handle<scrollbar> _hsb;
    point             pos;
    //bool              b->flags.virtual_v_scrollbar = false;
    //bool              b->flags.virtual_h_scrollbar = false;
    bool              layout_valid = false;

    scrollbars() {}
    ~scrollbars() {}

    // auto-created scrollbars
    scrollbar * vsb() const { return _vsb && !_vsb->is_external() ? _vsb : nullptr; }
    scrollbar * hsb() const { return _hsb && !_hsb->is_external() ? _hsb : nullptr; }

    // externally mounted scrollbars
    scrollbar * evsb() const { return _vsb && _vsb->is_external() ? _vsb : nullptr; }
    scrollbar * ehsb() const { return _hsb && _hsb->is_external() ? _hsb : nullptr; }

    inline bool need_to_show(int mi, int ma, int pg) {
      if (mi >= ma) return false;
      if ((ma - mi) <= pg) return false;
      return true;
    }

    bool set_h(view &v, element *b, range rval, int page, SB_MODE mode);
    bool set_h(view &v, element *b, range rval, int page, int pos, SB_MODE mode);

    bool remove_h(view &v, element *b) {
      layout_valid = false;
      if (hsb()) {
        _hsb->dismiss(v, b);
        _hsb          = 0;
        layout_valid = false;
        pos.x        = 0;
#ifdef SCROLL_INDICATOR_ONLY
        return false;
#else
        return true;
#endif
      }
      return false;
    }

    bool set_v(view &v, element *b, range rval, int page, SB_MODE mode);
    bool set_v(view &v, element *b, range rval, int page, int pos, SB_MODE mode);

    bool remove_v(view &v, element *b) {
      if (vsb()) {
        _vsb->dismiss(v, b);
        _vsb          = 0;
        layout_valid = false;
        pos.y        = 0;
#ifdef SCROLL_INDICATOR_ONLY
        return false;
#else
        return true;
#endif
      }
      return false;
    }

    bool visible() const { return _hsb || _vsb; }
    bool has_external_scrollbars() const { return ehsb() || evsb(); }
    bool can_scroll_h(view &v) const { return !_hsb.is_null(); }
    bool can_scroll_v(view &v) const { return !_vsb.is_null(); }
    void replace(view &v, element *b);
    void refresh(view &v, element *b);
    rect client_rect(view &v, element *b) const;
    //rect client_rect(element *b) const;

    bool hit_test(point zpt) const { 
      if (vsb() && vsb()->place().contains(zpt)) return true;
      if (hsb() && hsb()->place().contains(zpt)) return true;
      return false;
    }

    int v_width(view &v, element *b);
    int h_height(view &v, element *b);

    bool on_timer_tick(view &v, element *el, timer_id id) {
      int r = 0;
      if (auto sb = hsb()) { r += int(sb->on_timer_tick(v, el, id)); }
      if (auto sb = vsb()) { r += int(sb->on_timer_tick(v, el, id)); }
      return r != 0;
    }

    // point scroll_pos() const { return point(hsb?hsb->get_value():pos_x,
    // vsb?vsb->get_value():pos_y); }
    point scroll_pos(const element *b) const;
    void  scroll_pos(view &v, element *b, point pt, bool allow_out_of_bounds = false);

    void draw(view &v, graphics *sf, element *b, gool::point p, bool horz, bool vert);

    bool on(view &v, element *b, event_mouse &evt);

    void on_mouse_enters_element(view &v, element *b);
    void on_mouse_leaves_element(view &v, element *b);

    void clear_style() {
      if (auto sb = hsb()) sb->clear_styling();
      if (auto sb = vsb()) sb->clear_styling();
    }

  };

} // namespace html

#endif
