#include "html.h"

namespace html {

  uint pump::max_http_data_length = MAX_HTTP_DATA_LENGTH;

  // multipart/form-data composer, stack only object
  static tool::chars CRLF = CHARS("\r\n");
  static void        append(pump::request *rq, const tool::chars &v) {
    rq->data.push((const byte *)v.start, v.length);
  }
  static void append(pump::request *rq, const tool::bytes &v) {
    rq->data.push(v);
  }

  pump::multipart_composer::multipart_composer(request *prq) : rq(prq) {
    boundary = CHARS("----------sciter-") + tool::unique_id();
    rq->data.clear();
    rq->data_content_type =
      string::format("multipart/form-data; boundary=%s", boundary.c_str());
  }
  pump::multipart_composer::~multipart_composer() {
    append(rq, CHARS("--"));
    append(rq, boundary);
    append(rq, CHARS("--\r\n"));
    append(rq, CHARS("\r\n"));
  }
  void pump::multipart_composer::add(chars name, wchars data) {
    append(rq, CHARS("--"));
    append(rq, boundary);
    append(rq, CHARS("\r\n"));
    append(rq, CHARS("Content-Disposition: form-data; name=\""));
    append(rq, name);
    append(rq, CHARS("\"\r\n"));
    append(rq, CHARS("Content-Type: text/plain; charset=utf-8\r\n"));
    append(rq, CHARS("\r\n"));
    append(rq, u8::cvt(data));
    append(rq, CHARS("\r\n"));
  }
  void pump::multipart_composer::add(chars name, bytes data, chars filename,
    chars mimetype) {
    append(rq, CHARS("--"));
    append(rq, boundary);
    append(rq, CHARS("\r\n"));
    if (mimetype.length == 0) mimetype = CHARS("application/octet-stream");
    append(rq, CHARS("Content-Disposition: form-data; name=\""));
    append(rq, name);
    append(rq, CHARS("\";"));
    append(rq, CHARS(" filename=\""));
    append(rq, filename);
    append(rq, CHARS("\"\r\n"));
    append(rq, CHARS("Content-Type: "));
    append(rq, mimetype);
    append(rq, CHARS("\r\n"));
    append(rq, CHARS("\r\n"));
    append(rq, data);
    append(rq, CHARS("\r\n"));
  }

  uint pump::connection_timeout =
#if defined(HTTP_CONNECTION_TIMEOUT)
    HTTP_CONNECTION_TIMEOUT;
#else 
    10000;
#endif
    
  string pump::_user_agent;

  string pump::get_user_agent() {
    if (_user_agent.is_empty()) {
      _user_agent = 
#if defined(HTTP_USER_AGENT)
      HTTP_USER_AGENT;
#elif defined(SCITERJS)
      string::format("Sciter.JS/%s (%s;%s;%s)", SCITER_VERSION_FULL_STR,
        environment::platform_name(),
        environment::get_os_version_name(),
        environment::get_os_lang().c_str());
#else 
      string::format("sciter %s; %s; www.sciter.com)", SCITER_VERSION_FULL_STR, environment::get_os_version_name());
#endif
    }
    return _user_agent;
  }

  void pump::set_user_agent(string s) {
    _user_agent = s;
  }

  uint pump::https_error_action = 1;

  static const char *VERB_GET = "GET";
  static const char *VERB_POST = "POST";
  static const char *VERB_PUT = "PUT";
  static const char *VERB_DELETE = "DELETE";

  const char *get_http_verb(const pump::request *prq) {
    switch (prq->rq_type) {
    case RQ_GET: return VERB_GET;
    case RQ_POST: return VERB_POST;
    case RQ_PUT: return VERB_PUT;
    case RQ_DELETE: return VERB_DELETE;
    }
    assert(false);
    return VERB_GET;
  }

  // const char*  HTTP_POST_HDR = "Content-Type:
  // application/x-www-form-urlencoded;charset=utf-8";  static const char*
  // POST_BINARY_HDR = "Content-Type: multipart/form-data";  static const char*
  // POST_JSON_HDR = "Content-Type: application/json;charset=utf-8";
  const char *HTTP_ACCEPT_DATA[2] = { "*/*", 0 };

  bool verify_content_type(pump::request *rq, const char *content_type) {
    rq->data_content_type = content_type;
    // array<string> parts;
    // rq->data_content_type.tokens(parts,'/');

    chars data_content_type = rq->data_content_type;

    chars mimetype;
    chars charset = data_content_type.chop(CHARS(";"), mimetype);
    mimetype = trim(mimetype);
    charset = trim(charset);
    // charset=utf-8
    if (charset.like("charset=*"))
      rq->data_content_encoding = charset.tail('=');

    if (mimetype.length == 0) {
      assert(false);
      return true; // unknown?
    }

    switch (rq->data_type) {
    case DATA_STYLE:
    case DATA_HTML:
      return mimetype.like("text/*") ||
        mimetype == CHARS("application/x-zip-compressed") ||
        mimetype == CHARS("application/zip") ||
        mimetype == CHARS("application/xhtml+xml");

    case DATA_IMAGE: return mimetype.like("image/*");
    case DATA_CURSOR:
      return mimetype.like("image/*") || mimetype.like("application/*");
    case DATA_SCRIPT:
      if (mimetype.like("text/*")) return true;
      if (mimetype.like("application/json")) return true;
      if (mimetype.like("application/*script")) return true;
      return false;
    case DATA_RAW_DATA: return true;
    default: break;
    }
    return true;
  }

#if defined(ZIP_SUPPORT)
  handle<cabinet> make_cabinet_and_fetch_root(html::pump::request *rq) {
    array<byte> &d = rq->data;
    index_t      fn_pos = rq->url().last_index_of('#');

    string zip_url;
    string root_url;
    if (fn_pos > 0) {
      zip_url = rq->url().sub(0, fn_pos);
      zip_url += "/";
      root_url = zip_url + rq->url().sub(fn_pos + 1);
    }
    else {
      zip_url = rq->url;
      if (!zip_url.like("*/")) zip_url += "/";
    }

    handle<tool::cabinet> cab = new tool::cabinet();

    if (cab->unzip(d, zip_url) == 0) return handle<cabinet>();

    cab->url = zip_url;

    bytes td;
    bool  is_dir;

    if (root_url.length() && cab->fetch_file(root_url, td)) {
      rq->data = td;
      rq->content_url = root_url;
      return cab;
    }

    static const char *indexes[] = { "index.htm", "index.html", "main.htm", "main.html" };

    for (uint i = 0; i < items_in(indexes); ++i) {
      root_url = zip_url + indexes[i];
      if (cab->fetch(root_url, td, is_dir)) {
        if (!is_dir) {
          rq->data = td;
          rq->content_url = root_url;
          return cab;
        }
      }
    }
    return handle<tool::cabinet>();
  }
#endif

  bool view::postprocess_loaded_data(html::pump::request *rq) {
    if (rq->data_type != DATA_HTML && rq->data_type != DATA_STYLE) return false;
#if defined(ZIP_SUPPORT)
    if (!cabinet::is_zip_data(rq->data())) // it is not a zip file
    {
      // cab.clear();
      return true;
    }
    else // it's a zip file
    {
      handle<cabinet> cab = make_cabinet_and_fetch_root(rq);
      if (cab) {
        zip_cache[cab->url] = cab;
        return true;
      }
    }
#endif
    return false;
  }

  bool view::do_load_data(pump::request *rq)
  {
    // process zip content
#if defined(ZIP_SUPPORT)
    for (int n = 0; n < zip_cache.size(); ++n) {
      string surl = zip_cache(n)->url;
      if (rq->url().starts_with(zip_cache(n)->url())) {
        bytes data;
        rq->ready_flag = true;
        if (zip_cache(n)->fetch_file(rq->url, data)) {
          rq->data = data;
          rq->success_flag = true;
          rq->status = 200;
          on_data_loaded(rq);
          return true;
        }
        else {
          rq->status = 404;
          return true;
        }
      }
    }
#endif

    auto mark_busy = [this, rq]() {
      document *dst_doc = this->doc();
      if (rq->dst) {
        rq->dst->state_on(*this, S_BUSY | S_INCOMPLETE);
        dst_doc = rq->dst->doc();
      }

      switch (rq->data_type) {
      case DATA_STYLE:
        dst_doc->num_styles_requested = dst_doc->num_styles_requested + 1;
      case DATA_IMAGE:
      case DATA_CURSOR:
      case DATA_SCRIPT:
        dst_doc->num_resources_requested = dst_doc->num_resources_requested + 1;
      default: break;
      }
    };

    rq->start(this);

    // give a chance to callback to handle it

    if (callback) {
      if (callback->load_data(this, rq)) {
        rq->success_flag = true;
        // rq->ready_flag = true; - set by on_data_loaded
        on_data_loaded(rq);
        return true;
      }
      else if (rq->external_load) {
        mark_busy();
        return false;
      }
    }

    if (rq->external_load) return false;

    // process it

    if (parent()) {
      return parent()->do_load_data(
        rq); // must be do_load_data() but not load_data() - events?
    }

    url u;
    u.parse(rq->url);

    if (u.is_external()) {
#if defined(HTTP_SUPPORT)
      mark_busy();
      pump::send(rq);
#else
      rq->done_with_failure(uint(-1));
#endif
      return false;
    }

    // that is local resource so it will be in ready (delivered) state no matter
    // what its result will be
    rq->ready_flag = true;

    if (u.protocol == CHARS("sciter") || u.protocol == CHARS("sres")) {
      ustring     name = u.filename;
      tool::bytes data = html::app()->get_resource(name);
      if (data.length) {
        rq->data = data;
        rq->success_flag = true;
        on_data_loaded(rq);
        return true;
      }
#ifdef _DEBUG
      dbg_printf("sciter: unknown resource %s\n", rq->url.c_str());
#endif
      rq->status = 404;
      on_data_loaded(rq);
      return true;
    }
    
    if (u.protocol == CHARS("res")) {
      dbg_printf("res: unknown resource %s\n", rq->url.c_str());
      rq->status = 404;
      return true;
    }

    if (u.protocol == CHARS("data")) {
      if (crack_data_url(u.filename(), rq->data_content_type, rq->data)) {
        rq->success_flag = true;
        on_data_loaded(rq);
        return true;
      }
      return true;
    }

    if (u.protocol == CHARS("home")) {
      ustring rp = url::unescape(rq->url(7)); // home://
      ustring path = tool::get_home_dir(rp);
      u.parse(url::path_to_file_url(path));
      rq->content_url = u.src;
    }

    if (!u.is_local()) {
      rq->status = 404;
      return true;
    }

    mm_file mmf;
    if (mmf.open(url::unescape(u.filename))) {
      bytes data((byte *)mmf.data(), mmf.size());
      rq->data = data;
      postprocess_loaded_data(rq);
      rq->success_flag = true;
    }
    else // if(!mmf.data())
    {
#if defined(WINDOWS)
      rq->status = ::GetLastError();
#else
      // const char *str = u.filename;
      // size_t strl = strlen(str);
      rq->status = errno;
      if (!rq->status) rq->status = 404;
      // fprintf(stderr,"failed to load %s errno=%d\n", str, rq->status);

#endif
      rq->success_flag = false;
      this->debug_printf(OT_DOM, OS_WARNING,
        "failed to load \"%s\" file, error=%d\n",
        u.filename.c_str(), rq->status);
      //return true;
    }
    on_data_loaded(rq);
    return true;
  }

} // namespace html
