#include "tool/tool.h"
#include "html.h"

namespace html {
  typedef markup::scanner<wchar> scanner;

  struct parser;

  // check if istream is SVG document markup
  bool is_svg_markup(istream &inp) {
    bool r = false;
    {
      scanner scan(inp);
      for (int tt = scan.get_token(); tt; tt = scan.get_token()) {
        if (tt == scanner::TT_TAG_START) {
          tag::symbol_t tag = tag::symbol(scan.get_tag_name());
          r                 = tag == tag::T_SVG;
          break;
        } else if (tt == scanner::TT_DOCTYPE) {
          auto val = scan.get_value();
          r        = val.like(W("*svg *"));
          break;
        } else if (tt == scanner::TT_PI) {
          continue;
        } else if (tt == scanner::TT_SPACE) {
          continue;
        } else if (tt == scanner::TT_TEXT) {
          continue;
        } else if (tt == scanner::TT_COMMENT) {
          continue;
        }
        break;
      }
    }
    inp.rewind();
    return r;
  }

  struct dom_builder {
    dom_builder(document *pd, istream &inp, parser &prsr, view *pv = 0)
        : doc(pd), root(pd), top(pd), pview(pv), in(inp), parser(prsr),
          tag_start_line_no(0) {}

    void on_tag(tag::symbol_t sym, attribute_bag &atts, bool closed,
                bool seen_sel_start, bool seen_sel_end);
    void on_tail(tag::symbol_t sym, bool seen_sel_start, bool seen_sel_end);
    void on_text(wchars chars, int_v pos_sel_start, int_v pos_sel_end);
    void on_comment(wchars chars);
    bool check_state(tag::symbol_t sym);
    bool check_containment(tag::symbol_t sym, const tag::symbol_t *containers,
                           const tag::symbol_t *siblings);

    void on_include(attribute_bag &atts, bool closed);

    bool got_meta(element *m); // if true - include this meta into the DOM
    void skip_until_end_of(tag::symbol_t sym);

    bool fix_text_containment(text* pt);

    handle<document> doc;
    handle<element>  root;
    handle<element>  top;
    array<handle<element>>
        span_stack; // to handle <p><em>starts</p><p>end</em></p> cases as
                    //           <p><em>starts</em></p><p><em>end</em></p>
    view *        pview;
    istream &     in;
    html::parser &parser;
    int_v         index_at;
    int           tag_start_line_no;

    bookmark fragment_start;
    bookmark fragment_end;

    // bookmark         selection_start;
    // bookmark         selection_end;

    NONCOPYABLE(dom_builder)
  };

  enum INPUT_TYPE {
    TAG, // tag head or complete empty tag like <br/>
    TAIL,
    TEXT,
    COMMENT,
  };

  // static wchar BRchar = BR;

  // generates sequence of INPUT_TYPE tokens
  $generator(parser) {
    scanner       scan;
    int           tt;
    attribute_bag atts;
    tag::symbol_t tag;
    bool          empty_tag;
    int_v         saved_tt;
    wchars        text;
    ustring       tail;
    int           line_no;

    parser(istream & inp,
           function<bool(chars, found_wchars_t cb)> entity_resolver,
           function<void(chars, wchars)>            entity_registrator)
        : scan(inp, entity_resolver, entity_registrator) {}

    bool saw_selection_start() const { return scan.saw_selection_start(); }
    bool saw_selection_end() const { return scan.saw_selection_end(); }

    int_v pos_selection_start() const { 
      return scan.pos_selection_start(); 
    }
    int_v pos_selection_end() const { 
      return scan.pos_selection_end(); 
    }

    $emit(INPUT_TYPE) 
      for (tt = scan.get_token(); tt; tt = scan.get_token()) if (tt == scanner::TT_TAG_START) {
      text = wchars();
      tag  = tag::symbol(scan.get_tag_name());
      if (!tag) {
        string tag = scan.get_tag_name();
        view::debug_printf(OT_DOM, OS_WARNING,
                           "unrecognized tag <%s> at (%s(%d))\n", tag.c_str(),
                           scan.get_url().c_str(), scan.get_line_no());
      }
      empty_tag = false;
      atts.clear();
      for (tt = scan.get_token(); tt; tt = scan.get_token())
        if (tt == scanner::TT_ATTR) {
          chars name = scan.get_attr_name();

          // non-standard handling, sic!
          // multipe class attributes are concatenated rather than replaced by
          // last parsed.
          if (name == CHARS("class")) {
            ustring pv = atts.get_ustring(attr::a_class);
            if (pv.length()) {
              pv += W(" ");
              pv += scan.get_value();
              atts.set(attr::a_class, pv);
            } else
              atts.set(attr::a_class, scan.get_value());
          } else
            atts.set(name, scan.get_value());

        } else if (tt == scanner::TT_TAG_HEAD_END) {
          empty_tag = false;
          line_no   = scan.get_line_no();

          // if( tag == tag::T_BR )
          //{
          //  text = wchars(BRchar);
          //  $yield(TEXT);
          //  break;
          //}

          $yield(TAG);

          if (tag == tag::T_BR) {
            $yield(TAIL);
          } else if (tag::parsing_model(tag) == tag::PMODEL_CDATA) {
            tail = ustring::format(W("</%S>"), scan.get_tag_name().start);
            if (scan.get_cdata_until(tail)) {
              text = scan.get_value();
              $yield(TEXT);
              text = wchars();
              $yield(TAIL);
            }
          } else if (tag::parsing_model(tag) == tag::PMODEL_NO_TAIL) {
            $yield(TAIL);
          }
          break;
        } else if (tt == scanner::TT_EMPTY_TAG_END) {
          empty_tag = true;

          $yield(TAG);
          $yield(TAIL);
          break;
        }
    }
    else if (tt == scanner::TT_TAG_END) {
      text = wchars();
      tag  = tag::symbol(scan.get_tag_name());
      $yield(TAIL);
    }
    else if (tt == scanner::TT_TEXT) {
      text = scan.get_value();
      $yield(TEXT);
    }
    else if (tt == scanner::TT_COMMENT) {
      text = scan.get_value();
      $yield(COMMENT);
    }
    $stop
  };

  parsed_html_result parse_html(view &v, istream &inp, document *doc, bookmark *sel_start,
                  bookmark *sel_end) {
    auto entity_resolver = [&](chars name, found_wchars_t val_cb) -> bool {
      ustring eval;
      if (doc->resolve_entity(name, eval)) {
        val_cb(eval);
        return true;
      }
      return false;
    };

    auto entity_registrator = [&](chars name, wchars val) {
      doc->register_entity(name, val);
    };

    tristate_v is_full_document;

    parser      parse(inp, entity_resolver, entity_registrator);
    dom_builder builder(doc, inp, parse, &v);
    INPUT_TYPE  it;
    while (parse(it)) // initial scan:
    {
      switch (it) {
      case COMMENT: continue;
      case TAG:
        builder.tag_start_line_no = parse.line_no;
        if (parse.tag == tag::T_HTML) {
          if (is_full_document.is_undefined())
            is_full_document = true;
          doc->flags.is_synthetic = false; // full markup
          doc->atts               = parse.atts;
        } else if (parse.tag == tag::T_SVG && doc->is_svg_document()) {
          if (is_full_document.is_undefined())
            is_full_document = true;
          doc->flags.is_synthetic = false; // full markup
          doc->atts               = parse.atts;
          inp.set_utf8_encoding();
        }
        else if (parse.tag == tag::T_INCLUDE) {
          builder.tag_start_line_no = parse.line_no;
          builder.on_tag(parse.tag, parse.atts, parse.empty_tag, parse.saw_selection_start(), parse.saw_selection_end());
          continue;
        } else 
          goto GOT_SOMETHING;
        break;
      case TAIL: continue;
      case TEXT:
        if (trim(parse.text).length) goto GOT_SOMETHING;
        continue;
      }
      break;
    }

    while (parse(it)) {
    GOT_SOMETHING:
      switch (it) {
      case COMMENT: builder.on_comment(parse.text); break;
      case TAG:
        builder.tag_start_line_no = parse.line_no;
        builder.on_tag(parse.tag, parse.atts, parse.empty_tag,
                       parse.saw_selection_start(), parse.saw_selection_end());
        break;
      case TAIL:
        builder.on_tail(parse.tag, parse.saw_selection_start(),
                        parse.saw_selection_end());
        break;
      case TEXT:
        builder.on_text(parse.text, parse.pos_selection_start(),
                        parse.pos_selection_end());
        break;
      }
    }
    //     get_token(bool details = false)  { details_mode = details; return
    //     (this->*c_scan)(); }
    v.add_to_update(builder.root, true);
    if (sel_start && sel_end) {
      *sel_start = builder.fragment_start;
      *sel_end   = builder.fragment_end;
    }

    v.on_dom_parsed(doc);

    parsed_html_result res;

    res.full_document = !!is_full_document.val(0);

    if (is_full_document)
      res.content.push(doc);
    else       
      res.content = doc->nodes;

    return res;
  }

  bool insert_html(view &v, istream &inp, document *doc, helement &el,
                   SE_TYPE where, int_v at) {
    auto entity_resolver = [&](chars name, found_wchars_t val_cb) -> bool {
      ustring eval;
      if (doc->resolve_entity(name, eval)) {
        val_cb(eval);
        return true;
      }
      return false;
    };

    auto entity_registrator = [&](chars name, wchars val) {
      doc->register_entity(name, val);
    };

    parser parse(inp, entity_resolver, entity_registrator);

    dom_builder builder(doc, inp, parse, &v);
    int         idx;

    uint changed_bits = CONTENT_ADDED;

    switch (where) {
    case SE_REPLACE:
      builder.root = builder.top = el;
      el->clear(&v);
      changed_bits = CONTENT_ADDED | CONTENT_REMOVED;
      break;
    case SE_INSERT:
      builder.root = builder.top = el;
      builder.index_at = idx = at.val(0);
      break;
    case SE_APPEND:
      builder.root = builder.top = el;
      builder.index_at = idx = el->nodes.size();
      break;
    case SE_OUTER_REPLACE:
      builder.root = builder.top = el->parent;
      builder.index_at = idx = el->node_index;
      changed_bits           = CONTENT_ADDED | CONTENT_REMOVED;
      break;
    case SE_OUTER_INSERT_BEFORE:
      builder.root = builder.top = el->parent;
      builder.index_at = idx = el->node_index;
      break;
    case SE_OUTER_INSERT_AFTER:
      builder.root = builder.top = el->parent;
      builder.index_at = idx = el->node_index + 1;
      break;
    default: assert(false); return false;
    }

    helement parent = builder.top;

    for (INPUT_TYPE it; parse(it);)
      switch (it) {
      case COMMENT: builder.on_comment(parse.text); break;
      case TAG:
        builder.tag_start_line_no = parse.line_no;
        builder.on_tag(parse.tag, parse.atts, parse.empty_tag,
                       parse.saw_selection_start(), parse.saw_selection_end());
        break;
      case TAIL:
        builder.on_tail(parse.tag, parse.saw_selection_start(),
                        parse.saw_selection_end());
        break;
      case TEXT:
        builder.on_text(parse.text, parse.pos_selection_start(),
                        parse.pos_selection_end());
        break;
      }

    v.on_content_change(parent, changed_bits);
    v.add_to_update(parent, CHANGES_MODEL);

    switch (where) {
    case SE_REPLACE: break;
    case SE_INSERT: el = builder.root->first_element(); break;
    case SE_APPEND: el = builder.root->last_element(); break;
    case SE_OUTER_REPLACE: {
      helement src = el;
      el           = el->prev_element();
      src->remove(true);
    } break;
    case SE_OUTER_INSERT_BEFORE: el = el->prev_element(); break;
    case SE_OUTER_INSERT_AFTER: el = el->next_element(); break;
    }
    return true;
  }

  bool dom_builder::check_containment(tag::symbol_t        sym,
                                      const tag::symbol_t *containers,
                                      const tag::symbol_t *siblings) {
    element *t      = top;
    auto is_in_list = [](tag::symbol_t sym, const tag::symbol_t *list) -> bool {
      while (*list != 0)
        if (*list++ == sym) return true;
      return false;
    };

    for (; t && t != root; t = t->parent) {
      if (is_in_list(t->tag, containers)) break;
      if (is_in_list(t->tag, siblings)) {
        view::debug_printf(
            OT_DOM, OS_WARNING, "<%s> element is not allowed at (%s(%d))\n",
            tag::symbol_name(sym).c_str(), parser.scan.get_url().c_str(),
            parser.scan.get_line_no());
        on_tail(t->tag, parser.saw_selection_start(),
                parser.saw_selection_end());
        return false;
      }
    }
    return true;
  }

  bool dom_builder::fix_text_containment(text* pt) {
    static tag::symbol_t no_text_inside[] = { tag::T_TABLE, tag::T_TBODY, tag::T_THEAD, tag::T_TFOOT, tag::T_TR };
    if (items_of(no_text_inside).contains(top->tag) && !pt->is_space()) {
      element* p = nullptr; int node_index = 0;
      for (element* t = top; t; t = t->parent) {
        if (!(items_of(no_text_inside).contains(t->tag)))
        {
          p = t;
          break;
        }
        node_index = t->node_index;
      }
      if (p) {
        p->insert(node_index, pt);
        return true;
      }
    }
    return false;
  }


  bool dom_builder::check_state(tag::symbol_t sym) {
    // return;
    // peculiar HTML parsing rules (no closing tags)
    element *t = top;
    switch (sym) {
    case tag::T_P:
      for (; t && t != root; t = t->parent) {
        tag::TAG_TYPE tt = tag::type(t->tag);
        if (tt == tag::INLINE_TAG || tt == tag::INFO_TAG) continue;
        if (t->tag == tag::T_P) {
          view::debug_printf(
              OT_DOM, OS_WARNING, "<%s> element is not allowed at (%s(%d))\n",
              tag::symbol_name(sym).c_str(), parser.scan.get_url().c_str(),
              parser.scan.get_line_no());
          on_tail(tag::T_P, parser.saw_selection_start(),
                  parser.saw_selection_end());
          return true;
        }
        if (tt == tag::BLOCK_TAG || tt == tag::INLINE_BLOCK_TAG ||
            tt == tag::TABLE_TAG || tt == tag::TABLE_BODY_TAG ||
            tt == tag::TABLE_ROW_TAG || tt == tag::TABLE_CELL_TAG)
          break;
      }
      break;
    case tag::T_LI: {
      static tag::symbol_t C[] = {tag::T_UL, tag::T_OL, tag::T_MENU, tag::T_DIR,
                                  0};
      static tag::symbol_t S[] = {tag::T_LI, 0};
      check_containment(sym, C, S);
      goto CHECK_CLOSED_P;
    } break;
    case tag::T_DD:
    case tag::T_DT: {
      static tag::symbol_t C[] = {tag::T_DL, 0};
      static tag::symbol_t S[] = {tag::T_DD, tag::T_DT, 0};
      check_containment(sym, C, S);
      goto CHECK_CLOSED_P;
    } break;
    case tag::T_TR: {
      static tag::symbol_t C[] = {tag::T_TABLE, tag::T_TBODY, tag::T_THEAD,
                                  tag::T_TFOOT, 0};
      static tag::symbol_t S[] = {tag::T_TR, 0};
      check_containment(sym, C, S);
      goto CHECK_CLOSED_P;
    } break;
    case tag::T_TD:
    case tag::T_TH: {
      static tag::symbol_t C[] = {tag::T_TR, 0};
      static tag::symbol_t S[] = {tag::T_TD, tag::T_TH, 0};
      check_containment(sym, C, S);
      goto CHECK_CLOSED_P;
    } break;
    case tag::T_HEAD: {
      static tag::symbol_t C[] = {tag::T_HTML, 0};
      static tag::symbol_t S[] = {tag::T_HEAD, 0};
      if (!check_containment(sym, C, S)) return false;
    } break;
    case tag::T_BODY: {
      static tag::symbol_t C[] = {tag::T_HTML, 0};
      static tag::symbol_t S[] = {tag::T_BODY, 0};
      if (!check_containment(sym, C, S)) return false;
    } break;
    case tag::T_THEAD:
    case tag::T_TBODY:
    case tag::T_TFOOT: {
      static tag::symbol_t C[] = {tag::T_TABLE, 0};
      static tag::symbol_t S[] = {tag::T_THEAD, tag::T_TBODY, tag::T_TFOOT, 0};
      if (!check_containment(sym, C, S)) return false;
    } break;

    default: {
      tag::TAG_TYPE tt = tag::type(sym);
      if (tt == tag::BLOCK_TAG || tt == tag::TABLE_TAG) goto CHECK_CLOSED_P;
    } break;
    }
    return true;
  CHECK_CLOSED_P:
    for (; t && t != root; t = t->parent) {
      tag::TAG_TYPE tt = tag::type(t->tag);
      if (tt == tag::INLINE_TAG || tt == tag::INFO_TAG) continue;
      if (t->tag == tag::T_P) {
        view::debug_printf(
            OT_DOM, OS_WARNING, "<%s> element is not allowed at (%s(%d))\n",
            tag::symbol_name(sym).c_str(), parser.scan.get_url().c_str(),
            parser.scan.get_line_no());
        on_tail(tag::T_P, parser.saw_selection_start(),
                parser.saw_selection_end());
        return true;
      }
      if (tt == tag::BLOCK_TAG || tt == tag::INLINE_BLOCK_TAG ||
          tt == tag::TABLE_TAG || tt == tag::TABLE_BODY_TAG ||
          tt == tag::TABLE_ROW_TAG || tt == tag::TABLE_CELL_TAG)
        break;
    }
    return true;
  }

  void dom_builder::on_tag(tag::symbol_t sym, attribute_bag &atts, bool closed,
                           bool seen_sel_start, bool seen_sel_end) {
    if (sym == tag::T_INCLUDE) {
      on_include(atts, closed);
      return;
    }

    auto eltype = tag::type(sym);
    if (span_stack.length() &&
        (eltype == tag::INLINE_BLOCK_TAG || eltype == tag::INLINE_TAG))
      while (span_stack.length()) {
        helement cl = span_stack.pop()->clone_element(false);
        // top->dbg_report("top");
        top->append(cl);
        // cl->dbg_report("span");
        top = cl;
      }
    else {
      auto toptype = tag::type(top->tag);
      switch (eltype) {
      case tag::TABLE_BODY_TAG:
        if (toptype != tag::TABLE_TAG) {
          helement cl = new element(tag::T_TABLE);
          top->append(cl);
          top = cl;
        }
        break;
      case tag::TABLE_CELL_TAG:
        if (toptype == tag::TABLE_CELL_TAG) {
          on_tail(top->tag, parser.saw_selection_start(), parser.saw_selection_end());
          toptype = tag::type(top->tag);
        }
        if (toptype != tag::TABLE_ROW_TAG) {
          if (toptype != tag::TABLE_TAG && toptype != tag::TABLE_BODY_TAG) {
            helement cl = new element(tag::T_TABLE);
            top->append(cl);
            top = cl;
          }
          helement cl = new element(tag::T_TR);
          top->append(cl);
          top = cl;
        }
        break;
      case tag::TABLE_ROW_TAG:
        if (toptype == tag::TABLE_ROW_TAG) {
          on_tail(top->tag, parser.saw_selection_start(), parser.saw_selection_end());
          toptype = tag::type(top->tag);
        }
        if (/*toptype != tag::TABLE_TAG &&*/ toptype != tag::TABLE_BODY_TAG) {
          helement cl = new element(tag::T_TBODY);
          cl->state.synthetic(true);
          top->append(cl);
          top = cl;
        }
        break;
        /*          default:
                    if( toptype != tag::TABLE_TAG || toptype !=
           tag::TABLE_BODY_TAG ) { helement cl = new element(tag::T_TABLE);
                      top->append(cl);
                      top = cl;
                    } break; */
      }
    }

    helement el;

    el = new element(sym);

    el->atts = atts;

    if (sym == tag::T_META && !got_meta(el)) return;

    if (top == root && index_at.is_defined()) {
      top->insert(index_at, el);
      index_at = index_at + 1;
    } else {
      if (!check_state(sym)) return;
      top->append(el);
    }
    top = el;
#ifdef _DEBUG
//     top->dbg_report("dom_builder <");
#endif
    if (seen_sel_start) 
      fragment_start = top->start_pos();
    if (seen_sel_end) 
      fragment_end = top->start_pos();
  }
  void dom_builder::on_tail(tag::symbol_t sym, bool seen_sel_start,
                            bool seen_sel_end) {
    if (sym == tag::T_INCLUDE)
      return;
    if (seen_sel_start)
      fragment_start = top->end_pos();
    if (seen_sel_end)
      fragment_end = top->end_pos();

    helement t = top;
    for (; t && t != root; t = t->parent) {
      if (t->tag == sym) {
        top = t->parent;
        goto FOUND;
      }
      if (tag::type(t->tag) == tag::INLINE_TAG) span_stack.push(t);
    }
    // headless tail, check spans stack
    for (int n = span_stack.last_index(); n >= 0; --n)
      if (span_stack[n]->tag == sym) {
        span_stack.remove(n);
        break;
      }

    return;
  FOUND:
    //#ifdef _DEBUG
    //     top->dbg_report("dom_builder >");
    //#endif

    if (pview) switch (sym) {
      case tag::T_STYLE: {
        string id(char(++doc->style_serial_no), 1);
        t->sequential_id(id);
        t->line_no(tag_start_line_no);
        doc->handle_style(t);
      } break;
      case tag::T_LINK:
        if (t->atts[attr::a_rel] == WCHARS("stylesheet")) {
          ustring id(wchar(++doc->style_serial_no), 1);
          t->sequential_id(id);
          t->line_no(tag_start_line_no);
          doc->handle_style(t);
        }
        break;
      case tag::T_SCRIPT: {
        t->line_no(tag_start_line_no);
      } break;
        /*       case tag::T_META:
                 if (!got_meta(t))
                 {
                   

                 }
                 break; */
        // case tag::T_OPTGROUP:
      }
  }

  void dom_builder::on_text(wchars chars, int_v sel_start, int_v sel_end) 
  {
    if (span_stack.length() && trim_right(chars).length)
      while (span_stack.length()) {
        helement cl = span_stack.pop()->clone_element(false);
        // top->dbg_report("top");
        top->append(cl);
        // cl->dbg_report("span");
        top = cl;
      }

    text *t = 0;

    if (top->tag == tag::T_PLAINTEXT) goto PLAINTEXT;

    // assert(chars.length);

    t = new text(chars);
    if (top == root && index_at.is_defined()) {
      top->insert(index_at, t);
      index_at = index_at + 1;
    } else if(!fix_text_containment(t)) {
      top->append(t);
    }

    if (sel_start.is_defined())
      fragment_start = bookmark(t, limit(sel_start.val(0),0, chars.size()-1), false);
    if (sel_end.is_defined())
      fragment_end = bookmark(t, limit(sel_end.val(0), 0, chars.size() - 1), sel_end >= chars.size());

    return;

  PLAINTEXT:
    wchars line; chopline(chars, line);
    if (trim(line).length == 0)
      chopline(chars,line); // skip very first line if it is empty
    do {
      t = new text(line);
      element *te = new element(tag::T_TEXT);
      te->append(t);
      if (top == root && index_at.is_defined()) {
        top->insert(index_at, te);
        index_at = index_at + 1;
      } else {
        top->append(te);
      }
    } while (chopline(chars, line));
  }
  void dom_builder::on_comment(wchars chars) {
    /*if( chars == CHARS("StartFragment") ) {
      this->fragment_start = top->end_pos();
      return;
    }
    if( chars == CHARS("EndFragment") ) {
      this->fragment_end = top->end_pos();
      return;
    }*/
    /*if( chars == CHARS("StartSelection") ) {
      this->selection_start = top->end_pos();
      return;
    }
    if( chars == CHARS("EndSelection") ) {
      this->selection_end = top->end_pos();
      return;
    }*/

    comment *c = new comment(chars);
    if (top == root && index_at.is_defined()) {
      top->insert(index_at, c);
      index_at = index_at + 1;
    } else
      top->append(c);
  }

  bool dom_builder::got_meta(element *meta) {
    string http_equiv = meta->atts.get_string("http-equiv");
    if (http_equiv.is_defined() && lexical::ci::eq(http_equiv(),CHARS("content-type"))) {
      string content = meta->atts.get_string("content");
      content.to_lower();
      int    idx = content().index_of(CHARS("charset="));
      string cp  = "system";
      if (idx >= 0) {
        string cpt = trim(content().sub(idx + 8));
        if (cpt.length()) cp = cpt;
      }
      in.set_encoding(cp);
      return false;
    }
    string charset = meta->atts.get_string("charset");
    if (charset.is_defined()) {
      in.set_encoding(charset);
      return false;
    }

    return true;
  }

  bool check_mquery(dom_builder& dbu, wchars mquery)
  {
    handle<eval::conduit> parsed_media_expr;
    if (mquery.length) {
      parsed_media_expr = new eval::conduit();
      eval::parser p(*parsed_media_expr.ptr(), dbu.doc->uri().src, dbu.parser.line_no, nullptr);
      try {
        wchars rest = p.parse_mediaq(mquery);
        if (rest.length != 0) return true;
      }
      catch (const tool::eval::parse_error&) {
        view::debug_printf(OT_CSS, OS_WARNING,
          "error parsing media section at (%s(%d))\n",
          dbu.doc->uri().src.c_str(), dbu.parser.line_no);
        return false;
      }
      view *pv = dbu.doc->pview();
      if (pv) {
        eval_media_query(pv, dbu.doc, parsed_media_expr, parsed_media_expr->is_true);
        return parsed_media_expr->is_true;
      }
    }
    return true;
  }
  
  void dom_builder::on_include(attribute_bag &atts, bool empty_elem) {
    string url = atts.get_url(in.url, attr::a_src);
    ustring mediaq = atts.get_ustring(attr::a_media);
    if (url.length() && pview && check_mquery(*this,mediaq())) {
      handle<pump::request> rq = new pump::request(url, DATA_HTML);
      if (pview->load_data(rq, true) && rq->data.size()) {
        if (!empty_elem) 
          skip_until_end_of(tag::T_INCLUDE);
        in.push(doc->uri().src, url, rq->data);
        return;
      }
    }
    if (empty_elem) return;

    element *el = new element(tag::T_INCLUDE);
    el->atts    = atts;
    top->append(el);
    top = el;
  }

  void dom_builder::skip_until_end_of(tag::symbol_t sym) {
    /*if(start)
      *start = const_cast<char*>(_pos);
    if(end)
      *end = const_cast<char*>(_pos);)*/

    int level = 0;

    for (INPUT_TYPE it; parser(it);) {
      if (it == TAG && parser.tag == sym)
        ++level;
      else if (it == TAIL && parser.tag == sym) {
        if (level == 0) {
          // if(end)
          //  *end = const_cast<char*>(_pos);
          break;
        }
        --level;
      }
    }
  }

} // namespace html
