#include "html.h"
#if defined(USE_D2D)
#include "../d2d/d2d.h"
#endif
namespace html {

  handle<text_block> element::create_text_block(view &v, wchars chars) {
    handle<element> el = new element(tag::T__MARKER);
    if(chars.length)
      el->append(new text(chars));
    el->owner = el->parent = this;
    //el->c_style                      = this->c_style;
    //el->d_style                      = this->d_style;
    el->flags.disable_text_selection = true;
    text_block::setup_on(v, el, el->nodes());
    return handle<text_block>(el.ptr_of<text_block>());
  }
  
  void text_block::setup_on(view &v, element *el, slice<hnode> nodes) {
    text_block *tb;
    tb = turn_element_to<text_block>(el);
    tb->init(v, nodes);
  }

  void text_block::init(view &v, slice<hnode> nodes) {
#ifdef _DEBUG
    //string s = tag::symbol_name(tag);
    //if (tag == tag::T_P) nodes = nodes;
    //if (tag == tag::T_HTMLAREA) nodes = nodes;
    //if (tag == tag::T_LI && nodes.length == 1 && nodes[0]->is_text()) {
    //  text *pt = nodes[0].ptr_of<text>();
    //  if( pt->chars() == WCHARS("FOO") )
    //    nodes = nodes;
    //}
#endif
    handle<layout_data> pld = ldata.ptr_of<layout_data>();

#if defined(USE_INCREMENTAL_INIT)
    if( v.current_doc && v.current_doc->request_delayed_init(v,this) )
    {
      pld->dim.x = 0;
      pld->dim.y = 0;
      pld->dim_max.x = 0;
      pld->dim_max.y = 0;
      pld->dim_min.x = 0;
      pld->dim_min.y = 0;
      pld->used_dim_min = size();
      pld->used_dim_max = size();
      pld->inner_dim = pld->used_dim = size();
      pld->is_initialized = true;
      assert(flags.delayed_layout);
      return;
    }
#endif    
    auto_state<bool>    _(pld->is_initializing, true);

    pld->setup(v, this, nodes);
  }

  void text_block::drop_style(view *pv) {
    element *owner = get_owner();
    if (c_style->display == display_inline && owner) 
      owner->drop_style(pv);
    if (pv) {
      pv->refresh(this);
      clear_style();
      // if( this->is_of_type<text_block>() )
      // reset_styles(*pv);
    } else
      clear_style();
  }

  void text_block::drop_content_layout(view *pv) {
    if (pv && ldata && ldata->is_of_type<layout_data>()) {
      layout_data *pld = ldata.ptr_of<layout_data>();
      if (!pld->is_initializing && !pld->is_locked) {
        pld->drop();
      }
#ifdef _DEBUG
      else
        dbg_printf("WARNING:recursive text_block::drop_content_layout(view* pv) call");
        // assert(false);
#endif // _DEBUG
    }
    super::drop_content_layout(pv);
  }

  element *text_block::drop_layout(view *pv) {
    // if( c_style->display == display_inline && owner &&
    // owner->is_of_type<text_block>())
    //  return owner->drop_layout(pv);
    // drop_content_layout(pv);
    layout_data *pld   = ldata.ptr_of<layout_data>();

    if (ldata && ldata->is_of_type<layout_data>()) {
      layout_data *pld = ldata.ptr_of<layout_data>();
      if (pld->is_initializing || pld->is_locked)
        pld = pld;
    }

    pld->spell_checked = false;
    return super::drop_layout(pv);
  }

  bool perform_spell_check(view &v, spell_checker *sc, bookmark start,
                           bookmark end) {
    text *tx = nullptr;

    auto checkcb = [&](SPELL_CORRECTIVE_ACTION action, uint start, uint length,
                       ustring replacement) {

      bookmark ms_start(tx, start, false);
      bookmark ms_end(tx, start + length - 1, true);

      // TODO: SPELL_CORRECTIVE_ACTION_REPLACE, below works but 1) needs to be
      // undoable and 2) markable from secondary SC

      /*if (action == SPELL_CORRECTIVE_ACTION_REPLACE &&
      replacement.is_defined())
      {
        int delta = int(length) - replacement.size();
        tx->chars.remove(index_t(start), length);
        tx->chars.insert(index_t(start), replacement);
        end.pos = end.pos + delta;
      }
      else*/
      html::apply_marks(v, ms_start, ms_end, CHARS("misspell"));

    };

    auto process = [&](bookmark s, bookmark e) {
      tx        = s.node.ptr_of<text>();
      uint spos = s.linear_pos();
      uint epos;
      if (e.node != s.node)
        epos = (uint)tx->chars.length();
      else
        epos = e.linear_pos();

      sc->check_spelling(tx->chars(spos, epos), checkcb);
      // ts.dbg_report("start");
      // te.dbg_report("end");
    };

    html::clear_marks(v, start, end, string("misspell"));

    pos_iterator it(start, end);
    // handle<text> s;
    // uint         s_pos = 0;
    bookmark s, e, bm;
    while (it(bm)) {
      if (!bm.valid()) continue;
      if (!bm.node->is_text()) continue;
      if (!bm.node->get_element()->is_visible(v)) continue;

      if (!s.valid()) {
        s = bm;
      } else if (bm.node != s.node) {
        e = s.node->end_pos();
        process(s, e);
        s = bm;
      }
    }
    if (s.valid()) process(s, end);

    return true;
  }

  void text_block::check_spelling(view &v, spell_checker *sc) {
    super::check_spelling(v, sc);
    layout_data *pld = ldata.ptr_of<layout_data>();
    if (!pld->spell_checked) {
      pld->spell_checked = true;
      perform_spell_check(v, sc, this->start_pos(), this->end_pos());
    }
  }

  text_block::layout_data *text_block::get_layout_data(view &v) {
    if (!ldata->is_of_type<text_block::layout_data>()) {
      assert(false);
      ldata = turn_element_to<text_block>(this)->ldata;
    }
    // assert(ldata->is_initialized);
    if (!ldata->is_initialized && !ldata->is_initializing) {
      // setup_layout(v);
      array<hnode> transformed_nodes;
      init(v, get_nodes(v, transformed_nodes));
    }
    return ldata.ptr_of<layout_data>();
  }

  void text_block::calc_intrinsic_widths(view &v) {
    layout_data *pld = get_layout_data(v);
    if (pld->is_initializing || flags.delayed_layout) 
      return;
    // assert(pld->type() == flow_text);
#ifdef DEBUG
    if (this->is_css_element())
      pld = pld;
#endif 
    if (c_style->box_sizing_x() > box_sizing_content || c_style->box_sizing_y() > box_sizing_content) 
    {
      size container_dim = parent ? parent->dim() : v.client_dim();
      measure_borders_x(v, container_dim);
      measure_borders_y(v, container_dim);
    }

    pld->calc_min_max(v, this, pld->dim_min.x._v, pld->dim_max.x._v);
    int inners = pld->inner_borpad_x();

    pld->dim_min.x._v += inners;
    pld->dim_max.x._v += inners;
  }

  // float calc_descent(slice<DWRITE_LINE_METRICS> lms)
  //{
  //  return lms.last().height - lms.last().baseline;
  //}

  int text_block::layout_width(view &v, int width) {
    #ifdef _DEBUG
        //if( tag == tag::T__MARKER )
        //  width = width;
       if( this->is_id_test() )
          width = width;
       //if (tag == tag::T_TEXT && parent->tag == tag::T_PLAINTEXT && parent->last_element() == this)
       //  width = width;
    #endif
    // assert(pld->type() == flow_text);
    hstyle              cs  = get_style(v);
    handle<layout_data> pld = get_layout_data(v);
    if (pld->is_initializing || flags.delayed_layout) 
      return 0;

    pld->dim.x = width;

    rect crect = client_rect(v);
    size inner = crect.size();
    if (cs->overflow_x != overflow_none) inner.x = max(inner.x, pld->dim_min.x);

    if (pld->dim_min.x.is_defined() && pld->dim_min.y.is_defined() &&
        pld->inner_dim.x == inner.x)
      return pld->dim_min.y; // already measured

    if (pld->dim_min.x.is_undefined() || flags.has_percent_widths)
      calc_intrinsic_widths(v);

    auto_state<helement> BFC(v.bfc, is_floats_container(v) ? this : v.bfc);

    if (pld->fctx) pld->fctx->reset(v);

    if (!cs->can_wrap() && inner.x < pld->dim_max.x) inner.x = pld->dim_max.x;

    pld->inner_dim.x = inner.x;
    pld->inner_dim.y = 0;

    // element *floats_container = floats_parent(v);
    int h = pld->flow_text(v, this);
    if (ldata->fctx && !ldata->fctx->is_empty())
      h = max(h, ldata->fctx->get_max_y(v));

    return pld->dim_min.y = pld->dim_max.y = h;

  }

  int text_block::layout_height(view &v, int height) {
    layout_data *pld = get_layout_data(v);
    if (pld->is_initializing || flags.delayed_layout) return 0;

#ifdef DEBUG
    if (is_id_test())
      height = height;
#endif // DEBUG

    // assert(pld->type() == flow_text);
    pld->dim.y = height;

    auto valign = c_style->get_text_vertical_align();
    if(!is_inline_block_element(v)) 
      switch (valign) 
      {
        case valign_top: pld->offset.y = 0; break;
        case valign_bottom: pld->offset.y = pld->dim_min.y - height; break;
        case valign_middle: pld->offset.y = -(height - pld->dim_min.y) / 2; break;
      }
    pld->inner_dim = client_rect(v).size();
    return pld->dim.x;
  }

  bool text_block::get_first_line_metrics(view &v, int &y, int &height,
                                          int &ascent) {
    if (flags.delayed_layout)
      return false;
    layout_data *pld = get_layout_data(v);
    // assert(pld->type() == flow_text);
    if (!is_layout_valid())
      measure_inplace(v);
    if (pld->lines_count() == 0) return false;
    auto line = pld->get_line(0);
    y         = line.y;
    ascent    = line.baseline;
    height    = line.height;

    return true;
  }

  bool text_block::get_last_line_metrics(view &v, int &y, int &height,
                                         int &ascent) {
    if (get_style(v)->overflow_y >= overflow_hidden) return false;
    if (flags.delayed_layout) return false;

    if (!is_layout_valid())
      measure_inplace(v);

    layout_data *pld    = get_layout_data(v);
    uint         nlines = pld->lines_count();
    if (nlines == 0) return false;
    auto line = pld->get_line(nlines - 1);
    y         = line.y;
    ascent    = line.baseline;
    height    = line.height;
    // this->min_height(v);
    return true;
  }

  rect clip_rect(view &v, element *b);

  void text_block::draw_content_scrollable(view &v, graphics *pg, point pos,
                                           bool clip) {
    if (flags.delayed_layout)
      return;

    layout_data *pld = get_layout_data(v);
    // assert(pld->type() == flow_text);

    //if (is_id_test())
    //  pg = pg;

    rect clip_area = clip_rect(v, this);
    if (clip_area.empty() && clip) return;

    rect vr(v.dimension());

    bool clip_overflow = c_style->clip_overflow();

    point client_offset = client_rect(v).s;
    clip_area += pos;

    clipper _(pg, clip_area, clip_overflow && clip, !c_style->has_solid_background());

    pos = scroll_translate(v, pos);
    pos += client_offset;

    pld->zctx.draw(v, pg, pos, this, false);

    if (v._selection_ctx && v._selection_ctx->is_valid() &&
        v._selection_ctx->get_selection_type(v) <= TEXT_RANGE &&
        v._selection_ctx->allow_text_selection_on(this)) {
      draw_selection(v, pg, pos, v._selection_ctx);
    } else {
      draw_glyph_runs(v, this, *pld, pg,
                      pos /* - pld->offset , due to scroll_translate()*/,
                      nullptr);
    }

    if (pg->has_overlays(_)) 
      draw_outlines(v, pg, pos, true, false);

    pld->zctx.draw(v, pg, pos, this, true);
  }

  bool text_block::is_placeholder() const {
    return nodes.size() == 1 && nodes[0]->is_text() &&
           nodes[0].ptr_of<html::text>()->chars.size() == 0;
  }

  void text_block::draw_content(view &v, graphics *pg, point pos, bool clip) {
    if (flags.delayed_layout)
      return;

    handle<layout_data> pld = get_layout_data(v);

    if (c_style->clip_overflow())
      return draw_content_scrollable(v, pg, pos, clip);

    pld->zctx.draw(v, pg, pos, this, false);

    if (pld->_glyph_runs.length())
    {
      if (v.sticky_scrollable && !v.sticky_scrollable_anchor) {
        if (pg->get_clip_rc().y().contains(pos.y))
          v.sticky_scrollable_anchor = this;
      }

      if (v._selection_ctx && v._selection_ctx->is_valid() &&
        v._selection_ctx->get_selection_type(v) <= TEXT_RANGE &&
        v._selection_ctx->allow_text_selection_on(this)) {
        draw_selection(v, pg, pos, v._selection_ctx);
      }
      else {
        draw_glyph_runs(v, this, *pld, pg, pos - pld->offset, nullptr);
      }
    }
    pld->zctx.draw(v, pg, pos, this, true);
  }

  void text_block::draw_glyphs(view &v, graphics *pg, point pos) {
    if (flags.delayed_layout)
      return;
    handle<layout_data> pld = get_layout_data(v);
    draw_glyph_runs(v, this, *pld, pg, pos - pld->offset, nullptr);
  }

  /*  uint land_pos(text_block* base_tb, element* child)
    {
      assert(0);
      return 0;
    } */

  /*bool selection_ctx::get_sel_positions(text_block* for_tb, uint& start, uint&
  end)
  {
    start = 0;
    end = 0;
    return false;
  }*/

  void text_block::draw_selection(view &v, graphics *pg, point pos,
                                  selection_ctx *psi) {
    if (flags.delayed_layout)
      return;

    layout_data *pld = get_layout_data(v);

    tflow::sel_context sctx;

    if (psi->ime_start.valid() && psi->ime_end.valid())
      pld->get_sel_glyph_positions(this, psi->ime_start, psi->ime_end,
                                   sctx.ime_glyph_start, sctx.ime_glyph_end);

    if (!pld->get_sel_glyph_positions(this, psi->caret, psi->anchor,
                                      sctx.sel_glyph_start,
                                      sctx.sel_glyph_end) && !psi->target.valid()) {
      draw_glyph_runs(v, this, *pld, pg, pos - pld->offset, nullptr);
      return;
    }

    sctx.selection_back_color = psi->selection_back_color(v);
    sctx.selection_fore_color = psi->selection_fore_color(v);
    sctx.selection_text_color = psi->selection_text_color(v);
    
    draw_glyph_runs(v, this, *pld, pg, pos - pld->offset, &sctx);

    caret_metrics gm;

    if (psi->target.valid() && psi->target.node->owned_by(this) && get_caret_metrics(v, psi->target, gm)) {
      gm.move(pos - pld->offset);
      psi->draw_caret(v, pg, gm);
    } else if (psi->caret.valid() && psi->caret.node->owned_by(this) && get_caret_metrics(v, psi->caret, gm)) {
      gm.move(pos - pld->offset);
      psi->draw_caret(v, pg, gm);
    } else {
      super::draw_selection(v, pg, pos, psi);
    }
  }

  bool element::get_caret_metrics(view &v, const bookmark &bm,
                                  caret_metrics &gm) {
    if (flags.delayed_layout)
      return false;

    rect r    = border_box(v);
    gm.x1     = float(r.left());
    gm.x2     = float(r.right());
    gm.y1     = r.top();
    gm.y2     = r.bottom();
    gm.at_end = bm.after_it;

    gm.elem     = this;
    gm.line_no  = 0;
    gm.glyph_rc = r;
    gm.line_y1  = 0;
    gm.line_y2  = 10;
    gm.at_end = bm.at_element_end();

    gm.ctype = BLOCK_POSITION;
    return true;
  }

  bool text_block::get_caret_metrics(view &v, const bookmark &bm,
                                     caret_metrics &gm) {
    if (flags.delayed_layout)
      return false;

    if (!is_layout_valid()) return false;

    layout_data *pld = get_layout_data(v);

    if (pld->_runs.length() == 0) {
      super::get_caret_metrics(v, bm, gm);
      gm.x2           = gm.x1;
      gm.glyph_rc.e.x = gm.glyph_rc.e.y;
      return true;
    }

    // Gets the current caret position (in untransformed space).
    if (bm.node->owned_by(this)) {
      pld->get_metrics(v, this, bm, gm);
      gm.elem = this;
      return true;
    }

    return super::get_caret_metrics(v, bm, gm);

    /* this causes SO on <span><p>abc</p></span> constucts
    element* p = parent;
    while(p)
    {
      if(bm.node->belongs_to(p))
      {
        parent->get_caret_metrics(v, bm, gm);
        return true;
      }
      p = p->parent;
    }
    assert(false);
    return false;
    */
  }

  element *text_block::find_child_element(view &v,
                                    point pos /* local coordinate */,
                                    bool  exact) {
    layout_data *pld = get_layout_data(v);

    node *pn = pld->find_node_at(v, pos, exact);
    if (pn) { return pn->get_ui_element(); }

    // child element was not found in normal flow
    // try to find block sub element by brute force

    for (int n = 0; n < pld->_used_nodes.size(); ++n) {
      if (!pld->_used_nodes[n]->is_element()) continue;
      element* b = pld->_used_nodes[n].ptr_of<element>();
      if (!b->is_inline_block_element(v))
        continue;
      if (b->state.popup()) // it is handled already for the popups
        continue;
      if (b->state.moving() || b->state.copying())
        continue; 
      if (!b->floats(v) && !b->positioned(v) && b->is_visible(v)) {
        element *tb = b->find_element(v, pos - b->pos(), exact);
        if (tb) return tb;
      }
    }

    return nullptr;

  }

  bookmark text_block::find_text_pos(view &v,
                                     point zpos /*at is owner relative*/) {

    if (!is_layout_valid())
      measure_inplace(v);

    if (!super::find_element(v, zpos, false)) {
      assert(false);
      return bookmark();
    }
    if (is_placeholder()) return nodes[0]->start_pos();

    layout_data *pld = get_layout_data(v);
    // assert(pld->type() == flow_text);

    // find_text_pos
    // zpos = translate(zpos);

    zpos = inverse_translate(v, zpos);
    zpos += scroll_pos();
    zpos -= client_rect(v).s;

    // affine_mtx_f vmx;
    // view_mtx(v,vmx);
    // pointf sc;
    // vmx.scaling(&sc.x,&sc.y);
    // return pld->find_bookmark_at(v, point(pointf(zpos) / sc));

    return pld->find_bookmark_at(v, zpos);
  }

  style *text_block::get_style_at(view &v, uint n) {
    // layout_data* pld = get_layout_data(v);
    return get_style(v);
  }

  bool text_block::is_caret_pos_at(view& v,const bookmark &bm) const {
    layout_data *pld = const_cast<text_block*>(this)->get_layout_data(v);//ldata.ptr_of<layout_data>();
    if (tag == tag::T_BR) return false;
    return pld->is_caret_pos_at(bm);
  }

  bookmark text_block::start_caret_pos(view &v) const {
    layout_data *pld = const_cast<text_block *>(this)->get_layout_data(v);
    bookmark     bm  = start_pos();
    pld->advance_caret_pos(v, this, bm, ADVANCE_FIRST);
    return bm;
  }
  bookmark text_block::end_caret_pos(view &v) const {
    layout_data *pld = const_cast<text_block *>(this)->get_layout_data(v);
    bookmark     bm  = end_pos();
    pld->advance_caret_pos(v, this, bm, ADVANCE_LAST);
    return bm;
  }

  bool text_block::advance_caret_pos(view &v, bookmark &bm, ADVANCE_DIR dir,
                                     array<wchar> *out) {
    if (bm.node == this) 
      return super::advance_caret_pos(v, bm, dir, out);
    layout_data *pld = get_layout_data(v);
    return pld->advance_caret_pos(v, this, bm, dir, out);
  }

  node *text_block::first_visible_node() {
    if (!ldata->is_of_type<text_block::layout_data>()) return first_node();
    return ldata.ptr_of<layout_data>()->_runs.size()
               ? ldata.ptr_of<layout_data>()->_runs.first().node
               : nullptr;
  }
  node *text_block::last_visible_node() {
    if (!ldata->is_of_type<text_block::layout_data>()) return last_node();
    return ldata.ptr_of<layout_data>()->_runs.size()
               ? ldata.ptr_of<layout_data>()->_runs.last().node
               : nullptr;
  }

  void text_block::drop_pagination(view &v) {
    super::drop_pagination(v);
    layout_data *pld    = get_layout_data(v);
    uint         nlines = pld->lines_count();
    for (uint n = 0; n < nlines; ++n) {
      tflow::layout_line &line = pld->get_line_ref(n);
      line.page_no             = 0;
    }
  }

  void text_block::get_ui_text(view &v, array<wchar> &to) {
    layout_data *pld = const_cast<text_block *>(this)->get_layout_data(v);

    if (style_content && style_content->before)
      style_content->before->get_ui_text(v, to);
    pld->get_text(v, to);
    if (style_content && style_content->after)
      style_content->after->get_ui_text(v, to);
  }

  int text_block::paginate(view &v, range page, range &intersection,
                           int &elements_on_page, int page_no) {

    layout_data *pld    = get_layout_data(v);
    uint         nlines = pld->lines_count();
    if (nlines == 0)
      return super::paginate(v, page, intersection, elements_on_page, page_no);

    point vp  = this->doc_pos(v);
    rect  bb  = margin_box(v) + vp;
    range bby = bb.y();

    const style *cs = get_style(v);

    /* #ifdef _DEBUG
        if(tag == tag::T_P)
          cs = cs;

        if( page_no == 1 )
          cs = cs;

        if( cs->page_break_before.is_defined() )
          cs = cs;

        if (is_id_test())
          intersection = intersection;
    #endif */

    if (bby.e < page.s) return 0;

    if (bby.s >= page.e) {
      ldata->page_no_start = ldata->page_no_end = 0;
      return 0;
    }

    if (cs->page_break_before.is_defined() && (bby.s > page.s)) {
      int pr = cs->page_break_before;
      if (pr >= 0 && ((bby.s - page.s) > (pr * page.length()) / 100) &&
          elements_on_page > 0) {
        ldata->page_no_start = ldata->page_no_end = ushort(page_no + 1);
        intersection |= bby;
        return 0;
      }
    }

    if (cs->page_break_inside.is_defined() && bby.length() <= page.length() && bby.e >= page.e) {
      int pr = cs->page_break_inside;
      if (pr == page_break_avoid && elements_on_page > 0) {
        intersection.s = min(intersection.s, bby.s);
        return 0;
      }
    }

    if (page.covers(bby)) {

      if (cs->page_break_after.is_defined()) {
        int pr = cs->page_break_after;
        if (pr >= 0 && (bby.e - page.s) > ((pr * page.length()) / 100))
          intersection.s = min(intersection.s, bby.e);
      }

      ldata->page_no_start = ldata->page_no_end = uint16(page_no);
      ++elements_on_page;
      return 1;
    }

    if (ldata->page_no_start == 0) ldata->page_no_start = uint16(page_no);

#ifdef _DEBUG
    if (pld->offset.y && nlines == 1) nlines = nlines;
#endif

    uint n = 0;
    int  c = 0;

    for (; n < nlines; ++n, ++c) {
      tflow::layout_line &line = pld->get_line_ref(n);
      range               yr   = line.yr() + (vp.y - pld->offset.y);

      if (yr.e <= page.s) continue;
      if (yr.e > page.e) {
        intersection |= yr;
        //#if _DEBUG
        //        if(line.page_no)
        //          line.page_no = line.page_no;
        //#endif
        break;
      }
      line.page_no = uint16(page_no);
    }

    if (n == nlines) ldata->page_no_end = uint16(page_no);

    elements_on_page += c;

    return c;
  }

  float element::inline_baseline(view &v) {
    if (ldata->baseline.is_defined())
      return float(ldata->outer_top() + ldata->baseline);

    int ascent, descent, height;
    this->get_inline_block_metrics(v, ascent, descent, height);

    return float(ldata->outer_top() + ascent);
  }
  
  handle<gool::text_layout> view::create_text_layout(wchars t) {
    handle<gool::text_layout> tl = new text_layout();
    tl->set_text(t);
    tl->set_host(doc());
    return tl;
  }

} // namespace html


namespace gool {

using namespace html;
    
#define TB(th) th->tb.ptr_of<html::text_block>()

  text_layout::text_layout() {
    tb = new html::text_block(tag::T_TEXT);
  }
    
  void text_layout::set_host(resource* par, wchars classname) {
    if( classname )
      TB(this)->set_attr(html::attr::a_class, classname);
    TB(this)->parent = static_cast<element*>(par);
    TB(this)->owner = static_cast<element*>(par);
    //get_style();
  }

  void text_layout::set_style(wchars cssstyle) {
    TB(this)->set_attr(html::attr::a_style, cssstyle);
  }

  void text_layout::set_style(const style* pc) {
    TB(this)->c_style = pc;
    TB(this)->d_style = pc;
  }
  
  void text_layout::set_font(wchars cssfont) {
    ustring style = WCHARS("font:");
    TB(this)->set_attr(html::attr::a_style, style + cssfont);
  }

  wchars text_layout::get_font() const {
    ustring style;
    if (TB(this)->atts.exist(attr::a_style, style) && style.like(W("font:*")))
      return style()(5);
    return wchars();
  }

  wchars text_layout::get_style() const {
    ustring style;
    if (TB(this)->atts.exist(attr::a_style, style))
      return style();
    return wchars();
  }

  wchars text_layout::get_class() const {
    ustring cls;
    if (TB(this)->atts.exist(attr::a_class, cls))
      return cls();
    return wchars();
  }

  
  void text_layout::set_text(wchars t) {
    TB(this)->clear();
    TB(this)->append(new text(t));
    TB(this)->drop_layout();
  }

  void text_layout::set_class(wchars cls) {
    TB(this)->set_attr(html::attr::a_class, cls);
  }

  wchars text_layout::get_text() {
    if (TB(this)->nodes.size() && TB(this)->nodes[0]->is_text())
      return TB(this)->nodes[0].ptr_of<text>()->chars();
    return wchars();
  }

  void text_layout::set_width(float width) {
    if (view* pv = TB(this)->pview())
      TB(this)->set_width(*pv, int(width));
  }
  void text_layout::set_height(float height) {
    if (view* pv = TB(this)->pview())
      TB(this)->set_height(*pv, int(height));
  }

  void text_layout::set_color(color_v clr) {
    handle<text_block::layout_data> pld = TB(this)->ldata.ptr_of<text_block::layout_data>();
    pld->_default_text_color = clr;
  }

  bool check_measurement(const text_layout* ptl) {
    if (view* pv = TB(ptl)->pview()) {
      handle<text_block::layout_data> pld = TB(ptl)->get_layout_data(*pv);
      assert(pld->is_of_type<text_block::layout_data>() );
      if (!TB(ptl)->is_layout_valid())
        TB(ptl)->measure_inplace(*pv);
      if (pld->lines_count() == 0) return false;
      return true;
    }
    return false;
  }


  sizef text_layout::get_dim() const {
    if (!check_measurement(this))
      return sizef(0, 0);
    return TB(this)->content_dim();
  }
  sizef text_layout::get_box() const {
    if (!check_measurement(this))
      return sizef(0, 0);
    return TB(this)->dim();
  }
  float text_layout::width_min() const {
    if (!check_measurement(this))
      return 0;
    return float(TB(this)->ldata->dim_min.x);
  }
  float text_layout::width_max() const {
    if (!check_measurement(this))
      return 0;
    return float(TB(this)->ldata->dim_max.x);
  }
  float text_layout::height() const {
    if (!check_measurement(this))
      return 0;
    return float(TB(this)->ldata->dim_min.y);
  }
  
  float text_layout::ascent() const {
    if (!check_measurement(this))
      return 0;
    auto line = TB(this)->ldata.ptr_of<text_block::layout_data>()->get_line(0);
    return float(line.baseline);
  }

  uint text_layout::get_lines_count() const
  {
    if (!check_measurement(this))
      return 0;
    return TB(this)->ldata.ptr_of<text_block::layout_data>()->lines_count();
  }
  text_layout::line text_layout::get_line(int no) const {
    if (!check_measurement(this))
      return line{};
    auto tline = TB(this)->ldata.ptr_of<text_block::layout_data>()->get_line(no);
    line r;
    r.y = float(tline.y);
    r.baseline = float(tline.baseline);
    r.height = float(tline.height);
    r.start = tline.start_text_index;
    r.length = tline.end_text_index - tline.start_text_index;
    return r;
  }
  
  gool::TEXT_ALIGNMENT text_layout::get_text_alignment() const {
    switch (TB(this)->get_style()->get_text_align()) {
      default:
      case align_start:
      case align_left: return ALIGN_START;
      case align_center: return ALIGN_CENTER;
      case align_end:
      case align_right: return ALIGN_END;
    }
  }
  gool::TEXT_ALIGNMENT text_layout::get_lines_alignment() const {
    switch (TB(this)->get_style()->get_text_vertical_align()) {
    default:
    case valign_top: return ALIGN_START;
    case valign_middle: return ALIGN_CENTER;
    case valign_bottom: return ALIGN_END;
    }
  }
  void text_layout::set_alignment(gool::TEXT_ALIGNMENT text, gool::TEXT_ALIGNMENT lines) {
    halign_e ha;
    switch (text) {
      default:
      case ALIGN_START: ha = align_start; break;
      case ALIGN_END: ha = align_end; break;
      case ALIGN_CENTER: ha = align_center; break;
    }
    valign_e va;
    switch (lines) {
      default:
      case ALIGN_START: va = valign_top; break;
      case ALIGN_END: va = valign_bottom; break;
      case ALIGN_CENTER: va = valign_middle; break;
    }
    document* pd = TB(this)->doc();
    if (!pd) return;
    view* pv = pd->pview();
    if (!pv) return;

    TB(this)->update_a_style(*pv, pd, [&](style_prop_map& s) -> bool {
      s.set(cssa_text_align, halign_ev(ha));
      s.set(cssa_vertical_align, valign_ev(va));
      return true;
    });

  }

  handle<text_layout> application::create_text_layout(wchars t) {
    handle<text_layout> tl = new text_layout();
    tl->set_text(t);
    return tl;
  }
    
  void graphics::draw(gool::text_layout *tl, pointf dst, argb c) {
    if (html::view* pv = TB(tl)->pview()) {
      check_measurement(tl);
      if (!c.is_no_color()) tl->set_color(c);
      TB(tl)->draw_content(*pv, this, dst, false);
    }
  }



}


