#include "html.h"

namespace html {
  block_grid *block_grid::setup_on(view &v, element *el) {
    return turn_element_to<block_grid>(el);
  }

#define MAX_GRID_SIZE 10000

  static void cell_ranges(view &v, element *b, irange &y, irange &x) {
    hstyle cs = b->get_style(v);
    x.l       = cs->left.number(0); // 1-based
    x.h       = cs->right.number(0);
    if (x.l >= MAX_GRID_SIZE || x.l < 1) x.l = 0;
    if (x.h >= MAX_GRID_SIZE || x.h < 1) x.h = 0;
    if (!x.l && !x.h)
      x.clear();
    else if (x.l && !x.h)
      x.h = x.l;
    else if (!x.l && x.h)
      x.l = x.h;
    --x.l; // to make it zero based
    --x.h;

    y.l = cs->top.number(0);
    y.h = cs->bottom.number(0);
    if (y.l >= MAX_GRID_SIZE || y.l < 1) y.l = 0;
    if (y.h >= MAX_GRID_SIZE || y.h < 1) y.h = 0;
    if (!y.l && !y.h)
      y.clear();
    else if (y.l && !y.h)
      y.h = y.l;
    else if (!y.l && y.h)
      y.l = y.h;
    --y.l; // to make it zero based
    --y.h;
  }

  void block_grid::init(view &v) {
    hstyle cs = get_style(v);

    if (cs->flow.func) {
      if (cs->flow.func->name == CHARS("row"))
        return init_rows(v);
      else if (cs->flow.func->name == CHARS("grid"))
        return init_grid(v);
    } else
      assert(false);

    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    ld->rows.clear();
    ld->cols.clear();

    slice<helement> blocks = ld->blocks();

    // int n_undef = 0;

    if (!blocks.length) return;

    array<cell_def> &cells = ld->cells;
    cells.clear();

    for (uint n = 0; n < blocks.length; ++n) {
      cell_def cd;
      cell_ranges(v, cd.b = blocks[n], cd.rows, cd.cols);
      if (cd.cols.empty()) cd.cols.l = cd.cols.h = 0;
      if (cd.rows.empty()) cd.rows.l = cd.rows.h = ld->rows.size();
      //++n_undef;

      if (ld->cols.size() <= cd.cols.h) ld->cols.size(cd.cols.h + 1);
      if (cd.rows.empty()) cd.rows = irange(ld->rows.size(), ld->rows.size());
      if (ld->rows.size() <= cd.rows.h) ld->rows.size(cd.rows.h + 1);
      cells.push(cd);
    }
    /*if( n_undef )
      for(uint n = 0; n < cells.length(); ++n)
      {
        cell_def& cd = cells[n];
        if( cd.cols.empty() )
          cd.cols = range(0,ld->cols.last_index());
      } */

  }

  typedef array<tag::symbol_t> tag_symbol_list;

  void block_grid::init_rows(view &vi) {
    hstyle              cs = get_style(vi);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    ld->rows.clear();
    ld->cols.clear();

    slice<helement> blocks = ld->blocks();

    if (!blocks.length) return;

    array<cell_def> &cells = ld->cells;
    cells.clear();

    array<tag_symbol_list> row_template;

    handle<function_value> rd = cs->flow.func;

    for (int i = 0; i < rd->params.size(); ++i) {
      value v = rd->params.value(i);
      if (v.is_string()) {
        string          s = v.get(W(""));
        tag::symbol_t   t = tag::symbol(s);
        tag_symbol_list tl(1);
        tl[0] = t;
        row_template.push(tl);
      } else if (v.is_array()) {
        tag_symbol_list tl(v.size());
        for (uint k = 0; k < v.size(); ++k) {
          string s = v.get_element(k).to_string();
          tl[k]    = tag::symbol(s);
        }
        row_template.push(tl);
      } else {
        assert(false);
        row_template.push(tag_symbol_list());
      }
    }

    int row_template_size = row_template.size();

    if (row_template_size == 0) {
      row_template.push(tag::T__UNKNOWN);
      row_template_size = 1;
    }

    ld->cols.size(row_template_size);

    int current_col = row_template_size;

    auto match = [&](tag::symbol_t sym, int at) -> bool {
      const tag_symbol_list &list = row_template[at];
      FOREACH(i, list) {
        tag::symbol_t t = list[i];
        if (t == sym) return true;
        if (t == tag::T__UNKNOWN) return true;
      }
      return false;
    };

    bool rtl = cs->direction == direction_rtl;

    auto append = [&](element *el, cell_def &cd) {
      if (current_col >= row_template_size) {
        ld->rows.size(ld->rows.size() + 1);
        current_col = 0;
      }
      cd.b      = el;
      cd.rows.l = cd.rows.h = ld->rows.size() - 1;
      if (match(el->tag, current_col)) {
        if(rtl)
          cd.cols.l = cd.cols.h = row_template_size - 1 - current_col;
        else
          cd.cols.l = cd.cols.h = current_col;
        ++current_col;
      }
      else // the element is not in the row template
      {    // replace it in full row:
        if (current_col)
          ld->rows.size(ld->rows.size() + 1); // "close" current row
        cd.cols.l   = 0;
        cd.cols.h   = row_template_size - 1;
        current_col = row_template_size;
      }
    };


    for (uint n = 0; n < blocks.length; ++n) {
      cell_def cd;
      append(blocks[n], cd);
      cells.push(cd);
    }
  }

  void block_grid::calc_intrinsic_widths(view &v) {
    hstyle              cs = get_style(v);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    spring_board &cols = ld->cols;
    spring_board &rows = ld->rows;

    if (cols.size() == 0 || rows.size() == 0)
      init(v);
    else {
      cols.restart();
      rows.restart();
    }

    slice<cell_def> cells = ld->cells();

    if (cells.length == 0 || cols.size() == 0 || rows.size() == 0) {
      ld->dim_min.x = 0;
      ld->dim_max.x = 0;
      return;
    }

    flex_value spacing;

    cs->border_spacing_x.pixels_n_spring_w(v, this, ld->dim.x, spacing);

    // cols.accum(0,0,int_v(),0,0,1000,0,0);

    // flex_value zfv;
    // zfv.px = limits<int>::min_value();
    // zfv.px_max = limits<int>::min_value();

    // int min_w = 0;
    // int max_w = 0;
    // int max_min_w = 0;

    uint n_x_spanned = 0;
    for (uint n = 0; n < cells.length; ++n) {
      const cell_def &cd = cells[n];
      if (!cd.b) continue;
      assert(!cd.rows.empty());
      assert(!cd.cols.empty());
      style *bcs = cd.b->get_style(v);
      premeasure(v, cd.b, bcs, ld->dim);

      flex_value fv;
      flex_value mb;
      flex_value ma;

      bool is_inline_block = cd.b->is_inline_block_element(v);

#ifdef _DEBUG
      if (cd.b->is_id_test())
        cs = cs;
#endif

      int extra = 0;
      if (is_inline_block) {
        extra = cd.b->outer_int_x_extra(v, 0);
        fv.px     = cd.b->min_width(v) + extra;
        fv.flex   = bcs->width.flex1000();
      } else {
        extra = cd.b->borpad_int_x_extra(v, 0);
        fv.px      = cd.b->min_width(v) + extra;
        auto cap   = cd.b->max_width(v);
        if (cap.is_defined()) 
          fv.px_max = cap.val(0) + extra;
        bcs->margin[0].pixels_n_spring_w(v, cd.b, ld->dim.x, mb);
        bcs->margin[2].pixels_n_spring_w(v, cd.b, ld->dim.x, ma);
        fv.flex = bcs->width.flex1000();
      }

      fv.content_px_max = cd.b->get_style(v)->overflows_x() ? 0 : cd.b->max_content_width(v) + extra;

      if (cd.cols.length() == 1) {
        if (cd.cols.l > 0) mb.accum(spacing);
        if (cd.cols.h < (cols.size() - 1)) ma.accum(spacing);
        cols.accum(cd.cols.l, fv, mb, ma);
      } else // this thing spans more than one column so we need to distribute
             // its min/max/weight
      {
        ++n_x_spanned;
      }
    }

    if (n_x_spanned > 0) {
      for (uint n = 0; n_x_spanned > 0 && n < cells.length; ++n) {
        cell_def cd = cells[n];
        if (!cd.b) continue;
        if (cd.cols.length() == 1)
          continue;
        else // check constraints:
        {
          --n_x_spanned;

          flex_value fv;
          flex_value mb;
          flex_value ma;

          style *bcs = cd.b->get_style(v);

          bool is_inline_block = cd.b->is_inline_block_element(v);
          if (is_inline_block) {
            int outer = cd.b->outer_int_x_extra(v, 0);
            fv.px     = cd.b->min_width(v) + outer;
            fv.flex   = bcs->width.flex1000();
          } else {
            int borpad = cd.b->borpad_int_x_extra(v, 0);
            fv.px      = cd.b->min_width(v) + borpad;
            auto cap   = cd.b->max_width(v);
            if (cap.is_defined()) 
              fv.px_max = cap.val(0) + borpad;

            bcs->margin[0].pixels_n_spring_w(v, cd.b, ld->dim.x, mb);
            bcs->margin[2].pixels_n_spring_w(v, cd.b, ld->dim.x, ma);
            fv.flex = //max(1,bcs->width.flex1000());
              bcs->width.flex1000();
          }

          //fv.content_px_max = cd.b->max_content_width(v);
          fv.content_px_max = cd.b->get_style(v)->overflows_x() ? 0 : cd.b->max_content_width(v);

          cols.accum_span(cd.cols.l, cd.cols.h, fv, mb, ma);
        }
      }
    }

#ifdef DEBUG
    if (is_id_test()) {
      cs = cs;
      /*cols.solve(0, spring_board::ALIGN_START);
      int w1 = cols.span_width(0, 0);
      int w2 = cols.span_width(1, 1);
      int w3 = cols.span_width(2, 2);
      w3 = w3;*/
    }
#endif
        
    int extra = ld->inner_borpad_x();
    ld->dim_min.x = cols.total_min + extra;
    ld->dim_max.x = max(cols.total_max, cols.total_min) + extra;
  }

  inline bool is_inline_block(const style *cs) {
    return cs->display == display_inline ||
           cs->display == display_inline_block ||
           cs->display == display_inline_table;
  }

  // min_constraint

  int block_grid::layout_width(view &v, int width) {
    hstyle              cs = get_style(v);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    ld->dim.x = width;

    rect crect     = client_rect(v);
    size inner_dim = crect.size();

    if (ld->dim_min.x.is_defined() && ld->dim_min.y.is_defined() &&
        ld->inner_dim.x == inner_dim.x)
      return ld->dim_min.y; // already measured

    ld->inner_dim.x = inner_dim.x;
    ld->inner_dim.y = 0;
    ld->dim.y       = 0; // ?
    ld->dim_min.y.clear();
    ld->dim_max.y.clear();

    if (ld->dim_min.x.is_undefined() || flags.has_percent_widths) 
      calc_intrinsic_widths(v);

    slice<cell_def> cells = ld->cells();
    // slice<helement> blocks = ld->blocks();

    if (cells.length == 0) return ld->dim_min.y;

    spring_board &cols = ld->cols;
    spring_board &rows = ld->rows;

    if (cols.size() == 0 || rows.size() == 0) return 0;

    halign_e halign = cs->get_horizontal_align();
    ld->offset.x = -cols.solve(width, spring_board::align(halign));
    rows.restart();
    ld->rows_aligns.clear();

    // adjust rows

    flex_value spacing;
    cs->border_spacing_y.pixels_n_spring_h(v, this, ld->dim.x, spacing);

    int n_y_spanned = 0;
    int n_baselined = 0;

    for (uint n = 0; n < cells.length; ++n) {
      cell_def &cd = const_cast<cell_def &>(cells[n]);
      if (!cd.b) continue;
      int w = cols.span_width(cd.cols.l, cd.cols.h);

      if (cd.b->positioned(v)) cd.b->check_positioned_containment(v);

      if (is_inline_block(cd.b->c_style))
        replace_h(v, cd.b, w, true, cd.b->c_style->get_horizontal_align());
      else {
        cd.b->set_x_pos(cd.b->ldata->borpad_left());
        cd.b->set_border_width(v, w);
      }
      cd.shift.x = cd.b->pos().x;

      style *bcs = cd.b->get_style(v);

      flex_value fv;
      flex_value mb;
      flex_value ma;

      bool is_inline_block = cd.b->is_inline_block_element(v);

      if (is_inline_block) {
        int outer = cd.b->outer_int_y_extra(v, 0);
        fv.px     = cd.b->min_height(v) + outer;
        fv.flex   = bcs->height.flex1000();
      } else {
        int borpad = cd.b->borpad_int_y_extra(v, 0);
        fv.px      = cd.b->min_height(v) + borpad;
        auto cap   = cd.b->max_height(v);
        if (cap.is_defined()) fv.px_max = cap.val(0) + borpad;

        bcs->margin[1].pixels_n_spring_h(v, cd.b, ld->dim.x, mb);
        bcs->margin[3].pixels_n_spring_h(v, cd.b, ld->dim.x, ma);
        fv.flex = bcs->height.flex1000();
      }
      //fv.content_px_max = cd.b->max_content_height(v);
      fv.content_px_max = cd.b->get_style(v)->overflows_y() ? 0 : cd.b->max_content_height(v);

      if (cd.rows.length() == 1) {
        if (cd.rows.l > 0) mb.accum(spacing);
        if (cd.rows.h < (rows.size() - 1)) ma.accum(spacing);
        rows.accum(cd.rows.l, fv, mb, ma);

        if (bcs->vertical_align.is_defined() && bcs->vertical_align.val() == valign_baseline) {
          int a, d, h;
          cd.b->get_inline_block_metrics(v, a, d, h);
          if (a) {
            row_align &t = ld->rows_aligns.get(cd.rows.l);
            t.ascent     = max(t.ascent, a);
            t.descent    = max(t.descent, d);
            if (fv.px < t.ascent + t.descent) fv.px = t.ascent + t.descent;
            ++n_baselined;
          }
        }

      } else // this thing spans more than one row so we need to distribute its
             // min/max/weight
      {
        ++n_y_spanned;
      }
    }

#ifdef DEBUG
    if (is_id_test())
      cs = cs;
#endif

    for (uint n = 0; n_y_spanned > 0 && n < cells.length; ++n) {
      const cell_def &cd = cells[n];
      if (!cd.b) continue;
      // element *b = blocks[n];

      if (cd.rows.length() == 1) continue;

      // this thing spans more than one row so we need to distribute its
      // min/max/weight
      --n_y_spanned;

      flex_value fv;
      flex_value mb;
      flex_value ma;

      // int minv = cd.b->min_height(v) + cd.b->borpad_int_y_extra(v);
      style *bcs = cd.b->get_style(v);

      bool is_inline_block = cd.b->is_inline_block_element(v);
      if (is_inline_block) {
        int outer = cd.b->outer_int_y_extra(v, 0);
        fv.px     = cd.b->min_height(v) + outer;
      } else {
        int borpad = cd.b->borpad_int_y_extra(v, 0);
        fv.px      = cd.b->min_height(v) + borpad;
        auto cap   = cd.b->max_height(v);
        if (cap.is_defined()) fv.px_max = cap.val(0) + borpad;

        bcs->margin[1].pixels_n_spring_h(v, cd.b, ld->dim.x, mb);
        bcs->margin[3].pixels_n_spring_h(v, cd.b, ld->dim.x, ma);
      }
      fv.flex           = max(1, bcs->height.flex1000());
      //fv.content_px_max = cd.b->max_content_height(v);
      fv.content_px_max = cd.b->get_style(v)->overflows_y() ? 0 : cd.b->max_content_height(v);

      rows.accum_span(cd.rows.l, cd.rows.h, fv, mb, ma);
    }

    /*if(n_baselined)
      for(uint n = 0; n < ld->rows_aligns.length(); ++n)
      {
        row_align& ta = ld->rows_aligns[n];
        auto& t = rows.items[n];
        if(ta.ascent == 0)
          continue;
        if((ta.ascent + ta.descent) <= t.b_min)
          ta.ascent = (ta.ascent * t.b_min) / (ta.ascent + ta.descent);
        else
        {
          int delta = (ta.ascent + ta.descent) - t.b_min;
          t.b_min = ta.ascent + ta.descent;
          rows.total_min += delta;
        }
      }*/

    int inners = ld->inner_borpad_y();
    ld->dim_min.y = rows.total_min + inners;
    ld->dim_max.y = max(rows.total_min, rows.total_max) + inners;
    return ld->dim_min.y;
  }

  int block_grid::layout_height(view &v, int height) {
    hstyle              cs = get_style(v);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    slice<cell_def> cells = ld->cells();
    // slice<helement> blocks = ld->blocks();

    ld->dim.y = height;

    size inner_dim = client_rect(v).size();

    if (cells.length == 0) {
      ld->inner_dim.y = inner_dim.y;
      return ld->dim.x;
    }

    if (ld->dim_min.y.is_defined() && ld->dim_max.y.is_defined() &&
        ld->inner_dim.y == inner_dim.y)
      return ld->dim.x;

    ld->inner_dim = inner_dim;

    //   _layout_width(v,d,cs);

    spring_board &cols = ld->cols;
    spring_board &rows = ld->rows;

    if (cols.size() == 0 || rows.size() == 0) return 0;

    point inners = ld->inner_border_width.s + ld->inner_padding_width.s;

    auto valign = cs->get_block_vertical_align();
    ld->offset.y    = -rows.solve(height, spring_board::align(valign));

    for (uint n = 0; n < cells.length; ++n) // setup heights
    {
      cell_def &cd = const_cast<cell_def &>(cells[n]);
      if (!cd.b) continue;
      // element *b = blocks[n];
      int h = rows.span_width(cd.rows.l, cd.rows.h);
      if (is_inline_block(cd.b->c_style)) {
        if (cd.rows.l == cd.rows.h && cd.rows.l < ld->rows_aligns.size())
          replace_v(v, cd.b, h, true, cd.b->c_style->get_inline_vertical_align(),ld->rows_aligns[cd.rows.l].ascent);
        else
          replace_v(v, cd.b, h, true, cd.b->c_style->get_inline_vertical_align());
      } else {
        cd.b->set_y_pos(cd.b->ldata->borpad_top());
        cd.b->set_border_height(v, h);
      }
      cd.shift.y = cd.b->pos().y;

      cd.b->set_pos(point(cols.items[cd.cols.l].pos, rows.items[cd.rows.l].pos) + cd.shift + inners);
    }
    return ld->dim.x;
  }

  float block_grid::inline_baseline(view &v) {
    // for( element* le = last_ui_element(); le; le = le->prev_ui_element() )
    if (c_style->overflow_y > overflow_visible && !c_style->height.is_auto() &&
        !c_style->height.is_min_intrinsic() &&
        !c_style->height.is_max_intrinsic())
      return float(ldata->outer_top() + ldata->dim.y);

    float bl  = 0;
    auto  foo = [&bl, &v](element *le) -> bool {
      if (le->c_style->position > position_relative) return false;
      if (le->c_style->get_float()  > 0) return false;
      bl = le->inline_baseline(v) - le->ldata->outer_top() + le->pos().y;
      return true;
    };
    if (each_ui_child(foo)) return bl + ldata->outer_top();
    return super::inline_baseline(v);
  }

  int block_grid::n_rows() {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    return ld->rows.size();
  }
  int block_grid::n_cols() {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    return ld->cols.size();
  }
  void block_grid::get_row(int row, array<handle<element>> &els) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    for (int i = 0; i < ld->cells.size(); ++i) {
      const cell_def &cd = ld->cells[i];
      if (cd.rows.l == row) els.push(cd.b);
    }
  }
  void block_grid::get_col(int col, array<handle<element>> &els) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    for (int i = 0; i < ld->cells.size(); ++i) {
      const cell_def &cd = ld->cells[i];
      if (cd.cols.l == col) els.push(cd.b);
    }
  }
  bool block_grid::get_col_x(int col, gool::range &x) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    if (col < 0 || col >= ld->cols.size()) return false;

    x.s = ld->cols.items[col].pos; 
    x.e = ld->cols.items[col].pos + ld->cols.items[col].b_val;
    return true;
  }
  bool block_grid::get_row_y(int row, gool::range &y) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    if (row < 0 || row >= ld->rows.size()) return false;

    y.s = ld->rows.items[row].pos;
    y.e = ld->rows.items[row].pos + ld->rows.items[row].b_val;
    return true;
  }
  bool block_grid::get_row_at(view &v, int y, int &row) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    for (int r = 0; r < ld->rows.size(); ++r) {
      const auto &cd = ld->rows.items[r];
      if (y < cd.pos + cd.b_val) {
        row = r;
        return true;
      }
    }
    return false;
  }
  bool block_grid::get_col_at(view &v, int x, int &col) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    for (int c = 0; c < ld->cols.size(); ++c) {
      if (x < ld->cols.items[c + 1].pos) {
        col = c;
        return true;
      }
    }
    return false;
  }
  element *block_grid::at(int row, int col) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    for (int i = 0; i < ld->cells.size(); ++i) {
      const cell_def &cd = ld->cells[i];
      if (cd.cols.contains(col) && cd.rows.contains(row)) return cd.b;
    }
    return 0;
  }

// grid()
#define MAX_GRID_SIZE 10000

  typedef array<array<value>> cells_mx;

  inline int mx_cols(cells_mx &mx) { return mx.size() ? mx[0].size() : 0; }

  inline void mx_set_cell(cells_mx &mx, int r, int c, const value &str) {
    int w = mx_cols(mx);
    if (c >= w) {
      w = c + 1;
      for (int n = 0; n < mx.size(); ++n)
        mx[n].size(w);
    }
    while (r >= mx.size())
      mx.push(array<value>(max(w, 1)));
    mx[r][c] = str;
  }

  void block_grid::check_ranges(view &v, element *b, irange &rows, irange &cols) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    ld->rows.clear();
    ld->cols.clear();

    array<cell_def> &cells = ld->cells;

    hstyle cs = b->get_style(v);
    value  name;
    if (cs->floats.is_string()) 
      name = cs->floats;

    if (name.is_defined())
      for (int i = 0; i < cells.size(); ++i) {
        cell_def &cd = cells[i];
        if (cd.id == name) {
          cd.b = b;
          return;
        }
      }
    else
      for (int i = 0; i < cells.size(); ++i) {
        cell_def &cd = cells[i];
        if (cd.id.is_int()) {
          int n = cd.id.get(0) - 1;
          if (b->index() == n) {
            cd.b = b;
            return;
          }
        } else if (cd.id.is_defined())
          assert(false);
      }

    ++rows.h;
    cell_def cd;
    cd.b    = b;
    cd.cols = cols;
    cd.rows = irange(rows.h, rows.h);
    cells.push(cd);
  }

  void block_grid::init_grid(view &vi) {
    hstyle              cs = get_style(vi);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    ld->rows.clear();
    ld->cols.clear();

    slice<helement>  blocks = ld->blocks();
    array<cell_def> &cells  = ld->cells;

    // int n_undef = 0;

    cells.clear();

    if (!blocks.length) return;

    cells_mx mx;

    handle<function_value> gr = cs->flow.func;
    for (int r = 0; r < gr->params.size(); ++r) {
      value v = gr->params.value(r);
      if (v.is_array()) {
        for (uint c = 0; c < v.size(); ++c) {
          mx_set_cell(mx, r, c, v.get_element(c));
        }
      } else if (v.is_int()) {
        mx_set_cell(mx, r, 0, v);
      } else {
        assert(false);
      }
    }

    //// at least single cell
    // cols.l = cols.h = 1;
    // rows.l = rows.h = 1;

    int n_columns = mx_cols(mx);

    for (int rn = 0; rn < mx.size(); ++rn) {
      for (int cn = 0; cn < n_columns; ++cn) {
        value cd = mx[rn][cn];
        if (cd.is_undefined()) continue;
        int r, c, c_count = 1, r_count = 1;
        for (r = rn + 1; r < mx.size(); ++r) {
          value tcd = mx[r][cn];
          if (tcd != cd) break;
          ++r_count;
        }
        for (c = cn + 1; c < n_columns; ++c) {
          value tcd = mx[rn][c];
          if (tcd != cd) break;
          ++c_count;
        }
        for (r = rn; r < (rn + r_count); ++r)
          for (c = cn; c < (cn + c_count); ++c)
            mx[r][c].clear();

        cell_def celld;
        celld.cols.l = cn;
        celld.cols.h = cn + c_count - 1;
        celld.rows.l = rn;
        celld.rows.h = rn + r_count - 1;
        celld.id     = cd;
        cells.push(celld);
      }
    }

    irange cols(0, n_columns - 1);
    irange rows(0, mx.size() - 1);

    for (uint n = 0; n < blocks.length; ++n)
      check_ranges(vi, blocks[n], rows, cols);

    ld->cols.size(cols.length());
    ld->rows.size(rows.length());
  }


  rect clip_rect(view &v, element *b);
  bool drawing_under_transforms(view &v, element *el);

  struct find_row_pos_env // find_element traits for binary search
  {
    view &         v;
    point          pt;
    block_grid*    pgrid;

    find_row_pos_env(view &v_, block_grid* elem, point pt_)
      : v(v_), pt(pt_), pgrid(elem) {}

    int total() const { return int(pgrid->n_rows()); }

    bool is_less(int n) {
      gool::range yr;
      pgrid->get_row_y(n, yr);
      return yr.e < pt.y;
    }
    bool is_equal(int n) {
      gool::range yr;
      pgrid->get_row_y(n, yr);
      return yr && pt.y; // >= yr.s && pt.y < yr.e;
    }
    bool is_comparable(int n) {
      return true;
    }
    NONCOPYABLE(find_row_pos_env)
  };

  int find_first_row_to_draw(view &v, block_grid *pgrid, point pt) {

    find_row_pos_env env(v, pgrid, -pt);
    int              first_r = max(0, bsearch(env));
    return first_r;
  }

  void block_grid::draw_content(view &v, graphics *pg, point pos, bool clip)
  {
    layout_data *pld = ldata.ptr_of<layout_data>();
    int          n = pld->blocks.size();

    if (pld->zctx.is_empty() && n == 0) return;

    rect clip_area = clip_rect(v, this);
    if (clip_area.empty() && clip) return;

    // rect vr(-v.screen_pos(), v.screen_dim());

    bool  clip_overflow = c_style->clip_overflow();
    point client_offset = client_rect(v).s;
    clip_area += pos;

    clipper _(pg, clip_area, clip_overflow && clip, !c_style->has_solid_background());

    pos = scroll_translate(v, pos);
    pos += client_offset;

    pld->zctx.draw(v, pg, pos, this, false);

    int fn = find_first_row_to_draw(v, this, pos);

    for (int i = fn; i < n; ++i) {
      element *b = pld->blocks[i];
      hstyle   cs = b->get_style(v);
      if (!cs->display || cs->visibility != visibility_visible) continue;
      if (cs->position > position_static) continue;
      if (b->airborn) continue;
      if (b->state.popup()) continue;
      if (b->floats(v)) continue;
      rect r = b->border_box(v, TO_PARENT) + pos;
      if (r.overlaps_with(clip_area))
        b->draw(v, pg, b->pos() + pos);
    }

    if (pg->has_overlays(_)) draw_outlines(v, pg, pos, true, false);

    pld->zctx.draw(v, pg, pos, this, true);
  }


} // namespace html
