#include "html.h"

namespace html {

  block *block::setup_on(view &v, element *el) {
#ifdef _DEBUG
    if (el->is_id_test() && el->tag == tag::T_BODY)
      el = el;
#endif // _DEBUG
    block *tb = turn_element_to<block>(el);
    return tb;
  }

  int block::n_ui_children() const {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    return ld->blocks.size();
  }

  element *
  block::ui_child(int idx) const // n-th ui child (rendering tree child)
  {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    return ld->blocks[idx];
  }

  int block::n_rows() {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    return ld->blocks.size();
  }
  int  block::n_cols() { return 1; }
  void block::get_row(int row, array<handle<element>> &els) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    els.push(ld->blocks[row]);
  }
  void block::get_col(int col, array<handle<element>> &els) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    els                    = ld->blocks;
  }
  bool block::get_col_x(int col, gool::range &x) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    x = range(0, max(ld->dim_min.x, ld->dim.x));
    return true;
  }
  bool block::get_row_y(int row, gool::range &y) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    view *              pv = pview();
    if (!pv) return false;
    if (row < 0 || row >= ld->blocks.size()) return false;
    y = ld->blocks[row]->margin_box(*pv, TO_PARENT).y();
    return true;
  }
  bool block::get_row_at(view &v, int y, int &row) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    point               pos(ld->dim.x / 2, y);
    element *           child = find_child_element(v, pos, false);
    for (; child; child = child->get_owner()) {
      if (child->get_owner() == this) {
        row = int(child->flags.ui_index);
        return true;
      }
    }
    return false;
  }
  bool block::get_col_at(view &v, int x, int &col) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    if (!ld->blocks.size()) return false;
    col = 0;
    return true;
  }
  element *block::at(int row, int col) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    return col > 0 ? 0 : ld->blocks[row];
  }

  bool get_auto_width(view &v, element *b, int &w) {
    int_v val = b->auto_width(v);
    if (val.is_defined()) {
      w = val;
      return true;
    }
    return false;
  }
  bool get_auto_height(view &v, element *b, int &h) {
    int_v val = b->auto_height(v);
    if (val.is_defined()) {
      h = val;
      return true;
    }
    return false;
  }

  //int_v get_parent_declared_width(view &v, element *b);

  element::max_values element::measure_borders_x(view &v, size container_dim) {
    style *      cs = get_style(v);
    layout_data *ld = ldata;

    //int_v container_width;
    auto  pixels = [&](const size_v &w) -> int {
      size_v minw = w.get_min();
      return minw.is_defined() ?
        max(html::pixels(v, this, w, container_dim).width(), html::pixels(v, this, minw, container_dim).width()) :
        html::pixels(v, this, w,container_dim).width();
    };

    auto  max_pixels = [&](const size_v &w) -> int_v {
      size_v maxw = w.get_max();
      if(maxw.is_defined())
        return int_v(html::pixels(v, this, maxw, container_dim).width());
      return int_v();
    };

    element::max_values mv;

    mv.m1 = max_pixels(cs->margin[0]);
    mv.m2 = max_pixels(cs->margin[2]);

    ld->margin_width.s.x = pixels(cs->margin[0]);
    ld->margin_width.e.x = pixels(cs->margin[2]);

    auto box_sizing = cs->box_sizing_x();
    
    if (box_sizing >= box_sizing_padding) {
      ld->inner_padding_width.s.x = pixels(cs->padding_width[0]);
      ld->inner_padding_width.e.x = pixels(cs->padding_width[2]);
      ld->padding_width.s.x = 0;
      ld->padding_width.e.x = 0;
      mv.ip1 = max_pixels(cs->padding_width[0]);
      mv.ip2 = max_pixels(cs->padding_width[2]);
      mv.p1 = int_v(0);
      mv.p2 = int_v(0);
    }
    else {
      ld->padding_width.s.x = pixels(cs->padding_width[0]);
      ld->padding_width.e.x = pixels(cs->padding_width[2]);
      ld->inner_padding_width.s.x = 0;
      ld->inner_padding_width.e.x = 0;
      mv.p1 = max_pixels(cs->padding_width[0]);
      mv.p2 = max_pixels(cs->padding_width[2]);
      mv.ip1 = int_v(0);
      mv.ip2 = int_v(0);
    }

    if (box_sizing >= box_sizing_border) {
      auto t1 = cs->inner_used_border_width(0);
      auto t2 = cs->inner_used_border_width(2);
      ld->inner_border_width.s.x = pixels(t1);
      ld->inner_border_width.e.x = pixels(t2);
      ld->border_width.s.x = 0;
      ld->border_width.e.x = 0;
      mv.ib1 = max_pixels(t1);
      mv.ib2 = max_pixels(t2);
      mv.b1 = int_v(0);
      mv.b2 = int_v(0);
    }
    else {
      auto t1 = cs->used_border_width(0);
      auto t2 = cs->used_border_width(2);
      ld->border_width.s.x = pixels(t1);
      ld->border_width.e.x = pixels(t2);
      ld->inner_border_width.s.x = 0;
      ld->inner_border_width.e.x = 0;
      mv.b1 = max_pixels(t1);
      mv.b2 = max_pixels(t2);
      mv.ib1 = int_v(0);
      mv.ib2 = int_v(0);
    }

    return mv;

  }

  element::max_values element::measure_borders_y(view &v, size container_dim) {
    style *cs = get_style(v);
    auto   pixels = [&](const size_v &w) -> int {
      // note that paddings, borders and margins in percents are measured against *width* of the container
      // so .width() here is used instead of .height()
      return html::pixels(v, this, w).width();
    };

    auto  max_pixels = [&](const size_v &w) -> int_v {
      size_v maxw = w.get_max();
      if (maxw.is_defined())
        return int_v(html::pixels(v, this, maxw).width());
      return int_v();
    };

    // note: margins, paddings and borders in percents are calculated against
    // *width* of the container. Per CSS spec.
    layout_data *ld           = ldata;

    element::max_values mv;

    mv.m1 = max_pixels(cs->margin[1]);
    mv.m2 = max_pixels(cs->margin[3]);

    ld->margin_width.s.y = pixels(cs->margin[1]);
    ld->margin_width.e.y = pixels(cs->margin[3]);

    //bool paddings_content = cs->box_sizing_y() > box_sizing_content;
    auto box_sizing = cs->box_sizing_y();


    if (box_sizing >= box_sizing_padding) {
      ld->inner_padding_width.s.y = pixels(cs->padding_width[1]);
      ld->inner_padding_width.e.y = pixels(cs->padding_width[3]);
      ld->padding_width.s.y = 0;
      ld->padding_width.e.y = 0;
      mv.ip1 = max_pixels(cs->padding_width[1]);
      mv.ip2 = max_pixels(cs->padding_width[3]);
      mv.p1 = int_v(0);
      mv.p2 = int_v(0);
    }
    else {
      ld->padding_width.s.y = pixels(cs->padding_width[1]);
      ld->padding_width.e.y = pixels(cs->padding_width[3]);
      ld->inner_padding_width.s.y = 0;
      ld->inner_padding_width.e.y = 0;
      mv.p1 = max_pixels(cs->padding_width[1]);
      mv.p2 = max_pixels(cs->padding_width[3]);
      mv.ip1 = int_v(0);
      mv.ip2 = int_v(0);
    }

    if (box_sizing >= box_sizing_border) {
      auto t1 = cs->inner_used_border_width(1);
      auto t2 = cs->inner_used_border_width(3);
      ld->inner_border_width.s.y = pixels(t1);
      ld->inner_border_width.e.y = pixels(t2);
      ld->border_width.s.y = 0;
      ld->border_width.e.y = 0;
      mv.ib1 = max_pixels(t1);
      mv.ib2 = max_pixels(t2);
      mv.b1 = int_v(0);
      mv.b2 = int_v(0);
    }
    else {
      auto t1 = cs->used_border_width(1);
      auto t2 = cs->used_border_width(3);
      ld->border_width.s.y = pixels(t1);
      ld->border_width.e.y = pixels(t2);
      ld->inner_border_width.s.y = 0;
      ld->inner_border_width.e.y = 0;
      mv.b1 = max_pixels(t1);
      mv.b2 = max_pixels(t2);
      mv.ib1 = int_v(0);
      mv.ib2 = int_v(0);
    }

    return mv;
  }

  int premeasure(view &v, helement b, hstyle cs, size container_dim) {
    b->check_layout(v);

    h_layout_data ld = b->ldata;

#ifdef _DEBUG
    if (b->is_id_test())
      ld = ld;
#endif

    element::max_values mx = b->measure_borders_x(v, container_dim);
    element::max_values my = b->measure_borders_y(v, container_dim);

    if (b->fix_positioned(v)) container_dim = b->doc()->dim();

    int width_spring = cs->width.flex1000();

    size dim(0, 0);

    int_v max_w;
    int   min_w = 0;

    int  phase = 1;
    bool needed_intrinsic = false;

    auto min_content = [&]() -> int {
      if (ld->dim_min.x.is_undefined()) {
        if (phase == 1)
          needed_intrinsic = true;
        else
          b->calc_intrinsic_widths(v);
      }
      return ld->dim_min.x;
      //return b->min_width(v, container_dim.x); ????
    };
    auto max_content = [&]() -> int {
      if (ld->dim_max.x.is_undefined()) {
        if (phase == 1)
          needed_intrinsic = true;
        else
          b->calc_intrinsic_widths(v);
      }
      return ld->dim_max.x;
      //return b->max_width(v).val(ld->dim_max.x); ????
    };

    {
      auto_state<int> _(ld->dim.x);

      for (; phase <= 2; ++phase) {
        if (b->airborn && b->airborn->dim.x.is_defined())
          dim.x = b->airborn->dim.x;
        else if (cs->width.undefined_or_auto()) {
        WIDTH_AUTO:
          if (cs->width.is_auto() && get_auto_width(v, b, dim.x)) {
            //????? min_w = dim.x;
            // assert(min_w < 20000);
            if (cs->display == display_block || cs->display == display_list_item) {
              max_w = dim.x;
              dim.x = min_w = 0;
              width_spring = 1000;
            }
          } else if (cs->position > position_relative || cs->get_float() ) {
            // assert(ld->dim_max.x.is_defined());
            dim.x = max_content();
            max_w = max(dim.x,
                        container_dim.x - ld->outer_left() - ld->outer_right());
          } else if (cs->display == display_inline_block) {
            dim.x = max_content();
            max_w = max(dim.x,
                        container_dim.x - ld->outer_left() - ld->outer_right());
          } else if (cs->display == display_block ||
                     cs->display == display_list_item) {
            width_spring = 1000;
            dim.x        = 0;
          } else if (cs->display == display_table_cell) {
            width_spring = 1000;
            max_w = min(max_content(), container_dim.x - ld->borpad_left() -
                                           ld->borpad_right());
            min_w = min_content();
            //assert(min_w < 20000);
            dim.x = 0;
          } else if (cs->display == display_table) {
            width_spring = 1000;
            max_w        = max_content();
            min_w        = min_content();
            //assert(min_w < 20000);
          } else
            dim.x = 0;

        } else if (cs->width.is_min_intrinsic()) {
          dim.x = min_content();
          assert(min_w < 20000);
        } else if (cs->width.is_max_intrinsic()) {
          // assert(ld->dim_max.x.is_defined());
          dim.x = max_content();
          assert(min_w < 20000);
        } else if (cs->width.is_spring()) {
          width_spring = cs->width.flex1000();
          if (cs->get_float() )
            max_w = max_content() - ld->outer_left() - ld->outer_right();
          dim.x = 0;
        } else if (cs->width.is_percent()) {
          if(container_dim.x)
            dim.x = pixels(v, b, cs->width, container_dim).width();
          else 
            goto WIDTH_AUTO;

        } else {
          dim.x = pixels(v, b, cs->width).width();
        }
        if (!needed_intrinsic)
          break; // don't need calc_intrinsic_widths, done
                 // else set precalculated dim.x and do this again, so
                 // calc_intrinsic_width() can calc %'s properly
        ld->dim.x = dim.x;
      }
      phase = 2;
    }

    if (cs->min_width.is_undefined()) {
      if (cs->overflow_x == overflow_none) min_w = min_content();
    } else if (cs->min_width.is_auto() && get_auto_width(v, b, min_w)) {
      min_w = min_w;
      // min_w = min_content();
      assert(min_w < 20000);
    } else if (cs->min_width.is_auto() || cs->min_width.is_min_intrinsic()) {
      min_w = min_content();
      assert(min_w < 20000);
    } else if (cs->min_width.is_max_intrinsic()) {
      min_w = max_content();
      assert(min_w < 20000);
    } else if (cs->min_width.is_percent()) {
      //int_v t = get_parent_declared_width(v, b);
      if (container_dim.x)
        min_w   = pixels(v, b, cs->min_width,container_dim).width();
      assert(min_w < 20000);
    } else if (cs->min_width.is_defined()) {
      min_w = pixels(v, b, cs->min_width,container_dim).width();
      assert(min_w < 20000);
    }
    if (cs->max_width.is_undefined()) {
      if (cs->get_float()  /* || cs->position >= position_absolute*/)
        max_w = min(max(dim.x, max_content()), // shrink-to-fit kind of thing...
                    container_dim.x - ld->outer_left() - ld->outer_right());
    } else if (cs->max_width.is_auto()) {
      int t;
      if (get_auto_width(v, b, t))
        max_w = t;
      else if (cs->display != display_table && !cs->get_float()  &&  !cs->margin[0].is_spring() && !cs->margin[2].is_spring())
        max_w = container_dim.x - ld->outer_left() - ld->outer_right();
      else
        max_w = max_content();
    } else if (cs->max_width.is_max_intrinsic()) {
      max_w = max_content();
      // if(max_w == 0) {
      //  dim.x = 0;
      //  width_spring = 0;
      //}
    } else if (cs->max_width.is_min_intrinsic()) {
      max_w = min_content();
      // if(max_w == 0) {
      //  dim.x = 0;
      //  width_spring = 0;
      //}
    } else if (cs->max_width.is_percent()) {
      //int_v t = get_parent_declared_width(v, b);
      //if (t.is_defined())
      //  max_w = cs->max_width.pixels_width(v, b, t);
      //else if (container_dim.x)
      //  max_w = container_dim.x;
      if(container_dim.x)
        max_w = pixels(v, b, cs->max_width,container_dim).width();
    } else
      max_w = pixels(v, b, cs->max_width, container_dim).width();

    /* ???????????????????????????????????
    http://www.w3.org/TR/CSS21/visudet.html#min-max-widths
    none w h
    w > max-width   max-width   max(max-width * h/w, min-height)
    w < min-width   min-width   min(min-width * h/w, max-height)
    h > max-height  max(max-height * w/h, min-width) max-height
    h < min-height  min(min-height * w/h, max-width) min-height
    (w > max-width) and (h > max-height), where (max-width/w ? max-height/h)
    max-width max(min-height, max-width * h/w) (w > max-width) and (h >
    max-height), where (max-width/w > max-height/h) max(min-width, max-height *
    w/h) max-height (w < min-width) and (h < min-height), where (min-width/w ?
    min-height/h) min(max-width, min-height * w/h) min-height (w < min-width)
    and (h < min-height), where (min-width/w > min-height/h) min-width
    min(max-height, min-width * h/w) (w < min-width) and (h > max-height)
    min-width max-height (w > max-width) and (h < min-height) max-width
    min-height
    */

    dim.x = limit(dim.x, min_w, max_w.val(INT_MAX));
    // dim.y = computed_height(v); //cs->height.pixels( cs->font_size,
    // container_height);

    flow_e pflow = flow_default;

    if (element *owner = b->get_owner())
      pflow = owner->layout_type();

    if ((cs->display == display_inline ||
         cs->display == display_inline_block) &&
        pflow == flow_text)
      return dim.x;

    flex::engine se;

#if _DEBUG
    if (b->is_id_test())
      b = b;
#endif

    if (pflow == flow_default || pflow == flow_vertical || pflow == flow_stack) {
      se.add(ld->margin_width.s.x, mx.m1, cs->margin[0].flex1000());
      se.add(ld->border_width.s.x, mx.b1, cs->border_width[0].flex1000());
      se.add(ld->padding_width.s.x, mx.p1, cs->used_padding(0).flex1000());
      se.add(dim.x, max_w, width_spring);
      se.add(ld->padding_width.e.x, mx.p2, cs->used_padding(2).flex1000());
      se.add(ld->border_width.e.x, mx.b2, cs->border_width[2].flex1000());
      se.add(ld->margin_width.e.x, mx.m2, cs->margin[2].flex1000());
    } else {
      se.add(ld->margin_width.s.x);
      se.add(ld->border_width.s.x);
      se.add(ld->padding_width.s.x);
      se.add(dim.x, max_w, width_spring);
      se.add(ld->padding_width.e.x);
      se.add(ld->border_width.e.x);
      se.add(ld->margin_width.e.x);
    }

    if (se.ptotal) {
      int n = 0;
      se.calc(container_dim.x,false);
      ld->margin_width.s.x  = se.val(n++);
      ld->border_width.s.x  = se.val(n++);
      ld->padding_width.s.x = se.val(n++);
      dim.x                      = se.val(n++);
      ld->padding_width.e.x = se.val(n++);
      ld->border_width.e.x  = se.val(n++);
      ld->margin_width.e.x  = se.val(n++);
    }

    /*if( cs->want_scrollbar_x() )
    {
    bool is_rtl = cs->direction == direction_rtl;
    range r(0, ld->dim_min.x - 1 );
    ld->sb.set_h(r,dim.x, cs->overflow_x == overflow_scroll,is_rtl);
    }*/

    if (cs->transforms) {
      if (ld->xctx)
        ld->xctx->clear();
      else
        ld->xctx = new trans_ctx();
      cs->transforms->apply(v, b, ld->xctx->mtx);
    }

    //assert(dim.x < 20000); // such measured width looks suspicious.

    return dim.x;
  }

  /*
  The top margin of an in-flow block-level element
  is adjoining to its first in-flow block-level child's top margin
  if the element has no top border, no top padding, and the child has no
  clearance. http://www.w3.org/TR/CSS21/box.html#collapsing-margins
  */
  void calc_margin_top(view &v, helement b, int parent_height, int &pix,
                       int &spr, int &pref, bool for_self) {
    style *cs = b->get_style(v);

#ifdef _DEBUG
    if (b->is_id_test()) b = b;
      // if (b->tag == tag::T_P)
      //  b = b;
      // if (b->tag == tag::T_DIV)
      //  b = b;
#endif

    // bool is_first_child = b->prev_ui_element() == 0;

    pix = pixels(v, b, cs->margin[1]).height();
    spr = cs->margin[1].flex1000();

    if (b->tag == tag::T_BODY || b->is_document()) { return; }

      /*    if (!is_first_child
            || !for_self
            || (b->parent && !collapse_top_container(v, b->parent)))
            pix = cs->margin[1].pixels_height(v, b, parent_height);
          spr = cs->margin[1].flex1000();*/

#if defined(FLEX_PLUS)
    pref = cs->margin[1].preferred().pixels(cs->font_size, parent_height, true,
                                            &v);
#else
    pref = 0;
#endif

    element *first_ui_child = b->first_ui_element();

    if (first_ui_child && collapse_top_container(v, b) &&
        collapse_top_child(v, first_ui_child) && cs->flow == flow_default &&
        first_ui_child->get_style(v)->clears == 0) {
      int fpix  = 0;
      int fspr  = 0;
      int fpref = 0;
      calc_margin_top(v, first_ui_child, b->ldata->dim.y, fpix, fspr, fpref,
                      false);
      pix = overlap(fpix, pix);
    }
  }

  void calc_margin_bottom(view &v, helement b, int parent_height, int &pix,
                          int &spr, int &pref, bool for_self) {
    style *cs = b->get_style(v);

    // bool is_last_child = b->next_ui_element() == 0;

#ifdef _DEBUG
    if (b->is_id_test()) b = b;
    if (b->is_document()) b = b;

      // if (b->tag == tag::T_P)
      //  b = b;
      // if (b->tag == tag::T_DIV)
      //  b = b;
#endif

    pix = pixels(v, b, cs->margin[3]).height();
    spr = cs->margin[3].flex1000();

    if (b->tag == tag::T_BODY || b->is_document()) return;

      /*    if (!is_last_child
            || !for_self
            || b->tag == tag::T_BODY
            || (b->parent && !collapse_bottom_container(v, b->parent)))
            pix = cs->margin[3].pixels_height(v, b, parent_height);
          spr = cs->margin[3].flex1000();*/

#if defined(FLEX_PLUS)
    pref = cs->margin[3].preferred().pixels(cs->font_size, parent_height, true,
                                            &v);
#else
    pref = 0;
#endif

    element *last_ui_child = b->last_ui_element();
    if (last_ui_child && collapse_bottom_container(v, b) &&
        collapse_bottom_child(v, last_ui_child) && cs->flow == flow_default &&
        last_ui_child->get_style(v)->clears == 0) {
      int fpix  = 0;
      int fspr  = 0;
      int fpref = 0;
      calc_margin_bottom(v, last_ui_child, b->ldata->dim.y, fpix, fspr, fpref,
                         false);
      pix = overlap(fpix, pix);
    }
  }

  void overlapping_y(view &v, helement self, element *prev_blk, element *blk, int &pix, int &spr) {
    int ppix  = 0;
    int npix  = 0;
    int pspr  = 0;
    int nspr  = 0;
    int ppref = 0;
    int npref = 0;

    if (prev_blk) {
      calc_margin_bottom(v, prev_blk, self->ldata->dim.y, ppix, pspr, ppref);
      // if (!blk && ppix > 0 && self->is_BFC(v))
      //  ppix = 0; // last element in BFC - suppress its margins
    }
    if (blk) {
      calc_margin_top(v, blk, self->ldata->dim.y, npix, nspr, npref);
      // if (!prev_blk && npix > 0 && self->is_BFC(v))
      //  npix = 0; // first element in BFC - suppress its margins
    }
    if (!prev_blk && !blk) return;

    spr  = max(pspr, nspr);
    pix  = overlap(ppix, npix);
    //pref = max(ppref, npref);

    if (spr == 0) {
      if (prev_blk == 0 && pix > 0 && collapse_top_container(v, self) &&
          collapse_top_child(v, blk)) {
        // collapsed
        pix  = 0;
        spr  = 0;
        //pref = 0;
        return;
      }

      if (blk == 0 && pix > 0 && collapse_bottom_container(v, self) &&
          collapse_bottom_child(v, prev_blk)) {
        // collapsed
        pix  = 0;
        spr  = 0;
        //pref = 0;
        return;
      }
      if (prev_blk && blk) {
        if (blk->c_style->margin[1].is_auto())
          pix = overlap(pix, pixels(v, blk, size_v(12, size_v::unit_type::pt)).height());
        if (prev_blk->c_style->margin[3].is_auto())
          pix = overlap(pix, pixels(v, prev_blk, size_v(12, size_v::unit_type::pt)).height());
      }
    }

    if (self->c_style->border_spacing_y.is_defined() && prev_blk && blk) {
      int bpix, bspr;
      self->c_style->border_spacing_y.pixels_n_spring_h(v, self, self->ldata->dim.y, bpix, bspr);
      pix  = max(pix, bpix);
      spr  = max(spr, bspr);
      //pref = max(pref, bpref);
    }
  }

  void block::calc_intrinsic_widths(view &v) {
    hstyle              cs = get_style(v);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

#ifdef _DEBUG
    //if (is_id_test()) 
    //  cs = cs;
#endif

    if (ld->dim_min.x.is_defined() && ld->dim_max.x.is_defined())
      return; // already measured
     
    ld->dim_min.y.clear();
    ld->dim_max.y.clear();

    auto_state<helement> BFC(v.bfc, is_floats_container(v) ? this : v.bfc);

    int subs_size = ld->blocks.size();

    inc_max<int> miw;
    inc_max<int> maw;
    // int    v_springs = 0;

    for (int i = 0; i < subs_size; ++i) {
      if (subs_size != ld->blocks.size()) {
        miw.reset();
        maw.reset();
        i  = -1;
        subs_size = ld->blocks.size();
        continue;
      }

      helement b = ld->blocks[i];

      b->check_layout(v);

      hstyle bcs = b->get_style(v);

      if (bcs->is_display_none()) continue;

      h_layout_data bld = b->ldata;

      premeasure(v, b, bcs, ld->dim);

      if (b->oof_positioned(v) /* || b->floats(v)*/) continue;

      int outer = b->outer_int_x_extra(v, ld->dim.x);

      int vmin = b->min_width(v) + outer;
      int_v vmax = b->max_width(v);
      if (vmax.is_undefined()) 
        vmax = b->max_content_width(v);
      vmax = vmax.val(0) + outer;

      miw <<= vmin;
      maw <<= vmax;
    }
    if (ld->fctx) {
      int fmiw = ld->fctx->get_min_width(v);
      miw <<= fmiw;
      int fmaw = ld->fctx->get_max_width(v);
      maw <<= fmaw; //?
    }
    maw <<= miw;

    int extra_x = 0;

    if (cs->box_sizing_x() > box_sizing_content || cs->box_sizing_y() > box_sizing_content)
    {
      size container_dim = parent ? parent->dim() : v.client_dim();
      measure_borders_x(v, container_dim);
      measure_borders_y(v, container_dim);
      extra_x = ld->inner_borpad_x();
    }

    ld->dim_min.x = miw + extra_x;
    ld->dim_max.x = maw + extra_x;
  }

  int block::layout_width(view &v, int width) {

    hstyle              cs = get_style(v);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

#ifdef _DEBUG
//    if (is_id_test())
//      cs = cs;
#endif

    bool is_default_vertical_layout = this->is_of_type<block>();

    bool dim_changed = ld->used_dim.x != width;
    ld->dim.x        = width;

    rect crect = client_rect(v);  
    crect <<= ld->inner_border_width;
    crect <<= ld->inner_padding_width;

    size inner = crect.size();
    // ????? if (cs->overflow_x < overflow_scroll) inner.x = max(inner.x, ld->dim_min.x);

    if (ld->dim_min.x.is_defined() && ld->dim_min.y.is_defined() &&
        ld->inner_dim.x == inner.x) {
      if (!dim_changed) return ld->dim_min.y; // already measured
      if (is_default_vertical_layout) {
        do_h_align(v);
        return ld->dim_min.y; // already measured
      }
    }

    ld->inner_dim.x = inner.x;
    ld->inner_dim.y = 0;
    // ld->dim.y = 0; // ?
    ld->dim_min.y.clear();
    ld->dim_max.y.clear();

    if (tag == tag::T_BR) {
      font *pf        = v.get_font(cs);
      int   ysz       = pf->height();
      ld->dim.y       = ysz;
      ld->inner_dim.y = ysz;
      ld->dim_max.y   = ysz;
      ld->dim_min.y   = ysz;
      return ysz;
    }

    if (ld->dim_min.x.is_undefined() || flags.has_percent_widths) 
      calc_intrinsic_widths(v);
    if (cs->overflow_x >= overflow_scroll)
      inner.x = max(inner.x, ld->dim_min.x);

    inc_max<int> real_miw(ld->dim_min.x);
    inc_max<int> real_mih(ld->dim_min.y);

    layout_width_start(v, width);

    int subs_size = ld->blocks.size();
    // bool setup_height = layout_type() == flow_default;

    int ysz = ld->inner_borpad_top();
    int ysz_min = 0;
    int ysz_max_delta = 0;
    
    int xpos = ld->inner_borpad_left();

    auto_state<helement> BFC(v.bfc, is_floats_container(v) ? this : v.bfc);

    if (ld->fctx) ld->fctx->reset(v);

    int margin_pix = 0, margin_spring = 0;
    // int    miw = 0;
    // int    maw = 0;
    int v_springs = 0;

    range xx(0, inner.x - 1);

    element *prevb = 0;

    for (int i = 0; i < subs_size; i++) {
      element *b = ld->blocks[i];

      hstyle bcs = b->get_style(v);

      if (bcs->is_display_none()) {
        // b->pos.x = b->content_left();
        // b->pos.y = ysz;
        continue;
      }

      h_layout_data bld = b->ldata;

      auto b_floats = b->floats(v);
      if (b_floats) {
        b->measure_borders_x(v, ld->dim);
        b->measure_borders_y(v, ld->dim);
        b->set_width(v, b->computed_width(v, width));
        b->set_height(v, b->computed_height(v, ld->dim.y));
        real_miw <<= b->min_width(v,width);
        int nexty = ysz;
        if (prevb) nexty += prevb->ldata->margin_width.bottom();
        if (b_floats == float_left)
          v.push_left(b, nexty, this);
        else
          v.push_right(b, nexty, this);
        continue;
      }

      int bwidth = premeasure(v, b, bcs, inner );

      if (bcs->position) {
        b->check_positioned_containment(v);
        if (bcs->position > position_relative) {
          b->set_pos(b->ldata->outer_left(), ysz + bld->outer_top());
          continue;
        }
      }
      /*if( block_outer_width > d->dim.x ) ???
      {
      if(block_outer_width > miw)
      miw = block_outer_width;
      }*/

      if (bcs->visibility == visibility_collapse) {
        b->set_width(v, bwidth);
        bld->dim.y = 0;
        continue;
      }

      if (b->popup_positioned(v)) // it must be here to prevent collapsed popups repositioning.
        continue;

      v_springs += bcs->v_flex1000();

      overlapping_y(v, this, prevb, b, margin_pix, margin_spring);

      ysz += margin_pix;

      // int clears = bcs->clears;
      // ysz = floats_container->get_next_y(ysz,clears,this);

      ysz += bld->borpad_top();

      b->set_y_pos(ysz);

      halign_e halign = cs->get_horizontal_align();

      {
        int width_to_align = inner.x - bwidth - bld->outer_left() - bld->outer_right();
        if (width_to_align < 0 && cs->overflow_x == overflow_none)
          width_to_align = 0;
        if (halign == align_left)
          b->set_x_pos(xpos + bld->outer_left());
        else if (halign == align_right)
          b->set_x_pos(xpos + bld->outer_left() + width_to_align);
        else if (halign == align_center)
          b->set_x_pos(xpos + bld->outer_left() + width_to_align / 2);
      }

      b->set_width(v, bwidth);
      real_miw <<= b->min_width(v, width);
      int h = b->computed_height(v, 0);
      ysz += h;
      int mh = b->max_height(v);
      ysz_max_delta += max(0, mh - h);

      ysz += bld->borpad_bottom();
      prevb = b;

      if (is_default_vertical_layout && v_springs == 0)
        b->set_height(v, h);
      else
        bld->dim.y = h; // this will be recalculated by this->layout_height
    }

    if (prevb && !prevb->get_style(v)->is_display_none()) //
    {
#ifdef _DEBUG
      if (prevb->is_id_test()) prevb = prevb;
#endif
      overlapping_y(v, this, prevb, 0, margin_pix, margin_spring);
      ysz += margin_pix;
    }

    ysz += ld->inner_border_width.e.y + ld->inner_padding_width.e.y;

    ysz_min += ysz;

    if (ld->fctx && !ld->fctx->is_empty())
      ysz = max(ysz, ld->fctx->get_max_y(v));

    ld->dim_max.y = ld->dim_min.y = ysz;
    ld->dim_max.y = ld->dim_max.y + ysz_max_delta;
    ld->content_dim.y = ysz;

    //do_h_align(v); WRONG, see b->set_x_pos() above

    ld->can_v_flex = false;
    if (v_springs
        /*&& cs->flow == flow_default
      && !is_document()
      && is_of_type<block>()*/) {
      element *bfc = v.get_bfc(this);
      if (!bfc || !bfc->ldata->fctx || bfc->ldata->fctx->is_empty()) {
        // we've got vertical flexes in children and no floats around so
        // we can switch the element to flow:vertical
        // WRONG, WILL BREAK classify()! ::new(this) block_vertical(NO_INIT());
        ld->can_v_flex = true;
      }
    }

    return (ld->content_dim.x = real_miw);
  }

  int block::layout_height_std(view &v, int height) {
    hstyle       cs  = get_style(v);
#ifdef _DEBUG
    if (is_id_test())
      cs = cs;
#endif
    layout_data *pld = ldata.ptr_of<layout_data>();
    pld->dim.y       = height;
    size inner_dim   = client_rect(v).size();
    if (pld->inner_dim.y == inner_dim.y) return pld->dim.x;

    pld->inner_dim.y = inner_dim.y;

    do_v_align(v);

    return pld->dim.x;
  }

  void block::content_x_range_at(view &v, range y, element *of,
                                 range &in_out_xx) {
    hstyle              cs = get_style(v);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    if (!ld->fctx || ld->fctx->is_empty()) return;
    point offset = of->rel_pos(v, this);
    y += offset.y;
    range xx = ld->fctx->get_space_at(v, y, of);
    xx -= offset.x;
    xx &= of->content_box(v).x();
    in_out_xx &= xx;
  }

  element *block::find_child_element_generic(view &v, gool::point pos,
                                             bool exact) {
    get_style(v);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    if (exact)
      for (int i = ld->blocks.last_index(); i >= 0; --i) {
        element *b = ld->blocks[i];
        if (!b) continue;
        if (b->state.popup()) // it is handled already for the popups
          continue;
        if (b->state.moving() || b->state.copying())
          continue; // this is dragging block, skip it
        if (!b->floats(v) && !b->positioned(v) && b->is_visible(v)) {
          point    p  = b->pos();
          element *tb = b->find_element(v, pos - p, exact);
          if (tb) return tb;
        }
      }
    else // not exact
    {
      int      min_distance = limits<int>::max_value();
      element *closest      = 0;
      for (int i = ld->blocks.last_index(); i >= 0; --i) {
        element *b = ld->blocks[i];
        if (!b) continue;
        if (b->state.popup()) // it is handled already for the popups
          continue;
        if (b->state.moving() || b->state.copying())
          continue; // this is dragging block, skip it
        if (!b->floats(v) && !b->positioned(v) && b->is_visible(v)) {
          rect mb   = b->margin_box(v, TO_PARENT);
          int  dist = distance(mb, pos);
          if (dist < min_distance) {
            min_distance = dist;
            closest      = b;
          }
        }
      }
      if (closest) {
        point p = closest->pos();
        return closest->find_element(v, pos - p, exact);
      }
    }
    return 0;
  }

  struct find_pos_env_v // find_element traits for binary search
  {
    view &          v;
    point           pt;
    slice<helement> elements;

    find_pos_env_v(view &v_, slice<helement> elems, point pt_)
        : v(v_), pt(pt_), elements(elems) {}

    int total() const { return int(elements.length); }

    bool is_less(int n) {
      element *b  = elements[n];
      rect     sp = b->hit_box(v, element::TO_PARENT);
      return sp.bottom() < pt.y;
    }
    bool is_equal(int n) {
      element *b  = elements[n];
      rect     sp = b->hit_box(v, element::TO_PARENT);
      return sp.y().contains(pt.y);
    }
    bool is_comparable(int n) {
      element *b = elements[n];
      if (b->state.popup()) // it is handled already for the popups
        return false;
      if (b->state.moving() || b->state.copying())
        return false; // this is dragging element, skip it
                      // return b->c_style->get_float()  == 0 && b->c_style->position
                      // == 0 && !b->c_style->collapsed();
      return !b->floats(v) && !b->positioned(v) && b->is_it_visible(v);
    }
    NONCOPYABLE(find_pos_env_v)
  };

  struct find_closest_env_v // draw_content traits for binary search
  {
    view &          v;
    point           pt;
    slice<helement> elements;
    helement        exact;
    bool            to_view_space;

    find_closest_env_v(view &v_, slice<helement> elems, point pt_, bool to_view_space_)
        : v(v_), pt(pt_), elements(elems), to_view_space(to_view_space_) {}

    int total() const { return int(elements.length); }

    bool is_less(int n) {
      element *b = elements[n];
      rect     sp = b->hit_box(v, to_view_space ? element::TO_VIEW : element::TO_PARENT);
      if (sp.y().contains(pt.y)) exact = b;
      // return sp.pointOf(5).y < pt.y;
      return sp.bottom() < pt.y;
    }
    bool is_equal(int n) { return true; }
    bool is_comparable(int n) {
      element *b = elements[n];
      if (b->state.popup()) // it is handled already for the popups
        return false;
      // if( b->state.moving() || b->state.copying())
      //   return false; // this is dragging element, skip it
      return !b->floats(v) && !b->positioned(v) && b->is_it_visible(v);
    }
    NONCOPYABLE(find_closest_env_v)
  };

  element *block::find_child_element_vertical(view &v, gool::point pos,
                                              bool exact) {
    if (flags.has_transformed) return find_child_element_generic(v, pos, exact);

    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    int                 n  = -1;
    if (exact) {
      find_pos_env_v e(v, ld->blocks(), pos);
      n = bsearch(e);
      if (n >= 0) {
        element *ce = ld->blocks[n];
        point    p  = ce->pos();
        rect     r  = ce->hit_box(v) + p;
        if (r.contains(pos)) return ce->find_element(v, pos - p, exact);
      }
    } else {
      find_closest_env_v e(v, ld->blocks(), pos, false);
      n = bsearch(e);
      if (e.exact) {
        point p = e.exact->pos();
        return e.exact->find_element(v, pos - p, exact);
      }
      if (n >= 0) {
        element *ce = ld->blocks[n];
        point    p  = ce->pos();
        return ce->find_element(v, pos - p, exact);
      }
    }
    return 0;
  }

  bookmark block::find_text_pos(view &v, point zpos) {
    element *cel = find_element(v, zpos, false);
#ifdef _DEBUG
    assert(cel);
#endif
    while (cel && cel != this) {
      if (cel->is_block_element(v))
        return cel->find_text_pos(v, zpos - cel->rel_pos(v, this));
      cel = cel->get_owner();
    }

    // not found
    element *first = this->first_ui_element();
    if (first && zpos.y < first->hit_box(v, element::TO_PARENT).top()) {
      bookmark t = first->find_text_pos(v, zpos - first->pos());
      if (t.valid()) return t;
    }
    element *last = this->last_ui_element();
    if (last && zpos.y > last->hit_box(v, element::TO_PARENT).bottom()) {
      bookmark t = last->find_text_pos(v, zpos - last->pos());
      if (t.valid()) return t;
    }

    // return this_pos( zpos.x > ldata->dim.x / 2);
    // return bookmark( this, zpos.y > ldata->dim.y  );
    rect hb = this->hit_box(v);
    return bookmark(this, 0, zpos.x >= hb.pointOf(5).x);
  }

  rect clip_rect(view &v, element *b) {

    clip_box_e cb = b->get_style(v)->clip_box.val();

    rect cr;
    switch (cb) {
      default: cr = b->client_rect(v); break;
      case clip_box_content: cr = b->content_box(v); break;
      case clip_box_padding: cr = b->padding_box(v); break;
      case clip_box_border: cr = b->border_box(v); break;
      case clip_box_margin: cr = b->margin_box(v); break;
      case clip_box_hit_margin: cr = b->hit_box(v); break;
    }

    if (b == v.doc()) return cr;

    rect r = b->border_box(v); // big enough?

    hstyle cs = b->get_style(v);

    if (cs->overflow_x <= overflow_visible) {
      int w = v.dimension().x;
      r.s.x -= w;
      r.e.x += w;
    }
    else if (cs->overflow_x >= overflow_hidden) {
      // r = cr;
      r.s.x = cr.s.x;
      r.e.x = cr.e.x;
    }

    if (cs->overflow_y <= overflow_visible) {
      int h = v.dimension().y;
      r.s.y -= 10 * h;
      r.e.y += 10 * h;
    }
    else if (cs->overflow_y >= overflow_hidden) {
      // r = cr;
      r.s.y = cr.s.y;
      r.e.y = cr.e.y;
    }
    return r;
  }

  rect element::clip_box(view &v, RELATIVE_TO rt)
  {
    return clip_rect(v, this) + content_box(v, rt).s;
  }

  void element::draw_selection(view &v, graphics *pg, point pos,
                               selection_ctx *psi) {
    if (psi && psi->is_valid() && psi->caret.node == this &&
        psi->get_selection_type(v) == ONLY_CARET) {
      caret_metrics cm;
      cm.at_end = psi->caret.at_element_end();
      cm.elem   = this;
      cm.ctype  = BLOCK_POSITION;

      int sz      = pixels(v, this, c_style->font_size).height();
      cm.glyph_rc = size(sz, sz);
      rect box    = this->border_box(v);
      box += pos;

      if (cm.at_end)
        cm.glyph_rc.pointOf(1, box.pointOf(1));
      else
        cm.glyph_rc.pointOf(7, box.pointOf(7));

      cm.x1 = float(cm.glyph_rc.s.x);
      cm.x2 = float(cm.glyph_rc.e.x);
      cm.y1 = cm.glyph_rc.s.y;
      cm.y2 = cm.glyph_rc.e.y;

      psi->draw_caret(v, pg, cm);
    }
  }

  void block::draw_content(view &v, graphics *pg, point pos, bool clip) {
#ifdef DEBUG
    if (is_id_test())
      pg = pg;
#endif
    draw_content_vertical(v, pg, pos, clip);
    if (v._selection_ctx) draw_selection(v, pg, pos, v._selection_ctx);
  }

  void block::draw_content_generic(view &v, graphics *pg, point pos,
                                   bool clip) {
    handle<layout_data> pld = ldata.ptr_of<layout_data>();

    if (pld->zctx.is_empty() && pld->blocks.size() == 0) 
      return;

    rect clip_area = clip_rect(v, this);
    if (clip_area.empty() && clip) return;

    // rect vr(-v.screen_pos(), v.screen_dim());

    bool  clip_overflow = c_style->clip_overflow();
    point client_offset = client_rect(v).s;
    clip_area += pos;

    clipper _(pg, clip_area, clip_overflow && clip, !c_style->has_solid_background());

    pos = scroll_translate(v, pos);
    pos += client_offset;

    pld->zctx.draw(v, pg, pos, this, false);

    for (int i = 0; i < pld->blocks.size(); ++i) {
      helement b  = pld->blocks[i];
      if (!b) 
        break;
      hstyle   cs = b->get_style(v);
      //if (!cs->display || cs->visibility != visibility_visible) continue;
      if (cs->position > position_static) continue;
      if (b->airborn) continue;
      if (b->state.popup()) continue;
      if (b->floats(v)) continue;
      rect r = b->rendering_box(v, TO_PARENT) + pos;
      if (r.overlaps_with(clip_area)) 
        b->draw(v, pg, b->pos() + pos);
    }

    if (pg->has_overlays(_)) draw_outlines(v, pg, pos, true, false);

    pld->zctx.draw(v, pg, pos, this, true);
  }

  point rendering_pos(view &v, element *self, point pos) {
    element *w = self->get_windowed_container(v, true);
    if (!w) return point(0, 0); // stop it

    iwindow *pwnd = w->get_window(v, true);
    if (!pwnd) return point(0, 0);

    return pos - w->view_pos(v);
  }

  /*bool drawing_under_transforms(view &v, element *el) {

    while (el) {
      if (el->get_style(v)->transforms) return true;
      el = el->get_owner();
    }
    return false;
  }*/

  void block::draw_content_vertical(view &v, graphics *pg, point pos,
                                    bool clip) {
    
    handle<layout_data> pld = ldata.ptr_of<layout_data>();

    if (pld->blocks.size() == 0 && pld->zctx.is_empty()) return;

    rect clip_area = clip_rect(v, this);
    if (clip_area.empty() && clip) return;

    rect z_clip_area = clip_area;

    bool  clip_overflow = c_style->clip_overflow();
    point client_offset = client_rect(v).s;
    clip_area += pos;

    rect rendering_rc = rect(v.client_dim()) | pg->get_clip_rc();
    if (rendering_rc.empty()) return;

//#ifdef _DEBUG
//    if (c_style->has_solid_background()) {
//      dbg_report("non-tran");
//      c_style->has_solid_background();
//    }
//#endif // _DEBUG
   
    clipper _(pg, clip_area, clip_overflow && clip, !c_style->has_solid_background());

    pos = scroll_translate(v, pos);
    pos += client_offset;

    bool under_transforms = !pg->transform().is_identity(); //drawing_under_transforms(v, this);

    int fn = 0;

    if (!under_transforms) {
      point rpos = view_pos(v) + z_clip_area.s;
      find_closest_env_v e(v, pld->blocks(), rpos, true);
      fn = bsearch(e); // binary search of first visible element.
                       // we employ the fact that elements are strictly
                       // one-by-one in static layouts.
    }

    if (fn >= 0)
      for (int i = fn; i < pld->blocks.size(); ++i) {
        helement b = pld->blocks[i];
        if (!b)
          break;

        hstyle cs = b->get_style(v);
        if (!cs->display) continue;
        if (cs->position > position_static) continue;
        if (b->state.popup()) continue;
        if (b->airborn) continue;
        if (b->floats(v)) continue;

        if (!under_transforms && !cs->transforms && !v.is_printing()) {
          // attempt to optimize rendereing - do not draw anything above or
          // below clip area.
          rect r = b->rendering_box(v, TO_VIEW);
          if (r.top() > rendering_rc.bottom()) break;
          if (r.bottom() < rendering_rc.top()) 
            continue;
        }
        b->draw(v, pg, b->pos() + pos);
      }

    if (pg->has_overlays(_)) 
      draw_outlines(v, pg, pos, true, false);


    pld->zctx.draw(v, pg, pos, this, true);
  }

  float block::inline_baseline(view &v) {
    // for( element* le = last_ui_element(); le; le = le->prev_ui_element() )
    if (c_style->overflow_y > overflow_visible)
      return float(ldata->outer_top() + ldata->dim.y);

    float_v bl;
    float_v bl_explicit;

    auto foo = [&bl, &bl_explicit, &v](element *le) -> bool {
      if (le->c_style->position > position_relative) return false;
      if (le->c_style->get_float()  > 0) return false;
      if (bl.is_undefined() && le->is_of_type<text_block>())
        bl = le->inline_baseline(v) - le->ldata->outer_top() + le->pos().y;
      // if( le->flags.baseline_principal )
      //{
      //  bl_explicit = le->inline_baseline() - le->ldata->outer_top() +
      //  le->pos().y; return true;
      //}
      return false;
    };

    /*struct
    {
    float& bl;
    bool operator()(element* le)
    {
    if( le->c_style->position > position_relative ) return false;
    if( le->c_style->get_float()  > 0 ) return false;
    bl = le->inline_baseline() - le->ldata->outer_top() + le->pos().y;
    return true;
    }
    } foo = {bl};*/

    // if(each_ui_child(foo))
    // return bl + ldata->outer_top();

    each_ui_child(foo);
    if (bl_explicit.is_defined()) return bl_explicit + ldata->outer_top();
    if (bl.is_defined()) return bl + ldata->outer_top();
    return super::inline_baseline(v);
  }

  static element *prev_el(block *b, int &node_idx) {
    for (--node_idx; node_idx >= 0; --node_idx) {
      if (b->nodes[node_idx]->is_element())
        return b->nodes[node_idx].ptr_of<element>();
    }
    return 0;
  }
  static element *next_el(block *b, int &node_idx) {
    for (/*++node_idx*/; node_idx < b->nodes.size(); ++node_idx) {
      if (b->nodes[node_idx]->is_element())
        return b->nodes[node_idx].ptr_of<element>();
    }
    return 0;
  }

  void block::get_inline_block_metrics(view &v, int &ascent, int &descent,
                                       int &height) {
    layout_data *pld = ldata.ptr_of<layout_data>();
    if (!pld->baseline.is_defined())
      return super::get_inline_block_metrics(v, ascent, descent, height);
    rect od = outer_distance(v);
    height = pld->dim.y + od.top() + od.bottom();
    ascent = pld->baseline + od.top();  //(pld->baseline * height) / pld->dim.y;
    descent = height - ascent;
  }

} // namespace html