#ifndef __html_dom_h__
#define __html_dom_h__

#pragma once

#include "html-primitives.h"
#include "html-event-target.h"

#include "tool/tool.h"
#include "gool/gool.h"
#include "html-symbols.h"
#include "html-style.h"
#include "html-pump.h"
#include "html-animator.h"
#include "html-behavior.h"
#include "html-window.h"

#include <functional>

// abstract DOM

namespace html {
  using namespace tool;
  using namespace gool;

  class view;
  class application;
  struct element;
  struct document;
  struct bookmark;
  class floats_ctx;
  struct selection_ctx;
  struct timer_def;

//#define BR wchar(0xE000) // <br> marker.
#define ZWSP wchar(0x200B)
#define THSP wchar(0x2009) // thin space
#define EMPTY_STRING_PLACEHOLDER THSP // thin space

  enum BACKWARD_E { BACKWARD };

  enum ADVANCE_DIR {
    ADVANCE_LEFT,              // cluster left
    ADVANCE_RIGHT,             // cluster right
    ADVANCE_NEXT,              // cluster next
    ADVANCE_PREV,              // cluster prev
    ADVANCE_UP,                // line up
    ADVANCE_DOWN,              // line down
    ADVANCE_WORD_START,        // single word start
    ADVANCE_WORD_END,          // single word right
    ADVANCE_HOME,              // front of line
    ADVANCE_END,               // back of line
    ADVANCE_FIRST,             // very first position - document start
    ADVANCE_LAST,              // very last position - document end
    ADVANCE_ABSOLUTE_LEADING,  // explicit position (for mouse click)
    ADVANCE_ABSOLUTE_TRAILING, // explicit position, trailing edge

    ADVANCE_NEXT_CHAR, // char next
    ADVANCE_PREV_CHAR, // char prev
    ADVANCE_NEXT_CARET, // caret pos next
    ADVANCE_PREV_CARET, // caret pos prev

  };

#define MAX_CAP 32000

  // struct node;

  // typedef function< bool(node* pn, int pos, DOM_POS_E pt) > scan_cb;

  struct emit_ctx;

  struct script_expando_interface {
    static inline chars interface_name() { return CHARS("!seo"); }
    virtual bool get_expando(context& hc, script_expando& seo) = 0;
  };
    
  #define UNDEFINED_NODE_INDEX uint(-1) // set on disconnected nodes
  #define SYNTHETIC_NODE_INDEX uint(-2) // set on synthetic elements (anonymous paragraphs, popups)

  struct node : public resource, public ext_resource
#if defined(SCITERJS)
    , public script_expando_interface 
#endif
  {
    node()
        : parent(nullptr), owner(nullptr), node_index(UNDEFINED_NODE_INDEX), uid(next_node_uid())
#if defined(SCITER)
      , obj(0)
#endif

    {}
    node(NO_INIT ni) : resource(ni), ext_resource(ni), parent(ni), owner(ni)
#if defined(SCITERJS)
      , obj(ni)
#endif
    {}

    DEFINE_TYPE_ID(html::node)

    virtual void finalize() {
      if (ext_get_ref_count() <= 0) delete this;
    }
    virtual void ext_finalize() {
      if (get_ref_count() <= 0) delete this;
    }
    
    // official Node.nodeType values:
    enum NODE_TYPE {
      UNKNOWN = 0,
      ELEMENT_NODE =	1,
      ATTRIBUTE_NODE = 2, // NA
      TEXT_NODE	= 3, 
      CDATA_SECTION_NODE = 4, //NA
      ENTITY_REFERENCE_NODE = 5, // NA
      ENTITY_NODE =	6, // NA
      PROCESSING_INSTRUCTION_NODE =	7, //NA
      COMMENT_NODE = 8, 
      DOCUMENT_NODE	= 9, 
      DOCUMENT_TYPE_NODE = 10,
      DOCUMENT_FRAGMENT_NODE = 11,
      NOTATION_NODE =	12,

      TEXT = TEXT_NODE,
      COMMENT = COMMENT_NODE,
      ELEMENT = ELEMENT_NODE,
      DOCUMENT = DOCUMENT_NODE,
      DOCUMENT_FRAGMENT = DOCUMENT_FRAGMENT_NODE,
    };

    static long next_node_uid() {
      if (last_uid <= 0) last_uid = 1000;
      return -(++last_uid);
    }

    static bool is_node_uid(long l) {
      return (l < 0) && (-l > 1000);
    }
    
    virtual NODE_TYPE node_type() const { return UNKNOWN; }
    virtual string  node_name() const { return string(); }
    virtual wchars  node_value() const { return wchars(); }
    virtual void    node_set_value(wchars text, view* pv = nullptr ) {}

    //virtual ustring node_get_text() = 0;
    //virtual void    node_set_text(wchars text) = 0;
    virtual string  node_def() const { return string(); }
    virtual bool    node_append(node *n) { return false; }
    virtual bool    node_contains(node* pn) { return false; }
    virtual bool    node_has_child_nodes() { return false; }
    virtual bool    node_is_equal(node* pn);
    virtual bool    node_is_same(node* pn) { return this == pn; }
    virtual ustring node_text() const { return ustring(); }
    virtual void    node_set_text(ustring t, view* pv = nullptr) { assert(false); }
    
    virtual bool is_text() const { return false; }
    virtual bool is_element() const { return false; }
    virtual bool is_document() const { return false; }
    virtual bool is_space() const { return false; }
    virtual bool is_comment() const { return false; }
    virtual bool is_it_visible(view &v) { return false; } // visibility of the element itslef
    virtual bool is_inline_span_element(view &v) { return false; }
    virtual bool is_inline_block_element(view &v) { return false; }

    // overriden by document fragment
    virtual bool is_document_fragment() const { return false; }
    virtual bool wraps_nodes() const { return false; }
    virtual bool unwrap_nodes(array<handle<node>>& wp)  { return false; }

    virtual rect rbox(view &v); // view relative rendering box

    virtual bool      is_display_none_node(view &v) { return true; }
    virtual bool      is_collapsed_node(view &v) { return false; }
    virtual view *    pview() const;
    virtual document *doc() const;

    virtual node *first_node() const { return nullptr; }
    virtual node *last_node() const { return nullptr; }
    virtual node *next_node() const; // siblings
    virtual node *prev_node() const; // siblings
    virtual node *root_node() const;

    virtual style * get_style(view &v);
    virtual ustring get_lang() const;
    virtual ustring get_theme() const;

    virtual uint marks_at(int pos) const { return 0; }
    virtual bool marks_at(view &v, int pos, uint mid) { return false; }
    virtual bool marks_at_contains(int pos, chars markcls) const { return false; }

    virtual void emit_text(array<wchar> &to, bool verbose = false) {}
    virtual void get_ui_text(view &v, array<wchar> &to) { emit_text(to); } // returns rendered text
    virtual void emit(ostream &os, emit_ctx *pectx = nullptr) = 0;

    virtual node *split(view &v, int at, bool force = false) = 0;
    virtual int   unsplit(view &v)                           = 0;

    virtual void got_parent(element *p, uint nidx);
    // is connected to the DOM
    virtual bool is_connected() const;

    element *    dom_parent() const { 
      return is_connected() ? parent.ptr() : nullptr; 
    }

    element *    logical_parent() const;
    
    template <class T> T *cast() { return static_cast<T *>(this); }

    template <class T> const T *cast() const {
      return static_cast<const T *>(this);
    }

    virtual node *get_node_by_uid(long uid_) const {
      return (uid == uid_) ? const_cast<node *>(this) : 0;
    }

    virtual node *clone() = 0;
    virtual void  detach(const element *p) {
      if (parent == p) {  
        parent = nullptr;
        owner = nullptr;
        node_index = UNDEFINED_NODE_INDEX;
#ifdef SCITERJS
        obj.clear();
#endif // SCITERJS
      }
      else {
        // elements <text> 
        owner = parent;
      }
    }
    virtual void detach_owner() { 
      owner = parent; 
    }

    void index_stack(array<int> &indexes, const element *root = 0) const;

    virtual element *parent_block(view &v);
    virtual element *owner_block(view &v) { return parent_block(v); }
    virtual element *get_element() { return parent; }
    virtual element *get_ui_element() { if (element* o = owner) return o; return parent; }
    virtual element *get_owner() const { if (element* o = owner) return o; return parent; }
    virtual element *nearest_box(bool ui = true);        // closest box or this
    virtual element *nearest_block(view &v);             // closest box or this
    virtual element *nearest_box_or_row(bool ui = true); // closest box or this
    virtual element *nearest_text_box();                 // closest box or this

    static element *find_common_parent(node *b1, node *b2);
    static element *find_common_owner(node *b1, node *b2);
    static node *   find_base(node *b1, node *b2);

    virtual bookmark this_pos(bool tail) const;
    virtual bookmark start_pos() const                              = 0;
    virtual bookmark end_pos() const                                = 0;
    virtual bool     advance_forward(bookmark &bm, wchar &c) const  = 0;
    virtual bool     advance_backward(bookmark &bm, wchar &c) const = 0;
    virtual bool     is_char_pos(view& v,const bookmark &bm) const { return false; }
    virtual bool     is_caret_pos(view& v, const bookmark &bm) const { return false; }
    virtual bool     is_start_pos(const bookmark &bm) const = 0;
    virtual bool     is_end_pos(const bookmark &bm) const   = 0;
    virtual wchar    char_code(const bookmark &bm) const { return 0; }
    virtual wchar    ui_char_code(const bookmark &bm) const { return 0; }

    virtual bookmark start_caret_pos(view &v) const;
    virtual bookmark end_caret_pos(view &v) const;
    virtual bool     advance_caret_pos(view &v, bookmark &bm, ADVANCE_DIR dir,
                                       array<wchar> *out = 0) = 0;

    virtual bool belongs_to(element *pb, bool include_this = false);
    virtual bool owned_by(element *pb, bool include_this = false);

    virtual bool remove(bool finalize, view *pv = 0);

    virtual void normalize(bookmark &bm) const = 0; // check if in range + bm.after_it
    virtual void denormalize(bookmark &bm) const; // if bm.after_it pos += 1

    virtual bool each_any_child_node(const function<bool(node *)> &f) {
      return false;
    }

    virtual void stray(view &v) // a.k.a. on_document_unload, element is about
                                // to be completely removed from the DOM, clear
                                // all runtime structures
    {
#ifdef SCITERJS
      obj.clear();
#endif
    }

    virtual bool a11y_is_visible(view &v) { return is_it_visible(v); }

#ifdef _DEBUG
    virtual void dbg_report(const char *cap = "") = 0;
#endif

#if defined(SCITERJS)

    //script_expando_interface
    virtual bool get_expando(context& hc, script_expando& seo) override;

    virtual bool get_interface(tool::chars name, void **pp) override
    {
      if (pp && name == script_expando_interface::interface_name()) {
        *pp = static_cast<script_expando_interface*>(this);
        return true;
      }
      return resource::get_interface(name, pp);
    }

#endif

    weak_handle<element>   parent;
    weak_handle<element>   owner;        // ~~ layout parent, most of the time it is parent but in some contexts (text_block) it can be different.
    // mutable weak_handle<document>  parent_document;  does not work reliably, detachment requires full tree scan, sigh. 
    uint                   node_index;
#if defined(SCITER)
    tis::value             obj;
#elif defined(SCITERJS)
    qjs::hvalue            obj;
#endif
    long                   uid;
    static locked::counter last_uid;
  };
  typedef handle<node> hnode;

  //typedef uint16             char_mark_t;
  typedef slice<char_mark_t> char_marks;

  char_mark_t  span_class_bit(const string &cls);
  char_mark_t  apply_span_class(char_mark_t id, slice<string> additions);
  char_mark_t  remove_span_class(char_mark_t id, slice<string> removals);
         bool  mark_id_contains_class(char_mark_t  id, chars cls);
 array<string> span_class_names(char_mark_t b);

  struct text : public node {

    DEFINE_TYPE_ID_DERIVED(text,node)

    text(wchars t);
    text(NO_INIT ni) : super(ni) {}

    virtual NODE_TYPE node_type() const override { return TEXT; }
    virtual string    node_name() const override { return string("#text"); }
    virtual wchars    node_value() const override { return chars(); }
    virtual void      node_set_value(wchars text, view* pv) override { set_text(text, pv); }
    virtual string    node_def() const override { return string("Text"); }
    virtual ustring   node_text() const override { return chars(); }
    virtual void      node_set_text(ustring t, view* pv = nullptr) override { set_text(t, pv); }
        
    virtual bool is_text() const { return true; }
    virtual bool is_space() const { return chars.size() == 0 || only_spaces(chars()); }

    virtual bool is_it_visible(view &v) {  return true; } // visibility of the element itslef
    virtual bool is_display_none_node(view &v) { return false; }

    virtual void  emit_text(array<wchar> &to, bool verbose = false) override { to.push(chars()); }
    virtual void  emit(ostream &os, emit_ctx *pectx = nullptr);
    virtual node *clone() { return new text(chars()); }

    virtual bookmark start_pos() const;
    virtual bookmark end_pos() const;

    virtual bool  advance_forward(bookmark &bm, wchar &c) const;
    virtual bool  advance_backward(bookmark &bm, wchar &c) const;
    virtual bool  is_caret_pos(view& v, const bookmark &bm) const;
    virtual bool  is_char_pos(view& v, const bookmark &bm) const;
    virtual bool  is_start_pos(const bookmark &bm) const;
    virtual bool  is_end_pos(const bookmark &bm) const;
    virtual wchar char_code(const bookmark &bm) const;
    virtual wchar ui_char_code(const bookmark &bm) const;

    virtual bool advance_caret_pos(view &v, bookmark &bm, ADVANCE_DIR dir,
                                   array<wchar> *out = 0);

    virtual void normalize(bookmark &bm) const;
    virtual void denormalize(bookmark &bm) const;

    virtual node *split(view &v, int at, bool force = false) override;
    virtual int   unsplit(view &v) override;

    //        void     set_mark(int start, int end, char_mark_t m) { marks.size(
    //        chars.size() ); for(int n = max(0,start); n <
    //        min(end,marks.size()); ++n) marks[n] |= m; } void
    //        reset_mark(int start, int end, char_mark_t m) { marks.size(
    //        chars.size() ); for(int n = max(0,start); n <
    //        min(end,marks.size()); ++n) marks[n] &= ~m; } void
    //        clear_marks() { marks.clear(); }
    virtual uint marks_at(int pos) const override {
      return pos >= 0 && pos < marks.size() ? marks[pos] : 0;
    }
    virtual bool marks_at(view &v, int pos, uint mid) override;
    virtual bool marks_at_contains(int pos, chars markcls) const override;

#ifdef _DEBUG
    virtual void dbg_report(const char *cap = "");
#endif

    bool same_as(const node* other) const { 
      if (!other)
        return false;
      if (!other->is_text())
        return false;
      return chars() == static_cast<const text*>(other)->chars(); 
    }

    bool set_text(wchars text, view* pv = nullptr);

    // data
    array<wchar>     chars;
    array<char_mark_t> marks; // optional marks like misspelling, etc.
  };

  struct comment : public text {

    DEFINE_TYPE_ID_DERIVED(html::comment, html::text)

    comment(wchars t) : text(t) {}
    comment(NO_INIT ni) : super(ni) {}

    virtual NODE_TYPE node_type() const { return COMMENT; }
    virtual string    node_name() const { return string("#comment"); }
    virtual string    node_def() const override { return string("Comment"); }
    
    virtual bool is_text() const { return false; }
    virtual bool is_comment() const { return true; }
    virtual bool is_space() const { return true; }
    virtual bool is_it_visible(view &v) {
      return false;
    } // visibility of the element itslef
    virtual bool is_display_none_node(view &v) { return true; }

    virtual node *clone() { return new comment(chars()); }
    virtual rect rbox(view &v) { return rect(); } // view relative rendering box

    virtual void emit(ostream &os, emit_ctx *pectx = nullptr);

#ifdef _DEBUG
    virtual void dbg_report(const char *cap = "");
#endif
  };

  struct text_block;

  enum CARET_TYPE {
    CHAR_POSITION,
    BLOCK_POSITION,
    ROW_POSITION,
  };

  struct caret_metrics {
    // element/glyph box:
    float x1, x2; // if x1 > x2 then RTL
    int   y1, y2; // glyph y
    // line metrics:
    int  line_y1, line_y2; // line y
    uint line_no;

    bool at_end;   // true - caret bar is drawn at x2, otherwise at x1
    rect glyph_rc; // current glyph

    handle<element> elem;

    CARET_TYPE ctype;

    rect caret_outline() const {
      rect r = rect::make_ltrb((int)floorf(min(x1, x2)), y1, (int)ceilf(max(x1, x2))+1, y2);
      // r |= caret_rc;
      r |= glyph_rc;
      r >>= 2;
      return r;
    }

    void move(point pos) {
      x1 += pos.x;
      x2 += pos.x;
      y1 += pos.y;
      y2 += pos.y;
      line_y1 += pos.y;
      line_y2 += pos.y;
      glyph_rc += pos;
    }

    rect              caret_v_bar() const;
    rect              caret_h_bar() const;
    tuple<rect, rect> caret_bars(CARET_TYPE ct) const;

    // array<pointf> h_caret_path() const;
    // array<pointf> v_caret_path() const;
  };

  struct bookmark {
    handle<html::node> node;
    int_v pos; // that is caret position - in between elements: text node has
               // [0..chars.length] positions
    bool after_it; // used by find by coordinate only.

    //    text_block*  tb;
    //    int          tb_pos;

    bookmark() : /*tb(0),tb_pos(0),*/ after_it(false) {}
    bookmark(html::node *nd, int p, bool after);
    bookmark(const bookmark &rs)
        : node(rs.node), pos(rs.pos), after_it(rs.after_it)
    //, tb(rs.tb)
    //, tb_pos(rs.tb_pos)
    {}
    bool valid() const { return node && pos.is_defined() && node->parent; }
    void stack(array<int> &stack, const element *root = 0) const;

    int linear_pos() const { return pos + static_cast<int>(after_it); }

    bool operator<(const bookmark &rs) const;
    bool operator<(const array<int> &trs) const {
      array<int> tls;
      stack(tls);
      return tls < trs;
    }

    bool operator<=(const bookmark &rs) const;
    bool operator>(const bookmark &rs) const;
    bool operator>=(const bookmark &rs) const;
    bool operator==(const bookmark &rs) const {
      return node == rs.node && pos == rs.pos && after_it == rs.after_it;
    }
    bool operator!=(const bookmark &rs) const {
      return node != rs.node || pos != rs.pos || after_it != rs.after_it;
    }
    bookmark operator~() const {
      bookmark t(*this);
      t.linearize();
      return t;
    }

    bool is_between(const bookmark &ls, const bookmark &rs) const;
    bool is_between(pair<bookmark, bookmark> range) const {
      return is_between(range.first, range.second);
    }

    bool is_inside(const element *container) const;

    bool at_caret_pos(view& v) const { return valid() && node->is_caret_pos(v,*this); } 
    bool at_char_pos(view& v) const { return valid() && node->is_char_pos(v,*this); }

    bool at_br_pos() const;

    bool at_element_start() const {
      return valid() && node->is_element() && node->is_start_pos(*this);
    }

    bool at_element_start(element *&pel) const {
      if (!at_element_start()) return false;
      pel = node->cast<element>();
      return true;
    }
    bool at_element_end() const {
      return valid() && node->is_element() && node->is_end_pos(*this);
    }

    bool at_block_element_start(view& v) const;
    bool at_block_element_end(view& v) const;

    bool at_inline_block_element_start(view& v) const;
    bool at_inline_block_element_end(view& v) const;
    
    bool at_element_end(element *&pel) const {
      if (!at_element_end()) return false;
      pel = node->cast<element>();
      return true;
    }
    bool at_comment_start() const {
      return valid() && node->is_comment() && node->is_start_pos(*this);
    }
    bool at_comment_end() const {
      return valid() && node->is_comment() && node->is_end_pos(*this);
    }
    bool at_text_start() const {
      return valid() && node->is_text() && node->is_start_pos(*this);
    }
    bool at_text_end() const {
      return valid() && node->is_text() && node->is_end_pos(*this);
    }
    bool at_start() const { return valid() && node->is_start_pos(*this); }
    bool at_end() const { return valid() && node->is_end_pos(*this); }

    bool at_table_row_start() const;
    bool at_table_row_end() const;
    bool at_table_column_start() const;
    bool at_table_column_end() const;

    wchar char_code(view& v) const {
      return at_char_pos(v) ? node->char_code(*this) : 0;
    }
    wchar ui_char_code(view& v) const {
      return at_caret_pos(v) ? node->ui_char_code(*this) : 0;
    }

    uint marks() const;
    bool marks(view &v, uint mark);

    bool marks_contains(chars markclass) const;

    bool get_caret_metrics(view &v, caret_metrics &gm,
                           bool char_only = false) const;
    rect get_caret_rect(view &v) const; // view relative
    rect get_glyph_box(view &v) const;  // view relative

    void normalize() {
      if (valid()) node->normalize(*this);
    }
    void linearize() {
      if (valid()) node->denormalize(*this);
    }

    bookmark normalized() const {
      bookmark t = *this;
      t.normalize();
      return t;
    }
    bookmark linearized() const {
      bookmark t = *this;
      t.linearize();
      return t;
    }

    bool advance_caret_pos(view &v, ADVANCE_DIR dir);
    bool advance_caret_pos(view &v, ADVANCE_DIR dir, array<wchar> &out);

    bool advance_forward(wchar &c) {
      return valid() ? node->advance_forward(*this, c) : false;
    }
    bool advance_backward(wchar &c) {
      return valid() ? node->advance_backward(*this, c) : false;
    }

    bool advance(int npos) {
      if (!valid()) return false;
      wchar dummy;
      if (npos > 0) {
        for (int n = 0; n < npos; ++n)
          if (!advance_forward(dummy)) return false;
      } else {
        for (int n = 0; n < -npos; ++n)
          if (!advance_backward(dummy)) return false;
      }
      return valid();
    }

    static int delta(bookmark bm1, bookmark bm2) {
      int step = 1;
      if (bm1 > bm2) {
        swap(bm1, bm2);
        step = -1;
      }
      int   n = 0;
      wchar dummy;
      while (bm1 != bm2) {
        if (!bm1.advance_forward(dummy)) return n;
        n += step;
      }
      return n;
    }

#ifdef _DEBUG
    void dbg_report(const char *cap = "") const {
      if (valid()) {
        dbg_printf("%s: pos:%d/%s of", cap, int(pos),
                   after_it ? "after" : "before");
        node->dbg_report("\n");
      } else
        dbg_printf("%s: invalid!");
    }
#endif
  };

  class bookmark_p {
    array<int> _stack;

  public:
    bookmark_p() {}
    bookmark_p(const bookmark &bm, const element *root) { set(bm, root); }

    bool valid() const { return _stack.size() > 0; }
    int  pos() const { return _stack.size() > 0 ? _stack.last() >> 1 : 0; }
    bool after_it() const {
      return _stack.size() > 0 ? (_stack.last() & 1) != 0 : false;
    }

    void     set(const bookmark &bm, const element *root);
    bookmark get(const element *root) const;
  };

  struct layout_data; // layout specific data
  struct iwindow;

  struct ui_pos_enum {
    virtual bool on_element_head(const element *el) { return true; }
    virtual bool on_element_tail(const element *el) { return true; }
    virtual bool on_char(wchar c) { return true; }
  };

  // used by <script> and <style> elements
  struct line_no_holder : public resource_x<line_no_holder> {
    int val;
    line_no_holder() : val(0) {}
  };

  // used by <style> elements to calcualte order/specificity of rules inside.
  struct sequential_id_holder : public resource_x<sequential_id_holder> {
    string val;
    sequential_id_holder() {}
  };

  // used for delayed value initialization
  struct value_holder : public resource_x<value_holder> {
    tool::value val;
    value_holder() {}
  };


  // "forced" style set
  struct style_set_holder : public resource_x<style_set_holder> {
    string   name;
    string   url;
    bool     important = false;
    mutable weak_handle<style_bag> style_set;
    uint_ptr owner = 0;
  };
  
  struct css_content : resource // css generated content, holding ::before,
                                // ::after and content things
  {
    handle<element> before;
    handle<element> after;
    handle<element> marker;
    handle<element> shade;
    handle<node>    content;
    void clear_style();
  };

  struct style_provider {
    virtual style *get_style(view &v, document *pd = nullptr) = 0;
  };

  struct state_provider {
    virtual ui_state get_state(bool disabled_deep) const = 0;
  };

  struct class_list_provider {};
  struct node_list_provider {};
  struct child_list_provider {};
  struct attribute_list_provider {};

  struct block_table_row;
  struct block_table_body;
  struct block_table;

  struct element : public node, weakable
    , style_provider
    , state_provider
    , class_list_provider
    , node_list_provider
    , child_list_provider
    , attribute_list_provider
#if defined(SCITERJS)
    , public event_target
#endif
  {
  private:
    element(const element &) = delete;
    element &operator = (const element &) = delete;
  public:

    DEFINE_TYPE_ID_DERIVED(html::element, html::node)

    element() : element(tag::T__UNKNOWN) {}
    
    element(uint st);

    element(string t) : element(tag::symbol(t).val(0)) {}
    
    /*
     *  element(NO_INIT ni) constructor is used by turn_element_to<> that simply
     * changes VTBL of this particular instance. So can be switched to block,
     * block_vertical, block_horizontal, etc. according to the 'flow' property
     * in CSS declaration.
     *
     *  NOTE: in order turn_element_to<> to work the element class and *all* its
     * derivatives shall have "no init" ctor with signature name(NO_INIT). "no
     * init" ctor shall do *precisely nothing*. All non-trivial memebers
     * of element class shall have also "no init" ctors and shall be
     * listed in element(NO_INIT) constructor.
     */
    element(NO_INIT);

    virtual void finalize();
    virtual ~element();

    virtual NODE_TYPE node_type() const override { return ELEMENT; }
    virtual string    node_name() const override { return tag::symbol_name(tag); }
    virtual string    node_def() const override;
    virtual bool      node_contains(node* pn) override { if (!pn) return false; return pn->belongs_to(this); }
    virtual bool      node_has_child_nodes() override { return nodes.length() > 0u; }
    element *         logical_parent() const;
    
    typedef html::layout_data layout_data;

    virtual void detach(const element *p) {
      drop_layout();
      event_owner = nullptr;
      super::detach(p);
    }
    virtual void detach_owner() {
      if (flags.is_synthetic) {
        //WRONG: parent = 0 - will break v.mouse_over_element handling when synthetic will become invisible
        //parent = 0; // synthesized element looses parent too when detached.
        //owner  = 0;
        super::detach_owner();
        FOREACH(i, nodes) nodes[i]->detach_owner();
      }
      else
        super::detach_owner();
    }


    //virtual view *    pview() const override;
    //virtual document *doc() const override;

    virtual document *parent_doc() const { return doc(); }

    virtual element *get_element() override { return this; }
    virtual element *get_ui_element() override { return this; }
    virtual element *get_event_owner();

    virtual bool set_event_owner(element* powner)  {
      if (powner && powner->is_connected()) {
        event_owner = powner;
        return true;
      }
      return false;
    }

    virtual void got_parent(element *p, uint nidx) override {
      node::got_parent(p, nidx);
      flags.strayed = 0;
      if (!flags.state_initialized) state.data = 0;
      FOREACH(i, nodes) nodes[i]->got_parent(this, i);
    }

    virtual node *   clone() { return clone_element(); }
    virtual element *clone_element(bool and_content = true, bool and_attributes = true);
    virtual element *clone_element_for_editing(bool and_content = true, bool and_attributes = true) { return clone_element(and_content,and_attributes); }
    void             copy_content_from(view &v, const element *src);
   
    node *get_node_by_uid(long uid_) const;
    element* get_element_by_uid(long uid_) const;
    // turn_to things:
    // virtual void transforming();
    // virtual void transformed();

#ifdef _DEBUG
    virtual void dbg_report(const char *cap = "") {
      dbg_printf("%s <%s[%d]", cap, tag::symbol_name(tag).c_str(),
                 flags.ui_index);
      ustring id = attr_id();
      if (id.length()) dbg_printf(" #%s", string(id).c_str());
      ustring cls = attr_class();
      if (cls.length()) dbg_printf(" .%s", string(cls).c_str());
      dbg_printf(">\n");
    }
    virtual void dbg_stack(const char *cap = "") {
      if (cap && cap[0]) dbg_printf("\n%s %p", cap, this);

      if (parent && !is_document()) { parent->dbg_stack(); }

      dbg_printf(" <%s", tag::symbol_name(tag).c_str());
      ustring id = attr_id();
      if (id.length()) dbg_printf("#%s", string(id).c_str());
      ustring cls = attr_class();
      if (cls.length()) dbg_printf(".%s", string(cls).c_str());
      dbg_printf(">");
    }
#endif

    virtual ustring report() {
      ustring us = tag::symbol_name(tag);
      ustring id = attr_id();
      if (id.length()) {
        us += '#';
        us += id;
      }
      ustring cls = attr_class();
      if (cls.length()) {
        us += '.';
        us += cls;
      }
      return us;
    }

    bool set_style_variables(view &v, const dictionary<string, value>& vars);
    bool set_style_variable(view &v, const string &ns, value val);

    // indicators
    
    virtual bool is_element() const { return true; }
    virtual bool is_document() const { return false; }
    virtual bool is_svg_document() const { return false; }
    virtual bool is_svg_element() const { return false; }
    virtual bool is_in_svg_context() const { 
      if (parent && parent->is_in_svg_context()) return true;
      if (tag == tag::T_SVG) return true;
      return false; 
    }

    virtual bool is_table() const { return false; }
    virtual bool is_table_body() const { return false; }
    virtual bool is_table_row() const { return false; }
    virtual bool is_table_cell() const {
      element* o = get_owner();
      return (o && o->is_table_row()) || tag == tag::T_TD ||
             tag == tag::T_TH;
    }
    virtual bool is_break() const { return tag == tag::T_BR; }
    virtual bool is_body() const {
      return tag == tag::T_BODY && parent && parent->is_document();
    }

    virtual tag::symbol_t is_css_element() const // returns true if this is CSS generated element
    {
      return pseudo_element_id() ? tag : tag::T__UNKNOWN;
    }

    PSEUDO_ELEMENT_ID pseudo_element_id() const {
      switch (tag) {
        case tag::T__BEFORE:  return PSEUDO_BEFORE;
        case tag::T__AFTER:   return PSEUDO_AFTER;
        case tag::T__MARKER:  return PSEUDO_MARKER;
        case tag::T__SHADE:   return PSEUDO_SHADE;
        default: return PSEUDO_NONE;
      }
    }

    bool is_pseudo_element() const { return pseudo_element_id() != PSEUDO_NONE; }

    virtual bool is_anonymous_text_block() const { return false; }

    // a.k.a. replaced element: An element whose content is outside the scope of
    // the CSS formatting model, such as an image, embedded document, or applet.
    // For example, the content of the HTML IMG element is often replaced by the
    // image that its "src" attribute designates.

    virtual bool is_atomic_box() const; // true if element participates in text selection as a whole.
                                        // and pos_iterator shall skip its content.

    virtual bool is_box() const { return false; }
    virtual bool is_text_box() const { return false; }
    virtual bool is_horizontal_layout() const { return false; }
    virtual bool is_vertical_layout() const { return false; }

    virtual bool is_inline_element(view &v); // display:inline, inline-block ...;
    virtual bool is_display_none_node(view &v) {
      return get_style(v)->is_display_none();
    }
    virtual bool is_collapsed_node(view &v) {
      return get_style(v)->visibility == visibility_collapse;
    }
    virtual bool is_inline_span_element(view &v);
    virtual bool is_inline_block_element(view &v);
    virtual bool is_inline_block_element();

    virtual bool is_inside_text_flow(view &v); // element layout controlled by text_block flow

    virtual bool is_block_element(view &v); // display:block | table | list-item | etc.
    virtual bool is_box_element(view &v); // !display:none && !display:inline
    virtual bool is_row_element(view &v);
    virtual bool is_table_element(view &v);
    virtual bool is_inline_table_element(view &v);
    virtual bool is_cell_element(view &v);

    virtual bool is_sensitive_attr(const name_or_symbol &ns, bool &rem);

    virtual bool is_it_visible(view &v);  // visibility of the element itslef
    virtual bool is_it_drawable(view &v); // drawability of the element itslef
    virtual bool it_takes_space(view &v); // !is_display_none
    virtual bool is_visible(view &v, const element *up_to_parent = nullptr); // visible by itself and inside visible container
    virtual bool is_drawable(view &v, const element *up_to_parent = nullptr); // visible by itself and inside visible container
    virtual bool takes_space(view &v, const element *up_to_parent = nullptr); // visible by itself and inside visible container
    virtual bool abs_positioned(view &v) {
      return get_style(v)->position == position_absolute;
    }
    virtual bool rel_positioned(view &v) {
      return get_style(v)->position == position_relative;
    }
    virtual bool fix_positioned(view &v) {
      return get_style(v)->position == position_fixed;
    }
    virtual bool popup_positioned(view &v) { return parent && state.popup(); }
    virtual bool oof_positioned(view &v) {
      return airborn || get_style(v)->position >= position_absolute;
    }
    virtual bool positioned(view &v) { return get_style(v)->position > 0; }
    virtual bool position_detached(view &v) {
      return get_style(v)->is_position_detached();
    }
    virtual float_e floats(view &v) { return get_style(v)->used_float(); }
    virtual int h_flexes(view &v) { return get_style(v)->h_flex1000(); }
    virtual int v_flexes(view &v) { return get_style(v)->v_flex1000(); }

    virtual bool is_floats_container(view &v); // a.k.a. layout root - element
                                               // that establish BFC - block
                                               // formatting context

    // DOM access & manipulation:
    // DOM tree traversal
    virtual node *   first_node() const;          // child
    virtual node *   last_node() const;           // child
    virtual node *   first_nonspace_node() const; // child
    virtual node *   last_nonspace_node() const;  // child
    virtual element *first_element() const;       // child
    virtual element *last_element() const;        // child
    virtual element *next_element() const {
      return parent ? parent->next_child(this) : 0;
    } // sibling
    virtual element *prev_element() const {
      return parent ? parent->prev_child(this) : 0;
    }                                                     // sibling
    virtual element *next_child(const element *el) const; // siblings
    virtual element *prev_child(const element *el) const; // siblings

    int      index() const;
    int      n_children() const;   // number of children
    element *child(int idx) const; // n-th child

    virtual element *similar_neighbour(element *child) const {
      return prev_child(child);
    } // used by style resolver
      // ovveriden in table_row

    virtual element *get_element_by_id(const ustring &id) { return find_by_id(id); };

    virtual element *find_by_id(const ustring &id, bool this_too = false,
                                bool skip_frames = true);
    virtual element *find_by_tag(tag::symbol_t tag, bool this_too = false,
                                 bool skip_frames = true);
    virtual element *find_for_id(const ustring &id, const element *except = 0)
        const; // neighbour element that has for=id
    virtual element *find_neighbour_id(
        const ustring &id, const element *except = 0,
        bool and_name_too = false) const; // neighbour element that has id=id

    template<typename F> bool each_child(F f);
    template<typename F> bool each_child(F f, BACKWARD_E);

    // rendering tree traversal. Note: rendering tree is mostly a DOM tree but
    // my contain synthetic elements like anonymous paragraphs
    virtual bool each_ui_child(function<bool(element *)> f);
    virtual bool each_ui_child(function<bool(element *)> f, BACKWARD_E);

    virtual bool each_ui_child_node(const function<bool(node *)> &f) {
      // auto proxy = [&f]( element* el) -> bool { return f(el); };
      return each_ui_child(f);
    }

    template<typename F>
       bool each_any_child(F f);

    virtual bool each_any_child_node(const function<bool(node *)> &f) override {
      handle<element> holder = this;
      if (style_content) {
        if (style_content->marker && f(style_content->marker))
          return true;
        if (style_content->shade && f(style_content->shade))
            return true;
        if (style_content->after && f(style_content->after))
          return true;
        if (style_content->before && f(style_content->before))
          return true;
        if(style_content->content && f(style_content->content))
          return true;
      }
      return this->nodes.each(f);
    }

    virtual element *first_ui_element();
    virtual element *last_ui_element();
    virtual element *next_ui_element();
    virtual element *prev_ui_element();

    // DOM bookmark (DOM position)
    virtual void
                     normalize(bookmark &bm) const; // check if in range + bm.after_it
    virtual bookmark start_pos() const;
    virtual bookmark end_pos() const;
    virtual bool     advance_forward(bookmark &bm, wchar &c) const;
    virtual bool     advance_backward(bookmark &bm, wchar &c) const;
    virtual bool     is_start_pos(const bookmark &bm) const;
    virtual bool     is_end_pos(const bookmark &bm) const;
    virtual bool     advance_caret_pos(view &v, bookmark &bm, ADVANCE_DIR dir,
                                       array<wchar> *out = 0);
    virtual bookmark start_caret_pos(view &v) const;
    virtual bookmark end_caret_pos(view &v) const;

    virtual bookmark first_content_caret_pos(view &v) const;
    virtual bookmark last_content_caret_pos(view &v) const;

    virtual bool     is_caret_pos(view& v, const bookmark &bm) const;
    virtual bool     is_char_pos(view& v, const bookmark &bm) const;

    /*virtual element* next_ui_element() const { return owner?
    owner->next_ui_child(this):0; } virtual element* prev_ui_element() const {
    return owner? owner->prev_ui_child(this):0; } virtual element*
    next_ui_child(const element* el) const { return next_child(el); } virtual
    element* prev_ui_child(const element* el) const { return prev_child(el); }
  */

    virtual int n_ui_children() const {
      return n_children();
    } // number of ui children (rendering tree children)
    virtual element *ui_child(int idx) const {
      return child(idx);
    } // n-th ui child (rendering tree child)

    static element *find_common_parent(element *b1, element *b2);
    static element *find_common_parent(element *b1, element *b2,
                                       int layout_type);
    static element *find_base(element *b1, element *b2);
    static element *find_ui_base(element *b1, element *b2);
    static element *find_common_ui_parent(view &v, element *b1, element *b2);

    bool has_child_of_type(tag::symbol_t sym) const;
    bool has_children_of_type(tag::symbol_t sym) const;
    bool is_only_child_of_type() const;

    virtual bool belongs_to(element *pb, bool include_this = false);
    // same as belongs_to above but takes popups into account
    virtual bool belongs_to(view &v, element *pb, bool include_this = false);
    element *    ui_owner(view &v); // popups have anchors as their ui_parent
    element *    ui_parent(view &v); // popups have anchors as their ui_parent
    virtual element *owner_block(view &v);
    // virtual element* nearest_box(bool ui = true);
    virtual element *nearest_text_box() { return super::nearest_text_box(); }
    virtual element *nearest_known_box();
    virtual element *nearest_bfc(
        view &v); // closest block formatting context (<body>, <td> , float)

    virtual block_table_body *parent_table_body() const {
      return parent ? parent->parent_table_body() : nullptr;
    }
    virtual element *nearest_table_cell();

    virtual void on_layout_set(view &v); // called immediately after the element
                                         // got its layout controller

    virtual void append(node *n, view *pv = 0);
    virtual void append_nodes(slice<hnode> other_nodes, view *pv = 0);
    virtual void insert(int idx, node *n, view *pv = 0);
    virtual void insert_nodes(int idx, slice<hnode> other_nodes, view *pv = 0);
    virtual void remove_nodes(int from, int to, view *pv = 0);
    virtual void remove_child(node *child) {
      assert(child->parent == this);
      if( child->node_index != UNDEFINED_NODE_INDEX )
        nodes.remove(child->node_index);
      child->node_index = UNDEFINED_NODE_INDEX;
    }
    virtual void replace_child(node* child, node* by_this_one, view *pv = 0) {
      assert(child->parent == this);
      int index = int(child->node_index);
      //remove_child(child); 
      child->remove(true, pv);
      insert(index,by_this_one,pv);
    }

    virtual void commit_mutation(view& v, uint mutation_flags, bool root_mutation);

    virtual bool  remove(bool finalize, view *pv = 0) override;
    
    virtual node *split(view &v, int at, bool force = false) override;
    virtual int   unsplit(view &v) override;

    virtual void stray(view &v); // a.k.a. on_document_unload, element is about
                                 // to be completely removed from the DOM, clear
                                 // all runtime structures
    virtual void clear(view *pv = 0);
    // virtual void clear_value(view& v);

    handle<text_block> create_text_block(view &v, wchars text);

    static bool swap_locations(element *b1, element *b2, view *pv = 0);

    virtual ustring get_text(view &v);
    virtual void    get_ui_text(view &v, array<wchar> &to); // returns rendered text
    virtual void    emit_text(array<wchar> &to, bool verbose = false) override;
    virtual void    set_text(view &v, wchars text);
    virtual void    set_text_value(view &v, wchars text);
    virtual void    set_html_value(view &v, bytes html);

    virtual ustring node_text() const { 
      array<wchar> out;
      const_cast<element*>(this)->emit_text(out); 
      return out();
    }
    virtual void    node_set_text(ustring t, view* pv = nullptr) { 
      if (pv)
        set_text(*pv, t());
      else {
        nodes.clear();
        nodes.push(new text(t));
      }
    }


    // HTML emitter
    virtual void emit(ostream &os, emit_ctx *pectx = nullptr);
    virtual bool emit_head(ostream & os,emit_ctx *pectx = nullptr); // returns true if emit_content()/emit_tail() is needed.
    virtual void emit_tail(ostream & os,emit_ctx *pectx = nullptr); // returns true if emit_content()/emit_tail() is needed.
    virtual void emit_content(ostream &os, emit_ctx *pectx = nullptr);

    bool set_attr(const name_or_symbol &ns, const ustring &val, view* pv = nullptr); // return true if it is known attr and
                                       // some action was made
    bool remove_attr(const name_or_symbol &ns, view* pv = nullptr); // return true if it is known
                                                // attr and some action was made
    bool set_attr(view &v, const name_or_symbol &ns, const ustring &val); // return true if it is known attr and
                                       // some action was made
    bool remove_attr(view &v, const name_or_symbol &ns); // return true if it is
                                                         // known attr and some
                                                         // action was made
    bool set_attributes(const attribute_bag& other, view* pv = nullptr, bool allow_remove_id = true); // return true if it is known attr and
    bool set_states(const attribute_bag_v& other, view* pv = nullptr, bool skip_value_if_in_focus = false); // return true if it is known attr and

    // void   set_class(const ustring& val);
    // bool   remove_class(const ustring& val);
    // bool   add_class(const ustring& val);

    virtual bool on_set_attr(uint att_sym, const ustring &val); 
                                                  // return true if it is known
                                                  // attr and some action was
                                                  // made
    virtual bool on_remove_attr(uint att_sym, const ustring &val);    
                                               // return true if it is known
                                               // attr and some action was made

    virtual void on_lang_change(view* pv = nullptr);
    virtual void on_theme_change(view* pv = nullptr);

    bool get_attr(const char *name, ustring &val) const; // get attribute from
                                                         // atts map or from
                                                         // custom CSS
                                                         // attributes. name
                                                         // shall start from '-'
    ustring get_attr(const char *name) const {
      ustring val;
      get_attr(name, val);
      return val;
    }
    bool has_attr(const char *name) const; // get defined attribute in atts map
                                           // or in custom CSS attributes. name
                                           // shall start from '-'

    void  line_no(int ln);
    int_v line_no() const;

    void   sequential_id(string id);
    string sequential_id() const;

    void delay_value(tool::value val);
    bool delayed_value(tool::value& val);

    const style_set_holder* forced_style_set() const;
    void                    forced_style_set(const string& name, const string& cssurl, bool important = false);
    void                    forced_style_set(const style_set_holder* ph);
    void                    drop_forced_style_set() { forced_style_set(nullptr); }

    //  well known attributes
    ustring attr_id() const { return atts(attr::a_id); }
    ustring attr_key() const { return atts(attr::a_key); }
    ustring attr_class() const { return atts(attr::a_class); }
    ustring attr_type() const { return atts(attr::a_type); }
    ustring attr_title() const { return atts(attr::a_title); }
    ustring attr_titleid() const { return atts(attr::a_titleid); }
    ustring attr_tooltip() const { return atts(attr::a_tooltip); }
    bool    attr_disabled() const { return atts.exist(attr::a_disabled); }
    ustring attr_role() const { 
      ustring r; if (atts.exist(attr::a_role, r)) return r;
      return get_style()->role;
    }
    ustring attr_name() const { return atts(attr::a_name); }
    ustring attr_value() const { return atts(attr::a_value); }
    bool    attr_value(ustring &val) const { return atts.get(attr::a_value, val); }
    bool    attr_key(ustring &val) const { return atts.get(attr::a_key, val); }
    ustring attr_lang() const {
      ustring l = atts.get_ustring(attr::a_lang);
      if (l.length())
        return l.to_lower();
      return l;
    }
    ustring attr_novalue() const {
      ustring v = atts(attr::a_novalue);
      if (v.is_defined()) return v;
      return atts(attr::a_placeholder);
    }

    bool has_class(wchars name) const {
      return match_list_ci(name, attr_class()(), WCHARS(" "));
    }

    bool is_id_test() const { return attr_id() == WCHARS("test"); }

    uint get_tab_size() const;

    // UI state methods:
    virtual bool is_disabled() const;
    virtual bool is_readonly() const;
    virtual bool is_of_class(const char *cls) const;
    virtual bool is_of_class(const char *cls1, const char *cls2) const {
      return is_of_class(cls1) && is_of_class(cls2);
    }
    virtual bool     is_empty() const;
    virtual ui_state get_state(bool disabled_deep) const override;
    virtual void     check_states();
    virtual int_v    tab_index() const;

    virtual bool is_focusable(view &v);
    //virtual bool is_x_focusable(view &v);
    // style methods:
    virtual style *get_style(view &v, document *pd = nullptr) override;
    virtual style *get_style_no_cache(view &v, document *pd = nullptr);
    virtual style_provider* get_style_provider() { return this; }

    virtual style *get_style() const;
    virtual style *get_base_style(view &v);
    virtual void  clear_style();
    virtual void drop_styles(view &v); // of self and children
    virtual void drop_style_and_state(view *pv = nullptr); // of self
    virtual void drop_styles_and_states(view *pv = nullptr, bool  force_all = false); // of self and children
    virtual void resolve_styles(view &v); // of self and children
    virtual void reset_styles(view &v);   // of self and children
    virtual void reset_style(view &v) {
      clear_style();
      get_style(v);
    }
    void  get_applied_styles(view &v, style_def::callback &cb);
    value get_style_const(view &v, const string &name);

    virtual ustring get_lang() const;
    virtual ustring get_theme() const;

    virtual handle<style_prop_list> apply_attributes(view &v, document *pd); // applies attributes to the style

    virtual style *determine_style(view &v, document *d,
                                   bool call_on_changed = true,
                                   bool no_cache        = false);

    virtual style *apply_a_style(view &v, document *d, bool call_on_changed = true);
    virtual void   update_a_style(view &v, document *d, const function<bool(style_prop_map& s)>& updater, bool to_remove = false);
    
    virtual style *get_similar_style(view &v);

    // popuplates the list by all applicable style sources
    virtual hstyle get_style_list(view &v, document *d, style_list& sl); // returns resolved parent style 
    virtual void   resolve_style(view &v, document *d); // finds d_style and c_style
    virtual hstyle resolve_style_for_mark(view &v, char_mark_t cm);

    virtual void   on_style_changed(view &v, document *d); // style changed "event"
            void   check_animations(view &v, document *d, const style *prev_style, STYLE_CHANGE_TYPE sct); // style changed "event"
    void           set_style_generated_content(view &v, const style *from);
    virtual void   fixup_style(view &v, document *, style &s);
#if 0
    void           exec_assigned(view &v); // run CSSS! assigned! if needed.
#endif

    void set_style_attribute(cssa::name_or_symbol name, const ustring &v);
    void set_style_attribute(cssa::name_or_symbol name, const value &v);
    void set_style_attributes(const style_prop_list& list);
    //void set_style_attributes(const slice<named_value> &bag, style *target);
    void remove_style_attribute(cssa::name_or_symbol name);
    void remove_style_attributes();

    //void parse_attribute_as(view &v, document *d, cssa::symbol_t attr_sym, wchars val, style_prop_list *s);

    string doc_url() const;

    // behavior support:
    void attach_behavior(view &v, const string &name);
    void attach_behavior(view &v, ctl *pc); // attach secondary unnamed behavior
    void detach_named_behavior(view &v, const string &name);
    void get_behavior_names(array<string> &names);
    void detach_behavior(view &v, ctl *pc);
    void detach_behaviors(view &v);
    string behavior_name() const; // if any

    ctl *get_named_behavior(const string &name) const;

    template<class T> T *get_behavior() const {
      ctl *pbh = behavior;
      while (pbh) {
        if (pbh->is_of_type<T>()) return static_cast<T*>(pbh);
        pbh = pbh->next;
      }
      return nullptr;
    }
    ctl *get_behavior(CTL_TYPE t1, CTL_TYPE t2 = CTL_NO) const;

    // returns attached behavior as asset  
    som_asset_t* get_asset(som_atom_t name) const {
      ctl *pbh = behavior;
      while (pbh) {
        if (auto apsp = pbh->asset_get_passport()) {
          if (apsp->name == name)
            return static_cast<som_asset_t*>(pbh);
        }
        pbh = pbh->next;
      }
      return nullptr;
    }

    /*virtual bool get_interface(tool::chars name, void **pp) override
    {
      ctl *pbh = behavior;
      while (pbh) {
        if (pbh->get_interface(name, pp))
          return true;
        pbh = pbh->next;
      }
      return node::get_interface(name,pp);
    }*/
    
    void each_behavior(function<bool(ctl *)> f) const;

    uint             behavior_count() const;
    ctl *            behavior_no(uint n) const;
    bool             call_behavior_method(view &v, method_params *p);
    virtual CTL_TYPE ctl_type(view &v);
    virtual CTL_TYPE ctl_type() const;
    bool             is_input_ctl_type() const {
      CTL_TYPE ct = ctl_type();
      return (ct > CTL_UNKNOWN) && (ct < CTL_LAST_INPUT) && (ct != CTL_CLICKABLE);
    }

    template <typename ET> bool handles() const;

    bool applied_style_rules_report(view &v, value &retval);
    bool used_style_props_report(view &v, value &retval);

    bool send(const char *method_name, int argc, tool::value *argv,
              tool::value *retval, bool optional);

    selection_ctx *
    get_selection_ctx() const; // returns selection_ctx from it or its parents.

    // char marks support:
    //   each character in text node may have assisiated char_mark_t bit set;
    //   this function will find style for the mark derived from the base.
    virtual style *style_for_marks(char_mark_t marks, const style *base) {
      return nullptr;
    }

    // accessible helpers
    // these must be in sync with NAVDIR_*** definitions from oleacc.h
    enum NAVDIR {
      DIR_UP         = 0x00000001,
      DIR_DOWN       = 0x00000002,
      DIR_LEFT       = 0x00000003,
      DIR_RIGHT      = 0x00000004,
      DIR_NEXT       = 0x00000005,
      DIR_PREVIOUS   = 0x00000006,
      DIR_FIRSTCHILD = 0x00000007,
      DIR_LASTCHILD  = 0x00000008
    };
    virtual element *        a11y_navigate(NAVDIR dir);
    virtual bool             a11y_get_name(view &v, ustring &name);
    virtual bool             a11y_get_desc(view &v, ustring &name);
    virtual bool             a11y_get_value(view &v, ustring &name);
    virtual bool             a11y_get_children(array<hnode>& children);
    virtual handle<element>  a11y_get_focus(view &v);
    virtual bool             a11y_is_text() { return false; }
    virtual bool             a11y_is_atomic(view &v);
    virtual bool             a11y_get_state(view &v, ui_state& st);
    virtual element *        a11y_find_element(view &v, point zpos);
    virtual bool             a11y_do_default_action(view &v);
    virtual bool             a11y_is_visible(view &v) override;

        
    enum A11Y_LIVE {
      A11Y_LIVE_OFF,
      A11Y_LIVE_POLITE,
      A11Y_LIVE_ASSERTIVE,
    };
    virtual A11Y_LIVE get_a11y_live(view &v);


    // layout methods:
    virtual flow_e layout_type() const { return flow_null; }

    flow_e discover_flow_type(view &v);

    virtual element *drop_layout(view *pv = 0);
    //virtual void     reset_layout_tree(view& pv, bool and_drop_cache = false);
    virtual void     drop_layout_tree(view *pv = 0, bool and_drop_cache = false);
    virtual void     drop_minmax_dim();
    virtual void     drop_content_layout(view *pv = 0);
    virtual void     drop_style(view *pv = nullptr);
    virtual void     drop_positioned(view *pv = nullptr);
    virtual bool is_layout_valid();   // ldata contains valid values
    virtual bool is_x_minmax_valid(); // ldata contains valid values
    virtual bool is_y_minmax_valid(); // ldata contains valid values
    virtual bool scan_floats(view &v, element *float_container);

#if defined(SCITER) || defined(SCITERJS)
    virtual void  check_prototypes_tree(view& v);
#endif

    virtual point pos() const;
    virtual size  dim() const;
    virtual size  content_dim() const;
    virtual void  set_pos(point p);
            void  set_pos(int x, int y) { set_pos(point(x, y)); }
    virtual void  set_x_pos(int p);
    virtual void  set_y_pos(int p);
    virtual void  set_border_pos(point p); // p here is top/left of border box
    virtual void  set_cell_pos(point p, bool collapsed); // p here is top/left of border box
    virtual void  set_margin_pos(point p); // p here is top/left of border box
    virtual void  set_dim(size s);
    virtual void  set_border_dim(size s);
    virtual point screen_pos(view &v);
    virtual point view_pos(view &v);
    // virtual void  doc_mtx(view& v, affine_mtx_f& vmx); // computes
    // transformation matrix needed to render the element at its doc place.
    virtual void view_mtx(view &v, affine_mtx_f &vmx); // computes
                                                       // transformation matrix
                                                       // needed to render the
                                                       // element at its view
                                                       // place.
    virtual point transform_view_to_local(
        view &v,
        point vpt); // translates view point vpt into local dom coordinate
    virtual point transform_local_to_view(view &v, point pt);

    virtual point window_pos(view &v);
    virtual point doc_pos(view &v);

    virtual point rel_pos(view &v, element *other);
    virtual size  rel_shift(view &v, element *parent);
    virtual point scroll_pos() const;
    virtual void  scroll_pos(view &v, point off, bool allow_out_of_bounds = false);

    virtual point layout_parent_pos(view &v); // pos relative to layout_parent()
    virtual point positioned_pos(view &v, element *positioned);
    virtual bool  can_scroll_h(view &v) const;
    virtual bool  can_scroll_v(view &v) const;

    struct max_values {
      int_v m1, m2;
      int_v b1, b2, ib1, ib2;
      int_v p1, p2, ip1, ip2;
    };

    virtual max_values measure_borders_x(view &v, size container_dim);
    virtual max_values measure_borders_y(view &v, size container_dim);

    virtual int   min_width(view &v, int_v container_width = int_v());
    virtual int_v max_width(view &v, int_v container_width = int_v()); // called by calc_intrinsic_width, default max content width
    //virtual int_v max_cap_width(view &v, int container_width = 0); // called by spring environment

    virtual int outer_int_x_extra(view &v, int container_width = 0); // called by calc_intrinsic_width, default max content width
    virtual int borpad_int_x_extra(view &v, int container_width = 0); // called by calc_intrinsic_width, default max content width
    virtual int outer_int_y_extra(view &v, int container_width = 0); // called by calc_intrinsic_width, default max content width
    virtual int borpad_int_y_extra(view &v, int container_width = 0); // called by calc_intrinsic_width, default max content width
    virtual int cell_int_x_extra(view &v, int container_width,bool collapsed); // called by calc_intrinsic_width, default max content width
    virtual int cell_int_y_extra(view &v, int container_height,bool collapsed); // called by calc_intrinsic_width, default max content width

    //virtual int_v min_max_width(view &v); // min of (width and max-width), undefined if not set.
    //virtual int_v min_max_height(view &v); // min of (width and max-width), undefined if not set.

    virtual int   min_height(view &v, int_v container_width = int_v());
    virtual int_v max_height(view &v, int_v container_width = int_v()); // called by
                                                      // calc_intrinsic_height
                                                      // alike scenarios,
                                                      // default max content
                                                      // height
    //virtual int_v max_cap_height(view &v, int   container_width = 0); // called in spring contexts
    // virtual int min_width_outer(view& v, int container_width = 0);
    // virtual int max_width_outer(view& v, int container_width = 0);
    // virtual int min_height_outer(view& v);
    // virtual int max_height_outer(view& v);
    // virtual int min_width_border(view& v);
    // virtual int max_width_border(view& v, int cap = 32000);

    virtual int min_inline_margin_box_width(view &v);

    //virtual int_v min_max_width_border(view &v);
    //virtual int_v min_max_height_border(view &v);
    //virtual int_v min_max_width_outer(view &v);
    //virtual int_v min_max_height_outer(view &v);
    // virtual int min_height_border(view& v);
    // virtual int max_height_border(view& v, int cap = 32000);

    virtual int_v auto_width(view &v);
    virtual int_v auto_height(view &v);

    virtual int min_content_width(view &v);
    virtual int max_content_width(view &v);
    virtual int min_content_height(view &v);
    virtual int max_content_height(view &v);

    virtual int
                min_defined_width(view &v,
                                  bool including_width = false); // min-width defined in CSS
    virtual int min_defined_height(view &v, bool including_height = false); //

    virtual int declared_min_width(view &v, int container_width);
    virtual int declared_max_width(view &v, int container_width);
    virtual int declared_width(view &v, int container_width);
    virtual int declared_min_height(view &v, int container_height);
    virtual int declared_max_height(view &v, int container_height);
    virtual int declared_height(view &v, int container_height);

    virtual element *_find_element(view &v, point zpos, bool exact = true);
    virtual element *find_element(view &v, point zpos, bool exact = true);

    // virtual element* find_element(view& v, point zpos /*zpos is this
    // relative*/, bool exact = true);
    virtual element *find_child_element(view &v,
                                        point at /*at is this relative*/,
                                        bool  exact = true) {
      return nullptr;
    }
    virtual bookmark find_text_pos(view &v,
                                   point zpos /*at is owner relative*/);
    // virtual bool     caret_rect_of(view& v, const bookmark& bm, rectf& rect);
    // virtual bool     caret_rect_of(view& v, const bookmark& bm, rectf& rect);
    virtual bool get_caret_metrics(view &v, const bookmark &bm,
                                   caret_metrics &gm);

    virtual bool get_first_line_metrics(view &v, int &y, int &height,
                                        int &ascent);
    virtual bool get_last_line_metrics(view &v, int &y, int &height,
                                       int &ascent);
    virtual void get_inline_block_metrics(view &v, int &ascent, int &descent,
                                          int &height);
    virtual int  get_baseline_shift(view &v, element *parent_text_block);

    virtual element *layout_parent(view &v);
    virtual element *floats_parent(view &v);
    virtual element *pos_parent(view &v);
    virtual element *abs_pos_parent(view &v);

    element * check_positioned_containment(view &v);

    // virtual point    static_pos(view& v, element* child) { return
    // child->pos(); }

    virtual bool collapsed() const { return c_style->collapsed(); }

    virtual int  set_width(view &v, int width);   // returns intrinsic height
    virtual int  set_height(view &v, int height); // returns intrinsic width
    virtual void update_scrollbars(view &v);
    virtual bool _update_scrollbars(view *pv, bool reset = true) {
      update_scrollbars(*pv);
      return true;
    }
    // replace ::marker or ::shadow element
    void replace__marker_element(view& v, element* marker_element) {
      if (flags.marker_layout_valid) return;
      flags.marker_layout_valid = 1;
      replace_css_element(v, marker_element);
    }
    void replace__shadow_element(view& v, element* shadow_element) {
      if (flags.shadow_layout_valid) return;
      flags.shadow_layout_valid = 1;
      replace_css_element(v, shadow_element);
    }
    void replace_css_element(view& v, element* shadow_element);

    virtual bool do_delayed_measure(view &v);
    bool         request_delayed_measure(view &v, bool horizontal);

    virtual void set_border_width(view &v, int width);
    virtual void set_border_height(view &v, int height);

    virtual void set_cell_width(view &v, int width, bool collapsed);
    virtual void set_cell_height(view &v, int height, bool collapsed);

    // set width/height but without layout
    virtual void set_cell_width_nm(view &v, int width, bool collapsed);
    virtual void set_cell_height_nm(view &v, int height, bool collapsed);

    virtual int layout_width(view &v, int width) { /*assert(false);*/
      return 0;
    }
    virtual int layout_height(view &v, int height) { /*assert(false);*/
      return 0;
    }
    virtual bool do_v_align(view &v);
    virtual bool do_h_align(view &v);
    virtual void calc_intrinsic_widths(view &v) { /*assert(false);*/
    }
    virtual int  computed_width(view &v, int container_width);
    virtual int  computed_height(view &v, int container_height);
    virtual bool measure_inplace(view &v); // try to measure it in-place,
                                           // returns true if dimensions change.
    virtual bool try_update_inplace(view &v); // drop layout and try to measure it in-place,
                                              // if dimensions change then does v.add_to_update()
                                              // returns true if dimensions change.

    virtual void commit_measure(view &v);
    void         _commit_measure(view &v);
    void measure_oof(view &v, size parent_sz, size max_sz /*e.g. screen dim*/);
    void measure_oof(view &v, size forced_dim);

    void ensure_valid_layout(view& v);

    // animates updates of the element. If CSS transition is defined for the
    // element. If not - simply calls the func
    handle<animation> animated_update(view & v, const function<bool(view &, element *)> &state_changer, animated_effect* paeff = nullptr, effect_animator::STATE st = effect_animator::FORWARD);

    virtual int  paginate(view &v, range page, range &intersection,
                          int &elements_on_page, int page_no);
    virtual void drop_pagination(view &v);

    virtual void on_vscrollbar_show(view &v) {} // after scrollbar shown/hidden
    virtual void on_hscrollbar_show(view &v) {} // after scrollbar shown/hidden

    floats_ctx * fctx(view &v, bool create = false);
    virtual void content_x_range_at(view &v, range y, element *of,
                                    range &in_out_xx) {}

    // measured layout: columns/rows accessor
    virtual int      n_rows() { return 0; }
    virtual int      n_cols() { return 0; }
    virtual void     get_row(int row, array<handle<element>> &els) {}
    virtual void     get_col(int col, array<handle<element>> &els) {}
    virtual bool     get_col_x(int col, gool::range &x) { return false; }
    virtual bool     get_row_y(int row, gool::range &y) { return false; }
    virtual bool     get_row_at(view &v, int y, int &row) { return false; }
    virtual bool     get_col_at(view &v, int x, int &col) { return false; }
    virtual element *at(int row, int col) { return 0; }
    virtual bool     row_col_of(const element *el, int &row, int &col) {
      int nr = n_rows();
      int nc = n_cols();
      for (int r = 0; r < nr; ++r)
        for (int c = 0; c < nc; ++c)
          if (el == at(r, c)) {
            row = r;
            col = c;
            return true;
          }
      return false;
    }

    // caret position
    virtual bool advance(view &v, bookmark &bm, ADVANCE_DIR cmd,
                         point vpt = point());

    // drawing

    virtual void needs_draw(view &v,const rect& invalid) { 
      this->flags.was_drawn = 0; 
      if (behavior)
        behavior->invalidate_surface(this,invalid);
    }

    // virtual bool  drawing_context(view* &pv, graphics* &pgfx);

    virtual void draw(view &v, graphics *pg, point pos,
                      bool check_animator = true);

    void do_draw(view &v, graphics *pg, point pos, bool check_animator = true, bool check_layer = true);
    void do_draw_foreground(view &v, graphics *pg, point pos, bool check_animator = true);
    void do_draw_background(view &v, graphics *pg, point pos, bool check_animator = true);
    void do_draw_content(view &v, graphics *pg, point pos, bool check_animator = true);
    void do_draw_backdrop(view &v, graphics *pg, const rect& r, const filter_v& backdrop_filter);

    virtual void draw_background(view &v, graphics *pg, point pos);
    virtual void draw_border(view &v, graphics *pg, point pos);
    virtual void draw_outline(view &v, graphics *pg, point pos); // draw outline of the element itself
    virtual void draw_outlines(view &v, graphics *pg, point pos, bool draw_children,
                               bool draw_self,
                               bool apply_transform = false); // draw outlines of children
    virtual void draw_content(view &v, graphics *pg, point pos,
                              bool clip = true);
    virtual void draw_foreground(view &v, graphics *pg, point pos);
    virtual void draw_bullet(view &v, graphics *pg, point pos);
    //virtual void draw_selected_cover(view &v, graphics *pg, point pos);
    virtual void draw_selection(view &v, graphics *pg, point pos,
                                selection_ctx *psi);

    virtual void draw_shadow_dom(view &v, graphics *pg, point pos, int level);

    virtual bool is_fore_image_provider(view &v) const { return false; }

    // get surface where the element is drawn
    virtual graphics *surface(view &v, point &offset_on_surface);

    image *get_fore_image(view &v);
    image *get_back_image(view &v);

    virtual image *provide_fore_image(view &v) const;
    virtual image *provide_back_image(view &v) const;

    virtual point scroll_translate(view &v, point pos);

    virtual point translate(view &v, point pos); // translate - from "dom pixels" to "screen pixels"
    virtual point inverse_translate(view &v, point pos); // inverse translate -
                                                         // to flat, non
                                                         // transformed position
                                                         // : "screen pixels" to
                                                         // "DOM pixels"

    virtual bool compute_mtx(view &v, affine_mtx_f &mtx,point pos = point()); // compute transformation matrix (if any)
    virtual void transform(view &v, affine_mtx_f &mtx); // apply transformations to mtx

    virtual rect client_rect(view &v);
    enum RELATIVE_TO {
      TO_SELF,
      TO_PARENT,
      TO_LAYOUT_PARENT, // layout owner, sic!
      TO_DOC,
      TO_VIEW,
      TO_SCREEN,
      TO_WINDOW, // nearest window
    };

    rect outer_distance(view &v); // distance of margin_box from inner_box
    rect margin_box(view &v, RELATIVE_TO rt = TO_SELF);
    rect margin_distance(view &v); // distance of margin_box from content_box
    rect border_box(view &v, RELATIVE_TO rt = TO_SELF);
    rect border_distance(view &v); // distance of border_box from content_box
    rect padding_box(view &v, RELATIVE_TO rt = TO_SELF);
    rect padding_distance(view &v); // distance of padding_box from content_box
    rect content_box(view &v, RELATIVE_TO rt = TO_SELF);
    rect rendering_box(view &v, RELATIVE_TO rt = TO_SELF); // border_box + outlines
    rect hit_margin_distance(view &v); // distance of hit_margin_box from border_box

    rect background_clip_box(view &v, RELATIVE_TO rt = TO_SELF);
    rect foreground_clip_box(view &v, RELATIVE_TO rt = TO_SELF);

    rect outline_box(view &v, RELATIVE_TO rt = TO_SELF); // border_box + outlines + shadows
    rect outline_distance(view &v); // distance of outline_box from content_box
    rect outline_border_distance(view &v); // distance of outline_box from content_box

    //rect popup_margin_box(view& v, RELATIVE_TO rt = TO_SELF); // popup window box
    //rect popup_margin_border_distance(view &v); // distance of margin_box from border_box on popup
    //rect popup_margin_distance(view &v); // distance of margin_box from border_box on popup
    void set_popup_attachment(uint n); // 0-top, 1-right, 2-bottom, 3-left
    
    rect hit_box(view &v, RELATIVE_TO rt = TO_SELF); // border_box + hit margins
    rect image_z_box(view &v, const style::image_def &imd);
    bool get_caret_location(view &v, rect &r); // this relative.

    element* w3_offset_parent(view &v);
    // note: in dips
    pointf   w3_offset_origin(view &v); // https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/offsetTop
    sizef    w3_offset_size(view &v); // https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/offsetWidth
    pointf   w3_client_origin(view &v); // https://developer.mozilla.org/en-US/docs/Web/API/Element/clientTop
    sizef    w3_client_size(view &v); // https://developer.mozilla.org/en-US/docs/Web/API/Element/clientTop
    pointf   w3_scroll_position(view &v); // https://developer.mozilla.org/en-US/docs/Web/API/Element/scrollTop
    void     w3_scroll_position(view &v,pointf p); // https://developer.mozilla.org/en-US/docs/Web/API/Element/scrollTop
    sizef    w3_scroll_size(view &v); // https://developer.mozilla.org/en-US/docs/Web/API/Element/scrollTop

    rect rbox(view &v) override {
      if (is_box())
        return rendering_box(v, TO_VIEW); // view relative rendering box
      else
        return super::rbox(v); // view relative rendering box
    }

    rect clip_box(view &v, RELATIVE_TO rt = TO_SELF);

    virtual void refresh(view &v);

    rect client_box(view &v) { return client_rect(v); }
    rect foreground_image_box(view &v, RELATIVE_TO rt = TO_SELF);

    bool need_ellipsis(view &v);
    bool need_multiline_ellipsis(view& v);

    sizef pixels_per_dip();
    sizef pixels_per_dip(view &v);

    virtual void setup_layout(view &v);
    flow_e    check_layout(view &v);
    void         check_layout_tree(view &v);
    virtual void fixup_layout(view &v) { /* gets called as a last step of setup_layout*/ }

    virtual bool no_pixels_behind(view &v); // true if it and all its parents are transparent

    // inline entity interface
    /*virtual void  inline_draw(point pos, TEXT_DIRECTION td = TD_LTR );
    virtual size  inline_dim();
    virtual bool  inline_support_ttb();
    virtual BREAK_CONDITION inline_break_before();
    virtual BREAK_CONDITION inline_break_after();*/

    virtual float inline_baseline(view &v);

    // animation support
    template <typename TA> TA *get_animation(); // is_of_type
    bool has_animation(const animation* pa) const;

    // resource management methods:
    virtual bool on_data_arrived(view &v, pump::request *prq); // requested data just arrived
    virtual bool on_data_request(view &v, pump::request *prq); // data request is about to be sent
    virtual void accept_image(view &v, const image_ref &ir);
    virtual void used_images(view &v, function<void(const string &url, image *img)> cb);

    // scroll data
    virtual bool get_scroll_data(view &v, scroll_data &sd);
    virtual void set_scroll_pos(view &v, point pos, bool smooth,
                                bool allow_out_of_bounds);
    virtual void do_set_scroll_pos(view &v, point pos, bool smooth,
                                   bool allow_out_of_bounds);

#if 0
    // CSSS!
    virtual bool  eval_action(view &v, event &evt, eval::conduit *prg, value *prv = 0);
#endif
    virtual value eval_calc(eval::conduit *code, bool horizontal, pixels& pxc);

    // event handlers and behaviors stuff
    virtual bool on(view &v, event_mouse &evt);
    virtual bool on(view &v, event_key &evt);
    virtual bool on(view &v, event_focus &evt);
    virtual bool on(view &v, event_scroll &evt);
    virtual bool on(view &v, event_command &evt);
    virtual bool on(view &v, event_behavior &evt);
    virtual bool on_internal(view &v, event_behavior &evt);
    virtual bool on(view &v, event_gesture &evt);

    virtual bool on(view &v, event_exchange &evt);

    virtual void broadcast(view &v, event_behavior &evt);

    virtual bool on_timer(view &v, timer_def& td);
    virtual bool is_on_icon(view &v, point pt);
    virtual bool is_inside(view &v, point viewpt);

    virtual bool is_safe_to_wheel(view &v);

    bool state_on(view &v, ui_state st);
    bool state_off(view &v, ui_state st);

    bool state_focus_on(view &v, bool by_key);
    bool state_focus_off(view &v);

    // "smart" versions - delegates op to behavior (if any)
    bool set_state(ui_state st, view *pv);
    bool reset_state(ui_state st, view *pv);
    // bool _notify_disabled_status_change(view* v, bool disabled );
    void notify_disabled_status_change(view &v, bool disabled);
    void notify_readonly_status_change(view &v, bool readonly);
    void notify_attribute_change(view &v, const name_or_symbol &nm);

    virtual element *create_context_menu(view &v, const string &url, event_behavior &evt);

    // windowing and scrolling
    // virtual HWND hwindow();
    virtual iwindow *window(view &v);

    virtual element *get_scrollable_container(view &v,
                                              bool  including_this = false);
    virtual element *get_windowed_container(view &v,
                                            bool  including_this = false);
    virtual iwindow *get_window(view &v, bool including_this = false);
    virtual bool     scroll_to_view(view &v, handle<element> child, bool toTop,
                                    ENSURE_VISIBLE_MANNER manner);
    virtual bool     can_be_seen_in_full(view &v);

    virtual bool set_value(view &v, const value &val, bool ignore_text_value = false);
    virtual bool get_value(view &v, value &val, bool ignore_text_value = false);
    
            bool get_attr_value(value& val); // @value + @as, returns `false` if no value attribute 

    virtual bool default_set_value(view &v, const value &val, bool ignore_text_value);
    virtual bool default_get_value(view &v, value &val, bool ignore_text_value);

    virtual bool get_var(const element_context& ctx, name_or_symbol nm, value& v, const style* ps = nullptr) const;
    //virtual bool get_color_var(name_or_symbol nm, color_v& v, const style* ps = nullptr) const;
    //virtual bool get_length_var(name_or_symbol nm, size_v& v, const style* ps = nullptr) const;
    //virtual bool resolve_color(uint attr_sym, color_v& val) const override { return get_color_var(attr_sym, val); }
    //virtual bool resolve_current_color(color_v& val) const override { val = c_style->font_color; return true; }
         //color_v resolve_color(color_v v) const { return unname(v,this); }
         //size_v  resolve_length(size_v v) const { return unname(v,this); }
    
    element* get_marker() const {
      return style_content ? style_content->marker : nullptr;
    }
    element* get_shade() const {
      return style_content ? style_content->shade : nullptr;
    }

    bool has_shadow_dom() const 
    {
      if (!style_content) return false;
      return style_content->marker.is_defined() || style_content->shade.is_defined();
    }

    virtual bool wants_keyboard(view &v);

    void drop_caches() {
      // dbg_report("element::drop_caches");
      each_any_child([](element *el) -> bool {
        el->drop_caches();
        return false;
      });
      drop_cache();
    }

    virtual void drop_cache() {}
    
    virtual void check_spelling(view &v, spell_checker *sc) {
      each_ui_child([&v, sc](element *el) -> bool {
        el->check_spelling(v, sc);
        return false;
      });
    }

    bool allow_reconciliation() const
    {
#if defined(SCITER) || defined(SCITERJS)

      switch(flags.ssx_reconciliation) 
      {
        case 1: return false;
        case 2: return true;
        default: {
          if (behavior)
            return !!behavior->r13n_container(const_cast<element*>(this));
          //return atts.exist(attr::a_src);
          return true;
        }
      }
#else
      if (behavior)
        return !!behavior->r13n_container(const_cast<element*>(this));
      return true;
#endif
    }

    element* reconciliation_container()
    {
#if defined(SCITER) || defined(SCITERJS)
      switch (flags.ssx_reconciliation)
      {
        case 1: return nullptr;
        case 2: return this;
        default: {
          if (behavior)
            return behavior->r13n_container(const_cast<element*>(this));
          return this;
        }
      }
#else
      return const_cast<element*>(this);
#endif
    }

    // for <select|dropdown> it returns <popup> - container of <option>s
    element* logical_container() {
      if (behavior)
        return behavior->logical_container(const_cast<element*>(this));
      return this;
    }

    void allow_reconciliation(tristate_v v) {
#if defined(SCITER) || defined(SCITERJS)
      if (v.is_undefined())
        flags.ssx_reconciliation = 0;
      else if(v.val(0))
        flags.ssx_reconciliation = 2; // explicitly enable reconciliation 
      else 
        flags.ssx_reconciliation = 1; // explicitly disable reconciliation 
#endif
    }
    
    template <class TA> TA *get_animation_of_type();

    int get_list_index();
    // bool is_active_dd_target(view& v, element* dragging);
    element *get_target(view &v, bool only_visible) const; // get "for" element

    void post_custom_event(view &v, const ustring &name, uint_ptr reason = 0);
    bool send_custom_event(view &v, const ustring &name, uint_ptr reason = 0);

    slice<hnode> get_nodes(view &v, array<hnode> &buf);

#if defined(SCITERJS)
    //script_expando_interface
    virtual bool get_expando(context& hc, script_expando& seo) override;
#endif
    
    // data
    tag::symbol_t tag;
    attribute_bag atts;
    ui_state         state;
    mutable ui_state style_state; // has style rules that use these flags
    mutable ustring lang;
    mutable ustring theme;
    element_flags flags;
    weak_handle<element> positioned_parent; // container of position:relative | absolute | fixed;
    // size                  rendered_dim;
    array<hnode>         nodes;
    handle<layout_data>  ldata;

    hstyle          c_style; // current used style, is not null
    hstyle          d_style; // defined/determined style - computed from CSS
                             // declarations, is not null
    hstyle          p_style; // previously used style, is not null
    hstyle_prop_map a_style; // assigned style, can be null - element.style["..."] = ...;
    hstyle          p_drawn_style; // previously drawn style, is not null

    //OBSOLETE eval::conduit * last_act_assigned; // last act_assigned executed on the
                                            // element, the element does not own it,
                                            // this field is used only for address
                                            // comparison

    handle<animation>    animator; // list of animators
    handle<ctl>          behavior;
    handle<airborn_ctx>  airborn;
    handle<css_content>  style_content;
    attribute_bag_v      vars;     // css vars, set directly

    array<handle<resource>> meta; // various meta data (x_resources) associated with the element

    mutable handle<style_set_holder> forced_style_set_holder;

    weak_handle<element> event_owner;  // ~~ logical owner, most of the time it is an owner but in some contexts (popup, "portal") it can be different.

#ifdef _DEBUG
    tristate_v painted;
#endif

    static handle<style>         null_style;
    static handle<layout_data>   null_layout_data;

    static THREAD_LOCAL element* drawing_element;
  };

  typedef handle<element> helement;
  typedef weak_handle<element> weak_helement;

  struct timer_def {
    weak_helement blk;
    uint_ptr    uid = 0; // unique identifier
    timer_id    id = 0;
    uint        ms = 0;
    uint        until = 0; // cutoff time
    TIMER_KIND  kind = STOPPED_TIMER;
    uint_ptr    sys_id = 0;
#if defined(SCITERJS)
    handle<resource> pcb;
#endif
    //timer_def() {}

    bool is_script_timer() const {
      return (kind == SCRIPT_TIMER || kind == SCRIPT_INTERVAL || kind == SCRIPT_ELEMENT_TIMER);
    }

  };


  // template <typename TO_TYPE>
  //  inline bool is_ldata_of_type(element* p)
  //  {
  //    return p->ldata->is_of_type<TO_TYPE>();
  //  }

  /*  template <typename TO_T>
      inline TO_T* turn_element_to(element* p)
      {
        assert_static( sizeof(element) == sizeof(TO_T));
        if(   p->layout_type() == TO_T::LAYOUT_TYPE
          &&  p->ldata->is_of_type<TO_T::layout_data>())
          return static_cast<TO_T*>(p);

        //if(   p->layout_type() == TO_T::LAYOUT_TYPE
        //  &&  is_ldata_of_type<TO_T::layout_data>(p))
        //  return static_cast<TO_T*>(p);

        flow_e ft = p->layout_type();

        p->flags.layout_ctl_valid = 1;
        TO_T* pt = ft != TO_T::LAYOUT_TYPE
                    ? ::new(p) TO_T(NO_INIT())
                    : static_cast<TO_T*>(p);

        pt->ldata = new typename TO_T::layout_data(pt);

        return pt;
      }
   */

  // percentage calculation primitives, these are called when somethin is
  // defined in percents
  int  known_height(view &v, element *b);
  int  known_height_of_parent(view &v, element *b);
  bool is_height_of_parent_defined(view &v, element *b);
  int  known_width(view &v, element *b);
  int  known_width_of_parent(view &v, element *b);
  bool is_width_of_parent_defined(view &v, element *b);

  class tree_scanner {
    array<helement> roots;

  public:
    tree_scanner(element *root) {
      if (root) roots.push(root);
    }
    void add_root(element *root) {
      if (root) roots.push(root);
    }

    bool each_node(const function<bool(node *)> &f);
    bool each_element(const function<bool(element *)> &f);
  };

  element *find_first(view &v, element *root, wchars selector,
                      bool only_visible = false, bool inside_frames = false);
  void     find_all(view &v, array<helement> &r, element *root, wchars selector,
                    bool only_visible = false);
  void     find_all(view &v, element *root, wchars selector,
                    const function<bool(element *)> &f, bool only_visible = false);

  void            find_all_parents(view &v, array<helement> &r, element *b,
                                   wchars selector);
  element *       find_first_parent(view &v, element *b, wchars selector,
                                    uint depth = 0);
  element *       find_first_parent(view &v, element *root, element *b,
                                    wchars selector, uint depth = 0);
  element *       find_first_parent_or_root(view &v, element *root, element *b, wchars selector);
  element *       find_first_ui_parent(view &v, element *root, element *b,
                                       wchars selector, uint depth = 0);
  element *       find_first_ex(view &v, element *root, wchars selector,
                                wchars parent_selector, bool only_visible = false);
  inline element *matches(view &v, element *root, element *b, wchars selector) {
    return find_first_parent(v, root, b, selector, 1);
  }
  inline element *matches(view &v, element *b, wchars selector) {
    return find_first_parent(v, b, selector, 1);
  }

  void emit_range_html(view &v, ostream &out, const bookmark &start,
                       const bookmark &end, element* root);
  void emit_range_text(view &v, ostream &out, const bookmark &start,
                       const bookmark &end);
  void emit_cell_range_html(view &v, ostream &out, helement tbody,
                            slice<helement> cells);
  void emit_cell_range_text(view &v, ostream &out, helement tbody,
                            slice<helement> cells);

  // implementations

  inline element *node::nearest_box(bool ui) // closest box or this
  {
    if (ui)
      for (element *p = get_ui_element(); p; p= p->get_owner()) {
        if (p->is_box() || !p->parent) return p;
      }
    else
      for (element *p = get_element(); p; p = p->parent) {
        if (p->is_box() || !p->parent) return p;
      }

    return nullptr;
  }

  inline element *node::nearest_block(view &v) {
    for (element *p = get_element(); p; p = p->parent) {
      if (p->is_block_element(v) || !p->parent) return p;
    }
    return nullptr;
  }

  inline element *node::nearest_box_or_row(bool ui) // closest box or this
  {
    if (ui)
      for (element *p = get_ui_element(); p; p = p->get_owner()) {
        if (p->is_box() || p->is_table_row() || !p->parent) return p;
      }
    else
      for (element *p = get_element(); p; p = p->parent) {
        if (p->is_box() || p->is_table_row() || !p->parent) return p;
      }

    return 0;
  }

  inline element *node::nearest_text_box() // closest box or this
  {
    element *owner = get_owner();
    element *ptb = owner ? owner->nearest_box() : 0;
    if (ptb && ptb->is_text_box()) return ptb;
    return nullptr;
  }

  inline void node::denormalize(bookmark &bm) const {
    assert(bm.node == this);
    if (bm.after_it) {
      bm.pos      = bm.pos + 1;
      bm.after_it = false;
    }
  }

  inline bool node::belongs_to(element *pb, bool include_this) {
    node *t = include_this ? this : parent;
    while (t) {
      if (t == pb) return true;
      t = t->parent;
    }
    return false;
  }

  inline bool node::owned_by(element *pb, bool include_this) {
    node *t = include_this ? this : get_owner();
    while (t) {
      if (t == pb) return true;
      t = t->get_owner();
    }
    return false;
  }

  inline void node::index_stack(array<int> &   indexes,
                                const element *root) const {
    if (this == root) return;
    if (parent) {
      parent->index_stack(indexes, root);
      indexes.push(node_index << 1);
    } else
      indexes.push(node_index << 1);
  }

  inline ustring node::get_lang() const {
    if (parent) return parent->get_lang();
    return ustring();
  }
  inline ustring node::get_theme() const {
    if (parent) return parent->get_theme();
    return ustring();
  }


  inline void element::normalize(bookmark &bm) const {
    assert(bm.node.ptr() == this);
    if (bm.pos < 0) {
      bm.pos      = 0;
      bm.after_it = false;
    } else if (bm.pos >= this->nodes.size()) {
      bm.pos      = (int)this->nodes.last_index();
      bm.after_it = true;
    }
  }

  template <typename TA> inline TA *element::get_animation() {
    for (animation *pa = animator; pa; pa = pa->next)
      if (pa->is_of_type<TA>()) return static_cast<TA *>(pa);
    return 0;
  }

  inline bool element::has_animation(const animation* tpa) const
  {
    for (animation *pa = animator; pa; pa = pa->next)
      if (pa == tpa) return true;
    return false;
  }

  inline bool element::has_child_of_type(tag::symbol_t sym) const {
    int cnt = 0;
    for (const element *pc = first_element(); pc; pc = pc->next_element()) {
      if (pc->tag == sym) ++cnt;
    }
    return cnt == 1;
  }
  inline bool element::has_children_of_type(tag::symbol_t sym) const {
    int cnt = 0;
    for (const element *pc = first_element(); pc; pc = pc->next_element()) {
      if (pc->tag == sym) {
        if (++cnt >= 2) return true;
      }
    }
    return cnt > 0;
  }

  inline bool element::is_only_child_of_type() const {
    if (!parent) return false;
    const element *pc;
    for (pc = prev_element(); pc; pc = pc->prev_element()) {
      if (pc->tag == tag) return false;
    }
    for (pc = next_element(); pc; pc = pc->next_element()) {
      if (pc->tag == tag) return false;
    }
    return true;
  }

  template <class TA> inline TA *element::get_animation_of_type() {
    for (handle<animation> ta = animator; ta; ta = ta->next)
      if (ta->is_of_type<TA>()) return ta.ptr_of<TA>();
    return 0;
  }

  template<typename F>
  inline bool element::each_child(F func) {
    handle<element> holder = this;
    for (uint n = 0; n < nodes.length(); ++n) {
      if (!nodes[n]->is_element()) continue;
      handle<node> pn = nodes[n];
      if (func(pn.ptr_of<element>())) return true;
    }
    return false;
  }

  template<typename F>
  inline bool element::each_child(F func, BACKWARD_E) {
    handle<element> holder = this;
    for (index_t n = nodes.size(); n-- > 0;) {
      if (n >= nodes.size())
        return false; // model have changed while traversing?
      if (!nodes[n]->is_element()) continue;
      handle<node> pn = nodes[n];
      if (func(pn.ptr_of<element>())) return true;
    }
    return false;
  }

  inline element *node::parent_block(view &v) {
    for (element *p = parent; p; p = p->parent) {
      if (p->is_block_element(v)) return p;
    }
    return 0;
  }

  inline element *element::owner_block(view &v) {
    for (element *p = get_owner(); p; p = p->get_owner()) {
      if (p->is_block_element(v)) return p;
    }
    return 0;
  }
  /*inline element* element::nearest_box()
  {
    for(element* p = this; p; p = p->owner?p->owner:p->parent )
    {
      if( p->is_box() || !p->parent )
        return p;
    }
    return 0;
  }*/

  inline bool element::each_ui_child(function<bool(element *)> func) {
    if (!style_content) {
      if (each_child(func)) return true;
    }
    else {
      if (style_content->before && func(style_content->before))
        return true;
      if (each_child(func)) return true;
      if (style_content->after && func(style_content->after))
        return true;
      if (style_content->marker && func(style_content->marker))
        return true;
      if (style_content->shade) {
        if(func(style_content->shade))
          return true;
      }
    }
    return false;
  }
  inline bool element::each_ui_child(function<bool(element *)> func,
                                     BACKWARD_E                       b) {
    if (style_content && style_content->marker && func(style_content->marker))
      return true;
    if (style_content && style_content->shade && func(style_content->shade))
      return true;
    if (style_content && style_content->after && func(style_content->after))
      return true;
    if (each_child(func, b)) return true;
    if (style_content && style_content->before && func(style_content->before))
      return true;
    return false;
  }

  template<typename F>
  inline bool element::each_any_child(F f) {
    handle<element> holder = this;

#ifdef _DEBUG
    if(holder->is_document() && !parent)
      parent = parent;
#endif

    if (n_children() < 8) {  // see: https://terrainformatica.com/2017/10/15/when-linear-search-is-faster-than-stdmapfind-and-stdunordered_mapfind/
      array<element*> visited(n_children()); visited.clear(); 
      auto scan = [&](element* el) {
        if (visited.get_index(el) >= 0)
          return false;
        visited.push(el);
        return f(el);
      };
      return each_ui_child(scan) || each_child(scan);
    }
    else {
      hash_table<uint_ptr, bool> visited;
      auto scan = [&](element* el) {
        bool created = false;
        visited.get_ref(uint_ptr(el), created);
        if (created)
          return f(el);
        else
          return false;
      };
      return each_ui_child(scan) || each_child(scan);
    }
    //else
    //  return each_child(f);
  }


  template <typename ET> inline bool element::handles() const {
    ctl *pbh = behavior;
    while (pbh) {
      if ((pbh->subscription & ET::EVENT_GROUP) != 0) return true;
      pbh = pbh->next;
    }
    return false;
  }




  // inline bookmark::bookmark(element* el, bool after_it ):tb(0),tb_pos(0) {
  // node = el->parent; pos = after_it? el->node_index+1 : el->node_index; }
  // inline bookmark::bookmark(text* nd, int p ):tb(0),tb_pos(0) { node = nd;
  // pos = p; }
  inline bookmark::bookmark(html::node *nd, int p, bool after)
      : node(nd), pos(p), /*tb(0),tb_pos(0),*/ after_it(after) {}

  inline uint bookmark::marks() const {
    return valid() ? node->marks_at(pos) : 0;
  }
  inline bool bookmark::marks(view &v, uint mid) {
    return valid() ? node->marks_at(v, pos, mid) : false;
  }
  inline bool bookmark::marks_contains(chars markcls) const {
    return valid() ? node->marks_at_contains(pos, markcls) : false;
  }
  inline bool bookmark::at_block_element_end(view& v) const {
    return valid() && node->is_element() && node.ptr_of<element>()->is_block_element(v) && node->is_end_pos(*this);
  }
  inline bool bookmark::at_block_element_start(view& v) const {
    return valid() && node->is_element() && node.ptr_of<element>()->is_block_element(v) && node->is_start_pos(*this);
  }

  inline bool bookmark::at_inline_block_element_start(view& v) const {
    return valid() && node->is_element() && node.ptr_of<element>()->is_inline_block_element(v) && node->is_start_pos(*this);
  }
  inline bool bookmark::at_inline_block_element_end(view& v) const {
    return valid() && node->is_element() && node.ptr_of<element>()->is_inline_block_element(v) && node->is_end_pos(*this);
  }


  inline bool bookmark::at_br_pos() const {
    return valid() && node->is_element() && (node.ptr_of<element>()->tag == tag::T_BR);
  }



  inline bookmark node::this_pos(bool tail) const {
    return bookmark(parent, node_index, tail);
  }
  inline bookmark node::start_caret_pos(view &v) const { return start_pos(); }
  inline bookmark node::end_caret_pos(view &v) const { return end_pos(); }

  inline bookmark text::start_pos() const {
    return bookmark(const_cast<text *>(this), 0, false);
  }
  inline bookmark text::end_pos() const {
    return chars.size() != 0 ? bookmark(const_cast<text *>(this),
                                        (int)chars.last_index(), true)
                             : bookmark(const_cast<text *>(this), 0, false);
  }

#ifdef _DEBUG
  inline void text::dbg_report(const char *cap) {
    ustring t = chars();
    dbg_printf("%s text(%S)[%d] in", cap, t.c_str(), node_index);
    if (parent) parent->dbg_report();
  }
  inline void comment::dbg_report(const char *cap) {
    dbg_printf("%s {comment} in", cap);
    if (parent) parent->dbg_report();
  }
#endif

  rect rbox(view &v, bookmark start, bookmark end);

  uint apply_marks(view &v, bookmark start, bookmark end, slice<string> marks);
  inline uint apply_marks(view &v, bookmark start, bookmark end, string mark) { return apply_marks(v, start, end, slice<string>(mark)); }
  uint clear_marks(view &v, bookmark start, bookmark end, slice<string> marks);
  inline uint clear_marks(view &v, bookmark start, bookmark end, string mark) { return clear_marks(v, start, end, slice<string>(mark)); }
  bool get_marks_span(bookmark &start, bookmark &end, ustring &word,chars mark1);
  uint marks_id_of_range(view &v, bookmark start, bookmark end);

  bool advance(view &v, element *self, bookmark &bm, ADVANCE_DIR cmd,
               point vpt = point());

  struct emit_ctx {
    bookmark         anchor;
    bookmark         caret;
    function<void()> at_anchor; // called when anchor is encounetred
    function<void()> at_caret;  // called when caret is encounetred
    element *        root;      // emission root - document or element
  };

  void check_prop_used_by_style_content(view& v, element *pel, wchars name);

  struct morph_options {
    bool children_only = false;
  //virtual bool is_same(node* new_node, node* old_node) { return false; }
    virtual bool change_text(text* node, wchars to_text) { return true; }
    virtual bool remove_node(node* old_node) { return true; }
    virtual bool insert_node(node* el, int index, node* node) { return true; }
    virtual bool replace_node(node* old_node, node* by_node) { return true; }
    virtual bool reposition_node(node* el, int index, node* node) { return true; }
    virtual bool update_atts(node* of, const attribute_bag& from) { return true; }
  };


} // namespace html
#endif
