#ifndef __html_document_h__
#define __html_document_h__

#include "tool/tool.h"
#include "html-dom.h"
#include "html-style.h"
#include "html-pump.h"
#include "html-style-parser.h"

#if defined(SCITERJS)
namespace qjs{ 
  class xview;
}
#endif

namespace html {
  using namespace tool;

  struct media_vars_provider {
    virtual media_variables &media_vars();
  };

  extern media_vars_provider default_media_vars_provider;

  namespace behavior {
    struct frame_ctl;
    struct richtext_ctl;

    void init(bool start);

  } // namespace behavior

  class view;

  struct running_action;

  struct document : public block_vertical 
  {
    friend class view;
    friend class application;
    friend struct behavior::frame_ctl;
    friend struct behavior::richtext_ctl;

    bool allow_own_scripts = true; // <script>
    bool allow_own_styles = true; // <style>
    bool allow_script_behaviors = true; // prototype and aspect in CSS
    //bool is_reloading = false;

    document() : document(string("about:blank")) {}

    document(const string &url, tag::symbol_t t = tag::T_HTML)
      : super(t), _view(0), _media_vars_provider(0), _isolated(false),
      style_serial_no('A' - 1)
    {
      _uri.parse(url);
      // turn_element_to<document>(this);
      ldata = new layout_data(this);
    }

    document(NO_INIT ni) : super(ni) { assert(false); }
    document(const document *src);

    virtual ~document();

    DEFINE_TYPE_ID_DERIVED(document, block_vertical);

    virtual view *pview() const { return parent ? parent->pview() : _view.ptr(); }
    virtual void  pview(view *pv);
    // virtual bool  drawing_context(view* &pv, graphics* &pgfx);

    virtual document *doc() const override {
      return const_cast<document *>(this);
    }
    virtual document *parent_doc() const {
      return parent ? parent->doc() : nullptr;
    }

    virtual bool is_connected() const override {
      if (parent)
        return node_index != UNDEFINED_NODE_INDEX;
      return !_view.is_null();
    }

#ifdef _DEBUG
    virtual void dbg_report(const char *cap = "") {
      element::dbg_report(cap);
      //dbg_printf("doc url=%s\n",doc_url().c_str());
    }
#endif  


    virtual ustring get_lang() const override;
    virtual ustring get_theme() const override;
    virtual style * get_base_style(view &v) override;

    element* get_body();
    element* get_head();

    virtual point view_pos(view &v) override;
    virtual point doc_pos(view &v) override { return point(0, 0); }
    virtual void  view_mtx(view &v, affine_mtx_f &vmx) override;

    virtual const url &uri() const { return _uri; }
    virtual void       uri(const url &u) { _uri = u; }
    virtual void       base_uri(const url &u) { _base_uri = u; }
    virtual const url &base_uri() const { return _base_uri.is_defined() ? _base_uri : _uri; }

    virtual bool is_document() const { return true; }
    virtual bool is_floats_container(view &v) { return true; }
    virtual bool is_large() const;

    virtual bool each_ui_child(function<bool(element *)> func) override;

    virtual void     setup_layout(view &v);
    virtual element *drop_layout(view *pv = 0);

    /*virtual void     drop_layout_tree(view *pv, bool and_drop_cache = false) {
       super::drop_layout_tree(pv, and_drop_cache);
    }*/

    virtual void stray(view &v); // a.k.a. on_document_unload, element is about
                                 // to be completely removed from the DOM, clear
                                 // all runtime structures

    void measure(view &v, size sz);

    int paginate(view &v, gool::size page_content_size,
                 array<gool::range> &pages);

    // style management:
    virtual style_bag &styles() const {
      return const_cast<document *>(this)->_styles;
    }

    static void  destroy_stock_styles();
    virtual void reset_styles(view &v);
    bool         reset_styles(wchars text, const string &base_url = string());
    virtual void handle_style(element *style_elem);
    virtual bool load_style(element *style_elem, wchars text);
    virtual bool load_style(const string& url);
    style_bag *  get_named_style_set(const string &set_name);

    // resource management:
    virtual image_ref register_image(/*in-out*/ string &image_url);
    virtual void      release_image(image_ref &imr);
    virtual bool      register_image_map(ustring                  name,
                                         slice<pair<string, int>> urls_and_DPIs,
                                         size cells, slice<value> items);
    virtual image_ref register_image_fragment(ustring map_name,
                                              ustring part_name);

    virtual cursor *load_cursor(const string &cursor_url,
                                size *        photspot = nullptr);

    virtual bool on_data_arrived(view &v, pump::request *prq); // requested data just arrived

    virtual void image_data_arrived(view &v, handle<pump::request> hrq);
    virtual void style_data_arrived(view &v, handle<pump::request> hrq);
    virtual void cursor_data_arrived(view &v, handle<pump::request> hrq);

    void   image_arrived(view &v, image *pimg);
    image *get_image(const string &url);
    bool   drop_image(const string &url);

    // media vars
    bool find_media_var(wchars name, value &v);

    // other stuff
    //bool has_pseudo_classes_for(element *el, ui_state st, bool deep);

    virtual void     drop_cache() override;
    virtual iwindow *window(view &v);

    virtual bool is_root_document() const { return parent == 0 && _view; }
    virtual bool is_fragment_document() const {
      return (parent == 0) && flags.is_synthetic;
    }
    virtual void  update_scrollbars(view &v) override;
    virtual bool  get_scroll_data(view &v, scroll_data &sd) override;
    virtual point scroll_pos() const override;
    virtual void  scroll_pos(view &v, point off,
                             bool allow_out_of_bounds = false) override;
    virtual bool  can_scroll_h(view &v) const override;
    virtual bool  can_scroll_v(view &v) const override;

    bool start_eval(element *b, eval::conduit *prg);
    void stop_eval(element *b, eval::conduit *prg);

    // document specific event handlers and behaviors stuff
    virtual bool on(view &v, event_behavior &evt) override;
    virtual bool on(view &v, event_scroll &evt) override { return super::on(v, evt); }
    virtual bool on(view &v, event_mouse &evt) override;
    virtual bool on(view &v, event_key &evt) override;
    virtual bool on(view &v, event_focus &evt) override;
    virtual bool on_timer(view &v, timer_def& td) override;

    enum TOOLTIP_TYPE {
      ID_OF_TOOLTIP,
      PLAIN_TOOLTIP,
      HTML_TOOLTIP,
      ELEMENT_TEXT_TOOLTIP,
    };
    element *create_tooltip_element(view &v, ustring text, TOOLTIP_TYPE pupt,
                                    element *b);
    // void remove_tooltips(view& v);
    void register_popup(view &v, element *popup);
    void register_service_block(element *popup);

    // A11Y:
    bool a11y_get_name(view &v, ustring &name) override;

    // <!ENTITY ...>
    bool resolve_entity(chars name, ustring &val);
    void register_entity(chars name, wchars val);

    virtual node *clone() override { return new document(this); }

    bool debug_mode() const;
    void debug_mode(bool on) { 
      _debug_mode = on;
    }

    // get by ID optimization
    element *get_element_by_id(const ustring &id) override;
    void     add_element_id(element* pel, const ustring &id);
    void     remove_element_id(element* pel, const ustring &id);

#if defined(SCITERJS)
    //script_expando_interface
    virtual bool get_expando(context& hc, script_expando& seo) override;
#endif

#if defined(USE_INCREMENTAL_INIT)
    // delayed initialization
    bool request_delayed_init(view& v, element* pel);
    bool process_delayed_init(view& v);
#endif    

    // pools
    pool<string>                       string_pool;
    pool<ustring, ustring_ignore_case> font_name_pool;
    pool<ustring>                      ustring_pool;
    handle_pool<function_value>        function_pool;

  protected: // data
    doc_style_bag              _styles;
    url                        _uri;
    url                        _base_uri;
    handle_pool<eval::conduit> _actions;
    array<running_action>      _running_actions;
    array<helement>            _synthetic_elements; // tooltips
    weak_handle<view>          _view;
    media_vars_provider *      _media_vars_provider = nullptr;
    bool _isolated; // true if this document is isolated from normal flow, used
                    // by pager
    image_bag                          _images;
    image_map_bag                      _image_maps; // @image-map things
    hash_table<string, handle<cursor>> _cursors;

    hash_table<string, ustring> _entities;

    tool::hash_table<ustring, weak_handle<element>> id2element;

    weak_handle<element> _body;
    weak_handle<element> _head;

#if defined(USE_INCREMENTAL_INIT)
    // delayed initialization
    array<weak_handle<element>> _delayed_initialization_queue; // elements delayed for layout (text blocks mostly)
    uint                        _processed_text_blocks = 0;
#endif

  public:
    handle_pool<eval::conduit> media_expressions; // list of media expressions
                                                  // used in CSS, these
                                                  // expressions are evaluated
                                                  // on media_type(const char*
                                                  // t) change.

    int_v num_resources_requested;
    int_v num_styles_requested;

    uint style_serial_no; // starts from 'A' - 1

    tristate_v operational; // true after load finsihes and until pre unload

#if defined(SCITER)
    tis::value ns = 0; // namespace of loaded document
#elif defined(SCITERJS)
    qjs::hcontext ns;
#endif

    handle<gool::application> app; // the document holds application object as
                                   // it contains image_bag's that can use gfx
                                   // resources that need to be cleared before
                                   // the app gets to eternity.

    handle<request> pdocrq;
    bool            _debug_mode = false;

  };

  struct document_fragment : public element {
    DEFINE_TYPE_ID_DERIVED(document_fragment, element);
    document_fragment() : super(tag::T__FRAGMENT) { flags.is_synthetic = 1; }
    document_fragment(NO_INIT ni) : super(ni) {}

    virtual bool is_document_fragment() const override { return true; }
    virtual bool is_connected() const override { return false; }
    virtual bool wraps_nodes() const { return true; }
    virtual bool unwrap_nodes(array<handle<node>>& wn) { 
      nodes.swap(wn);
      for (auto n : wn) {
        n->parent = nullptr;
        n->owner = nullptr;
        n->node_index = UNDEFINED_NODE_INDEX;
      }
      return true;
    }
  };

  typedef tool::handle<document> hdocument;

  struct selector_context : public element_visitor_context {
    tool::array<tool::handle<html::style_def>> sds;
    helement                                   root;
    bool                                       deep;
    bool                                       only_visible;

    selector_context(element *root_el, wchars selector,
                     bool do_visible_only = true, bool do_deep = false);

    ~selector_context() {}

    bool is_valid() const { return sds.size() != 0; }

    virtual bool skip(view &v, element *b) {
      if (!deep && b->is_document())
        return true; // skip content of documents loaded in frames.
      return false;
    }

    virtual bool accept(view &v, element *b) {
      index_t n = sds.size();
      if (only_visible && !b->is_it_visible(v)) return false;
      for (index_t i = 0; i < n; ++i)
        if (sds[i]->match(b, root)) return true;
      return false;
    }
  };

  // used to prevent recursive invocation
  struct running_action {
    const element *el;
    eval::conduit *prg;
    bool           operator==(const running_action &rs) const {
      return el == rs.el && prg == rs.prg;
    }
  };

  bool              exec_command(view &v, element *source, element *target,
                                 const ustring &command, const value &params = value());
  pair<bool, value> query_command(view &v, element *source, element *target,
                                  const ustring &command,
                                  const value &  params = value());
  bool              value_to_attribute_bag(const value &v, attribute_bag &atts);
} // namespace html

#endif
