#include "html.h"
#include "html-behaviors.h"

#include "sdk-headers.h"
#ifdef SCITERJS
  #include "sdk.js/include/sciter-x-video-api.h"
#else
  #include "sdk/include/sciter-x-video-api.h"
#endif

namespace html {
  namespace behavior {

#if defined(USE_VIDEO)

    struct fragmented_video_destination : public sciter::fragmented_video_destination
    {
      value renderingSite() {
        return value::wrap_asset(this);
      }
    };

    class zero_video_ctl : public fragmented_video_destination, public video_ctl {
      typedef video_ctl super;

      handle<gool::color_space_converter> converter;

      sciter::om::hasset<sciter::video_source> _src;

      tristate_v _is_open;
      tristate_v _is_playing;
      long       _generation;
      rect       _invalid;
      
    public:
      zero_video_ctl() : _generation(0) {}
      virtual ~zero_video_ctl() {}

      long asset_add_ref() override {
        //return ctl::add_ref();
        return video_ctl::add_ref(); 
      }
      long asset_release() override {
        //return ctl::release();
        return video_ctl::release(); 
      }

      virtual bool attach(view &v, element *self) override {
        return super::attach(v, self);
      }

      // sciter::video_destination stuff:

      //virtual long add_ref() override { return super::add_ref(); }
      //virtual long release() override { return super::release(); }
      virtual long asset_get_interface(const char *name, void **out) override {
        if (strcmp(name, VIDEO_DESTINATION_INAME) == 0) {
          this->add_ref();
          *out = static_cast<sciter::fragmented_video_destination*>(this);// ->as_asset();
          return true;
        } else if (strcmp(name, FRAGMENTED_VIDEO_DESTINATION_INAME) == 0) {
          this->add_ref();
          *out = static_cast<sciter::fragmented_video_destination*>(this);// ->as_asset();
          return true;
        }
        return false;
      }

      virtual bool is_alive() override { return _view && _self; }
      virtual bool start_streaming(int frame_width // width
                                   , int frame_height // height
                                   , int color_space // COLOR_SPACE above
                                   , sciter::video_source *src = 0) override // video_source interface implementation, can be null
      {
        critical_section _(_guard);
        size             dim(frame_width, frame_height);
        frame = new bitmap(dim, color_space == COLORSPACE_RGB32, true);
        // frame->set(rect(dim),gool::argb(0,0,0,0));
        converter = gool::color_space_converter::factory((COLORSPACE)color_space, dim);
        _src     = src;
        _box     = rect();
        _invalid = rect();
        on_start_streaming();
        notify_video_frame_ready(); // "statechange" to notify new foreground-image
        return frame.ptr() && converter.ptr();
      }

      virtual bool stop_streaming() {
        critical_section _(_guard);
        frame     = nullptr;
        converter = nullptr;
        _src      = nullptr;
        on_stop_streaming(true);
        return true;
      }
      // render frame request, false - video_destination is not available (
      // isn't alive, document unloaded etc.)
      virtual bool render_frame(const BYTE *frame_data, UINT frame_data_size) {
        if (!is_alive()) return false;

        critical_section _(_guard);

        handle<gool::bitmap> _frame = frame;

        if (!_frame) return false;

        _frame->set_bits(bytes(frame_data, frame_data_size), converter);

        if(_view && _self)
          notify_frame_ready(*_view,_self);

        return true;
      }

      virtual bool render_frame_with_stride(const BYTE* frame_data, UINT frame_data_size, UINT stride) {
        if (!is_alive()) return false;

        critical_section _(_guard);

        handle<gool::bitmap> _frame = frame;

        if (!_frame) return false;

        _frame->set_bits(bytes(frame_data, frame_data_size), stride, converter);
        
        if (_view && _self)
          notify_frame_ready(*_view, _self);

        return true;
      }

      virtual bool render_frame_part(const BYTE *frame_data,
                                     UINT frame_data_size, int x, int y,
                                     int width, int height) {
        if (!is_alive()) return false;

        critical_section _(_guard);

        handle<gool::bitmap> _frame = frame;

        if (!_frame) return false;

        array<gool::argb> buffer;
        slice<gool::argb> data;

        buffer.size(width * height);

        if (converter->is_rgb32()) {
          buffer.target().copy(reinterpret_cast<const argb *>(frame_data));
          buffer.target().each([](gool::argb &c) { c = c.premultiply(); });
          // for(int n = 0; n < buffer.size(); ++n)
          //    buffer[n] = buffer[n].premultiply();
        } else {
          converter->convert_to_rgb32(bytes(frame_data, frame_data_size),
                                      buffer.begin());
        }
        data = buffer();

        _frame->copy(point(x, y), data, size(width, height));

        if (_invalid.empty())
          _invalid = rect(point(x, y), size(width, height));
        else
          _invalid |= rect(point(x, y), size(width, height));

        if (_view && _self)
          notify_frame_ready(*_view, _self);

        return true;
      }

      // end of sciter::video_destination stuff

      virtual bool load_movie(const tool::url &url) override { return true; }
      virtual void close_movie() override {
        if (_src) _src->stop();
      }

      virtual bool play() override {
        if (_src) return _src->play();
        return true;
      }
      virtual void stop() override {
        if (_src) _src->pause();
      }
      virtual bool is_movie_open() const override {
        return _is_open.val(0) != 0;
      }
      virtual bool is_playing() const override {
        return _is_playing.val(0) != 0;
      }
      virtual bool is_movie_ended() const override {
        bool at_end = false;
        if (_src) _src->get_is_ended(at_end);
        return at_end;
      }

      virtual double get_movie_duration() const override {
        double d = 0;
        if (_src && _src->get_duration(d)) return d;
        return 0;
      }
      virtual void goto_start() override { set_position(0); }
      virtual bool set_position(double seconds) override {
        if (_src) { _src->set_position(seconds); return true; }
        return false;
      }
      virtual double get_position() const override {
        double d;
        if (_src && _src->get_position(d)) return d;
        return 0;
      }

      virtual size get_movie_normal_size() const override {
        if (frame) return frame->dim();
        return size();
      }

      virtual bool set_volume(double new_volume) override {
        if (_src) { _src->set_volume(new_volume); return true; }
        return false;
      }
      virtual double get_volume() const override {
        double d;
        if (_src && _src->get_volume(d)) return d;
        return 0;
      }
      virtual bool set_balance(double new_balance) override {
        if (_src) { _src->set_balance(new_balance); return true; }
        return false;
      }
      virtual double get_balance() const override {
        double d;
        if (_src && _src->get_balance(d)) return d;
        return 0;
      }

      virtual void on_start_streaming() {
        if (!_is_playing) {
        _is_playing = TRUE;
        this->notify_video_started(false);
      }
      }
      virtual void on_stop_streaming(bool ended) {
        if (_is_playing) {
          this->notify_video_stopped(ended);
          _is_playing = FALSE;
        }
      }

      virtual bool on(view &v, element *self, event_behavior &evt) override;

      // animator
      virtual uint step(view &, element *, uint /*current_clock*/) override;
    };


    class custom_video_ctl : public zero_video_ctl 
    {
      typedef zero_video_ctl super;
    public:
      custom_video_ctl() {}
      virtual ~custom_video_ctl() {}

      virtual bool on(view &v, element *self, event_behavior &evt) override
      {
        if (evt.cmd == VIDEO_SOURCE_CREATED) {
          event_behavior evt(self, self, VIDEO_BIND_RQ, 0);
          if (self->on(v, evt)) {
            evt.cmd = VIDEO_BIND_RQ;
            evt.reason = (uint_ptr) static_cast<sciter::video_destination *>(this);
            if (self->on(v, evt)/* && vsite->get_ref_count() > 1*/) // host accepted it and add_ref'ed it.
              return true;
          }
          return true;
        }
        return super::on(v, self, evt);
      }
       
      virtual const string &behavior_name() const { 
        static string _name(CUSTOM_VIDEO_CTL_NAME);
        return _name;
      }

      SOM_PASSPORT_BEGIN_EX(video, fragmented_video_destination)
        SOM_FUNCS(SOM_FUNC(renderingSite))
      SOM_PASSPORT_END
    };
    
    bool zero_video_ctl::on(view &v, element *self, event_behavior &evt) {
      bool r = super::on(v, self, evt);
      switch (evt.cmd) {
        case VIDEO_STARTED | EVENT_SINKING:
        case VIDEO_STARTED | EVENT_SINKING | EVENT_HANDLED: {
          critical_section _(_guard);
          start_animation(&v, self);
          break;
        }
        case VIDEO_STOPPED | EVENT_SINKING:
        case VIDEO_STOPPED | EVENT_SINKING | EVENT_HANDLED: {
          critical_section _(_guard);
          stop_animation(&v, self);
          break;
        }
      }
      return r;
      
    }

    uint zero_video_ctl::step(view &v, element *self, uint current_clock) {

      critical_section _(_guard);

      {
        event_behavior evt(self, self, VIDEO_FRAME_REQUEST, 0);
        self->on(v, evt);
      }

      //for (ctl *ct = self->behavior; ct; ct = ct->next)
      //  ct->on_animation_frame(v, self);
            
      if (frame && (frame->_generation != _generation)) {
        _generation = frame->_generation;

        if (_invalid.empty())
          v.refresh(self);
        else {
          rect to_refresh = _invalid;
          // bool need_clipping;
          rect box = // get_target_box(need_clipping);
              self->foreground_image_box(v);

          if (box.empty())
            v.refresh(self);
          else {
#if 1
            // partial screen update
            float scale         = box.width() / float(frame->dim().x);
            to_refresh.s.x = int(to_refresh.s.x * scale);
            to_refresh.s.y = int(to_refresh.s.y * scale);
            to_refresh.e.x = int(to_refresh.e.x * scale + 0.5f);
            to_refresh.e.y = int(to_refresh.e.y * scale + 0.5f);
            v.refresh(self, to_refresh + box.s);
#else
            // full screen update
            v.refresh(self);
#endif
          }
        }
      }

      if (is_movie_ended()) return 0;

      return ANIMATION_TIMER_SPAN;
    }

    video_ctl *produce_zero_video_ctl(element *el) {
      view *pv = el->pview();
      if (!pv) return nullptr;

      handle<zero_video_ctl> vsite = new zero_video_ctl();

      event_behavior evt(el, el, VIDEO_BIND_RQ, 0);
      if (el->on(*pv, evt)) {
        evt.cmd = VIDEO_BIND_RQ;
        evt.reason = (uint_ptr) static_cast<sciter::video_destination *>(vsite.ptr_of<sciter::video_destination>());
        if (el->on(*pv, evt) && vsite->get_ref_count() > 1) // host accepted it and add_ref'ed it.
          return vsite;
      }
      return nullptr;
    }

    video_ctl *produce_custom_video_ctl(element *el) {
      view *pv = el->pview();
      if (!pv) return nullptr;
      return new custom_video_ctl();
    }

#endif
  }
} // namespace html
