
#include "html.h"
#include <math.h>

namespace html {
  namespace behavior {

    struct slider_ctl_factory : ctl_factory {
      slider_ctl_factory() : ctl_factory("slider") {}
      virtual ctl *create(element *el);
    };

    static slider_ctl_factory *_slider_ctl_factory = 0;

    value get_numeric_attr(element *self, name_or_symbol ns) {
      tool::ustring s = self->atts.get_ustring(ns);
      if (s.length() == 0) return value();
      value v = value::parse(s);
      if (v.is_number()) return v;
      return value();
    }

    struct slider_ctl : ctl {
      typedef ctl super;

      // int     offset;
      bool          tracking;
      bool          inverse;
      tag::symbol_t slider_tag;
      ustring buddy;

      float_v vmin;
      float_v vmax;
      float_v vstep;
      float_v vval;

      bool   is_float = false;

      slider_ctl(element* forel)
          : super(uint(-1)), tracking(false), inverse(false) 
      {
        if (forel->tag == tag::T_INPUT || forel->tag == tag::T_WIDGET)
          slider_tag = tag::T_BUTTON;
        else 
          slider_tag = tag::symbol(CHARS("knob"));

        value tmax = get_numeric_attr(forel, attr::a_max);
        value tmin = get_numeric_attr(forel, attr::a_min);
        value tval = get_numeric_attr(forel, attr::a_value);
        value tstep = get_numeric_attr(forel, attr::a_step);

        is_float = tmax.is_double() || tmin.is_double() || tval.is_double() || tstep.is_double();

        vmin = tmin;
        vmax = tmax;
        vstep = tstep;
        vval = tval;
      }

      virtual CTL_TYPE get_type() { return CTL_SLIDER; }

      virtual const string &behavior_name() const {
        return _slider_ctl_factory->name;
      }

      virtual bool focusable(const element *self) {
        return !self->attr_disabled();
      }

      void normalize(float_v &value) {
        if (is_float)
          normalize_float(value);
        else 
          normalize_int(value);
      }

      void normalize_int(float_v &value) {
        if (vmin > vmax) {
          inverse = true;
          swap(vmin, vmax);
        }
        else if (vmin == vmax)
          vmax = vmin + 100;

        int step = (int) limit(vstep.val(), 1.0f, vmax.val() - vmin.val());
        int val = (int) value.val(0);

        if (step > 1 && (val % step)) {
          if ((val % step) > (step / 2))
            val = ((val / step) + 1) * step;
          else
            val = (val / step) * step;
        }
        value = limit(float(val), vmin.val(), vmax.val());
      }

      void normalize_float(float_v &value) 
      {
        if (vmin > vmax) {
          inverse = true;
          swap(vmin, vmax);
        }
        else if (vmin == vmax)
          vmax = vmin + 100.0f;
        if (vstep.is_defined() && vstep.val() > 0)
          value = floorf(value.val() / vstep.val() + 0.5f) * vstep.val();
        value = limit(value.val(), vmin.val(), vmax.val());
      }

      float_v pos2val(int max_pos, int pos, bool inv)
      {
        return is_float 
          ? pos2val_float(max_pos, pos, inv) 
          : pos2val_int(max_pos, pos, inv);
      }

      float_v pos2val_float(int max_pos, int pos, bool inv) {
        if (inv)
          return vmin + ((vmax - vmin) * (max_pos - pos)) / max(max_pos, 1);
        else
          return vmin + ((vmax - vmin) * pos) / max(max_pos, 1);
      }

      float_v pos2val_int(int max_pos, int pos, bool inv) {
        float r;
        if (inv)
          r = vmin + (float(vmax - vmin) * (max_pos - pos)) / max(max_pos, 1);
        else
          r = vmin + (float(vmax - vmin) * pos) / max(max_pos, 1);
        return float(int(r + 0.5f));
      }

      virtual void detach(view &v, element *self) {}
      virtual bool attach(view &v, element *self) {
        if (!get_slider(self)) {
          element *pd = new element(slider_tag);
          pd->atts.set(attr::a_class, W("slider"));
          self->insert(0, pd);
          // pd->flags.out_of_flow = true;
          pd->state.current(true);
          pd->get_style(v);
        }
        buddy = self->atts("buddy");
        normalize(vval);
        set_value(v, self, vval, true);
        return true;
      }

      bool is_vertical(element *self) {
        return self->atts("type") == WCHARS("vslider") || self->has_class(WCHARS("vertical"));
      }

      element *get_slider(element *self) {
        element *p = self->first_element();
        while (p) {
          if (p->tag == slider_tag) break;
          p = p->next_element();
        }
        // assert(p);
        return p;
      }

OBSOLETE virtual bool on_x_method_call(view &v, element *self, const char *name,
                                    const value *argv, size_t argc,
                                    value &retval) override {
        chars fname = chars_of(name);
#define ACTION(ARGC, NAME) if (argc == ARGC && fname == CHARS(#NAME))
        ACTION(0, min) {
          retval = to_value(vmin);
          return true;
        }
        ACTION(0, sliderMin) {
          retval = to_value(vmin);
          return true;
        }
        ACTION(0, max) {
          retval = to_value(vmax);
          return true;
        }
        ACTION(0, sliderMax) {
          retval = to_value(vmax);
          return true;
        }
        ACTION(0, step) {
          retval = to_value(vstep);
          return true;
        }
        ACTION(0, sliderStep) {
          retval = to_value(vstep);
          return true;
        }
        ACTION(3, sliderRange) {
          if (!argv[0].is_number() || !argv[1].is_number() ||
              !argv[2].is_number())
            return false;
          vmin  = argv[0].get<float>();
          vmax  = argv[1].get<float>();
          vstep = argv[2].get<float>();
          normalize(vval);
          return true;
        }
        ACTION(2, sliderRange) {
          if (!argv[0].is_number() || !argv[1].is_number()) return false;
          vmin = argv[0].get<float>();
          vmax = argv[1].get<float>();
          vstep.clear();
          normalize(vval);
          return true;
        }

#undef ACTION
        return super::on_x_method_call(v, self, name, argv, argc, retval);
      }

      value to_value(float t) {
        if (is_float)
          return value(t);
        else
          return value((int)t);
      }

      //float from_value(value v) {
      //  return v.get<float>();
      //}

      value get_min() { return to_value(vmin); }
      value get_max() { return to_value(vmax); }
      value get_step() { return to_value(vstep); }

      bool set_min(value v) { vmin = 0; if (v.is_defined()) vmin = v.get<float>(); normalize(vval); return true; }
      bool set_max(value v) { vmax = 100; if (v.is_defined()) vmax = v.get<float>(); normalize(vval); return true; }
      bool set_step(value v) { vstep = 10; if (v.is_defined()) vstep = v.get<float>(); normalize(vval); return true; }

      SOM_PASSPORT_BEGIN_EX(slider, slider_ctl)
        SOM_PROPS(
          SOM_VIRTUAL_PROP(min, get_min, set_min),
          SOM_VIRTUAL_PROP(max, get_max, set_max),
          SOM_VIRTUAL_PROP(step, get_step, set_step),
        )
      SOM_PASSPORT_END

      virtual bool draw_background(view &v, element *self, graphics *sf,
                                   point pos) {
        on_size_changed(v, self);
        return ctl::draw_background(v, self, sf, pos);
      }

      virtual bool draw_content(view &v, element *self, graphics *sf,
                                point pos) {
#ifdef _DEBUG
      // element* ps = get_slider( self );
      // ps = ps;
#endif
        return ctl::draw_content(v, self, sf, pos);
      }

      int slider_pos(int max_pos, bool inv);


      virtual void on_size_changed(view &v, element *self) {
        size csize = self->content_box(v).size();
        if (csize.empty()) 
          return;

        element *slider = get_slider(self);
        if (!slider) return;

        /*int sx = slider->min_width(v);
        slider->set_width(v, sx);
        int sy = slider->min_height(v);

        h_layout_data slider_pld = slider->ldata;

        if (slider_pld->dim.y < sy) slider_pld->dim.y = sy;
        if (slider_pld->dim.x < sx) slider_pld->dim.x = sx;*/

        bool vertical = is_vertical(self);

        bool inv = inverse;

        if (vertical) {
          int max_pos = csize.y;
          int y = int(((vval - vmin) * max_pos) / (vmax - vmin));
          int y_pos;
          if (inv)
            y_pos = y;
          else
            y_pos = max_pos - y;
          //slider->set_y_pos(y_pos);
          //slider->set_x_pos(-(sx - self->ldata->dim.x) / 2);
          self->set_style_variable(v, string("slider-position"), value::make_ppx_length(y_pos));
        } else // horizontal
        {
          if (self->state.rtl()) inv = !inv;
          int max_pos = csize.x;
          int x = int(((vval - vmin) * max_pos) / (vmax - vmin));
          int x_pos;
          if (!inv)
            x_pos = x;
          else
            x_pos = max_pos - x;
          //slider->set_x_pos(x_pos);
          //slider->set_y_pos(-(sy - self->ldata->dim.y) / 2);
          self->set_style_variable(v, string("slider-position"), value::make_ppx_length(x_pos));
        }
      }

      void set_value_by_pos(view &v, element *self, int pos) {
        element *slider = get_slider(self);
        if (!slider) return;

        bool inv = inverse;

        int max_pos = 0;

        rect content_box = self->ldata->content_box();

        bool vertical = is_vertical(self);
        if (vertical) {
          inv     = !inv;
          max_pos = content_box.height();
        } else {
          max_pos = content_box.width();
          if (self->state.rtl()) inv = !inv;
        }

        float val;
        
        val = pos2val(max_pos, pos, inv);

        if (set_value(v, self, val)) 
          notify_changed(v, self, true);
      }

      void inc_value(view &v, element *self, int pos, bool by_mouse) {
        element *slider = get_slider(self);
        if (!slider) return;
        bool inv = inverse;
        if (!is_vertical(self) && self->state.rtl()) inv = !inv;

        if (inv) pos = -pos;

        float_v val = vval;

        switch (pos) {
        case -3: val = vmin; break;
        case -2:
          val = val - max(vstep, (vmax - vmin) / 6);
          break;
        case -1: val = val - vstep.val(1); break;
        case +3: val = vmax; break;
        case +2:
          val = val + max(vstep, (vmax - vmin) / 6);
          break;
        case +1: val = val + vstep.val(1); break;
        }
        normalize(val);
        if (set_value(v, self, val)) 
          notify_changed(v, self, by_mouse);
      }

      bool set_value(view &v, element *self, float n_value, bool force = false) {
        float_v oval = vval;
        float_v nval = n_value;
        
        normalize(nval);

        vval = nval;

        if (oval == vval && !force) return false;

        if (element *slider = get_slider(self))
          slider->set_attr(v, attr::a_value, to_value(vval).to_string());

        if (buddy.length() && self->doc()) {
          element *bu = self->doc()->find_by_id(buddy);
          if (bu) bu->set_text(v, to_value(vval).to_string());
        }
        
        on_size_changed(v, self);
        v.refresh(self);

        return true;
      }

      virtual bool set_value(view &v, element *self, const value &val) override {
        float vv = val.get<float>();
        set_value(v, self, vv);
        return true;
      }
      virtual bool get_value(view &v, element *self, value &val) override {
        if (vval.is_defined())
          val = to_value(vval);
        else
          val = value();
        return true;
      }

      virtual bool set_text(view &v, element *self, wchars text) {
        ustring us(text);
        value   val = value::parse(us);
        set_value(v, self, val);
        return true;
      }

      virtual bool max_intrinsic_width(view &v, element *self, int &value) {
        return min_intrinsic_width(v, self, value);
      }
      virtual bool min_intrinsic_width(view &v, element *self, int &value) {
        if (is_vertical(self)) {
          element *slider = get_slider(self);
          if (slider) {
            value = slider->min_width(v);
            return true;
          }
        }
        return false;
      }
      virtual bool max_intrinsic_height(view &v, element *self, int &value) {
        return min_intrinsic_height(v, self, value);
      }
      virtual bool min_intrinsic_height(view &v, element *self, int &value) {
        if (!is_vertical(self)) {
          element *slider = get_slider(self);
          if (slider) {
            value = slider->min_height(v);
            return true;
          }
        }
        return false;
      }

      virtual bool notify_changed(view &v, element *self,
                                  bool by_mouse = false) {
        event_behavior evt(self, self, BUTTON_STATE_CHANGED,
                           by_mouse ? CLICK_BY_MOUSE : CLICK_BY_KEY);
        v.post_behavior_event(evt);
        return true;
      }

      virtual bool notify_done(view &v, element *self, bool by_mouse = false) {
        event_behavior evt(self, self, BUTTON_CLICK,
                           by_mouse ? CLICK_BY_MOUSE : CLICK_BY_KEY);
        v.post_behavior_event(evt);
        return true;
      }

      virtual bool on(view &v, element *self, event_gesture &evt) override {
        switch (evt.cmd) {
        case GESTURE_REQUEST:
          evt.flags = is_vertical(self) ? GESTURE_FLAG_PAN_VERTICAL
                                        : GESTURE_FLAG_PAN_HORIZONTAL;
          return true;
        case GESTURE_START:
          v.set_focus(self, BY_MOUSE);
          v.set_capture(self);
          return true;
        case GESTURE_PAN:
          set_value_by_pos(v, self, is_vertical(self) ? evt.pos.y : evt.pos.x);
          // dbg_printf("PAN %d,%d\n",evt.delta_xy.x,evt.delta_xy.y);
          if (evt.flags & GESTURE_STATE_END)
            v.set_capture(nullptr);
          return true;
        }
        return false;
      }

      /*virtual bool on(view &v, element *self, event_behavior &evt) override {
        switch (evt.cmd) {
        case CONTENT_CHANGED:
          if ((evt.target == self) && ((evt.reason & ATTRIBUTES_CHANGED) != 0)) {
            //value vmax = get_numeric_attr(el, attr::a_max);
            //value vmin = get_numeric_attr(el, attr::a_min);
            //value vval = get_numeric_attr(el, attr::a_value);
            //value vstep = get_numeric_attr(el, attr::a_step);
          }
          break;
        }
        return false;
      }*/

      virtual void on_attr_change(view &v, element *self, const name_or_symbol &nm) {
        switch (nm.att_sym) {
          case attr::a_max: vmax = get_numeric_attr(self, attr::a_max); goto REFLECT;
          case attr::a_min: vmin = get_numeric_attr(self, attr::a_min); goto REFLECT;
          case attr::a_step: vstep = get_numeric_attr(self, attr::a_step); goto REFLECT;
          case attr::a_value: vval = get_numeric_attr(self, attr::a_value); goto REFLECT;
          default: return;
        }

        REFLECT:
          normalize(vval);
          on_size_changed(v, self);
      }


      virtual bool on(view &v, element *self, event_mouse &evt) override {
        if (is_vertical(self)) 
          return on_vertical(v, self, evt);
        element *slider = get_slider(self);
        if (!slider) return false;
        switch (evt.cmd) {
        case MOUSE_ENTER: v.refresh(self); return false;
        case MOUSE_LEAVE: v.refresh(self); return false;
        case MOUSE_DOWN:
        case MOUSE_DCLICK:
          if (!evt.is_point_button()) return false;
          v.set_focus(self, BY_MOUSE);
          if (evt.target != slider) set_value_by_pos(v, self, evt.pos.x);
          tracking = true;
          v.set_capture(self);
          /*          else if( evt.pos.x < slider->ldata->pos.x )
                      inc_value( v, self, -2, true );
                    else if( evt.pos.x > slider->ldata->pos.x )
                      inc_value( v, self, +2, true );*/

          return true;
        case MOUSE_UP:
        case MOUSE_UP | EVENT_HANDLED:
          //if (!evt.is_point_button()) return false;
          slider->on(v, evt);
          if (tracking) {
            tracking = false;
            v.refresh(self);
            v.set_capture(0);
            if (self->state.pressed()) {
              notify_done(v, self, true);
              return true;
            }
          }
          return false;
        case MOUSE_MOVE:
          if (tracking) {
            set_value_by_pos(v, self, evt.pos.x);
            v.update();
            return true;
          }
          break;
        case MOUSE_WHEEL:
          if (self->is_safe_to_wheel(v)) {
            inc_value(v, self, limit((int)evt.get_wheel_delta(), -1, 1), true);
            return true;
          }
          break;
        }
        return false;
      }

      virtual bool on_vertical(view &v, element *self, event_mouse &evt) {
        element *slider = get_slider(self);
        if (!slider) return false;

        switch (evt.cmd) {
        case MOUSE_ENTER: v.refresh(self); return false;
        case MOUSE_LEAVE: v.refresh(self); return false;
        case MOUSE_DOWN:
        case MOUSE_DCLICK:
          if (!evt.is_point_button()) return false;
          v.set_focus(self, BY_MOUSE);
          if (evt.target != slider) set_value_by_pos(v, self, evt.pos.y);
          tracking = true;

          v.set_capture(self);
          /*else if( evt.pos.y < slider->ldata->pos.y )
            inc_value( v, self, +2, true );
          else if( evt.pos.y > slider->ldata->pos.y )
            inc_value( v, self, -2, true );*/
          return true;
        case MOUSE_UP | EVENT_SINKING:
        case MOUSE_UP | EVENT_SINKING | EVENT_HANDLED:
          slider->on(v, evt);
          if (tracking) {
            tracking = false;
            v.refresh(self);
            v.set_capture(0);
            if (self->state.pressed()) {
              notify_done(v, self, true);
              return true;
            }
          }
          return false;
        case MOUSE_MOVE:
          if (tracking) {
            set_value_by_pos(v, self, evt.pos.y);
            return true;
          }
          break;
        case MOUSE_WHEEL:
          if (self->is_safe_to_wheel(v)) {
            inc_value(v, self, (int)evt.get_wheel_delta(), true);
            return true;
          }
          break;
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_key &evt) {
        if (evt.has_modifiers()) return false;
        switch (evt.cmd) {
        case KEY_DOWN:
          switch (evt.key_code) {
          case KB_LEFT:
            if (!is_vertical(self)) {
              inc_value(v, self, -1, false);
              notify_done(v, self, false);
              return true;
            }
            break;
          case KB_RIGHT:
            if (!is_vertical(self)) {
              inc_value(v, self, +1, false);
              notify_done(v, self, false);
              return true;
            }
            break;
          case KB_UP:
            if (is_vertical(self)) {
              inc_value(v, self, +1, false);
              notify_done(v, self, false);
              return true;
            }
            break;
          case KB_DOWN:
            if (is_vertical(self)) {
              inc_value(v, self, -1, false);
              notify_done(v, self, false);
              return true;
            }
            break;
          case KB_PRIOR:
            if (!is_vertical(self))
              inc_value(v, self, -2, false);
            else
              inc_value(v, self, +2, false);
            notify_done(v, self, false);
            return true;
          case KB_NEXT:
            if (!is_vertical(self))
              inc_value(v, self, +2, false);
            else
              inc_value(v, self, -2, false);
            notify_done(v, self, false);
            return true;
          case KB_HOME:
            if (!is_vertical(self))
              inc_value(v, self, -3, false);
            else
              inc_value(v, self, +3, false);
            notify_done(v, self, false);
            return true;
          case KB_END:
            if (!is_vertical(self))
              inc_value(v, self, +3, false);
            else
              inc_value(v, self, -3, false);
            notify_done(v, self, false);
            return true;
          }
          break;

        case KEY_UP:
          switch (evt.key_code) {
          case KB_LEFT:
          case KB_RIGHT:
          case KB_UP:
          case KB_DOWN:
          case KB_PRIOR:
          case KB_NEXT:
          case KB_HOME:
          case KB_END: notify_done(v, self, false); return true;
          }
          break;
        }
        return false;
      }

      virtual bool on_focus(view &v, element *self, event_focus &evt) {
        v.refresh(self);
        return true;
      }
    };

    int_v from_value(const value &v, int_v dv) {
      if (v.is_number()) return v.get<int>();
      return dv;
    }
    float_v from_value(const value &v, float_v dv) {
      if (v.is_number()) return (float)v.get<float>();
      return dv;
    }

    ctl *slider_ctl_factory::create(element *el) 
    {
      return new slider_ctl(el);
    }

    void init_slider_ctl() {
      ctl_factory::add(_slider_ctl_factory = new slider_ctl_factory());
    }
    extern void init_progress_ctl();

    void init_bars() {
      init_slider_ctl();
      init_progress_ctl();
    }

  } // namespace behavior
} // namespace html
