#include "html.h"
#include "html-actions-stack.h"
#include "html-clipboard.h"

namespace html {

  bool same_caret_position(const bookmark &bm1, const bookmark &bm2);

  namespace behavior {

    bool is_list_tag(tag::symbol_t t);

    struct richtext_ctl_factory : public ctl_factory {
      richtext_ctl_factory() : ctl_factory("richtext") {}
      virtual ctl *create(element *el);
    };

    static richtext_ctl_factory *_richtext_ctl = 0;
    ctl *                        richtext_ctl_factory::create(element *el) {
      return new richtext_ctl();
    }

    void init_richtext() {
      ctl_factory::add(_richtext_ctl = new richtext_ctl_factory());
    }

    const string &richtext_ctl::behavior_name() const {
      return _richtext_ctl->name;
    }

    void dbg_print(const bookmark &bm) {
      /*#ifdef _DEBUG
          if( bm.node->is_text() )
            dbg_printf("text:%S pos:%d after:%s\n",
               ustring(bm.node.ptr_of<text>()->chars()).c_str(),
               int(bm.pos),
               bm.after_it ? "true" : "false" );
          else if( bm.node->is_element() )
            dbg_printf("elem %s pos:%d after:%s\n",
      tag::symbol_name(bm.node.ptr_of<element>()->tag).c_str(), int(bm.pos),
               bm.after_it ? "true" : "false" );
      #endif*/
    }

    bool richtext_ctl::clear_comp_chars(view &v) {
      if (is_readonly(self)) return false;

      if (ime_start.valid() && ime_end.valid()) {
        remove_chars_at(v, ime_start, ime_end);
        caret = anchor = ime_start;
        ime_start = ime_end = bookmark();
        v.refresh(self);
        v.commit_update();
      }
      return true;
    }

    bookmark richtext_ctl::remove_chars_at(view &v, bookmark start,
                                           bookmark end) {
      start.linearize();
      end.linearize();

      // dbg_printf("richtext_ctl::remove_chars_at \n");

      assert(start.node->is_text());
      if (!start.node->is_text()) return start;

      if (start > end) swap(start, end);
      if (start.node != end.node) return start;

      int s = start.pos;
      int e = end.pos;

      array<wchar> &buffer = start.node.ptr_of<text>()->chars;
      s                    = limit(s, 0, buffer.size());
      e                    = limit(e, 0, buffer.size());
      wchars chars         = buffer()(s, e);

      buffer.remove(s, e - s);

      element *tb = start.node->nearest_text_box();
      if (tb) tb->drop_layout(&v);

      v.commit_update();
      // update(v,self);

      start.normalize();
      return start;
    }

    bookmark richtext_ctl::insert_chars_at(view &v, bookmark pos,
                                           wchars chars) {
      pos.linearize();

      assert(pos.node->is_text());
      if (!pos.node->is_text()) return pos;

      array<wchar> &buffer = pos.node.ptr_of<text>()->chars;
      int s = limit<int>(pos.pos, 0, buffer.size());

      // dbg_printf("ichtext_ctl::insert_chars_at before
      // (%S)\n",buffer.begin());

      if (this->is_plaintext())
        for (uint n = 0; n < chars.length; ++n) {
          wchar wc = chars[n];
          buffer.insert(s++, wc);
        }
      else
        for (uint n = 0; n < chars.length; ++n) {
          wchar wc = chars[n];
          buffer.insert(s++, wc == ' ' ? 0xA0 : wc);
        }

      // dbg_printf("ichtext_ctl::insert_chars_at \n");

      helement tb = pos.node->nearest_text_box();
      tb->drop_content_layout(&v);
      v.add_to_update(tb, CHANGES_DIMENSION);

      v.commit_update();
      // update(v,self);

      // if( buffer.size() == pos.pos + chars.length ) {
      pos.pos      = pos.pos + chars.size() - 1;
      pos.after_it = true;
      //} else
      //  pos.pos = pos.pos + chars.length;

      // pos.normalize();
      // dbg_printf("ichtext_ctl::insert_chars_at pos %d\n", int(pos.pos));

      return pos;
    }

    bool richtext_ctl::on_char(view &v, element *self, event_key &evt) {
      if (!this->self) return false;
      if (!is_editable(self)) return false;

      clear_comp_chars(v);

      //if (is_space(evt.key_code))
      //  rq_spell_check(v);
      //else if (caret.valid() && caret.marks_contains(CHARS("misspell")))
      //  rq_spell_check(v);

      if (evt.key_code < ' ' || evt.key_code == 127) // see: https://sciter.com/forums/topic/bug-altgr-inputs-not-working-on-plaintext/
        return false;
      wchar  buf[3];
      wchars chars(buf, 0);
      chars.length = u16::putc(evt.key_code, buf);
      return insert_text(v, chars);
    }

    bool richtext_ctl::on_key_down(view &v, element *self, event_key &evt) {
      if (!this->self) return false;
      wchars cmd;
      if (self->state.focus()) switch (evt.key_code) {
        case KB_RETURN: {
          if (!is_editable(self)) return false;
          static ustring command;
          if (evt.is_shift())
            command = WCHARS("edit:insert-soft-break");
          else if (evt.is_shortcut())
            command = WCHARS("edit:insert-block-break");
          else
            command = WCHARS("edit:insert-break");
          return html::exec_command(v, self, self, command);
        } break;

        case KB_TAB: {
          if (!is_editable(self)) return false;
          clear_comp_chars(v);
          wchar t = '\t';
          return insert_text(v, wchars(t));
        }

        case KB_DELETE:
          if (evt.is_alt()) {
            static ustring command = WCHARS("format:remove-span:*");
            return html::exec_command(v, self, self, command);
          }
          break;

        case KB_B:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:toggle-span:b|strong");
            return html::exec_command(v, self, self, command);
          }
          break;
        case KB_I:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:toggle-span:i|em");
            return html::exec_command(v, self, self, command);
          }
          break;
        case KB_U:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:toggle-span:u");
            return html::exec_command(v, self, self, command);
          }
          break;
        case KB_E:
          if(evt.is_shortcut()) {
            static ustring command = WCHARS("format:toggle-span:del|s|strike");
            return html::exec_command(v,self, self, command );
          }
          break;
        case KB_D:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:toggle-span:code");
            return html::exec_command(v, self, self, command);
          }
          break;
#if 1
        case KB_ADD:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:indent");
            return html::exec_command(v, self, self, command);
          }
          break;
        case KB_SUBTRACT:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:unindent");
            return html::exec_command(v, self, self, command);
          }
          break;
        case KB_MULTIPLY:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:toggle-list:ul");
            return html::exec_command(v, self, self, command);
          }
          break;
        case KB_DECIMAL:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:toggle-list:ol");
            return html::exec_command(v, self, self, command);
          }
          break;
        case KB_DIVIDE:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:toggle-list:dl");
            return html::exec_command(v, self, self, command);
          }
          break;

        case KB_NUMPAD0:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:morph-block:p");
            return html::exec_command(v, self, self, command);
          }
          break;
        case KB_NUMPAD1:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:morph-block:h1");
            return html::exec_command(v, self, self, command);
          }
          break;
        case KB_NUMPAD2:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:morph-block:h2");
            return html::exec_command(v, self, self, command);
          }
          break;
        case KB_NUMPAD3:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:morph-block:h3");
            return html::exec_command(v, self, self, command);
          }
          break;
        case KB_NUMPAD4:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:morph-block:h4");
            return html::exec_command(v, self, self, command);
          }
          break;
        case KB_NUMPAD5:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:morph-block:h5");
            return html::exec_command(v, self, self, command);
          }
          break;
        case KB_NUMPAD6:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:morph-block:h6");
            return html::exec_command(v, self, self, command);
          }
          break;
        case KB_NUMPAD7:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:morph-block:div");
            return html::exec_command(v, self, self, command);
          }
          break;
        case KB_NUMPAD9:
          if (evt.is_shortcut()) {
            static ustring command = WCHARS("format:toggle-pre");
            return html::exec_command(v, self, self, command);
          }
          break;
#endif

        }
      return super::on_key_down(v, self, evt);
    }

    bool richtext_ctl::on_ime_comp_string(view &v, element *self,
                                          event_key &evt) {
      if (!this->self) return false;
      if (!is_editable(self) || !need_ime(self)) return true;
      if (evt.ime.composition.length) remove_selection(v);
      clear_comp_chars(v);
#ifdef _DEBUG
      // caret.dbg_report("on_ime_comp_string");
#endif
      if (evt.ime.composition.length) {
        insert_ime_range(v, evt.ime.composition, evt.ime.caret_pos);
      }
      v.refresh(self);
      return true;
    }
    bool richtext_ctl::on_ime_result_string(view &v, element *self,
                                            event_key &evt) {
      if (!this->self) return false;
      if (!is_editable(self) || !need_ime(self)) return true;
      remove_selection(v);
      clear_comp_chars(v);
      if (evt.ime.composition.length) insert_text(v, evt.ime.composition);
      // notify_changed(v,BY_INS_CHAR);
      v.refresh(self);
      return true;
    }

    bool richtext_ctl::insert_ime_range(view &v, wchars s, int_v caret_pos) {
      // edit_auto_state _(this,b,v);
      if (s.length == 0 || !caret.valid()) return false;

      clear_comp_chars(v);

      // delete_char(v,0);

      bookmark pos;
      ime_start = ime_end = caret;

      hnode cn = caret.node;
      // bool is_empty = false;
      // this->is_empty(self,is_empty);

      ime_end = insert_chars_at(v, ime_end, s);

      if (caret_pos.is_defined()) {
        int n = caret_pos;
        // if(is_empty)
        //  ime_start = pos = bookmark(cn,0,false);
        // else
        pos = ime_start;
        while (n--)
          self->advance(v, pos, ADVANCE_NEXT);
      } else
        pos = ime_end;

      this->select(v, pos);

      return true;
    }

    bool richtext_ctl::cut(view &v) {
      if (!copy(v)) return false;
      assert(caret != anchor);
      bookmark bm = delete_range(v, anchor, caret,true);
      if (bm.valid()) {
        select(v, bm);
        return true;
      }
      return false;
    }

    bool richtext_ctl::can_paste(view &v) {
      if (!caret.valid() || !is_editable(self))
        return false;
      bool allow_images = true;
      uint fmt          = clipboard::cf_text | clipboard::cf_html;
      if (allow_images) fmt |= clipboard::cf_picture;
      return 0 != (clipboard::available_formats() & fmt);
    }

    bool richtext_ctl::paste(view &v) {
      handle<clipboard::data> data = clipboard::get();
      if (data) return paste(v, data);
      return false;
    }

    bool richtext_ctl::paste_text(view &v) {
      handle<clipboard::data> data = clipboard::get();
      if (auto ti = data->get<clipboard::text_item>()) {
        handle<clipboard::data> new_data = new clipboard::data();
        new_data->items.push(ti);
        return paste(v, new_data);
      }
      return false;
    }


    bool richtext_ctl::paste(view &v, clipboard::data *what, bookmark where) {

      clipboard::html_item * hi = 0;
      clipboard::text_item * ti = 0;
      clipboard::image_item *ii = 0;

      if ((hi = what->get<html::clipboard::html_item>()) != nullptr) {
        //rq_spell_check(v);
        return insert_html(v, hi->val.chars_as_bytes(), where, hi);
      } else if ((ii = what->get<html::clipboard::image_item>()) != nullptr) {
        return insert_image(v, ii->image, where);
      }
      if ((ti = what->get<html::clipboard::text_item>()) != nullptr) {
#if 1
        //rq_spell_check(v);
        return insert_text(v, ti->val(), where);
#else
        byte        br[5] = {'<', 'b', 'r', '/', '>'};
        array<byte> text  = ti->get_data();

        for (int n = text.last_index(); n >= 0; --n) {
          switch (text[n]) {
          case '<':
            text.remove(n);
            text.insert(n, to_bytes(CHARS("&lt;")));
            break;
          case '&':
            text.remove(n);
            text.insert(n, to_bytes(CHARS("&amp;")));
            break;
          default: break;
          }
        }

        for (int i = 0; i < text.size() - 1; ++i) {
          if (text[i] == '\r' && text[i + 1] == '\n') {
            text.remove(i, 2);
            text.insert(i, BYTES(br));
          } else if (text[i] == '\r' || text[i] == '\n') {
            text.remove(i, 1);
            text.insert(i, BYTES(br));
          }
        }
        rq_spell_check(v);
        return insert_html(v, string(), text(), where);
#endif
      }
      return false;
    }

    bool richtext_ctl::on(view &v, element *self, event_focus &evt) {
      if(!this->self) return false;
      switch (evt.cmd) {
      case GOT_FOCUS:
      case FOCUS_IN: evt.need_ime = need_ime(self); break;
      case REQUEST_FOCUS:
        if (evt.target->belongs_to(self) && evt.target->is_input_ctl_type()) {
          evt.target = self;
          return true;
        }
        break;
      }
      return super::on(v, self, evt);
    }

    bool richtext_ctl::advance_bookmark(view &v, bookmark &bm, bool forward,
                                        function<bool(bookmark &bm)> on_pos) {
      helement holder = self;
      if (!self->is_layout_valid()) v.commit_update();

      auto is_valid = [&](const bookmark &bm) -> bool {
        if (!bm.valid()) return false;
        if (!bm.node->belongs_to(root(), true)) return false;
        return true;
      };

      for (bookmark t = bm; is_valid(t);) {
        bookmark pt = t;
        t.advance_caret_pos(v, forward ? ADVANCE_RIGHT : ADVANCE_LEFT);
        if (!is_valid(t) || t == bm) break;
        if (t.at_caret_pos(v) && !same_caret_position(bm, t)) {
          if (on_pos(t)) {
            bm = t;
            return true;
          }
          // bm = t;
          // return true;
        } else if (pt == t)
          return false;
      }
      return false;
    }

    // bool is_inside_inline_block(view& v, )

    bool richtext_ctl::advance(view &v, bookmark &bm, ADVANCE_DIR cmd,
                               point vpt) {
      handle<element> root = this->root();
      while (root->advance(v, bm, cmd, vpt)) {
        if (!bm.valid()) return false;
        if (!bm.node->belongs_to(root, true)) return false;
        // element* pel = bm.node->get_element();
        // if(pel->is_inline_block_element(v))
        //  switch (cmd) {
        //    case ADVANCE_RIGHT:
        //    case ADVANCE_LEFT:
        //  }
        //
        return true;
      }
      return false;
    }

    bool richtext_ctl::advance_forward(view &v, bookmark &bm) {
      auto on_pos = [&](bookmark &bm) -> bool {
        if (!bm.node->belongs_to(root(),false))
          return false;
        if (bm.at_element_end() &&
            bm.node->get_element()->is_block_element(v)) {
          element *that         = bm.node->cast<element>();
          element *next_to_that = that->next_element();
          if (next_to_that && !next_to_that->is_text_box()) {
            // bm = this->insert_text_block_placholder(v, that, true);
            return true;
          } else if (!next_to_that && !that->is_text_box()) {
            // bm = this->insert_text_block_placholder(v, that, true);
            return true;
          }
          return false;
        } /*else if (bm.at_element_start() && bm.node->get_element()->is_block_element(v)) {
          return false;
        }*/
        return true;
      };
      return advance_bookmark(v, bm, true, on_pos);
    }
    bool richtext_ctl::advance_backward(view &v, bookmark &bm) {
      auto on_pos = [&](bookmark &bm) -> bool {
        if (!bm.node->belongs_to(root(), false))
          return false;
        if (bm.at_element_start() &&
            bm.node->get_element()->is_block_element(v)) {
          element *that         = bm.node->cast<element>();
          element *prev_to_that = that->prev_element();
          if (prev_to_that && !prev_to_that->is_text_box()) {
            // bm = this->insert_text_block_placholder(v, that, false);
            return true;
          } else if (!prev_to_that && !that->is_text_box()) {
            // bm = this->insert_text_block_placholder(v, that, false);
            return true;
          }
          return false;
        } /*else if (bm.at_element_end() &&
                   bm.node->get_element()->is_block_element(v)) {
          return false;
        }*/
        return true;
      };
      return advance_bookmark(v, bm, false, on_pos);
    }

    bool richtext_ctl::on(view &v, element *self, event_command &evt) {
      if (!this->self) return false;
#ifdef _DEBUG
            if (evt.command == event_command::EDIT_UNDO()) 
              evt.cmd = evt.cmd;
            if(evt.cmd == (event_command::EXEC))
              evt.cmd = evt.cmd;
#endif // _DEBUG

      if (!evt.doit() && !evt.checkit()) return false;

      if (!evt.target->belongs_to(v, self, true))
        return false;

      tag::symbol_t tag_sym;

      ustring cmd = evt.command;

      if (cmd == event_command::NAVIGATE_BACKWARD()) {
        bool     keep_anchor = evt.data.get(false);
        bookmark bm          = caret;
        if(evt.doit()) {
          if (caret != anchor && !keep_anchor) {
            select(v, caret < anchor ? caret : anchor);
            return true;
          }
          if (advance_backward(v, bm)) {
            if (keep_anchor)
              select(v, bm, anchor);
            else
              select(v, bm);
            return true;
          }
        }
      } else if (cmd == event_command::NAVIGATE_FORWARD()) {
        bool     keep_anchor = evt.data.get(false);
        bookmark bm          = caret;
        if (evt.doit()) {
          if ((caret != anchor) && !keep_anchor) {
            select(v, caret > anchor ? caret : anchor);
            return true;
          }
          if (advance_forward(v, bm)) {
            if (keep_anchor)
              select(v, bm, anchor);
            else
              select(v, bm);
            return true;
          }
        }
      }
      else if (cmd == event_command::EDIT_COPY()) {
        if (evt.doit()) return copy(v);
        evt.result = value(can_copy(v) ? CMD_AVAILABLE : CMD_DISABLED);
        return true;
      }
      else if (cmd == event_command::EDIT_CUT()) {
        if (evt.doit()) return cut(v);
        evt.result = value(can_cut(v) ? CMD_AVAILABLE : CMD_DISABLED);
        return true;
      } else if (cmd == event_command::EDIT_PASTE_TEXT()) {
        if (evt.doit()) return can_paste(v) && paste_text(v);
        evt.result = value(can_paste(v) ? CMD_AVAILABLE : CMD_DISABLED);
        return true;
      } else if (cmd == event_command::EDIT_PASTE()) {
        if (evt.doit()) return can_paste(v) && paste(v);
        evt.result = value(can_paste(v) ? CMD_AVAILABLE : CMD_DISABLED);
        return true;
      } else if (cmd == event_command::EDIT_UNDO()) {
        if (evt.doit()) return undo(v);
        evt.result = value(can_undo(v) ? CMD_AVAILABLE : CMD_DISABLED);
        return is_editable(self);
      } else if (cmd == event_command::EDIT_REDO()) {
        if (evt.doit()) return redo(v);
        evt.result = value(can_redo(v) ? CMD_AVAILABLE : CMD_DISABLED);
        return is_editable(self);
      } else if (cmd == event_command::EDIT_DELETE_WORD_PREV()) {
        if (!evt.doit()) return is_editable(self);
        if (move_caret(v, ADVANCE_WORD_START, true))
          return is_editable(self) && delete_backward(v);
        return false;
      } else if (cmd == event_command::EDIT_DELETE_LINE_START()) {
        if (!evt.doit()) return is_editable(self);
        if (move_caret(v, ADVANCE_HOME, true))
          return is_editable(self) && delete_backward(v);
        return false;
      } else if (cmd == event_command::EDIT_DELETE_PREV()) {
        if (!evt.doit()) return is_editable(self);
        return is_editable(self) && delete_backward(v);
      } else if (cmd == event_command::EDIT_DELETE_WORD_NEXT()) {
        if (!evt.doit()) return is_editable(self);
        if (move_caret(v, ADVANCE_WORD_END, true))
          return is_editable(self) && delete_forward(v);
        return false;
      } else if (cmd == event_command::EDIT_DELETE_LINE_END()) {
        if (!evt.doit()) return is_editable(self);
        if (move_caret(v, ADVANCE_END, true))
          return is_editable(self) && delete_forward(v);
        return false;
      } else if (cmd == event_command::EDIT_DELETE_NEXT()) {
        if (!evt.doit()) return is_editable(self);
        return is_editable(self) && delete_forward(v);
      } else if (cmd == WCHARS("edit:insert-break")) {
        if (evt.doit())
          return insert_break(v, anchor, caret);
        else {
          return is_editable(self);
        }
      } else if (cmd == WCHARS("edit:insert-text")) {
        if (evt.doit())
          return insert_text(v, evt.data.to_string());
        else
          return is_editable(self);
      } else if (is_plaintext()) {
        return super::on(v, self, evt);
        // rest of commands are richtext only
      }
      // only <richtext>
      else if (cmd == WCHARS("edit:insert-html")) {
        if (evt.doit())
          return insert_html(v, u8::cvt(evt.data.to_string()).chars_as_bytes());
        else
          return is_editable(self);
      } else if (cmd.like(W("format:apply-span:*"))) {
        // if(!evt.doit() ) return is_editable(self);
        string        acmd = cmd;
        chars         stag = chars(acmd).r_tail(':');
        tag::symbol_t t    = tag::symbol(stag);
        attribute_bag atts;
        value_to_attribute_bag(evt.data, atts);

        if (!evt.doit()) {
          tag::symbol_t sym = tag::symbol(stag);
          bool r  = !!selection_contains(v, slice<tag::symbol_t>(sym), atts);
          if (r) {
            if(span_shelve.remove_contains(sym))
              r = false;
          }
          else {
            if(span_shelve.apply_contains(sym))
              r = true;
          }
          int  rv = r ? CMD_SELECTED : CMD_AVAILABLE;
          //if (!has_non_empty_selection()) rv |= CMD_DISABLED;
          evt.result = value(rv);
          return is_editable(self);
        } else if (is_editable(self))
          return set_span(v, t, atts);
      } else if (cmd.like(W("format:pull-element:*"))) {
        string        acmd = cmd;
        chars         stag = chars(acmd).r_tail(':');
        tag::symbol_t t    = tag::symbol(stag);
        attribute_bag atts;
        value_to_attribute_bag(evt.data, atts);

        if (!evt.doit()) {
          tag::symbol_t sym = tag::symbol(stag);
          bool r  = !!selection_contains(v, slice<tag::symbol_t>(sym), atts);
          int  rv = r ? CMD_AVAILABLE : CMD_DISABLED;
          // if (!has_non_empty_selection())
          //  rv |= CMD_DISABLED;
          evt.result = value(rv);
          return is_editable(self);
        } else if (is_editable(self))
          return unwrap(v, t, atts);
      }
      /*else if (cmd.like(W("format:wrap-block:*"))) {
        string acmd = cmd;
        chars stag = chars(acmd).r_tail(':');
        tag::symbol_t t = tag::symbol(stag);
        attribute_bag atts;
        value_to_attribute_bag(evt.data, atts);

        if (!evt.doit()) {
          tag::symbol_t sym = tag::symbol(stag);
          bool r = !!selection_contains(v, slice<tag::symbol_t>(sym), atts);
          int rv = r ? CMD_AVAILABLE : CMD_DISABLED;
          //if (!has_non_empty_selection())
          //  rv |= CMD_DISABLED;
          evt.result = value(rv);
          return is_editable(self);
        }
        else if (is_editable(self))
          return unwrap(v, t, atts);
      }*/

      else if (cmd.like(W("format:toggle-span:*"))) {
        string               acmd  = cmd;
        chars                stags = chars(acmd).r_tail(':');
        atokens              stags_list(stags, CHARS("|"));
        chars                stag;
        array<tag::symbol_t> tlist;
        while (stags_list.next(stag))
          tlist.push(tag::symbol(stag));
        attribute_bag atts;
        value_to_attribute_bag(evt.data, atts);

        if (!evt.doit()) {

          bool r  = !!selection_contains(v, tlist(), atts);
          if (r) {
            if(span_shelve.remove_contains_one_of(tlist()))
              r = false;
          }
          else {
            if(span_shelve.apply_contains_one_of(tlist()))
              r = true;
          }

          int  rv = r ? CMD_SELECTED : CMD_AVAILABLE;
          //if (!has_non_empty_selection()) rv |= CMD_DISABLED;
          evt.result = value(rv);
          return is_editable(self);
        } else if (is_editable(self))
          return toggle_span(v, tlist(), atts);
      } else if (cmd.like(W("format:remove-span:*"))) {
        string               acmd  = cmd;
        chars                stags = chars(acmd).r_tail(':');
        array<tag::symbol_t> tlist;
        if (stags == CHARS("*")) {
          tag::all_formatting_spans(tlist);
        } else {
          atokens stags_list(stags, CHARS("|"));
          chars   stag;
          while (stags_list.next(stag))
            tlist.push(tag::symbol(stag));
        }

        attribute_bag atts;
        value_to_attribute_bag(evt.data, atts);

        if (!evt.doit()) {
          bool r = has_non_collapsed_selection() && selection_contains(v, tlist(), atts);
          int rv     = r ? CMD_AVAILABLE : CMD_DISABLED;
          evt.result = value(rv);
          return is_editable(self);
        } else if (is_editable(self))
          return remove_spans(v, tlist(), atts);
      } else if (cmd.like(W("format:toggle-list:*"))) {

        wchars        tag = wchars(cmd).r_tail(':');
        attribute_bag atts;
        wchars        selector;
        tag::symbol_t list_tag;
        if (tag == WCHARS("dl")) {
          selector = WCHARS("dl>dd,dl>dt");
          list_tag = tag::T_DL;
        } else if (tag == WCHARS("ol")) {
          selector = WCHARS("ol>li");
          list_tag = tag::T_OL;
        } else if (tag == WCHARS("ul")) {
          selector = WCHARS("ul>li");
          list_tag = tag::T_UL;
        } else if (tag == WCHARS("dir")) {
          selector = WCHARS("dir>li");
          list_tag = tag::T_DIR;
        } else if (tag == WCHARS("menu")) {
          selector = WCHARS("menu>li");
          list_tag = tag::T_MENU;
        } else
          return false;

        value_to_attribute_bag(evt.data, atts);
        bool has_list_items = !!selection_contains(v, selector);

        if (!evt.doit()) {
#pragma TODO("CMD_DISABLED if inside PLAIN TEXT!")
          evt.result = value(has_list_items ? CMD_SELECTED : CMD_AVAILABLE);
          return true;
        } else if (is_editable(self))
          return toggle_list(v, list_tag, atts, !has_list_items);
      } else if (cmd == WCHARS("format:toggle-pre")) {

        wchars selector = WCHARS("pre");
        bool   is_pre   = !!selection_contains(v, selector);
        // bool pre_disabled =

        if (!evt.doit()) {
          //#pragma TODO("CMD_DISABLED if inside PLAIN TEXT!")

          int rv = is_pre ? CMD_SELECTED : CMD_AVAILABLE;
          if (!can_pre(v) || is_readonly(self) )
            rv |= CMD_DISABLED;
          evt.result = value(rv);
          return true;
        } else if (is_editable(self)) {
          attribute_bag atts;
          value_to_attribute_bag(evt.data, atts);
          return toggle_pre(v, atts, !is_pre);
        }
      } else if (cmd == WCHARS("edit:insert-soft-break")) {
        if (evt.doit())
          return insert_soft_break(v, anchor, caret);
        else {
          return is_editable(self);
        }
      } else if (cmd == WCHARS("edit:insert-block-break")) {
        if (evt.doit())
          return insert_block_break(v, anchor, caret);
        else {
          return is_editable(self);
        }
      } else if (cmd == WCHARS("format:indent")) {
        if (evt.doit())
          return do_indent(v);
        else {
          evt.result = value(can_indent(v) ? CMD_AVAILABLE : CMD_DISABLED);
          return is_editable(self);
        }
      } else if (cmd == WCHARS("format:unindent")) {
        if (evt.doit())
          return do_unindent(v);
        else {
          evt.result = value(can_unindent(v) ? CMD_AVAILABLE : CMD_DISABLED);
          return true;
        }
      } else if (cmd.like(W("format:morph-block:*"))) {
        wchars tag_name = wchars(cmd).r_tail(':');
        // attribute_bag atts;
        tag::symbol_t tag_sym = tag::symbol(string(tag_name));
        if (!tag_sym) return false;
        if (evt.doit())
          return morph_blocks(v, anchor, caret, tag_sym, true);
        else
          evt.result = value(morph_blocks(v, anchor, caret, tag_sym, false)
                                 ? CMD_AVAILABLE
                                 : CMD_DISABLED);
        return true;
      }

      else if (cmd.like(W("format:toggle-block:*"))) {
        string               acmd = cmd;
        chars                stag = chars(acmd).r_tail(':');
        array<tag::symbol_t> tlist;
        tlist.push(tag::symbol(stag));

        tristate_v r = selection_contains_blocks(v, tlist());

        if (!evt.doit()) {
          if (is_table_selection()) {
            evt.result = value(CMD_DISABLED);
          } else {
            if (r.is_undefined())
              evt.result = value(CMD_DISABLED);
            else
              evt.result = value(r ? CMD_SELECTED : CMD_AVAILABLE);
          }
          return is_editable(self);
        } else if (is_editable(self) && r.is_defined()) {
          if (r)
            return remove_block(v, anchor, caret, tlist[0]);
          else
            return apply_block(v, anchor, caret, tlist[0]);
        }
      }

      else if (cmd.like(W("edit:merge-table-cells"))) {
        if (evt.checkit()) {
          if (is_table_range_selection() && is_editable(self))
            evt.result = value(CMD_AVAILABLE);
          else
            evt.result = value(CMD_DISABLED);
          return true;
        }

        if (evt.doit()) {
          if (!is_table_range_selection() || !is_editable(self)) return false;
          return merge_cells(v);
        }
      } else if (cmd.like(W("edit:delete-table-rows"))) {
        if (evt.checkit()) {
          if (!is_editable(self))
            evt.result = value(CMD_DISABLED);
          else if (is_table_range_selection())
            evt.result = value(CMD_AVAILABLE);
          else if (is_selection_in_table_cell())
            evt.result = value(CMD_AVAILABLE);
          else
            evt.result = value(CMD_DISABLED);
          return true;
        }

        if (evt.doit()) {
          if (!is_editable(self)) return false;
          if (!is_table_range_selection() && !is_selection_in_table_cell())
            return false;
          return delete_rows(v);
        }
      }

      else if (cmd.like(W("edit:delete-table-columns"))) {
        if (evt.checkit()) {
          if (!is_editable(self))
            evt.result = value(CMD_DISABLED);
          else if (is_table_range_selection())
            evt.result = value(CMD_AVAILABLE);
          else if (is_selection_in_table_cell())
            evt.result = value(CMD_AVAILABLE);
          else
            evt.result = value(CMD_DISABLED);
          return true;
        }

        if (evt.doit()) {
          if (!is_editable(self)) return false;
          if (!is_table_range_selection() && !is_selection_in_table_cell())
            return false;
          return delete_cols(v);
        }
      }

      else if (cmd.like(W("edit:insert-table-column:*"))) {
        if (evt.checkit()) {
          if (!is_editable(self))
            evt.result = value(CMD_DISABLED);
          else if (is_table_range_selection())
            evt.result = value(CMD_AVAILABLE);
          else if (is_selection_in_table_cell())
            evt.result = value(CMD_AVAILABLE);
          else
            evt.result = value(CMD_DISABLED);
          return true;
        }

        if (evt.doit()) {
          if (!is_editable(self)) return false;
          if (!is_table_range_selection() && !is_selection_in_table_cell())
            return false;
          return insert_column(v, cmd().ends_with(WCHARS(":after")));
        }
      } else if (cmd.like(W("edit:insert-table-row:*"))) {
        if (evt.checkit()) {
          if (!is_editable(self))
            evt.result = value(CMD_DISABLED);
          else if (is_table_range_selection())
            evt.result = value(CMD_AVAILABLE);
          else if (is_selection_in_table_cell())
            evt.result = value(CMD_AVAILABLE);
          else
            evt.result = value(CMD_DISABLED);
          return true;
        }

        if (evt.doit()) {
          if (!is_editable(self)) return false;
          if (!is_table_range_selection() && !is_selection_in_table_cell())
            return false;
          return insert_row(v, cmd().ends_with(WCHARS(":after")));
        }
      } else if (cmd.like(W("edit:split-table-cells"))) {
        if (evt.checkit()) {
          if (!is_editable(self))
            evt.result = value(CMD_DISABLED);
          else if (is_table_range_selection()) {
            FOREACH(i, selected) {
              auto cell = selected[i];
              if (cell->atts.get_colspan() > 1 || cell->atts.get_rowspan() > 1) {
                evt.result = value(CMD_AVAILABLE);
                return true;
              }
            }
            evt.result = value(CMD_DISABLED);
          }
          else if (element *cell = is_selection_in_table_cell())
            evt.result =
                cell->atts.get_colspan() > 1 || cell->atts.get_rowspan() > 1
                    ? value(CMD_AVAILABLE)
                    : value(CMD_DISABLED);
          else
            evt.result = value(CMD_DISABLED);
          return true;
        }

        if (evt.doit()) {
          if (!is_editable(self)) return false;
          if (!is_table_range_selection() && !is_selection_in_table_cell())
            return false;
          return split_cells(v);
        }
      }

      return super::on(v, self, evt);
    }

    bool richtext_ctl::set_target_to(view &v, bookmark pos) {
      refresh_selection(v);
      if (pos.valid()) pos.normalize();
      target = pos;
      refresh_selection(v);
      return true;
    }

    bool richtext_ctl::on(view &v, element *self, event_exchange &evt) {
      if (!this->self) return false;
      if (is_readonly(self)) return false;

      switch (evt.cmd) {

      case X_WILL_ACCEPT_DROP:
      {
        //dbg_report("X_WILL_ACCEPT_DROP\n");
        bool can_drop = false;
        evt.data->available_formats([&](clipboard::clipboard_format cf)->bool {
          if (is_plaintext()) {
            evt.cancel_event(!(cf == clipboard::cf_text));
          }
          else {
            evt.cancel_event(!(cf == clipboard::cf_html || cf == clipboard::cf_text || cf == clipboard::cf_picture));
          }
          return true;
        });
        return true;
      }

      case X_DRAG: {
        bookmark bm = self->find_text_pos(v, evt.pos);
        if (evt.dd_source == self) {
          auto selection = selection_ctx::normalized();
          selection.first.linearize();
          selection.second.linearize();
          if (bm.is_between(selection.first, selection.second)) return false;
        }
        set_target_to(v, bm);
        v.refresh(self);
      }
        return true;
      case X_DRAG_ENTER: return true;
      case X_DRAG_LEAVE: set_target_to(v, bookmark()); return true;
      case X_DROP: {
        bookmark bm = self->find_text_pos(v, evt.pos);
        if ((evt.dd_source == self) && bm.is_between(caret, anchor))
          return false;
        // insert_range(v,pti->val,bm,evt.dd_source == self);

        if (evt.dd_source != self || evt.dd_modes != dd_move)
          select(v, bookmark());
        paste(v, evt.data, bm);
        v.set_focus(self, BY_MOUSE);
        set_target_to(v, bookmark());
      }
        return true;
      }
      return false;
    }

    static bool needs_to_show_caret(view &v, element *self) {
      //return v.focus_element && v.focus_element->belongs_to(self, true);
      return self->state.focus() || self->state.owns_focus();
    }

    bool richtext_ctl::on_timer(view &v, element *, timer_id tid,
                                TIMER_KIND kind) {
      if (!this->self) return false;
      if (kind != INTERNAL_TIMER) return false;

      if (tid == CARET_TIMER_ID) {
#if defined(FANCY_CARET)
        ++caret_cycle;
#endif
        if (needs_to_show_caret(v, self)) {
          if (caret != anchor || !is_editable(self)) {
            caret_state.clear();
            refresh_selection(v);
            return false;
          } else if (!self->state.focus()) {
            caret_state = 3;
          } else if (caret_state == 2) {
            caret_state = 1;
          } else if (caret_state == 1) {
            caret_state = 2;
          }
          refresh_selection(v);
          // v.refresh(self);
          return true;
        } else {
          caret_state.clear();
          refresh_selection(v);
          // v.refresh(self);
          return false;
        }
      } else if (tid == SPELL_CHECK_TIMER_ID) {
        spell_check(v);
        return false;
      }
      return false;
    }

    void richtext_ctl::show_caret(view &v, bool onOff) {
      if (!self) return;
      if (onOff) {
#if defined(FANCY_CARET)
        caret_cycle = 0;
#endif
        if (needs_to_show_caret(v, self) && is_editable(self)) {
          caret_state = 2;
          v.start_timer(self, CARET_TIMER_MS, CARET_TIMER_ID, INTERNAL_TIMER);
        } else {
          caret_state = 1;
        }
      } else {
        caret_state.clear();
        v.stop_timer(self, CARET_TIMER_ID, INTERNAL_TIMER);
      }
      refresh_selection(v);
    }

    static argb get_color_behind_caret(view &v, element *el) {
      if (!el) return argb(255, 255, 255);
      style *cs = el->get_style(v);
      if (cs->has_background_color()) return cs->back_color.to_argb();
      return get_color_behind_caret(v, el->parent);
    }

    argb richtext_ctl::selection_caret_color(view &v) {
      argb c;
      if (caret.valid()) {
        const style *cs = caret.node->get_element()->get_style(v);
        if (cs->text_selection_caret_color.is_defined())
          return cs->text_selection_caret_color.to_argb();
        c = get_color_behind_caret(v, caret.node->get_element());
        c = c.inverse();
      } else {
        const style *cs = self->get_style(v);
        if (cs->text_selection_caret_color.is_defined())
          return cs->text_selection_caret_color.to_argb();
        argb c = cs->back_color.is_defined() ? cs->back_color.to_argb()
                                             : argb(255, 255, 255);
        if (c.alfa == 0) return argb(0, 0, 0);
        c.alfa = 255;
        return c.inverse();
      }
      return c;
    }

    argb richtext_ctl::selection_tail_back_color(view &v) {
      argb c = selection_back_color(v);
      // c.alfa = caret_tail_opacity;
      return c;
    }

    bool richtext_ctl::get_caret_place(view &v, element *self, rect &rc) {
      if (caret.valid()) {
        caret_metrics gm;
        if (caret.get_caret_metrics(v, gm)) {
          rc = gm.caret_v_bar() + gm.elem->rel_pos(v, self);
          return true;
          // rc.origin.x
        }
      }
      return false;
    }

    richtext_ctl::caret_def richtext_ctl::block_start(view& v)
    {
      if (_block_start.path.is_undefined()) {
        _block_start.path = v.app->create_path();
        //wchars svg_d = WCHARS("M0 0 H1 V3 H5 L3 1 L4 0 L9 5 L4 10 L3 9 L5 7 H1 V14 H0 Z");
        //_block_start.offset = point(-9, -7);
        //wchars svg_d = WCHARS("M1 0 L7 6 L1 12 L0 11 L5 6 L0 1 Z");
        //_block_start.offset = point(7, 6);
        wchars svg_d = WCHARS("M0 0 H2 V5 H9 V7 H2 V12 H0 Z");
        _block_start.offset = point(4, 9);
        html::parse_d_path(_block_start.path, svg_d);
      }
      return _block_start;
    }
    richtext_ctl::caret_def richtext_ctl::block_end(view& v)
    {
      if (_block_end.path.is_undefined()) {
        _block_end.path = v.app->create_path();
        //wchars svg_d = WCHARS("M0 0 H1 V7 H5 L3 5 L4 4 L9 9 L4 14 L3 13 L5 11 H1 V14 H0 Z");
        //_block_end.offset = point(-9, 5);
        //wchars svg_d = WCHARS("M0 2 L2 0 L8 6 L2 12 L0 10 L4 6 Z");
        //wchars svg_d = WCHARS("M1 0 L7 6 L1 12 L0 11 L5 6 L0 1 Z");
        //_block_end.offset = point(7, 6);
        wchars svg_d = WCHARS("M0 0 H2 V5 H9 V7 H2 V12 H0 Z");
        _block_end.offset = point(4, 3);
        html::parse_d_path(_block_end.path, svg_d);
      }
      return _block_end;
    }

    richtext_ctl::caret_def richtext_ctl::row_start(view& v)
    {
      if (_row_start.path.is_undefined()) {
        _row_start.path = v.app->create_path();
        //wchars svg_d = WCHARS("M0 4 H7 L5 2 L7 0 L13 6 L7 12 L5 10 L7 8 H0 Z M1 5 H4 V7 H1 Z M5 5 H8 V7 H5 Z");
        //_row_start.offset = point(13, 6);
        wchars svg_d = WCHARS("M0 0 H9 V2 H2 V9 H0 Z");
        _row_start.offset = point(3, 3);

        html::parse_d_path(_row_start.path, svg_d);
      }
      return _row_start;
    }
    richtext_ctl::caret_def richtext_ctl::row_end(view& v)
    {
      if (_row_end.path.is_undefined()) {
        _row_end.path = v.app->create_path();
        wchars svg_d = WCHARS("M0 7 H7 V0 H9 V9 H0 Z");
        _row_end.offset = point(-7, 6);
        html::parse_d_path(_row_end.path, svg_d);
      }
      return _row_end;
    }

    bool richtext_ctl::draw_caret(view &v, graphics *pg,
                                  const caret_metrics &cm) {
      if (!this->is_editable(self)) {
        if (caret_state) show_caret(v, false);
        return false;
      }

      switch (cm.ctype) {
      case CHAR_POSITION: {
#if defined(FANCY_CARET)
        if (caret_cycle == 0 && !cm.glyph_rc.empty() && caret == anchor) {
          argb caret_tail_color = selection_tail_back_color(v);
          if (caret_tail_color.alfa) {
            argb tail_color = selection_back_color(v);
            tail_color.alfa = 84;
            pg->fill(tail_color, cm.glyph_rc);
          }
        }
#endif
        if (caret_state == 2 || target.valid()) {
          argb caret_color = selection_caret_color(v);
          pg->fill(caret_color, cm.caret_v_bar());
        }
        else if (caret_state == 3) {
          argb caret_color = selection_caret_color(v);
          pg->fill(caret_color, cm.caret_v_bar());
        }
      } break;
      case BLOCK_POSITION: {
        if (caret_state == 2) {
          argb caret_color = selection_caret_color(v);
          //auto bars        = cm.caret_bars(BLOCK_POSITION);
          caret_def cd;

          point pt;
          if (cm.at_end) {
            cd = this->block_end(v);
            pt = cm.glyph_rc.pointOf(1);
          }
          else {
            cd = this->block_start(v);
            pt = cm.glyph_rc.pointOf(7);
          }

          size px_per_dip = v.pixels_per_dip(size(1, 1));
          cd.offset *= px_per_dip;

          pg->push_state();
          pg->set_fill(caret_color);
          pg->translate(pt - cd.offset);
          pg->scale(px_per_dip);
          pg->draw_path(cd.path,false,true);
          pg->pop_state();
          //pg->fill(caret_color, std::get<0>(bars)); // horz
          //pg->fill(caret_color, std::get<1>(bars)); // vert
        }
      } break;
      case ROW_POSITION: {
        if (caret_state == 2) {
          argb caret_color = selection_caret_color(v);
          //auto bars        = cm.caret_bars(ROW_POSITION);

          caret_def cd;

          point pt;
          if (cm.at_end) {
            cd = this->row_end(v);
            pt = cm.glyph_rc.pointOf(1);
          }
          else {
            cd = this->row_start(v);
            pt = cm.glyph_rc.pointOf(7);
          }

          size px_per_dip = v.pixels_per_dip(size(1, 1));
          cd.offset *= px_per_dip;

          pg->push_state();
          pg->set_fill(caret_color);
          pg->translate(pt - cd.offset);
          pg->scale(px_per_dip);
          pg->draw_path(cd.path, false, true);
          pg->pop_state();

          //pg->fill(caret_color, std::get<0>(bars));
          //pg->fill(caret_color, std::get<1>(bars));
        }
      } break;
      }
      return true;
    }

    void richtext_ctl::node_expand(node *n, int at, int nitems) {}
    void richtext_ctl::node_shrink(node *n, int at, int nitems) {}

    bool richtext_ctl::selection_each(
        view &v, function<bool(element *, bool &skip)> visitor)
    {
      if (selection_ctx::is_table_range_selection()) {
        for (int r = table_rows.l; r <= table_rows.h; ++r) {
          helement tr = selection_ctx::table->child(r);
          for (int c = table_cols.l; c <= table_cols.h; ++c) {
            helement td = tr->child(c);
            if (td && each_element(v, td->start_pos(), td->end_pos(), visitor))
              return true;
          }
        }
        return false;
      }
      return each_element(v, anchor, caret, visitor);
    }

    bool
    richtext_ctl::each_element(view &v, bookmark start, bookmark end,
                               function<bool(element *, bool &skip)> visitor) {
      bool skip = false;

      if (!start.valid() || !end.valid()) return false;

      if (start > end) swap(start, end);

      element *last = start.node->get_element();

      if (visitor(start.node->get_element(), skip))
        return true;
      else if (skip)
        start = last->end_pos();

      if (start >= end) return false;

      pos_iterator pi(start, end);
      for (bookmark bm; pi(bm);) {
        //#ifdef _DEBUG
        //      bm.dbg_report("each_element");
        //#endif
        element *el;
        bool     skip = false;
        if (bm.at_element_start(el)) {
          last = el;
          if (visitor(el, skip)) return true;
          if (skip) pi.bm = el->end_pos();
        }
      }

      if (end.node->get_element() != last) {
        if (visitor(end.node->get_element(), skip)) return true;
      }

      return false;
    }

    bool richtext_ctl::each_block_element(
        view &v, bookmark start, bookmark end,
        function<bool(element *, bool &skip)> visitor, bool ui) {
      bool skip = false;

      if (!start.valid() || !end.valid()) return false;

      if (start > end) swap(start, end);

      auto get_block_element = [this, &v, ui](node *pn) -> element * {
        element *el = ui ? pn->get_ui_element() : pn->get_element();
        while (el) {
          if (el == this->self) break;
          if (el == this->root()) break;
          if (el->is_block_element(v)) return el;
          el = ui ? el->get_owner() : el->parent.ptr();
        }
        return nullptr;
      };

      element *last = // start.node->get_element();
          ui ? start.node->get_ui_element() : start.node->get_element();

      element *first_block = get_block_element(start.node);
      if (first_block && visitor(first_block, skip))
        return true;
      else if (skip)
        start = last->end_pos();

      if (start >= end) return false;

      pos_ui_iterator pi(v, start, end);
      for (bookmark bm; pi(bm);) {
        //#ifdef _DEBUG
        //      bm.dbg_report("each_element");
        //#endif
        element *el;
        bool     skip = false;
        if (bm.at_element_start(el) && el->is_block_element(v)) {
          last = el;
          if (visitor(el, skip)) return true;
          if (skip) pi.bm = el->end_pos();
        }
      }

      element *last_block = get_block_element(end.node);
      if (last_block && (last_block != last)) {
        if (visitor(last_block, skip)) return true;
      }

      return false;
    }

    element *richtext_ctl::selection_contains(view &v,
                                              slice<tag::symbol_t> list,
                                              const attribute_bag &atts) {
      element *found      = nullptr;
      auto     match_list = [&](node *n, bool &skip) -> bool {
        helement t  = n->get_element();
        helement pt = n->nearest_text_box();
        while (t && t != pt) {
          if (list.contains(t->tag) && t->atts.contains(atts)) {
            found = t;
            return true;
          }
          t = t->parent;
        }
        return false;
      };
      selection_each(v, match_list);
      return found;
    }

    tristate_v
      richtext_ctl::selection_contains_blocks(view &v, slice<tag::symbol_t> list, const attribute_bag &atts) {
      element *bfc           = nullptr;
      bool     different_bfc = false;
      element *found         = nullptr;
      auto     match_list    = [&](node *n, bool &skip) -> bool {
        helement t  = n->get_element();
        helement pt = t->nearest_bfc(v);
        if (!bfc)
          bfc = pt;
        else if (bfc != pt) {
          different_bfc = true;
          return true;
        }
        while (t && t != pt) {
          if (list.contains(t->tag) && t->atts.contains(atts)) {
            found = t;
            return true;
          }
          t = t->parent;
        }
        return false;
      };
      each_block_element(v, caret, anchor, match_list);
      if (different_bfc) return tristate_v();
      return found ? tristate_v(1) : tristate_v(0);
    }

    element* richtext_ctl::selection_contains(view &v, wchars selector) {
      // helement base = element::find_base(caret.node->get_element(),
      element* found = nullptr;
      auto match = [&](node *n, bool &skip) -> bool {
        helement t = n->is_element() ? n->cast<element>() : n->parent.ptr();
        found = find_first_parent(v, self, t, selector);
        return found != nullptr;
      };
      selection_each(v, match);
      return found;
    }

    bool richtext_ctl::selection_contains(view & v, wchars selector /*:root is base parent*/, array<helement>& list)
    {
      auto match = [&](node *n, bool &skip) -> bool {
        helement t = n->is_element() ? n->cast<element>() : n->parent.ptr();
        element* found = find_first_parent(v, self, t, selector);
        if (found && !list().contains(found))
          list.push(found);
        return false;
      };
      selection_each(v, match);
      return list.size() > 0;
    }

    bool richtext_ctl::set_span(view &v, tag::symbol_t t,
                                const attribute_bag &atts) {
      tag::symbol_t tags[1] = {t};
      if (selection_contains(v, items_of(tags)))
        remove_spans(v, items_of(tags));
      return apply_span(v, t, atts);
      // return true;
    }

    /*bool richtext_ctl::remove_spans(view& v, slice<tag::symbol_t> list, const
    attribute_bag& atts)
    {
      if (selection_contains(v, list))
        return remove_spans(v, list);
      return false;
    }*/

    bool richtext_ctl::toggle_span(view &v, slice<tag::symbol_t> list,
                                   const attribute_bag &atts) {

      if (has_collapsed_selection()) {
        if (span_shelve.apply_contains_one_of(list))
          return remove_spans(v, list, atts);
        if (span_shelve.remove_contains_one_of(list))
          return apply_span(v, list[0], atts);
      }

      if (selection_contains(v, list, atts))
        return remove_spans(v, list, atts);
      else
        return apply_span(v, list[0], atts);
      return false;
    }

    bool richtext_ctl::unwrap(view &v, tag::symbol_t t,
                              const attribute_bag &atts) {
      tag::symbol_t tags[1] = {t};
      element *     el      = selection_contains(v, items_of(tags));
      if (el) return unwrap_element(v, el);
      return false;
    }

    bool richtext_ctl::toggle_list(view &v, tag::symbol_t list_tag,
                                   const attribute_bag &atts, bool on_off) {
      bool r = on_off ? apply_list(v, anchor, caret, list_tag, atts)
                      : remove_list(v, anchor, caret, list_tag, atts);
      if (r) select(v, caret, anchor);
      return r;
    }

    bool richtext_ctl::toggle_pre(view &v, const attribute_bag &atts,
                                  bool on_off) {
      bool r = on_off ? apply_pre(v, atts) : remove_pre(v);
      if (r) select(v, caret, anchor);
      return r;
    }

    bool richtext_ctl::insert_text(view &v, wchars t, bookmark where) {
      if (where.valid()) select(v, where);
      return insert_chars(v, anchor, caret, t);
    }

    pair<bookmark, bookmark>
    richtext_ctl::delete_cells(view &v, block_table_body *table_body,
                               irange table_rows, irange table_cols, bool merge) {
      uint rows, cols;
      if (!table_body->get_rows_cols(rows, cols))
        return pair<bookmark, bookmark>();

      auto decrement_rowspan = [&](action *group, element *cell, int by) {
        int n = cell->atts.get_rowspan();
        n -= by;
        ASSERT(n > 0);
        if (n == 1)
          change_attr::del(v, this, group, cell, "rowspan");
        else
          change_attr::set(v, this, group, cell, "rowspan", itow(n));
      };

      auto decrement_colspan = [&](action *group, element *cell, int by) {
        int n = cell->atts.get_colspan();
        n -= by;
        ASSERT(n > 0);
        if (n == 1)
          change_attr::del(v, this, group, cell, "colspan");
        else
          change_attr::set(v, this, group, cell, "colspan", itow(n));
      };

      auto find_node_position_for_cell = [&](int atrow, int atcol) -> int {
        helement row = table_body->ui_child(atrow);
        ASSERT(row);
        int cellnodepos = row->nodes.size();
        for (uint c = 0; c < cols; ++c) {
          if (int(c) == atcol) break;
          element *pc = table_body->get_cell_at(atrow, c);
          if (pc) cellnodepos = pc->node_index + 1;
        }
        return cellnodepos;
      };

      auto split_cell = [&](action *group, element *cell, irange cell_rows,
                            irange cell_cols) {
        for (int r = cell_rows.l; r <= cell_rows.h; ++r) {
          for (int c = cell_cols.l; c <= cell_cols.h; ++c) {
            if (r == cell_rows.l && c == cell_cols.l) continue;
            helement nc = new element(c > 0 && r > 0 ? tag::T_TD : cell->tag);
            int      cellnodepos = find_node_position_for_cell(r, c);
            helement row         = table_body->ui_child(r);
            ASSERT(row);
            insert_node::exec(v, this, group, row, cellnodepos, nc);
          }
        }
        change_attr::del(v, this, group, cell, "colspan");
        change_attr::del(v, this, group, cell, "rowspan");
      };

      auto do_delete = [&](action *group) -> pair<bookmark, bookmark> {

        if (table_cols.length() == int(cols) &&
            table_rows.length() == int(rows)) {

          delete_node::exec(v, this, group, table_body->parent_table());
          return pair<bookmark, bookmark>();
        } else if (!merge &&
                   table_cols.length() == int(cols)) // removal of full rows
        {
          // scan cells for row/colspans
          for (int r = int(rows) - 1; r >= 0; --r) {
            for (int c = int(cols) - 1; c >= 0; --c) {
              irange cell_r, cell_c;
              helement cell =
                  table_body->get_actual_cell_at(r, c, cell_r, cell_c);
              ASSERT(cell);
              irange r_intersection = table_rows & cell_r;
              if (!r_intersection.empty()) {
                if (cell_r.l < table_rows.l) //  cell starts outside of the
                                             //  range so we need to adjust
                                             //  rowspan
                  decrement_rowspan(group, cell, r_intersection.length());
                else if (cell_r.h > table_rows.h) //  cell starts inside the
                                                  //  range and ends outside of
                                                  //  it
                {
                  // we need to move cell down
                  helement nextrow = table_body->ui_child(table_rows.h + 1);
                  ASSERT(nextrow);
                  int cellnodepos =
                      find_node_position_for_cell(table_rows.h + 1, cell_c.l);
                  decrement_rowspan(group, cell, r_intersection.length());
                  delete_node::exec(v, this, group, cell);
                  insert_node::exec(v, this, group, nextrow, cellnodepos, cell);
                }
              }
            }
          }
          // removal of full rows
          array<helement> rows;
          for (int n = table_rows.h; n >= table_rows.l; --n)
            rows.push(table_body->ui_child(n));
          for (auto row : rows)
            delete_node::exec(v, this, group, row);
        } else if (!merge && table_rows.length() == int(rows)) {
          // removal of full columns

          array<helement> cells;

          // scan cells for row/colspans
          for (int r = int(rows) - 1; r >= 0; --r) {
            for (int c = int(cols) - 1; c >= 0; --c) {
              irange cell_r, cell_c;
              helement cell = table_body->get_actual_cell_at(r, c, cell_r, cell_c);
              ASSERT(cell);
              irange c_intersection = table_cols & cell_c;
              if (!c_intersection.empty()) {
                if (cell_c.l < table_cols.l) //  cell starts outside of the
                                             //  range so we need to adjust
                                             //  colspan
                  decrement_colspan(group, cell, c_intersection.length());
                else if (cell_c.h > table_cols.h) //  cell starts inside the
                                                  //  range and ends outside of
                                                  //  it
                {
                  // we need to move it right
                  helement therow = table_body->ui_child(cell_r.l);
                  ASSERT(therow);
                  int cellnodepos =
                      find_node_position_for_cell(cell_r.l, cell_c.h + 1);
                  // for (uint c = int(cols) - 1; c >= 0; --c) {
                  //  if (int(c) == cell_c.h) break;
                  //  element* pc = table_body->get_cell_at(cell_r.l, c);
                  //  if (pc) cellnodepos = pc->node_index - 1;
                  //}
                  decrement_colspan(group, cell, c_intersection.length());
                  delete_node::exec(v, this, group, cell);
                  insert_node::exec(v, this, group, therow, cellnodepos, cell);
                } else if (c == cell_c.l && r == cell_r.l) // origin of full
                                                           // cell that fully
                                                           // contained in the
                                                           // range - remove it
                  cells.push(cell);
              }
            }
          }
          for (auto cell : cells)
            delete_node::exec(v, this, group, cell);
        } else // region inside table_body - merging
        {
          array<helement> cells;
          // scan cells for row/colspans
          for (int r = int(rows) - 1; r >= 0; --r) {
            for (int c = int(cols) - 1; c >= 0; --c) {
              irange    cell_r, cell_c;
              helement cell =
                  table_body->get_actual_cell_at(r, c, cell_r, cell_c);
              ASSERT(cell);
              irange c_intersection = table_cols & cell_c;
              irange r_intersection = table_rows & cell_r;

              if (c_intersection.empty() || r_intersection.empty()) continue;

              if (cell_r != r_intersection || cell_c != c_intersection) {
                split_cell(group, cell, cell_r, cell_c);
                table_body->drop_layout();
                table_body->check_layout(v);
                // return pair<bookmark, bookmark>();
              }

              cell = table_body->get_cell_at(r, c);
              if (cell) cells.insert(0, cell);
            }
          }
          helement basecell = cells[0];
          for (int n = 1; n < cells.size(); ++n) {
            // insert_nodes::exec(v, this, group, therow, cellnodepos, cell);
            // delete_node::exec(v, this, group, cells[n]);
            helement     cell  = cells[n];
            array<hnode> nodes = cell->nodes();
            delete_nodes_range::exec(v, this, group, cell, 0, nodes.size());
            insert_nodes::exec(v, this, group, basecell, basecell->nodes.size(),
                               nodes());
            delete_node::exec(v, this, group, cell);
          }
          change_attr::set(v, this, group, basecell, "colspan",
                           itow(table_cols.length()));
          change_attr::set(v, this, group, basecell, "rowspan",
                           itow(table_rows.length()));
          return pair<bookmark, bookmark>(basecell->start_caret_pos(v),
                                          basecell->end_caret_pos(v));
        }
        return pair<bookmark, bookmark>();
      };

      if (transact_group) { return do_delete(transact_group); }

      handle<range_action> group =
          new range_action(this, WCHARS("delete cell range"));

      try {
        pair<bookmark, bookmark> bms = do_delete(group);

        push(v, group);

        v.commit_update();

        return bms;
      } catch (const tool::exception &) { group->undo(v, this); }
      // otherwise it is just a prolongation of previous op
      return pair<bookmark, bookmark>();
    }

    pair<bookmark, bookmark>
    richtext_ctl::insert_cells(view &v, block_table_body *table_body, int n,
                               bool column) {
      uint rows, cols;
      if (!table_body->get_rows_cols(rows, cols))
        return pair<bookmark, bookmark>();

      auto increment_rowspan = [&](action *group, element *cell, int by) {
        int n = cell->atts.get_rowspan();
        n += by;
        change_attr::set(v, this, group, cell, "rowspan", itow(n));
      };

      auto increment_colspan = [&](action *group, element *cell, int by) {
        int n = cell->atts.get_colspan();
        n += by;
        change_attr::set(v, this, group, cell, "colspan", itow(n));
      };

      auto find_node_position_for_cell = [&](int atrow, int atcol) -> int {
        helement row = table_body->ui_child(atrow);
        ASSERT(row);
        int cellnodepos = 0;
        for (uint c = 0; c < cols; ++c) {
          if (int(c) == atcol) break;
          element *pc = table_body->get_cell_at(atrow, c);
          if (pc) cellnodepos = pc->node_index + 1;
        }
        return cellnodepos;
      };

      auto insert_column = [&](action *group) -> pair<bookmark, bookmark> {
        helement first, last;
        for (int r = int(rows) - 1; r >= 0; --r) {
          irange    cell_r, cell_c;
          helement cell;
          if (n < int(cols))
            cell = table_body->get_actual_cell_at(r, n, cell_r, cell_c);
          else {
            cell = table_body->get_actual_cell_at(r, cols - 1, cell_r, cell_c);
            cell_c.l = cell_c.h = n;
            cell_r.l = cell_r.h = r;
          }
          if (cell_r.l < r) continue; // it starts above, skip it
          if (cell_c.l < n)           // spanned cell, need to expand it
            increment_colspan(group, cell, 1);
          else {
            helement newcell     = new element(cell->tag);
            int      cellnodepos = find_node_position_for_cell(r, n);
            helement row         = table_body->ui_child(r);
            ASSERT(row);
            insert_node::exec(v, this, group, row, cellnodepos, newcell);
            if (!first) first = newcell;
            last = newcell;
          }
        }
        if (last)
          return pair<bookmark, bookmark>(last->end_caret_pos(v),
                                          first->start_caret_pos(v));
        else
          return pair<bookmark, bookmark>();
      };

      auto insert_row = [&](action *group) -> pair<bookmark, bookmark> {
        helement first, last;
        helement row         = new element(tag::T_TR);
        int      ins_row_pos = 10000; //
        // table_row->node_index + (bm.at_element_end() ? 1 : 0);
        if (n < int(rows)) ins_row_pos = table_body->ui_child(n)->node_index;

        for (int c = int(cols) - 1; c >= 0; --c) {
          irange   cell_r, cell_c;
          helement cell;
          if (n < int(rows))
            cell = table_body->get_actual_cell_at(n, c, cell_r, cell_c);
          else {
            cell = table_body->get_actual_cell_at(rows - 1, c, cell_r, cell_c);
            cell_c.l = cell_c.h = c;
            cell_r.l = cell_r.h = n;
          }
          // if (cell_c.l < c) continue; // it starts before, skip it
          if (cell_r.l < n) // spanned cell, need to expand it
            increment_rowspan(group, cell, 1);
          else {
            helement newcell = new element(cell->tag);
            // insert_node::exec(v, this, group, row, cellnodepos, newcell);
            row->nodes.push(newcell.ptr());
            if (!first) first = newcell;
            last = newcell;
          }
        }
        insert_node::exec(v, this, group, table_body, ins_row_pos, row);
        if (last)
          return pair<bookmark, bookmark>(last->end_caret_pos(v),
                                          first->start_caret_pos(v));
        else
          return pair<bookmark, bookmark>();
      };

      if (transact_group) {
        return column ? insert_column(transact_group)
                      : insert_row(transact_group);
      }

      handle<range_action> group =
          new range_action(this, WCHARS("insert cells"));

      try {
        pair<bookmark, bookmark> bms =
            column ? insert_column(group) : insert_row(group);

        push(v, group);

        v.commit_update();

        return bms;
      } catch (const tool::exception &) { group->undo(v, this); }
      // otherwise it is just a prolongation of previous op
      return pair<bookmark, bookmark>();
    }

    bool richtext_ctl::merge_cells(view &v) {
      bookmark bm = caret;
      bookmark bma;
      if (!is_table_range_selection()) return false;

      delete_cells(v, selection_ctx::table, selection_ctx::table_rows,
                   selection_ctx::table_cols, true)
          .unpack(bma, bm);

      v.commit_update();

      if (bm.valid()) {
        select(v, bm, bma);
        return true;
      }
      return false;
    }

    bool richtext_ctl::delete_rows(view &v) {
      bookmark bm = caret;
      bookmark bma;

      block_table_body *table_body = nullptr;
      irange table_rows;
      irange table_cols;

      if (is_table_range_selection()) {
        table_body = selection_ctx::table;
        table_rows = selection_ctx::table_rows;
        uint rows, cols;
        table_body->get_rows_cols(rows, cols);
        table_cols = irange(0, int(cols) - 1);
      } else if (is_selection_in_table_cell()) {
        element *table_cell = caret.node->get_element()->nearest_table_cell();
        ASSERT(table_cell);
        table_body = table_cell->parent_table_body();
        ASSERT(table_body);
        uint r, c;
        table_body->get_cell_row_col(table_cell, r, c);
        uint rows, cols;
        table_body->get_rows_cols(rows, cols);
        table_rows.l = table_rows.h = int(r);
        table_cols = irange(0, int(cols) - 1);
      } else
        return false;

      delete_cells(v, table_body, table_rows, table_cols, false)
          .unpack(bma, bm);

      v.commit_update();

      if (bm.valid()) {
        select(v, bm, bma);
        return true;
      }
      return false;
    }

    bool richtext_ctl::delete_cols(view &v) {
      bookmark bm = caret;
      bookmark bma;

      block_table_body *table_body = nullptr;
      irange             table_rows;
      irange             table_cols;

      if (is_table_range_selection()) {
        table_body = selection_ctx::table;
        table_cols = selection_ctx::table_cols;
        uint rows, cols;
        table_body->get_rows_cols(rows, cols);
        table_rows = irange(0, int(rows) - 1);
      } else if (is_selection_in_table_cell()) {
        element *table_cell = caret.node->get_element()->nearest_table_cell();
        ASSERT(table_cell);
        table_body = table_cell->parent_table_body();
        ASSERT(table_body);
        uint r, c;
        table_body->get_cell_row_col(table_cell, r, c);
        uint rows, cols;
        table_body->get_rows_cols(rows, cols);
        table_cols.l = table_cols.h = int(c);
        table_rows = irange(0, int(rows) - 1);
      } else
        return false;

      delete_cells(v, table_body, table_rows, table_cols, false)
          .unpack(bma, bm);

      v.commit_update();

      if (bm.valid()) {
        select(v, bm, bma);
        return true;
      }
      return false;
    }

    bool richtext_ctl::insert_column(view &v, bool after) {
      bookmark bm = caret;
      bookmark bma;

      block_table_body *table_body = nullptr;
      irange table_rows;
      irange table_cols;

      if (is_table_range_selection()) {
        table_body = selection_ctx::table;
        table_cols = selection_ctx::table_cols;
        table_rows = selection_ctx::table_rows;
      } else if (is_selection_in_table_cell()) {
        element *table_cell = caret.node->get_element()->nearest_table_cell();
        ASSERT(table_cell);
        table_body = table_cell->parent_table_body();
        ASSERT(table_body);
        uint r, c;
        table_body->get_cell_row_col(table_cell, r, c);
        table_cols.l = table_cols.h = int(c);
        table_rows.l = table_rows.h = int(r);
      } else
        return false;

      int c;
      if (!after)
        c = table_cols.l;
      else
        c = table_cols.h + 1;

      insert_cells(v, table_body, c, true).unpack(bma, bm);

      v.commit_update();

      if (bm.valid()) {
        select(v, bm, bma);
        return true;
      }
      return false;
    }

    bool richtext_ctl::insert_row(view &v, bool after) {
      bookmark bm = caret;
      bookmark bma;

      block_table_body *table_body = nullptr;
      irange table_rows;
      irange table_cols;

      if (is_table_range_selection()) {
        table_body = selection_ctx::table;
        table_cols = selection_ctx::table_cols;
        table_rows = selection_ctx::table_rows;
      } else if (is_selection_in_table_cell()) {
        element *table_cell = caret.node->get_element()->nearest_table_cell();
        ASSERT(table_cell);
        table_body = table_cell->parent_table_body();
        ASSERT(table_body);
        uint r, c;
        table_body->get_cell_row_col(table_cell, r, c);
        table_cols.l = table_cols.h = int(c);
        table_rows.l = table_rows.h = int(r);
      } else
        return false;

      int r;
      if (!after)
        r = table_rows.l;
      else
        r = table_rows.h + 1;

      insert_cells(v, table_body, r, false).unpack(bma, bm);

      v.commit_update();

      if (bm.valid()) {
        select(v, bm, bma);
        return true;
      }
      return false;
    }

    pair<bookmark, bookmark>
    richtext_ctl::split_cells(view &v, block_table_body *table_body,
                              irange table_rows, irange table_cols) {
      uint rows, cols;
      if (!table_body->get_rows_cols(rows, cols))
        return pair<bookmark, bookmark>();

      auto find_node_position_for_cell = [&](int atrow, int atcol) -> int {
        helement row = table_body->ui_child(atrow);
        ASSERT(row);
        int cellnodepos = row->nodes.size();
        for (uint c = 0; c < cols; ++c) {
          if (int(c) == atcol) break;
          element *pc = table_body->get_cell_at(atrow, c);
          if (pc) cellnodepos = pc->node_index + 1;
        }
        return cellnodepos;
      };

      auto split_cell = [&](action *group, element *cell, irange cell_rows,
                            irange cell_cols) {
        for (int r = cell_rows.l; r <= cell_rows.h; ++r) {
          for (int c = cell_cols.l; c <= cell_cols.h; ++c) {
            if (r == cell_rows.l && c == cell_cols.l) continue;
            helement nc = new element(c > 0 && r > 0 ? tag::T_TD : cell->tag);
            int      cellnodepos = find_node_position_for_cell(r, c);
            helement row         = table_body->ui_child(r);
            ASSERT(row);
            insert_node::exec(v, this, group, row, cellnodepos, nc);
          }
        }
        change_attr::del(v, this, group, cell, "colspan");
        change_attr::del(v, this, group, cell, "rowspan");
      };

      auto do_split = [&](action *group) -> pair<bookmark, bookmark> {

        array<helement> cells;
        // scan cells for row/colspans
        for (int r = table_rows.h; r >= table_rows.l; --r) {
          for (int c = table_cols.h; c >= table_cols.l; --c) {
            irange   cell_r, cell_c;
            helement cell = table_body->get_actual_cell_at(r, c, cell_r, cell_c);
            ASSERT(cell);
            if (cell_r.l != r || cell_c.l != c)
              continue; // will be handled later

            if (cell_r.length() > 1 || cell_c.length() > 1) {
              split_cell(group, cell, cell_r, cell_c);
              table_body->drop_layout();
              table_body->check_layout(v);
            }

            cell = table_body->get_cell_at(r, c);
            if (cell) cells.insert(0, cell);
          }
        }
        return pair<bookmark, bookmark>();
      };

      if (transact_group) { return do_split(transact_group); }

      handle<range_action> group =
          new range_action(this, WCHARS("split cells"));

      try {
        pair<bookmark, bookmark> bms = do_split(group);

        push(v, group);

        v.commit_update();

        return bms;
      } catch (const tool::exception &) { group->undo(v, this); }
      // otherwise it is just a prolongation of previous op
      return pair<bookmark, bookmark>();
    }

    bool richtext_ctl::split_cells(view &v) {
      bookmark bm = caret;
      bookmark bma;

      block_table_body *table_body = nullptr;
      irange             table_rows;
      irange             table_cols;

      if (is_table_range_selection()) {
        table_body = selection_ctx::table;
        table_cols = selection_ctx::table_cols;
        table_rows = selection_ctx::table_rows;
      } else if (is_selection_in_table_cell()) {
        element *table_cell = caret.node->get_element()->nearest_table_cell();
        ASSERT(table_cell);
        table_body = table_cell->parent_table_body();
        ASSERT(table_body);
        uint r, c;
        table_body->get_cell_row_col(table_cell, r, c);
        table_cols.l = table_cols.h = int(c);
        table_rows.l = table_rows.h = int(r);
      } else
        return false;

      split_cells(v, table_body, table_rows, table_cols).unpack(bma, bm);

      v.commit_update();

      if (bm.valid()) {
        select(v, bm, bma);
        return true;
      }
      return false;
    }

    bool richtext_ctl::delete_forward(view &v) {
      bookmark bm = caret;
      bookmark bma;
      if (is_table_selection())
        delete_cells(v, selection_ctx::table, selection_ctx::table_rows,selection_ctx::table_cols, false).unpack(bma, bm);
      //else if(is_block_selection())
      //  bm = delete_range(v, anchor, caret, true);
      else if (caret != anchor)
        bm = delete_range(v, anchor, caret, true);
      else
        bm = delete_char(v, caret, true);

      //if(!check_empty(v, bm))
      //  rq_spell_check(v);

      v.commit_update();

      if (!bm.valid()) return false;

      if (!bma.valid() && !bm.at_char_pos(v))
        root()->advance(v, bm, ADVANCE_NEXT_CHAR);

      if (bm.valid()) {
        select(v, bm, bma);
        return true;
      }
      return false;
    }

    bool richtext_ctl::delete_backward(view &v) {
      bookmark bm = caret;
      if (caret != anchor)
        bm = delete_range(v, anchor, caret,false);
      else
        bm = delete_char(v, caret, false);

      //if (!check_empty(v, bm))
      //  rq_spell_check(v);

      v.commit_update();

      if (!bm.valid()) return false;

      if (!bm.at_char_pos(v))
        root()->advance(v, bm, ADVANCE_PREV_CHAR);

      if (bm.valid()) {
        select(v, bm);
        return true;
      }
      return false;
    }

    bool richtext_ctl::can_unindent(view &v) {
      // return selection_contains(v,unindents_selector());
      return can_unindent(v, anchor, caret);
    }
    bool richtext_ctl::can_indent(view &v) {
      // return selection_contains(v,WCHARS("blockquote,li,dd,dt");
      // return true;
      return can_indent(v, anchor, caret);
    }
    bool richtext_ctl::do_unindent(view &v) {
      return unindent(v, anchor, caret);
    }
    bool richtext_ctl::do_indent(view &v) { return indent(v, anchor, caret); }

    bool richtext_ctl::attach(view &v, element *self) {
      self->state.content_editable(true);

      bool r = super::attach(v, self);

      helement that = self;

      /*v.async([that, &v]() -> bool { -- breaks htmlarea.value = "<p></p>"; case
        if (!that->first_node()) {
          element *pe = new element(tag::T_TEXT);
          pe->state.synthetic(true);
          pe->append(new text(WCHARS("")));
          that->append(pe, &v);
        }
        return true;
      });*/

      if (self->is_layout_valid()) v.add_to_update(self, CHANGES_MODEL);
      self->reset_styles(v);

      document *d = self->doc();

      if (!d) return false;

      string src = self->atts.get_url(d->uri().src, attr::a_src);
      if (d && src.length()) {
        handle<pump::request> rq = new pump::request(src, DATA_HTML);
        _awaiting                = rq;
        rq->dst                  = self;
        v.request_data(rq);
      }
      else
      {
        //const byte *html = (const byte *)"<html><body></body></html>";
        //load(v, self, "", html, 13);
        bookmark bm;
        check_empty(v, bm);
        select(v, bm);
      }

      rq_spell_check(v);

      return r;
    }

    bool richtext_ctl::on_data_request(view &v, element *self,
                                       pump::request *rq) {
#if 0
      if (rq->data_type == DATA_IMAGE && rq->url == pasted_image_url && pasted_image_data.length()) {
        rq->data_content_type = pasted_image_mime_type;
        rq->data.transfer_from(pasted_image_data);
        rq->done_with_success(200);
      }
#endif
      if (!this->self) return false;
      if (rq->data_type == DATA_HTML && rq->dst == self) _awaiting = rq;
      v.notify_data_request(self, rq);
      return false;
    }

    bool richtext_ctl::on_data_arrived(view &v, element *self,
                                       pump::request *rq) {
      //#ifdef _DEBUG
      //    dbg_printf("on_data_arrived %s\n", rq->url.c_str());
      //#endif
      if (!this->self) return false;
      v.notify_data_arrived(self, rq);

      if (rq->data_type == DATA_HTML && _awaiting == rq && !rq->consumed) {
        load_html(v, self, rq->real_url(), rq->data(),rq->data_content_encoding);
        rq->consumed = true;
        _awaiting = nullptr;
      }
      return false;
    }

    bool richtext_ctl::load_html(view &v, element *self, const string &url,
                                 bytes html, const string &encoding) {
      bool r             = false;
      auto state_changer = [&](view &v, element *pel) -> bool {
        r = this->_load_html(v, pel, url, html, encoding);
        return r;
      };
      self->animated_update(v, state_changer);
      return r;
    }

    bool richtext_ctl::_load_html(view &v, element *self, const string &url,
                                  bytes html, const string &encoding) {
      bool focus_was_here = v.get_focus_element() &&
                            v.get_focus_element()->belongs_to(self, true);

      _awaiting = 0;

      select(v, bookmark());
      self->clear(&v);

      //if (html.length == 0) return true; check_cannonic_document_structure will handle it

      handle<document> d       = new document(url);
      d->allow_own_scripts     = false;
      d->allow_own_styles      = false;
      d->flags.is_synthetic    = true;
      d->_media_vars_provider  = this;

      ustring dir = self->atts(attr::a_dir);
      if (dir.length()) d->atts.set(attr::a_dir, dir);

      self->append(d, &v);

      string cs = get_attr(self, "-content-style");
      if (cs.length()) {
        cs = combine_url(self->doc()->uri().src, cs);

        string id("Z"); // simulate highest priority //
                        // (char(d->style_serial_no), 1); ++d->style_serial_no;

        handle<pump::request> hrq = new pump::request(cs, DATA_STYLE);
        hrq->rq_id = id;
        hrq->dst   = d.ptr();

        if (v.load_data(hrq, true)) {
          ustring      wdata = u8::cvt(hrq->data());
          style_parser stp(id, wdata, d, cs);
          stp.parse(wchars());
          // d->drop_layout_tree(&v);
          // d->resolve_styles(v);
        }
      }

      v.on_load_start(d); // this has nothing to do with xview

      istream m(html, url, v.debug_mode());
      if (encoding.is_defined())
        m.set_encoding(encoding);
      else
        m.set_utf8_encoding();

      bookmark sel_start, sel_end;

      parse_html(v, m, d, &sel_start, &sel_end);

      //if (d->is_empty()) return false; - check_cannonic_document_structure will handle it

      check_cannonic_document_structure(v, self);

      d->setup_layout(v);

      self->get_style(v);

      _body = find_first(v, d, WCHARS("body"));

      v.on_load_end(d, false); // we do need to load scripts
      d->operational = true;
      self->check_layout(v);
      size sz = self->dim();
      if (!sz.empty()) {
        d->measure(v, sz);
        v.refresh(self);
      }
      if (d->num_resources_requested == 0) v.on_document_complete(d);

      _body = find_first(v, d, WCHARS("body"));

      bookmark bm;
      check_empty(v,bm);

      if (sel_start.valid() && sel_end.valid())
        this->select(v, sel_start, sel_end);
      else if (sel_start.valid())
        this->select(v, sel_start);
      else if (sel_end.valid())
        this->select(v, sel_end);
      else
        this->select(v, bm);

      if (focus_was_here) v.set_focus(d, BY_CODE);

      rq_spell_check(v);

      return true;
    }

    bool richtext_ctl::set_html(view &v, element *self, bytes html) {

      bool focus_was_here = v.get_focus_element() &&
                            v.get_focus_element()->belongs_to(self, true);

      clear_comp_chars(v);
      select(v, bookmark());
      self->clear(&v);

      self->allow_reconciliation(false); // explicitly disable SSX/JSX reconciliation

      drop_undo_stack();

      if (html.length) {
        helement el = self;
        v.set_element_html(el, html, html::SE_REPLACE);
        rq_spell_check(v);
      }
      set_modified(v, false);

      bookmark bm;
      check_empty(v,bm);
      select(v, bm);

      if (focus_was_here) v.set_focus(self, BY_CODE);

      return true;
    }

    bool richtext_ctl::set_value(view &v, element *self, const value &val) {
      ustring us  = val.to_string();
      string  utf = u8::cvt(us);
      return set_html(v, self, utf.chars_as_bytes());
    }
    bool richtext_ctl::get_value(view &v, element *self, value &val) {
      html::ostream_w os;
      self->emit_content(os);
      val = value(os.data());
      return true;
    }

    bool richtext_ctl::set_text(view &v, element *self, wchars chars) {

      bool focus_was_here = v.get_focus_element() &&
        v.get_focus_element()->belongs_to(self, true);

      clear_comp_chars(v);
      select(v, bookmark());
      self->clear(&v);

      drop_undo_stack();

      array<wchar> line_buffer;

      bookmark anchor, caret;

      //const wchar *text_start = chars.start;
      wchars line; chopline(chars, line);
      do {
        // line_buffer.clear();
        // expand_tabs(line_buffer,line,st);
        text *   t = new text(line);
        element *te = new element(tag::T_TEXT);
        te->append(t);
        self->append(te);
      } while (chopline(chars, line));

      if (self->nodes.is_empty()) {
        text *   t = new text(WCHARS(""));
        element *te = new element(tag::T_TEXT);
        te->append(t);
        self->append(te);
      }
      v.add_to_update(self, CHANGES_MODEL);
      set_modified(v, false);

      bookmark bm;
      check_empty(v, bm);

      select(v, bm);

      if (focus_was_here) v.set_focus(self, BY_CODE);

      return true;

    }
    bool richtext_ctl::get_text(view &v, element *self, ustring &val) {
      array<wchar> to;
      for (int n = 0; n < self->nodes.size(); ++n)
        self->nodes[n]->get_ui_text(v,to);
      val = to();
      return true;
    }

    bool richtext_ctl::on(view &v, element *self, event_behavior &evt) {
      if (!this->self) return false;
      if (evt.cmd == INPUT_LANGUAGE_CHANGED) {
        handle<spell_checker> sc = app()->get_spell_checker(v.get_input_lang());
        if (sc) {
          _spell_checker = sc;
          spell_check(v);
        }
      } else if (evt.cmd == CONTEXT_MENU_SETUP) {
        handle<element> menu = evt.source;
        if (evt.data.get_prop("x").is_undefined()) {
          // seems like context menu is invoked by kbd
          rect loc;
          if (get_caret_place(v, self, loc)) {
            loc += self->view_pos(v);
            evt.data.set_prop("x", value(loc.e.x));
            evt.data.set_prop("y", value(loc.e.y));
          }
        }
        if (_spell_checker) {
          // bool get_marks_span(bookmark& start, bookmark& end, chars mark)
          bookmark start = caret;
          bookmark end;
          ustring  misspelled_word;
          if (get_marks_span(start, end, misspelled_word, CHARS("misspell"))) {
            this->select(v, end, start);
            auto list = _spell_checker->get_suggestions_for(misspelled_word());
            if (list.size()) {
              menu->set_attr(v, "has-misspells", ustring());
              handle<element> container = find_first(v, menu, WCHARS(".suggestions"));
              if (!container) {
                container = menu;
                container->insert(0, new element(tag::T_HR));
              }
              else
                container->clear();

              for (int n = list.last_index(); n >= 0; --n) {
                handle<element> repl = new element(tag::T_LI);
                repl->append(new text(list[n]()));
                repl->atts.set(attr::a_class, WCHARS("replacement"));
                container->insert(0, repl);
                // dbg_printf("repl: %S\n", list[i].c_str());
              }
            }
          }
        } else {
          evt.source->remove_attr(v, "has-misspells");
        }

        element *table_ops = find_first(v, menu, WCHARS("li.table-operations"));
        if (table_ops) {
          if (is_editable(self) &&
              (is_selection_in_table_cell() || is_table_selection()))
            table_ops->state_off(v, S_DISABLED);
          else
            table_ops->state_on(v, S_DISABLED);
        }

      } else if (evt.cmd == MENU_ITEM_CLICK && evt.target->attr_class() == WCHARS("replacement") && this->can_paste(v))
      {
        // this->paste((v, evt.target->first_node()->cast<text>()->chars());
        this->insert_chars(v, caret, anchor,
                           evt.target->first_node()->cast<text>()->chars());
        rq_spell_check(v);
        return true;
      }
      else if (evt.cmd_no_flags() == EDIT_VALUE_CHANGED) {
        rq_spell_check(v);
      }


      return super::on(v, self, evt);
    }

    bool richtext_ctl::check_empty(view &v, bookmark& bm) {
      helement root = this->root();
      self->state.content_editable(true);
      if (!root->first_node()) {
        assert(root->is_empty());
        handle<element> t = new element(tag::T_TEXT);
        t->state.content_editable(true);
        t->state.synthetic(true);
        root->append(t, &v);
        root->check_layout(v);
        t->check_layout(v);
        bm = t->start_caret_pos(v);
        return true;
      }
      else if(!bm.valid()) {
        root->advance(v, bm, ADVANCE_FIRST);
      }
      return false;
    }

    value richtext_ctl::api_load(value urlOrBytes, value url)
    {
      view* pv = self->pview();
      if (!pv) return value(false);

      if (urlOrBytes.is_string() && url.is_undefined()) {
        string                in = urlOrBytes.get<ustring>();
        string                out = combine_url(self->doc()->uri().src, in);
        handle<pump::request> rq = new pump::request(out, DATA_HTML);
        rq->dst = self;
        pv->request_data(rq);
        set_modified(*pv, false);
        return value(true);
      }
      if(urlOrBytes.is_string() && url.is_string()) {
        string  in_html = u8::cvt(urlOrBytes.get<ustring>(),true);
        ustring in_url = url.get<ustring>();
        string url = combine_url(self->doc()->uri().src, url::escape(in_url));
        bytes  html = in_html.chars_as_bytes();
        bool   r = load_html(*pv, self, url, html);
        set_modified(*pv, false);
        return value(r);
      }
      if (urlOrBytes.is_bytes() && url.is_string()) {
        tool::bytes html = urlOrBytes.get_bytes();
        ustring     in_url = url.get<ustring>();
        string url = combine_url(self->doc()->uri().src, url::escape(in_url));
        bool   r = load_html(*pv, self, url, html);
        set_modified(*pv, false);
        return value(r);
      }
      return value::make_error(W("html,url expected"));
    }

    bool richtext_ctl::api_loadEmpty() {
      view* pv = self->pview();
      if (!pv) return false;
      chars html = CHARS("<html><body><p></p></body></html>");
      load_html(*pv, self, "about:blank", to_bytes(html));
      set_modified(*pv, false);
      move_caret(*pv, ADVANCE_HOME, false);
      return true;
    }
    ustring richtext_ctl::api_get_url()
    {
      helement pel = self->first_element();
      if (pel && pel->is_document())
        return pel.ptr_of<document>()->uri().src;
      else
        return ustring();
    }
    bool    richtext_ctl::api_set_url(ustring url)
    {
      string   in = url;
      helement pel = self->first_element();
      if (pel && pel->is_document()) {
        pel.ptr_of<document>()->uri(tool::url(in));
        return true;
      }
      else
        return false;
    }

    value   richtext_ctl::api_contentToSource()
    {
      int_v           apos, cpos;
      html::emit_ctx  ectx;
      html::ostream_w os;

      ectx.root = self;
      ectx.anchor = this->anchor;
      ectx.caret = this->caret;
      ectx.at_anchor = [&apos, &os]() { apos = int(os.position()); };
      ectx.at_caret = [&cpos, &os]() { cpos = int(os.position()); };

      element *container = self;
      element *first_child = self->first_element();
      assert(first_child->is_document());
      if (first_child && first_child->flags.is_synthetic)
        container = self->first_element();

      container->emit_content(os, &ectx);

      value arr[] = {
        value(os.data()),
        value(api_get_url()),
        apos.is_defined() ? value(apos.val(0)) : value(),
        cpos.is_defined() ? value(cpos.val(0)) : value(),
      };

      return value::make_array(items_of(arr));
    }

    value   richtext_ctl::api_sourceToContent(ustring source, ustring toUrl, value selStart, value selEnd)
    {
      view* pv = self->pview();
      if (!pv) return value(false);

      //if (!argv[0].is_string()) { retval = value::make_error(W("bad source")); return true; }
      //if (!argv[1].is_string()) { retval = value::make_error(W("bad document url")); return true; }

      int pos1 = selStart.get(-1);
      int pos2 = selEnd.get(-1);
      if (pos1 > pos2)
        swap(pos1, pos2);

      ustring in = source;

      if (pos1 >= 0 && pos2 >= 0 && in.length()) {
        in.insert(tool::markup::CHARMARK_SELECTION_END, pos2);
        in.insert(tool::markup::CHARMARK_SELECTION_START, pos1);
      }

      //#ifdef _DEBUG
      //        v.dbg_print_stack_trace();
      //#endif

      string  url = combine_url(self->doc()->uri().src, url::escape(toUrl));
      bytes   html = in.chars_as_bytes();

      _body = nullptr;

      /*bool r = self->first_element()
        ? merge_html(*pv, self, url, html, CHARS("utf-16"))
        : load_html(*pv, self, url, html, CHARS("utf-16"));*/

      bool r = load_html(*pv, self, url, html, CHARS("utf-16"));

      return value(r);
    }

    value   richtext_ctl::api_contentToBytes()
    {
      int_v           apos, cpos;
      html::ostream_8 os;

      element *container = self;
      element *first_child = self->first_element();
      assert(first_child->is_document());
      if (first_child && first_child->flags.is_synthetic)
        container = self->first_element();

      container->emit_content(os);

      return value::make_bytes(os.data());
    }
    value   richtext_ctl::api_bytesToContent(value psrc, value purl)
    {
      if (!psrc.is_bytes()) return value::make_error(W("bad source"));
      if (!purl.is_string()) return value::make_error(W("bad document url"));

      view* pv = self->pview();
      if (!pv) return value(false);

      ustring in_url = purl.get<ustring>();
      string  url = combine_url(self->doc()->uri().src, url::escape(in_url));
      bytes   html = psrc.get_bytes();

      //bool r = self->first_element()
      //  ? merge_html(*pv, self, url, html, CHARS("utf-8"))
      //  : load_html(*pv, self, url, html, CHARS("utf-8"));
      //return value(r);

      load_html(*pv, self, url, html, CHARS("utf-8"));

      return value(true);

    }
    value    richtext_ctl::api_save(ustring fileurl)
    {
      ustring path = url::file_url_to_path(fileurl);

      view* pv = self->pview();
      if (!pv) return value(false);

      if (!pv->can_use_feature(FEATURE_FILE_IO))
        return value::make_error(W("file I/O is disabled"));

      html::ostream_8 os;
      os.data.push(BYTES(UTF8_BOM));
      self->emit_content(os);

      FILE *fp = 0;
#ifdef WINDOWS
      _wfopen_s(&fp, path.c_str(), L"wb");
#else
      fp = fopen(u8::cvt(path), "wb");
#endif
      if (!fp)
        return value(false);
      else {
        size_t written = fwrite(os.data.head(), os.data.length(), 1, fp);
        fclose(fp);
        if (written) {
          this->set_modified(*pv, false);
          return value(true);
        }
        else
          return value(false);
      }
    }

    value richtext_ctl::api_update(value updater, ustring name) {
      if (!updater.is_proxy_of_function())
        return value();

      auto pf = self->flags.suppress_change_event;
      self->flags.suppress_change_event = 1;

      sciter::om::hasset<behavior::transact_ctx> tctx = new transact_ctx(self, name);
      value args[1] = {value::wrap_asset(tctx)};
      bool r = updater.call(items_of(args)).get(false);
      if (r)
        tctx->commit();
      else
        tctx->rollback();

      self->flags.suppress_change_event = 0;

      return value(r);
    }

#ifndef SCITERJS
    bool richtext_ctl::on_x_method_call(view &v, element *self,
                                        const char *name, const value *argv,
                                        size_t argc, value &retval) {
      chars fname = chars_of(name);
#define METHOD(ARGC, NAME) if (argc == ARGC && fname == CHARS(#NAME))
      METHOD(1, loadDocument) {
        if (argv[0].is_string()) {
          string                in  = argv[0].get(W(""));
          string                out = combine_url(self->doc()->uri().src, in);
          handle<pump::request> rq  = new pump::request(out, DATA_HTML);
          rq->dst                   = self;
          v.request_data(rq);
          set_modified(v, false);
          retval = value(true);
        } else if (argv[0].is_bytes()) {
          tool::bytes html = argv[0].get_bytes();
          string      url  = self->doc()->uri().src;
          bool        r    = load_html(v, self, url, html);
          retval           = value(r);
          set_modified(v, false);
        } else
          retval = value::make_error(W("either string(url) or bytes"));
        return true;
      }
      METHOD(2, loadDocument) {
        if (argv[0].is_string() && argv[1].is_string()) {
          string  in_html = u8::cvt(wchars(argv[0].get(W(""))));
          ustring in_url  = argv[1].get(W(""));
          string url = combine_url(self->doc()->uri().src, url::escape(in_url));
          bytes  html = in_html.chars_as_bytes();
          bool   r    = load_html(v, self, url, html);
          retval      = value(r);
          set_modified(v, false);
        } else if (argv[0].is_bytes() && argv[1].is_string()) {
          tool::bytes html   = argv[0].get_bytes();
          ustring     in_url = argv[1].get(W(""));
          string url = combine_url(self->doc()->uri().src, url::escape(in_url));
          bool   r   = load_html(v, self, url, html);
          retval     = value(r);
          set_modified(v, false);
        } else
          retval = value::make_error(W("html,url expected"));
        return true;
      }
      METHOD(0, emptyDocument) {
        chars html = CHARS("<html><body><p></p></body></html>");
        load_html(v, self, "about:blank", to_bytes(html));
        set_modified(v, false);
        move_caret(v, ADVANCE_HOME, false);
        return true;
      }
      METHOD(1, setDocumentUrl) {
        if (argv[0].is_string()) {
          string   in  = argv[0].get(W(""));
          helement pel = self->first_element();
          if (pel && pel->is_document()) {
            pel.ptr_of<document>()->uri(url(in));
            retval = value(true);
          } else
            retval = value(false);
        }
        return true;
      }

      METHOD(1, setBaseUrl) {
        if (argv[0].is_string()) {
          string   in = argv[0].get(W(""));
          helement pel = self->first_element();
          if (pel && pel->is_document()) {
            pel.ptr_of<document>()->uri(url(in));
            retval = value(true);
          }
          else
            retval = value(false);
        }
        return true;
      }


      METHOD(0, getDocumentUrl) {
        helement pel = self->first_element();
        if (pel && pel->is_document())
          retval = ustring(pel.ptr_of<document>()->uri().src);
        else
          retval = value();
        return true;
      }

      METHOD(0, contentToSource) {
        int_v           apos, cpos;
        html::emit_ctx  ectx;
        html::ostream_w os;

        ectx.root      = self;
        ectx.anchor    = this->anchor;
        ectx.caret     = this->caret;
        ectx.at_anchor = [&apos, &os]() { apos = int(os.position()); };
        ectx.at_caret  = [&cpos, &os]() { cpos = int(os.position()); };

        element *container   = self;
        element *first_child = self->first_element();
        assert(first_child->is_document());
        if (first_child && first_child->flags.is_synthetic)
          container = self->first_element();

        container->emit_content(os, &ectx);

        value arr[3];
        arr[0] = value(os.data());
        arr[1] = apos.is_defined() ? value(apos.val(0)) : value();
        arr[2] = cpos.is_defined() ? value(cpos.val(0)) : value();

        retval = value::make_array(items_of(arr));

        return true;
      }

      METHOD(0, contentToBytes) {
        int_v           apos, cpos;
        html::ostream_8 os;

        element *container = self;
        element *first_child = self->first_element();
        assert(first_child->is_document());
        if (first_child && first_child->flags.is_synthetic)
          container = self->first_element();

        container->emit_content(os);

        retval = value::make_bytes(os.data());

        return true;
      }

      METHOD(4, sourceToContent) {
        if (!argv[0].is_string()) { retval = value::make_error(W("bad source")); return true; }
        if (!argv[1].is_string()) { retval = value::make_error(W("bad document url")); return true; }

        int pos1 = argv[2].get(-1);
        int pos2 = argv[3].get(-1);
        if (pos1 > pos2)
          swap(pos1, pos2);

        ustring in = argv[0].get_string();

        if (pos1 >= 0 && pos2 >= 0 && in.length()) {
          in.insert(tool::markup::CHARMARK_SELECTION_END, pos2);
          in.insert(tool::markup::CHARMARK_SELECTION_START, pos1);
        }

//#ifdef _DEBUG
//        v.dbg_print_stack_trace();
//#endif

        ustring in_url = argv[1].get(W(""));
        string  url  = combine_url(self->doc()->uri().src, url::escape(in_url));
        bytes   html = in.chars_as_bytes();

        _body = nullptr;

        bool r = self->first_element()
                     ? merge_html(v, self, url, html, CHARS("utf-16"))
                     : load_html(v, self, url, html, CHARS("utf-16"));

        retval = value(r);
        return true;
      }

      METHOD(2, bytesToContent) {
        if (!argv[0].is_bytes()) { retval = value::make_error(W("bad source")); return true; }
        if (!argv[1].is_string()) { retval = value::make_error(W("bad document url")); return true; }

        ustring in_url = argv[1].get(W(""));
        string  url = combine_url(self->doc()->uri().src, url::escape(in_url));
        bytes   html = argv[0].get_bytes();

        bool r = self->first_element()
          ? merge_html(v, self, url, html, CHARS("utf-8"))
          : load_html(v, self, url, html, CHARS("utf-8"));
        retval = value(r);
        return true;
      }


      METHOD(1, saveDocument) {
        if (!argv[0].is_string()) {
          retval = value::make_error(W("path is not a string"));
          return true;
        }

        ustring path = url::file_url_to_path(argv[0].get(W("")));

        html::ostream_8 os;
        os.data.push(BYTES(UTF8_BOM));
        self->emit_content(os);

        FILE *fp = 0;
#ifdef WINDOWS
        _wfopen_s(&fp, path.c_str(), L"wb");
#else
        fp = fopen(u8::cvt(path), "wb");
#endif
        if (!fp) {
          retval = value(false);
          return true;
        } else {
          size_t written = fwrite(os.data.head(), os.data.length(), 1, fp);
          fclose(fp);
          if (written) {
            retval = value(true);
            this->set_modified(v, false);
          } else
            retval = value(false);
          return true;
        }
      }

      METHOD(0, saveDocument) {
        html::ostream_8 os;
        os.data.push(BYTES(UTF8_BOM));
        self->emit_content(os);
        retval = value::make_bytes(os.data);
        return true;
      }

      METHOD(0, undoRedoPoint) {
        if (this->stack.length())
          retval = value((int)(uint_ptr)this->richtext_ctl::top());
        else
          retval = value::null_val();
        return true;
      }

      METHOD(0, mediaVars) {
        tool::value map = tool::value::make_map();
        for (int n = 0; n < _media_vars.size(); ++n) {
          tool::value k = tool::value(_media_vars.key(n));
          map.set_prop(k, _media_vars.value(n));
        }
        retval = map;
        return true;
      }
      METHOD(1, mediaVars) {
        if (argv[0].is_map() || argv[0].is_proxy_of_object())
          set_media_vars(v, self, argv[0], false, true);
        return true;
      }
      METHOD(2, mediaVars) {
        bool reset = argv[1].to_bool();
        if (argv[0].is_map() || argv[0].is_proxy_of_object())
          set_media_vars(v, self, argv[0], reset, true);
        return true;
      }
#undef METHOD
      return super::on_x_method_call(v, self, name, argv, argc, retval);
    }
#endif
    void richtext_ctl::update_media_vars(
        view &v, element *self,
        bool reset_styles) // accepts comma separated list of vars
    {
      // loop for all <html> nodes (e.g. in frames) including the root of the
      // frame itself.
      /*element* doc = root(v,self);
      element_iterator bi(v,doc,is_document_filter);
      element* b = doc;
      do
      {
        document *pd = static_cast<document*>(b);
        for( int n = 0; n < pd->media_expressions.size(); ++n )
        {
          handle< eval::conduit > code = pd->media_expressions.elements()[n];
          v.eval_media_query(pd,code,code->is_true);
        }
      } while( bi(b) );

      if( reset_styles )
        v.add_to_update(doc,true);*/
    }

    void richtext_ctl::set_media_vars(
        view &v, element *self, value map, bool reset,
        bool reapply_styles) // accepts comma separated list of vars
    {
      if (reset) _media_vars.clear();

      assert(map.is_map() || map.is_proxy_of_object());

      auto each_name_value = [&](const value &k, const value &v) -> bool {
        ustring us      = k.to_string();
        _media_vars[us] = v;
        return true; // true to continue enumeration, false - to stop
      };
      map.visit(each_name_value);
      update_media_vars(v, self, reapply_styles);
    }

    bool richtext_ctl::can_spell_check(view &v) {
      if (!self) return false;
      return self->atts.get_bool(attr::a_spellcheck, true);
    }

    bool richtext_ctl::rq_spell_check(view &v) {

      if (can_spell_check(v))
        v.start_timer(self, SPELL_CHECK_TIMER_MS, SPELL_CHECK_TIMER_ID,
                      INTERNAL_TIMER);
      return true;
    }

    bool richtext_ctl::spell_check(view &v) {
      if (!can_spell_check(v)) return false;
      if (!_spell_checker) {
        ustring l = self->attr_lang();
        if( l.is_defined() )
          _spell_checker = app()->get_spell_checker(l);
        else
          _spell_checker = app()->get_spell_checker(v.get_input_lang());
        if (!_spell_checker) return false;
      }
      self->check_spelling(v, _spell_checker);
      return false;
    }

  } // namespace behavior
} // namespace html
