#include "html.h"
#include "html-actions-stack.h"

namespace html {

  namespace behavior {

    node *split_at(view &v, editing_ctx *ectx, action *ra, bookmark &bm,
      handle<element> until, bool after, bool &created,
      bookmark &other, bool force_creation = false);

    element *get_inlines_container(element *pel, element *until);

    bookmark break_position(view &v, element *root, bookmark bm);

    helement split_element_at(view &v, editing_ctx *ectx, action *ra, bookmark &bm, handle<element> el, bool after, bool &created, bookmark &other);
    
    struct plaintext_ctl_factory : public ctl_factory {
      plaintext_ctl_factory() : ctl_factory("plaintext") {}
      virtual ctl *create(element *el);
    };

    static plaintext_ctl_factory *_plaintext_ctl = 0;
    ctl *                         plaintext_ctl_factory::create(element *el) {
      return new plaintext_ctl();
    }

    const string &plaintext_ctl::behavior_name() const {
      return _plaintext_ctl->name;
    }

    void init_plaintext() {
      ctl_factory::add(_plaintext_ctl = new plaintext_ctl_factory());
    }

    bool plaintext_ctl::is_empty() const { // empty for the purpose of initialization
      for (int n = 0; n < self->nodes.size(); ++n) {
        node *pn = self->nodes[n];
        if (pn->is_element()) {
          if (!pn->cast<element>()->is_empty()) return false;
        } else
          if (!pn->is_space()) return false;
      }
      return true;
    }

    bool plaintext_ctl::is_empty(const element *self, bool &yes) // empty in CSS :empty sence 
    {
      yes = false;
      each_child texts(self);
      int nlines = 0;
      for (element *el; texts(el);) {
        if (el->nodes.size() == 0) continue;    // ???
        if (!el->nodes[0]->is_text()) return true;
        if (el->nodes[0].ptr_of<text>()->chars().length) return true;
        if (++nlines > 1) return true;
      }
      yes = true;
      return true;
    }

    bool plaintext_ctl::attach(view &v, element *self) {
      this->self = self;
      self->state.content_editable(true);

      ustring val = self->atts.get_ustring(attr::a_value, W(""));

      if (val.length()) {
        set_text(v, self, val);
      }
      else if (is_empty()) {
        self->clear();
        text *   t  = new text(WCHARS(""));
        element *te = v.get_anonymous_para();
        te->append(t);
        self->append(te);
      }

      if (self->is_layout_valid()) v.add_to_update(self, CHANGES_MODEL);
      self->reset_styles(v);
      return htmlarea_ctl::attach(v, self);
    }

    // int  plaintext_ctl::spaces_per_tab() {
    //   return this->get_attr(self,"-spaces-per-tab",4);
    //}

    bool plaintext_ctl::set_text(view &v, element *self, wchars chars,
                                 int_v apos, int_v cpos) 
    {
      self->allow_reconciliation(false); // disable reconsiliation - text is set explicitly 
      // int st = spaces_per_tab();
      array<wchar> line_buffer;

      bookmark anchor, caret;

      const wchar *text_start = chars.start;

      bool crlf_seen = false;
      wchars line; chopline(chars, line,crlf_seen);

      do {
        text *   t  = new text(line);
        element *te = new element(tag::T_TEXT);
        te->append(t);
        self->append(te);

        if (apos.is_defined() && cpos.is_defined()) {
          int start = int(line.start - text_start);
          int end   = start + line.size();
          if (apos >= start && apos <= end) {
            anchor.node     = t;
            anchor.pos      = limit(apos - start, 0 ,line.size() - 1);
            anchor.after_it = apos == end;
          }
          if (cpos >= start && cpos <= end) {
            caret.node     = t;
            caret.pos      = limit(cpos - start, 0 ,line.size() - 1);
            caret.after_it = cpos == end;
          }
        }
      } while (chopline(chars, line, crlf_seen));
      if (self->nodes.is_empty()) {
        text *   t  = new text(wchars());
        element *te = new element(tag::T_TEXT);
        te->append(t);
        self->append(te);
      }
      else if (crlf_seen) {
        text *   t = new text(wchars());
        element *te = new element(tag::T_TEXT);
        te->append(t);
        self->append(te);
      }
      v.add_to_update(self, CHANGES_MODEL);

      if (!anchor.valid() || !caret.valid())
        caret = anchor = self->first_content_caret_pos(v);

      if (anchor.valid() && caret.valid())
        select(v, caret, anchor);


      stack.clear();
      depth = 0;

      return true;
    }

    bool plaintext_ctl::set_value(view &v, element *self, const value &val) {
      self->clear();
      ustring sval;
      if(!val.is_null() && val.is_defined())
        sval = val.get(W(""));
      wchars  chars = sval;
      return set_text(v, self, chars);
    }

    bool plaintext_ctl::get_text(view &v, element *self, array<wchar> &out) {
      each_child texts(self);
      for (element *el; texts(el);) {
        out.push(el->get_text(v));
        out.push(W("\r\n"), 2);
      }
      if(out.length())
      out.drop(2); // last \r\n
      return true;
    }

    bool plaintext_ctl::get_text(view &v, element *self, array<wchar> &out,
                                 int_v &apos, int_v &cpos) {
      each_child texts(self);
      for (element *el; texts(el);) {
        if (el->nodes.size() == 0) continue;    // ???
        if (!el->nodes[0]->is_text()) continue; // ???
        if (caret.node == el->nodes[0]) cpos = out.size() + caret.linear_pos();
        if (anchor.node == el->nodes[0]) apos = out.size() + anchor.linear_pos();
        out.push(el->nodes[0].ptr_of<text>()->chars());
        out.push(W("\r\n"), 2);
      }
      out.drop(2); // last \r\n
      return true;
    }

    bool plaintext_ctl::get_value(view &v, element *self, value &val) {
      array<wchar> out;
      if (get_text(v, self, out)) {
        val = value(out());
        return true;
      }
      return true;
    }

    bool plaintext_ctl::can_paste(view &v) {
      if (!caret.valid() || !is_editable(self)) return false;
      uint fmt = clipboard::cf_text;
      return 0 != (clipboard::available_formats() & fmt);
    }

    bool plaintext_ctl::paste(view &v) {
      ustring data;
      if (!clipboard::get(data)) return false;

      // array<wchar> buffer;
      // expand_tabs(buffer,data(),spaces_per_tab());

      return this->insert_chars(v, anchor, caret, data());
    }

    bool plaintext_ctl::copy(view &v) {
      if (~anchor == ~caret) return false;
      bookmark start = anchor;
      bookmark end   = caret;

      if (start > end) swap(start, end);

      ostream_w os;
      emit_range_text(v, os, start, end);
      clipboard::empty();
      clipboard::set_text(os.data());
      return true;
    }

    bool plaintext_ctl::draw_caret(view &v, graphics *pg,
                                   const caret_metrics &cm) {
      switch (cm.ctype) {
      case ROW_POSITION:
      case BLOCK_POSITION:
      case CHAR_POSITION: {
        if (caret_state == 2) {
          argb caret_color = selection_caret_color(v);
          pg->fill(caret_color, cm.caret_v_bar());
        }
      } break;
      }
      return true;
    }

    bool plaintext_ctl::on_data_arrived(view &v, element *self,
                                        pump::request *rq) {
      v.notify_data_arrived(self, rq);

      if (_awaiting == rq && !rq->consumed) {
        ustring text = u8::cvt(rq->data());
        self->clear();
        return set_text(v, self, text());
      }
      return false;
    }

    bool plaintext_ctl::check_empty(view &v,bookmark& bm) {
      if (!self->start_caret_pos(v).valid() || self->nodes.length() == 0) {
        handle<element> t = new element(tag::T_TEXT);
        self->append(t, &v);
        self->check_layout(v);
        t->check_layout(v);
        bm = t->start_caret_pos(v);
        return true;
      }
      return false;
    }

    bool plaintext_ctl::insert_chars(view &v, bookmark start, bookmark end, wchars text) {

      richtext_ctl *_this = static_cast<richtext_ctl *>(this);
      _this->clear_comp_chars(v);

      if (!start.valid() || !end.valid()) return false;
      if (!start.is_inside(self) || !end.is_inside(self)) return false;

      bool crlf_seen;

      wchars line; chopline(text,line, crlf_seen);

      if ((start == end) && start.at_char_pos(v) && start.node->is_text() && !crlf_seen /*not a multiline text*/) {
        behavior::insert_text *tit = top_insert_text();
        bookmark               bm = start;
        if (tit && tit->append(v, _this, top(), bm, line)) {
          _this->select(v, bookmark());
          v.commit_update();
          _this->select(v, bm);
          event_behavior evt(self, self, EDIT_VALUE_CHANGED, CHANGE_BY_INS_CONSECUTIVE_CHAR);
          v.post_behavior_event(evt, true);

          return true;
        }
      }

      handle<action> op = new action(_this, WCHARS("insert text"));

      _this->select(v, bookmark());

      op->change_reason = u16::codepoints(text) == 1 ? CHANGE_BY_INS_CHAR : CHANGE_BY_INS_CHARS;

      try {

        if (start != end) {
          if (start > end) swap(start, end);
          start = end = remove_range(v, _this, op, start, end, is_richtext(),false);
        }
        bookmark bm = start;
        
        helement lroot = this->root_at(v, bm);
        bookmark initial_bm = bm;
        do {
          insert_text::exec(v, _this, op, bm, line);
          element *p = get_inlines_container(bm.node->get_element(), lroot);
          if (p && p->parent && crlf_seen) {
            bool     created = false;
            bookmark nbm = bm;
            split_at(v, this, op, bm, p->parent.ptr(), true, created, nbm, true);
            if (!created) {
              element *   np = p->clone_element(false);
              html::text *nt = new html::text(wchars());
              np->append(nt);
              insert_node::exec(v, this, op, p->parent, p->node_index + 1, np);
              bm = nt->start_pos();
            }
            else
              bm = nbm;
          }
        } while (chopline(text, line, crlf_seen));

        v.commit_update(true);
        _this->select(v, bm);
        push(v, op);
        return true;
      }
      catch (const tool::exception &) { op->undo(v, _this); }
      return false;
    }

    bool plaintext_ctl::insert_break(view &v, bookmark start, bookmark end) {
      richtext_ctl *_this = static_cast<richtext_ctl *>(this);
      _this->clear_comp_chars(v);

      handle<action> op = new action(_this, WCHARS("insert break"));

      try {

        bookmark       bm = start;
        if (start != end) {
          if (start > end) swap(start, end);
          bm = remove_range(v, _this, op, start, end, false,false);
          if (!bm.valid()) return false;
        }

        bm = break_position(v, root(), bm);

        helement base = bm.node->nearest_box(false);
        if (!base || !base->belongs_to(_this->root(), true)) return false;

        // base->dbg_report("insert break in:");

        bool created = false;
        // bool advance = true;

        if (bm == base->end_pos() || bm == base->end_caret_pos(v)) {
          helement nel = base->clone_element_for_editing(false);
          insert_node::exec(v, _this, op, base->parent, base->node_index + 1, nel);
          bm = nel->start_pos();
          goto DONE;
        }
        else if (bm == base->start_pos() || bm == base->start_caret_pos(v)) {
          helement nel = base->clone_element_for_editing(false);
          insert_node::exec(v, _this, op, base->parent, base->node_index, nel);
          bm = nel->end_pos();
          goto DONE;
        }
        {
          bookmark dummy;
          split_element_at(v, _this, op, bm, base, false, created, dummy);
        }
      DONE:
        v.commit_update(true);
        // if( advance ) {
        if (_this->advance(v, bm, ADVANCE_NEXT)) _this->select(v, bm);
        //}
        // else
        //  _this->select(v,bm);

        push(v, op);
        return true;
      }
      catch (const tool::exception &) { op->undo(v, _this); }
      return false;
    }


    bool plaintext_ctl::on_x_method_call(view &v, element *self,
                                         const char *name, const value *argv,
                                         size_t argc, value &retval) {
      chars fname = chars_of(name);
#define METHOD(ARGC, NAME) if (argc == ARGC && fname == CHARS(#NAME))
      METHOD(1, loadDocument) {
        if (!argv[0].is_undefined()) {
          string                in  = argv[0].get(W(""));
          string                out = combine_url(self->doc()->uri().src, in);
          handle<pump::request> rq  = new pump::request(out, DATA_RAW_DATA);
          rq->dst                   = self;
          _awaiting = rq;
          v.request_data(rq);
          retval = value(true);
        }
        return true;
      }

      METHOD(3, setContent) {
        self->clear();
        ustring sval  = argv[0].get(W(""));
        int_v   apos  = argv[1].get(0);
        int_v   cpos  = argv[2].get(0);
        wchars  chars = sval;
        return set_text(v, self, chars, apos, cpos);
      }

      METHOD(1, setContent) {
        self->clear();
        ustring sval  = argv[0].get(W(""));
        wchars  chars = sval;
        return set_text(v, self, chars);
      }
      METHOD(0, getContent) {
        array<wchar> text;
        int_v        apos, cpos;
        this->get_text(v, self, text, apos, cpos);
        value rarr[3] = {value(text()), apos.to_value(), cpos.to_value()};
        retval        = value::make_array(items_of(rarr));
        return true;
      }

      METHOD(0, emptyDocument) {
        self->clear();
        text *   t  = new text(WCHARS(""));
        element *te = new element(tag::T_TEXT);
        te->append(t);
        self->append(te);
        return true;
      }

      METHOD(1, saveDocument) {
        if (!argv[0].is_undefined()) {
          string  in = argv[0].get(W(""));
          ustring file_name = url::unescape(combine_url(self->doc()->uri().src, in));
          file_name = url::file_url_to_path(file_name);
          FILE *f = nullptr;
#if defined(WINDOWS)
          if (0 != _wfopen_s(&f, file_name, L"w+")) f = nullptr;
#else
          f = fopen(string(file_name), "w+");
#endif
          if (f) {
            each_child texts(self);
            for (element *el; texts(el);) {
              if (el->nodes.size() == 0) continue;    // ???
              if (!el->nodes[0]->is_text()) continue; // ???
              wchars line = el->nodes[0].ptr_of<text>()->chars();
              string ln = u8::cvt(line); 
              if(el != el->parent->last_element())
                ln += "\r\n";
              fputs(ln.c_str(), f);
            }
            fclose(f);
            retval = value(true);
          }
          else
            retval = value::make_error("failed to save document");

          return true;
        }
      }
#undef METHOD
      return super::on_x_method_call(v, self, name, argv, argc, retval);
    }

    bool plaintext_ctl::load(ustring url) {
      string                out = combine_url(self->doc()->uri().src, url);
      handle<pump::request> rq = new pump::request(out, DATA_RAW_DATA);
      rq->dst = self;
      _awaiting = rq;
      if(auto pv = self->pview())
        pv->request_data(rq);
      return true;
    }
    bool plaintext_ctl::save(ustring url) 
    {
      ustring file_name = url::file_url_to_path(combine_url(self->doc()->uri().src, url));
      file_name = url::file_url_to_path(file_name);
      FILE *f = nullptr;
#if defined(WINDOWS)
      if (0 != _wfopen_s(&f, file_name, L"w+")) f = nullptr;
#else
      f = fopen(string(file_name), "w+");
#endif
      if (f) {
        each_child texts(self);
        for (element *el; texts(el);) {
          if (el->nodes.size() == 0) continue;    // ???
          if (!el->nodes[0]->is_text()) continue; // ???
          wchars line = el->nodes[0].ptr_of<text>()->chars();
          string ln = u8::cvt(line);
          if (el != el->parent->last_element())
            ln += "\r\n";
          fputs(ln.c_str(), f);
        }
        fclose(f);
        return true;
      }
      else
        //retval = value::make_error("failed to save document");
        return false;
    }

    bool plaintext_ctl::appendLine(tool::value text) {
      if (auto pv = self->pview()) {
        if (text.is_string()) {
          helement el = new element(tag::T_TEXT);
          el->append(new html::text(text.get<ustring>()()));
          self->append(el, pv);
          return true;
        }
        else if (text.is_array_like()) {
          for (uint n = 0; n < text.size(); ++n) {
            ustring s = text.get_element(n).get<ustring>();
            helement el = new element(tag::T_TEXT);
            el->append(new html::text(s()));
            self->append(el);
          }
          pv->add_to_update(self, true);
          return true;
        }
      }
      return false;
    }
    bool plaintext_ctl::insertLine(int at, tool::value text)
    {
      if (auto pv = self->pview()) {
        if (text.is_string()) {
          helement el = new element(tag::T_TEXT);
          el->append(new html::text(text.get<ustring>()()));
          self->insert(at, el, pv);
          return true;
        }
        else if (text.is_array_like()) {
          for (uint n = 0; n < text.size(); ++n) {
            ustring s = text.get_element(n).get<ustring>();
            helement el = new element(tag::T_TEXT);
            el->append(new html::text(s()));
            self->insert(at + n, el);
          }
          pv->add_to_update(self, true);
          return true;
        }
      }
      return false;
    }
    bool plaintext_ctl::removeLine(int at, int howmany) {
      if (auto pv = self->pview()) {
        //el->insert(at, new html::text(text()), pv);
        if (howmany <= 0) howmany = 1;
        self->remove_nodes(at, at + howmany - 1);
        return true;
      }
      return false;
    }
    bool plaintext_ctl::selectAll() {
      if (auto pv = self->pview()) {
        move_caret(*pv, ADVANCE_LAST, false);
        move_caret(*pv, ADVANCE_FIRST, true);
        return true;
      }
      return false;
    }
    bool plaintext_ctl::selectRange(int line1, int pos1, int line2, int pos2)
    {
      if (auto pv = self->pview()) {
        int tn = self->n_children();
        if (line1 < 0 || line1 >= tn) return false;
        if (line2 < 0 || line2 >= tn) return false;

        bookmark start, end;

        {
          helement el = self->child(line1);
          if (el->tag != tag::T_TEXT) return false;
          if (el->nodes.size() == 0) return false;
          handle<html::text> t = el->nodes[0].ptr_of<html::text>();
          if (pos1 < 0) start = bookmark(t, 0, false);
          else if (pos1 >= t->chars.size()) start = bookmark(t, t->chars.last_index(), true);
          else start = bookmark(t, pos1, false);
        }

        {
          helement el = self->child(line2);
          if (el->tag != tag::T_TEXT) return false;
          if (el->nodes.size() == 0) return false;
          handle<html::text> t = el->nodes[0].ptr_of<html::text>();
          if (pos2 < 0) start = bookmark(t, 0, false);
          else if (pos2 >= t->chars.size()) start = bookmark(t, t->chars.last_index(), true);
          else start = bookmark(t, pos2, false);
        }
        select(*pv, end, start);
        return true;
      }
      return false;
    }

    tool::ustring plaintext_ctl::getContent() {
      array<wchar> out;
      each_child texts(self);
      for (element *el; texts(el);) {
        if (el->nodes.size() == 0) continue;    // ???
        if (!el->nodes[0]->is_text()) continue; // ???
        wchars line = el->nodes[0].ptr_of<text>()->chars();
        if (el != el->parent->last_element())
          out.push(WCHARS("\r\n"));
        out.push(line);
      }
      return out();
    }
    bool    plaintext_ctl::setContent(tool::value stringOrArray)
    {
      if (auto pv = self->pview()) {
        self->clear();
        if (stringOrArray.is_array()) {
          
          for (uint n = 0; n < stringOrArray.size(); ++n)
          {
            ustring line = stringOrArray.get_element(n).get<ustring>();
            element *te = new element(tag::T_TEXT);
            text *   t = new text(line());
            te->append(t);
            self->append(te);
          }
          pv->add_to_update(self, true);
          return true;
        }
        else if (stringOrArray.is_string()) {
          set_text(*pv, self, stringOrArray.get<ustring>()());
          return true;
        }
      }
      return false;
    }
    int     plaintext_ctl::getLines() {
      return self->n_children();
    }
    bool    plaintext_ctl::getLine(int n, tool::ustring& text)
    {
      helement el = self->child(n);
      if (el->tag != tag::T_TEXT) return false;
      if (el->nodes.size() == 0) return false;
      html::text* t = el->nodes[0].ptr_of<html::text>();
      if (t) text = t->chars();
      return true;
    }
    bool    plaintext_ctl::setLine(int n, tool::ustring text) {
      int tn = self->n_children();
      if (n < 0 || n >= tn) return false;
      helement el = self->child(n);
      if (el->nodes.size() == 0) return false;
      if (!el->nodes[0]->is_text()) return false;
      el->nodes[0].ptr_of<html::text>()->chars = text();
      if (auto pv = self->pview())
        pv->add_to_update(el, true);
      return true;
    }
    bool    plaintext_ctl::nextLine(int& n, tool::ustring& text)
    {
      int tn = self->n_children();
      if (n < 0 || n >= tn) return false;
      helement el = self->child(n);
      if (el->tag != tag::T_TEXT) return false;
      if (el->nodes.size() == 0) return false;
      html::text* t = el->nodes[0].ptr_of<html::text>();
      if (t) text = t->chars();
      ++n;
      return true;
    }

    tool::array<int> plaintext_ctl::getSelectionStart()
    {
      if (!anchor.valid())
        return {};
      element* el = anchor.node->get_element();
      return { el->index(), anchor.linear_pos() };
    }

    tool::array<int> plaintext_ctl::getSelectionEnd()
    {
      if (!caret.valid())
        return{};
      element* el = caret.node->get_element();
      return{ el->index(), caret.linear_pos() };
    }

    tool::ustring    plaintext_ctl::getSelectionText()
    {
      if (!caret.valid()) return tool::ustring();
      auto        sel = normalized();
      html::view *pv = caret.node->pview();
      if (!pv) return tool::ustring();
      html::ostream_w os;
      html::emit_range_text(*pv, os, sel.first, sel.second);
      return os.data();
    }

  } // namespace behavior

} // namespace html