#include "html.h"

#if defined(WINDOWS) && !defined(WINDOWLESS)

#include <winspool.h>
#include <CommDlg.h>
#include "win/win-view.h"
#include "win/win-delayload.h"

#if defined(USE_GDI)
#include "gdi+/gdi+graphics.h"
#endif

#include "html-view-print.h"

namespace html {
  namespace behavior {

#define PRINTING_PPI 600

    struct pager_ctl_factory : ctl_factory {
      pager_ctl_factory() : ctl_factory("pager") {}
      virtual ctl *create(element *el);
    };
    struct page_box_ctl_factory : ctl_factory {
      page_box_ctl_factory() : ctl_factory("page-box") {}
      virtual ctl *create(element *el);
    };
    struct page_content_box_ctl_factory : ctl_factory {
      page_content_box_ctl_factory() : ctl_factory("page-content-box") {}
      virtual ctl *create(element *el);
    };

    static pager_ctl_factory *           _pager_ctl_factory            = 0;
    static page_box_ctl_factory *        _page_box_ctl_factory         = 0;
    static page_content_box_ctl_factory *_page_content_box_ctl_factory = 0;

    struct page_box_ctl;
    struct page_content_box_ctl;

    struct pager_ctl : ctl {
      typedef ctl super;

      element *self;

      handle<print_target> ptarget;

      handle<html::print_view>
          prv; // view holding page template and document in its frame

      int_v page_rows;
      int_v page_cols;
      int_v page_no; // first visible page no
      int_v total_pages;

      double page_ratio;
      size_v page_width;
      size_v page_height;
      // int        num_CSS_pixels; // number of "CSS pixels" in page width, 1px
      // = 1/96in
      double     scale;
      tristate_v auto_scale;
      size       rendered_dim;
      // size       page_device_dim; // size of page in PRINTING_PPI units

      array<handle<element>> page_boxes;

      array<page_content_box_ctl *>
          page_content_boxes; // number of page content boxes on the screen

      size                  paginated_dim; // page dimenesion used by paginator
      array<gool::range>    pages; // page y ranges discovered by the paginator
      page_content_box_ctl *paginator;

      handle<document> doc;      // document being printed;
      ustring          doc_name; // document caption to appear print spooler

      pager_ctl()
          : self(nullptr), scale(1.0), paginator(nullptr), page_ratio(1.0) {}

      virtual CTL_TYPE get_type() { return CTL_FRAME; }

      virtual const string &behavior_name() const {
        return _pager_ctl_factory->name;
      }

      // virtual bool focusable(const element* self) { return false; }

      virtual void detach(view &v, element *self) override;
      virtual bool attach(view &v, element *self) override;
      void         do_layout(view &v, element *self);

      virtual bool on_data_request(view &v, element *self, pump::request *rq) {
        v.notify_data_request(self, rq);
        return super::on_data_request(v, self, rq);
      }
      virtual bool on_data_arrived(view &v, element *self, pump::request *rq) {
        v.notify_data_arrived(self, rq);
        return super::on_data_arrived(v, self, rq);
      }

      virtual bool handle_data_request(pump::request *rq) {
        if (!self) return false;
        view *pv = self->pview();
        if (!pv) return false;
        return on_data_request(*pv, self, rq);
      }
      virtual bool handle_data_arrived(pump::request *rq) {
        if (!self) return false;
        view *pv = self->pview();
        if (!pv) return false;
        return on_data_arrived(*pv, self, rq);
      }

      string get_content_style_url() {
        string url = get_attr(self, "-content-style");
        if (url.is_undefined()) return url;
        return combine_url(self->doc()->uri().src, url);
      }

      virtual bool draw_background(view &v, element *self, graphics *sf,
                                   point pos) override {
        if (self && ptarget) do_layout(v, self);
        return false;
      }

      void relayout(view &v, element *self);

      void setup_page(int pn);

      bool load_template(string template_url);

      bool load_document(view &v, element *self, string url);
      bool load_html(view &v, element *self, bytes html, string url);

      page_content_box_ctl *page_content_frame(element **ppframe = nullptr);

      void notify_pagination_start() {
        if (!self) return;
        view *pview = self->pview();
        if (!pview) return;
        event_behavior evt(self, self, PAGINATION_STARTS, 0);
        pview->send_behavior_event(evt);
      }
      void notify_pagination_page(int npage) {
        if (!self) return;
        view *pview = self->pview();
        if (!pview) return;
        event_behavior evt(self, self, PAGINATION_PAGE, npage);
        pview->send_behavior_event(evt);
      }
      void notify_pagination_end(int npages) {
        if (!self) return;
        view *pview = self->pview();
        if (!pview) return;
        event_behavior evt(self, self, PAGINATION_ENDS, npages);
        pview->post_behavior_event(evt);
        drop_page_image_cache();
        pview->refresh(self);
      }

      void drop_page_image_cache();

      bool navigate_to(view &v, int npage) {
        // page_content_boxes; // number of page content boxes on the screen
        // size                         paginated_dim;      // page dimenesion
        // used by paginator
      }

      virtual bool on_x_method_call(view &v, element *self, const char *name,
                                    const value *argv, size_t argc,
                                    value &retval) override;

      handle<bitmap> page_image(view &v, element *self, graphics *sf,
                                int page_no);

      bool print(view &v, element *self, slice<int> pages = slice<int>());
      bool print_page(view &v, element *self, graphics *sf, int page_no);


      int  api_get_pages() { return pages.size(); }

      bool api_load_file(string doc_url, string template_url) {
        doc = nullptr;
        view* pv = self->pview();
        if (!pv) return false;

        if (template_url.length() && !load_template(template_url))
          return false;

        if (!load_document(*pv, self, doc_url))
          return false;

        relayout(*pv, self);
                
        return true;
      }
      bool api_load_html(string html, string doc_url, string template_url) {

        view* pv = self->pview();
        if (!pv) return false;

        if (template_url.length() && !load_template(template_url))
          return false;

        doc = nullptr;

        load_html(*pv, self, html.chars_as_bytes(), doc_url);

        return true;
      }

      int api_get_page() {
        return this->page_no;
      }
      bool api_set_page(int pn) {
        view* pv = self->pview();
        if (!pv) return false;
        page_no = limit(pn, 1, int(pages.size()));
        do_layout(*pv, self);
        pv->refresh(self);
        return true;
      }

      bool api_select_printer_dialog() {
        view* pv = self->pview();
        if (!pv) return false;
        handle<print_target> pt = pv->app->select_printer(pv);
        if (pt) {
          ptarget = pt;
          relayout(*pv, self);
        }
        return true;
      }

      bool api_select_default_printer() {
        view* pv = self->pview();
        if (!pv) return false;
        handle<print_target> pt = pv->app->select_printer(nullptr);
        if (pt) {
          ptarget = pt;
          relayout(*pv, self);
        }
        return true;
      }

      bool api_select_printer(ustring printer_id) {
        view* pv = self->pview();
        if (!pv) return false;
        handle<print_target> pt = pv->app->select_printer(pv, printer_id);
        if (pt) {
          ptarget = pt;
          relayout(*pv, self);
        }
        return true;
      }

      ustring api_get_doc_name() { return doc_name; }
      bool    api_set_doc_name(ustring n) { doc_name = n; return true; }

      bool    api_print(value pages_list) {
        view* pv = self->pview();
        if (!pv) return false;
        array<int> pages;
        if (pages_list.is_array_like()) {
          for (uint i = 0; i < pages_list.size(); ++i)
            pages.push(pages_list.get_element(i).get(0));
        }
        print(*pv, self, pages());
        return true;
      }

      value api_printers() {
        view* pv = self->pview();
        if (!pv) return value();

        array<printer_info> list = pv->app->get_printer_list();

        value out = value::make_array();
        for (int n = 0; n < list.size(); ++n) {
          printer_info  pi = list[n];
          value vpi = value::make_map();
          vpi.set_prop("id", value(pi.id));
          vpi.set_prop("name", value(pi.name));
          vpi.set_prop("shareName", value(pi.share_name));
          vpi.set_prop("comment", value(pi.comment));
          vpi.set_prop("location", value(pi.location));
          vpi.set_prop("isDefault", value(!!pi.is_default));
          out.push(vpi);
        }
        return out;
      }

      SOM_PASSPORT_BEGIN_EX(pager, pager_ctl)
        SOM_FUNCS(
          SOM_FUNC_EX(loadFile, api_load_file),
          SOM_FUNC_EX(loadHtml, api_load_html),
          SOM_FUNC_EX(selectPrinterDialog, api_select_printer_dialog),
          SOM_FUNC_EX(selectDefaultPrinter, api_select_default_printer),
          SOM_FUNC_EX(selectPrinter, api_select_printer),
          SOM_FUNC_EX(print, api_print),
          SOM_FUNC_EX(printers, api_printers),
        )
        SOM_PROPS(
          SOM_RO_VIRTUAL_PROP(pages, api_get_pages),
          SOM_VIRTUAL_PROP(page, api_get_page, api_set_page),
          SOM_VIRTUAL_PROP(documentName, api_get_doc_name, api_set_doc_name),
        )
      SOM_PASSPORT_END

    };

    // single page box inside the pager, renders page template with page frame
    // (below) inside
    struct page_box_ctl : public frame_ctl {
      typedef frame_ctl super;

      handle<bitmap> _page;
      int_v          _page_no;
      handle<bitmap> _img;

      page_box_ctl() {}

      // virtual CTL_TYPE get_type() { return CTL_FRAME; }
      virtual const string &behavior_name() const {
        return _page_box_ctl_factory->name;
      }

      // virtual bool focusable(const element* self) { return false; }

      virtual void detach(view &v, element *self) override {
        _img = nullptr;
        return super::detach(v, self);
      }
      virtual bool attach(view &v, element *self) override {
        super::attach(v, self);
        // int idx = self->index();
        return pager(self) != nullptr;
      }

      virtual bool draw_content(view &v, element *self, graphics *sf,
                                point pos) override {
        if (_img) sf->draw(_img, rect(pos, self->dim()));
        return true;
      }

      virtual bool draw_background(view &v, element *self, graphics *sf,
                                   point pos) override {

        auto pg = pager(self);
        if (!pg) return false;

        if (pg->pages.size() == 0) return false;

        int idx = pg->page_boxes.get_index(self);
        if (idx < 0) return false;

        if (pg->page_no + idx != _page_no || !_img ||
            _img->dim() != self->dim()) {
          _page_no = pg->page_no + idx;
          if (pg->pages.size() && _page_no > pg->pages.size()) {
            _img = nullptr;
            return true;
          }
          _img = pg->page_image(v, self, sf, _page_no);
        } else {
          if (pg->pages.size() && _page_no > pg->pages.size()) {
            _img = nullptr;
            return true;
          }
        }

        /*      if (pg->pages.size() && _page_no > pg->pages.size()) {
                _img = nullptr;
                return true;
              }
              else {
                _page_no = pg->page_no + idx;
                _img = pg->page_image(v, self, sf, _page_no);
              } */

        return false;
      }

      pager_ctl *pager(element *self) {
        pager_ctl *pp = static_cast<pager_ctl *>(
            self->parent->get_named_behavior(_pager_ctl_factory->name));
        assert(pp);
        return pp;
      }
    };

    // page content box inside the page box - renders fragment of the document
    struct page_content_box_ctl : public frame_ctl {
      typedef frame_ctl super;

      element *self;
      int      self_index; // index of this page_content_box_ctl in
                           // page_content_boxes array
      int page_no;         // page number of viewed document

      page_content_box_ctl() : self_index(0), page_no(0) {
        subscription = uint(-1);
      }

      // virtual CTL_TYPE get_type() { return CTL_FRAME; }
      virtual const string &behavior_name() const {
        return _page_content_box_ctl_factory->name;
      }

      // virtual bool focusable(const element* self) { return false; }

      virtual void detach(view &v, element *self) override {
        return super::detach(v, self);
      }
      virtual bool attach(view &v, element *self) override {
        super::attach(v, self);
        this->self = self;
        auto pg    = pager(self);
        if (!pg) return false;
        self_index = pg->page_content_boxes.size();
        page_no    = pg->page_no + self_index;
        pg->page_content_boxes.push(this);
        return true;
      }

      virtual bool draw_content(view &v, element *self, graphics *sf,
                                point pos) override;
      virtual void on_size_changed(view &v, element *self) override;

      pager_ctl *pager(element *self) {
        view *pv = self->pview();
        if (!pv || !pv->is_printing()) return nullptr;
        html::print_view *ppv = static_cast<html::print_view *>(pv);
        if (!ppv->host()) return nullptr;
        return static_cast<pager_ctl *>(ppv->host()->get_named_behavior(_pager_ctl_factory->name));
      }

      virtual bool on_data_request(view &v, element *self,
                                   pump::request *rq) override {
        pager(self)->handle_data_request(rq);
        return super::on_data_request(v, self, rq);
      }
      virtual bool on_data_arrived(view &v, element *self,
        pump::request *rq) override {
        if (auto pg = pager(self)) {
          pg->handle_data_arrived(rq);
          bool r = super::on_data_arrived(v, self, rq);
          view* pv = pg->self->pview();
          if(pv)
            pg->relayout(*pv, pg->self);
          return r;
        }
        return super::on_data_arrived(v, self, rq);
      }

      virtual string get_content_style_url(element *self) override {
        return pager(self)->get_content_style_url();
      }
    };

    ctl *pager_ctl_factory::create(element *el) { 
      html::view* pv = el->pview();
      if (!pv) return nullptr;
      if (pv->is_printing()) {
        view::debug_printf(OT_DOM, OS_ERROR, "behavior:pager cannot be created inside printint view");
        return nullptr;
      }
      return new pager_ctl(); 
    }
    ctl *page_box_ctl_factory::create(element *el) {
      return new page_box_ctl();
    }
    ctl *page_content_box_ctl_factory::create(element *el) {
      return new page_content_box_ctl();
    }

    void init_pager() {
      ctl_factory::add(_pager_ctl_factory = new pager_ctl_factory());
      ctl_factory::add(_page_box_ctl_factory = new page_box_ctl_factory());
      ctl_factory::add(_page_content_box_ctl_factory =
                           new page_content_box_ctl_factory());
    }

    void pager_ctl::setup_page(int pn) {
      if (!prv->doc()) return;
      element *pframe = find_first(*prv, prv->doc(), WCHARS("pageframe"));

      prv->page_no = pn;

      helement totalpages;
      int      reported_total = 0;

      prv->doc()->set_attr(*prv, "page-no", itow(pn));
      prv->doc()->set_attr(*prv, "page-parity", pn & 1 ? L"odd" : L"even");
      // prv->doc()->drop_layout();
      helement t = find_first(*prv, prv->doc(), WCHARS("#page-no"));
      if (t) t->set_value(*prv, value(pn));
      totalpages = find_first(*prv, prv->doc(), WCHARS("#total-pages"));
      if (totalpages) {
        reported_total = this->pages.size();
        totalpages->set_value(*prv, value(reported_total));
      }

      if (doc) {
        t = find_first(*prv, prv->doc(), WCHARS("#document-title"));
        if (t && pframe) {
          element *title = find_first(*prv, doc, WCHARS("head>title"));
          value    title_text;
          if (title && title->get_value(*prv, title_text))
            t->set_value(*prv, title_text);
        }
        t = find_first(*prv, prv->doc(), WCHARS("#document-url"));
        if (t) {
          ustring url = url::unescape(doc->uri().src);
          t->set_value(*prv, value(url));
        }
      }

      // prv->on_size( page_device_dim );
      prv->on_size(ptarget->page_device_size);
      prv->on_idle();

      if (totalpages && reported_total != this->pages.size()) {
        totalpages->set_value(*prv, value(this->pages.size()));
        prv->commit_update();
        prv->on_idle();
      }
    }

    bool pager_ctl::attach(view &v, element *self) {
      ctl::attach(v, self);

      self->allow_reconciliation(false);

      this->self = self;
      window_params params(PRINT_VIEW);
      params.parent = &v;
      prv = v.app->create_print_processor(params);
      if (!prv) return false;

      auto init = [this, &v]() -> bool {

        ptarget = prv->app->select_printer();

        if (!ptarget) return false;

        // prv->update_cb = [this](print_view* pv) {
        //  relayout(*pv->parent(), this->self);
        //};

        page_no = 1;

        window_params params(PRINT_VIEW);
        params.parent = &v;
        
        prv->host(this->self);
        prv->start(params);

        prv->pixels_per_inch(ptarget->device_dpi);

        string template_url = this->self->atts.get_url(
            this->self->doc()->uri().src, "page-template");

        if (!load_template(template_url)) return false;

        string doc_url =
            this->self->atts.get_url(this->self->doc()->uri().src, "src");

        if (doc_url.is_defined()) load_document(v, this->self, doc_url);

#if defined(SCITER)
        event_behavior evt(this->self, this->self, DOCUMENT_READY, 0);
        v.post_behavior_event(evt);
#else
        event_behavior evt(this->self, this->self, CUSTOM, 0);
        evt.name = W("paginationready");
        v.post_behavior_event(evt);
#endif
        return true;
      };

      v.async(init);

      return true;
    }

    void pager_ctl::relayout(view &v, element *self) {

      prv->pixels_per_inch(ptarget->device_dpi);
      rendered_dim = size();
      /*string page_box_name = "page-box";
      for (auto pb : page_boxes) {
        page_box_ctl* pc =
      static_cast<page_box_ctl*>(pb->get_named_behavior(page_box_name)); if (pc)
          pc->_img = nullptr;
      }*/
      page_boxes.clear();
      page_content_boxes.clear();
      paginated_dim = size();
      pages.clear();
      self->clear(&v);

      if (prv->doc()) {
        find_all(*prv, prv->doc(), WCHARS("pageframe"),
                 [](element *pb) -> bool {
                   // to force on_size_changed on the element
                   pb->ldata->used_dim = size();
                   return true;
                 });
      }

      setup_page(1);
      prv->on_idle();
      prv->commit_update();
      prv->on_idle();
      v.refresh(self);
    }

    bool pager_ctl::load_template(string template_url) {
      page_content_boxes.clear();
      paginated_dim = size();
      pages.clear();
      if (!template_url.length())
        template_url = this->self->atts.get_url(this->self->doc()->uri().src,
                                                "page-template");
      if (!template_url.length())
        template_url = "sciter:default-page-template.htm";
      handle<pump::request> rq = new pump::request(template_url, DATA_RAW_DATA);
      rq->dst                  = self;
      // rq->dst_view = prv;
      if (prv && prv->load_data(rq, true)) {
        prv->load(rq);
        return true;
      } else {
        prv->debug_printf(html::OT_DOM, html::OS_WARNING,
                          "cannot load page template %s\n",
                          template_url.c_str());
        return false;
      }
    }

    bool pager_ctl::load_document(view &v, element *self, string src) {
      element *page_content_box_el = 0;
      page_no                      = 1;
      page_content_box_ctl *pframe =
          this->page_content_frame(&page_content_box_el);
      if (!pframe || !prv) return false;

      /*if( page_content_boxes.size() ) {
        page_content_boxes.clear();
        paginated_dim = size();
        v.refresh(self);
      }*/

      handle<pump::request> rq = new pump::request(src, DATA_HTML);
      rq->dst                  = page_content_box_el;
      // rq->dst_view = prv;
      prv->request_data(rq);

      return true;
    }

    bool pager_ctl::load_html(view &v, element *self, bytes data, string src) {
      element *page_content_box_el = 0;
      page_no                      = 1;
      page_content_box_ctl *pframe =  this->page_content_frame(&page_content_box_el);
      if (!pframe) return false;

      pframe->load(*prv, page_content_box_el, src, data);
      relayout(v, self);

      return true;
    }

    void pager_ctl::detach(view &v, element *self) {
      doc = nullptr;
      if (self) 
        self->clear();
      if (prv) {
        prv->stop();
        prv = nullptr;
      }
      ptarget = nullptr;

      page_boxes.clear();
      page_content_boxes.clear();
      pages.clear();

      ctl::detach(v, self);
    }

    page_content_box_ctl *pager_ctl::page_content_frame(element **ppframe) {
      if (!prv) return nullptr;
      if (!prv->doc()) return nullptr;
      element *pageframe = find_first(*prv, prv->doc(), WCHARS("pageframe"));
      if (!pageframe) return nullptr;
      if (ppframe) *ppframe = pageframe;
      return static_cast<page_content_box_ctl *>(
          pageframe->get_named_behavior(_page_content_box_ctl_factory->name));
    }

    bool get_paper_size(const char *name, size_v &w, size_v &h);

    void pager_ctl::do_layout(view &v, element *self) {
      size dim = self->dim();
      if (rendered_dim == dim) return;

      if (!prv) return;

      page_rows = get_attr(self, "-rows", 1);
      page_cols = get_attr(self, "-cols", 1);

      const style *cs = self->get_style(v);

      int page_spacing_x = pixels(v, self, cs->border_spacing_x).width();
      int page_spacing_y = pixels(v, self, cs->border_spacing_y).width(); // % are of width

      if (page_width.is_undefined()) {
        string page_sz = get_attr(self, "-page-size");
        if (page_sz.length()) get_paper_size(page_sz, page_width, page_height);
        if (page_width.is_undefined()) {
          page_width  = size_v(8.5, size_v::unit_type::in);
          page_height = size_v(11, size_v::unit_type::in);
        }
      }

      // page_device_size;
      // page_device_dim.x = page_width.pixels_width(*this->prv,self,0);
      // page_device_dim.y = page_height.pixels_height(*this->prv,self,0);

      page_ratio = double(ptarget->page_device_size.x) /
                   double(ptarget->page_device_size.y);
      // double(page_width.d.value) / double(page_height.d.value);
      // num_CSS_pixels = page_width.pixels(num_CSS_pixels);

      if (page_rows <= 0) page_rows = 1;
      if (page_cols <= 0) page_cols = 1;

      size view_dim = dim;
      view_dim.x -= page_spacing_x * (page_cols - 1);
      view_dim.y -= page_spacing_y * (page_rows - 1);
      view_dim.x /= page_cols;
      view_dim.y /= page_rows;

      double view_ratio = double(view_dim.x) / double(view_dim.y);
      size   page_dim;

      if (view_ratio < page_ratio) {
        page_dim.x = view_dim.x;
        page_dim.y = int(page_dim.x / page_ratio);
      } else {
        page_dim.y = view_dim.y;
        page_dim.x = int(page_dim.y * page_ratio);
      }

      point prvo(0, 0);
      int   n       = 0;
      int   changes = 0;
      for (int r = 0; r < page_rows; ++r) {
        prvo.x = 0;
        for (int c = 0; c < page_cols; ++c) {
          rect pvrc(prvo, view_dim);
          rect prc(page_dim);
          prc.pointOf(5, pvrc.pointOf(5));

          element *pb = 0;
          if (n >= page_boxes.size()) {
            pb = new element(tag::T_PAGEBOX);
            page_boxes.push(pb);
            // pb->atts.set(attr::a_class,L"page");
            self->append(pb, &v);
            ++changes;
          } else
            pb = page_boxes[n];

          ++n;
          pb->check_layout(v);
          pb->set_border_width(v, prc.width());
          pb->set_border_height(v, prc.height());
          pb->set_pos(prc.s);
          prvo.x += view_dim.x + page_spacing_x;
        }
        prvo.y += view_dim.y + page_spacing_y;
        if (changes) v.refresh(self);
      }
    }

    struct paper_format_def {
      chars  name;
      size_v w;
      size_v h;
    };

    static paper_format_def paper_defs[] = {
        {CHARS("letter"), size_v(8.5, size_v::unit_type::in), size_v(11, size_v::unit_type::in)},
        {CHARS("legal"), size_v(8.5, size_v::unit_type::in), size_v(14, size_v::unit_type::in)},
        {CHARS("c-sheet"), size_v(17, size_v::unit_type::in), size_v(22, size_v::unit_type::in)},
        {CHARS("d-sheet"), size_v(22, size_v::unit_type::in), size_v(34, size_v::unit_type::in)},
        {CHARS("e-sheet"), size_v(34, size_v::unit_type::in), size_v(44, size_v::unit_type::in)},
        {CHARS("tabloid"), size_v(11, size_v::unit_type::in), size_v(17, size_v::unit_type::in)},
        {CHARS("ledger"), size_v(17, size_v::unit_type::in), size_v(11, size_v::unit_type::in)},
        {CHARS("statement"), size_v(5.5, size_v::unit_type::in), size_v(8.5, size_v::unit_type::in)},
        {CHARS("executive"), size_v(7.25, size_v::unit_type::in), size_v(10.5, size_v::unit_type::in)},
        {CHARS("a3"), size_v(297, size_v::unit_type::mm), size_v(420, size_v::unit_type::mm)},
        {CHARS("a4"), size_v(210, size_v::unit_type::mm), size_v(297, size_v::unit_type::mm)},
        {CHARS("a5"), size_v(148, size_v::unit_type::mm), size_v(210, size_v::unit_type::mm)},
        {CHARS("b4"), size_v(250, size_v::unit_type::mm), size_v(354, size_v::unit_type::mm)},
        {CHARS("b5"), size_v(182, size_v::unit_type::mm), size_v(354, size_v::unit_type::mm)},
        {CHARS("folio"), size_v(8.5, size_v::unit_type::in), size_v(13, size_v::unit_type::in)},
        {CHARS("quarto"), size_v(215, size_v::unit_type::mm), size_v(275, size_v::unit_type::mm)},
        {CHARS("10x14"), size_v(10, size_v::unit_type::in), size_v(14, size_v::unit_type::in)},
        {CHARS("10x17"), size_v(10, size_v::unit_type::in), size_v(17, size_v::unit_type::in)},
        {CHARS("note"), size_v(8.5, size_v::unit_type::in), size_v(11, size_v::unit_type::in)},
        {CHARS("envelope-9"), size_v(3.875, size_v::unit_type::in), size_v(8.875, size_v::unit_type::in)},
        {CHARS("envelope-10"), size_v(4.125, size_v::unit_type::in), size_v(9.5, size_v::unit_type::in)},
        {CHARS("envelope-11"), size_v(4.5, size_v::unit_type::in), size_v(10.375, size_v::unit_type::in)},
        {CHARS("envelope-12"), size_v(4.75, size_v::unit_type::in), size_v(11, size_v::unit_type::in)},
        {CHARS("envelope-14"), size_v(5, size_v::unit_type::in), size_v(11.5, size_v::unit_type::in)},
        {CHARS("envelope-dl"), size_v(110, size_v::unit_type::mm), size_v(220, size_v::unit_type::mm)},
        {CHARS("envelope-c5"), size_v(162, size_v::unit_type::mm), size_v(229, size_v::unit_type::mm)},
        {CHARS("envelope-c3"), size_v(324, size_v::unit_type::mm), size_v(258, size_v::unit_type::mm)},
        {CHARS("envelope-c4"), size_v(229, size_v::unit_type::mm), size_v(324, size_v::unit_type::mm)},
        {CHARS("envelope-c6"), size_v(114, size_v::unit_type::mm), size_v(162, size_v::unit_type::mm)},
        {CHARS("envelope-c65"), size_v(114, size_v::unit_type::mm), size_v(229, size_v::unit_type::mm)},
        {CHARS("envelope-b4"), size_v(250, size_v::unit_type::mm), size_v(353, size_v::unit_type::mm)},
        {CHARS("envelope-b5"), size_v(176, size_v::unit_type::mm), size_v(250, size_v::unit_type::mm)},
        {CHARS("envelope-b6"), size_v(176, size_v::unit_type::mm), size_v(125, size_v::unit_type::mm)},
        {CHARS("envelope-italy"), size_v(110, size_v::unit_type::mm), size_v(230, size_v::unit_type::mm)},
        {CHARS("envelope-monarch"), size_v(3.875, size_v::unit_type::in), size_v(7.5, size_v::unit_type::in)},
        {CHARS("envelope-personal"), size_v(3.625, size_v::unit_type::in), size_v(6.5, size_v::unit_type::in)},
        {CHARS("fanfold-us"), size_v(14.875, size_v::unit_type::in), size_v(11, size_v::unit_type::in)},
        {CHARS("fanfold-std-german"), size_v(8.5, size_v::unit_type::in), size_v(12, size_v::unit_type::in)},
        {CHARS("fanfold-lgl-german"), size_v(8.5, size_v::unit_type::in), size_v(13, size_v::unit_type::in)},
    };

    bool get_paper_size(const char *name, size_v &w, size_v &h) {
      tool::string lcname = name;
      lcname.to_lower();
      for (int n = 0; n < items_in(paper_defs); ++n) {
        if (lcname == paper_defs[n].name) {
          w = paper_defs[n].w;
          h = paper_defs[n].h;
          return true;
        }
      }
      array<chars> list = lcname.tokens(CHARS(" ,"));
      if (list.size() == 2) {
        from_string(w, ustring(list[0]));
        from_string(h, ustring(list[1]));
        return w.is_defined() && h.is_defined();
      }
      return false;
    }

    void pager_ctl::drop_page_image_cache() {
      static string page_box_name = "page-box";
      for (auto pb : page_boxes) {
        page_box_ctl *pc =
            static_cast<page_box_ctl *>(pb->get_named_behavior(page_box_name));
        if (pc) pc->_img = nullptr;
      }
    }

    handle<bitmap> pager_ctl::page_image(view &v, element *self, graphics *sf,
                                         int page_no) {
      // return nullptr;
      size dim = self->dim();

      handle<bitmap> img = new bitmap(dim, true, false);
      {
        handle<graphics> gfx =
            prv->app->create_bitmap_graphics(sf, img, argb(255, 255, 255));
        if (!gfx) return img;

        prv->set_graphics(gfx);

        setup_page(page_no);

        gfx->scale(sizef(float(dim.x) / ptarget->page_device_size.x,
                         float(dim.y) / ptarget->page_device_size.y));
        gfx->set_clip_rc(ptarget->page_device_size);

        {
          auto_state<html::graphics *> _1(prv->drawing_surface, gfx);
          prv->on_idle();
          prv->paint();
        }

        prv->set_graphics(nullptr);
      }
      return img;
    }

    bool pager_ctl::print_page(view &v, element *self, graphics *gfx,
                               int page_no) {

      size dim = self->dim();
      prv->set_graphics(gfx);

      setup_page(page_no);

      gfx->set_clip_rc(ptarget->page_device_size);

      {
        auto_state<html::graphics *> _1(prv->drawing_surface, gfx);
        prv->on_idle();
        prv->paint();
      }

      prv->set_graphics(nullptr);
      return true;
    }

    void page_content_box_ctl::on_size_changed(view &v, element *self) {
      auto pg     = pager(self);
      int  npages = pg->pages.size();

      if (pg->paginator) return;

      page_no = pg->prv->page_no;

      // dbg_printf("drawing page %d\n",page_no);
      document *pd = pg->doc;

      if (!pd) {
        element *el = self->child(0);
        if (!el) 
          return;
        if (!el->is_document()) 
          return;
        pd = el->cast<document>();
      }

      if (pg->paginated_dim != self->dim() || pg->pages.size() == 0) {
        auto_state<page_content_box_ctl *> _(pg->paginator, this);

        if (self->nodes.size() == 0) {
          self->nodes.push(pd);
          self->check_layout(*pg->prv);
          pd->drop_layout(pg->prv);
          pg->doc = nullptr;
        }

        pg->paginated_dim = self->dim();
        pd->set_width(*pg->prv, pg->paginated_dim.x);
        pd->set_height(*pg->prv, pg->paginated_dim.y);
        pd->commit_measure(*pg->prv);

        pd->ldata->offset = point(0, 0);

        pg->notify_pagination_start();
        npages = pd->paginate(v, pg->paginated_dim, pg->pages);
        pg->notify_pagination_end(npages);

        // we need to remove the document from the frame as we've done with the
        // pagination in order to pretect it from further remeasurments.
        pg->doc = pd;
        self->nodes.clear();
        self->ldata->drop();
        self->measure_inplace(*pg->prv);
      }
    }

    bool page_content_box_ctl::draw_content(view &v, element *self,
                                            graphics *sf, point pos) {
      auto pg = pager(self);
      // int npages = pg->pages.size();

      if (pg->paginator) return true;

      page_no = pg->prv->page_no;

      // dbg_printf("drawing page %d\n",page_no);
      document *pd = pg->doc;

      if (!pd) {
        element *el = self->child(0);
        if (!el) return false;
        if (!el->is_document()) return false;
        pd = el->cast<document>();
      }

      assert(pg->pages.size() > 0 && pg->pages.size() < 1000);

#ifdef _DEBUG
      if (page_no == 2) page_no = page_no;
#endif

      pd->ldata->offset = point(0, pg->pages[page_no - 1].s);

      gool::state _st(sf);

      sf->translate(pos);

      rect area = rect(self->dim());
      // sf->fill(argb(255,0,0),area);
      // area.corner.y = area.origin.y + 2000;

      clipper          _c(sf, area);
      auto_state<bool> _dc(pg->prv->drawing_content, true);

      pg->prv->page_y = area.y();

      pd->draw_content(v, sf, point(0, 0), false);

      return true;
    }

    bool pager_ctl::on_x_method_call(view &v, element *self, const char *name,
                                     const value *argv, size_t argc,
                                     value &retval) {
      chars fname = chars_of(name);
#define METHOD(ARGC, NAME) if (argc == ARGC && fname == CHARS(#NAME))
      METHOD(0, pagesTotal) {
        retval = int(pages.size());
        return true;
      }

      METHOD(1, loadDocument) {
        string doc_url = argv[0].to_string();

        doc = nullptr;

        if (doc_url.is_defined()) {
          load_document(v, self, doc_url);
        }
        retval = value(true);
        return true;
      }

      METHOD(2, loadDocument) {
        string template_url = argv[0].to_string();
        string doc_url      = argv[1].to_string();

        doc = nullptr;

        if (!load_template(template_url)) {
          retval = value::make_error("loading template");
          return true;
        }

        if (!load_document(v, self, doc_url)) {
          retval = value::make_error("loading document");
          return true;
        }

        relayout(v, self);

        retval = value(true);

        return true;
      }

      METHOD(2, loadHtml) {
        array<byte> data;

        string doc_url = argv[1].to_string();

        doc = nullptr;

        if (argv[0].is_bytes())
          data = argv[0].get_bytes();
        else if (argv[0].is_string())
          u8::from_utf16(argv[0].get_string(),data,true);
        else {
          retval = value::make_error("bytes or string expected");
          return true;
        }

        load_template(string());
        load_html(v, self, data(), doc_url);

        // relayout(v, self);
        retval = value(true);
        return true;
      }

      METHOD(1, pageNo) {
        int pn  = argv[0].to_int();
        page_no = limit(pn, 1, int(pages.size()));
        do_layout(v, self);
        v.refresh(self);
        retval = int(page_no);
        return true;
      }

      METHOD(0, pageNo) {
        retval = int(this->page_no);
        return true;
      }

      METHOD(1, selectPrinter) {
        ustring              dialog_caption = argv[0].to_string();
        handle<print_target> pt             = v.app->select_printer(&v);
        if (pt) {
          ptarget = pt;
          relayout(v, self);
          retval = value(true);
        }
        return true;
      }
      METHOD(0, selectPrinter) {
        handle<print_target> pt = v.app->select_printer(&v);
        if (pt) {
          ptarget = pt;
          relayout(v, self);
          retval = value(true);
        }
        return true;
      }
      METHOD(0, selectDefaultPrinter) {
        handle<print_target> pt = v.app->select_printer(nullptr);
        if (pt) {
          ptarget = pt;
          relayout(v, self);
          retval = value(true);
        }
        return true;
      }

      METHOD(1, setPrinter) {
        ustring              printer_id = argv[0].to_string();
        handle<print_target> pt         = v.app->select_printer(&v, printer_id);
        if (pt) {
          ptarget = pt;
          relayout(v, self);
          retval = value(true);
        }
        return true;
      }

      METHOD(1, setDocumentName) {
        doc_name = argv[0].to_string();
        return true;
      }

      METHOD(0, print) {
        print(v, self, slice<int>());
        return true;
      }

      METHOD(1, print) {
        value      pages_list = argv[0];
        array<int> pages;
        if (!pages_list.is_array_like()) {
          retval = value::make_error(WCHARS("expecting array of page numbers"));
          return true;
        }
        for (uint i = 0; i < pages_list.size(); ++i)
          pages.push(pages_list.get_element(i).get(0));

        print(v, self, pages());
        return true;
      }

      /*METHOD(1,print)
      {
        //bool dp = argv[0].to_bool();
        //print(v,self,dp);
        return true;
      }

      METHOD(0,getPaperSizes)
      {
        //make_paper_sizes(retval);
        return true;
      }
      METHOD(1,setPaperSize)
      {
        string paper_format = argv[0].to_string();
        if(get_paper_size(paper_format,page_width,page_height))
        {
          invalidate_layout(self);
          on_size_changed(v,self);
        }
        return true;
      }
      METHOD(0,autoScale)
      {
        retval = value(bool(auto_scale != 0));
        return true;
      }
      METHOD(1,autoScale)
      {
        bool as = argv[0].to_bool();
        if( bool(auto_scale != 0) != as )
        {
          auto_scale = as;
          invalidate_layout(self);
          on_size_changed(v,self);
        }
        return true;
      }

      METHOD(0,scale)
      {
        retval = value(scale);
        return true;
      }
      METHOD(1,scale)
      {
        double s = argv[0].to_float();
        if( scale != s )
        {
          scale = s;
          invalidate_layout(self);
          on_size_changed(v,self);
        }
        return true;
      }*/

      METHOD(0, printers) {
        auto list = v.app->get_printer_list();

        value out = value::make_array();
        for (int n = 0; n < list.size(); ++n) {
          auto  pi  = list[n];
          value vpi = value::make_map();
          vpi.set_prop(L"id", value(pi.id));
          vpi.set_prop(L"name", value(pi.name));
          vpi.set_prop(L"shareName", value(pi.share_name));
          vpi.set_prop(L"comment", value(pi.comment));
          vpi.set_prop(L"location", value(pi.location));
          vpi.set_prop(L"isDefault", value(!!pi.is_default));
          out.push(vpi);
        }
        retval = out;
        return true;
      }

#undef METHOD
      return ctl::on_x_method_call(v, self, name, argv, argc, retval);
    }

    bool pager_ctl::print(view &v, element *self, slice<int> page_numbers) {
              
      auto page_renderer_list = [this, &v, self, page_numbers](graphics *gfx, int no) -> bool {
        if (no < 1 || no > page_numbers.size())
          return false;
        int page_no = page_numbers[no - 1];
        if (page_no < 1 || page_no > this->pages.size())
          return false;
        this->print_page(v, self, gfx, page_no);
        return no < page_numbers.size();
      };

      auto page_renderer = [this, &v, self](graphics *gfx, int page_no) -> bool {
        this->print_page(v, self, gfx, page_no);
        return page_no < this->pages.size();
      };

      ustring dn;
      if (doc_name.length())
        dn = doc_name;
      else {
        element *title = find_first(*prv, doc, WCHARS("head>title"));
        value    title_text;
        if (title && title->get_value(*prv, title_text))
          dn = title_text.to_string();
      }
      if (!dn.length()) dn = WTEXT("Sciter Doc");

      if(page_numbers.length)
        return ptarget->print(dn, page_renderer_list);
      else 
        return ptarget->print(dn, page_renderer);
    }
  }
} // namespace html

#else

namespace html {
  namespace behavior {

    void init_pager() {}

  } // namespace behavior
} // namespace html

#endif
