#include "html.h"

namespace html {
  namespace behavior {

    struct menu_bar_ctl_factory : public ctl_factory {
      menu_bar_ctl_factory() : ctl_factory("menu-bar") {}
      virtual ctl *create(element *el) override;
    };

    struct menu_ctl_factory : public ctl_factory {
      menu_ctl_factory() : ctl_factory("menu") {}
      virtual ctl *create(element *el) override;
    };

    struct popup_menu_ctl_factory : public ctl_factory {
      popup_menu_ctl_factory() : ctl_factory("popup-menu") {}
      virtual ctl *create(element *el) override;
    };

    struct popup_selector_ctl_factory : public ctl_factory {
      popup_selector_ctl_factory() : ctl_factory("popup-selector") {}
      virtual ctl *create(element *el) override;
    };

    menu_bar_ctl_factory *      _menu_bar_ctl_factory       = 0;
    menu_ctl_factory *          _menu_ctl_factory           = 0;
    popup_menu_ctl_factory *    _popup_menu_ctl_factory     = 0;
    popup_selector_ctl_factory *_popup_selector_ctl_factory = 0;
    ustring                     role_menu_item = WCHARS("menu-item");

    inline bool is_menu_item(view &v, element *b) {
      if ((b->tag == tag::T_LI) || b->get_attr("-role") == role_menu_item)
        return !b->is_disabled();
      if ((b->tag == tag::T_OPTION) && b->parent &&
          b->parent->tag == tag::T_MENU)
        return !b->is_disabled();
      return false;
    }
    static bool is_visible_menu_item(view &v, element *b) {
      return is_menu_item(v, b) && b->is_visible(v);
    }
    static bool is_current_menu_item(view &v, element *b) {
      return is_menu_item(v, b) && b->state.current();
    }

    static bool skip_submenu(view &v, element *b) {
      return (b->tag == tag::T_MENU);
    }

    element *get_submenu(view &v, element *item) {
      element_iterator cit(v, item);
      for (element *b; cit(b);)
        if (b->tag == tag::T_MENU) return b;
      return 0;
    }

    bool is_submenu(view &v, element *item) {
      return get_submenu(v, item) != 0;
    }

    void prepare_popup(view &v, element *pup) {
      element_iterator bv(v, pup, &is_menu_item, &skip_submenu);
      for (element *t; bv(t);) {
        t->state.current(false);
        if (is_submenu(v, t))
          t->set_attr(attr::a_popup, ustring());
        else
          t->remove_attr(attr::a_popup);
      }
    }

    bool is_visible_menu_item_of(view &v, element *menu, element *t) {
      element_iterator bv(v, menu, &is_visible_menu_item, &skip_submenu);
      for (element *b; bv(b);)
        if (b == t) return true;
      return false;
    }

    element *get_item(view &v, element *self, element *target) {
      if (target == 0) return 0;
      if (is_visible_menu_item_of(v, self, target)) return target;
      return get_item(v, self, target->parent);
    }

    element *get_current_item(view &v, element *self) {
      element_iterator bv(v, self, &is_visible_menu_item, &skip_submenu);
      for (element *b; bv(b);) {
        if (!b) continue;
        if (is_visible_menu_item(v, b) && b->state.current()) return b;
      }
      return 0;
    }

    void setup_current_item(view &v, element *sub, bool by_mouse) {
      if (!by_mouse) {
        element *        bcur = 0;
        element_iterator bni(v, sub, is_menu_item);
        if (bni(bcur)) {
          bcur->state_on(v, S_CURRENT);
          return;
        }
      }
      sub->state_on(v, S_CURRENT);
    }

    struct menu_ctl : ctl {
      typedef ctl super;

      menu_ctl() {}

      virtual CTL_TYPE get_type() override { return CTL_MENU; }

      virtual element* r13n_container(element* self) override { return self; /*reconcilliation is enabled*/ }

      virtual bool focusable(const element *self) override {
        return !self->attr_disabled();
      }

      virtual const string &behavior_name() const override {
        return _menu_ctl_factory->name;
      }

      virtual bool attach(view &v, element *self) override {
        ctl::attach(v, self);
        return true;
      }

      virtual void detach(view &v, element *self) override {
        ctl::detach(v, self);
      }

      void remove_current_item(view &v, element *self, bool by_mouse = true) {
        element *ci = get_current_item(v, self);
        if (!ci) return;
        element *sub = get_submenu(v, ci);
        if (sub && sub->state.popup()) {
          if (by_mouse)
            v.start_timer(sub, SHOOT_SUBMENU_TIMER_MS, SHOOT_SUBMENU_TIMER_ID,
                          INTERNAL_TIMER);
          else
            v.close_popup(sub, false);
        }
        ci->state_off(v, S_CURRENT);
      }

      bool set_current_item(view &v, element *self, element *item,
                            bool by_mouse = true) {
        element *ci = get_current_item(v, self);
        if (ci == item) return false;

        if (ci) {
          /*#ifdef _DEBUG
                  if (ci->state.owns_popup())
                    ci = ci;
          #endif*/
          element *sub = get_submenu(v, ci);
          if (sub && sub->state.popup()) {
#ifdef HAS_MOUSE
            if (by_mouse)
              v.start_timer(sub, SHOOT_SUBMENU_TIMER_MS, SHOOT_SUBMENU_TIMER_ID,
                            INTERNAL_TIMER);
            else
#endif
              v.close_popup(sub, true);
          }
          // if(item)
          ci->state_off(v, S_CURRENT);
          if (!sub) set_focus_on_child_control(v, self, ci, false);
        }
        if (item) {
          if (!item->state.current()) {
            item->state_on(v, S_CURRENT);
            notify_current(v, self, item, by_mouse);
            v.update_element(item);
          }
          element *sub = get_submenu(v, item);
          if (!by_mouse && sub && !sub->state.popup()) {
            prepare_popup(v, sub);
            v.show_popup(sub, item, POPUP_WINDOW, sub_location(v, sub));
            // v.start_timer(sub,400);
            return true;
          }
        }
        return true;
      }

#ifdef HAS_MOUSE
      virtual bool on_timer(view &v, element *self, timer_id id,
                            TIMER_KIND kind) override {
        if (kind != INTERNAL_TIMER) return super::on_timer(v, self, id, kind);
        if (id == SHOOT_SUBMENU_TIMER_ID) {
          if (!self->state.hover() && self->state.popup()) {
            if (!self->parent || !self->parent->state.current())
              v.close_popup_tree(self);
            else
              return true;
          }
        } else if (id == PRESENT_SUBMENU_TIMER_ID) {
          if (self->state.popup()) {
            element *item = get_current_item(v, self);
            element *sub  = item ? get_submenu(v, item) : 0;
            if (sub && !sub->state.popup()) {
              prepare_popup(v, sub);
              v.show_popup(sub, item, POPUP_WINDOW, sub_location(v, sub));
              return true;
            }
          }
        }
        return false;
      }
#endif

      element *get_root(view &v, element *self) {
        while (self) {
          if (self->get_named_behavior(_menu_bar_ctl_factory->name) ||
              self->get_named_behavior(_popup_menu_ctl_factory->name)) {
            break;
          }
          self = self->parent;
        }
        return self;
      }

      void set_focus_on_child_control(view &v, element *self, element *item,
                                      bool on) {
        if (!item) return;
        if (is_submenu(v, item)) return;
        if (on) {
          element *child = find_first(v, item, WCHARS(":focusable"));
          if (child) v.set_focus(child, BY_CODE, true);
        }
        /*else
        {
          element *child = find_first(item,L":focus");
          if( child )
          {
            element* r = get_root(v,self);
            if(r)
              v.set_focus(r,BY_CODE,true);
          }
        }*/
      }

      bool set_next_current_item(view &v, element *self, bool next) {
        element *ci = get_current_item(v, self);

        if (next) {
          element_iterator bv(v, self, &is_visible_menu_item, &skip_submenu);
          if (ci) {
            bv.rewind(ci);
            element *sel = 0;
            if (bv(sel)) {
              set_current_item(v, self, sel);
              set_focus_on_child_control(v, self, sel, true);
              return true;
            }
          }
          bv.rewind();
          element *sel;
          if (bv(sel)) {
            set_current_item(v, self, sel);
            set_focus_on_child_control(v, self, sel, true);
          }
          return sel != 0;
        } else {
          element_iterator bv(false, v, self, &is_visible_menu_item,
                              &skip_submenu);
          if (ci) {
            bv.rewind(ci);
            element *sel = 0;
            if (bv(sel)) {
              set_current_item(v, self, sel);
              set_focus_on_child_control(v, self, sel, true);
              return true;
            }
          }
          element *sel = 0;
          if (bv(sel)) {
            set_current_item(v, self, sel);
            set_focus_on_child_control(v, self, sel, true);
          }
          return sel != 0;
        }
      }

      /*virtual bool on(view& v, element* self, event_focus& evt ) override
      {
        if(  self->state.popup()
          && (evt.cmd == FOCUS_OUT ||
               evt.cmd == (FOCUS_OUT | EVENT_SINKING)) )
        {
  #ifdef _DEBUG
          if(evt.target && evt.target->parent) {
            dbg_printf("focus %s\n", tag::symbol_name(evt.target->tag).c_str());
            dbg_printf("focus parent %s\n",
  tag::symbol_name(evt.target->parent->tag).c_str());
          }
  #endif
          if( !evt.target || !evt.target->belongs_to(v,self, true) )
            v.close_popup(self);
        }
        return false;
      }*/

      virtual bool on(view &v, element *self, event_mouse &evt) override {
        switch (evt.cmd) {
        case MOUSE_DCLICK:
        case MOUSE_DOWN: {
          element *item = get_item(v, self, evt.target);
          if (evt.is_point_button() && item) {
            set_current_item(v, self, item);
            return true;
          }
        }
          return true;
        case MOUSE_UP: {
          // element* item = get_item(self, self->find_block(evt.pos,v));
          element *item         = get_item(v, self, evt.target);
          element *current_item = get_current_item(v, self);
          if (evt.is_point_button()) {
            if (current_item && current_item == item) {
              do_click(v, self, current_item, true);
              return true;
            }
            set_current_item(v, self, 0, true);
          }
        } break;

        case MOUSE_MOVE: {
          element *item = get_item(v, self, evt.target);
          if (item && evt.is_point_button()) {
            v.ensure_visible(item, false, SCROLL);
            v.update_element(item);
          }
          set_current_item(v, self, item, true);
          element *sub = item ? get_submenu(v, item) : 0;
          if (sub && !sub->state.popup())
            v.start_timer(self, PRESENT_SUBMENU_TIMER_MS,
                          PRESENT_SUBMENU_TIMER_ID, INTERNAL_TIMER);
          else
            v.stop_timer(self, PRESENT_SUBMENU_TIMER_ID, INTERNAL_TIMER);
        } break;

          /*case MOUSE_IDLE:
            if(self->state.popup())
            {
              element* item = get_current_item(self);
              element *sub = item? get_submenu( item ):0;
              if(sub && !sub->state.popup())
              {
                prepare_popup(sub);
                v.show_popup( sub, item, sub_location(), false );
                //v.start_timer(sub,400);
                return true;
              }
            }
            break;*/

        case MOUSE_LEAVE | DRAGGING: {
          if (evt.dragging && self->state.popup()) {
            v.close_popup(self, false);
          }
        } break;
#ifdef HAS_MOUSE
        case MOUSE_ENTER:
          v.stop_timer(self, SHOOT_SUBMENU_TIMER_ID, INTERNAL_TIMER);
          break;

        case MOUSE_LEAVE:
          remove_current_item(v, self, true);
          v.stop_timer(self, PRESENT_SUBMENU_TIMER_ID, INTERNAL_TIMER);
          break;
#endif
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_key &evt) override {
        if (evt.has_modifiers()) return false;

        element *ci  = get_current_item(v, self);
        element *sub = ci ? get_submenu(v, ci) : 0;

        if (sub && sub->state.popup() && sub->state.focusable() &&
            sub->on(v, evt)) {
          return true;
        } else if (!sub && ci && ci->on(v, evt)) {
          return true;
        }

        switch (evt.cmd) {
        case KEY_DOWN:
          switch (evt.key_code) {
          case KB_RETURN: {
            return do_click(v, self, get_current_item(v, self));
          }
          case KB_UP: return set_next_current_item(v, self, false);
          case KB_DOWN: return set_next_current_item(v, self, true);
          case KB_RIGHT: {
            if (sub) {
              if (!sub->state.popup()) {
                prepare_popup(v, sub);
                v.show_popup(sub, ci, POPUP_WINDOW, sub_location(v, sub));
              }
              // if( !sub->state.focusable() )
              {
                sub->state.focusable(true);
                event_key e = evt;
                e.key_code  = KB_DOWN;
                return sub->on(v, e);
              }
              // else
              //  return false;
            }
            return false;
          }
          case KB_ESCAPE: {
            if (self->state.popup()) {
              self->state.focusable(false);
              v.close_popup(self, true);
              return true;
            }
          } break;

          case KB_LEFT: {
            if (sub && sub->state.popup()) {
              sub->state.focusable(false);
              v.close_popup(sub, false);
              v.set_focus(self, BY_KEY_PREV);
              return true;
            }
          }

          break;
          }
          break;
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_focus &evt) override {
        if (self->state.popup() && (evt.cmd == FOCUS_OUT ||
                                    evt.cmd == (FOCUS_OUT | EVENT_HANDLED))) {
          // dbg_printf("lost focus %s\n", (const char
          // *)symbol_name(evt.target->tag));
          if (!evt.target || !evt.target->belongs_to(v, self, true)) {
            v.close_popup(self, false);
            v.stop_timer(self, PRESENT_SUBMENU_TIMER_ID, INTERNAL_TIMER);
            v.stop_timer(self, SHOOT_SUBMENU_TIMER_ID, INTERNAL_TIMER);
          }
        }
        return false;
      }

      /*
      virtual bool on  (view& v, element* self, event_behavior& evt )
      {
        if( evt.cmd == POPUP_DISMISSED)
        {
          //if(!evt.target)
          //  return false;
          //if( !evt.target->parent->state.hover() )
            //set_current_item(v,self,0,false);
        }
        return false;
      }
      */
      /* virtual bool on(view& v, element* self, event_behavior& evt ) override
      {
        switch( evt.cmd ) {
          case MENU_ITEM_CLICK | EVENT_SINKING:
            if(evt.target->parent == self)
            {
              helement pup = self;
              v.close_popup_tree(pup);
              if( pup->parent )
                close_popup_parents(v,self);
            } break;
        }
        return false;
      }*/

      bool drop_menu(view *v, helement item, helement self) {
        if (item->parent) v->close_popup_tree(item->parent);
        if (self->parent) close_popup_parents(*v, self);
        return true;
      }

      virtual bool notify_current(view &v, element *self, element *item,
                                  bool by_mouse) {
        event_behavior evt(self, item, MENU_ITEM_ACTIVE,
                           by_mouse ? CLICK_BY_MOUSE : CLICK_BY_KEY);
        v.post_behavior_event(evt);
        return true;
      }
      virtual bool do_click(view &v, element *self, element *item,
                            bool by_mouse = false) {
        if (item && !item->is_disabled()) {
          helement a   = v.popup_anchor(self);
          element *sub = get_submenu(v, item);
          if (sub) {
            if (!sub->state.popup()) {
              prepare_popup(v, sub);
              v.show_popup(sub, item, POPUP_WINDOW, sub_location(v, sub));
            }
          } else {
            handle<view>    v_holder    = &v;
            handle<element> self_holder = self;
            set_current_item(v, self, 0, by_mouse);
            event_behavior evt(a, item, MENU_ITEM_CLICK,
                               by_mouse ? CLICK_BY_MOUSE : CLICK_BY_KEY);
            v.send_behavior_event(evt);
            v.post(delegate(this, &menu_ctl::drop_menu, &v, item, self), true);
            /*
                      //v.post_behavior_event( evt );
                      //v.process_posted_things(true);
                      if( item->parent )
                        v.close_popup_tree(item->parent);
                      if( self->parent )
                        close_popup_parents(v,self);*/
          }
        }
        return true;
      }

      void close_popup_parents(view &v, element *self) {
        while (self) {
          if (self->state.popup()) {
            if (!self->get_named_behavior("menu")) return; // stop it
            v.close_popup_tree(self);
            v.stop_timer(self, PRESENT_SUBMENU_TIMER_ID, INTERNAL_TIMER);
            v.stop_timer(self, SHOOT_SUBMENU_TIMER_ID, INTERNAL_TIMER);
            close_popup_parents(v, self->parent);
          }
          self = self->parent;
        }
      }

      virtual int sub_location(view &v, element *menu) {
        // if( menu->get_style(v)->direction == direction_rtl )
        //  return 0x14;
        // else
        return 0x19;
      }
    };

    struct menu_bar_ctl : ctl {
      typedef ctl super;

      element* pre_keyboard_item = nullptr;

      menu_bar_ctl() {}

      virtual CTL_TYPE get_type() override { return CTL_MENUBAR; }

      virtual bool focusable(const element *self) override {
        return !self->attr_disabled();
      }

      virtual element* r13n_container(element* self) override { return self; /*reconcilliation is enabled*/ }

      virtual const string &behavior_name() const override {
        return _menu_bar_ctl_factory->name;
      }

      virtual bool attach(view &v, element *self) override {
        ctl::attach(v, self);
        return true;
      }

      virtual void detach(view &v, element *self) override {
        ctl::detach(v, self);
      }

      virtual int sub_location(view &v, element *menu) { return 0x12; }

      virtual void show_popup(view &v, element *self, element *item,
                              bool by_mouse, bool animate = true) {
        element *sub = item ? get_submenu(v, item) : 0;
        if (sub && !sub->state.popup()) {
          prepare_popup(v, sub);
          sub->state.focusable(true);
          v.show_popup(sub, item, POPUP_WINDOW, sub_location(v, sub));
          // v.set_focus( sub, by_mouse? BY_MOUSE:BY_CODE );
          setup_current_item(v, sub, by_mouse);
        }
      }

      virtual bool on(view &v, element *self, event_key &evt) override {
        // if(evt.alt_state)
        //  return false;

        element *ci  = get_current_item(v, self);
        element *sub = ci ? get_submenu(v, ci) : 0;

        if (sub && sub->state.popup() && sub->state.focusable() && sub->on(v, evt)) {
          return true;
        } else if (!sub && ci && ci->on(v, evt)) {
          return true;
        }

        switch (evt.cmd) {
        case KEY_DOWN:
          switch (evt.key_code) {
          case KB_DOWN:
          case KB_RETURN:
          //case KB_SPACE: 
          {
            if(!pre_keyboard_item)
              pre_keyboard_item = ci;
            if (sub)
              show_popup(v, self, ci, false);
            else if (ci)
              return do_click(v, self, ci);
          } break;
          case KB_LEFT:  if (!pre_keyboard_item) pre_keyboard_item = ci; return set_next_current_item(v, self, false);
          case KB_RIGHT:  if (!pre_keyboard_item) pre_keyboard_item = ci; return set_next_current_item(v, self, true);
          }
          break;
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_behavior &evt) override {
        if (evt.cmd == POPUP_DISMISSING) {
          element *citem = get_current_item(v, self);
          if (!citem) return false;

          if (evt.target->parent != citem) return false;

          if (!citem->state.hover()) set_current_item(v, self, 0, true, false);
        } else if (evt.cmd == ACTIVATE_CHILD) {
          if (!evt.target->belongs_to(self)) return false;
          if (evt.target->tag != tag::T_LI) return false;
          if (evt.target->is_disabled()) return false;
          do_click(v, self, evt.target);
          return true;
        }

        return false;
      }

      virtual bool on(view &v, element *self, event_mouse &evt) override {
        switch (evt.cmd) {
        case MOUSE_DCLICK:
        case MOUSE_DOWN: {
          element *item = get_item(v, self, evt.target);
          if (evt.is_point_button() && item && !item->state.owns_popup()) {
            v.set_focus(self, BY_MOUSE);
            set_current_item(v, self, item, true, true);
          }
        }
          return true;
        case MOUSE_UP: {
          // element* item = get_item(self, self->find_block(evt.pos,v));
          element *item         = get_item(v, self, evt.target);
          element *current_item = get_current_item(v, self);
          if (evt.is_point_button()) {
            if (current_item && current_item == item) {
              if (!is_submenu(v, item)) do_click(v, self, current_item, true);
              return true;
            }
            // set_current_item(v, self, 0, true, false);
          }
        } break;
        case MOUSE_LEAVE: {
          element *citem = get_current_item(v, self);
          if (citem && !citem->state.owns_popup())
            set_current_item(v, self, 0, true, false);
        } break;
#if 0
        case MOUSE_MOVE | DRAGGING:
          if (evt.dragging) {
            /*element* item = get_item(v, self, evt.target);
            if( item && !item->state.owns_popup() && is_menu_item(v,item) )
            {
              element *sub = get_submenu( v, item );
              if(sub && sub->is_active_dd_target(v,evt.dragging))
                set_current_item(v,self,item, true, true, true);
            }*/
          }
          break;
#endif
        case MOUSE_MOVE: {
          element *hitem  = get_item(v, self, evt.target);
          element *citem = get_current_item(v, self);
          if (pre_keyboard_item == hitem)
            break;
          pre_keyboard_item = nullptr;
          if (hitem != citem && hitem && citem && is_menu_item(v, hitem)) {
            set_current_item(v, self, hitem, true, true, evt.cmd == (MOUSE_MOVE | DRAGGING));
          } else if (!hitem && citem && !citem->state.owns_popup()) {
            set_current_item(v, self, 0, true, false);
          }
        } break;

          /*
          case MOUSE_ENTER:
            v.stop_timer(self);
            break;

          */
        }
        return false;
      }

      struct delayed_show_popup : functor {
        handle<menu_bar_ctl> ctl;
        view &               v;
        helement             self;
        helement             item;
        bool                 by_mouse;
        bool                 animate;
        delayed_show_popup(menu_bar_ctl *this_, view &v_, element *self_,
                           element *item_, bool by_mouse_, bool animate_)
            : ctl(this_), v(v_), self(self_), item(item_), by_mouse(by_mouse_),
              animate(animate_) {}
        virtual bool operator()() override {
          ctl->show_popup(v, self, item, by_mouse, animate);
          return true; // complete
        }
      };

      bool set_current_item(view &v, element *self, element *item,
                            bool by_mouse, bool touch_popup,
                            bool dragging = false) {
        element *ci = get_current_item(v, self);
        // if( ci == item )
        //  return false;

        bool popup_shown = false;

        if (ci) {
          if (ci->state.current()) ci->state_off(v, S_CURRENT);
          element *sub = get_submenu(v, ci);
          if (sub && sub->state.popup()) {
            popup_shown = true;
            if (touch_popup && !dragging) v.close_popup(sub, false);
            // dbg_printf("popup off=%d %d\n", sub, touch_popup);
          }
        }
        if (item) {
          if (!item->state.current()) {
            item->state_on(v, S_CURRENT);
            notify_current(v, self, item, by_mouse);
          }

          element *sub = get_submenu(v, item);

          // dbg_printf("popup=%d %d %d\n", sub, popup_shown , touch_popup);
          bool has_focus = v.focus_element
                               ? v.focus_element->belongs_to(v, self, true)
                               : false; // self->state.focus();

          if (sub && touch_popup && (dragging || has_focus)) {
            // show_popup(v,self, item, by_mouse, !dragging);
            v.post(new delayed_show_popup(this, v, self, item, by_mouse,
                                          !dragging),
                   false);
            return true;
          }
        }
        return true;
      }

      bool set_next_current_item(view &v, element *self, bool next) {
        helement ci = get_current_item(v, self);
        v.set_focus(self, BY_CODE);

        if (next) {
          element_iterator bv(v, self, &is_visible_menu_item, &skip_submenu);
          if (ci) {
            bv.rewind(ci);
            element *sel = 0;
            if (bv(sel)) {
              set_current_item(v, self, sel, false, true);
              return true;
            }
          }
          bv.rewind();
          element *sel = 0;
          bv(sel);
          set_current_item(v, self, sel, false, true);
          return sel != 0;
        } else {
          element_iterator bv(false, v, self, &is_visible_menu_item,
                              &skip_submenu);
          if (ci) {
            bv.rewind(ci);
            element *sel = 0;
            if (bv(sel)) {
              set_current_item(v, self, sel, false, true);
              return true;
            }
          }
          element *sel = 0;
          bv(sel);
          set_current_item(v, self, sel, false, true);
          return sel != 0;
        }
      }

      bool notify_current(view &v, element *self, element *item,
                          bool by_mouse) {
        event_behavior evt(self, item, MENU_ITEM_ACTIVE,
                           by_mouse ? CLICK_BY_MOUSE : CLICK_BY_KEY);
        v.post_behavior_event(evt);
        return true;
      }
      bool do_click(view &v, element *self, element *item,
                    bool by_mouse = false) {
        if (item) {
          event_behavior evt(self, item, MENU_ITEM_CLICK,
                             by_mouse ? CLICK_BY_MOUSE : CLICK_BY_KEY);
          v.post_behavior_event(evt);
        }
        return true;
      }
    };

    struct popup_menu_ctl : ctl {
      bool popup_is_active;

      popup_menu_ctl() : popup_is_active(false) {}

      virtual const string &behavior_name() const override {
        return _popup_menu_ctl_factory->name;
      }

      virtual CTL_TYPE get_type() override { return CTL_MENUBUTTON; }

      virtual element* r13n_container(element* self) override { return self; /*reconcilliation is enabled*/ }

      virtual bool attach(view &v, element *self) override {
        ctl::attach(v, self);
        return true;
      }

      virtual void detach(view &v, element *self) override {
        ctl::detach(v, self);
      }

      virtual bool on_timer(view &v, element *self, timer_id tid, TIMER_KIND kind) { 
        if(tid == SHOOT_SUBMENU_TIMER_ID && kind == INTERNAL_TIMER)
          popup_is_active = false;
        return false; /*stop this timer*/ 
      }

      virtual bool on(view &v, element *self, event_behavior &evt) override {
        if (evt.cmd == POPUP_READY) {
          popup_is_active = true;
          return false;
        }
        if (evt.cmd == POPUP_DISMISSED) {
          v.start_timer(self, SHOOT_SUBMENU_TIMER_MS, SHOOT_SUBMENU_TIMER_ID, INTERNAL_TIMER);
          return false;
        }
        if (evt.cmd == MENU_ITEM_CLICK) {
          v.set_focus(self, BY_CODE);
          return false;
        }
        if (evt.target != self) return false;
        if (evt.cmd == BUTTON_PRESS) {
          if (evt.target->belongs_to(get_popup(v, self))) return false;
          if(!popup_is_active)
            show_popup(v, self, false, true);
          return true;
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_focus &evt) override {
        helement hpup = get_popup(v, self);
        if (hpup && hpup->state.popup() &&
            (evt.cmd == FOCUS_OUT ||
             evt.cmd == (FOCUS_OUT | EVENT_HANDLED))) {
          // dbg_printf("lost focus %s\n", (const char
          // *)symbol_name(evt.target->tag));
          if (!evt.target || !evt.target->belongs_to(v, hpup, true)) {
            v.close_popup(hpup, false);
            v.stop_timer(hpup, PRESENT_SUBMENU_TIMER_ID, INTERNAL_TIMER);
            v.stop_timer(hpup, SHOOT_SUBMENU_TIMER_ID, INTERNAL_TIMER);
          }
        }
        return false;
      }

      virtual bool on_method_call(view &v, element *self,
                                  method_params *p) override {
        switch (p->method_id) {
        case DO_CLICK: 
          show_popup(v, self, false, true); 
          return true;
        }
        return false;
      }

#if 0
      virtual bool on(view &v, element *self, event_mouse &evt) override {
        // if(!self->is_focusable())
        //  return false;

        switch (evt.cmd) {
        case MOUSE_DCLICK:
        case MOUSE_DOWN:
          if (evt.is_point_button()) {
            v.refresh(self);
            // v.set_capture(self);
            if (!evt.target->belongs_to(get_popup(v, self))) {
              if (popup_is_active) {
                element *pup = get_popup(v, self);
                if (pup) v.close_popup(pup, false);
                //popup_was_active = false;
              } else
                //popup_was_active = self->state.owns_popup();
              return true;
            }
          }
          break;
        case MOUSE_UP:
          if (evt.is_point_button()) {
            v.refresh(self);
            if (self->state.pressed() && self->state.hover() &&
                !evt.target->belongs_to(get_popup(v, self)) &&
                !popup_is_active) {
              self->state.pressed(false);
              show_popup(v, self, true, true);
              return true;
            }
          }
          break;

        case MOUSE_MOVE:
          if (evt.is_point_button()) {
            if (self->state.hover()) {
              if (!self->rendering_box(v).contains(evt.pos)) {
                self->state.hover(false);
                self->state_off(v, S_ACTIVE);
                //if (!self->state.owns_popup()) popup_was_active = false;
              }
            } else {
              if (self->rendering_box(v).contains(evt.pos)) {
                self->state.hover(true);
                self->state_on(v, S_ACTIVE);
              }
            }
          }
          break;
        case MOUSE_LEAVE: {
          self->state.pressed(false);
        } break;
        case MOUSE_WHEEL: return true;
        case MOUSE_ENTER | DRAGGING: {
          if (evt.dragging && !self->state.owns_popup()) {
            /*element* pup = get_popup(v,self);
            if( pup && pup->is_active_dd_target(v,evt.dragging) )
            {
              show_popup(v,self, true, false);
              v.update_element(self);
            }*/
          }
        } break;
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_key &evt) {
        // if(evt.alt_state)
        //  return false;

        helement hpup = get_popup(v, self);
        if (hpup && hpup->state.popup() && hpup->state.focusable() &&
            hpup->on(v, evt)) {
          return true;
        }

        switch (evt.cmd) {
        case KEY_DOWN | EVENT_SINKING:
          switch (evt.key_code) {
          case KB_DOWN:
          case KB_RETURN:
          case KB_SPACE: {
            helement hpup = get_popup(v, self);
            if (hpup) {
              if (!hpup->state.popup()) {
                show_popup(v, self, false, true);
                return true;
              }
              return hpup->on(v, evt);
            }
          }
          }
          break;
        }
        return false;
      }

#endif

      virtual void prepare_popup(view &v, element *self, element *pup) {
        element_iterator bv(v, pup, &is_menu_item, &skip_submenu);
        for (element *t; bv(t);) {
          t->state.current(false);
          if (is_submenu(v, t))
            t->set_attr(attr::a_popup, ustring());
          else
            t->remove_attr(attr::a_popup);
        }
      }

      virtual void show_popup(view &v, element *self, bool by_mouse,
                              bool animate) {
        helement hpup = get_popup(v, self);
        if (hpup && !hpup->state.popup()) {
          prepare_popup(v, self, hpup);
          int     alignment = 2;
          ustring align     = get_attr(self, "-align-popup");
          if (align == WCHARS("left"))
            alignment = 0x17;
          else if (align == WCHARS("right"))
            alignment = 0x19;
          else if (align == WCHARS("bottom"))
            alignment = 2;
          else if (align == WCHARS("top"))
            alignment = 8;
          v.show_popup(hpup, self, POPUP_WINDOW, alignment);
          setup_current_item(v, hpup, by_mouse);
          if (!by_mouse)
            v.set_focus(hpup, BY_CODE, true);
        }
      }

      /*virtual element* get_popup(view& v, element* self)
      {
        return find_first(self,L"popup,menu");
      }*/
      virtual element *get_popup(view &v, element *self) {
        element *b = find_first(v, self, WCHARS("popup,menu"), false);
        if (b) return b;
        ustring selector = self->atts("menu");
        if (selector.length()) b = find_first(v, self->doc(), selector, false);
        return b;
      }

      virtual element *get_current_item(view &v, element *self) {
        element *        pup = get_popup(v, self);
        element_iterator bv(v, pup, &is_visible_menu_item);
        for (element *b; bv(b);) {
          if (!b) continue;
          if (is_visible_menu_item(v, b) && b->state.current()) return b;
        }
        return 0;
      }
    };

    struct popup_selector_ctl : popup_menu_ctl {
      typedef popup_menu_ctl super;

      popup_selector_ctl() {}

      virtual bool focusable(const element *self) override { return true; }

      virtual const string &behavior_name() const override {
        return _popup_selector_ctl_factory->name;
      }

      virtual CTL_TYPE get_type() override { return CTL_MENUBUTTON; }

      virtual bool attach(view &v, element *self) override {
        ctl::attach(v, self);
        helement first = self->first_element();
        if (!first || first->tag != tag::T_CAPTION) {
          element *caption = new element(tag::T_CAPTION);
          self->insert(0, caption);
          set_caption(v, self);
        }
        return true;
      }

      virtual void detach(view &v, element *self) override {
        ctl::detach(v, self);
      }

      void notify_changed(view &v, element *self, uint reason) {
        event_behavior evt(self, self, BUTTON_STATE_CHANGED, reason);
        v.post_behavior_event(evt);
      }

      element *get_caption(element *self) {
        helement first = self->first_element();
        if (first && first->tag == tag::T_CAPTION) return first;
        return 0;
      }

      void prepare_popup(view &v, element *self, element *pup) {
        ustring val = self->atts(attr::a_value);
        pup->state.synthetic(true);
        if (val.length()) {
          element_iterator bv(v, pup, &is_visible_menu_item);
          for (element *t; bv(t);) {
            t->state.current(false);
            if (val == t->atts(attr::a_value)) {
              t->state.current(true);
              t->state.checked(true);
              // v.ensure_visible(t,false, SET_POS);
            }
          }
        }
      }

      void set_caption(view &v, element *self) {
        element *pup = get_popup(v, self);
        element *cap = get_caption(self);
        ustring  val = self->atts(attr::a_value);
        if (pup && cap && val.length()) {
          element_iterator bv(v, pup, &is_menu_item);
          for (element *t; bv(t);) {
            if (val == t->atts(attr::a_value)) {
              cap->copy_content_from(v, t);
              v.add_to_update(self, true);
              break;
            }
          }
        }
      }

      virtual bool set_value(view &v, element *self,
                             const value &val) override {
        element *pup = get_popup(v, self);
        element *cap = get_caption(self);
        if (pup && cap) {
          element *        item = 0;
          ustring          sv   = val.to_string();
          element_iterator bv(v, pup, &is_menu_item);
          for (element *t; bv(t);) {
            if (sv == t->atts(attr::a_value)) {
              item = t;
              break;
            }
          }
          if (item) {
            self->atts.set(attr::a_value, sv);
            cap->copy_content_from(v, item);
            v.add_to_update(self, true);
            return true;
          }
        }
        return false;
      }

      virtual bool get_value(view &v, element *self, value &val) override {
        val = value::parse(self->atts(attr::a_value));
        return true;
      }

      /*virtual bool on(view& v, element* self, event_mouse& evt )
      {
        if(!self->is_enable())
          return false;
        return super::on(v,self,evt);
      }*/

      virtual bool on(view &v, element *self, event_behavior &evt) override {
        if (evt.cmd == MENU_ITEM_CLICK) {
          element *cap = get_caption(self);
          if (evt.source && cap) {
            ustring val = evt.target->atts(attr::a_value);
            if (val.length()) {
              self->atts.set(attr::a_value, val);
              cap->copy_content_from(v, evt.target);
              v.add_to_update(cap, true);
              notify_changed(v, self, uint(evt.reason));
            }
          }
          return true;
        }
        if (evt.cmd == POPUP_READY) {
          element *pup = get_popup(v, self);
          if (pup) {
            element *c = find_first(v, pup, WCHARS(":current"), false, false);
            if (c) v.ensure_visible(c, false, SET_POS);
          }
          return false;
        }
        return false;
      }

      virtual bool get_auto_width(view &v, element *self, int &value) override {
        element *cap = get_caption(self);
        element *pup = get_popup(v, self);
        if (pup && cap) {
          int intrinsic_width =
              max(pup->max_content_width(v), cap->min_width(v));
          // intrinsic_width += cap->content_left() + cap->content_right();
          value = intrinsic_width;
          return true;
        }
        return false;
      }
    };

    ctl *menu_ctl_factory::create(element *el) { return new menu_ctl(); }

    ctl *menu_bar_ctl_factory::create(element *el) {
      return new menu_bar_ctl();
    }

    ctl *popup_menu_ctl_factory::create(element *el) {
      return new popup_menu_ctl();
    }

    ctl *popup_selector_ctl_factory::create(element *el) {
      return new popup_selector_ctl();
    }

    void init_menu() {
      ctl_factory::add(_menu_bar_ctl_factory = new menu_bar_ctl_factory());
      ctl_factory::add(_menu_ctl_factory = new menu_ctl_factory());
      ctl_factory::add(_popup_menu_ctl_factory = new popup_menu_ctl_factory());
      ctl_factory::add(_popup_selector_ctl_factory =
                           new popup_selector_ctl_factory());
    }

  } // namespace behavior
} // namespace html