#include "html.h"

#if defined(NATIVE_OSX_BACKEND)
gool::bitmap *osx_get_icon_for_path(const ustring &path, gool::size sz, bool must_exist);
#endif

namespace html {
  namespace behavior {

    struct fileicon_ctl_factory : public ctl_factory {
      fileicon_ctl_factory() : ctl_factory("file-icon") {}
      virtual ctl *create(element *) override;
    };

    struct shellicon_ctl_factory : public ctl_factory {
      shellicon_ctl_factory() : ctl_factory("shell-icon") {}
      virtual ctl *create(element *) override;
    };

    struct filethumbnail_ctl_factory : public ctl_factory {
      filethumbnail_ctl_factory() : ctl_factory("file-thumbnail") {}
      virtual ctl *create(element *) override;
    };

    static fileicon_ctl_factory *     _fileicon_ctl_factory      = 0;
    static shellicon_ctl_factory *    _shellicon_ctl_factory     = 0;
    static filethumbnail_ctl_factory *_filethumbnail_ctl_factory = 0;

#if defined(NATIVE_WINDOWS_BACKEND)

#define DECLARE_ICON_LIST(VARNAME, TYPE)                                       \
  static icon::file_icon_list *VARNAME(bool release = false) {                 \
    static icon::file_icon_list *list = 0;                                     \
    if (!list && !release) {                                                   \
      list = new icon::file_icon_list(icon::file_icon_list::TYPE);             \
    } else if (list && release) {                                              \
      delete list;                                                             \
      list = 0;                                                                \
    }                                                                          \
    return list;                                                               \
  }

    DECLARE_ICON_LIST(xx_large_file_icons, XX_LARGE)
    DECLARE_ICON_LIST(x_large_file_icons, X_LARGE)
    DECLARE_ICON_LIST(large_file_icons, LARGE)
    DECLARE_ICON_LIST(small_file_icons, SMALL)
    DECLARE_ICON_LIST(sys_small_file_icons, SYSTEM)

#elif defined(NATIVE_OSX_BACKEND)

    template <unsigned SZ> bitmap *get_icon_for_path(const ustring &path, bool must_exist) {
      static hash_table<ustring, hbitmap> file_icons;
      bool                                created = false;
      hbitmap &bm = file_icons.get_ref(path, created);
      if (created) bm = osx_get_icon_for_path(path, gool::size(SZ),must_exist);
      return bm.ptr();
    }

#endif

    void shutdown_ctl_image() {
#if defined(NATIVE_WINDOWS_BACKEND)
      xx_large_file_icons(true);
      x_large_file_icons(true);
      large_file_icons(true);
      small_file_icons(true);
      sys_small_file_icons(true);

#elif defined(NATIVE_OSX_BACKEND)

#endif
    }

#ifndef INVALID_FILE_ATTRIBUTES // wince does not have it
#define INVALID_FILE_ATTRIBUTES UINT(-1)
#endif

    struct fileicon_ctl : public ctl {
      int_v          factor;
      handle<bitmap> img;
      ustring        filename;
      tristate_v     attempt_made;

      enum FI_SIZE { SYS_DEFAULT, SMALL, LARGE, X_LARGE, XX_LARGE };

      virtual CTL_TYPE get_type() { return CTL_UNKNOWN; }

      fileicon_ctl() : factor() {}
      virtual const string &behavior_name() const override {
        return _fileicon_ctl_factory->name;
      }

      virtual bool attach(view &v, element *self) override {
        return ctl::attach(v, self);
      }
      virtual void detach(view &v, element *self) override {
        return ctl::detach(v, self);
      }

      virtual element* r13n_container(element* self) override { return self; /*reconcilliation is enabled*/ }

      virtual bool is_fore_image_provider(view &v) const { return true; }
      virtual bool null_if_unknown() const { return false; }

      // allows to read icon from real files
      virtual bool allow_read_from_fs() const { return true; }

      virtual void fetch_image(view &v, element *self) {
#if defined(NATIVE_WINDOWS_BACKEND)
        switch (factor) {
        default:
        case SYS_DEFAULT:
          img = sys_small_file_icons()->get(filename, allow_read_from_fs());
          break;
        case SMALL:
          img = small_file_icons()->get(filename, allow_read_from_fs());
          break;
        case LARGE:
          img = large_file_icons()->get(filename, allow_read_from_fs());
          break;
        case X_LARGE:
          img = x_large_file_icons()->get(filename, allow_read_from_fs());
          break;
        case XX_LARGE:
          img = xx_large_file_icons()->get(filename, allow_read_from_fs());
          break;
        }
#elif defined(NATIVE_OSX_BACKEND)
        // img = get_icon_for_path( filename );
        switch (factor) {
        default:
        case SMALL:
        case SYS_DEFAULT: img = get_icon_for_path<16>(filename,allow_read_from_fs()); break;
        case LARGE: img = get_icon_for_path<32>(filename,allow_read_from_fs()); break;
        case X_LARGE: img = get_icon_for_path<48>(filename,allow_read_from_fs()); break;
        case XX_LARGE: img = get_icon_for_path<128>(filename,allow_read_from_fs()); break;
        }
#endif
      }

      virtual bool is_loading(view &v, element *self) { return false; }

      virtual image *get_fore_image(view &v, element *self) override {
        // if(is_loading(v,self))
        //  return 0;

        // sb->draw(*v, self, sf, rc);
        ustring ufilename = get_attr(self, "-filename");

        if (ufilename == WCHARS(" ")) {
          img      = 0;
          filename = ufilename;
          return 0;
        }

        if (!ufilename.length()) {
          if (self->tag == tag::T_TEXT)
            ufilename = get_attr(self->get_owner(), "-filename");

          if (!ufilename.length()) {
            array<wchar> t;
            self->emit_text(t);
            ufilename = t();
          }
        }
        // ufilename.replace('/','\\');

        int     ufactor = 0;
        ustring sz      = get_attr(self, "-icon-size");
        if (sz == WCHARS("small"))
          ufactor = SMALL;
        else if (sz == WCHARS("sys-default"))
          ufactor = SYS_DEFAULT;
        else if (sz == WCHARS("large"))
          ufactor = LARGE;
        else if (sz == WCHARS("x-large"))
          ufactor = X_LARGE;
        else if (sz == WCHARS("xx-large"))
          ufactor = XX_LARGE;

        if (filename != ufilename || factor != ufactor)
          attempt_made = false;
        else if (attempt_made)
          return img;

        filename     = ufilename;
        factor       = ufactor;
        attempt_made = true;

        if (filename.length() == 0) {
          img = 0;
          return 0;
        }

        fetch_image(v, self);

        if (!img && !self->state.invalid()) {
          self->state_on(v, S_INVALID);
        } else if (img && self->state.invalid())
          self->state_off(v, S_INVALID);

        return img;
      }

      virtual bool set_value(view &v, element *self,
                             const value &val) override {
        ustring us = val.to_string();
        self->set_attr("filename", us);
        v.refresh(self);
        return true;
      }
      virtual bool get_value(view &v, element *self, value &val) override {
        ustring filename = get_attr(self, "-filename");
        val              = filename;
        return true;
      }
    };

    ctl *fileicon_ctl_factory::create(element *) { return new fileicon_ctl(); }

    struct shellicon_ctl : public fileicon_ctl {
      virtual const string &behavior_name() const override {
        return _shellicon_ctl_factory->name;
      }
      // disallow to read icon from real files - generic icons
      virtual bool allow_read_from_fs() const { return false; }
    };

    ctl *shellicon_ctl_factory::create(element *) {
      return new shellicon_ctl();
    }

    struct filethumbnail_ctl : public fileicon_ctl,
                               public load_file_thumbnail_callback {
      view *               _pv;
      element *            _this;
      typedef fileicon_ctl super;

      virtual const string &behavior_name() const override {
        return _filethumbnail_ctl_factory->name;
      }

      virtual bool attach(view &v, element *self) override {
        _this = self;
        _pv   = &v;
        return super::attach(v, self);
      }

      virtual void detach(view &v, element *self) override {
        _this = 0;
        super::detach(v, self);
      }

      virtual void fetch_image(view &v, element *self) override {
        uint size = 0;
        switch (factor) {
        default:
        case SYS_DEFAULT: size = 0; break;
        case SMALL: size = 16; break;
        case LARGE: size = 32; break;
        case X_LARGE: size = 48; break;
        case XX_LARGE: size = 256; break;
        }

        if (!load_file_thumbnail(filename, size, this))
          self->state_on(v, S_BUSY); // will be delivered later
      }

      // this method gets called from worker thread
      virtual void on_thumbnail_created(html::bitmap *pthumbnail) {
        if (_this && _pv)
          _pv->post(delegate(this, &filethumbnail_ctl::image_ready,
                             handle<bitmap>(pthumbnail)),
                    false);
        // else
        //  _this = _this;
      }

      // this method gets called by code in UI thread
      bool image_ready(handle<bitmap> pthumbnail) {
        img = pthumbnail;
        if (_this && _pv) {
          uint64 flagsoff = S_BUSY;
          if (img) flagsoff |= S_INVALID;
          _this->state_off(*_pv, flagsoff);
          // if(!img)
          //  dbg_printf("\nIMAGE READY: %S, NONE!\n",filename.c_str());
          // else
          //  dbg_printf("\nIMAGE READY: %S, %d %d\n",filename.c_str(),
          //  img->dim().x,img->dim().y);

          //_this->state_on(*_pv,S_VISITED);
          _pv->refresh(_this);
        }
        return true;
      }
    };

    ctl *filethumbnail_ctl_factory::create(element *) {
      return new filethumbnail_ctl();
    }

    void init_image() {
      // ctl_factory::add( _image_ctl_factory = new image_ctl_factory() );
      ctl_factory::add(_fileicon_ctl_factory = new fileicon_ctl_factory());
      ctl_factory::add(_shellicon_ctl_factory = new shellicon_ctl_factory());
      ctl_factory::add(_filethumbnail_ctl_factory =
                           new filethumbnail_ctl_factory());
    }

  } // namespace behavior
} // namespace html
