#include "html.h"

namespace html {
  namespace behavior {

    struct htmlarea_ctl_factory : public ctl_factory {
#ifdef SCITERJS
      htmlarea_ctl_factory() : ctl_factory("selectable") {}
#else
      htmlarea_ctl_factory() : ctl_factory("htmlarea") {}
#endif
      virtual ctl *create(element *el);
    };

    static htmlarea_ctl_factory *_htmlarea_ctl = 0;

    const string &htmlarea_ctl::behavior_name() const {
      return _htmlarea_ctl->name;
    }

    argb htmlarea_ctl::selection_back_color(view &v) {
      const style *cs = self->get_style(v);
      argb c = cs->text_selection_background_color.to_argb();
      return c;
    }
    argb htmlarea_ctl::selection_fore_color(view &v) {
      const style *cs = self->get_style(v);
      argb c = cs->text_selection_background_color.to_argb();
      c.alfa = 127;
      return c;
    }
    argb htmlarea_ctl::selection_text_color(view &v) {
      const style *cs = self->get_style(v);
      argb c = cs->text_selection_color.to_argb();
      return c;
    }
    argb htmlarea_ctl::selection_caret_color(view &v) {
      const style *cs = self->get_style(v);
      argb         c;
      if (cs->text_selection_caret_color.is_defined())
        c = cs->text_selection_caret_color.to_argb();
      else
        c = cs->text_selection_background_color.to_argb();
      c.alfa = argb::channel_t(caret_opacity);
      return c;
    }

    element *htmlarea_ctl::root_at(view &v, element *pel) {
      if (!pel) return nullptr;
      element *bfc = pel->nearest_bfc(v);
      if (!bfc || !bfc->belongs_to(root())) return root();
      return bfc;
    }

    element *htmlarea_ctl::root_at(view &v, bookmark bm) {
      return bm.valid() ? root_at(v, bm.node->get_element()) : root();
    }

    bool htmlarea_ctl::draw_content(view &v, element *self, graphics *pg, point pos) {
      auto_state<selection_ctx *> _(v._selection_ctx, this);
      self->draw_content(v, pg, pos);
      return true;
    }

    /*text_block* get_parent_tb(node* pn)
    {
      element* pe = pn->parent;
      while(pe)
      {
        if( pe->layout_type() == flow_text )
          return static_cast<text_block*>(pe);
      }
      assert(false);
      return 0;
    }*/

    bool htmlarea_ctl::ensure_visible(view &v, bookmark c,
                                      ENSURE_VISIBLE_MANNER how) {
      rect cr;
      if (!get_bookmark_place(v, caret, cr)) return false;
      element *scrollable = self->get_scrollable_container(v, true);
      if (scrollable && scrollable->is_visible(v)) {
        v.scroll_to_view(scrollable, cr + self->view_pos(v), false, SCROLL);
        return true;
      }
      return false;
    }

    bool htmlarea_ctl::select(view &v, bookmark c, bookmark a) {
      if (!self)
        return false;

      v.commit_update();

      refresh_selection(v);

      bookmark scaret  = caret;
      bookmark sanchor = anchor;

      // dbg_printf("caret %d/%d, anchor %d \n",caret.tb_pos, caret.after_it,
      // anchor.tb_pos);
#ifdef _DEBUG
      if (get_selection_type(v) == CELL_RANGE) {
        caret.dbg_report("htmlarea_ctl::select");
      }
#endif

      if (a.valid() && (c != a)) {
        set_caret_to(v, a, false); 
        set_caret_to(v, c, true);
      } else if (c.valid()) {
        set_caret_to(v, c, false);
      } else
        set_caret_to(v, bookmark(), false);

      if (caret.valid() && ensure_visible(v, caret, scroll_caret_manner(v)))
        show_caret(v, true);

      event_behavior evt(self, self, UI_STATE_CHANGED, 0);
      v.post_behavior_event(evt, true);
 
      return true;
    }

    void htmlarea_ctl::show_caret(view &v, bool on) {
      if (on) {
        if (caret.valid() && caret_opacity != 255) {
          caret_opacity = 255;
          v.start_timer(self, 20, CARET_TIMER_ID, INTERNAL_TIMER);
        }
      } else {
        caret_opacity = 0;
#if !defined(_DEBUG)
        v.stop_timer(self, CARET_TIMER_ID, INTERNAL_TIMER);
#endif
      }
    }

    bool htmlarea_ctl::draw_caret(view &v, graphics *pg,
                                  const caret_metrics &cm) {
      argb caret_color = selection_caret_color(v);
      if (caret_color.alfa) pg->fill(caret_color, cm.caret_v_bar());
      return true;
    }

    void htmlarea_ctl::refresh_selection(view &v) {
      v.refresh(self);
      rect rc;
      if (caret != anchor) {
        node *pn = node::find_common_parent(anchor.node, caret.node);
        if (pn) {
          element *el = pn->nearest_box();
          rc          = el->rendering_box(v) + el->rel_pos(v, self);
        }
      } else {
        get_bookmark_place(v, caret, rc);
      }
      if (!rc.empty()) v.refresh(self, rc);
    }

    bool htmlarea_ctl::get_bookmark_place(view &v, bookmark bm, rect &rc) {
      if (bm.valid() && bm.node->belongs_to(self)) {
        // get_caret_place
        // self->get_caret_rect(v,rf, this);
        caret_metrics gm;
        if (bm.get_caret_metrics(v, gm)) {
          rc = gm.caret_outline() + gm.elem->rel_pos(v, self);
          rc >>= 1;
          return true;
          // rc.s.x
        }
      }
      return false;
    }

    bool is_kindof_replaced(view &v, element *root, element *pel) {
      return pel->tag == tag::T_IMG || pel->tag == tag::T_INPUT ||
             pel->tag == tag::T_SELECT || pel->tag == tag::T_TEXTAREA;
    }

    bool htmlarea_ctl::perform_drag(view &v, element *self, event_mouse &evt) {
      // int s = anchor.linear_pos();
      // int e = caret.linear_pos();
      // if( s > e ) swap(s,e);
      handle<clipboard::data> cd = new clipboard::data();

      array<wchar> text;
      array<char>  html;
      clipboard::text_cf(v, *this, text);
      clipboard::html_cf(v, *this, html);

      cd->add(new html::clipboard::text_item(text()));
      cd->add(html::clipboard::html_item::from_cf_html(html()));

      rect vbox = self->content_box(v, element::TO_VIEW);
      rect vsrc = html::rbox(v, anchor, caret);
      // assert(!vsrc.empty());
      if (vsrc.empty()) return false;
      handle<bitmap> drag_image = new bitmap(vsrc.size(), true, false);
      if (drag_image) {
        handle<graphics> sfi =
            v.app->create_bitmap_bits_graphics(drag_image, argb(0, 0, 0, 0));
        if (!sfi) return false;
        sfi->offset(-vsrc.s);
        // sfi->set_clip_rc(vsrc);
        auto_state<graphics *>      _1(v.drawing_surface, sfi);
        auto_state<selection_ctx *> _2(v._selection_ctx, this);
        auto_state<tristate_v>      _3(v._draw_selection_only, TRUE);

        point vpt = self->view_pos(v);
        self->draw_content(v, sfi, vpt, false);
        v.commit_drawing_buffers();
      }
      uint ddm = is_editable(self) ? dd_copy_or_move : dd_copy;
      if (v.exec_drag(cd, ddm, self, drag_image, evt.pos_view - vsrc.s) && ddm == dd_move)
        return true;
      return false;
    }

    bool htmlarea_ctl::on(view &v, element *self, event_behavior &evt) {
      if (evt.cmd == CONTENT_CHANGED ||
          evt.cmd == (CONTENT_CHANGED | EVENT_HANDLED)) {
        if (caret.valid() && !caret.node->belongs_to(self)) {
          caret = anchor = bookmark();
          v.refresh(self);
        }
        if (anchor.valid() && !anchor.node->belongs_to(self)) {
          caret = anchor = bookmark();
          v.refresh(self);
        }
      }
      return false;
    }

    bool htmlarea_ctl::on(view &v, element *self, event_mouse &evt) {
      switch (evt.cmd) {
        // case MOUSE_DCLICK:
      case MOUSE_DOWN:
      {
        bookmark bm = self->find_text_pos(v, evt.pos);
        if (!bm.valid())
          return false;

        //#ifdef _DEBUG
        //          bm.dbg_report("down");
        //#endif

        if (!bm.node->belongs_to(self, true))
          break;

        if (evt.is_prop_button()) {
          v.refresh(self);
          v.set_focus(self, BY_MOUSE);

          bool on_selection = is_on_selection(v, bm);
          if (!on_selection) {
            if ((bm.at_element_start() || bm.at_element_end()) && is_kindof_replaced(v, self, bm.node->get_element()))
              select(v, bm.node->start_pos(), bm.node->end_pos());
            else
              select(v, bm, evt.is_shift() ? anchor : bookmark());
          }
          return true;
        }
        else if (evt.is_point_button())
        {
          v.refresh(self);
          v.set_focus(self, BY_MOUSE);
          v.set_capture(self);

          mouse_down_on_selection = caret.valid() && anchor.valid() && bm.is_between(selection_ctx::normalized());
          if (!mouse_down_on_selection) {
            //#ifdef _DEBUG
            //              bm.dbg_report("mouse_down");
            //#endif

            if ((bm.at_element_start() || bm.at_element_end()) && bm.node.ptr_of<element>()->is_atomic_box())
              select(v, bm.node->start_pos(), bm.node->end_pos());
            else if ((bm.at_element_start() || bm.at_element_end()) && bm.node.ptr_of<element>()->is_inline_block_element(v))
              select(v, bm.node->start_pos(), bm.node->end_pos());
            else
              select(v, bm, evt.is_shift() ? anchor : bookmark());
          }
          return true;
        }
      }
      break;
      case MOUSE_UP:
      case MOUSE_UP | EVENT_HANDLED:
        if (evt.is_point_button()) {
          v.refresh(self);
          v.set_capture(0);

          if (v.mouse_down_element && v.mouse_over_element) {

            if (!v.mouse_down_element->belongs_to(self, true)) break;

            node *n = node::find_base(v.mouse_down_element, v.mouse_over_element);

            bookmark bm = self->find_text_pos(v, evt.pos);
            if (mouse_down_on_selection)
              select(v, bm);
            else if (n && n->is_element() && is_kindof_replaced(v, self, n->cast<element>()))
              select(v, n->start_pos(), n->end_pos());
          }
          get_selection_type(v);
          mouse_down_on_selection.clear();

          if (self->state.pressed() && self->state.hover()) {
            self->state.pressed(false);
            return true;
          }
          else
            self->state.pressed(false);
        }
        break;
      case MOUSE_DRAG_REQUEST:
        if (mouse_down_on_selection && (selection_type == TEXT_RANGE)) {
          perform_drag(v, self, evt);
        }
        break;

      case MOUSE_DCLICK:
        if (evt.is_point_button()) {
          bookmark bm = self->find_text_pos(v, evt.pos);
          if (!bm.valid()) return false;
          //bm.linearize(); - wrong if click on bm.after_it == true
          bookmark bm_start = bm;
          bookmark bm_end = bm;

          //last_dbl_click_clock = v.get_ticks();

          bool r = false;

          if (evt.is_shortcut())
            r = html::advance(v, self, bm_start, ADVANCE_HOME) &&
            html::advance(v, self, bm_end, ADVANCE_END);
          else
            do {
              if (!html::advance(v, self, bm_start, ADVANCE_WORD_START)) break;
              bm_end = bm_start;
              if (!html::advance(v, self, bm_end, ADVANCE_WORD_END)) break;
              r = true;
            } while (false);

            if (r) { select(v, bm_start, bm_end); }
            return true;
        }
        break;

      case MOUSE_TCLICK:
        // tripple click
        if (evt.is_point_button()) {
          bookmark bm = self->find_text_pos(v, evt.pos);
          if (!bm.valid()) return false;
          //bm.linearize();

          helement hblock = bm.node->parent_block(v);
          if (!hblock)
            break;

          select(v, hblock->start_caret_pos(v), hblock->end_caret_pos(v));
          return true;
        }
        break;


      /*case MOUSE_TICK:
        if (evt.is_point_button() && self->state.focus()) {
          bookmark bm = self->find_text_pos(v, evt.pos);
          if (bm.valid()) select(v, bm, anchor);
        }
        break;*/

      case MOUSE_MOVE:
        if (evt.is_point_button() && self->state.focus() && !mouse_down_on_selection) {
          bookmark bm = self->find_text_pos(v, evt.pos);
          if (bm.valid()) select(v, bm, anchor);
        }
        /*else if (!evt.get_mouse_buttons() && evt.target)
        {
          if (evt.target->is_block_element(v)) {
            rect cbox = evt.target->content_box(v);
            rect pbox = evt.target->padding_box(v);
            if (!cbox.contains(evt.pos) && pbox.contains(evt.pos))
              //v.highlighted_ctl
          }
        }*/
        break;
      case MOUSE_LEAVE: {
        self->state.pressed(false);
      } break;
      }
      return false;
    }

    inline element *parent_block(view &v, node *n) {
      if (n->is_element()) {
        element *el = static_cast<element *>(n);
        if (el->is_block_element(v)) return el;
      }
      for (element *p = n->parent; p; p = p->parent) {
        if (p->is_block_element(v)) return p;
      }
      return 0;
    }

    bool htmlarea_ctl::allow_text_selection_on(text_block *tb) {
      element *t = tb;
      while (t) {
        if (t->flags.disable_text_selection) return false;
        if (t == self) break;
        t = t->parent;
      }
      return true;
    }

    bool htmlarea_ctl::move_caret(view &v, ADVANCE_DIR dir, bool keep_anchor,
                                  uint n) {
      bookmark bm = caret;
      if (html::advance(v, self, bm, dir)) {
        if (keep_anchor)
          select(v, bm, anchor);
        else
          select(v, bm);
        return true;
      }
      return false;
    }

    bool htmlarea_ctl::on_key_down(view &v, element *self, event_key &evt) {
      bookmark bm = caret;
      if (!bm.valid()) return false;
      ustring cmd;
      switch (evt.key_code) {
      case KB_LEFT: // seek left one cluster
        cmd = evt.is_shortcut() || evt.is_option() ? event_command::NAVIGATE_WORD_START() : 
              evt.is_command() ? event_command::NAVIGATE_LINE_START() : 
              event_command::NAVIGATE_BACKWARD();
        break;
      case KB_RIGHT: // seek right one cluster
        cmd = evt.is_shortcut() || evt.is_option() ? event_command::NAVIGATE_WORD_END() :
             evt.is_command() ? event_command::NAVIGATE_LINE_END() :
             event_command::NAVIGATE_FORWARD();
        break;
      case KB_UP: 
        cmd = evt.is_command()  ? event_command::NAVIGATE_START() // start of document, MacOS
                                : event_command::NAVIGATE_UP(); // up one line
        break;
      case KB_DOWN: 
        cmd = evt.is_command() ? event_command::NAVIGATE_END() // end of document, MacOS
                                : event_command::NAVIGATE_DOWN(); // down one line
        break;
      case KB_HOME: // beginning of line
        cmd = evt.is_shortcut() ? event_command::NAVIGATE_START()
                                : event_command::NAVIGATE_LINE_START();
        break;
      case KB_END: // end of line
        cmd = evt.is_shortcut() ? event_command::NAVIGATE_END()
                                : event_command::NAVIGATE_LINE_END();
        break;
      }
      if (cmd.is_defined())
        return exec_command(v, bm.node->get_element(), self, cmd,
                            value(evt.is_shift()));
      return false;
    }
    /*
      bool htmlarea_ctl::on_key_down(view& v, element* self, event_key& evt)
      {
        bookmark bm = caret;
        if (!bm.valid())
          return false;
        switch (evt.key_code)
        {
        case KB_LEFT: // seek left one cluster
          if (caret != anchor && !evt.is_shift()) {
            select(v, caret < anchor ? caret : anchor);
            return true;
          }
          else if (html::advance(v, self, bm, evt.is_shortcut() ?
      ADVANCE_WORD_START : ADVANCE_LEFT))
          {
            if (evt.is_shift())
              select(v, bm, anchor);
            else
              select(v, bm);
            return true;
          }
          break;

        case KB_RIGHT: // seek right one cluster
          if (caret != anchor && !evt.is_shift()) {
            select(v, caret > anchor ? caret : anchor);
            return true;
          }
          else if (html::advance(v, self, bm, evt.is_shortcut() ?
      ADVANCE_WORD_END : ADVANCE_RIGHT))
          {
            if (evt.is_shift())
              select(v, bm, anchor);
            else
              select(v, bm);
            return true;
          }
          break;
        case KB_UP: // up one line
          if (html::advance(v, self, bm, evt.is_shortcut() ? ADVANCE_UP :
      ADVANCE_UP, caret_pos))
          {
            point t = caret_pos;
            if (evt.is_shift())
              select(v, bm, anchor);
            else
              select(v, bm);
            caret_pos.x = t.x;
            return true;
          }
          break;
        case KB_DOWN: // down one line
          if (html::advance(v, self, bm, evt.is_shortcut() ? ADVANCE_DOWN :
      ADVANCE_DOWN, caret_pos))
          {
            point t = caret_pos;
            if (evt.is_shift())
              select(v, bm, anchor);
            else
              select(v, bm);
            caret_pos.x = t.x;
            return true;
          }
          break;

        case KB_HOME: // beginning of line
          if (html::advance(v, self, bm, evt.is_shortcut() ? ADVANCE_FIRST :
      ADVANCE_HOME))
          {
            if (evt.is_shift())
              select(v, bm, anchor);
            else
              select(v, bm);
            return true;
          }
          break;
        case KB_END: // end of line
          if (html::advance(v, self, bm, evt.is_shortcut() ? ADVANCE_LAST :
      ADVANCE_END))
          {
            if (evt.is_shift())
              select(v, bm, anchor);
            else
              select(v, bm);
            return true;
          }
          break;
        }
        return false;
      }
      */

    bool htmlarea_ctl::on(view &v, element *self, event_command &evt) {
      if (!evt.doit() && !evt.checkit()) return false;

      ustring cmd = evt.command;
      if (cmd == event_command::EDIT_COPY()) {
        if (evt.doit()) return copy(v);
        return can_copy(v);
      } else if (cmd == event_command::EDIT_SELECT_ALL()) {
        if (evt.doit()) return select_all(v);
        return !self->is_empty();
      }

      // navigational command handling

      bool     keep_anchor = evt.data.get(false);
      bool     r           = false;
      bookmark bm          = caret;

      if (evt.checkit()) return bm.valid();

      if (cmd == event_command::NAVIGATE_BACKWARD()) {
        if (caret != anchor && !keep_anchor) {
          select(v, caret < anchor ? caret : anchor);
          return true;
        }
        r = html::advance(v, root(), bm, ADVANCE_LEFT);
      } else if (cmd == event_command::NAVIGATE_FORWARD()) {
        if (caret != anchor && !keep_anchor) {
          select(v, caret > anchor ? caret : anchor);
          return true;
        }
        r = html::advance(v, root(), bm, ADVANCE_RIGHT);
      } else if (cmd == event_command::NAVIGATE_UP()) {
        if (html::advance(v, root(), bm, ADVANCE_UP, caret_pos)) {
          point t = caret_pos;
          if (keep_anchor)
            select(v, bm, anchor);
          else
            select(v, bm);
          caret_pos.x = t.x;
          return true;
        }
      } else if (cmd == event_command::NAVIGATE_DOWN()) {
        if (html::advance(v, root(), bm, ADVANCE_DOWN, caret_pos)) {
          point t = caret_pos;
          if (keep_anchor)
            select(v, bm, anchor);
          else
            select(v, bm);
          caret_pos.x = t.x;
          return true;
        }
      } else if (cmd == event_command::NAVIGATE_WORD_START()) {
        if(r = html::advance(v, root(), bm, ADVANCE_LEFT))
          r = html::advance(v, root(), bm, ADVANCE_WORD_START);
      } else if (cmd == event_command::NAVIGATE_WORD_END()) {
        if (r = html::advance(v, root(), bm, ADVANCE_RIGHT))
          r = html::advance(v, root(), bm, ADVANCE_WORD_END);
      } else if (cmd == event_command::NAVIGATE_START()) {
        r = html::advance(v, root(), bm, ADVANCE_FIRST);
      } else if (cmd == event_command::NAVIGATE_LINE_START()) {
        r = html::advance(v, root(), bm, ADVANCE_HOME);
      } else if (cmd == event_command::NAVIGATE_END()) {
        r = html::advance(v, root(), bm, ADVANCE_LAST);
      } else if (cmd == event_command::NAVIGATE_LINE_END()) {
        r = html::advance(v, root(), bm, ADVANCE_END);
      }
      if (r) {
        if (keep_anchor)
          select(v, bm, anchor);
        else
          select(v, bm);
        return true;
      }
      return false;
    }

    bool htmlarea_ctl::copy(view &v) {
      if (!caret.valid() || caret == anchor) return false;

      clipboard::empty();
      clipboard::set(v, *this);
      return true;
    }

    bool htmlarea_ctl::select_all(view &v) {
      // if( caret.valid() || caret == anchor )
      bookmark start = self->first_content_caret_pos(v);
      bookmark end   = self->last_content_caret_pos(v);
      if (start.valid() && end.valid()) { select(v, end, start); }
      return true;
    }

    bool htmlarea_ctl::on(view &v, element *self, event_focus &evt) {
      if (this->self != self)
        return false;
      if (evt.cmd == FOCUS_OUT && evt.target == self)
        show_caret(v, false);
      else if ((evt.cmd == FOCUS_IN || evt.cmd == GOT_FOCUS) && evt.target == self) {
        if (!caret.valid() && evt.cause != BY_MOUSE) {
          bookmark bm;
          if (html::advance(v, self, bm, ADVANCE_FIRST)) select(v, bm);
        }
        show_caret(v, true);
      }
      v.refresh(self);
      return false;
    }

    bool htmlarea_ctl::on_timer(view &v, element *self, timer_id tid,
                                TIMER_KIND kind) {
      if (caret_opacity <= 10 || tid != CARET_TIMER_ID) {
        caret_opacity = 0;
        // v.refresh(self);
        refresh_selection(v);
        return false; /*stop this timer*/
      }
      caret_opacity -= 5;
      refresh_selection(v);
      return true;
    }

    bool htmlarea_ctl::on_method_call(view &v, element *self,
                                      method_params *p) {
      switch (p->method_id) {
      case DO_CLICK:
        // v.set_focus(self,BY_CODE);
        // do_click(v,self,self, BY_CODE);
        return true;
      }
      return false;
    }

    ctl *htmlarea_ctl_factory::create(element *el) {
      return new htmlarea_ctl();
    }

    void init_htmlarea() {
      ctl_factory::add(_htmlarea_ctl = new htmlarea_ctl_factory());
    }

  } // namespace behavior
} // namespace html
