#include "html.h"
#include "html-behaviors.h"

namespace html {
  namespace behavior {

    struct clickable_factory : public ctl_factory {
      clickable_factory() : ctl_factory("clickable") {}
      virtual ctl *create(element *el) { return new clickable(); }
    };

    clickable_factory *_clickable_factory = 0;

    const string &clickable::behavior_name() const {
      return _clickable_factory->name;
    }

    //|
    //| clickable
    //|

    bool clickable::on(view &v, element *self, event_mouse &evt) {
      if (evt.cmd == MOUSE_DOWN && evt.is_point_button()) {
        self->state.pressed(true);
        self->get_style(v);
        // v.set_capture(self);
        event_behavior tevt(self, self, BUTTON_PRESS, CLICK_BY_MOUSE);
        v.send_behavior_event(tevt);
        v.dont_change_focus = true;
        evt.cancel_event(tevt.is_canceled());
        return true;
      } else if (evt.cmd == MOUSE_UP) {
        if (self->state.pressed() && evt.is_point_button()) {
          event_behavior tevt(self, self, BUTTON_CLICK, CLICK_BY_MOUSE);
          v.post_behavior_event(tevt);
          return true;
        }
      }
      /*else if(evt.cmd == MOUSE_LEAVE)
      {
        //g_warning("MOUSE_LEFT\n");
        return false;
      }*/
      else if (evt.cmd == MOUSE_DCLICK && evt.is_point_button()) {
        event_behavior tevt(self, self, BUTTON_PRESS, CLICK_BY_MOUSE);
        v.post_behavior_event(tevt);
        return true;
      }
      if (evt.cmd == MOUSE_TICK && evt.is_point_button() && self->state.pressed() && self->is_inside(v, evt.pos_view)) {
        event_behavior tevt(self, self, BUTTON_PRESS, CLICK_BY_MOUSE);
        v.post_behavior_event(tevt);
        return true;
      }
      return false;
    }
    bool clickable::on_method_call(view &v, element *self, method_params *p) {
      if (p->method_id != DO_CLICK) return false;
      event_behavior evt(self, self, BUTTON_CLICK, BY_CODE);
      v.post_behavior_event(evt);
      return true;
    }

    bool clickable::on(view &v, element *self, event_focus &evt) {
      // if( evt.cmd == GOT_FOCUS || evt.cmd == (GOT_FOCUS | EVENT_HANDLED) )
      //  self = self;
      // assert(false);
      return ctl::on(v, self, evt);
    }

    //|
    //| a button
    //|

    bool abutton::attach(view &v, element *self) {
      self->flags.disable_text_selection = 1;
      ctl::attach(v, self);
      return true;
    }

    void abutton::detach(view &v, element *self) {
      self->flags.disable_text_selection = 0;
      ctl::detach(v, self);
    }

    bool abutton::set_value(view &pv, element *self, const value &v) {
      return false;
    }
    bool abutton::get_value(view &pv, element *self, value &v) {
      v = value(self->state.pressed());
      return true;
    }

    rect area_to_refresh(view &v, element *pe) {
      rect rc = pe->rendering_box(v);
#ifdef OSX
      rc >>= 6;
#endif
      return rc | pe->hit_box(v);
    }

    bool abutton::on(view &v, element *self, event_focus &evt) {
      if (evt.cmd == LOST_FOCUS || evt.cmd == (LOST_FOCUS | EVENT_HANDLED)) {
        self->state.pressed(false);
        // self->dbg_report("abutton::on LOST_FOCUS");
      }

      // v.refresh(self, area_to_refresh(v, self));
      return false;
    }

    bool abutton::on(view &v, element *self, event_mouse &evt) {
      switch (evt.cmd) {
      case MOUSE_DCLICK:
      case MOUSE_DOWN:
        if (evt.is_point_button()) {
          self->state.pressed(true);
          v.refresh(self, area_to_refresh(v, self));
          v.set_focus(self, BY_MOUSE);
          v.set_capture(self);
          return do_press(v, self, self, true, true);
        }
        break;
      case MOUSE_UP:
        if (evt.is_point_button()) {
          v.refresh(self, area_to_refresh(v, self));
          v.set_capture(0);
          if (self->state.pressed() && self->state.hover()) {
            self->state.pressed(false);
            do_press(v, self, self, false, true);
            return do_click(v, self, self);
          }
          self->state.pressed(false);
          return true;
        }
        break;

      case MOUSE_MOVE:
        if (evt.is_point_button()) {
          bool is_here = self->is_inside(v, evt.pos_view);
          if (self->state.hover() != is_here) {
            if (!is_here) {
              // self->state.hover(false);
              self->state_off(v, S_ACTIVE | S_HOVER);
              // return false;
            } else {
              // self->state.hover(true);
              self->state_on(v, S_ACTIVE | S_HOVER);
            }
          }
          // return true;
        }
        break;
      case MOUSE_LEAVE: {
        self->state.pressed(false);
      } break;
      case MOUSE_TICK:
        if (evt.is_point_button() && self->is_inside(v, evt.pos_view)) {
          return do_press(v, self, self, true, true);
        }
        break;
      }
      return false;
    }

    /*static bool only_active(view &v, element *b) {
      hstyle cs = b->get_style(v);
      return cs->visible() && !b->is_disabled();
    }*/

    static bool skip_non_active(view &v, element *b) {
      hstyle cs = b->get_style(v);
      return !cs->visible() || b->is_disabled();
    }


    static bool only_tabstopable(view &v, element *b) {
      return b->tab_index().is_defined();
    }

    /*static element *get_logical_parent(element* self)
    {
      self = self->parent;
      while(self)
        if( self->is_element())
          return self;
        else
          self = self->parent;

      return 0;
    }*/

    static bool set_next_focus(view &v, element *self, bool forward) {
      if (self->tag != tag::T_INPUT && self->tag != tag::T_BUTTON) return false;
      array<helement>  tabstops;
      helement         parent = self->parent;
      element_iterator bv(v, parent, only_tabstopable, skip_non_active);
      for (element *b; bv(b);)
        tabstops.push(b);

      if (tabstops.size() == 0) return false;

      index_t prev_cidx = tabstops.get_index(self);
      index_t cidx      = prev_cidx;

      if (forward) {
        cidx += 1;
        if (cidx > tabstops.last_index()) cidx = 0;
      } else {
        cidx -= 1;
        if (cidx < 0) cidx = tabstops.last_index();
      }

      if (cidx == prev_cidx) return false;

      return v.set_focus(tabstops[cidx], forward ? BY_KEY_NEXT : BY_KEY_PREV,
                         true);
    }

    bool abutton::on(view &v, element *self, event_key &evt) {
      if (evt.target != self)
        return false; // handling only key events trgeted to this element!

      // if(evt.alt_state)
      //  return false;
      switch (evt.cmd) {
      case KEY_DOWN:
        if (evt.has_no_modifiers()) switch (evt.key_code) {
          case KB_RETURN:
            key_pressed = evt.key_code;
            if (!is_command(v, self)) break;
          case KB_SPACE:
            self->state_on(v, S_ACTIVE);
            self->state.pressed(true);
            key_pressed = evt.key_code;
            do_press(v, self, evt.target, true, false);
            return true;
          case KB_LEFT:
            // case KB_UP:
            return set_next_focus(v, self, false);
          case KB_RIGHT:
            // case KB_DOWN:
            return set_next_focus(v, self, true);
          }
        break;
      case KEY_UP:
        if (self->state.pressed() && key_pressed == evt.key_code)
          switch (evt.key_code) {
          case KB_RETURN:
            if (!is_command(v, self)) break;
          case KB_SPACE: {
            self->state_off(v, S_ACTIVE);
            self->state.pressed(false);
            key_pressed = 0;
            do_press(v, self, evt.target, false, false);
            do_click(v, self, evt.target);
            return true;
          }
          }
        break;
      }
      return false;
    }

    bool abutton::do_press(view &v, element *self, element *target, bool on,
                           bool by_mouse) {
      if (on) {
        event_behavior evt(self, target, BUTTON_PRESS,
                           by_mouse ? CLICK_BY_MOUSE : CLICK_BY_KEY);
        v.post_behavior_event(evt);
      }
      // v.update();
      return true;
    }
    bool abutton::do_click(view &v, element *self, element *target,
                           bool by_mouse) {
      event_behavior evt(self, target, BUTTON_CLICK,
                         by_mouse ? CLICK_BY_MOUSE : CLICK_BY_KEY);
      v.post_behavior_event(evt);
      return true;
    }
    bool abutton::on_method_call(view &v, element *self, method_params *p) {
      switch (p->method_id) {
      case DO_CLICK: return do_click(v, self, self, false);
      }
      return false;
    }

    struct button_factory : public ctl_factory {
      button_factory() : ctl_factory("button") {}
      virtual ctl *create(element *el);
    };

    static button_factory *_button_factory = 0;

    struct button : public abutton {
      button() {}

      virtual CTL_TYPE get_type() { return CTL_BUTTON; }

      virtual bool is_command(view &v, element *self) { return true; }

      virtual const string &behavior_name() const {
        return _button_factory->name;
      }

      virtual bool attach(view &v, element *self) {
        if (self->nodes.size() == 0 && self->tag == tag::T_INPUT) {
          ustring btext = self->attr_value();
          if (btext.length()) {
            element *tb = new element(tag::T_TEXT);
            tb->append(new text(btext));
            self->append(tb, &v);
          }
        }
        return abutton::attach(v, self);
      }
    };

    ctl *button_factory::create(element *el) { return new button(); }

    struct checkbox_factory : public ctl_factory {
      checkbox_factory() : ctl_factory("check") {}
      virtual ctl *create(element *el);
    };

    static checkbox_factory *_checkbox_factory = 0;

    struct checkbox : public abutton {
      checkbox() {}

      virtual CTL_TYPE get_type() { return CTL_CHECKBOX; }

      virtual bool is_command(view &v, element *self) { return true; }

      virtual const string &behavior_name() const {
        return _checkbox_factory->name;
      }

      virtual bool attach(view &v, element *self) override {
        abutton::attach(v, self);

        if (self->state.checked() || self->state.unchecked()) // states can be initialized before behavior attachment, e.g. Reactor
          return true;

        ustring sv = WCHARS("false");

        if (is_mixed(self)) {
          bool was_checked = self->state.checked();
          bool was_empty   = self->state.empty();
          bool checked     = false;
          bool empty       = false;

          if (self->atts.get(attr::a_checked, sv)) {
            sv = trim(sv());
            if (sv.length() == 0 || sv == WCHARS("true")) {
              checked = true;
              empty   = false;
            } else if (sv == WCHARS("false")) {
              checked = false;
              empty   = false;
            } else if (sv == WCHARS("undefined") || sv == WCHARS("null")) {
              checked = false;
              empty   = true;
            }
            if (was_checked != checked || was_empty != empty) {
              self->state.checked(checked);
              self->state.empty(empty);
              self->reset_styles(v);
            }
          } 
          return true;
        }

        bool unchecked     = false;
        bool checked       = false;
        bool was_checked   = self->state.checked();
        bool was_unchecked = self->state.unchecked();

        if (self->atts.get(attr::a_checked, sv)) {
          sv = trim(sv());
          if (sv.length() == 0 || sv == WCHARS("true")) {
            checked = true;
          } else if (sv == WCHARS("false")) {
            unchecked = true;
          } else if (sv == WCHARS("undefined")) {
            checked   = false;
            unchecked = false;
          }
          if (was_checked != checked || was_unchecked != unchecked) {
            self->state.checked(checked);
            self->state.unchecked(unchecked);
            self->reset_styles(v);
          }
        }
        return true;
      }

      virtual bool is_mixed(const element *self) {
        return self->atts.exist("mixed");
      }
      virtual bool is_empty(const element *self, bool &yes) {
        yes = is_mixed(self) ? self->state.empty() : false;
        return true;
      }

      virtual bool do_click(view &v, element *self, element *target,
                            bool by_mouse = false) {
        if (is_mixed(self)) {
          int state = self->state.empty() ? 0 : (self->state.checked() ? 1 : 2);
          switch (state) {
          case 0: // empty
            self->state_off(v, S_EMPTY);
            self->state_on(v, S_CHECKED);
            break;
          case 1:
            self->state_off(v, S_EMPTY);
            self->state_on(v, S_UNCHECKED);
            break;
          case 2:
            self->state_on(v, S_EMPTY);
            self->state_off(v, S_CHECKED | S_UNCHECKED);
            break;
          }
        } else {
          bool is_on = self->state.checked();
          self->state_on(v, S_EMPTY);
          if (is_on)
            self->state_on(v, S_UNCHECKED);
          else
            self->state_on(v, S_CHECKED);
        }
        v.refresh(self, area_to_refresh(v, self));
        event_behavior evt(target, self, BUTTON_STATE_CHANGED,
                           by_mouse ? CLICK_BY_MOUSE : CLICK_BY_KEY);
        v.post_behavior_event(evt);
        return abutton::do_click(v, self, target,
                                 by_mouse); // generate button click too.
      }

      virtual bool set_value(view &v, element *self, const value &val) {
        if (val.is_string()) {
/*
          ustring aval = get_attr(self, "-value");
          self->state_on(v, S_EMPTY);
          if (val.get_chars() == aval())
            self->state_on(v, S_CHECKED);
          else
            self->state_on(v, S_UNCHECKED);
  */        
          self->atts.set(html::attr::a_value, val.to_string());
          return true;
        }
        bool b = val.get(false);
        if (is_mixed(self) && (val.is_null() || val.is_undefined())) {
          self->state_on(v, S_EMPTY);
          self->state_off(v, S_CHECKED | S_UNCHECKED);
        } else if (b) {
          self->state_off(v, S_EMPTY);
          self->state_on(v, S_CHECKED);
        } else {
          self->state_off(v, S_EMPTY);
          self->state_on(v, S_UNCHECKED);
        }
        return true;
      }
      virtual bool get_value(view &pv, element *self, value &v) {
        if (is_mixed(self) && self->state.empty())
          v = value::null_val();
        else if (self->state.checked()) {
          if(!self->get_attr_value(v))
            v = value(true);
        }
        else if (self->state.unchecked()) {
          if (self->get_attr_value(v))
            v = value();
          else
            v = value(false);
        }
        return true;
      }
    };

    ctl *checkbox_factory::create(element *el) { return new checkbox(); }

    struct toggle_factory : public ctl_factory {
      toggle_factory() : ctl_factory("toggle") {}
      virtual ctl *create(element *el);
    };

    static toggle_factory *_toggle_factory = 0;

    struct toggle : public checkbox {
      toggle() {}

      virtual CTL_TYPE get_type() { return CTL_CHECKBOX; }

      virtual const string &behavior_name() const {
        return _toggle_factory->name;
      }

      virtual bool get_auto_width(view &v, element *self, int &value) override {
        tool::inc_max<int> max_width;
        int                cwidth   = self->dim().x;
        auto               calc_max = [&](element *el) -> bool {
          max_width <<= el->max_width(v, cwidth) + el->outer_int_x_extra(v, cwidth);
          return false;
        };
        self->each_ui_child(calc_max);
        value = max_width ? int(max_width) : 150;
        return true;
      }
    };

    ctl *toggle_factory::create(element *el) { return new toggle(); }

    struct radio_factory : public ctl_factory {
      radio_factory() : ctl_factory("radio") {}
      virtual ctl *create(element *el);
    };

    static radio_factory *_radio_factory = 0;

    struct radio : public abutton {
      radio() {}

      virtual CTL_TYPE get_type() { return CTL_RADIO; }

      virtual bool is_command(view &v, element *self) { return true; }

      virtual const string &behavior_name() const {
        return _radio_factory->name;
      }

      virtual bool attach(view &v, element *self) override {
        //if (!self->flags.state_initialized) 
        if( !self->state.checked() && !self->state.unchecked()) // states can be initialized before behavior attachment, e.g. Reactor
        {
          bool checked = self->atts.get_bool(attr::a_checked);
          bool was_checked = self->state.checked();
          if (was_checked != checked) {
            self->state.checked(checked);
            self->clear_style();
          }
          self->get_style(v);
        }
        abutton::attach(v, self);
        return true;
      }
      virtual bool is_empty(const element *self, bool &yes) {
        yes = false;
        return true;
      }

      void set(view &v, element *self, bool val) {
        if (val) {
          ustring name = self->attr_name();
          if (name.length())
            reset_by_name(v, self, name);
          else
            reset_siblings(v, self);
          self->state_on(v, S_CHECKED);
        } else {
          self->state_on(v, S_UNCHECKED);
        }
      }

      void reset_by_name(view &v, element *self, const ustring &name) {
        document *pd = self->doc();
        if (!pd) return;
        string       bhv_name = behavior_name();
        each_element it(pd);
        for (element *b; it(b);) {
          if (b->behavior && (b != self) && (b->state.checked()) &&
              (b->get_named_behavior(bhv_name) != 0) &&
              (b->attr_name() == name)) {
            b->state_on(v, S_UNCHECKED);
          }
        }
      }

      void reset_siblings(view &v, element *self) {
        if (!self->parent) return;
        string       bhv_name = behavior_name();
        each_element it(self->parent);
        for (element *b; it(b);) {
          if (b->behavior && (b != self) && (b->state.checked()) &&
              (b->get_named_behavior(bhv_name) != 0) && (b->tag == self->tag)) {
            b->state_on(v, S_UNCHECKED);
          }
        }
      }

      virtual bool do_click(view &v, element *self, element *target,
                            bool by_mouse = false) {
        if (!self->state.checked()) {
          set(v, self, true);
          event_behavior evt(target, self, BUTTON_STATE_CHANGED,
                             by_mouse ? CLICK_BY_MOUSE : CLICK_BY_KEY);
          v.post_behavior_event(evt);
        }
        return abutton::do_click(v, self, target,
                                 by_mouse); // generate button click too.
      }

      virtual bool set_value(view &v, element *self, const value &val) {
        bool b;
        if (val.is_bool()) {
          b = val.get(false);
          set(v, self, b);
        }
        else {
          self->atts.set(html::attr::a_value, val.to_string());
          //value ev;
          //self->get_attr_value(ev);
          //b = val == ev;
        }
        
        return true;
      }
      virtual bool get_value(view &pv, element *self, value &v) {
        if (self->state.checked()) {
          if (!self->get_attr_value(v))
            v = value(true);
        }
        else
          v = value();
        return true;
      }
    };

    ctl *radio_factory::create(element *el) { return new radio(); }

    struct switch_factory : public ctl_factory {
      switch_factory() : ctl_factory("switch") {}
      virtual ctl *create(element *el);
    };

    static switch_factory *_switch_factory = 0;

    struct Switch : public radio {
      virtual const string &behavior_name() const {
        return _switch_factory->name;
      }
      bool do_press(view &v, element *self, element *target, bool on,
                    bool by_mouse) {
        radio::do_press(v, self, target, on, by_mouse);
        if (!self->state.checked()) {
          set(v, self, true);
          event_behavior evt(target, self, BUTTON_STATE_CHANGED,
                             by_mouse ? CLICK_BY_MOUSE : CLICK_BY_KEY);
          v.post_behavior_event(evt);
        }
        return true;
      }
      virtual bool do_click(view &v, element *self, element *target,
                            bool by_mouse = false) {
        return abutton::do_click(v, self, target,
                                 by_mouse); // generate button click too.
      }
    };

    ctl *switch_factory::create(element *el) { return new Switch(); }

    struct label_factory : public ctl_factory {
      label_factory() : ctl_factory("label") {}
      virtual ctl *create(element *el);
    };

    static label_factory *_label_factory = 0;

    struct label : ctl {
      virtual CTL_TYPE get_type() { return CTL_LABEL; }

      virtual const string &behavior_name() const override;
      virtual bool focusable(const element *self) override { return false; }

      virtual element* r13n_container(element* self) override { return self; }

      element *principal(view &v, const element *self) {
        ustring pid = self->atts.get_ustring(attr::a_for);
        if (pid.length()) {
          return self->find_neighbour_id(pid, nullptr, true);
        } else // check containment
        {
          for (int i = 0; i < self->nodes.size(); ++i) {
            node *pn = self->nodes[i];
            if (pn->is_element()) {
              CTL_TYPE ctl_type = pn->cast<element>()->ctl_type(v);
              if (ctl_type != CTL_NO && ctl_type != CTL_UNKNOWN &&
                  ctl_type != CTL_LABEL)
                return pn->cast<element>();
            }
          }
        }
        return 0;
      }

      virtual bool on(view &v, element *self, event_mouse &evt) override {
        switch (evt.cmd) {
        case MOUSE_ENTER: {
          element *pr = principal(v, self);
          if (pr) pr->state_on(v, S_HOVER);
        } break;
        case MOUSE_LEAVE: {
          element *pr = principal(v, self);
          if (pr) pr->state_off(v, S_HOVER);
        } break;
        case MOUSE_DOWN:
          if (evt.is_point_button()) {
            element *pr = principal(v, self);
            if (!pr) break;
            self->state.pressed(true);
            pr->state_on(v, S_ACTIVE);
            return true;
          }
          break;
        case MOUSE_UP:
          if (evt.is_point_button() && self->state.pressed()) {
            element *pr = principal(v, self);
            if (!pr || pr->is_disabled()) break;
            pr->state_off(v, S_ACTIVE);
            if (self->state.pressed()) {
              method_params prm;
              prm.method_id = DO_CLICK;
              if (!v.call_behavior_method(pr, &prm)) v.set_focus(pr, BY_MOUSE);
            }
            return true;
          }
          break;
          /*case MOUSE_CLICK:
          case MOUSE_DCLICK:
            {
              element* pr = principal( v, self );
              if(pr) {
                 method_params prm; prm.method_id = DO_CLICK;
                 if(v.call_behavior_method(pr,&prm))
                   return true;
                 else {
                   v.set_focus(pr,BY_MOUSE);
                   return true;
                 }
              }
            } break;*/
        }
        /*else if(evt.cmd == MOUSE_DOWN && evt.is_point_button())
        {
          self->state.pressed(true);
          self->get_style(v);
          element* pr = principal( v, self );
          if( pr )
            return pr->on(v,evt);
        }
        else if(evt.cmd == MOUSE_UP)
        {
          if( self->state.pressed() && evt.is_point_button())
          {
            element* pr = principal( v, self );
            if(pr)
              return pr->on(v,evt);
          }
        }
        else if(evt.cmd == MOUSE_DCLICK && evt.is_point_button())
        {
          element* pr = principal( v, self );
          if(pr)
            return pr->on(v,evt);
        }
        */
        return false;
      }

      // virtual  bool on_method_call(view& v, element* self, method_params *p);
    };

    const string &label::behavior_name() const { return _label_factory->name; }

    ctl *label_factory::create(element *el) { return new label(); }

#if 0
  struct toggle_ctl_factory: public ctl_factory
  {
    toggle_ctl_factory():ctl_factory( "toggle") {}
    virtual ctl* create(element* el);
  };

  static toggle_ctl_factory* _toggle_ctl_factory = 0;

  struct toggle_ctl: public abutton
  {
    toggle_ctl() {}

    virtual CTL_TYPE get_type() { return CTL_SELECT_SINGLE; }

    virtual bool is_command (view& v, element* self) { return true; }

    virtual const string& behavior_name() const { return _toggle_ctl_factory->name; }

    virtual void attach (view& v, element* self)
    {
      abutton::attach(v,self);

      helement found;

      auto toggler = [&v,&found](element* el) -> bool {
        if( !found )
          found = el;
        if( el->atts.exist(attr::a_selected) )
          found = el;

      };

      self->each_child(
      bool checked = self->atts.exist(attr::a_checked);
      bool was_checked = self->state.checked();
      if(was_checked != checked)
      {
        self->state.checked(checked);
        self->clear_style();
      }
      self->get_style(v);
    }

    //virtual bool is_mixed (const element* self) { return self->atts.exist("mixed"); }
    //virtual bool is_empty(const element* self, bool& yes) { yes = is_mixed(self)? self->state.empty():false; return true; }

    virtual bool do_click(view& v, element* self, element* target, bool by_mouse = false)
    {
      if(is_mixed(self))
      {
        int state = self->state.empty()? 0: (self->state.checked()?1:2);
        switch(state)
        {
          case 0: // empty
            self->state_off(v,S_EMPTY);
            self->state_on(v,S_CHECKED);
            break;
          case 1:
            self->state_off(v,S_CHECKED);
            break;
          case 2:
            self->state_on(v,S_EMPTY);
            break;
        }
      }
      else
      {
        bool is_on = self->state.checked();
        if(is_on)
          self->state_off(v,S_CHECKED);
        else
          self->state_on(v,S_CHECKED);
      }
      //v.refresh(self);
      v.refresh(self, area_to_refresh(v, self));
      event_behavior evt( target, self, BUTTON_STATE_CHANGED, by_mouse? CLICK_BY_MOUSE:CLICK_BY_KEY );
      v.post_behavior_event( evt );
      return abutton::do_click(v, self, target, by_mouse); // generate button click too.
    }

    virtual bool set_value( view& v, element* self, const value& val )
    {
      bool b = val.get(false);
      if( is_mixed(self) && (val.is_null() || val.is_undefined()))
        self->state_on(v,S_EMPTY);
      else if(b)
        self->state_on(v,S_CHECKED);
      else
        self->state_off(v,S_CHECKED);
      return true;
    }
	  virtual bool get_value( view& pv, element* self, value& v )
	  {
      if( is_mixed(self) && self->state.empty() )
        v = value::null_val();
      else
        v = value(self->state.checked());
      return true;
	  }

  };

  ctl* toggle_factory::create(element* el)
  {
    return new toggle();
  }

#endif

    void init_buttons() {
      ctl_factory::add(_clickable_factory = new clickable_factory());
      ctl_factory::add(_button_factory = new button_factory());
      ctl_factory::add(_checkbox_factory = new checkbox_factory());
      ctl_factory::add(_radio_factory = new radio_factory());
      ctl_factory::add(_switch_factory = new switch_factory());
      ctl_factory::add(_label_factory = new label_factory());

      ctl_factory::add(_toggle_factory = new toggle_factory());

      // ctl_factory::add( _menu_checkbox_ctl = new menu_checkbox_ctl_factory()
      // );
      // ctl_factory::add( _menu_optionbox_ctl = new
      // menu_optionbox_ctl_factory() );  ctl_factory::add( _file_ctl = new
      // file_ctl_factory() );  ctl_factory::add( _path_ctl = new
      // path_ctl_factory() );
    }

  } // namespace behavior
} // namespace html
