#include "html.h"

#ifdef NATIVE_WINDOWS_BACKEND
#include <shlobj.h>

namespace mswin {
  HGLOBAL packed_dib(gool::application *app, const gool::image *pd) {
    handle<gool::bitmap> bmp;
    if (pd->is_bitmap() &&
        static_cast<const gool::bitmap *>(pd)->get_bits().length)
      bmp = (gool::bitmap *)pd;
    else {
      bmp = new gool::bitmap(pd->dim(), true, true);
      handle<gool::graphics> gfx =
          app->create_bitmap_bits_graphics(bmp, gool::argb());
      if (gfx) gfx->draw(const_cast<gool::image *>(pd), gool::pointf(0, 0));
    }

    BITMAPINFO bitmap_info;
    memzero(bitmap_info);
    bitmap_info.bmiHeader.biSize        = sizeof(bitmap_info.bmiHeader);
    bitmap_info.bmiHeader.biWidth       = bmp->dim().x;
    bitmap_info.bmiHeader.biHeight      = 0 - bmp->dim().y;
    bitmap_info.bmiHeader.biPlanes      = 1;
    bitmap_info.bmiHeader.biBitCount    = 32;
    bitmap_info.bmiHeader.biCompression = BI_RGB;

    int imdatasize = bmp->stride() * pd->dim().y;

    HGLOBAL hPackedDib =
        GlobalAlloc(GMEM_MOVEABLE | GMEM_DDESHARE,
                    sizeof(bitmap_info.bmiHeader) + imdatasize);
    if (hPackedDib) {
      byte *ptr = (byte *)GlobalLock(hPackedDib);
      target(ptr, sizeof(bitmap_info.bmiHeader) + imdatasize)
          .xcopy(&bitmap_info.bmiHeader, sizeof(bitmap_info.bmiHeader))
          .xcopy(bmp->get_bits().start, imdatasize);
      GlobalUnlock(hPackedDib);
    }

    return hPackedDib;
  }

} // namespace mswin

#endif

namespace html {
  using namespace tool;
  using namespace gool;

  bool parse_cf_html(bytes cf_html, int &start_html, int &end_html,
                     int &start_fragment, int &end_fragment,
                     string &source_url) {
    const char *start = (const char *)cf_html.start;
    const char *ptr   = start;
    const char *end   = (const char *)cf_html.end();

    start_html     = 0;
    end_html       = 0;
    start_fragment = 0;
    end_fragment   = 0;

    while (*ptr && ptr < end) {
      if (start_html && ((ptr - start) >= start_html)) return true;

      while (ptr < end && (*ptr <= ' '))
        ptr++;
      if (*ptr == 0 || ptr >= end) break;
      // if(!isalpha(*ptr))
      //  return false;
      string name;
      while (ptr < end && (*ptr != ':'))
        name += *ptr++;
      ptr++;
      string value;
      while (ptr < end && (*ptr >= ' '))
        value += *ptr++;
      ptr++;

      name.to_lower();

      if (name == CHARS("starthtml"))
        start_html = atoi(value);
      else if (name == CHARS("endhtml"))
        end_html = atoi(value);
      else if (name == CHARS("startfragment"))
        start_fragment = atoi(value);
      else if (name == CHARS("endfragment")) {
        end_fragment = atoi(value);
      } else if (name == CHARS("sourceurl"))
        source_url = value;
    }
    return start_html && end_html && start_fragment && end_fragment;
  }

#ifdef NATIVE_WINDOWS_BACKEND
  // Get clipboard id for HTML format...

  CLIPFORMAT clipboard::CF_HTML() {
    static UINT id = 0;
    if (!id) id = RegisterClipboardFormatW(L"HTML Format");
    return CLIPFORMAT(id);
  }

  CLIPFORMAT clipboard::CF_JSON() {
    static UINT id = 0;
    if (!id) id = RegisterClipboardFormatW(L"application/json");
    return CLIPFORMAT(id);
  }

  CLIPFORMAT clipboard::CF_HYPERLINK() {
    static UINT id = 0;
    if (!id) id = RegisterClipboardFormatW(CFSTR_INETURL);
    return CLIPFORMAT(id);
  }

  CLIPFORMAT clipboard::CF_FILEDESCRIPTOR() {
    static UINT id = 0;
    if (!id) id = RegisterClipboardFormatW(CFSTR_FILEDESCRIPTOR);
    // if(!id) id = RegisterClipboardFormatW(L"FileGroupDescriptorW");
    return CLIPFORMAT(id);
  }

  CLIPFORMAT clipboard::CF_FILECONTENTS() {
    static UINT id = 0;
    if (!id) id = RegisterClipboardFormatW(CFSTR_FILECONTENTS);
    return CLIPFORMAT(id);
  }

  void clipboard::empty() {
    if (OpenClipboard(0)) {
      // Empty what's in there...
      EmptyClipboard();
      CloseClipboard();
    }
  }

  
  bool parse_cf_html(bytes cf_html, int &start_html, int &end_html,
                     int &start_fragment, int &end_fragment,
                     string &source_url);

  void clipboard::available_formats(
      tool::function<bool(clipboard::clipboard_format cf)> cb) {
    int mask = 0;

    CF_HTML();

    // CF_UNICODETEXT

    if (CountClipboardFormats() == 0) return;
    if (!OpenClipboard(NULL)) return;
    for (UINT fmt = EnumClipboardFormats(0); fmt;
         fmt      = EnumClipboardFormats(fmt)) {
#ifdef _DEBUG
      char buf[256];
      buf[0] = 0;
      GetClipboardFormatNameA(fmt, buf, 256);
#endif
      switch (fmt) {
      case CF_TEXT:
      case CF_UNICODETEXT:
        if ((mask & cf_text) == 0) {
          if (!cb(cf_text)) break;
        }
        mask |= cf_text;
        break;
      //case CF_DIB:
      case CF_BITMAP:
        if ((mask & cf_picture) == 0) {
          if (!cb(cf_picture)) break;
        }
        mask |= cf_picture;
        break;
      case CF_HDROP:
        if ((mask & cf_file) == 0) {
          if (!cb(cf_file)) break;
        }
        mask |= cf_file;
        break;
      default:
        if (fmt == CF_HTML()) {
          if (!cb(cf_html)) break;
          mask |= cf_html;
        } else if (fmt == CF_HYPERLINK()) {
          if (!cb(cf_hyperlink)) break;
          mask |= cf_hyperlink;
        }
#if 0
        else if (fmt == CF_HYPERLINK() || fmt == CF_FILEDESCRIPTOR()) {
          if ((mask & cf_hyperlink) == 0) {
            if (!cb(cf_hyperlink)) break;
          }
          mask |= cf_hyperlink;
        }
#endif
        break;
      }
    }
    CloseClipboard();
    return;
  }

  void clipboard::set(view &v, selection_ctx &sctx) {
    CF_HTML();
    empty();

    array<wchar> text;
    array<char>  html;
    text_cf(v, sctx, text);
    html_cf(v, sctx, html);

    if (OpenClipboard(0)) {
      _set_text(text());
      if (html.length()) _set_html(html());
      CloseClipboard();
    }
  }

  bool clipboard::set_text(wchars us) {
    if (OpenClipboard(0)) {
      _set_text(us);
      CloseClipboard();
    }
    return true;
  }

  void clipboard::_set_html(chars cf_html) {
    // Allocate global memory for transfer...
    HGLOBAL hHtmlText =
        ::GlobalAlloc(GMEM_MOVEABLE | GMEM_DDESHARE, cf_html.length + 1);
    if (hHtmlText) {
      // Put your string in the global memory...
      char *ptext = (char *)::GlobalLock(hHtmlText);
      target(ptext, cf_html.length + 1).copy(cf_html).copy('\0');
      ::GlobalUnlock(hHtmlText);
      ::SetClipboardData(CF_HTML(), hHtmlText);
    }
  }

  bool clipboard::_set_text(wchars us) {
    string s(us.start, us.length);

    // Allocate global memory for transfer...
    HGLOBAL hTextUnicode = GlobalAlloc(GMEM_MOVEABLE | GMEM_DDESHARE,
                                       (us.length + 1) * sizeof(wchar));
    HGLOBAL hText = GlobalAlloc(GMEM_MOVEABLE | GMEM_DDESHARE, s.length() + 1);

    if (hTextUnicode) {
      // Put ustring in the global memory...
      wchar *ptextunicode = (wchar *)GlobalLock(hTextUnicode);
      target(ptextunicode, us.length + 1).copy(us).copy(UCHAR('\0'));
      GlobalUnlock(hTextUnicode);
      ::SetClipboardData(CF_UNICODETEXT, hTextUnicode);
    }

    if (hText) {
      // Put string in the global memory...
      char *ptext = (char *)GlobalLock(hText);
      target(ptext, s.length() + 1).copy(s()).copy('\0');
      GlobalUnlock(hText);
      ::SetClipboardData(CF_TEXT, hText);
    }

    return true;
  }

  bool clipboard::get(ustring &data) {
    if (IsClipboardFormatAvailable(CF_UNICODETEXT)) {
      if (!OpenClipboard(NULL)) return false;
      HGLOBAL hglb = GetClipboardData(CF_UNICODETEXT);
      if (hglb != NULL) {
        wchar *pstr = (wchar *)GlobalLock(hglb);
        if (pstr != NULL) {
          size_t sz = min(str_len(pstr), 0xFFFFFFF);

          wchar *p   = pstr;
          wchar *end = p + sz;
          for (; p < end; ++p) {
            if (*p >= ' ') continue;
            if (*p == '\t' || *p == '\r' || *p == '\n') continue;
            *p = '?';
          }
          data = ustring(pstr, sz);
          GlobalUnlock(hglb);
        }
      }
      CloseClipboard();
    } else if (IsClipboardFormatAvailable(CF_TEXT)) {
      if (!OpenClipboard(NULL)) return false;
      HGLOBAL hglb = GetClipboardData(CF_TEXT);
      if (hglb != NULL) {
        char *pstr = (char *)GlobalLock(hglb);
        if (pstr != NULL) {
          size_t sz  = min(strlen(pstr), 0xFFFFFFF);
          char * p   = pstr;
          char * end = p + sz;
          for (; p < end; ++p) {
            if (*p >= ' ') continue;
            if (*p == '\t' || *p == '\r' || *p == '\n') continue;
            *p = '?';
          }
          data = string(pstr, sz);
          GlobalUnlock(hglb);
        }
      }
      CloseClipboard();
    } else
      return false;

    return true;
  }

  bool clipboard::get_image(handle<image> &data) {
    
    if (IsClipboardFormatAvailable(CF_DIB)) {
      if (!OpenClipboard(NULL))
        return false;

      HANDLE h = GetClipboardData(CF_DIB);
      if (!h)
        return false;

      byte * packed_dib = (byte *)GlobalLock(h);
      size_t packed_dib_size = GlobalSize(h);


      LPBITMAPINFOHEADER pBIH;
      void*           pDIBBits;

      pBIH = (LPBITMAPINFOHEADER)packed_dib;
      // point to DIB bits after BITMAPINFO object
      pDIBBits = (byte*)(pBIH + 1);

      gool::size sz(abs(pBIH->biWidth), abs(pBIH->biHeight));

      const gool::argb* pbits = (gool::argb*)pDIBBits;
      size_t            nbits = sz.x * sz.y;

      if (pBIH->biBitCount == 32) {
        handle<gool::bitmap> bmp = new gool::bitmap(sz);
        bmp->set_bits(slice<argb>(pbits, nbits));
        if (pBIH->biHeight > 0)
          bmp->flip_y();
        data = bmp;
      }
      else {
        dib32 dib(sz);
        ::StretchDIBits(dib.DC(),0, 0, dib.width(), dib.height(), 0, 0, pBIH->biWidth, pBIH->biHeight, pDIBBits, (LPBITMAPINFO)pBIH, DIB_RGB_COLORS, SRCCOPY);
        data = new bitmap(&dib, false);
      }

      GlobalUnlock(h);

      CloseClipboard();
      if (data)
        return true;
    }

    if (IsClipboardFormatAvailable(CF_BITMAP)) {
      if (!OpenClipboard(NULL)) return false;

      HBITMAP hbmp = (HBITMAP)GetClipboardData(CF_BITMAP);
      if (hbmp) {
        data = new bitmap(hbmp, false);
        CloseClipboard();
        return true;
      }
    }

    return false;
  }

  bool clipboard::get_files(array<ustring> &filenames) 
  {
    if (IsClipboardFormatAvailable(CF_HDROP)) {
      if (!OpenClipboard(NULL)) return false;

      //LPCWSTR pstr = (LPCWSTR)GetClipboardData(CF_HDROP);
      //if (!pstr) return false;

      HGLOBAL hglb = GetClipboardData(CF_HDROP);
      if (!hglb) return false;
      
      byte* pb = (byte*)GlobalLock(hglb);
      
      DROPFILES df;
      memcpy(&df, pb, sizeof(df));

      const wchar *pstr = (const wchar *) (pb + df.pFiles);
      
      for (;pstr;) {
        size_t l = wcsnlen_s(pstr, MAX_PATH);
        if (l == 0) break;
        filenames.push(wchars(pstr,l));
        pstr += (l + 1);
      }

      GlobalUnlock(hglb);

      CloseClipboard();

      return true;
    }
    else
      return false;

    return true;
  }


  void clipboard::set(const image *pd) {
    if (OpenClipboard(0)) {
      _set_image(pd);
      CloseClipboard();
    }
  }

  void clipboard::_set_image(const image *pd) {
   HGLOBAL hPackedDib = mswin::packed_dib(app(), pd);
   ::SetClipboardData(CF_DIB, hPackedDib);
   handle<bitmap> bmp = const_cast<image*>(pd)->get_bitmap(nullptr, pd->dim());
   HBITMAP hbmp = bmp->create_win_bitmap();
   ::SetClipboardData(CF_BITMAP, hbmp);
  }

  void clipboard::_set_files(const array<ustring>& filenames) {

    DROPFILES df;
    memzero(df);
    df.pFiles = sizeof(df);
    df.fWide = TRUE;

    array<byte> data;
    data.push((const byte *)&df, sizeof(df));

    for (int i = 0; i < filenames.size(); ++i) {
      ustring filename = filenames[i];
      if (filename.like(W("file://*"))) filename = filename().sub(7);
      filename.replace_all('/', '\\');
      data.push((const byte *)filename.c_str(), (filename.size() + 1) * sizeof(wchar));
    }
    data.push(0);
    data.push(0);
    HGLOBAL hg = GlobalAlloc(GMEM_MOVEABLE | GMEM_DDESHARE,data.size());
    if (hg) {
      byte *ptr = (byte *)GlobalLock(hg);
      target(ptr, data.size()).copy(data());
      GlobalUnlock(hg);
    }
    ::SetClipboardData(CF_HDROP, hg);
  }

  uint clipboard::get_sequence_num() {
#ifdef UNDER_CE
    static uint fix_me = 0;
    return ++fix_me;
#else
    return GetClipboardSequenceNumber();
#endif
  }

  bool clipboard::get_cf_html(array<byte> &text) {
    int formats = available_formats();
    if (formats & cf_html) {
      if (!OpenClipboard(NULL)) return false;
      HANDLE hTextHtml = ::GetClipboardData(CF_HTML());
      if (hTextHtml) {
        byte *ptext = (byte *)GlobalLock(hTextHtml);
#ifdef _DEBUG
        // FILE *f = 0; fopen_s(&f,"c:/clipboard.txt","w+");
        // fwrite(ptext,1,strlen((char*)ptext),f);
        // fclose(f);
#endif
        text.clear();
        text.push(ptext, strlen((char *)ptext));
        text.push(0);
        GlobalUnlock(hTextHtml);
        CloseClipboard();
        return true;
      }
      CloseClipboard();
    }
    return false;
  }

  bool clipboard::get_html(array<byte> &text, string &url) {
    if (!get_cf_html(text)) return false;

    int start_html     = 0;
    int end_html       = 0;
    int start_fragment = 0;
    int end_fragment   = 0;

    if (!parse_cf_html(text(), start_html, end_html, start_fragment,
                       end_fragment, url))
      return true;

    if (start_fragment < end_fragment) {
      text.remove(0, start_fragment);
      text.size(end_fragment - start_fragment);
    } else {
      text.remove(0, start_html);
      text.size(end_html - start_html);
    }

    // array<byte> html;
    // html.push(text(start_html,end_html));
    // text.swap(html);

    //<!--StartSelection-->
    //<!--EndSelection-->
    //?

    return true;
  }

  bool clipboard::get_link(ustring &url, ustring &caption) {
    int formats = available_formats();
    if (formats & cf_hyperlink) {
      if (!OpenClipboard(NULL)) return false;
      HANDLE hTextHyperlink = ::GetClipboardData(CF_HYPERLINK());
      if (hTextHyperlink) {
        ustring path = (wchar *)GlobalLock(hTextHyperlink);
        ustring uri = url::path_to_file_url(path);
        caption = path;
        GlobalUnlock(hTextHyperlink);

        if (hTextHyperlink = ::GetClipboardData(CF_FILEDESCRIPTOR())) {
          FILEGROUPDESCRIPTORW *fd = (FILEGROUPDESCRIPTORW *)GlobalLock(hTextHyperlink);
          if (fd) {
            caption = fd->fgd[0].cFileName;
            if (url.is_undefined())
              url = url::path_to_file_url(caption);
          }
          GlobalUnlock(hTextHyperlink);
          if (caption.like(L"*.url"))
            caption = caption().sub(0, caption.size() - 4);
          if (caption.length() == 0) caption = uri;
        }
      }
      CloseClipboard();
      return url.is_defined() && caption.is_defined();
    }
    return false;
  }
#endif

#ifdef POSIX
#define sprintf_s snprintf
#endif

  void clipboard::text_cf(view &v, selection_ctx &sctx, array<wchar> &text) {
    ostream_w os;
    switch (sctx.get_selection_type(v)) {
    case TEXT_RANGE:
      emit_range_text(v, os, sctx.normalized().first, sctx.normalized().second);
      break;
    case CELL_RANGE:
      emit_cell_range_text(v, os, sctx.table, sctx.selected());
      break;
    default:
      emit_range_text(v, os, sctx.normalized().first, sctx.normalized().second);
      break;
    }

    os.data.swap(text);
  }

  void clipboard::html_cf(view &v, const bookmark &start, const bookmark &end,
                          array<char> &html) {
    ostream_8 osb;
    document *doc = start.node->doc();

    emit_range_html(v, osb, start, end, doc);

    chars range_chars;
    range_chars.start  = (char *)osb.data.head();
    range_chars.length = osb.data.length();


    html_cf(range_chars, doc->uri().src(), html);
  }

  void clipboard::html_cf(view &v, selection_ctx &sctx, array<char> &html) {
    ostream_8 osb;

    element* root = sctx.selection_root();

    switch (sctx.get_selection_type(v)) {
    case TEXT_RANGE:
      emit_range_html(v, osb, sctx.normalized().first,
                      sctx.normalized().second,root);
      break;
    case CELL_RANGE:
      emit_cell_range_html(v, osb, sctx.table, sctx.selected());
      break;
    }

    chars range_chars;
    range_chars.start  = (char *)osb.data.head();
    range_chars.length = osb.data.length();

    document *doc = sctx.caret.node->doc();
#if defined(WINDOWS)
    html_cf(range_chars, doc->uri().src(), html);
#else
    html = range_chars;
#endif
  }

  void clipboard::html_cf(chars html_fragment, chars url, array<char> &html) {
    array<char> buffer;
    chars       range_chars = html_fragment;

    if (!html_fragment.contains(chars_of(MARKER_START_FRAGMENT))) {
      buffer.push(chars_of(MARKER_START_FRAGMENT));
      buffer.push(html_fragment);
      buffer.push(chars_of(MARKER_END_FRAGMENT));
      range_chars = buffer();
    }

    // auto nr = range_chars.find('\0');
    // assert(!nr);

    html.length(400 + range_chars.length);
    html.length(0);

    // Create a template string for the HTML header...

    html += CHARS("Version:1.0\r\n"
                  "StartHTML:00000000\r\n"
                  "EndHTML:00000000\r\n"
                  "StartFragment:00000000\r\n"
                  "EndFragment:00000000\r\n");
    html += CHARS("SourceUrl:");
    html += url;
    html += CHARS("\r\n");

    int header_length = html.size();

    // if(doc->get_style(v)->direction == direction_rtl )
    //  html += CHARS("<html dir=\"rtl\">");
    // else
    html += CHARS("<html>");

    // Append the HTML...
    html += range_chars;
    // html += CHARS("\r\n");
    html += CHARS("</html>");

    // Now go back, calculate all the lengths, and write out the
    // necessary header information. Note, wsprintf() truncates the
    // string when you overwrite it so you follow up with code to replace
    // the 0 appended at the end with a '\r'...
    char *ptr = strstr(html.head(), "StartHTML:");
    memcpy(ptr + 10, tool::itoa(header_length, 10, 8, '0').start, 8);
    // sprintf_s(ptr+10, 10, "%08u", header_length);
    //   *(ptr+10+8) = '\r';

    ptr = strstr(html.head(), "EndHTML:");
    memcpy(ptr + 8, tool::itoa(html.size(), 10, 8, '0').start, 8);
    // sprintf_s(ptr+8,10, "%08u", html.size());
    //*(ptr+8+8) = '\r';

    const char *start_fragment_ptr = strstr(html.head(), MARKER_START_FRAGMENT);
    const char *end_fragment_ptr   = strstr(html.head(), MARKER_END_FRAGMENT);

    if (!start_fragment_ptr) goto FAILURE;
    ptr = strstr(html.head(), "StartFragment:");
    // sprintf_s(ptr+14, 10, "%08u", unsigned(start_fragment_ptr -
    // html.head()));
    //   *(ptr+14+8) = '\r';
    memcpy(ptr + 14,
           tool::itoa(unsigned(start_fragment_ptr - html.head()), 10, 8, '0')
               .start,
           8);
    if (!end_fragment_ptr) goto FAILURE;
    assert(end_fragment_ptr);
    ptr = strstr(html.head(), "EndFragment:");
    memcpy(
        ptr + 12,
        tool::itoa(unsigned(end_fragment_ptr - html.head()), 10, 8, '0').start,
        8);
    // sprintf_s(ptr+12, 10, "%08u", unsigned(end_fragment_ptr - html.head()));
    //   *(ptr+12+8) = '\r';

    // Now we have everything in place ready to put on the
    // clipboard.

    #if defined(_DEBUG) && defined(WINDOWS)
        FILE * f = 0; fopen_s(&f,"d:/htmlarea.html.txt","wb");
        if(f)
        {
          fwrite(html.head(),html.size(),1,f);
          fclose(f);
        }
    #endif
    return;
  FAILURE:
    chars  html_chars = html();
    size_t np         = html_chars.find('\0').start - html_chars.start;
    assert(np == html.length());
    np = np;
    html.clear();
  }

  bool clipboard::get_text(array<byte> &text) {
    ustring us;
    if (!get(us)) return false;
    u8::x_from(us, text);
    return true;
  }

  bool clipboard::get_json(tool::value &json) {
#pragma TODO("JSON clipboard format")
    return false;
  }

  clipboard::html_item *clipboard::html_item::from_cf_html(bytes data) {
    clipboard::html_item *hi = new clipboard::html_item();

    int start_html     = 0;
    int end_html       = 0;
    int start_fragment = 0;
    int end_fragment   = 0;

    parse_cf_html(data, start_html, end_html, start_fragment, end_fragment,
                  hi->url);

    // if( start_fragment < end_fragment )
    //  html = val.bytes()(start_fragment,end_fragment);
    // else
    if (start_html < end_html)
      hi->val = data(start_html, end_html);
    else
      hi->val = data;

    return hi;
  }

  clipboard::data *clipboard::get(uint formats) {

    array<clipboard::clipboard_format> list;
    available_formats([formats,&list](clipboard::clipboard_format cf) -> bool {
      if((formats & cf) != 0)
        list.push(cf);
      return true;
    });

    clipboard::data *dt = new clipboard::data();

    FOREACH(i, list)
    switch (list[i]) 
    {
      case cf_html: 
      {
        array<byte> html;
#if defined(WINDOWS)
        clipboard::get_cf_html(html);
        dt->add(html_item::from_cf_html(html()));
#elif defined(OSX) && !defined(WINDOWLESS)
        // clipboard::get_html(html);
        // dt->add(html_item::from_html(html()));
        dt->add(html_item::from_pasteboard(nullptr));
#elif defined(LINUX)
        string url;
        clipboard::get_html(html, url);

        html_item *pit = new html_item();
        pit->val       = chars((const char *)html.cbegin(), html.length());
        ;
        pit->url = url;
        dt->add(pit);
  #endif
      } break;
      case cf_text: {
        ustring text;
        clipboard::get(text);
        dt->add(new text_item(text));
      } break;
      case cf_hyperlink: {
        ustring url, caption;
        clipboard::get_link(url, caption);
        dt->add(new link_item(caption, url));
      } break;
      case cf_json: {
        ustring text;
        clipboard::get(text);
        dt->add(new text_item(text));
      } break;

      case cf_picture: {
        handle<gool::image> img;
        if (clipboard::get_image(img)) dt->add(new image_item(img));
      } break;

      case cf_file: {
        array<ustring> filenames;
        if (clipboard::get_files(filenames))
          dt->add(new file_item(filenames));
      } break;

    }

    return dt;
  }

#if defined(WINDOWS) || defined(OSX)
  bool clipboard::set(clipboard::data *cbdata) {
    empty();
#ifdef WINDOWS
    OpenClipboard(0);
#endif
    int cnt = 0;
    for (int n = 0; n < cbdata->items.size(); ++n) {
      switch (cbdata->items[n]->data_type) {
      case clipboard::cf_text: {
        clipboard::text_item *it =
            cbdata->items[n].ptr_of<clipboard::text_item>();
        clipboard::_set_text(it->val());
        ++cnt;
      } break;
      case clipboard::cf_html: {
        clipboard::html_item *it =
            cbdata->items[n].ptr_of<clipboard::html_item>();
        clipboard::_set_html(it->val());
        ++cnt;
      } break;
      // case clipboard::cf_hyperlink: {
      //  clipboard::link_item* it =
      //  cbdata->items[n].ptr_of<clipboard::link_item>();
      //  clipboard::_set_link(it->url(),it->caption());
      //  ++cnt;
      //} break;
      // case clipboard::cf_json: {
      //} break;
      case clipboard::cf_picture: {
        clipboard::image_item *it =
            cbdata->items[n].ptr_of<clipboard::image_item>();
        clipboard::_set_image(it->image);
        ++cnt;
      } break;

      case clipboard::cf_file: {
        clipboard::file_item *it =
          cbdata->items[n].ptr_of<clipboard::file_item>();
        clipboard::_set_files(it->filenames);
        ++cnt;
      } break;

      }
    }

#ifdef WINDOWS
    CloseClipboard();
#endif
    return cnt > 0;
  }
#endif

} // namespace html
