#ifndef __html_behavior_h__
#define __html_behavior_h__

#include "tool/tool.h"
#include "html-primitives.h"
#include "html-events.h"
#include "html-scrollbar.h"

#include "sdk-headers.h"

namespace html {
  class view;
  struct document;
  struct element;
  struct iwindow;
  struct selection_ctx;

  enum CTL_TYPE {
    CTL_NO      = 0, ///< This dom element has no behavior at all.
    CTL_UNKNOWN = 1, ///< This dom element has behavior but its type is unknown.

    CTL_EDIT            = 2,  ///< Single line edit box.
    CTL_NUMERIC         = 3,  ///< Numeric input with optional spin buttons.
    CTL_CLICKABLE       = 4,  ///< toolbar button, behavior:clickable.
    CTL_BUTTON          = 5,  ///< Command button.
    CTL_CHECKBOX        = 6,  ///< CheckBox (button).
    CTL_RADIO           = 7,  ///< OptionBox (button).
    CTL_SELECT_SINGLE   = 8,  ///< Single select, ListBox or TreeView.
    CTL_SELECT_MULTIPLE = 9,  ///< Multiselectable select, ListBox or TreeView.
    CTL_DD_SELECT       = 10, ///< Dropdown single select.
    CTL_TEXTAREA        = 11, ///< Multiline TextBox.
    CTL_HTMLAREA        = 12, ///< HTML selection behavior.
    CTL_PASSWORD        = 13, ///< Password input element.
    CTL_PROGRESS        = 14, ///< Progress element.
    CTL_SLIDER          = 15, ///< Slider input element.
    CTL_DECIMAL         = 16, ///< Decimal number input element.
    CTL_CURRENCY        = 17, ///< Currency input element.
    CTL_SCROLLBAR       = 18,
    CTL_LIST            = 19,
    CTL_RICHTEXT        = 20,
    CTL_CALENDAR        = 21,
    CTL_DATE            = 22,
    CTL_TIME            = 23,
    CTL_FILE            = 24, ///< file input element.
    CTL_PATH            = 25, ///< path input element.

    CTL_LAST_INPUT = 26,

    CTL_HYPERLINK = CTL_LAST_INPUT,
    CTL_FORM      = 27,

    CTL_MENUBAR    = 28,
    CTL_MENU       = 29,
    CTL_MENUBUTTON = 30,

    CTL_FRAME    = 31,
    CTL_FRAMESET = 32,

    CTL_TOOLTIP = 33,

    CTL_HIDDEN  = 34,
    CTL_URL     = 35, ///< URL input element.
    CTL_TOOLBAR = 36,

    CTL_WINDOW = 37, ///< has HWND attached to it

    CTL_LABEL     = 38,
    CTL_IMAGE     = 39, ///< image/video object.
    CTL_PLAINTEXT = 40, ///< Multiline TextBox + colorizer.
  };

  enum BEHAVIOR_METHOD_IDENTIFIERS {
    DO_CLICK       = 0,
    GET_TEXT_VALUE = 1,
    SET_TEXT_VALUE,
    TEXT_EDIT_GET_SELECTION,
    TEXT_EDIT_SET_SELECTION,
    TEXT_EDIT_REPLACE_SELECTION,
    SCROLL_BAR_GET_VALUE,
    SCROLL_BAR_SET_VALUE,
    TEXT_EDIT_GET_CARET_POSITION,
    TEXT_EDIT_GET_SELECTION_TEXT, // p - TEXT_SELECTION_PARAMS
    TEXT_EDIT_GET_SELECTION_HTML, // p - TEXT_SELECTION_PARAMS
    TEXT_EDIT_CHAR_POS_AT_XY,
    FRAME_RESTORE   = 0xfe,
    XCALL_METHOD_ID = 0xff, // p - XCALL_PARAMS
  };

  struct method_params {
    uint method_id;
  };

  struct selection_params : method_params {
    uint selection_start;
    uint selection_end;
  };

  struct position_params : method_params {
    int left;
    int top;
    int width;
    int height;
  };

  struct text_params : method_params {
    const wchar *text;
    uint         text_length;
  };

  struct xcall_params : method_params {
    const char *method_name;
    uint        argc;
    value *     argv;
    value       retval;
  };

  struct frame_restore_params : method_params {
    frame_restore_params() { method_id = FRAME_RESTORE; }
    handle<document> doc;
    handle<element>  focus;
    point            pos;
    size             dim;
  };

  struct text_selection_params;
  typedef uint CALLBACK outs_proc(text_selection_params *params,
                                  uint data /* BYTE or WCHAR */);

  struct text_selection_params : method_params {
    outs_proc *outs; // output stream

    void output(wchars text) {
      if (outs)
        for (uint n = 0; n < text.length; ++n)
          outs(this, text[n]);
    }
    void output(bytes html) {
      if (outs)
        for (uint n = 0; n < html.length; ++n)
          outs(this, html[n]);
    }
  };

  struct text_char_pos_at_xy_params : method_params {
    int      x, y;     // in
    int      char_pos; // out
    element *he;       // out
    int      he_pos;   // out
  };

  struct scripting_params : method_params {
    const char * method_name;
    uint         argc;
    const value *argv;
    value        retval;
    bool         csss_call;
    scripting_params(bool by_csss = false) {
      method_id = XCALL_METHOD_ID;
      csss_call = by_csss;
    }
    static bool is_csss_call(method_params *prms) {
      return prms->method_id == XCALL_METHOD_ID &&
             ((scripting_params *)prms)->csss_call;
    }
  };

  struct event_handler {
    virtual bool on(view &v, element *self, event_mouse &evt) { return false; }
    virtual bool on(view &v, element *self, event_key &evt) { return false; }
    virtual bool on(view &v, element *self, event_focus &evt) { return false; }
    virtual bool on(view &v, element *self, event_command &evt) { return false; }
    virtual bool on(view &v, element *self, event_scroll &evt) { return false; }
    virtual bool on(view &v, element *self, event_behavior &evt) { return false; }
    virtual bool on(view &v, element *self, event_exchange &evt) { return false; }
    virtual bool on(view &v, element *self, event_gesture &evt) { return false; }
  };

  struct ctl : 
      public sciter::om::asset<ctl>,
      public event_handler, public virtual resource {
    handle<ctl>     next;
    uint            subscription;

    ctl() : next(0), subscription(0xFFFFFFFF & ~HANDLE_SIZE) {}
    ctl(uint subs) : next(0), subscription(subs) {}
    virtual ~ctl() {}

    virtual long  asset_add_ref() override { 
      return resource::add_ref(); 
    }
    virtual long  asset_release() override { 
      return resource::release(); 
    }

    virtual som_asset_t* as_asset() { return static_cast<som_asset_t*>(this); }

    virtual const string &behavior_name() const = 0;
    virtual CTL_TYPE      get_type()            = 0;
    virtual bool          is_intrinsic() const { return false; } // true means intrinsic, non removable behavior

    virtual bool is_ext() const { return false; }
    virtual bool is_atomic() const { return true; }

    bool will_handle(uint flag) { return (flag & subscription) != 0; }

    virtual bool get_expando(context&, script_expando& seo) { return false; }

    virtual void scan_owned_elements(const function<bool(element *)> &scanner) {}

    virtual bool attach(view &v, element *self) { return true; }
    virtual void detach(view &v, element *self) {}
    // virtual bool on_paint   (view& v, element* self, surface& sf, const rect&
    // rc ) { return false; }
    virtual bool on_timer(view &v, element *self, timer_id tid, TIMER_KIND kind) { return false; /*stop this timer*/ }

    virtual void clear_style() { }

    virtual void accept_image(view &v, element *self, const image_ref &ir){};

    virtual void invalidate_layout(element *self) {}
    virtual void invalidate_surface(element *self, const rect& invalid_rc) {}

    virtual bool a11y_get_focus(view &v, element *self, handle<element>& pf) { return false; }
    virtual bool a11y_get_children(element *self, array<handle<node>>& children) { return false; }
    virtual bool a11y_get_state(view &v, element *self, ui_state& st) { return false; }
    virtual bool a11y_is_atomic(view &v, element *self) { return false; }
    virtual bool a11y_get_value(view &v, element *self, ustring& sval) { 
      value val;
      if (get_value(v, self, val) && val.is_defined()) {
        sval = val.to_string();
        return true;
      } 
      return false;
    }

    virtual bool handle_method_call(view &v, element *self,
                                    method_params *params) {
      if (params->method_id == XCALL_METHOD_ID) {
        xcall_params *xp = (xcall_params *)params;
        if (on_x_method_call(v, self, xp->method_name, xp->argv, xp->argc,
                             xp->retval))
          return true;
      }
      return on_method_call(v, self, params);
    }

    virtual bool on_method_call(view &v, element *self, method_params *params) {
      return false;
    }

    // behavior method call from script
    virtual bool on_x_method_call(view &v, element *self, const char *name,
                                  const value *argv, size_t argc,
                                  value &retval) {
      return false;
    }
#ifdef SCITER
    /*virtual bool on_script_method_call(view &v, element *self, tis::VM *c,
                                       tis::value tag, tis::value &retval) {
      return false;
    }*/
#endif

    virtual bool on_data_arrived(view &v, element *self, pump::request *rq) {
      return false;
    }
    // data request is about to be sent:
    virtual bool on_data_request(view &v, element *self, pump::request *rq) {
      return false;
    }

    virtual void on_document_loaded(view &v, element *self) {}

    virtual void on_size_changed(view &v, element *self) {}

    // called when this behavior gets new companions on the element
    // (css::behavior changes)
    virtual void on_behaviors_changed(view &v, element *self) {}

    virtual void on_attr_change(view &v, element *self, const name_or_symbol &nm);

    virtual bool get_auto_width(view &v, element *self, int &value) {
      return false;
    }
    virtual bool get_auto_height(view &v, element *self, int &value) {
      return false;
    }

    virtual bool line_ascent(view &v, element *self, int &value) {
      return false;
    }

    virtual bool get_scroll_data(view &v, element *self, scroll_data &sd) {
      return false;
    }
    virtual bool set_scroll_pos(view &v, element *self, point pos,
                                bool smooth) {
      return false;
    }
    virtual bool update_scroll_position(view &v, element *self) {
      return false;
    }

    virtual image *get_fore_image(view &v, element *self) { return 0; }

    // for behavior:layer 
    virtual bool draw(view &v, graphics *sf, element *self, point pos) { return false; }

    virtual bool draw_foreground(view &v, element *self, graphics *sf,
                                 point pos) {
      return false;
    }
    virtual bool draw_background(view &v, element *self, graphics *sf,
                                 point pos) {
      return false;
    }
    virtual bool draw_content(view &v, element *self, graphics *sf, point pos) {
      return false;
    }

    virtual bool draw_outline(view &v, element *self, graphics *sf, point pos) {
      return false;
    }

    virtual bool draw_scrollbars(view &v, element *self, graphics *sf,
                                 point pos) {
      return false;
    }

    inline bool _draw_foreground(view &v, element *self, graphics *sf,
                                 point pos) {
      for (ctl *ct = this; ct; ct = ct->next)
        if (ct->draw_foreground(v, self, sf, pos)) return true; // done
      return false;
    }
    inline bool _draw_background(view &v, element *self, graphics *sf,
                                 point pos) {
      for (ctl *ct = this; ct; ct = ct->next)
        if (ct->draw_background(v, self, sf, pos)) return true; // done
      return false;
    }
    inline bool _draw_content(view &v, element *self, graphics *sf, point pos) {
      for (ctl *ct = this; ct; ct = ct->next)
        if (ct->draw_content(v, self, sf, pos)) return true; // done
      return false;
    }

    inline bool _draw_outline(view &v, element *self, graphics *sf, point pos) {
      for (ctl *ct = this; ct; ct = ct->next)
        if (ct->draw_outline(v, self, sf, pos)) return true; // done
      return false;
    }


    virtual bool focusable(const element *self) { return false; }

    virtual bool set_value(view &v, element *self, const value &val) { return false; }
    virtual bool get_value(view &v, element *self, value &val) { return false; }

    virtual bool set_text(view &v, element *self, wchars val) { return false; }
    virtual bool get_text(view &v, element *self, ustring &val) { return false; }

    virtual bool set_html(view &v, element *self, bytes html) { return false; }

    // surface&     get_surface( view& v, element* b );

    virtual bool get_caret_place(view &v, element *self, rect &rc) {
      return false;
    }

    virtual bool set_child_state(view &v, element *self, element *child,
                                 const ui_state &st);
    virtual bool reset_child_state(view &v, element *self, element *child,
                                   const ui_state &st);

    // <select> returns self, <select|dropdown> returns <popup>
    virtual element *get_options_container(element *self) { return 0; }

    virtual bool is_readonly(const element *b) const;
    virtual bool is_disabled(const element *b) const;

    virtual bool is_editable(const element *b) const { return !is_disabled(b) && !is_readonly(b); }

    virtual bool wants_keyboard(const element *b) const { return false; }

    virtual bool is_empty(const element *self, bool &yes) { return false; }

    // get attribute from CSS custom attributes collection or from "atts rack",
    // name here shall start from "-"
    // static bool    get_attr(const element* self, const char* name, ustring&
    // val);  static bool    has_attr(const element* self, const char* name);

    ustring get_attr(element *self, const char *name);
    int     get_attr(element *self, const char *name, int dv);
    double  get_attr(element *self, const char *name, double dv);

    virtual void on_lang_change(view& pv, element* self) {}
    virtual void on_theme_change(view& pv, element* self) {}

    virtual bool get_window(iwindow *&hw) { return false; }

    virtual selection_ctx *get_selection_ctx() const { return 0; }

    virtual element* r13n_container(element* self) { return nullptr; /* reconcilliation of children is prohibited 
                                                                        by default on all elements having behavior applied. */ }
    virtual element* logical_container(element* self) {
      return self; 
    }

  };

  typedef handle<ctl> behavior_h;

  struct ctl_factory : resource {
    // behavior list support
    string name; // name must be pointer to a static string

    ctl_factory(const char *n);
    virtual ~ctl_factory() {}

    // returns behavior implementation by name.
    static ctl *produce(element *el, const string &name);
    static hash_table<string, handle<ctl_factory>> all;

    static void add(ctl_factory *cf) { all[cf->name] = cf; }

    virtual ctl *create(element *el) = 0;

    static void clear_all() { all.clear(); }
  };

} // namespace html

#endif
