#include "html.h"

namespace html {

  scroll_animator::scroll_animator(view& v, point fpos, scroll_mode smode)
    : final_scroll_pos(fpos), step_no(0), prev_clock(0), mode(smode),
    inertia(0) {
#ifdef USE_TOUCH
    is_touch = v.is_touched;
#endif
  }
  
  void scroll_animator::set_final_pos(view &v, point fp) {
    final_scroll_pos = fp;
  }

  uint scroll_animator::start(view &v, element *b, const style *new_style,
                              const style *old_style) {
    self = b;
    scroll_pos = b->scroll_pos();
    //?? wrong: v.check_mouse(true); - will generate MOUSE_LEAVE needlessly.

    b->commit_measure(v);
    v.commit_update();

    b->flags.animating_scroll = 1;

    event_scroll evt(b, SCROLL_ANIMATION_START, true, scroll_pos.y, SCROLL_SOURCE_ANIMATOR, 0);
    b->on(v, evt);

    return ANIMATION_TIMER_SPAN;
  }

  
  uint scroll_animator::step(view &v, element *b, uint current_clock) {
    if (!b->is_drawable(v)) return 0;

    b->commit_measure(v);
    v.commit_update();

    switch (mode) {
    case WHEEL: return step_wheel(v, b, current_clock);
    case INERTIAL: return step_inertial(v, b, current_clock);
    }

    return 0;
  }

#if 1
  // less inertia + more fading
  uint scroll_animator::step_wheel(view &v, element *b, uint current_clock) {
    point spos = b->scroll_pos();

    if (spos == final_scroll_pos)
      return check_stop(v,b, current_clock);

    size velocity(final_scroll_pos.x - spos.x, final_scroll_pos.y - spos.y);

#ifdef USE_TOUCH
    bool allow_overscroll = USE_OVERSCROLL && is_touch && (b->get_style(v)->overflow() == overflow_scroll_indicator);
#else
    bool allow_overscroll = false;
#endif
    

    int dt = prev_clock ? max(1, int(next_clock) - int(prev_clock)) : int(ANIMATION_TIMER_SPAN);

    velocity.x = (velocity.x * dt) / ANIMATION_TIMER_SPAN;
    velocity.y = (velocity.y * dt) / ANIMATION_TIMER_SPAN;

    delta.x = limit(velocity.x, -1, 1);
    delta.y = limit(velocity.y, -1, 1);
    int maxx = b->ldata->dim_min.x * 2;
    int maxy = b->ldata->dim_min.y * 2;
    const int multiplier =
#if defined(OSX) && defined(USE_TOUCH)
      is_touch ? 6 : 3;
#else
      3;
#endif
      
    const float inertia_factor = multiplier * v.pixels_per_dip(size(1, 1)).y; // 1/7th

    velocity.x = limit(int(abs(velocity.x) / inertia_factor), 1, maxx) * delta.x;
    velocity.y = limit(int(abs(velocity.y) / inertia_factor), 1, maxy) * delta.y;
    //  velocity.y *= 99; velocity.y /= 100;
    //  velocity.x *= 99; velocity.x /= 100;

    point nspos = spos;
    nspos.x += velocity.x; 
    nspos.y += velocity.y;
        
    b->scroll_pos(v, nspos, allow_overscroll);
    nspos = b->scroll_pos();
    point d = spos - nspos;
    if (d.x == 0 && d.y == 0) 
      return check_stop(v, b, current_clock);

    if (d.x) {
      event_scroll evt(b, SCROLL_POS, false, nspos.x, SCROLL_SOURCE_ANIMATOR, 0);
      b->on(v, evt);
    }
    if (d.y) {
      event_scroll evt(b, SCROLL_POS, true, nspos.y, SCROLL_SOURCE_ANIMATOR, 0);
      b->on(v, evt);
    }
    v.refresh(b);

    prev_clock = current_clock;
    return ANIMATION_TIMER_SPAN;
  }

#else

  uint scroll_animator::step_wheel(view &v, element *b, uint current_clock) {
    point spos = b->scroll_pos();

    if (spos == final_scroll_pos) return 0;

    size velocity(final_scroll_pos.x - spos.x, final_scroll_pos.y - spos.y);

    bool allow_overscroll = USE_OVERSCROLL && (b->get_style(v)->overflow() == overflow_scroll_indicator);

    int dt = prev_clock ? max(1, int(next_clock) - int(prev_clock)) : int(ANIMATION_TIMER_SPAN);
    
    velocity.x = (velocity.x * dt) / ANIMATION_TIMER_SPAN;
    velocity.y = (velocity.y * dt) / ANIMATION_TIMER_SPAN;

    delta.x  = limit(velocity.x, -1, 1);
    delta.y  = limit(velocity.y, -1, 1);
    int maxx = b->ldata->dim_min.x * 2; // / 7;
    int maxy = b->ldata->dim_min.y * 2; // / 7;

    const int inertia_factor = 7 * v.pixels_per_dip(size(1,1)).y; // 1/7th

    velocity.x  = limit(abs(velocity.x) / inertia_factor, 1, maxx) * delta.x;
    velocity.y  = limit(abs(velocity.y) / inertia_factor, 1, maxy) * delta.y;

    if (velocity.x == 0 && velocity.y == 0) {
      final_scroll_pos = spos;
      return 0;
    }

    point nspos = spos;
    nspos.x += velocity.x;
    nspos.y += velocity.y;

    b->scroll_pos(v, nspos, allow_overscroll);
    nspos   = b->scroll_pos();
    point d = spos - nspos;
    if (d.x == 0 && d.y == 0) return 0;

    //#ifdef _DEBUG
    // uint nticks = v.get_ticks();
    //_dprintf("tdelta=%d ydelta=%d\n",nticks - pticks, d.y);
    // pticks = nticks;
    //#endif

    if (d.x) {
      event_scroll evt(b, SCROLL_POS, false, nspos.x);
      b->on(v, evt);
    }
    if (d.y) {
      event_scroll evt(b, SCROLL_POS, true, nspos.y);
      b->on(v, evt);
    }
    v.refresh(b);

    if (allow_overscroll) {
      scroll_data sd;
      b->get_scroll_data(v, sd);

      point _final_scroll_pos = final_scroll_pos;

      final_scroll_pos.x = limit(final_scroll_pos.x, sd.content_outline.left(),
        sd.content_outline.right() + 1 - sd.dim_view.x);
      final_scroll_pos.y = limit(final_scroll_pos.y, sd.content_outline.top(),
        sd.content_outline.bottom() + 1 - sd.dim_view.y);
    }
    prev_clock = current_clock;
    return ANIMATION_TIMER_SPAN;
  }
#endif

  inline float sign1(float x) {
    if (x > 0.0001f) return 1.f;
    if (x < -0.0001f) return -1.f;
    return 0.f;
  }

  sizef speed_at_t(
      sizef v0, uint t,
      float k =
          0.0972f /*  friction coefficient of iPhone sliding on office table surface - iCurling, eh? */) {
    // v(t)=v0-F*t/m  where F is friction force = k*m*g
    // v(t)=v0-(k*m*g)*t/m
    // v(t)=v0-k*g*t;
    const float g = 9.8f; // gravitation acceleration at Earth surface.
    float       dv =
        k * g * float(t) / 1000.f; // 1000. as time and speed use milliseconds
    sizef v;
    v.x = (v0.x < 0.f) ? min(v0.x + dv, 0.f) : max(v0.x - dv, 0.f);
    v.y = (v0.y < 0.f) ? min(v0.y + dv, 0.f) : max(v0.y - dv, 0.f);
    return v;
  }

  uint scroll_animator::step_inertial(view &v, element *b, uint current_clock) {
    // const uint duration = 800; // ms

    point spos = b->scroll_pos();
    if (spos == final_scroll_pos) return 0;

    uint step_time = current_clock - prev_clock;
    if (prev_clock == 0) step_time = ANIMATION_TIMER_SPAN;
    prev_clock = current_clock;

    uint delay = ANIMATION_TIMER_SPAN;

    sizef step_speed = speed_at_t(speed, current_clock - start_clock);

    sizef distance = step_speed * float(step_time);

    sizef adistance(abs(distance.x), abs(distance.y));

    if (adistance.x < 1.0f && adistance.y < 1.0f) {
      if (adistance.x < 0.3f && adistance.y < 0.3f) return 0;

      uint delay_y =
          adistance.y > 0.f ? uint(float(delay) / adistance.y) : 10000;
      uint delay_x =
          adistance.x > 0.f ? uint(float(delay) / adistance.x) : 10000;

      distance.x = sign1(distance.x);
      distance.y = sign1(distance.y);

      delay = min(delay_x, delay_y);
    }

    point nspos = spos;
    nspos.x += int(distance.x);
    nspos.y += int(distance.y);
    b->scroll_pos(v, nspos);
    nspos   = b->scroll_pos();
    point d = spos - nspos;

    if (d.x) {
      event_scroll evt(b, SCROLL_POS, false, nspos.x, SCROLL_SOURCE_ANIMATOR, 0);
      b->on(v, evt);
    }
    if (d.y) {
      event_scroll evt(b, SCROLL_POS, true, nspos.y, SCROLL_SOURCE_ANIMATOR, 0);
      b->on(v, evt);
    }
    v.refresh(b);

    scroll_data sd;
    b->get_scroll_data(v, sd);

    final_scroll_pos.x = limit(final_scroll_pos.x, sd.content_outline.left(),
                               sd.content_outline.right() + 1 - sd.dim_view.x);
    final_scroll_pos.y = limit(final_scroll_pos.y, sd.content_outline.top(),
                               sd.content_outline.bottom() + 1 - sd.dim_view.y);

    return delay;
  }

  uint scroll_animator::check_stop(view& v, element* b, uint current_clock) {
    //return 0;
    hstyle st = b->get_style(v);
    if (!st->smooth_scroll_wheel(true))
      return 0;

#ifdef USE_TOUCH
    bool allow_overscroll = USE_OVERSCROLL && is_touch && (b->get_style(v)->overflow() == overflow_scroll_indicator);
    if (!allow_overscroll)
      return 0;

    scroll_data sd;
    b->get_scroll_data(v, sd);

    bool rollback_y = false;
    bool rollback_x = false;

    prev_clock = current_clock;

    //if(v.is_touched)
    //  return ANIMATION_TIMER_SPAN;

    if (b->can_scroll_v(v) && sd.content_outline.height() > sd.dim_view.y) {
      if (sd.pos.y < 0) {
        rollback_y = true;
        sd.pos.y = 0;
      }
      if (sd.pos.y > (sd.content_outline.bottom() - sd.dim_view.y + 1))
      {
        rollback_y = true;
        sd.pos.y = sd.content_outline.bottom() - sd.dim_view.y + 1;
      }
    }

    if (b->can_scroll_h(v) && sd.content_outline.width() > sd.dim_view.x) {
      if (sd.pos.x < 0) {
        rollback_x = true;
        sd.pos.x = 0;
      }
      if (sd.pos.x > (sd.content_outline.right() - sd.dim_view.x + 1))
      {
        rollback_x = true;
        sd.pos.x = sd.content_outline.right() - sd.dim_view.x + 1;
      }
    }

    if (rollback_x || rollback_y) {
      set_final_pos(v, sd.pos);
      return ANIMATION_TIMER_SPAN;
    }
    else
#endif
      return 0;
  }

  
  void scroll_animator::stop(view &v, element *b) {

    event_scroll evt(b, SCROLL_ANIMATION_END, true, b->scroll_pos().y, SCROLL_SOURCE_ANIMATOR, 0);
    b->on(v, evt);

    b->flags.animating_scroll = 0;
    v.check_mouse(true);
  }

} // namespace html
