#include "html.h"
#include "html-behaviors.h"
#include "html-actions-stack.h"
#include "html-dom-merge.h"

namespace html {

  namespace behavior {

    // wchar br[2] = { BR, 0 };
    // wchars br_chars(br,1);

    extern bool  is_empty_element(element *el);
    extern bool  is_empty_sequence(slice<hnode> nodes);
    extern node *split_at(view &v, editing_ctx *ectx, action *ra, bookmark &bm,
                          handle<element> until, bool after, bool &created,
                          bookmark &other, bool force_creation = false);
    extern helement split_element_at(view &v, editing_ctx *ectx, action *ra,
                                     bookmark &bm, handle<element> el,
                                     bool after, bool &created,
                                     bookmark &other);

    static tag::symbol_t phrasing_blocks[] = {
        tag::T_P,  tag::T_H1, tag::T_H2, tag::T_H3,
        tag::T_H4, tag::T_H5, tag::T_H6, tag::T_BLOCKQUOTE};

    static tag::symbol_t lists[] = {tag::T_UL, tag::T_OL, tag::T_MENU,
                                    tag::T_DIR};

    static tag::symbol_t text_containers[] = {
      tag::T_DIV,     tag::T_TD,      tag::T_TH,       tag::T_BLOCKQUOTE,
      tag::T_CAPTION, tag::T_ADDRESS, tag::T_FIELDSET, tag::T_LEGEND,
      tag::T_DD,      tag::T_DT,      tag::T_LI,
    };

    // the thing that contains <li>s
    bool is_list_tag(tag::symbol_t t) { return items_of(lists).contains(t); }

    // the thing that contains text (predominantly)
    bool is_text_container_tag(tag::symbol_t t) { return items_of(text_containers).contains(t); }

    // the thing that contains text (predominantly)
    bool is_phrasing_block_tag(tag::symbol_t t) { return items_of(phrasing_blocks).contains(t); }


    void richtext_ctl::push(view &v, action *act) {
      // assert(act->chain);
      if (!act->chain) return;

      bool was_modified = get_modified();

      drop_tail();

      stack.push(act);
      depth = stack.size();

      // v->current_undo_redo_status();
      // v->update_controls();

      act->close(static_cast<richtext_ctl *>(this));
      // act->after(v);

      if (!was_modified)
        static_cast<richtext_ctl *>(this)->on_document_status_changed(v, true);

      {
        uint reason = CHANGE_BY_CODE;
        if (act->change_reason.is_defined())
          reason = act->change_reason.val(0);
        else if (act->caption == WCHARS("delete range"))
          reason = CHANGE_BY_DEL_CHARS;
        else if (act->caption == WCHARS("delete character"))
          reason = CHANGE_BY_DEL_CHAR;
        else if (act->caption == WCHARS("insert text") ||
                 act->caption == WCHARS("insert plaintext"))
          reason = CHANGE_BY_INS_CHARS;
        event_behavior evt(self,self,EDIT_VALUE_CHANGED, reason);
        v.post_behavior_event(evt, true);
      }
    }

    void richtext_ctl::pop() {
      if (depth > 0) {
        depth--;
        drop_tail();
      }
    }

    void richtext_ctl::drop_tail() {
      while (depth >= 0 && depth < stack.size())
        stack.pop();
    }

    action *richtext_ctl::top() const {
      if (depth) return stack[depth - 1];
      return 0;
    }

    insert_text *richtext_ctl::top_insert_text() {
      action *t = top();
      if (!t) return 0;
      if (!t->chain->is_insert_text()) return 0;
      return t->chain.ptr_of<html::behavior::insert_text>();
    }

    remove_char_backward *richtext_ctl::top_remove_char_backward() {
      action *t = top();
      if (!t) return 0;
      if (!t->chain->is_remove_char_backward()) return 0;
      return t->chain.ptr_of<remove_char_backward>();
    }

    remove_char_forward *richtext_ctl::top_remove_char_forward() {
      action *t = top();
      if (!t) return 0;
      if (!t->chain->is_remove_char_forward()) return 0;
      return t->chain.ptr_of<remove_char_forward>();
    }

    void dbg_print(const bookmark &bm);

    // isolate_range prepares the range so this will be true
    //   pos.node == end.node - common base of original pos.node and end.node
    //   pos.after_it == false
    //   end.after_it == true
    //   NOTE: most of the time it changes the DOM.

    static helement isolate_range(view &v, editing_ctx *ectx, action *ra,
                                  bookmark &pos, bookmark &end,
                                  bool force = false) {
      pos.linearize();
      end.linearize();
      if (pos > end) swap(pos, end);

      helement root_s = ectx->root_at(v, pos);
      helement root_e = ectx->root_at(v, end);

      ASSERT((root_s == root_e) && root_e);

      helement base =
          element::find_base(pos.node->get_element(), end.node->get_element());
      if (force && pos.node->get_element() == end.node->get_element() &&
          base == pos.node->get_element() && base != root_s) {
        base = pos.node->get_element()->parent;
      }

      ASSERT(base->belongs_to(root_s, true));

      bool     created_at_start = false;
      bool     created_at_end   = false;
      bookmark dummy;
      split_at(v, ectx, ra, end, base, true, created_at_end, dummy);
      split_at(v, ectx, ra, pos, base, false, created_at_start, end);
      assert(pos.node == end.node);
      assert(pos.node->is_element());
      // if( !created_at_end && created_at_start )
      //  end.pos = end.pos + 1;
      return base;
    }

    bool apply_span_1(view &v, editing_ctx *ectx, action *ra, bookmark &start,
                      bookmark &end, tag::symbol_t t,
                      const attribute_bag &atts) {
      helement base = isolate_range(v, ectx, ra, start, end);

      if (start.node->is_element() && start.node == end.node && !start.after_it && end.after_it && start.node == base) {
        v.refresh(ectx->root());
        base->drop_layout_tree(&v);
        end.linearize();
        helement wrapper = wrap_nodes::exec(v, ectx, ra, base, start.pos, end.pos, t, atts);

        v.commit_update();
        start = wrapper->start_caret_pos(v);
        end   = wrapper->end_caret_pos(v);
        return true;
      }
      return false;
    }

    bool wrap_into(view &v, editing_ctx *ectx, range_action *ra,
                   bookmark &start, bookmark &end, element *el) {
      helement base = isolate_range(v, ectx, ra, start, end);

      if (start.node->is_element() && start.node == end.node &&
          !start.after_it && end.after_it && start.node == base) {
        v.refresh(ectx->root());
        base->drop_layout_tree(&v);
        end.linearize();

        helement wrapper =
            wrap_nodes::exec(v, ectx, ra, base, start.pos, end.pos, el);

        v.commit_update();
        start = wrapper->start_caret_pos(v);
        end   = wrapper->end_caret_pos(v);
        return true;
      }
      return false;
    }

    void unwrap_elements(view &v, editing_ctx *ectx, action *ra, element *el,
                         bookmark &s, bookmark &e, slice<tag::symbol_t> t,
                         const attribute_bag &atts) {
      if (t.contains(el->tag) && el->atts.contains(atts)) {
        pair<bookmark, bookmark> sel = unwrap_element::exec(v, ectx, ra, el);
        s = sel.first;
        e = sel.second;
      } else {
        s.linearize();
        e.linearize();
        int start = s.pos;
        for (int i = e.pos - 1; i >= start; --i) {
          if (!el->nodes[i]->is_element()) continue;
          helement cel = el->nodes[i].ptr_of<element>();
          s            = cel->start_pos();
          e            = cel->end_pos();
          unwrap_elements(v, ectx, ra, cel, s, e, t, atts);
        }
      }
    }

    bookmark zip_at(view &v, editing_ctx *ectx, action *group, helement base, int pos, bookmark &other, bool forward);

    bool remove_spans_1(view &v, editing_ctx *ectx, action *ra, element *tb,
                        bookmark &s, bookmark &e, slice<tag::symbol_t> tlist,
                        const attribute_bag &atts) {
      auto match_list = [&](node *n) -> bool {
        helement t = n->get_element();
        while (t && t != tb) {
          if (tlist.contains(t->tag) && t->atts.contains(atts)) return true;
          t = t->parent;
        }
        return false;
      };

      bookmark start       = s;
      bool     start_old   = false;
      bookmark end         = e;
      bool     end_old     = false;
      bool     match_found = false;

      if (match_list(start.node))
        match_found = true;
      else {
        start_old = true;
        pos_iterator si(start, end, true);
        for (bookmark bm; si(bm);) {
          element *el;
          if (bm.at_element_start(el) && match_list(el)) {
            start       = el->start_caret_pos(v);
            match_found = true;
            break;
          }
        }
      }

      if (match_list(end.node))
        match_found = true;
      else {
        end_old = true;
        pos_iterator si(start, end, false);
        for (bookmark bm; si(bm);) {
          element *el;
          if (bm.at_element_end(el) && match_list(el)) {
            end         = el->end_caret_pos(v);
            match_found = true;
            break;
          }
        }
      }

      if (!match_found) return false;

      assert(start < end);
      if (start >= end) return false;

      helement base = isolate_range(v, ectx, ra, start, end, true);

      if (start.node->is_element() && start.node == end.node &&
          !start.after_it && end.after_it && start.node == base) {
        v.refresh(ectx->root());
        base->drop_layout_tree(&v);
        start.linearize();
        end.linearize();

        // int split_start = start.linear_pos();
        // int split_end = end.linear_pos();

        unwrap_elements(v, ectx, ra, base, start, end, tlist, atts);

        if (!start_old) s = start;
        if (!end_old) e = end;

        // does not work, wrong bookmarks, investigate as it creates consequent
        // text nodes that is not a problem per se. Just to keep DOM clean:

        // e = zip_at(v,ectx,ra,base,split_end, s);
        // s = zip_at(v,ectx,ra,base,split_start,e);

        v.commit_update();

        return true;
      }
      return false;
    }

    inline element *nearest_box_element(node *n) // closest box or this
    {
      // return parent? parent->nearest_box():0;
      for (element *p = n->get_element(); p; p = p->parent) {
        if (p->is_box() || !p->parent) return p;
      }
      return 0;
    }

    /*element* immediate_child(element* base, node* descendant) {
      if( !descendant)
        return 0;
      if( descendant->parent == base )
        return descendant->get_element();
      return immediate_child(base, descendant->parent);
    }

    element* compatible_target(element* root, element* target, node *n) {
      if(!target || root == target)
        return root;
      if(!n->is_element())
        return target; // no change needed;
      element* te = n->cast<element>();
      tag::CMODEL_TYPE tcm = tag::content_model(target->tag);
      if( tcm != tag::CMODEL_BLOCKS && tag::require_block_container(te->tag))
        return compatible_target(root, target->parent, n);
      return target;
    }*/

    static void remove_contained_nodes(view &v, editing_ctx *ectx,
                                       action *group, helement root,
                                       bookmark start, bookmark end) {
      each_node_backward it(root);
      it.n = end.node;
      array<hnode> toremove;
      for (hnode n; it(n);) {
        bool starts_inside = n->start_pos() > start;
        bool ends_inside   = n->end_pos() < end;
        if (starts_inside && ends_inside && start.node != n && end.node != n)
          toremove.push(n);
      }
      for (int n = 0; n < toremove.size(); ++n)
        delete_node::exec(v, ectx, group, toremove[n]);
    }

    static bool compatible_to_zip(node *l, node *r) {
      if (l->is_text() && r->is_text()) return true;
      if (l->is_element() && r->is_element() &&
          !l->cast<element>()->is_atomic_box() &&
          l->cast<element>()->tag == r->cast<element>()->tag &&
          l->cast<element>()->atts == r->cast<element>()->atts)
        return true;
      return false;
    }

    bookmark zip_at(view &v, editing_ctx *ectx, action *group, helement base, int pos, bookmark &other, bool forward) {
      hnode left =
          pos <= 0 || pos > base->nodes.size() ? 0 : base->nodes[pos - 1];
      hnode right = pos < 0 || pos >= base->nodes.size() ? 0 : base->nodes[pos];
      if (!left && !right) {
        // assert(false);
        base->check_layout(v);
        return base->start_caret_pos(v);
      }
      if (left && right) {
        if (compatible_to_zip(left, right)) {
          if (left->is_element()) {
            helement le = left.ptr_of<element>();
            helement re = right.ptr_of<element>();
#ifdef _DEBUG
            le->dbg_report("left");
            re->dbg_report("right");
#endif
            array<hnode> nodes = re->nodes();
            delete_nodes_range::exec(v, ectx, group, re, 0, nodes.size());
            base = le;
            pos  = le->nodes.size();
            insert_nodes::exec(v, ectx, group, le, le->nodes.size(), nodes());
            delete_node::exec(v, ectx, group, re);
            // return nodes[0]->start_caret_pos(v);
            return zip_at(v, ectx, group, base, pos, other,forward);
          } else if (left->is_text()) {
            handle<text> lt = left.ptr_of<text>();
            handle<text> rt = right.ptr_of<text>();
            bookmark     bm = lt->end_pos();
            bookmark     at = bm;
            insert_text::exec(v, ectx, group, at, rt->chars());
            delete_node::exec(v, ectx, group, rt);
            if (other.node == rt) {
              other = bookmark(lt, lt->chars.size() + other.linear_pos(), false);
            }
            return bm;
          }

          // insert_nodes
        }
        else {
          if (forward)
            return right->start_caret_pos(v);
          else
            return left->end_caret_pos(v);
        }
      } else if (left)
        return left->end_caret_pos(v);
      else if (right)
        return right->start_caret_pos(v);
      return bookmark(base, pos, false);
      // if(  )
    }

    bookmark richtext_ctl::insert_text_block_placholder(view &v, element *pel,
                                                        bool after) {
      handle<action>  group = new placeholder_action(this);
      handle<element> t     = new element(tag::T_P);
      html::text *    nt    = new html::text(wchars());
      t->append(nt);
      // t->state.synthetic(true);
      insert_node::exec(v, this, group, pel->parent,
                        pel->node_index + int(after), t);
      push(v, group);
      root()->commit_measure(v);
      return nt->start_pos();
    }

    bool richtext_ctl::select(view &v, bookmark c, bookmark a) {

      span_shelve.clear();
            
      element* r = root();
      if (!r)
        a = c = bookmark();
      if (c.valid()) {
        if (!c.node->belongs_to(r, false))
          c = bookmark();
      }
      if (a.valid()) {
        if (!a.node->belongs_to(r, false))
          a = bookmark();
      }
      return super::select(v, c, a);

/*      if (!this->top() || !this->top()->is_placeholder_action())
        return super::select(v, c, a);
      if (!c.valid()) {
        this->undo(v);
        return super::select(v, c, a);
      }
      placeholder_action *pa  = static_cast<placeholder_action *>(this->top());
      node *              ppn = pa->placeholder();
      if (ppn->is_element() && !c.node->belongs_to(ppn->cast<element>(), true))
        this->undo(v);
      else if (!ppn->is_element() && c.node != ppn)
        this->undo(v);
      return super::select(v, c, a); */
    }

    bookmark check_caret_position(view &v, editing_ctx *ectx, action *group, bookmark bm, bool prefer_backward = false) {
      helement root = ectx->root();
      ASSERT(root);
      //root->dbg_report("check empty");

      richtext_ctl *  rt = static_cast<richtext_ctl *>(ectx);
      bool yes = false;
      if (rt->is_empty(root,yes) && yes ) 
      {
        if (root->nodes.length() != 0)
          delete_nodes_range::exec(v, ectx, group, root, 0, root->nodes.size());
        handle<element> t = v.get_anonymous_para();
        t->nodes.push(new text(WCHARS("")));
        if (rt->is_plaintext())
            t->state.synthetic(true);
        insert_node::exec(v, ectx, group, root, 0, t);
        root->check_layout(v);
        bm = t->start_caret_pos(v);
        return bm;
      }

      /*auto is_valid = [&](const bookmark &bm) -> bool {
        if (!bm.valid()) return false;
        if (!bm.node->belongs_to(ectx->root(), true)) return false;
        return true;
      };*/

      root->commit_measure(v);

      if (bm.valid()) {
                
        if (!bm.at_caret_pos(v)) {
          if (bm.node->is_text()) {
            ectx->advance(v, bm, prefer_backward ? ADVANCE_LEFT : ADVANCE_RIGHT);
            ectx->advance(v, bm, prefer_backward ? ADVANCE_RIGHT : ADVANCE_LEFT);
          }
          else {
            ectx->advance(v, bm, prefer_backward ? ADVANCE_LEFT : ADVANCE_RIGHT);
          }
        }
        if (!bm.at_caret_pos(v)) {
          if (ectx->advance(v, bm, ADVANCE_LEFT)) 
            bm.after_it = true;
        }
      }
      if(!bm.at_caret_pos(v))
        ectx->advance(v, bm, ADVANCE_HOME);

      return bm;
    }

    bookmark remove_range(view &v, editing_ctx *ectx, action *group,
                        bookmark rstart, bookmark rend, bool is_richtext, bool forward) {

      bookmark start = rstart;
      bookmark end   = rend;

      // ??? start.linearize();
      // ??? end.linearize();

      if (start == end) return rstart;

      if (start > end) swap(start, end);

      bookmark bm = start;

      bookmark very_first = ectx->root()->start_caret_pos(v);
      very_first.linearize();
      bookmark very_last = ectx->root()->end_caret_pos(v);
      very_last.linearize();

      auto delete_inline_node = [&](hnode n) -> bookmark {
        helement p = n->parent;
        if( n->is_text() && p->nodes.size() == 1 && p->is_block_element(v))
          return delete_text_range::exec(v, ectx, group, n.ptr_of<text>(),  0, n.ptr_of<text>()->chars.size()); // just empty it
        bookmark bm = delete_node::exec(v, ectx, group, n);
        while (p && p->is_empty() && p->is_inline_element(v) /*&& p != ectx->root_at(v,bm)*/) {
          helement t = p->parent;
          bm = delete_node::exec(v, ectx, group, p);
          p = t;
          if (p == ectx->root_at(v, bm))
            break;
        }
        return bm;
      };

      if (start == very_first && end == very_last) {
        // CTRL+A -> DEL
        delete_nodes_range::exec(v, ectx, group, ectx->root(), 0,
                                 ectx->root()->nodes.size());
        goto CHECK_EMPTY;
      }

      if (start.node == end.node) {
        if (start.node->is_text()) {
          if (start == start.node->start_pos() && end == start.node->end_pos())
            bm = delete_inline_node(start.node);
          else
            bm = delete_text_range::exec(v, ectx, group, start.node.ptr_of<text>(),
                                      start.linear_pos(), end.linear_pos());
          goto CHECK_EMPTY;
        } else if (start.node->is_element()) {
          element *pe = start.node.ptr_of<element>();
          if (start.linear_pos() <= 0 && end.linear_pos() >= pe->nodes.size())
            //bm = delete_node::exec(v, ectx, group, start.node);
            bm = delete_inline_node(start.node);
          else
            delete_nodes_range::exec(v, ectx, group,
                                     start.node.ptr_of<element>(),
                                     start.linear_pos(), end.linear_pos());
          goto CHECK_EMPTY;
        }
      } else // more sophisticated case
      {
        // helement base = isolate_range(v, ectx, group,start,end,true);
        // delete_nodes_range::exec(v,ectx,group,base,start.linear_pos(),
        // end.linear_pos());  bm = zip_at(v,ectx,group,base,start.linear_pos());

        hnode first = start.node;
        hnode last  = end.node;
        hnode base  = node::find_base(first, last);

        if (!base) return bookmark();
        if (!base->belongs_to(ectx->root(), true)) return bookmark();

        if (first->is_text() && last->is_text()) // so we need to glue them
        {
          handle<text> tf   = first.ptr_of<text>();
          handle<text> tl   = last.ptr_of<text>();
          helement     tftb = tf->nearest_box(false);
          helement     tltb = tl->nearest_box(false);

          if (base->belongs_to(tftb, true) && base->belongs_to(tltb, true)) {
            helement base = isolate_range(v, ectx, group, start, end, true);
            if (!base->belongs_to(ectx->root(), true)) return bookmark();
            if(start.linear_pos() != end.linear_pos())
              delete_nodes_range::exec(v, ectx, group, base, start.linear_pos(), end.linear_pos());
            bookmark dummy;
            bm = zip_at(v, ectx, group, base, start.linear_pos(), dummy,forward);
          } else {
            bool created_at_start = false;
            bool created_at_end   = false;
            // helement cp = node::find_common_parent(tf, tl);
            bookmark dummy;
            split_at(v, ectx, group, end, tltb, true, created_at_end, dummy);
            split_at(v, ectx, group, start, tftb, false, created_at_start, end);

#ifdef _DEBUG
            tftb->dbg_report("first");
            tltb->dbg_report("second");
            assert(end.node == tltb);
#endif
            int          end_idx_1  = end.linear_pos();
            int          end_idx_2  = tltb->nodes.size();
            array<hnode> tail_nodes = tltb->nodes(end_idx_1, end_idx_2);
            if (end_idx_1 < end_idx_2)
              delete_nodes_range::exec(v, ectx, group, tltb, end_idx_1,
                                       end_idx_2);
            remove_contained_nodes(v, ectx, group, base->get_element(), start,
                                   end);
            int pos = start.linear_pos();
            insert_nodes::exec(v, ectx, group, tftb, pos, tail_nodes());
            //-----
            helement n = tltb;
            while (n != base) {
              helement p = n->parent;
              if (!p) break;
              delete_node::exec(v, ectx, group, n);
              if (!p || !is_empty_element(p)) break;
              n = p;
            }
            //-----
            tftb->drop_layout(&v);
            bm = zip_at(v, ectx, group, tftb, pos, dummy,forward);
          }

        } else if (base->is_element()) {
          bool     created_at_start = false;
          bool     created_at_end   = false;
          helement cont             = base.ptr_of<element>();
          bookmark dummy;
          split_at(v, ectx, group, end, cont, end.after_it, created_at_end,dummy);
          split_at(v, ectx, group, start, cont, start.after_it,created_at_start, end);
          int pos  = start.linear_pos();
          int tail = end.linear_pos();
          if (pos < tail)
            delete_nodes_range::exec(v, ectx, group, cont, pos, tail);
          cont->drop_layout(&v);
          bm = zip_at(v, ectx, group, cont, pos, dummy,forward);

        }
      }

    CHECK_EMPTY:
      return check_caret_position(v, ectx, group, bm, !forward);
    }

#if 0
  bookmark remove_range(view& v, editing_ctx* ectx, action* group, bookmark start, bookmark end, bool is_richtext)
  {
    start.linearize();
    end.linearize();
    if( start > end ) swap(start,end);

    bookmark bm = start;

    if( start.node == end.node )
    {
      if( start.node->is_text() )
      {
        delete_text_range::exec(v,ectx, group, start.node.ptr_of<text>(),start.linear_pos(),end.linear_pos());
        goto CHECK_EMPTY;
      }
      else if( start.node->is_element() )
      {
        delete_nodes_range::exec(v,ectx, group, start.node.ptr_of<element>(),start.linear_pos(),end.linear_pos());
        goto CHECK_EMPTY;
      }
    }
    else // more sophisticated case
    {
      array<hnode> toremove;

      hnode first = start.node;
      hnode last = end.node;
      hnode base = node::find_base(first,last);

      each_node_backward it(base);
      it.n = last;

      if(first->is_text() && last->is_text()) // so we need to glue them
      {
        text* tf = first.ptr_of<text>();
        text* tl = last.ptr_of<text>();
        // remove tail and append by last head
        delete_text_range::exec(v,ectx,group,tf,start.linear_pos());
        bm = tf->end_caret_pos(v);
        wchars toinsert = trim(tl->chars(end.linear_pos()));
        if(toinsert.length) {
          if(bm.linear_pos() == 0 && !toinsert)
            toinsert = WCHARS("?");
          insert_text::exec(v,ectx,group,bm,toinsert);
        }
        helement target = tf->parent;
        helement tm_parent = tl->parent;
        hnode    tm = tl->next_node();
        delete_node::exec(v,ectx,group,tl);
        while(tm) {
          tm->dbg_report("reparent ");
          target = compatible_target(ectx->root(),target,tm);
          hnode tmn = tm->next_node();
          delete_node::exec(v,ectx,group,tm);
          insert_node::exec(v,ectx,group,target,target->nodes.size(),tm);
          tm = tmn;
        }
        while( tm_parent != base )
        {
          if(!is_empty_element(tm_parent))
            break;
          helement t = tm_parent->parent;
          delete_node::exec(v,ectx,group,tm_parent);
          tm_parent = t;
        }

        last = 0;

        bm = start;
      }
      else if(first->is_element() && last->is_text())
      {
        text* tl = last.ptr_of<text>();
        delete_text_range::exec(v,ectx,group,tl, 0, end.linear_pos());
        last = 0; // don't remove last
      }
      else if(first->is_text() && last->is_element())
      {
        text* tf = first.ptr_of<text>();
        delete_text_range::exec(v,ectx, group,tf, start.linear_pos());
        if(!end.at_element_end())
          last = 0; // don't remove last
      }
      else if(first->is_element() && last->is_element())
      {
        if(!end.at_element_end())
          last = 0; // don't remove last
      }
      else {
        assert(false);
        return bookmark();
      }

      for(node* n; it(n);)
      {
        bool starts_inside =  n->start_pos() > start;
        bool ends_inside =  n->end_pos() < end;
        if( starts_inside && ends_inside
            && start.node != n
            && end.node != n )
          toremove.push(n);
      }
      // note: elements in toremove are in backward order, first is latest in the DOM.

      // toremove list contains elements that need to be removed untouched.
      // toremove list does not contain first element
      // first need to appended by tail of the last

      for(int n = 0; n < toremove.size(); ++n)
        delete_node::exec(v,ectx,group,toremove[n]);

      if(last)
      {
        hnode n = last;
        while(n != base) {
          helement p = n->parent;
          delete_node::exec(v,ectx,group,n);
          if(!p || !is_empty_element(p))
            break;
          n = p;
        }
      }
    }

CHECK_EMPTY:
    if( ectx->root()->is_empty() )
    {
      handle<text> t = new text(WCHARS(" "));
      insert_node::exec(v,ectx,group,ectx->root(), 0, t);
      bm = bookmark(t,0,false);
    }

    return bm;
  }

#endif

    bool richtext_ctl::shelve_apply_span(view &v, tag::symbol_t t)
    {
      if (span_shelve.apply_contains_one_of(slice<tag::symbol_t>(t)))
        return false;
      if (!span_shelve.unshelve_remove(slice<tag::symbol_t>(t)))
        span_shelve.push_apply(t);

      event_behavior evt(self, self, UI_STATE_CHANGED, 0);
      v.post_behavior_event(evt, true);

      return true;
    }

    bool richtext_ctl::shelve_remove_spans(view &v, slice<tag::symbol_t> t)
    {
      if (span_shelve.remove_contains_one_of(t))
        return false;

      if( !span_shelve.unshelve_apply(t) )
        span_shelve.push_remove(t);

      event_behavior evt(self, self, UI_STATE_CHANGED, 0);
      v.post_behavior_event(evt, true);

      return true;
    }

    bool richtext_ctl::apply_span(view &v, tag::symbol_t t,
                                  const attribute_bag &atts) {
      clear_comp_chars(v);

      if (has_collapsed_selection()) {
        assert(atts.is_empty());
        if(shelve_apply_span(v, t))
          return true;
        return false;
      }

      bookmark start = anchor;
      bookmark end   = caret;

      if (start > end) swap(start, end);

      handle<action> group =
          transact_group
              ? transact_group
              : new range_action(this,
                                 ustring::format(W("apply <%S> span"),
                                                 tag::symbol_name(t).c_str()));

      if (is_table_range_selection() && selected.size()) {
        start = bookmark();
        for (helement cell : selected) {
          bookmark s = cell->start_caret_pos(v);
          end        = cell->end_caret_pos(v);
          apply_span(v, group, s, end, t, atts);
          if (!start.valid()) start = s;
        }
      } else
        apply_span(v, group, start, end, t, atts);

      select(v, start, end);

      if (!transact_group) push(v, group);

      return true;
    }

    bool richtext_ctl::remove_spans(view &v, slice<tag::symbol_t> tlist,
                                    const attribute_bag &atts) {
      clear_comp_chars(v);

      if (has_collapsed_selection())
      {
        if (shelve_remove_spans(v, tlist))
          return true;
        return false;
      }

      bookmark start = anchor;
      bookmark end   = caret;

      if (start > end) swap(start, end);

      handle<action> group =
          transact_group
              ? transact_group
              : new range_action(
                    this, ustring::format(W("remove <%S> spans"),
                                          tag::symbol_name(tlist[0]).c_str()));

      if (is_table_range_selection() && selected.size()) {
        start = bookmark();
        for (helement cell : selected) {
          bookmark s = cell->start_caret_pos(v);
          end        = cell->end_caret_pos(v);
          remove_spans(v, group, s, end, tlist, atts);
          if (!start.valid()) start = s;
        }
      } else
        remove_spans(v, group, start, end, tlist, atts);

      select(v, start, end);

      if (!transact_group) push(v, group);

      return false;
    }

    bool richtext_ctl::apply_span(view &v, handle<action> group,
                                  bookmark &start, bookmark &end,
                                  tag::symbol_t t, const attribute_bag &atts) {
      try {

        helement base = element::find_ui_base(start.node->get_element(),
                                              end.node->get_element());

        helement tb_start = start.node->nearest_text_box();
        helement tb_end   = end.node->nearest_text_box();

        ASSERT(tb_start && tb_end);
        //#ifdef _DEBUG
        //      tb_start->dbg_report("start");
        //      tb_end->dbg_report("end");
        //      base->dbg_report("base");
        //#endif
        ASSERT(start.valid() && end.valid());

        if (tb_start == tb_end) {
          // simple case: start and end inside the same text block
          apply_span_1(v, this, group, start, end, t, atts);
        } else {
          // start and end belong to different text blocks
          // scan through all text blocks in the range and apply span to them.
          element_ui_iterator cit(v, base);
          cit.shallow = true;
          cit.pfilter = [](view &v, element *el) -> bool {
            return el->is_of_type<text_block>();
          };
          cit.rewind(tb_start);

          array<pair<bookmark, bookmark>> ranges;

          element *b = tb_start;
          do {
            bookmark s     = b->start_caret_pos(v);
            bookmark e     = b->end_caret_pos(v);
            bool     first = false;
            bool     last  = false;
            if (start.node->owned_by(b)) {
              first = true;
              s     = start;
            }
            if (end.node->owned_by(b)) {
              e    = end;
              last = true;
            }
            ranges.push(pair<bookmark, bookmark>(s, e));
            if (first) start = s;
            if (last) {
              end = e;
              break;
            }
          } while (cit(b));

          start = bookmark();
          for (auto &r : ranges) {
            bookmark first = r.first;
            end            = r.second;
            ASSERT(first.valid() && end.valid());
            apply_span_1(v, this, group, first, end, t, atts);
            if (!start.valid()) start = first;
          }
        }
        return true;
      } catch (const tool::exception &) { group->undo(v, this); }
      return false;
    }

    bool richtext_ctl::unwrap_element(view &v, element *el) {
      clear_comp_chars(v);

      handle<range_action> group = new range_action(
          this, ustring::format(W("pull <%S> element"),
                                tag::symbol_name(el->tag).c_str()));

      try {

        auto sel = unwrap_element::exec(v, this, group, el);

        select(v, sel.first, sel.second);

        this->push(v, group);
        return true;
      } catch (const tool::exception &) { group->undo(v, this); }
      return false;
    }

    bool richtext_ctl::remove_spans(view &v, handle<action> group,
                                    bookmark &start, bookmark &end,
                                    slice<tag::symbol_t> tlist,
                                    const attribute_bag &atts) {
      try {

        helement base = element::find_base(start.node->get_element(),
                                           end.node->get_element());

        helement tb_start = start.node->nearest_text_box();
        helement tb_end   = end.node->nearest_text_box();
        if (tb_start == tb_end) {
          // simple case: start and end inside the same text block
          remove_spans_1(v, this, group, tb_start, start, end, tlist, atts);
        } else {
          // start and end belong to different text blocks
          // scan through all text blocks in the range and apply span to them.
          element_iterator cit(v, base);
          cit.shallow = true;
          cit.pfilter = [](view &v, element *el) -> bool {
            return el->is_of_type<text_block>();
          };
          cit.rewind(tb_start);
          element *b = tb_start;
          do {
            bookmark s     = b->start_caret_pos(v);
            bookmark e     = b->end_caret_pos(v);
            bool     first = false;
            bool     last  = false;
            if (start.node->belongs_to(b)) {
              first = true;
              s     = start;
            }
            if (end.node->belongs_to(b)) {
              e    = end;
              last = true;
            }
            remove_spans_1(v, this, group, b, s, e, tlist, atts);
            if (first) start = s;
            if (last) {
              end = e;
              break;
            }
          } while (cit(b));
        }
        return true;
      } catch (const tool::exception &) { group->undo(v, this); }
      return false;
    }

    element *   get_inlines_container(element *pel, element *until);
    element *   get_inlines_container(view& v, bookmark& bm, element *until);
    element *   get_block(node *n, element *until);
    static bool is_at_pre(view &v, const bookmark &bm) {
      if (!bm.valid() || !bm.at_char_pos(v)) return false;
      element *p = bm.node->nearest_text_box();
      if (p) {
        return p->tag == tag::T_PRE;
        // return !p->get_style()->collapse_ws();
      }
      return false;
    }

    bool is_last_child(element *el) {
      if (el->next_element()) return false;
      node *pn = el->next_node();
      if (!pn || pn->is_space()) return true;
      return false;
    }
    bool is_first_child(element *el) {
      if (el->prev_element()) return false;
      node *pn = el->prev_node();
      if (!pn || pn->is_space()) return true;
      return false;
    }

    // returns true if text container(p,li,etc) needs to be inserted to contain
    // text
    static bool needs_text_block(view &v, bookmark bm) {
      if (bm.at_caret_pos(v)) return false;
      if (!bm.node->is_element()) return false;
      element *el = bm.node->get_element();
      if (el->is_inline_element(v)) return false;
      // this elements can contain text if it goes before first child block
      if (is_first_child(el) &&
          CONST_SLICE(text_containers).contains(el->parent->tag))
        return false;
      return true;
    }

  element *get_inlines_container(view& v, bookmark& bm, element *until) {
      if (!bm.valid()) return nullptr;
      element *pel = bm.node->get_element();
      if (pel == until) {
          if(!bm.node->is_text())
            bm = until->end_caret_pos(v);
          return until;
      }
      auto tt = tag::type(pel->tag);
      if (tt == tag::TABLE_BODY_TAG || tt == tag::TABLE_ROW_TAG || tt == tag::TABLE_TAG) {
          bm = pel->this_pos(true);
          return get_inlines_container(v,bm, until);
      }
      if (tt == tag::INLINE_BLOCK_TAG || tt == tag::BLOCK_TAG || tt == tag::TABLE_CELL_TAG) {
          if (is_list_tag(pel->tag)) {
              if (bm.at_element_start()) {
                  bm = pel->this_pos(false);
                  return get_inlines_container(v,bm, until);
              }
              else if (bm.at_element_end()) {
                  bm = pel->this_pos(true);
                  return get_inlines_container(v,bm, until);
              }
              return nullptr;
          }
          auto cm = tag::content_model(pel->tag);
          if (cm == tag::CMODEL_BLOCKS || cm == tag::CMODEL_INLINES)
              return pel;
      }
      if (tt == tag::INLINE_TAG || tt == tag::INLINE_BLOCK_TAG) {
        return bm.node->nearest_text_box();
      }
      bm = pel->this_pos(true);
      return get_inlines_container(v,bm, until);
  }

static bool needs_text_node(view &v, bookmark& bmr)
    {
      bookmark bm = bmr;
      if (bm.at_element_start())
        bm = bm.node->this_pos(false);
      else if (bm.at_element_end())
        bm = bm.node->this_pos(true);
      else
        return false;
      element *pel = bm.node->get_element();
      //if (el->is_inline_element(v)) return false;
      // this elements can contain text if it goes before first child block
      if (is_text_container_tag(pel->tag)) {
        bmr = bm;
        return true;
      }
      return false;
    }


    static bool only_text_container(view &v, bookmark bm) {
      element *el = bm.node->nearest_box();
      ASSERT(el);
      // these elements can contain only text
      static tag::symbol_t text_containers[] = {tag::T_P, tag::T_DIV};
      if (CONST_SLICE(text_containers).contains(el->tag)) return true;
      return false;
    }

    static element *make_empty_p(element *proto) {
      // text* nt = new text(wchars());
      element *nel = proto->clone_element(false, false);
      // np->append(nt);
      switch (nel->tag) {
      case tag::T_DD: nel->tag = tag::T_DT; break;
      case tag::T_DT: nel->tag = tag::T_DD; break;
      case tag::T_LI: /*nel->tag = tag::T_DD;*/ break;
      default: nel->tag = tag::T_P; break;
      }
      return nel;
    };

    bool richtext_ctl::insert_chars(view &v, bookmark start, bookmark end,
                                    wchars text) {

      richtext_ctl *_this = static_cast<richtext_ctl *>(this);
      _this->clear_comp_chars(v);

//#ifdef _DEBUG
//      if (text == WCHARS("w"))
//        text = text;
//#endif

      if (!start.valid() || !end.valid()) return false;
      if (!start.is_inside(self) || !end.is_inside(self)) return false;

      //_this->select(v, bookmark());

      if ((start == end) && start.at_char_pos(v) && start.node->is_text() && span_shelve.is_empty()) {
        behavior::insert_text *tit = top_insert_text();
        bookmark               bm  = start;
        if (tit && tit->append(v, _this, top(), bm, text)) {
          event_behavior evt(self, self, EDIT_VALUE_CHANGED,
                             CHANGE_BY_INS_CONSECUTIVE_CHAR);
          v.post_behavior_event(evt, true);
          _this->select(v, bookmark());
          v.commit_update(true);
          _this->select(v, bm);
          return true;
        }
      }
      if ((start == end) &&
          (start.at_element_end() || start.at_element_start())) {
        bookmark bm = start;
        helement el = bm.node.ptr_of<element>();
        if (el->is_of_type<block_table_row>()) 
          return insert_row(v, bm);
      }

      handle<action> op = new action(_this, WCHARS("insert text"));

      tag_shelve span_shelve = this->span_shelve;

      _this->select(v, bookmark());

      op->change_reason = u16::codepoints(text) == 1 ? CHANGE_BY_INS_CHAR
                                                     : CHANGE_BY_INS_CHARS;

      try {

        if (start != end) {
          if (start > end) swap(start, end);
          start = end = remove_range(v, _this, op, start, end, is_richtext(),false);
        }
        bookmark bm = start;

        bool plain_text = is_at_pre(v, bm);

        if (plain_text) {
          wchars line;
          while(chopline(text,line)) {
            array<wchar> t = line;
            if (text.length) t.push('\n');
            insert_text::exec(v, _this, op, bm, t());
            if (!text.length) break;
          }
        } else {

          // for (wchars line = chopline(text);; line = chopline(text)) {
          //  lines.push(line);
          //  if (!text.length)
          //    break;
          //}

          wchars line; chopline(text,line);
          bool   multiple_lines = text.length > 0;

          if (bm.at_br_pos())
          {
            html::text *nt = new html::text(wchars());
            insert_node::exec(v, this, op, bm.node->parent, bm.node->node_index, nt);
            bm = nt->start_caret_pos(v);
          }
          helement lroot = this->root_at(v, bm);
          helement box   = get_inlines_container(v,bm, lroot);
          /*if (!box) {
            html::text *nt = new html::text(wchars());
            insert_node::exec(v, this, op, box, bm.linear_pos(), nt);
            bm = nt->start_pos();
          }
          else */
          if (needs_text_node(v, bm)) // will update bm
          {
            // bm is an insertion point here
            ASSERT(bm.node->is_element());
            element* pel = bm.node->get_element();
            int      pos = bm.linear_pos();

            //pel->dbg_report("text node");
            //node *pn = pel->nodes[pos + 1];
            //node *pp = pel->nodes[pos - 1];
            if ((pos > 0) && pel->nodes[pos - 1]->is_text())
              bm = pel->nodes[pos - 1]->end_pos();
            else if ((pos < pel->nodes.size()) && pel->nodes[pos]->is_text())
              bm = pel->nodes[pos]->start_pos();
            else {
              html::text *nt = new html::text(wchars());
              insert_node::exec(v, this, op, pel, pos, nt);
              bm = nt->start_pos();
            }
          }
          else if (needs_text_block(v, bm)) {
            // element *np = new element(tag::T_P);
            element *   np = make_empty_p(bm.node->get_element());
            html::text *nt = new html::text(wchars());
            np->append(nt);
            if (bm.at_element_start())
              insert_node::exec(v, this, op, bm.node->parent,
                                bm.node->node_index, np);
            else if (bm.at_element_end())
              insert_node::exec(v, this, op, bm.node->parent,
                                bm.node->node_index + 1, np);
            else
              insert_node::exec(v, this, op, bm.node->parent,
                                bm.node->node_index, nt);
            bm = nt->start_pos();
          } else if (multiple_lines && lroot == box) {
            ASSERT(bm.node->is_text());
            // box->dbg_report("insert text");
            element *np = make_empty_p(bm.node->get_element());
            wrap_nodes::exec(v, this, op, box, 0, box->nodes.size(), np);
          }
          bookmark initial_bm = bm;
          do {
            insert_text::exec(v, _this, op, bm, line);
            if (text.length == 0) break;
            element *p = get_inlines_container(bm.node->get_element(), lroot);
            if (p && p->parent) {
              bool     created = false;
              bookmark nbm     = bm;
              split_at(v, this, op, bm, p->parent.ptr(), true, created, nbm);
              if (!created) {
                element *   np = p->clone_element(false);
                html::text *nt = new html::text(wchars());
                np->append(nt);
                insert_node::exec(v, this, op, p->parent, p->node_index + 1,
                                  np);
                bm = nt->start_pos();
              } else
                bm = nbm;
            }
          } while (chopline(text, line));

          /*WTF?: if (box && box->parent && box->tag == tag::T_TEXT) { // temporary text container
            auto bm2 = unwrap_element::exec(v, this, op, box);
            initial_bm = bm2.first;
            bm = bm2.second;
          }*/

          if (span_shelve.has_removes())
            span_shelve.each_remove([&](slice<tag::symbol_t> t) {
               this->remove_spans(v, op, initial_bm, bm, t);
          });

          if (span_shelve.has_applies())
            span_shelve.each_apply([&](tag::symbol_t t) {
              this->apply_span(v, op, initial_bm, bm, t);
          });

        }

        v.commit_update(true);
        _this->select(v, bm);
        push(v, op);
        return true;
      } catch (const tool::exception &) { op->undo(v, _this); }
      return false;
    }

    /*bool richtext_ctl::insert_plaintext_chars(view &v, bookmark start,
                                              bookmark end, wchars text) {
      richtext_ctl *_this = static_cast<richtext_ctl *>(this);
      _this->clear_comp_chars(v);

      if (!start.valid() || !end.valid()) return false;

      handle<action> op = new action(_this, WCHARS("insert plaintext"));

      if (start != end) {
        if (start > end) swap(start, end);
        start = end = remove_range(v, _this, op, start, end, false);
      }
      bookmark bm = start;

      if (text.index_of('\r') < 0 && text.index_of('\n') < 0) {
        if (insert_text::exec(v, _this, op, bm, text)) {
          v.commit_update(true);
          _this->select(v, bm);
          push(v, op);
          return true;
        }
      } else { // multiline

        hnode tail = split_node::exec(v, this, op, bm.node, bm.linear_pos(),
                                      true); // split the text node
        hnode tail_block = split_node::exec( v, this, op, tail->parent.ptr(), tail->node_index); // split the <text>

        array<hnode> texts;
        wchars       line = chopline(text);

        bm = bm.node->end_pos();

        if (line.length) insert_text::exec(v, this, op, bm, line);

        for (line = chopline(text); text.length > 0; line = chopline(text)) {
          html::text *   t  = new html::text(line);
          html::element *te = new element(tag::T_TEXT);
          te->append(t);
          texts.push(te);
        }
        insert_nodes::exec(v, this, op, self, tail_block->node_index, texts());
        bm = tail->start_pos();
        if (line.length) insert_text::exec(v, this, op, bm, line);

        v.commit_update(true);
        _this->select(v, bm);
        push(v, op);
        return true;
      }

      return false;
    }*/

    bool richtext_ctl::insert_row(view &v, bookmark bm) {
      richtext_ctl *_this = static_cast<richtext_ctl *>(this);
      _this->clear_comp_chars(v);

      if (!bm.valid()) return false;

      handle<action> op = new action(_this, WCHARS("insert row"));

      assert(bm.node->is_of_type<block_table_row>());
      if (!bm.node->is_of_type<block_table_row>()) return false;

      handle<block_table_row>  table_row  = bm.node.ptr_of<block_table_row>();
      handle<block_table_body> table_body = table_row->parent_table_body();
      if (!table_body) return false;

      int ins_pos = table_row->flags.ui_index + (bm.at_element_end() ? 1 : 0);
      int ins_node_pos = table_row->node_index + (bm.at_element_end() ? 1 : 0);

      handle<element> new_row = new element(tag::T_TR);

      uint rows, cols;
      if (!table_body->get_rows_cols(rows, cols)) return false;

      helement first, last;

      for (uint r = 0; r < rows; ++r)
        for (uint c = 0; c < cols; ++c) {
          element *cell = table_body->get_cell_at(r, c);
          if (!cell) continue;

          uint r1, r2, c1, c2;
          if (!table_body->get_cell_rows_cols(cell, r1, r2, c1, c2)) continue;

          if (int(r1) < ins_pos && ins_pos <= int(r2)) { // the cell spans over
                                                         // the insertion row,
                                                         // increase its rowspan
            int row_span = cell->atts.get_rowspan() + 1;
            change_attr::set(v, _this, op, cell, "rowspan",
                             tool::itoa(row_span));
          } else if (int(r) == table_row->flags.ui_index) {
            helement t = cell->clone_element(false);
            t->atts.remove(attr::a_rowspan);
            new_row->append(t);
            if (!first) first = t;
            last = t;
          }
        }

      insert_node::exec(v, _this, op, table_row->parent, ins_node_pos, new_row);
      v.commit_update();
      if (first) bm = first->start_caret_pos(v);

      _this->select(v, bm);
      push(v, op);
      return true;
    }

    static element *is_at_empty_para(view &v, const bookmark &bm) {
      if (!bm.valid()) return 0;
      element *p = bm.node->nearest_text_box();
      if (!p) return nullptr;
      if (bm.node->is_element() &&
          bm.node.ptr_of<element>()->tag == tag::T_BR) {
        if (is_empty_element(p)) return p;
        return nullptr;
      }
      if (!bm.node->is_text()) return nullptr;

      if (p->nodes.size() != 1) return nullptr;
      wchars txt = bm.node.ptr_of<text>()->chars();
      if (txt.length == 0) return p;
      if (txt.length == 1 && is_space(txt[0])) return p;
      return nullptr;
    }

    // remove empty blocks on the path
    void purge_empty_blocks(view &v, editing_ctx *ectx, action *group,
                            helement p) {
      helement pp = p->parent;
      while (p && p != ectx->root() && is_empty_element(p)) {
        pp = p->parent;
        delete_node::exec(v, ectx, group, p);
        p = pp;
      }
    }

    /*bool is_empty_node(node* pn);

    int split_block(view& v, editing_ctx *ectx, action* group, helement p, int
    pos, helement until)
    {
      if (p == until) return pos;
      if (pos == p->nodes.last_index() && is_empty_node(p->nodes.last()) )
        return split_block(v, ectx, group, p->parent, p->node_index, until);
      if (pos > p->nodes.last_index())
        return split_block(v, ectx, group, p->parent, p->node_index, until);
      else if (pos == 1 && is_empty_node(p->nodes.first()))
        return split_block(v, ectx, group, p->parent, p->node_index, until);
      else if (pos == 0)
        return split_block(v, ectx, group, p->parent, p->node_index, until);
      else {
        // split the list onto two
        hnode hnl = split_node::exec(v, ectx, group, p.ptr(), pos);
        ASSERT(hnl);
        return split_block(v, ectx, group, hnl.ptr_of<element>(),
    hnl->node_index, until);
      }
    }*/

    element *unlist_list_item(view &v, editing_ctx *ectx, action *group,
                              helement el) {
      helement p = el->parent; // the list
      if (!p) return nullptr;
      helement originalp = p;
      helement pp        = p->parent;
      if (!pp) return nullptr;
      //#ifdef  _DEBUG
      //    p->dbg_report("p=");
      //    pp->dbg_report("pp=");
      //#endif //  _DEBUG

      tag::symbol_t target_tag = el->tag; // tag::T_P;
      if (el->tag == tag::T_BLOCKQUOTE) {
        // target_tag = tag::T_P;
        p  = el;
        pp = el->parent;
      } else if (p->tag == tag::T_BLOCKQUOTE) {
        target_tag = tag::T_P;
      } else if (p->tag == tag::T_LI) {
        target_tag = tag::T_LI;
      } else if (el->tag == tag::T_LI) {
        // pp->dbg_report("pp");
        // p->dbg_report("p");
        if (is_list_tag(pp->tag)) {
          target_tag = tag::T_LI;
        } else if (pp->parent && is_list_tag(pp->parent->tag)) {
          // cannonical unlist

          int pos = el->node_index;
          //        hnode new_list;
          //        if (pos == 0)
          //          new_list = new element(pp->parent->tag);
          //        else
          //          new_list = split_node::exec(v, ectx, group, p.ptr(), pos);
          hnode new_list = split_node::exec(v, ectx, group, p.ptr(), pos, true);
          ASSERT(new_list);
          // this el is its first child of new list
          hnode new_list_item = split_node::exec(
              v, ectx, group, new_list->parent.ptr(), new_list->node_index, true);
          ASSERT(new_list_item);
          delete_node::exec(v, ectx, group, new_list);
          delete_node::exec(v, ectx, group, el);
          insert_node::exec(v, ectx, group, new_list_item->parent,
                            new_list_item->node_index, el);
          insert_node::exec(v, ectx, group, el, 0xffff, new_list);
          if (is_empty_element(new_list.ptr_of<element>()))
            delete_node::exec(v, ectx, group, new_list);
          if (is_empty_element(new_list_item.ptr_of<element>()))
            delete_node::exec(v, ectx, group, new_list_item);
          return el;
        } else {
          auto is_block_element = [](node *pn) -> bool {
            if (!pn->is_element()) return false;
            element *     el = static_cast<element *>(pn);
            tag::TAG_TYPE tt = tag::type(el->tag);
            return (tt != tag::INLINE_TAG && tt != tag::UNKNOWN_TAG);
          };

          int end   = el->nodes.size();
          int start = end;
          for (int n = 0; n < el->nodes.size(); ++n) {
            if (is_block_element(el->nodes[n])) {
              start = n;
              break;
            }
          }

          if (start < end)
            target_tag = tag::T_DIV;
          else
            target_tag = tag::T_P;
        }
      } else if (pp->tag == tag::T_DD) {
        target_tag = tag::T_DT;
        p          = pp;
        pp         = pp->parent;
      } else if (pp->tag == tag::T_DT) {
        target_tag = tag::T_DD;
        p          = pp;
        pp         = pp->parent;
      } else if (p->tag == tag::T_DD) {
        target_tag = tag::T_DT;
      } else if (p->tag == tag::T_DT) {
        target_tag = tag::T_DD;
      } else if (el->tag == tag::T_DD) {
        target_tag = tag::T_P;
      } else if (el->tag == tag::T_DT) {
        target_tag = tag::T_P;
      }

      if (is_last_child(el)) {
        // that's the last one
        delete_node::exec(v, ectx, group, el);
        insert_node::exec(v, ectx, group, pp, p->node_index + 1, el);
      } else if (is_first_child(el)) {
        // that's the first one
        delete_node::exec(v, ectx, group, el);
        insert_node::exec(v, ectx, group, pp, p->node_index, el);
      } else {
        // split the list onto two
        int   pos = el->node_index;
        hnode hnl = split_node::exec(v, ectx, group, p.ptr(), pos);
        if (!hnl.is_null()) {
          delete_node::exec(v, ectx, group, el);
          insert_node::exec(v, ectx, group, pp, hnl->node_index, el);
        }
      }
      if (target_tag != el->tag)
        morph_element::exec(v, ectx, group, el, target_tag);

      purge_empty_blocks(v, ectx, group, originalp);
      return el;
    }

    element *unblock(view &v, editing_ctx *ectx, action *group, helement el,
                     tag::symbol_t t) {
      helement p = el->parent; // the list
      if (!p) return nullptr;
      helement originalp = p;
      helement pp        = p->parent;
      if (!pp) return nullptr;

      tag::symbol_t target_tag = el->tag; // tag::T_P;
      if (el->tag == t) {
        p          = el;
        pp         = el->parent;
        target_tag = tag::T_P;
      } else if (p->tag == t) {
        target_tag = tag::T_P;
      }

      if (is_last_child(el)) {
        // that's the last one
        delete_node::exec(v, ectx, group, el);
        insert_node::exec(v, ectx, group, pp, p->node_index + 1, el);
      } else if (is_first_child(el)) {
        // that's the first one
        delete_node::exec(v, ectx, group, el);
        insert_node::exec(v, ectx, group, pp, p->node_index, el);
      } else {
        // split the list onto two
        int pos = el->node_index;
        // hnode hnl = split_node::exec(v, ectx, group, p.ptr(), pos);
        // if (!hnl.is_null()) {
        delete_node::exec(v, ectx, group, el);
        insert_node::exec(v, ectx, group, pp, pos, el);
        //}
      }
      if (target_tag != el->tag)
        morph_element::exec(v, ectx, group, el, target_tag);

      purge_empty_blocks(v, ectx, group, originalp);
      return el;
    }

    element *handle_cr_in_empty_para(view &v, editing_ctx *ectx, action *group,
                                     helement el) {
      helement p = el->parent; // the list
      if (!p) return nullptr;
      helement pp = p->parent;
      if (!pp) return nullptr;
      if (el->tag == tag::T_LI && CONST_SLICE(lists).contains(el->parent->tag))
        goto UNLIST;
      else if (el->tag == tag::T_DT && el->parent->tag == tag::T_DL)
        goto UNLIST;
      else if (el->tag == tag::T_DD && el->parent->tag == tag::T_DL)
        goto UNLIST;
      else if (el->parent->tag == tag::T_BLOCKQUOTE)
        goto UNLIST;
      return nullptr;
    UNLIST:
      return unlist_list_item(v, ectx, group, el);
    }

      // this breaks text of block into two parts:
      //  1. text before bm and
      //  2. paragraph after it so this
      //   <blockquote>one|two</blockquote>
      //  will be this
      //   <blockquote>one<p>two</p></blockquote>

#if 0
  element* break_block(view& v, editing_ctx* ectx, action* op, element* base, bookmark& bm )
  {
    bool created = false;
    bookmark dummy;
    split_at(v,ectx,op,bm, base, false, created, dummy);
    //if(!created)
    //  return nullptr;

    int si = bm.linear_pos();
    int ei = base->nodes.size();
    for( int i = si; i < base->nodes.size(); ++i )
    {
      if( base->nodes[i]->is_element()
        && base->nodes[i].ptr_of<element>()->is_block_element(v) ) {
          ei = i;
          break;
      }
    }
    array<hnode> nodes = base->nodes(si,ei);
#ifdef _DEBUG
    base->dbg_report("break_block");
#endif // _DEBUG

    delete_nodes_range::exec(v,ectx,op,base,si,ei);
    helement ne = new element(tag::T_P);
    if( !is_empty_sequence(nodes()) )
      ne->append_nodes(nodes());
    insert_node::exec(v,ectx,op,base,si,ne);
    bm = ne->start_pos();
    return ne;
  }
#else
    element *break_block(view &v, editing_ctx *ectx, action *op, element *base,
                         bookmark &bm) {
      //bool base_can_contain_subblocks =
      //    base->tag != tag::T_P && base->tag != tag::T_TEXT;
      bool     created = false;
      bookmark dummy;
      split_at(v, ectx, op, bm, base, false, created, dummy);
      // if(!created)
      //  return nullptr;

      int si = bm.linear_pos();
      int ei = base->nodes.size();
      int zi = 0;

      for (int i = si - 1; i >= 0; --i) {
        if (base->nodes[i]->is_element() &&
            base->nodes[i].ptr_of<element>()->is_block_element(v))
          break;
        zi = i;
      }

      for (int i = si; i < base->nodes.size(); ++i) {
        if (base->nodes[i]->is_element() &&
            base->nodes[i].ptr_of<element>()->is_block_element(v)) {
          ei = i;
          break;
        }
      }

      array<hnode> nodes_after  = base->nodes(si, ei);
      array<hnode> nodes_before = base->nodes(zi, si);
#ifdef _DEBUG
      base->dbg_report("break_block");
#endif // _DEBUG

      helement el;

      if (si < ei) delete_nodes_range::exec(v, ectx, op, base, si, ei);

      // if (!is_empty_sequence(nodes_after()))
      {
        helement ne = new element(tag::T_P);
        ne->append_nodes(nodes_after());
        insert_node::exec(v, ectx, op, base, si, ne);
        bm = ne->start_pos();
        el = ne;
      }

      /*if (base_can_contain_subblocks) {

        if (zi < si) delete_nodes_range::exec(v, ectx, op, base, zi, si);

        // if (!is_empty_sequence(nodes_before()))
        {
          helement ns = new element(tag::T_P);
          ns->append_nodes(nodes_before());
          insert_node::exec(v, ectx, op, base, zi, ns);
          if (zi >= si) bm = ns->start_pos();
          // el = ns;
        }
      }*/

      return el;
    }
#endif

    bool richtext_ctl::remove_selection(view &v) {
      richtext_ctl *_this = static_cast<richtext_ctl *>(this);
      if (_this->anchor.normalized() == _this->caret.normalized()) return false;
      bookmark bm = delete_range(v, _this->anchor, _this->caret,true);
      if (bm.valid()) {
        _this->select(v, bm);
        return true;
      }
      return false;
    }

    bookmark break_position(view &v, element *root, bookmark bm) {
      element *el = bm.node->get_element();
      ASSERT(el);
      if (el->is_atomic_box() && el != root)
        return el->this_pos(bm.at_element_end());
      switch (el->tag) {
        case tag::T_TABLE: return el->this_pos(bm.at_element_end());
        case tag::T_TBODY: return bookmark();
        case tag::T_THEAD: return bookmark();
        case tag::T_TFOOT: return bookmark();
      }

      return bm;
    }

    // gets containment level
    uint get_element_level(element *self, element *el) {
      if (el == self || !el) return 0;
      if (el->tag == tag::T_BODY) return 0;
      return 1 + get_element_level(self, el->parent);
    }

    bool richtext_ctl::insert_break(view &v, bookmark start, bookmark end) {
      if (start == end && start.valid() &&
          start.node->is_of_type<block_table_row>())
        return insert_row(v, start);

      richtext_ctl *_this = static_cast<richtext_ctl *>(this);
      _this->clear_comp_chars(v);

      static tag::symbol_t _containers[] = {
          tag::T_TD,      tag::T_TH,       tag::T_BLOCKQUOTE, tag::T_CAPTION,
          tag::T_ADDRESS, tag::T_FIELDSET, tag::T_LEGEND,     tag::T_OL,
          tag::T_UL,      tag::T_DL,       tag::T_MENU,       tag::T_DIR,
          tag::T_PRE,     tag::T_TABLE,
      };
      static slice<tag::symbol_t> containers = CONST_SLICE(_containers);

      handle<action> op = new action(_this, WCHARS("insert break"));

      try {
        bookmark bm = start;
        if (start != end) {
          if (start > end) swap(start, end);
          bm = remove_range(v, _this, op, start, end, is_richtext(),false);
        }

        if (!bm.valid()) return false;

        bm = break_position(v, root(), bm);

        helement base = bm.node->nearest_block(v);
        if (!base || !base->belongs_to(root_at(v, base), true)) return false;

        // base->dbg_report("insert break in:");

        bool     created = false;
        bool     advance = true;
        helement nel;

        if (is_at_pre(v, bm) &&
            insert_text::exec(v, _this, op, bm, WCHARS("\n"))) {
          advance = false;
          goto DONE;
        }

        if (element *ep = is_at_empty_para(v, bm)) {
          nel = handle_cr_in_empty_para(v, _this, op, ep);
          if (nel) {
            bm = ep->start_pos();
            goto PRETTIFY;
          }
          //return false;
        }

        if (bm == base->end_pos())
        {
          nel = make_empty_p(base);
          insert_node::exec(v, _this, op, base->parent, base->node_index + 1,
                            nel);
          bm = nel->start_pos();
          goto PRETTIFY;
        }
        else if (bm == base->start_pos())
        {
          nel = make_empty_p(base);
          if (base == root())
            insert_node::exec(v, _this, op, base, 0, nel);
          else
            insert_node::exec(v, _this, op, base->parent, base->node_index,
                              nel);
          bm = nel->start_pos();
          goto PRETTIFY;
        }
        else if (bm == base->end_caret_pos(v) || is_at_empty_para(v, bm))
        {
          // base->dbg_report("break");
          if (containers.contains(base->tag)) {
            // nel = new element(tag::T_P);
            // insert_node::exec(v, _this, op, base, base->nodes().size(), nel);
            // bm = nel->start_pos();
            nel = break_block(v, _this, op, base, bm);
            goto PRETTIFY;
          } else if (base->is_inline_block_element(v))
            base = base->parent;
          else {
            nel = make_empty_p(base);
            if (base == root())
              insert_node::exec(v, _this, op, base, base->nodes.size(), nel);
            else
              insert_node::exec(v, _this, op, base->parent,
                                base->node_index + 1, nel);
            bm = nel->start_pos();
            goto PRETTIFY;
          }
        }
        else if (bm == base->start_caret_pos(v))
        {
          if (containers.contains(base->tag)) {
            nel = break_block(v, _this, op, base, bm);
            // nel = new element(tag::T_P);
            // insert_node::exec(v, _this, op, base, 0, nel);
            // bm = nel->start_pos();
            goto PRETTIFY;
          } else if (base->is_inline_block_element(v))
            base = base->parent;
          else {
            nel = make_empty_p(base);
            if (base == root())
              insert_node::exec(v, _this, op, base, 0, nel);
            else
              insert_node::exec(v, _this, op, base->parent, base->node_index,
                                nel);
            bm = nel->start_pos();
            goto PRETTIFY;
          }
        }

        if (containers.contains(base->tag) ||  base == root_at(v,base) ) {
          nel = break_block(v, _this, op, base, bm);
        } else {
          bookmark dummy;
          nel = split_element_at(v, _this, op, bm, base, false, created, dummy);
          nel->check_layout(v);
          if (!nel->is_text_box() && !nel->is_empty()) {
            // <li><ul>...</ul></li> is kind of prohibited so we need text container at the beginning, kind of caption, eh?
            // <li><text>...</text>
            //     <ul>...</ul></li>
            if (is_text_container_tag(nel->tag)) {
              hnode pfn = nel->first_node();
              if (pfn->is_space())
              {
                delete_node::exec(v, _this, op, nel->first_node());
                helement br = new element(tag::T_TEXT);
                insert_node::exec(v, _this, op, nel, 0, br);
                bm = br->this_pos(false);
              }
              else if (pfn->is_element() && pfn.ptr_of<element>()->is_block_element(v))
              {
                helement br = new element(tag::T_TEXT);
                insert_node::exec(v, _this, op, nel, 0, br);
                bm = br->this_pos(false);
              }
            }
          }
        }

      PRETTIFY:
#if 1
        if (nel) {
          tag::TAG_TYPE tt = tag::type(nel->tag);
          if (tt == tag::BLOCK_TAG || tt == tag::TABLE_TAG ||
              tt == tag::TABLE_BODY_TAG || tt == tag::TABLE_ROW_TAG ||
              tt == tag::TABLE_CELL_TAG) {
            // string s = el->get_text(v);
            handle<text> pn    = new text(WCHARS("\r\n"));
            uint         level = get_element_level(self, nel);
            while (level) {
              pn->chars.push(WCHARS("\t"));
              --level;
            }
            insert_node::exec(v, _this, op, nel->parent, nel->node_index, pn);
          }
        }
#endif
      DONE:
        v.commit_update(true);
        if (advance) {
          if (_this->advance(v, bm, ADVANCE_NEXT)) _this->select(v, bm);
        } else
          _this->select(v, bm);

        if (op->chain) push(v, op);
        return true;
      } catch (const tool::exception &) { op->undo(v, _this); }
      return false;
    }


    bool richtext_ctl::insert_element(view &v, bookmark start, bookmark end,
                                      helement el) {
      richtext_ctl *_this = static_cast<richtext_ctl *>(this);
      _this->clear_comp_chars(v);

      handle<action> op = new action(_this, WCHARS("insert element"));

      try {

        bookmark bm = start;
        if (start != end) {
          if (start > end) swap(start, end);
          bm = remove_range(v, _this, op, start, end, is_richtext(),false);
          if (!bm.valid()) return false;
        }

        helement base;

        if (tag::require_block_container(el->tag))
          base = bm.node->nearest_box(false);
        else
          base = bm.node->parent;

        if (!base || !base->belongs_to(_this->root(), true)) return false;

        bool     created = false;
        bookmark dummy;
        split_at(v, _this, op, bm, base, false, created, dummy);
        insert_node::exec(v, _this, op, base, bm.linear_pos(), el);
        if (el->tag == tag::T_BR) {
          hnode nn = el->next_node();
          if (!nn) {
            //wchar nbsp = NBSP_CHAR;
            //nn = new text(nbsp);
            nn = new text(WCHARS(" "));
            insert_node::exec(v, _this, op, base, el->node_index+1, nn );
          }
          bm = nn->start_caret_pos(v);//el->start_caret_pos(v);
        }
        else 
          bm = el->start_caret_pos(v);

        v.commit_update();

        if (!bm.at_caret_pos(v)) 
          _this->advance(v, bm, ADVANCE_NEXT);
        _this->select(v, bm);

        push(v, op);
        return true;
      } catch (const tool::exception &) { op->undo(v, _this); }
      return false;
    }

    bool richtext_ctl::insert_soft_break(view &v, bookmark start,
                                         bookmark end) {
            
      if (is_at_pre(v, start))
        return insert_chars(v, start, end, WCHARS("\n"));
      else {
        return insert_element(v, start, end, new element(tag::T_BR));
        /*helement he = new element(tag::T_BR);
        if (insert_element(v, start, end, he))
        {
          bookmark bm = he->end_caret_pos(v);
          if(advance_forward(v, bm))
            select(v, bm);
        }*/
      }
    }

    bool richtext_ctl::insert_block_break(view &v, bookmark start,
                                          bookmark end) {
      if (start == end && start.valid() &&
          start.node->is_of_type<block_table_row>())
        return insert_row(v, start);

      richtext_ctl *_this = static_cast<richtext_ctl *>(this);
      _this->clear_comp_chars(v);

      handle<action> op = new action(_this, WCHARS("split paragraph"));

      try {

        bookmark bm = start;
        if (start != end) {
          if (start > end) swap(start, end);
          bm = remove_range(v, _this, op, start, end, is_richtext(),false);
          if (!bm.valid()) return false;
        }

        // bm.linearize();
        helement until;
        helement base = get_inlines_container(bm.node->get_element(), root());
        if (!base || !base->belongs_to(_this->root(), true)) return false;

        // static tag::symbol_t phrasing_content[] = { tag::T_P, tag::T_H1,
        // tag::T_H2, tag::T_H3, tag::T_H4, tag::T_H5, tag::T_H6 };

        if (is_phrasing_block_tag(base->tag))
          return insert_break(v, start, end);

        // bool created = false;

        if (bm == base->end_caret_pos(v)) {
          helement nel = new element(tag::T_P);
          // nel->append(new element(tag::T_BR)); //
          insert_node::exec(v, _this, op, base, base->nodes.size(), nel);
          bm = nel->start_pos();
          goto DONE;
        } else if (bm == base->start_caret_pos(v)) {
          helement nel = new element(tag::T_P);
          // nel->append(new element(tag::T_BR)); //
          insert_node::exec(v, _this, op, base, 0, nel);
          bm = nel->start_pos();
          goto DONE;
        }

        break_block(v, _this, op, base, bm);

      DONE:
        v.commit_update();
        if (_this->advance(v, bm, ADVANCE_NEXT)) _this->select(v, bm);
        push(v, op);

        return true;
      } catch (const tool::exception &) { op->undo(v, _this); }
      return false;
    }

    bookmark richtext_ctl::delete_range_in(view &v, action *act, bookmark start, bookmark end, bool forward) {
      this->clear_comp_chars(v);

      this->refresh_selection(v);

      if (root_at(v, start) != root_at(v, end)) {
        beep(); // selection spans different cells ?
        return bookmark();
      }

      this->caret  = bookmark();
      this->anchor = bookmark();

      bookmark bm = remove_range(v, this, act, start, end, is_richtext(),forward);

      return bm;
    }

    bookmark richtext_ctl::delete_range(view &v, bookmark start, bookmark end, bool forward) {
      if (transact_group) return delete_range_in(v, transact_group, start, end,forward);

      handle<range_action> group =
          new range_action(this, WCHARS("delete range"));

      try {

        bookmark bm = delete_range_in(v, group, start, end, forward);

        push(v, group);

        v.commit_update();

        return bm;
      } catch (const tool::exception &) { group->undo(v, this); }
      // otherwise it is just a prolongation of previous op
      return start;
    }

    void fix_glue_positions(view& v, bookmark& start, bookmark& end)
    {
      auto compatible_with_text_glue = [](element* pel) {
        if (!pel->is_text_box()) return false;
        if (pel->nodes.length() == 0) return false;
        return true;
      };
      helement s_block = start.node->nearest_box(true); //s_block->dbg_report("start"); start.dbg_report("bm start");
      helement e_block = end.node->nearest_box(true); //e_block->dbg_report("end"); end.dbg_report("bm end");
      if ((s_block != e_block)
        && compatible_with_text_glue(e_block)
        && compatible_with_text_glue(s_block))
      { // the case when two paragraphs are glued together
        start = s_block->last_node()->end_pos();
        end = e_block->first_node()->start_pos();
      }
    }

    bookmark richtext_ctl::delete_char(view &v, bookmark bm, bool forward) {
        
      this->clear_comp_chars(v);

      if (forward) {
        //this->advance(v, bm, NORMALIZE_INS_POS);

        bookmark start = bm;
        bookmark end   = bm;

        //if( bm.at_element_start() && bm.node.ptr_of<element>()->state.selected() )


        if (this->advance(v, end, ADVANCE_RIGHT) && start.valid() && end.valid()) {
          if (end.node == start.node && start.node->is_text()) {

            handle<text> t = start.node.ptr_of<text>();
            int          s = start.linear_pos();
            int          e = end.linear_pos();
            start = t->this_pos(false);
            end = t->this_pos(true);

            remove_char_forward *rcf = this->top_remove_char_forward();
            if (rcf && rcf->is_compatible(t, s, e) && rcf->append(v, this, top(), t,s, e, bm))
            {
              event_behavior evt(self, self, EDIT_VALUE_CHANGED, CHANGE_BY_DEL_CONSECUTIVE_CHAR);
              v.post_behavior_event(evt, true);
              return bm;
            }
            else {
              handle<action> act = new action(this, WCHARS("delete character"));
              try {
                bm = remove_char_forward::exec(v, this, act, t, s, e);
                if (!bm.valid()) bm = end;
                bm = check_caret_position(v, this, act, bm);
                if (bm.valid()) {
                  select(v, bm);
                  push(v, act.ptr());
                  return bm;
                }
              } catch (const tool::exception &) { act->undo(v, this); }
              return start;
            }
          }
          fix_glue_positions(v, start, end);
          return delete_range(v, start, end, forward);
        } else if (!advance(v, start, ADVANCE_PREV_CHAR)) // seems like we have
                                                          // the only valid
                                                          // caret position here
        {
          // seems like we have the only valid caret position here, clear
          // content
          bookmark very_first = root()->start_caret_pos(v);
          bookmark very_last  = root()->end_caret_pos(v);
          return delete_range(v, very_first, very_last,forward);
        }
      } else // backward
      {
        // advance(v,bm,NORMALIZE_DEL_POS);

        bookmark start = bm;
        bookmark end   = bm;
        if (advance(v, start, ADVANCE_LEFT) && start.valid() && end.valid()) {
          if (end.node == start.node && start.node->is_text()) {
            // if( end == end.node->end_caret_pos(v) && start ==
            // end.node->start_caret_pos(v) )
            //  return
            //  delete_range(v,start.node->this_pos(false),start.node->this_pos(true));
            handle<text> t = start.node.ptr_of<text>();
            int          s = start.linear_pos();
            int          e = end.linear_pos();
            start = t->this_pos(false);
            end = t->this_pos(true);

            remove_char_backward *rcf = top_remove_char_backward();
            if (rcf && rcf->is_compatible(t, s, e) && rcf->append(v, this, top(), t, s, e, bm)) {
              event_behavior evt(self, self, EDIT_VALUE_CHANGED,CHANGE_BY_DEL_CONSECUTIVE_CHAR);
              v.post_behavior_event(evt, true);
              return bm;
            }
            else {
              handle<action> act = new action(this, WCHARS("delete character"));
              try {
                bm = remove_char_backward::exec(v, this, act, t, s, e);
                if (!bm.valid()) bm = start;
                bm = check_caret_position(v, this, act, bm, true);
                if (bm.valid()) {
                  select(v, bm);
                  push(v, act.ptr());
                  return bm;
                }
              } catch (const tool::exception &) { act->undo(v, this); }
              return start;
            }
          }
          fix_glue_positions(v, start, end);
          return delete_range(v, start, end, forward);
        } else if (!advance(v, start, ADVANCE_NEXT_CHAR)) {
          // seems like we have the only valid caret position here, clear
          // content
          bookmark very_first = root()->start_caret_pos(v);
          bookmark very_last  = root()->end_caret_pos(v);
          return delete_range(v, very_first, very_last, forward);
        }
      }

      return bookmark();
    }

    void flatten_list(array<helement> &list_items) {
      for (int i = list_items.last_index(); i >= 0; --i) {
        for (int k = i - 1; k >= 0; --k)
          if (list_items[i]->owned_by(list_items[k], true)) {
            list_items.remove(i);
            break;
          }
      }
    }

    bool do_apply_list(view &v, editing_ctx *ectx, action *act, bookmark start,
                       bookmark end, tag::symbol_t t,
                       const attribute_bag &atts) {
      richtext_ctl *_this = static_cast<richtext_ctl *>(ectx);
      _this->clear_comp_chars(v);
      array<helement> list_items;
      array<helement> cell_items;

      auto match = [&](node *n, bool &skip) -> bool {
        helement      t  = n->nearest_box(false);
        tag::TAG_TYPE tt = tag::type(t->tag);
        if (tt == tag::BLOCK_TAG || tt == tag::TABLE_TAG) {
          skip = true;
          list_items.push(t);
        }
        else if (tt == tag::TABLE_CELL_TAG)
        {
          cell_items.push(t);
          skip = true;
        }
        return false;
      };
      _this->selection_each(v, match);

      // check "flatness" of the table
      flatten_list(list_items);

      if (list_items.size())
        goto DO_LIST;
      else if (cell_items.size()) {
        for (int n = 0; n < cell_items.size(); ++n) {
          helement he = cell_items[n];
          helement we = wrap_nodes::exec(v, ectx, act, he, 0, he->nodes.size(), tag::T_P, attribute_bag());
          list_items.push(we);
        }
      }
      else
        return false;
DO_LIST:
      for (int i = list_items.last_index(); i >= 0; --i) {
        helement el = list_items[i];
        if (el->tag == tag::T_LI || el->tag == tag::T_DT ||
            el->tag == tag::T_DD)
          unlist_list_item(v, _this, act, list_items[i]);
      }

      helement list;
      int      list_at = 0;

      do {
        helement prev = list_items.first()->prev_element();
        if (prev && prev->tag == t) {
          list    = prev;
          list_at = list->nodes.size();
          break;
        }
        helement next = list_items.last()->next_element();
        if (next && next->tag == t) {
          list = next;
          break;
        }

        list       = new element(t);
        list->atts = atts;

        helement p       = list_items[0]->parent;
        int      p_index = list_items[0]->node_index;

        helement prior;
        for (int n = p_index; n >= 0; --n) {
          if (p->nodes[n]->is_element()) {
            prior = p->nodes[n].ptr_of<element>();
            break;
          }
        }

        if (prior && prior->tag == tag::T_LI)
          insert_node::exec(v, _this, act, prev, 0xffff, list);
        else
          insert_node::exec(v, _this, act, p, p_index, list);

      } while (0);

      for (int i = 0; i < list_items.size(); ++i) {
        delete_node::exec(v, _this, act, list_items[i]);
        insert_node::exec(v, _this, act, list, list_at++, list_items[i]);
        tag::symbol_t tt = tag::T_LI;
        if (t == tag::T_DL)
          tt = (list_items[i]->index() % 2) ? tag::T_DD : tag::T_DT;
        morph_element::exec(v, _this, act, list_items[i], tt);
      }
      return true;
    }

    bool richtext_ctl::apply_list(view &v, bookmark start, bookmark end,
                                  tag::symbol_t t, const attribute_bag &atts) {

      if (transact_group)
        return do_apply_list(v, this, transact_group, start, end, t, atts);

      handle<action> group = new range_action(this, WCHARS("apply list"));
      try {

        if (!do_apply_list(v, this, group, start, end, t, atts)) return false;

        push(v, group);

        v.commit_update();

        return true;
      } catch (const tool::exception &) { group->undo(v, this); }
      return false;
    }

    bool do_remove_list(view &v, editing_ctx *ectx, action *act, bookmark start,
                        bookmark end, tag::symbol_t t,
                        const attribute_bag &atts) {
      richtext_ctl *_this = static_cast<richtext_ctl *>(ectx);
      _this->clear_comp_chars(v);

      array<helement> list_items;
      wchars          selector;
      switch (t) {
      case tag::T_DL: selector = WCHARS("dl>dd,dl>dt"); break;
      case tag::T_OL: selector = WCHARS("ol>li"); break;
      case tag::T_UL: selector = WCHARS("ul>li"); break;
      case tag::T_DIR: selector = WCHARS("dir>li"); break;
      case tag::T_MENU: selector = WCHARS("menu>li"); break;
      }

      auto match = [&](node *n, bool &skip) -> bool {
        helement t = n->get_element();
        if (matches(v, _this->root(), t, selector)) {
          skip = true;
          list_items.push(t);
        }
        return false;
      };
      _this->selection_each(v, match);

      if (list_items.size() == 0) return false;

      // check "flatness" of the table
      flatten_list(list_items);

      if (!list_items.size()) return false;

      for (int i = list_items.last_index(); i >= 0; --i)
        unlist_list_item(v, _this, act, list_items[i]);

      return true;
    }

    bool richtext_ctl::remove_list(view &v, bookmark &start, bookmark &end,
                                   tag::symbol_t t, const attribute_bag &atts) {
      if (transact_group)
        return do_remove_list(v, this, transact_group, start, end, t, atts);

      handle<action> group = new range_action(this, WCHARS("remove list"));
      try {

        if (!do_remove_list(v, this, group, start, end, t, atts)) return false;

        push(v, group);

        v.commit_update();

        return true;
      } catch (const tool::exception &) { group->undo(v, this); }
      return false;
    }

    bool richtext_ctl::morph_blocks(view &v, bookmark start, bookmark end,
                                    tag::symbol_t ts, bool do_it) {
      richtext_ctl *_this = static_cast<richtext_ctl *>(this);
      _this->clear_comp_chars(v);

      array<helement> blocks;
      auto            match = [&](node *n, bool &skip) -> bool {
        helement t = n->nearest_box(true);
        // tag::TAG_TYPE tt = tag::type(t->tag);
        helement root = root_at(v, t);
        if (t->is_text_box() /*&& (CONST_SLICE(phrasing_blocks).contains(t->tag) || t == root)*/) {
          skip = true;
          if (t->tag != ts) blocks.push(t);
        }
        return false;
      };
      _this->each_block_element(v, anchor, caret, match, true);
      // assert(blocks.size());

      if (do_it) {

        if (blocks.size() == 0) return true;

        handle<action> group =
            transact_group ? transact_group
                           : new range_action(_this, WCHARS("morph block"));

        for (int i = blocks.last_index(); i >= 0; --i) {
          helement root = this->root_at(v, blocks[i]);
          if (blocks[i] == root) {
            helement     p     = new element(ts);
            array<hnode> nodes = root->nodes();
            delete_nodes_range::exec(v, this, group, root, 0, nodes.size());
            insert_node::exec(v, this, group, root, 0, p);
            insert_nodes::exec(v, this, group, p, 0, nodes());
          } else if (blocks[i]->is_anonymous_text_block()) {
            helement     p      = new element(ts);
            array<hnode> nodes  = blocks[i]->nodes();
            helement     parent = nodes[0]->parent;
            int          idx    = nodes[0]->node_index;
            for (int n = nodes.last_index(); n >= 0; --n)
              delete_node::exec(v, this, group, nodes[n]);
            insert_node::exec(v, this, group, parent, idx, p);
            insert_nodes::exec(v, this, group, p, 0, nodes());
          } else
            morph_element::exec(v, _this, group, blocks[i], ts);
        }

        if (!transact_group) push(v, group);
        v.commit_update();
      } else
        return blocks.size() != 0;

      return true;
    }

    bool shift_element(view &v, editing_ctx *ectx, action *group, helement el,
                       tag::symbol_t cont, bool only_this_one) {
      if (el->prev_element() && el->prev_element()->tag == cont) {
        helement prev = el->prev_element();

        delete_node::exec(v, ectx, group, el);
        insert_node::exec(v, ectx, group, prev, 0xffff, el);
      } else if (el->is_of_type<text_block>() && el->tag != cont &&
                 only_this_one) {
        morph_element::exec(v, ectx, group, el, cont);
      } else {
        helement p   = el->parent;
        int      pos = el->node_index;
        helement n   = new element(cont);
        delete_node::exec(v, ectx, group, el);
        insert_node::exec(v, ectx, group, p, pos, n);
        insert_node::exec(v, ectx, group, n, 0, el);
      }

      return true;
    }

    bool richtext_ctl::apply_block(view &v, bookmark start, bookmark end,
                                   tag::symbol_t ts) {
      this->clear_comp_chars(v);

      helement root = root_at(v,start);
      if (root != root_at(v, end))
        return false;

      if (start > end) swap(start, end);

      helement nb = start.node->nearest_box(false);
      if (nb == root)
      {
        helement wrapper = new element(ts);
        handle<range_action> act = new range_action(this, WCHARS("wrap block"));
        if (wrap_into(v, this, act, start, end, wrapper))
        {
          push(v, act);
          v.commit_update();
          this->select(v, end, start);
          return true;
        }
        return false;
      }

      array<helement> elements;
      auto            store_elem = [&](element *n, bool &skip) -> bool {
        helement t = n->nearest_box(false);
        if (t->belongs_to(root, false)) {
          skip = true;
          elements.push(t);
        }
        return false;
      };
      this->each_block_element(v, start, end, store_elem);
      assert(elements.size());
      if (elements.size() == 0)
        return false;


      // check "flatness" of the table
      flatten_list(elements);

      if (!elements.size()) return false;

      handle<action> act = new range_action(this, WCHARS("apply block"));
      for (int i = 0; i < elements.size(); ++i)
        shift_element(v, this, act, elements[i], ts, elements.size() == 1);

      push(v, act);

      v.commit_update();

      return true;
    }

    bool richtext_ctl::remove_block(view &v, bookmark start, bookmark end,
                                    tag::symbol_t tag) {
      this->clear_comp_chars(v);

      if (start > end) swap(start, end);
      array<helement> list_items;

      auto match = [&](element *el, bool &skip) -> bool {
        helement t  = el;
        helement pt = t->nearest_bfc(v);
        while (t && t != pt) {
          if (t->tag == tag) {
            list_items.push(t);
            // return true;
            break;
          }
          t = t->parent;
        }
        return false;
      };
      this->each_block_element(v, start, end, match);
      assert(list_items.size());

      // check "flatness" of the table
      flatten_list(list_items);

      if (!list_items.size()) return false;

      // element* ip = list_items[0]->parent;
      // if (ip == this->root_at(v, list_items[0]))
      //  return false;

      handle<action> group = new range_action(this, WCHARS("reset block"));

      try {

        for (int i = list_items.last_index(); i >= 0; --i) {
          unblock(v, this, group, list_items[i], tag);
        }

        push(v, group);

        v.commit_update();

        return true;
      } catch (const tool::exception &) { group->undo(v, this); }
      return false;
    }

    bool richtext_ctl::can_unindent(view &v, bookmark start, bookmark end) {
      if (!start.valid() || !end.valid()) return false;
      if (is_table_selection()) return false;

      if (start > end) swap(start, end);
      array<helement> list_items;

      // wchars selector = WCHARS("blockquote,blockquote>*,dd,dt,li");
      auto store_elem = [&](node *n, bool &skip) -> bool {
        helement t = n->get_element();
        // if(matches(v, _this->root(), t, selector))
        {
          skip = true;
          list_items.push(t);
        }
        return false;
      };
      this->each_block_element(v, start, end, store_elem);
      if (!list_items.size()) return false;

      // check "flatness" of the list
      flatten_list(list_items);

      element *proot = root_at(v, list_items[0]);

      for (int n = 0; n < list_items.size(); ++n) {
        element *pe = list_items[n];
        if (pe == proot || pe->parent == proot) return false;
        element *troot = root_at(v, pe);
        if (troot != proot) return false; // only lists of the same level
      }
      return true;
    }

    bool richtext_ctl::unindent(view &v, bookmark start, bookmark end) {
      this->clear_comp_chars(v);

      if (start > end) swap(start, end);
      array<helement> list_items;
      // wchars selector = unindents_selector();
      auto match = [&](node *n, bool &skip) -> bool {
        helement t = n->get_element();
        // if(matches(v, _this->root(), t, selector))
        {
          skip = true;
          list_items.push(t);
        }
        return false;
      };
      this->each_block_element(v, start, end, match);
      assert(list_items.size());

      // check "flatness" of the table
      flatten_list(list_items);

      if (!list_items.size()) return false;

      element *ip = list_items[0]->parent;
      if (ip == this->root_at(v, list_items[0])) return false;

      handle<action> group = new range_action(this, WCHARS("unindent"));

      try {

        for (int i = list_items.last_index(); i >= 0; --i) {
          unlist_list_item(v, this, group, list_items[i]);
        }

        push(v, group);

        v.commit_update();

        return true;
      } catch (const tool::exception &) { group->undo(v, this); }
      return false;
    }

    bool indent_element(view &v, editing_ctx *ectx, action *group, helement el,
                        bool only_this_one) {
      // auto is_box = [](view& v, element* el) -> bool { return
      // el->is_box_element(v); };

      /*    auto next_box = [&](element* self, slice<tag::symbol_t>
         compatible_containers) -> element* { element* t = self->next_element();
            while(t)
              if( compatible_containers.contains(t->tag) )
                return t;
              else {
                node* nn = t->first_nonspace_node();
                if( !nn->is_element() )
                  return 0;
                t = nn->cast<element>();
              }
            return 0;
          };
          auto prev_box = [&](element* self, slice<tag::symbol_t>
         compatible_containers) -> element* { element* t = self->prev_element();
            while(t)
              if( compatible_containers.contains(t->tag) )
                return t;
              else {
                node* nn = t->last_nonspace_node();
                if( !nn->is_element() )
                  return 0;
                t = nn->cast<element>();
              }
            return 0;
          }; */

      auto shift = [&](helement             el,
                       slice<tag::symbol_t> compatible_containers) {
        /*helement next = next_box(el,compatible_containers);
      helement prev = prev_box(el,compatible_containers);

      if( next.is_defined() && prev.is_defined() && compatible(next,prev) && (next->parent == prev->parent)) {
        // append to prev
        delete_node::exec(v,ectx,group,el);
        insert_node::exec(v,ectx,group,prev,prev->nodes.size(),el);
        // glue prev and next
        array<hnode> nodes = next->nodes();
        delete_nodes_range::exec(v,ectx,group,next,0,nodes.size());
        insert_nodes::exec(v,ectx,group,prev,prev->nodes.size(),nodes());
        delete_node::exec(v,ectx,group,next);
        if( CONST_SLICE(lists).contains(next->tag) && el->tag != tag::T_LI )
          morph_element::exec(v,ectx,group,el,tag::T_LI);
      }
      else if( prev.is_defined() ) {
        delete_node::exec(v,ectx,group,el);
        insert_node::exec(v,ectx,group,prev,prev->nodes.size(),el);
        if( CONST_SLICE(lists).contains(prev->tag) && el->tag != tag::T_LI )
          morph_element::exec(v,ectx,group,el,tag::T_LI);
      }
      else if( next.is_defined() ) {
        delete_node::exec(v,ectx,group,el);
        insert_node::exec(v,ectx,group,next,0,el);
        if( CONST_SLICE(lists).contains(next->tag) && el->tag != tag::T_LI )
          morph_element::exec(v,ectx,group,el,tag::T_LI);
      }
      else*/ if (el->tag == tag::T_LI) {
          if (!el->prev_element() || el->prev_element()->tag != tag::T_LI)
            return;
          helement prevli = el->prev_element();
          helement prevl  = prevli->last_element();
          helement p;

          delete_node::exec(v, ectx, group, el);

          if (prevl && is_list_tag(prevl->tag))
            p = prevl;
          else {
            helement n = new element(prevli->parent->tag);
            insert_node::exec(v, ectx, group, prevli, 0xffff, n);
            p = n;
          }
          insert_node::exec(v, ectx, group, p, 0xffff, el);

          // insert_node::exec(v,ectx,group,p,pos,n);
          // insert_node::exec(v,ectx,group,n,0,el);
        } else if (el->is_of_type<text_block>() &&
                   el->tag != tag::T_BLOCKQUOTE && only_this_one) {
          morph_element::exec(v, ectx, group, el, tag::T_BLOCKQUOTE);
        } else {
          helement p   = el->parent;
          int      pos = el->node_index;
          helement n   = new element(tag::T_BLOCKQUOTE);
          delete_node::exec(v, ectx, group, el);
          insert_node::exec(v, ectx, group, p, pos, n);
          insert_node::exec(v, ectx, group, n, 0, el);
        }

      };

      if (el->tag == tag::T_LI) {
        shift(el, items_of(lists));
      } else if (el->tag == tag::T_DD) {
      } else if (el->tag == tag::T_DT) {
        // ????
        morph_element::exec(v, ectx, group, el, tag::T_DD);
      } else {
        static tag::symbol_t containers[] = {tag::T_UL, tag::T_OL, tag::T_MENU,
                                             tag::T_DIR, tag::T_BLOCKQUOTE};
        shift(el, CONST_SLICE(containers));
      }

      return true;
    }

    bool richtext_ctl::can_indent(view &v, bookmark start, bookmark end) {
      if (!start.valid() || !end.valid()) return false;
      if (is_table_selection()) return false;

      if (start > end) swap(start, end);
      array<helement> elements;
      // wchars selector = WCHARS("blockquote,blockquote>*,dd,dt,li");
      auto store_elem = [&](node *n, bool &skip) -> bool {
        helement t = n->nearest_box(false);
        skip       = true;
        elements.push(t);
        return false;
      };
      this->each_block_element(v, start, end, store_elem);
      // assert(elements.size());

      // check "flatness" of the list
      flatten_list(elements);

      if (!elements.size()) return false;

      // int cnt = 0;
      for (int n = 0; n < elements.size(); ++n) {
        element *pe = elements[n];
        if (pe->tag != tag::T_LI) return false; // only li's can be shifted
        if (pe->index() == 0)
          return false; // only non-first li's can be shifted
      }
      return true;
    }

    bool richtext_ctl::indent(view &v, bookmark start, bookmark end) {
      this->clear_comp_chars(v);

      if (start > end) swap(start, end);
      array<helement> elements;
      // wchars selector = WCHARS("blockquote,blockquote>*,dd,dt,li");
      auto store_elem = [&](node *n, bool &skip) -> bool {
        helement t = n->nearest_box(false);
        skip       = true;
        elements.push(t);
        return false;
      };
      this->each_block_element(v, start, end, store_elem);
      assert(elements.size());

      // check "flatness" of the table
      flatten_list(elements);

      if (!elements.size()) return false;

      handle<action> act = new range_action(this, WCHARS("indent"));
      for (int i = 0; i < elements.size(); ++i)
        indent_element(v, this, act, elements[i], elements.size() == 1);

      push(v, act);

      v.commit_update();

      return true;
    }


    bool richtext_ctl::can_pre(view &v) {
      helement first_el;

      bool yes = false;
      bookmark start, end;
      this->normalized().unpack(start, end);

      auto check_block = [&](element *el, bool &skip) -> bool {
        if (el->is_of_type<text_block>()) {
          skip = true;
          if (!first_el) {
            first_el = el;
            yes = true;
          }
          else if (el->get_owner() != first_el->get_owner()) {
            yes = false;
            return true;
          }
        }
        return false;
      };

      each_block_element(v, start, end, check_block, true);
      return yes;
    }

    bool richtext_ctl::apply_pre(view &v, const attribute_bag &atts) {
      this->clear_comp_chars(v);

      handle<action> op = new range_action(this, WCHARS("apply pre"));

      bookmark start, end;
      this->normalized().unpack(start, end);

      helement pre_root = root_at(v,start);

      try {

        // helement base =
        // element::find_base(this->get_caret().node->get_element(),this->get_anchor().node->get_element());
        auto collapse_ws = [](array<wchar> &buf, int l) {
          while (l < buf.size()) {
            if (buf[l] == NBSP_CHAR) {
              buf[l] = ' ';
              ++l;
              continue;
            }
            if (buf[l] == '\r' || buf[l] == '\n') buf[l] = ' ';
            if (!is_space(buf[l])) {
              ++l;
              continue;
            } else if ((l + 1) < buf.size() && is_space(buf[l + 1])) {
              buf.remove(l);
              continue;
            } else
              ++l;
          }

        };

        array<helement> texts;

        auto reg_text = [&](element *el, bool &skip) -> bool {
          if (el && el->is_of_type<text_block>()) {
            skip = true;
            texts.push(el);
          }
          return false;
        };
        //this->selection_each(v, reg_text);
        each_block_element(v, start, end, reg_text, true);

        array<wchar> chars;
        int          idx = 0;
        helement     pa;

        for (int n = 0; n < texts.size(); ++n) {
          if (n) chars.push(WCHARS("\n"));
          int l = chars.size();
          texts[n]->emit_text(chars);
          collapse_ws(chars, l);
        }
        for (int n = 0; n < texts.size(); ++n) {
          if (texts[n]->is_anonymous_text_block())
          {
            ASSERT(texts[n]->parent);
            pa = texts[n]->parent;
            idx = int(pa->node_index);
            int start = int(texts[n]->nodes.first()->node_index);
            int end = int(texts[n]->nodes.last()->node_index) + 1;
            delete_nodes_range::exec(v, this, op, texts[n]->parent, start,end);
          }
          else {
            if (!pa) {
              if (texts[n] == pre_root) {
                pa = pre_root;
                idx = 0;
                delete_nodes_range::exec(v, this, op, pre_root,0, pre_root->nodes.size());
                break;
              }
              else {
                pa = texts[n]->parent;
                idx = texts[n]->node_index;
              }
            }
            delete_node::exec(v, this, op, texts[n]);
          }
        }

        ASSERT(pa);
        ASSERT(pa->belongs_to(pre_root,true));

        if ((pa->is_empty() || is_space_seq(v, pa->nodes())) && pa != pre_root)
        {
          // nothing meaningful in it left, so
          helement t = pa;
          pa = pa->parent;
          idx = int(pa->node_index);
          delete_node::exec(v, this, op, t);
        }

        helement pre = new element(tag::T_PRE);
        pre->append(new text(chars()));
        insert_node::exec(v, this, op, pa, idx, pre);

        v.commit_update();

        this->select(v, pre->end_caret_pos(v), pre->start_caret_pos(v));

        push(v, op);

        return true;
      } catch (const tool::exception &) { op->undo(v, this); }
      return false;
    }

    /*static bool unpre(view& v, editing_ctx *ectx, action* group, helement pre,
    bookmark& start, bookmark& end)
    {

      each_node it(pre);
      for( node* n; it(n); )
      {
        if( !n->is_text() )
          continue;
        handle<text> t = n->cast<text>();

        while(true)
        {
          int bri = t->chars().last_index_of('\n');
          if( bri < 0 )
            break;
          bookmark bm(t,bri,false);
          bool created = false; bookmark other;
          split_at(v,ectx,group,bm,pre->parent,true,created,other);
               assert( pre->parent == bm.node );
          handle<node> tail = bm.node.ptr_of<element>()->nodes[bm.linear_pos()];
               assert(tail->is_element());
          morph_element::exec(v,ectx,group,tail.ptr_of<element>(), tag::T_P);
        }
      }

      morph_element::exec(v,ectx,group,pre, tag::T_P);

      return true;
    }*/

    static pair<helement, helement> unpre(view &v, editing_ctx *ectx,
                                          action *group, helement pre) {
      element *cont = pre->parent;
      int      idx  = pre->node_index;

      // html::ostream_8 os;
      // pre->emit_content(os);
      //   os.data();
      array<wchar> ptx;
      pre->emit_text(ptx);

      pair<helement, helement> els;

      wtokens wt(ptx(), WCHARS("\n"));
      for (wchars t; wt.next(t);) {
        t          = trim(t);
        helement p = new element(tag::T_P);
        p->append(new text(t));
        insert_node::exec(v, ectx, group, cont, idx++, p);
        if (!els.first) els.first = p;
        els.second = p;
      }

      delete_node::exec(v, ectx, group, pre);

      return els;
    }

    bool richtext_ctl::remove_pre(view &v) {
      richtext_ctl *_this = static_cast<richtext_ctl *>(this);
      _this->clear_comp_chars(v);

      array<helement> pres;

      auto reg_pre = [&](node *n, bool &skip) -> bool {
        helement t = n->nearest_box(false);
        if (t && t->tag == tag::T_PRE) {
          skip = true;
          pres.push(t);
        }
        return false;
      };

      _this->selection_each(v, reg_pre);

      if (!pres.size()) return false;

      handle<action> act = new range_action(_this, WCHARS("remove pre"));

      try {
        helement first, last;
        _this->select(v, bookmark());
        for (int i = 0; i < pres.size(); ++i) {
          pair<helement, helement> els = unpre(v, _this, act, pres[i] /*,start,end*/);
          if (!first) first = els.first;
          last = els.second;
        }
        v.commit_update();

        ASSERT(first && last);
        ASSERT(first->is_connected() && last->is_connected());

        _this->select(v, last->end_caret_pos(v), first->start_caret_pos(v));

        push(v, act);
        return true;
      } catch (const tool::exception &) { act->undo(v, this); }
      return false;
    }

    bool richtext_ctl::insert_image(view &v, handle<gool::image> image, bookmark where) {

      helement target = where.valid() ? where.node->get_element() : self;
      event_behavior evt(target, target, PASTE_IMAGE, 0);
      evt.data = value::wrap_resource(image);
      if (v.send_behavior_event(evt))
        return true;
      if (!self)
        return false;
#if 0
      this->pasted_image_data = image->get_data();
      this->pasted_image_mime_type = image->mime_type();
      this->pasted_image_url = CHARS("cid:") + md5(this->pasted_image_data()).to_string();
      tool::markup::mem_ostream os;
      os << "<html><body>" MARKER_START_FRAGMENT "<img src='"
        << pasted_image_url() << "' />" MARKER_END_FRAGMENT "</body></html>";
      return insert_html(v, os.data()(), where);
#else 
      array<char> data_url = image->get_data_url();
      tool::markup::mem_ostream os;
      os << "<html><body>" MARKER_START_FRAGMENT "<img src='"
        << data_url() << "' />" MARKER_END_FRAGMENT "</body></html>";
      return insert_html(v, os.data()(), where);
#endif
    }

#if 1

    // get element that is block and can contain blocks - (<div> but not <p> )
    element *get_block_container(element *pel, element *until) {
      if (!pel) return nullptr;
      if (pel == until) return until;
      auto cm = tag::content_model(pel->tag);
      if (cm != tag::CMODEL_BLOCKS)
        return get_block_container(pel->parent, until);
      switch (pel->tag) {
      case tag::T_LI:
      case tag::T_DD:
      case tag::T_DT:
        ASSERT(pel->parent && pel->parent != until);
        return pel->parent;
      default: return pel;
      }
    }

    element *get_block(node *n, element *until) {
      if (!n) return nullptr;
      if (n == until) return until;
      if (n->is_element()) {
        element *pe = n->cast<element>();
        auto     tt = tag::type(pe->tag);
        if (tt == tag::BLOCK_TAG || tt == tag::TABLE_CELL_TAG) return pe;
      }
      return get_block(n->parent, until);
    }

    bool is_list(element *el) { return is_list_tag(el->tag); }

    // get element that is block and contain blocks - (<p> but not <strong> )
    element *get_inlines_container(element *pel, element *until) {
      if (!pel) return nullptr;
      if (pel == until) return until;
      auto tt = tag::type(pel->tag);
      if (tt == tag::INLINE_BLOCK_TAG || tt == tag::BLOCK_TAG ||
          tt == tag::TABLE_CELL_TAG) {
        if (is_list(pel)) return nullptr;
        auto cm = tag::content_model(pel->tag);
        if (cm == tag::CMODEL_BLOCKS || cm == tag::CMODEL_INLINES) return pel;
      }
      return get_inlines_container(pel->parent, until);
    }

    slice<hnode> get_inline_nodes(slice<hnode> &nodes) {
      slice<hnode> r;
      r.start = nodes.start;
      for (; nodes.length; ++nodes) {
        hnode n = *nodes;
        if (n->is_text() || n->is_comment()) {
          ++r.length;
          continue;
        }
        // element
        auto tt = tag::type(n.ptr_of<element>()->tag);
        if (tt == tag::INLINE_BLOCK_TAG || tt == tag::INLINE_TAG) {
          ++r.length;
          continue;
        }
        break;
      }
      return r;
    }

    slice<hnode> get_block_nodes(slice<hnode> &nodes) {
      slice<hnode> r;
      r.start = nodes.start;
      for (; nodes.length; ++nodes) {
        hnode n = *nodes;
        if (n->is_space()) {
          ++r.length;
          continue;
        }
        // element
        auto tt = tag::type(n.ptr_of<element>()->tag);
        if (tt != tag::INLINE_BLOCK_TAG && tt != tag::INLINE_TAG) {
          ++r.length;
          continue;
        }
        break;
      }
      return r;
    }

    bool get_fragment_positions(view &v, document *pd, bookmark &start,
                                bookmark &end) {
      hnode     cstart;
      hnode     cend;
      helement  body;
      each_node gen(pd);
      for (hnode n; gen(n);) {
        if (n->is_element()) {
          if (n->cast<element>()->tag == tag::T_BODY) body = n->cast<element>();
          continue;
        }
        if (!n->is_comment()) continue;
        comment *c = n->cast<comment>();
        if (c->chars().like(W("*StartFragment*"))) {
          if (cend) {
            cstart = cend;
            cend   = c;
            goto MARKERS_FOUND;
          } else
            cstart = c;
        } else if (c->chars().like(W("*EndFragment*"))) {
          cend = c;
          if (cstart) goto MARKERS_FOUND;
        }
      }

      if (body && body->first_node()) {
        start = body->first_node()->start_pos();
        end   = body->last_node()->end_pos();
        return true;
      } else if (pd && pd->first_node()) {
        start = pd->first_node()->start_pos();
        end   = pd->last_node()->end_pos();
        return true;
      }
      return false;

    MARKERS_FOUND:
      if (cstart->next_node() == cend) return false;
      start = cstart->next_node() ? cstart->next_node()->start_caret_pos(v) : cstart->start_caret_pos(v);
      end   = cend->prev_node() ? cend->prev_node()->end_caret_pos(v) : cend->end_caret_pos(v);
      cstart->remove(true);
      if (cstart != cend) cend->remove(true);
      return true;
    }

    array<hnode> get_rows_of(document *pd) {
      array<hnode> rows;

      element *tr = pd->find_by_tag(tag::T_TR);
      if (tr) {
        rows.push(tr);
        for (tr = tr->next_element(); tr; tr = tr->next_element())
          rows.push(tr);
        return rows;
      }
      /*else
      {
        array<hnode> nodes_to_insert;
        element * body = pd->find_by_tag(tag::T_BODY);
        if (body) nodes_to_insert = body->nodes();
        else      nodes_to_insert = pd->nodes();
        rows.push(new element(tag::T_TR));


        //rows.push(new element(tag::T_TR));
      }*/
      return rows;
    }

    bool richtext_ctl::insert_html(view &v, bytes html, bookmark where,
                                   const clipboard::html_item *phi) {
      clear_comp_chars(v);

      bookmark start = anchor;
      bookmark end   = caret;

      handle<action> op = new range_action(this, WCHARS("insert html"));

      array<hnode> nodes_to_insert;

      string source_url;
      if (phi) source_url = phi->url;

      handle<document> nd = new document(source_url);
      nd->flags.is_synthetic = true;
      nd->pview(&v);
      istream m(html, source_url, v.debug_mode());
      m.set_utf8_encoding();

#if defined(_DEBUG) && defined(WINDOWS)
      FILE *f = fopen("d:\\pasted.htm", "wt+");
      if (f) {
        fwrite(html.start, 1, html.length, f);
        fclose(f);
      }
#endif // _DEBUG

      parse_html(v, m, nd);

      // nd->atts.set(attr::a_src,source_url);

      bookmark sel_start, sel_end;

      helement target        = where.valid() ? where.node->get_element() : self;
      helement paste_content = nd.ptr();

      if (phi) {
        event_behavior evt(nd, target, PASTE_HTML, 0);
        evt.data = value::make_map();
        if (phi->url.is_defined())
          evt.data.set_prop("url", value(ustring(phi->url)));
        if (phi->title.is_defined())
          evt.data.set_prop("title", value(ustring(phi->title)));
        if (phi->generator.is_defined())
          evt.data.set_prop("generator", value(ustring(phi->generator)));
        if (v.send_behavior_event(evt)) return true;
        if (!self) return false;
        paste_content = evt.source;
        if (!paste_content) return false;
      }

      if (!get_fragment_positions(v, nd, sel_start, sel_end)) return false;

      //      sel_start.dbg_report("sel_start");
      //      sel_end.dbg_report("sel_end");

      bool inlines_insertion = false;

      try {
        if (start != end) {
          if (start > end) swap(start, end);

          if (where.valid()) {
            int offset = bookmark::delta(end, where);
            if (offset > 0) { // insertion is after the selection so we need to
                              // adjust position.
              where = remove_range(v, this, op, start, end, is_richtext(),false);
              where.advance(offset);
            }
          }
          else {
            where = remove_range(v, this, op, start, end, is_richtext(),false);
          }
        } else if (!where.valid())
          where = start;

        bookmark bm = where;
        if (!bm.valid()) return false;

        helement root = root_at(v, bm);

        if (bm.at_table_row_start() || bm.at_table_row_end()) {
          array<hnode> rows = get_rows_of(nd);
          if (rows.length()) {
            helement tbody = bm.node.ptr_of<element>()->parent;
            int      pos   = bm.node->node_index + int(bm.at_table_row_end());
            insert_nodes::exec(v, this, op, tbody, pos, rows());
            if (bm.at_table_row_end())
              bm = rows.last()->end_pos();
            else
              bm = rows.first()->start_pos();
          }
        } else {
          element *pic;
          if (sel_start.at_element_start() && sel_end.at_element_end() &&
              sel_start.node->parent.ptr() == sel_end.node->parent.ptr()) {
            nodes_to_insert = sel_start.node->parent->nodes()(
                sel_start.node->node_index,
                sel_end.node->node_index + 1); // selection contains inlines in
                                               // the same inlines container
          } else if (((pic = get_inlines_container(
                           sel_start.node->get_element(), root)) != nullptr) &&
                     (pic == get_inlines_container(sel_end.node->get_element(),
                                                   root))) {
#ifdef _DEBUG
            pic->dbg_report("inlines_container!");
#endif
            nodes_to_insert = pic->nodes(); // selection contains inlines in the
                                            // same inlines container
            inlines_insertion = true;
          } else {
            element *body = paste_content->find_by_tag(tag::T_BODY);
            if (body)
              nodes_to_insert = body->nodes();
            else
              nodes_to_insert = paste_content->nodes();
          }

          hnode insertion_point;
          if (bm.at_text_start() && bm.at_text_end()) {
            insertion_point = get_block(bm.node->get_element(), root);
            if (insertion_point == root) insertion_point = bm.node;
          }

          for (int i = 0; i < nodes_to_insert.size(); ++i)
            nodes_to_insert[i]->remove(false);

          slice<hnode> nodes = nodes_to_insert();

          while (nodes.length) {
            slice<hnode> inline_run = get_inline_nodes(nodes);
            if (inline_run.length) {
              helement pbc =
                  get_inlines_container(bm.node->get_element(), root);
              // inline_run[0]->dbg_report("c");
              ASSERT(pbc);
              bool     created = false;
              bookmark dummy;
              split_at(v, this, op, bm, pbc, false, created, dummy);
              ASSERT(bm.valid() && bm.node->is_element());
              helement el  = bm.node.ptr_of<element>();
              int      pos = bm.linear_pos();
              insert_nodes::exec(v, this, op, el, pos, inline_run);
              bm              = inline_run.last()->this_pos(true);
              insertion_point = nullptr;
            }
            slice<hnode> block_run = get_block_nodes(nodes);
            while (block_run.length) {
              if (insertion_point) {
                if (is_list(insertion_point->parent) &&
                    is_list(block_run[0]->get_element())) {
                  array<hnode> other_items = block_run[0]->get_element()->nodes;
                  for (int i = 0; i < other_items.size(); ++i)
                    other_items[i]->remove(false);
                  insert_nodes::exec(v, this, op, insertion_point->parent,
                                     insertion_point->node_index,
                                     other_items());
                  bm = other_items.last()->this_pos(true);
                  block_run.prune(1);
                } else {
                  insert_nodes::exec(v, this, op, insertion_point->parent,
                                     insertion_point->node_index, block_run);
                  bm               = block_run.last()->this_pos(true);
                  block_run.length = 0;
                }
                delete_node::exec(v, this, op, insertion_point);
                insertion_point = nullptr;
              } else {
                helement pbc =
                    get_block_container(bm.node->get_element(), root);
                ASSERT(pbc);
                bool     created = false;
                bookmark dummy;
                hnode    split_tail =
                    split_at(v, this, op, bm, pbc, false, created, dummy);
                ASSERT(bm.valid() && bm.node->is_element());
                helement el  = bm.node.ptr_of<element>();
                int      pos = bm.linear_pos();
                insert_nodes::exec(v, this, op, el, pos, block_run);
                bm               = block_run.last()->this_pos(true);
                block_run.length = 0;
              }
            }
          }
        }

        v.commit_update();

        while (bm.valid() && !bm.at_caret_pos(v))
          bm.advance_caret_pos(v, ADVANCE_PREV);

        // bm = nn->end_caret_pos(v);

        this->select(v, bm);

        push(v, op);
        return true;
      } catch (const tool::exception &) { op->undo(v, this); }
      return false;
    }

#else
    bool richtext_ctl::insert_html(view &v, const string &source_url,
                                   bytes html, bookmark where) {
      clear_comp_chars(v);

      bookmark start = anchor;
      bookmark end   = caret;

      handle<action> op = new range_action(this, WCHARS("insert html"));

      if (start != end) {
        if (start > end) swap(start, end);

        if (where.valid()) {
          int offset = bookmark::delta(end, where);
          where      = remove_range(v, this, op, start, end, is_richtext());
          if (offset > 0) where.advance(offset);
        } else
          where = remove_range(v, this, op, start, end, is_richtext());
      } else if (!where.valid())
        where = start;

      bookmark bm = where;
      if (!bm.valid()) return false;

      helement el  = bm.node->parent;
      int      pos = 0;
      if (bm.at_start())
        pos = bm.node->node_index;
      else if (bm.at_end())
        pos = bm.node->node_index + 1;
      else {
        split_node::exec(v, this, op, bm.node, bm.linear_pos());
        pos = bm.node->node_index + 1;
      }

      bm = bookmark(el, pos, false);

      if (!insert_html_at(v, this, op, bm, source_url, html)) return false;

      v.commit_update();

      this->select(v, bm);

      push(v, op);
      return true;
    }
#endif

    bool richtext_ctl::undo(view &v) {
      richtext_ctl *_this = static_cast<richtext_ctl *>(this);
      _this->clear_comp_chars(v);

      bool was_modified = get_modified();

      action *act = top();
      if (!act) return false;
      --depth;

      try {
        act->undo(v, static_cast<richtext_ctl *>(this));
      } catch (const tool::exception &) { ; }

      bool is_modified = get_modified();
      if (was_modified != is_modified)
        on_document_status_changed(v, is_modified);

      {
        event_behavior evt(self,self, EDIT_VALUE_CHANGED, CHANGE_BY_UNDO_REDO);
        v.post_behavior_event(evt, true);
      }
      return true;
    }

    bool richtext_ctl::redo(view &v) {
      richtext_ctl *_this = static_cast<richtext_ctl *>(this);
      _this->clear_comp_chars(v);

      if (depth >= stack.size()) return false;

      bool was_modified = get_modified();

      action *act = stack[depth++];

      act->redo(v, static_cast<richtext_ctl *>(this));

      bool is_modified = get_modified();

      if (was_modified != is_modified)
        static_cast<richtext_ctl *>(this)->on_document_status_changed(
            v, is_modified);

      {
        event_behavior evt(static_cast<richtext_ctl *>(this)->self,
                           static_cast<richtext_ctl *>(this)->self,
                           EDIT_VALUE_CHANGED, CHANGE_BY_UNDO_REDO);
        v.post_behavior_event(evt, true);
      }

      return true;
    }

    bool richtext_ctl::check_cannonic_document_structure(view& v, element *self)
    {
      handle<document> d;
      if (self->first_element() && self->first_element()->is_document())
        d = self->first_element()->cast<document>();
      if (!d)
        return false;

      helement head = find_first(v, d, WCHARS("head"));
      helement body = find_first(v, d, WCHARS("body"));

      array<hnode> metas;
      find_all(v, d, WCHARS("style,link,meta,title,base"), [&](element* el) {
        if (!head || !el->belongs_to(head)) {
          metas.push(el);
          hnode hn = el->next_node();
          if (hn && hn->is_space())
            metas.push(hn);
        }
        return false;
      });

      for (hnode hn : d->nodes()) {
        if (hn->is_comment())
          continue;
        if (hn->is_text() && hn->is_space())
          continue;
        if (hn == head || hn == body)
          continue;
        goto BAD;
      }
      if(metas.length())
        goto BAD;
      if(body)
        return false;
    BAD:

      for (auto me : metas)
        me->remove(false);

      if (head)
        head->remove(false);
      if (body)
        body->remove(false);

      array<hnode> nodes; nodes.swap(d->nodes);

      if (metas.length() && !head)
        head = new element(tag::T_HEAD);

      if (head) {
        d->append(head, &v);
        if (metas.length())
          head->append_nodes(metas(), &v);
      }
      if (!body)
        body = new element(tag::T_BODY);

      d->append(body, &v);
      body->append_nodes(nodes(), &v);

      each_node nenum(d);
      for (hnode pn; nenum(pn);) // join adjancent text nodes
      {
        hnode pnn = pn->next_node();
        if (pnn && pnn->is_text() && pn->is_text()) {
          pn->cast<text>()->chars.push(pnn.ptr_of<text>()->chars() );
          pnn->remove(true);
        }
      }

      return true;
    }

    bool richtext_ctl::check_cannonic_document_structure(view& v, element *self, handle<action> op)
    {
      handle<document> d;
      if (self->first_element() && self->first_element()->is_document())
        d = self->first_element()->cast<document>();
      if (!d)
        return false;

      helement head = find_first(v, d, WCHARS("head"));
      helement body = find_first(v, d, WCHARS("body"));

      array<hnode> metas;
      find_all(v, d, WCHARS("style,link,meta,title,base"), [&](element* el) {
        if (!head || !el->belongs_to(head)) {
          metas.push(el);
          hnode hn = el->next_node();
          if (hn && hn->is_space())
            metas.push(hn);
        }
        return false;
      });

      for (hnode hn : d->nodes()) {
        if (hn->is_comment())
          continue;
        if (hn->is_text() && hn->is_space())
          continue;
        if (hn == head || hn == body)
          continue;
        goto BAD;
      }

      if (metas.length())
        goto BAD;

      return false;
    BAD:

      for (auto me : metas) delete_node::exec(v, this, op, me);

      if(head)
        delete_node::exec(v, this, op, head);
      if(body)
        delete_node::exec(v, this, op, body);

      array<hnode> nodes = d->nodes();
      if(nodes.size())
        delete_nodes_range::exec(v, this, op, d, 0, nodes.size());

      if (metas.length() && !head) {
        head = new element(tag::T_HEAD);
      }
      if (head) {
        insert_node::exec(v, this, op, d, 0, head);
        if (metas.length())
          insert_nodes::exec(v, this, op, head, head->nodes.size(), metas());
      }
      if (!body)
        body = new element(tag::T_BODY);
      insert_node::exec(v, this, op, d, 1, body);
      insert_nodes::exec(v, this, op, body, body->nodes.size(), nodes());

      each_node nenum(d);
      for (hnode pn; nenum(pn);) // join adjancent text nodes
      {
        hnode pnn = pn->next_node();
        if (pnn && pnn->is_text() && pn->is_text()) {
          bookmark t = pn->end_pos();
          insert_text::exec(v, this, op, t, pnn.ptr_of<text>()->chars());
          delete_node::exec(v, this, op, pnn);
        }
      }
      return true;
    }


    bool richtext_ctl::morph_ctx::change_text(text* node, wchars to_text) {
      replace_text::exec(v, rt, op, node, to_text);
      return false;  // don't do def processing
    }
    
    bool richtext_ctl::morph_ctx::remove_node(node* old_node) {
      delete_node::exec(v, rt, op, old_node);
      return false;   // don't do def processing
    }
    
    bool richtext_ctl::morph_ctx::insert_node(node* el, int index, node* node) {
      insert_node::exec(v, rt, op, el->cast<element>(), index, node);
      return false;   // don't do def processing
    }
    
    bool richtext_ctl::morph_ctx::replace_node(node* old_node, node* by_node)
    { 
      int index = int(old_node->node_index);
      helement p = old_node->parent;
      delete_node::exec(v, rt, op, old_node);
      insert_node::exec(v, rt, op, p, index, by_node);
      return false;   // don't do def processing
    }

    bool richtext_ctl::morph_ctx::reposition_node(node* el, int index, node* node) { 
      int oldindex = int(node->node_index);
      helement p = el->cast<element>();
      hnode n = node;
      if (oldindex < index)
        --index;
      delete_node::exec(v, rt, op, n);
      insert_node::exec(v, rt, op, p, index, n);
      return false;   // don't do def processing
    }

    bool richtext_ctl::morph_ctx::update_atts(node* of, const html::attribute_bag& from) {
      attributes_changed::record(v, rt, op, of->cast<element>(), of->cast<element>()->atts, from);
      return true;   // DO def processing!
    }

    bool richtext_ctl::merge_html(view &v, element *self, const string &url, bytes html, const string &encoding, morph_options* pmr) {
      richtext_ctl *_this = static_cast<richtext_ctl *>(this);
      _this->clear_comp_chars(v);

      // parsed bookmark could be still in new document - move it to the same
      // location in new document
      auto move_bookmark = [](bookmark &bm, node *nn, node *on) {
        auto is_inside = [](node *cont, node *n) -> bool {
          while (n) {
            if (n == cont) return true;
            n = n->parent;
          }
          return false;
        };
        if (!is_inside(nn, bm.node)) return;
        if (bm.node == nn)
          bm.node = on;
        else {
          uint      cnt = 0;
          each_node en(nn);
          for (hnode t; en(t); ++cnt) {
            if (t == bm.node) break;
          }
          uint      cnn = 0;
          each_node eo(on);
          for (hnode t; eo(t); ++cnn)
            if (cnn == cnt) {
              bm.node = t;
              break;
            }
        }
      };

      handle<action> op = this->transact_group ? this->transact_group : new range_action(_this, WCHARS("merge html"));

      try {

        bool focus_was_here = v.get_focus_element() &&
                              v.get_focus_element()->belongs_to(self, true);

        _awaiting = 0;

        select(v, bookmark());

        if (html.length == 0) return true;

        handle<document> d;

        if (self->first_element() && self->first_element()->is_document())
          d = self->first_element()->cast<document>();

        handle<document> nd    = new document(url);
        nd->flags.is_synthetic = true;
        nd->pview(&v);

        istream m(html, url, v.debug_mode());
        if (encoding.is_defined()) m.set_encoding(encoding);

        bookmark sel_start, sel_end;
        parse_html(v, m, nd, &sel_start, &sel_end);

        if (d && pmr) {
          document_context dc(d);
          html::morph::exec(dc, d, nd, pmr);
          if (sel_start.valid()) move_bookmark(sel_start, nd, d);
          if (sel_end.valid()) move_bookmark(sel_end, nd, d);
        } if (d) {
          morph_ctx ctx(v, this, op);
          document_context dc(d);
          html::morph::exec(dc, d, nd, &ctx);
          if (sel_start.valid()) move_bookmark(sel_start, nd, d);
          if (sel_end.valid()) move_bookmark(sel_end, nd, d);
        }
        else {
          //merge(self, nd, mut);
          self->clear();
          nd->remove(false);
          self->append(nd);
          if (sel_start.valid()) move_bookmark(sel_start, nd, self);
          if (sel_end.valid()) move_bookmark(sel_end, nd, self);
          if (self->first_element() && self->first_element()->is_document())
            d = self->first_element()->cast<document>();
        }

        if (d) {
          if (check_cannonic_document_structure(v, self, op))
            sel_start = sel_end = bookmark();
          d->setup_layout(v);
          d->operational = true;
          self->check_layout(v);
          size sz = self->dim();
          if (!sz.empty()) {
            d->measure(v, sz);
            v.refresh(self);
          }
          if (d->num_resources_requested == 0) v.on_document_complete(d);
        }

        _body = find_first(v, d, WCHARS("body"));

        // d->commit_measure(v);
        rq_spell_check(v);

        if (sel_start.valid() && sel_end.valid())
        {
          if (sel_start > sel_end)
            swap(sel_start, sel_end);
          if (!sel_start.at_caret_pos(v) && !sel_start.at_block_element_start(v))
            this->advance(v, sel_start, ADVANCE_NEXT_CARET);
          if (!sel_end.at_caret_pos(v) && !sel_end.at_block_element_end(v))
            this->advance(v, sel_end, ADVANCE_PREV_CARET);
          this->select(v, sel_start, sel_end);
        }
        else if (sel_start.valid())
          this->select(v, sel_start);
        else if (sel_end.valid())
          this->select(v, sel_end);

        if (focus_was_here) v.set_focus(self, BY_CODE);

        if (!this->transact_group && op->chain) // had changes
          push(v, op);

        return true;
      } catch (const tool::exception &) { op->undo(v, _this); }
      return false;
    }

  } // namespace behavior
} // namespace html
