//|
//|
//| Copyright (c) 2001-2010
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| graphics primitives
//|
//|

#ifndef __gool_h__
#define __gool_h__

#include "config.h"
#include "tool/tool.h"

#include "gool-types.h"
#include "gool-geometry.h"
#include "gool-affine.h"
#include "gool-graphics.h"
#include "gool-image.h"
#include "gool-image-filters.h"
#include "gool-theme.h"
#include "gool-figures.h"
#include "gool-icon.h"
#include <algorithm>

#if defined(USE_SKIA)
  #include "xgl/xgl-ref.h"
  #include "xgl/skia/include/core/SkTypeface.h"
#endif

struct window_params;

namespace html {
  class view;
  class print_view;
} // namespace html

namespace gool {

  enum WINDOW_STATE {
    WINDOW_STATE_NA = 0, // not a window?
    WINDOW_SHOWN    = 1,
    WINDOW_MINIMIZED,
    WINDOW_MAXIMIZED,
    WINDOW_HIDDEN,
    WINDOW_FULL_SCREEN,
  };

  enum WINDOW_TYPE {
    UNDEFINED_WINDOW_TYPE,
    TOOLTIP_WINDOW,
    POPUP_WINDOW,
    TOOL_WINDOW,
    CHILD_WINDOW,
    FRAME_WINDOW,
    DIALOG_WINDOW,
    PRINT_VIEW,
  };

  struct interval_node;

  class application : public resource {
    handle<font> _system_font;

  public:
    bool normal_shutdown = true;
    //interval_node *font_family_range_map; // interval tree root, holds unicode
    //                                      // ranges per family

    static GRAPHICS_CAPS requested_gfx_layer; // 0 - GDI+/CoreGraphics, 1-d2d-warp, 2-d2d-hw

    application();
    virtual ~application() {}

#if 0 && defined(DEBUG)
    virtual long add_ref() { 
      long r = resource::add_ref();
      dbg_printf("app ++ %d\n", r);
      if (r == 3)
        r = r;
      return r;
    }
    virtual long release() {
      long r = resource::release();
      dbg_printf("app -- %d\n", r);
      return r;
    }
#endif // DEBUG

    virtual bool is_valid() = 0;
    virtual bool mirrored_bitmaps() const { return false; }

    virtual font *create_font(const ustring &name, float size,
                              uint weight = 400, bool italic = false,
                              FONT_RENDERING mode = FR_VECTOR) = 0;
    virtual void  clear_font_cache()                           = 0;

    virtual bool font_families_for_char_code(tool::wchars lang_id, WRITING_SCRIPT script, uint char_code, array<ustring>& out) { return false; }

    virtual tool::ustring
    get_supported_font_family(const tool::ustring &family_name_list) = 0;

    virtual bool install_font(const tool::ustring &name, int weight,
                              bool italic, tool::bytes data)              = 0;
    virtual bool get_used_font(tool::handle<gool::font> &pf,
                               tool::wchars              family_name_list,
                               const gool::font *proto, tool::wchars lang_id,
                               WRITING_SCRIPT script, uint char_code = 0) = 0;
    virtual void get_system_font(/*inout*/ ustring &name, /*out*/ int &size,
                                 /*out*/ uint &weight,
                                 /*out*/ bool &italic)                    = 0;
    font *       get_system_font();

    // list all installed font families:
    virtual bool
    each_font_family(function<bool(wchars name)>  cb,
                     function<void(ucode, ucode)> cb_ranges_or_null) {
      return false;
    }

    virtual path *create_path() = 0;

    // returns application instance that is compatible with printing
    virtual application *printing_app() { return this; }
    virtual handle<print_target>
    select_printer(html::view *ui_parent  = nullptr,
                   ustring     printer_id = ustring()) {
      return nullptr;
    } /*if ui_parent == null then it returns default printer params*/

    virtual array<printer_info> get_printer_list() {
      return array<printer_info>();
    }

    virtual void final_release() override {
      resource::final_release();
      clear_font_cache();
      clear_bitmap_cache();
      //clear_font_family_cache();
    }

    virtual graphics *create_bitmap_graphics(gool::graphics *proto, gool::bitmap *bmp, gool::argb initc) = 0;
    virtual graphics * create_bitmap_bits_graphics(gool::bitmap *pb, gool::argb initc, bool high_quality = false) = 0;

      virtual handle<text_layout> create_text_layout(tool::wchars text);

#ifdef WIC_SUPPORT
    virtual IWICImagingFactory *wic_factory() const = 0;
#endif

    virtual GRAPHICS_CAPS graphics_caps() const { return UNKNOWN_GRAPHICS; }

    virtual bool supports_filters() const {
      return false;
    } // true if the platform is capable of filters
    virtual bool supports_printing() const {
      return false;
    } // true if the platform is capable of printing and so create_print_view()
      // returns something

    virtual handle<html::view> create_frame(const window_params &params)  = 0;
    virtual handle<html::view> create_dialog(const window_params &params) = 0;

    virtual tool::bytes get_resource(const wchar *path);

    virtual void clear_bitmap_cache();

    virtual html::print_view* create_print_processor(const window_params& params) {
      return nullptr;
    }

    static mutex guard;

    handle<bitmap> bitmap_list_head;

#if defined(USE_D2D)
    typedef d2d::asset<IDWriteFontCollection> font_collection_t;
#elif defined(USE_CGX)
    typedef CTFontRef font_collection_t;
#else
    typedef void *font_collection_t;
#endif

    struct memory_font_variant {
      int               weight;
      bool              italic;
      font_collection_t coll;
#if defined(USE_SKIA)
      xgl::ref<SkTypeface> typeface;
#endif
      array<byte> data;

      bool operator==(const memory_font_variant &rs) {
        return weight == rs.weight && italic == rs.italic;
      }
      bool operator!=(const memory_font_variant &rs) {
        return weight != rs.weight || italic != rs.italic;
      }
    };

    struct memory_font_family : public resource {
      array<memory_font_variant> variants;
    };

  protected:
    virtual bool init_font(gool::font *pf) = 0;

    hash_table<tool::ustring, handle<memory_font_family>> memory_fonts;

    virtual bool construct_font_collection_for(memory_font_variant &fv) {
      return false;
    };

    bool install_memory_font(const tool::ustring &family, int weight,
                             bool italic, tool::bytes data) {
      handle<memory_font_family> fd;

      if (!memory_fonts.find(family, fd)) {
        fd = new memory_font_family();
        memory_fonts[family] = fd;
      }

      memory_font_variant fv  = {weight, italic, nullptr};
      int                 idx = fd->variants.get_index(fv);

      if (idx >= 0 && fd->variants[idx].coll) return true;

      fv.data = data;

      if (!construct_font_collection_for(fv)) return false;

      fd->variants.push(fv);
      return true;
    }

  public:
    memory_font_variant *
    closest_memory_font_collection(const tool::ustring &family, int weight,
                                   bool italic) {
      handle<memory_font_family> fd;

      if (!memory_fonts.find(family, fd)) return nullptr;

      auto distance = [=](const memory_font_variant &fv) -> int {
        int d = abs(int(italic) - int(fv.italic)) * 10000;
        d += abs(weight - fv.weight);
        return d;
      };

      memory_font_variant *fv   = nullptr;
      int                  mind = 0;
      for (int i = 0; i < fd->variants.size(); ++i) {
        int td = distance(fd->variants[i]);
        if (!fv || td < mind) {
          mind = td;
          fv   = &fd->variants[i];
        }
      }
      return fv;
    }

    bool has_memory_family(const tool::ustring &family) const {
      handle<memory_font_family> fd;
      return memory_fonts.find(family, fd) && fd->variants.size();
    }

  protected:
    handle<tool::spell_checker_factory> _spell_checker_factory;

  public:
    tool::spell_checker *get_spell_checker(const ustring &lang) {
#ifdef SPELL_CHECK_SUPPORT
      static bool spell_checker_factory_requested = false;
      if (!spell_checker_factory_requested) {
        spell_checker_factory_requested = true;
        _spell_checker_factory = spell_checker_factory::create_factory();
      }
      if (_spell_checker_factory) return _spell_checker_factory->create(lang);
#endif
      return nullptr;
    }

    //void    clear_font_family_cache();
    //ustring get_family_for_ucode(ucode uc);
  };

  extern application *app();

  extern application *printing_app();

#if defined(WINDOWS)

  class window_dc {
    HWND  hwnd;
    HDC   hdc;
    HFONT hfont;

  public:
    window_dc(HWND hw) : hfont(0), hwnd(hw) { hdc = ::GetDC(hwnd); }
    ~window_dc() {
      if (hfont) ::SelectObject(hdc, hfont);
      ::ReleaseDC(hwnd, hdc);
    }
         operator HDC() { return hdc; }
    void select(HFONT hf) {
      HFONT pf = (HFONT)SelectObject(hdc, hf);
      if (!hfont) hfont = pf;
    }
  };

  // dc used for measurements
  class screen_dc {
    HDC   hdc;
    HFONT hfont;

  public:
    screen_dc() : hfont(0) { hdc = ::CreateCompatibleDC(nullptr); }
    ~screen_dc() {
      if (hfont) ::SelectObject(hdc, hfont);
      ::DeleteDC(hdc);
    }
         operator HDC() { return hdc; }
    void select(HFONT hf) {
      HFONT pf = (HFONT)SelectObject(hdc, hf);
      if (!hfont) hfont = pf;
    }
  };
#endif

  inline ustring system_font_name(wchars name = WCHARS("system")) {
    static ustring sys_name;
    if (sys_name.is_defined()) return sys_name;
    int     size;
    uint    weight;
    bool    italic;
    app()->get_system_font(sys_name, size, weight, italic);
    return sys_name;
  }


} // namespace gool

#if defined(WINDOWS)
  #include "ports/windows_specifics.h"
#elif defined(OSX)
  #include "ports/macosx_specifics.h"
#elif defined(LINUX)
  #include "ports/linux_specifics.h"
#endif


#endif
