#ifndef __gool_types_h__
#define __gool_types_h__

#include "tool/tool.h"
#include "gool-geometry.h"
#include "gool-affine.h"


namespace gool {
  using namespace tool;

  extern bool treat_px_as_dip;

  struct resolution_provider {
    virtual size pixels_per_inch() = 0;
    virtual void pixels_per_inch(size sz) = 0;
    virtual void reset_resolution() = 0;
    virtual bool px_as_dip() const { return treat_px_as_dip; }

    sizef pixels_per_dip(sizef sz) {
      /*size res;
      size  ppi = pixels_per_inch();
      res.x = (int)roundf(sz.x * ppi.x / 96);
      res.y = (int)roundf(sz.y * ppi.y / 96);
      return res;*/
      sizef res;
      sizef ppi = pixels_per_inch();
      res.x = (sz.x * ppi.x / 96);
      res.y = (sz.y * ppi.y / 96);
      return res;
    }

#if 0
    size pixels_per_dip(size sz) {
      /*size res;
      size ppi = pixels_per_inch();
      res.x = max(1, muldiv(abs(sz.x), ppi.x, 96)) * (sz.x < 0 ? -1 : 1); // correct rounding of negative argX
      res.y = max(1, muldiv(abs(sz.y), ppi.y, 96)) * (sz.y < 0 ? -1 : 1); // correct rounding of negative argY
      return res;*/
      size  res;
      size  ppi = pixels_per_inch();
      res.x = (int)roundf(sz.x * ppi.x / 96.0f);
      res.y = (int)roundf(sz.y * ppi.y / 96.0f);
      return res;
    }

    int pixels_per_dip(int sz) {
      return pixels_per_dip(size(sz, 0)).x;
    }
#endif

    float pixels_per_dip(float sz) {
      return pixels_per_dip(sizef(sz, 0)).x;
    }


    point dip_to_ppx(pointf pt) {
      size res = pixels_per_dip(pt);
      return point(res.x,res.y);
    }

    size dip_to_ppx(sizef pt) {
      return pixels_per_dip(pt);
    }

    int dip_to_ppx(float w) {
      return dip_to_ppx(sizef(w,w)).x;
    }


    sizef ppx_to_dip(sizef sz) {
      //sizef ppd = pixels_per_dip(sizef(1, 1));
      //sizef res;
      //res.x = sz.x / ppd.x;
      //res.y = sz.y / ppd.y;
      sizef res;
      sizef  ppi = pixels_per_inch();
      res.x = (sz.x * 96 / ppi.x);
      res.y = (sz.y * 96 / ppi.y);
      return res;
    }

    sizef ppx_to_dip(size sz) {
      //size ppd = pixels_per_dip(size(1, 1));
      sizef res;
      sizef  ppi = pixels_per_inch();
      res.x = (sz.x * 96.0f / ppi.x );
      res.y = (sz.y * 96.0f / ppi.y );
      return res;
    }

    float ppx_to_dip(int sz) {
      sizef  ppi = pixels_per_inch();
      return (sz * 96.0f / ppi.x);
    }


    rectf ppx_to_dip(rectf r) {
      //sizef ppd = pixels_per_dip(sizef(1, 1));
      sizef res;
      sizef  ppi = pixels_per_inch();
      r.s.x = (r.s.x * 96 / ppi.x);
      r.s.y = (r.s.y * 96 / ppi.y);
      r.e.x = (r.e.x * 96 / ppi.x);
      r.e.y = (r.e.y * 96 / ppi.y);
      return r;
    }

    sizef ppx_to_px(sizef sz) {
      return px_as_dip() ? ppx_to_dip(sz) : sz;
    }

    sizef ppx_to_px(size sz) {
      return px_as_dip() ? ppx_to_dip(sz) : sizef(sz);
    }
 

    // on Windows and Linux pixels_per_inch === physical_pixels_per_inch 
    // on MacOS all screen measurments are in logical pixels (actually dips)
    // so physical_pixels_per_inch is redefined there
    virtual size physical_pixels_per_inch() { return pixels_per_inch(); }
    virtual size physical_pixels_per_dip(size sz) {
      size res;
      size ppi = physical_pixels_per_inch();
      res.x = muldiv(abs(sz.x), ppi.x, 96) *
        (sz.x < 0 ? -1 : 1); // correct rounding of negative argX
      res.y = muldiv(abs(sz.y), ppi.y, 96) *
        (sz.y < 0 ? -1 : 1); // correct rounding of negative argY
      return res;
    }

    // scaling factor - layout pixels to physical backend pixels
    // this makes sense only on MacOS where layout pixels are actually dips
    // Mac/Retina this returns size(2,2)
    size pixels_scaling() {
#if defined(OSX)
      size ps = physical_pixels_per_inch();
      size pn = pixels_per_inch();
      return size(ps.x / pn.x, ps.y / pn.y);
#else
      return size(1,1);
#endif
    }

    // metrics
    virtual int get_window_metrics(tool::value::length_special_values what);

    // unknown dimension 
    virtual size dimension() const { return size(300, 150); }

    static resolution_provider &desktop();
  };

  struct argb;

  typedef uint32 color; // t8b8g8r8 color packed to int, note it uses
                        // transparency rather than opacity so 0x000000FF is
                        // fully opaque red and 0x7F0000FF is semi-transparent
                        // (50%) red

  typedef void *ghandle;

  inline byte r(color c) { return byte(c & 0xff); }
  inline byte g(color c) { return byte((c >> 8) & 0xff); }
  inline byte b(color c) { return byte((c >> 16) & 0xff); }
  inline byte a(color c) { return byte((c >> 24) & 0xff); }

  inline color rgba(uint r, uint g, uint b, uint a) {
    return ((a & 0xff) << 24) | ((b & 0xff) << 16) | ((g & 0xff) << 8) |
           (r & 0xff);
  }
  inline color rgba(uint r, uint g, uint b) {
    return ((b & 0xff) << 16) | ((g & 0xff) << 8) | (r & 0xff);
  }

  // static table "string" -> uint
  namespace name 
  {
    typedef uint_v symbol_t;

    symbol_t symbol(chars str); // intern it and return id. if not found returns undefined;
    symbol_t symbol(const string &str); // intern it and return id. if not found
                                        // returns undefined;
    string symbol_name(symbol_t sym);

  } // namespace name

#pragma warning(push)
#pragma warning(disable : 4480) // warning C4480: nonstandard extension used:
                                // specifying underlying type for enum
                                // 'gool::SYSTEM_COLORS'

  enum SPECIAL_COLORS {
    NO_COLOR = 0, // a.k.a. NULL
    INHERIT_COLOR = 1, // inherit value
    CURRENT_COLOR = 3, // currentcolor value
  };

#define SYSCOL(c) (0xFF000000 | c)
#define SYS_COLOR_IDX(c) (0xFF000000 | c)

  enum SYSTEM_COLORS {

    SC_WINDOW              = 0xFF000005,
    SC_WINDOW_TEXT         = 0xFF000008,
    SC_DIALOG_BRIGHT       = 0xFF000014,
    SC_DIALOG_LIGHT        = 0xFF000016,
    SC_DIALOG_SURFACE      = 0xFF00000F,
    SC_DIALOG_SHADOW       = 0xFF000010,
    SC_DIALOG_DKSHADOW     = 0xFF000015,
    SC_DIALOG_TEXT         = 0xFF000012,
    SC_DIALOG_LIGHTSHADOW  = 0xFF000020,

    SC_SELECTION           = 0xFF00000D,
    SC_SELECTION_TEXT      = 0xFF00000E,
    SC_MENU_SELECTION      = 0xFF000022,
    SC_MENU_SELECTION_TEXT = 0xFF000023,
    SC_WORKSPACE           = 0xFF00000C,
    SC_CAPTION             = 0xFF00001C,
    SC_CAPTION_TEXT        = 0xFF000013,
    SC_ACTIVE_CAPTION      = 0xFF000002,
    SC_ACTIVE_CAPTION_TEXT = 0xFF000009,
    SC_INFO                = 0xFF000018,
    SC_INFO_TEXT           = 0xFF000017,
    SC_SCROLLBAR           = 0xFF00001F,
    SC_DESKTOP             = 0xFF000001,
    SC_GRAY_TEXT           = 0xFF000011,
    SC_WINDOW_BORDER       = 0xFF000024,
    SC_BORDER              = 0xFF000025,
    SC_ACCENT              = 0xFF000026,
    SC_ACCENT_TEXT         = 0xFF000027,
    SC_HYPERLINK           = 0x00007fff,
  };

#pragma warning(pop)

  // bool is_sys_color(color cs);

  // translate color to RGBA
  color xcolor(SYSTEM_COLORS c);

  typedef argb (*get_current_system_color_t)(SYSTEM_COLORS sc);

  extern get_current_system_color_t  get_current_system_color;

  // medium color of two
  inline color mcolor(color c1, color c2) {
    return rgba((r(c1) + r(c2)) / 2, (g(c1) + g(c2)) / 2, (b(c1) + b(c2)) / 2);
  }

  inline color brighter(color c) {
    uint cc = c;
    return rgba((r(cc) + 255) / 2, (g(cc) + 255) / 2, (b(cc) + 255) / 2);
  }

  inline color darker(color c) {
    uint cc = c;
    return rgba(r(cc) / 2, g(cc) / 2, b(cc) / 2);
  }

  /*inline bool is_transparent( color c ) { return a(c) == 0xFF; }
  inline bool no_color(color c)
  {
    if(c == NO_COLOR || c == TRANSPARENT_COLOR)
      return true;
    return is_transparent(xcolor(c));
  }
  */

  inline bool is_semi_transparent(color c) {
    uint al = a(c);
    return al > 0 && al <= 0xFF;
  }

    // parse color, formats supported:
    //  #fff
    //  #fafafa
    //  rgb(i|%,i|%,i|%)

#pragma pack(push, 1)
  struct rgb;
  struct hsv;
  struct hsl;
  struct cmyk;

  INLINE uint uround(float v) { return uint(v + 0.5f); }

  struct argb {
    typedef byte channel_t;

    channel_t blue;
    channel_t green;
    channel_t red;
    channel_t alfa;

    argb() {
      blue  = 0;
      red   = 0;
      green = 0;
      alfa  = 255;
    }
    argb(rgb c, channel_t alfa = 255);
    argb(hsv v);
    argb(color cf) { operator=(cf); }
    argb(SYSTEM_COLORS cf);
    //argb(const value& v) :argb(color_v(v).to_argb()) {}
    argb(int r, int g, int b, int a = 255) {
      blue  = b & 0xff;
      red   = r & 0xff;
      green = g & 0xff;
      alfa  = a & 0xff;
    }
    argb &operator=(color cf) {
      alfa  = 255 - a(cf);
      blue  = b(cf);
      red   = r(cf);
      green = g(cf);
      return *this;
    }

    color to_color() const { return rgba(red, green, blue, 255 - alfa); }
          operator color() const { return to_color(); }
    value to_value() const { return value::make_packed_color(to_color()); }

    void set(int r, int g, int b, int a = 0) {
      blue  = b & 0xff;
      red   = r & 0xff;
      green = g & 0xff;
      alfa  = a & 0xff;
    }

    bool operator==(const argb &rs) const {
      return blue == rs.blue && green == rs.green && red == rs.red &&
             alfa == rs.alfa;
    }
    bool operator!=(const argb &rs) const {
      return blue != rs.blue || green != rs.green || red != rs.red ||
             alfa != rs.alfa;
    }

    // INLINE void premultiply_inv() {  alfa = 255 - alfa; uint a = alfa + 1;
    // blue = channel_t((blue * a) >> 8); green = channel_t((green * a) >> 8);
    // red = channel_t((red * a) >> 8); }  INLINE argb premultiply() const {  argb
    // t; uint a = 1 + alfa; t.alfa = alfa; t.blue = channel_t((blue * a) >> 8);
    // t.green = channel_t((green * a) >> 8); t.red = channel_t((red * a) >> 8);
    // return t; }

    INLINE void premultiply_inv() {
      alfa   = 255 - alfa;
      uint a = alfa;
      blue   = channel_t((blue * a) / 255);
      green  = channel_t((green * a) / 255);
      red    = channel_t((red * a) / 255);
    }
    INLINE argb premultiply() const {
      argb t;
      uint a  = alfa;
      t.alfa  = alfa;
      t.blue  = channel_t((blue * a) / 255);
      t.green = channel_t((green * a) / 255);
      t.red   = channel_t((red * a) / 255);
      return t;
    }

    bool        is_no_color() const { return *this == no_color(); }
    static argb no_color() { return argb(0, 0, 0, 0); }
    static argb no_color_white() { return argb(255, 255, 255, 0); }

    bool is_opaque() const { return alfa == 255; }
    bool is_transparent() const { return alfa == 0; }
    // normalized versions
    double n_red() const { return red / 255.0; }
    double n_green() const { return green / 255.0; }
    double n_blue() const { return blue / 255.0; }
    double n_alpha() const { return alfa / 255.0; }

    INLINE argb demultiply() const {
      argb t;
      uint a = t.alfa = alfa;
      if (a != 0) {
        uint r  = (red << 8) / a;
        t.red   = channel_t(min(255, r));
        uint g  = (green << 8) / a;
        t.green = channel_t(min(255, g));
        uint b  = (blue << 8) / a;
        t.blue  = channel_t(min(255, b));
      }
      return t;
    }
    inline uint luminance_fast() const {
      return (uint(red) + (uint(green) << 1) + uint(blue)) >> 2;
    }

    inline uint luminance() const {
      return (uint(red) * 77 + uint(green) * 151 + uint(blue) * 28) / 256;
    }

    INLINE void blend(const argb &with) // blend in premultiplied space
    {
      uint alpha = 255 - with.alfa;
      red        = channel_t((red * alpha + with.red * with.alfa) / 255);
      green      = channel_t((green * alpha + with.green * with.alfa) / 255);
      blue       = channel_t((blue * alpha + with.blue * with.alfa) / 255);
      alfa       = channel_t(255 - ((alpha * (255 - alfa)) / 255));
      // alfa  =  ((alfa * alpha) >> 8) + with.alfa;
      //*this = premultiply();
    }

    inline argb inverse() const // inverse color
    {
      uint l = luminance();
      if (l < 64 || l > 196)
        return argb(255 - red, 255 - green, 255 - blue, alfa);
      return l < 128 ? argb(255, 255, 255, alfa) : argb(0, 0, 0, alfa);
    }

    INLINE argb gradient(const argb &c, float k) const {
      argb ret;
      uint ik   = uround(k * 256);
      ret.red   = channel_t(red + (((int(c.red) - red) * ik) >> 8));
      ret.green = channel_t(green + (((int(c.green) - green) * ik) >> 8));
      ret.blue  = channel_t(blue + (((int(c.blue) - blue) * ik) >> 8));
      ret.alfa  = channel_t(alfa + (((int(c.alfa) - alfa) * ik) >> 8));
      return ret;
    }

    INLINE argb opacity(uint op_byte) const {
      argb ret = *this;
      if (op_byte < 255) ret.alfa = byte((uint(ret.alfa) * op_byte) >> 8);
      return ret;
    }

    unsigned int hash() const { return tool::hash(*((uint *)this)); }

    ustring to_string() const {
      if (alfa == 255)
        return ustring::format(W("rgb(%d,%d,%d)"), red, green, blue);
      else
        return ustring::format(W("rgba(%d,%d,%d,%d%%)"), red, green, blue,
                               (alfa * 100) / 255);
    }

    static argb morph(argb c1, argb c2, double ratio) {

      if (c1.alfa == c2.alfa) {
        argb c3;
        c3.alfa = c1.alfa;
        c3.red = channel_t(unsigned(c1.red + (c2.red - c1.red) * ratio));
        c3.green = channel_t(unsigned(c1.green + (c2.green - c1.green) * ratio));
        c3.blue = channel_t(unsigned(c1.blue + (c2.blue - c1.blue) * ratio));
        return c3;
      }
      else if (c1.red == c2.red && c1.green == c2.green && c1.blue == c2.blue) {
        argb c3 = c1;
        c3.alfa = channel_t(unsigned(c1.alfa + (c2.alfa - c1.alfa) * ratio));
        return c3;
      }
      else {
        c1 = c1.premultiply();
        c2 = c2.premultiply();
        argb c3;
        c3.alfa = channel_t(unsigned(c1.alfa + (c2.alfa - c1.alfa) * ratio));
        c3.red = channel_t(unsigned(c1.red + (c2.red - c1.red) * ratio));
        c3.green = channel_t(unsigned(c1.green + (c2.green - c1.green) * ratio));
        c3.blue = channel_t(unsigned(c1.blue + (c2.blue - c1.blue) * ratio));
        return c3.demultiply();
      }
    }

    // special no color provided value
    static argb undefined() { return argb(0xAF, 0xAF, 0xAF, 0); }
  };

  struct rgb {
    byte blue;
    byte green;
    byte red;
    rgb() {
      blue  = 0;
      red   = 0;
      green = 0;
    }
    rgb(int r, int g, int b) { set(r, g, b); }
    rgb(const argb &c) { set(c.red, c.green, c.blue); }
    rgb(hsv v);
    rgb(hsl v);
    rgb(cmyk v);
    rgb(color cf) {
      blue  = b(cf);
      red   = r(cf);
      green = g(cf);
    }
    //rgb(const value& v):rgb(color_v(v).to_rgb()) {}


    rgb &operator=(color cf) {
      blue  = b(cf);
      red   = r(cf);
      green = g(cf);
      return *this;
    }
          operator color() { return to_color(); }
    color to_color() const { return rgba(red, green, blue); }
    value to_value() const { return value::make_packed_color(to_color()); }

    void set(int r, int g, int b) {
      blue  = b & 0xff;
      red   = r & 0xff;
      green = g & 0xff;
    }
  };

  inline argb::argb(rgb c, argb::channel_t a) {
    blue  = c.blue;
    red   = c.red;
    green = c.green;
    alfa  = a;
  }

  // CMYK color space
  struct cmyk {
    byte c, m, y, k;

    cmyk(byte C = 0, byte M = 0, byte Y = 0, byte K = 0)
        : c(C), m(M), y(Y), k(K) {}
    cmyk(float C, float M, float Y, float K)
        : c(byte(C * 256)), m(byte(M * 256)), y(byte(Y * 256)),
          k(byte(K * 256)) {}
    ///////////////////////////////////////////////////////////////////////////////
    //
    // The algorithms for these routines were taken from:
    //     http://www.neuro.sfc.keio.ac.jp/~aly/polygon/info/color-space-faq.html
    //
    // RGB --> CMYK                              CMYK --> RGB
    // ---------------------------------------
    // -------------------------------------------- Black   =
    // minimum(1-Red,1-Green,1-Blue)   Red   = 1-minimum(1,Cyan*(1-Black)+Black)
    // Cyan    = (1-Red-Black)/(1-Black)         Green =
    // 1-minimum(1,Magenta*(1-Black)+Black) Magenta = (1-Green-Black)/(1-Black)
    // Blue  = 1-minimum(1,Yellow*(1-Black)+Black) Yellow  =
    // (1-Blue-Black)/(1-Black)
    //

    // RGB2CMYK
    template <typename Trgb> void set_rgb(const Trgb &rgb) {
      float R = 1.0 - (float(rgb.red) / 255.0);
      float G = 1.0 - (float(rgb.green) / 255.0);
      float B = 1.0 - (float(rgb.blue) / 255.0);

      float C, M, Y, K;
      if (R < G)
        K = R;
      else
        K = G;
      if (B < K) K = B;

      C = (R - K) / (1.0 - K);
      M = (G - K) / (1.0 - K);
      Y = (B - K) / (1.0 - K);

      C = (C * 100) + 0.5;
      M = (M * 100) + 0.5;
      Y = (Y * 100) + 0.5;
      K = (K * 100) + 0.5;

      c = (byte)C;
      m = (byte)M;
      y = (byte)Y;
      k = (byte)K;
    }

    template <typename Trgb> void get_rgb(Trgb &rgb) const {
      float C = float(c) / 255.0f;
      float M = float(m) / 255.0f;
      float Y = float(y) / 255.0f;
      float K = float(k) / 255.0f;

      float R = C * (1.0f - K) + K;
      float G = M * (1.0f - K) + K;
      float B = Y * (1.0f - K) + K;

      R = (1.0f - R) * 255.0f + 0.5f;
      G = (1.0f - G) * 255.0f + 0.5f;
      B = (1.0f - B) * 255.0f + 0.5f;

      rgb.red   = (byte)R;
      rgb.green = (byte)G;
      rgb.blue  = (byte)B;
    }
  };

  struct rgb565 {
    unsigned short data;

    rgb565() { data = 0; }
    rgb565(int r, int g, int b) { set(r, g, b); }
    rgb565(color cf) { set(r(cf), g(cf), b(cf)); }
    rgb565(argb cf) { set(cf.red, cf.green, cf.blue); }
    rgb565(rgb cf) { set(cf.red, cf.green, cf.blue); }

    INLINE void set(uint r, uint g, uint b) {
      data = (unsigned short)(((r & 0xf8) << 8) | ((g & 0xfc) << 3) | (b >> 3));
    }

    INLINE byte red() const { return byte((data >> 8) & 0xf8); }
    INLINE byte green() const { return byte((data & 0x7e0) >> 3); }
    INLINE byte blue() const { return byte((data & 31) << 3); }

    rgb565 &operator=(color cf) {
      set(r(cf), g(cf), b(cf));
      return *this;
    }
    operator color() const { return rgba(red(), green(), blue()); }
    operator argb() const { return argb(red(), green(), blue()); }
    operator rgb() const { return rgb(red(), green(), blue()); }
  };

#pragma pack(pop)

  struct hsv {
    float h; // 0..360
    float s; // 0..1
    float v; // 0..1
    hsv(float _h = 0.0f, float _s = 0.0f, float _v = 0.0f)
        : h(_h), s(_s), v(_v) {}
    hsv(const hsv &x) {
      h = x.h;
      v = x.v;
      s = x.s;
    }
    hsv(const rgb &x) { set(x.red, x.green, x.blue); }
    hsv(const argb &x) { set(x.red, x.green, x.blue); }
    void set(byte red, byte green, byte blue);
    void get(byte &red, byte &green, byte &blue);
    hsv &operator=(const hsv &x) {
      h = x.h;
      v = x.v;
      s = x.s;
      return *this;
    }
  };

  inline rgb::rgb(hsv v) : blue(0), green(0), red(0) {
    v.get(red, green, blue);
  }
  inline argb::argb(hsv v) : blue(0), green(0), red(0), alfa(255) {
    v.get(red, green, blue);
  }

  struct hsl {
    float h; // 0..360
    float s; // 0..1
    float l; // 0..1
    hsl(float _h = 0.0f, float _s = 0.0f, float _l = 0.0f)
        : h(_h), s(_s), l(_l) {}
    hsl(const hsl &x) {
      h = x.h;
      l = x.l;
      s = x.s;
    }
    hsl(const rgb &x);
    hsl(const argb &x);
    hsl &operator=(const hsl &x) {
      h = x.h;
      l = x.l;
      s = x.s;
      return *this;
    }
  };

  struct length {
    enum TYPE { UNDEFINED, PERCENT, PX, DIPX };
    TYPE  type;
    float value;
    length() : type(UNDEFINED), value(0) {}
    length(const length &r) : type(r.type), value(r.value) {}
    length &operator=(const length &r) {
      type  = r.type;
      value = r.value;
      return *this;
    }
    bool operator==(const length &r) const {
      return type == r.type && value == r.value;
    }
    bool operator!=(const length &r) const {
      return type != r.type || value != r.value;
    }
    // float pixels_width( size sz )
  };

  class graphics;

  class brush : public resource_x<brush> {
  public:
    enum TYPE {
      HOLLOW,
      SOLID,
      LINEAR,
      RADIAL,
      TILE,
      EXPAND,
    };
    virtual ~brush() {}

    virtual TYPE     type() const { return HOLLOW; }
    virtual unsigned hash() const { return type(); }

    virtual void set_fill_to(graphics *pg) const;
    virtual void set_stroke_to(graphics *pg) const;

    virtual void         transform(const affine_mtx_f &m) {}
    virtual affine_mtx_f transform() const { return affine_mtx_f(); }

    // virtual bool eq(const brush& b) const { return true; }
    virtual bool is_gradient() const { return false; }
  };

  class solid_brush : public brush {
  public:
    argb color;
    solid_brush(argb c) : color(c) {}
    virtual ~solid_brush() {}

    virtual TYPE     type() const { return SOLID; }
    virtual unsigned hash() const { return type() + color.to_color(); }

    virtual void set_fill_to(graphics *pg) const;
    virtual void set_stroke_to(graphics *pg) const;
  };

  struct color_stop {
    argb         color;
    float_v      position;
    unsigned int hash() const {
      return tool::hash((uint)color) ^ tool::hash((float)position);
    }
    bool operator==(const color_stop &rs) const {
      return color == rs.color && position == rs.position;
    }
    bool operator!=(const color_stop &rs) const {
      return color != rs.color || position != rs.position;
    }
  };


  class gradient_brush : public brush {
  public:
    virtual ~gradient_brush() {}

    bool add_stop(float_v position, argb color) {
      color_stop cs;
      cs.position = position;
      cs.color    = color;
      _stops.push(cs);
      return true;
    };

    void              stops(slice<color_stop> cs) { _stops = cs; }
    slice<color_stop> stops() const { return _stops(); }

    virtual bool is_gradient() const { return true; }

    void generate_stops(argb c1, argb c2, float profile);

    bool is_none() const { return _stops.length() == 0; }

    bool normalize_stops() {
      if (is_none()) return false;
      float   miv   = 0.f;
      float   mav   = 1.f;
      index_t first = 0;
      index_t last  = _stops.size() - 1;
      for (index_t n = 0; n < _stops.size();) {
        if (_stops[n].position.is_defined()) {
          _stops[n].position = max(miv, _stops[n].position);
          miv                = _stops[n].position;
          first              = n++;
          continue;
        }
        mav  = 1.f;
        last = n;
        for (++n; n < _stops.size(); ++n)
          if (_stops[n].position.is_defined()) {
            mav  = max(miv, _stops[n].position);
            last = n;
            break;
          } else {
            mav  = 1.f;
            last = n;
          }
        index_t i              = 0;
        _stops[first].position = miv;
        for (index_t k = first + 1; k < last; ++k)
          _stops[k].position = miv + ((mav - miv) * ++i) / (last - first);
        _stops[last].position = mav;
      }
      return true;
    }

    virtual void         transform(const affine_mtx_f &m) { _mx = m; }
    virtual affine_mtx_f transform() const { return _mx; }

  protected:
    affine_mtx_f      _mx;
    array<color_stop> _stops;
    mutable uint_v    _hash;
    gradient_brush() {}
  };

  class linear_brush : public gradient_brush {
  public:
    pointf start; // in 0.0 .. 1.0 range
    pointf end;   // in 0.0 .. 1.0 range
    // float_v angle;
    linear_brush(pointf start_ = pointf(0, 0), pointf end_ = pointf(0, 0))
        : start(start_), end(end_) {}
    virtual ~linear_brush() {}
    virtual TYPE type() const { return LINEAR; }

    linear_brush &operator=(const linear_brush &rs) {
      start  = rs.start;
      end    = rs.end;
      _mx    = rs._mx;
      _stops = rs._stops;
      return *this;
    }

    virtual unsigned hash() const {
      if (_hash.is_defined()) return _hash;
      unsigned h = type();
      hash_combine(h, tool::hash(stops()));
      hash_combine(h, tool::hash(start.x));
      hash_combine(h, tool::hash(start.y));
      hash_combine(h, tool::hash(end.x));
      hash_combine(h, tool::hash(end.y));
      _hash = h;
      return h;
    }
    virtual bool operator==(const brush &r) const {
      if (r.type() != LINEAR) return false;
      const linear_brush *pb = static_cast<const linear_brush *>(&r);
      return start == pb->start && end == pb->end && stops() == pb->stops();
    }

    virtual void set_fill_to(graphics *pg) const;
    virtual void set_stroke_to(graphics *pg) const;
  };

  class radial_brush : public gradient_brush {
  public:
    pointf center; // center
    sizef  radius; // radius
    sizef  focus;  // focus offset from center
    radial_brush(pointf c = pointf(0, 0), sizef r = pointf(0, 0),
                 sizef f = sizef(0, 0))
        : center(c), radius(r), focus(f) {}

    radial_brush &operator=(const radial_brush &rs) {
      center = rs.center;
      radius = rs.radius;
      focus  = rs.focus;
      _mx    = rs._mx;
      _stops = rs._stops;
      return *this;
    }

    virtual ~radial_brush() {}
    virtual TYPE     type() const { return RADIAL; }
    virtual unsigned hash() const {
      if (_hash.is_defined()) return _hash;
      unsigned h = type();
      hash_combine(h, tool::hash(stops()));
      hash_combine(h, tool::hash(radius.x));
      hash_combine(h, tool::hash(radius.y));
      hash_combine(h, tool::hash(focus.x));
      hash_combine(h, tool::hash(focus.y));
      hash_combine(h, tool::hash(center.x));
      hash_combine(h, tool::hash(center.y));
      _hash = h;
      return h;
    }
    virtual bool operator==(const brush &rb) const {
      if (rb.type() != RADIAL) return false;
      const radial_brush *pb = static_cast<const radial_brush *>(&rb);
      return radius == pb->radius && focus == pb->focus &&
             center == pb->center && stops() == pb->stops();
    }

    virtual void set_fill_to(graphics *pg) const;
    virtual void set_stroke_to(graphics *pg) const;
  };

  /*struct color_v;
  struct color_name_resolver {
    virtual bool resolve_color(uint attr_sym, color_v& val) const = 0;
    virtual bool resolve_current_color(color_v& val) const = 0;
  };

  bool reduce_color_function(function_value *vf, color_v &c, const color_name_resolver* pr);
  */

  struct color_v {
    //uint64  sc;
    //int   sc; // SYSTEM_COLORS
    //color ac;

    enum TYPE {
      RGBT  = 0, // name (any), data -> rgbt (32 bit for now) rgba with alfa inversed 
      //NAME  = 1, // name -> attr::symbol , data -> rgba (fallback)
      //FUNC  = 2, // data -> function_value* 
      SYST  = 1, // name -> SYSTEM_COLOR enum
    };

    /*union {
      struct{
        uint   type : 2;
        uint   name : 14;
        uint64 data : 48;
      } s;
      uint64   bits;
    };*/

    uint64   bits;

    void _pack(TYPE t, uint n) { bits = (uint64(t & 0x1) << 63) | n; }

    TYPE            _type() const { return TYPE(uint(bits >> 63) & 0x1); }
    uint            _syst() const { return uint(bits & 0xFFFFFFFFull); }
    color           _color() const { return (color)(bits & 0xFFFFFFFFull); }

    explicit color_v(NO_INIT) {}

    color_v(SPECIAL_COLORS n = NO_COLOR)    { _pack(SYST,n); }
    color_v(color c)                        { _pack(RGBT, c); }
    color_v(argb c)                         { _pack(RGBT, c.to_color());  }
    color_v(rgb c)                          { _pack(RGBT, c.to_color());  }
    color_v(SYSTEM_COLORS n)                { _pack(SYST,n & 0xFF);  }
    color_v(uint64 packed_color) : bits(packed_color) { ; }
    color_v(const color_v &iv) : bits(iv.bits) { }
    color_v(const value &v): bits(0)
    {
      set(v);
      assert(!is_current());
    }

    bool set(const value &v) {
      if (v.is_undefined())
        _pack(SYST, NO_COLOR);
      else if (v.is_inherit())
        _pack(SYST, INHERIT_COLOR);
      else if (v.is_color())
        bits = v.get_packed_color();
      else {
        _pack(SYST, NO_COLOR);
        return false;
      }
      return true;
    }

    argb  to_argb(/*const color_name_resolver* pr = nullptr*/) const;
    rgb   to_rgb(/*const color_name_resolver* pr = nullptr*/) const;
    color to_color(/*const color_name_resolver* pr = nullptr*/) const { return to_argb().to_color(); }
    value to_value() const {
      if (is_undefined()) return value();
      return value::make_packed_color(bits);
    }

    operator value() const { return to_value(); }

    color_v &operator=(const color_v &nv) {
      if (&nv != this) {
        clear();
        bits = nv.bits;
      }
      return *this;
    }

    color_v &operator=(const value &v) {
      clear();
      if (v.is_inherit())
        *this = color_v::inherit_val();
      else if (v.is_none())
        *this = color_v::transparent_val();
      else if (v.is_color())
        bits = v.get_packed_color();
      return *this;
    }

    color_v &operator=(const gool::argb &nv) {
      clear();
      _pack(RGBT,nv.to_color());
      return *this;
    }
    color_v &operator=(const gool::rgb &nv) {
      clear();
      _pack(RGBT, nv.to_color());
      return *this;
    }

    color_v &operator=(const color &nv) {
      clear();
      _pack(RGBT, nv);
      return *this;
    }

    bool operator==(const color_v &rs) const {
      return bits == rs.bits;
    }
    bool operator!=(const color_v &rs) const {
      return bits != rs.bits;
    }

    static color_v null_val() { return color_v(NO_COLOR); }
    static color_v inherit_val() { return color_v(INHERIT_COLOR); }
    static color_v current_val() { return color_v(CURRENT_COLOR); }
    static color_v transparent_val() { return color_v(argb::no_color_white()); }

    bool is_undefined() const { return _type() == SYST && _syst() == NO_COLOR; }
    bool is_defined() const { return _type() != SYST || _syst() != NO_COLOR; }
    bool is_inherit() const { return _type() == SYST && _syst() == INHERIT_COLOR; }
    bool is_current() const { return _type() == SYST && _syst() == CURRENT_COLOR; }

    bool is_literal() const { return _type() == RGBT; }
    
    bool is_transparent() const {
      return is_undefined() || ( _type() == RGBT && ((_color() & 0xFF000000) == 0xFF000000));
    }

    void clear() { _pack(SYST,NO_COLOR); }

    void inherit(const color_v &v) {
      if (v.is_defined()) *this = v;
    }

    color_v val(const color_v &v1) const {
      return is_defined() ? *this : v1;
    }
    color_v val(const color_v &v1, const color_v &v2) const {
      return is_defined() ? *this
                          : (v1.is_defined() ? v1 : v2);
    }

    unsigned int hash() const { return tool::hash(bits); }

    ustring to_string() const {
      if (is_undefined()) return ustring();
      if (is_transparent())
        return WCHARS("transparent");
      argb c = _color();
      if (c.alfa == 255)
        return ustring::format(W("rgb(%d,%d,%d)"), c.red, c.green, c.blue);
      else
        return ustring::format(W("rgba(%d,%d,%d,%d%%)"), c.red, c.green, c.blue, (c.alfa * 100) / 255);
      // return ustring::format(L"#%02x%02x%02x",r(v._v),g(v._v),b(v._v));
    }
  };

  color_v parse_color(tool::chars ss);

  bool find_named_color(tool::chars name, color_v &cv);

  inline argb::argb(SYSTEM_COLORS cf) { 
    if (cf == SC_ACCENT_TEXT) {
      *this = get_current_system_color(SC_ACCENT);
      if (luminance_fast() < 128)
        *this = argb(255, 255, 255);
      else 
        *this = argb(0, 0, 0);
    } else 
      *this = get_current_system_color(cf);
  }

  struct print_target : public resource {
    ustring     device_name;
    size        device_dpi;
    size        page_paper_size;  // in millimeters
    size        page_device_size; // in device pixels
    rect        page_device_rect; // printable rect
    array<byte> device_mode;      // DEVMODE on Windows
#if defined(WINDOWS)
    HDC device_dc;
#endif

    virtual bool
    print(const ustring &                                 doc_name,
          function<bool(graphics *page_gfx, int page_no)> renderer) = 0;
  };

  struct printer_info {
    ustring     id;
    ustring     name;
    ustring     share_name;
    ustring     comment;
    ustring     location;
    tristate_v  is_default;
    array<byte> device_mode; // DEVMODE on Windows
  };

} // namespace gool

namespace tool {
  TYPE_IS_POD(gool::argb);
  TYPE_IS_POD(gool::rgb);
  TYPE_IS_POD(gool::hsv);
  TYPE_IS_POD(gool::hsl);
  TYPE_IS_POD(gool::cmyk);

} // namespace tool

#endif
