#ifndef __gool_image_h__
#define __gool_image_h__

#include "config.h"
#include "tool/tool.h"
#include "gool-types.h"
#include "gool-geometry.h"
#include "gool-image-convertor.h"


#ifdef USE_D2D
#include "d2d/d2d-primitives.h"
#include <wincodec.h>
#endif
#ifdef USE_GDI
#include "gdi+/gdi+plus.h"
#endif
#ifdef USE_CAIRO
#include "../gtk/gtk-types.h"
#endif
#ifdef OSX
#include <ApplicationServices/ApplicationServices.h>
#include "osx/osx-ref.h"
#endif

#ifdef USE_SKIA
#include "xgl/skia/include/core/SkBitmap.h"
#endif

namespace html {
  class view;
  struct document;
} // namespace html

namespace gool {
  using namespace tool;

  class image;
  class bitmap;
  class graphics;
  struct image_filter;

  enum SECTION_RENDERING_MODE {
    SRM_UNDEFINED       = 0,
    SRM_TILE            = 1,
    SRM_STRETCH         = 2, // stretch
    SRM_MIDDLE          = 3, // no-repeat, 50% 50%
    SRM_INTEGRAL_REPEAT = 4, // repeat
  };

  struct SECTION_DEFS {
    rect                   margins;
    SECTION_RENDERING_MODE top;
    SECTION_RENDERING_MODE left;
    SECTION_RENDERING_MODE right;
    SECTION_RENDERING_MODE bottom;
    SECTION_RENDERING_MODE center;
  };

  enum BLEND_MODE {
    BLEND_NORMAL,
    BLEND_MULTIPLY,
    BLEND_SCREEN,
    BLEND_OVERLAY,
    BLEND_DARKEN,
    BLEND_LIGHTEN,
    BLEND_COLOR_DODGE,
    BLEND_COLOR_BURN,
    BLEND_HARD_LIGHT,
    BLEND_SOFT_LIGHT,
    BLEND_DIFFERNCE,
    BLEND_EXCLUSION,
    BLEND_HUE,
    BLEND_SATURATION,
    BLEND_COLOR,
    BLEND_LUMINOSITY,
  };

  class image_link {
    friend class image;
    friend class graphics;
    graphics *  for_gfx;
    image *     for_img;
    image_link *next_in_graphics; // in graphics
    image_link *next_in_image;    // in graphics
  public:
    long generation; // used by bitmap::set_bits
#ifdef USE_D2D
    d2d::asset<ID2D1Bitmap> d2d_bmp;
#endif
    image_link()
        : for_gfx(0), for_img(0), next_in_graphics(0), next_in_image(0),
          generation(0) {}
    void detach_from_graphics();
    void detach_from_image();
  };

  typedef function<void(rect dst, rect src)> image_renderer_f;

  class image : public resource_x<image> {
    friend class image_link;

  public:
    enum PACKAGING {
      UNKNOWN,
      PNG,
      JPG,
      GIF,
      BMP,
      SVG,
      ICO,
      WEBP,
    };
    static handle<image> create(bytes data, const string &url,
                                html::document *pd = 0);
    static handle<image> create(
        const array<byte> &data, const string &url,
        html::document *pd = 0); // creates an image and stores the data in it

    virtual bitmap *create_compatible_bitmap(size) { return 0; }

    virtual bool is_valid() const { return true; }
    virtual bool is_transparent() const { return false; } // has alpha bits
    virtual bool is_animated(uint_ptr /*site_id*/) const { return false; }
    virtual bool is_fragment() const {
      return false;
    } // true if it is a fragment of base

    virtual image *get_fragment(const string &fragment_id) { return this; }

    virtual handle<bitmap> get_bitmap(graphics *pg, size sz) = 0;

    virtual void draw(graphics *gfx, rect dst, rect src, byte opacity);
    virtual void draw(graphics *gfx, rectf dst, rect src, byte opacity);
    virtual void expand(graphics *gfx, const rect &dst, const SECTION_DEFS &sds, rect src = rect());
    virtual bool is_solid_color(rect /*src*/, argb & /*clr*/) { return false; }

    virtual void tile(graphics *gfx, const rect &dst, const rect &src, point offset, size dstcell = size());

    size dimension() const { return dim(); }

    virtual size dim() const { return gool::size(); }
    virtual size dim(const rect & /*for_dst*/) const { return dim(); }
    virtual void drop_cache();
    virtual void clear(argb clr) { return; }

    size_t save(array<byte> &out, PACKAGING type, int compression_level);

    string get_url() const { return url; }
    void   set_url(const string &s) { url = s; }

    virtual bool set_frame_no(int no, uint_ptr site_id) { return false; }

    virtual bool is_bitmap() const { return false; }
    virtual bool is_vector() const { return false; } // pure vector image

    virtual ~image();

    virtual image *transform(const image_filter * /*ptran*/) { return this; }
    virtual image *mapped_left_to_right() { return this; }
    virtual image *mapped_top_to_right() { return this; }

    image_link *get_link_for(graphics *pg);

    virtual long generation() const { return 0; }

    string mime_type() const;

    virtual bytes       get_data() { return data(); }
    virtual array<char> get_data_url();

#ifdef WINDOWS
    HBITMAP create_win_bitmap();
#endif

#ifdef _DEBUG
    static void report_number_of_instances();
#endif

  protected:
    image() : packaging(UNKNOWN), image_links(nullptr) {
#ifdef _DEBUG
      ++image_instances;
#endif
    }

    PACKAGING   packaging;
    string      url;
    array<byte> data;
    image_link *image_links;

#ifdef _DEBUG
    static int image_instances;
#endif
  };

  class tile_brush : public brush
  {
  public:
    tile_brush(image* img) : tile(img) {}
    virtual ~tile_brush() {}
    virtual TYPE type() const { return TILE; }
    handle<image> tile;

    virtual void set_fill_to(graphics *pg) const;
    virtual void set_stroke_to(graphics *pg) const;
  };



  const char *mime_type_of(image::PACKAGING ip);

  inline string image::mime_type() const {
    return gool::mime_type_of(packaging);
  }

  bool mirrored_bitmaps();

  typedef handle<image> himage;

  class dib32;

  struct pixmap {
    argb *buf;
    size  dim;

    pixmap(argb *p = nullptr, size d = size()) : buf(p), dim(d) {}
    pixmap(slice<argb> pixels, size pixels_dim) : buf(const_cast<argb*>(pixels.start)), dim(pixels_dim) { assert(pixels_dim.x*pixels_dim.y == pixels.size()); }

    int width() const { return dim.x; }
    int height() const { return dim.y; }

    argb &pixel(int x, int y) const { return buf[x + target_y(y) * dim.x]; }
    void  pixel(int x, int y, argb c) { buf[x + target_y(y) * dim.x] = c; }

    int target_y(int y) const {
#ifdef USE_CGX
      if (mirrored_bitmaps()) return dim.y - y - 1;
#endif
      return y;
    }

    void copy_color_hspan(int x, int y, unsigned len, const argb *colors) {
      argb *p = buf + x + target_y(y) * dim.x;
      do {
        *p++ = (*colors++);
      } while (--len);
    }

    void copy_color_vspan(int x, int y, unsigned len, const argb *colors) {
      do {
        argb *p = buf + x + target_y(y++) * dim.x;
        *p      = (*colors++);
      } while (--len);
    }

    template <typename F>
    void overlay_hspan(int x, int y, const argb *colors, int len, F f) {
      argb *p = buf + x + target_y(y) * dim.x;
      do {
        f(*p++, *colors++);
      } while (--len);
    }

    template <typename F>
    void transform_hspan(int x, int y, int len, F f) {
      argb *p = buf + x + target_y(y) * dim.x;
      do {
        f(*p++);
      } while (--len);
    }


    /*template <typename F> void overlay(const pixmap &src, point to, F functor) {
      rect tr(dim);
      rect ir = tr & rect(to, src.dim);
      if (ir.empty()) return;

      range y  = ir.y();
      range sy = y - to.y;
      range x  = ir.x();
      range sx = x - to.x;

      for (int r = y.l; r <= y.h; ++r, ++sy.l)
        overlay_hspan(x.l, r, &src.pixel(sx.l, sy.l), sx.length(), functor);
    }*/

    template <typename F> void overlay(const pixmap &src, rect srcr, point to, F functor) {
      rect sr = srcr & rect(src.dim); // source clipped
      rect tr = rect(dim) & rect(to, sr.size()); // target clipped
      if (tr.empty()) return;

      range ty = tr.y();
      range sy = sr.y();
      range tx = tr.x();
      range sx = sr.x();
      int tx_length = tx.length();

      for (int dr = ty.s, sr = sy.s; dr < ty.e; ++dr, ++sr)
        overlay_hspan(tx.s, dr, &src.pixel(sx.s, sr), tx_length, functor);
    }

    template <typename F> void transform(rect tr, F functor) {
      tr &= rect(dim);
      if (tr.empty()) return;

      range y = tr.y();
      range x = tr.x();

      for (int r = y.s; r < y.e; ++r)
        transform_hspan(x.s, r, x.length(), functor);
    }

    void set(rect dst, argb c)                         { transform(dst, [c](argb& p) { p = c; }); }
    void copy(point dst, const pixmap &src, rect srcr) { overlay(src, srcr, dst, [](argb& t, const argb& c) { t = c; }); }
    void copy(point dst, const pixmap &src)            { copy(dst, src, rect(src.dim)); }
    void blend(point dst, const pixmap& src, rect srcr){ overlay(src, srcr, dst, [](argb& t, const argb& c) { t.blend(c); }); }
    void blend(point dst, const pixmap& src)           { blend(dst, src, rect(src.dim)); }

  };

  template <class PixMap>
  class transposer // wdith -> height transposer / 90deg rotator
  {
    PixMap &m_pixmap;

  public:
    //--------------------------------------------------------------------
    transposer(PixMap &pixm) : m_pixmap(pixm) {}

    //--------------------------------------------------------------------
    inline int width() const { return m_pixmap.height(); }
    inline int height() const { return m_pixmap.width(); }

    //--------------------------------------------------------------------
    inline argb &pixel(int x, int y) const { return m_pixmap.pixel(y, x); }
    inline void  pixel(int x, int y, argb c) { m_pixmap.pixel(y, x, c); }

    //--------------------------------------------------------------------
    template <typename C>
    inline void copy_color_hspan(int x, int y, unsigned len, const C *colors) {
      m_pixmap.copy_color_vspan(y, x, len, colors);
    }

    //--------------------------------------------------------------------
    template <typename C>
    inline void copy_color_vspan(int x, int y, unsigned len, const C *colors) {
      m_pixmap.copy_color_hspan(y, x, len, colors);
    }
    NONCOPYABLE(transposer)
  };

  // is a dib, device independent bitmap;
  class bitmap : public image, public l2elem<bitmap> {
    friend class image;
    friend class application;
    friend struct png_reader;
    friend struct wic_reader;

  public:

    DEFINE_TYPE_ID_DERIVED(bitmap, image)

    bitmap(size sz = size(0,0), bool transparent = true, bool with_pixel_buf = true);
    bitmap(const bitmap *src);
    bitmap(const dib32 *src, bool transparent = true);
#ifdef WINDOWS
    bitmap(HBITMAP hbmp, bool has_alpha = true);
    static bitmap *create(const dib32 &src, bool transparent);
#endif
#ifdef OSX
    static bitmap *create(CGImageRef src, bool transparent,
                          gool::size sz = gool::size());
#endif
    virtual ~bitmap() {
      drop_cache();
      l2elem<bitmap>::unlink();
    }

    virtual bool is_valid() const override { return _pixels.size() == _dim.x * _dim.y; }
    virtual bool is_transparent() const override { return _has_alpha_bits; }
    virtual handle<bitmap> get_bitmap(graphics * /*pg*/, size /*sz*/) override { return this; }

    template <typename PA> inline void foreach_pixel(PA pa) {
      argb *pp   = _pixels.head();
      argb *pend = _pixels.tail();
      for (; pp < pend; ++pp)
        pa(*pp);
    }

    inline bool draws_nothing() {
      argb *pp   = _pixels.head();
      argb *pend = _pixels.tail();
      for (; pp < pend; ++pp)
        if (pp->alfa != 0) return false;
      return true;
    }

    inline void premultiply() {
      if (_has_alpha_bits) foreach_pixel([](argb &p) { p = p.premultiply(); });
      // FOREACH(i,_pixels) { _pixels[i] = _pixels[i].premultiply(); }
      else
        foreach_pixel([](argb &p) { p.alfa = 255; });
      // FOREACH(i,_pixels) { _pixels[i].alfa = 255; }
    }

    inline void demultiply() {
      foreach_pixel([](argb &p) { p = p.demultiply(); });
    }

    virtual size dim() const override { return _dim; }

    virtual pixmap pmap() const {
      size sz = dim();
      if (_pixels.size() == sz.x * sz.y)
        return pixmap(_pixels.begin(), dim());
      else
        return pixmap();
    }

    point mapped(point pt) const { // mapped point - returns the point in image
                                   // buffer space. on OSX image buffers are
                                   // bottom to top
#ifdef USE_CGX
      if (mirrored_bitmaps()) return point(pt.x, _dim.y - pt.y - 1);
#endif
      return pt;
    }
    rect mapped(rect rc) const { // mapped point - returns the point in image
                                 // buffer space. on OSX image buffers are
                                 // bottom to top
#ifdef USE_CGX
      if (mirrored_bitmaps()) {
        rect t(mapped(rc.s), mapped(rc.e));
        std::swap(t.s.y, t.e.y);
        return t;
      }
#endif
      return rc;
    }

    slice<argb> operator[](int nrow) {
#ifdef USE_CGX
      if (mirrored_bitmaps())
        return slice<argb>(&_pixels[(_dim.y - nrow - 1) * _dim.x],
                           uint(_dim.x));

#endif
      return slice<argb>(&_pixels[nrow * _dim.x], uint(_dim.x));
    }
    argb *operator()(int nrow) {
#ifdef USE_CGX
      if (mirrored_bitmaps()) return &_pixels[(_dim.y - nrow - 1) * _dim.x];
#endif
      return &_pixels[nrow * _dim.x];
    }

    tslice<argb> target_row(int nrow) {
      if (nrow >= 0 && nrow < _dim.y && _pixels.length()) {
#ifdef USE_CGX
        if (mirrored_bitmaps())
          return tslice<argb>(&_pixels[(_dim.y - nrow - 1) * _dim.x], _dim.x);
#endif
        return tslice<argb>(&_pixels[nrow * _dim.x], _dim.x);
      }
      return tslice<argb>();
    }

    argb &pixel(int x, int y) {
      return operator()(y)[x];
    }

    virtual void drop_cache() override {
      super::drop_cache();
      transformed_cache.clear();
#ifdef USE_CGX
      cg_image = nullptr;
#endif
#ifdef USE_GDI
      gdi_bmp = nullptr;
#endif
#ifdef USE_CAIRO
      cr_surface = nullptr;
#endif
#ifdef USE_SKIA
      sk_bitmap.reset();
#endif
    }

    virtual bitmap *create_compatible_bitmap(size sz) override {
      return new bitmap(sz, is_transparent());
    }
    // void clear(argb clr);
    void set(rect dst, argb c)                          { pmap().set(dst, c);  ++_generation; }
    void copy(point dst, bitmap *src, rect srcr)        { pmap().copy(dst, src->pmap(), srcr);  ++_generation; }
    void copy(point dst, slice<argb> pixels, size pixels_dim) { pmap().copy(dst, pixmap(pixels, pixels_dim));  ++_generation; }
    void copy(point dst, bitmap *src)                   { copy(dst, src, rect(src->dim())); }
    void blend(point dst, bitmap *src, rect srcr)       { pmap().blend(dst, src->pmap(), srcr); ++_generation; }
    void blend(point dst, bitmap *src)                  { blend(dst, src, rect(src->dim())); }

    void rotate_90() {
      array<argb> buffer; buffer.length(get_bits().length);
      int n = 0;
      for (int c = 0; c < dim().x; ++c)
        for (int r = 0; r < dim().y; ++r)
          buffer[n++] = pixel(c, r);
      swap(_dim.x, _dim.y);
      swap(_pixels, buffer);
    }
    void flip_x() {
      int rows = dim().y;
      for (int r = 0; r < rows; ++r) {
        slice<argb> row = (*this)[r];
        argb *      s = const_cast<argb *>(row.start);
        argb *      l = const_cast<argb *>(row.end() - 1);
        while (s < l)
          swap(*s++, *l--);
      }
    }

    void flip_y() { top_to_bottom_inplace(); }


    template <typename F>
    void combine(point dst, bitmap *src, rect srcr, F functor);

    virtual bool is_solid_color(rect src, argb &clr) override;

    virtual image *transform(const image_filter *ptran) override;
    virtual image *mapped_left_to_right() override;
    virtual image *mapped_top_to_right() override;

    uint uid() { return _uid; }
    virtual long generation() const override { return _generation; }

    virtual bool is_bitmap() const override { return true; }

    virtual bytes get_data() override;

    bool set_bits(slice<argb> data, bool adjust = true) {
      //tool::critical_section _(gool::lock); - WRONG, must be locked by caller.
      if (data.size() != _dim.x * _dim.y) return false;
#ifdef USE_CGX
      if (adjust && mirrored_bitmaps()) {
        _pixels.size(_dim.x * _dim.y);
        for (int r = 0; r < _dim.y; ++r) {
          target_row(r).copy(data.start);
          data.prune(_dim.x);
        }
      } else
        _pixels = data;
#else
      _pixels = data;
#endif
      ++_generation;
      return true;
    }
    slice<argb> get_bits() const {
      if (_pixels.length() == 0) {
        const_cast<bitmap *>(this)->_pixels.length(
            _dim.x * _dim.y,
            is_transparent() ? argb(0, 0, 0, 0) : argb(0, 0, 0, 255));
      }
      return _pixels();
    }

    bool set_bits(bytes data, color_space_converter *csx) {
      //tool::critical_section _(gool::lock); == must be locked by caller

      assert(csx->dim() == dim());
      if (_pixels.length() == 0)
        _pixels.length(_dim.x * _dim.y, is_transparent() ? argb(0, 0, 0, 0)
                                                         : argb(0, 0, 0, 255));

      csx->convert_to_rgb32(data, _pixels.head());
#ifdef USE_CGX
      if (mirrored_bitmaps()) {
        // make it bottom to top
        array<argb> row(_dim.x);
        int         s = 0;
        int         e = _dim.y - 1;
        for (; s < e; ++s, --e) {
          row = (*this)[s];
          target_row(s).copy((*this)[e]);
          target_row(e).copy(row());
        }
      }
#endif
      ++_generation;
      return true;
    }

    bool set_bits(bytes data, size_t stride, color_space_converter *csx) {
      //tool::critical_section _(gool::lock);

      assert(csx->dim() == dim());
      if (_pixels.length() == 0)
        _pixels.length(_dim.x * _dim.y, is_transparent() ? argb(0, 0, 0, 0) : argb(0, 0, 0, 255));

      size_t row_chunk_length = csx->pixel_size() * _dim.x;
      if (row_chunk_length == 0) return false;
      if (row_chunk_length > stride) return false;
            
      for (int r = 0; r < _dim.y; ++r) {
        auto tr = target_row(r);
        bytes row_data = data(0, int(row_chunk_length));
        if (row_data.length == 0) break;
        csx->convert_to_rgb32(row_data, tr.start);
        data.prune(stride);
      }
      ++_generation;
      return true;
    }

    virtual void clear(argb clr) override {
      //tool::critical_section _(gool::lock);
      if (_pixels.size() != _dim.x * _dim.y) return;
      _pixels.set_all_to(clr.premultiply());
      ++_generation;
    }

//#ifdef USE_CGX
    void top_to_bottom_inplace() {
      tool::critical_section _(gool::lock);

      if (_pixels.length() == 0) return;

      // make it bottom to top
      array<argb> row(_dim.x);
      int         s = 0;
      int         e = _dim.y - 1;
      for (; s < e; ++s, --e) {
        row = (*this)[s];
        target_row(s).copy((*this)[e]);
        target_row(e).copy(row());
      }
      //++_generation;
    }
//#endif

    void blur(size radius);

    tslice<byte> pixels_as_target_bytes() { return tslice<byte>((byte*)_pixels.begin(), _pixels.length() * sizeof(argb)); }
    slice<byte>  pixels_as_bytes() { return slice<byte>((const byte*)_pixels.cbegin(), _pixels.length() * sizeof(argb)); }

    int stride() const { return dim().x * sizeof(argb); }

#ifdef WINDOWS
    HICON create_win_icon(size offset = size());
#endif

  public:
    array<argb>     _pixels; // width*height, premultiplied
    size            _dim;
    bool            _has_alpha_bits;
    uint            _uid;
    locked::counter _generation; // pixel validity generation number, each
                                 // modification of bits shall increase this
                                 // number
    // hash_table<uint_ptr, handle<bitmap>> transformed_cache;

    typedef pair<handle<image_filter>, handle<bitmap>> filter_and_bitmap;
    circular_buffer<filter_and_bitmap>                 transformed_cache;

    handle<bitmap> _mapped_ltr;
    handle<bitmap> _mapped_ttr;

#ifdef USE_GDI
    handle<Gdiplus::Bitmap> gdi_bmp;
#endif
#ifdef OSX
    CF::ref<CGImageRef> cg_image;
#endif
#ifdef USE_CAIRO
    cairo::ref<cairo_surface_t> cr_surface;
#endif

#ifdef USE_SKIA
    SkBitmap sk_bitmap;
#endif
    long                   _used_generation;
    static locked::counter _uid_;
  };

  typedef handle<bitmap> hbitmap;

  class expandable_bitmap : public bitmap {
  public:
    expandable_bitmap(const dib32 *src, bool transparent,
                      const SECTION_DEFS &sd)
        : bitmap(src, transparent), section_defs(sd) {}
    expandable_bitmap(size sz, bool transparent, bool with_pixel_buf,
                      const SECTION_DEFS &sd)
        : bitmap(sz, transparent, with_pixel_buf), section_defs(sd) {}

    SECTION_DEFS section_defs;

    virtual handle<bitmap> get_bitmap(graphics *pg, size sz) { return this; }

  };

  /*class shadow_bitmap : public expandable_bitmap {
  public:
    shadow_bitmap(size sz = size(0, 0)) : expandable_bitmap(sz, true, true) {}
    point offset;
  };*/

  class shadow_bitmap : public expandable_bitmap {
  public:
    shadow_bitmap(size sz) : expandable_bitmap(sz, true, true, SECTION_DEFS()) {}
    point offset_origin;
    point offset_corner;
  };


  /*inline
    void image::render( graphics* gfx, rect dst, rect src, byte opacity)
  {
    bitmap *bmp = get_bitmap();
    if(bmp) bmp->render(gfx, dst, src, opacity);
  }*/

  class animated_image : public image {
  public:
    enum framemode { nothing = 0, nodispose = 1, clear = 2, restore = 3 };
    enum blendmode {
      blend = 0,
      copy  = 1,
    };
    struct state {
      uint     iter_no;
      int      frame_no;
      uint     clock; // next clock
      uint_ptr site_id;
      handle<bitmap> rendered;
      state() : iter_no(0), frame_no(-1), clock(0), site_id(0) {}
    };

    animated_image(size dim) : sz(dim), iters(0) {}
    virtual ~animated_image() { frames.clear(); }

    virtual size dim() const override { return sz; }

    virtual handle<bitmap> get_bitmap(graphics * /*pg*/, size /*sz*/) override {
      //return frames[0].img;
      //return get_bitmap_for_frame(0);
      return nullptr;
    }

    virtual handle<bitmap> get_bitmap(graphics * /*pg*/, size /*sz*/,
                               uint_ptr site_id) {
      const state &st = get_state_for(site_id);
      //return frames[st.frame_no].img;
      return get_bitmap_for_frame(st.frame_no,site_id);
    }

    handle<bitmap> get_bitmap_for_frame(uint frame_no, uint_ptr site_id);
    
    state &get_state_for(uint_ptr site_id) {
      FOREACH(i, sites) {
        if (sites[i].site_id == site_id) return sites[i];
      }
      state &st  = sites.push();
      st.site_id = site_id;
      _restart(st);
      return st;
    }

    void remove_state_for(uint_ptr site_id) {
      FOREACH(i, sites)
      if (sites[i].site_id == site_id) sites.remove(i);
    }

    uint next_step(uint_ptr site_id) // returns new frame duration
    {
      state &st = get_state_for(site_id);
      return _set_next_frame(st);
    }
    uint step_duration(uint_ptr site_id) // returns current frame duration
    {
      state &st = get_state_for(site_id);
      return _frame_duration(st);
    }

    void add(bitmap *frame_img, uint duration, uint mode, point pt,
             blendmode blend_mode = animated_image::blend);

    uint _set_next_frame(state &st); // returns new frame duration
    void _restart(state &st);
    uint _frame_duration(state &st) const {
      return frames[st.frame_no].duration;
    }

    void iterations(uint it) { iters = it; }

    virtual bool is_animated(uint_ptr site_id) const override {
      if (iters == 0) return true;
      state &st = const_cast<animated_image *>(this)->get_state_for(site_id);
      if (frames.size() <= 1) return false;

      return st.iter_no < iters;
    }

    virtual bool set_frame_no(int no, uint_ptr site_id) override {
      return get_bitmap_for_frame(no, site_id) != nullptr;
    }

    virtual void drop_cache() override {
      //FOREACH(i, frames) {
      //  if (frames[i].img) frames[i].img->drop_cache();
      //}
      FOREACH(i, sites) {
        if (sites[i].rendered)
          sites[i].rendered->drop_cache();
      }
    }

    int            n_frames() const { return frames.size(); }
    handle<bitmap> frame_image(int n) const { return n < frames.size() ? frames[n].frame_img: handle<bitmap>(); }

  protected:

    void set_next_frame(state& st);

    state st;
    struct frame {
      point          pos;
      framemode      prev_dispose_mode;
      uint           duration;
      blendmode      blend_mode;
      handle<bitmap> frame_img;
      bool           restore_after;
    };
    tool::array<frame> frames;
    tool::array<state> sites;

    size sz;
    uint iters;
  };

  enum IMAGE_MAPPING {
    IMAGE_MAPPING_NONE,
    IMAGE_MAPPING_L2R,
    IMAGE_MAPPING_T2R,
  };

  class stock_image // stock symbols
      : public image {
    // tool::string _name;
    int id;
    stock_image(int i) : id(i) {}

  public:

    DEFINE_TYPE_ID_DERIVED(stock_image, image)

    // stock_image(const tool::string& name);
    static stock_image *get(const tool::string &name);

    virtual bool is_valid() const { return true; }
    virtual bool is_transparent() const { return true; } // has alpha bits
    virtual bool is_animated(uint_ptr /*site_id*/) const { return false; }
    virtual bool is_vector() const { return true; } // pure vector image

    virtual handle<bitmap> get_bitmap(graphics * /*pg*/, size /*sz*/) { return 0; }

    virtual void draw(graphics *gfx, rect dst, rect src, byte opacity) {
      draw(gfx, rectf(dst), src, opacity);
    }
    virtual void draw(graphics *gfx, rectf dst, rect src, byte opacity);
    virtual void expand(graphics * /*gfx*/, const rect & /*dst*/, const SECTION_DEFS & /*sds*/, rect src = rect()) override {
      assert(false);
    }

    size dimension() const;

    virtual size dim() const { return dimension(); }
    virtual size dim(const rect & /*for_dst*/) const { return dim(); }
    virtual void drop_cache() {}

    virtual image *mapped_left_to_right();
    virtual image *mapped_top_to_right();
  };

  class morphing_image // image being morphed
      : public image {
    handle<image> _from, _to;
    float         _progress;

  public:
    DEFINE_TYPE_ID_DERIVED(morphing_image, image)

    morphing_image(image *from, image *to, float progress = 0.0f)
        : _from(from), _to(to), _progress(progress) {}

    virtual bool is_valid() const override { return true; }
    virtual bool is_transparent() const override { return true; } // has alpha bits
    virtual bool is_animated(uint_ptr /*site_id*/) const override { return false; }

    virtual handle<bitmap> get_bitmap(graphics * /*pg*/, size /*sz*/) override {
      return nullptr;
    }

    virtual void draw(graphics *gfx, rect dst, rect src, byte opacity) override {
      draw(gfx, rectf(dst), src, opacity);
    }
    virtual void draw(graphics *gfx, rectf dst, rect src, byte opacity) override {
      uint p = uint(limit<float>(_progress, 0.0f, 1.0f) * 255);
      if (_from)
        _from->draw(gfx, dst, src, byte(uint(opacity) * (255 - p) / 255));
      if (_to) _to->draw(gfx, dst, src, byte(uint(opacity) * p / 255));
    }
    virtual void expand(graphics *gfx, const rect &dst, const SECTION_DEFS &sds, rect src = rect()) override {
      if (_from) _from->expand(gfx, dst, sds, src);
      if (_to) _to->expand(gfx, dst, sds, src);
    }

    size dimension() const {
      size f;
      if (_from) f = _from->dimension();
      size t;
      if (_to) t = _to->dimension();
      return size(max(f.x, t.x), max(f.y, t.y));
    }

    virtual size dim() const override { return dimension(); }
    virtual size dim(const rect & /*for_dst*/) const override { return dim(); }
    virtual void drop_cache() override {
      if (_from) _from->drop_cache();
      if (_to) _to->drop_cache();
    }
    // virtual image* mapped_left_to_right();
    // virtual image* mapped_top_to_right();

    void morph(image *from, image *to, float n) {
      _from     = from;
      _to       = to;
      _progress = n;
    }
  };

  enum ICT_TYPE {
    ICT_NONE = 0,     // image-transformation:none;
    ICT_COLOR_SCHEMA, // ICT_COLORIZE but with selective magic
    ICT_CBG,          // contrast, brightness, gamma
    ICT_COLORIZE,     // like converting to gray but using color
    ICT_OPACITY,      // alpha = alpha * p1
    ICT_HUE,          // dst.hue = src.hue
    ICT_SATURATION,   // dst.saturation = src.saturation
    ICT_FLIP_X,       // mirror x
    ICT_FLIP_Y,       // mirror y

    ICT_MULTIPLY,
    ICT_SCREEN,
    ICT_OVERLAY,
    ICT_DARKEN,
    ICT_LIGHTEN,
    ICT_DODGE,
    ICT_COLOR_BURN,
    ICT_HARD_LIGHT,
    ICT_SOFT_LIGHT,
    ICT_DIFFERNCE,
    ICT_EXCLUSION,
    //BLEND_HUE,
    //BLEND_SATURATION,
    ICT_COLOR,
    ICT_LUMINOSITY,
    
  };

 
  struct image_filter : public resource {
    handle<image_filter> next;

    virtual ICT_TYPE      type() const  = 0;
    virtual image_filter *clone() const = 0;
    virtual uint          sub_type() const { return 0; }

    bitmap *apply(bitmap *src) const {
      bitmap *out = new bitmap(src);
      for (const image_filter *p = this; p; p = p->next)
        p->this_apply(out);
      return out;
    }

    bool is_equal(const image_filter *other) const {
      if (type() != other->type()) return false;
      if (sub_type() != other->sub_type()) return false;
      if (!this_value_is_equal(other)) return false;
      if (!next && !(other->next)) return true;
      if (next && other->next) return next->is_equal(other->next);
      return false;
    }

    uint hash() const {
      uint h = hash_value((uint)type());
      hash_combine(h, hash_value(sub_type()));
      hash_combine(h, this_value_hash());
      if(next) hash_combine(h, next->hash());
      return h;
    }

    virtual bool this_value_is_equal(const image_filter *to) const = 0;
    virtual uint this_value_hash() const = 0;
    virtual void this_apply(bitmap *src) const               = 0;
    virtual void emit(wchar_stream_o &out) const             = 0;

    bool operator==(const image_filter &other) const { return is_equal(&other); }
    bool operator!=(const image_filter &other) const { return !is_equal(&other); }
       
    static handle<image_filter> clone(handle<image_filter>  head,
                                      handle<image_filter> &tail) {
      handle<image_filter> t, h;
      tail = 0;
      while (head) {
        t = head->clone();
        if (!h) h = t;
        if (tail) tail->next = t;
        tail = t;
        head = head->next;
      }
      return h;
    }
  };
   

  struct image_filters {
    handle<image_filter> first;
    uint additive; // not bool as it must fill the struct without holes.
    image_filters() : additive(0) {}

    uint hash() const {
      return tool::hash(first) + tool::hash(additive);
    }
    bool operator==(const image_filters& rs) const {
      return first.is_identical(rs.first) && additive == rs.additive;
    }

    bool is_undefined() const { return !first; }
    bool is_defined() const { return !!first; }
    void clear() {
      first    = 0;
      additive = false;
    }
    void inherit(const image_filters &v) {
      if (v.is_undefined()) return;
      additive = additive || v.additive;
      if (additive) {
        handle<image_filter> last, dummy;
        first = image_filter::clone(first, last);
        if (last)
          last->next = image_filter::clone(v.first, dummy);
        else
          first = image_filter::clone(v.first, dummy);
      } else {
        first = v.first;
      }
    }
    ustring to_string() const {
      if (!first) return ustring();
      wchar_stream_o out;
      for (image_filter *t = first; t; t = t->next) {
        if (t != first) out << WCHARS(" ");
        t->emit(out);
      }
      return out.text();
    }
  };

  struct image_reader {
    bytes        data;
    const string url;

    image_reader(bytes dt, const string &u) : data(dt), url(u) {}
    bool read_bytes(void *bf, size_t bfsize) {
      if (bfsize > data.length) return false;
      data.pull(target((byte *)bf, bfsize));
      return true;
    }
    virtual handle<image> read(html::document *pv = nullptr) = 0;

    NONCOPYABLE(image_reader)
  };
  struct image_writer {
    tool::array<byte> &out;
    image_writer(tool::array<byte> &buf) : out(buf) {}
    void write_bytes(const void *bf, size_t bfsize) {
      out.push((const byte *)bf, int(bfsize));
    }
    virtual size_t write(image *img, int compression_level) = 0;

    NONCOPYABLE(image_writer)
  };

#ifdef SVG_SUPPORT
  struct svg_reader : public image_reader {
    svg_reader(bytes dt, const string &url) : image_reader(dt, url) {}
    virtual handle<image> read(html::document *pv) override;
  };
#endif

#ifdef WINDOWS
  class dib32 : public resource {
    const uint      _width;
    const uint      _height;
    const uint      _stride;
    void *          _bits;
    mutable HBITMAP _old_bitmap;
    mutable HDC     _dc;
    HBITMAP         _bitmap;

    BITMAPINFO _bitmap_info;

  public:
    dib32(__in const size &sz)
        : _width(sz.x), _height(sz.y), _stride((sz.x * 32 + 31) / 32 * 4),
          _bits(0), _old_bitmap(0), _dc(0) {

      memzero(_bitmap_info);
      _bitmap_info.bmiHeader.biSize        = sizeof(_bitmap_info.bmiHeader);
      _bitmap_info.bmiHeader.biWidth       = _width;
      _bitmap_info.bmiHeader.biHeight      = 0 - int(_height);
      _bitmap_info.bmiHeader.biPlanes      = 1;
      _bitmap_info.bmiHeader.biBitCount    = 32;
      _bitmap_info.bmiHeader.biCompression = BI_RGB;

      _bitmap = ::CreateDIBSection(NULL, // device context
                                   &_bitmap_info, DIB_RGB_COLORS, &_bits,
                                   0,  // file mapping object
                                   0); // file offset

      if (0 == _bits) {
        return;
        // throw std::bad_alloc();
      }

      memset(_bits, 0, _width * _height * 4);
    }

    ~dib32() {
      if (_dc) {
        ::SelectObject(_dc, _old_bitmap);
        ::DeleteDC(_dc);
      }
      if (_bitmap) ::DeleteObject(_bitmap);
    }

    void set_white() { memset(_bits, 0xff, _width * _height * 4); }

    gool::size dim() const { return gool::size(_width, _height); }
    uint       width() const { return _width; }
    uint       height() const { return _height; }
    uint       stride() const { return _stride; }
    void *     bits() const { return _bits; }

    bool       is_valid() const { return _bits != 0; }

    HDC        DC() const {
      if (!_dc) {
        _dc = ::CreateCompatibleDC(NULL);
        if (_dc) _old_bitmap = (HBITMAP)::SelectObject(_dc, _bitmap);
      }
      return _dc;
    }

    HBITMAP BITMAP() const { return _bitmap; }

    HBITMAP detach() {
      auto r  = _bitmap;
      _bitmap = 0;
      return r;
    }

    const BITMAPINFO &get_info() const { return _bitmap_info; }

    slice<argb> row(int r) const {
      assert(r < int(_height));
      if (r >= 0 && r < int(_height)) {
        argb *pc = static_cast<argb *>(_bits);
        return slice<argb>(pc + r * _width, _width);
      }
      return slice<argb>();
    }
    slice<argb> pixels() const {
      argb *pc = static_cast<argb *>(_bits);
      return slice<argb>(pc, _width * _height);
    }
    tslice<argb> target_pixels() {
      argb *pc = static_cast<argb *>(_bits);
      return tslice<argb>(pc, _width * _height);
    }

    NONCOPYABLE(dib32)
  };
#endif

  struct bitmap_list {
    hash_table<ustring, handle<bitmap>> bitmaps;
    mutex lock;
  public:
    bitmap_list() {}
    bitmap *        get(const ustring &name, bool allow_read_from_disk);
    virtual bitmap *create(const ustring &name, bool allow_read_from_disk) = 0;

    void clear() { bitmaps.clear(); }
  };

  struct cursor : public resource {
    DEFINE_TYPE_ID(cursor);

    uint   id;
    string url;
#if defined(OSX)
    #if defined(WINDOWLESS)
      void* pcur;
      cursor() : id(0),pcur(0){}
      virtual ~cursor() {}
    #else
      void *pcur;
      cursor() : id(0), pcur(nullptr) {}
      cursor(void *nscursor);
      virtual ~cursor();
      void set(); // set as current
    #endif
#elif defined(LINUX)
    #if defined(WINDOWLESS)
	  void* pcur;
      cursor() : id(0),pcur(0){}
      virtual ~cursor() {}
	#else
	  gtk::ref<GdkCursor> pcur;
      cursor() : id(0) {}
      virtual ~cursor() {}
	#endif  
#elif defined(WINDOWS)
    HCURSOR hcur;
    HCURSOR object() const { return hcur; }
    cursor() : hcur(NULL), id(uint(-1)) {}
    virtual ~cursor();
    static cursor *copy();
    static cursor *move();
#endif

    static cursor *system(uint id);
    static cursor *from_data(bytes data, string url);
    static cursor *from_bitmap(bitmap *bmp, string url, size offset);

    static handle<cursor> inherit;

    value to_value() const {
              
      static const char* names[] = {
        "default",
        "text",
        "wait",
        "crosshair",
        "uparrow",
        "nw-resize",
        "ne-resize",
        "w-resize",
        "n-resize",
        "move",
        "no",
        "starting",
        "help",
        "pointer",
        "drag-copy",
        "drag-move",
      };
      if (id >= 0 && id < items_in(names))
        return value(names[id]);
      return url.to_value(value::UT_URL);
    }

  };

  // NOTE: works in premultiplied color space
  template <typename F>
  inline void bitmap::combine(point dst, bitmap *src, rect srcr, F f) {
    srcr &= rect(src->dim());
    rect dstr(dst, srcr.size());
    dstr &= rect(dim());

    int sr = srcr.s.y;
    int dr = dstr.s.y;
    for (; dr < dstr.e.y; ++sr, ++dr) {
      argb *s = (*src)(sr) + srcr.s.x;
      argb *d = (*this)(dr) + dstr.s.x;
      for (int x = dstr.s.x; x < dstr.e.x; ++x, ++d, ++s)
        *d = f(*d, *s);
    }
    ++_generation;
  }

  namespace combine {
    namespace premultiplied {
// https://dev.w3.org/SVG/modules/compositing/master/
#define MUL(c1, c2) byte((uint(c1) * uint(c2)) / 255)

      inline argb src_over(argb dst, argb src) {
        byte sai = 255 - src.alfa;
        argb t;
        t.red   = src.red + MUL(dst.red, sai);
        t.green = src.green + MUL(dst.green, sai);
        t.blue  = src.blue + MUL(dst.blue, sai);
        t.alfa  = src.alfa + dst.alfa - MUL(src.alfa, dst.alfa);
        return t;
      }
      inline argb dst_over(argb dst, argb src) {
        byte dai = 255 - dst.alfa;
        argb t;
        t.red   = dst.red + MUL(dai, src.red);
        t.green = dst.green + MUL(dai, src.green);
        t.blue  = dst.blue + MUL(dai, src.blue);
        t.alfa  = src.alfa + dst.alfa - MUL(src.alfa, dst.alfa);
        return t;
      }
      inline argb src_in(argb dst, argb src) {
        argb t;
        t.red   = MUL(dst.alfa, src.red);
        t.green = MUL(dst.alfa, src.green);
        t.blue  = MUL(dst.alfa, src.blue);
        t.alfa  = MUL(dst.alfa, src.alfa);
        return t;
      }
      inline argb dst_in(argb dst, argb src) {
        argb t;
        t.red   = MUL(src.alfa, dst.red);
        t.green = MUL(src.alfa, dst.green);
        t.blue  = MUL(src.alfa, dst.blue);
        t.alfa  = MUL(src.alfa, dst.alfa);
        return t;
      }
      inline argb src_out(argb dst, argb src) {
        argb t;
        byte dai = 255 - dst.alfa;
        t.red    = MUL(dai, src.red);
        t.green  = MUL(dai, src.green);
        t.blue   = MUL(dai, src.blue);
        t.alfa   = MUL(dai, src.alfa);
        return t;
      }
      inline argb dst_out(argb dst, argb src) {
        argb t;
        byte sai = 255 - src.alfa;
        t.red    = MUL(sai, dst.red);
        t.green  = MUL(sai, dst.green);
        t.blue   = MUL(sai, dst.blue);
        t.alfa   = MUL(sai, dst.alfa);
        return t;
      }
      inline argb src_atop(argb dst, argb src) {
        argb t;
        byte sai = 255 - src.alfa;
        t.red    = MUL(dst.alfa, src.red) + MUL(sai, dst.red);
        t.green  = MUL(dst.alfa, src.green) + MUL(sai, dst.green);
        t.blue   = MUL(dst.alfa, src.blue) + MUL(sai, dst.blue);
        t.alfa   = dst.alfa;
        return t;
      }
      inline argb dst_atop(argb dst, argb src) {
        argb t;
        byte dai = 255 - dst.alfa;
        t.red    = MUL(dai, src.red) + MUL(src.alfa, dst.red);
        t.green  = MUL(dai, src.green) + MUL(src.alfa, dst.green);
        t.blue   = MUL(dai, src.blue) + MUL(src.alfa, dst.blue);
        t.alfa   = MUL(dai, src.alfa) + MUL(src.alfa, dst.alfa);
        return t;
      }
      inline argb dst_xor_src(argb dst, argb src) {
        argb t;
        byte dai = 255 - dst.alfa;
        byte sai = 255 - src.alfa;
        t.red    = MUL(dai, src.red) + MUL(sai, dst.red);
        t.green  = MUL(dai, src.green) + MUL(sai, dst.green);
        t.blue   = MUL(dai, src.blue) + MUL(sai, dst.blue);
        t.alfa   = src.alfa + dst.alfa - 2 * MUL(src.alfa, dst.alfa);
        return t;
      }
      inline argb dst_copy_src(argb dst, argb src) { return src; }

#undef MUL
    }
  } // namespace combine
} // namespace gool

#endif
