#include "gool.h"
#include <cmath>

namespace html {
  struct element;
  float setup_fill_and_stroke(gool::graphics *gfx, const html::element*el, html::style *ps,
                              gool::sizef sz, bool filled, byte opacity,
                              float scale = 1.0f);
}

namespace gool {

  bool mirrored_bitmaps() {
    static bool mi = app()->mirrored_bitmaps();
    return mi;
  }

  locked::counter bitmap::_uid_;
#ifdef _DEBUG

  int image::image_instances = 0;

  void image::report_number_of_instances() {
    dbg_printf("number of images:%d\n", image_instances);
  }
#endif
  image::~image() {
    drop_cache();
#ifdef _DEBUG
    --image_instances;
#endif
  }

  void image::drop_cache() {
    image_link *pl = image_links;
    while (pl) {
      image_link *plt = pl;
      pl->detach_from_graphics();
      pl = pl->next_in_image;
      delete plt;
    }
    image_links = 0;
  }

  image_link *image::get_link_for(graphics *pg) {
    image_link *pl = image_links;
    while (pl) {
      if (pl->for_gfx == pg && pl->for_gfx->get_uid() == pg->get_uid())
        return pl;
      pl = pl->next_in_image;
    }
    pl                   = new image_link;
    pl->for_gfx          = pg;
    pl->for_img          = this;
    pl->next_in_graphics = pg->_image_links;
    pg->_image_links     = pl;
    pl->next_in_image    = image_links;
    image_links          = pl;
    return pl;
  }

  void image_link::detach_from_graphics() {
    if (for_gfx->_image_links == this)
      for_gfx->_image_links = this->next_in_graphics;
    else {
      image_link *pl = for_gfx->_image_links;
      while (pl) {
        if (pl->next_in_graphics == this) {
          pl->next_in_graphics = this->next_in_graphics;
          return;
        }
        pl = pl->next_in_graphics;
      }
      assert(false); // must be in!
    }
  }

  void image_link::detach_from_image() {
    if (for_img->image_links == this)
      for_img->image_links = this->next_in_image;
    else {
      image_link *pl = for_img->image_links;
      while (pl) {
        if (pl->next_in_image == this) {
          pl->next_in_image = this->next_in_image;
          return;
        }
        pl = pl->next_in_image;
      }
      assert(false); // must be in!
    }
  }

  void image::draw(graphics *gfx, rect dst, rect src, byte opacity) {
    if(!dst.empty() && !src.empty() )
      gfx->do_draw(this, dst, src, opacity);
  }
  void image::draw(graphics *gfx, rectf dst, rect src, byte opacity) {
    if (!dst.empty() && !src.empty())
      gfx->do_draw(this, dst, src, opacity);
  }

  void image::tile(graphics *gfx, const rect &dst, const rect &src, point offset, size dstcell)
  {
    if (!dst.empty() && !src.empty())
      gfx->fill(this, dst, src, offset, dstcell);
  }

  void image::expand(graphics *gfx, const rect &dst, const SECTION_DEFS &sds,
                     rect area) {
    gfx->do_expand(this, dst, sds, area);
  }

  inline void intersection(
      rect &dst,
      rect &src) // modifies both, dst & src to contian intersected areas.
  {
    rect cr = dst & src;
    if (cr.s.x > dst.s.x) src.s.x += cr.s.x - dst.s.x;
    if (cr.s.y > dst.s.y) src.s.y += cr.s.y - dst.s.y;
    if (cr.e.x < dst.e.x) src.e.x -= dst.e.x - cr.e.x;
    if (cr.e.y < dst.e.y) src.e.y -= dst.e.y - cr.e.y;
    dst = cr;
  }

  /*void bitmap::copy(point dst, bitmap *src, rect srcr) {
    srcr &= rect(src->dim());
    rect dstr(dst, srcr.size());
    dstr &= rect(dim());

    int sr = srcr.origin.y;
    int dr = dstr.origin.y;
    for (; dr <= dstr.corner.y; ++sr, ++dr) {
      argb *s = (*src)(sr) + srcr.origin.x;
      argb *d = (*this)(dr) + dstr.origin.x;
      for (int x = dstr.origin.x; x <= dstr.corner.x; ++x, ++d, ++s)
        *d = *s;
    }
    ++_generation;
  }*/

  /*void bitmap::blend(point dst, bitmap *src, rect srcr) {
    srcr &= rect(src->dim());
    rect dstr(dst, srcr.size());
    dstr &= rect(dim());
    int sr = srcr.origin.y;
    int dr = dstr.origin.y;
    for (; dr <= dstr.corner.y; ++sr, ++dr) {
      argb *s = (*src)(sr) + srcr.origin.x;
      argb *d = (*this)(dr) + dstr.origin.x;
      for (int x = dstr.origin.x; x <= dstr.corner.x; ++x, ++d, ++s)
        d->blend(*s);
    }
  }*/

  /*void bitmap::set(rect srcr, argb c) {
    rect dstr(dim());
    dstr &= srcr;
    // intersection(dstr,srcr);
    for (int dr = dstr.origin.y; dr <= dstr.corner.y; ++dr) {
      argb *d = (*this)(dr) + dstr.origin.x;
      for (int x = dstr.origin.x; x <= dstr.corner.x; ++x, ++d)
        *d = c;
    }
  }*/

  /*void bitmap::copy(point dst, slice<argb> src, size dim) {
    pixmap pmd(_pixels.begin(), _dim);
    pixmap pms(const_cast<argb *>(src.start), dim);

    pmd.overlay(pms, dst, [](argb &d, const argb &s) { d = s; });

    ++_generation;
  }*/

  bool bitmap::is_solid_color(rect src, argb &clr) {
    rect d(this->dim());
    d = d & src;
    if (d.empty()) {
      clr = argb(0, 0, 0, 0);
      return true;
    }
    argb t = (*this)[d.s.y][d.s.x];
    for (int y = d.s.y; y < d.e.y; ++y) {
      auto r = (*this)[y];
      for (int x = d.s.x; x < d.e.x; ++x)
        if (r[x] != t) return false;
    }
    clr = t;
    return true;
  }

  image *bitmap::mapped_left_to_right() {
    if (_mapped_ltr) return _mapped_ltr;
    _mapped_ltr = new bitmap(this);
    if (_mapped_ltr) {
      int rows = _mapped_ltr->dim().y;
      for (int r = 0; r < rows; ++r) {
        slice<argb> row = (*_mapped_ltr)[r];
        argb *      s   = const_cast<argb *>(row.start);
        argb *      l   = const_cast<argb *>(row.end() - 1);
        while (s < l)
          swap(*s++, *l--);
      }
    }
    return _mapped_ltr;
  }
  image *bitmap::mapped_top_to_right() {
    if (_mapped_ttr) return _mapped_ttr;
    _mapped_ttr = new bitmap(this);
    if (_mapped_ttr) {
      swap(_mapped_ttr->_dim.x, _mapped_ttr->_dim.y);
      int rows          = _dim.y;
      int dst_row_width = _mapped_ttr->_dim.x;

      for (int r = 0; r < rows; ++r) {
        slice<argb> row = (*this)[r];
        for (uint c = 0; c < row.length; ++c) {
          int dst_index                   = c * dst_row_width + r;
          _mapped_ttr->_pixels[dst_index] = row[c];
        }
      }
    }
    return _mapped_ttr;
  }

  void animated_image::set_next_frame(state& st)
  {
    ++st.frame_no;
    if (st.frame_no >= frames.size())
      st.frame_no = 0;

    handle<bitmap> fimg = st.rendered ? st.rendered.ptr() : new bitmap(dim());

    frame f = frames[st.frame_no];
    if (st.frame_no) {
      if (f.prev_dispose_mode == restore && st.frame_no >= 2) // restore previous
      {
        int restore_point_index = st.frame_no - 2;
        while (restore_point_index > 0) {
          frame &t = frames[restore_point_index];
          if (!t.restore_after) break;
          --restore_point_index;
        }
        frame &restore_point = frames[restore_point_index];
        frame &prev = frames[st.frame_no - 1];
        prev.restore_after = true;
        fimg->copy(point(0, 0), prev.frame_img);
        fimg->copy(prev.pos, restore_point.frame_img, rect(prev.pos, prev.frame_img->dim()));
      }
      else if (f.prev_dispose_mode == clear) {
        frame &prev = frames[st.frame_no - 1];
        if (prev.frame_img->dim() != dim())
          fimg->copy(point(0, 0), st.rendered);
        argb trans(0, 0, 0, 0);
        fimg->set(rect(prev.pos, prev.frame_img->dim()), trans);
      }
      else { // prev_dispose_mode == nodispose
        //frame &prev = frames[frame_no - 1];
        //fimg->copy(point(0, 0), st.rendered);
      }
    }

    if (f.blend_mode == copy || st.frame_no == 0)
      fimg->copy(f.pos, f.frame_img, rect(f.frame_img->dim()));
    else
      fimg->blend(f.pos, f.frame_img, rect(f.frame_img->dim()));

    st.rendered = fimg;

  }

  handle<bitmap> animated_image::get_bitmap_for_frame(uint frame_no, uint_ptr site_id)
  {
    state &st = get_state_for(site_id);
    
    for (int n = 0; n < frames.size(); ++n)
    {
      if (st.frame_no == int(frame_no))
        break;
      set_next_frame(st);
    }
    return st.rendered;

/*
    frame f = frames[frame_no];

    state &st = get_state_for(site_id);

    handle<bitmap> fimg = st.rendered ? st.rendered.ptr() : new bitmap(dim());

    if (frame_no) {
      if (f.prev_dispose_mode == restore && frame_no >= 2) // restore previous
      {
        int restore_point_index = frame_no - 2;
        while (restore_point_index > 0) {
          frame &t = frames[restore_point_index];
          if (!t.restore_after) break;
          --restore_point_index;
        }
        frame &restore_point = frames[restore_point_index];
        frame &prev = frames[frame_no - 1];
        prev.restore_after = true;
        fimg->copy(point(0, 0), prev.frame_img);
        fimg->copy(prev.pos, restore_point.frame_img, rect(prev.pos, prev.frame_img->dim()));
      }
      else if (f.prev_dispose_mode == clear) {
        frame &prev = frames[frame_no - 1];
        if (prev.frame_img->dim() != dim())
          fimg->copy(point(0, 0), st.rendered);
        argb trans(0, 0, 0, 0);
        fimg->set(rect(prev.pos, prev.frame_img->dim()), trans);
      }
      else { // if( prev_dispose_mode == nodispose )
             //frame &prev = frames[frame_no - 1];
             //fimg->copy(point(0, 0), st.rendered);
      }
    }

    if (f.blend_mode == copy || frame_no == 0)
      fimg->copy(f.pos, f.frame_img, rect(f.frame_img->dim()));
    else
      fimg->blend(f.pos, f.frame_img, rect(f.frame_img->dim()));

    st.rendered = fimg;

    return st.rendered;
*/

  }



#if 0

  void animated_image::add(bitmap *frame_img, uint duration,
                           uint prev_dispose_mode, point pt,
                           blendmode blend_mode) {
    frame f;
    f.duration      = duration;
    f.restore_after = false;

    argb trans(0, 0, 0, 0);

    handle<bitmap> fimg     = frame_img->create_compatible_bitmap(dim());
    int            frame_no = frames.size();

    if (frame_no) {
      if (prev_dispose_mode == restore && frame_no >= 2) // restore previous
      {
        int restore_point_index = frame_no - 2;
        while (restore_point_index > 0) {
          frame &t = frames[restore_point_index];
          if (!t.restore_after) break;
          --restore_point_index;
        }
        frame &restore_point = frames[restore_point_index];
        frame &prev          = frames[frame_no - 1];
        prev.restore_after   = true;
        fimg->copy(point(0, 0), prev.img);
        fimg->copy(prev.area.origin, restore_point.img, prev.area);
      } else if (prev_dispose_mode == clear) {
        frame &prev = frames[frame_no - 1];
        if (prev.area != rect(dim())) fimg->copy(point(0, 0), prev.img);
        fimg->set(prev.area, trans);
      } else { // if( prev_dispose_mode == nodispose )
        frame &prev = frames[frame_no - 1];
        fimg->copy(point(0, 0), prev.img);
      }
    }

    if (blend_mode == copy || frame_no == 0)
      fimg->copy(pt, frame_img, rect(frame_img->dim()));
    else
      fimg->blend(pt, frame_img, rect(frame_img->dim()));

    f.img  = fimg;
    f.area = rect(pt, frame_img->dim());

    frames.push(f);
  }
#else
  void animated_image::add(bitmap *frame_img, uint duration, uint prev_dispose_mode, point pt, blendmode blend_mode) {
    frame f;
    f.duration = duration;
    f.restore_after = false;
    f.blend_mode = blend_mode;
    f.prev_dispose_mode = framemode(prev_dispose_mode);
    f.pos = pt;
    f.frame_img = frame_img;
    frames.push(f);
  }
#endif

  uint animated_image::_set_next_frame(state &st) // returns new frame duration
  {
    if (frames.size() == 0) return 0;

    set_next_frame(st);

    if (st.frame_no >= frames.size()) return 0;
    if (frames.length() == 1) return 100000;
    frame &next = frames[st.frame_no];
    return min(next.duration, 100000);
  }

  void animated_image::_restart(state &st) { 
    st.frame_no = -1; 
    set_next_frame(st);
  }

  path_image::path_image(const tool::string &spath)
      : path_app(nullptr), mapping(IMAGE_MAPPING_NONE), is_expandable(false) {
    path_chars = spath;
  }

  path_image::~path_image() {}

  bool lessthanzero(float f) { return *((int32 *)&f) < 0; }

  bool path_image::parse(gool::path *path, chars d,
    size sz) // returns true if it uses relative commands -
             // defines expandable shape
  {
    path->reset();

    pointf subpath_start, last, last2, p1, p2, p3;
    pointf reference_point; bool rp_active = false;
    wchar  last_command_char = 0;
    bool   is_relative = true;
    bool   carry_on = true;

#ifdef _DEBUG
    if (!sz.empty())
      sz = sz;
#endif 

    bool expandable = false;

    const chars command_chars = CHARS("EMmLlHhVvCcSsQqTtAaZzRr");

    auto expand_x = [&sz, &expandable](pointf &p) {
      if (expandable && lessthanzero(p.x)) { p.x += sz.x; }
    };
    auto expand_y = [&sz, &expandable](pointf &p) {
      if (expandable && lessthanzero(p.y)) { p.y += sz.y; }
    };
    auto expand = [&sz, &expandable](pointf &p) {
      if (expandable && lessthanzero(p.x)) { p.x += sz.x; }
      if (expandable && lessthanzero(p.y)) { p.y += sz.y; }
    };

    auto relate_x = [&reference_point, &last, rp_active](pointf &p) {
      if (rp_active) p.x += reference_point.x;
      else p.x += last.x;
    };

    auto relate_y = [&reference_point, &last, rp_active](pointf &p) {
      if (rp_active) p.y += reference_point.y;
      else p.y += last.y;
    };

    auto relate = [&reference_point, &last, rp_active](pointf &p) {
      if (rp_active) p += reference_point;
      else p += last;
    };

    auto parse_real = [](chars &text, float &num) -> bool {
      while (!!text && (is_space(*text) || *text == ','))
        ++text;
      return tool::parse_real(text, num);
    };

    auto parse_int = [](chars &text, int &num) -> bool {
      while (!!text && (is_space(*text) || *text == ','))
        ++text;
      return tool::parse_int(text, num, 10U);
    };

    auto parse_coord_x = [&parse_real, sz, &expandable](chars & text,
      pointf &pt) -> bool {
      if (!parse_real(text, pt.x)) return false;
      if (*text == '%') {
        ++text;
        expandable = true;
        pt.x = sz.x * pt.x / 100.0f;
      }
      if (*text == '*') {
        ++text;
        expandable = true;
        pt.x = sz.x * pt.x;
      }
      return true;
    };

    auto parse_coord_y = [&parse_real, sz, &expandable](chars & text,
      pointf &pt) -> bool {
      if (!parse_real(text, pt.y)) return false;
      if (*text == '%') {
        ++text;
        expandable = true;
        pt.y = sz.y * pt.y / 100.0f;
      }
      if (*text == '*') {
        ++text;
        expandable = true;
        pt.y = sz.y * pt.y;
      }
      return true;
    };

    auto parse_coord = [&parse_coord_x, &parse_coord_y, sz, &expandable](chars & text, pointf &pt) -> bool {
      if (!parse_coord_x(text, pt)) return false;
      if (!parse_coord_y(text, pt)) return false;
      return true;
    };

    while (!!d) {
      while (!!d && ::is_space(*d))
        ++d;
      if (command_chars.index_of(*d) >= 0) {
        last_command_char = d++;
        is_relative = (last_command_char >= 'a' && last_command_char <= 'z');
      }

      switch (last_command_char) {
      case 'E':
        expandable = true;
        break; // explicit E'xpandable path
      case 'M':
      case 'm':
      case 'L':
      case 'l':
        if (parse_coord(d, p1)) {
          if (is_relative)
            relate(p1);
          else
            expand(p1);

          if (last_command_char == 'M' || last_command_char == 'm') {
            // if( path-> )
            path->move_to(subpath_start = p1);
            last_command_char = 'l';
          }
          else
            path->line_to(p1);

          last2 = last = p1;
        }
        else
          goto PARSING_ERROR;
        break;

      case 'R':
        if (parse_coord(d, p1))
        {
          expand(p1);
          reference_point = p1;
          rp_active = true;
        }
        else
          goto PARSING_ERROR;
        break;
      case 'r':
        rp_active = false;
        break;
      case 'H':
      case 'h':
        if (parse_coord_x(d, p1)) {
          if (is_relative)
            relate_x(p1);
          else
            expand_x(p1);

          path->line_to(pointf(p1.x, last.y));

          last2.x = last.x = p1.x;
        }
        // else
        //    ++d;
        else
          goto PARSING_ERROR;
        break;

      case 'V':
      case 'v':
        if (parse_coord_y(d, p1)) {
          if (is_relative)
            relate_y(p1);
          else
            expand_y(p1);

          path->line_to(pointf(last.x, p1.y));

          last2.y = last.y = p1.y;
        }
        // else
        //    ++d;
        else
          goto PARSING_ERROR;
        break;

      case 'C':
      case 'c':
        if (parse_coord(d, p1) && parse_coord(d, p2) && parse_coord(d, p3)) {
          if (is_relative) {
            relate(p1);
            relate(p2);
            relate(p3);
          }
          else {
            expand(p1);
            expand(p2);
            expand(p3);
          }

          path->cubic_to(p3, p1, p2);

          last2 = p2;
          last = p3;
        }
        else
          goto PARSING_ERROR;
        break;

      case 'S':
      case 's':

        if (parse_coord(d, p1) && parse_coord(d, p3)) {
          if (is_relative) {
            relate(p1);
            relate(p3);
          }
          else {
            expand(p1);
            expand(p3);
          }

          p2 = last + (last - last2);
          path->cubic_to(p3, p2, p1);

          last2 = p1;
          last = p3;
        }
        else
          goto PARSING_ERROR;
        break;

      case 'Q':
      case 'q':
        if (parse_coord(d, p1) && parse_coord(d, p2)) {
          if (is_relative) {
            relate(p1);
            relate(p2);
          }
          else {
            expand(p1);
            expand(p2);
          }

          path->quadratic_to(p2, p1);

          last2 = p1;
          last = p2;
        }
        else
          goto PARSING_ERROR;
        break;

      case 'T':
      case 't':
        if (parse_coord(d, p1)) {
          if (is_relative)
            relate(p1);
          else
            expand(p1);

          p2 = last + (last - last2);
          path->quadratic_to(p1, p2);

          last2 = p2;
          last = p1;
        }
        else
          goto PARSING_ERROR;
        break;

      case 'A':
      case 'a':
        if (parse_coord(d, p1)) {
          float num;
          int   inum;

          if (parse_real(d, num)) {
            float angle = num / 180.0f * real_traits<float>::pi();

            if (parse_int(d, inum)) {
              bool largeArc = inum != 0;

              if (parse_int(d, inum)) {
                bool sweep = inum != 0;

                if (parse_coord(d, p2)) {
                  if (is_relative)
                    relate(p2);
                  else
                    expand(p2);

                  if (last != p2)
                    path->arc_to(p2, p1, angle, largeArc, sweep);

                  last2 = last;
                  last = p2;
                }
              }
            }
          }
        }
        else
          goto PARSING_ERROR;

        break;

      case 'Z':
      case 'z':
        path->close();
        last = last2 = subpath_start;

        while (!!d && ::is_space(*d))
          ++d;
        last_command_char = 'M';
        break;

      default: carry_on = false; break;
      }

      if (!carry_on) break;
    }
    return expandable;
  PARSING_ERROR:
    path = nullptr;
    expandable = false;
    return false;
  }


  gool::path *path_image::get_path(gool::application *for_app /* = nullptr*/,
                                   size for_sz /* = size()*/) const {
    if (path && is_expandable && for_sz != path_size) {
      path      = nullptr;
      path_size = for_sz;
    }
    else {
      path_size = path_size;
    }

    gool::application *used_app = for_app ? for_app : app();

    if (!path || (used_app != path_app)) {
      this->path                  = used_app->create_path();
      this->path_app              = used_app;
      is_expandable = parse(this->path, this->path_chars, path_size);
    }
    return this->path;
  }

  size path_image::dimension() const {
    get_path();
    rectf rc = path->bounds();
    if (is_expandable) return size(10, 10);
    return rc.size()/* + sizef(1.0f, 1.0f)*/;
  }

  void path_image::draw(graphics *gfx, rectf dst, rect src, byte opacity) {
    sizef dstsz = dst.size();
    get_path(gfx->app(), dstsz);

    rectf rc = path->bounds();

    sizef srcsz = is_expandable ? sizef(10, 10) : sizef(rc.size());

    if (!gfx->current_style) return;

    gool::state _(gfx);

    gfx->translate(dst.s);
    if (!dstsz.empty() && !srcsz.empty()) {
      if (!is_expandable) {
        float storke_width = setup_fill_and_stroke(gfx, gfx->current_element, gfx->current_style, dst.size(), false, opacity, srcsz.x / dstsz.x);
        gfx->scale(dstsz / (srcsz + sizef(storke_width)));
        gfx->translate(-rc.s + pointf(storke_width/2, storke_width/2));
      } else
        setup_fill_and_stroke(gfx, gfx->current_element, gfx->current_style, dst.size(), false, opacity);
      gfx->draw_path(path, true, true);
    }
  }

  image *path_image::mapped_left_to_right() {
#pragma TODO("path_image::mapping please")
    return this;
  }
  image *path_image::mapped_top_to_right() {
#pragma TODO("path_image::mapping please")
    return this;
  }

#include "gool-stock-image-names-ph.h"

  stock_image *stock_image::get(const tool::string &name) {
    const stock_image_def *def =
        stock_image_names::find_def(name, (uint)name.length());
    if (!def) return 0;

#define MAX_STOCK_IMAGES 28

    static handle<stock_image> images[200];

    if (!images[def->id]) {
      stock_image *si = new stock_image(def->id);
      images[def->id] = si;
      si->set_url(string::format("stock:%s", name.c_str()));
    }
    return images[def->id];
  }

  size stock_image::dimension() const {
    // size s = resolution_provider::desktop().pixels_per_dip(size(7));
    // s.x |= 1;
    // s.y |= 1;

    size s;

    switch (id % 100) // hollow/filled pairs
    {
    case 0: // arrow-left, 0
      s = size(3, 6);
      break;
    case 1: // arrow-right, 1
      s = size(3, 6);
      break;
      break;
    case 3: // arrow-up, 3
      s = size(6, 3);
      break;
    case 2: // arrow-down, 2
      s = size(6, 3);
      break;
    case 4: // arrow-ne, 4
      s = size(4, 4);
      break;
      break;
    case 5: // arrow-se, 5
      s = size(4, 4);
      break;
      break;
    case 6: // arrow-sw, 6
      s = size(4, 4);
      break;
      break;
    case 7: // arrow-nw, 7
      s = size(4, 4);
      break;
      break;
    case 8: // arrow-n, 8
      s = size(6, 3);
      break;
    case 9: // arrow-s, 9
      s = size(6, 3);
      break;
    case 10: // arrow-w, 10
      s = size(3, 6);
      break;
      break;
    case 11: // arrow-e, 11
      s = size(3, 6);
      break;
      break;
    case 12: // checkmark
      s = size(6, 6);
      break;
    case 13: // cross-x
      s = size(6, 6);
      break;
    case 14: // circle / disk
      s = size(6, 6);
      break;
    case 15: // frame / block
      s = size(6, 6);
      break;
    case 16: // chevron-right
    case 17: // chevron-left
      s = size(5, 6);
      break;
    case 18: // chevron-up
    case 19: // chevron-down
      s = size(6, 5);
      break;
    default: assert(false); break;
    }
    s += size(1, 1);
    return resolution_provider::desktop().pixels_per_dip(s);
  }

  image *stock_image::mapped_left_to_right() {
    tool::string name;

    switch (id % 100) // hollow/filled pairs
    {
    case 0: // arrow-left, 0
      name = "arrow-right";
      break;
    case 1: // arrow-right, 1
      name = "arrow-left";
      break;
    case 3: // arrow-up, 3
      break;
    case 2: // arrow-down, 2
      break;
    case 4: // arrow-ne, 4
      name = "arrow-nw";
      break;
    case 5: // arrow-se, 5
      name = "arrow-sw";
      break;
    case 6: // arrow-sw, 6
      name = "arrow-se";
      break;
    case 7: // arrow-nw, 7
      name = "arrow-ne";
      break;
    case 8: // arrow-n, 8
      break;
    case 9: // arrow-s, 9
      break;
    case 10: // arrow-w, 10
      name = "arrow-e";
      break;
    case 11: // arrow-e, 11
      name = "arrow-w";
      break;
    case 12: // checkmark
      //?
      return this;
    case 13:
      return this; // cross-x
    case 14:
      return this; // circle / disk
    case 15:
      return this; // frame / block
    case 16: name = "chevron-left"; break;
    case 17: name = "chevron-right"; break;
    case 18:
    case 19:
      return this; // chevron-up/down
    default: assert(false); break;
    }
    if (name.is_defined())
      return get(name);
    else
      return this;
  }
  image *stock_image::mapped_top_to_right() { return this; }

  void stock_image::draw(graphics *gfx, rectf dst, rect src, byte opacity) {
    if (!gfx->current_style) return;

    rectf r = dst;

    gool::state _(gfx);

    // if(id == 114)
    // id = id;

    float stroke_width = setup_fill_and_stroke(gfx, gfx->current_element, gfx->current_style,
                                               r.size(), id < 100, opacity);

    // argb c = gfx->get_text_color();
    polygonf vertices;

    /*    float dx = floorf((r.width() - stroke_width) / 4);
        float dy = floorf((r.height() - stroke_width) / 4);

        float dx1 = (r.width() - stroke_width) / 4 - stroke_width / 2;
        float dy1 = (r.height() - stroke_width) / 4 - stroke_width / 2;

        //int x4 = dst.width() / 4;

        const float m = 0.70710678f; */

    handle<gool::path> path = gfx->create_path();

    switch (id % 100) // paired hollow/filled
    {
    case 0: // arrow-left, 0
      r <<= sizef(stroke_width / 2, stroke_width / 2);
      vertices.add(r.pointOf(4));
      vertices.add(r.pointOf(9));
      vertices.add(r.pointOf(3));
      break;
    case 1: // arrow-right, 1
      r <<= sizef(stroke_width / 2, stroke_width / 2);
      vertices.add(r.pointOf(7));
      vertices.add(r.pointOf(6));
      vertices.add(r.pointOf(1));
      break;
    case 3: // arrow-up, 3
      r <<= sizef(stroke_width / 2, stroke_width / 2);
      vertices.add(r.pointOf(8));
      vertices.add(r.pointOf(3));
      vertices.add(r.pointOf(1));
      break;
    case 2: // arrow-down, 2
      r <<= sizef(stroke_width / 2, stroke_width / 2);
      vertices.add(r.pointOf(7));
      vertices.add(r.pointOf(9));
      vertices.add(r.pointOf(2));
      break;
    case 4: // arrow-ne, 4
      // r.origin.x = r.corner.x - (m*r.width());
      // r.corner.y = r.origin.y + (m*r.height());
      r <<= sizef(stroke_width / 2, stroke_width / 2);
      vertices.add(r.pointOf(7));
      vertices.add(r.pointOf(9));
      vertices.add(r.pointOf(3));
      break;
    case 5: // arrow-se, 5
      // r.origin.x = r.corner.x - (m*r.width());
      // r.origin.y = r.corner.y - (m*r.height());
      r <<= sizef(stroke_width / 2, stroke_width / 2);
      vertices.add(r.pointOf(9));
      vertices.add(r.pointOf(3));
      vertices.add(r.pointOf(1));
      break;
    case 6: // arrow-sw, 6
      // r.corner.x = r.origin.x + (m*r.width());
      // r.origin.y = r.corner.y - (m*r.height());
      r <<= sizef(stroke_width / 2, stroke_width / 2);
      vertices.add(r.pointOf(7));
      vertices.add(r.pointOf(3));
      vertices.add(r.pointOf(1));
      break;
    case 7: // arrow-nw, 7
      // r.corner.x = r.origin.x + (m*r.width());
      // r.corner.y = r.origin.y + (m*r.height());
      r <<= sizef(stroke_width / 2, stroke_width / 2);
      vertices.add(r.pointOf(7));
      vertices.add(r.pointOf(9));
      vertices.add(r.pointOf(1));
      break;
    case 8: // arrow-n, 8
      // r <<= sizef(0,dy1);
      r <<= sizef(stroke_width / 2, stroke_width / 2);
      vertices.add(r.pointOf(8));
      vertices.add(r.pointOf(3));
      vertices.add(r.pointOf(1));
      break;
    case 9: // arrow-s, 9
      // r <<= sizef(0,dy1);
      r <<= sizef(stroke_width / 2, stroke_width / 2);
      vertices.add(r.pointOf(7));
      vertices.add(r.pointOf(9));
      vertices.add(r.pointOf(2));
      break;
    case 10: // arrow-w, 10
      // r <<= sizef(dx1,0);
      r <<= sizef(stroke_width / 2, stroke_width / 2);
      vertices.add(r.pointOf(4));
      vertices.add(r.pointOf(9));
      vertices.add(r.pointOf(3));
      break;
    case 11: // arrow-e, 11
      // r <<= sizef(dx1,0);
      r <<= sizef(stroke_width / 2, stroke_width / 2);
      vertices.add(r.pointOf(7));
      vertices.add(r.pointOf(6));
      vertices.add(r.pointOf(1));
      break;

    case 12: // checkmark
    {
      float  d   = max(1.0f, (r.width() - 1) / 3);
      pointf org = pointf(r.s) + pointf(0, -1);
      vertices.add(org + pointf(0 * d, 1 * d));
      vertices.add(org + pointf(0 * d, 2 * d));
      vertices.add(org + pointf(1 * d, 3 * d));
      vertices.add(org + pointf(3 * d, 1 * d));
      vertices.add(org + pointf(3 * d, 0 * d));
      vertices.add(org + pointf(1 * d, 2 * d));
      vertices.add(org + pointf(0 * d, 1 * d));
      break;
    }

    case 13: // cross-x
    {
      pointf org = pointf(r.s);
      float  d   = max(1.0f, r.width() / 6);
      vertices.add(org + pointf(1 * d, 0 * d));
      vertices.add(org + pointf(3 * d, 2 * d));
      vertices.add(org + pointf(5 * d, 0 * d));
      vertices.add(org + pointf(6 * d, 1 * d));
      vertices.add(org + pointf(4 * d, 3 * d));
      vertices.add(org + pointf(6 * d, 5 * d));
      vertices.add(org + pointf(5 * d, 6 * d));
      vertices.add(org + pointf(3 * d, 4 * d));
      vertices.add(org + pointf(1 * d, 6 * d));
      vertices.add(org + pointf(0 * d, 5 * d));
      vertices.add(org + pointf(2 * d, 3 * d));
      vertices.add(org + pointf(0 * d, 1 * d));
      // vertices.add(org + pointf(1*d,0*d));
      break;
    }

    case 14: {
      r <<= stroke_width / 2;
      gfx->draw_ellipse(r.pointOf(5), sizef(r.width() / 2, r.height() / 2));
      return;
    }
    case 15: {
      r <<= stroke_width / 2;
      gfx->draw_rectangle(r.s, r.dimension());
      return;
    }

    case 16: // chevron-right
    {
      pointf org = pointf(r.s); // + pointf(0.5f,0.5f);
      float  d   = max(1.0f, r.width() / 5);
      vertices.add(org + pointf(1 * d, 0 * d));
      vertices.add(org + pointf(4 * d, 3 * d));
      vertices.add(org + pointf(1 * d, 6 * d));
      vertices.add(org + pointf(0 * d, 5 * d));
      vertices.add(org + pointf(2 * d, 3 * d));
      vertices.add(org + pointf(0 * d, 1 * d));
      break;
    }
    case 17: // chevron-left
    {
      pointf org = pointf(r.s); // + pointf(0.5f,0.5f);
      float  d   = max(1.0f, r.width() / 5);
      vertices.add(org + pointf(3 * d, 0 * d));
      vertices.add(org + pointf(0 * d, 3 * d));
      vertices.add(org + pointf(3 * d, 6 * d));
      vertices.add(org + pointf(4 * d, 5 * d));
      vertices.add(org + pointf(2 * d, 3 * d));
      vertices.add(org + pointf(4 * d, 1 * d));
      break;
    }

    case 18: // chevron-up
    {
      pointf org = pointf(r.s); // + pointf(0.5f,0.5f);
      float  d   = max(1.0f, r.width() / 6);
      vertices.add(org + pointf(0 * d, 3 * d));
      vertices.add(org + pointf(3 * d, 0 * d));
      vertices.add(org + pointf(6 * d, 3 * d));
      vertices.add(org + pointf(5 * d, 4 * d));
      vertices.add(org + pointf(3 * d, 2 * d));
      vertices.add(org + pointf(1 * d, 4 * d));
      break;
    }

    case 19: // chevron-down
    {
      pointf org = pointf(r.s); // + pointf(0.5f,0.5f);
      float  d   = max(1.0f, r.width() / 6);
      vertices.add(org + pointf(1 * d, 0 * d));
      vertices.add(org + pointf(3 * d, 2 * d));
      vertices.add(org + pointf(5 * d, 0 * d));
      vertices.add(org + pointf(6 * d, 1 * d));
      vertices.add(org + pointf(3 * d, 4 * d));
      vertices.add(org + pointf(0 * d, 1 * d));
      break;
    }

    default: assert(false); break;
    }
    // else switch( id ) {
    //
    //}

    if (path) {
      if (vertices.length()) path->set(vertices);
      if (id >= 100)
        gfx->draw_path(path, true, false);
      else
        gfx->draw_path(path, true, true);
    }
  }

  bitmap *bitmap_list::get(const ustring &name, bool allow_read_from_disk) {
    critical_section _(lock);
    bool             new_added = false;
    handle<bitmap> & bm        = bitmaps.get_ref(name, new_added);
    if (new_added) bm = create(name, allow_read_from_disk);
    return bm;
  }

  bitmap::bitmap(size sz, bool transparent, bool with_pixel_buf)
      : _has_alpha_bits(transparent), _dim(sz), transformed_cache(8),
        _generation(0)
#if defined(USE_GDI) || defined(USE_CGX)
        ,
        _used_generation(0)
#endif
  {
    _uid = (uint)++_uid_;
    if (with_pixel_buf)
      _pixels.length(_dim.x * _dim.y,
                     transparent ? argb(0, 0, 0, 0) : argb(0, 0, 0, 255));
    if (!sz.empty()) // empty size is a special case, see
                     // application::application
      this->link_after(app()->bitmap_list_head);
  }

  bitmap::bitmap(const bitmap *src)
      : _has_alpha_bits(src->_has_alpha_bits), _dim(src->_dim),
        _pixels(src->_pixels), transformed_cache(8), _generation(0)
#if defined(USE_GDI)
        ,
        _used_generation(0)
#endif
  {
    _uid = (uint)++_uid_;
    this->link_after(app()->bitmap_list_head);
  }

#ifdef WINDOWS
  bitmap::bitmap(const dib32 *pd, bool transparent)
      : _has_alpha_bits(transparent), _dim(pd->dim()), transformed_cache(8),
        _generation(0)
#if defined(USE_GDI)
        ,
        _used_generation(0)
#endif
  {
    _uid = ++_uid_;
    this->link_after(app()->bitmap_list_head);
    _pixels.reserve(_dim.x * _dim.y);
    for (int n = 0; n < _dim.y; ++n)
      _pixels.push(pd->row(n));
    premultiply();
  }

  bitmap *bitmap::create(const dib32 &src, bool transparent) {
    return new bitmap(&src, transparent);
  }

  bitmap::bitmap(HBITMAP hbmp, bool has_alpha)
      : _has_alpha_bits(has_alpha), transformed_cache(8), _generation(0)
#if defined(USE_GDI)
        ,
        _used_generation(0)
#endif
  {
    _uid = (uint)++_uid_;
    this->link_after(app()->bitmap_list_head);
    BITMAP def = {0};
    GetObject(hbmp, sizeof(def), &def);

    BITMAPINFO bmi              = {0};
    bmi.bmiHeader.biSize        = sizeof(bmi.bmiHeader);
    bmi.bmiHeader.biBitCount    = 32;
    bmi.bmiHeader.biHeight      = -def.bmHeight;
    bmi.bmiHeader.biWidth       = def.bmWidth;
    bmi.bmiHeader.biPlanes      = 1;
    bmi.bmiHeader.biCompression = BI_RGB;

    _dim.x = def.bmWidth;
    _dim.y = def.bmHeight;

    _pixels.size(def.bmWidth * def.bmHeight);
    {
      screen_dc sdc;
      GetDIBits(sdc, hbmp, 0, def.bmHeight, _pixels.head(), &bmi,
                DIB_RGB_COLORS);
    }

    if (!has_alpha) FOREACH(i, _pixels)
    _pixels[i].alfa = 0xff;

    // bool has_alpha = false;

    //_pixels.each([&has_alpha](argb& p) -> bool { if( p.alfa ) has_alpha =
    //true; return false; });  if( !has_alpha ) { _has_alpha_bits = false;
    //_pixels.each([&has_alpha](argb& p) -> bool { p.alfa = 0xff; return false;
    //});
    // FOREACH(i,_pixels) {
    //  _pixels[i].alfa = 0xff;
    //}
    //}
    // else
    //   premultiply();
  }
#endif

  handle<cursor> cursor::inherit = new cursor();

#ifdef WINDOWS

  cursor::~cursor() {
    if (hcur) DestroyCursor(hcur);
  }

  cursor *cursor::system(uint id) {
    static uint_ptr cursor_ids[] = {
        (uint_ptr)IDC_ARROW,       // 0
        (uint_ptr)IDC_IBEAM,       // 1
        (uint_ptr)IDC_WAIT,        // 2
        (uint_ptr)IDC_CROSS,       // 3
        (uint_ptr)IDC_UPARROW,     // 4
        (uint_ptr)IDC_SIZENWSE,    // 5
        (uint_ptr)IDC_SIZENESW,    // 6
        (uint_ptr)IDC_SIZEWE,      // 7
        (uint_ptr)IDC_SIZENS,      // 8
        (uint_ptr)IDC_SIZEALL,     // 9
        (uint_ptr)IDC_NO,          // 10
        (uint_ptr)IDC_APPSTARTING, // 11
        (uint_ptr)IDC_HELP,        // 12
        (uint_ptr)IDC_HAND,        // 13
        (uint_ptr)L"DRAG-COPY",    // 14
        (uint_ptr)L"DRAG-MOVE",    // 15
        (uint_ptr)0,               // 16 - none

    };
    static handle<cursor> cursors[items_in(cursor_ids)];
    if (id < 14) {
      if (!cursors[id]) {
        cursors[id] = new cursor();
        cursors[id]->id = id;
        cursors[id]->hcur =
            cursor_ids[id] ? LoadCursor(NULL, MAKEINTRESOURCE(cursor_ids[id]))
                           : NULL;
      }
      return cursors[id];
    }
    else if (id < items_in(cursor_ids)) {

      if (!cursors[id]) {
        cursors[id] = new cursor();
        cursors[id]->id = id;
        cursors[id]->hcur = cursor_ids[id] ? LoadCursorW(THIS_HINSTANCE, (LPCWSTR)cursor_ids[id]) : NULL;
      }
      return cursors[id];
    }

    return 0;
  }

  cursor *cursor::from_data(bytes data, string url) {
    char tmppath[MAX_PATH];
    char tmpname[MAX_PATH];
    GetTempPathA(MAX_PATH, tmppath);
    GetTempFileNameA(tmppath, "cur", 0, tmpname);

    FILE *f = 0;
    fopen_s(&f, tmpname, "wb+");
    if (!f) return 0;
    fwrite(data.start, data.length, 1, f);
    fclose(f);

    HCURSOR h = LoadCursorFromFileA(tmpname);

    ::remove(tmpname);

    // assert(h);
    if (!h) {
      auto img = image::create(data, url);
      if (!img) return nullptr;
      if (!img->is_bitmap()) return nullptr;
      cursor *cur = new cursor();
      cur->hcur = img.ptr_of<bitmap>()->create_win_icon(size());
      cur->url = url;
      return cur;
    }

    cursor *out = new cursor();
    out->hcur   = h;
    out->url    = url;
    return out;
  }

  cursor *cursor::from_bitmap(bitmap *bmp, string url, size offset) {
    cursor *cur = new cursor();
    cur->hcur   = bmp->create_win_icon(offset);
    cur->url    = url;
    return cur;
  }

#endif

} // namespace gool
