#pragma once

#include "gool-image.h"

namespace gool {

  // static image_filter* construct( wchars name, tool::slice<tool::value>
  // params);

  extern void cvt_grey(bitmap *dst, argb c);
  extern void cvt_grey(bitmap *dst, argb c0, argb c1, argb c2, argb c3,
                       argb c4);
  extern void cvt_cbg(bitmap *dst, float contrast, float brightness,
                      float gamma_exponent);
  extern void cvt_colorize(bitmap *dst, argb c);
  extern void cvt_hue(bitmap *dst, float hue);
  extern void cvt_saturation(bitmap *dst, float sat);
  extern void cvt_opacity(bitmap *dst, float opacity);
  extern void cvt_flip_x(bitmap *dst);
  extern void cvt_flip_y(bitmap *dst);

  struct image_filter_none : public image_filter {
    virtual ICT_TYPE      type() const { return ICT_NONE; }
    virtual image_filter *clone() const { return new image_filter_none(); }

    virtual uint this_value_hash() const {
      return 0;
    }

    virtual bool this_value_is_equal(const image_filter *to) const {
      return true;
    }
    virtual void this_apply(bitmap * /*src*/) const {}
    virtual void emit(wchar_stream_o &out) const { out << WCHARS("none"); }
  };

  struct image_filter_hue : public image_filter {
    float                 hue; // 0...360
    virtual ICT_TYPE      type() const { return ICT_HUE; }
    virtual image_filter *clone() const { return new image_filter_hue(*this); }

    virtual uint this_value_hash() const {
      return hash_value(hue);
    }

    virtual bool this_value_is_equal(const image_filter *to) const {
      return to->type() == ICT_HUE &&
             static_cast<const image_filter_hue *>(to)->hue == hue;
    }
    virtual void this_apply(bitmap *dst) const { cvt_hue(dst, hue); }
    virtual void emit(wchar_stream_o &out) const {
      out << ustring::format(W("hue(%d)"), int(hue));
    }
  };

  struct image_filter_saturation : public image_filter {
    float                 saturation; // 0...360
    virtual ICT_TYPE      type() const { return ICT_SATURATION; }
    virtual image_filter *clone() const {
      return new image_filter_saturation(*this);
    }

    virtual uint this_value_hash() const {
      return hash_value(saturation);
    }

    virtual bool this_value_is_equal(const image_filter *to) const {
      return static_cast<const image_filter_saturation *>(to)->saturation == saturation;
    }
    virtual void this_apply(bitmap *dst) const {
      cvt_saturation(dst, saturation);
    }
    virtual void emit(wchar_stream_o &out) const {
      out << ustring::format(W("saturation(%f)"), saturation);
    }
  };

  struct image_filter_color_t : public image_filter
  {
    argb                  src; // 0...360

    virtual uint this_value_hash() const {
      uint h = hash_value(src);
      return h;
    }
    virtual bool this_value_is_equal(const image_filter *to) const {
      return static_cast<const image_filter_color_t *>(to)->src == src;
    }
  };

  struct image_filter_colorize : public image_filter_color_t {
    virtual ICT_TYPE      type() const { return ICT_COLORIZE; }
    virtual image_filter *clone() const {
      return new image_filter_colorize(*this);
    }

    virtual void this_apply(bitmap *dst) const { cvt_colorize(dst, src); }
    virtual void emit(wchar_stream_o &out) const {
      out << WCHARS("colorize(") << src.to_string() << WCHARS(")");
    }
  };

  struct image_filter_opacity : public image_filter {
    float                 opacity; // 0...1
    virtual ICT_TYPE      type() const { return ICT_OPACITY; }
    virtual image_filter *clone() const {
      return new image_filter_opacity(*this);
    }

    virtual uint this_value_hash() const {
      return hash_value(opacity);
    }

    virtual bool this_value_is_equal(const image_filter *to) const {
      return to->type() == ICT_OPACITY &&
             static_cast<const image_filter_opacity *>(to)->opacity == opacity;
    }
    virtual void this_apply(bitmap *dst) const { cvt_opacity(dst, opacity); }
    virtual void emit(wchar_stream_o &out) const {
      out << ustring::format(W("opacity(%f)"), opacity);
    }
  };

  struct image_filter_color_schema_5 : public image_filter {
    argb clr0, clr1, clr2, clr3, clr4;

    virtual ICT_TYPE      type() const { return ICT_COLOR_SCHEMA; }
    virtual uint          sub_type() const { return 5; }
    virtual image_filter *clone() const {
      return new image_filter_color_schema_5(*this);
    }

    virtual uint this_value_hash() const {
      uint h = hash_value(clr0);
      hash_combine(h, hash_value(clr1));
      hash_combine(h, hash_value(clr2));
      hash_combine(h, hash_value(clr3));
      hash_combine(h, hash_value(clr4));
      return h;
    }
    
    virtual bool this_value_is_equal(const image_filter *to) const {
      return static_cast<const image_filter_color_schema_5 *>(to)->clr0 ==
                 clr0 &&
             static_cast<const image_filter_color_schema_5 *>(to)->clr1 ==
                 clr1 &&
             static_cast<const image_filter_color_schema_5 *>(to)->clr2 ==
                 clr2 &&
             static_cast<const image_filter_color_schema_5 *>(to)->clr3 ==
                 clr3 &&
             static_cast<const image_filter_color_schema_5 *>(to)->clr4 == clr4;
    }
    virtual void this_apply(bitmap *dst) const {
      cvt_grey(dst, clr0, clr1, clr2, clr3, clr4);
    }
    virtual void emit(wchar_stream_o &out) const {
      out << WCHARS("color-schema(") << clr0.to_string() << WCHARS(",")
          << clr1.to_string() << WCHARS(",") << clr2.to_string() << WCHARS(",")
          << clr3.to_string() << WCHARS(",") << clr4.to_string() << WCHARS(")");
    }
  };

  struct image_filter_color_schema_1 : public image_filter_color_t {

    virtual ICT_TYPE      type() const { return ICT_COLOR_SCHEMA; }
    virtual uint          sub_type() const { return 1; }
    virtual image_filter *clone() const {
      return new image_filter_color_schema_1(*this);
    }

    virtual void this_apply(bitmap *dst) const { cvt_grey(dst, src); }
    virtual void emit(wchar_stream_o &out) const {
      out << WCHARS("color-schema(") << src.to_string() << WCHARS(")");
    }
  };

  struct image_filter_contrast_brightness_gamma : public image_filter {
    float contrast;
    float brightness;
    float gamma_exponent;

    image_filter_contrast_brightness_gamma()
        : contrast(0.5f), brightness(0.5f), gamma_exponent(1.0f) {
      assert(int(contrast * 256) == 128);
      assert(int(brightness * 256) - 128 == 0);
    }

    virtual image_filter *clone() const {
      return new image_filter_contrast_brightness_gamma(*this);
    }
        
    virtual ICT_TYPE type() const { return ICT_CBG; }
    virtual uint     sub_type() const { return 0; }

    virtual uint this_value_hash() const override {
      uint h = hash_value(contrast);
      hash_combine(h, hash_value(brightness));
      hash_combine(h, hash_value(gamma_exponent));
      return h;
    }

    virtual bool this_value_is_equal(const image_filter *to) const override {
      return to->type() == ICT_CBG &&
             static_cast<const image_filter_contrast_brightness_gamma *>(to)
                     ->contrast == contrast &&
             static_cast<const image_filter_contrast_brightness_gamma *>(to)
                     ->brightness == brightness &&
             static_cast<const image_filter_contrast_brightness_gamma *>(to)
                     ->gamma_exponent == gamma_exponent;
    }
    virtual void this_apply(bitmap *dst) const {
      cvt_cbg(dst, contrast, brightness, gamma_exponent);
    }
    virtual void emit(wchar_stream_o &out) const {
      out << WCHARS("contrast-brightness-gamma(") << ftow(contrast)
          << WCHARS(",") << ftow(brightness) << WCHARS(",")
          << ftow(gamma_exponent) << WCHARS(")");
    }
  };

  struct image_filter_contrast : public image_filter_contrast_brightness_gamma {
    virtual void emit(wchar_stream_o &out) const {
      out << WCHARS("contrast(") << ftow(contrast) << WCHARS(")");
    }
    virtual image_filter *clone() const {
      return new image_filter_contrast(*this);
    }
  };
  struct image_filter_brightness
      : public image_filter_contrast_brightness_gamma {
    virtual void emit(wchar_stream_o &out) const {
      out << WCHARS("brightness(") << ftow(brightness) << WCHARS(")");
    }
    virtual image_filter *clone() const {
      return new image_filter_brightness(*this);
    }
  };
  struct image_filter_gamma : public image_filter_contrast_brightness_gamma {
    virtual void emit(wchar_stream_o &out) const {
      out << WCHARS("gamma(") << ftow(gamma_exponent) << WCHARS(")");
    }
    virtual image_filter *clone() const {
      return new image_filter_gamma(*this);
    }
  };

  struct image_filter_flip_x : public image_filter {
    virtual ICT_TYPE      type() const { return ICT_FLIP_X; }
    virtual image_filter *clone() const {
      return new image_filter_flip_x(*this);
    }

    virtual uint this_value_hash() const {
      return 0;
    }

    virtual bool this_value_is_equal(const image_filter *to) const {
      return to->type() == ICT_FLIP_X;
    }
    virtual void this_apply(bitmap *dst) const { cvt_flip_x(dst); }
    virtual void emit(wchar_stream_o &out) const { out << WCHARS("flip-x()"); }
  };

  struct image_filter_flip_y : public image_filter {
    virtual ICT_TYPE      type() const { return ICT_FLIP_Y; }
    virtual image_filter *clone() const {
      return new image_filter_flip_y(*this);
    }

    virtual uint this_value_hash() const {
      return 0;
    }

    virtual bool this_value_is_equal(const image_filter *to) const {
      return to->type() == ICT_FLIP_Y;
    }
    virtual void this_apply(bitmap *dst) const { cvt_flip_y(dst); }
    virtual void emit(wchar_stream_o &out) const { out << WCHARS("flip-y()"); }
  };

  /*inline image_filter* image_filter::construct(const function_value* fv)
  {

  }*/

  namespace bmagic 
  {
    // SkAlphaMulAlpha
    inline uint mul_bytes(int a, int b) {
      unsigned prod = a * b + 128;
      return (prod + (prod >> 8)) >> 8;
    }

    inline uint div_255_round(unsigned prod) {
      prod += 128;
      return (prod + (prod >> 8)) >> 8;
    }
        
    static inline int clamp_div_255round(int prod) {
      if (prod <= 0) {
        return 0;
      }
      else if (prod >= 255 * 255) {
        return 255;
      }
      else {
        return div_255_round(prod);
      }
    }

    inline int srcover_byte(int a, int b) {
      return a + b - mul_bytes(a, b);
    }

    inline int clamp_signed_byte(int n) {
      if (n < 0) {
        n = 0;
      }
      else if (n > 255) {
        n = 255;
      }
      return n;
    }


  }
  
  struct image_filter_multiply : public image_filter_color_t {

    virtual ICT_TYPE      type() const { return ICT_MULTIPLY; }
    virtual image_filter *clone() const { return new image_filter_multiply(*this); }

    virtual void this_apply(bitmap *dst) const { 
      // borrowed from Skia
      // kMultiply_Mode
      // B(Cb, Cs) = Cb x Cs
      // multiply uses its own version of blendfunc_byte because sa and da are not needed
      auto blendfunc_multiply_byte = [](int sc, int dc, int sa, int da) {
        return bmagic::clamp_div_255round(sc * (255 - da) + dc * (255 - sa) + sc * dc);
      };

      auto xpixel = [this,blendfunc_multiply_byte](argb &dst) -> argb {
        int sa = src.alfa;
        int da = dst.alfa;
        int a = bmagic::srcover_byte(sa, da);
        int r = blendfunc_multiply_byte(src.red, dst.red, sa, da);
        int g = blendfunc_multiply_byte(src.green, dst.green, sa, da);
        int b = blendfunc_multiply_byte(src.blue, dst.blue, sa, da);
        return argb(r, g, b, a);
      };
      dst->foreach_pixel(xpixel);
    }
    virtual void emit(wchar_stream_o &out) const { out << WCHARS("multiply(") << src.to_string() << WCHARS(")");  }
  };

  struct image_filter_screen : public image_filter_color_t {
    
    virtual ICT_TYPE      type() const { return ICT_SCREEN; }
    virtual image_filter *clone() const { return new image_filter_screen(*this); }

    virtual void this_apply(bitmap *dst) const {

      auto xpixel = [this](argb &dst) -> argb {
        int a = bmagic::srcover_byte(src.alfa, dst.alfa);
        int r = bmagic::srcover_byte(src.red, dst.red);
        int g = bmagic::srcover_byte(src.green, dst.green);
        int b = bmagic::srcover_byte(src.blue, dst.blue);
        return argb(r, g, b, a);
      };
      dst->foreach_pixel(xpixel);
    }
    virtual void emit(wchar_stream_o &out) const { out << WCHARS("screen(") << src.to_string() << WCHARS(")"); }
  };
  
  struct image_filter_overlay : public image_filter_color_t {

    virtual ICT_TYPE      type() const { return ICT_OVERLAY; }
    virtual image_filter *clone() const { return new image_filter_overlay(*this); }

    virtual void this_apply(bitmap *dst) const {

      // kOverlay_Mode
      auto overlay_byte = [](int sc, int dc, int sa, int da) -> int {
        int tmp = sc * (255 - da) + dc * (255 - sa);
        int rc;
        if (2 * dc <= da) {
          rc = 2 * sc * dc;
        }
        else {
          rc = sa * da - 2 * (da - dc) * (sa - sc);
        }
        return bmagic::clamp_div_255round(rc + tmp);
      };

      auto xpixel = [this, overlay_byte](argb &dst) -> argb {
        int sa = src.alfa;
        int da = dst.alfa;
        int a = bmagic::srcover_byte(sa, da);
        int r = overlay_byte(src.red, dst.red, sa, da);
        int g = overlay_byte(src.green, dst.green, sa, da);
        int b = overlay_byte(src.blue, dst.blue, sa, da);
        return argb(r, g, b, a);
      };
      dst->foreach_pixel(xpixel);
    }
    virtual void emit(wchar_stream_o &out) const { out << WCHARS("overlay(") << src.to_string() << WCHARS(")"); }
  };

  struct image_filter_darken : public image_filter_color_t {

    virtual ICT_TYPE      type() const { return ICT_DARKEN; }
    virtual image_filter *clone() const { return new image_filter_darken(*this); }

    virtual void this_apply(bitmap *dst) const {

      // kDarken_Mode
      auto darken_byte = [](int sc, int dc, int sa, int da) -> int {
        int sd = sc * da;
        int ds = dc * sa;
        if (sd < ds) {
          // srcover
          return sc + dc - bmagic::div_255_round(ds);
        }
        else {
          // dstover
          return dc + sc - bmagic::div_255_round(sd);
        }
      };

      auto xpixel = [this, darken_byte](argb &dst) -> argb {
        int sa = src.alfa;
        int da = dst.alfa;
        int a = bmagic::srcover_byte(sa, da);
        int r = darken_byte(src.red, dst.red, sa, da);
        int g = darken_byte(src.green, dst.green, sa, da);
        int b = darken_byte(src.blue, dst.blue, sa, da);
        return argb(r, g, b, a);
      };
      dst->foreach_pixel(xpixel);
    }
    virtual void emit(wchar_stream_o &out) const { out << WCHARS("darken(") << src.to_string() << WCHARS(")"); }
  };

  struct image_filter_lighten : public image_filter_color_t {
    
    virtual ICT_TYPE      type() const { return ICT_LIGHTEN; }
    virtual image_filter *clone() const { return new image_filter_lighten(*this); }

    virtual void this_apply(bitmap *dst) const {
      // kLighten_Mode
      auto lighten_byte = [](int sc, int dc, int sa, int da) -> int {
        int sd = sc * da;
        int ds = dc * sa;
        if (sd > ds) {
          // srcover
          return sc + dc - bmagic::div_255_round(ds);
        }
        else {
          // dstover
          return dc + sc - bmagic::div_255_round(sd);
        }
      };

      auto xpixel = [this, lighten_byte](argb &dst) -> argb {
        int sa = src.alfa;
        int da = dst.alfa;
        int a = bmagic::srcover_byte(sa, da);
        int r = lighten_byte(src.red, dst.red, sa, da);
        int g = lighten_byte(src.green, dst.green, sa, da);
        int b = lighten_byte(src.blue, dst.blue, sa, da);
        return argb(r, g, b, a);
      };
      dst->foreach_pixel(xpixel);
    }
    virtual void emit(wchar_stream_o &out) const { out << WCHARS("lighten(") << src.to_string() << WCHARS(")"); }
  };

  struct image_filter_dodge : public image_filter_color_t {

    virtual ICT_TYPE      type() const { return ICT_DODGE; }
    virtual image_filter *clone() const { return new image_filter_dodge(*this); }

    virtual void this_apply(bitmap *dst) const {
      // kColorDodge_Mode
      auto colordodge_byte = [](int sc, int dc, int sa, int da) -> int {
        int diff = sa - sc;
        int rc;
        if (0 == dc) {
          return bmagic::mul_bytes(sc, 255 - da);
        }
        else if (0 == diff) {
          rc = sa * da + sc * (255 - da) + dc * (255 - sa);
        }
        else {
          diff = dc * sa / diff;
          rc = sa * ((da < diff) ? da : diff) + sc * (255 - da) + dc * (255 - sa);
        }
        return bmagic::clamp_div_255round(rc);
      };

      auto xpixel = [this, colordodge_byte](argb &dst) -> argb {
        int sa = src.alfa;
        int da = dst.alfa;
        int a = bmagic::srcover_byte(sa, da);
        int r = colordodge_byte(src.red, dst.red, sa, da);
        int g = colordodge_byte(src.green, dst.green, sa, da);
        int b = colordodge_byte(src.blue, dst.blue, sa, da);
        return argb(r, g, b, a);
      };
      dst->foreach_pixel(xpixel);
    }
    virtual void emit(wchar_stream_o &out) const { out << WCHARS("dodge(") << src.to_string() << WCHARS(")"); }
  };

  struct image_filter_color_burn : public image_filter_color_t {

    virtual ICT_TYPE      type() const { return ICT_COLOR_BURN; }
    virtual image_filter *clone() const { return new image_filter_color_burn(*this); }

    virtual void this_apply(bitmap *dst) const {

      auto colorburn_byte = [](int sc, int dc, int sa, int da) -> int {
        int rc;
        if (dc == da) {
          rc = sa * da + sc * (255 - da) + dc * (255 - sa);
        }
        else if (0 == sc) {
          return bmagic::mul_bytes(dc, 255 - sa);
        }
        else {
          int tmp = (da - dc) * sa / sc;
          rc = sa * (da - ((da < tmp) ? da : tmp))
            + sc * (255 - da) + dc * (255 - sa);
        }
        return bmagic::clamp_div_255round(rc);
      };

      auto xpixel = [this, colorburn_byte](argb &dst) -> argb {
        int sa = src.alfa;
        int da = dst.alfa;
        int a = bmagic::srcover_byte(sa, da);
        int r = colorburn_byte(src.red, dst.red, sa, da);
        int g = colorburn_byte(src.green, dst.green, sa, da);
        int b = colorburn_byte(src.blue, dst.blue, sa, da);
        return argb(r, g, b, a);
      };
      dst->foreach_pixel(xpixel);
    }
    virtual void emit(wchar_stream_o &out) const { out << WCHARS("color-burn(") << src.to_string() << WCHARS(")"); }
  };

  struct image_filter_hard_light : public image_filter_color_t {
    
    virtual ICT_TYPE      type() const { return ICT_HARD_LIGHT; }
    virtual image_filter *clone() const { return new image_filter_hard_light(*this); }

    virtual void this_apply(bitmap *dst) const {

      // kHardLight_Mode
      auto hardlight_byte = [](int sc, int dc, int sa, int da) -> int {
        int rc;
        if (2 * sc <= sa) {
          rc = 2 * sc * dc;
        }
        else {
          rc = sa * da - 2 * (da - dc) * (sa - sc);
        }
        return bmagic::clamp_div_255round(rc + sc * (255 - da) + dc * (255 - sa));
      };

      auto xpixel = [this, hardlight_byte](argb &dst) -> argb {
        int sa = src.alfa;
        int da = dst.alfa;
        int a = bmagic::srcover_byte(sa, da);
        int r = hardlight_byte(src.red, dst.red, sa, da);
        int g = hardlight_byte(src.green, dst.green, sa, da);
        int b = hardlight_byte(src.blue, dst.blue, sa, da);
        return argb(r, g, b, a);
      };
      dst->foreach_pixel(xpixel);
    }
    virtual void emit(wchar_stream_o &out) const { out << WCHARS("hard-light(") << src.to_string() << WCHARS(")"); }
  };

  struct image_filter_soft_light : public image_filter_color_t {

    virtual ICT_TYPE      type() const { return ICT_SOFT_LIGHT; }
    virtual image_filter *clone() const { return new image_filter_soft_light(*this); }

    virtual void this_apply(bitmap *dst) const {

      auto sqrt_bits = [](int x, int count) -> int {
        assert(x >= 0 && count > 0 && (unsigned)count <= 30);

        uint    root = 0;
        uint    remHi = 0;
        uint    remLo = x;

        do {
          root <<= 1;

          remHi = (remHi << 2) | (remLo >> 30);
          remLo <<= 2;

          uint testDiv = (root << 1) + 1;
          if (remHi >= testDiv) {
            remHi -= testDiv;
            root++;
          }
        } while (--count >= 0);

        return root;
      };


      // returns 255 * sqrt(n/255)
      auto sqrt_unit_byte = [sqrt_bits](int n) {
        return sqrt_bits(n, 15 + 4);
      };

      // kSoftLight_Mode
      auto softlight_byte = [sqrt_unit_byte](int sc, int dc, int sa, int da) -> int {
        int m = da ? dc * 256 / da : 0;
        int rc;
        if (2 * sc <= sa) {
          rc = dc * (sa + ((2 * sc - sa) * (256 - m) >> 8));
        }
        else if (4 * dc <= da) {
          int tmp = (4 * m * (4 * m + 256) * (m - 256) >> 16) + 7 * m;
          rc = dc * sa + (da * (2 * sc - sa) * tmp >> 8);
        }
        else {
          int tmp = sqrt_unit_byte(m) - m;
          rc = dc * sa + (da * (2 * sc - sa) * tmp >> 8);
        }
        return bmagic::clamp_div_255round(rc + sc * (255 - da) + dc * (255 - sa));
      };

      auto xpixel = [this, softlight_byte](argb &dst) -> argb {
        int sa = src.alfa;
        int da = dst.alfa;
        int a = bmagic::srcover_byte(sa, da);
        int r = softlight_byte(src.red, dst.red, sa, da);
        int g = softlight_byte(src.green, dst.green, sa, da);
        int b = softlight_byte(src.blue, dst.blue, sa, da);
        return argb(r, g, b, a);
      };
      dst->foreach_pixel(xpixel);
    }
    virtual void emit(wchar_stream_o &out) const { out << WCHARS("soft-light(") << src.to_string() << WCHARS(")"); }
  };

  struct image_filter_difference : public image_filter_color_t {

    virtual ICT_TYPE      type() const { return ICT_DIFFERNCE; }
    virtual image_filter *clone() const { return new image_filter_difference(*this); }

    virtual void this_apply(bitmap *dst) const {

      // kDifference_Mode
      auto difference_byte = [](int sc, int dc, int sa, int da) -> int {
        int tmp = std::min(sc * da, dc * sa);
        return bmagic::clamp_signed_byte(sc + dc - 2 * bmagic::div_255_round(tmp));
      };

      auto xpixel = [this, difference_byte](argb &dst) -> argb {
        int sa = src.alfa;
        int da = dst.alfa;
        int a = bmagic::srcover_byte(sa, da);
        int r = difference_byte(src.red, dst.red, sa, da);
        int g = difference_byte(src.green, dst.green, sa, da);
        int b = difference_byte(src.blue, dst.blue, sa, da);
        return argb(r, g, b, a);
      };
      dst->foreach_pixel(xpixel);
    }
    virtual void emit(wchar_stream_o &out) const { out << WCHARS("difference(") << src.to_string() << WCHARS(")"); }
  };

  struct image_filter_exclusion : public image_filter_color_t {

    virtual ICT_TYPE      type() const { return ICT_EXCLUSION; }
    virtual image_filter *clone() const { return new image_filter_exclusion(*this); }

    virtual void this_apply(bitmap *dst) const {

      // kExclusion_Mode
      auto exclusion_byte = [](int sc, int dc, int, int) -> int {
        int r = 255 * (sc + dc) - 2 * sc * dc;
        return bmagic::clamp_div_255round(r);
      };

      auto xpixel = [this, exclusion_byte](argb &dst) -> argb {
        int sa = src.alfa;
        int da = dst.alfa;
        int a = bmagic::srcover_byte(sa, da);
        int r = exclusion_byte(src.red, dst.red, sa, da);
        int g = exclusion_byte(src.green, dst.green, sa, da);
        int b = exclusion_byte(src.blue, dst.blue, sa, da);
        return argb(r, g, b, a);
      };
      dst->foreach_pixel(xpixel);
    }
    virtual void emit(wchar_stream_o &out) const { out << WCHARS("exclusion(") << src.to_string() << WCHARS(")"); }
  };

  /*struct image_filter_color : public image_filter_color_t {

    virtual ICT_TYPE      type() const { return ICT_COLOR; }
    virtual image_filter *clone() const { return new image_filter_color(*this); }

    virtual void this_apply(bitmap *dst) const {

      // kExclusion_Mode
      auto exclusion_byte = [](int sc, int dc, int, int) -> int {
        int r = 255 * (sc + dc) - 2 * sc * dc;
        return bmagic::clamp_div_255round(r);
      };

      auto xpixel = [this, exclusion_byte](argb &dst) -> argb {
        int sa = src.alfa;
        int da = dst.alfa;
        int a = bmagic::srcover_byte(sa, da);
        int r = exclusion_byte(src.red, dst.red, sa, da);
        int g = exclusion_byte(src.green, dst.green, sa, da);
        int b = exclusion_byte(src.blue, dst.blue, sa, da);
        return argb(r, g, b, a);
      };
      dst->foreach_pixel(xpixel);
    }
    virtual void emit(wchar_stream_o &out) const { out << WCHARS("exclusion(") << src.to_string() << WCHARS(")"); }
  };*/


} // namespace gool
