#pragma once

#include "gool-types.h"

namespace gool {

  enum COLORSPACE {
    COLORSPACE_UNKNOWN,
    COLORSPACE_YV12,
    COLORSPACE_IYUV, // a.k.a. I420
    COLORSPACE_NV12,
    COLORSPACE_YUY2,
    COLORSPACE_RGB24,
    COLORSPACE_RGB555,
    COLORSPACE_RGB565,
    COLORSPACE_RGB32, // with alpha, sic!
    COLORSPACE_BGR32  // with alpha premultiplied, sic!
  };

  class color_space_converter : public resource {
  public:
    color_space_converter(size sz) : width(sz.x), height(sz.y) {}
    ~color_space_converter() {}

    virtual bool is_rgb32() const { return false; }

    virtual void convert_to_rgb32(bytes input, argb *output) {
      assert(false);
      target(output, width * height).xcopy(input.start, input.length);
    }

    static color_space_converter *factory(COLORSPACE cspace, size sz);

    size dim() const { return size(width, height); }
    virtual size_t pixel_size() const { return 0; } // number of bytes in pixel representation

  protected:
    int width;
    int height;
  };

  class yv_base_converter : public color_space_converter {
  public:
    yv_base_converter(size sz) : color_space_converter(sz) {
      int crv, cbu, cgu, cgv;
      int i, ind;

      crv = 104597;
      cbu = 132201;
      cgu = 25675;
      cgv = 53279;

      for (i = 0; i < 256; i++) {
        crv_tab[i]   = (i - 128) * crv;
        cbu_tab[i]   = (i - 128) * cbu;
        cgu_tab[i]   = (i - 128) * cgu;
        cgv_tab[i]   = (i - 128) * cgv;
        tab_76309[i] = 76309 * (i - 16);
      }

      for (i = 0; i < 384; i++)
        clp[i] = 0;
      ind = 384;
      for (i = 0; i < 256; i++)
        clp[ind++] = byte(i);
      ind = 640;
      for (i = 0; i < 384; i++)
        clp[ind++] = 255;

      u_plane_pos = width * height;
      v_plane_pos = u_plane_pos + u_plane_pos / 4;
    }
    ~yv_base_converter() {}

    virtual void convert_to_rgb32(bytes input, argb *output) = 0;

    // static color_space_converter* factory(const GUID mediaType, size sz);

  protected:
    size          dim;
    int           u_plane_pos;
    int           v_plane_pos;
    int           crv_tab[256];
    int           cbu_tab[256];
    int           cgu_tab[256];
    int           cgv_tab[256];
    int           tab_76309[256];
    unsigned char clp[1024];
  };

  //#define clamp(x) max(min(255, x), 0)

  class YUV420_space_converter : public yv_base_converter {
  public:
    YUV420_space_converter(size sz) : yv_base_converter(sz) {}

    virtual void convert_to_rgb32(bytes input, argb *output) {
      const byte *src0 = input.start;
      const byte *src1 = input.start + v_plane_pos;
      const byte *src2 = input.start + u_plane_pos;
      byte *      dst  = (byte *)output;

      int         y1, y2, u, v;
      const byte *py1, *py2;
      int         i, j, c1, c2, c3, c4;
      byte *      d1, *d2;

      py1 = src0;
      py2 = py1 + width;
      d1  = dst;
      d2  = d1 + 4 * width;
      for (j = 0; j < height; j += 2) {
        for (i = 0; i < width; i += 2) {
          u = *src1++;
          v = *src2++;

          c1 = crv_tab[v];
          c2 = cgu_tab[u];
          c3 = cgv_tab[v];
          c4 = cbu_tab[u];

          // up-left
          y1    = tab_76309[*py1++];
          *d1++ = clp[384 + ((y1 + c1) >> 16)];
          *d1++ = clp[384 + ((y1 - c2 - c3) >> 16)];
          *d1++ = clp[384 + ((y1 + c4) >> 16)];
          *d1++ = 0xff;

          // down-left
          y2    = tab_76309[*py2++];
          *d2++ = clp[384 + ((y2 + c1) >> 16)];
          *d2++ = clp[384 + ((y2 - c2 - c3) >> 16)];
          *d2++ = clp[384 + ((y2 + c4) >> 16)];
          *d2++ = 0xff;

          // up-right
          y1    = tab_76309[*py1++];
          *d1++ = clp[384 + ((y1 + c1) >> 16)];
          *d1++ = clp[384 + ((y1 - c2 - c3) >> 16)];
          *d1++ = clp[384 + ((y1 + c4) >> 16)];
          *d1++ = 0xff;

          // down-right
          y2    = tab_76309[*py2++];
          *d2++ = clp[384 + ((y2 + c1) >> 16)];
          *d2++ = clp[384 + ((y2 - c2 - c3) >> 16)];
          *d2++ = clp[384 + ((y2 + c4) >> 16)];
          *d2++ = 0xff;
        }
        d1 += 4 * width;
        d2 += 4 * width;
        py1 += width;
        py2 += width;
      }
    }
  };

  class YV12_space_converter : public yv_base_converter {
  public:
    YV12_space_converter(size sz) : yv_base_converter(sz) {}

    virtual void convert_to_rgb32(bytes input, argb *output) {
      const byte *src0 = input.start;
      const byte *src1 = input.start + u_plane_pos;
      const byte *src2 = input.start + v_plane_pos;
      byte *      dst  = (byte *)output;

      int         y1, y2, u, v;
      const byte *py1, *py2;
      int         i, j, c1, c2, c3, c4;
      byte *      d1, *d2;

      py1 = src0;
      py2 = py1 + width;
      d1  = dst;
      d2  = d1 + 4 * width;
      for (j = 0; j < height; j += 2) {
        for (i = 0; i < width; i += 2) {
          u = *src1++;
          v = *src2++;

          c1 = crv_tab[v];
          c2 = cgu_tab[u];
          c3 = cgv_tab[v];
          c4 = cbu_tab[u];

          // up-left
          y1    = tab_76309[*py1++];
          *d1++ = clp[384 + ((y1 + c1) >> 16)];
          *d1++ = clp[384 + ((y1 - c2 - c3) >> 16)];
          *d1++ = clp[384 + ((y1 + c4) >> 16)];
          *d1++ = 0xff;

          // down-left
          y2    = tab_76309[*py2++];
          *d2++ = clp[384 + ((y2 + c1) >> 16)];
          *d2++ = clp[384 + ((y2 - c2 - c3) >> 16)];
          *d2++ = clp[384 + ((y2 + c4) >> 16)];
          *d2++ = 0xff;

          // up-right
          y1    = tab_76309[*py1++];
          *d1++ = clp[384 + ((y1 + c1) >> 16)];
          *d1++ = clp[384 + ((y1 - c2 - c3) >> 16)];
          *d1++ = clp[384 + ((y1 + c4) >> 16)];
          *d1++ = 0xff;

          // down-right
          y2    = tab_76309[*py2++];
          *d2++ = clp[384 + ((y2 + c1) >> 16)];
          *d2++ = clp[384 + ((y2 - c2 - c3) >> 16)];
          *d2++ = clp[384 + ((y2 + c4) >> 16)];
          *d2++ = 0xff;
        }
        d1 += 4 * width;
        d2 += 4 * width;
        py1 += width;
        py2 += width;
      }
    }
  };

  class NV12_space_converter : public yv_base_converter {
  public:
    NV12_space_converter(size sz) : yv_base_converter(sz) {}

    virtual void convert_to_rgb32(bytes input, argb *output) {
      // BYTE* input_buffer = const_cast<BYTE*>(input.start);
      // NV12_to_RGB32(input_buffer, &input_buffer[u_plane_pos],
      // rgba_buffer.head(), dim.x, dim.y);

      const byte *luma = input.start;
      const byte *uv   = input.start + u_plane_pos;
      byte *      dst  = (byte *)output;

      int         y1, y2, u, v;
      const byte *py1, *py2;
      int         i, j, c1, c2, c3, c4;
      byte *      d1, *d2;

      py1 = luma;
      py2 = py1 + width;
      d1  = dst;
      d2  = d1 + 4 * width;
      for (j = 0; j < height; j += 2) {
        for (i = 0; i < width; i += 2) {
          v = *uv++;
          u = *uv++;

          c1 = crv_tab[v];
          c2 = cgu_tab[u];
          c3 = cgv_tab[v];
          c4 = cbu_tab[u];

          // up-left
          y1    = tab_76309[*py1++];
          *d1++ = clp[384 + ((y1 + c1) >> 16)];
          *d1++ = clp[384 + ((y1 - c2 - c3) >> 16)];
          *d1++ = clp[384 + ((y1 + c4) >> 16)];
          *d1++ = 0xff;

          // down-left
          y2    = tab_76309[*py2++];
          *d2++ = clp[384 + ((y2 + c1) >> 16)];
          *d2++ = clp[384 + ((y2 - c2 - c3) >> 16)];
          *d2++ = clp[384 + ((y2 + c4) >> 16)];
          *d2++ = 0xff;

          // up-right
          y1    = tab_76309[*py1++];
          *d1++ = clp[384 + ((y1 + c1) >> 16)];
          *d1++ = clp[384 + ((y1 - c2 - c3) >> 16)];
          *d1++ = clp[384 + ((y1 + c4) >> 16)];
          *d1++ = 0xff;

          // down-right
          y2    = tab_76309[*py2++];
          *d2++ = clp[384 + ((y2 + c1) >> 16)];
          *d2++ = clp[384 + ((y2 - c2 - c3) >> 16)];
          *d2++ = clp[384 + ((y2 + c4) >> 16)];
          *d2++ = 0xff;
        }

        d1 += 4 * width;
        d2 += 4 * width;
        py1 += width;
        py2 += width;
      }
    }
  };

  class RGB24_space_converter : public color_space_converter {
  public:
    RGB24_space_converter(size sz) : color_space_converter(sz) {}

    virtual void convert_to_rgb32(bytes input, argb *output) {
      uint        n   = uint(width * height);
      const byte *src = input.start;
      for (uint i = 0; i < n; ++i, ++output) {
        output->red   = *(src++);
        output->green = *(src++);
        output->blue  = *(src++);
        output->alfa  = 0xff;
      }
    }

    virtual size_t pixel_size() const override { return 3; } // number of bytes in pixel representation
    
  };

  class RGB555_space_converter : public color_space_converter {
  public:
    RGB555_space_converter(size sz) : color_space_converter(sz) {}

    virtual void convert_to_rgb32(bytes input, argb *output) {
      uint          n = width * height;
      const uint16 *p = (const uint16 *)input.start;
      for (uint i = 0; i < n; i++) {
        *(output++) = byte((*p & 0x001F) << 3);
        *(output++) = byte((*p & 0x03E0) >> 2);
        *(output++) = byte((*p & 0x7C00) >> 7);
        *(output++) = 0xff; // Alpha = 1
        p++;
      }
    }
    virtual size_t pixel_size() const override { return 2; } // number of bytes in pixel representation
  };

  class RGB565_space_converter : public color_space_converter {
  public:
    RGB565_space_converter(size sz) : color_space_converter(sz) {}

    virtual void convert_to_rgb32(bytes input, argb *output) {
      uint          n = width * height;
      const uint16 *p = (const uint16 *)input.start;
      for (uint i = 0; i < n; i++) {
        *(output++) = byte((*p & 0x001F) << 3);
        *(output++) = byte((*p & 0x07E0) >> 3);
        *(output++) = byte((*p & 0xF800) >> 8);
        *(output++) = 0xff; // Alpha = 1
      }
    }
    virtual size_t pixel_size() const override { return 2; } // number of bytes in pixel representation
  };

  class YUY2_space_converter : public yv_base_converter {
  public:
    YUY2_space_converter(size sz) : yv_base_converter(sz) {}

    inline void get_rgb_from_yuv(int y, int u, int v, argb *out) {
      int c1 = crv_tab[v];
      int c2 = cgu_tab[u];
      int c3 = cgv_tab[v];
      int c4 = cbu_tab[u];

      int y1     = tab_76309[y];
      out->red   = clp[384 + ((y1 + c1) >> 16)];
      out->green = clp[384 + ((y1 - c2 - c3) >> 16)];
      out->blue  = clp[384 + ((y1 + c4) >> 16)];
      out->alfa  = 0xFF;
    }

    virtual void convert_to_rgb32(bytes input, argb *output) {
      uint        n   = width * height / 2;
      const byte *src = input.start;

      for (uint i = 0; i < n; i++) {
        int y0 = *(src++);
        int u  = *(src++);
        int y1 = *(src++);
        int v  = *(src++);
        get_rgb_from_yuv(y0, u, v, output);
        ++output;
        get_rgb_from_yuv(y1, u, v, output);
        ++output;
      }
    }

    virtual size_t pixel_size() const override { return 4; } // number of bytes in pixel representation
  };

  class RGB32_space_converter : public color_space_converter {
  public:
    RGB32_space_converter(size sz) : color_space_converter(sz) {}
    ~RGB32_space_converter() {}

    virtual bool is_rgb32() const { return true; }

    virtual void convert_to_rgb32(bytes input, argb *output) {
      target(output, width * height).xcopy(input.start, input.length);
    }

    virtual size_t pixel_size() const override { return 4; } // number of bytes in pixel representation
  };

  // BGRA-premultiplied color space
  class BGR32_space_converter : public color_space_converter {
  public:
    BGR32_space_converter(size sz) : color_space_converter(sz) {}
    ~BGR32_space_converter() {}

    virtual bool is_rgb32() const { return true; }

    virtual void convert_to_rgb32(bytes input, argb *output) {

      // input:
      // channel_t    red;
      // channel_t    green;
      // channel_t    blue;
      // channel_t    alfa;
      //...

      for (size_t n = 0; n < input.length; n += 4, ++output) {
        output->red   = input[n + 0];
        output->green = input[n + 1];
        output->blue  = input[n + 2];
        output->alfa  = input[n + 3];
      }
    }

    virtual void convert_from_rgb32(slice<gool::argb> input, byte *data) {

      // input:
      // channel_t    red;
      // channel_t    green;
      // channel_t    blue;
      // channel_t    alfa;
      //...
      for (size_t n = 0; n < input.length; ++n, data += 4) {
        argb ic = input[n];
        data[0] = ic.red;
        data[1] = ic.green;
        data[2] = ic.blue;
        data[3] = ic.alfa;
      }
    }

    virtual size_t pixel_size() const override { return 4; } // number of bytes in pixel representation
  };

  inline color_space_converter *
  color_space_converter::factory(COLORSPACE cspace, size sz) {
    switch (cspace) {
    default: return new color_space_converter(sz);
    case COLORSPACE_YV12: return new YV12_space_converter(sz);
    case COLORSPACE_IYUV: return new YUV420_space_converter(sz);
    case COLORSPACE_NV12: return new NV12_space_converter(sz);
    case COLORSPACE_YUY2: return new YUY2_space_converter(sz);
    case COLORSPACE_RGB24: return new RGB24_space_converter(sz);
    case COLORSPACE_RGB555: return new RGB555_space_converter(sz);
    case COLORSPACE_RGB565: return new RGB565_space_converter(sz);
    case COLORSPACE_RGB32: return new RGB32_space_converter(sz);
    case COLORSPACE_BGR32: return new BGR32_space_converter(sz);
    }
  }

} // namespace gool