//|
//|
//| Copyright (c) 2001-2010
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| graphics primitives
//|
//|

#ifndef __gool_graphics_h__
#define __gool_graphics_h__

#include "tool/tool.h"
#include "gool-types.h"
#include "gool-image.h"
#include "gool-figures.h"
#include "gool-font.h"

namespace html {
  struct style;
  struct element;
}

namespace gool {
  using namespace tool;

  extern const float pi;

  //extern bool  set_gpu_black_list(const char *bl_json);
  //extern value get_gpu_black_list();

  enum GRAPHICS_CAPS {
    UNKNOWN_GRAPHICS,
    SOFTWARE_GRAPHICS,
    WARP_GRAPHICS, // better than pure SOFTWARE_GRAPHICS but not of
                   // HARDWARE_GRAPHICS acceleration level
    HARDWARE_GRAPHICS,
    SKIA_GRAPHICS,
    SKIA_OPENGL_GRAPHICS,
    ANY_GRAPHICS = 0xFFFF
  };

  enum TEXT_DIRECTION {
    TD_DEFAULT,
    TD_LTR,
    TD_RTL,
    TD_TTB,
    // TD_BTT,
  };

  /*enum STROKE_STYLE
  {
    STROKE_NONE,
    STROKE_SOLID,
    STROKE_DOTTED,
    STROKE_DASHED,
    MAX_STROKE_VALUE = STROKE_DASHED,
  };*/

  enum CAP_STYLE { // D2D1_CAP_STYLE compatible
    CAP_BUTT     = 0,
    CAP_SQUARE   = 1,
    CAP_ROUND    = 2,
    CAP_TRIANGLE = 3
  };

  enum LINE_JOIN { // D2D1_LINE_JOIN compatible
    JOIN_MITER          = 0,
    JOIN_BEVEL          = 1,
    JOIN_ROUND          = 2,
    JOIN_MITER_OR_BEVEL = 3
  };

  enum DASH_STYLE {
    DASH_STYLE_SOLID        = 0,
    DASH_STYLE_DASH         = 1,
    DASH_STYLE_DOT          = 2,
    DASH_STYLE_DASH_DOT     = 3,
    DASH_STYLE_DASH_DOT_DOT = 4,
    DASH_STYLE_CUSTOM       = 5,
  };

  enum TEXT_ALIGNMENT {
    ALIGN_DEFAULT,
    ALIGN_START,
    ALIGN_END,
    ALIGN_CENTER,
    ALIGN__MAX,
  };

  enum FILL_STROKE_MODES {
    FILL_ONLY          = 1,
    STROKE_ONLY        = 2,
    FILL_AND_STROKE    = 3,
    FILL_BY_LINE_COLOR = 7
  };

  enum IMAGE_RENDERING {
    image_rendering_default,
    image_rendering_speed,
    image_rendering_quality,
    image_rendering_undefined
  };

  class application;
  class theme_image;
  class path;
  class text_layout;
  class clipper;
  class brush;


  template <typename S>
  void normailize_round_rect(geom::size_t<S> rect_dim, geom::size_t<S> &tl,
                             geom::size_t<S> &tr, geom::size_t<S> &br,
                             geom::size_t<S> &bl) {
    float dx = float(rect_dim.x);
    float dy = float(rect_dim.y);

    float k = 1.0f;
    float t = 0;
    if (tl.x + tr.x) {
      t = dx / (tl.x + tr.x);
      if (t < k) k = t;
    }
    if (bl.x + br.x) {
      t = dx / (bl.x + br.x);
      if (t < k) k = t;
    }
    if (tl.y + bl.y) {
      t = dy / (tl.y + bl.y);
      if (t < k) k = t;
    }
    if (tr.y + br.y) {
      t = dy / (tr.y + br.y);
      if (t < k) k = t;
    }

    if (k < 1.0f) {
      tl.x = S(tl.x * k);
      tl.y = S(tl.y * k);
      tr.x = S(tr.x * k);
      tr.y = S(tr.y * k);
      br.x = S(br.x * k);
      br.y = S(br.y * k);
      bl.x = S(bl.x * k);
      bl.y = S(bl.y * k);
    }
  }

  class path : public resource_x<path> {
  public:
    friend class layer;

    enum COMBINE_MODE {
      //
      // Produce a geometry representing the set of points contained in either
      // the first or the second geometry.
      //
      COMBINE_MODE_UNION = 0,

      //
      // Produce a geometry representing the set of points common to the first
      // and the second geometries.
      //
      COMBINE_MODE_INTERSECT = 1,

      //
      // Produce a geometry representing the set of points contained in the
      // first geometry or the second geometry, but not both.
      //
      COMBINE_MODE_XOR = 2,

      //
      // Produce a geometry representing the set of points contained in the
      // first geometry but not the second geometry.
      //
      COMBINE_MODE_EXCLUDE = 3,

    };

    path() : filled(false) {}

    virtual void set(const polygonf &vertices, bool even_odd = true) {
      slice<pointf> points = vertices();

      if (points.length == 0) return;

      set_even_odd(even_odd);

      pointf p = points++;
      start(p, true);
      while (!!points) {
        p = points++;
        line_to(p);
      }
      close();
    }
    virtual void set(const polygonf &v1, const polygonf &v2) {
      slice<pointf> points = v1();

      if (points.length == 0) return;

      set_even_odd(true);

      pointf p = points++;
      start(p, true);
      while (!!points) {
        p = points++;
        line_to(p);
      }
      close();

      points = v2();
      p      = points++;
      move_to(p);
      while (!!points) {
        p = points++;
        line_to(p);
      }
      close();
    }
    virtual void add_rect(const rectf &r) {
      start(r.pointOf(7), true);
      line_to(r.pointOf(9));
      line_to(r.pointOf(3));
      line_to(r.pointOf(1));
      close();
    }
    virtual void add_rect(pointf pos, sizef dim) {
      add_rect(gool::rectf(pos, dim));
    }

    // round rect
    virtual void add_round_rect(pointf org, sizef dim, sizef tl, sizef tr,
                                sizef br, sizef bl) {
      // reset();
      rectf r(org, dim);

      const float kappa1 = 1.0f - 0.5522847493f; // length proportional to
                                                 // radius of a cubic bezier
                                                 // handle for 90deg arcs

      this->move_to(pointf(r.s.x, r.s.y + tl.y));
      this->line_to(pointf(r.s.x, r.e.y - bl.y));
      this->cubic_to(pointf(r.s.x + bl.x, r.e.y),
                     pointf(r.s.x, r.e.y - bl.y * kappa1),
                     pointf(r.s.x + bl.x * kappa1, r.e.y));
      this->line_to(pointf(r.e.x - br.x, r.e.y));
      this->cubic_to(pointf(r.e.x, r.e.y - br.y),
                     pointf(r.e.x - br.x * kappa1, r.e.y),
                     pointf(r.e.x, r.e.y - br.y * kappa1));
      this->line_to(pointf(r.e.x, r.s.y + tr.y));
      this->cubic_to(pointf(r.e.x - tr.x, r.s.y),
                     pointf(r.e.x, r.s.y + tr.y * kappa1),
                     pointf(r.e.x - tr.x * kappa1, r.s.y));
      this->line_to(pointf(r.s.x + tl.x, r.s.y));
      this->cubic_to(pointf(r.s.x, r.s.y + tl.y),
                     pointf(r.s.x + tl.x * kappa1, r.s.y),
                     pointf(r.s.x, r.s.y + tl.y * kappa1));
      // this->line_to( pointf(r.s.x,r.s.y + tl.y ));
      this->close();
    }

    virtual bool is_inside(pointf pt) = 0;

    virtual void reset() = 0;
    virtual void seal()  = 0;

    virtual void start(pointf start, bool filled)     = 0;
    virtual void move_to(pointf pt, bool rel = false) = 0;
    virtual void line_to(pointf pt, bool rel = false) = 0;
    virtual void hline_to(float x, bool rel = false) {
      line_to(pointf(x, rel ? 0 : lastp().y), rel);
    }
    virtual void vline_to(float y, bool rel = false) {
      line_to(pointf(rel ? 0 : lastp().x, y), rel);
    }
    virtual void add_arc(pointf c, sizef r, float angle_start,
                         float angle_swipe)                           = 0;
    virtual void quadratic_to(pointf pt, pointf cp, bool rel = false) = 0;
    virtual void cubic_to(pointf pt, pointf cp1, pointf cp2,
                          bool rel = false) = 0; // bezier
    virtual void set_even_odd(bool yes)     = 0;
    virtual void close()                    = 0;
    virtual bool is_empty() const           = 0;
    virtual bool is_open() const { return !is_empty(); }

    void circ_arc_to(pointf pc, pointf pe, float radius, bool rel_to = false);
    void arc(pointf c, sizef r, float angle_start, float angle_swipe, bool rel_to = false);

    virtual void arc_to(pointf to, sizef r, float rotation_angle,
                        bool large_arc, bool clockwise, bool rel_to = false);

    virtual rectf bounds() const = 0;

    // bool  last(pointf& pos) { if (!is_empty()) { pos = lastp(); return true;
    // } return false; }
    virtual pointf lastp() = 0;
    virtual void   lastp(pointf p) {}

    virtual path *combine(COMBINE_MODE mode, const path *other) {
      return nullptr;
    }

  protected:
    bool filled;
  };

  class image_link;

  enum COMPOSITION_OP {
    no_composition,
    src_over,
    dst_over,
    src_in,
    dst_in,
    src_out,
    dst_out,
    src_atop,
    dst_atop,
    dst_src_xor,
    dst_copy_src
  };

  struct filter_graph_builder {
    virtual bool add_blur(float p) { return false; }
    virtual bool add_brightness(float p) { return false; }
    virtual bool add_contrast(float p) { return false; }
    virtual bool add_grayscale(float p) { return false; }
    virtual bool add_hue_rotate(float a) { return false; }
    virtual bool add_invert() { return false; }
    virtual bool add_opacity(float p) { return false; }
    virtual bool add_saturate(float p) { return false; }
    virtual bool add_sepia(float p) { return false; }
    virtual bool add_drop_shadow(float offx, float offy, float blur_radius,
                                 float spread_radius, gool::argb clr) {
      return false;
    }
    virtual bool add_composition(COMPOSITION_OP op) { return false; }
  };

  class graphics : public resource_x<graphics>, public weakable {
  public:
    graphics(bool transparent)
      : _clipper(0), _overlay_requests(0), _image_links(nullptr),
          _drawing_root(nullptr),
          _transparent(transparent),
          current_style(nullptr) /*, text_color(argb::no_color())*/ {
      _uid = ++_uid_counter;
    }
    ~graphics() { free_assets(); }

    virtual void final_release() override {
      while (n_layers() > 0)
        pop_layer();
      free_assets();
    }

    long get_uid() const { return _uid; }

    virtual void begin_drawing() { _overlay_requests = 0; }
    virtual void end_drawing() {}

    virtual void  set_tag(void * /*ptr*/) {}
    virtual void *get_tag() { return 0; }

    // shapes need 0.5px adjustment to draw in the middle of px, GDI+
    virtual bool needs_half_pixel_adjustment() const { return false; }

    virtual void clear(const rect& rc) {}
    virtual void commit() {}
    virtual void set_immediate_update_mode(bool onoff) { }

    // push/pop_opacity is called only on transparent graphics when engine knows that it will draw opaque container.
    // Used to enable ClearType on such areas.
    virtual void push_opacity() { _opaque = true; }
    virtual void pop_opacity() { _opaque.clear(); }
    virtual bool is_opaque() { return _opaque.val(!_transparent) != 0; }

    // true if surface supports alpha
    virtual bool is_transparent() const { return _transparent; }

    // bitmap graphics
    virtual bool is_bitmap_graphics() const { return false; }

    // window graphics
    virtual void resize(gool::size /*nsz*/) { assert(false); }

    virtual void fill(argb c, const rect &dst)                     = 0;
    virtual void fill(argb c, const rect &dst, const size &radius) = 0;
    virtual void fill(argb c, const path *dst)                     = 0;
    virtual void fill(image *img, const path *dst)                 = 0;
    virtual void fill(const brush *lb, const rect &dst)            = 0;
    virtual void fill(image *img, const rect &dst, const rect &src,
                      point offset = point(0, 0), size dstcell = size());
    virtual void fill(image *img, const rect &dst, point offset = point(0, 0)) {
      fill(img, dst, rect(img->dim()), offset);
    }
    void expand(image *img, const rect &dst, const SECTION_DEFS &sds) {
      img->expand(this, dst, sds);
    }

    virtual image_renderer_f image_renderer(image *img);

    // virtual void fill( argb c, const path& dst) = 0;
    // virtual void fill( image* img, const rect& dst, point offset =
    // point(0,0)) { fill(img, dst,rect(img->dim()),offset); }

    virtual void draw(image *img, const rect &dst, const rect &src,
                      byte opacity = 255) {
      img->draw(this, dst, src, opacity);
    }
    virtual void draw(image *img, const rect &dst, byte opacity = 255) {
      img->draw(this, dst, rect(img->dim(dst)), opacity);
    }
    virtual void draw(image *img, const rectf &dst, byte opacity = 255) {
      img->draw(this, dst, rect(img->dim(dst)), opacity);
    }

    virtual void draw(image *img, const point &dst, byte opacity = 255) {
      img->draw(this, rect(dst, img->dim()), rect(img->dim()), opacity);
    }
    virtual void draw(image *img, const pointf &dst, byte opacity = 255) {
      img->draw(this, rectf(dst, img->dim()), rect(img->dim()), opacity);
    }

    virtual void draw(text_layout *tl, pointf dst, argb c = argb::no_color());
    virtual void draw(text_layout *tl, point dst, argb c = argb::no_color()) { draw(tl, pointf(dst), c); }

    virtual void draw(const path *dst, argb c, float width = 1.0f) = 0;

    // virtual void draw_rect( rect rc, int border_width, argb c, STROKE_STYLE
    // ss = STROKE_SOLID ) = 0;
    virtual void draw_line_v(const rect &rc, argb c, int stroke, int step,
                             int_v off = int_v()) = 0;
    virtual void draw_line_h(const rect &rc, argb c, int stroke, int step,
                             int_v off = int_v()) = 0;

    virtual void draw_line(pointf p1, pointf p2, argb c, int width) = 0;

    virtual path *       create_path() const = 0;
    virtual application *app() const         = 0;

    virtual bool push_clip(const rect &r, clipper *cl);
    virtual bool push_clip(const path *clip, clipper *cl);
    virtual void pop_clip();

    virtual void flush() {}

    virtual void            image_rendering_mode(IMAGE_RENDERING irm) {}
    virtual IMAGE_RENDERING image_rendering_mode() const {
      return image_rendering_default;
    }

    // virtual void set_text_color(argb c) { _text_color = c; }
    // virtual argb get_text_color() const { return _text_color; }

    virtual point offset(point noff) = 0;

    virtual bool is_under_transform() { return false; }

    virtual bool set_antialiasing(bool /*on*/) { return false; }

    // virtual void draw( html::text_layout* ptl, point pt, argb c ) = 0;

    // virtual bitmap* create_bitmap(argb* pbits, uint stride, gool::size dim,
    // bool has_alpha) = 0;  virtual void draw_bits(argb* pbits, uint stride,
    // gool::size dim, bool has_alpha, const rect&  dst) = 0;

    void request_overlay();
    bool has_overlays(clipper &cl);

    //|
    //| Graphin alike interface
    //|

    virtual uint               push_state()                           = 0;
    virtual bool               pop_state()                            = 0;
    virtual void               restore_state(uint level)              = 0;
    //virtual const text_format &get_text_format()                      = 0;
    //virtual void               set_text_format(const text_format &tf) = 0;

    // must be overriden
    virtual void
                 push_layer(const rect &clip, byte opacity = 255,
                            function<bool(filter_graph_builder *)> filters = nullptr) = 0;
    virtual void push_layer(const path *clip, byte opacity = 255) = 0;
    virtual void push_layer(const bitmap *clip, bool draw1a,
                            byte opacity = 255) {
    } // draw1a = true - pixels with alpha == 255 are visible, draw1a = false -
      // pixels with alpha == 255 are clipped out.
    virtual void pop_layer() = 0;
    virtual uint n_layers()  = 0;

    virtual CAP_STYLE  cap_style()                                      = 0;
    virtual void       cap_style(CAP_STYLE cs)                          = 0;
    virtual LINE_JOIN  line_join()                                      = 0;
    virtual void       line_join(LINE_JOIN ncs, float mitter_limit = 4) = 0;
    virtual void       dash_style(DASH_STYLE st)                        = 0;
    virtual DASH_STYLE dash_style()                                     = 0;
    virtual void       custom_dash_style(slice<float> pattern, float offset) {}

    virtual void set_fill(argb c = argb::no_color()) = 0; // set fill brush
    virtual void set_fill(const linear_brush &lb)    = 0;
    virtual void set_fill(const radial_brush &rb)    = 0;
    virtual void set_fill(const image *ib)           = 0;

    virtual void set_fill(const brush *pbr) {
      if (pbr)
        pbr->set_fill_to(this);
      else
        set_fill();
    }

    virtual void set_fill_even_odd(bool even_odd /*false - non-null*/) = 0;

    // virtual void set_fill( brush* pb ) = 0; // set fill brush
    virtual void set_stroke(argb c = argb::no_color()) = 0; // set stroke brush
    virtual void set_stroke(const linear_brush &lb)    = 0;
    virtual void set_stroke(const radial_brush &rb)    = 0;
    virtual void set_stroke_width(float w)             = 0; // set stroke brush

    virtual void set_stroke(const brush *pbr, float width, CAP_STYLE cp,
                            LINE_JOIN lj, float mitter_limit = 4) {
      if (pbr) {
        pbr->set_stroke_to(this);
        set_stroke_width(width);
        cap_style(cp);
        line_join(lj, mitter_limit);
        dash_style(DASH_STYLE_SOLID);
      } else
        set_stroke();
    }

    // virtual void set_stroke( brush* pb, float width = 0, STROKE_STYLE st =
    // STROKE_SOLID) = 0; // set stroke brush
    virtual void draw_ellipse(pointf center, sizef radius, bool stroke = true, bool fill = true) = 0;
    virtual void draw_arc(pointf center, sizef radius, float angle_start, float angle_sweep, bool stroke = true, bool fill = true);
    virtual void draw_star(pointf center, sizef r1, sizef r2, float start_angle, int num_rays, bool stroke = true, bool fill = true);

    virtual void draw_rectangle(pointf org, sizef dim, bool stroke = true, bool fill = true) = 0;
    virtual void draw_rectangle(pointf org, sizef dim, sizef rtl, sizef rtr, sizef rbr, sizef rbl, bool stroke = true, bool fill = true);
    virtual void draw_line(pointf start, pointf end) = 0;
    virtual void draw_path(const gool::path *p, bool stroke = true, bool fill = true) = 0;
    virtual void draw_path(const gool::polygonf &p, bool stroke = true, bool fill = false)  = 0;

    enum line_end_type {
      LET_NONE,
      LET_ARROW,
      LET_CIRCLE,
    };

    struct line_end_style {
      line_end_type type;
      sizef         dim;
      line_end_style() : type(LET_NONE) {}

      static line_end_style none() {
        line_end_style t;
        return t;
      }
      static line_end_style arrow(sizef sz) {
        line_end_style t;
        if (!sz.empty()) {
          t.dim  = sz;
          t.type = LET_ARROW;
        }
        return t;
      }
      static line_end_style arrow(float w, float h) {
        return arrow(sizef(w, h));
      }
      static line_end_style circle(float r) {
        line_end_style t;
        if (r != 0) {
          t.dim  = sizef(r);
          t.type = LET_CIRCLE;
        }
        return t;
      }
    };

    virtual void draw_line(pointf origin, pointf end, line_end_style sorigin,
                           line_end_style send);

    virtual void draw_image(image *img, const rectf &dst, const rect &src,
                            byte opacity = 255) {
      img->draw(this, dst, src, opacity);
    }
    // virtual void draw_image( image* img, const rectf&  dst, byte opacity =
    // 255) { img->draw(this,dst,rect(img->dim()),opacity); }  virtual void
    // draw_image( image* img, const pointf& dst, byte opacity = 255) {
    // img->draw(this,rect(dst,img->dim()),rect(img->dim()),opacity); }

    virtual void   transform(const affine_mtx_f &m)                = 0;
    virtual affine_mtx_f transform() const                         = 0;
    virtual void   reset_transform()                               = 0;

    virtual void   translate(pointf pt)                            = 0;
    virtual void   rotate(float radians, pointf center = pointf()) = 0;
    virtual void   scale(sizef sz, pointf center = pointf())       = 0;
    virtual void   skew(sizef sz, pointf center = pointf())        = 0;
    virtual pointf world_to_screen(pointf pt) const                = 0;
    virtual float  world_to_screen(float scalar) const {
      pointf s(0, 0);
      pointf e(scalar, scalar);
      s = world_to_screen(s);
      e = world_to_screen(e);
      return sqrtf((e.x - s.x) * (e.x - s.x) + (e.y - s.y) * (e.y - s.y)) *
             0.7071068f;
    }
    virtual pointf screen_to_world(pointf pt) const = 0;
    virtual float  screen_to_world(float scalar) const {
      pointf s(0, 0);
      pointf e(scalar, scalar);
      s = screen_to_world(s);
      e = screen_to_world(e);
      return sqrtf((e.x - s.x) * (e.x - s.x) + (e.y - s.y) * (e.y - s.y)) *
             0.7071068f;
    }

    // virtual void reset_path() = 0;
    // virtual void close_path() = 0;

    //#ifdef USE_D2D
    //    virtual ID2D1Brush* solid_brush(const argb& c) = 0;
    //#endif

    friend class image;
    friend class image_link;
    friend class theme_image;
    virtual void do_draw(image *img, const rectf &dst, const rect &src,
                         byte opacity) = 0;
    virtual void do_expand(image *img, const rect &dst, const SECTION_DEFS &sds,
                           rect area);
    virtual void free_assets();

    rect get_clip_rc() const { return _clip_rc; }
    void set_clip_rc(rect rc) { _clip_rc = rc; }
    rect get_clip_world_rc() const {
      rect rf   = _clip_rc;
      rf.s = screen_to_world(rf.s);
      rf.e = screen_to_world(rf.e);
      return rf;
    }
    void  set_drawing_root(void *tag) { _drawing_root = tag; }
    void *get_drawing_root() const { return _drawing_root; }

    html::style*    current_style;
    html::element*  current_element;
    size            pixels_scaling;

    // argb         text_color; // text color to force

    resolution_provider* get_resolution_provider() const;

    resolution_provider* pview = nullptr;

    bool drawing_backdrop = false; // it is drawing backdrop

  protected:
    clipper *  _clipper;
    uint       _overlay_requests;
    void *     _drawing_root;
    bool       _transparent;
    tristate_v _opaque;

    image_link *_image_links;
    rect        _clip_rc; // for drawing optimizations only
    long        _uid;

    static locked::counter _uid_counter;
  };

  inline image_renderer_f graphics::image_renderer(image *img) {
    return [=](rect dst, rect src) { draw(img, dst, src); };
  }

  class aa_mode {
    handle<graphics> gfx;
    bool             pmode;
    aa_mode();
    aa_mode(const aa_mode &);

  public:
    aa_mode(graphics *pg) : gfx(pg) { pmode = gfx->set_antialiasing(true); }
    ~aa_mode() { gfx->set_antialiasing(pmode); }
  };

  class state {
    uint      level;
    graphics *gfx;
    state() {}

  public:
    state(graphics *pg) : gfx(pg) { level = pg->push_state(); }
    ~state() {
      if (gfx) gfx->restore_state(level);
    }
  };

  class clipper {
    friend class graphics;
    graphics *  gfx;
    rect        rc;
    bool        clip;
    bool        overflow;
    uint        overlay_requests;
    //tristate_v  transparency;
    //tristate_v  prev_transparency;
    bool        pushed_opacity;
    clipper*    prev;

  public:
    uint container_id;

    clipper(graphics *pg, const rect &rc, bool overflows = true, tristate_v trans = tristate_v())
        : gfx(pg), clip(false), overflow(overflows), overlay_requests(0),
          container_id(0), prev(0), pushed_opacity(false)
    {
      assert(gfx);
      if (overflow) { clip = gfx->push_clip(rc, this); }

      bool requested_opacity = trans.is_defined() && !trans;
      bool current_opacity = gfx->is_opaque();
      if (requested_opacity && !current_opacity)
      {
        pushed_opacity = true;
        gfx->push_opacity();
      }

    };
    ~clipper() {
      if (pushed_opacity)
        gfx->pop_opacity();
      if (clip)
        gfx->pop_clip();
    }
  };

  inline bool graphics::push_clip(const rect &r, clipper *cl) {
    cl->prev = _clipper;
    cl->rc   = _clip_rc;
    _clip_rc &= r;
    _clipper = cl;
    return true;
  }
  inline bool graphics::push_clip(const path *clip, clipper *cl) {
    cl->prev = _clipper;
    cl->rc   = _clip_rc;
    _clip_rc &= rect(clip->bounds());
    _clipper = cl;
    return true;
  }

  inline void graphics::pop_clip() {
    if (_clipper) {
      _clip_rc = _clipper->rc;
      _clipper = _clipper->prev;
    }
  }
  inline void graphics::request_overlay() {
    if (_clipper)
      ++_clipper->overlay_requests;
    else
      ++_overlay_requests;
  }
  inline bool graphics::has_overlays(clipper &cl) {
    if (_clipper == &cl)
      return _clipper->overlay_requests > 0;
    else
      return _overlay_requests > 0;
  }


  class text_layout: public resource_x<text_layout> {
    
  public:
    handle<resource> tb;

    struct line {
      float        y;
      float        baseline;
      float        height;
      uint         length;      // number of characters in the line
      uint         start;       // line start
      //array<wchar> visual_text; // line text in visual order
    };

    text_layout();

    void   set_host(resource* par, wchars classname = wchars());
    void   set_style(wchars name);
    void   set_style(const html::style* cs);
    wchars get_style() const;
    void   set_font(wchars cssfont);
    wchars get_font() const;
    void   set_class(wchars name);
    wchars get_class() const;
    void   set_text(wchars text);
    wchars get_text();

    void   set_color(color_v clr);

    void   set_width(float width);
    void   set_height(float height);
    sizef  get_dim() const;
    sizef  get_box() const;

    float  width_min() const;
    float  width_max() const;
    float  height() const;
    float  ascent() const;

    uint   get_lines_count() const;
    line   get_line(int n) const;

    gool::TEXT_ALIGNMENT get_text_alignment() const;
    gool::TEXT_ALIGNMENT get_lines_alignment() const;

    void   set_alignment(gool::TEXT_ALIGNMENT text, gool::TEXT_ALIGNMENT line);

  };

  class layer : public resource {
    graphics *pgfx;

  public:
    layer(graphics *pg, rect clip, byte opacity = 255,
          function<bool(filter_graph_builder *)> filters = nullptr)
        : pgfx(pg) {
      pgfx->push_layer(clip, opacity, filters);
    }
    layer(graphics *pg, const path *clip, byte opacity = 255)
        : pgfx(pg) {
      pgfx->push_layer(clip, opacity);
    }
    layer(graphics *pg, const bitmap *clip, bool draw1a, byte opacity = 255)
        : pgfx(pg) {
      pgfx->push_layer(clip, draw1a, opacity);
    }
    ~layer() { pgfx->pop_layer(); }
  };

  // class brush_collection
  //{
  //  handle_pool<brush> coll;
  // public:
  //  const brush* instantiate(const brush* pb) { return coll-> }
  //};

  //"system", "system-caption", "system-menu", "system-status"
  // extern bool get_system_font(/*inout*/ ustring& name, /*out*/int& size,
  // /*out*/ uint& weight, /*out*/bool& italic);
  extern bool install_font(const string &name, bytes data);

  /*class bitmap_graphics: virtual public graphics
  {
  protected:
    bitmap_graphics() {}
  public:
    virtual bool is_bitmap_graphics() const override { return true; }
    virtual bitmap* get_bitmap() = 0;
    virtual ~bitmap_graphics() {}
  };

  class dib_graphics: virtual public gool::graphics
  {
  protected:
    dib_graphics() {}
  public:
    virtual ~dib_graphics();
    virtual bool is_bitmap_graphics() const override { return true; }
    //virtual dib32* get_dib() = 0;
  }; */

  inline void brush::set_fill_to(graphics *pg) const { pg->set_fill(); }
  inline void brush::set_stroke_to(graphics *pg) const { pg->set_stroke(); }

  inline void solid_brush::set_fill_to(graphics *pg) const {
    pg->set_fill(this->color);
  }
  inline void solid_brush::set_stroke_to(graphics *pg) const {
    pg->set_stroke(this->color);
  }

  inline void linear_brush::set_fill_to(graphics *pg) const {
    pg->set_fill(*this);
  }
  inline void linear_brush::set_stroke_to(graphics *pg) const {
    pg->set_stroke(*this);
  }

  inline void radial_brush::set_fill_to(graphics *pg) const {
    pg->set_fill(*this);
  }
  inline void radial_brush::set_stroke_to(graphics *pg) const {
    pg->set_stroke(*this);
  }

  inline void tile_brush::set_fill_to(graphics *pg) const {
    pg->set_fill(tile);
  }
  inline void tile_brush::set_stroke_to(graphics *pg) const
  {
    assert(false);
  }



  class path_image // path:{ svg path d commands }
      : public image {
    IMAGE_MAPPING              mapping;
    tool::string               path_chars;
    mutable handle<gool::path> path;
    mutable application *      path_app;
    mutable size               path_size; // path was generated for that size
    mutable bool               is_expandable;

  public:
    path_image(const tool::string &spath);
    virtual ~path_image();

    static bool parse(gool::path *p, chars svg_d_path, size sz = size(0, 0));

    virtual bool is_valid() const { return true; }
    virtual bool is_transparent() const { return true; } // has alpha bits
    virtual bool is_vector() const { return true; }
    virtual bool is_animated(uint_ptr /*site_id*/) const { return false; }

    virtual handle<bitmap> get_bitmap(graphics * /*pg*/, size /*sz*/) override { return 0; }

    virtual void draw(graphics *gfx, rect dst, rect src, byte opacity) {
      draw(gfx, rectf(dst), src, opacity);
    }
    virtual void draw(graphics *gfx, rectf dst, rect src, byte opacity);
    virtual void expand(graphics * /*gfx*/, const rect & /*dst*/,
                        const SECTION_DEFS & /*sds*/) {
      assert(false);
    }

    virtual void tile(graphics *gfx, const rect &dst, const rect &src, point offset, size dstcell) override
    {
      size sz = src.size();
      if (sz.empty())
        return;
      if (dstcell.empty())
        dstcell = sz;
      clipper _c(gfx, dst);
      for (int y = dst.s.y - dstcell.y; y <= dst.e.y; y += dstcell.y)
        for (int x = dst.s.x - dstcell.x; x <= dst.e.x; x += dstcell.x)
        {
          rect r_dst(point(x, y) + offset, dstcell);
          draw(gfx, r_dst, src, 255);
        }
    }

    size dimension() const;

    virtual size dim() const { return dimension(); }
    virtual size dim(const rect & /*for_dst*/) const { return dim(); }
    virtual void drop_cache() {
      path     = nullptr;
      path_app = nullptr;
    }

    virtual image *mapped_left_to_right() override;
    virtual image *mapped_top_to_right() override;

    gool::path *get_path(gool::application *app    = nullptr,
                         size               for_sz = size()) const;
  };



} // namespace gool

#endif
