//|
//|
//| Copyright (c) 2006-2018
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| font abstraction
//|
//|

#ifndef __gool_font_h__
#define __gool_font_h__

#include "tool/tool.h"
#include "gool-types.h"

namespace html {
  struct style;
}

namespace gool {
  using namespace tool;

  enum FONT_RENDERING {
    FR_UI,
    FR_VECTOR,
  };

  struct font : public resource {
    ustring        name;
    uint           weight;
    bool           italic;
    float          size; // pixels
    FONT_RENDERING mode;
    int            ascent;
    int            descent;
    int              x_height;
    mutable float_v  _ch;
    bool           serifs; // like Times New Roman

    font()
        : weight(400), italic(false), size(0), mode(FR_VECTOR), ascent(),
          descent(), x_height(), serifs(false) {}
    font(const font &f)
        : name(f.name), weight(f.weight), italic(f.italic), size(f.size),
          mode(f.mode), serifs(f.serifs), ascent(), descent(), x_height() {}
    virtual ~font() {}

    virtual bool is_bitmap_font() const { return false; }

    virtual ustring used_font_name() const { return name; }

    bool operator==(const font &rs) const {
      return name == rs.name && weight == rs.weight && italic == rs.italic &&
             size == rs.size && mode == rs.mode;
    }
    bool operator!=(const font &rs) const { return !operator==(rs); }

    virtual unsigned int hash() const {
      uint v = 17;
      hash_combine(v, hash_value(name));
      hash_combine(v, hash_value(weight));
      hash_combine(v, hash_value((uint)italic));
      hash_combine(v, hash_value(size));
      hash_combine(v, hash_value((uint)mode));
      return v;
    }

    // these are not abstract as font can be used as a key in hash table.
    virtual bool has_glyph_for(uint ucodepoint) const = 0;
    virtual uint glyph_index(uint ucodepoint) const   = 0;
    virtual bool glyph_metrics(uint16 /*glyph_index*/, float /*em_size*/,
                               float & /*width*/, float & /*height*/) const {
      return false;
    }
    virtual void metrics(int &a, int &d, int &x, float sz = 0) const {
      a  = ascent;
      d  = descent;
      x  = x_height;
      sz = size;
    }

    int height(float sz = 0) const {
      int a, d, x;
      metrics(a, d, x, sz);
      return a + d;
    }

    float ch() const {
      if (_ch.is_undefined()) {
        auto        glyph = glyph_index('0');
        float       height;
        float       width;
        glyph_metrics(uint16(glyph), size, width, height);
        _ch = width;
      }
      return _ch;
    }

  };
 

}

#endif