//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| affine transformations
//|
//|

#ifndef __GL_AFFINE_H
#define __GL_AFFINE_H

#include "tool/tool.h"
#include "gool/gool-geometry.h"
#include <math.h>

//#ifdef USE_D2D
//#include <d2d1.h>
//#endif

namespace gool {
  namespace geom {

    template <class T> inline bool is_equal_eps(T v1, T v2, T epsilon) {
      return fabs(double(v1) - double(v2)) <= double(epsilon);
    }

    // borrowed from AGG

    //============================================================trans_affine
    //
    // See Implementation agg_trans_affine.cpp
    //
    // Affine transformation are linear transformations in Cartesian coordinates
    // (strictly speaking not only in Cartesian, but for the beginning we will
    // think so). They are rotation, scaling, translation and skewing.
    // After any affine transformation a line segment remains a line segment
    // and it will never become a curve.
    //
    // There will be no math about matrix calculations, since it has been
    // described many times. Ask yourself a very simple question:
    // "why do we need to understand and use some matrix stuff instead of just
    // rotating, scaling and so on". The answers are:
    //
    // 1. Any combination of transformations can be done by only 4
    // multiplications
    //    and 4 additions in floating point.
    // 2. One matrix transformation is equivalent to the number of consecutive
    //    discrete transformations, i.e. the matrix "accumulates" all
    //    transformations in the order of their settings. Suppose we have 4
    //    transformations:
    //       * rotate by 30 degrees,
    //       * scale X to 2.0,
    //       * scale Y to 1.5,
    //       * move to (100, 100).
    //    The result will depend on the order of these transformations,
    //    and the advantage of matrix is that the sequence of discret calls:
    //    rotate(30), scaleX(2.0f), scaleY(1.5), move(100,100)
    //    will have exactly the same result as the following matrix
    //    transformations:
    //
    //    affine_matrix m;
    //    m *= rotate_matrix(30);
    //    m *= scaleX_matrix(2.0f);
    //    m *= scaleY_matrix(1.5);
    //    m *= move_matrix(100,100);
    //
    //    m.transform_my_point_at_last(x, y);
    //
    // What is the good of it? In REAL life we will set-up the matrix only once
    // and then transform many points, let alone the convenience to set any
    // combination of transformations.
    //
    // So, how to use it? Very easy - literally as it's shown above. Not quite,
    // let us write a correct example:
    //
    // agg::trans_affine m;
    // m *= agg::trans_affine_rotation(30.0f * 3.1415926 / 180.0f);
    // m *= agg::trans_affine_scaling(2.0, 1.5);
    // m *= agg::trans_affine_translation(100.0f, 100.0f);
    // m.transform(&x, &y);
    //
    // The affine matrix is all you need to perform any linear transformation,
    // but all transformations have origin point (0,0). It means that we need to
    // use 2 translations if we want to rotate someting around (100,100):
    //
    // m *= agg::trans_affine_translation(-100.0f, -100.0f);         // move to
    // (0,0) m *= agg::trans_affine_rotation(30.0f * 3.1415926 / 180.0f);  //
    // rotate m *= agg::trans_affine_translation(100.0f, 100.0f);           //
    // move back to (100,100)
    //----------------------------------------------------------------------

    template <typename REAL> struct trans_affine {
      REAL sx, shy, shx, sy, tx, ty;

      //------------------------------------------ Construction
      // Identity matrix
      trans_affine() : sx(1), shy(0), shx(0), sy(1), tx(0), ty(0) {}

      // Custom matrix. Usually used in derived classes
      trans_affine(REAL v0, REAL v1, REAL v2, REAL v3, REAL v4, REAL v5)
        : sx(v0), shy(v1), shx(v2), sy(v3), tx(v4), ty(v5) {}

      // Custom matrix from m[6]
      explicit trans_affine(const REAL *m)
        : sx(m[0]), shy(m[1]), shx(m[2]), sy(m[3]), tx(m[4]), ty(m[5]) {}

      // Rectangle to a parallelogram.
      trans_affine(REAL x1, REAL y1, REAL x2, REAL y2, const REAL *parl) {
        rect_to_parl(x1, y1, x2, y2, parl);
      }

      // Parallelogram to a rectangle.
      trans_affine(const REAL *parl, REAL x1, REAL y1, REAL x2, REAL y2) {
        parl_to_rect(parl, x1, y1, x2, y2);
      }

      // Arbitrary parallelogram transformation.
      trans_affine(const REAL *src, const REAL *dst) { parl_to_parl(src, dst); }

      //---------------------------------- Parellelogram transformations
      // transform a parallelogram to another one. Src and dst are
      // pointers to arrays of three points (double[6], x1,y1,...) that
      // identify three corners of the parallelograms assuming implicit
      // fourth point. The arguments are arrays of double[6] mapped
      // to x1,y1, x2,y2, x3,y3  where the coordinates are:
      //        *-----------------*
      //       /          (x3,y3)/
      //      /                 /
      //     /(x1,y1)   (x2,y2)/
      //    *-----------------*
      const trans_affine<REAL> &parl_to_parl(const REAL *src, const REAL *dst);

      const trans_affine<REAL> &rect_to_parl(REAL x1, REAL y1, REAL x2, REAL y2,
        const REAL *parl);

      const trans_affine<REAL> &parl_to_rect(const REAL *parl, REAL x1, REAL y1,
        REAL x2, REAL y2);

      //------------------------------------------ Operations
      // Reset - load an identity matrix
      trans_affine<REAL> &reset();

      // Direct transformations operations
      trans_affine<REAL> &translate(REAL x, REAL y);
      trans_affine<REAL> &translate(point_t<REAL> pt) {
        return translate(pt.x, pt.y);
      }
      trans_affine<REAL> &translate(point_t<int> pt) {
        return translate(REAL(pt.x), REAL(pt.y));
      }
      trans_affine<REAL> &rotate(REAL a);
      trans_affine<REAL> &scale(REAL s);
      trans_affine<REAL> &scale(REAL x, REAL y);
      trans_affine<REAL> &scale(size_t<REAL> sz) { return scale(sz.x, sz.y); }

      // Multiply matrix to another one
      trans_affine<REAL> &multiply(const trans_affine<REAL> &m);

      // Multiply "m" to "this" and assign the result to "this"
      const trans_affine<REAL> &premultiply(const trans_affine<REAL> &m);

      // Multiply matrix to inverse of another one
      const trans_affine<REAL> &multiply_inv(const trans_affine<REAL> &m);

      // Multiply inverse of "m" to "this" and assign the result to "this"
      const trans_affine<REAL> &premultiply_inv(const trans_affine<REAL> &m);

      // returns clone of this translated to x,y keepeing the rest
      trans_affine<REAL> absolute_translation(REAL x, REAL y) const {
        return trans_affine<REAL>(sx, shy, shx, sy, x, y);
      }

      // Invert matrix. Do not try to invert degenerate matrices,
      // there's no check for validity. If you set scale to 0 and
      // then try to invert matrix, expect unpredictable result.
      const trans_affine<REAL> &invert();

      // Mirroring around X
      const trans_affine<REAL> &flip_x();

      // Mirroring around Y
      const trans_affine<REAL> &flip_y();

      //------------------------------------------- Load/Store
      // Store matrix to an array [6] of double
      void store_to(REAL *m) const {
        *m++ = sx;
        *m++ = shy;
        *m++ = shx;
        *m++ = sy;
        *m++ = tx;
        *m++ = ty;
      }

      // Load matrix from an array [6] of double
      const trans_affine<REAL> &load_from(const REAL *m) {
        sx = *m++;
        shy = *m++;
        shx = *m++;
        sy = *m++;
        tx = *m++;
        ty = *m++;
        return *this;
      }

      //------------------------------------------- Operators

      // Multiply the matrix by another one
      const trans_affine<REAL> &operator*=(const trans_affine<REAL> &m) {
        return multiply(m);
      }

      // Multiply the matrix by inverse of another one
      const trans_affine<REAL> &operator/=(const trans_affine<REAL> &m) {
        return multiply_inv(m);
      }

      // Multiply the matrix by another one and return
      // the result in a separete matrix.
      trans_affine operator*(const trans_affine<REAL> &m) {
        return trans_affine(*this).multiply(m);
      }

      // Multiply the matrix by inverse of another one
      // and return the result in a separete matrix.
      trans_affine operator/(const trans_affine<REAL> &m) {
        return trans_affine(*this).multiply_inv(m);
      }

      // Calculate and return the inverse matrix
      trans_affine operator~() const {
        trans_affine ret = *this;
        return ret.invert();
      }

      // Equal operator with default epsilon
      bool operator==(const trans_affine<REAL> &m) const {
        return is_equal(m, real_traits<REAL>::epsilon());
      }

      // Not Equal operator with default epsilon
      bool operator!=(const trans_affine<REAL> &m) const {
        return !is_equal(m, real_traits<REAL>::epsilon());
      }

      //-------------------------------------------- Transformations
      // Direct transformation of x and y
      void transform(REAL *x, REAL *y) const;
      void transform(point_t<REAL> &pt) const {
        return transform(&pt.x, &pt.y);
      }
      void transform(point_t<int> &pt) const {
        REAL x = REAL(pt.x);
        REAL y = REAL(pt.y);
        transform(&x, &y);
        pt.x = int(x);
        pt.y = int(y);
      }

      // Direct transformation of x and y, 2x2 matrix only, no translation
      void transform_2x2(REAL *x, REAL *y) const;

      // Inverse transformation of x and y. It works slower than the
      // direct transformation. For massive operations it's better to
      // invert() the matrix and then use direct transformations.
      void inverse_transform(REAL *x, REAL *y) const;
      void inverse_transform(point_t<REAL> &pt) const {
        return inverse_transform(&pt.x, &pt.y);
      }
      void inverse_transform(point_t<int> &pt) const {
        REAL x = REAL(pt.x);
        REAL y = REAL(pt.y);
        inverse_transform(&x, &y);
        pt.x = int(x);
        pt.y = int(y);
      }

      //-------------------------------------------- Auxiliary
      // Calculate the determinant of matrix
      REAL determinant() const { return sx * sy - shy * shx; }

      // Calculate the reciprocal of the determinant
      REAL determinant_reciprocal() const {
        return 1.0f / (sx * sy - shy * shx);
      }

      // Get the average scale (by X and Y).
      // Basically used to calculate the approximation_scale when
      // decomposinting curves into line segments.
      REAL scale() const;

      // Check to see if the matrix is not degenerate
      bool is_valid(REAL epsilon = real_traits<REAL>::epsilon()) const;

      // Check to see if it's an identity matrix
      bool is_identity(REAL epsilon = real_traits<REAL>::epsilon()) const;

      // Check to see if two matrices are equal
      bool is_equal(const trans_affine<REAL> &m,
        REAL epsilon = real_traits<REAL>::epsilon()) const;

      // Determine the major parameters. Use with caution considering
      // possible degenerate cases.
      REAL rotation() const;
      void translation(REAL *dx, REAL *dy) const;
      void scaling(REAL *x, REAL *y) const;
      void scaling_abs(REAL *x, REAL *y) const;
    };

    //------------------------------------------------------------------------
    template <typename REAL>
    inline void trans_affine<REAL>::transform(REAL *x, REAL *y) const {
      REAL tmp = *x;
      *x = tmp * sx + *y * shx + tx;
      *y = tmp * shy + *y * sy + ty;
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    inline void trans_affine<REAL>::transform_2x2(REAL *x, REAL *y) const {
      REAL tmp = *x;
      *x = tmp * sx + *y * shx;
      *y = tmp * shy + *y * sy;
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    inline void trans_affine<REAL>::inverse_transform(REAL *x, REAL *y) const {
      REAL d = determinant_reciprocal();
      REAL a = (*x - tx) * d;
      REAL b = (*y - ty) * d;
      *x = a * sy - b * shx;
      *y = b * sx - a * shy;
    }

    //------------------------------------------------------------------------
    template <typename REAL> inline REAL trans_affine<REAL>::scale() const {
      REAL x =
        real_traits<REAL>::sincos45 * sx + real_traits<REAL>::sincos45 * shx;
      REAL y =
        real_traits<REAL>::sincos45 * shy + real_traits<REAL>::sincos45 * sy;
      return real_traits<REAL>::sqrt(x * x + y * y);
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    inline trans_affine<REAL> &trans_affine<REAL>::translate(REAL x, REAL y) {
      tx += x;
      ty += y;
      return *this;
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    inline trans_affine<REAL> &trans_affine<REAL>::rotate(REAL a) {
      REAL ca = real_traits<REAL>::cos(a);
      REAL sa = real_traits<REAL>::sin(a);
      REAL t0 = sx * ca - shy * sa;
      REAL t2 = shx * ca - sy * sa;
      REAL t4 = tx * ca - ty * sa;
      shy = sx * sa + shy * ca;
      sy = shx * sa + sy * ca;
      ty = tx * sa + ty * ca;
      sx = t0;
      shx = t2;
      tx = t4;
      return *this;
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    inline trans_affine<REAL> &trans_affine<REAL>::scale(REAL x, REAL y) {
      REAL mm0 = x; // Possible hint for the optimizer
      REAL mm3 = y;
      sx *= mm0;
      shx *= mm0;
      tx *= mm0;
      shy *= mm3;
      sy *= mm3;
      ty *= mm3;
      return *this;
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    inline trans_affine<REAL> &trans_affine<REAL>::scale(REAL s) {
      REAL m = s; // Possible hint for the optimizer
      sx *= m;
      shx *= m;
      tx *= m;
      shy *= m;
      sy *= m;
      ty *= m;
      return *this;
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    inline const trans_affine<REAL> &
      trans_affine<REAL>::premultiply(const trans_affine<REAL> &m) {
      trans_affine t = m;
      return *this = t.multiply(*this);
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    inline const trans_affine<REAL> &
      trans_affine<REAL>::multiply_inv(const trans_affine<REAL> &m) {
      trans_affine t = m;
      t.invert();
      return multiply(t);
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    inline const trans_affine<REAL> &
      trans_affine<REAL>::premultiply_inv(const trans_affine<REAL> &m) {
      trans_affine t = m;
      t.invert();
      return *this = t.multiply(*this);
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    inline void trans_affine<REAL>::scaling_abs(REAL *x, REAL *y) const {
      // Used to calculate scaling coefficients in image resampling.
      // When there is considerable shear this method gives us much
      // better estimation than just sx, sy.
      *x = real_traits<REAL>::sqrt(sx * sx + shx * shx);
      *y = real_traits<REAL>::sqrt(shy * shy + sy * sy);
    }

    //====================================================trans_affine_rotation
    // Rotation matrix. (REAL)sin() and (REAL)cos() are calculated twice for the
    // same angle. There's no harm because the performance of
    // (REAL)sin()/(REAL)cos() is very good on all modern processors. Besides,
    // this operation is not going to be invoked too often.
    template <typename REAL>
    class trans_affine_rotation : public trans_affine<REAL> {
    public:
      trans_affine_rotation(REAL a)
        : trans_affine<REAL>(
          real_traits<REAL>::cos(a), real_traits<REAL>::sin(a),
          -real_traits<REAL>::sin(a), real_traits<REAL>::cos(a), 0, 0) {}
    };

    //====================================================trans_affine_scaling
    // Scaling matrix. x, y - scale coefficients by X and Y respectively
    template <typename REAL>
    class trans_affine_scaling : public trans_affine<REAL> {
    public:
      trans_affine_scaling(REAL x, REAL y)
        : trans_affine<REAL>(x, 0, 0, y, 0, 0) {}

      trans_affine_scaling(REAL s) : trans_affine<REAL>(s, 0, 0, s, 0, 0) {}
    };

    //================================================trans_affine_translation
    // Translation matrix
    template <typename REAL>
    class trans_affine_translation : public trans_affine<REAL> {
    public:
      trans_affine_translation(REAL x, REAL y)
        : trans_affine<REAL>(1, 0, 0, 1, x, y) {}
    };

    //====================================================trans_affine_skewing
    // Sckewing (shear) matrix
    template <typename REAL>
    class trans_affine_skewing : public trans_affine<REAL> {
    public:
      trans_affine_skewing(REAL x, REAL y)
        : trans_affine<REAL>(1, real_traits<REAL>::tan(y),
          real_traits<REAL>::tan(x), 1, 0, 0) {}
    };

    //===============================================trans_affine_line_segment
    // Rotate, Scale and Translate, associating 0...dist with line segment
    // x1,y1,x2,y2
    template <typename REAL>
    class trans_affine_line_segment : public trans_affine<REAL> {
    public:
      trans_affine_line_segment(REAL x1, REAL y1, REAL x2, REAL y2, REAL dist) {
        REAL dx = x2 - x1;
        REAL dy = y2 - y1;
        if (dist > 0) {
          multiply(trans_affine_scaling<REAL>(
            real_traits<REAL>::sqrt(dx * dx + dy * dy) / dist));
        }
        multiply(trans_affine_rotation<REAL>(real_traits<REAL>::atan2(dy, dx)));
        multiply(trans_affine_translation<REAL>(x1, y1));
      }
    };

    //============================================trans_affine_reflection_unit
    // Reflection matrix. Reflect coordinates across the line through
    // the origin containing the unit vector (ux, uy).
    // Contributed by John Horigan
    template <typename REAL>
    class trans_affine_reflection_unit : public trans_affine<REAL> {
    public:
      trans_affine_reflection_unit(REAL ux, REAL uy)
        : trans_affine<REAL>(2 * ux * ux - 1, 2 * ux * uy, 2 * ux * uy,
          2 * uy * uy - 1, 0, 0) {}
    };

    //=================================================trans_affine_reflection
    // Reflection matrix. Reflect coordinates across the line through
    // the origin at the angle a or containing the non-unit vector (x, y).
    // Contributed by John Horigan
    template <typename REAL>
    class trans_affine_reflection : public trans_affine_reflection_unit<REAL> {
    public:
      trans_affine_reflection(REAL a)
        : trans_affine_reflection_unit<REAL>(real_traits<REAL>::cos(a),
          real_traits<REAL>::sin(a)) {}

      trans_affine_reflection(REAL x, REAL y)
        : trans_affine_reflection_unit<REAL>(
          x / real_traits<REAL>::sqrt(x * x + y * y),
          y / real_traits<REAL>::sqrt(x * x + y * y)) {}
    };

    //------------------------------------------------------------------------
    template <typename REAL>
    const trans_affine<REAL> &
      trans_affine<REAL>::parl_to_parl(const REAL *src, const REAL *dst) {
      sx = src[2] - src[0];
      shy = src[3] - src[1];
      shx = src[4] - src[0];
      sy = src[5] - src[1];
      tx = src[0];
      ty = src[1];
      invert();
      multiply(trans_affine<REAL>(dst[2] - dst[0], dst[3] - dst[1],
        dst[4] - dst[0], dst[5] - dst[1], dst[0],
        dst[1]));
      return *this;
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    const trans_affine<REAL> &
      trans_affine<REAL>::rect_to_parl(REAL x1, REAL y1, REAL x2, REAL y2,
        const REAL *parl) {
      REAL src[6];
      src[0] = x1;
      src[1] = y1;
      src[2] = x2;
      src[3] = y1;
      src[4] = x2;
      src[5] = y2;
      parl_to_parl(src, parl);
      return *this;
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    const trans_affine<REAL> &
      trans_affine<REAL>::parl_to_rect(const REAL *parl, REAL x1, REAL y1,
        REAL x2, REAL y2) {
      REAL dst[6];
      dst[0] = x1;
      dst[1] = y1;
      dst[2] = x2;
      dst[3] = y1;
      dst[4] = x2;
      dst[5] = y2;
      parl_to_parl(parl, dst);
      return *this;
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    trans_affine<REAL> &
      trans_affine<REAL>::multiply(const trans_affine<REAL> &m) {
      REAL t0 = sx * m.sx + shy * m.shx;
      REAL t2 = shx * m.sx + sy * m.shx;
      REAL t4 = tx * m.sx + ty * m.shx + m.tx;
      shy = sx * m.shy + shy * m.sy;
      sy = shx * m.shy + sy * m.sy;
      ty = tx * m.shy + ty * m.sy + m.ty;
      sx = t0;
      shx = t2;
      tx = t4;
      return *this;
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    const trans_affine<REAL> &trans_affine<REAL>::invert() {
      REAL d = determinant_reciprocal();

      REAL t0 = sy * d;
      sy = sx * d;
      shy = -shy * d;
      shx = -shx * d;

      REAL t4 = -tx * t0 - ty * shx;
      ty = -tx * shy - ty * sy;

      sx = t0;
      tx = t4;
      return *this;
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    const trans_affine<REAL> &trans_affine<REAL>::flip_x() {
      sx = -sx;
      shy = -shy;
      tx = -tx;
      return *this;
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    const trans_affine<REAL> &trans_affine<REAL>::flip_y() {
      shx = -shx;
      sy = -sy;
      ty = -ty;
      return *this;
    }

    //------------------------------------------------------------------------
    template <typename REAL> trans_affine<REAL> &trans_affine<REAL>::reset() {
      sx = sy = 1;
      shy = shx = tx = ty = 0;
      return *this;
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    bool trans_affine<REAL>::is_identity(REAL epsilon) const {
      return is_equal_eps(sx, REAL(1), epsilon) &&
        is_equal_eps(shy, REAL(0), epsilon) &&
        is_equal_eps(shx, REAL(0), epsilon) &&
        is_equal_eps(sy, REAL(1), epsilon) &&
        is_equal_eps(tx, REAL(0), epsilon) &&
        is_equal_eps(ty, REAL(0), epsilon);
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    bool trans_affine<REAL>::is_valid(REAL epsilon) const {
      //return std::abs(sx) > epsilon &&
      //       std::abs(sy) > epsilon;
      return std::abs(determinant()) > epsilon;
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    bool trans_affine<REAL>::is_equal(const trans_affine<REAL> &m,
      REAL                      epsilon) const {
      return is_equal_eps(sx, m.sx, epsilon) &&
        is_equal_eps(shy, m.shy, epsilon) &&
        is_equal_eps(shx, m.shx, epsilon) &&
        is_equal_eps(sy, m.sy, epsilon) &&
        is_equal_eps(tx, m.tx, epsilon) && is_equal_eps(ty, m.ty, epsilon);
    }

    //------------------------------------------------------------------------
    template <typename REAL> REAL trans_affine<REAL>::rotation() const {
      REAL x1 = 0;
      REAL y1 = 0;
      REAL x2 = 1;
      REAL y2 = 0;
      transform(&x1, &y1);
      transform(&x2, &y2);
      return real_traits<REAL>::atan2(y2 - y1, x2 - x1);
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    void trans_affine<REAL>::translation(REAL *dx, REAL *dy) const {
      *dx = tx;
      *dy = ty;
    }

    //------------------------------------------------------------------------
    template <typename REAL>
    void trans_affine<REAL>::scaling(REAL *x, REAL *y) const {
      REAL               x1 = 0;
      REAL               y1 = 0;
      REAL               x2 = 1;
      REAL               y2 = 1;
      trans_affine<REAL> t(*this);
      t *= trans_affine_rotation<REAL>(-rotation());
      t.transform(&x1, &y1);
      t.transform(&x2, &y2);
      *x = x2 - x1;
      *y = y2 - y1;
    }

  } // namespace geom

  typedef geom::trans_affine<double> affine_mtx_d;
  typedef geom::trans_affine<float>  affine_mtx_f;

  typedef geom::trans_affine_rotation<double>    affine_rotation_d;
  typedef geom::trans_affine_scaling<double>     affine_scaling_d;
  typedef geom::trans_affine_translation<double> affine_translation_d;
  typedef geom::trans_affine_skewing<double>     affine_skewing_d;

  typedef geom::trans_affine_rotation<float>    affine_rotation_f;
  typedef geom::trans_affine_scaling<float>     affine_scaling_f;
  typedef geom::trans_affine_translation<float> affine_translation_f;
  typedef geom::trans_affine_skewing<float>     affine_skewing_f;

  class transform3d {
  public:
    float m[16];

    transform3d() { identity(); }

    transform3d(float m0, float m4, float m8, float m12, float m1, float m5,
      float m9, float m13, float m2, float m6, float m10, float m14,
      float m3, float m7, float m11, float m15);

    void identity(void);
    void translate(float x, float y, float z);
    void translate_x(float dist);
    void translate_y(float dist);
    void translate_z(float dist);
    void rotate(float angle, float x, float y, float z);
    void perspective(float dist);
    void rotate_x(float angle);
    void rotate_y(float angle);
    void rotate_z(float angle);
    void scale(float x, float y, float z);
    void transform_point(float &x, float &y, float &z);
    void transform_vector(float &x, float &y, float &z);

    // Operators...
    transform3d operator+(const transform3d &other);
    transform3d operator-(const transform3d &other);
    transform3d operator*(const transform3d &other);

    transform3d operator*(float scalar);
  };

  inline transform3d::transform3d(float m0, float m4, float m8, float m12,
    float m1, float m5, float m9, float m13,
    float m2, float m6, float m10, float m14,
    float m3, float m7, float m11, float m15) {
    m[0] = m0;
    m[4] = m4;
    m[8] = m8;
    m[12] = m12;
    m[1] = m1;
    m[5] = m5;
    m[9] = m9;
    m[13] = m13;
    m[2] = m2;
    m[6] = m6;
    m[10] = m10;
    m[14] = m14;
    m[3] = m3;
    m[7] = m7;
    m[11] = m11;
    m[15] = m15;
  }

  inline void transform3d::identity(void) {
    m[0] = 1.0f;
    m[4] = 0.0f;
    m[8] = 0.0f;
    m[12] = 0.0f;
    m[1] = 0.0f;
    m[5] = 1.0f;
    m[9] = 0.0f;
    m[13] = 0.0f;
    m[2] = 0.0f;
    m[6] = 0.0f;
    m[10] = 1.0f;
    m[14] = 0.0f;
    m[3] = 0.0f;
    m[7] = 0.0f;
    m[11] = 0.0f;
    m[15] = 1.0f;
  }

  inline void transform3d::perspective(float depth) {
    identity();
    float proj = 0;

    if (depth > 0) { proj = -1 / depth; }
    m[14] = proj;
  }

  inline void transform3d::translate(float x, float y, float z) {
    identity();

    m[12] = x;
    m[13] = y;
    m[14] = z;
  }

  inline void transform3d::translate_x(float dist) {
    identity();
    m[12] = dist;
  }

  inline void transform3d::translate_y(float dist) {
    identity();
    m[13] = dist;
  }

  inline void transform3d::translate_z(float dist) {
    identity();
    m[14] = dist;
  }

  inline void transform3d::rotate(float angle, float ux, float uy, float uz) {
    float s = sinf(angle);
    float c = cosf(angle);

    // normalize vector
    float vlen = sqrtf(ux * ux + uy * uy + uz * uz);
    ux /= vlen;
    uy /= vlen;
    uz /= vlen;

    m[0] = c + (1 - c) * ux;
    m[1] = (1 - c) * ux * uy + s * uz;
    m[2] = (1 - c) * ux * uz - s * uy;
    m[3] = 0;

    m[4] = (1 - c) * uy * ux - s * uz;
    m[5] = c + (1 - c) * powf(uy, 2);
    m[6] = (1 - c) * uy * uz + s * ux;
    m[7] = 0;

    m[8] = (1 - c) * uz * ux + s * uy;
    m[9] = (1 - c) * uz * uz - s * ux;
    m[10] = c + (1 - c) * powf(uz, 2);
    m[11] = 0;

    m[12] = 0;
    m[13] = 0;
    m[14] = 0;
    m[15] = 1;
  }

  inline void transform3d::rotate_x(float angle) {
    float s = sinf(angle);
    float c = cosf(angle);

    identity();

    m[5] = c;
    m[6] = s;
    m[9] = -s;
    m[10] = c;
  }

  inline void transform3d::rotate_y(float angle) {
    float s = sinf(angle);
    float c = cosf(angle);

    identity();

    m[0] = c;
    m[2] = -s;
    m[8] = s;
    m[10] = c;
  }

  inline void transform3d::rotate_z(float angle) {
    float s = sinf(angle);
    float c = cosf(angle);

    identity();

    m[0] = c;
    m[1] = s;
    m[4] = -s;
    m[5] = c;
  }

  inline void transform3d::scale(float x, float y, float z) {
    identity();

    m[0] = x;
    m[5] = y;
    m[10] = z;
  }

  inline void transform3d::transform_point(float &px, float &py, float &pz) {
    float x = px;
    float y = py;
    float z = pz;

    px = x * m[0] + y * m[4] + z * m[8] + m[12];

    py = x * m[1] + y * m[5] + z * m[9] + m[13];

    pz = x * m[2] + y * m[6] + z * m[10] + m[14];
  }

  inline void transform3d::transform_vector(float &px, float &py, float &pz) {
    float x = px;
    float y = py;
    float z = pz;

    px = x * m[0] + y * m[4] + z * m[8];

    py = x * m[1] + y * m[5] + z * m[9];

    pz = x * m[2] + y * m[6] + z * m[10];
  }

  inline transform3d transform3d::operator+(const transform3d &other) {
    transform3d result;

    result.m[0] = m[0] + other.m[0];
    result.m[1] = m[1] + other.m[1];
    result.m[2] = m[2] + other.m[2];
    result.m[3] = m[3] + other.m[3];

    result.m[4] = m[4] + other.m[4];
    result.m[5] = m[5] + other.m[5];
    result.m[6] = m[6] + other.m[6];
    result.m[7] = m[7] + other.m[7];

    result.m[8] = m[8] + other.m[8];
    result.m[9] = m[9] + other.m[9];
    result.m[10] = m[10] + other.m[10];
    result.m[11] = m[11] + other.m[11];

    result.m[12] = m[12] + other.m[12];
    result.m[13] = m[13] + other.m[13];
    result.m[14] = m[14] + other.m[14];
    result.m[15] = m[15] + other.m[15];

    return result;
  }

  inline transform3d transform3d::operator-(const transform3d &other) {
    transform3d result;

    result.m[0] = m[0] - other.m[0];
    result.m[1] = m[1] - other.m[1];
    result.m[2] = m[2] - other.m[2];
    result.m[3] = m[3] - other.m[3];

    result.m[4] = m[4] - other.m[4];
    result.m[5] = m[5] - other.m[5];
    result.m[6] = m[6] - other.m[6];
    result.m[7] = m[7] - other.m[7];

    result.m[8] = m[8] - other.m[8];
    result.m[9] = m[9] - other.m[9];
    result.m[10] = m[10] - other.m[10];
    result.m[11] = m[11] - other.m[11];

    result.m[12] = m[12] - other.m[12];
    result.m[13] = m[13] - other.m[13];
    result.m[14] = m[14] - other.m[14];
    result.m[15] = m[15] - other.m[15];

    return result;
  }

  inline transform3d transform3d::operator*(const transform3d &other) {
    transform3d result;

    result.m[0] = (m[0] * other.m[0]) + (m[4] * other.m[1]) +
      (m[8] * other.m[2]) + (m[12] * other.m[3]);
    result.m[1] = (m[1] * other.m[0]) + (m[5] * other.m[1]) +
      (m[9] * other.m[2]) + (m[13] * other.m[3]);
    result.m[2] = (m[2] * other.m[0]) + (m[6] * other.m[1]) +
      (m[10] * other.m[2]) + (m[14] * other.m[3]);
    result.m[3] = (m[3] * other.m[0]) + (m[7] * other.m[1]) +
      (m[11] * other.m[2]) + (m[15] * other.m[3]);

    result.m[4] = (m[0] * other.m[4]) + (m[4] * other.m[5]) +
      (m[8] * other.m[6]) + (m[12] * other.m[7]);
    result.m[5] = (m[1] * other.m[4]) + (m[5] * other.m[5]) +
      (m[9] * other.m[6]) + (m[13] * other.m[7]);
    result.m[6] = (m[2] * other.m[4]) + (m[6] * other.m[5]) +
      (m[10] * other.m[6]) + (m[14] * other.m[7]);
    result.m[7] = (m[3] * other.m[4]) + (m[7] * other.m[5]) +
      (m[11] * other.m[6]) + (m[15] * other.m[7]);

    result.m[8] = (m[0] * other.m[8]) + (m[4] * other.m[9]) +
      (m[8] * other.m[10]) + (m[12] * other.m[11]);
    result.m[9] = (m[1] * other.m[8]) + (m[5] * other.m[9]) +
      (m[9] * other.m[10]) + (m[13] * other.m[11]);
    result.m[10] = (m[2] * other.m[8]) + (m[6] * other.m[9]) +
      (m[10] * other.m[10]) + (m[14] * other.m[11]);
    result.m[11] = (m[3] * other.m[8]) + (m[7] * other.m[9]) +
      (m[11] * other.m[10]) + (m[15] * other.m[11]);

    result.m[12] = (m[0] * other.m[12]) + (m[4] * other.m[13]) +
      (m[8] * other.m[14]) + (m[12] * other.m[15]);
    result.m[13] = (m[1] * other.m[12]) + (m[5] * other.m[13]) +
      (m[9] * other.m[14]) + (m[13] * other.m[15]);
    result.m[14] = (m[2] * other.m[12]) + (m[6] * other.m[13]) +
      (m[10] * other.m[14]) + (m[14] * other.m[15]);
    result.m[15] = (m[3] * other.m[12]) + (m[7] * other.m[13]) +
      (m[11] * other.m[14]) + (m[15] * other.m[15]);

    return result;
  }

  inline transform3d transform3d::operator*(const float scalar) {
    transform3d result;

    result.m[0] = m[0] * scalar;
    result.m[1] = m[1] * scalar;
    result.m[2] = m[2] * scalar;
    result.m[3] = m[3] * scalar;

    result.m[4] = m[4] * scalar;
    result.m[5] = m[5] * scalar;
    result.m[6] = m[6] * scalar;
    result.m[7] = m[7] * scalar;

    result.m[8] = m[8] * scalar;
    result.m[9] = m[9] * scalar;
    result.m[10] = m[10] * scalar;
    result.m[11] = m[11] * scalar;

    result.m[12] = m[12] * scalar;
    result.m[13] = m[13] * scalar;
    result.m[14] = m[14] * scalar;
    result.m[15] = m[15] * scalar;

    return result;
  }

} // namespace gool

#endif
