﻿#include "gdi+.h"

#include "tool/tool.h"
#include "d2d/d2d.h"

#include "win/win-application.h"
#include "win/win-delayload.h"

#if defined(USE_GDI)

#include <mlang.h>


namespace gdi {

  using namespace Gdiplus;

  void setup_text_flow( html::view& v, html::element* elem, html::tflow::text_flow& tf, tool::slice< tool::handle<html::node> > nodes );

  static GdiplusStartupInput gdiplus_startupInput;
  static ULONG_PTR           gdiplus_token = 0;
  static tristate_v          gdiplus_ok;   
    
  application* app_factory()
  {
    auto_ptr<application> pa;  
    do
    {
      pa = auto_ptr<application>(new application());

      if(Gdiplus::Ok != GdiplusStartup(&gdiplus_token, &gdiplus_startupInput, NULL))
        break;

      //pa->_installed_fonts = new Gdiplus::InstalledFontCollection();
      //if(!pa->_installed_fonts)
      //  break;

      gdiplus_ok = true;
      return pa.release();
    
    } while(false);

    return nullptr;
  }

  static handle_pool<gdi::font> _fonts;

  application::application()
  {
    //platform::setup_text_flow = setup_text_flow;
    //platform::draw_glyph_runs = draw_glyph_runs;
    //frame::init(true);
  }

  void application::final_release() {
    super::final_release();
    if(gdiplus_ok)
      GdiplusShutdown(gdiplus_token);
  }

  bool application::is_valid() { return gdiplus_ok != 0; }

  void gdi_clear_font_cache() {
    for(int n = 0; n < _fonts.size();++n) {
      _fonts.elements()[n]->_gdi_font = 0;
      _fonts.elements()[n]->hfont = 0;
    }
    _fonts.clear();
  }

  gool::font* gdi_create_font(const ustring& name
                    , float size
                    , uint weight
                    , bool italic
                    , FONT_RENDERING  mode)
  {
    gdi::font key;
    key.name = name;
    key.size = size;
    key.weight = weight;
    key.italic = italic;
    key.mode = mode;

    gdi::font* pf = _fonts.intern(key);
    if( pf ) gdi_init_font(pf);
    return pf;
  }

  bool          gdi_init_font(gool::font* gf);
  tool::ustring gdi_supported_font_family(const tool::ustring& family_name_list /*comma separated list*/ );


  tool::ustring family_name( const FontFamily* pff ) {
    WCHAR name[LF_FACESIZE] = {0};
    if( Ok == pff->GetFamilyName(name) )
      return name;
    return tool::ustring();
  }


  static pool<ustring> families;
  static pool<ustring> installed_families;

  static int CALLBACK EnumAllFontFamiliesProcW(
    ENUMLOGFONTEXW *lpelfe,    // logical-font data
    NEWTEXTMETRICEXW *lpntme,  // physical-font data
    DWORD FontType,            // type of font
    LPARAM lParam              // application-defined data
  )
  {
    if(FontType & TRUETYPE_FONTTYPE)
      families.intern( ustring(lpelfe->elfLogFont.lfFaceName).to_lower() );
    return 1;
  }
  bool known_font_family(wchars name)
  {
    if( families.size() == 0 ) {
      LOGFONTW logfont;
      memset(&logfont,0,sizeof(logfont));
      logfont.lfCharSet = DEFAULT_CHARSET;
      HDC sDC = GetDC(0);
      EnumFontFamiliesExW(sDC, &logfont, (FONTENUMPROCW)EnumAllFontFamiliesProcW,0,0);
      ReleaseDC(0,sDC);
    }
    ustring nm(name); nm.to_lower();
    return families.exists(nm) || installed_families.exists(nm);
  }


  bool gdi_init_font(gool::font* gf)
  {
    gdi::font* pf = static_cast<gdi::font*>(gf);

    if(pf->_gdi_font && pf->hfont) return true;

    ustring font_name = gdi_supported_font_family(pf->name);

    pf->hfont = 
          CreateFont(
            -int(gf->size + 0.5f),      // height of the font
            0,                          // average character width
            0,                          // angle of escapement
            0,                          // base-line orientation angle
            gf->weight,                 // font weight
            gf->italic?TRUE:FALSE,         // italic attribute option
            FALSE,                      // underline attribute option
            FALSE,                      // strikeout attribute option
            DEFAULT_CHARSET,            // character set identifier
            OUT_DEFAULT_PRECIS,         // output precision
            CLIP_DEFAULT_PRECIS,        // clipping precision
            CLEARTYPE_QUALITY,          // output quality
            DEFAULT_PITCH | FF_DONTCARE,// pitch and family
            font_name                   // typeface name
          );

    uint font_style = 0;
    if( gf->weight > 400 ) font_style |= 1;
    if( gf->italic ) font_style |= 2;

    screen_dc sdc;

    sdc.select(pf->hfont);

    handle<Font> f = new Font(sdc,pf->hfont); 
                     //new Font(font_name,(gf->size * 72) / 96,font_style,Gdiplus::UnitPoint);
    Gdiplus::FontFamily ff;
    f->GetFamily(&ff);
    pf->_gdi_font = f;
    //float p_height = f->GetHeight(float(pixels_per_inch().y));
    //float du_height = ff.GetLineSpacing(font_style);
    float ratio = pf->_du_ratio = //p_height / du_height;
      f->GetSize()  / ff.GetEmHeight(font_style);

    uint lineh = (int)ceilf(ff.GetLineSpacing(font_style) * ratio);
    pf->ascent = (int)floorf(ff.GetCellAscent(font_style) * ratio);
    pf->descent = lineh - pf->ascent;
    pf->x_height = lineh / 2;


    /*if(pf->ascent < 0) {
      pf->ascent = 16; 
      pf->descent = 4;
      pf->x_height = 20;
    }*/

    DWORD sz = GetFontUnicodeRanges(sdc,NULL);
    if (sz) {
      array<byte> glyphsets_bytes(sz);
      GetFontUnicodeRanges(sdc, (LPGLYPHSET)glyphsets_bytes.head());
      LPGLYPHSET glyphsets = (LPGLYPHSET)glyphsets_bytes.head();

      pf->_ranges.size(glyphsets->cRanges);

      font::char_range* pr = pf->_ranges.head();
    
      for(uint r = 0; r < glyphsets->cRanges; ++r, ++pr)
      {
        auto ra = glyphsets->ranges[r];
        pr->first = ra.wcLow;
        pr->length = ra.cGlyphs;
      }
    }
    else {
      // corrupted font?
      // assume it has at least ASCII printable range
      font::char_range cr;
      cr.first = ' ';
      cr.length = '~' - ' ';
      pf->_ranges.push(cr);
    }

    //tool::sort(pf->_ranges.head(),pf->_ranges.length());
    std::sort(pf->_ranges.begin(),pf->_ranges.end());

    return pf->_gdi_font != nullptr;
  }

  tool::ustring gdi_supported_font_family(const tool::ustring& family_name_list /*comma separated list*/ )
  {
    wtokens tz( family_name_list,WCHARS(",;"));
    wchars name;
  
    while(tz.next(name))
    {
      name = trim(name);
      if( icmp(name,WCHARS("monospace"))  ) 
        return gdi_supported_font_family(WCHARS("consolas,courier new")); 
        //return family_name(FontFamily::GenericMonospace());
      else if( icmp(name,WCHARS("sans-serif")) )
        return gdi_supported_font_family(WCHARS("segoe ui,tahoma,arial")); 
        //return family_name(FontFamily::GenericSansSerif());
      else if( icmp(name,WCHARS("serif")) )
        return WCHARS("times new roman"); 
        //return family_name(FontFamily::GenericSerif());
      else if( icmp(name,WCHARS("cursive")) )
        return gdi_supported_font_family(WCHARS("vladimir script,freestyle script")); 
      else if( icmp(name,WCHARS("fantasy")) )
        return gdi_supported_font_family(WCHARS("jokeman,comic sans ms")); 
      else if( name.like(L"system*"))
        return mswin::system_font_name(name);
      else if( known_font_family(name) )
        return name;
      /*
      FontFamily families[2048];
      int        nfamilies;
      _installed_fonts->GetFamilies(items_in(families), families, &nfamilies);

      for( int i = 0; i < nfamilies; ++i ) 
      {
        ustring fn = family_name(&families[i]);
        if( icmp(name,fn()) )
          return fn;
      }*/
    }
    return  mswin::system_font_name();
  }

  bool gdi_font_subst(
        tool::handle<gool::font>& pf,
        const gool::font* proto,
        tool::wchars lang_id, 
        WRITING_SCRIPT script, 
        uint char_code)
  {
    assert(char_code);

    pf = proto;
    static IMLangFontLink* font_link = 0;

    if (!font_link) {
      com::asset<IMultiLanguage> multi_language;
      if (::CoCreateInstance(CLSID_CMultiLanguage, 0, CLSCTX_ALL,
                             IID_IMultiLanguage,
                             reinterpret_cast<void**>(multi_language.target())) != S_OK)
        return false;
      if (multi_language->QueryInterface(&font_link) != S_OK)
        return false;
    }

    DWORD code_pages[2] = {0,0};
    long  n_code_pages = 0;

    wchar w2[2] = {0,0};

    uint n = u16::putc(char_code,w2);

    font_link->GetStrCodePages(w2,n,0,code_pages,&n_code_pages);
    //font_link->GetCharCodePages(WCHAR(char_code), &code_pages);

    screen_dc sdc; 

    sdc.select( static_cast<const gdi::font*>(proto)->hfont );

    HFONT mlang_font = NULL;
    LOGFONTW lf = {0};

    if( SUCCEEDED( font_link->MapFont( sdc, code_pages[0], static_cast<const gdi::font*>(proto)->hfont, &mlang_font))) {
      ::GetObject(mlang_font, sizeof(LOGFONT), &lf);
      font_link->ReleaseFont(mlang_font);
    }
    else 
      return false;

    //if ( SUCCEEDED(font_link->MapFont(sdc, code_pages[0], WCHAR(char_code), &mlang_font)) && mlang_font) {
    //  ::GetObject(mlang_font, sizeof(LOGFONT), &lf);
    //  font_link->ReleaseFont(mlang_font);
    //}
    //else 
    //  return false;

    pf = static_cast<gdi::font*> ( gdi_create_font(
                      lf.lfFaceName
                    , proto->size
                    , proto->weight
                    , proto->italic
                    , proto->mode) );
    
    assert(pf->has_glyph_for(char_code));

    return true;

  }

  wchars get_fallback_list(WRITING_SCRIPT script, wchars lang_id, bool serif )
  {
     switch(script)
     {
        default:
        case WS_UNKNOWN: 
        case WS_CYRILLIC: return serif
                            ? WCHARS("Times New Roman,!")
                            : WCHARS("Segoe UI,Arial,!");
        case WS_HANZI:
          if(lang_id.starts_with(WCHARS("ja")))
            return serif
               ? WCHARS("MS Mincho,MS明朝,!")
               : WCHARS("Meiryo UI,メイリオ UI,!");
          else if(lang_id.ends_with(WCHARS("tw")) || lang_id.ends_with(WCHARS("hant")))
            return serif  // traditional
               ? WCHARS("MingLiU-ExtB,MingLiU,名流,名流-ExtB,!")
               : WCHARS("MingLiU-ExtB,Microsoft JhengHei,微軟正黑體,名流-ExtB,!");
          else 
            return serif  // simplified
               ? WCHARS("NSimSun,SimSun-ExtB,SimSun,!")
               : WCHARS("SimSun-ExtB,SimSun,Microsoft YaHei,微软雅黑,!");
        case WS_KANA:
          return serif
             ? WCHARS("MS Mincho,MS明朝,!")
             : WCHARS("Meiryo UI,メイリオ UI,!");
        case WS_ARABIC:
          return serif
             ? WCHARS("Arabic Typesetting,!")
             : WCHARS("Arabic Simplified,!");
        case WS_HEBREW:
          return serif
             ? WCHARS("Narkisim,!")
             : WCHARS("Miriam,!");
        case WS_HANGUL:
          return serif
             ? WCHARS("Batang,!")
             : WCHARS("Malgun Gothic,!");
     }      
  }
  
  bool gdi_used_font(
        tool::handle<gool::font>& pf,
        tool::wchars family_name_list, const gool::font* proto,
        tool::wchars lang_id, 
        WRITING_SCRIPT script, 
        uint char_code)
  {
    assert(char_code);

    uint iterations = 0;

    std::function<bool(tool::wchars family_name_list)> lookup;

    lookup = [&](tool::wchars family_name_list) -> bool
    {
      ++iterations;
    
      if(script == WS_UNKNOWN && char_code != 0)
        script = writing_script(char_code);
        
      wtokens tz( family_name_list,WCHARS(",;"));
      wchars name;

      while(tz.next(name))
      {
        name = trim(name);
     
        if( icmp(name,WCHARS("monospace")) )
          return lookup(WCHARS("consolas,courier new,sans-serif"));

        else if( icmp(name,WCHARS("sans-serif")) )
          return lookup(get_fallback_list(script,lang_id,false));

        else if( icmp(name,WCHARS("serif")) )
          return lookup(get_fallback_list(script,lang_id,true));

        else if( icmp(name,WCHARS("cursive")) )
          return lookup(WCHARS("vladimir script,freestyle script,sans-serif"));

        else if( icmp(name,WCHARS("fantasy")) )
          return lookup(WCHARS("jokeman,comic sans ms,sans-serif"));

        else if( name.like(L"system*") )
        {
          ustring sys_name = mswin::system_font_name(name) + ",!";
          return lookup(sys_name);
        }
        else if( name == WCHARS("!") ) // misericorde
        {
          ustring sys_name = mswin::system_font_name(name);
          pf = mswin::app()->create_font(sys_name,proto->size,proto->weight,proto->italic,proto->mode);
          if(!pf->has_glyph_for(char_code))
            gdi_font_subst(pf,proto,lang_id,script,char_code);
          return true;
        }
                
        if(known_font_family(name))
        {
          tool::handle<gool::font> tpf = mswin::app()->create_font(name,proto->size,proto->weight,proto->italic,proto->mode);
          if( tpf->has_glyph_for(char_code) ) {
            pf = tpf;
            return true;
          }
        }
      
      }
      return lookup(get_fallback_list(script,lang_id,false));
    };

    return lookup(family_name_list);
  }

  bool gdi_install_font(const tool::ustring& name, tool::bytes data)
  {
    ustring nm = name;
    nm.to_lower();
    if(installed_families.exists(nm))
      return true;

    DWORD n = 0;
    HANDLE h = AddFontMemResourceEx((LPVOID)data.start,DWORD(data.length),0,&n);
    if(h && n)
    {
      installed_families.intern(nm);
      //families.clear();
      return true;
    } 
    return false;
  }
  
#if 0
  static printing_application* _instance = nullptr;

  printing_application::printing_application()
  {
    //platform::setup_text_flow = setup_text_flow;
    //platform::draw_glyph_runs = draw_glyph_runs;
    assert(_instance == nullptr);
    _instance = this;
  }

  void printing_application::final_release() {
    assert(_instance == this);
    super::final_release();
    if(gdiplus_ok)
      GdiplusShutdown(gdiplus_token);
    _instance = nullptr;
  }

  bool printing_application::is_valid() { return gdiplus_ok != 0; }


  gool::application* printing_application::create() {
    critical_section _(guard);
    if(_instance) return _instance;

    if(Gdiplus::Ok != GdiplusStartup(&gdiplus_token, &gdiplus_startupInput, NULL))
       return nullptr;

    gdiplus_ok = true;
    return new printing_application();
  }
#endif

  html::print_view* application::create_print_processor(const window_params& params)
  {
    html::print_view* ppv = new gdi::print_view(params);
    ppv->app = this;
    return ppv;
  }


}

#endif