#include "d2d.h"
#include "d2d-application.h"
//#include "d2d-window.h"
#include "d2d-text-analysis.h"
#include <numeric>

uint32 estimate_glyph_count(uint32 text_length) {
  return 3 * text_length / 2 + 16;
}

namespace d2d {

  void shape_glyph_runs(view &v, tflow::text_flow &tf, element *elem,
                        IDWriteTextAnalyzer *textAnalyzer);

  bool text_analysis::exec(view &v, element *el, tflow::text_flow &tf,
                           slice<hnode> nodes) {
    text_analysis inst;

    d2d::asset<IDWriteTextAnalyzer> dw_text_analyzer;
    
    HRESULT hr = dw_factory()->CreateTextAnalyzer(dw_text_analyzer.target());
    if (FAILED(hr)) return false;

    if (inst.generate_results(v, el, nodes, dw_text_analyzer, tf._text,
                              tf._runs, tf._breakpoints)) {
      inst.shape_glyph_runs(v, el, tf, dw_text_analyzer);
      return true;
    }
    return false;
  }

  uint next_ucp(wchars rc) {
    return u16::getc(rc);
  }

  bool text_analysis::generate_results(
      view &v, element *elem, slice<hnode> nodes,
      IDWriteTextAnalyzer *textAnalyzer, OUT tool::array<wchar> &text,
      OUT tool::array<tflow::text_run> &runs,
      OUT tool::array<LINE_BREAKPOINT> &breakpoints) {
    hstyle  es   = elem->get_style(v);
    ustring lang = elem->get_lang();

    _reading_direction = es->direction == direction_rtl
                             ? READING_DIRECTION_RIGHT_TO_LEFT
                             : READING_DIRECTION_LEFT_TO_RIGHT;

#ifdef DEBUG
    //if (elem->parent->is_id_test())
    //  es = es;
#endif

    uint next_run_index = 0;
    text.clear();
    runs.clear();
    breakpoints.clear();

    auto push_run = [&](node *pnode, uint start, uint length, uint dom_start, char_mark_t marks) {
      linked_text_run &run = _runs.push();
      run.node             = pnode;
      run.next_run_index   = ++next_run_index;
      run.text_start       = start;
      run.text_length      = length;
      run.bidi_level(pnode->get_style(v)->direction == direction_rtl ? 1 : 0);
      run.text_node_start = dom_start;
      run.marks           = marks;
    };

    if (elem->c_style->collapse_ws()) {
      auto receiver = [&](wchars span, char_marks marks, node *pnode) {
        uint start = static_cast<uint32>(text.length());
        if (span.length == 1 && span[0] == 0) {
          text.push(span); // inline block
          push_run(pnode, start, static_cast<uint32>(text.length()) - start, 0, char_mark_t());
        } else
          visible_chars_collapse_ws(pnode, span, marks, text, push_run, lang);
      };
      flatten_nodes(v, elem, nodes, receiver);

    } else {
      auto receiver = [&](wchars span, char_marks marks, node *pnode) {
        uint start = static_cast<uint32>(text.length());
        if (span.length == 1 && span[0] == 0) {
          text.push(span); // inline block
          push_run(pnode, start, static_cast<uint32>(text.length()) - start, 0,
                   char_mark_t());
        } else
          // visible_chars(span,text);
          visible_chars_break_nl(pnode, span, marks, text, push_run, lang);

        // push_run(pnode, start, static_cast<uint32>(text.length()) - start,0);
      };
      flatten_nodes(v, elem, nodes, receiver);
    }

    if (text.length() == 0 && nodes.length == 1 && nodes[0]->is_text() &&
        nodes[0].ptr_of<html::text>()->is_space()) {
      // special case for empty paragraphs with forced flow:text
      // note: needed for content-editable
      text.push(EMPTY_STRING_PLACEHOLDER);
      push_run(nodes[0], 0, 1, 0, char_mark_t());
    }

    _text = text();
    if (_text.length == 0) return true;

    // Analyzes the text using each of the analyzers and returns
    // their results as a series of runs.

    HRESULT hr = S_OK;

    try {
      // Initially start out with one result that covers the entire range.
      // This result will be subdivided by the analysis processes.

      // Allocate enough room to have one breakpoint per code unit.
      _breakpoints.length(_text.length);

      // Call each of the analyzers in sequence, recording their results.
      if (FAILED(hr = textAnalyzer->AnalyzeLineBreakpoints(
                     this, 0, UINT32(_text.length), this)))
        return false;
      if (FAILED(hr = textAnalyzer->AnalyzeBidi(this, 0, UINT32(_text.length),
                                                this)))
        return false;
      if (FAILED(hr = textAnalyzer->AnalyzeScript(this, 0, UINT32(_text.length),
                                                  this)))
        return false;
      // if (FAILED(hr = textAnalyzer->AnalyzeNumberSubstitution(this, 0,
      // UINT32(_text.length), this))) return false;

      /*        d2d::text_layout tl(v,_text,
         elem->get_style(v)->get_text_format(v), d2d_app()); for( uint32 pos =
         0; pos < uint32(_text.length); ++pos ) {
              }*/

      for (index_t i = _runs.last_index(); i >= 0; --i) // new runs (if any) will have numbers total_runs
      {
        // uint rl = (uint)_runs.length();
        linked_text_run &r  = _runs[i];
        html::style *    rs = r.get_style(v);
        if (r.is_inline_block_pos(_text)) {
          r.used_font = v.get_font(rs);
          r.no_visual(false);
          // r.script.shapes = DWRITE_SCRIPT_SHAPES_NO_VISUAL;
          continue;
        }

        if (rs != es && !rs->can_wrap() && r.text_length) {
          uint nm = r.text_start + r.text_length - 1;
          for (uint n = r.text_start + 1; n < nm; ++n) {
            LINE_BREAKPOINT &bp = _breakpoints[n];
            if (bp.break_condition_after != BREAK_CONDITION_MUST_BREAK)
              bp.break_condition_after = BREAK_CONDITION_MAY_NOT_BREAK;
            if (bp.break_condition_before != BREAK_CONDITION_MUST_BREAK)
              bp.break_condition_before = BREAK_CONDITION_MAY_NOT_BREAK;
          }
          LINE_BREAKPOINT &bp1 = _breakpoints[r.text_start];
          LINE_BREAKPOINT &bp2 = _breakpoints[nm];
          if (bp1.break_condition_after != BREAK_CONDITION_MUST_BREAK)
            bp1.break_condition_after = BREAK_CONDITION_MAY_NOT_BREAK;
          if (bp2.break_condition_after != BREAK_CONDITION_MUST_BREAK)
            bp2.break_condition_before = BREAK_CONDITION_MAY_NOT_BREAK;
        }

        wchars run_chars = r.chars(_text.start);

        // if(!setup_used_run_font(v, rs, r, run_chars, lang))
        //  view::debug_printf(OT_DOM, OS_ERROR,"Font fallback failure for font
        //  %S fallback %S.\n",
        //    r.decl_font->name.c_str(),
        //    r.used_font->name.c_str());

        wchars font_list = rs->font_family();

#ifdef _DEBUG
        if (!rs->font_size.is_points())
          r.node->dbg_report("wrong font size");
#endif

        handle<font> decl_font = static_cast<d2d::font *>(v.get_font(rs));

        // check each character in the run for supported glyphs, split the run
        // and set substitution font.

        while (run_chars.length) {
          const wchar *start = run_chars.start;
          const wchar *end   = start;

          uint ucp = u16::getc(run_chars);

          if (decl_font->has_glyph_for(ucp)) continue;
          end = run_chars.start;

          handle<gool::font> fb_font;

          auto script = writing_script(ucp);

          if (!app()->get_used_font(fb_font, font_list, decl_font, lang, script, ucp))
            continue;

          while (run_chars.length)
          {
            end = run_chars.start;
            if (script == WS_EMOTICON) 
            {
              if (next_ucp(run_chars) != 0x200d) // ZWJ
                break; // one glyph per run, due to problems with dw_factory4->TranslateColorGlyphRun
            }
            auto t = run_chars;
            ucp = u16::getc(run_chars);
            if (ucp != 0x200d && (decl_font->has_glyph_for(ucp) || !fb_font->has_glyph_for(ucp)))
            {
              run_chars = t;
              break;
            }
            script = writing_script(ucp);
            end = run_chars.start;
          }

          uint text_position = uint(start - _text.start);
          uint text_length   = uint(end - start);
          set_current_run(text_position);
          split_current_run(text_position);
          while (text_length > 0) {
            linked_text_run &run = fetch_next_run(&text_length);
            run.used_font        = fb_font.ptr();
          }
        }
      }

      // Exchange our results with the caller's.
      breakpoints.swap(_breakpoints);

      size_t total_runs = _runs.length();

      // Resequence the resulting runs in order before returning to caller.
      runs.length(total_runs);

      UINT32 nextRunIndex = 0;
      for (int i = 0; i < int(total_runs); ++i) {
        runs[i] = _runs[nextRunIndex];
        // wchars tt = runs[i].chars(_text.start);
        nextRunIndex = _runs[nextRunIndex].next_run_index;
      }

    } catch (...) { return false; }

    return true;
  }

  // IMPLEMENTATION

  ////////////////////////////////////////////////////////////////////////////////
  // IDWriteTextAnalysisSource source implementation

  IFACEMETHODIMP
  text_analysis::GetTextAtPosition(UINT32 text_position,
                                   OUT WCHAR const **text_string,
                                   OUT UINT32 *text_length) throw() {
    if (text_position >= _text.length) {
      // Return no text if a query comes for a position at the end of
      // the range. Note that is not an error and we should not return
      // a failing HRESULT. Although the application is not expected
      // to return any text that is outside of the given range, the
      // analyzers may probe the ends to see if text exists.
      *text_string = NULL;
      *text_length = 0;
    } else {
      *text_string = &_text[text_position];
      *text_length = UINT32(_text.length - text_position);
    }
    return S_OK;
  }

  IFACEMETHODIMP
  text_analysis::GetTextBeforePosition(UINT32 text_position,
                                       OUT WCHAR const **text_string,
                                       OUT UINT32 *text_length) throw() {
    if (text_position == 0 || text_position > _text.length) {
      // Return no text if a query comes for a position at the end of
      // the range. Note that is not an error and we should not return
      // a failing HRESULT. Although the application is not expected
      // to return any text that is outside of the given range, the
      // analyzers may probe the ends to see if text exists.
      *text_string = NULL;
      *text_length = 0;
    } else {
      *text_string = &_text[0];
      *text_length = text_position -
                     0; // text length is valid from current position backward
    }
    return S_OK;
  }

  DWRITE_READING_DIRECTION STDMETHODCALLTYPE
                           text_analysis::GetParagraphReadingDirection() throw() {
    return (DWRITE_READING_DIRECTION)_reading_direction;
  }

  IFACEMETHODIMP
  text_analysis::GetLocaleName(UINT32 text_position, OUT UINT32 *text_length,
                               OUT WCHAR const **localeName) throw() {
    // The pointer returned should remain valid until the next call,
    // or until analysis ends. Since only one locale name is supported,
    // the text length is valid from the current position forward to
    // the end of the string.
    set_current_run(text_position);
    if (_current_run_index < _runs.length()) {
      *localeName  = _runs[_current_run_index].node->get_lang();
      *text_length = UINT32(_text.length - text_position);
    } else {
      *localeName  = 0;
      *text_length = 0;
    }
    return S_OK;
  }

  IFACEMETHODIMP text_analysis::GetNumberSubstitution(
      UINT32 text_position, OUT UINT32 *text_length,
      OUT IDWriteNumberSubstitution **numberSubstitution) throw() {
    // if (_number_substitution != NULL)
    //    _number_substitution->AddRef();
    //*numberSubstitution = _number_substitution;
    *numberSubstitution = NULL;
    *text_length        = UINT32(_text.length - text_position);

    return S_OK;
  }

  ////////////////////////////////////////////////////////////////////////////////
  // IDWriteTextAnalysisSink implementation

  IFACEMETHODIMP text_analysis::SetLineBreakpoints(
      UINT32 text_position, UINT32 text_length,
      DWRITE_LINE_BREAKPOINT const *lineBreakpoints // [text_length]
      ) throw() {
    if (text_length > 0) {
      static_assert(sizeof(tflow::LINE_BREAKPOINT) ==
                        sizeof(DWRITE_LINE_BREAKPOINT),
                    "types does not match");
      _breakpoints.target().copy(
          index_t(text_position),
          (const tflow::LINE_BREAKPOINT *)lineBreakpoints, size_t(text_length));
      // used to be:   mem cpy(&_breakpoints[text_position], lineBreakpoints,
      // text_length * sizeof(lineBreakpoints[0]));
    }
    return S_OK;
  }

  IFACEMETHODIMP text_analysis::SetScriptAnalysis(
      UINT32 text_position, UINT32 text_length,
      DWRITE_SCRIPT_ANALYSIS const *scriptAnalysis) throw() {

    // dbg_printf("text_analysis::SetScriptAnalysis %d/%d uid
    // %d\n",text_position,text_length,scriptAnalysis->script);

    try {
      set_current_run(text_position);
      split_current_run(text_position);
      while (text_length > 0) {
        linked_text_run &run        = fetch_next_run(&text_length);
        run.script_analysis.eScript = scriptAnalysis->script;
        run.no_visual(scriptAnalysis->shapes == DWRITE_SCRIPT_SHAPES_NO_VISUAL);
      }
    } catch (...) {
      return E_FAIL; // Unknown error, probably out of memory.
    }

    return S_OK;
  }

  IFACEMETHODIMP text_analysis::SetBidiLevel(UINT32 text_position,
                                             UINT32 text_length,
                                             UINT8  explicitLevel,
                                             UINT8  resolvedLevel) throw() {
    try {
      set_current_run(text_position);
      split_current_run(text_position);
      while (text_length > 0) {
        linked_text_run &run = fetch_next_run(&text_length);
        run.bidi_level(resolvedLevel);
      }
    } catch (...) {
      return E_FAIL; // Unknown error, probably out of memory.
    }

    return S_OK;
  }

  IFACEMETHODIMP text_analysis::SetNumberSubstitution(
      UINT32 text_position, UINT32 text_length,
      IDWriteNumberSubstitution *numberSubstitution) throw() {
    try {
      set_current_run(text_position);
      split_current_run(text_position);
      while (text_length > 0) {
        linked_text_run &run = fetch_next_run(&text_length);
        run.is_number_substituted(numberSubstitution != NULL);
      }
    } catch (...) {
      return E_FAIL; // Unknown error, probably out of memory.
    }

    return S_OK;
  }

  void shape_glyph_run(view &v, tflow::text_flow &tf, element *elem,
                       IDWriteTextAnalyzer *textAnalyzer, uint32 run_index,
                       IN OUT uint32 &glyph_start);

  void text_analysis::shape_glyph_runs(view &v, element *elem,
                                       tflow::text_flow &   tf,
                                       IDWriteTextAnalyzer *textAnalyzer) {
    // Shapes all the glyph runs in the layout.

#ifdef DEBUG
    //if (elem->parent->is_id_test())
    //  elem = elem;
#endif

    uint32 text_length = static_cast<UINT32>(tf._text.length());
    if (!text_length) return;

    // Estimate the maximum number of glyph indices needed to hold a string.
    uint32 estimated_glyph_count = estimate_glyph_count(text_length);

    tf._glyph_indices.size(estimated_glyph_count);
    tf._glyph_offsets.size(estimated_glyph_count);
    tf._glyph_advances.size(estimated_glyph_count);
    tf._glyph_clusters.size(text_length);

    uint32 glyph_start = 0;

    // uint total = _runs.length();

    // Shape each run separately. This is needed whenever script, locale,
    // or reading direction changes.
    for (UINT32 runIndex = 0; runIndex < tf._runs.length(); ++runIndex)
      shape_glyph_run(v, tf, elem, textAnalyzer, runIndex, glyph_start);

    tf._glyph_indices.size(glyph_start);
    tf._glyph_offsets.size(glyph_start);
    tf._glyph_advances.size(glyph_start);
    tf._glyph_justified_advances = tf._glyph_advances;
  }

  /*inline physical_font* used_font_face(font* pf, const text_run& run)
  {
    //return run.do_glyphs_fallback
    //        ? mswin::app().get_fallback_font(run.script.script)
    //        : pf;
    return run.used_font;
  }*/

  static_assert(sizeof(LINE_BREAKPOINT) == sizeof(DWRITE_LINE_BREAKPOINT),
                "LINE_BREAKPOINT != DWRITE_LINE_BREAKPOINT");

  void shape_glyph_run(view &v, tflow::text_flow &tf, element *elem,
                       IDWriteTextAnalyzer *textAnalyzer, uint32 run_index,
                       IN OUT uint32 &glyph_start) {
    // Shapes a single run of text into glyphs.
    // Alternately, you could iteratively interleave shaping and line
    // breaking to reduce the number glyphs held onto at once. It's simpler
    // for this demostration to just do shaping and line breaking as two
    // separate processes, but realize that this does have the consequence that
    // certain advanced fonts containing line specific features (like Gabriola)
    // will shape as if the line is not broken.

    if (run_index >= tf._runs.length()) // how come?
      return;

    text_run &run = tf._runs[run_index];

    UINT32 text_start  = run.text_start;
    UINT32 text_length = run.text_length;
    UINT32 maxglyph_count = static_cast<UINT32>(tf._glyph_indices.size() - glyph_start);
    UINT32 actual_glyph_count = 0;

    element *el = run.get_element();

    if (run.is_inline_block_pos(tf._text())) {
      // el->check_layout(v);
      // assert(el->is_inline_block_element(v));
      run.glyph_start = glyph_start;
      run.glyph_count = 1;
      if (text_length > maxglyph_count) {
        UINT32 totalGlyphsArrayCount = glyph_start + 1;
        tf._glyph_indices.size(totalGlyphsArrayCount);
      }
      tf._glyph_advances.length(max(static_cast<size_t>(glyph_start + 1),
                                    tf._glyph_advances.length()));
      tf._glyph_offsets.length(max(static_cast<size_t>(glyph_start + 1),
                                   tf._glyph_offsets.length()));
      tf._glyph_advances[glyph_start] =
          static_cast<float>(el->min_inline_margin_box_width(v));
      tf._glyph_indices[glyph_start] = 0;
      glyph_start += 1;
      return;
    }

    ustring      locale_name = el->get_lang();
    html::style *style       = el->get_style(v);
    font *       pf          = static_cast<d2d::font *>(run.get_used_font(v));

    float font_em_size = pf->size;
    bool  gdi_glyphs   = style->font_rendering_mode != 0;

    // pf = used_font_face(pf,run);
    // d2d::asset<IDWriteFontFace> font_face = dw_font_face(pf, run);

    run.glyph_start = glyph_start;
    run.glyph_count = 0;

    if (text_length == 0) return; // Nothing to do..

    HRESULT hr = S_OK;

    ////////////////////
    // Allocate space for shaping to fill with glyphs and other information,
    // with about as many glyphs as there are text characters. We'll actually
    // need more glyphs than codepoints if they are decomposed into separate
    // glyphs, or fewer glyphs than codepoints if multiple are substituted
    // into a single glyph. In any case, the shaping process will need some
    // room to apply those rules to even make that determintation.

    if (text_length > maxglyph_count) {
      maxglyph_count               = estimate_glyph_count(text_length);
      UINT32 totalGlyphsArrayCount = glyph_start + maxglyph_count;
      tf._glyph_indices.size(totalGlyphsArrayCount);
    }

    tool::buffer<DWRITE_SHAPING_TEXT_PROPERTIES, 256>  textProps(text_length);
    tool::buffer<DWRITE_SHAPING_GLYPH_PROPERTIES, 256> glyphProps(maxglyph_count);

    ////////////////////
    // Get the glyphs from the text, retrying if needed.

    int                    tries      = 0;
    DWRITE_SCRIPT_ANALYSIS run_script = {uint16(run.script_analysis.eScript),
                                         run.no_visual()
                                             ? DWRITE_SCRIPT_SHAPES_NO_VISUAL
                                             : DWRITE_SCRIPT_SHAPES_DEFAULT};

    DWRITE_FONT_FEATURE font_features[16] = {};
    DWRITE_TYPOGRAPHIC_FEATURES typo_features = { font_features, 0 };
    const DWRITE_TYPOGRAPHIC_FEATURES* font_features_list = &typo_features;

    UINT feature_ranges = 0;

    if (uint ligatures = style->font_variant_ligatures.val())  
    {
      BOOL common = ligatures & font_variant_ligatures_no_common ? FALSE : TRUE;
      typo_features.features[typo_features.featureCount].parameter = common;
      typo_features.features[typo_features.featureCount++].nameTag = DWRITE_FONT_FEATURE_TAG_STANDARD_LIGATURES;
      typo_features.features[typo_features.featureCount].parameter = common;
      typo_features.features[typo_features.featureCount++].nameTag = DWRITE_FONT_FEATURE_TAG_CONTEXTUAL_LIGATURES;

      BOOL discretionary = ligatures & font_variant_ligatures_no_discretionary ? FALSE : TRUE;
      typo_features.features[typo_features.featureCount].parameter = discretionary;
      typo_features.features[typo_features.featureCount++].nameTag = DWRITE_FONT_FEATURE_TAG_DISCRETIONARY_LIGATURES;

      BOOL historical = ligatures & font_variant_ligatures_no_historical ? FALSE : TRUE;
      typo_features.features[typo_features.featureCount].parameter = historical;
      typo_features.features[typo_features.featureCount++].nameTag = DWRITE_FONT_FEATURE_TAG_HISTORICAL_LIGATURES;

      BOOL contextual = ligatures & font_variant_ligatures_no_contextual ? FALSE : TRUE;
      typo_features.features[typo_features.featureCount].parameter = contextual;
      typo_features.features[typo_features.featureCount++].nameTag = DWRITE_FONT_FEATURE_TAG_CONTEXTUAL_ALTERNATES;
      
      assert(typo_features.featureCount <= items_in(font_features));

      feature_ranges = 1;
    }

    if (uint caps = style->font_variant_caps.val())
    {
      switch (caps) {
        //case font_variant_caps_normal: 
        case font_variant_caps_small: 
          typo_features.features[typo_features.featureCount].nameTag = DWRITE_FONT_FEATURE_TAG_SMALL_CAPITALS;
          typo_features.features[typo_features.featureCount++].parameter = TRUE;
          break;
        case font_variant_caps_all_small:
          typo_features.features[typo_features.featureCount].nameTag = DWRITE_FONT_FEATURE_TAG_SMALL_CAPITALS;
          typo_features.features[typo_features.featureCount++].parameter = TRUE;
          typo_features.features[typo_features.featureCount].nameTag = DWRITE_FONT_FEATURE_TAG_SMALL_CAPITALS_FROM_CAPITALS;
          typo_features.features[typo_features.featureCount++].parameter = TRUE;
          break;
        case font_variant_caps_petite:
          typo_features.features[typo_features.featureCount].nameTag = DWRITE_FONT_FEATURE_TAG_PETITE_CAPITALS;
          typo_features.features[typo_features.featureCount++].parameter = TRUE;
          break;
        case font_variant_caps_all_petite:
          typo_features.features[typo_features.featureCount].nameTag = DWRITE_FONT_FEATURE_TAG_PETITE_CAPITALS;
          typo_features.features[typo_features.featureCount++].parameter = TRUE;
          typo_features.features[typo_features.featureCount].nameTag = DWRITE_FONT_FEATURE_TAG_PETITE_CAPITALS_FROM_CAPITALS;
          typo_features.features[typo_features.featureCount++].parameter = TRUE;
          break;
        case font_variant_caps_unicase:
          typo_features.features[typo_features.featureCount].nameTag = DWRITE_FONT_FEATURE_TAG_UNICASE;
          typo_features.features[typo_features.featureCount++].parameter = TRUE;
          break;
        case font_variant_caps_titling:
          typo_features.features[typo_features.featureCount].nameTag = DWRITE_FONT_FEATURE_TAG_TITLING;
          typo_features.features[typo_features.featureCount++].parameter = TRUE;
          break;
      }
      assert(typo_features.featureCount <= items_in(font_features));

      feature_ranges = 1;
    }
    

    do {
      wchars text = tf._text(text_start, text_start + text_length);

      hr = textAnalyzer->GetGlyphs(
          text.start, text_length, pf->dw_font_face(),
          run.is_sideways(),  // is_sideways,
          BOOL(run.is_rtl()), // isRightToLeft
          &run_script, locale_name,
          NULL, //(run.is_number_substituted) ? _number_substitution : NULL,
          &font_features_list, // features
          &text_length, // featureLengths
          feature_ranges, //1, featureRanges
          maxglyph_count,  // maxglyph_count
          &tf._glyph_clusters[text_start], &textProps[0],
          &tf._glyph_indices[glyph_start], &glyphProps[0], &actual_glyph_count);

      tries++;

      if (hr == E_NOT_SUFFICIENT_BUFFER) {
        // Try again using a larger buffer.
        maxglyph_count               = estimate_glyph_count(maxglyph_count);
        UINT32 totalGlyphsArrayCount = glyph_start + maxglyph_count;
        glyphProps.reset(maxglyph_count);
        tf._glyph_indices.size(totalGlyphsArrayCount);
      } else {
        if (tf._glyph_indices[glyph_start] == 0 && text[0] > ' ') {
          for (uint i = 0; i < actual_glyph_count; ++i)
            tf._glyph_indices[glyph_start + i] = 0xfffd;
        }
        break;
      }
    } while (tries < 2); // We'll give it two chances.

    if (FAILED(hr)) return;

    ////////////////////
    // Get the placement of the all the glyphs.

    tf._glyph_advances.length(
        max(static_cast<size_t>(glyph_start + actual_glyph_count),
            tf._glyph_advances.length()));
    tf._glyph_offsets.length(
        max(static_cast<size_t>(glyph_start + actual_glyph_count),
            tf._glyph_offsets.length()));

    if (gdi_glyphs)

      hr = textAnalyzer->GetGdiCompatibleGlyphPlacements(
          &tf._text[text_start], &tf._glyph_clusters[text_start], &textProps[0],
          text_length, &tf._glyph_indices[glyph_start], &glyphProps[0],
          actual_glyph_count, pf->dw_font_face(), font_em_size,
          FLOAT(v.pixels_per_dip(sizef(1)).y), NULL,
          FALSE, // use GDI natural
          run.is_sideways(),
          BOOL(run.is_rtl()), // isRightToLeft
          &run_script, locale_name,
          NULL, // features
          NULL, // featureRangeLengths
          0,    // featureRanges
          &tf._glyph_advances[glyph_start],
          (DWRITE_GLYPH_OFFSET *)&tf._glyph_offsets[glyph_start]);

    else

      hr = textAnalyzer->GetGlyphPlacements(
          &tf._text[text_start], &tf._glyph_clusters[text_start], &textProps[0],
          text_length, &tf._glyph_indices[glyph_start], &glyphProps[0],
          actual_glyph_count, pf->dw_font_face(), font_em_size,
          run.is_sideways(),
          BOOL(run.is_rtl()), // isRightToLeft
          &run_script, locale_name,
          NULL, // features
          NULL, // featureRangeLengths
          0,    // featureRanges
          &tf._glyph_advances[glyph_start],
          (DWRITE_GLYPH_OFFSET *)&tf._glyph_offsets[glyph_start]);

    // auto dtest = _glyph_indices();

    if (FAILED(hr)) return;

    ////////////////////
    // Certain fonts, like Batang, contain glyphs for hidden control
    // and formatting characters. So we'll want to explicitly force their
    // advance to zero.
    if (run.no_visual())
      tf._glyph_advances.set(glyph_start, glyph_start + actual_glyph_count, 0.0f);

    ////////////////////
    // Set the final glyph count of this run and advance the starting glyph.
    run.glyph_count = actual_glyph_count;
    glyph_start += actual_glyph_count;
  }

#if !defined(WINDOWLESS)
  void window::setup_text_flow(html::element *elem, html::tflow::text_flow &tf,
                               tool::slice<tool::handle<html::node>> nodes) {
    d2d::text_analysis::exec(*this, elem, tf, nodes);
  }

  void
  print_view::setup_text_flow(html::element *elem, html::tflow::text_flow &tf,
                              tool::slice<tool::handle<html::node>> nodes) {
    d2d::text_analysis::exec(*this, elem, tf, nodes);
  }

  void print_view::draw_glyph_run_back(gool::graphics *              gfx,
                                       const html::tflow::text_flow &tf,
                                       const html::tflow::glyph_run &run,
                                       gool::rectf rc, argb back_color) {
    if (this->drawing_content) {
      auto ln = tf.get_line(run.line_no);
      if (ln.page_no && this->page_no != ln.page_no) return;
      if (!page_y.covers(rect(rc).y())) return;
      ln.page_no = uint16(this->page_no);
    }
    gfx->fill(back_color, rc);
  }
#endif

} // namespace d2d
