#ifndef __d2d_graphics_h__
#define __d2d_graphics_h__

#include <d2d1.h>
#include <d2d1helper.h>
#include <dwrite.h>

#include "tool/tool.h"
#include "gool/gool.h"
#include "d2d-objects.h"
#include "d2d-primitives.h"
#include <math.h>

namespace html {
  struct iwindow;
}

namespace d2d {
  using namespace tool;
  using namespace gool;

  GRAPHICS_CAPS get_graphics_caps();

  struct image_brush_def {
    handle<image>           img;
    locked::counter_t       generation;
    rect                    src;
    com::asset<ID2D1BitmapBrush> brush;
  };
    

  // class state;
  // class text_layout;

  interface __declspec(uuid("7165B7A1-AB2B-4423-908B-E42A1EA76799")) __declspec(
      novtable) IFillAndStroke : public IUnknown {
    virtual ID2D1Brush *      fillBrush()   = 0;
    virtual ID2D1Brush *      strokeBrush() = 0;
    virtual float             strokeWidth() = 0;
    virtual ID2D1StrokeStyle *strokeStyle() = 0;
  };

  struct font : public gool::font {
    virtual ~font() {}

    virtual bool has_glyph_for(uint ucodepoint) const override {
      if (ucodepoint <= ' ') return true;
      BOOL    exists = FALSE;
      HRESULT hr     = dw_font()->HasCharacter(ucodepoint, &exists);
      assert(SUCCEEDED(hr));
      hr;
      return exists != FALSE;
    }

    virtual uint glyph_index(uint ucodepoint) const override {
      uint16  index = 0;
      HRESULT hr    = dw_font_face()->GetGlyphIndicesW(&ucodepoint, 1, &index);
      assert(SUCCEEDED(hr));
      hr;
      return index;
    }

    virtual bool glyph_metrics(uint16 glyph_index, float em_size, float &width,
                               float &height) const override {
      DWRITE_FONT_METRICS fm = {0};
      _font->GetMetrics(&fm);

      HRESULT              hr = 0;
      DWRITE_GLYPH_METRICS glyph_metrics;
      if (mode == FR_UI)
        hr = dw_font_face()->GetGdiCompatibleGlyphMetrics(
            size, FLOAT(resolution_provider::desktop().pixels_per_dip(sizef(1)).y), 0,
            FALSE, &glyph_index, 1, &glyph_metrics);
      else {
        hr = dw_font_face()->GetDesignGlyphMetrics(&glyph_index, 1,
                                                   &glyph_metrics);
      }
      assert(SUCCEEDED(hr));
      if (FAILED(hr)) return false;
      float m = em_size / fm.designUnitsPerEm;
      width   = glyph_metrics.advanceWidth * m;
      height  = glyph_metrics.advanceHeight * m;
      return true;
    }

    virtual void metrics(int &a, int &d, int &x, float sz = 0) const override {
      if (!ascent) {
        DWRITE_FONT_METRICS fm = {0};
        _font->GetMetrics(&fm);
        float m = (sz ? sz : size) / fm.designUnitsPerEm;
        x       = int(fm.xHeight * m + 0.5f);
        a       = int(fm.ascent * m + 0.5f);
        d       = int(fm.descent * m + 0.5f);
      } else {
        x = x_height;
        a = ascent;
        d = descent;
      }
    }

    void dw_font(IDWriteFont *pf) {
      assert(pf);
      _font                  = pf;
      _font_face             = 0;
      DWRITE_FONT_METRICS fm = {0};
      _font->GetMetrics(&fm);
      float m  = size / fm.designUnitsPerEm;
      x_height = int(fm.xHeight * m + 0.5f);
      ascent   = int(fm.ascent * m);
      descent = int((fm.ascent + fm.descent) * m + 0.5f) - ascent;

      dw_font_face();
      
/*
      int caph = int(fm.capHeight * m + 0.5f);
      int delta = (int(fm.ascent * m + 0.5f) - caph) / 2;
      x_height = int(fm.xHeight * m + 0.5f);
      ascent = caph + delta; //int(fm.ascent * m);
      descent  = int((fm.ascent + fm.descent) * m + 0.5) - ascent;
*/

    }

    IDWriteFont *    dw_font() const { return _font; }
    IDWriteFontFace *dw_font_face() const {
      if (!_font_face) _font->CreateFontFace(_font_face.target());
      return _font_face;
    }
    //void dw_font_face(IDWriteFontFace *ff) const { _font_face = ff; }

    void drop_cache() {
      _font      = 0;
      _font_face = 0;
    }

  private:
    mutable com::asset<IDWriteFont>     _font;
    mutable com::asset<IDWriteFontFace> _font_face;
  };

  class path : public gool::path {
  public:
    friend class d2d::graphics;
    path() : _opened(false), _sealed(false) {}
    virtual ~path() {
      ps = 0;
      pg = 0;
    }

    virtual void set(const polygonf &vertices, bool even_odd = true);
    virtual void set(const polygonf &v1, const polygonf &v2);

    virtual bool is_inside(pointf pt);

    virtual void reset() override;
    void         check_sink();
    virtual void seal() override;

    ID2D1PathGeometry *get_geometry() const {
      const_cast<path *>(this)->seal();
      return pg;
    }

    virtual void start(pointf start, bool filled) override;
    virtual void move_to(pointf pt, bool rel = false) override;
    virtual void line_to(pointf pt, bool rel = false) override;
    virtual void hline_to(float x, bool rel = false) override {
      line_to(pointf(x, rel ? 0 : lastp().y), rel);
    }
    virtual void vline_to(float y, bool rel = false) override {
      line_to(pointf(rel ? 0 : lastp().x, y), rel);
    }

    virtual void  add_arc(pointf c, sizef r, float angle_start,
                          float angle_swipe) override;
    virtual void  quadratic_to(pointf pt, pointf cp, bool rel = false) override;
    virtual void  cubic_to(pointf pt, pointf cp1, pointf cp2,
                           bool rel = false) override; // bezier
    virtual void  arc_to(pointf to, sizef r, float rotation_angle,
                         bool large_arc, bool clockwise,
                         bool rel_to = false) override;
    virtual rectf bounds() const override;

    virtual void set_even_odd(bool yes) override;
    virtual void close() override;
    virtual bool is_empty() const override { return !_opened; }

    virtual pointf lastp() { return _lastp; }
    virtual void   lastp(pointf p) { 
      _lastp = p; 
    }

    virtual gool::path *combine(COMBINE_MODE      mode,
                                const gool::path *other) override;

  protected:
    bool   _sealed;
    bool   _opened;
    pointf _firstp;
    pointf _lastp;

    com::asset<ID2D1PathGeometry> pg;
    com::asset<ID2D1GeometrySink> ps;
  };

  // GRAPHICS class
  class graphics : public gool::graphics,
                   public com::base<graphics, com::list<IDWriteTextRenderer>> {
    typedef gool::graphics super;
    // friend class state;

    enum { MAX_CACHE_ENTRIES = 512 };

    gool::point _offset;
#if defined(USE_D2D_PLUS)
    D2D1_INTERPOLATION_MODE _image_rendering_mode;
#endif

  public:
    ID2D1RenderTarget *render_target() const
    {
#if defined(USE_D2D_PLUS)
      if (_device_context)
        return _device_context;
#endif
      return _render_target;
    }
    void               render_target(ID2D1RenderTarget *prt) {
      _render_target = prt;
#if defined(USE_D2D_PLUS)
      device_context();
#endif
    }

#if defined(USE_D2D_PLUS)
    ID2D1DeviceContext *device_context() const { 
      if(!_device_context && _render_target)
        _render_target->QueryInterface<ID2D1DeviceContext>(const_cast<graphics*>(this)->_device_context.target());
      return _device_context;
    }

    ID2D1DeviceContext4 *device_context_4() const {
      if (!_device_context_4 && device_context())
        device_context()->QueryInterface<ID2D1DeviceContext4>(const_cast<graphics*>(this)->_device_context_4.target());
      return _device_context_4;
    }

    void              device_context(ID2D1DeviceContext *pdc)
    {
      _render_target = pdc;
      _device_context = pdc;
      _device_context_4 = nullptr;
    }
#endif


    virtual GRAPHICS_CAPS graphics_caps() const { return HARDWARE_GRAPHICS; }

    graphics(bool transparent)
        : super(transparent)
#if defined(USE_D2D_PLUS)
      , _image_rendering_mode(D2D1_INTERPOLATION_MODE_LINEAR)
#endif
    {
    }
    virtual ~graphics() { }

    virtual void final_release() 
    {
      _path.clear();
      _registers.clear();
      _registers_stack.clear();
      _dotted_stroke  = nullptr;
      _dashed_stroke  = nullptr;
      _rounded_stroke = nullptr;
      _solid_stroke   = nullptr;
      reset(); // that must be destroyed before render_target.
      super::final_release();
    }

    //virtual ID2D1SolidColorBrush *get_brush(const argb &c) {
    //  return get_solid_brush(c);
    //}
    // virtual ID2D1LinearGradientBrush*
    //    get_brush(const linear_brush& c) { return
    //    get_linear_gradient_brush(c); }

    virtual void set_tag(void *ptr) {
      if (render_target()) render_target()->SetTags(D2D1_TAG(ptr), 0);
    }
    virtual void *get_tag() {
      D2D1_TAG t1 = 0, t2 = 0;
      if (render_target()) {
        render_target()->GetTags(&t1, &t2);
        return (void *)t1;
      }
      return 0;
    }

    virtual void clear(const rect& rc) override {
      if (auto rt = render_target()) {
        D2D1_RECT_F clip = g2f(rc);
        rt->PushAxisAlignedClip(clip, D2D1_ANTIALIAS_MODE_ALIASED);
        rt->Clear(D2D1::ColorF(0x000000, 0.0f));
        rt->PopAxisAlignedClip();
      }
    }

    virtual void on_window_position_change() {}
    virtual void on_window_active_state_change(bool active) {}

    virtual gool::path *create_path() const override { return new d2d::path(); }
    virtual gool::application *app() const override;

    virtual point offset(point noff) {
      point off = _offset;
      _offset   = noff;
      _clip_rc -= noff;
      render_target()->SetTransform(
          D2D1::Matrix3x2F::Translation(float(_offset.x), float(_offset.y)));
      return off;
    }

    virtual void fill(argb c, const rect &rc) override {
      com::asset<ID2D1SolidColorBrush> brush;
      if (get_solid_brush(c, brush))
        render_target()->FillRectangle(g2f(rc), brush);
    }

    virtual void fill(argb c, const rect &rc, const size &radius) override {
      D2D1_ROUNDED_RECT rr;
      rr.rect    = g2f(rc);
      rr.radiusX = float(radius.x);
      rr.radiusY = float(radius.y);

      com::asset<ID2D1SolidColorBrush> brush;
      if (get_solid_brush(c, brush))
        render_target()->FillRoundedRectangle(rr, brush);
    }

    virtual void fill(image *img, const rect &dst, const rect &src,
      point offset = point(0, 0), size dstcell = size()) override;


    virtual void fill(const brush *pgb, const rect &rc) {
      switch (pgb->type()) {
      case brush::LINEAR:
        fill_linear_gradient(*static_cast<const linear_brush *>(pgb), rc);
        break;
      case brush::RADIAL:
        fill_radial_gradient(*static_cast<const radial_brush *>(pgb), rc);
        break;
      }
    }

#if 0

    ID2D1SolidColorBrush *get_solid_brush(const gool::argb &k) {
      bool                              created = false;
      com::asset<ID2D1SolidColorBrush> &pa = solid_brushes.get_ref(k, created);
      HRESULT                           hr =
          created ? render_target()->CreateSolidColorBrush(g2f(k), pa.target())
                  : S_OK;
      assert(SUCCEEDED(hr));
      hr = hr;

      if (solid_brushes.size() > MAX_CACHE_ENTRIES) {
        com::asset<ID2D1SolidColorBrush> tpa = pa;
        solid_brushes.clear();
        bool                              created = false;
        com::asset<ID2D1SolidColorBrush> &npa =
            solid_brushes.get_ref(k, created);
        assert(created);
        npa = tpa;
      }
      return pa;
    }

    hash_table<gool::argb, com::asset<ID2D1SolidColorBrush>> solid_brushes;
#else

    circular_buffer<pair<gool::argb, com::asset<ID2D1SolidColorBrush>>,8> solid_brushes;

    bool get_solid_brush(const gool::argb &k, com::asset<ID2D1SolidColorBrush>& pb ) {

      HRESULT hr = render_target()->CreateSolidColorBrush(g2f(k), pb.target());
      return SUCCEEDED(hr);

      FOREACH(i, solid_brushes) {
        if (solid_brushes[i].first == k)
          return solid_brushes[i].second;
      }
     
      pair<gool::argb, com::asset<ID2D1SolidColorBrush>> np;
      hr = render_target()->CreateSolidColorBrush(g2f(k), np.second.target());
      if(SUCCEEDED(hr)) {
        solid_brushes.push(np);
        return np.second;
      }
      return solid_brushes[0].second;
    }

#endif 

    bool get_linear_gradient_brush(const linear_brush& k, com::asset<ID2D1LinearGradientBrush> &rpa) {
      HRESULT hr = S_OK;

      bool created = false;
      com::asset<ID2D1LinearGradientBrush>& pa = linear_gradient_brushes.get_ref(k, created);

      if (!created) {
        rpa = pa;
        return true;
      }

      ID2D1RenderTarget *                   prt = render_target();
      D2D1_LINEAR_GRADIENT_BRUSH_PROPERTIES bp;
      bp.startPoint = g2f(k.start);
      bp.endPoint   = g2f(k.end);

      slice<gool::color_stop>   src_sv = k.stops();
      array<D2D1_GRADIENT_STOP> sv(src_sv.length);

      for (uint n = 0; n < src_sv.length; ++n) {
        D2D1_GRADIENT_STOP cs;
        cs.color    = g2f(argb(src_sv[n].color));
        cs.position = src_sv[n].position;
        sv[n]       = cs;
      }

      com::asset<ID2D1GradientStopCollection> pstops;
      hr = prt->CreateGradientStopCollection(sv.head(), uint(sv.length()),
                                             pstops.target());
      assert(SUCCEEDED(hr));
      if (SUCCEEDED(hr)) {
        hr = prt->CreateLinearGradientBrush(bp, pstops, pa.target());
        assert(SUCCEEDED(hr));
      } else
        return false;

      rpa = pa;

      if (linear_gradient_brushes.size() > MAX_CACHE_ENTRIES) {
        linear_gradient_brushes.remove_at(0); // remove oldest entry
        bool created = false;
        com::asset<ID2D1LinearGradientBrush> &npa = linear_gradient_brushes.get_ref(k, created);
        //assert(created);
        npa = rpa;
      }
      return true;
    }

    hash_table<linear_brush, com::asset<ID2D1LinearGradientBrush>> linear_gradient_brushes;

    bool get_radial_gradient_brush(const radial_brush& k, com::asset<ID2D1RadialGradientBrush> &rpa) {
      HRESULT hr = S_OK;

      bool created = false;
      com::asset<ID2D1RadialGradientBrush>& pa = radial_gradient_brushes.get_ref(k, created);

      if (!created) {
        rpa = pa;
        return true;
      }

      ID2D1RenderTarget *                   prt = render_target();
      D2D1_RADIAL_GRADIENT_BRUSH_PROPERTIES bp;
      bp.center                 = g2f(k.center);
      bp.radiusX                = k.radius.x;
      bp.radiusY                = k.radius.y;
      bp.gradientOriginOffset.x = k.focus.x;
      bp.gradientOriginOffset.y = k.focus.y;

      slice<gool::color_stop>   src_sv = k.stops();
      array<D2D1_GRADIENT_STOP> sv(src_sv.length);

      for (uint n = 0; n < src_sv.length; ++n) {
        D2D1_GRADIENT_STOP cs;
        cs.color    = g2f(argb(src_sv[n].color));
        cs.position = src_sv[n].position;
        sv[n]       = cs;
      }

      com::asset<ID2D1GradientStopCollection> pstops;
      hr = prt->CreateGradientStopCollection(sv.head(), uint(sv.length()),
                                             pstops.target());
      assert(SUCCEEDED(hr));
      if (SUCCEEDED(hr)) {
        hr = prt->CreateRadialGradientBrush(bp, pstops, pa.target());
        assert(SUCCEEDED(hr));
      } else
        return false;

      rpa = pa;

      if (radial_gradient_brushes.size() > MAX_CACHE_ENTRIES) {
        radial_gradient_brushes.remove_at(0); // remove oldest entry
        bool created = false;
        com::asset<ID2D1RadialGradientBrush> &npa = radial_gradient_brushes.get_ref(k, created);
        //assert(created);
        npa = rpa;
      }

      return true;
    }

    hash_table<radial_brush, com::asset<ID2D1RadialGradientBrush>>
        radial_gradient_brushes;

    void fill_linear_gradient(const linear_brush &lb, const rect &rc) {
      com::asset<ID2D1LinearGradientBrush> lgb;
      if (!get_linear_gradient_brush(lb, lgb)) return;
      render_target()->FillRectangle(g2f(rc), lgb);
    }

    void fill_radial_gradient(const radial_brush &lb, const rect &rc) {
      com::asset<ID2D1RadialGradientBrush> lgb;
      if (!get_radial_gradient_brush(lb, lgb)) return;
      render_target()->FillRectangle(g2f(rc), lgb);
    }

    circular_buffer<image_brush_def,MAX_CACHE_ENTRIES>   image_brushes;
    bool get_bitmap_brush(image* pimg, rect src, com::asset<ID2D1BitmapBrush> &bb) {
      //circular_buffer<image_brush_def>   _image_brushes;
      FOREACH(n,image_brushes)
      {
        image_brush_def& ibd = image_brushes[n];
        if (ibd.img != pimg) continue;
        if (ibd.generation != pimg->generation()) {
          ibd.brush = nullptr;
          ibd.img = nullptr;
          continue;
        }
        if (ibd.src != src) continue;
        bb = ibd.brush;
        return true;
      }

      handle<bitmap> bmp = pimg->get_bitmap(this, src.size());
      if (!bmp) return false;

      image_brush_def ibd;
      ibd.src = src;
      ibd.generation = bmp->generation();
      ibd.img = pimg;

      com::asset<ID2D1Bitmap> d2bmp = D2D1Bitmap(bmp);
      if (!d2bmp) return false;

      size src_size = src.size();

      if (src_size != bmp->dim())
      {
        com::asset<ID2D1Bitmap> fragment;

        D2D1_BITMAP_PROPERTIES props = D2D1_BITMAP_PROPERTIES();
        props.dpiX = 96.0f;
        props.dpiY = 96.0f;
        props.pixelFormat.format = DXGI_FORMAT_B8G8R8A8_UNORM;
        props.pixelFormat.alphaMode =
          bmp->_has_alpha_bits
          ? D2D1_ALPHA_MODE_PREMULTIPLIED
          : D2D1_ALPHA_MODE_IGNORE; // D2D1_ALPHA_MODE_UNKNOWN;

        HRESULT  hr = render_target()->CreateBitmap(g2u(src.size()), nullptr, bmp->dim().x * sizeof(argb), props, fragment.target());
        if (SUCCEEDED(hr)) {
          D2D1_POINT_2U dp = { 0,0 };
          D2D1_RECT_U sr = g2u(src);
          if (FAILED(fragment->CopyFromBitmap(&dp, d2bmp, &sr)))
            return false;
          d2bmp = fragment;
        } else 
          return false;
      }

      D2D1_BITMAP_BRUSH_PROPERTIES bbp;
      bbp.extendModeX = D2D1_EXTEND_MODE_WRAP;
      bbp.extendModeY = D2D1_EXTEND_MODE_WRAP;
      bbp.interpolationMode = //D2D1_BITMAP_INTERPOLATION_MODE_LINEAR;
        D2D1_BITMAP_INTERPOLATION_MODE_NEAREST_NEIGHBOR;

      HRESULT hr = render_target()->CreateBitmapBrush(d2bmp, bbp, ibd.brush.target());
      if (SUCCEEDED(hr)) {
        bb = ibd.brush;
        image_brushes.push(ibd);
        return true;
      }
      return false;
    }

    /*virtual void fill( image* img, const rect& dst, const rect& src,point
offset = point(0,0))
    {
      size sz = src.size();
      if( sz.empty() )
        return;

      //assert(sz.x > 1 && sz.y > 1);
#pragma TODO("investigate better options to repeat 1px bitmaps: \
CreateBitmapBrush a) does not work properly under non trivial transformations
and \ b) does not allow to create brishes from fragments of base bitmap" )

      argb clr;
      if( (sz.x <= 8 || sz.y <= 8) && img->is_solid_color(src,clr) )
        return fill(clr,dst);

      for(int y = dst.origin.y + offset.y; y <= dst.corner.y; y += sz.y)
        for(int x = dst.origin.x + offset.x; x <= dst.corner.x; x += sz.x)
        {
          rect r_dst( point(x, y), sz);
          rect r_target = r_dst & dst;
          if( r_target == r_dst )
            do_draw(img,r_target,src);
          else
          {
            rect r_src( src );
            r_src.origin.x += r_target.origin.x - r_dst.origin.x;
            r_src.origin.y += r_target.origin.y - r_dst.origin.y;
            r_src.corner.x -= r_dst.corner.x - r_target.corner.x;
            r_src.corner.y -= r_dst.corner.y - r_target.corner.y;
            if( !r_src.empty() )
              do_draw(img,r_target,r_src);
          }
        }
    }*/

    ID2D1Bitmap *D2D1Bitmap(bitmap *bmp) {
      image_link *il = bmp->get_link_for(this);
      if (!il) return 0;
      if (il->d2d_bmp) {
        if (il->generation != bmp->_generation) {
          critical_section _(gool::lock);
          il->generation = bmp->_generation;
          HRESULT hr = il->d2d_bmp->CopyFromMemory(NULL, bmp->_pixels.head(),
                                                   bmp->_dim.x * sizeof(argb));
          assert(SUCCEEDED(hr));
          hr = hr;
        }
        return il->d2d_bmp;
      }

      HRESULT     hr     = S_OK;
      uint32      max_sz = render_target()->GetMaximumBitmapSize();
      D2D1_SIZE_U sz     = g2u(bmp->_dim);
      if (sz.width > max_sz) { sz.width = max_sz; }
      if (sz.height > max_sz) { sz.height = max_sz; }
      D2D1_BITMAP_PROPERTIES props = D2D1_BITMAP_PROPERTIES();
      props.dpiX                   = 96.0f;
      props.dpiY                   = 96.0f;
      props.pixelFormat.format     = DXGI_FORMAT_B8G8R8A8_UNORM;
      props.pixelFormat.alphaMode =
          bmp->_has_alpha_bits
              ? D2D1_ALPHA_MODE_PREMULTIPLIED
              : D2D1_ALPHA_MODE_IGNORE; // D2D1_ALPHA_MODE_UNKNOWN;

      critical_section _(gool::lock);

      hr = render_target()->CreateBitmap(sz, bmp->_pixels.head(),
                                         bmp->_dim.x * sizeof(argb), props,
                                         il->d2d_bmp.target());
      assert(SUCCEEDED(hr));
      if (FAILED(hr)) return 0;

      il->generation = bmp->_generation;
      return il->d2d_bmp;
    }

    virtual void fill(image *img, const rect &dst, point offset) {
      fill(img, dst, rect(img->dimension()), offset);

      /*
      Seems like FillRectangle by brush always uses
      D2D1_BITMAP_INTERPOLATION_MODE_NEAREST_NEIGHBOR to provide seamless
      tiling. And D2D1_BITMAP_INTERPOLATION_MODE_NEAREST_NEIGHBOR looks badly on
      sciling + animation
            

      handle<bitmap> bmp = img->get_bitmap(this, dst.size());
      if( !bmp )
      {
        img->draw(this,dst,img->dim(dst),255);
        return;
      }
          

      com::asset<ID2D1BitmapBrush> bitmapBrush;
      D2D1_BITMAP_BRUSH_PROPERTIES bbp;
      bbp.extendModeX = D2D1_EXTEND_MODE_WRAP;
      bbp.extendModeY = D2D1_EXTEND_MODE_WRAP;
      bbp.interpolationMode = D2D1_BITMAP_INTERPOLATION_MODE_LINEAR;
      //D2D1_BITMAP_INTERPOLATION_MODE_NEAREST_NEIGHBOR;

      HRESULT hr = render_target()->CreateBitmapBrush( D2D1Bitmap(bmp), bbp,
      bitmapBrush.target()); if( SUCCEEDED(hr) )
      {
        offset += dst.origin;
        bitmapBrush->SetTransform(D2D1::Matrix3x2F::Translation(D2D1::SizeF(float(offset.x),float(offset.y))));
        render_target()->FillRectangle(g2f(dst),bitmapBrush);
      }*/
    }

    virtual void fill(argb c, const gool::path *dst) override {

      com::asset<ID2D1SolidColorBrush> brush;
      if(get_solid_brush(c, brush))
        render_target()->FillGeometry( static_cast<const d2d::path *>(dst)->get_geometry(), brush);
    }

    virtual void fill(gool::image *img, const gool::path *dst) override {

      com::asset<ID2D1BitmapBrush>      bitmapBrush;
      if (get_bitmap_brush(img,img->dim(), bitmapBrush))
      {
        render_target()->FillGeometry(
            static_cast<const d2d::path *>(dst)->get_geometry(), bitmapBrush);
      }
    }

    /*virtual void draw_rect( rect rc, int stroke_width, argb c, STROKE_STYLE ss
    = STROKE_SOLID )
    {
      ID2D1StrokeStyle *pss = 0;
      D2D1_RECT_F r = g2f(rc);
      float w2 = float(stroke_width)/2;
      r.left += w2;
      r.right -= w2;
      r.top += w2;
      r.bottom -= w2;
      switch(ss)
      {
        case STROKE_NONE: return;
        case STROKE_DOTTED: pss = dotted_stroke(); break;
        case STROKE_DASHED: pss = dashed_stroke(); break;
      }
      render_target()->DrawRectangle(r,get_brush(c),float(stroke_width),pss);
    }*/

    com::asset<ID2D1StrokeStyle> get_d_stroke(int pos, int w, int stroke, int step,
                                         int_v off);

    virtual void draw_line_v(const rect &rc, argb c, int stroke, int step,
                             int_v off) override {
      if (c.alfa == 0) return;

      com::asset<ID2D1StrokeStyle> pss =
          get_d_stroke(rc.top(), rc.width(), stroke, step, off);

      com::asset<ID2D1SolidColorBrush> pbr;
      get_solid_brush(c, pbr);


      // render_target()->DrawRectangle(r, get_brush(c), float(stroke_width),
      // pss);
      float w  = float(rc.width());
      float w2 = w / 2.0f;
      // float h2 = rc.width() == 1 ? 0 : w / 2.0f;

      D2D1_POINT_2F pts = {rc.left() + w2, float(rc.top())};
      D2D1_POINT_2F pte = {rc.left() + w2, float(rc.bottom())};

      render_target()->DrawLine(pts, pte, pbr, w, pss);

      /*       int s1 = stroke;
             //rect r(rc.origin.x, 0,rc.corner.x, 0);
             ID2D1RenderTarget* prt = render_target();
             ID2D1Brush* pbr = get_brush(c);
             ID2D1StrokeStyle* pst = rc.width() == 1?
         solid_stroke():rounded_stroke();

             float w = float(rc.width());
             float w2 = w / 2.0f;
             float h2 = rc.width() == 1? 0: w / 2.0f;

             D2D1_POINT_2F pts = {rc.left() + w2,rc.top() + h2};
             D2D1_POINT_2F pte = {rc.left() + w2,rc.top() + s1 - h2};

             prt->DrawLine(pts,pte,pbr,w,pst);
             

             int space = rc.height() - step - stroke + 1;
             int num_steps = space / step;
             for(int ly = rc.origin.y + step; num_steps > 0;)
             {
               pts.y = ly + h2;
               pte.y = ly + s1 - h2;
               prt->DrawLine(pts,pte,pbr,w,pst);
               int delta = space / num_steps;
               ly += delta;
               space -= delta;
               --num_steps;
             }
             pte.y = rc.corner.y + 1 - h2;
             pts.y = rc.corner.y + 1 - s1 + h2;
             prt->DrawLine(pts,pte,pbr,w,pst);*/
    }

    virtual void draw_line_h(const rect &rc, argb c, int stroke, int step,
                             int_v off) override {
      if (c.alfa == 0) return;

      com::asset<ID2D1StrokeStyle> pss =
          get_d_stroke(rc.left(), rc.height(), stroke, step, off);

      com::asset<ID2D1SolidColorBrush> pbr;
      get_solid_brush(c, pbr);

      float w  = float(rc.height());
      float w2 = w / 2.0f;
      // float h2 = rc.height() == 1 ? 0 : w / 2.0f;

      D2D1_POINT_2F pts = {float(rc.left()), rc.top() + w2};
      D2D1_POINT_2F pte = {float(rc.right()), rc.top() + w2};

      render_target()->DrawLine(pts, pte, pbr, w, pss);

      /*
      int s1 = stroke;
      //rect r(rc.origin.x, 0,rc.corner.x, 0);
      ID2D1RenderTarget* prt = render_target();
      ID2D1Brush* pbr = get_brush(c);
      ID2D1StrokeStyle* pst = rc.height() == 1? solid_stroke():rounded_stroke();

      float w = float(rc.height());
      float h2 = w / 2.0f;
      float w2 = rc.height() == 1? 0: w / 2.0f;
      //float x = rc.left() + w2;
      //float y = rc.top();

      D2D1_POINT_2F pts = {rc.left() + w2,rc.top() + h2};
      D2D1_POINT_2F pte = {rc.left() + s1 - w2,rc.top() + h2};
      

      prt->DrawLine(pts,pte,pbr,w,pst);
      

      int space = rc.width() - step - stroke + 1;
      int num_steps = space / step;
      for(int lx = rc.origin.x + step; num_steps > 0;)
      {
        pts.x = lx + w2;
        pte.x = lx + s1 - w2;
        prt->DrawLine(pts,pte,pbr,w,pst);
        int delta = space / num_steps;
        lx += delta;
        space -= delta;
        --num_steps;
      }
      pte.x = rc.corner.x + 1 - w2;
      pts.x = rc.corner.x + 1 - s1 + w2;
      prt->DrawLine(pts,pte,pbr,w,pst);
      */
    }

    virtual void draw_line(pointf p1, pointf p2, argb c, int width) override {
      ID2D1RenderTarget *prt = render_target();
      ID2D1StrokeStyle * pst = solid_stroke();
      com::asset<ID2D1SolidColorBrush> pbr;
      if (get_solid_brush(c, pbr))
        prt->DrawLine(g2f(p1), g2f(p2), pbr, float(width), pst);
    }

    virtual void draw(const gool::path *dst, argb c,
                      float width = 1.0f) override {
      ID2D1RenderTarget *prt = render_target();
      ID2D1StrokeStyle * pst = solid_stroke();
      com::asset<ID2D1SolidColorBrush> pbr;
      if (get_solid_brush(c, pbr))
        prt->DrawGeometry(static_cast<const d2d::path *>(dst)->get_geometry(), pbr, width, pst);
    }

    virtual void draw_ellipse(pointf center, sizef radius, argb c, int width) {
      ID2D1RenderTarget *prt = render_target();
      ID2D1StrokeStyle * pst = solid_stroke();
      D2D1_ELLIPSE       ed;
      ed.point   = g2f(center);
      ed.radiusX = radius.x;
      ed.radiusY = radius.y;
      com::asset<ID2D1SolidColorBrush> pbr;
      if (get_solid_brush(c, pbr))
        prt->DrawEllipse(ed, pbr, float(width), pst);
    }

    /*
    

    void draw_arrow(const linef& l, argb c, int width)
    {
      const float h = 7.5f;
      const float w = 5.f;
      const float r = w * 0.7f;
      pointf v1 = l.unit();        // unit vector
      pointf o = l.end - h * v1;
      pointf s = l.origin + r * v1;
      pointf n = pointf(-v1.y,v1.x); // normal to the vector
      pointf a1 = o + w * n;
      pointf a2 = o  -w * n;
      draw_ellipse(l.origin,sizef(r,r),c,width);
      draw_line(s,o,c,width);
      draw_line(a2,a1,c,width);
      draw_line(a2,l.end,c,width);
      draw_line(a1,l.end,c,width);
    }*/

    /*virtual void draw( image* img, const rect&  dst, byte opacity = 255)
    {

    }*/

    virtual void draw(image *img, const rect &dst, const rect &src,
                      byte opacity = 255) {
      super::draw(img, dst, src, opacity);
    }
    virtual void draw(image *img, const rect &dst, byte opacity = 255) {
      super::draw(img, dst, opacity);
    }
    virtual void draw(image *img, const point &dst, byte opacity = 255) {
      super::draw(img, dst, opacity);
    }

    // checks if we have some non-trivial transformations that may lead to
    // border effects.
    virtual bool is_under_transform() override {
      D2D1::Matrix3x2F transform;
      render_target()->GetTransform(&transform);
      return transform.Determinant() != 1.0f ||
             fabs(fmodf(transform._31, 1.0f)) > 0.01f ||
             fabs(fmodf(transform._32, 1.0f)) > 0.01f;
      // return !transform.IsIdentity();
    }

    virtual void image_rendering_mode(IMAGE_RENDERING irm) override {
#if defined(USE_D2D_PLUS)
      switch (irm) {
      default:
      case image_rendering_default:
        _image_rendering_mode = D2D1_INTERPOLATION_MODE_LINEAR;
        break;
      case image_rendering_speed:
        _image_rendering_mode = D2D1_INTERPOLATION_MODE_NEAREST_NEIGHBOR;
        break;
      case image_rendering_quality:
        _image_rendering_mode = D2D1_INTERPOLATION_MODE_HIGH_QUALITY_CUBIC;
        break;
      }
#endif
    }

    virtual IMAGE_RENDERING image_rendering_mode() const override {
#if defined(USE_D2D_PLUS)
      switch (_image_rendering_mode) {
      default:
      case D2D1_INTERPOLATION_MODE_LINEAR: return image_rendering_default;
      case D2D1_INTERPOLATION_MODE_NEAREST_NEIGHBOR:
        return image_rendering_speed;
      case D2D1_INTERPOLATION_MODE_HIGH_QUALITY_CUBIC:
        return image_rendering_quality;
      }
#else
      return image_rendering_default;
#endif
    }

    virtual image_renderer_f image_renderer(image *img) override {

      handle<bitmap> bmp = img->get_bitmap(this, img->dimension());
      if (!bmp) return super::image_renderer(img);

      com::asset<ID2D1Bitmap> nbmp = D2D1Bitmap(bmp);

      if (!nbmp) // bitmap is invalid for some reasons
        return [=](rect dst, rect src) {};

#if defined(USE_D2D_PLUS)
      auto interpolation_mode = this->_image_rendering_mode;

      ID2D1RenderTarget *target = render_target();

      ID2D1DeviceContext *dc = device_context();

      if (dc)
        return [=](rect dst, rect src) {
          D2D1_RECT_F dstf = g2f(dst);
          D2D1_RECT_F srcf = g2f(src);
          dc->DrawBitmap(nbmp, dstf, 1.0f, interpolation_mode, srcf);
        };
      else
        return [=](rect dst, rect src) {
          D2D1_RECT_F dstf = g2f(dst);
          D2D1_RECT_F srcf = g2f(src);
          target->DrawBitmap(nbmp, dstf, 1.0f,
                             D2D1_BITMAP_INTERPOLATION_MODE_LINEAR, srcf);
        };
#endif
      return [=](rect dst, rect src) {
        D2D1_RECT_F dstf = g2f(dst);
        D2D1_RECT_F srcf = g2f(src);
        render_target()->DrawBitmap(
            nbmp, dstf, 1.0f, D2D1_BITMAP_INTERPOLATION_MODE_LINEAR, srcf);
      };
    }

    virtual void do_draw(image *img, const rectf &dst, const rect &src,
                         byte opacity = 255) override {

      if (dst.empty() || src.empty()) return;

      handle<bitmap> bmp = img->get_bitmap(this, dst.size());
      if (!bmp) {
        //-- img->draw(this,dst,src,opacity); -- ATTN: this will cause recursive
        //call!
        return;
      }

      D2D1_RECT_F dstf = g2f(dst);
      D2D1_RECT_F srcf = g2f(src);

      // swap(srcf.bottom,srcf.top);

      /*if( src.dimension() != img->dimension() )
      {
        D2D1::Matrix3x2F transform;
        render_target()->GetTransform(&transform);

        bool scaleOrRotation =
            transform._11 != 1.0 || transform._12 != 0.0 ||
            transform._21 != 0.0 || transform._22 != 1.0;
        

        if( scaleOrRotation)
        {
          dstf.bottom += .5f; // this black magic is required to overcome border
      effects on rotations. dstf.right += .5f; dstf.top -= .5f; // this black
      magic is required to overcome border effects on rotations. dstf.left -=
      .5f;
          //dstf.bottom += 1.0f; // this black magic is required to overcome
      border effects on rotations.
          //dstf.right += 1.0f;
        }
      }*/
      ID2D1Bitmap *nbmp = D2D1Bitmap(bmp);
      if (nbmp) {
#if defined(USE_D2D_PLUS)
        ID2D1DeviceContext *dc = device_context();
        if (dc)
          dc->DrawBitmap(nbmp, dstf /*g2f(dst)*/, opacity / 255.0f,
                         _image_rendering_mode, srcf);
        else
          render_target()->DrawBitmap(nbmp, dstf /*g2f(dst)*/, opacity / 255.0f,
                                      D2D1_BITMAP_INTERPOLATION_MODE_LINEAR,
                                      srcf);
#else
        render_target()->DrawBitmap(nbmp, dstf /*g2f(dst)*/, opacity / 255.0f,
                                    D2D1_BITMAP_INTERPOLATION_MODE_LINEAR,
                                    srcf);
#endif
      }
    }

    virtual bool set_antialiasing(bool onoff) override {
      D2D1_ANTIALIAS_MODE aa = render_target()->GetAntialiasMode();
      bool                r  = aa == D2D1_ANTIALIAS_MODE_PER_PRIMITIVE;
      render_target()->SetAntialiasMode(onoff
                                            ? D2D1_ANTIALIAS_MODE_PER_PRIMITIVE
                                            : D2D1_ANTIALIAS_MODE_ALIASED);
      return r;
    }

    virtual void reset() {
      solid_brushes.clear();
      linear_gradient_brushes.clear();
      radial_gradient_brushes.clear();
      image_brushes.clear();
      // bitmaps.clear();
    }

    virtual void push_opacity() override { 
      //D2D1_ANTIALIAS_MODE aa = render_target()->GetAntialiasMode();
      //if(aa == D2D1_ANTIALIAS_MODE_PER_PRIMITIVE)
      render_target()->SetTextAntialiasMode(D2D1_TEXT_ANTIALIAS_MODE_CLEARTYPE);
      super::push_opacity();
    }
    virtual void pop_opacity()  override {
      render_target()->SetTextAntialiasMode(D2D1_TEXT_ANTIALIAS_MODE_GRAYSCALE);
      super::pop_opacity();
    }

    virtual bool push_clip(const rect &r, clipper *cl) {
      render_target()->PushAxisAlignedClip(g2f(r), D2D1_ANTIALIAS_MODE_PER_PRIMITIVE);
      super::push_clip(r, cl);
      return true;
    }
    virtual void pop_clip() {
      super::pop_clip();
      render_target()->PopAxisAlignedClip();
    }

    //virtual void draw(gool::text_layout &tl, point dst, argb c) override;
    //virtual void draw(gool::text_layout &tl, pointf dst, argb c) override;
    //virtual void draw(gool::text_layout &tl, pointf dst) override; // using current fill and stroke

    // IDWriteTextRenderer implementation

    IFACEMETHOD(DrawGlyphRun)
    (void *clientDrawingContext, FLOAT baselineOriginX, FLOAT baselineOriginY,
     DWRITE_MEASURING_MODE measuringMode, const DWRITE_GLYPH_RUN *glyphRun,
     const DWRITE_GLYPH_RUN_DESCRIPTION *glyphRunDescription,
     IUnknown *                          clientDrawingEffect);

    IFACEMETHOD(DrawUnderline)
    (void *clientDrawingContext, FLOAT baselineOriginX, FLOAT baselineOriginY,
     const DWRITE_UNDERLINE *underline, IUnknown *clientDrawingEffect);

    IFACEMETHOD(DrawStrikethrough)
    (void *clientDrawingContext, FLOAT baselineOriginX, FLOAT baselineOriginY,
     const DWRITE_STRIKETHROUGH *strikethrough, IUnknown *clientDrawingEffect);

    IFACEMETHOD(DrawInlineObject)
    (void *clientDrawingContext, FLOAT originX, FLOAT originY,
     IDWriteInlineObject *inlineObject, BOOL isSideways, BOOL isRightToLeft,
     IUnknown *clientDrawingEffect);

    IFACEMETHOD(IsPixelSnappingDisabled)
    (void *clientDrawingContext, OUT BOOL *isDisabled);

    IFACEMETHOD(GetCurrentTransform)
    (void *clientDrawingContext, OUT DWRITE_MATRIX *transform);

    IFACEMETHOD(GetPixelsPerDip)
    (void *clientDrawingContext, OUT FLOAT *pixelsPerDip);
    // end of IDWriteTextRenderer implementation

    //|
    //| Graphin stuff
    //|

    virtual CAP_STYLE cap_style() override {
      return (CAP_STYLE)_registers.cap_style;
    }
    virtual void cap_style(CAP_STYLE ncs) override {
      _registers.cap_style    = (D2D1_CAP_STYLE)ncs;
      _registers.stroke_style = 0;
    }
    virtual LINE_JOIN line_join() override {
      return (LINE_JOIN)_registers.line_join;
    }
    virtual void line_join(LINE_JOIN ncs, float mitter_limit = 4) override {
      _registers.line_join    = (D2D1_LINE_JOIN)ncs;
      _registers.mitter_limit = mitter_limit;
      _registers.stroke_style = 0;
    }

    virtual void dash_style(DASH_STYLE st) override {
      _registers.dash_style   = (D2D1_DASH_STYLE)(st);
      _registers.stroke_style = 0;
    }
    virtual DASH_STYLE dash_style() override {
      return (DASH_STYLE)_registers.dash_style;
    }
    virtual void custom_dash_style(slice<float> pattern,
                                   float        offset) override {
      _registers.dash_style        = D2D1_DASH_STYLE_CUSTOM;
      _registers.custom_dash_style = pattern;
      _registers.dash_offset       = offset;
      if (_registers.stroke_width > 0)
        FOREACH(i, _registers.custom_dash_style) {
          _registers.custom_dash_style[i] /= _registers.stroke_width;
        }
      _registers.stroke_style = 0;
    }

    virtual void set_stroke(argb c = argb::no_color()) override;
    virtual void set_stroke(const linear_brush &lb) override;
    virtual void set_stroke(const radial_brush &rb) override;
    virtual void set_stroke_width(float w) override;

    virtual void set_fill(argb c = argb::no_color()) override;
    virtual void set_fill(const linear_brush &lb) override;
    virtual void set_fill(const radial_brush &rb) override;
    virtual void set_fill(const image *ib) override;
    virtual void
    set_fill_even_odd(bool even_odd /*false - non-null*/) override {
      _registers.fill_even_odd = even_odd;
    }

    virtual void draw_ellipse(pointf center, sizef radius, bool stroke = true, bool fill = true) override;
    //virtual void draw_arc(pointf center, sizef radius, float angle_start,
    //                      float angle_sweep) override;
    virtual void draw_rectangle(pointf org, sizef dim, bool stroke = true, bool fill = true) override;
    // virtual void draw_rectangle(rectf r)  override {
    // super::draw_rectangle(r); }
    virtual void draw_rectangle(pointf org, sizef dim, sizef rtl, sizef rtr,
                                sizef rbr, sizef rbl, bool stroke = true, bool fill = true) override {
      super::draw_rectangle(org, dim, rtl, rtr, rbr, rbl);
    }

    virtual void transform(const affine_mtx_f &m) override {
      D2D1::Matrix3x2F current;
      render_target()->GetTransform(&current);
      render_target()->SetTransform(g2f(m) * current);
    }
    
    virtual affine_mtx_f transform() const override {
      D2D1::Matrix3x2F current;
      render_target()->GetTransform(&current);
      return f2g(current);
    }

    virtual void reset_transform() override {
      render_target()->SetTransform(D2D1::Matrix3x2F::Identity());
    }

    virtual void translate(pointf pt) override;
    virtual void rotate(float radians, pointf center = pointf()) override;
    virtual void scale(sizef sz, pointf center = pointf()) override;
    virtual void skew(sizef sz, pointf center = pointf()) override;

    virtual pointf world_to_screen(pointf pt) const override;
    virtual float  world_to_screen(float scalar) const override {
      return super::world_to_screen(scalar);
    }

    virtual pointf screen_to_world(pointf pt) const override;
    virtual float  screen_to_world(float scalar) const override {
      return super::screen_to_world(scalar);
    }

    virtual void draw_path(const gool::path *dst, bool stroke = true,
                           bool fill = true) override {
      com::asset<ID2D1PathGeometry> geometry =
          static_cast<const path *>(dst)->get_geometry();
      if (!geometry) return;
      UINT c1 = 0;
      UINT c2 = 0;
      geometry->GetFigureCount(&c1);
      geometry->GetSegmentCount(&c2);
      if (c1 == 0 && c2 == 0) return;
      if (fill && _registers.fill_drawable())
        render_target()->FillGeometry(geometry, _registers.fill_brush);
      if (stroke && _registers.stroke_drawable())
        render_target()->DrawGeometry(geometry, _registers.stroke_brush,
                                      _registers.stroke_width,
                                      get_stroke_style());
    }

    virtual void draw_path(const gool::polygonf &p, bool stroke = true, bool fill = true);

    virtual void draw_line(pointf start, pointf end) override {
      if (_registers.stroke_drawable())
        render_target()->DrawLine(g2f(start), g2f(end), _registers.stroke_brush,
                                  _registers.stroke_width, get_stroke_style());
    }

    virtual void draw_line(pointf origin, pointf end, line_end_style sorigin,
                           line_end_style send) {
      return super::draw_line(origin, end, sorigin, send);
    }

    /*virtual void reset_path() override
    {
      _path.reset(_path_sink);
    }
    virtual void close_path() override
    {
      if(_path_sink)
        _path_sink->close();
    }*/

    struct state_registers {
      com::asset<ID2D1DrawingStateBlock> state_block;

      com::asset<ID2D1StrokeStyle> stroke_style;
      com::asset<ID2D1Brush>       stroke_brush;
      //text_format             text_format;
      float                   stroke_width;

      com::asset<ID2D1Brush> fill_brush;
      bool              fill_even_odd;

      D2D1_DASH_STYLE dash_style;
      array<float> custom_dash_style; // D2D1_DASH_STYLE_CUSTOM
      float        dash_offset;

      D2D1_CAP_STYLE cap_style;
      D2D1_LINE_JOIN line_join;

      float mitter_limit;

      D2D1::Matrix3x2F transform;

      state_registers()
          : stroke_width(0), cap_style(D2D1_CAP_STYLE_FLAT),
            line_join(D2D1_LINE_JOIN_MITER), dash_style(D2D1_DASH_STYLE_SOLID),
            dash_offset(0), mitter_limit(0), fill_even_odd(true) {}

      state_registers(const state_registers& rs)
        : state_block(rs.state_block)
        , stroke_style(rs.stroke_style)
        , stroke_brush(rs.stroke_brush)
        , stroke_width(rs.stroke_width)
        , fill_brush(rs.fill_brush)
        , fill_even_odd(rs.fill_even_odd)
        , dash_style(rs.dash_style)
        , custom_dash_style(rs.custom_dash_style)
        , dash_offset(rs.dash_offset)
        , cap_style(rs.cap_style)
        , line_join(rs.line_join)
        , transform(rs.transform) {}

      bool drawable() const { return stroke_drawable() || fill_drawable(); }
      bool stroke_drawable() const {
        return stroke_brush && stroke_width > 0.f;
      }
      bool fill_drawable() const { return fill_brush; }

      void clear() {
        state_block  = nullptr;
        stroke_style = nullptr;
        stroke_brush = nullptr;
        fill_brush   = nullptr;
      }
    };

    virtual uint push_state() override;
    virtual bool pop_state() override;
    virtual void restore_state(uint level) override;

    struct layer_t {
#ifdef USE_D2D_PLUS
      com::asset<ID2D1Image>    bitmap;
      com::asset<ID2D1Effect>   effect;  // effect head
      com::asset<ID2D1DeviceContext> otarget; // original target
#endif
      com::asset<ID2D1Layer> layer; // if any
      rect                   position;
      byte                   opacity;
    };
    array<layer_t> layers_stack;

    virtual void push_layer(
        const gool::rect &clip, byte opacity = 255,
        function<bool(filter_graph_builder *)> filters = nullptr) override;
    virtual void push_layer(const gool::path *clip,
                            byte              opacity = 255) override;
    virtual void push_layer(const gool::bitmap *clip, bool draw1a,
                            byte opacity = 255) override;
    virtual void pop_layer() override;
    virtual uint n_layers() override { return uint(layers_stack.length()); }

    ID2D1StrokeStyle *dotted_stroke();
    ID2D1StrokeStyle *dashed_stroke();
    ID2D1StrokeStyle *rounded_stroke();
    ID2D1StrokeStyle *solid_stroke();

  protected:
    ID2D1StrokeStyle *get_stroke_style();

    handle<gool::path>     _path;
    state_registers        _registers;
    array<state_registers> _registers_stack;

    com::asset<ID2D1StrokeStyle> _dotted_stroke;
    com::asset<ID2D1StrokeStyle> _dashed_stroke;
    com::asset<ID2D1StrokeStyle> _rounded_stroke;
    com::asset<ID2D1StrokeStyle> _solid_stroke;

#if defined(USE_D2D_PLUS)
    com::asset<ID2D1DeviceContext>  _device_context;
    com::asset<ID2D1DeviceContext4> _device_context_4;
#endif

    com::asset<ID2D1RenderTarget>   _render_target;
  };

#if defined(USE_D2D_PLUS)
  class dx_surface_graphics : public graphics {
    friend class application;
    friend class window;
    typedef graphics super;
  protected:
    dx_surface_graphics(bool transparent) : super(transparent), _graphics_caps() {}

  public:
    static dx_surface_graphics *create(IDXGISurface *pSurf);
    static dx_surface_graphics *create(ID2D1DeviceContext *pSurf) {
      dx_surface_graphics *pgfx = new dx_surface_graphics(true);
      pgfx->device_context(pSurf);
      return pgfx;
    }

    virtual ~dx_surface_graphics() {}

    virtual GRAPHICS_CAPS graphics_caps() const { return _graphics_caps; }

  protected:
    GRAPHICS_CAPS _graphics_caps;
  };
#endif

  inline bool is_rt_transparent(ID2D1RenderTarget *prt) {
    D2D1_PIXEL_FORMAT fmt = prt->GetPixelFormat();
    return fmt.alphaMode == D2D1_ALPHA_MODE_PREMULTIPLIED || fmt.alphaMode == D2D1_ALPHA_MODE_STRAIGHT;
  }

  class ext_graphics : public graphics {
    friend class application;
    friend class window;
    typedef graphics super;
  public:
    ext_graphics(ID2D1RenderTarget *prt):super(is_rt_transparent(prt))  {

      render_target(prt);
      D2D1_SIZE_U sz = _render_target->GetPixelSize();
      set_clip_rc(d2g(sz));
    }
    virtual ~ext_graphics() { }

    virtual gool::size size() {
      if (render_target()) {
        D2D1_SIZE_U sz = render_target()->GetPixelSize();
        return d2g(sz);
      }
      return gool::size();
    }
  };

  class dc_graphics : public graphics {
    friend class application;
    friend class window;
    typedef graphics super;
  public:
    dc_graphics(HDC hdc, rect rc, bool wants_alpha = false);
    virtual ~dc_graphics() {}

  };

  class offscreen_graphics : public graphics {
    typedef graphics super;

  public:
    offscreen_graphics(graphics *pg, size sz): super(true) {
      D2D1_PIXEL_FORMAT pf;
      pf.format    = DXGI_FORMAT_B8G8R8A8_UNORM;
      pf.alphaMode = D2D1_ALPHA_MODE_PREMULTIPLIED;

      com::asset<ID2D1BitmapRenderTarget> rt;

      HRESULT hr   = pg->render_target()->CreateCompatibleRenderTarget(
          g2f(sz), g2u(sz), pf,
          D2D1_COMPATIBLE_RENDER_TARGET_OPTIONS_GDI_COMPATIBLE, // we will
          rt.target());                                         // request HDC
        
      if (SUCCEEDED(hr)) {
        render_target(rt);
        rt->QueryInterface<ID2D1GdiInteropRenderTarget>(_interop_target.target());
      }
      
      set_clip_rc(sz);
    }
    ~offscreen_graphics() {}

    void begin() {
      _render_target->BeginDraw();
      super::begin_drawing();
    }

    void end(function<void(HDC)> flush) {
      if (_interop_target) {
        HDC hdc = 0;
        _interop_target->GetDC(D2D1_DC_INITIALIZE_MODE_COPY, &hdc);

        flush(hdc);

        RECT rect = {};
        _interop_target->ReleaseDC(&rect);
      }
      super::end_drawing();
      _render_target->EndDraw();
    }

  protected:
    com::asset<ID2D1GdiInteropRenderTarget> _interop_target;
  };

  class bitmap_graphics : public graphics {
    typedef graphics super;

  public:
    bitmap_graphics(graphics *pg, gool::bitmap *pb, argb initc = argb());
    virtual ~bitmap_graphics();

    virtual bool is_bitmap_graphics() const override { return true; }

  protected:
    graphics *                     pproto;
    handle<gool::bitmap>           pbitmap;
  };

  class bitmap_bits_graphics : public d2d::graphics {
    typedef d2d::graphics super;
    bitmap_bits_graphics(size sz) : super(true), dib(sz) {}

  public:
    static bitmap_bits_graphics *create(gool::bitmap *pb, argb initc = argb());

    virtual ~bitmap_bits_graphics() {}
    virtual void final_release() override;

    virtual bool is_bitmap_graphics() const override { return true; }

  protected:
    handle<gool::bitmap> pbitmap;
    gool::dib32          dib;

  };


  void draw_glyph_run_shadow(graphics *gfx, const DWRITE_GLYPH_RUN *glyph_run,
                             DWRITE_MEASURING_MODE dw_measuring_mode,
                             D2D1_POINT_2F dst_shadow, argb c, float radius,
                             float spread);

} // namespace d2d

#endif