#include "d2d.h"
#include "html/html.h"
#include "win/win-application.h"
#include "win/win-delayload.h"
#include <numeric>
#include <comdef.h>

DLOADV(_D2D1MakeRotateMatrix, D2D1MakeRotateMatrix, d2d1.dll,
       void(WINAPI *)(FLOAT angle, D2D1_POINT_2F center,
                      D2D1_MATRIX_3X2_F *matrix));

DLOADV(_D2D1MakeSkewMatrix, D2D1MakeSkewMatrix, d2d1.dll,
       void(WINAPI *)(FLOAT angleX, FLOAT angleY, D2D1_POINT_2F center,
                      D2D1_MATRIX_3X2_F *matrix));

DLOADV(_D2D1InvertMatrix, D2D1InvertMatrix, d2d1.dll,
       BOOL(WINAPI *)(D2D1_MATRIX_3X2_F *matrix));

//DLOADV(_CreateDXGIFactory, CreateDXGIFactory, dxgi.dll,
//       HRESULT(WINAPI *)(REFIID riid, void **ppFactory));

//DLOADV(_D3D10CreateDevice1, D3D10CreateDevice1, d3d10_1.dll,
//       HRESULT(WINAPI *)(IDXGIAdapter *pAdapter, D3D10_DRIVER_TYPE DriverType,
//                         HMODULE Software, UINT Flags,
//                         D3D10_FEATURE_LEVEL1 HardwareLevel, UINT SDKVersion,
//                         ID3D10Device1 **ppDevice));

#if defined(USE_D2D_PLUS)
EXTERN_DLOADV(_D3D11CreateDevice, D3D11CreateDevice, d3d11.dll,
       HRESULT(WINAPI *)(IDXGIAdapter *pAdapter, D3D_DRIVER_TYPE DriverType,
                         HMODULE Software, UINT Flags,
                         const D3D_FEATURE_LEVEL *pFeatureLevels,
                         UINT FeatureLevels, UINT SDKVersion,
                         ID3D11Device **       ppDevice,
                         D3D_FEATURE_LEVEL *   pFeatureLevel,
                         ID3D11DeviceContext **ppImmediateContext));
#endif

namespace gool {

  /*static value blacklist;

  bool set_gpu_black_list(const char *bl_json) {
    if (!bl_json) {
      blacklist.clear();
      return true;
    } else {
      tool::bytes blacklist_text((byte *)bl_json, strlen(bl_json));
      if (!blacklist_text) {
        blacklist.clear();
        return false;
      }
      ustring us   = u8::cvt(blacklist_text);
      wchars  usws = us;
      blacklist    = xjson::parse(usws, false);
      return blacklist.is_defined();
    }
  }
  value get_gpu_black_list() {
    if (blacklist.is_defined()) return blacklist;
    tool::bytes blacklist_text = app()->get_resource(W("gpu-blacklist.json"));
    if (!blacklist_text) return value();
    ustring us   = u8::cvt(blacklist_text);
    wchars  usws = us;
    blacklist    = xjson::parse(usws, false);
    return blacklist;
  }*/

} // namespace gool

namespace d2d {
  using namespace tool;
  using namespace gool;
  using namespace html;
  using namespace mswin;

  struct fill_stroke_def
      : public resource,
        public com::base<fill_stroke_def, com::list<IFillAndStroke>> {
    ID2D1Brush *      fillBrush() override { return fill_brush; }
    ID2D1Brush *      strokeBrush() override { return stroke_brush; }
    float             strokeWidth() override { return stroke_width; }
    ID2D1StrokeStyle *strokeStyle() override { return stroke_style; }

    com::asset<ID2D1Brush>       fill_brush;
    com::asset<ID2D1Brush>       stroke_brush;
    float                        stroke_width;
    com::asset<ID2D1StrokeStyle> stroke_style;
  };

  bool is_display_hw_capable();

  // IDWriteTextRenderer implementation

  /*html::element* dom_element(IUnknown* clientDrawingEffect)
  {
     IDomElement* pel = 0;
     if(SUCCEEDED(clientDrawingEffect->QueryInterface(&pel)))
       return static_cast<html::element*>(pel);
     //assert(pel);
     return 0;
  }*/

  IFillAndStroke *fill_stroke(IUnknown *clientDrawingEffect) {
    IFillAndStroke *pel = 0;
    if (SUCCEEDED(clientDrawingEffect->QueryInterface(&pel))) return pel;
    return 0;
  }

  void draw_glyph_run(graphics *gfx, pointf at,
                      const DWRITE_GLYPH_RUN *            glyphRun,
                      const DWRITE_GLYPH_RUN_DESCRIPTION *glyphRunDescription,
                      DWRITE_MEASURING_MODE               measuringMode,
                      IFillAndStroke *                    fill_stroke) {
    // D2D1_POINT_2F org = d2d::g2f(at);

    com::asset<ID2D1Brush> fill   = fill_stroke->fillBrush();
    com::asset<ID2D1Brush> stroke = fill_stroke->strokeBrush();

    HRESULT hr = S_OK;

    // Create the path geometry.
    com::asset<ID2D1PathGeometry> pPathGeometry = NULL;
    hr = d2_factory()->CreatePathGeometry(pPathGeometry.target());

    // Write to the path geometry using the geometry sink.
    com::asset<ID2D1GeometrySink> pSink;
    if (SUCCEEDED(hr)) { hr = pPathGeometry->Open(pSink.target()); }

    // Get the glyph run outline geometries back from DirectWrite and place them
    // within the geometry sink.
    if (SUCCEEDED(hr)) {
      hr = glyphRun->fontFace->GetGlyphRunOutline(
          glyphRun->fontEmSize, glyphRun->glyphIndices, glyphRun->glyphAdvances,
          glyphRun->glyphOffsets, glyphRun->glyphCount, glyphRun->isSideways,
          glyphRun->bidiLevel % 2, pSink);
    }

    // Close the geometry sink
    if (SUCCEEDED(hr)) { hr = pSink->Close(); }

    // Initialize a matrix to translate the s of the glyph run.
    D2D1::Matrix3x2F const matrix =
        D2D1::Matrix3x2F(1.0f, 0.0f, 0.0f, 1.0f, at.x, at.y);

    // Create the transformed geometry
    com::asset<ID2D1TransformedGeometry> pTransformedGeometry;
    if (SUCCEEDED(hr)) {
      hr = d2_factory()->CreateTransformedGeometry(
          pPathGeometry, &matrix, pTransformedGeometry.target());
    }

    if (fill) // Fill in the glyph run
      gfx->render_target()->FillGeometry(pTransformedGeometry, fill);
    if (stroke) // Draw the outline of the glyph run
      gfx->render_target()->DrawGeometry(pTransformedGeometry, stroke,
                                         fill_stroke->strokeWidth(),
                                         fill_stroke->strokeStyle());
  }

  /*void draw_glyph_run(view& v, graphics* gfx,
                     element* el,
                     style* cs,
                     pointf at,
                     const DWRITE_GLYPH_RUN* glyphRun,
                     const DWRITE_GLYPH_RUN_DESCRIPTION* glyphRunDescription,
                     DWRITE_MEASURING_MODE measuringMode,
                     argb color)
  {
    D2D1_POINT_2F org = d2d::g2f(at);
    com::asset<ID2D1Brush> br = gfx->get_brush(color);
    gfx->render_target()->DrawGlyphRun(org,glyphRun,br,measuringMode);
    if(cs->text_decoration)
    {
      slice<float> glyph_widths(glyphRun->glyphAdvances, glyphRun->glyphCount);
      DWRITE_FONT_METRICS fm;
      glyphRun->fontFace->GetMetrics(&fm);
      float du = v.pixels_per_dip(sizef(glyphRun->fontEmSize)).y /
  fm.designUnitsPerEm; float width = glyph_widths.accumulate(0.0f);
      

      if(cs->text_decoration & text_decoration_double_underline ) // includes
  text_decoration_underline
      {
        float thicknes = ceilf(du * fm.underlineThickness);
        float shift    = ceilf(du * abs(fm.underlinePosition));
        gool::rectf rc(at.x,
                       at.y + shift,
                       at.x + width - 1.f,
                       at.y + shift + thicknes - 1.f);
        gfx->render_target()->FillRectangle(d2d::g2f(rc),br);
        if( (cs->text_decoration & text_decoration_double_underline) ==
  text_decoration_double_underline )
        {
          gool::rectf rc(at.x,
                         at.y + shift + 2*thicknes,
                         at.x + width - 1.f,
                         at.y + shift + 2*thicknes + thicknes - 1.f);
          gfx->render_target()->FillRectangle(d2d::g2f(rc),br);
        }
      }
      if(cs->text_decoration & text_decoration_linethrough )
      {
        float thicknes = ceilf(du * fm.strikethroughThickness);
        float shift    = ceilf(du * abs(fm.strikethroughPosition));
        gool::rectf rc(at.x,
                       at.y - shift,
                       at.x + width - 1.f,
                       at.y - shift + thicknes - 1.f);
        gfx->render_target()->FillRectangle(d2d::g2f(rc),br);
      }
      if(cs->text_decoration & text_decoration_overline )
      {
        float thicknes = ceilf(du * fm.underlineThickness);
        float shift    = ceilf(du * fm.ascent);
        gool::rectf rc(at.x,
                       at.y - shift,
                       at.x + width - 1.f,
                       at.y - shift + thicknes - 1.f);
        gfx->render_target()->FillRectangle(d2d::g2f(rc),br);
      }
    }
  } */

  void draw_glyph_run_shadow(graphics *gfx, const DWRITE_GLYPH_RUN *glyph_run,
                             DWRITE_MEASURING_MODE dw_measuring_mode,
                             D2D1_POINT_2F dst_shadow, argb c, float radius,
                             float spread) {
    int iradius = int(radius + 0.49f);
    if (iradius == 0) {
      com::asset<ID2D1SolidColorBrush> fore;
      gfx->get_solid_brush(c, fore);
      gfx->render_target()->DrawGlyphRun(dst_shadow, glyph_run, fore, dw_measuring_mode);
      return;
    }

    // pure man text shadow implementation.
    // this really should be implemented using ID2D1Effect when it will be
    // available on W7

    D2D1_POINT_2F org = dst_shadow;

    HRESULT hr = S_OK;

    // Create the path geometry.
    com::asset<ID2D1PathGeometry> pPathGeometry;
    hr = d2_factory()->CreatePathGeometry(pPathGeometry.target());

    // Write to the path geometry using the geometry sink.
    com::asset<ID2D1GeometrySink> pSink;
    {
      if (SUCCEEDED(hr)) {
        hr = pPathGeometry->Open(pSink.target());
        pSink->SetFillMode(D2D1_FILL_MODE_WINDING);
      }

      // Get the glyph run outline geometries back from DirectWrite and place them
      // within the geometry sink.
      if (SUCCEEDED(hr)) {
        hr = glyph_run->fontFace->GetGlyphRunOutline(
          glyph_run->fontEmSize, glyph_run->glyphIndices,
          glyph_run->glyphAdvances, glyph_run->glyphOffsets,
          glyph_run->glyphCount, glyph_run->isSideways,
          glyph_run->bidiLevel % 2, pSink);
      }

      // Close the geometry sink
      if (SUCCEEDED(hr)) { hr = pSink->Close(); }
    }

    // Initialize a matrix to translate the s of the glyph run.
    D2D1::Matrix3x2F const matrix =
        D2D1::Matrix3x2F(1.0f, 0.0f, 0.0f, 1.0f, org.x, org.y);

    // Create the transformed geometry
    com::asset<ID2D1TransformedGeometry> pTransformedGeometry;
    if (SUCCEEDED(hr)) {
      hr = d2_factory()->CreateTransformedGeometry(
          pPathGeometry, &matrix, pTransformedGeometry.target());
    }

    argb cc = c;
    cc.alfa = argb::channel_t((cc.alfa * (128 / max(iradius, 1)) >> 8));

    D2D1_STROKE_STYLE_PROPERTIES props = {};
    props.startCap = D2D1_CAP_STYLE_ROUND;
    props.endCap = D2D1_CAP_STYLE_ROUND;
    props.dashCap = D2D1_CAP_STYLE_SQUARE; 
    props.lineJoin = D2D1_LINE_JOIN_ROUND;  
    props.miterLimit = 4.0f;
    props.dashStyle = D2D1_DASH_STYLE_SOLID;
    props.dashOffset = 0.0f;

    com::asset<ID2D1StrokeStyle> _stroke;

    hr = d2_factory()->CreateStrokeStyle(props, 0, 0, _stroke.target());

    com::asset<ID2D1SolidColorBrush> fore;
    gfx->get_solid_brush(cc, fore);

    if (SUCCEEDED(hr)) {
      for (int i = 0; i <= iradius; ++i) {
        if (i) {
          float stroke_width = float(i);
          gfx->render_target()->DrawGeometry(pTransformedGeometry, fore, stroke_width, gfx->rounded_stroke());
        }
        gfx->render_target()->FillGeometry(pTransformedGeometry, fore);
      }
    }
  }

  com::asset<ID2D1StrokeStyle> graphics::get_d_stroke(int pos, int w, int stroke,
                                                 int step, int_v off) {
    com::asset<ID2D1StrokeStyle>      dotted_stroke;
    D2D1_STROKE_STYLE_PROPERTIES props;
    props.startCap = D2D1_CAP_STYLE_FLAT;
    props.endCap   = D2D1_CAP_STYLE_FLAT;
    props.dashCap  = w <= 2 ? D2D1_CAP_STYLE_SQUARE
                           : D2D1_CAP_STYLE_ROUND; // D2D1_CAP_STYLE_SQUARE
    props.lineJoin   = D2D1_LINE_JOIN_MITER;       // D2D1_LINE_JOIN_BEVEL
    props.miterLimit = 4.0f;
    props.dashStyle  = D2D1_DASH_STYLE_CUSTOM;
    props.dashOffset = -0.5f + off.val(0); // float(pos % (step * w));
    float   dashes[2] = {float(stroke) / w - 1.0f,
                       float(step - stroke) / w + 1.f};
    HRESULT hr        = d2_factory()->CreateStrokeStyle(props, dashes, 2,
                                                 dotted_stroke.target());
    assert(SUCCEEDED(hr));
    hr = hr;
    return dotted_stroke;
  }

  HRESULT graphics::DrawGlyphRun(
      void *clientDrawingContext, FLOAT baselineOriginX, FLOAT baselineOriginY,
      DWRITE_MEASURING_MODE measuringMode, const DWRITE_GLYPH_RUN *glyphRun,
      const DWRITE_GLYPH_RUN_DESCRIPTION *glyphRunDescription,
      IUnknown *                          clientDrawingEffect) {
    pointf org(baselineOriginX, baselineOriginY);
    // html::element* pel = clientDrawingEffect?
    // dom_element(clientDrawingEffect):0;  if(!pel)
    {
      IFillAndStroke *fs = fill_stroke(clientDrawingEffect);
      if (fs) {
        draw_glyph_run(this, org, glyphRun, glyphRunDescription, measuringMode,
                       fs);
        fs->Release();
        return S_OK;
      }
      com::asset<ID2D1Brush> br;
      clientDrawingEffect->QueryInterface(br.target());
      if (br)
        this->render_target()->DrawGlyphRun(g2f(org), glyphRun, br,
                                            measuringMode);
      return S_OK;
    }
    //     html::view*  pv =
    //     reinterpret_cast<html::view*>(clientDrawingContext); html::style* cs
    //     = pel->get_style(*pv); if(cs->text_shadow.is_defined())
    //     {
    //       for( int n = cs->text_shadow.items->list.size() - 1; n >= 0; --n )
    //       {
    //         const shadow_v::item_t it = cs->text_shadow.items->list[n];
    //         gool::argb c(it.color.val(cs->font_color));
    //         pointf org_shadow = org;
    //         org_shadow.x += it.offset_x.pixels_width_f(*pv,pel);
    //         org_shadow.y += it.offset_y.pixels_height_f(*pv,pel);
    //         draw_glyph_run(*pv,this,pel,cs,org_shadow,glyphRun,glyphRunDescription,measuringMode,c);
    //         //this->render_target()->DrawGlyphRun(org_shadow,glyphRun,this->get_brush(c),measuringMode);
    //       }
    //     }
    //     gool::argb c(cs->font_color);
    //     //this->render_target()->DrawGlyphRun(org,glyphRun,this->get_brush(c),measuringMode);
    //     draw_glyph_run(*pv,this,pel,cs,org,glyphRun,glyphRunDescription,measuringMode,c);
    //     return S_OK;
  }

  HRESULT graphics::DrawUnderline(void *clientDrawingContext,
                                  FLOAT baselineOriginX, FLOAT baselineOriginY,
                                  const DWRITE_UNDERLINE *underline,
                                  IUnknown *              clientDrawingEffect) {
    // html::element* pel = dom_element(clientDrawingEffect);
    gool::rectf rc(0 + baselineOriginX, underline->offset + baselineOriginY,
                   underline->width + baselineOriginX,
                   underline->offset + underline->thickness + baselineOriginY);

    // if(!pel)
    {
      com::asset<ID2D1Brush> br;
      clientDrawingEffect->QueryInterface(br.target());
      if (br) this->render_target()->FillRectangle(d2d::g2f(rc), br);
      return S_OK;
    }
    /*html::view*  pv = reinterpret_cast<html::view*>(clientDrawingContext);
    html::style* cs = pel->c_style;
    gool::argb c(cs->font_color);

    this->fill(c,rc); */
    // return S_OK;
  }

  HRESULT graphics::DrawStrikethrough(void *clientDrawingContext,
                                      FLOAT baselineOriginX,
                                      FLOAT baselineOriginY,
                                      const DWRITE_STRIKETHROUGH *strikethrough,
                                      IUnknown *clientDrawingEffect) {
    gool::rectf rc(0 + baselineOriginX, strikethrough->offset + baselineOriginY,
                   strikethrough->width + baselineOriginX,
                   strikethrough->offset + strikethrough->thickness +
                       baselineOriginY);
    // html::element* pel = dom_element(clientDrawingEffect);
    // if(!pel)
    {
      com::asset<ID2D1Brush> br;
      clientDrawingEffect->QueryInterface(br.target());
      if (br) this->render_target()->FillRectangle(d2d::g2f(rc), br);
      return S_OK;
    }
    // html::view*  pv = reinterpret_cast<html::view*>(clientDrawingContext);
    // html::style* cs = pel->c_style;
    // gool::argb c(cs->font_color);
    // this->fill(c,rc);
    // return S_OK;
  }

  HRESULT graphics::DrawInlineObject(void *clientDrawingContext, FLOAT originX,
                                     FLOAT                originY,
                                     IDWriteInlineObject *inlineObject,
                                     BOOL isSideways, BOOL isRightToLeft,
                                     IUnknown *clientDrawingEffect) {
    // html::element* pel = dom_element(inlineObject);
    // if(!pel)
    return E_NOINTERFACE;
    // html::view*  pv = reinterpret_cast<html::view*>(clientDrawingContext);
    ////html::style* cs = pel->c_style;
    //
    // if( pv && !pel->floats(*pv) && !pel->positioned(*pv) &&
    // pel->is_it_visible(*pv) )
    //{
    //  pointf pos(originX,originY);
    //  pos.x += pel->ldata->outer_left();
    //  pos.y += pel->ldata->outer_top();
    //  pel->draw(*pv,this,pos);
    //}
    // return S_OK;
  }

  HRESULT graphics::IsPixelSnappingDisabled(void *clientDrawingContext,
                                            OUT BOOL *isDisabled) {
    *isDisabled = FALSE;
    return S_OK;
  }

  HRESULT graphics::GetCurrentTransform(void *clientDrawingContext,
                                        OUT DWRITE_MATRIX *transform) {
    this->render_target()->GetTransform(
        reinterpret_cast<D2D1_MATRIX_3X2_F *>(transform));
    return S_OK;
  }

  HRESULT graphics::GetPixelsPerDip(void *clientDrawingContext,
                                    OUT FLOAT *pixelsPerDip) {
    // html::view*  pv = reinterpret_cast<html::view*>(clientDrawingContext);
    // pixels_per_dip();
    float x, yUnused;
    this->render_target()->GetDpi(&x, &yUnused);
    *pixelsPerDip = x / 96;
    return S_OK;
  }
  // end of IDWriteTextRenderer implementation

  uint graphics::push_state() {
    uint lv = uint(_registers_stack.length());
    _registers.state_block = 0;
    HRESULT hr = d2_factory()->CreateDrawingStateBlock(_registers.state_block.target());
    if (SUCCEEDED(hr))
      render_target()->SaveDrawingState(_registers.state_block);
    else
      assert(false);
#ifdef _DEBUG
    D2D1_DRAWING_STATE_DESCRIPTION dsd;
    _registers.state_block->GetDescription(&dsd);
#endif
    _registers_stack.push(_registers);
    return lv;
  }
  bool graphics::pop_state() {
    if (!_registers_stack.size()) return false;
    _registers = _registers_stack.pop();
    if (_registers.state_block)
      render_target()->RestoreDrawingState(_registers.state_block);
    //#ifdef _DEBUG
    //    D2D1_DRAWING_STATE_DESCRIPTION dsd;
    //    _registers.state_block->GetDescription(&dsd);
    //#endif
    return true;
  }
  void graphics::restore_state(uint level) {
    while (_registers_stack.length() && _registers_stack.length() > level)
      pop_state();
  }

  pointf graphics::world_to_screen(pointf pt) const {
    D2D1::Matrix3x2F current;
    render_target()->GetTransform(&current);
    return d2g(current.TransformPoint(g2f(pt)));
  }

  pointf graphics::screen_to_world(pointf pt) const {
    D2D1::Matrix3x2F current;
    render_target()->GetTransform(&current);
    if (_D2D1InvertMatrix()(&current))
      return d2g(current.TransformPoint(g2f(pt)));
    return pt;
  }

  ID2D1StrokeStyle *graphics::dotted_stroke() {
    if (!_dotted_stroke) {
      D2D1_STROKE_STYLE_PROPERTIES props;
      props.startCap   = D2D1_CAP_STYLE_FLAT;
      props.endCap     = D2D1_CAP_STYLE_FLAT;
      props.dashCap    = D2D1_CAP_STYLE_ROUND; // D2D1_CAP_STYLE_SQUARE
      props.lineJoin   = D2D1_LINE_JOIN_MITER; // D2D1_LINE_JOIN_BEVEL
      props.miterLimit = 4.0f;
      props.dashStyle  = D2D1_DASH_STYLE_DOT;
      props.dashOffset = 0.0f;
      HRESULT hr =
          d2_factory()->CreateStrokeStyle(props, 0, 0, _dotted_stroke.target());
      assert(SUCCEEDED(hr));
      hr = hr;
    }
    return _dotted_stroke;
  }
  ID2D1StrokeStyle *graphics::dashed_stroke() {
    if (!_dashed_stroke) {
      D2D1_STROKE_STYLE_PROPERTIES props;
      props.startCap   = D2D1_CAP_STYLE_FLAT;
      props.endCap     = D2D1_CAP_STYLE_FLAT;
      props.dashCap    = D2D1_CAP_STYLE_ROUND; // D2D1_CAP_STYLE_SQUARE
      props.lineJoin   = D2D1_LINE_JOIN_MITER;
      props.miterLimit = 4.0f;
      props.dashStyle  = D2D1_DASH_STYLE_DASH;
      props.dashOffset = 0.0f;
      HRESULT hr =
          d2_factory()->CreateStrokeStyle(props, 0, 0, _dashed_stroke.target());
      assert(SUCCEEDED(hr));
      hr = hr;
    }
    return _dashed_stroke;
  }
  ID2D1StrokeStyle *graphics::rounded_stroke() {
    if (!_rounded_stroke) {
      D2D1_STROKE_STYLE_PROPERTIES props;
      props.startCap   = D2D1_CAP_STYLE_ROUND;
      props.endCap     = D2D1_CAP_STYLE_ROUND;
      props.dashCap    = D2D1_CAP_STYLE_SQUARE; // D2D1_CAP_STYLE_SQUARE
      props.lineJoin   = D2D1_LINE_JOIN_MITER;  // D2D1_LINE_JOIN_BEVEL
      props.miterLimit = 4.0f;
      props.dashStyle  = D2D1_DASH_STYLE_SOLID;
      props.dashOffset = 0.0f;
      HRESULT hr       = d2_factory()->CreateStrokeStyle(props, 0, 0,
                                                   _rounded_stroke.target());
      assert(SUCCEEDED(hr));
      hr = hr;
    }
    return _rounded_stroke;
  }

  ID2D1StrokeStyle *graphics::solid_stroke() {
    if (!_solid_stroke) {
      D2D1_STROKE_STYLE_PROPERTIES props;
      props.startCap   = D2D1_CAP_STYLE_FLAT;
      props.endCap     = D2D1_CAP_STYLE_FLAT;
      props.dashCap    = D2D1_CAP_STYLE_SQUARE; // D2D1_CAP_STYLE_SQUARE
      props.lineJoin   = D2D1_LINE_JOIN_MITER;  // D2D1_LINE_JOIN_BEVEL
      props.miterLimit = 4.0f;
      props.dashStyle  = D2D1_DASH_STYLE_SOLID;
      props.dashOffset = 0.0f;
      HRESULT hr =
          d2_factory()->CreateStrokeStyle(props, 0, 0, _solid_stroke.target());
      assert(SUCCEEDED(hr));
      hr = hr;
    }
    return _solid_stroke;
  }

  ID2D1StrokeStyle *graphics::get_stroke_style() {
    if (!_registers.stroke_style) {
      D2D1_STROKE_STYLE_PROPERTIES props;
      props.startCap = _registers.cap_style;
      props.endCap   = _registers.cap_style;
      if (_registers.dash_style == D2D1_DASH_STYLE_DOT &&
          _registers.cap_style == D2D1_CAP_STYLE_FLAT)
        props.dashCap = D2D1_CAP_STYLE_SQUARE;
      else
        props.dashCap = _registers.cap_style;  // D2D1_CAP_STYLE_SQUARE
      props.lineJoin   = _registers.line_join; // D2D1_LINE_JOIN_BEVEL
      props.miterLimit = _registers.mitter_limit;
      props.dashStyle  = _registers.dash_style;
      props.dashOffset = _registers.dash_offset / _registers.stroke_width;
      HRESULT hr;
      if (_registers.custom_dash_style.length() &&
          _registers.dash_style == D2D1_DASH_STYLE_CUSTOM)
        hr = d2_factory()->CreateStrokeStyle(
            props, _registers.custom_dash_style.cbegin(),
            (UINT32)_registers.custom_dash_style.length(),
            _registers.stroke_style.target());
      else
        hr = d2_factory()->CreateStrokeStyle(props, 0, 0,
                                             _registers.stroke_style.target());
      assert(SUCCEEDED(hr));
      hr = hr;
    }
    return _registers.stroke_style;
  }

  void graphics::set_fill(argb c) // set fill brush
  {
    if (c.is_no_color())
      _registers.fill_brush = 0;
    else {
      com::asset<ID2D1SolidColorBrush> fore;  
      if(get_solid_brush(c, fore))
        _registers.fill_brush = fore;
    }
  }

  void graphics::set_fill(const linear_brush &lb) // set fill brush
  {
    com::asset<ID2D1LinearGradientBrush> lgb;
    if (!get_linear_gradient_brush(lb, lgb)) return;

    if (!lb.transform().is_identity()) lgb->SetTransform(g2f(lb.transform()));

    _registers.fill_brush = lgb;
  }

  void graphics::set_fill(const radial_brush &rb) {
    com::asset<ID2D1RadialGradientBrush> rgb;
    if (!get_radial_gradient_brush(rb, rgb)) return;

    if (!rb.transform().is_identity()) rgb->SetTransform(g2f(rb.transform()));

    _registers.fill_brush = rgb;
  }

  void graphics::set_fill(const image *ib) // set fill brush
  {
    com::asset<ID2D1BitmapBrush>      bitmapBrush;
    if (get_bitmap_brush(const_cast<image*>(ib), ib->dim(), bitmapBrush))
    {
      _registers.fill_brush = bitmapBrush;
    }
  }

  void graphics::set_stroke_width(float w) { _registers.stroke_width = w; }

  void graphics::set_stroke(argb c) // set stroke brush
  {
    if (c.is_no_color())
      _registers.stroke_brush = 0;
    else {
      com::asset<ID2D1SolidColorBrush> fore;
      if (get_solid_brush(c, fore))
        _registers.stroke_brush = fore;
    }
  }

  void graphics::set_stroke(const linear_brush &lb) // set fill brush
  {
    com::asset<ID2D1LinearGradientBrush> lgb;
    if (!get_linear_gradient_brush(lb, lgb)) return;
    _registers.stroke_brush = lgb;
  }

  void graphics::set_stroke(const radial_brush &rb) // set fill brush
  {
    com::asset<ID2D1RadialGradientBrush> rgb;
    if (!get_radial_gradient_brush(rb, rgb)) return;
    _registers.stroke_brush = rgb;
  }

  void graphics::draw_ellipse(pointf center, sizef radius, bool stroke, bool fill) {
    if (!_registers.drawable()) return;
    D2D1_ELLIPSE d2el;
    d2el.point   = g2f(center);
    d2el.radiusX = radius.x;
    d2el.radiusY = radius.y;
    if (fill && _registers.fill_drawable())
      this->render_target()->FillEllipse(d2el, _registers.fill_brush);
    if (stroke && _registers.stroke_drawable())
      this->render_target()->DrawEllipse(d2el, _registers.stroke_brush,
                                         _registers.stroke_width,
                                         get_stroke_style());
  }

  /*void graphics::draw_arc(pointf center, sizef radius, float angle_start,
                          float angle_sweep) {

    handle<d2d::path> p = new d2d::path();

    com::asset<ID2D1PathGeometry> pg;
    HRESULT                  hr = d2_factory()->CreatePathGeometry(pg.target());
    assert(SUCCEEDED(hr));
    com::asset<ID2D1GeometrySink> ps;
    hr = pg->Open(ps.target());
    assert(SUCCEEDED(hr));

    pointf start;
    start.x = center.x + radius.x * cosf(angle_start);
    start.y = center.y + radius.y * sinf(angle_start);

    // D2D1_POINT_2F pt = d2d::g2f(center);

    const float PI = 3.14159265359f;

    if (fabs(angle_sweep) >= 2 * PI) {
      ps->BeginFigure(g2f(center), _registers.fill_drawable() ? D2D1_FIGURE_BEGIN_FILLED : D2D1_FIGURE_BEGIN_HOLLOW);
    }
    else if (_registers.fill_drawable()) {
      ps->BeginFigure(g2f(center), D2D1_FIGURE_BEGIN_FILLED);
      ps->AddLine(g2f(start));
    } else
      ps->BeginFigure(g2f(start), D2D1_FIGURE_BEGIN_HOLLOW);

    D2D1_ARC_SEGMENT as;
    as.point.x        = center.x + radius.x * cosf(angle_start + angle_sweep);
    as.point.y        = center.y + radius.y * sinf(angle_start + angle_sweep);
    as.size.width     = radius.x;
    as.size.height    = radius.y;
    as.rotationAngle  = fabsf(fmodf(angle_sweep, 2.0f * gool::pi));
    as.sweepDirection = angle_sweep > 0
                            ? D2D1_SWEEP_DIRECTION_CLOCKWISE
                            : D2D1_SWEEP_DIRECTION_COUNTER_CLOCKWISE;
    as.arcSize =
        angle_sweep > gool::pi ? D2D1_ARC_SIZE_LARGE : D2D1_ARC_SIZE_SMALL;
    ps->AddArc(as);
    ps->EndFigure(_registers.fill_drawable() ? D2D1_FIGURE_END_CLOSED
                                             : D2D1_FIGURE_END_OPEN);
    // ps->SetFillMode(fill_even_odd? D2D1_FILL_MODE_ALTERNATE :
    // D2D1_FILL_MODE_WINDING );
    hr = ps->Close();
    assert(SUCCEEDED(hr));
    if (_registers.fill_drawable())
      this->render_target()->FillGeometry(pg, _registers.fill_brush);
    if (_registers.stroke_drawable())
      this->render_target()->DrawGeometry(pg, _registers.stroke_brush,
                                          _registers.stroke_width,
                                          get_stroke_style());
  }*/

  void graphics::draw_rectangle(pointf org, sizef dim, bool stroke, bool fill) {
    if (!_registers.drawable()) return;
    D2D1_RECT_F d2r;
    d2r.left   = org.x;
    d2r.right  = org.x + dim.x;
    d2r.top    = org.y;
    d2r.bottom = org.y + dim.y;
    if (fill && _registers.fill_drawable())
      this->render_target()->FillRectangle(d2r, _registers.fill_brush);
    if (stroke && _registers.stroke_drawable())
      this->render_target()->DrawRectangle(d2r, _registers.stroke_brush, _registers.stroke_width, get_stroke_style());
  }
   
  void graphics::fill(image *img, const rect &dst, const rect &src, point offset, size dstcell)
  {
    size src_size = src.size();
    size dst_size = dstcell;

    assert(!src.empty());
    com::asset<ID2D1BitmapBrush>      bitmapBrush;
    if (get_bitmap_brush(img, src, bitmapBrush))
    {
      offset += dst.s;

      auto mtx = D2D1::Matrix3x2F::Translation(D2D1::SizeF(float(offset.x), float(offset.y)));
      if (!dst_size.empty() && (dst_size != src_size))
        mtx = D2D1::Matrix3x2F::Scale(FLOAT(dst_size.x) / src_size.x, FLOAT(dst_size.y) / src_size.y) * mtx;
      bitmapBrush->SetTransform(mtx);

      render_target()->FillRectangle(g2f(dst), bitmapBrush);
    }
    else
    {
      size sz = src.size();
      if (sz.empty())
        return;
      if (dstcell.empty())
        dstcell = sz;
      //assert(sz.x > 1 && sz.y > 1);
      argb clr;
      if ((sz.x <= 8 || sz.y <= 8) && img->is_solid_color(src, clr))
        return fill(clr, dst);

      clipper _c(this, dst);
      for (int y = dst.s.y - dstcell.y; y < dst.e.y; y += dstcell.y)
        for (int x = dst.s.x - dstcell.x; x < dst.e.x; x += dstcell.x)
        {
          rect r_dst(point(x, y) + offset, dstcell);
          do_draw(img, r_dst, src);
        }
    }
  }


  // bool is_display_hw_capable();

  dc_graphics::dc_graphics(HDC hdc, rect rc, bool wants_alpha): super(wants_alpha) {
    // Create a DC render target.
    D2D1_RENDER_TARGET_PROPERTIES props = D2D1::RenderTargetProperties(
        is_display_hw_capable() ? D2D1_RENDER_TARGET_TYPE_DEFAULT
                                : D2D1_RENDER_TARGET_TYPE_SOFTWARE,
        D2D1::PixelFormat(DXGI_FORMAT_B8G8R8A8_UNORM,
                          wants_alpha ? D2D1_ALPHA_MODE_PREMULTIPLIED
                                      : D2D1_ALPHA_MODE_IGNORE),
        0, 0, D2D1_RENDER_TARGET_USAGE_NONE, D2D1_FEATURE_LEVEL_DEFAULT);

    com::asset<ID2D1DCRenderTarget> rt;

    HRESULT hr =
        d2_factory()->CreateDCRenderTarget(&props, rt.target());
    assert(SUCCEEDED(hr));
    hr = hr;

    if (rt) {
      render_target(rt);
      hr = rt->BindDC(hdc, PRECT(rc));
    }

    assert(SUCCEEDED(hr));
    hr = hr;
  }

  gool::graphics *application::create_bitmap_graphics(gool::graphics *proto,
                                                      gool::bitmap *  bmp,
                                                      gool::argb      initc) {
    uint32 max_sz = static_cast<d2d::graphics *>(proto)
                        ->render_target()
                        ->GetMaximumBitmapSize();
    D2D1_SIZE_U sz = g2u(bmp->_dim);
    if (sz.width > max_sz) { return nullptr; }
    if (sz.height > max_sz) { return nullptr; }
    return new bitmap_graphics(static_cast<d2d::graphics *>(proto), bmp, initc);
  }

  bitmap_graphics::bitmap_graphics(graphics *pg, gool::bitmap *pb, argb initc): super(!initc.is_opaque()) {
    pproto  = pg;
    pbitmap = pb;
    size sz = pb->dimension();
    // D2D1_PIXEL_FORMAT pf;
    // pf.format = DXGI_FORMAT_B8G8R8A8_UNORM;
    // pf.alphaMode = D2D1_ALPHA_MODE_PREMULTIPLIED;

    com::asset<ID2D1BitmapRenderTarget> rt;

    HRESULT hr = pg->render_target()->CreateCompatibleRenderTarget(g2f(sz), rt.target());
    D2D1_COLOR_F c = g2f(initc);
    // c.a = 1;
    // c.b = 1;
    // c.g = 0;
    // c.r = 0;
    assert(SUCCEEDED(hr));
    hr = hr;

    render_target(rt);

    // Retrieve the bitmap from the render target.
    rt->BeginDraw();
    rt->Clear(c);

    set_clip_rc(pb->dim());

    super::begin_drawing();
  }
  bitmap_graphics::~bitmap_graphics() {
    super::end_drawing();
    HRESULT hr = render_target()->EndDraw();
    assert(SUCCEEDED(hr));
#ifdef DEBUG_DRAW_PIPELINE
    if (FAILED(hr))
      view::debug_printf(html::OT_DOM, html::OS_ERROR,
                         "bitmap_graphics drawing failure(A): 0x%x\n", hr);
#endif

    com::asset<ID2D1BitmapRenderTarget> rt;
    hr = render_target()->QueryInterface(rt.target());
    assert(SUCCEEDED(hr));
    if (FAILED(hr))
      return;
    assert(SUCCEEDED(hr));
    gool::image_link *il = pbitmap->get_link_for(pproto);
    il->d2d_bmp          = 0;
    // com::asset<ID2D1Bitmap> dst_bmp;
    hr = rt->GetBitmap(il->d2d_bmp.target());
    assert(SUCCEEDED(hr));
#ifdef DEBUG_DRAW_PIPELINE
    if (FAILED(hr))
      view::debug_printf(html::OT_DOM, html::OS_ERROR,
                         "bitmap_graphics drawing failure(B): 0x%x\n", hr);
#endif

    // D2D1_SIZE_U sz = il->d2d_bmp->GetPixelSize();
    /*if (SUCCEEDED(hr))
    {
      //D2D1_POINT_2U dst_p = {0,0};
      //dst_bmp->CopyFromBitmap(&dst_p,pg->D2D1Bitmap(pb),0);
      //il->d2d_bmp = dst_bmp;
    }*/
  }

  gool::graphics *application::create_bitmap_bits_graphics(gool::bitmap *bmp,
                                                           gool::argb    initc,
                                                           bool high_quality) {
    bitmap_bits_graphics *pg = bitmap_bits_graphics::create(bmp, initc);
    if (!pg) return nullptr;
    if (!pg->render_target()) {
      delete pg;
      return nullptr;
    }
    return pg;
  }

  bitmap_bits_graphics *bitmap_bits_graphics::create(gool::bitmap *pb,
                                                     argb          initc) {
    size                           sz = pb->dimension();
    auto_ptr<bitmap_bits_graphics> bbg =
        auto_ptr<bitmap_bits_graphics>(new bitmap_bits_graphics(sz));
    if (!bbg->dib.bits()) return nullptr;

    bbg->pbitmap = pb;

    D2D1_RENDER_TARGET_PROPERTIES props = D2D1::RenderTargetProperties(
        D2D1_RENDER_TARGET_TYPE_SOFTWARE,
        D2D1::PixelFormat(DXGI_FORMAT_B8G8R8A8_UNORM,
                          D2D1_ALPHA_MODE_PREMULTIPLIED),
        96.0f, 96.0f, D2D1_RENDER_TARGET_USAGE_GDI_COMPATIBLE,
        D2D1_FEATURE_LEVEL_DEFAULT);

    com::asset<ID2D1DCRenderTarget> rt;

    HRESULT hr = d2_factory()->CreateDCRenderTarget(&props,rt.target());
    assert(SUCCEEDED(hr));
    hr = hr;
    if (FAILED(hr)) return nullptr;

    bbg->render_target(rt);

    rect dst(sz);

    if (initc == argb::undefined()) {
      tslice<argb>((argb *)bbg->dib.bits(), pb->dim().x * pb->dim().y)
          .copy(pb->get_bits());
      hr = rt->BindDC(bbg->dib.DC(), PRECT(dst));
      if (FAILED(hr)) return nullptr;
      bbg->_render_target->BeginDraw();
    } else {
      hr = rt->BindDC(bbg->dib.DC(), PRECT(dst));
      if (FAILED(hr)) return nullptr;
      bbg->_render_target->BeginDraw();
      bbg->_render_target->Clear(g2f(initc));
    }

    bbg->set_clip_rc(sz);

    bbg->super::begin_drawing();
    return bbg.release();
  }

  void bitmap_bits_graphics::final_release() {

    if (!render_target()) {
#ifdef DEBUG_DRAW_PIPELINE
      view::debug_printf(html::OT_DOM, html::OS_ERROR,
                         "bitmap_bits_graphics drawing failure(A)\n");
#endif
      return super::final_release();
    }
    super::end_drawing();

    HRESULT hr;

    hr = render_target()->EndDraw();

    if (!dib.bits()) {
#ifdef DEBUG_DRAW_PIPELINE
      view::debug_printf(html::OT_DOM, html::OS_ERROR,
                         "bitmap_bits_graphics drawing failure(B)\n");
#endif
      return;
    }

    if (FAILED(hr)) {
#ifdef DEBUG_DRAW_PIPELINE
      view::debug_printf(html::OT_DOM, html::OS_ERROR,
                         "bitmap_bits_graphics drawing failure(C): 0x%x\n", hr);
#endif
      assert(false);
    }
#if defined(USE_D2D_PLUS)
    _device_context = nullptr;
#endif
    _render_target = nullptr;
    pbitmap->set_bits(
        slice<argb>((argb *)dib.bits(), dib.dim().x * dib.dim().y));
    // pbitmap->_dim = dib.dim();
    return super::final_release();
  }


#ifdef USE_D2D_PLUS
  dx_surface_graphics *dx_surface_graphics::create(IDXGISurface *surface) {
    auto_ptr<dx_surface_graphics> pg = auto_ptr<dx_surface_graphics>(new dx_surface_graphics(true));

    D2D1_RENDER_TARGET_PROPERTIES rtp = D2D1::RenderTargetProperties(
        D2D1_RENDER_TARGET_TYPE_DEFAULT,
        D2D1::PixelFormat(DXGI_FORMAT_UNKNOWN, D2D1_ALPHA_MODE_PREMULTIPLIED),
        96.0, 96.0);

    HRESULT hr;

    com::asset<ID2D1RenderTarget> target;

    hr = d2_factory()->CreateDxgiSurfaceRenderTarget(surface, &rtp, target.target());
    if (FAILED(hr)) return nullptr;

    pg->render_target(target);

    return pg.release();
  }
#endif

  static uint64 ver64(value ver) {
    uint    a = 0, b = 0, c = 0, d = 0;
    ustring us = ver.to_string();
    if (swscanf_s(us, L"%d.%d.%d.%d", &a, &b, &c, &d) != 4) return 0;
    return (uint64(a) << 48) + (uint64(b) << 32) + (uint64(c) << 16) +
           uint64(d);
  }

#if 0

  bool is_display_hw_capable()
  {
    static tristate_v _is_display_hw_capable;

    if(_is_display_hw_capable.is_defined())
      return !!_is_display_hw_capable;
    
    if(!_Direct3DCreate9)
      return !!(_is_display_hw_capable = false);
    com::asset<IDirect3D9> d3d9; d3d9.attach(_Direct3DCreate9()(D3D9_SDK_VERSION));
    if(!d3d9)
      return !!(_is_display_hw_capable = false);

    critical_section _(gool::lock);

    D3DADAPTER_IDENTIFIER9 adapt = {0};

    try 
    {
      // some Intel drivers (e.g. igdumdx32) are written so badly that they produce AV inside this call
      HRESULT hr = d3d9->GetAdapterIdentifier(D3DADAPTER_DEFAULT,0,&adapt);
      if(FAILED(hr))
        return !!(_is_display_hw_capable = false);
    } 
    catch(...) 
    {
      return !!(_is_display_hw_capable = false);
    }


    value blacklist = get_gpu_black_list();
    if(blacklist.is_undefined())
      return !!(_is_display_hw_capable = true); // assume it is hw capable

    static value any = value(L"*");
    
    for( uint n = 0; n < blacklist.size(); ++n)
    {
      value bli = blacklist[n];

      value vendorId = bli.get_prop("vendorId");
      value deviceId = bli.get_prop("deviceID");
      if( vendorId != any && adapt.VendorId != (DWORD) vendorId.to_int() )
          continue;
      if( deviceId != any && adapt.DeviceId != (DWORD) deviceId.to_int() )
          continue;
     
      value ver_start, ver_end;
      ver_start = bli.get_prop("startMatchInclusive");
      ver_end = bli.get_prop("endMatchExclusive");
      if( ver_start.is_string() && ver_end.is_string() )
      {
         if( uint64(adapt.DriverVersion.QuadPart) < ver64(ver_start) ||
             uint64(adapt.DriverVersion.QuadPart) >=  ver64(ver_end))
             continue;
      }

      value subSysID = bli.get_prop("subSysID");
      if( subSysID.is_defined() && (DWORD)subSysID.to_int() != adapt.SubSysId )
        continue;
      
      value umdFilename = bli.get_prop("umdFilename");
      if(umdFilename.is_string() && umdFilename.get_string() != ustring(adapt.Driver))
        continue;

      value os = bli.get_prop("os");
      if(os.is_defined() && os.to_int() != environment::get_os_version())
        continue;

      return !!(_is_display_hw_capable = false);
    }
    return !!(_is_display_hw_capable = true);
  }
#else
  // note this version does not use UMD filename for detection
  bool is_display_hw_capable() {
#if defined(USE_HARDWARE_BLACKLIST)
    static tristate_v _is_display_hw_capable;

    if (_is_display_hw_capable.is_defined()) return !!_is_display_hw_capable;

    if (!_CreateDXGIFactory) return !!(_is_display_hw_capable = false);

    com::asset<IDXGIFactory> pFactory;
    HRESULT             hr = _CreateDXGIFactory()(__uuidof(IDXGIFactory),
                                      (void **)pFactory.target());
    if (FAILED(hr)) return !!(_is_display_hw_capable = false);
    com::asset<IDXGIAdapter> pAdapter;
    hr = pFactory->EnumAdapters(0, pAdapter.target());
    if (FAILED(hr)) return !!(_is_display_hw_capable = false);

    DXGI_ADAPTER_DESC adapt;
    hr = pAdapter->GetDesc(&adapt);
    if (FAILED(hr)) return !!(_is_display_hw_capable = false);

    LARGE_INTEGER UMDVersion;

    hr = pAdapter->CheckInterfaceSupport(__uuidof(ID3D10Device), &UMDVersion);
    if (FAILED(hr)) return !!(_is_display_hw_capable = false);

    critical_section _(gool::lock);

    static DXGI_ADAPTER_DESC _dd = {0};
    if (memcmp(&_dd, &adapt, sizeof(_dd)) == 0) return !!_is_display_hw_capable;

    _dd = adapt;

    static value any = value(L"*");

    value blacklist = get_gpu_black_list();

    if (blacklist.is_undefined())
      return !!(_is_display_hw_capable = true); // assume it is hw capable

    for (uint n = 0; n < blacklist.size(); ++n) {
      value bli = blacklist[n];

      value vendorId = bli.get_prop("vendorId");
      value deviceId = bli.get_prop("deviceID");
      if (vendorId != any && adapt.VendorId != (DWORD)vendorId.to_int())
        continue;
      if (deviceId != any && adapt.DeviceId != (DWORD)deviceId.to_int())
        continue;

      value ver_start, ver_end;
      ver_start = bli.get_prop("startMatchInclusive");
      ver_end   = bli.get_prop("endMatchExclusive");
      if (ver_start.is_string() && ver_end.is_string()) {
        if (uint64(UMDVersion.QuadPart) < ver64(ver_start) ||
            uint64(UMDVersion.QuadPart) >= ver64(ver_end))
          continue;
      }

      value subSysID = bli.get_prop("subSysID");
      if (subSysID.is_defined() && (DWORD)subSysID.to_int() != adapt.SubSysId)
        continue;

      // value umdFilename = bli.get_prop("umdFilename");
      // if(umdFilename.is_string() && umdFilename.get_string() !=
      // ustring(adapt.Driver))
      //  continue;

      value os = bli.get_prop("os");
      if (os.is_defined() && os.to_int() != environment::get_os_version())
        continue;

      return !!(_is_display_hw_capable = false);
    }
    return !!(_is_display_hw_capable = true);
#else
    return true;
#endif
  }
#endif

#if defined(USE_HARDWARE_BLACKLIST) || defined(USE_DX_ON_LAYERED)

#ifdef USE_D2D_PLUS
  bool create_DX_device(com::asset<ID3D11Device> &device,
                        GRAPHICS_CAPS        requested_caps,
                        GRAPHICS_CAPS &      actual_caps) {
    if (!_D3D11CreateDevice) return false;

    D3D_DRIVER_TYPE drivers[] = {
        D3D_DRIVER_TYPE_HARDWARE,
        D3D_DRIVER_TYPE_WARP,
    };

    D3D_FEATURE_LEVEL levels[] = {
        D3D_FEATURE_LEVEL_11_1, D3D_FEATURE_LEVEL_11_0, D3D_FEATURE_LEVEL_10_1,
        D3D_FEATURE_LEVEL_10_0, D3D_FEATURE_LEVEL_9_3,  D3D_FEATURE_LEVEL_9_2,
        D3D_FEATURE_LEVEL_9_1};

    int m_start = 0;
    if (requested_caps == WARP_GRAPHICS || !is_display_hw_capable())
      m_start = 1; // the driver is known as bad so force to use WARP instead.

    // D3D_FEATURE_LEVEL  used_level;

    for (int m = m_start; m < items_in(drivers); ++m) {
      HRESULT hr = _D3D11CreateDevice()(
          0, // adapter
          drivers[m], NULL, D3D11_CREATE_DEVICE_BGRA_SUPPORT, levels,
          items_in(levels), D3D11_SDK_VERSION, device.target(), NULL, NULL);

      if (SUCCEEDED(hr)) {
        actual_caps = m == 0 ? HARDWARE_GRAPHICS : WARP_GRAPHICS;
        return true;
      }
    }
    return false;
  }

#else
  bool create_DX_device(com::asset<ID3D10Device1> &device,
                        GRAPHICS_CAPS         requested_caps,
                        GRAPHICS_CAPS &       actual_caps) {
    if (!_D3D10CreateDevice1) return false;

    D3D10_DRIVER_TYPE drivers[] = {D3D10_DRIVER_TYPE_HARDWARE,
                                   D3D10_DRIVER_TYPE_WARP};

    D3D10_FEATURE_LEVEL1 levels[] = {
        // D3D11_FEATURE_LEVEL_11_0,
        D3D10_FEATURE_LEVEL_10_1, D3D10_FEATURE_LEVEL_10_0,
        /*D3D10_FEATURE_LEVEL_9_3,
        D3D10_FEATURE_LEVEL_9_2,
        D3D10_FEATURE_LEVEL_9_1,*/
    };

    int m_start = 0;
    if (requested_caps == WARP_GRAPHICS || !is_display_hw_capable())
      m_start = 1; // the driver is known as bad so force to use WARP instead.

    for (int m = m_start; m < items_in(drivers); ++m) {
      for (int n = 0; n < items_in(levels); ++n) {
        HRESULT hr =
            _D3D10CreateDevice1()(0, // adapter
                                  drivers[m],
                                  0, // reserved
                                  D3D10_CREATE_DEVICE_BGRA_SUPPORT, levels[n],
                                  D3D10_1_SDK_VERSION, device.target());

        if (SUCCEEDED(hr)) {
          actual_caps = m == 0 ? HARDWARE_GRAPHICS : WARP_GRAPHICS;
          return true;
        }
      }
    }
    return false;
  }
#endif

#endif

  GRAPHICS_CAPS get_graphics_caps() {
    GRAPHICS_CAPS actual_caps = application::requested_gfx_layer;
#if defined(USE_HARDWARE_BLACKLIST)
#ifdef USE_D2D_PLUS
    com::asset<ID3D11Device> device;
#else
    com::asset<ID3D10Device1> device;
#endif
    if (!create_DX_device(device, application::requested_gfx_layer,
                          actual_caps))
      return UNKNOWN_GRAPHICS;
#endif
    return actual_caps;
  }

  GRAPHICS_CAPS application::graphics_caps() const {
    return get_graphics_caps();
  }

  void graphics::draw_path(const gool::polygonf &p, bool stroke, bool fill) {
    handle<gool::path> pa = app()->create_path();
    pa->set(p);
    draw_path(pa,stroke,fill);
  }

  gool::path *application::create_path() { return new d2d::path(); }

  gool::application *graphics::app() const { return d2d_app(); }

  void path::reset() {
    pg         = 0;
    HRESULT hr = d2d::d2_factory()->CreatePathGeometry(pg.target());
    assert(SUCCEEDED(hr));
    hr = hr;
  }

  void path::check_sink() {
    if (!pg) reset();
    if (!ps) {

      if (_sealed) {
      TRY_SEALED:
        com::asset<ID2D1PathGeometry> tpg;
        HRESULT hr = d2d::d2_factory()->CreatePathGeometry(tpg.target());
        assert(SUCCEEDED(hr));
        if (FAILED(hr)) return;

        UINT32 count = 0;
        pg->GetFigureCount(&count);

        if (count) {
          hr = tpg->Open(ps.target());
          assert(SUCCEEDED(hr));
          if (FAILED(hr)) return;
          hr = pg->Stream(ps);
          assert(SUCCEEDED(hr));
          if (FAILED(hr)) return;
        }
        pg      = tpg;
        _sealed = false;
      } else {
        HRESULT hr = pg->Open(ps.target());
        if (hr == 0x88990001) // object is not in correct state
          goto TRY_SEALED;
        assert(SUCCEEDED(hr));
        hr = hr;
      }
    }
  }

  bool path::is_inside(pointf pt) {
    seal();
    BOOL             contains = FALSE;
    D2D1::Matrix3x2F current  = D2D1::Matrix3x2F::Identity();
    // D2D1::Matrix3x2F current; render_target()->GetTransform( &current );
    HRESULT hr = pg->FillContainsPoint(d2d::g2f(pt), current, &contains);
    // assert(SUCCEEDED(hr));
    hr = hr;
    return !!contains;
  }

  void path::set(const polygonf &vertices, bool fill_even_odd) {
    if (!pg) {
      HRESULT hr = d2_factory()->CreatePathGeometry(pg.target());
      assert(SUCCEEDED(hr));
      hr = hr;
    }
    slice<pointf> points = vertices();

    if (points.length == 0) return;

    com::asset<ID2D1GeometrySink> ps;
    HRESULT                       hr = pg->Open(ps.target());
    assert(SUCCEEDED(hr));
    hr = hr;

    D2D1_POINT_2F pt = d2d::g2f(points++);
    ps->BeginFigure(pt, D2D1_FIGURE_BEGIN_FILLED);
    while (!!points) {
      pt = d2d::g2f(points++);
      ps->AddLine(pt);
    }
    ps->EndFigure(D2D1_FIGURE_END_CLOSED);
    ps->SetFillMode(fill_even_odd ? D2D1_FILL_MODE_ALTERNATE
                                  : D2D1_FILL_MODE_WINDING);
    hr = ps->Close();
    assert(SUCCEEDED(hr));
    hr = hr;
    // is_inside(pointf(0,0));
  }

  void path::set(const polygonf &v1, const polygonf &v2) {
    if (!pg) {
      HRESULT hr = d2d::d2_factory()->CreatePathGeometry(pg.target());
      assert(SUCCEEDED(hr));
      hr = hr;
    }

    slice<pointf> points = v1();

    if (points.length == 0) return;

    com::asset<ID2D1GeometrySink> ps;
    HRESULT                       hr = pg->Open(ps.target());
    assert(SUCCEEDED(hr));

    D2D1_POINT_2F pt = d2d::g2f(points++);
    ps->BeginFigure(pt, D2D1_FIGURE_BEGIN_FILLED);
    while (!!points) {
      pt = d2d::g2f(points++);
      ps->AddLine(pt);
    }
    ps->EndFigure(D2D1_FIGURE_END_CLOSED);

    points = v2();

    pt = d2d::g2f(points++);
    ps->BeginFigure(pt, D2D1_FIGURE_BEGIN_FILLED);
    while (!!points) {
      pt = d2d::g2f(points++);
      ps->AddLine(pt);
    }
    ps->EndFigure(D2D1_FIGURE_END_CLOSED);

    ps->SetFillMode(D2D1_FILL_MODE_ALTERNATE);
    hr = ps->Close();
    assert(SUCCEEDED(hr));
    hr = hr;
  }

  /*void path::sink_t::reset()
  {
    ps->Close();
    this->opened = false;
  }*/

  void path::start(pointf start, bool filled) {
    check_sink();
    if (_opened) {
      // ps->EndFigure( this->filled? D2D1_FIGURE_END_OPEN:
      // D2D1_FIGURE_END_CLOSED );
      ps->EndFigure(D2D1_FIGURE_END_OPEN);
    }
    this->_sealed = false;
    this->_opened = true;
    this->filled  = filled;
    ps->BeginFigure(d2d::g2f(_firstp = _lastp = start),
                    filled ? D2D1_FIGURE_BEGIN_FILLED
                           : D2D1_FIGURE_BEGIN_HOLLOW);
  }
  void path::move_to(pointf pt, bool rel) {
    check_sink();

    if (rel)
      _lastp += pt;
    else
      _lastp = pt;

    // if( _opened )
    //  ps->EndFigure( D2D1_FIGURE_END_OPEN );
    // else
    path::start(_lastp, true);
  }
  void path::line_to(pointf pt, bool rel) {
    _sealed = false;
    if (!_opened) start(_lastp, true);
    if (rel) pt += _lastp;
    ps->AddLine(d2d::g2f(_lastp = pt));
  }

  void path::close() {
    if (_opened) {
      // ps->SetFillMode(D2D1_FILL_MODE_ALTERNATE);
      _sealed = false;
      if (ps) ps->EndFigure(D2D1_FIGURE_END_CLOSED);
      _opened = false;
    }
  }

  void path::seal() {
    if (!_sealed) {
      if (ps) {
        if (_opened) {
          ps->EndFigure(D2D1_FIGURE_END_OPEN);
          _opened = false;
        }
        (void)ps->Close();
        ps = 0;
      }
      _sealed = true;
      _opened = false;
    }
  }

  rectf path::bounds() const {
    const_cast<path *>(this)->seal();
    D2D1_RECT_F b  = {0};
    HRESULT     hr = pg->GetBounds(D2D1::IdentityMatrix(), &b);
    // assert(SUCCEEDED(hr));
    hr = hr;
    return d2d::d2f(b);
  }

  gool::path *path::combine(COMBINE_MODE mode, const gool::path *other) {
    seal();

    com::asset<ID2D1PathGeometry> pPathGeometry;
    HRESULT hr = d2_factory()->CreatePathGeometry(pPathGeometry.target());

    if (FAILED(hr)) return nullptr;

    com::asset<ID2D1GeometrySink> pGeometrySink;
    hr = pPathGeometry->Open(pGeometrySink.target());

    if (FAILED(hr)) return nullptr;

    hr = this->pg->CombineWithGeometry(
        (static_cast<const path *>(other))->get_geometry(),
        D2D1_COMBINE_MODE(mode), NULL, NULL, pGeometrySink);

    if (FAILED(hr)) return nullptr;

    hr = pGeometrySink->Close();

    path *pp = new path();
    pp->pg   = pPathGeometry;
    pp->seal();
    return pp;
  }

  void path::arc_to(pointf to, sizef r, float rotation_angle, bool large_arc,
                    bool clockwise, bool rel_to) {
    if (rel_to) { to += _lastp; }

    D2D1_ARC_SEGMENT as;

    as.point = d2d::g2f(to);
    as.size  = d2d::g2f(r);

    as.rotationAngle = rotation_angle * 180.0f / real_traits<float>::pi();

    as.arcSize        = large_arc ? D2D1_ARC_SIZE_LARGE : D2D1_ARC_SIZE_SMALL;
    as.sweepDirection = clockwise ? D2D1_SWEEP_DIRECTION_CLOCKWISE
                                  : D2D1_SWEEP_DIRECTION_COUNTER_CLOCKWISE;
    ps->AddArc(as);
  }

  void path::add_arc(pointf c, sizef r, float angle_start, float angle_swipe) {
    D2D1_ARC_SEGMENT as;
    // as.point = d2d::g2f(_lastp);

    as.point.x = c.x + r.x * std::cos(angle_start + angle_swipe);
    as.point.y = c.y + r.y * std::sin(angle_start + angle_swipe);

    as.size = d2d::g2f(r);

    as.rotationAngle = angle_swipe;

    bool large = as.rotationAngle > pi;

    as.arcSize        = large ? D2D1_ARC_SIZE_LARGE : D2D1_ARC_SIZE_SMALL;
    as.sweepDirection = angle_swipe > 0
                            ? D2D1_SWEEP_DIRECTION_CLOCKWISE
                            : D2D1_SWEEP_DIRECTION_COUNTER_CLOCKWISE;

    _lastp = pointf(as.point.x,as.point.y);
    ps->AddArc(as);
  }

  void path::quadratic_to(pointf pt, pointf cp, bool rel) {
    if (!_opened) start(_lastp, true);
    _sealed = false;
    if (rel) {
      pt += _lastp;
      cp += _lastp;
    }
    D2D1_QUADRATIC_BEZIER_SEGMENT bs;
    bs.point1 = d2d::g2f(cp);
    bs.point2 = d2d::g2f(_lastp = pt);
    ps->AddQuadraticBezier(bs);
  }

  void path::cubic_to(pointf pt, pointf cp1, pointf cp2, bool rel) {
    if (!_opened) start(_lastp, true);
    _sealed = false;
    if (rel) {
      pt += _lastp;
      cp1 += _lastp;
      cp2 += _lastp;
    }
    D2D1_BEZIER_SEGMENT bs;
    bs.point1 = d2d::g2f(cp1);
    bs.point2 = d2d::g2f(cp2);
    bs.point3 = d2d::g2f(_lastp = pt);
    ps->AddBezier(bs);
  }

  void path::set_even_odd(bool even_odd) {
    check_sink();
    ps->SetFillMode(even_odd ? D2D1_FILL_MODE_ALTERNATE
                             : D2D1_FILL_MODE_WINDING);
  }

#if defined(USE_D2D_PLUS)
  struct d2d_filter_graph_builder : public filter_graph_builder {
    com::asset<ID2D1DeviceContext> pdc;
    com::asset<ID2D1Image>         bitmap;
    com::asset<ID2D1Effect>        tail;

    virtual bool add_blur(float p) {

      com::asset<ID2D1Effect> effect;
      HRESULT hr = pdc->CreateEffect(CLSID_D2D1GaussianBlur, effect.target());
      if (FAILED(hr)) return false;
      if (tail)
        effect->SetInputEffect(0, tail);
      else
        effect->SetInput(0, bitmap);

      hr = effect->SetValue(D2D1_GAUSSIANBLUR_PROP_STANDARD_DEVIATION, p);
      if (FAILED(hr)) return false;

      effect->SetValue(D2D1_GAUSSIANBLUR_PROP_BORDER_MODE,
                       D2D1_BORDER_MODE_HARD);

      tail = effect;
      return true;
    }

    virtual bool add_brightness(float p) {
      com::asset<ID2D1Effect> effect;
      HRESULT hr = pdc->CreateEffect(CLSID_D2D1ColorMatrix, effect.target());
      if (FAILED(hr)) return false;
      if (tail)
        effect->SetInputEffect(0, tail);
      else
        effect->SetInput(0, bitmap);

      D2D1_MATRIX_5X4_F matrix = D2D1::Matrix5x4F(1, 0, 0, 0, 0, 1, 0, 0, 0, 0,
                                                  1, 0, 0, 0, 0, 1, p, p, p, 0);
      hr = effect->SetValue(D2D1_COLORMATRIX_PROP_COLOR_MATRIX, matrix);
      if (FAILED(hr)) return false;

      tail = effect;
      return true;
    }

    virtual bool add_contrast(float p) {
      com::asset<ID2D1Effect> effect;
      HRESULT hr = pdc->CreateEffect(CLSID_D2D1ColorMatrix, effect.target());
      if (FAILED(hr)) return false;
      if (tail)
        effect->SetInputEffect(0, tail);
      else
        effect->SetInput(0, bitmap);

      float t = (1.0f - p) / 2.0f;

      D2D1_MATRIX_5X4_F matrix = D2D1::Matrix5x4F(p, 0, 0, 0, 0, p, 0, 0, 0, 0,
                                                  p, 0, 0, 0, 0, 1, t, t, t, 0);
      hr = effect->SetValue(D2D1_COLORMATRIX_PROP_COLOR_MATRIX, matrix);
      if (FAILED(hr)) return false;

      tail = effect;
      return true;
    }
    virtual bool add_grayscale(float p) {
      com::asset<ID2D1Effect> effect;
      HRESULT hr = pdc->CreateEffect(CLSID_D2D1ColorMatrix, effect.target());
      if (FAILED(hr)) return false;
      if (tail)
        effect->SetInputEffect(0, tail);
      else
        effect->SetInput(0, bitmap);

      p = 1.0f - p; // grayscale level to saturation

      float r_weight = 0.299f;
      float g_weight = 0.587f;
      float b_weight = 0.114f;

      float a = (1.0f - p) * r_weight + p;
      float b = (1.0f - p) * r_weight;
      float c = (1.0f - p) * r_weight;
      float d = (1.0f - p) * g_weight;
      float e = (1.0f - p) * g_weight + p;
      float f = (1.0f - p) * g_weight;
      float g = (1.0f - p) * b_weight;
      float h = (1.0f - p) * b_weight;
      float i = (1.0f - p) * b_weight + p;

      D2D1_MATRIX_5X4_F matrix = D2D1::Matrix5x4F(a, b, c, 0, d, e, f, 0, g, h,
                                                  i, 0, 0, 0, 0, 1, 0, 0, 0, 0);

      // D2D1_MATRIX_5X4_F matrix = D2D1::Matrix5x4F(0.299f, 0.299f, 0.299f, 0,
      //  0.587f, 0.587f, 0.587f, 0,
      //  0.114f, 0.114f, 0.114f, 0,
      //  0, 0, 0, 1,
      //  0, 0, 0, 0);

      hr = effect->SetValue(D2D1_COLORMATRIX_PROP_COLOR_MATRIX, matrix);
      if (FAILED(hr)) return false;

      tail = effect;
      return true;
    }
    virtual bool add_hue_rotate(float a) {
      com::asset<ID2D1Effect> effect;
      HRESULT hr = pdc->CreateEffect(CLSID_D2D1HueRotation, effect.target());
      if (FAILED(hr)) return false;
      if (tail)
        effect->SetInputEffect(0, tail);
      else
        effect->SetInput(0, bitmap);
      const float PI = 3.14159265359f;
      hr = effect->SetValue(D2D1_HUEROTATION_PROP_ANGLE, a / PI * 180);
      if (FAILED(hr)) return false;
      tail = effect;
      return true;
    }
    virtual bool add_invert() {
      com::asset<ID2D1Effect> effect;
      HRESULT hr = pdc->CreateEffect(CLSID_D2D1ColorMatrix, effect.target());
      if (FAILED(hr)) return false;
      if (tail)
        effect->SetInputEffect(0, tail);
      else
        effect->SetInput(0, bitmap);

      D2D1_MATRIX_5X4_F matrix = D2D1::Matrix5x4F(
          -1, 0, 0, 0, 0, -1, 0, 0, 0, 0, -1, 0, 0, 0, 0, 1, 1, 1, 1, 0);
      hr = effect->SetValue(D2D1_COLORMATRIX_PROP_COLOR_MATRIX, matrix);
      if (FAILED(hr)) return false;

      tail = effect;
      return true;
    }
    virtual bool add_opacity(float p) {
      com::asset<ID2D1Effect> effect;
      HRESULT hr = pdc->CreateEffect(CLSID_D2D1ColorMatrix, effect.target());
      if (FAILED(hr)) return false;
      if (tail)
        effect->SetInputEffect(0, tail);
      else
        effect->SetInput(0, bitmap);

      D2D1_MATRIX_5X4_F matrix = D2D1::Matrix5x4F(1, 0, 0, 0, 0, 1, 0, 0, 0, 0,
                                                  1, 0, 0, 0, 0, p, 0, 0, 0, 0);
      hr = effect->SetValue(D2D1_COLORMATRIX_PROP_COLOR_MATRIX, matrix);
      if (FAILED(hr)) return false;

      tail = effect;
      return true;
    }
    virtual bool add_saturate(float p) {
      com::asset<ID2D1Effect> effect;
      HRESULT hr = pdc->CreateEffect(CLSID_D2D1Saturation, effect.target());
      if (FAILED(hr)) return false;
      if (tail)
        effect->SetInputEffect(0, tail);
      else
        effect->SetInput(0, bitmap);

      hr = effect->SetValue(D2D1_SATURATION_PROP_SATURATION, p);
      if (FAILED(hr)) return false;

      tail = effect;
      return true;
    }
    virtual bool add_sepia(float p) {

      com::asset<ID2D1Effect> effect;
      HRESULT hr = pdc->CreateEffect(CLSID_D2D1ColorMatrix, effect.target());
      if (FAILED(hr)) return false;
      if (tail)
        effect->SetInputEffect(0, tail);
      else
        effect->SetInput(0, bitmap);

      D2D1_MATRIX_5X4_F matrix =
          D2D1::Matrix5x4F(0.393f, 0.349f, 0.272f, 0, 0.769f, 0.686f, 0.534f, 0,
                           0.189f, 0.168f, 0.131f, 0, 0, 0, 0, 1, 0, 0, 0, 0);
      hr = effect->SetValue(D2D1_COLORMATRIX_PROP_COLOR_MATRIX, matrix);
      if (FAILED(hr)) return false;

      tail = effect;
      return true;
    }
    virtual bool add_drop_shadow(float offx, float offy, float blur_radius,
                                 float spread_radius, gool::argb clr) {
      com::asset<ID2D1Effect> shadow_effect;
      HRESULT hr = pdc->CreateEffect(CLSID_D2D1Shadow, shadow_effect.target());
      if (FAILED(hr)) return false;
      if (tail)
        shadow_effect->SetInputEffect(0, tail);
      else
        shadow_effect->SetInput(0, bitmap);

      D2D1_VECTOR_4F color = {clr.red / 255.0f, clr.green / 255.0f,
                              clr.blue / 255.0f, clr.alfa / 255.0f};

      hr = shadow_effect->SetValue(D2D1_SHADOW_PROP_COLOR, color);
      if (FAILED(hr)) return false;
      hr = shadow_effect->SetValue(D2D1_SHADOW_PROP_BLUR_STANDARD_DEVIATION, blur_radius);
      if (FAILED(hr)) return false;

      com::asset<ID2D1Effect> affine_transform;
      hr = pdc->CreateEffect(CLSID_D2D12DAffineTransform,
                             affine_transform.target());
      if (FAILED(hr)) return false;

      D2D1_MATRIX_3X2_F matrix = D2D1::Matrix3x2F::Translation(offx, offy);
      affine_transform->SetValue(D2D1_2DAFFINETRANSFORM_PROP_TRANSFORM_MATRIX,
                                 matrix);

      affine_transform->SetInputEffect(0, shadow_effect);

      com::asset<ID2D1Effect> composite_effect;
      pdc->CreateEffect(CLSID_D2D1Composite, composite_effect.target());

      composite_effect->SetInputEffect(0, affine_transform);

      if (tail)
        composite_effect->SetInputEffect(1, tail);
      else
        composite_effect->SetInput(1, bitmap);

      tail = composite_effect;

      return true;
    }

    virtual bool add_composition(COMPOSITION_OP op) override {
      return false;

      // NOTE: this requires second bitmap for composition

      /*      com::asset<ID2D1Effect> effect;
            HRESULT hr = pdc->CreateEffect(CLSID_D2D1Composite,
         effect.target()); if( FAILED(hr) ) return false; if( tail )
              effect->SetInputEffect(0, tail);
            else
              effect->SetInput(0, bitmap);

            D2D1_COMPOSITE_MODE md;

            switch(op) {
            default:
              case src_over : md = D2D1_COMPOSITE_MODE_SOURCE_OVER; break;
              case dst_over : md = D2D1_COMPOSITE_MODE_DESTINATION_OVER; break;
              case src_in   : md = D2D1_COMPOSITE_MODE_SOURCE_IN; break;
              case dst_in   : md = D2D1_COMPOSITE_MODE_DESTINATION_IN; break;
              case src_out  : md = D2D1_COMPOSITE_MODE_SOURCE_OUT; break;
              case dst_out  : md = D2D1_COMPOSITE_MODE_DESTINATION_OUT; break;
              case src_atop : md = D2D1_COMPOSITE_MODE_SOURCE_ATOP; break;
              case dst_atop : md = D2D1_COMPOSITE_MODE_DESTINATION_ATOP; break;
              case dst_src_xor  : md = D2D1_COMPOSITE_MODE_XOR; break;
              case dst_copy_src : md = D2D1_COMPOSITE_MODE_SOURCE_COPY; break;
            }
            

            hr = effect->SetValue(D2D1_COMPOSITE_PROP_MODE, md);
            if( FAILED(hr) )
              return false;

            tail = effect;
            return true;  */
    }
  };
#endif

  void graphics::push_layer(const rect &clip, byte opacity,
                            function<bool(filter_graph_builder *)> filters) {
    layer_t ld;
    ld.opacity = opacity;
#ifdef USE_D2D_PLUS
    while (filters && device_context()) {
      D2D1_SIZE_F bitmap_size =  D2D1::SizeF(float(clip.width()), float(clip.height()));

      com::asset<ID2D1BitmapRenderTarget> rt;
      com::asset<ID2D1DeviceContext>      dc;

      HRESULT hr = device_context()->CreateCompatibleRenderTarget(bitmap_size,
                                                                  rt.target());

      if (FAILED(hr)) break;

      rt->QueryInterface<ID2D1DeviceContext>(dc.target());

      // Create the bitmap to which the effects will be applied.
      /*      hr = device_context()->CreateBitmap(
                    bitmap_size,
                    nullptr,
                    0,
                    D2D1::BitmapProperties1(
                        D2D1_BITMAP_OPTIONS_TARGET,
                        D2D1::PixelFormat(
                            DXGI_FORMAT_B8G8R8A8_UNORM,
                            D2D1_ALPHA_MODE_PREMULTIPLIED
                            ),
                        96.0,
                        96.0
                        ),
                        ld.bitmap.target() ); */

      if (FAILED(hr)) break;

      // device_context()->GetTarget(ld.otarget.target());

      ld.otarget  = device_context();
      ld.position = clip;
      ld.opacity  = opacity;

      dc->GetTarget(ld.bitmap.target());

      device_context(dc);

      dc->SetTransform(D2D1::Matrix3x2F::Translation(FLOAT(-clip.s.x),
                                                     FLOAT(-clip.s.y)));

      dc->BeginDraw();

      // Clear the bitmap with transparent white.
      dc->Clear(D2D1::ColorF(D2D1::ColorF::White, 0.0f));

      d2d_filter_graph_builder builder;
      builder.bitmap = ld.bitmap;
      builder.pdc    = dc;

      if (!filters(&builder))
        // if(!produce_filter_graph(v, element, filters, &builder))
        break;

      ld.effect = builder.tail;

      // layers_stack.push(ld);
      break;
    }
#endif

    if (opacity < 255 /* || ld.effect*/) {
      D2D1_SIZE_F szf = d2d::g2f(clip.size());

      HRESULT hr = render_target()->CreateLayer(&szf, ld.layer.target());

      D2D1_RECT_F rcf = d2d::g2f(clip);
      render_target()->PushLayer(
          D2D1::LayerParameters(rcf, NULL, D2D1_ANTIALIAS_MODE_PER_PRIMITIVE,
                                D2D1::IdentityMatrix(),
                                float(opacity) / 255.0f,
                                nullptr,
                                D2D1_LAYER_OPTIONS_INITIALIZE_FOR_CLEARTYPE),
          ld.layer);

      assert(SUCCEEDED(hr));
      hr = hr;
    } else {
      auto clip2d = d2d::g2f(clip);
      render_target()->PushAxisAlignedClip(clip2d, D2D1_ANTIALIAS_MODE_ALIASED);
    }

    layers_stack.push(ld);
  }

  void graphics::push_layer(const gool::path *clip, byte opacity) {
    layer_t ld;
    ld.opacity = opacity;

    // super::push_layer(clip,opacity);

    HRESULT hr = render_target()->CreateLayer(0, ld.layer.target());

    D2D1_RECT_F rcf = d2d::g2f(clip->bounds());

    render_target()->PushLayer(
        D2D1::LayerParameters(
            rcf, static_cast<const d2d::path *>(clip)->get_geometry(),
            D2D1_ANTIALIAS_MODE_PER_PRIMITIVE, D2D1::IdentityMatrix(),
            float(opacity) / 255.0f,
            nullptr,
            D2D1_LAYER_OPTIONS_INITIALIZE_FOR_CLEARTYPE),
        ld.layer);
    assert(SUCCEEDED(hr));
    hr = hr;

    layers_stack.push(ld);
  }

  void graphics::push_layer(const gool::bitmap *src_clip, bool draw1a,
                            byte opacity) {
    layer_t ld;
    ld.opacity = opacity;

    handle<bitmap> clip;

    if (draw1a)
      clip = src_clip;
    else {
      clip = new gool::bitmap(src_clip->dim(), true, true);
      clip->set_bits(src_clip->_pixels());
      clip->foreach_pixel(
          [](argb &p) { p = argb(0, 0, 0, byte(255) - p.alfa); });
    }

    // super::push_layer(clip,opacity);

    HRESULT hr = render_target()->CreateLayer(0, ld.layer.target());

    // D2D1_RECT_F rcf = d2d::g2f(clip->bounds());
    assert(SUCCEEDED(hr));

    if (FAILED(hr)) return;

    D2D1_BITMAP_BRUSH_PROPERTIES propertiesXClampYClamp =
        D2D1::BitmapBrushProperties(
            D2D1_EXTEND_MODE_CLAMP, D2D1_EXTEND_MODE_CLAMP,
            D2D1_BITMAP_INTERPOLATION_MODE_NEAREST_NEIGHBOR);

    ID2D1Bitmap *d2bmp = D2D1Bitmap(clip);
    assert(d2bmp);
    if (!d2bmp) return;
    com::asset<ID2D1BitmapBrush> pib;

    hr = render_target()->CreateBitmapBrush(d2bmp, propertiesXClampYClamp,
                                            pib.target());
    assert(SUCCEEDED(hr));
    hr = hr;

    render_target()->PushLayer(
        D2D1::LayerParameters(
            D2D1::InfiniteRect(), NULL, D2D1_ANTIALIAS_MODE_PER_PRIMITIVE,
            D2D1::IdentityMatrix(), float(opacity) / 255.0f, pib, D2D1_LAYER_OPTIONS_INITIALIZE_FOR_CLEARTYPE),
        ld.layer);

    layers_stack.push(ld);
  }

  void graphics::pop_layer() {

    assert(layers_stack.size());

    layer_t ld = layers_stack.pop();

    if (ld.layer)
      render_target()->PopLayer();
    else
      render_target()->PopAxisAlignedClip();

#ifdef USE_D2D_PLUS
    if (ld.otarget && ld.otarget != device_context()) {
      HRESULT hr = device_context()->EndDraw();
      assert(SUCCEEDED(hr));
      hr = hr;
      device_context(ld.otarget);
      if (ld.effect) {
        D2D1_POINT_2F pos;
        pos.x = FLOAT(ld.position.s.x);
        pos.y = FLOAT(ld.position.s.y);
        if (ld.effect)
          device_context()->DrawImage(ld.effect, pos);
        else
          device_context()->DrawImage(ld.bitmap, pos);
      }
    }
#endif
  }

  void graphics::translate(pointf pt) {
    D2D1::Matrix3x2F current;
    render_target()->GetTransform(&current);
    render_target()->SetTransform(D2D1::Matrix3x2F::Translation(pt.x, pt.y) *
                                  current);
  }

  void graphics::rotate(float radians, pointf center) {
    D2D1::Matrix3x2F current;
    render_target()->GetTransform(&current);
    radians = 360.f * radians / real_traits<float>::pi2(); // to degree

    D2D1::Matrix3x2F rotation;
    _D2D1MakeRotateMatrix()(radians, g2f(center), &rotation);
    render_target()->SetTransform(rotation * current);
  }

  void graphics::scale(sizef sz, pointf center) {
    D2D1::Matrix3x2F current;
    render_target()->GetTransform(&current);
    render_target()->SetTransform(
        D2D1::Matrix3x2F::Scale(sz.x, sz.y, g2f(center)) * current);
  }
  void graphics::skew(sizef sz, pointf center) {
    D2D1::Matrix3x2F current;
    render_target()->GetTransform(&current);

    D2D1::Matrix3x2F skew;
    _D2D1MakeSkewMatrix()(sz.x, sz.y, g2f(center), &skew);

    render_target()->SetTransform(skew * current);
  }

} // namespace d2d

