﻿#include "d2d.h"

#include <wincodec.h>

#include "tool/tool.h"
#include "win/win-application.h"
#include "win/win-delayload.h"

DLOADV(_D2D1CreateFactory, D2D1CreateFactory, d2d1.dll, 
            HRESULT (WINAPI *)( D2D1_FACTORY_TYPE factoryType, REFIID riid, CONST D2D1_FACTORY_OPTIONS *pFactoryOptions, void **ppIFactory ) );

DLOADV(_DWriteCreateFactory, DWriteCreateFactory, dwrite.dll, 
            HRESULT (WINAPI *)( DWRITE_FACTORY_TYPE factoryType, REFIID riid, IUnknown **ppIFactory ) );

DLOADV(_GetUserDefaultLocaleName, GetUserDefaultLocaleName, kernel32.dll, 
            int (WINAPI *)( LPWSTR lpLocaleNam, int cchLocaleName ) );

#ifdef USE_D2D_PLUS
DLOADV(_D3D11CreateDevice, D3D11CreateDevice, d3d11.dll,
  HRESULT(WINAPI *)(IDXGIAdapter *pAdapter, D3D_DRIVER_TYPE DriverType,
    HMODULE Software, UINT Flags,
    const D3D_FEATURE_LEVEL *pFeatureLevels,
    UINT FeatureLevels, UINT SDKVersion,
    ID3D11Device **       ppDevice,
    D3D_FEATURE_LEVEL *   pFeatureLevel,
    ID3D11DeviceContext **ppImmediateContext));

DLOADV(_DCompositionCreateDevice3, DCompositionCreateDevice3, dcomp.dll,
  HRESULT(WINAPI *)(
    IUnknown *renderingDevice,
    REFIID   iid,
    void     **dcompositionDevice));
#endif

namespace d2d
{
  using namespace tool;

    
  application::application()
  {
    //platform::setup_text_flow = &d2d::setup_text_flow;
    //platform::draw_glyph_runs = &d2d::draw_glyph_runs;
    //frame::init(true);
  }

  void application::clear_font_cache() {
    for(int n = 0; n < _fonts.size();++n)
      _fonts.elements()[n]->drop_cache();
    _fonts.clear();
  }

  void application::final_release() {
    super::final_release();

    p_fonts_collection = 0;
    font_loading_context.Shutdown();
    //else 
    // ?  CoUninitialize();

    _DW_factory = 0;
    if( normal_shutdown ) {
      _D2_factory = 0;
#ifdef USE_D2D_PLUS
      _D2_1_factory = 0;
#endif
    }
    else {
      _D2_factory.detach(); // that's 
#ifdef USE_D2D_PLUS
      _D2_1_factory.detach();
#endif
    }

  }

  application::~application() { 
  }

  application* app_factory()
  {
    auto_ptr<application> pa;  
    do
    {
      if(!_D2D1CreateFactory || !_DWriteCreateFactory)
        break;

      D2D1_FACTORY_OPTIONS options;
#if defined(USE_D2D_DEBUG) 
      options.debugLevel = D2D1_DEBUG_LEVEL_INFORMATION;
#else 
      options.debugLevel = D2D1_DEBUG_LEVEL_NONE;
#endif

      HRESULT hr = S_OK;

      com::asset<IDWriteFactory> _dw_factory;
#if defined(USE_D2D_PLUS)
      com::asset<IDWriteFactory4> _dw_factory4;

      // Create a shared DirectWrite factory.
      hr = _DWriteCreateFactory()(DWRITE_FACTORY_TYPE_SHARED, __uuidof(IDWriteFactory4), reinterpret_cast<IUnknown**>(_dw_factory4.target()));
      
      if (FAILED(hr)) {
        hr = _DWriteCreateFactory()(DWRITE_FACTORY_TYPE_SHARED, __uuidof(IDWriteFactory), reinterpret_cast<IUnknown**>(_dw_factory.target()));
        if (FAILED(hr)) break;
        pa = auto_ptr<application>(new application());
        hr = _dw_factory->GetSystemFontCollection(pa->p_fonts_collection.target());
        if (FAILED(hr)) break;
      }
      else {
        _dw_factory = _dw_factory4;
        application_plus* pp = new application_plus();
        pp->_dw_factory4 = _dw_factory4;
        pa = auto_ptr<application>(pp);
        hr = _dw_factory4->GetSystemFontCollection(pa->p_fonts_collection.target());
        if (FAILED(hr)) break;

      }
#else 
      hr = _DWriteCreateFactory()(DWRITE_FACTORY_TYPE_SHARED, __uuidof(IDWriteFactory), reinterpret_cast<IUnknown**>(_dw_factory.target()));
      if (FAILED(hr)) break;
      pa = auto_ptr<application>(new application());
      hr = _dw_factory->GetSystemFontCollection(pa->p_fonts_collection.target());
      if (FAILED(hr)) break;
#endif
      pa->_DW_factory = _dw_factory;

      hr = _D2D1CreateFactory()(D2D1_FACTORY_TYPE_MULTI_THREADED, __uuidof(ID2D1Factory) ,
        &options,
        (void**)pa->_D2_factory.target());
      if (FAILED(hr)) break;

#if defined(USE_D2D_PLUS)
      hr = pa->_D2_factory->QueryInterface<ID2D1Factory1>(pa->_D2_1_factory.target());
      if(SUCCEEDED(hr)) {
        hr = pa->_D2_factory->QueryInterface<ID2D1Factory2>(pa->_D2_1_factory2.target());
      }
#endif

      return pa.release();

    } while(false);
    return nullptr;
  }

#ifdef USE_D2D_PLUS
  ID3D11Device* application::device_3d()
  {
    if (_device_3D)
      return _device_3D;

    if (!_D3D11CreateDevice) 
      return nullptr;

    //D3D_FEATURE_LEVEL feature_levels[] = {
    //  D3D_FEATURE_LEVEL_11_1, D3D_FEATURE_LEVEL_11_0, D3D_FEATURE_LEVEL_10_1,
    //  D3D_FEATURE_LEVEL_10_0, D3D_FEATURE_LEVEL_9_3,  D3D_FEATURE_LEVEL_9_2,
    //  D3D_FEATURE_LEVEL_9_1 };

    // This flag adds support for surfaces with a different color channel
    // ordering than the API default. It is required for compatibility with
    // Direct2D.
    UINT creation_flags = D3D11_CREATE_DEVICE_BGRA_SUPPORT;

    // Create Direct3D device and context
    
    HRESULT hr;

    if (requested_gfx_layer == HARDWARE_GRAPHICS)
      hr = _D3D11CreateDevice()(
        nullptr, D3D_DRIVER_TYPE_HARDWARE, 0, creation_flags, nullptr,0, //feature_levels, items_in(feature_levels), 
        D3D11_SDK_VERSION, _device_3D.target(), nullptr, nullptr);
    else
      hr = DXGI_ERROR_UNSUPPORTED;

    if (FAILED(hr)) {
      hr = _D3D11CreateDevice()(
        nullptr, D3D_DRIVER_TYPE_WARP, 0, creation_flags, nullptr, 0, //feature_levels, items_in(feature_levels), 
        D3D11_SDK_VERSION, _device_3D.target(), nullptr, nullptr);
      requested_gfx_layer = WARP_GRAPHICS;
    }

    if (FAILED(hr)) return nullptr;

    return _device_3D;

  }
  IDCompositionDesktopDevice* application::dcomp_device()
  {
    if (_device_dcomp)
      return _device_dcomp;

    if (!_DCompositionCreateDevice3)
      return nullptr;

    com::asset<ID3D11Device> d3d = device_3d();
    if (!d3d)
      return nullptr;

    com::asset<ID2D1Factory2> d2d_1_factory = d2_1_factory2();
    if (!d2d_1_factory)
      return nullptr;

    HRESULT hr;

    com::asset<IDXGIDevice3> deviceX;
        
    hr = d3d.QueryInterface(deviceX); if (FAILED(hr)) return nullptr;
    
    com::asset<ID2D1Device> device2D;

    hr = d2d_1_factory->CreateDevice(deviceX, device2D.target()); if (FAILED(hr)) return nullptr;
    hr = _DCompositionCreateDevice3()(deviceX, __uuidof(_device_dcomp),reinterpret_cast<void **>(_device_dcomp.target()));
    if (FAILED(hr)) return nullptr;

    return _device_dcomp;
  }

#endif

  gool::font* application::create_font(const ustring& name
                    , float size
                    , uint weight
                    , bool italic
                    , FONT_RENDERING  mode)
  {
    d2d::font key;
    key.name = name;
    key.size = size;
    key.weight = weight;
    key.italic = italic;
    key.mode = mode;
    
    d2d::font* pf = _fonts.intern(key);
    if( pf ) init_font(pf);
    return pf;
  }
 

 struct font_list_scanner
 {
   wchar_t                           localeName[LOCALE_NAME_MAX_LENGTH];
   int                               defaultLocaleSuccess;
   com::asset<IDWriteFontCollection> fonts_collection;

   font_list_scanner():defaultLocaleSuccess(0) {
      memzero(localeName);
      if(_GetUserDefaultLocaleName )
        defaultLocaleSuccess = _GetUserDefaultLocaleName()(localeName, LOCALE_NAME_MAX_LENGTH);
      fonts_collection = d2d::fonts_collection();
   }

   bool check_name(wchars name_to_check)
   {
     HRESULT hr;
     
     UINT32 index = 0;
     BOOL   exists = FALSE;
     
     hr = fonts_collection->FindFamilyName(ustring(name_to_check).c_str(), &index, &exists);
     return !!exists;

/*  uint familyCount = fonts_collection->GetFontFamilyCount();
      for (uint i = 0; i < familyCount; ++i) - No idea why I decided to do lookup this way initially:
      {
        com::asset<IDWriteFontFamily> pFontFamily;

        // Get the font family.
        hr = fonts_collection->GetFontFamily(i, pFontFamily.target());

        com::asset<IDWriteLocalizedStrings> pFamilyNames;

        // Get a list of localized strings for the family name.
        if (SUCCEEDED(hr))
        {
            hr = pFontFamily->GetFamilyNames(pFamilyNames.target());
        }

        uint loc_index = 0;
        uint en_index = 0;
        BOOL exists = false;
        
        if (SUCCEEDED(hr))
        {
            hr = E_FAIL;
            // If the default locale is returned, find that locale name, otherwise use "en-us".
            if (defaultLocaleSuccess)
            {
                hr = pFamilyNames->FindLocaleName(localeName, &loc_index, &exists);
            }
            if (FAILED(hr) || !exists) // if the above find did not find a match, retry with US English
            {
                hr = pFamilyNames->FindLocaleName(L"en-us", &en_index, &exists);
            }
            
            // we are using only english font names here:   
            //hr = pFamilyNames->FindLocaleName(L"en-us", &index, &exists);
        }
        
        // If the specified locale doesn't exist, select the first on the list.
        if (!exists)
            en_index = 0;

        wchar_t loc_name[256] = {0};
        // Get the family name.
        if (!SUCCEEDED(hr))
          continue;

        if(loc_index != UINT_MAX) {
          hr = pFamilyNames->GetString(loc_index, loc_name, 255);
          if ( SUCCEEDED(hr) && icmp(name_to_check, chars_of(loc_name)) )
            return true;
        }

        wchar_t en_name[256] = {0};
        if( en_index != UINT_MAX ) {
          hr = pFamilyNames->GetString(en_index, en_name, 255);
          if ( SUCCEEDED(hr) && icmp(name_to_check, chars_of(en_name)) )
            return true;
        }

      }*/
      return false;
   };


   
   bool operator()(wchars name_to_check)
   {
     critical_section _( application::guard ); 

     fonts_collection = d2d::fonts_collection();
     if( check_name(name_to_check) )
         return true;

     return false;
   };

 };

 tool::ustring application::get_supported_font(
         const tool::ustring& family_name_list /*comma separated list*/
       , int  weight
       , bool italic
       , com::asset<IDWriteFontCollection>& found_in /*out*/)
 {
   //HRESULT hr;
   critical_section _(guard); 

   font_list_scanner scan_font_list;

   if( !scan_font_list.fonts_collection )
   {
     return family_name_list;
   }

   wtokens tz( family_name_list,WCHARS(",;"));
   wchars name;

   while(tz.next(name))
   {
     name = trim(name);
     if( icmp(name,WCHARS("monospace"))  )
       return get_supported_font(WCHARS("consolas,courier new"),weight,italic,found_in); 
     else if( icmp(name,WCHARS("sans-serif")) )
       return get_supported_font(WCHARS("segoe ui,arial"),weight,italic,found_in); 
     else if( icmp(name,WCHARS("serif")) )
       //return WCHARS("times new roman"); 
       return get_supported_font(WCHARS("times new roman"),weight,italic,found_in); 
     else if( icmp(name,WCHARS("cursive")) )
       return get_supported_font(WCHARS("vladimir script,freestyle script"),weight,italic,found_in); 
     else if( icmp(name,WCHARS("fantasy")) )
       return get_supported_font(WCHARS("jokeman,comic sans ms"),weight,italic,found_in); 
     else if( name.like(L"system*")) {
       found_in = fonts_collection();
       return gool::system_font_name(name);
     }
     memory_font_variant* fv = this->closest_memory_font_collection(name, weight, italic);
     if (fv) {
       found_in = fv->coll;
       return name;
     }
     if(scan_font_list(name))
     {
       found_in = scan_font_list.fonts_collection;
       return name;
     }
   }
   found_in = fonts_collection();
   return  gool::system_font_name();
 }

  bool application::init_font(gool::font* gf)
  {
    d2d::font* pf = static_cast<d2d::font*>(gf);

    if(pf->dw_font()) return true;

    critical_section _( guard); 

    com::asset<IDWriteFontCollection> font_collection = d2d::fonts_collection();

    com::asset<IDWriteFontFamily>     font_family;

    if( !font_collection )
    {
      return false;
    }

    HRESULT hr = S_OK;

    uint font_index  = 0;
    BOOL font_exists = false;
    ustring font_name = get_supported_font(pf->name,int(pf->weight),pf->italic, font_collection);
    
    if( font_collection == d2d::fonts_collection() ) {
    
      hr = font_collection->FindFamilyName(font_name, &font_index, &font_exists);
      if (!font_exists)
      {
          //view::debug_printf(html::OT_DOM, html::OS_WARNING, "font %S does not exist\n", font_name.c_str());
          // If the given font does not exist, take what we can get
          // (displaying something instead nothing), choosing the foremost
          // font in the collection.
          font_index = 0;
      }
    } else if(!font_collection)
      hr = E_FAIL;

    if (SUCCEEDED(hr))
    {
      hr = font_collection->GetFontFamily(font_index, font_family.target());
    }

    com::asset<IDWriteFont> dw_font;

    if (SUCCEEDED(hr))
    {
      hr = font_family->GetFirstMatchingFont(
              (DWRITE_FONT_WEIGHT)pf->weight,
              DWRITE_FONT_STRETCH_NORMAL,
              (pf->italic? DWRITE_FONT_STYLE_ITALIC: DWRITE_FONT_STYLE_NORMAL),
              dw_font.target());
      if (SUCCEEDED(hr))
      {
        pf->dw_font(dw_font);
        //com::asset<IDWriteFontFace>  font_face;
        //dw_font->CreateFontFace(font_face.target());
        //pf->dw_font_face(font_face);
        //view::debug_printf(html::OT_DOM, html::OS_INFO, "font used %S\n", font_name.c_str());
      }
    }
    return SUCCEEDED(hr);
  }

  wchars get_fallback_list(WRITING_SCRIPT script, wchars lang_id, bool serif )
  {
     switch(script)
     {
        default:
        case WS_UNKNOWN: 
        case WS_CYRILLIC: return serif
                            ? WCHARS("Times New Roman,!")
                            : WCHARS("Segoe UI,Arial,!");
        case WS_HANZI:
          if(lang_id.starts_with(WCHARS("ja")))
            return serif
               ? WCHARS("MS Mincho,MS明朝,!")
               : WCHARS("Meiryo UI,メイリオ UI,Yu Gothic UI,!");
          else if(lang_id.ends_with(WCHARS("tw")) || lang_id.ends_with(WCHARS("hant")))
            return serif  // traditional
               ? WCHARS("MingLiU,名流,MingLiU-ExtB,名流-ExtB,!")
               : WCHARS("Microsoft JhengHei,微軟正黑體,MingLiU-ExtB,名流-ExtB,!");
          else 
            return serif  // simplified
               ? WCHARS("NSimSun,SimSun-ExtB,!")
               : WCHARS("Microsoft YaHei,微软雅黑,SimSun-ExtB,!");
        case WS_KANA:
          return serif
             ? WCHARS("MS Mincho,MS明朝,!")
             : WCHARS("Meiryo UI,メイリオ UI,Yu Gothic UI,!");
        case WS_ARABIC:
          return serif
             ? WCHARS("Arabic Typesetting,!")
             : WCHARS("Arabic Simplified,!");
        case WS_HEBREW:
          return serif
             ? WCHARS("Narkisim,!")
             : WCHARS("Miriam,!");
        case WS_HANGUL:
          return serif
             ? WCHARS("Batang,!")
             : WCHARS("Malgun Gothic,!");
     }      
  }

  ustring application::get_font_family_for_ucode(uint char_code)
  {
    // last resort, quite heavy 

    com::asset<IDWriteFontCollection> font_collection = d2d::fonts_collection();
    com::asset<IDWriteFontFamily>     font_family;
    uint32 cnt = font_collection->GetFontFamilyCount();

    for (uint32 i = 0; i < cnt; ++i)
    {
      font_collection->GetFontFamily(i, font_family.target());
      com::asset<IDWriteLocalizedStrings> pFamilyNames;
      font_family->GetFamilyNames(pFamilyNames.target());
      uint en_index = 0;
      BOOL exists;
      HRESULT hr = pFamilyNames->FindLocaleName(L"en-us", &en_index, &exists);
      ustring name;
      if (en_index != UINT_MAX)
      {

        wchar_t en_name[256] = { 0 };
        hr = pFamilyNames->GetString(en_index, en_name, 255);

        com::asset<IDWriteFont> font;

        HRESULT hr = font_family->GetFirstMatchingFont(
          DWRITE_FONT_WEIGHT_NORMAL,
          DWRITE_FONT_STRETCH_NORMAL,
          DWRITE_FONT_STYLE_NORMAL,
          font.target());

        if (SUCCEEDED(hr)) {
          BOOL exists = FALSE;
          font->HasCharacter(char_code, &exists);
          if (exists)
            return en_name;
        }
      }
    }
    return ustring();
  }

  bool application::each_font_family( function<bool(wchars name)> cb, function<void(ucode,ucode)> cb_ranges)
  {
    com::asset<IDWriteFontCollection> font_collection = d2d::fonts_collection();
    com::asset<IDWriteFontFamily>     font_family;
    uint32 cnt = font_collection->GetFontFamilyCount();

    if( cb_ranges ) {
      for( uint32 i = 0; i < cnt; ++i ) 
      {
        font_collection->GetFontFamily(i,font_family.target());
        com::asset<IDWriteLocalizedStrings> pFamilyNames;
        font_family->GetFamilyNames(pFamilyNames.target());
        uint en_index = 0;
        BOOL exists;
        HRESULT hr = pFamilyNames->FindLocaleName(L"en-us", &en_index, &exists);
        ustring name;
        if( en_index != UINT_MAX ) 
        {

            wchar_t en_name[256] = {0};
            hr = pFamilyNames->GetString(en_index, en_name, 255);
            if( !cb( chars_of(en_name)))
              break;

            com::asset<IDWriteFont> font;

            HRESULT hr = font_family->GetFirstMatchingFont(
              DWRITE_FONT_WEIGHT_NORMAL,
              DWRITE_FONT_STRETCH_NORMAL,
              DWRITE_FONT_STYLE_NORMAL,
              font.target());

#if defined(USE_D2D_PLUS)
            com::asset<IDWriteFont1> font_ex;

            if( SUCCEEDED(hr) && SUCCEEDED(font.QueryInterface( font_ex ))) {
              DWRITE_UNICODE_RANGE ranges[1000];
              uint32 count = 0;
              hr = font_ex->GetUnicodeRanges(items_in(ranges),ranges,&count);
              if( SUCCEEDED(hr) && count ) {
                for( uint32 n = 0; n < count; ++n ) {
                  cb_ranges( ranges[n].first, ranges[n].last );
                }
              }
            }
#else 
            hr = hr;
#endif

        }
      }
    }
    else {
      for( uint32 i = 0; i < cnt; ++i ) 
      {
        font_collection->GetFontFamily(i,font_family.target());
        com::asset<IDWriteLocalizedStrings> pFamilyNames;
        font_family->GetFamilyNames(pFamilyNames.target());
        uint en_index = 0;
        BOOL exists;
        HRESULT hr = pFamilyNames->FindLocaleName(L"en-us", &en_index, &exists);
        ustring name;
        if( en_index != UINT_MAX ) 
        {
            wchar_t en_name[256] = {0};
            hr = pFamilyNames->GetString(en_index, en_name, 255);
            if( !cb( chars_of(en_name)))
              break;
        }
      }
    }
    return true;
  }
  
  /*const gool::font* find_font_for_char(uint char_code,const gool::font* proto) 
  {
    com::asset<IDWriteFontCollection> font_collection = d2d::fonts_collection();
    com::asset<IDWriteFontFamily>     font_family;
    com::asset<IDWriteFont>           font;

    uint32 cnt = font_collection->GetFontFamilyCount();
    BOOL has;
    for( uint32 i = 0; i < cnt; ++i ) {
      font_collection->GetFontFamily(i,font_family.target());
      HRESULT hr = font_family->GetFirstMatchingFont(
              (DWRITE_FONT_WEIGHT)proto->weight,
              DWRITE_FONT_STRETCH_NORMAL,
              proto->italic? DWRITE_FONT_STYLE_ITALIC: DWRITE_FONT_STYLE_NORMAL,
              font.target());
      if (SUCCEEDED(hr)) {
          font->HasCharacter(char_code,&has);
          if( has ) 
          {
            com::asset<IDWriteLocalizedStrings> pFamilyNames;
            font_family->GetFamilyNames(pFamilyNames.target());
            uint en_index = 0;
            BOOL exists;
            hr = pFamilyNames->FindLocaleName(L"en-us", &en_index, &exists);
            ustring name;
            if( en_index != UINT_MAX ) {
              wchar_t en_name[256] = {0};
              hr = pFamilyNames->GetString(en_index, en_name, 255);
              name = en_name;
            }
            gool::font* pf = mswin::app()->create_font(name,proto->size,proto->weight,proto->italic,proto->mode);
            static_cast<d2d::font*>(pf)->dw_font(font);
            return pf;
          }
      }
    }
    return proto;
  }*/


#pragma warning( push )
#pragma warning( disable : 4102 ) // warning C4102: 'START_ELLIPSIS' : unreferenced label - bug in MS compiler.

  bool application::get_used_font(
       tool::handle<gool::font>& pf,
       tool::wchars family_name_list, 
       const gool::font* proto,
       tool::wchars lang_id, 
       WRITING_SCRIPT script, uint char_code)
  {
    critical_section _(guard); 

    font_list_scanner scan_font_list;
  
    if( !scan_font_list.fonts_collection )
    {
      return false;
    }

    uint iterations = 0;

    std::function<bool(tool::wchars family_name_list)> lookup;

    lookup = [&](tool::wchars family_name_list) -> bool
    {
      ++iterations;
    
      if(script == WS_UNKNOWN && char_code != 0)
        script = writing_script(char_code);
        
      wtokens tz( family_name_list,WCHARS(",;") );
      wchars name;

      while(tz.next(name))
      {
        name = trim(name);
     
        if( icmp(name,WCHARS("monospace")) )
          return lookup(WCHARS("consolas,courier new,!"));

        else if( icmp(name,WCHARS("sans-serif")) )
          return lookup(get_fallback_list(script,lang_id,false));

        else if( icmp(name,WCHARS("serif")) )
          return lookup(get_fallback_list(script,lang_id,true));

        else if( icmp(name,WCHARS("cursive")) )
          return lookup(WCHARS("vladimir script,freestyle script,sans-serif"));

        else if( icmp(name,WCHARS("fantasy")) )
          return lookup(WCHARS("jokeman,comic sans ms,sans-serif"));

        else if( name.like(L"system*") )
        {
          ustring sys_name = gool::system_font_name(name) + ",!";
          return lookup(sys_name);
        }
        else if( name == WCHARS("!") ) // misericorde
        {
          ustring sys_name = gool::system_font_name(name);
          pf = app()->create_font(sys_name,proto->size,proto->weight,proto->italic,proto->mode);
#if defined(USE_D2D_PLUS)
          if( !pf->has_glyph_for( char_code ))
          {
            ustring fam = get_font_family_for_ucode(char_code);
            if( fam.is_defined() ) {
              pf = app()->create_font(fam,proto->size,proto->weight,proto->italic,proto->mode);
              assert( pf->has_glyph_for(char_code) );
            } else {
              pf = proto;
            }
            //pf = find_font_for_char(char_code,proto);
            return true;
          }
#endif
          return true;
        }
              
        com::asset<IDWriteFontCollection> font_collection;
        com::asset<IDWriteFontFamily>     font_family;
        com::asset<IDWriteFont>           font;
        uint font_index  = 0;

        HRESULT hr;

        do {
          memory_font_variant* fv = closest_memory_font_collection(name, proto->weight, proto->italic);
          if (fv) {
            font_collection = fv->coll;
            font_index = 0;
            hr = font_collection->GetFontFamily(font_index, font_family.target());
            if (SUCCEEDED(hr))
              break;
          }

          if(scan_font_list(name)) {
            font_collection = scan_font_list.fonts_collection;
            BOOL font_exists = false;
            hr = font_collection->FindFamilyName(ustring(name), &font_index, &font_exists);
            if(!font_exists)
               font_index = 0;
            hr = font_collection->GetFontFamily(font_index, font_family.target());
          }
        }
        while (false);

        if (font_family )
        {
          hr = font_family->GetFirstMatchingFont(
                  (DWRITE_FONT_WEIGHT)proto->weight,
                  DWRITE_FONT_STRETCH_NORMAL,
                  proto->italic? DWRITE_FONT_STYLE_ITALIC: DWRITE_FONT_STYLE_NORMAL,
                  font.target());
          if (SUCCEEDED(hr))
          {
            if(char_code && iterations < 3)
            {
              BOOL has = false;
              font->HasCharacter(char_code,&has);
              if( has )
                goto FOUND;
            } else {
              goto FOUND;
            }

          }
          continue;
FOUND:
          pf = app()->create_font(name,proto->size,proto->weight,proto->italic,proto->mode);
          static_cast<d2d::font*>(pf.ptr())->dw_font(font);
          return true;
        }

      }
      bool r = lookup(get_fallback_list(script,lang_id,false));
      return r;
    };

    return lookup(family_name_list);
  }

#pragma warning( pop )

#if !defined(WINDOWLESS)
  handle<html::view> application::create_frame(const window_params& params)
  {
    return handle<html::view>(mswin::frame<d2d::window>::construct(params).ptr());
  }
  handle<html::view> application::create_dialog(const window_params& params)
  {
    return handle<html::view>(mswin::dialog<d2d::window>::construct(params).ptr());
  }

  gool::application*  application::printing_app() {
    return app();
  }
#endif

}

