#include "api-defs.h"
#include "tool/tool.h"
#include "gool/gool.h"
#include "sdk-headers.h"

#if defined(SCITERJS)
  #include "xdomjs/xcontext.h"
#endif // defined(SCITERJS)

#define VREF(pval) ((tool::value *)pval)

UINT SCAPI ValueInit_api(VALUE *pval) {
  assert_static(sizeof(VALUE) == sizeof(tool::value));

  if (!pval) return 1;
  memset(pval, 0, sizeof(VALUE));
  return 0;
}

UINT SCAPI ValueClear_api(VALUE *pval) {
  if (!pval) return 1;
  VREF(pval)->clear();
  return 0;
}

UINT SCAPI ValueCopy_api(VALUE *pdst, const VALUE *psrc) {
  if (!pdst || !psrc) return 1;
  *VREF(pdst) = *VREF(psrc);
  return 0;
}

UINT SCAPI ValueCompare_api(const VALUE *pval1, const VALUE *pval2) {
  if (!pval1 || !pval2) return 1;
  return *VREF(pval1) == *VREF(pval2) ? HV_OK_TRUE : HV_OK;
}

UINT SCAPI ValueType_api(const VALUE *pval, UINT *pType, UINT *pUnits) {
  if (!pval || !pType) return 1;
  *pType = VREF(pval)->type();
  if (pUnits) *pUnits = VREF(pval)->units();
  return 0;
}

UINT SCAPI ValueIsolate_api(VALUE *pval) {
  VREF(pval)->isolate();
  return HV_OK;
}

UINT SCAPI ValueStringData_api(const VALUE *pval, LPCWSTR *pChars,
                               UINT *pNumChars) {
  if (!pval || !pChars) return 1;
  tool::wchars wc;
  if (VREF(pval)->is_function())
    wc = VREF(pval)->get_function()->name;
  else if (VREF(pval)->is_string())
    wc = VREF(pval)->get_chars();
  else
    return 2;
  *pChars = LPCWSTR(wc.start);
  if (pNumChars) *pNumChars = UINT(wc.length);
  return 0;
}

UINT SCAPI ValueStringDataSet_api(VALUE *pval, LPCWSTR chars, UINT numChars,
                                  UINT units) {
  if (!pval || !chars) return 1;
  static_assert(sizeof(VALUE) == sizeof(tool::value),
                "tool::value and VALUE sizeof mismatch");
  tool::ustring us((const wchar *)chars, numChars);
  *VREF(pval) = tool::value(us, units);
  return 0;
}

UINT SCAPI ValueIntData_api(const VALUE *pval, INT *pData) {
  if (!pval || !pData) return 1;

  if (VREF(pval)->is_number() || VREF(pval)->is_bool())
    *pData = VREF(pval)->get_int();
  else if (VREF(pval)->is_color()) {
    gool::color_v cv = *VREF(pval);
    *pData = cv.to_color();
  } else
    return 2;
  return 0;
}

UINT SCAPI ValueIntDataSet_api(VALUE *pval, INT data, UINT type, UINT units) {
  if (!pval) return 1;
  switch (type) {
  case tool::value::t_int: *VREF(pval) = tool::value(int(data), units); break;
  case tool::value::t_color:
    *VREF(pval) = gool::color_v((gool::color)data).to_value();
    break;
  case tool::value::t_bool: *VREF(pval) = tool::value(data != 0, units); break;
  case tool::value::t_length:
    *VREF(pval) = tool::value::make_length(data, units);
    break;
  case tool::value::t_array: *VREF(pval) = tool::value::make_array(data); break;
  case tool::value::t_map: *VREF(pval) = tool::value::make_map(); break;
  default: return 2;
  }
  return 0;
}

UINT SCAPI ValueBinaryData_api(const VALUE *pval, LPCBYTE *pBytes,
                               UINT *pnBytes) {
  if (!pval || !pBytes || !pnBytes) return 1;
  switch (VREF(pval)->type()) {
  case tool::value::t_bytes: {
    tool::bytes bs = VREF(pval)->get_bytes();
    *pBytes        = bs.start;
    *pnBytes       = UINT(bs.length);
    return 0;
  } break;
  case tool::value::t_object_proxy: {
    tool::object_proxy *pp = VREF(pval)->get_proxy();
    void *              pv = 0;
    if (pp->get_user_data(&pv)) {
      *pBytes  = (byte *)pv;
      *pnBytes = UINT(sizeof(pv));
      return 0;
    }
  } break;
  case tool::value::t_resource: {
    tool::resource *pp = VREF(pval)->get_resource();
    *pBytes  = (byte *)pp;
    *pnBytes = sizeof(pp);
    return 0;
  } break;
  }
  return 2;
}

UINT SCAPI ValueBinaryDataSet_api(VALUE *pval, LPCBYTE pBytes, UINT nBytes,
                                  UINT type, UINT units) {
  if (!pval) return 1;
  switch (type) {
  case tool::value::t_bytes:
    *VREF(pval) = tool::value::make_bytes(tool::bytes(pBytes, nBytes), units);
    return 0;
  case tool::value::t_object_proxy: {
    if (VREF(pval)->is_proxy()) {
      tool::object_proxy *pp = VREF(pval)->get_proxy();
      if (pp->set_user_data((void *)pBytes)) return 0;
    }
  } break;
  }
  return 2;
}

UINT SCAPI ValueInt64Data_api(const VALUE *pval, INT64 *pData) {
  if (!pval || !pData) return 1;
  if (VREF(pval)->is_asset())
    *pData = (uint_ptr)VREF(pval)->get_asset();
  else if (VREF(pval)->is_date())
    *pData = VREF(pval)->get_int64();
  else if (VREF(pval)->is_currency())
    *pData = VREF(pval)->get_int64();
  else 
    return 2;
  return 0;
}

UINT SCAPI ValueInt64DataSet_api(VALUE *pval, INT64 data, UINT type,
                                 UINT units) {
  if (!pval) return 1;
  switch (type) {
  case tool::value::t_asset:
    *VREF(pval) = tool::value::wrap_asset(reinterpret_cast<som_asset_t*>(data));
    return 0;
  case tool::value::t_currency:
    *VREF(pval) = tool::value::make_currency(data);
    break;
  case tool::value::t_date:
    *VREF(pval) = tool::value::make_date(data, units ? tool::date_time::DT_UTC : 0);
    break;
  default: return 2;
  }
  return 0;
}

UINT SCAPI ValueFloatData_api(const VALUE *pval, double *pData) {
  if (!pval || !pData) return 1;
  if (VREF(pval)->is_int() || VREF(pval)->is_double() ||
      VREF(pval)->is_length())
    *pData = VREF(pval)->get_double();
  else if (VREF(pval)->is_angle())
    *pData = VREF(pval)->get_angle();
  else if (VREF(pval)->is_duration())
    *pData = VREF(pval)->get_duration();
  else
    return 2;
  return 0;
}

UINT SCAPI ValueFloatDataSet_api(VALUE *pval, double data, UINT type,
                                 UINT units) {
  if (!pval) return 1;
  switch (type) {
  case tool::value::t_double: *VREF(pval) = tool::value(data, units); break;
  case tool::value::t_length: *VREF(pval) = tool::value::make_length(data, units); break;
  case tool::value::t_angle: *VREF(pval) = tool::value::make_angle(data); break;
  case tool::value::t_duration: *VREF(pval) = tool::value::make_duration(data); break;
  default: return 2;
  }
  return 0;
}

UINT SCAPI ValueElementsCount_api(const VALUE *pval, INT *pn) {
  if (!pval || !pn) return 1;
  switch (VREF(pval)->type()) {
  case tool::value::t_function:
    *pn = VREF(pval)->get_function()->params.size();
    break;
  case tool::value::t_map: *pn = VREF(pval)->get_map()->params.size(); break;
  case tool::value::t_array:
    *pn = VREF(pval)->get_array()->elements.size();
    break;
  case tool::value::t_object_proxy:
    *pn = VREF(pval)->get_proxy()->size();
    break;
  default: return 2;
  }
  return 0;
}

UINT SCAPI ValueNthElementValue_api(const VALUE *pval, INT n, VALUE *pretval) {
  if (!pval || !pretval) return 1;
  n = abs(n);
  switch (VREF(pval)->type()) {
  case tool::value::t_function:
    if (n >= VREF(pval)->get_function()->params.size())
      *VREF(pretval) = tool::value();
    else
      *VREF(pretval) = VREF(pval)->get_function()->params.value(n);
    break;
  case tool::value::t_map:
    if (n >= VREF(pval)->get_map()->params.size())
      *VREF(pretval) = tool::value();
    else
      *VREF(pretval) = VREF(pval)->get_map()->params.value(n);
    break;
  case tool::value::t_array:
    if (n >= VREF(pval)->get_array()->elements.size())
      *VREF(pretval) = tool::value();
    else
      *VREF(pretval) = VREF(pval)->get_array()->elements[n];
    break;
  case tool::value::t_object_proxy:
    *VREF(pretval) = VREF(pval)->get_proxy()->get_by_index(n);
    break;
  default: return 2;
  }
  return 0;
}

UINT SCAPI ValueNthElementValueSet_api(VALUE *pval, INT n,
                                       const VALUE *psetval) {
  if (!pval || !psetval) return 1;
  n = abs(n);
  switch (VREF(pval)->type()) {
  case tool::value::t_function:
    if (n >= VREF(pval)->get_function()->params.size())
      VREF(pval)->get_function()->params.push(*VREF(psetval));
    else
      VREF(pval)->get_function()->params.value(n) = *VREF(psetval);
    break;
  case tool::value::t_map:
    if (n >= VREF(pval)->get_map()->params.size())
      VREF(pval)->get_map()->params.push(*VREF(psetval));
    else
      VREF(pval)->get_map()->params.value(n) = *VREF(psetval);
    break;

  default:
    VREF(pval)->clear();
    *VREF(pval) = tool::value::make_array(n + 1);
    // fall through
  case tool::value::t_array:
    if (n >= VREF(pval)->get_array()->elements.size())
      VREF(pval)->get_array()->elements.push(*VREF(psetval));
    else
      VREF(pval)->get_array()->elements[n] = *VREF(psetval);
    break;
  case tool::value::t_object_proxy:
    if (!VREF(pval)->get_proxy()->set_by_index(n, *VREF(psetval))) return 2;
    break;
  }
  return 0;
}

// UINT SCAPI ValueEnum( const VALUE* pval, CALLBACK )

UINT SCAPI ValueNthElementKey_api(const VALUE *pval, INT n, VALUE *pretval) {
  if (!pval || !pretval) return 1;
  n = abs(n);
  switch (VREF(pval)->type()) {
  case tool::value::t_function:
    if (n >= VREF(pval)->get_function()->params.size())
      *VREF(pretval) = tool::value();
    else
      *VREF(pretval) = VREF(pval)->get_function()->params.key(n);
    break;
  case tool::value::t_map:
    if (n >= VREF(pval)->get_map()->params.size())
      *VREF(pretval) = tool::value();
    else
      *VREF(pretval) = VREF(pval)->get_map()->params.key(n);
    break;
  default: return 2;
  }
  return 0;
}

UINT SCAPI ValueEnumElements_api(const VALUE *pval, KeyValueCallback *penum,
                                 LPVOID param) {
  if (!pval || !penum) return 1;

  auto thunk = [&](const tool::value &k, const tool::value &v) -> bool {
    return 0 != penum(param, (VALUE *)&k, (VALUE *)&v);
  };

  if (VREF(pval)->visit(thunk)) return 0;
  return 2;
}

UINT SCAPI ValueSetValueToKey_api(VALUE *pval, const VALUE *pkey,
                                  const VALUE *psetval) {
  if (!pval || !pkey) return 1;
  switch (VREF(pval)->type()) {
  default: *VREF(pval) = tool::value::make_map(new tool::map_value());
  case tool::value::t_map: {
    // if(!VREF(pkey)->is_string()) - this limitation is too strict.
    //  return 2;
    tool::value key = *VREF(pkey);
    if (!VREF(psetval) || VREF(psetval)->is_undefined())
      VREF(pval)->get_map()->params.remove(key);
    else
      VREF(pval)->get_map()->params[key] = *VREF(psetval);
  } break;
  case tool::value::t_function: {
    if (!VREF(pkey)->is_string()) return 2;
    tool::value key = *VREF(pkey);
    if (!VREF(psetval) || VREF(psetval)->is_undefined())
      VREF(pval)->get_function()->params.remove(key);
    else
      VREF(pval)->get_function()->params[key] = *VREF(psetval);
  } break;
  case tool::value::t_object_proxy:
    if (!VREF(pval)->get_proxy()->set_by_key(*VREF(pkey), *VREF(psetval)))
      return 2;
  }
  return 0;
}

UINT SCAPI ValueGetValueOfKey_api(const VALUE *pval, const VALUE *pkey,
                                  VALUE *pretval) {
  if (!pval || !pkey || !pretval) return 1;
  tool::value v;
  switch (VREF(pval)->type()) {
  case tool::value::t_map: {
    // if(!VREF(pkey)->is_string()) // this limitation is too strict.
    //  return 2;
    tool::value key = *VREF(pkey);
    VREF(pval)->get_map()->params.find(key, v);
  } break;
  case tool::value::t_function: {
    if (!VREF(pkey)->is_string()) return 2;
    tool::value key = *VREF(pkey);
    VREF(pval)->get_function()->params.find(key, v);
  } break;
  case tool::value::t_object_proxy:
    v = VREF(pval)->get_proxy()->get_by_key(*VREF(pkey));
    break;
  default: return 2;
  }
  *VREF(pretval) = v;
  return 0;
}

UINT SCAPI ValueToString_api(VALUE *pval, UINT how) {
  if (!pval) return 1;
  tool::ustring s;
  switch (how) {
  case CVT_SIMPLE: s = VREF(pval)->to_string(); break;
  case CVT_JSON_LITERAL:
    s = tool::xjson::emit(*VREF(pval), tool::xjson::JSON, false);
    break;
  case CVT_JSON_MAP:
    s = tool::xjson::emit(*VREF(pval), tool::xjson::JSON, true);
    break;
  case CVT_XJSON_LITERAL:
    s = tool::xjson::emit(*VREF(pval), tool::xjson::XJSON, false);
    break;
  }
  *VREF(pval) = tool::value(s);
  return 0;
}

UINT SCAPI ValueFromString_api(VALUE *pval, LPCWSTR str, UINT strLength,
                               UINT how) {
  tool::wchars input((const wchar *)str, strLength);
  switch (how) {
  case CVT_SIMPLE:
    *VREF(pval) = tool::value::parse(tool::ustring(input));
    return 0;
  case CVT_JSON_LITERAL:
  case CVT_XJSON_LITERAL:
    *VREF(pval) = tool::xjson::parse(input, false);
    return strLength - UINT(input.length); // return number of chars left
                                           // unparsed. So 0 is ok
  case CVT_JSON_MAP:
    *VREF(pval) = tool::xjson::parse(input, true);
    return strLength - UINT(input.length); // return number of chars left
                                           // unparsed. So 0 is ok
  }
  return strLength;
}

UINT SCAPI ValueInvoke_api(const VALUE *pval, VALUE *pthis, UINT argc,
                           const VALUE *argv, VALUE *pretval, LPCWSTR url) {
  if (!pval || !pthis || !pretval) return 1;
  if (argc && !argv) return 1;

  tool::value v;
  switch (VREF(pval)->type()) {
  /*
  case tool::value::t_function:
    {
      if(!VREF(pkey)->is_string())
        return 2;
      tool::ustring key = VREF(pkey)->get(L"");
      VREF(pval)->get_function()->params.find(key,v);
    } break;
  */
  case tool::value::t_resource: {
    tool::native_functor_holder *pf = VREF(pval)->get_native_functor();
    if (!pf) return 2;
    v = pf->call(argc, (const tool::value *)argv);
    break;
  }
  case tool::value::t_object_proxy:
    v = VREF(pval)->get_proxy()->invoke(*VREF(pthis), argc,
                                        (const tool::value *)argv);
    break;
  default: return 2;
  }
  *VREF(pretval) = v;
  return 0;
}

struct ext_native_functor_holder : public tool::native_functor_holder {
  ext_native_functor_holder()
      : _invoke(nullptr), _release(nullptr), _tag(nullptr) {}
  ext_native_functor_holder(NATIVE_FUNCTOR_INVOKE * pinvoke,
                            NATIVE_FUNCTOR_RELEASE *prelease, VOID *tag)
      : _invoke(pinvoke), _release(prelease), _tag(tag) {}
  virtual ~ext_native_functor_holder() {
    if (_release) _release(_tag);
  }
  virtual tool::value call(uint argc, const tool::value *argv) {
    if (_invoke) {
      tool::value r;
      _invoke(_tag, argc, (const VALUE *)argv, (VALUE *)&r);
      return r;
    }
    return tool::value();
  }
  NATIVE_FUNCTOR_INVOKE * _invoke;
  NATIVE_FUNCTOR_RELEASE *_release;
  VOID *                  _tag;
};

UINT SCAPI ValueNativeFunctorSet_api(VALUE *                 pval,
                                     NATIVE_FUNCTOR_INVOKE * pinvoke,
                                     NATIVE_FUNCTOR_RELEASE *prelease,
                                     VOID *                  tag) {
  if (!pval || !pinvoke) return 1;

  VREF(pval)->set_resource(
      new ext_native_functor_holder(pinvoke, prelease, tag));

  return 0;
}

SBOOL SCAPI ValueIsNativeFunctor_api(const VALUE *pval) {
  return VREF(pval)->is_native_functor();
}

UINT64 SCAPI SciterAtomValue_api(const char* name)
{
  if (name)
#if defined(SCITER)
    return tis::CsSymbolOf(name);
#elif defined(SCITERJS)
    return qjs::hruntime::current().symbol_of(tool::chars_of(name));
#endif
  else
    return 0;
}

SBOOL SCAPI SciterAtomNameCB_api(UINT64 atomv, LPCSTR_RECEIVER* rcv, LPVOID rcv_param)
{
#if defined(SCITER)
  if (atomv && tis::CsSymbolP(atomv)) {
    tool::string tag = u8::cvt(tis::CsSymbolName(atomv));
    rcv(tag.c_str(), UINT(tag.length()), rcv_param);
    return TRUE;
  }
#elif defined(SCITERJS)
  if (atomv) {
    tool::string tag = qjs::hruntime::current().symbol_name_a(uint(atomv));
    rcv(tag.c_str(), UINT(tag.length()), rcv_param);
    return TRUE;
  }
#endif
  return FALSE;
}

